/* *********************************************************************** *
 *                                                                         *
 * xpress_objects.cpp                                                      *
 *                                                                         *
 * C++ API for Xpress-Optimizer                                            *
 *                                                                         *
 * (c) Copyright Fair Isaac Corporation 2024-2025. All rights reserved     *
 * For FICO Xpress Optimizer v45.01.02                                     *
 *                                                                         *
 * *********************************************************************** */

#ifndef XPRESS_OBJECTS_H
#define XPRESS_DO_NOT_INCLUDE_CXX_SOURCE
#include "xpress_objects.hpp"
#endif
#include "xpress.hpp"
#include "xpress_maps.hpp"
#include "xpress_objects.hpp"

;
xpress::objects::SymbolicEvalStack::SymbolicEvalStack(int capacity)
    : stack(std::vector<Expression>(capacity)), pos(0) {}

auto xpress::objects::SymbolicEvalStack::push(Expression e) -> void {
  if (pos >= xpress::toInt(stack.size()))
    throw std::invalid_argument("malformed formula");
  stack[pos++] = e;
}

auto xpress::objects::SymbolicEvalStack::pop() -> xpress::objects::Expression {
  if (pos <= 0)
    throw std::invalid_argument("malformed formula");
  return stack[--pos];
}

auto xpress::objects::SymbolicEvalStack::peek() -> xpress::objects::Expression {
  if (pos <= 0)
    throw std::invalid_argument("malformed formula");
  return stack[pos - 1];
}

auto xpress::objects::SymbolicEvalStack::size() -> int { return pos; }

xpress::objects::Token::Token(int type, double value)
    : type(type), value(value) {}

xpress::objects::Token::Token(int type) : Token(type, (double)type) {}

xpress::objects::Token::Token() : Token(XPRS_TOK_EOF) {}

auto xpress::objects::Token::equals(Token const &other) const -> bool {
  return type == other.type && value == other.value;
}

auto xpress::objects::Token::compareTo(Token const &other) const -> int {

  int delta = type - other.type;
  if (delta != 0)
    return delta;
  else {
    if (value < other.value)
      return -1;
    else if (value > other.value)
      return 1;
    else
      return 0;
  }
}

auto xpress::objects::Token::getHashCode() const -> std::size_t {
  return xpress::combineHash(type, value);
}

auto xpress::objects::Token::operator==(Token const &other) const -> bool {
  return compareTo(other) == 0;
}
auto xpress::objects::Token::operator!=(Token const &other) const -> bool {
  return compareTo(other) != 0;
}

;

;

;

;

;

;

;

;

;

;

;

;

;

;

auto xpress::objects::Token::constant(double value) -> xpress::objects::Token {
  return Token(XPRS_TOK_CON, value);
}

auto xpress::objects::Token::column(int col) -> xpress::objects::Token {
  if (col < 0)
    throw std::invalid_argument("illegal column value");
  return Token(XPRS_TOK_COL, col);
}

auto xpress::objects::Token::function(int id) -> xpress::objects::Token {
  if (id < 1)
    throw std::invalid_argument("illegal user function value");
  return Token(XPRS_TOK_FUN, id);
}

auto xpress::objects::Token::internalFunction(int id)
    -> xpress::objects::Token {
  return of(XPRS_TOK_IFUN, id);
}

auto xpress::objects::Token::of(int type,
                                double value) -> xpress::objects::Token {
  switch (type) {
  case XPRS_TOK_EOF:
    return EOFORMULA;
  case XPRS_TOK_LB:
    return LB;
  case XPRS_TOK_RB:
    return RB;
  case XPRS_TOK_OP:
    if ((int)value != value)
      throw std::invalid_argument("unknown operator");
    switch ((int)value) {
    case XPRS_OP_UMINUS:
      return UMINUS;
    case XPRS_OP_EXPONENT:
      return EXPONENT;
    case XPRS_OP_MULTIPLY:
      return MULTIPLY;
    case XPRS_OP_DIVIDE:
      return DIVIDE;
    case XPRS_OP_PLUS:
      return PLUS;
    case XPRS_OP_MINUS:
      return MINUS;
    default:
      throw std::invalid_argument("unknown operator");
    }
  case XPRS_TOK_DEL:
    if ((int)value != value)
      throw std::invalid_argument("unknown delimiter");
    switch ((int)value) {
    case XPRS_DEL_COMMA:
      return COMMA;
    case XPRS_DEL_COLON:
      return COLON;
    default:
      throw std::invalid_argument("unknown delimiter");
    }
  case XPRS_TOK_CON:
    return Token(XPRS_TOK_CON, value);
  case XPRS_TOK_COL:
    if ((int)value != value || value < 0)
      throw std::invalid_argument("illegal column value");
    return Token(XPRS_TOK_COL, value);
  case XPRS_TOK_FUN:
    if ((int)value != value || value < 1)
      throw std::invalid_argument("illegal user function value");
    return Token(XPRS_TOK_FUN, value);
  case XPRS_TOK_IFUN:
    if ((int)value != value)
      throw std::invalid_argument("unknown internal function");
    switch ((int)value) {
    case XPRS_IFUN_LOG10:
      return LOG10;
    case XPRS_IFUN_LN:
      return LN;
    case XPRS_IFUN_EXP:
      return EXP;
    case XPRS_IFUN_ABS:
      return ABS;
    case XPRS_IFUN_SQRT:
      return SQRT;
    case XPRS_IFUN_SIN:
      return SIN;
    case XPRS_IFUN_COS:
      return COS;
    case XPRS_IFUN_TAN:
      return TAN;
    case XPRS_IFUN_ARCSIN:
      return ARCSIN;
    case XPRS_IFUN_ARCCOS:
      return ARCCOS;
    case XPRS_IFUN_ARCTAN:
      return ARCTAN;
    case XPRS_IFUN_MIN:
      return MIN;
    case XPRS_IFUN_MAX:
      return MAX;
    case XPRS_IFUN_PWL:
      return PWL;
    case XPRS_IFUN_SUM:
      return SUM;
    case XPRS_IFUN_PROD:
      return PROD;
    case XPRS_IFUN_SIGN:
      return SIGN;
    default:
      throw std::invalid_argument("unknown internal function");
    }
  default:
    throw std::invalid_argument("unknown token type");
  }
}

auto xpress::objects::Token::toString() const -> std::string {
  switch (type) {
  case XPRS_TOK_EOF:
    return "EOFORMULA";
  case XPRS_TOK_LB:
    return "LB";
  case XPRS_TOK_RB:
    return "RB";
  case XPRS_TOK_OP:
    switch ((int)value) {
    case XPRS_OP_UMINUS:
      return "UMINUS";
    case XPRS_OP_EXPONENT:
      return "EXPONENT";
    case XPRS_OP_MULTIPLY:
      return "MULTIPLY";
    case XPRS_OP_DIVIDE:
      return "DIVIDE";
    case XPRS_OP_PLUS:
      return "PLUS";
    case XPRS_OP_MINUS:
      return "MINUS";
    }
    break;
  case XPRS_TOK_DEL:
    switch ((int)value) {
    case XPRS_DEL_COMMA:
      return "COMMA";
    case XPRS_DEL_COLON:
      return "COLON";
    }
    break;
  case XPRS_TOK_CON:
    return InternalUtils::floatToString(value);
  case XPRS_TOK_COL:
    if (std::round(value) == value)
      return xpress::concatStrings("column", "[", (int)std::round(value), "]");
    else
      return xpress::concatStrings("column", "[",
                                   InternalUtils::floatToString(value), "]");
  case XPRS_TOK_FUN:
    if (std::round(value) == value)
      return xpress::concatStrings("func", "[", (int)std::round(value), "]");
    else
      return xpress::concatStrings("func", "[",
                                   InternalUtils::floatToString(value), "]");
  case XPRS_TOK_IFUN:
    switch ((int)value) {
    case XPRS_IFUN_LOG10:
      return "LOG10";
    case XPRS_IFUN_LN:
      return "LN";
    case XPRS_IFUN_EXP:
      return "EXP";
    case XPRS_IFUN_ABS:
      return "ABS";
    case XPRS_IFUN_SQRT:
      return "SQRT";
    case XPRS_IFUN_SIN:
      return "SIN";
    case XPRS_IFUN_COS:
      return "COS";
    case XPRS_IFUN_TAN:
      return "TAN";
    case XPRS_IFUN_ARCSIN:
      return "ARCSIN";
    case XPRS_IFUN_ARCCOS:
      return "ARCCOS";
    case XPRS_IFUN_ARCTAN:
      return "ARCTAN";
    case XPRS_IFUN_MIN:
      return "MIN";
    case XPRS_IFUN_MAX:
      return "MAX";
    case XPRS_IFUN_PWL:
      return "PWL";
    case XPRS_IFUN_SUM:
      return "SUM";
    case XPRS_IFUN_PROD:
      return "PROD";
    case XPRS_IFUN_SIGN:
      return "SIGN";
    }
    break;
  }
  return xpress::concatStrings("(", type, ",",
                               InternalUtils::floatToString(value), ")");
}

;

;

template <typename C>
xpress::objects::ConstraintDefinitionImplementation<
    C>::ConstraintDefinitionImplementation(std::optional<std::string> name)
    : name(name), extracted(false) {}

template <typename C>
auto xpress::objects::ConstraintDefinitionImplementation<C>::checkModify()
    -> void {
  if (extracted)
    throw std::runtime_error("cannot modify constraint after extraction");
}

template <typename C>
auto xpress::objects::ConstraintDefinitionImplementation<C>::getName() const
    -> std::optional<std::string> {
  return name;
}

template <typename C>
auto xpress::objects::ConstraintDefinitionImplementation<C>::setConstraintName(
    std::optional<std::string> newName)
    -> xpress::objects::ConstraintDefinitionImplementation<C> & {
  checkModify();
  this->name = newName;
  return *this;
}

;

template <typename C>
auto xpress::objects::ConstraintDefinitionImplementation<C>::create(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  if (extracted)
    throw std::runtime_error("constraint can be extracted only once");
  createInternal(creator);
  extracted = true;
}

;

auto xpress::objects::IIS::getConstraints() const
    -> std::vector<IISConstraint> {
  return constraints;
}

auto xpress::objects::IIS::getVariables() const -> std::vector<IISVariable> {
  return variables;
}

template <typename T> auto xpress::objects::IIS::get_begin(T const &t) -> auto {
  using std::begin;
  return begin(t);
}
template <typename T> auto xpress::objects::IIS::get_end(T const &t) -> auto {
  using std::end;
  return end(t);
}

xpress::objects::IIS::IIS(
    xpress::SizedArray<IISConstraint const> const &constraints,
    xpress::SizedArray<IISVariable const> const &variables)
    : constraints(get_begin(constraints), get_end(constraints)),
      variables(get_begin(variables), get_end(variables)){}

;

;

;

;

;
;
template <typename T>
xpress::objects::IndexHandler<T>::IndexHandler(int attribute)
    : attribute(attribute), serial(0), objects({}), synchronizedMutex(){};

template <typename T>
auto xpress::objects::IndexHandler<T>::objectForIndex(XpressProblem *prob,
                                                      int index) const -> T {
  std::lock_guard<std::recursive_mutex> synchronized(synchronizedMutex);

  int count = prob->getIntAttrib(attribute);
  if (index < 0 || index >= count)
    throw NotInProblemException("invalid index", index);
  if (xpress::toInt(objects.size()) < count)
    objects.resize(count, (T) nullptr);
  T i = objects[index];
  if (!i) {
    i = makeInstance(prob, serial++, index);
    objects[index] = i;
  }
  return i;
}

template <typename T>
auto xpress::objects::IndexHandler<T>::fillObjects(
    XpressProblem *prob, int first, int last,
    std::vector<T> &objs) const -> void {
  std::lock_guard<std::recursive_mutex> synchronized(synchronizedMutex);

  int count = prob->getIntAttrib(attribute);
  if (last < first)
    return;
  if (first < 0 || first >= count)
    throw NotInProblemException("invalid index", first);
  if (last < 0 || last >= count)
    throw NotInProblemException("invalid index", last);
  if (xpress::toInt(objects.size()) < count)
    objects.resize(count, (T) nullptr);
  for (int i = 0; i < last - first + 1; ++i) {
    int index = first + i;
    T o = objects[index];
    if (!o) {
      o = makeInstance(prob, serial++, index);
      objects[index] = o;
    }
    objs[i] = o;
  }
}

;

template <typename T>
auto xpress::objects::IndexHandler<T>::deleteIndices(
    XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  std::lock_guard<std::recursive_mutex> synchronized(synchronizedMutex);

  if ((!indices) || count == 0)
    return;
  std::vector<int> sorted(0);
  int origCount = 0;
  if (true) {

    origCount = prob->getIntAttrib(attribute);
    for (int i = 0; i < count; ++i) {
      if (indices[i] < 0 || indices[i] >= origCount)
        throw NotInProblemException("invalid index", indices[i]);
    }

    sorted = xpress::copyOfArray(indices, count);
  }

  delIndices(prob, count, indices);

  if (xpress::toInt(sorted.size()) > 0) {
    int shift = 0;
    int next = 0;
    std::sort(sorted.begin(), sorted.end());
    int outIndex = sorted[0];
    for (int inIndex = sorted[0];
         inIndex < origCount && inIndex < xpress::toInt(objects.size());
         ++inIndex) {
      T i = objects[inIndex];
      if (next < count && inIndex == sorted[next]) {
        ++shift;
        ++next;
        if (!(!i)) {

          i.markDeleted();
          assert(i.getIndex() == -1);
        }
      } else {
        if (!(!i)) {
          i.updateIndex(-shift);
          assert(i.getIndex() == outIndex);
          objects[outIndex] = i;
          objects[inIndex] = nullptr;
          ++outIndex;
        }
      }
    }
    while (outIndex < xpress::toInt(objects.size())) {
      objects[outIndex] = nullptr;
      ++outIndex;
    }
  }
}
template <typename T>
template <typename O>
auto xpress::objects::IndexHandler<T>::deleteObjects(
    XpressProblem *prob, xpress::SizedArray<O const> const &objs) -> void {
  std::lock_guard<std::recursive_mutex> synchronized(synchronizedMutex);

  if (xpress::toInt(objs.size()) == 0)
    return;

  std::vector<int> indices(xpress::toInt(objs.size()));
  int next = 0;
  for (auto &i : objs) {
    if (i.getProblem() != prob)
      throw NotInProblemException(
          "cannot delete object that is not in this problem");
    indices[next++] = i.getIndex();
  }
  deleteIndices(prob, xpress::toInt(indices.size()), indices);
}

template <typename T> auto xpress::objects::IndexHandler<T>::drop() -> void {
  std::lock_guard<std::recursive_mutex> synchronized(synchronizedMutex);

  for (auto &t : objects) {
    if (!(!t)) {
      t.markDeleted();
      assert(t.getIndex() == -1);
    }
  }
  objects.clear();
}

auto xpress::objects::LinQuadUtils::index4Var(
    xpress::objects::XpressProblem *prob, xpress::objects::Variable v) -> int {
  return v.getIndexForProb(prob);
}

auto xpress::objects::LinQuadUtils::index4Row(
    xpress::objects::XpressProblem *prob,
    xpress::objects::Inequality r) -> int {
  return r.getIndexForProb(prob);
}

auto xpress::objects::LinQuadUtils::unifyConstantKey(
    xpress::objects::Variable v) -> xpress::objects::Variable {
  return (!v) ? xpress::objects::XpressProblem::NULL_VARIABLE : v;
}

auto xpress::objects::LinQuadUtils::isVariablePairOrdered(
    xpress::objects::Variable v1, xpress::objects::Variable v2) -> bool {
  if (XpressProblem::isNullVariable(v1))

    return XpressProblem::isNullVariable(v2);
  else if (XpressProblem::isNullVariable(v2))
    return true;
  else if (v1.getIndex() <= v2.getIndex())
    return true;
  else
    return false;
}

auto xpress::objects::LinQuadUtils::isLinear(Variable v1, Variable v2) -> bool {
  bool c1 = XpressProblem::isNullVariable(v1);
  bool c2 = XpressProblem::isNullVariable(v2);
  return c1 != c2;
}

auto xpress::objects::LinQuadUtils::isConstant(Variable x) -> bool {
  return XpressProblem::isNullVariable(x);
}

auto xpress::objects::LinQuadUtils::isConstant(Variable v1,
                                               Variable v2) -> bool {
  return XpressProblem::isNullVariable(v1) && XpressProblem::isNullVariable(v2);
}

auto xpress::objects::LinQuadUtils::getQConstantKey()
    -> xpress::objects::QPair const & {
  static QPair qConstantKey(XpressProblem::NULL_VARIABLE,
                            XpressProblem::NULL_VARIABLE);
  return qConstantKey;
}

;
;

xpress::objects::NotInProblemException::NotInProblemException(
    std::string message)
    : xpress::XPRSException(message), index(std::numeric_limits<int>::min()) {}

xpress::objects::NotInProblemException::NotInProblemException(
    std::string message, int index)
    : xpress::XPRSException(message), index(index){}

;
;

template <typename A, typename I, typename IIsIntegral, typename Func0,
          typename Func0IsInvocable>
auto xpress::objects::utils::sum(I const &count, Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms(count);
  for (I i = 0; i < count; ++i)
    terms[i] = makeTerm(i);
  return SumExpression(terms);
}

template <typename Strm0, typename A, typename T, typename Func0,
          typename Strm0IsStream, typename Func0IsInvocable>
auto xpress::objects::utils::sum(Strm0 const &data, Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (auto &d : data)
    terms.push_back(makeTerm(d));
  return SumExpression(terms);
}

template <typename Strm0, typename T, typename Strm0IsStream>
auto xpress::objects::utils::sum(Strm0 const &terms)
    -> xpress::objects::SumExpression {
  std::vector<Expression> t(terms.begin(), terms.end());
  return SumExpression(t);
}

template <typename... MORETERMS, typename RestrictPack>
auto xpress::objects::utils::sum(Expression firstTerm, MORETERMS... moreTerms)
    -> xpress::objects::SumExpression {
  std::vector<Expression> more = xpress::packToArray<Expression>(moreTerms...);
  if (xpress::toInt(more.size()) == 0)
    return SumExpression(std::vector<Expression>({firstTerm}));
  else {
    std::vector<Expression> terms(xpress::toInt(more.size()) + 1);
    terms[0] = firstTerm;
    xpress::arrayCopy(more, 0, terms, 1, xpress::toInt(more.size()));
    return SumExpression(terms);
  }
}

auto xpress::objects::utils::uminus(Expression value)
    -> xpress::objects::Expression {
  return value.mul(-1.0);
}

auto xpress::objects::utils::plus(Expression augend, Expression addend)
    -> xpress::objects::Expression {
  return augend.plus(addend);
}

auto xpress::objects::utils::plus(double augend, Expression addend)
    -> xpress::objects::Expression {
  return ConstantExpression(augend).plus(addend);
}

auto xpress::objects::utils::plus(Expression augend, double addend)
    -> xpress::objects::Expression {
  return augend.plus(addend);
}

auto xpress::objects::utils::minus(Expression minuend, Expression subtrahend)
    -> xpress::objects::Expression {
  return minuend.minus(subtrahend);
}

auto xpress::objects::utils::minus(double minuend, Expression subtrahend)
    -> xpress::objects::Expression {
  return ConstantExpression(minuend).minus(subtrahend);
}

auto xpress::objects::utils::minus(Expression minuend, double subtrahend)
    -> xpress::objects::Expression {
  return minuend.minus(subtrahend);
}

auto xpress::objects::utils::div(Expression dividend, Expression divisor)
    -> xpress::objects::Expression {
  return dividend.div(divisor);
}

auto xpress::objects::utils::div(double dividend, Expression divisor)
    -> xpress::objects::Expression {
  return ConstantExpression(dividend).div(divisor);
}

auto xpress::objects::utils::div(Expression dividend, double divisor)
    -> xpress::objects::Expression {
  return dividend.div(divisor);
}

auto xpress::objects::utils::mul(Expression factor1, Expression factor2)
    -> xpress::objects::Expression {
  return factor1.mul(factor2);
}

auto xpress::objects::utils::mul(double factor1, Expression factor2)
    -> xpress::objects::Expression {
  return ConstantExpression(factor1).mul(factor2);
}

auto xpress::objects::utils::mul(Expression factor1, double factor2)
    -> xpress::objects::Expression {
  return factor1.mul(factor2);
}

auto xpress::objects::utils::pow(Expression basis, Expression exp)
    -> xpress::objects::BinaryExpression {
  return BinaryExpression(basis, BinaryExpression::Operator::Exponent, exp);
}

auto xpress::objects::utils::pow(double basis, Expression exp)
    -> xpress::objects::BinaryExpression {
  return pow(ConstantExpression(basis), exp);
}

auto xpress::objects::utils::pow(Expression basis, double exp)
    -> xpress::objects::BinaryExpression {
  return pow(basis, ConstantExpression(exp));
}

auto xpress::objects::utils::log10(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Log10, value);
}

auto xpress::objects::utils::ln(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Ln, value);
}

auto xpress::objects::utils::exp(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Exp, value);
}

auto xpress::objects::utils::abs(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Abs, value);
}

auto xpress::objects::utils::sqrt(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Sqrt, value);
}

auto xpress::objects::utils::sin(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Sin, value);
}

auto xpress::objects::utils::cos(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Cos, value);
}

auto xpress::objects::utils::tan(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Tan, value);
}

auto xpress::objects::utils::arcsin(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::ArcSin, value);
}

auto xpress::objects::utils::arccos(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::ArcCos, value);
}

auto xpress::objects::utils::arctan(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::ArcTan, value);
}

auto xpress::objects::utils::sign(Expression value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Sign, value);
}

template <typename... VALUE, typename RestrictPack>
auto xpress::objects::utils::min(VALUE... value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Min,
      xpress::packToArray<Expression>(value...));
}

auto xpress::objects::utils::min(
    xpress::SizedArray<Expression const> const &value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Min, value);
}

template <typename... VALUE, typename RestrictPack>
auto xpress::objects::utils::max(VALUE... value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Max,
      xpress::packToArray<Expression>(value...));
}

auto xpress::objects::utils::max(
    xpress::SizedArray<Expression const> const &value)
    -> xpress::objects::InternalFunctionExpression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Max, value);
}

template <typename... VALUE, typename RestrictPack>
auto xpress::objects::utils::prod(VALUE... value)
    -> xpress::objects::Expression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Prod,
      xpress::packToArray<Expression>(value...));
}

auto xpress::objects::utils::prod(xpress::SizedArray<Expression const> const
                                      &value) -> xpress::objects::Expression {
  return InternalFunctionExpression(
      InternalFunctionExpression::FunctionType::Prod, value);
}

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::utils::scalarProduct(Strm0 const &variables,
                                           Strm1 const &coefs)
    -> xpress::objects::LinExpression {
  return scalarProduct(LinExpression::create(), variables, coefs);
}

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::utils::scalarProduct(
    LinExpression result, Strm0 const &variables,
    Strm1 const &coefs) -> xpress::objects::LinExpression {
  using std::begin;
  using std::end;
  auto vCurrent = begin(variables);
  auto vEnd = end(variables);
  using std::begin;
  using std::end;
  auto coefCurrent = begin(coefs);
  auto coefEnd = end(coefs);
  while (vCurrent != vEnd && coefCurrent != coefEnd) {
    auto const &v = *vCurrent;
    auto const &coef = *coefCurrent;

    if (LinQuadUtils::isConstant(v))
      result.addConstant(coef);
    else
      result.addTerm(v, coef);

    ++vCurrent;
    ++coefCurrent;
  }
  if (vCurrent != vEnd || coefCurrent != coefEnd)
    throw std::invalid_argument("input data has different dimensions");
  return result;
}

template <typename K>
auto xpress::objects::utils::scalarProduct(
    std::unordered_map<K, Variable> variables,
    std::unordered_map<K, double> coefs) -> xpress::objects::LinExpression {
  return scalarProduct(LinExpression::create(), variables, coefs);
}

template <typename K>
auto xpress::objects::utils::scalarProduct(
    LinExpression result, std::unordered_map<K, Variable> variables,
    std::unordered_map<K, double> coefs) -> xpress::objects::LinExpression {
  for (auto &e : variables) {
    K key = e.first;
    double coef = 0.0;
    coef = (coefs.find(key) == coefs.end()) ? 0.0 : coefs[key];
    if (coef != 0.0) {
      Variable v = e.second;
      if (LinQuadUtils::isConstant(v))
        result.addConstant(coef);
      else
        result.addTerm(v, coef);
    }
  }

  result.normalizeTerms();
  return result;
}

template <typename Strm0, typename Strm1, typename Strm2,
          typename Strm0IsStream, typename Strm1IsStream,
          typename Strm2IsStream>
auto xpress::objects::utils::scalarProduct(
    Strm0 const &variables1, Strm1 const &variables2,
    Strm2 const &coefs) -> xpress::objects::QuadExpression {
  return scalarProduct(QuadExpression::create(), variables1, variables2, coefs);
}

template <typename Strm0, typename Strm1, typename Strm2,
          typename Strm0IsStream, typename Strm1IsStream,
          typename Strm2IsStream>
auto xpress::objects::utils::scalarProduct(
    QuadExpression result, Strm0 const &variables1, Strm1 const &variables2,
    Strm2 const &coefs) -> xpress::objects::QuadExpression {
  using std::begin;
  using std::end;
  auto v1Current = begin(variables1);
  auto v1End = end(variables1);
  using std::begin;
  using std::end;
  auto v2Current = begin(variables2);
  auto v2End = end(variables2);
  using std::begin;
  using std::end;
  auto coefCurrent = begin(coefs);
  auto coefEnd = end(coefs);
  while (v1Current != v1End && v2Current != v2End && coefCurrent != coefEnd) {
    auto const &v1 = *v1Current;
    auto const &v2 = *v2Current;
    auto const &coef = *coefCurrent;

    if (LinQuadUtils::isConstant(v1, v2))
      result.addConstant(coef);
    else
      result.addTerm(v1, v2, coef);

    ++v1Current;
    ++v2Current;
    ++coefCurrent;
  }
  if (v1Current != v1End || v2Current != v2End || coefCurrent != coefEnd)
    throw std::invalid_argument("input data has different dimensions");
  return result;
}

template <typename A, typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral, typename Func0, typename Func0IsInvocable>
auto xpress::objects::utils::sum(C1 count1, C2 count2, Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (C1 i1 = 0; i1 < count1; ++i1)
    for (C2 i2 = 0; i2 < count2; ++i2)
      terms.push_back(makeTerm(i1, i2));
  return SumExpression(terms);
}

template <typename Iter0, typename Iter1, typename A, typename T1, typename T2,
          typename Func0, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Func0IsInvocable>
auto xpress::objects::utils::sum(Iter0 const &data1, Iter1 const &data2,
                                 Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (auto &i1 : data1)
    for (auto &i2 : data2)
      terms.push_back(makeTerm(i1, i2));
  return SumExpression(terms);
}

template <typename A, typename C1, typename C2, typename C3,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename Func0, typename Func0IsInvocable>
auto xpress::objects::utils::sum(C1 count1, C2 count2, C3 count3,
                                 Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (C1 i1 = 0; i1 < count1; ++i1)
    for (C2 i2 = 0; i2 < count2; ++i2)
      for (C3 i3 = 0; i3 < count3; ++i3)
        terms.push_back(makeTerm(i1, i2, i3));
  return SumExpression(terms);
}

template <typename Iter0, typename Iter1, typename Iter2, typename A,
          typename T1, typename T2, typename T3, typename Func0,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Func0IsInvocable>
auto xpress::objects::utils::sum(Iter0 const &data1, Iter1 const &data2,
                                 Iter2 const &data3, Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (auto &i1 : data1)
    for (auto &i2 : data2)
      for (auto &i3 : data3)
        terms.push_back(makeTerm(i1, i2, i3));
  return SumExpression(terms);
}

template <typename A, typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename Func0, typename Func0IsInvocable>
auto xpress::objects::utils::sum(C1 count1, C2 count2, C3 count3, C4 count4,
                                 Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (C1 i1 = 0; i1 < count1; ++i1)
    for (C2 i2 = 0; i2 < count2; ++i2)
      for (C3 i3 = 0; i3 < count3; ++i3)
        for (C4 i4 = 0; i4 < count4; ++i4)
          terms.push_back(makeTerm(i1, i2, i3, i4));
  return SumExpression(terms);
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename A, typename T1, typename T2, typename T3, typename T4,
          typename Func0, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Func0IsInvocable>
auto xpress::objects::utils::sum(
    Iter0 const &data1, Iter1 const &data2, Iter2 const &data3,
    Iter3 const &data4, Func0 makeTerm) -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (auto &i1 : data1)
    for (auto &i2 : data2)
      for (auto &i3 : data3)
        for (auto &i4 : data4)
          terms.push_back(makeTerm(i1, i2, i3, i4));
  return SumExpression(terms);
}

template <typename A, typename C1, typename C2, typename C3, typename C4,
          typename C5, typename C1IsIntegral, typename C2IsIntegral,
          typename C3IsIntegral, typename C4IsIntegral, typename C5IsIntegral,
          typename Func0, typename Func0IsInvocable>
auto xpress::objects::utils::sum(C1 count1, C2 count2, C3 count3, C4 count4,
                                 C5 count5, Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (C1 i1 = 0; i1 < count1; ++i1)
    for (C2 i2 = 0; i2 < count2; ++i2)
      for (C3 i3 = 0; i3 < count3; ++i3)
        for (C4 i4 = 0; i4 < count4; ++i4)
          for (C5 i5 = 0; i5 < count5; ++i5)
            terms.push_back(makeTerm(i1, i2, i3, i4, i5));
  return SumExpression(terms);
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename A, typename T1, typename T2, typename T3,
          typename T4, typename T5, typename Func0, typename Iter0IsIterable,
          typename Iter1IsIterable, typename Iter2IsIterable,
          typename Iter3IsIterable, typename Iter4IsIterable,
          typename Func0IsInvocable>
auto xpress::objects::utils::sum(Iter0 const &data1, Iter1 const &data2,
                                 Iter2 const &data3, Iter3 const &data4,
                                 Iter4 const &data5, Func0 makeTerm)
    -> xpress::objects::SumExpression {
  std::vector<Expression> terms;
  for (auto &i1 : data1)
    for (auto &i2 : data2)
      for (auto &i3 : data3)
        for (auto &i4 : data4)
          for (auto &i5 : data5)
            terms.push_back(makeTerm(i1, i2, i3, i4, i5));
  return SumExpression(terms);
}

;

;

;

;

;

;

;

;

auto xpress::objects::ExpressionImplementation::uminus() const
    -> xpress::objects::Expression {
  return UnaryExpression(UnaryExpression::Operator::UMinus, this);
}

auto xpress::objects::ExpressionImplementation::plus(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  return BinaryExpression(this, BinaryExpression::Operator::Plus, arg);
}

auto xpress::objects::ExpressionImplementation::plus(double arg) const
    -> xpress::objects::Expression {
  return plus(ConstantExpression(arg));
}

auto xpress::objects::ExpressionImplementation::minus(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  return BinaryExpression(this, BinaryExpression::Operator::Minus, arg);
}

auto xpress::objects::ExpressionImplementation::minus(double arg) const
    -> xpress::objects::Expression {
  return minus(ConstantExpression(arg));
}

auto xpress::objects::ExpressionImplementation::mul(Expression arg) const
    -> xpress::objects::Expression {
  return BinaryExpression(this, BinaryExpression::Operator::Multiply, arg);
}

auto xpress::objects::ExpressionImplementation::mul(double arg) const
    -> xpress::objects::Expression {
  return mul(ConstantExpression(arg));
}

auto xpress::objects::ExpressionImplementation::div(Expression arg) const
    -> xpress::objects::Expression {
  return BinaryExpression(this, BinaryExpression::Operator::Divide, arg);
}

auto xpress::objects::ExpressionImplementation::div(double arg) const
    -> xpress::objects::Expression {
  return div(ConstantExpression(arg));
}

xpress::objects::InequalityDefinition
xpress::objects::operator<=(Expression const &lhs, double rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::LEQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator>=(Expression const &lhs, double rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::GEQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator<=(double lhs, Expression const &rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::LEQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator>=(double lhs, Expression const &rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::GEQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator<=(Expression const &lhs, Expression const &rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::LEQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator>=(Expression const &lhs, Expression const &rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::GEQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator==(Expression const &lhs, double rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::EQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator!=(Expression const &, double const &) {
  throw std::runtime_error("Cannot have 'not equal' for constraints");
}
xpress::objects::InequalityDefinition
xpress::objects::operator==(double lhs, Expression const &rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::EQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator!=(double, Expression const &) {
  throw std::runtime_error("Cannot have 'not equal' for constraints");
}
xpress::objects::InequalityDefinition
xpress::objects::operator==(Expression const &lhs, Expression const &rhs) {
  return xpress::objects::InequalityDefinition(lhs, xpress::RowType::EQ, rhs,
                                               std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator!=(Expression const &, Expression const &) {
  throw std::runtime_error("Cannot have 'not equal' for constraints");
}

xpress::objects::Expression xpress::objects::operator-(Expression const &expr) {
  return expr.uminus();
}

xpress::objects::Expression xpress::objects::operator+(Expression const &expr,
                                                       Expression const &e) {
  return expr.plus(e);
}
xpress::objects::Expression xpress::objects::operator+(Expression const &expr,
                                                       double d) {
  return expr.plus(d);
}
xpress::objects::Expression xpress::objects::operator+(double d,
                                                       Expression const &expr) {
  return ConstantExpression(d).plus(expr);
}

xpress::objects::Expression xpress::objects::operator-(Expression const &expr,
                                                       Expression const &e) {
  return expr.minus(e);
}
xpress::objects::Expression xpress::objects::operator-(Expression const &expr,
                                                       double d) {
  return expr.minus(d);
}
xpress::objects::Expression xpress::objects::operator-(double d,
                                                       Expression const &expr) {
  return ConstantExpression(d).minus(expr);
}

xpress::objects::Expression xpress::objects::operator*(Expression const &expr,
                                                       Expression const &e) {
  return expr.mul(e);
}
xpress::objects::Expression xpress::objects::operator*(Expression const &expr,
                                                       double d) {
  return expr.mul(d);
}
xpress::objects::Expression xpress::objects::operator*(double d,
                                                       Expression const &expr) {
  return ConstantExpression(d).mul(expr);
}

xpress::objects::Expression xpress::objects::operator/(Expression const &expr,
                                                       Expression const &e) {
  return expr.div(e);
}
xpress::objects::Expression xpress::objects::operator/(Expression const &expr,
                                                       double d) {
  return expr.div(d);
}
xpress::objects::Expression xpress::objects::operator/(double d,
                                                       Expression const &expr) {
  return ConstantExpression(d).div(expr);
}

;

;

auto xpress::objects::InequalityImplementation::getIndex() const -> int {
  return index;
}

auto xpress::objects::InequalityImplementation::updateIndex(int delta) -> void {
  this->index += delta;
}

auto xpress::objects::InequalityImplementation::getIndexForProb(
    XpressProblem const *xprob) const -> int {
  if (!this->prob)
    throw NotInProblemException("deleted");

  if (this->prob != xprob && this->prob != xprob->toplevel)
    throw NotInProblemException("not in problem");
  return index;
}

auto xpress::objects::InequalityImplementation::getProblem() const
    -> xpress::objects::XpressProblem * {
  return prob;
}

auto xpress::objects::InequalityImplementation::getSerial() const -> XPRSint64 {
  return serial;
}

auto xpress::objects::InequalityImplementation::markDeleted() -> void {
  prob = nullptr;
  index = -1;
}

auto xpress::objects::InequalityImplementation::compareTo(
    Inequality const &other) const -> int {
  assert(index >= -1);

  if (other.isSameImpl(this)) {

    return 0;
  } else if (!(!prob) && !(!other.getProblem()) && prob != other.getProblem()) {

    throw std::invalid_argument(
        "cannot compare objects from different problems");
  }

  else {

    if (serial < other.getSerial())
      return -1;
    else if (serial > other.getSerial())
      return 1;
    else
      return 0;
  }
}

auto xpress::objects::InequalityImplementation::getHashCode() const
    -> std::size_t {
  union {
    XPRSint64 i;
    unsigned long long u;
  } u;
  static_assert(sizeof(u.i) == sizeof(serial), "");
  u.i = serial;
  return u.u;
}

auto xpress::objects::InequalityImplementation::check(
    char const *operation) const -> void {
  if (!prob)
    throw NotInProblemException("Inequality is deleted");
  if (!(!operation) && !prob->isOriginal())
    throw xpress::XPRSProblem::CannotPerformOperationException(operation,
                                                               "Inequality");
}
xpress::objects::InequalityImplementation::InequalityImplementation(
    XpressProblem *prob, long serial, int index)
    : prob(prob), serial(serial), index(index) {}

auto xpress::objects::InequalityImplementation::getRhs() -> double {
  check("get right-hand side");
  return prob->getRhs(getIndex());
}

auto xpress::objects::InequalityImplementation::getRhsRange() -> double {
  check("get right-hand side range");
  return prob->getRhsRange(getIndex());
}

auto xpress::objects::InequalityImplementation::getName() const -> std::string {
  check("get name");
  return prob->getRowName(getIndex());
}

auto xpress::objects::InequalityImplementation::getType() -> xpress::RowType {
  check("get type");
  char type = prob->getRowType(getIndex());
  return (xpress::RowType)type;
}

auto xpress::objects::InequalityImplementation::setRhs(double newRhs)
    -> xpress::objects::InequalityImplementation & {
  check("set right-hand side");
  prob->chgRhs(getIndex(), newRhs);
  return *this;
}

auto xpress::objects::InequalityImplementation::setRhsRange(double newRange)
    -> xpress::objects::InequalityImplementation & {
  check("set right-hand side range");
  prob->chgRhsRange(getIndex(), newRange);
  return *this;
}

auto xpress::objects::InequalityImplementation::setType(xpress::RowType newType)
    -> xpress::objects::InequalityImplementation & {
  check("set type");
  prob->chgRowType(getIndex(), (char)newType);
  return *this;
}

auto xpress::objects::InequalityImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::InequalityImplementation & {
  check("set name");
  int idx = getIndex();
  prob->addNames(xpress::Namespaces::Row,
                 std::vector<std::optional<std::string>>({newName}), idx, idx);
  return *this;
}

auto xpress::objects::InequalityImplementation::getValue(
    xpress::SizedArray<double const> const &data) const -> double {
  check(nullptr);
  return data[getIndex()];
}

auto xpress::objects::InequalityImplementation::getSlack() const -> double {
  check(nullptr);
  return prob->getSlack(getIndex());
}

auto xpress::objects::InequalityImplementation::getDual() const -> double {
  check(nullptr);
  return prob->getDual(getIndex());
}

auto xpress::objects::InequalityImplementation::getCallbackSlack() -> double {
  check(nullptr);
  return prob->getCallbackSlack(getIndex());
}

auto xpress::objects::InequalityImplementation::getCallbackDual() -> double {
  check(nullptr);
  return prob->getCallbackDual(getIndex());
}

auto xpress::objects::InequalityImplementation::chgCoef(
    Variable v, double newCoef) -> xpress::objects::InequalityImplementation & {
  check("change coefficient");
  prob->chgCoef(this, v, newCoef);
  return *this;
}

auto xpress::objects::InequalityImplementation::chgCoefs(
    xpress::SizedArray<Variable const> const &v,
    xpress::SizedArray<double const> const &newCoef)
    -> xpress::objects::InequalityImplementation & {
  check("change coefficients");
  std::vector<Inequality> me(xpress::toInt(v.size()));
  for (int i = 0; i < xpress::toInt(v.size()); ++i)
    me[i] = this;
  prob->chgCoefs(me, v, newCoef);
  return *this;
}

auto xpress::objects::InequalityImplementation::getLhs()
    -> xpress::objects::Expression {
  check("get left-hand side");
  return prob->getLhsExpression(getIndexForProb(prob));
}

auto xpress::objects::AbstractExpressionImplementation::getConstantView() const
    -> double {
  throw std::invalid_argument("not a constant expression");
}

auto xpress::objects::AbstractExpressionImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  throw std::invalid_argument("not a linear expression");
}

auto xpress::objects::AbstractExpressionImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  throw std::invalid_argument("not a quadratic expression");
}

auto xpress::objects::AbstractExpressionImplementation::leq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(this, xpress::RowType::LEQ, rhs, std::nullopt);
}

auto xpress::objects::AbstractExpressionImplementation::leq(
    Expression const &rhs) const -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(this, xpress::RowType::LEQ, rhs, std::nullopt);
}

auto xpress::objects::AbstractExpressionImplementation::geq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(this, xpress::RowType::GEQ, rhs, std::nullopt);
}

auto xpress::objects::AbstractExpressionImplementation::geq(
    Expression const &rhs) const -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(this, xpress::RowType::GEQ, rhs, std::nullopt);
}

auto xpress::objects::AbstractExpressionImplementation::eq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(this, xpress::RowType::EQ, rhs, std::nullopt);
}

auto xpress::objects::AbstractExpressionImplementation::eq(
    Expression const &rhs) const -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(this, xpress::RowType::EQ, rhs, std::nullopt);
}

auto xpress::objects::AbstractExpressionImplementation::in(double lb, double ub)
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(this, lb, ub, std::nullopt);
}

xpress::objects::InequalityDefinitionImplementation::
    InequalityDefinitionImplementation(xpress::objects::Expression lhs,
                                       xpress::RowType type,
                                       xpress::objects::Expression rhs,
                                       std::optional<std::string> name)
    : ConstraintDefinitionImplementation<Inequality>(name), lhs(lhs),
      type(type), rhs(rhs), range(std::vector<double>(0)) {
  if (type == xpress::RowType::Range)
    throw std::invalid_argument("cannot create ranges with this constructor");
  assert(!(!lhs));
  assert(!(!rhs));
}

xpress::objects::InequalityDefinitionImplementation::
    InequalityDefinitionImplementation(xpress::objects::Expression lhs,
                                       xpress::RowType type, double rhs,
                                       std::optional<std::string> name)
    : InequalityDefinitionImplementation(lhs, type, ConstantExpression(rhs),
                                         name) {}

xpress::objects::InequalityDefinitionImplementation::
    InequalityDefinitionImplementation(double lhs, xpress::RowType type,
                                       xpress::objects::Expression rhs,
                                       std::optional<std::string> name)
    : InequalityDefinitionImplementation(ConstantExpression(lhs), type, rhs,
                                         name) {}

xpress::objects::InequalityDefinitionImplementation::
    InequalityDefinitionImplementation(Expression expr, double lb, double ub,
                                       std::optional<std::string> name)
    : ConstraintDefinitionImplementation<Inequality>(name), lhs(expr),
      type(xpress::RowType::Range), rhs(nullptr),
      range(std::vector<double>({lb, ub})) {
  assert(!(!expr));
}

auto xpress::objects::InequalityDefinitionImplementation::createRow(
    xpress::XPRSProblem::RowCreator *creator) const -> void {
  XpressProblem *prob = (XpressProblem *)creator->prob;
  bool normalize = prob->addToRow(1.0, getLhs(), creator);

  if (getType() == xpress::RowType::Range) {
    std::vector<double> bounds = getRange();
    creator->applyRange(bounds[0], bounds[1]);
  } else {
    normalize = prob->addToRow(-1.0, getRhs(), creator) || normalize;
    creator->setType((char)getType());
  }
  if (!(!(getName())))
    creator->setName(getName());
  creator->finishRow(normalize || true);
}

auto xpress::objects::InequalityDefinitionImplementation::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  createRow(creator);
}

auto xpress::objects::InequalityDefinitionImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::InequalityDefinitionImplementation & {
  setConstraintName(newName);
  return *this;
}

auto xpress::objects::InequalityDefinitionImplementation::getIndexHandler(
    XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::Inequality> const * {
  return prob->rowHandler.get();
}

;

;

auto xpress::objects::VariableImplementation::getIndex() const -> int {
  return index;
}

auto xpress::objects::VariableImplementation::updateIndex(int delta) -> void {
  this->index += delta;
}

auto xpress::objects::VariableImplementation::getIndexForProb(
    XpressProblem const *xprob) const -> int {
  if (!this->prob)
    throw NotInProblemException("deleted");

  if (this->prob != xprob && this->prob != xprob->toplevel)
    throw NotInProblemException("not in problem");
  return index;
}

auto xpress::objects::VariableImplementation::getProblem() const
    -> xpress::objects::XpressProblem * {
  return prob;
}

auto xpress::objects::VariableImplementation::getSerial() const -> XPRSint64 {
  return serial;
}

auto xpress::objects::VariableImplementation::markDeleted() -> void {
  prob = nullptr;
  index = -1;
}

auto xpress::objects::VariableImplementation::compareTo(
    Variable const &other) const -> int {

  assert(index >= 0 || index == XpressProblem::NULL_VARIABLE_INDEX ||
         index == -1);

  if (other.isSameImpl(this)) {

    return 0;
  }

  else if (index == XpressProblem::NULL_VARIABLE_INDEX) {
    return other.getIndex() == XpressProblem::NULL_VARIABLE_INDEX ? 0 : -1;
  } else if (other.getIndex() == XpressProblem::NULL_VARIABLE_INDEX) {
    return 1;
  } else if (!(!prob) && !(!other.getProblem()) && prob != other.getProblem()) {

    throw std::invalid_argument(
        "cannot compare objects from different problems");
  }

  else {

    if (serial < other.getSerial())
      return -1;
    else if (serial > other.getSerial())
      return 1;
    else
      return 0;
  }
}

auto xpress::objects::VariableImplementation::getHashCode() const
    -> std::size_t {
  union {
    XPRSint64 i;
    unsigned long long u;
  } u;
  static_assert(sizeof(u.i) == sizeof(serial), "");
  u.i = serial;
  return u.u;
}

auto xpress::objects::VariableImplementation::check(char const *operation) const
    -> void {
  if (!prob)
    throw NotInProblemException("Variable is deleted");
  if (!(!operation) && !prob->isOriginal())
    throw xpress::XPRSProblem::CannotPerformOperationException(operation,
                                                               "Variable");
}
xpress::objects::VariableImplementation::VariableImplementation(
    XpressProblem *prob, long serial, int index)
    : prob(prob), serial(serial), index(index) {}

auto xpress::objects::VariableImplementation::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return ExpressionRTTI::Variable;
}

auto xpress::objects::VariableImplementation::leq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(xpress::objects::LinTerm(this, 1.0),
                              xpress::RowType::LEQ, rhs, std::nullopt);
}

auto xpress::objects::VariableImplementation::leq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(xpress::objects::LinTerm(this, 1.0),
                              xpress::RowType::LEQ, rhs, std::nullopt);
}

auto xpress::objects::VariableImplementation::geq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(xpress::objects::LinTerm(this, 1.0),
                              xpress::RowType::GEQ, rhs, std::nullopt);
}

auto xpress::objects::VariableImplementation::geq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(xpress::objects::LinTerm(this, 1.0),
                              xpress::RowType::GEQ, rhs, std::nullopt);
}

auto xpress::objects::VariableImplementation::eq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(xpress::objects::LinTerm(this, 1.0),
                              xpress::RowType::EQ, rhs, std::nullopt);
}

auto xpress::objects::VariableImplementation::eq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(xpress::objects::LinTerm(this, 1.0),
                              xpress::RowType::EQ, rhs, std::nullopt);
}

auto xpress::objects::VariableImplementation::in(double lb, double ub)
    -> xpress::objects::InequalityDefinition {
  return InequalityDefinition(this, lb, ub, std::nullopt);
}

auto xpress::objects::VariableImplementation::square()
    -> xpress::objects::QuadTerm {
  return QuadTerm(this, this, 1.0);
}

auto xpress::objects::VariableImplementation::getLB() const -> double {
  check("get lower bound");
  return prob->getLB(getIndex());
}

auto xpress::objects::VariableImplementation::getUB() const -> double {
  check("get upper bound");
  return prob->getUB(getIndex());
}

auto xpress::objects::VariableImplementation::getName() const -> std::string {
  check("get name");
  return prob->getColumnName(getIndex());
}

auto xpress::objects::VariableImplementation::getType() const
    -> xpress::ColumnType {
  check("get type");
  char type = prob->getColType(getIndex());
  return (xpress::ColumnType)type;
}

auto xpress::objects::VariableImplementation::fix(double value)
    -> xpress::objects::VariableImplementation & {
  check("fix");
  prob->chgBounds(1, std::vector<int>({getIndex()}),
                  std::vector<char>({(char)'B'}), std::vector<double>({value}));
  return *this;
}

auto xpress::objects::VariableImplementation::setLB(double newLB)
    -> xpress::objects::VariableImplementation & {
  check("set lower bound");
  prob->chgLB(getIndex(), newLB);
  return *this;
}

auto xpress::objects::VariableImplementation::setUB(double newUB)
    -> xpress::objects::VariableImplementation & {
  check("set upper bound");
  prob->chgUB(getIndex(), newUB);
  return *this;
}

auto xpress::objects::VariableImplementation::setType(
    xpress::ColumnType newType) -> xpress::objects::VariableImplementation & {
  check("set type");
  prob->chgColType(getIndex(), (char)newType);
  return *this;
}

auto xpress::objects::VariableImplementation::setLimit(double newLimit)
    -> xpress::objects::VariableImplementation & {
  check("set limit");
  prob->chgGlbLimit(1, std::vector<int>({getIndex()}),
                    std::vector<double>({newLimit}));
  return *this;
}

auto xpress::objects::VariableImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::VariableImplementation & {
  check("set name");
  int idx = getIndex();
  prob->addNames(xpress::Namespaces::Column,
                 std::vector<std::optional<std::string>>({newName}), idx, idx);
  return *this;
}

auto xpress::objects::VariableImplementation::getValue(
    xpress::SizedArray<double const> const &data) const -> double {
  check(nullptr);
  return data[getIndex()];
}

auto xpress::objects::VariableImplementation::getSolution() const -> double {
  check(nullptr);
  return prob->getSolution(getIndex());
}

auto xpress::objects::VariableImplementation::getRedCost() const -> double {
  check(nullptr);
  return prob->getRedCost(getIndex());
}

auto xpress::objects::VariableImplementation::getCallbackSolution() -> double {
  check(nullptr);
  return prob->getCallbackSolution(getIndex());
}

auto xpress::objects::VariableImplementation::getCallbackRedCost() -> double {
  check(nullptr);
  return prob->getCallbackRedCost(getIndex());
}

auto xpress::objects::VariableImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  creator->checkForVariables();
  creator->addNonzero(
      getIndexForProb((xpress::objects::XpressProblem *)creator->prob), factor);
  return true;
}

auto xpress::objects::VariableImplementation::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  extractor->addToken(XPRS_TOK_COL, extractor->dereferenceVariable(this));
  if (factor != 1.0) {
    extractor->addToken(XPRS_TOK_CON, factor);
    extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
  }
  extractor->endExpression();
}

auto xpress::objects::VariableImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  check(nullptr);
  return solution[getIndex()];
}

auto xpress::objects::VariableImplementation::chgCoef(Inequality r,
                                                      double newCoef) -> void {
  check("change coefficient");
  prob->chgCoef(r, this, newCoef);
}

auto xpress::objects::VariableImplementation::chgCoefs(
    xpress::SizedArray<Inequality const> const &r,
    xpress::SizedArray<double const> const &newCoef) -> void {
  check("change coefficients");
  std::vector<Variable> me(xpress::toInt(r.size()));
  for (int i = 0; i < xpress::toInt(r.size()); ++i)
    me[i] = this;
  prob->chgCoefs(r, me, newCoef);
}

auto xpress::objects::VariableImplementation::chgObj(double newCoef)
    -> xpress::objects::VariableImplementation & {
  check("change objective coefficient");
  prob->chgObj(1, std::vector<int>({getIndex()}),
               std::vector<double>({newCoef}));
  return *this;
}

auto xpress::objects::VariableImplementation::getObj() const -> double {
  check("get objective coefficient");
  return prob->getObj(getIndex());
}

auto xpress::objects::VariableImplementation::chgObjN(
    int objidx, double newCoef) -> xpress::objects::VariableImplementation & {
  check("change objective coefficient");
  prob->chgObjN(objidx, 1, std::vector<int>({getIndex()}),
                std::vector<double>({newCoef}));
  return *this;
}

auto xpress::objects::VariableImplementation::getObjN(int objIdx) const
    -> double {
  check("get objective coefficient");
  return prob->getObjN(objIdx, getIndex(), getIndex())[0];
}

auto xpress::objects::VariableImplementation::pwlOf(
    Variable input, xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval) const
    -> xpress::objects::PWLDefinition {
  std::optional<std::string> name;
  return pwlOf(input, xval, yval, name);
}

auto xpress::objects::VariableImplementation::pwlOf(
    Variable input, xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval,
    std::optional<std::string> name) const -> xpress::objects::PWLDefinition {
  return xpress::objects::PWLDefinition(input, this, xval, yval, name);
}

template <typename... BREAKPOINTS, typename RestrictPack>
auto xpress::objects::VariableImplementation::pwlOf(
    Variable input,
    BREAKPOINTS... breakpoints) const -> xpress::objects::PWLDefinition {
  return pwlOf(input, xpress::packToArray<PwlBreakpoint>(breakpoints...),
               std::nullopt);
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::VariableImplementation::pwlOf(
    Variable input,
    Strm0 const &breakpoints) const -> xpress::objects::PWLDefinition {
  return pwlOf(input, breakpoints, std::nullopt);
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::VariableImplementation::pwlOf(
    Variable input, Strm0 const &breakpoints,
    std::optional<std::string> name) const -> xpress::objects::PWLDefinition {
  std::vector<std::vector<double>> xy =
      xpress::PwlBreakpoint::convert(breakpoints);
  return xpress::objects::PWLDefinition(input, this, xy[0], xy[1], name);
}

template <typename... VALUES, typename RestrictPack>
auto xpress::objects::VariableImplementation::maxOf(
    xpress::SizedArray<Variable const> const &variables,
    VALUES... values) const -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Max, variables,
      xpress::packToArray<double>(values...), std::nullopt);
}

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::VariableImplementation::maxOf(
    Strm0 const &variables, Strm1 const &values,
    std::optional<std::string> name) const
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Max, xpress::toArray<Variable>(variables),
      xpress::toArray<double>(values), name);
}

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::VariableImplementation::maxOf(Strm0 const &variables,
                                                    Strm1 const &values) const
    -> xpress::objects::GeneralConstraintDefinition {
  return maxOf(variables, values, std::nullopt);
}

template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::VariableImplementation::maxOf(VARIABLES... variables)
    const -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Max,
      xpress::packToArray<Variable>(variables...), std::vector<double>(0),
      std::nullopt);
}

template <typename... VALUES, typename RestrictPack>
auto xpress::objects::VariableImplementation::minOf(
    xpress::SizedArray<Variable const> const &variables,
    VALUES... values) const -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Min, variables,
      xpress::packToArray<double>(values...), std::nullopt);
}

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::VariableImplementation::minOf(
    Strm0 const &variables, Strm1 const &values,
    std::optional<std::string> name) const
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Min, xpress::toArray<Variable>(variables),
      xpress::toArray<double>(values), name);
}

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::VariableImplementation::minOf(Strm0 const &variables,
                                                    Strm1 const &values) const
    -> xpress::objects::GeneralConstraintDefinition {
  return minOf(variables, values, std::nullopt);
}

template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::VariableImplementation::minOf(VARIABLES... variables)
    const -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Min,
      xpress::packToArray<Variable>(variables...), std::vector<double>(0),
      std::nullopt);
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::VariableImplementation::orOf(
    Strm0 const &variables, std::optional<std::string> name) const
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Or, xpress::toArray<Variable>(variables),
      std::vector<double>(0), name);
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::VariableImplementation::orOf(Strm0 const &variables) const
    -> xpress::objects::GeneralConstraintDefinition {
  return orOf(variables, std::nullopt);
}

template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::VariableImplementation::orOf(VARIABLES... variables) const
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Or,
      xpress::packToArray<Variable>(variables...), std::vector<double>(0),
      std::nullopt);
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::VariableImplementation::andOf(
    Strm0 const &variables, std::optional<std::string> name) const
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::And, xpress::toArray<Variable>(variables),
      std::vector<double>(0), name);
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::VariableImplementation::andOf(Strm0 const &variables)
    const -> xpress::objects::GeneralConstraintDefinition {
  return andOf(variables, std::nullopt);
}

template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::VariableImplementation::andOf(VARIABLES... variables)
    const -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::And,
      xpress::packToArray<Variable>(variables...), std::vector<double>(0),
      std::nullopt);
}

auto xpress::objects::VariableImplementation::absOf(Variable x) const
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      this, xpress::GenConsType::Abs, std::vector<Variable>({x}),
      std::vector<double>(0), std::nullopt);
}

auto xpress::objects::VariableImplementation::ifThen(
    InequalityDefinition implied) const
    -> xpress::objects::IndicatorDescription {
  return IndicatorDescription(this, true, implied);
}

auto xpress::objects::VariableImplementation::ifNotThen(
    InequalityDefinition implied) const
    -> xpress::objects::IndicatorDescription {
  return IndicatorDescription(this, false, implied);
}

auto xpress::objects::VariableImplementation::toString() const -> std::string {

  if (!(!prob) && prob->isOriginal()) {
    return getName();
  } else {
    return xpress::concatStrings("X", "[", index, "]");
  }
}

auto xpress::objects::VariableImplementation::getMaxDegree() const -> int {
  return 1;
}

auto xpress::objects::VariableImplementation::getConstantView() const
    -> double {
  throw std::invalid_argument("not a constant expression");
}

auto xpress::objects::VariableImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return InternalUtils::getSingletonLinView(this, 1.0);
}

auto xpress::objects::VariableImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return InternalUtils::getSingletonQuadView(this, XpressProblem::NULL_VARIABLE,
                                             1.0);
}

auto xpress::objects::VariableImplementation::uminus() const
    -> xpress::objects::Expression {
  return LinTerm(this, -1.0);
}

auto xpress::objects::VariableImplementation::mul(Expression arg) const
    -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return mul(arg.cast<ConstantExpression>());
  case ExpressionRTTI::Variable:
    return mul(arg.cast<Variable>());
  case ExpressionRTTI::LinTerm:
    return mul(arg.cast<LinTerm>());
  default:
    return ExpressionImplementation::mul(arg);
  }
}

auto xpress::objects::VariableImplementation::mul(ConstantExpression arg) const
    -> xpress::objects::LinTerm {
  return LinTerm(this, arg.getValue());
}

auto xpress::objects::VariableImplementation::mul(double arg) const
    -> xpress::objects::Expression {
  return LinTerm(this, arg);
}

auto xpress::objects::VariableImplementation::mul(Variable arg) const
    -> xpress::objects::QuadTerm {
  return QuadTerm(this, arg, 1.0);
}

auto xpress::objects::VariableImplementation::mul(LinTerm arg) const
    -> xpress::objects::QuadTerm {
  return QuadTerm(this, arg.getVariable(), arg.getCoefficient());
}

auto xpress::objects::VariableImplementation::div(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return div(arg.cast<ConstantExpression>());
  default:
    return ExpressionImplementation::div(arg);
  }
}

auto xpress::objects::VariableImplementation::div(ConstantExpression arg) const
    -> xpress::objects::LinTerm {
  return LinTerm(this, 1.0 / arg.getValue());
}

auto xpress::objects::VariableImplementation::div(double arg) const
    -> xpress::objects::Expression {
  return LinTerm(this, 1.0 / arg);
}

xpress::objects::InequalityDefinition
xpress::objects::operator<=(Variable const &lhs, double rhs) {
  return InequalityDefinition(lhs, xpress::RowType::LEQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator>=(Variable const &lhs, double rhs) {
  return InequalityDefinition(lhs, xpress::RowType::GEQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator<=(double lhs, Variable const &rhs) {
  return InequalityDefinition(rhs, xpress::RowType::GEQ, lhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator>=(double lhs, Variable const &rhs) {
  return InequalityDefinition(rhs, xpress::RowType::LEQ, lhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator<=(Variable const &lhs, Variable const &rhs) {
  return InequalityDefinition(lhs, xpress::RowType::LEQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator>=(Variable const &lhs, Variable const &rhs) {
  return InequalityDefinition(lhs, xpress::RowType::GEQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator<=(Variable const &lhs, Expression const &rhs) {
  return InequalityDefinition(lhs, xpress::RowType::LEQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator>=(Variable const &lhs, Expression const &rhs) {
  return InequalityDefinition(lhs, xpress::RowType::GEQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator==(Variable const &lhs, double rhs) {
  return InequalityDefinition(lhs, xpress::RowType::EQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition xpress::objects::operator!=(Variable,
                                                                  double) {
  throw std::runtime_error("Cannot have 'not equal' for constraints");
}
xpress::objects::InequalityDefinition
xpress::objects::operator==(double lhs, Variable const &rhs) {
  return InequalityDefinition(rhs, xpress::RowType::EQ, lhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator!=(double, Variable const &) {
  throw std::runtime_error("Cannot have 'not equal' for constraints");
}
xpress::objects::InequalityDefinition
xpress::objects::operator==(Variable const &lhs, Variable const &rhs) {
  return InequalityDefinition(lhs, xpress::RowType::EQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator!=(Variable const &, Variable const &) {
  throw std::runtime_error("Cannot have 'not equal' for constraints");
}
xpress::objects::InequalityDefinition
xpress::objects::operator==(Variable const &lhs, Expression const &rhs) {
  return InequalityDefinition(lhs, xpress::RowType::EQ, rhs, std::nullopt);
}
xpress::objects::InequalityDefinition
xpress::objects::operator!=(Variable const &, Expression const &) {
  throw std::runtime_error("Cannot have 'not equal' for constraints");
}

xpress::objects::LinTerm xpress::objects::operator-(Variable v) {
  return v.uminus().cast<LinTerm>();
}

xpress::objects::LinTerm xpress::objects::operator*(Variable const &v,
                                                    double d) {
  return v.mul(d).cast<LinTerm>();
}
xpress::objects::LinTerm xpress::objects::operator*(double d,
                                                    Variable const &v) {
  return v.mul(d).cast<LinTerm>();
}
xpress::objects::LinTerm
xpress::objects::operator*(Variable const &v, ConstantExpression const &d) {
  return v.mul(d).cast<LinTerm>();
}
xpress::objects::QuadTerm xpress::objects::operator*(Variable const &v,
                                                     Variable const &v2) {
  return v.mul(v2).cast<QuadTerm>();
}
xpress::objects::QuadTerm xpress::objects::operator*(Variable const &v,
                                                     LinTerm const &t) {
  return v.mul(t).cast<QuadTerm>();
}

xpress::objects::LinTerm xpress::objects::operator/(Variable const &v,
                                                    double d) {
  return v.div(d).cast<LinTerm>();
}
xpress::objects::LinTerm
xpress::objects::operator/(Variable const &v, ConstantExpression const &d) {
  return v.div(d).cast<LinTerm>();
}

xpress::objects::ConstantExpressionImplementation::
    ConstantExpressionImplementation(double value)
    : value(value) {}

auto xpress::objects::ConstantExpressionImplementation::constant(double cnst)
    -> xpress::objects::ConstantExpression {
  return ConstantExpression(cnst);
}

auto xpress::objects::ConstantExpressionImplementation::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return ExpressionRTTI::Constant;
}

auto xpress::objects::ConstantExpressionImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  creator->addRhs(-factor * getValue());
  return false;
}

auto xpress::objects::ConstantExpressionImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  (void)solution;
  return getValue();
}

auto xpress::objects::ConstantExpressionImplementation::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  extractor->addToken(XPRS_TOK_CON, factor * getValue());
  extractor->endExpression();
}

auto xpress::objects::ConstantExpressionImplementation::toString() const
    -> std::string {
  return InternalUtils::floatToString(getValue());
}

auto xpress::objects::ConstantExpressionImplementation::getMaxDegree() const
    -> int {
  return 0;
}

auto xpress::objects::ConstantExpressionImplementation::getConstantView() const
    -> double {
  return getValue();
}

auto xpress::objects::ConstantExpressionImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return InternalUtils::getSingletonLinView(XpressProblem::NULL_VARIABLE,
                                            getValue());
}

auto xpress::objects::ConstantExpressionImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return InternalUtils::getSingletonQuadView(
      XpressProblem::NULL_VARIABLE, XpressProblem::NULL_VARIABLE, getValue());
}

auto xpress::objects::ConstantExpressionImplementation::uminus() const
    -> xpress::objects::Expression {
  return ConstantExpression(-getValue());
}

auto xpress::objects::ConstantExpressionImplementation::plus(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return plus(arg.cast<ConstantExpression>());
  default:
    return AbstractExpressionImplementation::plus(arg);
  }
}

auto xpress::objects::ConstantExpressionImplementation::plus(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::ConstantExpression {
  return ConstantExpression(getValue() + arg.getValue());
}

auto xpress::objects::ConstantExpressionImplementation::plus(double arg) const
    -> xpress::objects::Expression {
  return ConstantExpression(getValue() + arg);
}

auto xpress::objects::ConstantExpressionImplementation::minus(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return minus(arg.cast<ConstantExpression>());
  default:
    return AbstractExpressionImplementation::minus(arg);
  }
}

auto xpress::objects::ConstantExpressionImplementation::minus(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::ConstantExpression {
  return ConstantExpression(getValue() - arg.getValue());
}

auto xpress::objects::ConstantExpressionImplementation::minus(double arg) const
    -> xpress::objects::Expression {
  return ConstantExpression(getValue() - arg);
}

auto xpress::objects::ConstantExpressionImplementation::mul(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return mul(arg.cast<ConstantExpression>());
  case ExpressionRTTI::Variable:
    return mul(arg.cast<Variable>());
  case ExpressionRTTI::LinTerm:
    return mul(arg.cast<LinTerm>());
  case ExpressionRTTI::QuadTerm:
    return mul(arg.cast<QuadTerm>());
  default:
    return AbstractExpressionImplementation::mul(arg);
  }
}

auto xpress::objects::ConstantExpressionImplementation::mul(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::ConstantExpression {
  return ConstantExpression(getValue() * arg.getValue());
}

auto xpress::objects::ConstantExpressionImplementation::mul(double arg) const
    -> xpress::objects::Expression {
  return ConstantExpression(getValue() * arg);
}

auto xpress::objects::ConstantExpressionImplementation::mul(
    xpress::objects::Variable arg) const -> xpress::objects::LinTerm {
  return LinTerm(arg, getValue());
}

auto xpress::objects::ConstantExpressionImplementation::mul(
    xpress::objects::LinTerm arg) const -> xpress::objects::LinTerm {
  return LinTerm(arg.getVariable(), arg.getCoefficient() * getValue());
}

auto xpress::objects::ConstantExpressionImplementation::mul(
    xpress::objects::QuadTerm arg) const -> xpress::objects::QuadTerm {
  return QuadTerm(arg.getVariable1(), arg.getVariable2(),
                  arg.getCoefficient() * getValue());
}

auto xpress::objects::ConstantExpressionImplementation::div(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return div(arg.cast<ConstantExpression>());
  default:
    return AbstractExpressionImplementation::div(arg);
  }
}

auto xpress::objects::ConstantExpressionImplementation::div(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::ConstantExpression {
  return ConstantExpression(getValue() / arg.getValue());
}

auto xpress::objects::ConstantExpressionImplementation::div(double arg) const
    -> xpress::objects::Expression {
  return ConstantExpression(getValue() / arg);
}

xpress::objects::ConstantExpression
xpress::objects::operator-(ConstantExpression const &c) {
  return c.uminus().cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator+(ConstantExpression const &v, double d) {
  return v.plus(d).cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator+(double d, ConstantExpression const &v) {
  return v.plus(d).cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator+(ConstantExpression const &v,
                           ConstantExpression const &d) {
  return v.plus(d).cast<ConstantExpression>();
}

xpress::objects::ConstantExpression
xpress::objects::operator-(ConstantExpression const &v, double d) {
  return v.minus(d).cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator-(double d, ConstantExpression const &v) {
  return ConstantExpression(d).minus(v).cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator-(ConstantExpression const &v,
                           ConstantExpression const &d) {
  return v.minus(d).cast<ConstantExpression>();
}

xpress::objects::ConstantExpression
xpress::objects::operator*(ConstantExpression const &v, double d) {
  return v.mul(d).cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator*(double d, ConstantExpression const &v) {
  return v.mul(d).cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator*(ConstantExpression const &v,
                           ConstantExpression const &d) {
  return v.mul(d).cast<ConstantExpression>();
}
xpress::objects::LinTerm xpress::objects::operator*(ConstantExpression const &c,
                                                    Variable const &v) {
  return c.mul(v).cast<LinTerm>();
}
xpress::objects::LinTerm xpress::objects::operator*(ConstantExpression const &v,
                                                    LinTerm const &t) {
  return v.mul(t).cast<LinTerm>();
}
xpress::objects::QuadTerm
xpress::objects::operator*(ConstantExpression const &v, QuadTerm const &t) {
  return v.mul(t).cast<QuadTerm>();
}

xpress::objects::ConstantExpression
xpress::objects::operator/(ConstantExpression const &v, double d) {
  return v.div(d).cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator/(double d, ConstantExpression const &v) {
  return ConstantExpression(d).div(v).cast<ConstantExpression>();
}
xpress::objects::ConstantExpression
xpress::objects::operator/(ConstantExpression const &v,
                           ConstantExpression const &d) {
  return v.div(d).cast<ConstantExpression>();
}

;

;

auto xpress::objects::GeneralConstraintImplementation::getIndex() const -> int {
  return index;
}

auto xpress::objects::GeneralConstraintImplementation::updateIndex(int delta)
    -> void {
  this->index += delta;
}

auto xpress::objects::GeneralConstraintImplementation::getIndexForProb(
    XpressProblem const *xprob) const -> int {
  if (!this->prob)
    throw NotInProblemException("deleted");

  if (this->prob != xprob && this->prob != xprob->toplevel)
    throw NotInProblemException("not in problem");
  return index;
}

auto xpress::objects::GeneralConstraintImplementation::getProblem() const
    -> xpress::objects::XpressProblem * {
  return prob;
}

auto xpress::objects::GeneralConstraintImplementation::getSerial() const
    -> XPRSint64 {
  return serial;
}

auto xpress::objects::GeneralConstraintImplementation::markDeleted() -> void {
  prob = nullptr;
  index = -1;
}

auto xpress::objects::GeneralConstraintImplementation::compareTo(
    GeneralConstraint const &other) const -> int {
  assert(index >= -1);

  if (other.isSameImpl(this)) {

    return 0;
  } else if (!(!prob) && !(!other.getProblem()) && prob != other.getProblem()) {

    throw std::invalid_argument(
        "cannot compare objects from different problems");
  }

  else {

    if (serial < other.getSerial())
      return -1;
    else if (serial > other.getSerial())
      return 1;
    else
      return 0;
  }
}

auto xpress::objects::GeneralConstraintImplementation::getHashCode() const
    -> std::size_t {
  union {
    XPRSint64 i;
    unsigned long long u;
  } u;
  static_assert(sizeof(u.i) == sizeof(serial), "");
  u.i = serial;
  return u.u;
}

auto xpress::objects::GeneralConstraintImplementation::check(
    char const *operation) const -> void {
  if (!prob)
    throw NotInProblemException("GeneralConstraint is deleted");
  if (!(!operation) && !prob->isOriginal())
    throw xpress::XPRSProblem::CannotPerformOperationException(
        operation, "GeneralConstraint");
}
xpress::objects::GeneralConstraintImplementation::
    GeneralConstraintImplementation(XpressProblem *prob, long serial, int index)
    : prob(prob), serial(serial), index(index) {}

auto xpress::objects::GeneralConstraintImplementation::getType()
    -> xpress::GenConsType {
  check("get type");
  std::vector<xpress::GenConsType> type(1);

  std::vector<double> dummyVal(1);
  std::vector<int> dummyValStart(2);
  int p_ncols;
  int p_nvals;
  prob->getGenCons(type, nullptr, static_cast<int *>(nullptr), nullptr, 0,
                   &p_ncols, dummyValStart, dummyVal, 1, &p_nvals, getIndex(),
                   getIndex());
  return type[0];
}

auto xpress::objects::GeneralConstraintImplementation::getResultant()
    -> xpress::objects::Variable {
  check("get resultant");
  std::vector<int> resultant(1);

  std::vector<double> dummyVal(1);
  std::vector<int> dummyValStart(2);
  int p_ncols;
  int p_nvals;
  prob->getGenCons(nullptr, resultant, static_cast<int *>(nullptr), nullptr, 0,
                   &p_ncols, dummyValStart, dummyVal, 1, &p_nvals, getIndex(),
                   getIndex());
  return prob->variableForIndex(resultant[0]);
}

auto xpress::objects::GeneralConstraintImplementation::getVariables()
    -> std::vector<xpress::objects::Variable> {
  check("get variables");
  int p_ncols;
  int p_nvals;
  prob->getGenCons(nullptr, nullptr, nullptr, nullptr, 0, &p_ncols, nullptr,
                   nullptr, 0, &p_nvals, getIndex(), getIndex());
  std::vector<int> colind(p_ncols);
  prob->getGenCons(nullptr, nullptr, nullptr, colind, p_ncols, &p_ncols,
                   nullptr, nullptr, 0, &p_nvals, getIndex(), getIndex());
  std::vector<Variable> ret(xpress::toInt(colind.size()));
  for (int i = 0; i < xpress::toInt(colind.size()); ++i)
    ret[i] = prob->variableForIndex(colind[i]);
  return ret;
}

auto xpress::objects::GeneralConstraintImplementation::getValues()
    -> std::vector<double> {
  check("get values");
  int p_ncols;
  int p_nvals;
  prob->getGenCons(nullptr, nullptr, nullptr, nullptr, 0, &p_ncols, nullptr,
                   nullptr, 0, &p_nvals, getIndex(), getIndex());
  std::vector<double> value(p_nvals);
  prob->getGenCons(nullptr, nullptr, nullptr, nullptr, 0, &p_ncols, nullptr,
                   value, p_nvals, &p_nvals, getIndex(), getIndex());
  return value;
}

auto xpress::objects::GeneralConstraintImplementation::getName() const
    -> std::string {
  check("get name");
  return prob->getGenConsName(getIndex());
}

auto xpress::objects::GeneralConstraintImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::GeneralConstraintImplementation & {
  check("set name");
  int idx = getIndex();
  prob->addNames(xpress::Namespaces::GenCons,
                 std::vector<std::optional<std::string>>({newName}), idx, idx);
  return *this;
}

auto xpress::objects::GeneralConstraintImplementation::Abs(Variable resultant,
                                                           Variable variable)
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Abs, variable);
}

template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::GeneralConstraintImplementation::And(
    Variable resultant,
    VARIABLES... variables) -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::And,
      xpress::packToArray<Variable>(variables...));
}

auto xpress::objects::GeneralConstraintImplementation::And(
    Variable resultant, xpress::SizedArray<Variable const> const &variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::And, variables);
}

template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::GeneralConstraintImplementation::Or(
    Variable resultant,
    VARIABLES... variables) -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Or,
      xpress::packToArray<Variable>(variables...));
}

auto xpress::objects::GeneralConstraintImplementation::Or(
    Variable resultant, xpress::SizedArray<Variable const> const &variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Or, variables);
}

template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::GeneralConstraintImplementation::Max(
    Variable resultant,
    VARIABLES... variables) -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Max,
      xpress::packToArray<Variable>(variables...));
}

auto xpress::objects::GeneralConstraintImplementation::Max(
    Variable resultant, xpress::SizedArray<Variable const> const &variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Max, variables);
}

template <typename... VALUES, typename RestrictPack>
auto xpress::objects::GeneralConstraintImplementation::Max(
    Variable resultant, xpress::SizedArray<Variable const> const &variables,
    VALUES... values) -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Max, variables,
      xpress::packToArray<double>(values...));
}

auto xpress::objects::GeneralConstraintImplementation::Max(
    Variable resultant, xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &values)
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Max, variables, values);
}

template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::GeneralConstraintImplementation::Min(
    Variable resultant,
    VARIABLES... variables) -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Min,
      xpress::packToArray<Variable>(variables...));
}

auto xpress::objects::GeneralConstraintImplementation::Min(
    Variable resultant, xpress::SizedArray<Variable const> const &variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Min, variables);
}

template <typename... VALUES, typename RestrictPack>
auto xpress::objects::GeneralConstraintImplementation::Min(
    Variable resultant, xpress::SizedArray<Variable const> const &variables,
    VALUES... values) -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Min, variables,
      xpress::packToArray<double>(values...), std::nullopt);
}

auto xpress::objects::GeneralConstraintImplementation::Min(
    Variable resultant, xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &values)
    -> xpress::objects::GeneralConstraintDefinition {
  return xpress::objects::GeneralConstraintDefinition(
      resultant, xpress::GenConsType::Min, variables, values);
}

;
;

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
xpress::objects::GeneralConstraintDefinitionImplementation::
    GeneralConstraintDefinitionImplementation(Variable resultant,
                                              xpress::GenConsType type,
                                              Strm0 const &variables,
                                              Strm1 const &value,
                                              std::optional<std::string> name)
    : ConstraintDefinitionImplementation<GeneralConstraint>(name),
      variables(variables.begin(), variables.end()),
      value(value.begin(), value.end()), resultant(resultant), type(type) {}

template <typename Strm0, typename Strm0IsStream>
xpress::objects::GeneralConstraintDefinitionImplementation::
    GeneralConstraintDefinitionImplementation(Variable resultant,
                                              xpress::GenConsType type,
                                              Strm0 const &variables,
                                              std::optional<std::string> name)
    : GeneralConstraintDefinitionImplementation(
          resultant, type, variables,
          xpress::Iterable<double>::fromArray(std::vector<double>()), name) {}

template <typename... VARIABLES, typename RestrictPack>
xpress::objects::GeneralConstraintDefinitionImplementation::
    GeneralConstraintDefinitionImplementation(Variable resultant,
                                              xpress::GenConsType type,
                                              VARIABLES... variables)
    : GeneralConstraintDefinitionImplementation(
          resultant, type,
          xpress::Iterable<Variable>::fromArray(
              xpress::packToArray<Variable>(variables...)),
          xpress::Iterable<double>::fromArray(std::vector<double>()),
          std::nullopt) {}

xpress::objects::GeneralConstraintDefinitionImplementation::
    GeneralConstraintDefinitionImplementation(
        Variable resultant, xpress::GenConsType type,
        xpress::SizedArray<Variable const> const &variables)
    : GeneralConstraintDefinitionImplementation(
          resultant, type, xpress::Iterable<Variable>::fromArray(variables),
          xpress::Iterable<double>::fromArray(std::vector<double>()),
          std::nullopt) {}

xpress::objects::GeneralConstraintDefinitionImplementation::
    GeneralConstraintDefinitionImplementation(
        Variable resultant, xpress::GenConsType type,
        xpress::SizedArray<Variable const> const &variables,
        xpress::SizedArray<double const> const &values)
    : GeneralConstraintDefinitionImplementation(
          resultant, type, xpress::Iterable<Variable>::fromArray(variables),
          xpress::Iterable<double>::fromArray(values), std::nullopt) {}

auto xpress::objects::GeneralConstraintDefinitionImplementation::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  creator->checkForVariables();
  std::vector<int> colind(xpress::toInt(variables.size()));
  for (int i = 0; i < xpress::toInt(variables.size()); ++i)
    colind[i] = variables[i].getIndex();
  creator->addGeneralConstraint(resultant.getIndex(), type, colind, value,
                                getName());
}

auto xpress::objects::GeneralConstraintDefinitionImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::GeneralConstraintDefinitionImplementation & {
  setConstraintName(newName);
  return *this;
}

auto xpress::objects::GeneralConstraintDefinitionImplementation::
    getIndexHandler(xpress::objects::XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::GeneralConstraint> const
        * {
  return prob->generalConstraintHandler.get();
}

xpress::objects::IISVariable::IISVariable(InfeasibleDomain domain,
                                          Variable variable, double dj,
                                          IIS::Isolation isolation)
    : domain(domain), variable(variable), dj(dj), isolation(isolation){}

;

;

xpress::objects::IndicatorDescriptionImplementation::
    IndicatorDescriptionImplementation(Variable indicatorVariable,
                                       bool indicatorValue,
                                       InequalityDefinition implied)
    : ConstraintDefinitionImplementation<Inequality>(implied.getName()) {
  this->indicatorVariable = indicatorVariable;
  this->indicatorValue = indicatorValue;
  this->implied = implied;
}

auto xpress::objects::IndicatorDescriptionImplementation::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {

  implied.create(creator);

  creator->commit();

  assert(creator->getRowCount() == creator->prob->attributes.getRows());
  int row = creator->getRowCount() - 1;
  creator->prob->setIndicators(
      1, std::vector<int>({row}),
      std::vector<int>(
          {indicatorVariable.getIndexForProb((XpressProblem *)creator->prob)}),
      std::vector<int>({indicatorValue ? 1 : -1}));
}

auto xpress::objects::IndicatorDescriptionImplementation::getName() const
    -> std::optional<std::string> {
  return implied.getName();
}

auto xpress::objects::IndicatorDescriptionImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::IndicatorDescriptionImplementation & {
  implied.setName(newName);
  return *this;
}

auto xpress::objects::IndicatorDescriptionImplementation::getIndexHandler(
    xpress::objects::XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::Inequality> const * {
  return prob->rowHandler.get();
}

xpress::objects::IndicatorObjects::IndicatorObjects(Variable indicatorVariable,
                                                    bool indicatorValue,
                                                    Inequality row)
    : indicatorVariable(indicatorVariable), indicatorValue(indicatorValue),
      row(row) {}

xpress::objects::LinCoef::LinCoef(Variable variable, double coefficient)
    : variable(variable), coefficient(coefficient) {}

xpress::objects::LinCoef::LinCoef(LinCoef const &other)
    : LinCoef(other.getVariable(), other.getCoefficient()) {}

xpress::objects::LinCoef::LinCoef()
    : LinCoef(XpressProblem::NULL_VARIABLE, 0.0) {}

auto xpress::objects::LinExpressionImplementation::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return ExpressionRTTI::LinExpression;
}

auto xpress::objects::LinExpressionImplementation::normalizeTerms()
    -> xpress::objects::LinExpressionImplementation & {

  return *this;
}

;

;

;

auto xpress::objects::LinExpressionImplementation::prepareToAdd(int newTerms)
    -> void {

  (void)newTerms;
}

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::LinExpressionImplementation::addTerms(
    Strm0 const &variables, Strm1 const &coefficients)
    -> xpress::objects::LinExpressionImplementation & {
  prepareToAdd(std::max(0, std::max(xpress::sizeHint(variables),
                                    xpress::sizeHint(coefficients))));
  using std::begin;
  using std::end;
  auto vCurrent = begin(variables);
  auto vEnd = end(variables);
  using std::begin;
  using std::end;
  auto cCurrent = begin(coefficients);
  auto cEnd = end(coefficients);
  while (vCurrent != vEnd && cCurrent != cEnd) {
    auto const &v = *vCurrent;
    auto const &c = *cCurrent;

    addTerm(v, c);

    ++vCurrent;
    ++cCurrent;
  }
  if (vCurrent != vEnd || cCurrent != cEnd)
    throw std::invalid_argument("input data has different dimensions");
  return *this;
}

auto xpress::objects::LinExpressionImplementation::addTerms(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &coefficients, int offset,
    int count) -> xpress::objects::LinExpressionImplementation & {
  prepareToAdd(count);
  for (int i = 0; i < count; ++i)
    addTerm(variables[offset + i], coefficients[offset + i]);
  return *this;
}

;

auto xpress::objects::LinExpressionImplementation::addTerm(LinTerm const &term)
    -> xpress::objects::LinExpressionImplementation & {
  return addTerm(term.getVariable(), term.getCoefficient());
}

;

;

auto xpress::objects::LinExpressionImplementation::square()
    -> xpress::objects::Expression {
  return mul(this);
}

auto xpress::objects::LinExpressionImplementation::create()
    -> xpress::objects::LinExpression {
  return LinTermMap();
}

auto xpress::objects::LinExpressionImplementation::create(double constant)
    -> xpress::objects::LinExpression {
  return LinTermMap(constant);
}

auto xpress::objects::LinExpressionImplementation::getMaxDegree() const -> int {
  return 1;
}

auto xpress::objects::LinExpressionImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return std::make_shared<xpress::TransformedView<LinearView, QuadView>>(
      getLinearView(), [](LinCoef const &c) {
        return QuadCoef(c.getVariable(), XpressProblem::NULL_VARIABLE,
                        c.getCoefficient());
      });
}

xpress::objects::LinTermImplementation::LinTermImplementation(
    Variable variable, double coefficient)
    : variable(variable), coefficient(coefficient) {
  if (LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("linear terms must have a variable");
}

auto xpress::objects::LinTermImplementation::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return ExpressionRTTI::LinTerm;
}

auto xpress::objects::LinTermImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  creator->checkForVariables();
  creator->addNonzero(xpress::objects::LinQuadUtils::index4Var(
                          (XpressProblem *)creator->prob, getVariable()),
                      factor * getCoefficient());
  return false;
}

auto xpress::objects::LinTermImplementation::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  double coef = factor * getCoefficient();
  if (coef == 0.0)
    extractor->addToken(XPRS_TOK_CON, 0.0);
  else {
    if (coef != 1.0)
      extractor->addToken(XPRS_TOK_CON, coef);
    extractor->addToken(XPRS_TOK_COL,
                        extractor->dereferenceVariable(getVariable()));
    if (coef != 1.0)
      extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
  }
  extractor->endExpression();
}

auto xpress::objects::LinTermImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return getCoefficient() * solution[getVariable().getIndex()];
}

auto xpress::objects::LinTermImplementation::lTerm(
    xpress::objects::Variable variable,
    double coefficient) -> xpress::objects::LinTerm {
  return LinTerm(variable, coefficient);
}

auto xpress::objects::LinTermImplementation::lTerm(Variable variable)
    -> xpress::objects::LinTerm {
  return LinTerm(variable, 1.0);
}

auto xpress::objects::LinTermImplementation::square() const
    -> xpress::objects::QuadTerm {
  return LinTermImplementation::mul(LinTerm(this));
}

auto xpress::objects::LinTermImplementation::toString() const -> std::string {
  return toString(getVariable(), getCoefficient());
}

auto xpress::objects::LinTermImplementation::toString(
    xpress::objects::Variable v, double coef) -> std::string {
  if (coef == 0.0)
    return "0.0";
  else if (LinQuadUtils::isConstant(v))
    return InternalUtils::floatToString(coef);
  else if (coef == 1.0)
    return v.toString();
  else if (coef == -1.0)
    return xpress::concatStrings("-", v.toString());
  else
    return xpress::concatStrings(InternalUtils::floatToString(coef), "*",
                                 v.toString());
}

auto xpress::objects::LinTermImplementation::getMaxDegree() const -> int {
  return 1;
}

auto xpress::objects::LinTermImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return InternalUtils::getSingletonLinView(getVariable(), getCoefficient());
}

auto xpress::objects::LinTermImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return InternalUtils::getSingletonQuadView(
      getVariable(), XpressProblem::NULL_VARIABLE, getCoefficient());
}

auto xpress::objects::LinTermImplementation::uminus() const
    -> xpress::objects::Expression {
  return LinTerm(getVariable(), -getCoefficient());
}

auto xpress::objects::LinTermImplementation::mul(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return mul(arg.cast<ConstantExpression>());
  case ExpressionRTTI::Variable:
    return mul(arg.cast<Variable>());
  case ExpressionRTTI::LinTerm:
    return mul(arg.cast<LinTerm>());
  default:
    return AbstractExpressionImplementation::mul(arg);
  }
}

auto xpress::objects::LinTermImplementation::mul(ConstantExpression arg) const
    -> xpress::objects::LinTerm {
  return LinTerm(getVariable(), getCoefficient() * arg.getValue());
}

auto xpress::objects::LinTermImplementation::mul(double arg) const
    -> xpress::objects::Expression {
  return LinTerm(getVariable(), getCoefficient() * arg);
}

auto xpress::objects::LinTermImplementation::mul(Variable arg) const
    -> xpress::objects::QuadTerm {
  return QuadTerm(getVariable(), arg, getCoefficient());
}

auto xpress::objects::LinTermImplementation::mul(LinTerm arg) const
    -> xpress::objects::QuadTerm {
  return QuadTerm(getVariable(), arg.getVariable(),
                  getCoefficient() * arg.getCoefficient());
}

auto xpress::objects::LinTermImplementation::div(Expression arg) const
    -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return div(arg.cast<ConstantExpression>());
  default:
    return AbstractExpressionImplementation::div(arg);
  }
}

auto xpress::objects::LinTermImplementation::div(ConstantExpression arg) const
    -> xpress::objects::LinTerm {
  return LinTerm(getVariable(), getCoefficient() / arg.getValue());
}

auto xpress::objects::LinTermImplementation::div(double arg) const
    -> xpress::objects::Expression {
  return LinTerm(getVariable(), getCoefficient() / arg);
}

xpress::objects::LinTerm xpress::objects::operator-(LinTerm const &term) {
  return term.uminus().cast<LinTerm>();
}

xpress::objects::LinTerm xpress::objects::operator*(LinTerm const &l,
                                                    double d) {
  return l.mul(d).cast<LinTerm>();
}
xpress::objects::LinTerm xpress::objects::operator*(double d,
                                                    LinTerm const &l) {
  return l.mul(d).cast<LinTerm>();
}
xpress::objects::LinTerm
xpress::objects::operator*(LinTerm const &l, ConstantExpression const &d) {
  return l.mul(d).cast<LinTerm>();
}
xpress::objects::QuadTerm xpress::objects::operator*(LinTerm const &l,
                                                     Variable const &v) {
  return l.mul(v).cast<QuadTerm>();
}
xpress::objects::QuadTerm xpress::objects::operator*(LinTerm const &l,
                                                     LinTerm const &t) {
  return l.mul(t).cast<QuadTerm>();
}

xpress::objects::LinTerm xpress::objects::operator/(LinTerm const &l,
                                                    double d) {
  return l.div(d).cast<LinTerm>();
}
xpress::objects::LinTerm
xpress::objects::operator/(LinTerm const &l, ConstantExpression const &d) {
  return l.div(d).cast<LinTerm>();
}

;

;

auto xpress::objects::PWLImplementation::getIndex() const -> int {
  return index;
}

auto xpress::objects::PWLImplementation::updateIndex(int delta) -> void {
  this->index += delta;
}

auto xpress::objects::PWLImplementation::getIndexForProb(
    XpressProblem const *xprob) const -> int {
  if (!this->prob)
    throw NotInProblemException("deleted");

  if (this->prob != xprob && this->prob != xprob->toplevel)
    throw NotInProblemException("not in problem");
  return index;
}

auto xpress::objects::PWLImplementation::getProblem() const
    -> xpress::objects::XpressProblem * {
  return prob;
}

auto xpress::objects::PWLImplementation::getSerial() const -> XPRSint64 {
  return serial;
}

auto xpress::objects::PWLImplementation::markDeleted() -> void {
  prob = nullptr;
  index = -1;
}

auto xpress::objects::PWLImplementation::compareTo(PWL const &other) const
    -> int {
  assert(index >= -1);

  if (other.isSameImpl(this)) {

    return 0;
  } else if (!(!prob) && !(!other.getProblem()) && prob != other.getProblem()) {

    throw std::invalid_argument(
        "cannot compare objects from different problems");
  }

  else {

    if (serial < other.getSerial())
      return -1;
    else if (serial > other.getSerial())
      return 1;
    else
      return 0;
  }
}

auto xpress::objects::PWLImplementation::getHashCode() const -> std::size_t {
  union {
    XPRSint64 i;
    unsigned long long u;
  } u;
  static_assert(sizeof(u.i) == sizeof(serial), "");
  u.i = serial;
  return u.u;
}

auto xpress::objects::PWLImplementation::check(char const *operation) const
    -> void {
  if (!prob)
    throw NotInProblemException("PWL is deleted");
  if (!(!operation) && !prob->isOriginal())
    throw xpress::XPRSProblem::CannotPerformOperationException(operation,
                                                               "PWL");
}
xpress::objects::PWLImplementation::PWLImplementation(XpressProblem *prob,
                                                      long serial, int index)
    : prob(prob), serial(serial), index(index) {}

auto xpress::objects::PWLImplementation::getInput()
    -> xpress::objects::Variable {
  check("get input");
  std::vector<int> input(1);
  int npoints;

  std::vector<double> dummyXval(1);
  std::vector<double> dummyYval(1);
  std::vector<int> dummyStart(2);
  prob->getPwlCons(input, nullptr, dummyStart, dummyXval, dummyYval, 1,
                   &npoints, getIndex(), getIndex());
  return prob->variableForIndex(input[0]);
}

auto xpress::objects::PWLImplementation::getResultant()
    -> xpress::objects::Variable {
  check("get resultant");
  std::vector<int> resultant(1);
  int npoints;

  std::vector<double> dummyXval(1);
  std::vector<double> dummyYval(1);
  std::vector<int> dummyStart(2);
  prob->getPwlCons(nullptr, resultant, dummyStart, dummyXval, dummyYval, 1,
                   &npoints, getIndex(), getIndex());
  return prob->variableForIndex(resultant[0]);
}

auto xpress::objects::PWLImplementation::getBreakpoints()
    -> std::vector<xpress::PwlBreakpoint> {
  check("get breakpoints");
  int npoints;
  std::vector<int> start(2);
  prob->getPwlCons(nullptr, nullptr, start, nullptr, nullptr, 0, &npoints,
                   getIndex(), getIndex());
  std::vector<double> xval(npoints);
  std::vector<double> yval(npoints);
  prob->getPwlCons(nullptr, nullptr, start, xval, yval, npoints, &npoints,
                   getIndex(), getIndex());
  return xpress::PwlBreakpoint::convert(xval, yval);
}

auto xpress::objects::PWLImplementation::getName() const -> std::string {
  check("get name");
  return prob->getName(xpress::Namespaces::PwlCons, getIndex());
}

auto xpress::objects::PWLImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::PWLImplementation & {
  check("set name");
  int idx = getIndex();
  prob->addNames(xpress::Namespaces::PwlCons,
                 std::vector<std::optional<std::string>>({newName}), idx, idx);
  return *this;
}

;

;

xpress::objects::PWLDefinitionImplementation::PWLDefinitionImplementation(
    xpress::objects::Variable input, xpress::objects::Variable resultant,
    xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval,
    std::optional<std::string> name)
    : ConstraintDefinitionImplementation<PWL>(name), input(input),
      resultant(resultant), xval(xval.toVector()), yval(yval.toVector()) {
  if (xpress::toInt(xval.size()) != xpress::toInt(yval.size()))
    throw std::invalid_argument("xval and yval have different length");
}

auto xpress::objects::PWLDefinitionImplementation::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  creator->addPWL(
      input.getIndexForProb((xpress::objects::XpressProblem *)creator->prob),
      resultant.getIndexForProb(
          (xpress::objects::XpressProblem *)creator->prob),
      xval, yval, getName());
}

auto xpress::objects::PWLDefinitionImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::PWLDefinitionImplementation & {
  setConstraintName(newName);
  return *this;
}

auto xpress::objects::PWLDefinitionImplementation::getIndexHandler(
    XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::PWL> const * {
  return prob->pwlHandler.get();
}

xpress::objects::QPair::QPair(xpress::objects::Variable v1,
                              xpress::objects::Variable v2)
    : xpress::maps::ComparableMapKey2<xpress::objects::Variable,
                                      xpress::objects::Variable>(
          xpress::objects::LinQuadUtils::isVariablePairOrdered(v1, v2) ? v1
                                                                       : v2,
          xpress::objects::LinQuadUtils::isVariablePairOrdered(v1, v2) ? v2
                                                                       : v1) {}

xpress::objects::QPair::QPair(QPair const &q)
    : QPair(q.getKey1(), q.getKey2()) {}

xpress::objects::QPair::QPair()
    : xpress::maps::ComparableMapKey2<xpress::objects::Variable,
                                      xpress::objects::Variable>(
          XpressProblem::NULL_VARIABLE, XpressProblem::NULL_VARIABLE) {}

auto xpress::objects::QPair::getVar1() const -> xpress::objects::Variable {
  return getKey1();
}

auto xpress::objects::QPair::getVar2() const -> xpress::objects::Variable {
  return getKey2();
}

auto xpress::objects::QPair::isLinear() -> bool {
  return XpressProblem::isNullVariable(getVar2()) &&
         !XpressProblem::isNullVariable(getVar1());
}

auto xpress::objects::QPair::isConstant() -> bool {

  assert(!XpressProblem::isNullVariable(getVar1()) ||
         XpressProblem::isNullVariable(getVar2()));
  return XpressProblem::isNullVariable(getVar1());
}

auto xpress::objects::QPair::toString() const -> std::string {
  std::stringstream buffer;
  buffer << "(";
  buffer << getVar1().toString();
  buffer << ",";
  buffer << getVar2().toString();
  buffer << ")";
  return buffer.str();
}

xpress::objects::QuadCoef::QuadCoef(QPair variables, double coefficient)
    : variables(variables), coefficient(coefficient) {}

xpress::objects::QuadCoef::QuadCoef(Variable var1, Variable var2,
                                    double coefficient)
    : QuadCoef(QPair(var1, var2), coefficient) {}
xpress::objects::QuadCoef::QuadCoef(QuadCoef const &other)
    : QuadCoef(other.getVariables(), other.getCoefficient()) {}

xpress::objects::QuadCoef::QuadCoef() : QuadCoef(QPair(), 0.0) {}

auto xpress::objects::QuadCoef::getVar1() -> xpress::objects::Variable {
  return getVariables().getVar1();
}

auto xpress::objects::QuadCoef::getVar2() -> xpress::objects::Variable {
  return getVariables().getVar2();
}

auto xpress::objects::QuadExpressionImplementation::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return ExpressionRTTI::QuadExpression;
}

auto xpress::objects::QuadExpressionImplementation::normalizeTerms()
    -> xpress::objects::QuadExpressionImplementation & {

  return *this;
}

;

;

;

auto xpress::objects::QuadExpressionImplementation::prepareToAdd(int newTerms)
    -> void {

  (void)newTerms;
}

template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::QuadExpressionImplementation::addTerms(
    Strm0 const &variables, Strm1 const &coefficients)
    -> xpress::objects::QuadExpressionImplementation & {
  prepareToAdd(std::max(0, std::max(xpress::sizeHint(variables),
                                    xpress::sizeHint(coefficients))));
  using std::begin;
  using std::end;
  auto vCurrent = begin(variables);
  auto vEnd = end(variables);
  using std::begin;
  using std::end;
  auto cCurrent = begin(coefficients);
  auto cEnd = end(coefficients);
  while (vCurrent != vEnd && cCurrent != cEnd) {
    auto const &v = *vCurrent;
    auto const &c = *cCurrent;

    addTerm(v, c);

    ++vCurrent;
    ++cCurrent;
  }
  if (vCurrent != vEnd || cCurrent != cEnd)
    throw std::invalid_argument("input data has different dimensions");
  return *this;
}

template <typename Strm0, typename Strm1, typename Strm2,
          typename Strm0IsStream, typename Strm1IsStream,
          typename Strm2IsStream>
auto xpress::objects::QuadExpressionImplementation::addTerms(
    Strm0 const &variables1, Strm1 const &variables2, Strm2 const &coefficients)
    -> xpress::objects::QuadExpressionImplementation & {
  prepareToAdd(std::max(0, std::max(std::max(xpress::sizeHint(variables1),
                                             xpress::sizeHint(variables2)),
                                    xpress::sizeHint(coefficients))));
  using std::begin;
  using std::end;
  auto v1Current = begin(variables1);
  auto v1End = end(variables1);
  using std::begin;
  using std::end;
  auto v2Current = begin(variables2);
  auto v2End = end(variables2);
  using std::begin;
  using std::end;
  auto cCurrent = begin(coefficients);
  auto cEnd = end(coefficients);
  while (v1Current != v1End && v2Current != v2End && cCurrent != cEnd) {
    auto const &v1 = *v1Current;
    auto const &v2 = *v2Current;
    auto const &c = *cCurrent;

    addTerm(v1, v2, c);

    ++v1Current;
    ++v2Current;
    ++cCurrent;
  }
  if (v1Current != v1End || v2Current != v2End || cCurrent != cEnd)
    throw std::invalid_argument("input data has different dimensions");
  return *this;
}

auto xpress::objects::QuadExpressionImplementation::addTerms(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &coefficients, int offset,
    int count) -> xpress::objects::QuadExpressionImplementation & {
  prepareToAdd(count);
  for (int i = 0; i < count; ++i)
    addTerm(variables[offset + i], coefficients[offset + i]);
  return *this;
}

auto xpress::objects::QuadExpressionImplementation::addTerms(
    xpress::SizedArray<Variable const> const &variables1,
    xpress::SizedArray<Variable const> const &variables2,
    xpress::SizedArray<double const> const &coefficients, int offset,
    int count) -> xpress::objects::QuadExpressionImplementation & {
  prepareToAdd(count);
  for (int i = 0; i < count; ++i)
    addTerm(variables1[offset + i], variables2[offset + i],
            coefficients[offset + i]);
  return *this;
}

;

;

auto xpress::objects::QuadExpressionImplementation::addTerm(LinTerm const &term)
    -> xpress::objects::QuadExpressionImplementation & {
  return addTerm(term.getVariable(), term.getCoefficient());
}

auto xpress::objects::QuadExpressionImplementation::addTerm(
    QuadTerm const &term) -> xpress::objects::QuadExpressionImplementation & {
  return addTerm(term.getVariable1(), term.getVariable2(),
                 term.getCoefficient());
}

;

;

;

;

auto xpress::objects::QuadExpressionImplementation::create()
    -> xpress::objects::QuadExpression {
  return QuadTermMap();
}

auto xpress::objects::QuadExpressionImplementation::create(double constant)
    -> xpress::objects::QuadExpression {
  return QuadTermMap(constant);
}

auto xpress::objects::QuadExpressionImplementation::getMaxDegree() const
    -> int {
  return 2;
}

xpress::objects::QuadTermImplementation::QuadTermImplementation(
    xpress::objects::Variable variable1, xpress::objects::Variable variable2,
    double coefficient)
    : variable1(xpress::objects::LinQuadUtils::isVariablePairOrdered(variable1,
                                                                     variable2)
                    ? variable1
                    : variable2),
      variable2(xpress::objects::LinQuadUtils::isVariablePairOrdered(variable1,
                                                                     variable2)
                    ? variable2
                    : variable1),
      coefficient(coefficient) {
  if (LinQuadUtils::isConstant(variable1, variable2) ||
      LinQuadUtils::isLinear(variable1, variable2))
    throw std::invalid_argument("quadratic terms must have two variables");
}

auto xpress::objects::QuadTermImplementation::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return ExpressionRTTI::QuadTerm;
}

auto xpress::objects::QuadTermImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  creator->checkForVariables();
  int x1 = xpress::objects::LinQuadUtils::index4Var(
      (XpressProblem *)creator->prob, getVariable1());
  int x2 = xpress::objects::LinQuadUtils::index4Var(
      (XpressProblem *)creator->prob, getVariable2());
  creator->addNonzero(x1, x2,

                      ((x1 != x2) ? 0.5 : 1.0) * factor * getCoefficient());
  return false;
}

auto xpress::objects::QuadTermImplementation::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  double coef = factor * getCoefficient();
  if (coef == 0.0)
    extractor->addToken(XPRS_TOK_CON, 0.0);
  else {
    if (coef != 1.0)
      extractor->addToken(XPRS_TOK_CON, coef);
    extractor->addToken(XPRS_TOK_COL,
                        extractor->dereferenceVariable(getVariable1()));
    extractor->addToken(XPRS_TOK_COL,
                        extractor->dereferenceVariable(getVariable2()));
    extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
    if (coef != 1.0)
      extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
  }
  extractor->endExpression();
}

auto xpress::objects::QuadTermImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  if (xpress::objects::LinQuadUtils::isConstant(getVariable1(), getVariable2()))
    return getCoefficient();
  else if (xpress::objects::LinQuadUtils::isLinear(getVariable1(),
                                                   getVariable2()))
    return getCoefficient() * solution[getVariable1().getIndex()];
  else
    return getCoefficient() * solution[getVariable1().getIndex()] *
           solution[getVariable2().getIndex()];
}

auto xpress::objects::QuadTermImplementation::qTerm(
    xpress::objects::Variable variable1, xpress::objects::Variable variable2,
    double coefficient) -> xpress::objects::QuadTerm {
  return QuadTerm(variable1, variable2, coefficient);
}

auto xpress::objects::QuadTermImplementation::qTerm(
    xpress::objects::Variable variable1,
    xpress::objects::Variable variable2) -> xpress::objects::QuadTerm {
  return QuadTerm(variable1, variable2, 1.0);
}

auto xpress::objects::QuadTermImplementation::toString() const -> std::string {
  return toString(getVariable1(), getVariable2(), getCoefficient());
}

auto xpress::objects::QuadTermImplementation::toString(
    xpress::objects::Variable v1, xpress::objects::Variable v2,
    double coef) -> std::string {
  if (coef == 0.0)
    return "0.0";

  bool ordered = LinQuadUtils::isVariablePairOrdered(v1, v2);
  if (!ordered) {
    Variable tmp = v1;
    v1 = v2;
    v2 = tmp;
  }

  if (LinQuadUtils::isConstant(v1, v2))
    return InternalUtils::floatToString(coef);
  else if (LinQuadUtils::isLinear(v1, v2)) {
    if (coef == 1.0)
      return v1.toString();
    else if (coef == -1.0)
      return xpress::concatStrings("-", v1.toString());
    else
      return xpress::concatStrings(InternalUtils::floatToString(coef), "*",
                                   v1.toString());
  } else {
    if (coef == 1.0)
      return xpress::concatStrings(v1.toString(), "*", v2.toString());
    else if (coef == -1.0)
      return xpress::concatStrings("-", v1.toString(), "*", v2.toString());
    else
      return xpress::concatStrings(InternalUtils::floatToString(coef), "*",
                                   v1.toString(), "*", v2.toString());
  }
}

auto xpress::objects::QuadTermImplementation::getMaxDegree() const -> int {
  return 2;
}

auto xpress::objects::QuadTermImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return InternalUtils::getSingletonQuadView(getVariable1(), getVariable2(),
                                             getCoefficient());
}

auto xpress::objects::QuadTermImplementation::uminus() const
    -> xpress::objects::Expression {
  return QuadTerm(getVariable1(), getVariable2(), -getCoefficient());
}

auto xpress::objects::QuadTermImplementation::mul(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return mul(arg.cast<ConstantExpression>());
  default:
    return AbstractExpressionImplementation::mul(arg);
  }
}

auto xpress::objects::QuadTermImplementation::mul(ConstantExpression arg) const
    -> xpress::objects::QuadTerm {
  return QuadTerm(getVariable1(), getVariable2(),
                  getCoefficient() * arg.getValue());
}

auto xpress::objects::QuadTermImplementation::mul(double arg) const
    -> xpress::objects::Expression {
  return QuadTerm(getVariable1(), getVariable2(), getCoefficient() * arg);
}

auto xpress::objects::QuadTermImplementation::div(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  switch (arg.getRTTI()) {
  case ExpressionRTTI::Constant:
    return div(arg.cast<ConstantExpression>());
  default:
    return AbstractExpressionImplementation::div(arg);
  }
}

auto xpress::objects::QuadTermImplementation::div(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::QuadTerm {
  return QuadTerm(getVariable1(), getVariable2(),
                  getCoefficient() / arg.getValue());
}

auto xpress::objects::QuadTermImplementation::div(double arg) const
    -> xpress::objects::Expression {
  return QuadTerm(getVariable1(), getVariable2(), getCoefficient() / arg);
}

xpress::objects::QuadTerm xpress::objects::operator-(QuadTerm const &term) {
  return term.uminus().cast<QuadTerm>();
}

xpress::objects::QuadTerm xpress::objects::operator*(QuadTerm const &q,
                                                     double d) {
  return q.mul(d).cast<QuadTerm>();
}
xpress::objects::QuadTerm xpress::objects::operator*(double d,
                                                     QuadTerm const &q) {
  return q.mul(d).cast<QuadTerm>();
}
xpress::objects::QuadTerm
xpress::objects::operator*(QuadTerm const &q, ConstantExpression const &d) {
  return q.mul(d).cast<QuadTerm>();
}

xpress::objects::QuadTerm xpress::objects::operator/(QuadTerm const &q,
                                                     double d) {
  return q.div(d).cast<QuadTerm>();
}

;

;

auto xpress::objects::SOSImplementation::getIndex() const -> int {
  return index;
}

auto xpress::objects::SOSImplementation::updateIndex(int delta) -> void {
  this->index += delta;
}

auto xpress::objects::SOSImplementation::getIndexForProb(
    XpressProblem const *xprob) const -> int {
  if (!this->prob)
    throw NotInProblemException("deleted");

  if (this->prob != xprob && this->prob != xprob->toplevel)
    throw NotInProblemException("not in problem");
  return index;
}

auto xpress::objects::SOSImplementation::getProblem() const
    -> xpress::objects::XpressProblem * {
  return prob;
}

auto xpress::objects::SOSImplementation::getSerial() const -> XPRSint64 {
  return serial;
}

auto xpress::objects::SOSImplementation::markDeleted() -> void {
  prob = nullptr;
  index = -1;
}

auto xpress::objects::SOSImplementation::compareTo(SOS const &other) const
    -> int {
  assert(index >= -1);

  if (other.isSameImpl(this)) {

    return 0;
  } else if (!(!prob) && !(!other.getProblem()) && prob != other.getProblem()) {

    throw std::invalid_argument(
        "cannot compare objects from different problems");
  }

  else {

    if (serial < other.getSerial())
      return -1;
    else if (serial > other.getSerial())
      return 1;
    else
      return 0;
  }
}

auto xpress::objects::SOSImplementation::getHashCode() const -> std::size_t {
  union {
    XPRSint64 i;
    unsigned long long u;
  } u;
  static_assert(sizeof(u.i) == sizeof(serial), "");
  u.i = serial;
  return u.u;
}

auto xpress::objects::SOSImplementation::check(char const *operation) const
    -> void {
  if (!prob)
    throw NotInProblemException("SOS is deleted");
  if (!(!operation) && !prob->isOriginal())
    throw xpress::XPRSProblem::CannotPerformOperationException(operation,
                                                               "SOS");
}
xpress::objects::SOSImplementation::SOSImplementation(XpressProblem *prob,
                                                      long serial, int index)
    : prob(prob), serial(serial), index(index) {}

auto xpress::objects::SOSImplementation::getName() const -> std::string {
  check("get name");
  return prob->getName(xpress::Namespaces::Set, getIndex());
}

auto xpress::objects::SOSImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::SOSImplementation & {
  check("set name");
  int idx = getIndex();
  prob->addNames(xpress::Namespaces::Set,
                 std::vector<std::optional<std::string>>({newName}), idx, idx);
  return *this;
}

auto xpress::objects::SOSImplementation::sos(
    xpress::SetType type, xpress::SizedArray<Variable const> const &items,
    xpress::SizedArray<double const> const &weights,
    std::optional<std::string> name) -> xpress::objects::SOSDefinition {
  return xpress::objects::SOSDefinition(type, items, weights, name);
}

auto xpress::objects::SOSImplementation::sos1(
    xpress::SizedArray<Variable const> const &items,
    xpress::SizedArray<double const> const &weights,
    std::optional<std::string> name) -> xpress::objects::SOSDefinition {
  return sos(xpress::SetType::SOS1, items, weights, name);
}

auto xpress::objects::SOSImplementation::sos2(
    xpress::SizedArray<Variable const> const &items,
    xpress::SizedArray<double const> const &weights,
    std::optional<std::string> name) -> xpress::objects::SOSDefinition {
  return sos(xpress::SetType::SOS2, items, weights, name);
}

;

;

xpress::objects::SOSDefinitionImplementation::SOSDefinitionImplementation(
    xpress::SetType type, xpress::SizedArray<Variable const> const &items,
    xpress::SizedArray<double const> const &weights,
    std::optional<std::string> name)
    : ConstraintDefinitionImplementation<SOS>(name), type(type),
      sosElems(items.toVector()),
      sosWeights(weights ? weights.toVector() : std::vector<double>()),
      defaultWeights((!weights)) {
  if (!(!weights) &&
      xpress::toInt(weights.size()) != xpress::toInt(items.size()))
    throw std::invalid_argument(
        "special ordered set items and weights have different length");
}

auto xpress::objects::SOSDefinitionImplementation::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  if (!(dynamic_cast<XpressProblem *>(creator->prob) != nullptr))
    throw std::invalid_argument(
        "cannot create SOS constraints that reference Variable objects");
  XpressProblem *xprob = (XpressProblem *)creator->prob;
  if (!xprob->isOriginal())
    throw xpress::XPRSProblem::CannotReferenceVariablesException();
  std::vector<int> elems(xpress::toInt(sosElems.size()));
  for (int i = 0; i < xpress::toInt(elems.size()); ++i)
    elems[i] = sosElems[i].getIndex();
  if (defaultWeights)
    creator->addSet(type, elems, nullptr, getName());
  else
    creator->addSet(type, elems, sosWeights, getName());
}

auto xpress::objects::SOSDefinitionImplementation::setName(
    std::optional<std::string> newName)
    -> xpress::objects::SOSDefinitionImplementation & {
  setConstraintName(newName);
  return *this;
}

auto xpress::objects::SOSDefinitionImplementation::getIndexHandler(
    XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::SOS> const * {
  return prob->sosHandler.get();
}

;

xpress::objects::SumExpressionImplementation::SumExpressionImplementation(
    xpress::SizedArray<Expression const> const &terms)
    : terms(terms.begin(), terms.end()), maxDegree(-1) {}

auto xpress::objects::SumExpressionImplementation::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return ExpressionRTTI::Sum;
}

auto xpress::objects::SumExpressionImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  if (xpress::toInt(terms.size()) == 0)
    return false;

  bool normalize = false;
  bool first = true;
  for (auto &t : terms) {
    if (t.getMaxDegree() > 2) {
      t.extract(factor, creator->getPostfixExtractor());
    } else {
      if (t.extract(factor, creator) || !first)
        normalize = true;
      first = false;
    }
  }
  return normalize;
}

auto xpress::objects::SumExpressionImplementation::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  if (xpress::toInt(terms.size()) == 0)
    return;
  extractor->startExpression();
  extractor->addToken(XPRS_TOK_RB, 0.0);
  bool comma = false;
  for (int i = xpress::toInt(terms.size()) - 1; i >= 0; --i) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    terms[i].extract(1.0, extractor);
    comma = true;
  }
  extractor->addToken(XPRS_TOK_IFUN, XPRS_IFUN_SUM);
  if (factor != 1.0) {
    extractor->addToken(XPRS_TOK_CON, factor);
    extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
  }
  extractor->endExpression();
}

auto xpress::objects::SumExpressionImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  double result = 0.0;
  for (auto &t : terms)
    result += t.evaluate(solution);
  return result;
}

auto xpress::objects::SumExpressionImplementation::toString() const
    -> std::string {
  if (xpress::toInt(terms.size()) == 0)
    return "0";
  std::stringstream buffer;
  buffer << terms[0].toString();
  for (int i = 1; i < xpress::toInt(terms.size()); ++i) {
    buffer << " + ";
    buffer << terms[i].toString();
  }
  return buffer.str();
}

auto xpress::objects::SumExpressionImplementation::getMaxDegree() const -> int {

  int result = maxDegree;
  if (result < 0) {
    result = 0;
    for (int i = 0; i < xpress::toInt(terms.size()); ++i) {
      int degree = terms[i].getMaxDegree();
      if (degree > result) {
        result = degree;
        if (result == std::numeric_limits<int>::max())
          break;
      }
    }
    maxDegree = result;
  }
  return result;
}

auto xpress::objects::SumExpressionImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  if (getMaxDegree() > 1)
    throw std::invalid_argument("not a linear expression");
  return std::make_shared<
      xpress::SumView<LinearView, Expression, ExpressionImplementation>>(
      const_cast<SumExpressionImplementation *>(this)->shared_from_this(),
      terms.data(), xpress::toInt(terms.size()),
      [](Expression const &e) -> std::shared_ptr<LinearView> {
        return e.getLinearView();
      });
}

auto xpress::objects::SumExpressionImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  if (getMaxDegree() > 2)
    throw std::invalid_argument("not a quadratic expression");
  return std::make_shared<
      xpress::SumView<QuadView, Expression, ExpressionImplementation>>(
      const_cast<SumExpressionImplementation *>(this)->shared_from_this(),
      terms.data(), xpress::toInt(terms.size()),
      [](Expression const &e) -> std::shared_ptr<QuadView> {
        return e.getQuadView();
      });
}

xpress::objects::FormulaExpressionImplementation::
    FormulaExpressionImplementation() {}

auto xpress::objects::FormulaExpressionImplementation::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return ExpressionRTTI::Formula;
}

;

auto xpress::objects::FormulaExpressionImplementation::extract(
    double factor, PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  extract(extractor);
  if (factor != 1.0) {
    extractor->addToken(XPRS_TOK_CON, factor);
    extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
  }
  extractor->endExpression();
}

auto xpress::objects::FormulaExpressionImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  extract(factor, creator->getPostfixExtractor());
  return false;
}

;

template <typename T>
auto xpress::objects::FormulaExpressionImplementation::parseUnaryFunction(
    SymbolicEvalStack &stack,
    T type) -> xpress::objects::InternalFunctionExpression {
  Expression arg = stack.pop();
  if (!stack.pop().isSameImpl(dummyRB))
    throw std::invalid_argument("malformed formula");
  return InternalFunctionExpression(type, arg);
}

template <typename T, typename Func0, typename Func0IsInvocable>
auto xpress::objects::FormulaExpressionImplementation::reverseArgs(
    std::vector<T> const &args, Func0 mkArray) -> std::vector<T> {
  std::vector<T> reversed = mkArray(args.size());
  int next = xpress::toInt(reversed.size());
  for (auto &e : args) {
    reversed[--next] = e;
  }
  return reversed;
}

template <typename T>
auto xpress::objects::FormulaExpressionImplementation::parseNaryFunction(
    SymbolicEvalStack &stack,
    T type) -> xpress::objects::InternalFunctionExpression {
  std::vector<Expression> args;
  for (Expression arg = stack.pop(); !arg.isSameImpl(dummyRB);
       arg = stack.pop()) {
    args.push_back(arg);
  }
  return InternalFunctionExpression(type, args);
}

auto xpress::objects::FormulaExpressionImplementation::parse(
    XpressProblem *prob, xpress::SizedArray<int const> const &type,
    xpress::SizedArray<double const> const &value)
    -> xpress::objects::Expression {

  if (xpress::toInt(type.size()) != xpress::toInt(value.size()))
    throw std::invalid_argument("arrays have different length");

  std::vector<Token> tokens;
  for (int i = 0; i < xpress::toInt(type.size()); ++i) {
    if (type[i] == XPRS_TOK_EOF)
      break;
    tokens.push_back(Token::of(type[i], value[i]));
  }

  SymbolicEvalStack stack = SymbolicEvalStack(xpress::toInt(tokens.size()));
  Expression left, right;
  for (auto &t : tokens) {
    switch (t.getType()) {
    case XPRS_TOK_EOF:
      assert(false);
      break;
    case XPRS_TOK_LB:
      throw std::invalid_argument(xpress::concatStrings(
          "invalid token ", t.toString(), " for parsed formula"));
    case XPRS_TOK_RB:
      stack.push(dummyRB);
      break;
    case XPRS_TOK_OP:
      right = stack.pop();
      switch ((int)t.getValue()) {
      case XPRS_OP_UMINUS:
        stack.push(UnaryExpression(UnaryExpression::Operator::UMinus, right));
        break;
      case XPRS_OP_EXPONENT:
        [[fallthrough]];
      case XPRS_OP_MULTIPLY:
        [[fallthrough]];
      case XPRS_OP_DIVIDE:
        [[fallthrough]];
      case XPRS_OP_PLUS:
        [[fallthrough]];
      case XPRS_OP_MINUS:
        left = stack.pop();
        stack.push(BinaryExpression(left,
                                    static_cast<BinaryExpression::Operator>(
                                        static_cast<int>(t.getValue())),
                                    right));
        break;
      default:
        throw std::invalid_argument(
            xpress::concatStrings("unknown operator ", t.toString()));
      }
      break;
    case XPRS_TOK_DEL:
      switch ((int)t.getValue()) {
      case XPRS_DEL_COMMA:

        break;
      case XPRS_DEL_COLON:
        stack.push(dummyCOLON);
        break;
      default:
        throw std::invalid_argument(
            xpress::concatStrings("unknown delimiter ", t.toString()));
      }
      break;
    case XPRS_TOK_CON:
      stack.push(ConstantExpression(t.getValue()));
      break;
    case XPRS_TOK_COL:
      stack.push(prob->variableForIndex((int)t.getValue()));

      break;
    case XPRS_TOK_FUN: {
      std::vector<Expression> args;
      while (!stack.peek().isSameImpl(dummyRB) &&
             !stack.peek().isSameImpl(dummyCOLON))
        args.push_back(stack.pop());
      int output = 1;
      if (stack.peek().isSameImpl(dummyCOLON)) {
        stack.pop();
        Expression e = stack.pop();
        if (!e.isInstance<ConstantExpressionImplementation>())
          throw std::invalid_argument("malformed user function");
        ConstantExpression c = e.cast<ConstantExpression>();
        if ((int)std::round(c.getValue()) != c.getValue())
          throw std::invalid_argument("malformed user function");
        output = (int)std::round(c.getValue());
      }
      if (!stack.pop().isSameImpl(dummyRB))
        throw std::invalid_argument("malformed formula");

      if (true)
        throw std::runtime_error("user functions currently not supported");
      (void)output;
    } break;
    case XPRS_TOK_IFUN: {
      Expression func = nullptr;

      switch ((int)t.getValue()) {

      case XPRS_IFUN_LOG10:
        [[fallthrough]];
      case XPRS_IFUN_LN:
        [[fallthrough]];
      case XPRS_IFUN_EXP:
        [[fallthrough]];
      case XPRS_IFUN_ABS:
        [[fallthrough]];
      case XPRS_IFUN_SQRT:
        [[fallthrough]];
      case XPRS_IFUN_SIN:
        [[fallthrough]];
      case XPRS_IFUN_COS:
        [[fallthrough]];
      case XPRS_IFUN_TAN:
        [[fallthrough]];
      case XPRS_IFUN_ARCSIN:
        [[fallthrough]];
      case XPRS_IFUN_ARCCOS:
        [[fallthrough]];
      case XPRS_IFUN_ARCTAN:
        [[fallthrough]];
      case XPRS_IFUN_SIGN:
        func = parseUnaryFunction(
            stack, static_cast<InternalFunctionExpression::FunctionType>(
                       static_cast<int>(t.getValue())));
        break;

      case XPRS_IFUN_MIN:
        [[fallthrough]];
      case XPRS_IFUN_MAX:
        [[fallthrough]];
      case XPRS_IFUN_SUM:
        [[fallthrough]];
      case XPRS_IFUN_PROD:
        func = parseNaryFunction(
            stack, static_cast<InternalFunctionExpression::FunctionType>(
                       static_cast<int>(t.getValue())));
        break;

      case XPRS_IFUN_PWL:
        if (!stack.peek().isInstance<VariableImplementation>())
          throw std::invalid_argument("malformed PWL");
        Variable x = stack.pop().cast<Variable>();
        std::vector<xpress::PwlBreakpoint> breaks;
        while (!stack.peek().isSameImpl(dummyRB)) {
          Expression breakX = stack.pop();
          Expression breakY = stack.pop();
          if (!breakX.isInstance<ConstantExpressionImplementation>() ||
              !breakX.isInstance<ConstantExpressionImplementation>())
            throw std::invalid_argument("malformed PWL");
          breaks.push_back(xpress::PwlBreakpoint(
              breakX.cast<ConstantExpression>().getValue(),
              breakY.cast<ConstantExpression>().getValue()));
        }
        stack.pop();
        func = PWLExpression(x, reverseArgs(breaks, [=](auto l) {
                               return std::vector<xpress::PwlBreakpoint>(l);
                             }));
        break;
      }
      if (!func)
        throw std::invalid_argument(
            xpress::concatStrings("unknown internal function ", t.toString()));
      stack.push(func);
    } break;
    default:
      throw std::invalid_argument(
          xpress::concatStrings("unknown token ", t.toString()));
    }
  }
  if (stack.size() != 1)
    throw std::invalid_argument("malformed formula");
  return stack.pop();
}

;
xpress::objects::LinExprProductView::LinExprProductView(
    std::shared_ptr<LinearView> e1, std::shared_ptr<LinearView> e2)
    : e1(e1), e2(e2){}

;
;
;
;
xpress::objects::LinExprProductView::WrappedIteratorImpl::WrappedIteratorImpl(
    std::shared_ptr<LinearView> outer, std::shared_ptr<LinearView> inner,
    bool isEnd)
    : outer(outer), outerCurrent(nullptr), outerEnd(nullptr), inner(inner),
      innerCurrent(nullptr), innerEnd(nullptr), value(QuadCoef()) {
  if (isEnd) {
    outerCurrent = outer->end();
    outerEnd = outer->end();
    innerCurrent = inner->end();
    innerEnd = inner->end();
  } else {
    outerCurrent = outer->begin();
    outerEnd = outer->end();
    innerCurrent = inner->begin();
    innerEnd = inner->end();

    if (!(innerCurrent != innerEnd))
      outerCurrent = outerEnd;

    else if (!(outerCurrent != outerEnd))
      innerCurrent = innerEnd;
    if (outerCurrent != outerEnd && innerCurrent != innerEnd) {
      value = QuadCoef(
          (*outerCurrent).getVariable(), (*innerCurrent).getVariable(),
          (*outerCurrent).getCoefficient() * (*innerCurrent).getCoefficient());
    }
  }
}
auto xpress::objects::LinExprProductView::WrappedIteratorImpl::operator!=(
    WrappedIterator<QuadCoef> const &other) const -> bool {
  WrappedIteratorImpl const *o =
      dynamic_cast<WrappedIteratorImpl const *>(other.get());
  return o->outerCurrent != outerCurrent || o->innerCurrent != innerCurrent;
}
auto xpress::objects::LinExprProductView::WrappedIteratorImpl::operator*()
    -> xpress::objects::QuadCoef & {
  return value;
}
auto xpress::objects::LinExprProductView::WrappedIteratorImpl::operator++()
    -> xpress::objects::QuadCoef & {
  if (outerCurrent != outerEnd) {
    ++innerCurrent;
    if (innerCurrent != innerEnd) {
      value = QuadCoef(
          (*outerCurrent).getVariable(), (*innerCurrent).getVariable(),
          (*outerCurrent).getCoefficient() * (*innerCurrent).getCoefficient());
    } else {
      ++outerCurrent;
      if (outerCurrent != outerEnd) {
        innerCurrent = inner->begin();
        innerEnd = inner->end();
        assert(innerCurrent != innerEnd);
        value = QuadCoef((*outerCurrent).getVariable(),
                         (*innerCurrent).getVariable(),
                         (*outerCurrent).getCoefficient() *
                             (*innerCurrent).getCoefficient());
      }
    }
  }
  return value;
}

auto xpress::objects::LinExprProductView::begin() const
    -> xpress::WrappedIterator<xpress::objects::QuadCoef> {
  return WrappedIterator<QuadCoef>(
      std::make_shared<WrappedIteratorImpl>(e1, e2, false));
}
auto xpress::objects::LinExprProductView::end() const
    -> xpress::WrappedIterator<xpress::objects::QuadCoef> {
  return WrappedIterator<QuadCoef>(
      std::make_shared<WrappedIteratorImpl>(e1, e2, true));
}

template <typename I>
xpress::objects::IISConstraint::IISConstraint(
    I constraint, std::optional<xpress::RowType> direction, double dual,
    IIS::Isolation isolation)
    : constraint(constraint), direction(direction), dual(dual),
      isolation(isolation) {}

template <typename M, typename T>
auto xpress::objects::InternalUtils::mergeCoefficient(M &m, T const &key,
                                                      double value) -> void {
  if (value == 0.0)
    return;
  typename M::iterator it = m.find(key);
  if (it == m.end()) {
    m[key] = value;
  } else {
    it->second += value;
    if (it->second == 0.0)
      m.erase(it);
  }
}

auto xpress::objects::InternalUtils::incsForSort() -> std::vector<int> const & {
  static std::vector<int> incs({1391376, 463792, 198768, 86961, 33936, 13776,
                                4592, 1968, 861, 336, 112, 48, 21, 7, 3, 1});
  return incs;
}

auto xpress::objects::InternalUtils::sortParallelArrays(
    xpress::SizedArray<int> const &mvec, xpress::SizedArray<double> const &dvec,
    int offset, int size) -> void {
  std::vector<int> const &incs = incsForSort();

  int i, j, k, h;

  for (k = 0; k < xpress::toInt(incs.size()); k++)
    for (h = incs[k], i = h - 1; i < size; i++) {
      int v = mvec[offset + i];
      double d = dvec[offset + i];
      j = i;
      while ((j >= h) && (mvec[offset + j - h] > v)) {
        mvec[offset + j] = mvec[offset + j - h];
        dvec[offset + j] = dvec[offset + j - h];
        j -= h;
      }
      mvec[offset + j] = v;
      dvec[offset + j] = d;
    }
}

auto xpress::objects::InternalUtils::sortParallelArrays(
    xpress::SizedArray<int> const &key1, xpress::SizedArray<int> const &key2,
    xpress::SizedArray<double> const &dvec, int offset, int size) -> void {
  std::vector<int> const &incs = incsForSort();

  int i, j, k, h;

  for (k = 0; k < xpress::toInt(incs.size()); k++)
    for (h = incs[k], i = h - 1; i < size; i++) {
      int v1 = key1[offset + i];
      int v2 = key2[offset + i];
      double d = dvec[offset + i];
      j = i;
      while ((j >= h) &&
             (key1[offset + j - h] > v1 ||
              (key1[offset + j - h] == v1 && key2[offset + j - h] > v2))) {
        key1[offset + j] = key1[offset + j - h];
        key2[offset + j] = key2[offset + j - h];
        dvec[offset + j] = dvec[offset + j - h];
        j -= h;
      }
      key1[offset + j] = v1;
      key2[offset + j] = v2;
      dvec[offset + j] = d;
    }
}

template <typename T, typename NoPrimitive>
auto xpress::objects::InternalUtils::sortParallelArrays(
    std::vector<T> &mvec, xpress::SizedArray<double> const &dvec, int offset,
    int size) -> void {

  std::vector<int> const &incs = incsForSort();
  int i, j, k, h;

  for (k = 0; k < xpress::toInt(incs.size()); k++) {
    for (h = incs[k], i = h - 1; i < size; i++) {
      T g = mvec[offset + i];
      double d = dvec[offset + i];
      j = i;
      while ((j >= h) && (mvec[offset + j - h].compareTo(g) > 0)) {
        mvec[offset + j] = mvec[offset + j - h];
        dvec[offset + j] = dvec[offset + j - h];
        j -= h;
      }
      mvec[offset + j] = g;
      dvec[offset + j] = d;
    }
  }
}

template <typename T, typename NoPrimitive>
auto xpress::objects::InternalUtils::sortParallelArrays(
    std::vector<T> &key1, std::vector<T> &key2,
    xpress::SizedArray<double> const &dvec, int offset, int size) -> void {

  std::vector<int> const &incs = incsForSort();
  int i, j, k, h;

  for (k = 0; k < xpress::toInt(incs.size()); k++) {
    for (h = incs[k], i = h - 1; i < size; i++) {
      T g1 = key1[offset + i];
      T g2 = key2[offset + i];
      double d = dvec[offset + i];
      j = i;
      while ((j >= h) && (key1[offset + j - h].compareTo(g1) > 0 ||
                          (key1[offset + j - h].compareTo(g1) == 0 &&
                           key2[offset + j - h].compareTo(g2) > 0))) {
        key1[offset + j] = key1[offset + j - h];
        key2[offset + j] = key2[offset + j - h];
        dvec[offset + j] = dvec[offset + j - h];
        j -= h;
      }
      key1[offset + j] = g1;
      key2[offset + j] = g2;
      dvec[offset + j] = d;
    }
  }
}

auto xpress::objects::InternalUtils::sortParallelArraysLex(
    xpress::SizedArray<double> const &key1,
    xpress::SizedArray<double> const &key2, int offset, int size) -> void {
  std::vector<int> const &incs = incsForSort();

  int i, j, k, h;

  for (k = 0; k < xpress::toInt(incs.size()); k++)
    for (h = incs[k], i = h - 1; i < size; i++) {
      double v1 = key1[offset + i];
      double v2 = key2[offset + i];
      j = i;
      while ((j >= h) &&
             (key1[offset + j - h] > v1 ||
              (key1[offset + j - h] == v1 && key2[offset + j - h] > v2))) {
        key1[offset + j] = key1[offset + j - h];
        key2[offset + j] = key2[offset + j - h];
        j -= h;
      }
      key1[offset + j] = v1;
      key2[offset + j] = v2;
    }
}

auto xpress::objects::InternalUtils::calcExtendedArraySize(int old,
                                                           int minAdd) -> int {
  return std::max(old * 2, std::max(old + minAdd, 8));
}

template <typename T>
auto xpress::objects::InternalUtils::extendArray(std::vector<T> &old,
                                                 int minAdd,
                                                 bool) -> std::vector<T> & {
  int newLength = calcExtendedArraySize(xpress::toInt(old.size()), minAdd);
  old.resize(newLength);
  return old;
}

auto xpress::objects::InternalUtils::getSingletonLinView(Variable v, double c)
    -> std::shared_ptr<xpress::objects::LinearView> {
  LinCoef value = LinCoef(v, c);
  return std::make_shared<xpress::SingletonView<LinearView>>(
      [=]() { return LinCoef(v, c); });
}

auto xpress::objects::InternalUtils::getSingletonQuadView(Variable v1,
                                                          Variable v2, double c)
    -> std::shared_ptr<xpress::objects::QuadView> {
  QuadCoef value = QuadCoef(QPair(v1, v2), c);
  return std::make_shared<xpress::SingletonView<QuadView>>(
      [=]() { return QuadCoef(v1, v2, c); });
}

auto xpress::objects::InternalUtils::floatToString(double value)
    -> std::string {
  if (std::isnan(value))
    return "NaN";
  else {

    std::stringstream s;
    s.imbue(std::locale("C"));
    if (std::round(value) == value) {
      s << std::fixed << std::setprecision(1) << value;
      return s.str();
    } else {
      s << std::fixed << std::setprecision(16) << value;
      std::string res = s.str();

      std::string::size_type i = res.size();
      while (i > 2 && res[i - 1] == '0')
        --i;
      res.resize(i);
      return res;
    }
  }
}

;

auto xpress::objects::LinTermListImplementation::setAutoCompress(bool compress)
    -> void {
  autoCompress = compress;
}

auto xpress::objects::LinTermListImplementation::isAutoCompress() -> bool {
  return autoCompress;
}

;

;

xpress::objects::LinTermListImplementation::LinTermListImplementation()
    : constant(0.0), length(0), ind({}), val({}) {}

xpress::objects::LinTermListImplementation::LinTermListImplementation(double c)
    : constant(c), length(0), ind({}), val({}) {}

xpress::objects::LinTermListImplementation::LinTermListImplementation(
    double c, int capacity)
    : constant(c), length(0), ind({}), val({}) {
  if (capacity > 0) {
    ind = std::vector<Variable>(capacity);
    val = std::vector<double>(capacity);
  }
}

auto xpress::objects::LinTermListImplementation::deepCopy()
    -> xpress::objects::LinExpression {
  return deepCopy(1.0);
}

auto xpress::objects::LinTermListImplementation::deepCopy(double factor)
    -> xpress::objects::LinExpression {
  if (factor == 0.0)
    return LinTermList();
  int len = length;
  LinTermList ret = LinTermList(constant * factor, length);
  for (int i = 0; i < len; ++i)
    ret.addTerm(ind[i], factor * val[i]);
  return ret;
}

auto xpress::objects::LinTermListImplementation::prepareToAdd(int newTerms)
    -> void {
  if (newTerms > 0)
    reserve(newTerms);
}

auto xpress::objects::LinTermListImplementation::reserve(int l)
    -> xpress::objects::LinTermListImplementation & {
  if (l <= 0)
    return *this;
  if (xpress::toInt(ind.size()) < length + l)
    ind = xpress::objects::InternalUtils::extendArray(ind, l, length > 0);
  if (xpress::toInt(val.size()) < length + l)
    val = xpress::objects::InternalUtils::extendArray(val, l, length > 0);
  return *this;
}

auto xpress::objects::LinTermListImplementation::reset()
    -> xpress::objects::LinTermListImplementation & {
  constant = 0;
  length = 0;
  return *this;
}

auto xpress::objects::LinTermListImplementation::addTerm(
    Variable const &variable,
    double const &coefficient) -> xpress::objects::LinTermListImplementation & {
  if (xpress::objects::LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("cannot add constant terms");
  reserve(1);
  ind[length] = variable;
  val[length] = coefficient;
  ++length;
  return *this;
}

auto xpress::objects::LinTermListImplementation::addTerm(
    double const &coefficient,
    Variable const &variable) -> xpress::objects::LinTermListImplementation & {
  return addTerm(variable, coefficient);
}

auto xpress::objects::LinTermListImplementation::addTerm(Variable const &x)
    -> xpress::objects::LinTermListImplementation & {
  return addTerm(x, 1.0);
}

auto xpress::objects::LinTermListImplementation::addTerms(
    LinExpression const &expr) -> xpress::objects::LinTermListImplementation & {
  return addTerms(expr, 1.0);
}

auto xpress::objects::LinTermListImplementation::addTerms(
    LinExpression const &expr,
    double factor) -> xpress::objects::LinTermListImplementation & {
  int start = length;
  constant += factor * expr.getConstant();
  std::shared_ptr<LinearView> view = expr.getLinearView();
  for (auto &e : *view) {
    if (!LinQuadUtils::isConstant(e.getVariable()))
      addTerm(e.getVariable(), factor * e.getCoefficient());
  }

  if (!expr.isInstance<LinTermListImplementation>()) {
    xpress::objects::InternalUtils::sortParallelArrays(ind, val, start,
                                                       length - start);
  }
  return *this;
}

auto xpress::objects::LinTermListImplementation::addConstant(double c)
    -> xpress::objects::LinTermListImplementation & {
  constant += c;
  return *this;
}

auto xpress::objects::LinTermListImplementation::setConstant(double c)
    -> xpress::objects::LinTermListImplementation & {
  constant = c;
  return *this;
}

auto xpress::objects::LinTermListImplementation::getConstant() const -> double {
  return constant;
}

auto xpress::objects::LinTermListImplementation::normalizeTerms()
    -> xpress::objects::LinTermListImplementation & {
  compress();
  return *this;
}

auto xpress::objects::LinTermListImplementation::compress() const -> void {
  xpress::objects::InternalUtils::sortParallelArrays(ind, val, 0, length);
  int outIdx = 0;
  Variable last = xpress::objects::XpressProblem::NULL_VARIABLE;
  for (int inIdx = 0; inIdx < length; ++inIdx) {
    Variable v = ind[inIdx];
    if (!v.equals(last)) {
      last = v;

      if (outIdx > 0 && val[outIdx - 1] == 0.0)
        --outIdx;
      ind[outIdx] = v;
      val[outIdx] = val[inIdx];
      ++outIdx;
    } else {
      val[outIdx - 1] += val[inIdx];
    }
  }

  if (outIdx > 0 && val[outIdx - 1] == 0.0)
    --outIdx;
  length = outIdx;
}

auto xpress::objects::LinTermListImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  if (isAutoCompress())
    compress();
  if (length > 0)
    creator->checkForVariables();
  XpressProblem *prob = (XpressProblem *)creator->prob;
  for (int i = 0; i < length; ++i) {
    if (val[i] != 0.0)
      creator->addNonzero(
          xpress::objects::LinQuadUtils::index4Var(prob, ind[i]),
          factor * val[i]);
  }
  creator->addRhs(-factor * constant);

  return false;
}

auto xpress::objects::LinTermListImplementation::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  if (length == 0 && constant == 0.0) {

    extractor->startExpression();
    extractor->addToken(XPRS_TOK_CON, 0.0);
    extractor->endExpression();
    return;
  }
  extractor->startExpression();
  if (isAutoCompress())
    compress();
  extractor->addToken(XPRS_TOK_RB, 0.0);
  bool comma = false;
  for (int i = length - 1; i >= 0; --i) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);

    double coef = factor * val[i];

    if (coef != 1.0)
      extractor->addToken(XPRS_TOK_CON, coef);
    if (coef != 0.0) {
      extractor->addToken(XPRS_TOK_COL, extractor->dereferenceVariable(ind[i]));
      if (coef != 1.0)
        extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
    }
    comma = true;
  }
  if (constant != 0.0) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    extractor->addToken(XPRS_TOK_CON, factor * constant);
    comma = true;
  }
  extractor->addToken(XPRS_TOK_IFUN, XPRS_IFUN_SUM);
  extractor->endExpression();
}

auto xpress::objects::LinTermListImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  double result = constant;
  for (int i = 0; i < length; ++i)
    result += val[i] * solution[ind[i].getIndex()];
  return result;
}

auto xpress::objects::LinTermListImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  if (constant == 0.0) {
    return std::make_shared<
        xpress::IndexedView<LinearView, ExpressionImplementation>>(
        const_cast<LinTermListImplementation *>(this)->shared_from_this(),
        length, [&](int idx) { return LinCoef(ind[idx], val[idx]); });
  } else {
    LinCoef initial = LinCoef(XpressProblem::NULL_VARIABLE, constant);

    return std::make_shared<
        xpress::IndexedView<LinearView, ExpressionImplementation>>(
        const_cast<LinTermListImplementation *>(this)->shared_from_this(),
        length, initial, [&](int idx) { return LinCoef(ind[idx], val[idx]); });
  }
}

auto xpress::objects::LinTermListImplementation::toString() const
    -> std::string {
  std::stringstream buffer;
  bool useOperator = false;
  if (getConstant() != 0.0) {
    buffer << InternalUtils::floatToString(getConstant());
    useOperator = true;
  }
  for (int i = 0; i < length; ++i) {
    Variable v = ind[i];
    double c = val[i];
    if (useOperator) {
      buffer << (c < 0.0 ? "-" : "+");
      c = std::abs(c);
    }
    useOperator = true;
    buffer << LinTerm::toString(v, c);
  }
  if (!useOperator) {

    return "0.0";
  }
  return buffer.str();
}

;

xpress::objects::LinTermMapImplementation::LinTermMapImplementation()
    : map(std::unordered_map<Variable, double>()) {}

xpress::objects::LinTermMapImplementation::LinTermMapImplementation(
    double constant)
    : LinTermMapImplementation() {
  this->map[xpress::objects::XpressProblem::NULL_VARIABLE] = constant;
}

auto xpress::objects::LinTermMapImplementation::deepCopy()
    -> xpress::objects::LinExpression {
  return deepCopy(1.0);
}

auto xpress::objects::LinTermMapImplementation::deepCopy(double factor)
    -> xpress::objects::LinExpression {
  if (factor == 0.0) {
    LinTermMap retval = LinTermMap();
    return retval;
  }
  LinTermMap ret = LinTermMap();
  for (auto &e : map) {
    Variable v = e.first;
    if (LinQuadUtils::isConstant(v))
      ret.setConstant(factor * e.second);
    else
      ret.setCoefficient(v, factor * e.second);
  }
  return ret;
}

auto xpress::objects::LinTermMapImplementation::reset()
    -> xpress::objects::LinTermMapImplementation & {
  map.clear();
  return *this;
}

auto xpress::objects::LinTermMapImplementation::addTerm(
    Variable const &variable,
    double const &coefficient) -> xpress::objects::LinTermMapImplementation & {
  if (xpress::objects::LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("cannot add constant terms");
  xpress::objects::InternalUtils::mergeCoefficient(map, variable, coefficient);
  return *this;
}

auto xpress::objects::LinTermMapImplementation::addTerm(
    double const &coefficient,
    Variable const &variable) -> xpress::objects::LinTermMapImplementation & {
  return addTerm(variable, coefficient);
}

auto xpress::objects::LinTermMapImplementation::addTerm(Variable const &x)
    -> xpress::objects::LinTermMapImplementation & {
  return addTerm(x, 1.0);
}

auto xpress::objects::LinTermMapImplementation::addTerms(
    LinExpression const &expr) -> xpress::objects::LinTermMapImplementation & {
  return addTerms(expr, 1.0);
}

auto xpress::objects::LinTermMapImplementation::addTerms(
    LinExpression const &expr,
    double factor) -> xpress::objects::LinTermMapImplementation & {
  InternalUtils::mergeCoefficient(map,
                                  xpress::objects::XpressProblem::NULL_VARIABLE,
                                  factor * expr.getConstant());
  std::shared_ptr<LinearView> view = expr.getLinearView();
  for (auto &e : *view) {
    if (!LinQuadUtils::isConstant(e.getVariable()))
      addTerm(e.getVariable(), factor * e.getCoefficient());
  }
  return *this;
}

auto xpress::objects::LinTermMapImplementation::setCoefficient(
    Variable variable,
    double coefficient) -> xpress::objects::LinTermMapImplementation & {
  if (xpress::objects::LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("cannot add constant terms");
  if (coefficient == 0.0) {

    map.erase(variable);
  } else {
    map[variable] = coefficient;
  }
  return *this;
}

auto xpress::objects::LinTermMapImplementation::getCoefficient(
    Variable variable) const -> double {
  if (xpress::objects::LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("cannot add constant terms");
  double value;
  value = (const_cast<LinTermMapImplementation *>(this)->map.find(variable) ==
           const_cast<LinTermMapImplementation *>(this)->map.end())
              ? ZERO
              : const_cast<LinTermMapImplementation *>(this)->map[variable];
  return value;
}

auto xpress::objects::LinTermMapImplementation::addConstant(double c)
    -> xpress::objects::LinTermMapImplementation & {
  InternalUtils::mergeCoefficient(
      map, xpress::objects::XpressProblem::NULL_VARIABLE, c);
  return *this;
}

auto xpress::objects::LinTermMapImplementation::setConstant(double c)
    -> xpress::objects::LinTermMapImplementation & {
  map[XpressProblem::NULL_VARIABLE] = c;
  return *this;
}

auto xpress::objects::LinTermMapImplementation::getConstant() const -> double {
  double value = 0.0;
  value = (const_cast<LinTermMapImplementation *>(this)->map.find(
               xpress::objects::XpressProblem::NULL_VARIABLE) ==
           const_cast<LinTermMapImplementation *>(this)->map.end())
              ? ZERO
              : const_cast<LinTermMapImplementation *>(this)
                    ->map[xpress::objects::XpressProblem::NULL_VARIABLE];
  return value;
}

auto xpress::objects::LinTermMapImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  XpressProblem *prob = (XpressProblem *)creator->prob;
  bool haveVariables = false;
  for (auto &e : map) {
    Variable v = e.first;
    if (XpressProblem::isNullVariable(v))
      creator->addRhs(-factor * e.second);
    else {
      haveVariables = true;
      creator->addNonzero(xpress::objects::LinQuadUtils::index4Var(prob, v),
                          factor * e.second);
    }
  }
  if (haveVariables)
    creator->checkForVariables();

  return true;
}

auto xpress::objects::LinTermMapImplementation::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  if (map.size() == 0) {

    extractor->startExpression();
    extractor->addToken(XPRS_TOK_CON, 0.0);
    extractor->endExpression();
    return;
  }
  extractor->startExpression();
  extractor->addToken(XPRS_TOK_RB, 0.0);
  bool comma = false;
  double constant = 0.0;
  for (auto &e : xpress::sortMap(map, true)) {
    Variable v = e.first;
    double coef = factor * e.second;
    if (XpressProblem::isNullVariable(v)) {
      constant = coef;
      continue;
    }
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);

    if (coef != 1.0)
      extractor->addToken(XPRS_TOK_CON, coef);
    if (coef != 0.0) {
      extractor->addToken(XPRS_TOK_COL, extractor->dereferenceVariable(v));
      if (coef != 1.0)
        extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
    }
    comma = true;
  }
  if (constant != 0.0) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    extractor->addToken(XPRS_TOK_CON, constant);
    comma = true;
  }
  if (!comma) {

    extractor->addToken(XPRS_TOK_CON, 0.0);
  }
  extractor->addToken(XPRS_TOK_IFUN, XPRS_IFUN_SUM);
  extractor->endExpression();
}

auto xpress::objects::LinTermMapImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  double result = 0.0;
  for (auto &e : map) {
    Variable v = e.first;
    if (XpressProblem::isNullVariable(v))
      result += e.second;
    else
      result += solution[v.getIndex()] * e.second;
  }
  return result;
}

auto xpress::objects::LinTermMapImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return std::make_shared<
      xpress::STLView<LinearView, std::unordered_map<Variable, double>,
                      ExpressionImplementation>>(
      const_cast<LinTermMapImplementation *>(this)->shared_from_this(), &map,
      [](std::pair<Variable, double> p) { return LinCoef(p.first, p.second); });
}

auto xpress::objects::LinTermMapImplementation::toString() const
    -> std::string {
  std::stringstream buffer;

  bool useOperator = false;
  if (getConstant() != 0.0) {
    buffer << InternalUtils::floatToString(getConstant());
    useOperator = true;
  }
  auto ordered = xpress::sortMap(map, false);
  for (auto const &e : ordered) {
    Variable const &v = e.first;
    double c = e.second;
    if (LinQuadUtils::isConstant(v))
      continue;
    if (useOperator) {
      buffer << (c < 0.0 ? "-" : "+");
      c = std::abs(c);
    }
    useOperator = true;
    buffer << LinTerm::toString(v, c);
  }
  if (!useOperator) {

    return "0.0";
  }
  return buffer.str();
}

;

auto xpress::objects::QuadTermListImplementation::setAutoCompress(bool compress)
    -> void {
  autoCompress = compress;
}

auto xpress::objects::QuadTermListImplementation::isAutoCompress() -> bool {
  return autoCompress;
}

;

;

;

xpress::objects::QuadTermListImplementation::QuadTermListImplementation()
    : constant(0.0), length(0), ind1({}), ind2({}), val({}) {}

xpress::objects::QuadTermListImplementation::QuadTermListImplementation(
    double c)
    : constant(c), length(0), ind1({}), ind2({}), val({}) {}

xpress::objects::QuadTermListImplementation::QuadTermListImplementation(
    double c, int capacity)
    : constant(c), length(0), ind1({}), ind2({}), val({}) {
  if (capacity > 0) {
    ind1 = std::vector<Variable>(capacity);
    ind2 = std::vector<Variable>(capacity);
    val = std::vector<double>(capacity);
  }
}

auto xpress::objects::QuadTermListImplementation::deepCopy()
    -> xpress::objects::QuadExpression {
  return deepCopy(1.0);
}

auto xpress::objects::QuadTermListImplementation::deepCopy(double factor)
    -> xpress::objects::QuadExpression {
  int len = length;
  QuadTermList ret = QuadTermList(factor * constant, len);
  for (int i = 0; i < len; ++i)
    ret.addTerm(ind1[i], ind2[i], factor * val[i]);
  return ret;
}

auto xpress::objects::QuadTermListImplementation::reserve(int l)
    -> xpress::objects::QuadTermListImplementation & {
  if (l <= 0)
    return *this;
  if (xpress::toInt(ind1.size()) < length + l)
    ind1 = InternalUtils::extendArray(ind1, l, length > 0);
  if (xpress::toInt(ind2.size()) < length + l)
    ind2 = InternalUtils::extendArray(ind2, l, length > 0);
  if (xpress::toInt(val.size()) < length + l)
    val = InternalUtils::extendArray(val, l, length > 0);
  return *this;
}

auto xpress::objects::QuadTermListImplementation::reset()
    -> xpress::objects::QuadTermListImplementation & {
  constant = 0;
  length = 0;
  return *this;
}

auto xpress::objects::QuadTermListImplementation::addTerm(
    Variable const &variable, double const &coefficient)
    -> xpress::objects::QuadTermListImplementation & {
  reserve(1);
  if (LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("cannot add constant terms");
  ind1[length] = LinQuadUtils::unifyConstantKey(variable);
  ind2[length] = XpressProblem::NULL_VARIABLE;
  val[length] = coefficient;
  ++length;
  return *this;
}

auto xpress::objects::QuadTermListImplementation::addTerm(
    double const &coefficient,
    Variable const &variable) -> xpress::objects::QuadTermListImplementation & {
  return addTerm(variable, coefficient);
}

auto xpress::objects::QuadTermListImplementation::addTerm(Variable const &x)
    -> xpress::objects::QuadTermListImplementation & {
  return addTerm(x, 1.0);
}

auto xpress::objects::QuadTermListImplementation::addTerm(
    Variable const &variable1, Variable const &variable2,
    double const &coefficient)
    -> xpress::objects::QuadTermListImplementation & {
  reserve(1);
  Variable v1 = LinQuadUtils::unifyConstantKey(variable1);
  Variable v2 = LinQuadUtils::unifyConstantKey(variable2);
  if (!LinQuadUtils::isVariablePairOrdered(v1, v2)) {
    Variable tmp = v1;
    v1 = v2;
    v2 = tmp;
  }
  if (LinQuadUtils::isConstant(v1, v2))
    throw std::invalid_argument("cannot add constant terms");
  ind1[length] = v1;
  ind2[length] = v2;
  val[length] = coefficient;
  ++length;
  return *this;
}

auto xpress::objects::QuadTermListImplementation::addTerm(
    double const &coefficient, Variable const &variable1,
    Variable const &variable2)
    -> xpress::objects::QuadTermListImplementation & {
  return addTerm(variable1, variable2, coefficient);
}

auto xpress::objects::QuadTermListImplementation::addTerm(Variable const &x1,
                                                          Variable const &x2)
    -> xpress::objects::QuadTermListImplementation & {
  return addTerm(x1, x2, 1.0);
}

auto xpress::objects::QuadTermListImplementation::addTerms(
    LinExpression const &expr)
    -> xpress::objects::QuadTermListImplementation & {
  return addTerms(expr, 1.0);
}

auto xpress::objects::QuadTermListImplementation::addTerms(
    QuadExpression const &expr)
    -> xpress::objects::QuadTermListImplementation & {
  return addTerms(expr, 1.0);
}

auto xpress::objects::QuadTermListImplementation::addTerms(
    LinExpression const &expr,
    double const &factor) -> xpress::objects::QuadTermListImplementation & {
  int start = length;
  constant += factor * expr.getConstant();
  std::shared_ptr<LinearView> view = expr.getLinearView();
  for (auto &e : *view) {
    if (!LinQuadUtils::isConstant(e.getVariable()))
      addTerm(e.getVariable(), factor * e.getCoefficient());
  }

  if (!expr.isInstance<LinTermListImplementation>()) {
    InternalUtils::sortParallelArrays(ind1, ind2, val, start, length - start);
  }
  return *this;
}

auto xpress::objects::QuadTermListImplementation::addTerms(
    QuadExpression const &expr,
    double const &factor) -> xpress::objects::QuadTermListImplementation & {
  int start = length;
  constant += factor * expr.getConstant();
  std::shared_ptr<QuadView> view = expr.getQuadView();
  for (auto &e : *view) {
    if (!e.getVariables().isConstant())
      addTerm(e.getVar1(), e.getVar2(), factor * e.getCoefficient());
  }

  if (!expr.isInstance<QuadTermListImplementation>()) {
    InternalUtils::sortParallelArrays(ind1, ind2, val, start, length - start);
  }
  return *this;
}

auto xpress::objects::QuadTermListImplementation::addConstant(double c)
    -> xpress::objects::QuadTermListImplementation & {
  constant += c;
  return *this;
}

auto xpress::objects::QuadTermListImplementation::setConstant(double c)
    -> xpress::objects::QuadTermListImplementation & {
  constant = c;
  return *this;
}

auto xpress::objects::QuadTermListImplementation::getConstant() const
    -> double {
  return constant;
}

auto xpress::objects::QuadTermListImplementation::normalizeTerms()
    -> xpress::objects::QuadTermListImplementation & {
  compress();
  return *this;
}

auto xpress::objects::QuadTermListImplementation::compress() const -> void {
  InternalUtils::sortParallelArrays(ind1, ind2, val, 0, length);
  int outIdx = 0;
  Variable last1 = XpressProblem::NULL_VARIABLE;
  Variable last2 = XpressProblem::NULL_VARIABLE;
  for (int inIdx = 0; inIdx < length; ++inIdx) {
    Variable v1 = ind1[inIdx];
    Variable v2 = ind2[inIdx];
    if (!v1.equals(last1) || !v2.equals(last2)) {
      last1 = v1;
      last2 = v2;

      if (outIdx > 0 && val[outIdx - 1] == 0.0)
        --outIdx;
      ind1[outIdx] = v1;
      ind2[outIdx] = v2;
      val[outIdx] = val[inIdx];
      ++outIdx;
    } else {
      val[outIdx - 1] += val[inIdx];
    }
  }

  if (outIdx > 0 && val[outIdx - 1] == 0.0)
    --outIdx;
  length = outIdx;
}

auto xpress::objects::QuadTermListImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  if (isAutoCompress())
    compress();
  if (length > 0)
    creator->checkForVariables();
  XpressProblem *prob = (XpressProblem *)creator->prob;
  creator->addRhs(-factor * constant);
  for (int i = 0; i < length; ++i) {
    assert(!LinQuadUtils::isConstant(ind1[i], ind2[i]));
    if (LinQuadUtils::isLinear(ind1[i], ind2[i])) {
      if (val[i] != 0.0)
        creator->addNonzero(LinQuadUtils::index4Var(prob, ind1[i]),
                            factor * val[i]);
    } else {
      if (val[i] != 0.0) {
        int x1 = LinQuadUtils::index4Var(prob, ind1[i]);
        int x2 = LinQuadUtils::index4Var(prob, ind2[i]);
        creator->addNonzero(x1, x2,

                            ((x1 != x2) ? 0.5 : 1.0) * factor * val[i]);
      }
    }
  }

  return false;
}

auto xpress::objects::QuadTermListImplementation::extract(
    double factor, PostfixExtractor *extractor) const -> void {
  if (length == 0 && constant == 0.0) {

    extractor->startExpression();
    extractor->addToken(XPRS_TOK_CON, 0.0);
    extractor->endExpression();
    return;
  }
  extractor->startExpression();
  if (isAutoCompress())
    compress();
  extractor->addToken(XPRS_TOK_RB, 0.0);
  bool comma = false;
  for (int i = length - 1; i >= 0; --i) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    double coef = factor * val[i];

    if (coef != 1.0)
      extractor->addToken(XPRS_TOK_CON, coef);
    if (coef != 0.0) {
      if (LinQuadUtils::isLinear(ind1[i], ind2[i])) {
        int v1 = extractor->dereferenceVariable(ind1[i]);
        extractor->addToken(XPRS_TOK_COL, v1);
        if (coef != 1.0)
          extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
      } else {
        int v1 = extractor->dereferenceVariable(ind1[i]);
        int v2 = extractor->dereferenceVariable(ind2[i]);
        extractor->addToken(XPRS_TOK_COL, v1);
        extractor->addToken(XPRS_TOK_COL, v2);
        extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
        if (coef != 1.0)
          extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
      }
    }
    comma = true;
  }
  if (constant != 0.0) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    extractor->addToken(XPRS_TOK_CON, factor * constant);
    comma = true;
  }
  if (!comma) {

    extractor->addToken(XPRS_TOK_CON, 0.0);
  }
  extractor->addToken(XPRS_TOK_IFUN, XPRS_IFUN_SUM);
  extractor->endExpression();
}

auto xpress::objects::QuadTermListImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  double result = constant;
  for (int i = 0; i < length; ++i) {
    assert(!LinQuadUtils::isConstant(ind1[i], ind2[i]));
    if (LinQuadUtils::isLinear(ind1[i], ind2[i]))
      result += val[i] * solution[ind1[i].getIndex()];
    else
      result +=
          val[i] * solution[ind1[i].getIndex()] * solution[ind2[i].getIndex()];
  }
  return result;
}

auto xpress::objects::QuadTermListImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  if (constant == 0.0) {
    return std::make_shared<
        xpress::IndexedView<QuadView, ExpressionImplementation>>(
        const_cast<QuadTermListImplementation *>(this)->shared_from_this(),
        length, [&](int idx) {
          return QuadCoef(QPair(ind1[idx], ind2[idx]), val[idx]);
        });
  } else {
    QuadCoef initial = QuadCoef(
        QPair(XpressProblem::NULL_VARIABLE, XpressProblem::NULL_VARIABLE),
        constant);
    return std::make_shared<
        xpress::IndexedView<QuadView, ExpressionImplementation>>(
        const_cast<QuadTermListImplementation *>(this)->shared_from_this(),
        length, initial, [&](int idx) {
          return QuadCoef(QPair(ind1[idx], ind2[idx]), val[idx]);
        });
  }
}

auto xpress::objects::QuadTermListImplementation::toString() const
    -> std::string {
  std::stringstream buffer;
  bool useOperator = false;
  if (getConstant() != 0.0) {
    buffer << InternalUtils::floatToString(getConstant());
    useOperator = true;
  }
  for (int i = 0; i < length; ++i) {
    Variable v1 = ind1[i];
    Variable v2 = ind2[i];
    double c = val[i];
    if (useOperator) {
      buffer << (c < 0.0 ? "-" : "+");
      c = std::abs(c);
    }
    useOperator = true;
    buffer << QuadTerm::toString(v1, v2, c);
  }
  if (!useOperator) {

    return "0.0";
  }
  return buffer.str();
}

;

xpress::objects::QuadTermMapImplementation::QuadTermMapImplementation()
    : map(std::unordered_map<QPair, double>()) {}

xpress::objects::QuadTermMapImplementation::QuadTermMapImplementation(
    double constant)
    : QuadTermMapImplementation() {
  if (constant != 0.0)
    this->map[LinQuadUtils::getQConstantKey()] = constant;
}

auto xpress::objects::QuadTermMapImplementation::deepCopy()
    -> xpress::objects::QuadExpression {
  return deepCopy(1.0);
}

auto xpress::objects::QuadTermMapImplementation::deepCopy(double factor)
    -> xpress::objects::QuadExpression {
  if (factor == 0.0)
    return QuadTermMap();
  QuadTermMap ret = QuadTermMap();
  for (auto &e : map) {
    QPair q = e.first;
    Variable v1 = q.getVar1();
    Variable v2 = q.getVar2();
    if (LinQuadUtils::isConstant(v1, v2))
      ret.setConstant(factor * e.second);
    else if (LinQuadUtils::isLinear(v1, v2))
      ret.setCoefficient(v1, factor * e.second);
    else
      ret.setCoefficient(v1, v2, factor * e.second);
  }
  return ret;
}

auto xpress::objects::QuadTermMapImplementation::reset()
    -> xpress::objects::QuadTermMapImplementation & {
  map.clear();
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::addTerm(
    Variable const &variable,
    double const &coefficient) -> xpress::objects::QuadTermMapImplementation & {
  if (LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("cannot add constant terms");
  InternalUtils::mergeCoefficient(
      map, QPair(variable, XpressProblem::NULL_VARIABLE), coefficient);
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::addTerm(
    double const &coefficient,
    Variable const &variable) -> xpress::objects::QuadTermMapImplementation & {
  return addTerm(variable, coefficient);
}

auto xpress::objects::QuadTermMapImplementation::addTerm(Variable const &x)
    -> xpress::objects::QuadTermMapImplementation & {
  return addTerm(x, 1.0);
}

auto xpress::objects::QuadTermMapImplementation::addTerm(
    Variable const &variable1, Variable const &variable2,
    double const &coefficient) -> xpress::objects::QuadTermMapImplementation & {
  bool ordered = LinQuadUtils::isVariablePairOrdered(variable1, variable2);
  Variable v1 = ordered ? variable1 : variable2;
  Variable v2 = ordered ? variable2 : variable1;
  if (LinQuadUtils::isConstant(v1, v2))
    throw std::invalid_argument("cannot add constant terms");
  InternalUtils::mergeCoefficient(map, QPair(v1, v2), coefficient);
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::addTerm(
    double const &coefficient, Variable const &variable1,
    Variable const &variable2) -> xpress::objects::QuadTermMapImplementation & {
  return addTerm(variable1, variable2, coefficient);
}

auto xpress::objects::QuadTermMapImplementation::addTerm(Variable const &x1,
                                                         Variable const &x2)
    -> xpress::objects::QuadTermMapImplementation & {
  return addTerm(x1, x2, 1.0);
}

auto xpress::objects::QuadTermMapImplementation::addTerms(
    LinExpression const &expr) -> xpress::objects::QuadTermMapImplementation & {
  return addTerms(expr, 1.0);
}

auto xpress::objects::QuadTermMapImplementation::addTerms(
    QuadExpression const &expr)
    -> xpress::objects::QuadTermMapImplementation & {
  return addTerms(expr, 1.0);
}

auto xpress::objects::QuadTermMapImplementation::addTerms(
    LinExpression const &expr,
    double const &factor) -> xpress::objects::QuadTermMapImplementation & {
  InternalUtils::mergeCoefficient(map, LinQuadUtils::getQConstantKey(),
                                  factor * expr.getConstant());
  std::shared_ptr<LinearView> view = expr.getLinearView();
  for (auto &e : *view) {
    if (!LinQuadUtils::isConstant(e.getVariable()))
      addTerm(e.getVariable(), factor * e.getCoefficient());
  }
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::addTerms(
    QuadExpression const &expr,
    double const &factor) -> xpress::objects::QuadTermMapImplementation & {
  InternalUtils::mergeCoefficient(map, LinQuadUtils::getQConstantKey(),
                                  factor * expr.getConstant());
  std::shared_ptr<QuadView> view = expr.getQuadView();
  for (auto &e : *view) {
    if (!e.getVariables().isConstant())
      addTerm(e.getVar1(), e.getVar2(), factor * e.getCoefficient());
  }
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::setCoefficient(
    Variable variable,
    double coefficient) -> xpress::objects::QuadTermMapImplementation & {
  if (LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("cannot add constant terms");
  QPair key = QPair(variable, XpressProblem::NULL_VARIABLE);
  if (coefficient == 0.0) {

    map.erase(key);
  } else {
    map[key] = coefficient;
  }
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::setCoefficient(
    Variable variable1, Variable variable2,
    double coefficient) -> xpress::objects::QuadTermMapImplementation & {
  bool ordered = LinQuadUtils::isVariablePairOrdered(variable1, variable2);
  Variable v1 = ordered ? variable1 : variable2;
  Variable v2 = ordered ? variable2 : variable1;
  if (LinQuadUtils::isConstant(v1, v2))
    throw std::invalid_argument("cannot add constant terms");
  QPair key = QPair(v1, v2);
  if (coefficient == 0.0) {

    map.erase(key);
  } else {
    map[key] = coefficient;
  }
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::getCoefficient(
    Variable variable) const -> double {
  if (LinQuadUtils::isConstant(variable))
    throw std::invalid_argument("cannot add constant terms");
  double value;
  value = (const_cast<QuadTermMapImplementation *>(this)->map.find(
               QPair(variable, XpressProblem::NULL_VARIABLE)) ==
           const_cast<QuadTermMapImplementation *>(this)->map.end())
              ? ZERO
              : const_cast<QuadTermMapImplementation *>(this)
                    ->map[QPair(variable, XpressProblem::NULL_VARIABLE)];
  return value;
}

auto xpress::objects::QuadTermMapImplementation::getCoefficient(
    Variable variable1, Variable variable2) const -> double {
  bool ordered = LinQuadUtils::isVariablePairOrdered(variable1, variable2);
  Variable v1 = ordered ? variable1 : variable2;
  Variable v2 = ordered ? variable2 : variable1;
  if (LinQuadUtils::isConstant(v1, v2))
    throw std::invalid_argument("cannot add constant terms");
  double value = 0.0;
  value =
      (const_cast<QuadTermMapImplementation *>(this)->map.find(QPair(v1, v2)) ==
       const_cast<QuadTermMapImplementation *>(this)->map.end())
          ? ZERO
          : const_cast<QuadTermMapImplementation *>(this)->map[QPair(v1, v2)];
  return value;
}

auto xpress::objects::QuadTermMapImplementation::addConstant(double c)
    -> xpress::objects::QuadTermMapImplementation & {
  InternalUtils::mergeCoefficient(map, LinQuadUtils::getQConstantKey(), c);
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::setConstant(double c)
    -> xpress::objects::QuadTermMapImplementation & {
  map[LinQuadUtils::getQConstantKey()] = c;
  return *this;
}

auto xpress::objects::QuadTermMapImplementation::getConstant() const -> double {
  double value = 0.0;
  value = (const_cast<QuadTermMapImplementation *>(this)->map.find(
               LinQuadUtils::getQConstantKey()) ==
           const_cast<QuadTermMapImplementation *>(this)->map.end())
              ? ZERO
              : const_cast<QuadTermMapImplementation *>(this)
                    ->map[LinQuadUtils::getQConstantKey()];
  return value;
}

auto xpress::objects::QuadTermMapImplementation::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  XpressProblem *prob = (XpressProblem *)creator->prob;
  bool haveVariables = false;
  for (auto &e : map) {
    QPair k = e.first;
    if (k.isConstant())
      creator->addRhs(-factor * e.second);
    else if (k.isLinear()) {
      haveVariables = true;
      creator->addNonzero(LinQuadUtils::index4Var(prob, k.getVar1()),
                          factor * e.second);
    } else {
      haveVariables = true;
      int x1 = LinQuadUtils::index4Var(prob, k.getVar1());
      int x2 = LinQuadUtils::index4Var(prob, k.getVar2());
      creator->addNonzero(x1, x2,

                          ((x1 != x2) ? 0.5 : 1.0) * factor * e.second);
    }
  }
  if (haveVariables)
    creator->checkForVariables();

  return true;
}

auto xpress::objects::QuadTermMapImplementation::extract(
    double factor, PostfixExtractor *extractor) const -> void {
  if (map.size() == 0) {

    extractor->startExpression();
    extractor->addToken(XPRS_TOK_CON, 0.0);
    extractor->endExpression();
    return;
  }
  extractor->startExpression();
  extractor->addToken(XPRS_TOK_RB, 0.0);
  bool comma = false;
  double constant = 0.0;
  for (auto &e : xpress::sortMap(map, true)) {
    QPair k = e.first;
    double coef = factor * e.second;
    if (k.isConstant()) {
      constant = coef;
      continue;
    }
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);

    if (coef != 1.0)
      extractor->addToken(XPRS_TOK_CON, coef);
    if (coef != 0.0) {
      if (k.isLinear()) {
        int v1 = extractor->dereferenceVariable(k.getVar1());
        extractor->addToken(XPRS_TOK_COL, v1);
        if (coef != 1.0)
          extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
      } else {
        int v1 = extractor->dereferenceVariable(k.getVar1());
        int v2 = extractor->dereferenceVariable(k.getVar2());
        extractor->addToken(XPRS_TOK_COL, v1);
        extractor->addToken(XPRS_TOK_COL, v2);
        extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
        if (coef != 1.0)
          extractor->addToken(XPRS_TOK_OP, XPRS_OP_MULTIPLY);
      }
    }
    comma = true;
  }
  if (constant != 0.0) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    extractor->addToken(XPRS_TOK_CON, constant);
    comma = true;
  }
  if (!comma) {

    extractor->addToken(XPRS_TOK_CON, 0.0);
  }
  extractor->addToken(XPRS_TOK_IFUN, XPRS_IFUN_SUM);
  extractor->endExpression();
}

auto xpress::objects::QuadTermMapImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  double result = 0.0;
  for (auto &e : map) {
    QPair k = e.first;
    if (k.isConstant())
      result += e.second;
    else if (k.isLinear())
      result += e.second * solution[k.getVar1().getIndex()];
    else
      result += e.second * solution[k.getVar1().getIndex()] *
                solution[k.getVar2().getIndex()];
  }
  return result;
}

auto xpress::objects::QuadTermMapImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return std::make_shared<xpress::STLView<
      QuadView, std::unordered_map<QPair, double>, ExpressionImplementation>>(
      const_cast<QuadTermMapImplementation *>(this)->shared_from_this(), &map,
      [](std::pair<QPair, double> p) { return QuadCoef(p.first, p.second); });
}

auto xpress::objects::QuadTermMapImplementation::toString() const
    -> std::string {
  std::stringstream buffer;

  bool useOperator = false;
  if (getConstant() != 0.0) {
    buffer << InternalUtils::floatToString(getConstant());
    useOperator = true;
  }
  auto ordered = xpress::sortMap(map, false);
  for (auto const &e : ordered) {
    Variable const &v1 = e.first.getVar1();
    Variable const &v2 = e.first.getVar2();
    double c = e.second;
    if (LinQuadUtils::isConstant(v1, v2))
      continue;
    if (useOperator) {
      buffer << (c < 0.0 ? "-" : "+");
      c = std::abs(c);
    }
    useOperator = true;
    buffer << QuadTerm::toString(v1, v2, c);
  }
  if (!useOperator) {

    return "0.0";
  }
  return buffer.str();
}

;

xpress::objects::BinaryExpressionImplementation::BinaryExpressionImplementation(
    xpress::objects::Expression left,
    xpress::objects::BinaryExpression::Operator op,
    xpress::objects::Expression right)
    : left(left), op(op), right(right), maxDegree(-1) {}

auto xpress::objects::BinaryExpressionImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  double result = 0.0;
  switch (op) {
  case xpress::objects::BinaryExpression::Operator::Plus:
    result = left.evaluate(solution) + right.evaluate(solution);
    break;
  case xpress::objects::BinaryExpression::Operator::Minus:
    result = left.evaluate(solution) - right.evaluate(solution);
    break;
  case xpress::objects::BinaryExpression::Operator::Multiply:
    result = left.evaluate(solution) * right.evaluate(solution);
    break;
  case xpress::objects::BinaryExpression::Operator::Divide:
    result = left.evaluate(solution) / right.evaluate(solution);
    break;
  case xpress::objects::BinaryExpression::Operator::Exponent:
    result = std::pow(left.evaluate(solution), right.evaluate(solution));
    break;
  }
  return result;
}

auto xpress::objects::BinaryExpressionImplementation::extract(
    PostfixExtractor *extractor) const -> void {

  extractor->startExpression();
  left.extract(1.0, extractor);
  right.extract(1.0, extractor);
  extractor->addToken(XPRS_TOK_OP, (int)op);
  extractor->endExpression();
}

auto xpress::objects::BinaryExpressionImplementation::toString() const
    -> std::string {
  std::stringstream buffer;
  buffer << "(";
  buffer << left.toString();
  buffer << ")";
  switch (op) {
  case xpress::objects::BinaryExpression::Operator::Exponent:
    buffer << " ^ ";
    break;
  case xpress::objects::BinaryExpression::Operator::Multiply:
    buffer << " * ";
    break;
  case xpress::objects::BinaryExpression::Operator::Divide:
    buffer << " / ";
    break;
  case xpress::objects::BinaryExpression::Operator::Plus:
    buffer << " + ";
    break;
  case xpress::objects::BinaryExpression::Operator::Minus:
    buffer << " - ";
    break;
  }
  buffer << "(";
  buffer << right.toString();
  buffer << ")";
  return buffer.str();
}

auto xpress::objects::BinaryExpressionImplementation::getMaxDegree() const
    -> int {

  int result = maxDegree;
  if (result < 0) {
    int leftMax = left.getMaxDegree();
    int rightMax = right.getMaxDegree();
    result = std::numeric_limits<int>::max();
    switch (op) {
    case xpress::objects::BinaryExpression::Operator::Exponent:

      break;
    case xpress::objects::BinaryExpression::Operator::Multiply:
      if (leftMax < std::numeric_limits<int>::max() - rightMax)
        result = leftMax + rightMax;
      break;
    case xpress::objects::BinaryExpression::Operator::Divide:

      result = (rightMax == 0) ? leftMax : std::numeric_limits<int>::max();
      break;
    case xpress::objects::BinaryExpression::Operator::Plus:
      result = std::max(leftMax, rightMax);
      break;
    case xpress::objects::BinaryExpression::Operator::Minus:
      result = std::max(leftMax, rightMax);
      break;
    }
    maxDegree = result;
  }
  return result;
}

auto xpress::objects::BinaryExpressionImplementation::getConstantView() const
    -> double {
  if (getMaxDegree() != 0)
    throw std::invalid_argument("not a constant expression");
  switch (op) {
  case xpress::objects::BinaryExpression::Operator::Exponent:
    assert(false);
    throw std::invalid_argument("not a constant expression");
  case xpress::objects::BinaryExpression::Operator::Multiply:
    return left.getConstantView() * right.getConstantView();
  case xpress::objects::BinaryExpression::Operator::Divide:
    return left.getConstantView() / right.getConstantView();
  case xpress::objects::BinaryExpression::Operator::Plus:
    return left.getConstantView() + right.getConstantView();
  case xpress::objects::BinaryExpression::Operator::Minus:
    return left.getConstantView() - right.getConstantView();
  }
  return std::numeric_limits<double>::quiet_NaN();
}

auto xpress::objects::BinaryExpressionImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  if (getMaxDegree() > 1)
    throw std::invalid_argument("not a linear expression");

  int leftMax;
  int rightMax;
  switch (op) {
  case xpress::objects::BinaryExpression::Operator::Exponent:
    assert(false);
    break;
  case xpress::objects::BinaryExpression::Operator::Multiply:
    leftMax = left.getMaxDegree();
    rightMax = right.getMaxDegree();
    if (leftMax + rightMax == 0)
      return InternalUtils::getSingletonLinView(XpressProblem::NULL_VARIABLE,
                                                left.getConstantView() *
                                                    right.getConstantView());
    else if (leftMax == 0)
      return UnaryExpression::scaledLinView(right.getLinearView(),
                                            left.getConstantView());
    else if (rightMax == 0)
      return UnaryExpression::scaledLinView(left.getLinearView(),
                                            right.getConstantView());
    assert(false);
    break;
  case xpress::objects::BinaryExpression::Operator::Divide:
    assert(right.getMaxDegree() == 0);
    return UnaryExpression::scaledLinView(left.getLinearView(),
                                          1.0 / right.getConstantView());
  case xpress::objects::BinaryExpression::Operator::Plus:
    return std::make_shared<xpress::SumView<LinearView, Expression, void>>(
        [](Expression const &e) -> std::shared_ptr<LinearView> {
          return e.getLinearView();
        },
        left, right);
  case xpress::objects::BinaryExpression::Operator::Minus:
    return std::make_shared<
        xpress::SumView<LinearView, std::shared_ptr<LinearView>, void>>(
        [](std::shared_ptr<LinearView> const &v)
            -> std::shared_ptr<LinearView> { return v; },
        left.getLinearView(),
        UnaryExpression::scaledLinView(right.getLinearView(), -1.0));
  }

  throw std::invalid_argument("not a linear expression");
}

auto xpress::objects::BinaryExpressionImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  if (getMaxDegree() > 2)
    throw std::invalid_argument("not a quadratic expression");

  int leftMax;
  int rightMax;
  switch (op) {
  case xpress::objects::BinaryExpression::Operator::Exponent:
    assert(false);
    break;
  case xpress::objects::BinaryExpression::Operator::Multiply:
    leftMax = left.getMaxDegree();
    rightMax = right.getMaxDegree();
    if (leftMax + rightMax == 0)
      return InternalUtils::getSingletonQuadView(
          XpressProblem::NULL_VARIABLE, XpressProblem::NULL_VARIABLE,
          left.getConstantView() * right.getConstantView());
    else if (leftMax == 0)
      return UnaryExpression::scaledQuadView(right.getQuadView(),
                                             left.getConstantView());
    else if (rightMax == 0)
      return UnaryExpression::scaledQuadView(left.getQuadView(),
                                             right.getConstantView());
    else if (leftMax == 1 && rightMax == 1) {
      return std::make_shared<xpress::objects::LinExprProductView>(
          left.getLinearView(), right.getLinearView());
    }
    assert(false);
    break;
  case xpress::objects::BinaryExpression::Operator::Divide:
    assert(right.getMaxDegree() == 0);
    return UnaryExpression::scaledQuadView(left.getQuadView(),
                                           1.0 / right.getConstantView());
  case xpress::objects::BinaryExpression::Operator::Plus:
    return std::make_shared<xpress::SumView<QuadView, Expression, void>>(
        [](Expression const &e) -> std::shared_ptr<QuadView> {
          return e.getQuadView();
        },
        left, right);
  case xpress::objects::BinaryExpression::Operator::Minus:
    return std::make_shared<
        xpress::SumView<QuadView, std::shared_ptr<QuadView>, void>>(
        [](std::shared_ptr<QuadView> const &v) -> std::shared_ptr<QuadView> {
          return v;
        },
        left.getQuadView(),
        UnaryExpression::scaledQuadView(right.getQuadView(), -1.0));
  }

  throw std::invalid_argument("not a quadratic expression");
}

;

xpress::objects::InternalFunctionExpressionImplementation::
    InternalFunctionExpressionImplementation(
        xpress::objects::InternalFunctionExpression::FunctionType function,
        std::vector<xpress::objects::Expression> arguments)
    : function(function), arguments(arguments) {
  if ((unary & (((XPRSint64)1) << (int)function)) != 0 &&
      xpress::toInt(arguments.size()) != 1)
    throw std::invalid_argument("unary function requires exactly one argument");
}

xpress::objects::InternalFunctionExpressionImplementation::
    InternalFunctionExpressionImplementation(
        xpress::objects::InternalFunctionExpression::FunctionType function,
        xpress::objects::Expression argument)
    : InternalFunctionExpressionImplementation(
          function, std::vector<xpress::objects::Expression>({argument})) {}

xpress::objects::InternalFunctionExpressionImplementation::
    InternalFunctionExpressionImplementation(
        xpress::objects::InternalFunctionExpression::FunctionType function,
        xpress::objects::Expression firstArgument,
        xpress::objects::Expression secondArgument)
    : InternalFunctionExpressionImplementation(
          function, std::vector<xpress::objects::Expression>(
                        {firstArgument, secondArgument})) {}

auto xpress::objects::InternalFunctionExpressionImplementation::
    collectArguments(Expression firstArgument, Expression secondArgument,
                     xpress::SizedArray<Expression const> const &more)
        -> std::vector<xpress::objects::Expression> {
  std::vector<Expression> args(2 + xpress::toInt(more.size()));
  args[0] = firstArgument;
  args[1] = secondArgument;
  xpress::arrayCopy(more, 0, args, 2, xpress::toInt(more.size()));
  return args;
}

template <typename... MOREARGUMENTS, typename RestrictPack>
xpress::objects::InternalFunctionExpressionImplementation::
    InternalFunctionExpressionImplementation(
        xpress::objects::InternalFunctionExpression::FunctionType function,
        xpress::objects::Expression firstArgument,
        xpress::objects::Expression secondArgument,
        MOREARGUMENTS... moreArguments)
    : InternalFunctionExpressionImplementation(
          function,
          collectArguments(firstArgument, secondArgument,
                           xpress::packToArray<xpress::objects::Expression>(
                               moreArguments...))) {}

xpress::objects::InternalFunctionExpressionImplementation::
    InternalFunctionExpressionImplementation(
        xpress::objects::InternalFunctionExpression::FunctionType function,
        xpress::SizedArray<Expression const> const &args)
    : function(function), arguments(args.begin(), args.end()) {}

auto xpress::objects::InternalFunctionExpressionImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  double result = 0.0;
  std::vector<double> args(xpress::toInt(arguments.size()));
  for (int i = 0; i < xpress::toInt(arguments.size()); ++i)
    args[i] = arguments[i].evaluate(solution);
  switch (function) {
  case xpress::objects::InternalFunctionExpression::FunctionType::Log10:
    result = std::log10(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Ln:
    result = std::log(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Exp:
    result = std::exp(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Abs:
    result = std::abs(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Sqrt:
    result = std::sqrt(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Sin:
    result = std::sin(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Cos:
    result = std::cos(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Tan:
    result = std::tan(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcSin:
    result = std::asin(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcCos:
    result = std::acos(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcTan:
    result = std::atan(args[0]);
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Min: {
    result = std::numeric_limits<double>::infinity();
    for (auto &d : args)
      result = std::min(result, d);
  } break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Max: {
    result = -std::numeric_limits<double>::infinity();
    for (auto &d : args)
      result = std::max(result, d);
  } break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Sum: {
    result = 0.0;
    for (auto &d : args)
      result += d;
  } break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Prod: {
    result = 1.0;
    for (auto &d : args)
      result *= d;
  } break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Sign:
    if (args[0] > 0)
      result = 1.0;
    else if (args[0] < 0)
      result = -1.0;
    else
      result = 0.0;
    break;
  }
  return result;
}

auto xpress::objects::InternalFunctionExpressionImplementation::extract(
    PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  extractor->addToken(XPRS_TOK_RB, 0.0);
  bool comma = false;
  for (int i = xpress::toInt(arguments.size()) - 1; i >= 0; --i) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    arguments[i].extract(1.0, extractor);
    comma = true;
  }
  extractor->addToken(XPRS_TOK_IFUN, (int)function);
  extractor->endExpression();
}

auto xpress::objects::InternalFunctionExpressionImplementation::toString() const
    -> std::string {
  std::stringstream buffer;
  switch (function) {
  case xpress::objects::InternalFunctionExpression::FunctionType::Log10:
    buffer << "log10";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Ln:
    buffer << "ln";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Exp:
    buffer << "exp";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Abs:
    buffer << "abs";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Sqrt:
    buffer << "sqrt";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Sin:
    buffer << "sin";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Cos:
    buffer << "cos";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Tan:
    buffer << "tan";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcSin:
    buffer << "arcsin";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcCos:
    buffer << "arccos";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcTan:
    buffer << "arctan";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Min:
    buffer << "min";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Max:
    buffer << "max";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Sum:
    buffer << "sum";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Prod:
    buffer << "prod";
    break;
  case xpress::objects::InternalFunctionExpression::FunctionType::Sign:
    buffer << "sign";
    break;
  default:
    buffer << "InternalFunction";
    buffer << (int)function;
  }
  std::string comma = "";
  buffer << "(";
  for (int i = 0; i < xpress::toInt(arguments.size()); ++i) {
    buffer << comma;
    comma = ", ";
    buffer << arguments[i].toString();
  }
  buffer << ")";
  return buffer.str();
}

auto xpress::objects::InternalFunctionExpressionImplementation::getMaxDegree()
    const -> int {

  return std::numeric_limits<int>::max();
}

auto xpress::objects::PWLExpressionImplementation::breakToX(
    xpress::SizedArray<PwlBreakpoint const> const &points)
    -> std::vector<double> {
  int len = xpress::toInt(points.size());
  std::vector<double> x(len);
  for (int i = 0; i < len; ++i)
    x[i] = points[i].getX();
  return x;
}

auto xpress::objects::PWLExpressionImplementation::breakToY(
    xpress::SizedArray<PwlBreakpoint const> const &points)
    -> std::vector<double> {
  int len = xpress::toInt(points.size());
  std::vector<double> y(len);
  for (int i = 0; i < len; ++i)
    y[i] = points[i].getY();
  return y;
}

xpress::objects::PWLExpressionImplementation::PWLExpressionImplementation(
    Variable argument, xpress::SizedArray<double const> const &breakX,
    xpress::SizedArray<double const> const &breakY)
    : argumentColumn(-1), argumentVariable(argument), breakX(breakX.toVector()),
      breakY(breakY.toVector()) {
  if (xpress::toInt(breakX.size()) != xpress::toInt(breakY.size()))
    throw std::invalid_argument("arrays have different lengths");
  if (xpress::toInt(breakX.size()) < 2)
    throw std::invalid_argument("at least two breakpoints are required");

  xpress::objects::InternalUtils::sortParallelArraysLex(
      this->breakX, this->breakY, 0, xpress::toInt(this->breakX.size()));
}

template <typename... POINTS, typename RestrictPack>
xpress::objects::PWLExpressionImplementation::PWLExpressionImplementation(
    Variable argument, POINTS... points)
    : PWLExpressionImplementation(
          argument, breakToX(xpress::packToArray<PwlBreakpoint>(points...)),
          breakToY(xpress::packToArray<PwlBreakpoint>(points...))) {}

xpress::objects::PWLExpressionImplementation::PWLExpressionImplementation(
    Variable argument, xpress::SizedArray<PwlBreakpoint const> const &points)
    : PWLExpressionImplementation(argument, breakToX(points),
                                  breakToY(points)) {}

xpress::objects::PWLExpressionImplementation::PWLExpressionImplementation(
    int argument, xpress::SizedArray<double const> const &breakX,
    xpress::SizedArray<double const> const &breakY)
    : argumentColumn(argument), argumentVariable(nullptr),
      breakX(breakX.toVector()), breakY(breakY.toVector()) {
  if (xpress::toInt(breakX.size()) != xpress::toInt(breakY.size()))
    throw std::invalid_argument("arrays have different lengths");
  if (xpress::toInt(breakX.size()) < 2)
    throw std::invalid_argument("at least two breakpoints are required");

  xpress::objects::InternalUtils::sortParallelArraysLex(
      this->breakX, this->breakY, 0, xpress::toInt(this->breakX.size()));
}

template <typename... POINTS, typename RestrictPack>
xpress::objects::PWLExpressionImplementation::PWLExpressionImplementation(
    int argument, POINTS... points)
    : PWLExpressionImplementation(
          argument, breakToX(xpress::packToArray<PwlBreakpoint>(points...)),
          breakToY(xpress::packToArray<PwlBreakpoint>(points...))) {}

xpress::objects::PWLExpressionImplementation::PWLExpressionImplementation(
    int argument, xpress::SizedArray<PwlBreakpoint const> const &points)
    : PWLExpressionImplementation(argument, breakToX(points),
                                  breakToY(points)) {}

auto xpress::objects::PWLExpressionImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {

  double val = (!argumentVariable) ? solution[argumentColumn]
                                   : argumentVariable.getValue(solution);
  int last = xpress::toInt(breakX.size()) - 1;
  if (val <= breakX[0]) {

    double slope = (breakY[1] - breakY[0]) / (breakX[1] - breakX[0]);
    return breakY[0] - slope * (breakX[0] - val);
  } else if (val >= breakX[last]) {

    double slope =
        (breakY[last] - breakY[last - 1]) / (breakX[last] - breakX[last - 1]);
    return breakY[last] + slope * (val - breakX[last]);
  } else {
    int left = 0;
    while (breakX[left + 1] < val)
      ++left;
    assert(left < last);
    double slope =
        (breakY[left + 1] - breakY[left]) / (breakX[left + 1] - breakX[left]);
    return breakY[left] + slope * (val - breakX[left]);
  }
}

auto xpress::objects::PWLExpressionImplementation::extract(
    PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  extractor->addToken(XPRS_TOK_RB, 0.0);
  for (int i = xpress::toInt(breakX.size()) - 1; i >= 0; --i) {
    extractor->addToken(XPRS_TOK_CON, breakY[i]);
    extractor->addToken(XPRS_TOK_CON, breakX[i]);
  }
  if (!argumentVariable)
    extractor->addToken(XPRS_TOK_COL, argumentColumn);
  else
    extractor->addToken(XPRS_TOK_COL,
                        extractor->dereferenceVariable(argumentVariable));
  extractor->addToken(XPRS_TOK_IFUN, XPRS_IFUN_PWL);
  extractor->endExpression();
}

auto xpress::objects::PWLExpressionImplementation::toString() const
    -> std::string {
  std::stringstream buffer;
  buffer << "pwl";
  buffer << "(";
  if (!argumentVariable) {
    buffer << "column[";
    buffer << argumentColumn;
    buffer << "]";
  } else
    buffer << argumentVariable.toString();
  std::string comma = ", ";
  buffer << comma;
  comma = "";
  for (int i = 0; i < xpress::toInt(breakX.size()); ++i) {
    buffer << comma;
    comma = ", ";
    buffer << "(";
    buffer << InternalUtils::floatToString(breakX[i]);
    buffer << comma;
    buffer << InternalUtils::floatToString(breakY[i]);
    buffer << ")";
  }
  buffer << ")";
  return buffer.str();
}

auto xpress::objects::PWLExpressionImplementation::getMaxDegree() const -> int {
  return std::numeric_limits<int>::max();
}

xpress::objects::UnaryExpressionImplementation::UnaryExpressionImplementation(
    xpress::objects::UnaryExpression::Operator op,
    xpress::objects::Expression argument)
    : op(op), argument(argument) {}

auto xpress::objects::UnaryExpressionImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  double result = 0.0;
  switch (op) {
  case xpress::objects::UnaryExpression::Operator::UMinus:
    result = -argument.evaluate(solution);
    break;
  }
  return result;
}

auto xpress::objects::UnaryExpressionImplementation::extract(
    PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  argument.extract(1.0, extractor);
  extractor->addToken(XPRS_TOK_OP, (int)op);
  extractor->endExpression();
}

auto xpress::objects::UnaryExpressionImplementation::toString() const
    -> std::string {
  std::stringstream buffer;
  switch (op) {
  case xpress::objects::UnaryExpression::Operator::UMinus:
    buffer << "-";
    buffer << argument.toString();
    break;
  }
  return buffer.str();
}

auto xpress::objects::UnaryExpressionImplementation::getMaxDegree() const
    -> int {
  switch (op) {
  case xpress::objects::UnaryExpression::Operator::UMinus:
    return argument.getMaxDegree();
  }
  return std::numeric_limits<int>::max();
}

auto xpress::objects::UnaryExpressionImplementation::getConstantView() const
    -> double {
  return -argument.getConstantView();
}

auto xpress::objects::UnaryExpressionImplementation::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  if (getMaxDegree() > 1)
    throw std::invalid_argument("not a linear expression");
  switch (op) {
  case xpress::objects::UnaryExpression::Operator::UMinus:
    return scaledLinView(argument.getLinearView(), -1.0);
  }
  throw std::invalid_argument("not a linear expression");
}

auto xpress::objects::UnaryExpressionImplementation::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  if (getMaxDegree() > 2)
    throw std::invalid_argument("not a quadratic expression");
  switch (op) {
  case xpress::objects::UnaryExpression::Operator::UMinus:
    return scaledQuadView(argument.getQuadView(), -1.0);
  }
  throw std::invalid_argument("not a quadratic expression");
}

auto xpress::objects::UnaryExpressionImplementation::scaledLinView(
    std::shared_ptr<LinearView> data,
    double factor) -> std::shared_ptr<xpress::objects::LinearView> {
  return std::make_shared<xpress::TransformedView<LinearView, LinearView>>(
      data, [=](LinCoef const &c) {
        return LinCoef(c.getVariable(), factor * c.getCoefficient());
      });
}

auto xpress::objects::UnaryExpressionImplementation::scaledQuadView(
    std::shared_ptr<QuadView> data,
    double factor) -> std::shared_ptr<xpress::objects::QuadView> {
  return std::make_shared<xpress::TransformedView<QuadView, QuadView>>(
      data, [=](QuadCoef const &c) {
        return QuadCoef(c.getVariables(), factor * c.getCoefficient());
      });
}

template <typename... ARGUMENTS, typename RestrictPack>
xpress::objects::UserFunctionExpressionImplementation::
    UserFunctionExpressionImplementation(
        xpress::XPRSProblem::AbstractUserFunction *function,
        ARGUMENTS... arguments)
    : function(function),
      arguments(xpress::packToArray<Expression>(arguments...)), output(1) {}

template <typename... ARGUMENTS, typename RestrictPack>
xpress::objects::UserFunctionExpressionImplementation::
    UserFunctionExpressionImplementation(
        xpress::XPRSProblem::AbstractUserFunction *function, int output,
        ARGUMENTS... arguments)
    : function(function),
      arguments(xpress::packToArray<Expression>(arguments...)), output(output) {
}

xpress::objects::UserFunctionExpressionImplementation::
    UserFunctionExpressionImplementation(
        xpress::XPRSProblem::AbstractUserFunction *function,
        SizedArray<Expression> const &arguments)
    : function(function), arguments(arguments.begin(), arguments.end()),
      output(1) {}

xpress::objects::UserFunctionExpressionImplementation::
    UserFunctionExpressionImplementation(
        xpress::XPRSProblem::AbstractUserFunction *function, int output,
        SizedArray<Expression> const &arguments)
    : function(function), arguments(arguments.begin(), arguments.end()),
      output(output) {}

auto xpress::objects::UserFunctionExpressionImplementation::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  std::vector<double> args(xpress::toInt(arguments.size()));
  for (int i = 0; i < xpress::toInt(arguments.size()); ++i)
    args[i] = arguments[i].evaluate(solution);
  return function->evaluate(output, args);
}

auto xpress::objects::UserFunctionExpressionImplementation::extract(
    PostfixExtractor *extractor) const -> void {
  extractor->startExpression();
  extractor->addToken(XPRS_TOK_RB, 0.0);
  if (function->isMultiOutput()) {
    extractor->addToken(XPRS_TOK_CON, output);
    extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COLON);
  }
  bool comma = false;
  for (int i = xpress::toInt(arguments.size()) - 1; i >= 0; --i) {
    if (comma)
      extractor->addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    arguments[i].extract(1.0, extractor);
    comma = true;
  }
  extractor->addToken(XPRS_TOK_FUN, function->getId());
  extractor->endExpression();
}

auto xpress::objects::UserFunctionExpressionImplementation::toString() const
    -> std::string {
  std::stringstream buffer;
  buffer << function->getName();
  buffer << "(";
  if (function->isMultiOutput()) {
    buffer << xpress::concatStrings(output, ": ");
  }
  std::string comma = "";
  for (int i = 0; i < xpress::toInt(arguments.size()); ++i) {
    buffer << comma;
    comma = ", ";
    buffer << arguments[i].toString();
  }
  buffer << ")";
  return buffer.str();
}

auto xpress::objects::UserFunctionExpressionImplementation::getMaxDegree() const
    -> int {
  return std::numeric_limits<int>::max();
}

namespace xpress {
namespace objects {

class AfterObjectiveCallbackHolder : public CallbackHolder {
  AfterObjectiveCallbackHolder(AfterObjectiveCallbackHolder const &) = delete;
  AfterObjectiveCallbackHolder &
  operator=(AfterObjectiveCallbackHolder const &) = delete;
  AfterObjectiveCallbackHolder(AfterObjectiveCallbackHolder &&) = delete;
  AfterObjectiveCallbackHolder &
  operator=(AfterObjectiveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  AfterObjectiveCallbackHolder(CallbackExceptionHandler *e,
                               std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapAfterObjectiveCallback(XPRSprob cbprob, void *context) {
    AfterObjectiveCallbackHolder *holder =
        reinterpret_cast<AfterObjectiveCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class BarIterationCallbackHolder : public CallbackHolder {
  BarIterationCallbackHolder(BarIterationCallbackHolder const &) = delete;
  BarIterationCallbackHolder &
  operator=(BarIterationCallbackHolder const &) = delete;
  BarIterationCallbackHolder(BarIterationCallbackHolder &&) = delete;
  BarIterationCallbackHolder &operator=(BarIterationCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int *)> callback;
  BarIterationCallbackHolder(CallbackExceptionHandler *e,
                             std::function<void(XpressProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapBarIterationCallback(XPRSprob cbprob, void *context,
                                       int *p_action) {
    BarIterationCallbackHolder *holder =
        reinterpret_cast<BarIterationCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_action);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class BarlogCallbackHolder : public CallbackHolder {
  BarlogCallbackHolder(BarlogCallbackHolder const &) = delete;
  BarlogCallbackHolder &operator=(BarlogCallbackHolder const &) = delete;
  BarlogCallbackHolder(BarlogCallbackHolder &&) = delete;
  BarlogCallbackHolder &operator=(BarlogCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  BarlogCallbackHolder(CallbackExceptionHandler *e,
                       std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapBarlogCallback(XPRSprob cbprob, void *context) {
    BarlogCallbackHolder *holder =
        reinterpret_cast<BarlogCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class BeforeObjectiveCallbackHolder : public CallbackHolder {
  BeforeObjectiveCallbackHolder(BeforeObjectiveCallbackHolder const &) = delete;
  BeforeObjectiveCallbackHolder &
  operator=(BeforeObjectiveCallbackHolder const &) = delete;
  BeforeObjectiveCallbackHolder(BeforeObjectiveCallbackHolder &&) = delete;
  BeforeObjectiveCallbackHolder &
  operator=(BeforeObjectiveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  BeforeObjectiveCallbackHolder(CallbackExceptionHandler *e,
                                std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapBeforeObjectiveCallback(XPRSprob cbprob, void *context) {
    BeforeObjectiveCallbackHolder *holder =
        reinterpret_cast<BeforeObjectiveCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class BeforeSolveCallbackHolder : public CallbackHolder {
  BeforeSolveCallbackHolder(BeforeSolveCallbackHolder const &) = delete;
  BeforeSolveCallbackHolder &
  operator=(BeforeSolveCallbackHolder const &) = delete;
  BeforeSolveCallbackHolder(BeforeSolveCallbackHolder &&) = delete;
  BeforeSolveCallbackHolder &operator=(BeforeSolveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  BeforeSolveCallbackHolder(CallbackExceptionHandler *e,
                            std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapBeforeSolveCallback(XPRSprob cbprob, void *context) {
    BeforeSolveCallbackHolder *holder =
        reinterpret_cast<BeforeSolveCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class ChangeBranchObjectCallbackHolder : public CallbackHolder {
  ChangeBranchObjectCallbackHolder(ChangeBranchObjectCallbackHolder const &) =
      delete;
  ChangeBranchObjectCallbackHolder &
  operator=(ChangeBranchObjectCallbackHolder const &) = delete;
  ChangeBranchObjectCallbackHolder(ChangeBranchObjectCallbackHolder &&) =
      delete;
  ChangeBranchObjectCallbackHolder &
  operator=(ChangeBranchObjectCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, BranchObject *, BranchObject **)>
      callback;
  ChangeBranchObjectCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XpressProblem &, BranchObject *, BranchObject **)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapChangeBranchObjectCallback(XPRSprob cbprob, void *context,
                                             XPRSbranchobject branch,
                                             XPRSbranchobject *p_newbranch) {
    ChangeBranchObjectCallbackHolder *holder =
        reinterpret_cast<ChangeBranchObjectCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      xpress::BranchObject *branchObject =
          branch ? new xpress::BranchObject(branch) : nullptr;
      xpress::BranchObject *p_newbranchObject = nullptr;
      holder->callback(*cbprobObject, branchObject, &p_newbranchObject);
      if (p_newbranchObject && p_newbranchObject != branchObject) {
        *p_newbranch = p_newbranchObject->ptr;
        p_newbranchObject->dtorFunction = nullptr;
        delete p_newbranchObject;
        ObjectMap<void>::removeObject<XPRSbranchobject>(*p_newbranch);
      }

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class CheckTimeCallbackHolder : public CallbackHolder {
  CheckTimeCallbackHolder(CheckTimeCallbackHolder const &) = delete;
  CheckTimeCallbackHolder &operator=(CheckTimeCallbackHolder const &) = delete;
  CheckTimeCallbackHolder(CheckTimeCallbackHolder &&) = delete;
  CheckTimeCallbackHolder &operator=(CheckTimeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  CheckTimeCallbackHolder(CallbackExceptionHandler *e,
                          std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapCheckTimeCallback(XPRSprob cbprob, void *context) {
    CheckTimeCallbackHolder *holder =
        reinterpret_cast<CheckTimeCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class ChgbranchCallbackHolder : public CallbackHolder {
  ChgbranchCallbackHolder(ChgbranchCallbackHolder const &) = delete;
  ChgbranchCallbackHolder &operator=(ChgbranchCallbackHolder const &) = delete;
  ChgbranchCallbackHolder(ChgbranchCallbackHolder &&) = delete;
  ChgbranchCallbackHolder &operator=(ChgbranchCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int *, int *, double *)> callback;
  ChgbranchCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XpressProblem &, int *, int *, double *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapChgbranchCallback(XPRSprob cbprob, void *context,
                                    int *p_entity, int *p_up,
                                    double *p_estdeg) {
    ChgbranchCallbackHolder *holder =
        reinterpret_cast<ChgbranchCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_entity, p_up, p_estdeg);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class ChgnodeCallbackHolder : public CallbackHolder {
  ChgnodeCallbackHolder(ChgnodeCallbackHolder const &) = delete;
  ChgnodeCallbackHolder &operator=(ChgnodeCallbackHolder const &) = delete;
  ChgnodeCallbackHolder(ChgnodeCallbackHolder &&) = delete;
  ChgnodeCallbackHolder &operator=(ChgnodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int *)> callback;
  ChgnodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XpressProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapChgnodeCallback(XPRSprob cbprob, void *context, int *p_node) {
    ChgnodeCallbackHolder *holder =
        reinterpret_cast<ChgnodeCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_node);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class ComputeRestartCallbackHolder : public CallbackHolder {
  ComputeRestartCallbackHolder(ComputeRestartCallbackHolder const &) = delete;
  ComputeRestartCallbackHolder &
  operator=(ComputeRestartCallbackHolder const &) = delete;
  ComputeRestartCallbackHolder(ComputeRestartCallbackHolder &&) = delete;
  ComputeRestartCallbackHolder &
  operator=(ComputeRestartCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  ComputeRestartCallbackHolder(CallbackExceptionHandler *e,
                               std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapComputeRestartCallback(XPRSprob cbprob, void *context) {
    ComputeRestartCallbackHolder *holder =
        reinterpret_cast<ComputeRestartCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class CutRoundCallbackHolder : public CallbackHolder {
  CutRoundCallbackHolder(CutRoundCallbackHolder const &) = delete;
  CutRoundCallbackHolder &operator=(CutRoundCallbackHolder const &) = delete;
  CutRoundCallbackHolder(CutRoundCallbackHolder &&) = delete;
  CutRoundCallbackHolder &operator=(CutRoundCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int, int *)> callback;
  CutRoundCallbackHolder(CallbackExceptionHandler *e,
                         std::function<void(XpressProblem &, int, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapCutRoundCallback(XPRSprob cbprob, void *context,
                                   int ifxpresscuts, int *p_action) {
    CutRoundCallbackHolder *holder =
        reinterpret_cast<CutRoundCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, ifxpresscuts, p_action);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class CutlogCallbackHolder : public CallbackHolder {
  CutlogCallbackHolder(CutlogCallbackHolder const &) = delete;
  CutlogCallbackHolder &operator=(CutlogCallbackHolder const &) = delete;
  CutlogCallbackHolder(CutlogCallbackHolder &&) = delete;
  CutlogCallbackHolder &operator=(CutlogCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  CutlogCallbackHolder(CallbackExceptionHandler *e,
                       std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapCutlogCallback(XPRSprob cbprob, void *context) {
    CutlogCallbackHolder *holder =
        reinterpret_cast<CutlogCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class CutmgrCallbackHolder : public CallbackHolder {
  CutmgrCallbackHolder(CutmgrCallbackHolder const &) = delete;
  CutmgrCallbackHolder &operator=(CutmgrCallbackHolder const &) = delete;
  CutmgrCallbackHolder(CutmgrCallbackHolder &&) = delete;
  CutmgrCallbackHolder &operator=(CutmgrCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  CutmgrCallbackHolder(CallbackExceptionHandler *e,
                       std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapCutmgrCallback(XPRSprob cbprob, void *context) {
    CutmgrCallbackHolder *holder =
        reinterpret_cast<CutmgrCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class GapNotifyCallbackHolder : public CallbackHolder {
  GapNotifyCallbackHolder(GapNotifyCallbackHolder const &) = delete;
  GapNotifyCallbackHolder &operator=(GapNotifyCallbackHolder const &) = delete;
  GapNotifyCallbackHolder(GapNotifyCallbackHolder &&) = delete;
  GapNotifyCallbackHolder &operator=(GapNotifyCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, double *, double *, double *, double *)>
      callback;
  GapNotifyCallbackHolder(CallbackExceptionHandler *e,
                          std::function<void(XpressProblem &, double *,
                                             double *, double *, double *)>
                              c)
      : exceptionHandler(e), callback(c) {}
  static void wrapGapNotifyCallback(XPRSprob cbprob, void *context,
                                    double *p_relgapnotifytarget,
                                    double *p_absgapnotifytarget,
                                    double *p_absgapnotifyobjtarget,
                                    double *p_absgapnotifyboundtarget) {
    GapNotifyCallbackHolder *holder =
        reinterpret_cast<GapNotifyCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_relgapnotifytarget,
                       p_absgapnotifytarget, p_absgapnotifyobjtarget,
                       p_absgapnotifyboundtarget);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class InfnodeCallbackHolder : public CallbackHolder {
  InfnodeCallbackHolder(InfnodeCallbackHolder const &) = delete;
  InfnodeCallbackHolder &operator=(InfnodeCallbackHolder const &) = delete;
  InfnodeCallbackHolder(InfnodeCallbackHolder &&) = delete;
  InfnodeCallbackHolder &operator=(InfnodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  InfnodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapInfnodeCallback(XPRSprob cbprob, void *context) {
    InfnodeCallbackHolder *holder =
        reinterpret_cast<InfnodeCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class IntsolCallbackHolder : public CallbackHolder {
  IntsolCallbackHolder(IntsolCallbackHolder const &) = delete;
  IntsolCallbackHolder &operator=(IntsolCallbackHolder const &) = delete;
  IntsolCallbackHolder(IntsolCallbackHolder &&) = delete;
  IntsolCallbackHolder &operator=(IntsolCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  IntsolCallbackHolder(CallbackExceptionHandler *e,
                       std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapIntsolCallback(XPRSprob cbprob, void *context) {
    IntsolCallbackHolder *holder =
        reinterpret_cast<IntsolCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class LplogCallbackHolder : public CallbackHolder {
  LplogCallbackHolder(LplogCallbackHolder const &) = delete;
  LplogCallbackHolder &operator=(LplogCallbackHolder const &) = delete;
  LplogCallbackHolder(LplogCallbackHolder &&) = delete;
  LplogCallbackHolder &operator=(LplogCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  LplogCallbackHolder(CallbackExceptionHandler *e,
                      std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapLplogCallback(XPRSprob cbprob, void *context) {
    LplogCallbackHolder *holder =
        reinterpret_cast<LplogCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class MessageCallbackHolder : public CallbackHolder {
  MessageCallbackHolder(MessageCallbackHolder const &) = delete;
  MessageCallbackHolder &operator=(MessageCallbackHolder const &) = delete;
  MessageCallbackHolder(MessageCallbackHolder &&) = delete;
  MessageCallbackHolder &operator=(MessageCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, char const *, int, int)> callback;
  MessageCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XpressProblem &, char const *, int, int)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapMessageCallback(XPRSprob cbprob, void *context,
                                  char const *msg, int msglen, int msgtype) {
    MessageCallbackHolder *holder =
        reinterpret_cast<MessageCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, msg, msglen, msgtype);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class MipThreadCallbackHolder : public CallbackHolder {
  MipThreadCallbackHolder(MipThreadCallbackHolder const &) = delete;
  MipThreadCallbackHolder &operator=(MipThreadCallbackHolder const &) = delete;
  MipThreadCallbackHolder(MipThreadCallbackHolder &&) = delete;
  MipThreadCallbackHolder &operator=(MipThreadCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, XpressProblem &)> callback;
  MipThreadCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XpressProblem &, XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapMipThreadCallback(XPRSprob cbprob, void *context,
                                    XPRSprob threadprob) {
    MipThreadCallbackHolder *holder =
        reinterpret_cast<MipThreadCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      XpressProblem *threadprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              threadprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(threadprob);
              });
      holder->callback(*cbprobObject, *threadprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class MipThreadDestroyCallbackHolder : public CallbackHolder {
  MipThreadDestroyCallbackHolder(MipThreadDestroyCallbackHolder const &) =
      delete;
  MipThreadDestroyCallbackHolder &
  operator=(MipThreadDestroyCallbackHolder const &) = delete;
  MipThreadDestroyCallbackHolder(MipThreadDestroyCallbackHolder &&) = delete;
  MipThreadDestroyCallbackHolder &
  operator=(MipThreadDestroyCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  MipThreadDestroyCallbackHolder(CallbackExceptionHandler *e,
                                 std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapMipThreadDestroyCallback(XPRSprob cbprob, void *context) {
    MipThreadDestroyCallbackHolder *holder =
        reinterpret_cast<MipThreadDestroyCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class MiplogCallbackHolder : public CallbackHolder {
  MiplogCallbackHolder(MiplogCallbackHolder const &) = delete;
  MiplogCallbackHolder &operator=(MiplogCallbackHolder const &) = delete;
  MiplogCallbackHolder(MiplogCallbackHolder &&) = delete;
  MiplogCallbackHolder &operator=(MiplogCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  MiplogCallbackHolder(CallbackExceptionHandler *e,
                       std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMiplogCallback(XPRSprob cbprob, void *context) {
    MiplogCallbackHolder *holder =
        reinterpret_cast<MiplogCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class MsJobEndCallbackHolder : public CallbackHolder {
  MsJobEndCallbackHolder(MsJobEndCallbackHolder const &) = delete;
  MsJobEndCallbackHolder &operator=(MsJobEndCallbackHolder const &) = delete;
  MsJobEndCallbackHolder(MsJobEndCallbackHolder &&) = delete;
  MsJobEndCallbackHolder &operator=(MsJobEndCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, void *, char const *, int *)> callback;
  MsJobEndCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(XpressProblem &, void *, char const *, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMsJobEndCallback(XPRSprob cbprob, void *context, void *jobdata,
                                  char const *jobdesc, int *p_status) {
    MsJobEndCallbackHolder *holder =
        reinterpret_cast<MsJobEndCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn =
          holder->callback(*cbprobObject, jobdata, jobdesc, p_status);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class MsJobStartCallbackHolder : public CallbackHolder {
  MsJobStartCallbackHolder(MsJobStartCallbackHolder const &) = delete;
  MsJobStartCallbackHolder &
  operator=(MsJobStartCallbackHolder const &) = delete;
  MsJobStartCallbackHolder(MsJobStartCallbackHolder &&) = delete;
  MsJobStartCallbackHolder &operator=(MsJobStartCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, void *, char const *, int *)> callback;
  MsJobStartCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(XpressProblem &, void *, char const *, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMsJobStartCallback(XPRSprob cbprob, void *context,
                                    void *jobdata, char const *jobdesc,
                                    int *p_status) {
    MsJobStartCallbackHolder *holder =
        reinterpret_cast<MsJobStartCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn =
          holder->callback(*cbprobObject, jobdata, jobdesc, p_status);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class MsWinnerCallbackHolder : public CallbackHolder {
  MsWinnerCallbackHolder(MsWinnerCallbackHolder const &) = delete;
  MsWinnerCallbackHolder &operator=(MsWinnerCallbackHolder const &) = delete;
  MsWinnerCallbackHolder(MsWinnerCallbackHolder &&) = delete;
  MsWinnerCallbackHolder &operator=(MsWinnerCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, void *, char const *)> callback;
  MsWinnerCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(XpressProblem &, void *, char const *)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMsWinnerCallback(XPRSprob cbprob, void *context, void *jobdata,
                                  char const *jobdesc) {
    MsWinnerCallbackHolder *holder =
        reinterpret_cast<MsWinnerCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, jobdata, jobdesc);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class NewnodeCallbackHolder : public CallbackHolder {
  NewnodeCallbackHolder(NewnodeCallbackHolder const &) = delete;
  NewnodeCallbackHolder &operator=(NewnodeCallbackHolder const &) = delete;
  NewnodeCallbackHolder(NewnodeCallbackHolder &&) = delete;
  NewnodeCallbackHolder &operator=(NewnodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int, int, int)> callback;
  NewnodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XpressProblem &, int, int, int)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapNewnodeCallback(XPRSprob cbprob, void *context,
                                  int parentnode, int node, int branch) {
    NewnodeCallbackHolder *holder =
        reinterpret_cast<NewnodeCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, parentnode, node, branch);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class NlpCoefEvalErrorCallbackHolder : public CallbackHolder {
  NlpCoefEvalErrorCallbackHolder(NlpCoefEvalErrorCallbackHolder const &) =
      delete;
  NlpCoefEvalErrorCallbackHolder &
  operator=(NlpCoefEvalErrorCallbackHolder const &) = delete;
  NlpCoefEvalErrorCallbackHolder(NlpCoefEvalErrorCallbackHolder &&) = delete;
  NlpCoefEvalErrorCallbackHolder &
  operator=(NlpCoefEvalErrorCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, int, int)> callback;
  NlpCoefEvalErrorCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(XpressProblem &, int, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapNlpCoefEvalErrorCallback(XPRSprob cbprob, void *context,
                                          int col, int row) {
    NlpCoefEvalErrorCallbackHolder *holder =
        reinterpret_cast<NlpCoefEvalErrorCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, col, row);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class NodeLPSolvedCallbackHolder : public CallbackHolder {
  NodeLPSolvedCallbackHolder(NodeLPSolvedCallbackHolder const &) = delete;
  NodeLPSolvedCallbackHolder &
  operator=(NodeLPSolvedCallbackHolder const &) = delete;
  NodeLPSolvedCallbackHolder(NodeLPSolvedCallbackHolder &&) = delete;
  NodeLPSolvedCallbackHolder &operator=(NodeLPSolvedCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  NodeLPSolvedCallbackHolder(CallbackExceptionHandler *e,
                             std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapNodeLPSolvedCallback(XPRSprob cbprob, void *context) {
    NodeLPSolvedCallbackHolder *holder =
        reinterpret_cast<NodeLPSolvedCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class NodecutoffCallbackHolder : public CallbackHolder {
  NodecutoffCallbackHolder(NodecutoffCallbackHolder const &) = delete;
  NodecutoffCallbackHolder &
  operator=(NodecutoffCallbackHolder const &) = delete;
  NodecutoffCallbackHolder(NodecutoffCallbackHolder &&) = delete;
  NodecutoffCallbackHolder &operator=(NodecutoffCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int)> callback;
  NodecutoffCallbackHolder(CallbackExceptionHandler *e,
                           std::function<void(XpressProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapNodecutoffCallback(XPRSprob cbprob, void *context, int node) {
    NodecutoffCallbackHolder *holder =
        reinterpret_cast<NodecutoffCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, node);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class OptnodeCallbackHolder : public CallbackHolder {
  OptnodeCallbackHolder(OptnodeCallbackHolder const &) = delete;
  OptnodeCallbackHolder &operator=(OptnodeCallbackHolder const &) = delete;
  OptnodeCallbackHolder(OptnodeCallbackHolder &&) = delete;
  OptnodeCallbackHolder &operator=(OptnodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int *)> callback;
  OptnodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XpressProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapOptnodeCallback(XPRSprob cbprob, void *context,
                                  int *p_infeasible) {
    OptnodeCallbackHolder *holder =
        reinterpret_cast<OptnodeCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_infeasible);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class PreIntsolCallbackHolder : public CallbackHolder {
  PreIntsolCallbackHolder(PreIntsolCallbackHolder const &) = delete;
  PreIntsolCallbackHolder &operator=(PreIntsolCallbackHolder const &) = delete;
  PreIntsolCallbackHolder(PreIntsolCallbackHolder &&) = delete;
  PreIntsolCallbackHolder &operator=(PreIntsolCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int, int *, double *)> callback;
  PreIntsolCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XpressProblem &, int, int *, double *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapPreIntsolCallback(XPRSprob cbprob, void *context, int soltype,
                                    int *p_reject, double *p_cutoff) {
    PreIntsolCallbackHolder *holder =
        reinterpret_cast<PreIntsolCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, soltype, p_reject, p_cutoff);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class PrenodeCallbackHolder : public CallbackHolder {
  PrenodeCallbackHolder(PrenodeCallbackHolder const &) = delete;
  PrenodeCallbackHolder &operator=(PrenodeCallbackHolder const &) = delete;
  PrenodeCallbackHolder(PrenodeCallbackHolder &&) = delete;
  PrenodeCallbackHolder &operator=(PrenodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, int *)> callback;
  PrenodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XpressProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapPrenodeCallback(XPRSprob cbprob, void *context,
                                  int *p_infeasible) {
    PrenodeCallbackHolder *holder =
        reinterpret_cast<PrenodeCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_infeasible);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class PresolveCallbackHolder : public CallbackHolder {
  PresolveCallbackHolder(PresolveCallbackHolder const &) = delete;
  PresolveCallbackHolder &operator=(PresolveCallbackHolder const &) = delete;
  PresolveCallbackHolder(PresolveCallbackHolder &&) = delete;
  PresolveCallbackHolder &operator=(PresolveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &)> callback;
  PresolveCallbackHolder(CallbackExceptionHandler *e,
                         std::function<void(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapPresolveCallback(XPRSprob cbprob, void *context) {
    PresolveCallbackHolder *holder =
        reinterpret_cast<PresolveCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class SlpCascadeEndCallbackHolder : public CallbackHolder {
  SlpCascadeEndCallbackHolder(SlpCascadeEndCallbackHolder const &) = delete;
  SlpCascadeEndCallbackHolder &
  operator=(SlpCascadeEndCallbackHolder const &) = delete;
  SlpCascadeEndCallbackHolder(SlpCascadeEndCallbackHolder &&) = delete;
  SlpCascadeEndCallbackHolder &
  operator=(SlpCascadeEndCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  SlpCascadeEndCallbackHolder(CallbackExceptionHandler *e,
                              std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpCascadeEndCallback(XPRSprob cbprob, void *context) {
    SlpCascadeEndCallbackHolder *holder =
        reinterpret_cast<SlpCascadeEndCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpCascadeStartCallbackHolder : public CallbackHolder {
  SlpCascadeStartCallbackHolder(SlpCascadeStartCallbackHolder const &) = delete;
  SlpCascadeStartCallbackHolder &
  operator=(SlpCascadeStartCallbackHolder const &) = delete;
  SlpCascadeStartCallbackHolder(SlpCascadeStartCallbackHolder &&) = delete;
  SlpCascadeStartCallbackHolder &
  operator=(SlpCascadeStartCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  SlpCascadeStartCallbackHolder(CallbackExceptionHandler *e,
                                std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpCascadeStartCallback(XPRSprob cbprob, void *context) {
    SlpCascadeStartCallbackHolder *holder =
        reinterpret_cast<SlpCascadeStartCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpCascadeVarCallbackHolder : public CallbackHolder {
  SlpCascadeVarCallbackHolder(SlpCascadeVarCallbackHolder const &) = delete;
  SlpCascadeVarCallbackHolder &
  operator=(SlpCascadeVarCallbackHolder const &) = delete;
  SlpCascadeVarCallbackHolder(SlpCascadeVarCallbackHolder &&) = delete;
  SlpCascadeVarCallbackHolder &
  operator=(SlpCascadeVarCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, int)> callback;
  SlpCascadeVarCallbackHolder(CallbackExceptionHandler *e,
                              std::function<int(XpressProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpCascadeVarCallback(XPRSprob cbprob, void *context,
                                       int col) {
    SlpCascadeVarCallbackHolder *holder =
        reinterpret_cast<SlpCascadeVarCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, col);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpCascadeVarFailCallbackHolder : public CallbackHolder {
  SlpCascadeVarFailCallbackHolder(SlpCascadeVarFailCallbackHolder const &) =
      delete;
  SlpCascadeVarFailCallbackHolder &
  operator=(SlpCascadeVarFailCallbackHolder const &) = delete;
  SlpCascadeVarFailCallbackHolder(SlpCascadeVarFailCallbackHolder &&) = delete;
  SlpCascadeVarFailCallbackHolder &
  operator=(SlpCascadeVarFailCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, int)> callback;
  SlpCascadeVarFailCallbackHolder(CallbackExceptionHandler *e,
                                  std::function<int(XpressProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpCascadeVarFailCallback(XPRSprob cbprob, void *context,
                                           int col) {
    SlpCascadeVarFailCallbackHolder *holder =
        reinterpret_cast<SlpCascadeVarFailCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, col);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpConstructCallbackHolder : public CallbackHolder {
  SlpConstructCallbackHolder(SlpConstructCallbackHolder const &) = delete;
  SlpConstructCallbackHolder &
  operator=(SlpConstructCallbackHolder const &) = delete;
  SlpConstructCallbackHolder(SlpConstructCallbackHolder &&) = delete;
  SlpConstructCallbackHolder &operator=(SlpConstructCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  SlpConstructCallbackHolder(CallbackExceptionHandler *e,
                             std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpConstructCallback(XPRSprob cbprob, void *context) {
    SlpConstructCallbackHolder *holder =
        reinterpret_cast<SlpConstructCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpDrColCallbackHolder : public CallbackHolder {
  SlpDrColCallbackHolder(SlpDrColCallbackHolder const &) = delete;
  SlpDrColCallbackHolder &operator=(SlpDrColCallbackHolder const &) = delete;
  SlpDrColCallbackHolder(SlpDrColCallbackHolder &&) = delete;
  SlpDrColCallbackHolder &operator=(SlpDrColCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, int, int, double, double *, double,
                    double)>
      callback;
  SlpDrColCallbackHolder(CallbackExceptionHandler *e,
                         std::function<int(XpressProblem &, int, int, double,
                                           double *, double, double)>
                             c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpDrColCallback(XPRSprob prob, void *context, int col,
                                  int detcol, double detval, double *p_value,
                                  double lb, double ub) {
    SlpDrColCallbackHolder *holder =
        reinterpret_cast<SlpDrColCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *probObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              prob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(prob);
              });
      callbackReturn =
          holder->callback(*probObject, col, detcol, detval, p_value, lb, ub);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(prob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpIntSolCallbackHolder : public CallbackHolder {
  SlpIntSolCallbackHolder(SlpIntSolCallbackHolder const &) = delete;
  SlpIntSolCallbackHolder &operator=(SlpIntSolCallbackHolder const &) = delete;
  SlpIntSolCallbackHolder(SlpIntSolCallbackHolder &&) = delete;
  SlpIntSolCallbackHolder &operator=(SlpIntSolCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  SlpIntSolCallbackHolder(CallbackExceptionHandler *e,
                          std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpIntSolCallback(XPRSprob cbprob, void *context) {
    SlpIntSolCallbackHolder *holder =
        reinterpret_cast<SlpIntSolCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpIterEndCallbackHolder : public CallbackHolder {
  SlpIterEndCallbackHolder(SlpIterEndCallbackHolder const &) = delete;
  SlpIterEndCallbackHolder &
  operator=(SlpIterEndCallbackHolder const &) = delete;
  SlpIterEndCallbackHolder(SlpIterEndCallbackHolder &&) = delete;
  SlpIterEndCallbackHolder &operator=(SlpIterEndCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  SlpIterEndCallbackHolder(CallbackExceptionHandler *e,
                           std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpIterEndCallback(XPRSprob cbprob, void *context) {
    SlpIterEndCallbackHolder *holder =
        reinterpret_cast<SlpIterEndCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class SlpIterStartCallbackHolder : public CallbackHolder {
  SlpIterStartCallbackHolder(SlpIterStartCallbackHolder const &) = delete;
  SlpIterStartCallbackHolder &
  operator=(SlpIterStartCallbackHolder const &) = delete;
  SlpIterStartCallbackHolder(SlpIterStartCallbackHolder &&) = delete;
  SlpIterStartCallbackHolder &operator=(SlpIterStartCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &)> callback;
  SlpIterStartCallbackHolder(CallbackExceptionHandler *e,
                             std::function<int(XpressProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpIterStartCallback(XPRSprob cbprob, void *context) {
    SlpIterStartCallbackHolder *holder =
        reinterpret_cast<SlpIterStartCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class SlpIterVarCallbackHolder : public CallbackHolder {
  SlpIterVarCallbackHolder(SlpIterVarCallbackHolder const &) = delete;
  SlpIterVarCallbackHolder &
  operator=(SlpIterVarCallbackHolder const &) = delete;
  SlpIterVarCallbackHolder(SlpIterVarCallbackHolder &&) = delete;
  SlpIterVarCallbackHolder &operator=(SlpIterVarCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, int)> callback;
  SlpIterVarCallbackHolder(CallbackExceptionHandler *e,
                           std::function<int(XpressProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpIterVarCallback(XPRSprob cbprob, void *context, int col) {
    SlpIterVarCallbackHolder *holder =
        reinterpret_cast<SlpIterVarCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, col);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpPreUpdateLinearizationCallbackHolder : public CallbackHolder {
  SlpPreUpdateLinearizationCallbackHolder(
      SlpPreUpdateLinearizationCallbackHolder const &) = delete;
  SlpPreUpdateLinearizationCallbackHolder &
  operator=(SlpPreUpdateLinearizationCallbackHolder const &) = delete;
  SlpPreUpdateLinearizationCallbackHolder(
      SlpPreUpdateLinearizationCallbackHolder &&) = delete;
  SlpPreUpdateLinearizationCallbackHolder &
  operator=(SlpPreUpdateLinearizationCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XpressProblem &, int *)> callback;
  SlpPreUpdateLinearizationCallbackHolder(
      CallbackExceptionHandler *e, std::function<int(XpressProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpPreUpdateLinearizationCallback(XPRSprob cbprob,
                                                   void *context,
                                                   int *ifRepeat) {
    SlpPreUpdateLinearizationCallbackHolder *holder =
        reinterpret_cast<SlpPreUpdateLinearizationCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, ifRepeat);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class UserSolNotifyCallbackHolder : public CallbackHolder {
  UserSolNotifyCallbackHolder(UserSolNotifyCallbackHolder const &) = delete;
  UserSolNotifyCallbackHolder &
  operator=(UserSolNotifyCallbackHolder const &) = delete;
  UserSolNotifyCallbackHolder(UserSolNotifyCallbackHolder &&) = delete;
  UserSolNotifyCallbackHolder &
  operator=(UserSolNotifyCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XpressProblem &, char const *, int)> callback;
  UserSolNotifyCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XpressProblem &, char const *, int)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapUserSolNotifyCallback(XPRSprob cbprob, void *context,
                                        char const *solname, int status) {
    UserSolNotifyCallbackHolder *holder =
        reinterpret_cast<UserSolNotifyCallbackHolder *>(context);
    try {
      XpressProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XpressProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XpressProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, solname, status);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};
} // namespace objects
} // namespace xpress
auto xpress::objects::XpressProblem::CallbackAPI::addLplogCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeLplogCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::LplogCallbackHolder> wrapper =
      std::make_unique<xpress::objects::LplogCallbackHolder>(prob, callback);
  prob->check(XPRSaddcblplog(
      prob->ptr, xpress::objects::LplogCallbackHolder::wrapLplogCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeLplogCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeLplogCallbacks();
  else
    prob->check(XPRSremovecblplog(prob->ptr,
                                  LplogCallbackHolder::wrapLplogCallback,
                                  callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeLplogCallbacks() {
  prob->check(XPRSremovecblplog(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addMiplogCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMiplogCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::MiplogCallbackHolder> wrapper =
      std::make_unique<xpress::objects::MiplogCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbmiplog(
      prob->ptr, xpress::objects::MiplogCallbackHolder::wrapMiplogCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeMiplogCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMiplogCallbacks();
  else
    prob->check(XPRSremovecbmiplog(prob->ptr,
                                   MiplogCallbackHolder::wrapMiplogCallback,
                                   callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeMiplogCallbacks() {
  prob->check(XPRSremovecbmiplog(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addCutlogCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeCutlogCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::CutlogCallbackHolder> wrapper =
      std::make_unique<xpress::objects::CutlogCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbcutlog(
      prob->ptr, xpress::objects::CutlogCallbackHolder::wrapCutlogCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeCutlogCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeCutlogCallbacks();
  else
    prob->check(XPRSremovecbcutlog(prob->ptr,
                                   CutlogCallbackHolder::wrapCutlogCallback,
                                   callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeCutlogCallbacks() {
  prob->check(XPRSremovecbcutlog(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addBarlogCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeBarlogCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::BarlogCallbackHolder> wrapper =
      std::make_unique<xpress::objects::BarlogCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbbarlog(
      prob->ptr, xpress::objects::BarlogCallbackHolder::wrapBarlogCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeBarlogCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeBarlogCallbacks();
  else
    prob->check(XPRSremovecbbarlog(prob->ptr,
                                   BarlogCallbackHolder::wrapBarlogCallback,
                                   callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeBarlogCallbacks() {
  prob->check(XPRSremovecbbarlog(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addOptnodeCallback(
    std::function<void(xpress::objects::XpressProblem &, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeOptnodeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::OptnodeCallbackHolder> wrapper =
      std::make_unique<xpress::objects::OptnodeCallbackHolder>(prob, callback);
  prob->check(XPRSaddcboptnode(
      prob->ptr, xpress::objects::OptnodeCallbackHolder::wrapOptnodeCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeOptnodeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeOptnodeCallbacks();
  else
    prob->check(XPRSremovecboptnode(prob->ptr,
                                    OptnodeCallbackHolder::wrapOptnodeCallback,
                                    callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeOptnodeCallbacks() {
  prob->check(XPRSremovecboptnode(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addPrenodeCallback(
    std::function<void(xpress::objects::XpressProblem &, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removePrenodeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::PrenodeCallbackHolder> wrapper =
      std::make_unique<xpress::objects::PrenodeCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbprenode(
      prob->ptr, xpress::objects::PrenodeCallbackHolder::wrapPrenodeCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removePrenodeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removePrenodeCallbacks();
  else
    prob->check(XPRSremovecbprenode(prob->ptr,
                                    PrenodeCallbackHolder::wrapPrenodeCallback,
                                    callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removePrenodeCallbacks() {
  prob->check(XPRSremovecbprenode(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addInfnodeCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeInfnodeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::InfnodeCallbackHolder> wrapper =
      std::make_unique<xpress::objects::InfnodeCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbinfnode(
      prob->ptr, xpress::objects::InfnodeCallbackHolder::wrapInfnodeCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeInfnodeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeInfnodeCallbacks();
  else
    prob->check(XPRSremovecbinfnode(prob->ptr,
                                    InfnodeCallbackHolder::wrapInfnodeCallback,
                                    callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeInfnodeCallbacks() {
  prob->check(XPRSremovecbinfnode(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addNodecutoffCallback(
    std::function<void(xpress::objects::XpressProblem &, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeNodecutoffCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::NodecutoffCallbackHolder> wrapper =
      std::make_unique<xpress::objects::NodecutoffCallbackHolder>(prob,
                                                                  callback);
  prob->check(XPRSaddcbnodecutoff(
      prob->ptr,
      xpress::objects::NodecutoffCallbackHolder::wrapNodecutoffCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeNodecutoffCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeNodecutoffCallbacks();
  else
    prob->check(XPRSremovecbnodecutoff(
        prob->ptr, NodecutoffCallbackHolder::wrapNodecutoffCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeNodecutoffCallbacks() {
  prob->check(XPRSremovecbnodecutoff(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addIntsolCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeIntsolCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::IntsolCallbackHolder> wrapper =
      std::make_unique<xpress::objects::IntsolCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbintsol(
      prob->ptr, xpress::objects::IntsolCallbackHolder::wrapIntsolCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeIntsolCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeIntsolCallbacks();
  else
    prob->check(XPRSremovecbintsol(prob->ptr,
                                   IntsolCallbackHolder::wrapIntsolCallback,
                                   callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeIntsolCallbacks() {
  prob->check(XPRSremovecbintsol(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addPreIntsolCallback(
    std::function<void(xpress::objects::XpressProblem &, int, int *, double *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removePreIntsolCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::PreIntsolCallbackHolder> wrapper =
      std::make_unique<xpress::objects::PreIntsolCallbackHolder>(prob,
                                                                 callback);
  prob->check(XPRSaddcbpreintsol(
      prob->ptr,
      xpress::objects::PreIntsolCallbackHolder::wrapPreIntsolCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removePreIntsolCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removePreIntsolCallbacks();
  else
    prob->check(XPRSremovecbpreintsol(
        prob->ptr, PreIntsolCallbackHolder::wrapPreIntsolCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removePreIntsolCallbacks() {
  prob->check(XPRSremovecbpreintsol(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addMessageCallback(
    std::function<void(xpress::objects::XpressProblem &, char const *, int,
                       int)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMessageCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::MessageCallbackHolder> wrapper =
      std::make_unique<xpress::objects::MessageCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbmessage(
      prob->ptr, xpress::objects::MessageCallbackHolder::wrapMessageCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeMessageCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMessageCallbacks();
  else
    prob->check(XPRSremovecbmessage(prob->ptr,
                                    MessageCallbackHolder::wrapMessageCallback,
                                    callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeMessageCallbacks() {
  prob->check(XPRSremovecbmessage(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addMipThreadCallback(
    std::function<void(xpress::objects::XpressProblem &,
                       xpress::objects::XpressProblem &)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMipThreadCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::MipThreadCallbackHolder> wrapper =
      std::make_unique<xpress::objects::MipThreadCallbackHolder>(prob,
                                                                 callback);
  prob->check(XPRSaddcbmipthread(
      prob->ptr,
      xpress::objects::MipThreadCallbackHolder::wrapMipThreadCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeMipThreadCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMipThreadCallbacks();
  else
    prob->check(XPRSremovecbmipthread(
        prob->ptr, MipThreadCallbackHolder::wrapMipThreadCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeMipThreadCallbacks() {
  prob->check(XPRSremovecbmipthread(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addMipThreadDestroyCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMipThreadDestroyCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::MipThreadDestroyCallbackHolder> wrapper =
      std::make_unique<xpress::objects::MipThreadDestroyCallbackHolder>(
          prob, callback);
  prob->check(
      XPRSaddcbdestroymt(prob->ptr,
                         xpress::objects::MipThreadDestroyCallbackHolder::
                             wrapMipThreadDestroyCallback,
                         wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeMipThreadDestroyCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMipThreadDestroyCallbacks();
  else
    prob->check(XPRSremovecbdestroymt(
        prob->ptr, MipThreadDestroyCallbackHolder::wrapMipThreadDestroyCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeMipThreadDestroyCallbacks() {
  prob->check(XPRSremovecbdestroymt(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addNewnodeCallback(
    std::function<void(xpress::objects::XpressProblem &, int, int, int)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeNewnodeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::NewnodeCallbackHolder> wrapper =
      std::make_unique<xpress::objects::NewnodeCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbnewnode(
      prob->ptr, xpress::objects::NewnodeCallbackHolder::wrapNewnodeCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeNewnodeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeNewnodeCallbacks();
  else
    prob->check(XPRSremovecbnewnode(prob->ptr,
                                    NewnodeCallbackHolder::wrapNewnodeCallback,
                                    callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeNewnodeCallbacks() {
  prob->check(XPRSremovecbnewnode(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addBarIterationCallback(
    std::function<void(xpress::objects::XpressProblem &, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeBarIterationCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::BarIterationCallbackHolder> wrapper =
      std::make_unique<xpress::objects::BarIterationCallbackHolder>(prob,
                                                                    callback);
  prob->check(XPRSaddcbbariteration(
      prob->ptr,
      xpress::objects::BarIterationCallbackHolder::wrapBarIterationCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeBarIterationCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeBarIterationCallbacks();
  else
    prob->check(XPRSremovecbbariteration(
        prob->ptr, BarIterationCallbackHolder::wrapBarIterationCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeBarIterationCallbacks() {
  prob->check(XPRSremovecbbariteration(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addPresolveCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removePresolveCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::PresolveCallbackHolder> wrapper =
      std::make_unique<xpress::objects::PresolveCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbpresolve(
      prob->ptr, xpress::objects::PresolveCallbackHolder::wrapPresolveCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removePresolveCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removePresolveCallbacks();
  else
    prob->check(XPRSremovecbpresolve(
        prob->ptr, PresolveCallbackHolder::wrapPresolveCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removePresolveCallbacks() {
  prob->check(XPRSremovecbpresolve(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addChangeBranchObjectCallback(
    std::function<void(xpress::objects::XpressProblem &, xpress::BranchObject *,
                       xpress::BranchObject **)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeChangeBranchObjectCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::ChangeBranchObjectCallbackHolder> wrapper =
      std::make_unique<xpress::objects::ChangeBranchObjectCallbackHolder>(
          prob, callback);
  prob->check(XPRSaddcbchgbranchobject(
      prob->ptr,
      xpress::objects::ChangeBranchObjectCallbackHolder::
          wrapChangeBranchObjectCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeChangeBranchObjectCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeChangeBranchObjectCallbacks();
  else
    prob->check(XPRSremovecbchgbranchobject(
        prob->ptr,
        ChangeBranchObjectCallbackHolder::wrapChangeBranchObjectCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeChangeBranchObjectCallbacks() {
  prob->check(XPRSremovecbchgbranchobject(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addComputeRestartCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeComputeRestartCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::ComputeRestartCallbackHolder> wrapper =
      std::make_unique<xpress::objects::ComputeRestartCallbackHolder>(prob,
                                                                      callback);
  prob->check(XPRSaddcbcomputerestart(
      prob->ptr,
      xpress::objects::ComputeRestartCallbackHolder::wrapComputeRestartCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeComputeRestartCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeComputeRestartCallbacks();
  else
    prob->check(XPRSremovecbcomputerestart(
        prob->ptr, ComputeRestartCallbackHolder::wrapComputeRestartCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeComputeRestartCallbacks() {
  prob->check(XPRSremovecbcomputerestart(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addNodeLPSolvedCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeNodeLPSolvedCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::NodeLPSolvedCallbackHolder> wrapper =
      std::make_unique<xpress::objects::NodeLPSolvedCallbackHolder>(prob,
                                                                    callback);
  prob->check(XPRSaddcbnodelpsolved(
      prob->ptr,
      xpress::objects::NodeLPSolvedCallbackHolder::wrapNodeLPSolvedCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeNodeLPSolvedCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeNodeLPSolvedCallbacks();
  else
    prob->check(XPRSremovecbnodelpsolved(
        prob->ptr, NodeLPSolvedCallbackHolder::wrapNodeLPSolvedCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeNodeLPSolvedCallbacks() {
  prob->check(XPRSremovecbnodelpsolved(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addGapNotifyCallback(
    std::function<void(xpress::objects::XpressProblem &, double *, double *,
                       double *, double *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeGapNotifyCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::GapNotifyCallbackHolder> wrapper =
      std::make_unique<xpress::objects::GapNotifyCallbackHolder>(prob,
                                                                 callback);
  prob->check(XPRSaddcbgapnotify(
      prob->ptr,
      xpress::objects::GapNotifyCallbackHolder::wrapGapNotifyCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeGapNotifyCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeGapNotifyCallbacks();
  else
    prob->check(XPRSremovecbgapnotify(
        prob->ptr, GapNotifyCallbackHolder::wrapGapNotifyCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeGapNotifyCallbacks() {
  prob->check(XPRSremovecbgapnotify(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addUserSolNotifyCallback(
    std::function<void(xpress::objects::XpressProblem &, char const *, int)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeUserSolNotifyCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::UserSolNotifyCallbackHolder> wrapper =
      std::make_unique<xpress::objects::UserSolNotifyCallbackHolder>(prob,
                                                                     callback);
  prob->check(XPRSaddcbusersolnotify(
      prob->ptr,
      xpress::objects::UserSolNotifyCallbackHolder::wrapUserSolNotifyCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeUserSolNotifyCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeUserSolNotifyCallbacks();
  else
    prob->check(XPRSremovecbusersolnotify(
        prob->ptr, UserSolNotifyCallbackHolder::wrapUserSolNotifyCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeUserSolNotifyCallbacks() {
  prob->check(XPRSremovecbusersolnotify(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addBeforeSolveCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeBeforeSolveCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::BeforeSolveCallbackHolder> wrapper =
      std::make_unique<xpress::objects::BeforeSolveCallbackHolder>(prob,
                                                                   callback);
  prob->check(XPRSaddcbbeforesolve(
      prob->ptr,
      xpress::objects::BeforeSolveCallbackHolder::wrapBeforeSolveCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeBeforeSolveCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeBeforeSolveCallbacks();
  else
    prob->check(XPRSremovecbbeforesolve(
        prob->ptr, BeforeSolveCallbackHolder::wrapBeforeSolveCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeBeforeSolveCallbacks() {
  prob->check(XPRSremovecbbeforesolve(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addBeforeObjectiveCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeBeforeObjectiveCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::BeforeObjectiveCallbackHolder> wrapper =
      std::make_unique<xpress::objects::BeforeObjectiveCallbackHolder>(
          prob, callback);
  prob->check(
      XPRSaddcbbeforeobjective(prob->ptr,
                               xpress::objects::BeforeObjectiveCallbackHolder::
                                   wrapBeforeObjectiveCallback,
                               wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeBeforeObjectiveCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeBeforeObjectiveCallbacks();
  else
    prob->check(XPRSremovecbbeforeobjective(
        prob->ptr, BeforeObjectiveCallbackHolder::wrapBeforeObjectiveCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeBeforeObjectiveCallbacks() {
  prob->check(XPRSremovecbbeforeobjective(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addAfterObjectiveCallback(
    std::function<void(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeAfterObjectiveCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::AfterObjectiveCallbackHolder> wrapper =
      std::make_unique<xpress::objects::AfterObjectiveCallbackHolder>(prob,
                                                                      callback);
  prob->check(XPRSaddcbafterobjective(
      prob->ptr,
      xpress::objects::AfterObjectiveCallbackHolder::wrapAfterObjectiveCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeAfterObjectiveCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeAfterObjectiveCallbacks();
  else
    prob->check(XPRSremovecbafterobjective(
        prob->ptr, AfterObjectiveCallbackHolder::wrapAfterObjectiveCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeAfterObjectiveCallbacks() {
  prob->check(XPRSremovecbafterobjective(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addCheckTimeCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeCheckTimeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::CheckTimeCallbackHolder> wrapper =
      std::make_unique<xpress::objects::CheckTimeCallbackHolder>(prob,
                                                                 callback);
  prob->check(XPRSaddcbchecktime(
      prob->ptr,
      xpress::objects::CheckTimeCallbackHolder::wrapCheckTimeCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeCheckTimeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeCheckTimeCallbacks();
  else
    prob->check(XPRSremovecbchecktime(
        prob->ptr, CheckTimeCallbackHolder::wrapCheckTimeCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeCheckTimeCallbacks() {
  prob->check(XPRSremovecbchecktime(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addCutRoundCallback(
    std::function<void(xpress::objects::XpressProblem &, int, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeCutRoundCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::CutRoundCallbackHolder> wrapper =
      std::make_unique<xpress::objects::CutRoundCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbcutround(
      prob->ptr, xpress::objects::CutRoundCallbackHolder::wrapCutRoundCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeCutRoundCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeCutRoundCallbacks();
  else
    prob->check(XPRSremovecbcutround(
        prob->ptr, CutRoundCallbackHolder::wrapCutRoundCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeCutRoundCallbacks() {
  prob->check(XPRSremovecbcutround(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpCascadeEndCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpCascadeEndCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpCascadeEndCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpCascadeEndCallbackHolder>(prob,
                                                                     callback);
  prob->check(XPRSaddcbslpcascadeend(
      prob->ptr,
      xpress::objects::SlpCascadeEndCallbackHolder::wrapSlpCascadeEndCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpCascadeEndCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpCascadeEndCallbacks();
  else
    prob->check(XPRSremovecbslpcascadeend(
        prob->ptr, SlpCascadeEndCallbackHolder::wrapSlpCascadeEndCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpCascadeEndCallbacks() {
  prob->check(XPRSremovecbslpcascadeend(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpCascadeStartCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpCascadeStartCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpCascadeStartCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpCascadeStartCallbackHolder>(
          prob, callback);
  prob->check(
      XPRSaddcbslpcascadestart(prob->ptr,
                               xpress::objects::SlpCascadeStartCallbackHolder::
                                   wrapSlpCascadeStartCallback,
                               wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpCascadeStartCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpCascadeStartCallbacks();
  else
    prob->check(XPRSremovecbslpcascadestart(
        prob->ptr, SlpCascadeStartCallbackHolder::wrapSlpCascadeStartCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpCascadeStartCallbacks() {
  prob->check(XPRSremovecbslpcascadestart(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpCascadeVarCallback(
    std::function<int(xpress::objects::XpressProblem &, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpCascadeVarCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpCascadeVarCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpCascadeVarCallbackHolder>(prob,
                                                                     callback);
  prob->check(XPRSaddcbslpcascadevar(
      prob->ptr,
      xpress::objects::SlpCascadeVarCallbackHolder::wrapSlpCascadeVarCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpCascadeVarCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpCascadeVarCallbacks();
  else
    prob->check(XPRSremovecbslpcascadevar(
        prob->ptr, SlpCascadeVarCallbackHolder::wrapSlpCascadeVarCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpCascadeVarCallbacks() {
  prob->check(XPRSremovecbslpcascadevar(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpCascadeVarFailCallback(
    std::function<int(xpress::objects::XpressProblem &, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpCascadeVarFailCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpCascadeVarFailCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpCascadeVarFailCallbackHolder>(
          prob, callback);
  prob->check(XPRSaddcbslpcascadevarfail(
      prob->ptr,
      xpress::objects::SlpCascadeVarFailCallbackHolder::
          wrapSlpCascadeVarFailCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpCascadeVarFailCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpCascadeVarFailCallbacks();
  else
    prob->check(XPRSremovecbslpcascadevarfail(
        prob->ptr,
        SlpCascadeVarFailCallbackHolder::wrapSlpCascadeVarFailCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpCascadeVarFailCallbacks() {
  prob->check(XPRSremovecbslpcascadevarfail(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpConstructCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpConstructCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpConstructCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpConstructCallbackHolder>(prob,
                                                                    callback);
  prob->check(XPRSaddcbslpconstruct(
      prob->ptr,
      xpress::objects::SlpConstructCallbackHolder::wrapSlpConstructCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpConstructCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpConstructCallbacks();
  else
    prob->check(XPRSremovecbslpconstruct(
        prob->ptr, SlpConstructCallbackHolder::wrapSlpConstructCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpConstructCallbacks() {
  prob->check(XPRSremovecbslpconstruct(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpIntSolCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpIntSolCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpIntSolCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpIntSolCallbackHolder>(prob,
                                                                 callback);
  prob->check(XPRSaddcbslpintsol(
      prob->ptr,
      xpress::objects::SlpIntSolCallbackHolder::wrapSlpIntSolCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpIntSolCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpIntSolCallbacks();
  else
    prob->check(XPRSremovecbslpintsol(
        prob->ptr, SlpIntSolCallbackHolder::wrapSlpIntSolCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpIntSolCallbacks() {
  prob->check(XPRSremovecbslpintsol(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpIterEndCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpIterEndCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpIterEndCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpIterEndCallbackHolder>(prob,
                                                                  callback);
  prob->check(XPRSaddcbslpiterend(
      prob->ptr,
      xpress::objects::SlpIterEndCallbackHolder::wrapSlpIterEndCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpIterEndCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpIterEndCallbacks();
  else
    prob->check(XPRSremovecbslpiterend(
        prob->ptr, SlpIterEndCallbackHolder::wrapSlpIterEndCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpIterEndCallbacks() {
  prob->check(XPRSremovecbslpiterend(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpIterStartCallback(
    std::function<int(xpress::objects::XpressProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpIterStartCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpIterStartCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpIterStartCallbackHolder>(prob,
                                                                    callback);
  prob->check(XPRSaddcbslpiterstart(
      prob->ptr,
      xpress::objects::SlpIterStartCallbackHolder::wrapSlpIterStartCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpIterStartCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpIterStartCallbacks();
  else
    prob->check(XPRSremovecbslpiterstart(
        prob->ptr, SlpIterStartCallbackHolder::wrapSlpIterStartCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpIterStartCallbacks() {
  prob->check(XPRSremovecbslpiterstart(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpIterVarCallback(
    std::function<int(xpress::objects::XpressProblem &, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpIterVarCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpIterVarCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpIterVarCallbackHolder>(prob,
                                                                  callback);
  prob->check(XPRSaddcbslpitervar(
      prob->ptr,
      xpress::objects::SlpIterVarCallbackHolder::wrapSlpIterVarCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpIterVarCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpIterVarCallbacks();
  else
    prob->check(XPRSremovecbslpitervar(
        prob->ptr, SlpIterVarCallbackHolder::wrapSlpIterVarCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpIterVarCallbacks() {
  prob->check(XPRSremovecbslpitervar(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addSlpDrColCallback(
    std::function<int(xpress::objects::XpressProblem &, int, int, double,
                      double *, double, double)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpDrColCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpDrColCallbackHolder> wrapper =
      std::make_unique<xpress::objects::SlpDrColCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbslpdrcol(
      prob->ptr, xpress::objects::SlpDrColCallbackHolder::wrapSlpDrColCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpDrColCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpDrColCallbacks();
  else
    prob->check(XPRSremovecbslpdrcol(
        prob->ptr, SlpDrColCallbackHolder::wrapSlpDrColCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeSlpDrColCallbacks() {
  prob->check(XPRSremovecbslpdrcol(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addMsJobStartCallback(
    std::function<int(xpress::objects::XpressProblem &, void *, char const *,
                      int *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMsJobStartCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::MsJobStartCallbackHolder> wrapper =
      std::make_unique<xpress::objects::MsJobStartCallbackHolder>(prob,
                                                                  callback);
  prob->check(XPRSaddcbmsjobstart(
      prob->ptr,
      xpress::objects::MsJobStartCallbackHolder::wrapMsJobStartCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeMsJobStartCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMsJobStartCallbacks();
  else
    prob->check(XPRSremovecbmsjobstart(
        prob->ptr, MsJobStartCallbackHolder::wrapMsJobStartCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeMsJobStartCallbacks() {
  prob->check(XPRSremovecbmsjobstart(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addMsJobEndCallback(
    std::function<int(xpress::objects::XpressProblem &, void *, char const *,
                      int *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMsJobEndCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::MsJobEndCallbackHolder> wrapper =
      std::make_unique<xpress::objects::MsJobEndCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbmsjobend(
      prob->ptr, xpress::objects::MsJobEndCallbackHolder::wrapMsJobEndCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeMsJobEndCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMsJobEndCallbacks();
  else
    prob->check(XPRSremovecbmsjobend(
        prob->ptr, MsJobEndCallbackHolder::wrapMsJobEndCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeMsJobEndCallbacks() {
  prob->check(XPRSremovecbmsjobend(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addMsWinnerCallback(
    std::function<int(xpress::objects::XpressProblem &, void *, char const *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMsWinnerCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::MsWinnerCallbackHolder> wrapper =
      std::make_unique<xpress::objects::MsWinnerCallbackHolder>(prob, callback);
  prob->check(XPRSaddcbmswinner(
      prob->ptr, xpress::objects::MsWinnerCallbackHolder::wrapMsWinnerCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::removeMsWinnerCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMsWinnerCallbacks();
  else
    prob->check(XPRSremovecbmswinner(
        prob->ptr, MsWinnerCallbackHolder::wrapMsWinnerCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::removeMsWinnerCallbacks() {
  prob->check(XPRSremovecbmswinner(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::addNlpCoefEvalErrorCallback(
    std::function<int(xpress::objects::XpressProblem &, int, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeNlpCoefEvalErrorCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::NlpCoefEvalErrorCallbackHolder> wrapper =
      std::make_unique<xpress::objects::NlpCoefEvalErrorCallbackHolder>(
          prob, callback);
  prob->check(XPRSaddcbnlpcoefevalerror(
      prob->ptr,
      xpress::objects::NlpCoefEvalErrorCallbackHolder::
          wrapNlpCoefEvalErrorCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeNlpCoefEvalErrorCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeNlpCoefEvalErrorCallbacks();
  else
    prob->check(XPRSremovecbnlpcoefevalerror(
        prob->ptr, NlpCoefEvalErrorCallbackHolder::wrapNlpCoefEvalErrorCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeNlpCoefEvalErrorCallbacks() {
  prob->check(XPRSremovecbnlpcoefevalerror(prob->ptr, nullptr, nullptr));
}
auto xpress::objects::XpressProblem::CallbackAPI::
    addSlpPreUpdateLinearizationCallback(
        std::function<int(xpress::objects::XpressProblem &, int *)> callback,
        int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpPreUpdateLinearizationCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::objects::SlpPreUpdateLinearizationCallbackHolder>
      wrapper = std::make_unique<
          xpress::objects::SlpPreUpdateLinearizationCallbackHolder>(prob,
                                                                    callback);
  prob->check(XPRSaddcbslppreupdatelinearization(
      prob->ptr,
      xpress::objects::SlpPreUpdateLinearizationCallbackHolder::
          wrapSlpPreUpdateLinearizationCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpPreUpdateLinearizationCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpPreUpdateLinearizationCallbacks();
  else
    prob->check(XPRSremovecbslppreupdatelinearization(
        prob->ptr,
        SlpPreUpdateLinearizationCallbackHolder::
            wrapSlpPreUpdateLinearizationCallback,
        callback.getHolder()));
}
void xpress::objects::XpressProblem::CallbackAPI::
    removeSlpPreUpdateLinearizationCallbacks() {
  prob->check(
      XPRSremovecbslppreupdatelinearization(prob->ptr, nullptr, nullptr));
}

;

auto xpress::objects::XpressProblem::getTopLevel()
    -> xpress::objects::XpressProblem * {
  return !(!toplevel) ? toplevel : this;
}

xpress::objects::XpressProblem::XpressProblem()
    : XpressProblem(std::nullopt, std::nullopt) {}

xpress::objects::XpressProblem::XpressProblem(
    std::optional<std::string> problemName)
    : XpressProblem(problemName, std::nullopt) {}

xpress::objects::XpressProblem::XpressProblem(
    std::optional<std::string> problemName,
    std::optional<std::string> licensePath)
    : XPRSProblem((!(problemName)) ? nullptr : problemName.value().c_str(),
                  (!(licensePath)) ? nullptr : licensePath.value().c_str()),
      callbacks(this), toplevel(nullptr),
      generalConstraintHandler(std::make_shared<GeneralConstraintHandler>()),
      rowHandler(std::make_shared<InequalityHandler>()),
      pwlHandler(std::make_shared<PWLHandler>()),
      sosHandler(std::make_shared<SOSHandler>()),
      variableHandler(std::make_shared<VariableHandler>()) {

  if (!(!(problemName)) && problemName.value().size() > 0) {
    bool error = true;
    {
      xpress::RaiiHelper finally([&]() {
        if (error) {
        }
      });

      setProbName(problemName);
      error = false;
    }
  }
}

xpress::objects::XpressProblem::XpressProblem(XPRSprob callbackProblem,
                                              XpressProblem *toplevel)
    : XPRSProblem(callbackProblem), callbacks(this), toplevel(toplevel),
      generalConstraintHandler(nullptr), rowHandler(nullptr),
      pwlHandler(nullptr), sosHandler(nullptr), variableHandler(nullptr) {
  assert(!(!toplevel));
  assert(!(!toplevel->variableHandler));
  assert(!(!toplevel->rowHandler));
  assert(!(!toplevel->pwlHandler));
  assert(!(!toplevel->generalConstraintHandler));
  assert(!(!toplevel->sosHandler));
}

auto xpress::objects::XpressProblem::makeChild(XPRSprob childPtr)
    -> xpress::objects::XpressProblem * {
  return new XpressProblem(childPtr, this);
}

auto xpress::objects::XpressProblem::isOriginal() -> bool {
  if (!(!toplevel))
    return false;
  return (attributes.getPresolveState() & ((1 << 1) | (1 << 2))) == 0;
}

xpress::objects::XpressProblem::NoCommitRowCreator::NoCommitRowCreator(
    XPRSProblem *prob, int initialInequalitiesSize, int initialNzsSize,
    bool hasNames, bool errorIfVariables)
    : RowCreator(prob, initialInequalitiesSize, initialNzsSize, hasNames,
                 errorIfVariables) {}
auto xpress::objects::XpressProblem::NoCommitRowCreator::commit() -> void {}

auto xpress::objects::XpressProblem::addCut(int cuttype,
                                            InequalityDefinition cut) -> int {

  switch (cut.getType()) {
  case xpress::RowType::LEQ:
    break;
  case xpress::RowType::GEQ:
    break;
  case xpress::RowType::EQ:
    break;
  default:
    throw std::invalid_argument("cuts must be <=, >= or ==");
  }
  Expression lhs = cut.getLhs();
  Expression rhs = cut.getRhs();
  if (lhs.getMaxDegree() > 1 || rhs.getMaxDegree() > 1)
    throw std::invalid_argument("cuts must be linear");
  RowCreator creator = NoCommitRowCreator(this, 1, 16, false, false);
  addToRow(1.0, lhs, &creator);
  addToRow(-1.0, rhs, &creator);

  creator.normalizeCurrentRow();
  assert((!creator.qind1) || creator.qind1.getLength() == 0);

  int precols = attributes.getCols();
  std::vector<int> colind(precols);
  std::vector<double> rowcoef(precols);
  int ncoefs;
  double prerhs;
  int status;
  char cutSense = (char)cut.getType();
  presolveRow(cutSense, creator.colinds.getLength(), creator.colinds.getData(),
              creator.rowcoefs.getData(), creator.rhsvals.getData()[0], precols,
              &ncoefs, colind, rowcoef, &prerhs, &status);
  if (status != 0)
    return status;
  addCuts(1, std::vector<int>({cuttype}), std::vector<char>({cutSense}),
          std::vector<double>({prerhs}), std::vector<int>({0, ncoefs}), colind,
          rowcoef);
  return 0;
}

auto xpress::objects::XpressProblem::addManagedCut(
    bool globallyValid, InequalityDefinition cut) -> void {
  switch (cut.getType()) {
  case xpress::RowType::LEQ:
    break;
  case xpress::RowType::GEQ:
    break;
  case xpress::RowType::EQ:
    break;
  default:
    throw std::invalid_argument("cuts must be <=, >= or ==");
  }
  Expression lhs = cut.getLhs();
  Expression rhs = cut.getRhs();
  if (lhs.getMaxDegree() > 1 || rhs.getMaxDegree() > 1)
    throw std::invalid_argument("cuts must be linear");
  RowCreator creator = NoCommitRowCreator(this, 1, 16, false, false);
  addToRow(1.0, lhs, &creator);
  addToRow(-1.0, rhs, &creator);

  creator.normalizeCurrentRow();
  assert((!creator.qind1) || creator.qind1.getLength() == 0);

  addManagedCuts(globallyValid ? 1 : 0, 1,
                 std::vector<char>({(char)cut.getType()}),
                 creator.rhsvals.getData(),
                 std::vector<int>({0, creator.colinds.getLength()}),
                 creator.colinds.getData(), creator.rowcoefs.getData());
}

;
xpress::objects::XpressProblem::BranchObject::BranchObject(XpressProblem *prob)
    : xpress::BranchObject(prob->ptr, true), prob(prob) {}

auto xpress::objects::XpressProblem::BranchObject::addBounds(
    int branch, int nbounds, xpress::SizedArray<char const> const &bndtype,
    std::vector<xpress::objects::Variable> variables,
    xpress::SizedArray<double const> const &bndval) -> void {
  std::vector<int> colind(nbounds);
  for (int i = 0; i < nbounds; ++i)
    colind[i] = variables[i].getIndexForProb(prob);
  xpress::BranchObject::addBounds(branch, nbounds, bndtype, colind, bndval);
}

auto xpress::objects::XpressProblem::BranchObject::addBound(
    int branch, char bndtype, xpress::objects::Variable variable,
    double bndval) -> void {
  addBounds(branch, 1, std::vector<char>({bndtype}),
            std::vector<xpress::objects::Variable>({variable}),
            std::vector<double>({bndval}));
}

auto xpress::objects::XpressProblem::BranchObject::addConstraint(
    int branch, InequalityDefinition constraint) -> void {
  switch (constraint.getType()) {
  case xpress::RowType::LEQ:
    break;
  case xpress::RowType::GEQ:
    break;
  case xpress::RowType::EQ:
    break;
  default:
    throw std::invalid_argument("branch constraints must be <=, >= or ==");
  }
  Expression lhs = constraint.getLhs();
  Expression rhs = constraint.getRhs();
  if (lhs.getMaxDegree() > 1 || rhs.getMaxDegree() > 1)
    throw std::invalid_argument("branch constraints must be linear");
  RowCreator creator = NoCommitRowCreator(prob, 1, 16, false, false);
  prob->addToRow(1.0, lhs, &creator);
  prob->addToRow(-1.0, rhs, &creator);

  creator.normalizeCurrentRow();
  assert((!creator.qind1) || creator.qind1.getLength() == 0);

  xpress::BranchObject::addRows(
      branch, 1, creator.rowcoefs.getLength(),
      std::vector<char>({(char)constraint.getType()}),
      creator.rhsvals.getData(), creator.start.getData(),
      creator.colinds.getData(), creator.rowcoefs.getData());
}

auto xpress::objects::XpressProblem::createBranchObject()
    -> xpress::objects::XpressProblem::BranchObject * {
  return new xpress::objects::XpressProblem::BranchObject(this);
}

auto xpress::objects::XpressProblem::fixupQObjForExpression(
    int length, xpress::SizedArray<double> const &qcoef) -> void {
  for (int i = 0; i < length; ++i)
    qcoef[i] *= 2;
}

auto xpress::objects::XpressProblem::setObjective(
    xpress::objects::Expression obj) -> void {
  std::shared_ptr<RowCreator> creator = extractObjective(obj, true);

  clearObjective();

  bool clear = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (clear)
        clearObjective();
    });

    if (creator->colinds.getLength() > 0)
      XPRSProblem::chgObj(creator->colinds.getLength(),
                          creator->colinds.getData().data(),
                          creator->rowcoefs.getData().data());
    if (!(!creator->qind1) && creator->qind1.getLength() > 0) {

      int qLength = creator->qind1.getLength();
      std::vector<double> qcoef = creator->qcoef.getData();
      fixupQObjForExpression(qLength, qcoef);
      chgMQObj(qLength, creator->qind1.getData().data(),
               creator->qind2.getData().data(), qcoef.data());
    }
    clear = false;
  }
}

auto xpress::objects::XpressProblem::setObjective(
    xpress::objects::Expression obj, xpress::ObjSense sense) -> void {
  bool clear = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (clear)
        clearObjective();
    });

    setObjective(obj);
    chgObjSense(sense);
    clear = false;
  }
}

auto xpress::objects::XpressProblem::extractObjective(
    xpress::objects::Expression obj,
    bool isFirst) -> std::shared_ptr<xpress::XPRSProblem::RowCreator> {
  int maxDegree = obj.getMaxDegree();
  if (maxDegree > 2)
    throw std::invalid_argument("cannot add objective with formula");
  if (!isFirst && maxDegree > 1)
    throw std::invalid_argument("cannot add quadratic multi-objectives");
  std::shared_ptr<RowCreator> creator =
      std::make_shared<RowCreator>(this, 1, 128, false, !isOriginal());

  bool normalize = addToRow(1.0, obj, creator.get());
  assert((!creator->formulaRows) || creator->formulaRows.getLength() == 0);
  if (!(!creator->formulaRows) && creator->formulaRows.getLength() > 0) {

    assert(false);
    throw std::invalid_argument("cannot add objective with formula");
  }
  if (!isFirst && !(!creator->qind1) && creator->qind1.getLength() > 0) {

    assert(false);
    throw std::invalid_argument("cannot add quadratic multi-objectives");
  }
  if (normalize)
    creator->normalizeCurrentRow();

  double rhs = creator->rhsvals.getData()[0];
  if (rhs != 0) {

    creator->addNonzero(-1, rhs);
  }

  creator->normalizeCurrentRow();
  return creator;
}

auto xpress::objects::XpressProblem::addObjective(
    xpress::objects::Expression obj, int priority, double weight) -> void {
  std::shared_ptr<RowCreator> creator = extractObjective(obj, false);
  addObj(creator->colinds.getLength(), creator->colinds.getData(),
         creator->rowcoefs.getData(),

         priority, weight);
}

auto xpress::objects::XpressProblem::setObjective(
    int idx, xpress::objects::Expression obj, int priority, double weight,
    double abstol, double reltol) -> void {
  if (idx < 0)
    throw std::invalid_argument("negative objective index");

  std::vector<int> allCols(attributes.getCols() + 1);
  std::vector<double> zero(xpress::toInt(allCols.size()));

  int oldCount = attributes.getObjectives();
  bool isNew = idx >= oldCount;

  int origPriority = 0;
  double origWeight = 0, origAbsTol = 0, origRelTol = 0;
  if (!isNew || idx == 0) {
    origPriority = getObjIntControl(idx, xpress::ObjControl::Priority);
    origWeight = getObjDblControl(idx, xpress::ObjControl::Weight);
    origAbsTol = getObjDblControl(idx, xpress::ObjControl::AbsTol);
    origRelTol = getObjDblControl(idx, xpress::ObjControl::RelTol);
  }
  bool undo = true;
  if (idx == 0) {
    setObjective(obj);
    {
      xpress::RaiiHelper finally([&]() {
        if (undo) {

          allCols[0] = -1;
          for (int i = 1; i < xpress::toInt(allCols.size()); ++i)
            allCols[i] = i - 1;
          ::XPRSchgobj(ptr, xpress::toInt(allCols.size()), allCols.data(),
                       zero.data());
          ::XPRSdelqmatrix(ptr, -1);
          ::XPRSsetobjintcontrol(ptr, 0,
                                 static_cast<int>(xpress::ObjControl::Priority),
                                 origPriority);
          ::XPRSsetobjdblcontrol(
              ptr, 0, static_cast<int>(xpress::ObjControl::Weight), origWeight);
          ::XPRSsetobjdblcontrol(
              ptr, 0, static_cast<int>(xpress::ObjControl::AbsTol), origAbsTol);
          ::XPRSsetobjdblcontrol(
              ptr, 0, static_cast<int>(xpress::ObjControl::RelTol), origRelTol);
        }
      });

      setObjIntControl(0, xpress::ObjControl::Priority, priority);
      setObjDblControl(0, xpress::ObjControl::Weight, weight);
      setObjDblControl(0, xpress::ObjControl::AbsTol, abstol);
      setObjDblControl(0, xpress::ObjControl::RelTol, reltol);
      undo = false;
    }

  } else {
    std::shared_ptr<RowCreator> creator = extractObjective(obj, false);

    if (!isNew) {
      allCols[0] = -1;
      for (int i = 1; i < xpress::toInt(allCols.size()); ++i)
        allCols[i] = i - 1;
      XPRSProblem::chgObjN(idx, xpress::toInt(allCols.size()), allCols.data(),
                           zero.data());
    }

    XPRSProblem::chgObjN(idx, creator->colinds.getLength(),
                         creator->colinds.getData().data(),
                         creator->rowcoefs.getData().data());
    {
      xpress::RaiiHelper finally([&]() {
        if (undo) {

          if (isNew) {

            while (idx >= oldCount) {
              ::XPRSdelobj(ptr, idx);
              --idx;
            }
          } else {

            ::XPRSchgobjn(ptr, idx, xpress::toInt(allCols.size()),
                          allCols.data(), zero.data());

            ::XPRSsetobjintcontrol(
                ptr, idx, static_cast<int>(xpress::ObjControl::Priority),
                origPriority);
            ::XPRSsetobjdblcontrol(ptr, idx,
                                   static_cast<int>(xpress::ObjControl::Weight),
                                   origWeight);
            ::XPRSsetobjdblcontrol(ptr, idx,
                                   static_cast<int>(xpress::ObjControl::AbsTol),
                                   origAbsTol);
            ::XPRSsetobjdblcontrol(ptr, idx,
                                   static_cast<int>(xpress::ObjControl::RelTol),
                                   origRelTol);
          }
        }
      });

      setObjIntControl(idx, xpress::ObjControl::Priority, priority);
      setObjDblControl(idx, xpress::ObjControl::Weight, weight);
      setObjDblControl(idx, xpress::ObjControl::AbsTol, abstol);
      setObjDblControl(idx, xpress::ObjControl::RelTol, reltol);
      undo = false;
    }
  }
}

auto xpress::objects::XpressProblem::nlpSetInitVal(
    std::unordered_map<Variable, double> values) -> void {
  std::vector<int> idx(values.size());
  std::vector<double> val(values.size());
  int next = 0;
  for (auto &e : values) {
    idx[next] = e.first.getIndexForProb(this);
    val[next] = e.second;
    ++next;
  }

  xpress::objects::InternalUtils::sortParallelArrays(idx, val, 0,
                                                     xpress::toInt(idx.size()));
  XPRSProblem::nlpSetInitVal(xpress::toInt(idx.size()), idx, val);
}

auto xpress::objects::XpressProblem::nlpSetInitVal(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &values) -> void {
  if (xpress::toInt(variables.size()) != xpress::toInt(values.size()))
    throw std::invalid_argument(
        "arrays variables and values have different length");
  int count = xpress::toInt(variables.size());
  std::vector<int> idx(count);
  for (int i = 0; i < count; ++i)
    idx[i] = variables[i].getIndexForProb(this);
  XPRSProblem::nlpSetInitVal(count, idx, values);
}

auto xpress::objects::XpressProblem::addToRow(
    double factor, xpress::objects::Expression expr,
    xpress::XPRSProblem::RowCreator *creator) -> bool {
  assert(creator->prob == this);
  switch (expr.getRTTI()) {
  case ExpressionRTTI::Constant:
    [[fallthrough]];
  case ExpressionRTTI::Variable:
    [[fallthrough]];
  case ExpressionRTTI::LinTerm:
    [[fallthrough]];
  case ExpressionRTTI::QuadTerm:
    [[fallthrough]];
  case ExpressionRTTI::LinExpression:
    [[fallthrough]];
  case ExpressionRTTI::QuadExpression:

    return expr.extract(factor, creator);
  default:
    break;
  }

  switch (expr.getMaxDegree()) {
  case 0:
    creator->addRhs(-factor * expr.getConstantView());
    return false;
  case 1: {

    std::shared_ptr<LinearView> view = expr.getLinearView();
    for (auto &e : *view.get()) {
      Variable v = e.getVariable();
      if (LinQuadUtils::isConstant(v))
        creator->addRhs(-factor * e.getCoefficient());
      else
        creator->addNonzero(e.getVariable().getIndexForProb(this),
                            factor * e.getCoefficient());
    }
  }
    return true;
  case 2: {

    std::shared_ptr<QuadView> view = expr.getQuadView();
    for (auto &e : *view.get()) {
      Variable v1 = e.getVar1();
      Variable v2 = e.getVar2();
      if (LinQuadUtils::isConstant(v1, v2))
        creator->addRhs(-factor * e.getCoefficient());
      else if (LinQuadUtils::isLinear(v1, v2))
        creator->addNonzero(v1.getIndexForProb(this),
                            factor * e.getCoefficient());
      else {
        int x1 = v1.getIndexForProb(this);
        int x2 = v2.getIndexForProb(this);
        creator->addNonzero(x1, x2,

                            ((x1 != x2) ? 0.5 : 1.0) * factor *
                                e.getCoefficient());
      }
    }
  }
    return true;
  }

  return expr.extract(factor, creator);
}

template <typename T>
auto xpress::objects::XpressProblem::toIndexArray(
    XpressProblem *prob,
    xpress::SizedArray<T const> const &indices) -> std::vector<int> {
  if (!indices)
    return std::vector<int>(0);
  return toIndexArray(xpress::toInt(indices.size()), prob, indices);
}

template <typename T>
auto xpress::objects::XpressProblem::toIndexArray(
    int len, XpressProblem *prob,
    xpress::SizedArray<T const> const &indices) -> std::vector<int> {
  if (!indices)
    return std::vector<int>(0);
  std::vector<int> ret(len);
  for (int i = 0; i < len; ++i)
    ret[i] = indices[i].getIndexForProb(prob);
  return ret;
}

template <typename Strm0, typename T, typename Strm0IsStream>
auto xpress::objects::XpressProblem::toIndexArray(
    XpressProblem *prob, Strm0 const &indices,
    int indicesLength) -> std::vector<int> {
  if (indicesLength >= 0) {
    std::vector<int> ret(indicesLength);
    int i = 0;
    for (auto &t : indices)
      ret[i++] = t.getIndexForProb(prob);
    return ret;
  } else {
    std::vector<int> ret;
    for (auto &t : indices)
      ret.push_back(t.getIndexForProb(prob));
    return ret;
  }
}

template <typename Strm0, typename T, typename Strm0IsStream>
auto xpress::objects::XpressProblem::toIndexArray(
    XpressProblem *prob, Strm0 const &indices) -> std::vector<int> {
  return toIndexArray(prob, indices, xpress::sizeHint(indices));
}

template <typename Strm0, typename I, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getIndicesForProb(
    XpressProblem *prob, Strm0 const &objs, int objsSize) -> std::vector<int> {
  if (objsSize >= 0) {

    std::vector<int> indices(objsSize + 2);
    int minIndex = 0;
    int maxIndex = 0;
    int i = 0;
    for (auto &idx : objs) {
      int index = idx.getIndexForProb(prob);
      indices[2 + i] = index;
      if (i == 0) {
        minIndex = index;
        maxIndex = index;
      } else {
        if (index < minIndex)
          minIndex = index;
        if (index > maxIndex)
          maxIndex = index;
      }
      ++i;
    }
    indices[0] = minIndex;
    indices[1] = maxIndex;
    return indices;
  } else {

    std::vector<int> data;
    data.push_back(0);
    data.push_back(0);
    int minIndex = 0;
    int maxIndex = 0;
    int i = 0;
    for (auto &idx : objs) {
      int index = idx.getIndexForProb(prob);
      if (i == 0) {
        minIndex = index;
        maxIndex = index;
      } else {
        if (index < minIndex)
          minIndex = index;
        if (index > maxIndex)
          maxIndex = index;
      }
      data.push_back(index);
    }
    data[0] = minIndex;
    data[1] = maxIndex;
    return data;
  }
}

template <typename Strm0, typename I, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getIndicesForProb(
    XpressProblem *prob, Strm0 const &objs) -> std::vector<int> {
  return getIndicesForProb(prob, objs, xpress::sizeHint(objs));
}

auto xpress::objects::XpressProblem::getLhsExpression(int row)
    -> xpress::objects::Expression {
  if (!isOriginal())
    throw CannotPerformOperationException(
        "cannot fetch left-hand side as Expression from callback or for "
        "presolved model");
  std::vector<int> start(2);
  std::vector<int> colind(0);
  std::vector<double> colcoef(0);
  int linearCoefs;
  getRows(start, nullptr, nullptr, 0, &linearCoefs, row, row);
  if (linearCoefs > 0) {
    colind = std::vector<int>(linearCoefs);
    colcoef = std::vector<double>(linearCoefs);
    getRows(start, colind, colcoef, linearCoefs, &linearCoefs, row, row);
  }

  int cols = attributes.getCols();
  std::vector<int> qstart(cols + 1);
  std::vector<int> qcolind(0);
  std::vector<double> qcoef(0);
  int qcoefs;
  getQRowQMatrix(row, qstart, qcolind, qcoef, 0, &qcoefs, 0, cols - 1);
  if (qcoefs > 0) {
    qcolind = std::vector<int>(qcoefs);
    qcoef = std::vector<double>(qcoefs);
    getQRowQMatrix(row, qstart, qcolind, qcoef, qcoefs, &qcoefs, 0, cols - 1);
  }
  Expression part1 = nullptr;
  if (qcoefs > 0) {

    QuadExpression q = QuadExpression::create();
    for (int j = 0; j < cols; ++j) {
      Variable vj = variableForIndex(j);
      for (int k = qstart[j]; k < qstart[j + 1]; ++k) {

        double factor = (qcolind[k] == j) ? 1.0 : 2.0;
        q.addTerm(vj, variableForIndex(qcolind[k]), factor * qcoef[k]);
      }
    }
    if (linearCoefs > 0) {
      for (int k = 0; k < linearCoefs; ++k)
        q.addTerm(variableForIndex(colind[k]), colcoef[k]);
    }
    part1 = q;
  } else if (linearCoefs > 0) {

    LinExpression l = LinExpression::create();
    for (int k = 0; k < linearCoefs; ++k)
      l.addTerm(variableForIndex(colind[k]), colcoef[k]);
    part1 = l;
  }

  int ntypes;
  nlpGetFormula(row, 1, 0, &ntypes, nullptr, nullptr);
  if (ntypes > 1) {
    std::vector<int> type(ntypes);
    std::vector<double> value(ntypes);
    nlpGetFormula(row, 1, ntypes, &ntypes, type, value);

    Expression formula = FormulaExpression::parse(this, type, value);
    if (!part1)
      return formula;
    else
      return BinaryExpression(formula, BinaryExpression::Operator::Plus, part1);
  } else {
    if (!part1)
      return ConstantExpression(0.0);
    else
      return part1;
  }
}

auto xpress::objects::XpressProblem::checkOperation(char const *operation,
                                                    char const *clazz) -> void {
  if (!isOriginal())
    throw CannotPerformOperationException(operation, clazz);
}

template <typename T>
auto xpress::objects::XpressProblem::appendArray(
    std::vector<T> &array, xpress::SizedArray<T const> const &data) -> void {
  array.insert(array.end(), data.begin(), data.end());
}

template <typename Strm0, typename C, typename T, typename Strm0IsStream>
auto xpress::objects::XpressProblem::dereferenceIndices(
    XpressProblem *prob, IndexHandler<C> const *indexHandler, int oldCount,
    int newCount, Strm0 const &) -> xpress::Iterable<C> {
  return xpress::Iterable<C>::fromArray(
      indexHandler->objectsForIndices(prob, oldCount, newCount - 1));
}

template <typename R, typename C, typename I, typename Func0, typename Func1,
          typename Func0IsInvocable, typename Func1IsInvocable>
auto xpress::objects::XpressProblem::makeConstraints(I dataBegin, I dataEnd,
                                                     Func0 generator,
                                                     Func1 makeResult,
                                                     int keyHint) -> R {
  if (!isOriginal())
    throw CannotPerformOperationException(
        "cannot add constraints from callback or in presolved model");
  int oldCount = -1;
  xpress::objects::IndexHandler<C> const *indexHandler = nullptr;

  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator(true);
  std::vector<typename xpress_iteratorvaluetype(I)> keys;
  bool collectKeys = false;
  if (keyHint >= 0) {
    keys = std::vector<typename xpress_iteratorvaluetype(I)>();
    collectKeys = true;
  } else if (keyHint == -1) {
    keys = std::vector<typename xpress_iteratorvaluetype(I)>();
    collectKeys = true;
  }

  bool undo = true;
  R result;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    while (dataBegin != dataEnd) {
      typename xpress_iteratorvaluetype(I) const &t = *dataBegin;
      if (collectKeys)
        keys.push_back(t);
      ConstraintDefinition<C> def = generator(t);
      if (!indexHandler) {
        indexHandler = def.getIndexHandler(this);
        oldCount = getIntAttrib(indexHandler->attribute);
      }
      def.create(creator.get());
      ++dataBegin;
    }
    creator->commit();
    result = makeResult(
        this, indexHandler, oldCount,
        (!indexHandler) ? (oldCount - 1)
                        : getIntAttrib(indexHandler->attribute),
        collectKeys
            ? xpress::Iterable<typename xpress_iteratorvaluetype(I)>::fromArray(
                  keys)
            : xpress::Iterable<typename xpress_iteratorvaluetype(
                  I)>::makeNull());
    undo = false;
  }

  return result;
}

template <typename C, typename I, typename T>
auto xpress::objects::XpressProblem::makeConstraints(I dataBegin, I dataEnd)
    -> xpress::Iterable<C> {

  if (!isOriginal())
    throw CannotPerformOperationException(
        "cannot add constraints from callback or in presolved model");
  int oldCount = -1;
  IndexHandler<C> const *indexHandler = nullptr;

  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator(true);
  bool undo = true;
  xpress::Iterable<C> result;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    while (dataBegin != dataEnd) {
      ConstraintDefinition<C> const &def = *dataBegin;
      if (!indexHandler) {
        indexHandler = def.getIndexHandler(this);
        oldCount = getIntAttrib(indexHandler->attribute);
      }
      def.create(creator.get());
      ++dataBegin;
    }
    creator->commit();
    std::vector<C> resultList;
    if (!indexHandler) {

      resultList = std::vector<C>();
    } else {
      int newCount = getIntAttrib(indexHandler->attribute);
      resultList = std::vector<C>();
      XpressProblem::appendArray<C>(
          resultList,
          indexHandler->objectsForIndices(this, oldCount, newCount - 1));
    }
    result = xpress::Iterable<C>::fromArray(resultList);
    undo = false;
  }

  return result;
}

template <typename I, typename T, typename Func0, typename C,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    I dataBegin, I dataEnd, Func0 defs) -> xpress::Iterable<std::pair<T, C>> {

  std::vector<std::pair<T, C>> result;
  return makeConstraints<xpress::Iterable<std::pair<T, C>>, C>(
      dataBegin, dataEnd, defs,

      [&](XpressProblem *p, IndexHandler<C> const *h, int oldCount, int,
          xpress::Iterable<T> keys) {
        int next = oldCount;
        for (auto &t : keys) {
          result.push_back(std::pair<T, C>(t, h->objectForIndex(p, next)));
          ++next;
        }
        return xpress::Iterable<std::pair<T, C>>::fromArray(result);
      },
      -1);
}

template <typename C>
auto xpress::objects::XpressProblem::addConstraint(ConstraintDefinition<C> def)
    -> C {
  std::vector<ConstraintDefinition<C>> data{def};
  xpress::Iterable<C> s = makeConstraints<C>(data.begin(), data.end());
  return *s.begin();
}

template <typename Strm0, typename C, typename Strm0IsStream>
auto xpress::objects::XpressProblem::addConstraints(Strm0 const &defs)
    -> xpress::Iterable<C> {
  using std::begin;
  using std::end;
  return

      makeConstraints<C>(begin(defs), end(defs));
}

template <typename I, typename IIsIntegral, typename Func0, typename C,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(I const &count, Func0 defs)
    -> xpress::Iterable<C> {
  xpress::RangeIterable<I> it(0, count);

  using std::begin;
  using std::end;
  return makeConstraints<xpress::Iterable<C>, C,
                         typename xpress::RangeIterable<I>::iterator>(
      begin(it), end(it), defs, dereferenceIndices<xpress::Iterable<I>, C, I>,
      -2);
}

template <typename Strm0, typename T, typename Func0, typename C,
          typename Strm0IsStream, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    Strm0 const &data, Func0 defs) -> xpress::Iterable<C> {

  using std::begin;
  using std::end;
  return makeConstraints<xpress::Iterable<C>, C>(
      begin(data), end(data), defs,
      dereferenceIndices<xpress::Iterable<T>, C, T>, -2);
}

auto xpress::objects::XpressProblem::getGeneralConstraints()
    -> std::vector<xpress::objects::GeneralConstraint> {
  int gencons = attributes.getOriginalGencons();
  return generalConstraintForIndices(0, gencons - 1);
}

xpress::objects::XpressProblem::GeneralConstraintHandler::
    GeneralConstraintHandler()
    : IndexHandler<GeneralConstraint>(XPRS_ORIGINALGENCONS) {}
auto xpress::objects::XpressProblem::GeneralConstraintHandler::makeInstance(
    XpressProblem *prob, long serialNo,
    int index) const -> xpress::objects::GeneralConstraint {
  return GeneralConstraint(prob, serialNo, index);
}

auto xpress::objects::XpressProblem::GeneralConstraintHandler::lowLevelDelete(
    XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  prob->checkNullObject();

  if (count <= 0)
    return;
  int rval = ::XPRSdelgencons(prob->ptr, count, indices.data);

  prob->consumeCBException();
  if (rval != 0) {
    throw XPRSException::error(prob->ptr, rval);
  }
}

auto xpress::objects::XpressProblem::GeneralConstraintHandler::delIndices(
    XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  lowLevelDelete(prob, count, indices);
}

auto xpress::objects::XpressProblem::GeneralConstraintHandler::
    objectsForIndices(XpressProblem *prob, int first, int last) const
    -> std::vector<xpress::objects::GeneralConstraint> {
  if (last < first)
    return std::vector<GeneralConstraint>(0);
  std::vector<GeneralConstraint> ret(last - first + 1);
  IndexHandler<GeneralConstraint>::fillObjects(prob, first, last, ret);
  return ret;
}

;

auto xpress::objects::XpressProblem::generalConstraintForIndex(int index) const
    -> xpress::objects::GeneralConstraint {
  return generalConstraintHandler->objectForIndex(
      const_cast<XpressProblem *>(this), index);
}

auto xpress::objects::XpressProblem::generalConstraintForIndices(int first,
                                                                 int last) const
    -> std::vector<xpress::objects::GeneralConstraint> {
  std::vector<GeneralConstraint> objs(std::max(0, last - first + 1));
  generalConstraintHandler->fillObjects(const_cast<XpressProblem *>(this),
                                        first, last, objs);
  return objs;
}

auto xpress::objects::XpressProblem::delGenCons(
    int nconstraints, xpress::Array<int const> const &conind) -> void {
  checkNullObject();

  if (!(!generalConstraintHandler))
    generalConstraintHandler->deleteIndices(this, nconstraints, conind);
  else {
    GeneralConstraintHandler::lowLevelDelete(this, nconstraints, conind);
  }
}

auto xpress::objects::XpressProblem::delGeneralConstraints(
    xpress::SizedArray<GeneralConstraint const> const &constraints) -> void {
  checkOperation("delete", "GeneralConstraint");
  generalConstraintHandler->deleteObjects(this, constraints);
}

;

template <typename Strm0, typename T, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getValues(
    std::shared_ptr<IndexHandler<T>>, Strm0 const &objs,
    GetValuesFunctor functor, int objsSize) -> std::vector<double> {
  if (objsSize == 0)
    return std::vector<double>(0);

  std::vector<int> indices =
      getIndicesForProb((!(!toplevel)) ? toplevel : this, objs);
  int first = indices[0];
  std::vector<double> dense = functor(this, first, indices[1]);
  int resultSize = xpress::toInt(indices.size()) - 2;
  std::vector<double> ret(resultSize);
  for (int i = 0; i < resultSize; ++i)
    ret[i] = dense[indices[2 + i] - first];
  return ret;
}

template <typename Strm0, typename T, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getValues(
    std::shared_ptr<IndexHandler<T>> handler, Strm0 const &objs,
    GetValuesFunctor functor) -> std::vector<double> {
  return getValues(handler, objs, functor, xpress::sizeHint(objs));
}

auto xpress::objects::XpressProblem::delIndicator(Inequality row) -> void {
  checkOperation("delete indicator", "Inequality");
  int idx = row.getIndexForProb(this);
  delIndicators(idx, idx);
}

auto xpress::objects::XpressProblem::getIndicator(Inequality row)
    -> std::optional<xpress::objects::IndicatorObjects> {
  checkOperation("query indicator", "Inequality");
  std::vector<int> colind(1);
  std::vector<int> complement(1);
  getIndicators(colind, complement, row.getIndexForProb(this),
                row.getIndexForProb(this));
  if (complement[0] > 0)
    return IndicatorObjects(variableForIndex(colind[0]), true, row);
  else if (complement[0] < 0)
    return IndicatorObjects(variableForIndex(colind[0]), false, row);
  else
    return std::nullopt;
}

auto xpress::objects::XpressProblem::setIndicator(Variable indicatorVariable,
                                                  bool indicatorValue,
                                                  Inequality row) -> void {
  setIndicators(std::vector<Variable>({indicatorVariable}),
                std::vector<bool>({indicatorValue}),
                std::vector<Inequality>({row}));
}

auto xpress::objects::XpressProblem::setIndicators(
    xpress::SizedArray<Variable const> const &indicatorVariable,
    std::vector<bool> const &indicatorValue,
    xpress::SizedArray<Inequality const> const &row) -> void {
  int count = xpress::toInt(indicatorVariable.size());
  if (xpress::toInt(indicatorValue.size()) != 0 &&
      count != xpress::toInt(indicatorValue.size()))
    throw std::invalid_argument(
        "arrays indicatorVariable and indicatorValue have different length");
  if (count != xpress::toInt(row.size()))
    throw std::invalid_argument(
        "arrays indicatorVariable and row have different length");
  if (count <= 0)
    return;
  std::vector<int> comp(count);
  std::vector<int> rowind(count);
  std::vector<int> colind(count);
  for (int i = 0; i < count; ++i) {
    rowind[i] = row[i].getIndexForProb(this);
    colind[i] = indicatorVariable[i].getIndexForProb(this);
    comp[i] = (xpress::toInt(indicatorValue.size()) != 0 && !indicatorValue[i])
                  ? -1
                  : 1;
  }
  XPRSProblem::setIndicators(count, rowind, colind, comp);
}

template <typename Strm0, typename T, typename Func0, typename Func1,
          typename Func2, typename Strm0IsStream, typename Func0IsInvocable,
          typename Func1IsInvocable, typename Func2IsInvocable>
auto xpress::objects::XpressProblem::setIndicators(Strm0 const &data,
                                                   int dataLength,
                                                   Func0 indicatorVariable,
                                                   Func1 indicatorValue,
                                                   Func2 row) -> void {

  std::function<bool(T)> indicatorValueFunc = indicatorValue;
  checkOperation("set indicators", "Inequality");
  xpress::ExtendableIntArray comp =
      ExtendableIntArray(std::vector<int>(dataLength >= 0 ? dataLength : 8));
  xpress::ExtendableIntArray rowind =
      ExtendableIntArray(std::vector<int>(dataLength >= 0 ? dataLength : 8));
  xpress::ExtendableIntArray colind =
      ExtendableIntArray(std::vector<int>(dataLength >= 0 ? dataLength : 8));
  for (auto &t : data) {
    rowind.append(row(t).getIndexForProb(this));
    colind.append(indicatorVariable(t).getIndexForProb(this));
    comp.append((!(!indicatorValueFunc) && !indicatorValue(t)) ? -1 : 1);
  }
  XPRSProblem::setIndicators(comp.getLength(), rowind.getData(),
                             colind.getData(), comp.getData());
}

template <typename Strm0, typename T, typename Func0, typename Func1,
          typename Func2, typename Strm0IsStream, typename Func0IsInvocable,
          typename Func1IsInvocable, typename Func2IsInvocable>
auto xpress::objects::XpressProblem::setIndicators(Strm0 const &data,
                                                   Func0 indicatorVariable,
                                                   Func1 indicatorValue,
                                                   Func2 row) -> void {
  setIndicators(data, xpress::sizeHint(data), indicatorVariable, indicatorValue,
                row);
}

template <typename I, typename IIsIntegral, typename Func0, typename Func1,
          typename Func2, typename Func0IsInvocable, typename Func1IsInvocable,
          typename Func2IsInvocable>
auto xpress::objects::XpressProblem::setIndicators(I count,
                                                   Func0 indicatorVariable,
                                                   Func1 indicatorValue,
                                                   Func2 row) -> void {
  setIndicators(
      xpress::Iterable<I>(std::make_shared<xpress::RangeIterable<I>>(0, count)),
      count, indicatorVariable, indicatorValue, row);
}

xpress::objects::XpressProblem::InequalityHandler::InequalityHandler()
    : xpress::objects::IndexHandler<xpress::objects::Inequality>(
          XPRS_INPUTROWS) {}
auto xpress::objects::XpressProblem::InequalityHandler::makeInstance(
    xpress::objects::XpressProblem *prob, long serialNo,
    int index) const -> xpress::objects::Inequality {
  return Inequality(prob, serialNo, index);
}

auto xpress::objects::XpressProblem::InequalityHandler::lowLevelDelete(
    xpress::objects::XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  prob->checkNullObject();

  if (count <= 0)
    return;
  int rval = ::XPRSdelrows(prob->ptr, count, indices.data);

  prob->consumeCBException();
  if (rval != 0) {
    throw XPRSException::error(prob->ptr, rval);
  }
}

auto xpress::objects::XpressProblem::InequalityHandler::delIndices(
    xpress::objects::XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  lowLevelDelete(prob, count, indices);
}

auto xpress::objects::XpressProblem::InequalityHandler::objectsForIndices(
    XpressProblem *prob, int first,
    int last) const -> std::vector<xpress::objects::Inequality> {
  if (last < first)
    return std::vector<Inequality>(0);
  std::vector<Inequality> ret(last - first + 1);
  IndexHandler<Inequality>::fillObjects(prob, first, last, ret);
  return ret;
}

;

auto xpress::objects::XpressProblem::inequalityForIndex(int index) const
    -> xpress::objects::Inequality {
  return rowHandler->objectForIndex(const_cast<XpressProblem *>(this), index);
}

auto xpress::objects::XpressProblem::inequalitiesForIndices(
    int first, int last) const -> std::vector<xpress::objects::Inequality> {
  std::vector<Inequality> objs(std::max(0, last - first + 1));
  rowHandler->fillObjects(const_cast<XpressProblem *>(this), first, last, objs);
  return objs;
}

auto xpress::objects::XpressProblem::getInequalities()
    -> std::vector<xpress::objects::Inequality> {
  int rows = attributes.getInputRows();
  return inequalitiesForIndices(0, rows - 1);
}

auto xpress::objects::XpressProblem::delRows(
    int nrows, xpress::Array<int const> const &rowind) -> void {
  checkNullObject();

  if (!(!rowHandler))
    rowHandler->deleteIndices(this, nrows, rowind);
  else {
    InequalityHandler::lowLevelDelete(this, nrows, rowind);
  }
}

auto xpress::objects::XpressProblem::delInequalities(
    xpress::SizedArray<Inequality const> const &rows) -> void {
  rowHandler->deleteObjects(this, rows);
}

auto xpress::objects::XpressProblem::getValuesSlacks(
    XPRSProblem *prob, int first, int last) -> std::vector<double> {
  std::vector<double> values(std::max(0, last - first + 1));
  int status;
  prob->getSlacks(&status, values, first, last);
  if (status == static_cast<int>(xpress::SolStatus::NotFound))
    throw std::invalid_argument("no slacks available");
  return values;
}
auto xpress::objects::XpressProblem::getValuesDuals(
    XPRSProblem *prob, int first, int last) -> std::vector<double> {
  std::vector<double> values(std::max(0, last - first + 1));
  int status;
  prob->getSlacks(&status, values, first, last);
  if (status == static_cast<int>(xpress::SolStatus::NotFound))
    throw std::invalid_argument("no duals available");
  return values;
}
auto xpress::objects::XpressProblem::getValuesCallbackSlacks(
    XPRSProblem *prob, int first, int last) -> std::vector<double> {
  bool avail;
  std::vector<double> values = prob->getCallbackSlacks(&avail, first, last);
  if (!avail)
    throw std::invalid_argument("no slacks available");
  return values;
}
auto xpress::objects::XpressProblem::getValuesCallbackDuals(
    XPRSProblem *prob, int first, int last) -> std::vector<double> {
  bool avail;
  std::vector<double> values = prob->getCallbackDuals(&avail, first, last);
  if (!avail)
    throw std::invalid_argument("no duals available");
  return values;
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getSlacks(Strm0 const &rows)
    -> std::vector<double> {
  return getValues<Strm0, Inequality>(rowHandler, rows, getValuesSlacks);
}

auto xpress::objects::XpressProblem::getSlack(Inequality r) -> double {
  return XPRSProblem::getSlack(
      r.getIndexForProb(!(!toplevel) ? toplevel : this));
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getDuals(Strm0 const &rows)
    -> std::vector<double> {
  return getValues<Strm0, Inequality>(rowHandler, rows, getValuesDuals);
}

auto xpress::objects::XpressProblem::getDual(Inequality r) -> double {
  return XPRSProblem::getDual(
      r.getIndexForProb(!(!toplevel) ? toplevel : this));
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getCallbackSlacks(Strm0 const &rows)
    -> std::vector<double> {
  return getValues<Strm0, Inequality>(rowHandler, rows,
                                      getValuesCallbackSlacks);
}

auto xpress::objects::XpressProblem::getCallbackSlack(Inequality r) -> double {
  return getCallbackSlack(r.getIndexForProb(!(!toplevel) ? toplevel : this));
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getCallbackDuals(Strm0 const &rows)
    -> std::vector<double> {
  return getValues<Strm0, Inequality>(rowHandler, rows, getValuesCallbackDuals);
}

auto xpress::objects::XpressProblem::getCallbackDual(Inequality r) -> double {
  return getCallbackDual(r.getIndexForProb(!(!toplevel) ? toplevel : this));
}

auto xpress::objects::XpressProblem::rhsSA(
    int len, xpress::SizedArray<Inequality const> const &rows,
    xpress::SizedArray<double> const &lower,
    xpress::SizedArray<double> const &upper) -> void {
  checkOperation("perform sensitivity analysis", "Inequality");
  std::vector<int> idx = toIndexArray(len, this, rows);
  XPRSProblem::rhsSA(xpress::toInt(idx.size()), idx, lower, upper);
}

auto xpress::objects::XpressProblem::rhsSA(
    xpress::SizedArray<Inequality const> const &rows,
    xpress::SizedArray<double> const &lower,
    xpress::SizedArray<double> const &upper) -> void {
  rhsSA(xpress::toInt(rows.size()), rows, lower, upper);
}

auto xpress::objects::XpressProblem::chgRhs(
    xpress::SizedArray<Inequality const> const &rows,
    xpress::SizedArray<double const> const &newRHS) -> void {
  checkOperation("change right-hand side", "Inequality");
  if (xpress::toInt(rows.size()) != xpress::toInt(newRHS.size()))
    throw std::invalid_argument("arrays have different length");
  XPRSProblem::chgRhs(xpress::toInt(rows.size()),
                      toIndexArray(xpress::toInt(rows.size()), this, rows),
                      newRHS);
}

auto xpress::objects::XpressProblem::chgRowType(
    xpress::SizedArray<Inequality const> const &rows,
    xpress::SizedArray<RowType const> const &rowType) -> void {
  checkOperation("change type", "Inequality");
  if (!rows)
    throw std::invalid_argument("variables cannot be nullptr");
  if (!rowType)
    throw std::invalid_argument("colType cannot be nullptr");
  if (xpress::toInt(rows.size()) != xpress::toInt(rowType.size()))
    throw std::invalid_argument("arrays have different length");
  XPRSProblem::chgRowType(xpress::toInt(rows.size()),
                          toIndexArray(xpress::toInt(rows.size()), this, rows),
                          rowTypeToArray(rowType));
}

auto xpress::objects::XpressProblem::chgRhsRange(
    xpress::SizedArray<Inequality const> const &rows,
    xpress::SizedArray<double const> const &newRange) -> void {
  checkOperation("change right-hand side ranges", "Inequality");
  if (xpress::toInt(rows.size()) != xpress::toInt(newRange.size()))
    throw std::invalid_argument("arrays have different length");
  XPRSProblem::chgRhsRange(xpress::toInt(rows.size()),
                           toIndexArray(xpress::toInt(rows.size()), this, rows),
                           newRange);
}

auto xpress::objects::XpressProblem::loadDelayedRows(
    xpress::SizedArray<Inequality const> const &rows) -> void {
  checkOperation("load delayed rows", "Inequality");
  XPRSProblem::loadDelayedRows(
      xpress::toInt(rows.size()),
      toIndexArray(xpress::toInt(rows.size()), this, rows));
}

auto xpress::objects::XpressProblem::loadModelCuts(
    xpress::SizedArray<Inequality const> const &rows) -> void {
  checkOperation("load model cuts", "Inequality");
  XPRSProblem::loadModelCuts(
      xpress::toInt(rows.size()),
      toIndexArray(xpress::toInt(rows.size()), this, rows));
}

auto xpress::objects::XpressProblem::chgObj(
    Variable variable, double coefficient) -> xpress::objects::XpressProblem & {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  XPRSProblem::chgObj(
      1, std::vector<int>({variable.getIndexForProb(getTopLevel())}),
      std::vector<double>({coefficient}));
  return *this;
}

auto xpress::objects::XpressProblem::chgObj(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &coefficients)
    -> xpress::objects::XpressProblem & {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  if ((!variables) && (!coefficients))
    return *this;
  if (xpress::toInt(variables.size()) != xpress::toInt(coefficients.size()))
    throw std::invalid_argument("arrays have different length");
  XPRSProblem::chgObj(xpress::toInt(variables.size()),
                      toIndexArray(getTopLevel(), variables), coefficients);
  return *this;
}

auto xpress::objects::XpressProblem::chgObjN(int objidx, Variable variable,
                                             double coefficient)
    -> xpress::objects::XpressProblem & {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  XPRSProblem::chgObjN(
      objidx, 1, std::vector<int>({variable.getIndexForProb(getTopLevel())}),
      std::vector<double>({coefficient}));
  return *this;
}

auto xpress::objects::XpressProblem::chgObjN(
    int objidx, xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &coefficients)
    -> xpress::objects::XpressProblem & {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  if ((!variables) && (!coefficients))
    return *this;
  if ((!variables) != ((!coefficients)))
    throw std::invalid_argument(
        "one of the arguments is nullptr the other not");
  if (xpress::toInt(variables.size()) != xpress::toInt(coefficients.size()))
    throw std::invalid_argument("arrays have different length");
  XPRSProblem::chgObjN(objidx, xpress::toInt(variables.size()),
                       toIndexArray(getTopLevel(), variables), coefficients);
  return *this;
}

auto xpress::objects::XpressProblem::chgCoef(Inequality row, Variable variable,
                                             double coefficient)
    -> xpress::objects::XpressProblem & {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  XpressProblem *tl = getTopLevel();
  XPRSProblem::chgCoef(row.getIndexForProb(tl), variable.getIndexForProb(tl),
                       coefficient);
  return *this;
}

auto xpress::objects::XpressProblem::chgCoefs(
    xpress::SizedArray<Inequality const> const &row,
    xpress::SizedArray<Variable const> const &variable,
    xpress::SizedArray<double const> const &coefficient)
    -> xpress::objects::XpressProblem & {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  if ((!variable) && (!row) && (!coefficient))
    return *this;
  if ((!variable) != ((!row)) || ((!variable)) != ((!coefficient)))
    throw std::invalid_argument("some arguments are nullptr some are not");
  if (xpress::toInt(variable.size()) != xpress::toInt(row.size()) ||
      xpress::toInt(variable.size()) != xpress::toInt(coefficient.size()))
    throw std::invalid_argument("arrays have different sizes");
  XpressProblem *tl = getTopLevel();
  XPRSProblem::chgMCoef(

      (XPRSint64)xpress::toInt(row.size()), toIndexArray(tl, row),
      toIndexArray(tl, variable), coefficient);
  return *this;
}

auto xpress::objects::XpressProblem::getPWLs()
    -> std::vector<xpress::objects::PWL> {
  int pwls = attributes.getOriginalPwls();
  return pwlsForIndices(0, pwls - 1);
}

xpress::objects::XpressProblem::PWLHandler::PWLHandler()
    : IndexHandler<PWL>(XPRS_ORIGINALPWLS) {}
auto xpress::objects::XpressProblem::PWLHandler::makeInstance(
    xpress::objects::XpressProblem *prob, long serialNo,
    int index) const -> xpress::objects::PWL {
  return PWL(prob, serialNo, index);
}

auto xpress::objects::XpressProblem::PWLHandler::lowLevelDelete(
    xpress::objects::XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  prob->checkNullObject();

  if (count <= 0)
    return;
  int rval = ::XPRSdelpwlcons(prob->ptr, count, indices.data);

  prob->consumeCBException();
  if (rval != 0) {
    throw XPRSException::error(prob->ptr, rval);
  }
}

auto xpress::objects::XpressProblem::PWLHandler::delIndices(
    xpress::objects::XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  lowLevelDelete(prob, count, indices);
}

auto xpress::objects::XpressProblem::PWLHandler::objectsForIndices(
    XpressProblem *prob, int first,
    int last) const -> std::vector<xpress::objects::PWL> {
  if (last < first)
    return std::vector<PWL>(0);
  std::vector<PWL> ret(last - first + 1);
  IndexHandler<PWL>::fillObjects(prob, first, last, ret);
  return ret;
}

;

auto xpress::objects::XpressProblem::pwlForIndex(int index) const
    -> xpress::objects::PWL {
  return pwlHandler->objectForIndex(const_cast<XpressProblem *>(this), index);
}

auto xpress::objects::XpressProblem::pwlsForIndices(int first, int last) const
    -> std::vector<xpress::objects::PWL> {
  std::vector<PWL> objs(std::max(0, last - first + 1));
  pwlHandler->fillObjects(const_cast<XpressProblem *>(this), first, last, objs);
  return objs;
}

auto xpress::objects::XpressProblem::delPwlCons(
    int npwls, xpress::Array<int const> const &pwlind) -> void {
  checkNullObject();

  if (!(!pwlHandler))
    pwlHandler->deleteIndices(this, npwls, pwlind);
  else
    PWLHandler::lowLevelDelete(this, npwls, pwlind);
}

auto xpress::objects::XpressProblem::delPwlConstraints(
    xpress::SizedArray<PWL const> const &pwls) -> void {
  pwlHandler->deleteObjects(this, pwls);
}

auto xpress::objects::XpressProblem::getCoef(Inequality row,
                                             Variable variable) -> double {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  XpressProblem *tl = getTopLevel();
  return XPRSProblem::getCoef(row.getIndexForProb(tl),
                              variable.getIndexForProb(tl));
}

auto xpress::objects::XpressProblem::getObj(Variable variable) -> double {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  int idx = variable.getIndexForProb(getTopLevel());

  return XPRSProblem::getObj(idx, idx)[0];
}

auto xpress::objects::XpressProblem::getObjN(int objidx,
                                             Variable variable) -> double {
  if (!isOriginal())
    throw CannotReferenceVariablesException();
  int idx = variable.getIndexForProb(getTopLevel());
  return XPRSProblem::getObjN(objidx, idx, idx)[0];
}

auto xpress::objects::XpressProblem::getSOSs()
    -> std::vector<xpress::objects::SOS> {
  int sos = attributes.getOriginalSets();
  return sosForIndices(0, sos - 1);
}

xpress::objects::XpressProblem::SOSHandler::SOSHandler()
    : xpress::objects::IndexHandler<SOS>(XPRS_ORIGINALSETS) {}
auto xpress::objects::XpressProblem::SOSHandler::makeInstance(
    XpressProblem *prob, long serialNo,
    int index) const -> xpress::objects::SOS {
  return SOS(prob, serialNo, index);
}

auto xpress::objects::XpressProblem::SOSHandler::lowLevelDelete(
    XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  prob->checkNullObject();

  if (count <= 0)
    return;
  int rval = ::XPRSdelsets(prob->ptr, count, indices.data);

  prob->consumeCBException();
  if (rval != 0) {
    throw XPRSException::error(prob->ptr, rval);
  }
}

auto xpress::objects::XpressProblem::SOSHandler::delIndices(
    XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  lowLevelDelete(prob, count, indices);
}

auto xpress::objects::XpressProblem::SOSHandler::objectsForIndices(
    XpressProblem *prob, int first,
    int last) const -> std::vector<xpress::objects::SOS> {
  if (last < first)
    return std::vector<SOS>(0);
  std::vector<SOS> ret(last - first + 1);
  IndexHandler<SOS>::fillObjects(prob, first, last, ret);
  return ret;
}

;

auto xpress::objects::XpressProblem::sosForIndex(int index) const
    -> xpress::objects::SOS {
  return sosHandler->objectForIndex(const_cast<XpressProblem *>(this), index);
}

auto xpress::objects::XpressProblem::sosForIndices(int first, int last) const
    -> std::vector<xpress::objects::SOS> {
  std::vector<SOS> objs(std::max(0, last - first + 1));
  sosHandler->fillObjects(const_cast<XpressProblem *>(this), first, last, objs);
  return objs;
}

auto xpress::objects::XpressProblem::delSets(
    int nsets, xpress::Array<int const> const &sosind) -> void {
  checkNullObject();

  if (!(!sosHandler))
    sosHandler->deleteIndices(this, nsets, sosind);
  else {
    SOSHandler::lowLevelDelete(this, nsets, sosind);
  }
}

auto xpress::objects::XpressProblem::delSOS(
    xpress::SizedArray<SOS const> const &soss) -> void {
  sosHandler->deleteObjects(this, soss);
}

auto xpress::objects::XpressProblem::delSOS(SOS sos) -> void {
  delSOS(std::vector<SOS>({sos}));
}

;

xpress::objects::XpressProblem::VariableHandler::VariableHandler()
    : xpress::objects::IndexHandler<Variable>(XPRS_INPUTCOLS) {}
auto xpress::objects::XpressProblem::VariableHandler::makeInstance(
    XpressProblem *prob, long serialNo,
    int index) const -> xpress::objects::Variable {
  return Variable(prob, serialNo, index);
}

auto xpress::objects::XpressProblem::VariableHandler::lowLevelDelete(
    XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  prob->checkNullObject();

  if (count <= 0)
    return;
  int rval = ::XPRSdelcols(prob->ptr, count, indices.data);

  prob->consumeCBException();
  if (rval != 0) {
    throw XPRSException::error(prob->ptr, rval);
  }
}

auto xpress::objects::XpressProblem::VariableHandler::delIndices(
    XpressProblem *prob, int count,
    xpress::Array<int const> const &indices) -> void {
  lowLevelDelete(prob, count, indices);
}

auto xpress::objects::XpressProblem::VariableHandler::objectsForIndices(
    XpressProblem *prob, int first,
    int last) const -> std::vector<xpress::objects::Variable> {
  if (last < first)
    return std::vector<Variable>(0);
  std::vector<Variable> ret(last - first + 1);
  IndexHandler<Variable>::fillObjects(prob, first, last, ret);
  return ret;
}

;

auto xpress::objects::XpressProblem::variableForIndex(int index) const
    -> xpress::objects::Variable {
  return variableHandler->objectForIndex(const_cast<XpressProblem *>(this),
                                         index);
}

auto xpress::objects::XpressProblem::variablesForIndices(
    int first, int last) const -> std::vector<xpress::objects::Variable> {
  std::vector<Variable> objs(std::max(0, last - first + 1));
  variableHandler->fillObjects(const_cast<XpressProblem *>(this), first, last,
                               objs);
  return objs;
}

auto xpress::objects::XpressProblem::delCols(
    int ncols, xpress::Array<int const> const &colind) -> void {
  checkNullObject();

  if (!(!variableHandler))
    variableHandler->deleteIndices(this, ncols, colind);
  else {
    VariableHandler::lowLevelDelete(this, ncols, colind);
  }
}

auto xpress::objects::XpressProblem::delVariables(
    xpress::SizedArray<Variable const> const &vars) -> void {
  variableHandler->deleteObjects(this, vars);
}

auto xpress::objects::XpressProblem::getValuesSolution(
    XPRSProblem *prob, int first, int last) -> std::vector<double> {
  std::vector<double> values(std::max(0, last - first + 1));
  int status;
  prob->getSolution(&status, values, first, last);
  if (status == static_cast<int>(xpress::SolStatus::NotFound))
    throw std::invalid_argument("no solution available");
  return values;
}
auto xpress::objects::XpressProblem::getValuesRedCost(
    XPRSProblem *prob, int first, int last) -> std::vector<double> {
  std::vector<double> values(std::max(0, last - first + 1));
  int status;
  prob->getRedCosts(&status, values, first, last);
  if (status == static_cast<int>(xpress::SolStatus::NotFound))
    throw std::invalid_argument("no reduced costs available");
  return values;
}
auto xpress::objects::XpressProblem::getValuesCallbackSolution(
    XPRSProblem *prob, int first, int last) -> std::vector<double> {
  bool avail;
  std::vector<double> values = prob->getCallbackSolution(&avail, first, last);
  if (!avail)
    throw std::invalid_argument("no solution available");
  return values;
}
auto xpress::objects::XpressProblem::getValuesCallbackRedCost(
    XPRSProblem *prob, int first, int last) -> std::vector<double> {
  bool avail;
  std::vector<double> values = prob->getCallbackRedCosts(&avail, first, last);
  if (!avail)
    throw std::invalid_argument("no reduced costs available");
  return values;
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getSolution(Strm0 const &vars)
    -> std::vector<double> {

  return getValues<Strm0, Variable>(variableHandler, vars, getValuesSolution);
}

auto xpress::objects::XpressProblem::getSolution(Variable v) -> double {

  return XPRSProblem::getSolution(
      v.getIndexForProb(!(!toplevel) ? toplevel : this));
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getRedCosts(Strm0 const &vars)
    -> std::vector<double> {

  return getValues<Strm0, Variable>(variableHandler, vars, getValuesRedCost);
}

auto xpress::objects::XpressProblem::getRedCost(Variable v) -> double {

  return XPRSProblem::getRedCost(
      v.getIndexForProb(!(!toplevel) ? toplevel : this));
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getCallbackSolution(Strm0 const &vars)
    -> std::vector<double> {

  return getValues(variableHandler, vars, getValuesCallbackSolution);
}

auto xpress::objects::XpressProblem::getCallbackSolution(Variable v) -> double {

  return getCallbackSolution(v.getIndexForProb(!(!toplevel) ? toplevel : this));
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::XpressProblem::getCallbackRedCosts(Strm0 const &vars)
    -> std::vector<double> {

  return getValues<Strm0, Variable>(variableHandler, vars,
                                    getValuesCallbackRedCost);
}

auto xpress::objects::XpressProblem::getCallbackRedCost(Variable v) -> double {

  return getCallbackRedCost(v.getIndexForProb(!(!toplevel) ? toplevel : this));
}

auto xpress::objects::XpressProblem::objSA(
    int len, xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double> const &lower,
    xpress::SizedArray<double> const &upper) -> void {
  checkOperation("perform sensitivity", "Variable");
  std::vector<int> idx = toIndexArray(len, this, variables);
  XPRSProblem::objSA(len, idx, lower, upper);
}

auto xpress::objects::XpressProblem::objSA(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double> const &lower,
    xpress::SizedArray<double> const &upper) -> void {
  objSA(xpress::toInt(variables.size()), variables, lower, upper);
}

auto xpress::objects::XpressProblem::bndSA(
    int len, xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double> const &lblower,
    xpress::SizedArray<double> const &lbupper,
    xpress::SizedArray<double> const &ublower,
    xpress::SizedArray<double> const &ubupper) -> void {
  checkOperation("perform sensitivity", "Variable");
  std::vector<int> idx = toIndexArray(len, this, variables);

  XPRSProblem::bndSA(len, idx, lblower, lbupper, ublower, ubupper);
}

auto xpress::objects::XpressProblem::bndSA(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double> const &lblower,
    xpress::SizedArray<double> const &lbupper,
    xpress::SizedArray<double> const &ublower,
    xpress::SizedArray<double> const &ubupper) -> void {
  bndSA(xpress::toInt(variables.size()), variables, lblower, lbupper, ublower,
        ubupper);
}

auto xpress::objects::XpressProblem::addMipSol(
    xpress::SizedArray<double const> const &values,
    xpress::SizedArray<Variable const> const &variables,
    std::optional<std::string> name) -> void {

  XPRSProblem::addMipSol(values, toIndexArray(this, variables), name);
}

auto xpress::objects::XpressProblem::getVariables()
    -> std::vector<xpress::objects::Variable> {
  int cols = attributes.getInputCols();
  return variablesForIndices(0, cols - 1);
}

auto xpress::objects::XpressProblem::chgBounds(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<char const> const &bndType,
    xpress::SizedArray<double const> const &bndValue) -> void {
  checkOperation("change bounds", "Variable");
  if (xpress::toInt(variables.size()) != xpress::toInt(bndType.size()) ||
      xpress::toInt(variables.size()) != xpress::toInt(bndValue.size()))
    throw std::invalid_argument("arrays have different length");
  XPRSProblem::chgBounds(
      xpress::toInt(variables.size()),
      toIndexArray(xpress::toInt(variables.size()), this, variables), bndType,
      bndValue);
}

auto xpress::objects::XpressProblem::chgColType(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<ColumnType const> const &colType) -> void {
  checkOperation("change type", "Variable");
  if (!variables)
    throw std::invalid_argument("variables cannot be nullptr");
  if (!colType)
    throw std::invalid_argument("colType cannot be nullptr");
  if (xpress::toInt(variables.size()) != xpress::toInt(colType.size()))
    throw std::invalid_argument("arrays have different length");
  XPRSProblem::chgColType(
      xpress::toInt(variables.size()),
      toIndexArray(xpress::toInt(variables.size()), this, variables),
      columnTypeToArray(colType));
}

auto xpress::objects::XpressProblem::isNullVariable(Variable v) -> bool {
  return v.isSameImpl(NULL_VARIABLE);
}

auto xpress::objects::XpressProblem::slpSetDetRow(Variable v,
                                                  Inequality i) -> void {
  checkOperation("set determining row", "Variable");
  slpSetDetRow(1, std::vector<int>({v.getIndexForProb(this)}),
               std::vector<int>({i.getIndexForProb(this)}));
}

template <typename C, typename CIsIntegral>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::ArrayBuilder<C> builder)
    -> std::vector<xpress::objects::Variable> {
  std::vector<xpress::objects::Variable> ret =
      xpress::makeArray<xpress::objects::Variable>(builder.getDim(1 - 1));
  buildColumns(builder, [&](std::vector<int> const &indices) {
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1) {
      ret[i1] = variableForIndex(indices[i1]);
    }
  });
  return ret;
}

template <typename I, typename C, typename CIsIntegral, typename Func0,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::ArrayBuilder<C> builder, Func0 makeResult,
    std::function<void(I &, C, xpress::objects::Variable)> addResult) -> I {
  I ret = makeResult();
  buildColumns(builder, [=](std::vector<int> const &indices) {
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1) {
      addResult(ret, i1, variableForIndex(indices[i1]));
    }
  });
  return ret;
}

template <typename K1>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::MapBuilder<K1> builder)
    -> std::unordered_map<K1, xpress::objects::Variable> {
  std::unordered_map<K1, xpress::objects::Variable> ret =
      std::unordered_map<K1, xpress::objects::Variable>();
  std::vector<K1> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1>(
      builder,
      [&](K1 k1, int idx) {
        keys.push_back(k1);
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          K1 key = keys[i];
          int idx = indices[i];
          ret[key] = variableForIndex(idx);
        }
      });
  return ret;
}

template <typename I, typename K1, typename Func0, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::MapBuilder<K1> builder, Func0 makeResult,
    std::function<void(I &, K1, xpress::objects::Variable)> addResult) -> I {
  I ret = makeResult();
  std::vector<K1> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1>(
      builder,
      [&](K1 k1, int idx) {
        keys.push_back(k1);
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          K1 key = keys[i];
          addResult(ret, key, variableForIndex(indices[i]));
        }
      });
  return ret;
}

template <typename C, typename CIsIntegral>
auto xpress::objects::XpressProblem::addVariables(C dim)
    -> xpress::VariableBuilder::VariableArrayBuilder<C> {
  return VariableBuilder::VariableArrayBuilder<C>(xpress::toInt(dim), this);
}

template <typename Iter0, typename K1, typename Iter0IsIterable>
auto xpress::objects::XpressProblem::addVariables(Iter0 const &iterable1)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> {
  return VariableBuilder::VariableMapBuilder<K1>(iterable1, this);
}

auto xpress::objects::XpressProblem::addVariable(
    double lb, double ub, ColumnType type,
    std::optional<std::string> name) -> xpress::objects::Variable {

  ColumnCreator creator = ColumnCreator(this, 1, !(!(name)));
  bool undo = true;
  Variable v;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    int idx = creator.addColumn(this, lb, ub, (char)type, 0.0, name);
    creator.commit(this);
    v = variableForIndex(idx);
    undo = false;
  }

  return v;
}

auto xpress::objects::XpressProblem::addVariable(
    double lb, double ub, ColumnType type, double limit,
    std::optional<std::string> name) -> xpress::objects::Variable {

  ColumnCreator creator = ColumnCreator(this, 1, !(!(name)));
  bool undo = true;
  Variable v;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    int idx = creator.addColumn(this, lb, ub, (char)type,
                                (type == ColumnType::SemiContinuous ||
                                 type == ColumnType::SemiInteger ||
                                 type == ColumnType::PartialInteger)
                                    ? limit
                                    : 0.0,
                                name);
    creator.commit(this);
    v = variableForIndex(idx);
    undo = false;
  }

  return v;
}

auto xpress::objects::XpressProblem::addVariable(
    double lb, double ub, ColumnType type) -> xpress::objects::Variable {
  return addVariable(lb, ub, type, std::nullopt);
}

auto xpress::objects::XpressProblem::addVariable(
    std::optional<std::string> name) -> xpress::objects::Variable {
  return addVariable(0, std::numeric_limits<double>::infinity(),
                     ColumnType::Continuous, name);
}

auto xpress::objects::XpressProblem::addVariable(ColumnType type)
    -> xpress::objects::Variable {
  return addVariable(0, std::numeric_limits<double>::infinity(), type,
                     std::nullopt);
}

auto xpress::objects::XpressProblem::addVariable(
    ColumnType type,
    std::optional<std::string> name) -> xpress::objects::Variable {
  return addVariable(0, std::numeric_limits<double>::infinity(), type, name);
}

auto xpress::objects::XpressProblem::addVariable()
    -> xpress::objects::Variable {
  return addVariable(ColumnType::Continuous);
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Array2Builder<C1, C2> builder)
    -> std::vector<std::vector<xpress::objects::Variable>> {
  std::vector<std::vector<xpress::objects::Variable>> ret =
      xpress::makeArray<xpress::objects::Variable>(builder.getDim(1 - 1),
                                                   builder.getDim(2 - 1));
  buildColumns(builder, [&](std::vector<std::vector<int>> const &indices) {
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
      for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2) {
        ret[i1][i2] = variableForIndex(indices[i1][i2]);
      }
  });
  return ret;
}

template <typename I, typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral, typename Func0, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Array2Builder<C1, C2> builder, Func0 makeResult,
    std::function<void(I &, C1, C2, xpress::objects::Variable)> addResult)
    -> I {
  I ret = makeResult();
  buildColumns(builder, [=](std::vector<std::vector<int>> const &indices) {
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
      for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2) {
        addResult(ret, i1, i2, variableForIndex(indices[i1][i2]));
      }
  });
  return ret;
}

template <typename K1, typename K2>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Map2Builder<K1, K2> builder)
    -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable> {
  xpress::maps::HashMap2<K1, K2, xpress::objects::Variable> ret =
      xpress::maps::HashMap2<K1, K2, xpress::objects::Variable>();
  std::vector<xpress::maps::MapKey2<K1, K2>> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1, K2>(
      builder,
      [&](K1 k1, K2 k2, int idx) {
        keys.push_back(xpress::maps::MapKey2<K1, K2>(k1, k2));
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          xpress::maps::MapKey2<K1, K2> key = keys[i];
          int idx = indices[i];
          ret[key] = variableForIndex(idx);
        }
      });
  return ret;
}

template <typename I, typename K1, typename K2, typename Func0,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Map2Builder<K1, K2> builder, Func0 makeResult,
    std::function<void(I &, K1, K2, xpress::objects::Variable)> addResult)
    -> I {
  I ret = makeResult();
  std::vector<xpress::maps::MapKey2<K1, K2>> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1, K2>(
      builder,
      [&](K1 k1, K2 k2, int idx) {
        keys.push_back(xpress::maps::MapKey2<K1, K2>(k1, k2));
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          xpress::maps::MapKey2<K1, K2> key = keys[i];
          addResult(ret, key.getKey1(), key.getKey2(),
                    variableForIndex(indices[i]));
        }
      });
  return ret;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::objects::XpressProblem::addVariables(C1 dim1, C2 dim2)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2> {
  return VariableBuilder::VariableArray2Builder<C1, C2>(
      xpress::toInt(dim1), xpress::toInt(dim2), this);
}

template <typename Iter0, typename Iter1, typename K1, typename K2,
          typename Iter0IsIterable, typename Iter1IsIterable>
auto xpress::objects::XpressProblem::addVariables(Iter0 const &iterable1,
                                                  Iter1 const &iterable2)
    -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> {
  return VariableBuilder::VariableMap2Builder<K1, K2>(iterable1, iterable2,
                                                      this);
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Array3Builder<C1, C2, C3> builder)
    -> std::vector<std::vector<std::vector<xpress::objects::Variable>>> {
  std::vector<std::vector<std::vector<xpress::objects::Variable>>> ret =
      xpress::makeArray<xpress::objects::Variable>(
          builder.getDim(1 - 1), builder.getDim(2 - 1), builder.getDim(3 - 1));
  buildColumns(builder,
               [&](std::vector<std::vector<std::vector<int>>> const &indices) {
                 for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
                   for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
                     for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3) {
                       ret[i1][i2][i3] = variableForIndex(indices[i1][i2][i3]);
                     }
               });
  return ret;
}

template <typename I, typename C1, typename C2, typename C3,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename Func0, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Array3Builder<C1, C2, C3> builder,
    Func0 makeResult,
    std::function<void(I &, C1, C2, C3, xpress::objects::Variable)> addResult)
    -> I {
  I ret = makeResult();
  buildColumns(
      builder, [=](std::vector<std::vector<std::vector<int>>> const &indices) {
        for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
          for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
            for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3) {
              addResult(ret, i1, i2, i3, variableForIndex(indices[i1][i2][i3]));
            }
      });
  return ret;
}

template <typename K1, typename K2, typename K3>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Map3Builder<K1, K2, K3> builder)
    -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable> {
  xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable> ret =
      xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable>();
  std::vector<xpress::maps::MapKey3<K1, K2, K3>> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1, K2, K3>(
      builder,
      [&](K1 k1, K2 k2, K3 k3, int idx) {
        keys.push_back(xpress::maps::MapKey3<K1, K2, K3>(k1, k2, k3));
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          xpress::maps::MapKey3<K1, K2, K3> key = keys[i];
          int idx = indices[i];
          ret[key] = variableForIndex(idx);
        }
      });
  return ret;
}

template <typename I, typename K1, typename K2, typename K3, typename Func0,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Map3Builder<K1, K2, K3> builder, Func0 makeResult,
    std::function<void(I &, K1, K2, K3, xpress::objects::Variable)> addResult)
    -> I {
  I ret = makeResult();
  std::vector<xpress::maps::MapKey3<K1, K2, K3>> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1, K2, K3>(
      builder,
      [&](K1 k1, K2 k2, K3 k3, int idx) {
        keys.push_back(xpress::maps::MapKey3<K1, K2, K3>(k1, k2, k3));
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          xpress::maps::MapKey3<K1, K2, K3> key = keys[i];
          addResult(ret, key.getKey1(), key.getKey2(), key.getKey3(),
                    variableForIndex(indices[i]));
        }
      });
  return ret;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::objects::XpressProblem::addVariables(C1 dim1, C2 dim2, C3 dim3)
    -> xpress::VariableBuilder::VariableArray3Builder<C1, C2, C3> {
  return VariableBuilder::VariableArray3Builder<C1, C2, C3>(
      xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3), this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename K1,
          typename K2, typename K3, typename Iter0IsIterable,
          typename Iter1IsIterable, typename Iter2IsIterable>
auto xpress::objects::XpressProblem::addVariables(Iter0 const &iterable1,
                                                  Iter1 const &iterable2,
                                                  Iter2 const &iterable3)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> {
  return VariableBuilder::VariableMap3Builder<K1, K2, K3>(iterable1, iterable2,
                                                          iterable3, this);
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4> builder)
    -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>> {
  std::vector<std::vector<std::vector<std::vector<xpress::objects::Variable>>>>
      ret = xpress::makeArray<xpress::objects::Variable>(
          builder.getDim(1 - 1), builder.getDim(2 - 1), builder.getDim(3 - 1),
          builder.getDim(4 - 1));
  buildColumns(
      builder, [&](std::vector<std::vector<std::vector<std::vector<int>>>> const
                       &indices) {
        for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
          for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
            for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3)
              for (int i4 = 0; i4 < builder.getDim(4 - 1); ++i4) {
                ret[i1][i2][i3][i4] = variableForIndex(indices[i1][i2][i3][i4]);
              }
      });
  return ret;
}

template <typename I, typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename Func0, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4> builder,
    Func0 makeResult,
    std::function<void(I &, C1, C2, C3, C4, xpress::objects::Variable)>
        addResult) -> I {
  I ret = makeResult();
  buildColumns(builder,
               [=](std::vector<std::vector<std::vector<std::vector<int>>>> const
                       &indices) {
                 for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
                   for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
                     for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3)
                       for (int i4 = 0; i4 < builder.getDim(4 - 1); ++i4) {
                         addResult(ret, i1, i2, i3, i4,
                                   variableForIndex(indices[i1][i2][i3][i4]));
                       }
               });
  return ret;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> builder)
    -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable> {
  xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable> ret =
      xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable>();
  std::vector<xpress::maps::MapKey4<K1, K2, K3, K4>> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1, K2, K3, K4>(
      builder,
      [&](K1 k1, K2 k2, K3 k3, K4 k4, int idx) {
        keys.push_back(xpress::maps::MapKey4<K1, K2, K3, K4>(k1, k2, k3, k4));
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          xpress::maps::MapKey4<K1, K2, K3, K4> key = keys[i];
          int idx = indices[i];
          ret[key] = variableForIndex(idx);
        }
      });
  return ret;
}

template <typename I, typename K1, typename K2, typename K3, typename K4,
          typename Func0, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> builder,
    Func0 makeResult,
    std::function<void(I &, K1, K2, K3, K4, xpress::objects::Variable)>
        addResult) -> I {
  I ret = makeResult();
  std::vector<xpress::maps::MapKey4<K1, K2, K3, K4>> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1, K2, K3, K4>(
      builder,
      [&](K1 k1, K2 k2, K3 k3, K4 k4, int idx) {
        keys.push_back(xpress::maps::MapKey4<K1, K2, K3, K4>(k1, k2, k3, k4));
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          xpress::maps::MapKey4<K1, K2, K3, K4> key = keys[i];
          addResult(ret, key.getKey1(), key.getKey2(), key.getKey3(),
                    key.getKey4(), variableForIndex(indices[i]));
        }
      });
  return ret;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::objects::XpressProblem::addVariables(C1 dim1, C2 dim2, C3 dim3,
                                                  C4 dim4)
    -> xpress::VariableBuilder::VariableArray4Builder<C1, C2, C3, C4> {
  return VariableBuilder::VariableArray4Builder<C1, C2, C3, C4>(
      xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
      xpress::toInt(dim4), this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename K1, typename K2, typename K3, typename K4,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
auto xpress::objects::XpressProblem::addVariables(Iter0 const &iterable1,
                                                  Iter1 const &iterable2,
                                                  Iter2 const &iterable3,
                                                  Iter3 const &iterable4)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> {
  return VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>(
      iterable1, iterable2, iterable3, iterable4, this);
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder)
    -> std::vector<std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>> {
  std::vector<std::vector<
      std::vector<std::vector<std::vector<xpress::objects::Variable>>>>>
      ret = xpress::makeArray<xpress::objects::Variable>(
          builder.getDim(1 - 1), builder.getDim(2 - 1), builder.getDim(3 - 1),
          builder.getDim(4 - 1), builder.getDim(5 - 1));
  buildColumns(
      builder,
      [&](std::vector<std::vector<
              std::vector<std::vector<std::vector<int>>>>> const &indices) {
        for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
          for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
            for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3)
              for (int i4 = 0; i4 < builder.getDim(4 - 1); ++i4)
                for (int i5 = 0; i5 < builder.getDim(5 - 1); ++i5) {
                  ret[i1][i2][i3][i4][i5] =
                      variableForIndex(indices[i1][i2][i3][i4][i5]);
                }
      });
  return ret;
}

template <typename I, typename C1, typename C2, typename C3, typename C4,
          typename C5, typename C1IsIntegral, typename C2IsIntegral,
          typename C3IsIntegral, typename C4IsIntegral, typename C5IsIntegral,
          typename Func0, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder,
    Func0 makeResult,
    std::function<void(I &, C1, C2, C3, C4, C5, xpress::objects::Variable)>
        addResult) -> I {
  I ret = makeResult();
  buildColumns(
      builder,
      [=](std::vector<std::vector<
              std::vector<std::vector<std::vector<int>>>>> const &indices) {
        for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
          for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
            for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3)
              for (int i4 = 0; i4 < builder.getDim(4 - 1); ++i4)
                for (int i5 = 0; i5 < builder.getDim(5 - 1); ++i5) {
                  addResult(ret, i1, i2, i3, i4, i5,
                            variableForIndex(indices[i1][i2][i3][i4][i5]));
                }
      });
  return ret;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder)
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, xpress::objects::Variable> {
  xpress::maps::HashMap5<K1, K2, K3, K4, K5, xpress::objects::Variable> ret =
      xpress::maps::HashMap5<K1, K2, K3, K4, K5, xpress::objects::Variable>();
  std::vector<xpress::maps::MapKey5<K1, K2, K3, K4, K5>> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1, K2, K3, K4, K5>(
      builder,
      [&](K1 k1, K2 k2, K3 k3, K4 k4, K5 k5, int idx) {
        keys.push_back(
            xpress::maps::MapKey5<K1, K2, K3, K4, K5>(k1, k2, k3, k4, k5));
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          xpress::maps::MapKey5<K1, K2, K3, K4, K5> key = keys[i];
          int idx = indices[i];
          ret[key] = variableForIndex(idx);
        }
      });
  return ret;
}

template <typename I, typename K1, typename K2, typename K3, typename K4,
          typename K5, typename Func0, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::buildVariables(
    xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder,
    Func0 makeResult,
    std::function<void(I &, K1, K2, K3, K4, K5, xpress::objects::Variable)>
        addResult) -> I {
  I ret = makeResult();
  std::vector<xpress::maps::MapKey5<K1, K2, K3, K4, K5>> keys;
  std::vector<int> indices;
  XPRSProblem::buildColumns<K1, K2, K3, K4, K5>(
      builder,
      [&](K1 k1, K2 k2, K3 k3, K4 k4, K5 k5, int idx) {
        keys.push_back(
            xpress::maps::MapKey5<K1, K2, K3, K4, K5>(k1, k2, k3, k4, k5));
        indices.push_back(idx);
      },
      [&]() {
        int len = xpress::toInt(keys.size());
        for (int i = 0; i < len; ++i) {
          xpress::maps::MapKey5<K1, K2, K3, K4, K5> key = keys[i];
          addResult(ret, key.getKey1(), key.getKey2(), key.getKey3(),
                    key.getKey4(), key.getKey5(), variableForIndex(indices[i]));
        }
      });
  return ret;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::objects::XpressProblem::addVariables(C1 dim1, C2 dim2, C3 dim3,
                                                  C4 dim4, C5 dim5)
    -> xpress::VariableBuilder::VariableArray5Builder<C1, C2, C3, C4, C5> {
  return VariableBuilder::VariableArray5Builder<C1, C2, C3, C4, C5>(
      xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
      xpress::toInt(dim4), xpress::toInt(dim5), this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename K1, typename K2, typename K3, typename K4,
          typename K5, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
auto xpress::objects::XpressProblem::addVariables(Iter0 const &iterable1,
                                                  Iter1 const &iterable2,
                                                  Iter2 const &iterable3,
                                                  Iter3 const &iterable4,
                                                  Iter4 const &iterable5)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> {
  return VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>(
      iterable1, iterable2, iterable3, iterable4, iterable5, this);
}

auto xpress::objects::XpressProblem::makeConstraintCreator()
    -> std::shared_ptr<xpress::XPRSProblem::ConstraintCreator> {
  return makeConstraintCreator(true);
}

auto xpress::objects::XpressProblem::makeConstraintCreator(bool hasNames)
    -> std::shared_ptr<xpress::XPRSProblem::ConstraintCreator> {
  if (isOriginal())
    return std::make_shared<ConstraintCreator>(this, hasNames, false);
  else
    return std::make_shared<ConstraintCreator>(this, hasNames, true);
}

template <typename I1, typename I2, typename I1IsIntegral,
          typename I2IsIntegral, typename Func0, typename C,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    I1 count1, I2 count2, Func0 makeConstraint) -> void {
  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator();
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    for (I1 k1 = 0; k1 < count1; ++k1)
      for (I2 k2 = 0; k2 < count2; ++k2) {
        makeConstraint(k1, k2).create(creator.get());
      }
    creator->commit();
    undo = false;
  }
}

template <typename Iter0, typename Iter1, typename K1, typename K2,
          typename Func0, typename C, typename Iter0IsIterable,
          typename Iter1IsIterable, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    Iter0 const &iterable1, Iter1 const &iterable2,
    Func0 makeConstraint) -> void {
  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator();
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    for (auto &k1 : iterable1)
      for (auto &k2 : iterable2) {
        makeConstraint(k1, k2).create(creator.get());
      }
    creator->commit();
    undo = false;
  }
}

template <typename I1, typename I2, typename I3, typename I1IsIntegral,
          typename I2IsIntegral, typename I3IsIntegral, typename Func0,
          typename C, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    I1 count1, I2 count2, I3 count3, Func0 makeConstraint) -> void {
  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator();
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    for (I1 k1 = 0; k1 < count1; ++k1)
      for (I2 k2 = 0; k2 < count2; ++k2)
        for (I3 k3 = 0; k3 < count3; ++k3) {
          makeConstraint(k1, k2, k3).create(creator.get());
        }
    creator->commit();
    undo = false;
  }
}

template <typename Iter0, typename Iter1, typename Iter2, typename K1,
          typename K2, typename K3, typename Func0, typename C,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
    Func0 makeConstraint) -> void {
  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator();
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    for (auto &k1 : iterable1)
      for (auto &k2 : iterable2)
        for (auto &k3 : iterable3) {
          makeConstraint(k1, k2, k3).create(creator.get());
        }
    creator->commit();
    undo = false;
  }
}

template <typename I1, typename I2, typename I3, typename I4,
          typename I1IsIntegral, typename I2IsIntegral, typename I3IsIntegral,
          typename I4IsIntegral, typename Func0, typename C,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    I1 count1, I2 count2, I3 count3, I4 count4, Func0 makeConstraint) -> void {
  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator();
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    for (I1 k1 = 0; k1 < count1; ++k1)
      for (I2 k2 = 0; k2 < count2; ++k2)
        for (I3 k3 = 0; k3 < count3; ++k3)
          for (I4 k4 = 0; k4 < count4; ++k4) {
            makeConstraint(k1, k2, k3, k4).create(creator.get());
          }
    creator->commit();
    undo = false;
  }
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename K1, typename K2, typename K3, typename K4, typename Func0,
          typename C, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
    Iter3 const &iterable4, Func0 makeConstraint) -> void {
  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator();
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    for (auto &k1 : iterable1)
      for (auto &k2 : iterable2)
        for (auto &k3 : iterable3)
          for (auto &k4 : iterable4) {
            makeConstraint(k1, k2, k3, k4).create(creator.get());
          }
    creator->commit();
    undo = false;
  }
}

template <typename I1, typename I2, typename I3, typename I4, typename I5,
          typename I1IsIntegral, typename I2IsIntegral, typename I3IsIntegral,
          typename I4IsIntegral, typename I5IsIntegral, typename Func0,
          typename C, typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    I1 count1, I2 count2, I3 count3, I4 count4, I5 count5,
    Func0 makeConstraint) -> void {
  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator();
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    for (I1 k1 = 0; k1 < count1; ++k1)
      for (I2 k2 = 0; k2 < count2; ++k2)
        for (I3 k3 = 0; k3 < count3; ++k3)
          for (I4 k4 = 0; k4 < count4; ++k4)
            for (I5 k5 = 0; k5 < count5; ++k5) {
              makeConstraint(k1, k2, k3, k4, k5).create(creator.get());
            }
    creator->commit();
    undo = false;
  }
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename K1, typename K2, typename K3, typename K4,
          typename K5, typename Func0, typename C, typename Iter0IsIterable,
          typename Iter1IsIterable, typename Iter2IsIterable,
          typename Iter3IsIterable, typename Iter4IsIterable,
          typename Func0IsInvocable>
auto xpress::objects::XpressProblem::addConstraints(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
    Iter3 const &iterable4, Iter4 const &iterable5,
    Func0 makeConstraint) -> void {
  std::shared_ptr<ConstraintCreator> creator = makeConstraintCreator();
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator->undo();
    });

    for (auto &k1 : iterable1)
      for (auto &k2 : iterable2)
        for (auto &k3 : iterable3)
          for (auto &k4 : iterable4)
            for (auto &k5 : iterable5) {
              makeConstraint(k1, k2, k3, k4, k5).create(creator.get());
            }
    creator->commit();
    undo = false;
  }
}

auto xpress::objects::XpressProblem::drop() -> void {
  if (getTopLevel() != this)
    throw CannotPerformOperationException(
        "cannot do this with a presolved or callback model");
  variableHandler->drop();
  rowHandler->drop();
  pwlHandler->drop();
  generalConstraintHandler->drop();
  sosHandler->drop();
}

auto xpress::objects::XpressProblem::mapIsolation(char isolation)
    -> xpress::objects::IIS::Isolation {
  if (isolation == (char)0)
    return IIS::Isolation::NotIsolated;
  else if (isolation == (char)1)
    return IIS::Isolation::Isolated;

  else if (static_cast<signed char>(isolation) == -1)
    return IIS::Isolation::NotAvailable;
  else
    throw std::invalid_argument("invalid isolation status");
}

auto xpress::objects::XpressProblem::getIIS(int iis) -> xpress::objects::IIS {
  std::vector<IISConstraint> constraints;
  std::vector<IISVariable> variables;
  int nrows;
  int ncols;
  XPRSProblem::getIISData(iis, &nrows, &ncols, nullptr, nullptr, nullptr,
                          nullptr, nullptr, nullptr, nullptr, nullptr);
  std::vector<int> rowind(nrows);
  std::vector<char> rowtype(nrows);
  std::vector<double> duals(nrows);
  std::vector<char> isolationrows(nrows);
  std::vector<int> colind(ncols);
  std::vector<char> bndtype(ncols);
  std::vector<double> djs(ncols);
  std::vector<char> isolationcols(ncols);

  XPRSProblem::getIISData(iis, &nrows, &ncols, rowind, colind, rowtype, bndtype,
                          duals, djs, isolationrows, isolationcols);
  for (int i = 0; i < nrows; ++i) {
    IIS::Isolation isolation = mapIsolation(isolationrows[i]);
    switch (rowtype[i]) {
    case 'L':
      constraints.push_back(IISConstraint(inequalityForIndex(rowind[i]),
                                          RowType::LEQ, duals[i], isolation));
      break;
    case 'G':
      constraints.push_back(IISConstraint(inequalityForIndex(rowind[i]),
                                          RowType::GEQ, duals[i], isolation));
      break;
    case 'E':
      constraints.push_back(IISConstraint(inequalityForIndex(rowind[i]),
                                          RowType::EQ, duals[i], isolation));
      break;
    case '1':
      constraints.push_back(IISConstraint(sosForIndex(rowind[i]), std::nullopt,
                                          duals[i], isolation));
      break;
    case '2':
      constraints.push_back(IISConstraint(sosForIndex(rowind[i]), std::nullopt,
                                          duals[i], isolation));
      break;
    case 'W':
      constraints.push_back(IISConstraint(pwlForIndex(rowind[i]), std::nullopt,
                                          duals[i], isolation));
      break;
    case 'X':
      constraints.push_back(IISConstraint(generalConstraintForIndex(rowind[i]),
                                          std::nullopt, duals[i], isolation));
      break;
    case 'I':

      constraints.push_back(IISConstraint(inequalityForIndex(rowind[i]),
                                          std::nullopt, duals[i], isolation));
      break;
    default:
      throw std::invalid_argument("unknown constraint type in IIS data");
    }
  }

  for (int j = 0; j < ncols; ++j) {
    IISVariable::InfeasibleDomain property;
    switch (bndtype[j]) {
    case 'U':
      property = IISVariable::InfeasibleDomain::UpperBound;
      break;
    case 'L':
      property = IISVariable::InfeasibleDomain::LowerBound;
      break;
    case 'F':
      property = IISVariable::InfeasibleDomain::Fixing;
      break;
    case 'B':
      property = IISVariable::InfeasibleDomain::Binary;
      break;
    case 'I':
      property = IISVariable::InfeasibleDomain::Integral;
      break;
    case 'P':
      property = IISVariable::InfeasibleDomain::PartialInteger;
      break;
    case 'S':
      property = IISVariable::InfeasibleDomain::SemiContinuous;
      break;
    case 'R':
      property = IISVariable::InfeasibleDomain::SemiInteger;
      break;
    default:
      throw std::invalid_argument("unknown bound type in IIS data");
    }
    variables.push_back(IISVariable(property, variableForIndex(colind[j]),
                                    djs[j], mapIsolation(isolationcols[j])));
  }

  return IIS(constraints, variables);
}

template <typename M>
auto xpress::objects::XpressProblem::convertPreferences(int count, M prefs)
    -> std::vector<double> {
  std::vector<double> ret(count, 0.0);
  for (auto &e : prefs) {
    ret[e.first.getIndex()] = e.second;
  }
  return ret;
}

template <typename IMap, typename VMap>
auto xpress::objects::XpressProblem::repairWeightedInfeas(
    VMap const &lepref, VMap const &gepref, IMap const &lbpref,
    IMap const &ubpref, char phase2, double delta,
    std::optional<std::string> flags) -> int {
  if ((lepref.size() > 0) || (gepref.size() > 0))
    checkOperation("run repairWeightedInfeas", "Inequality");
  if ((lbpref.size() > 0) || (ubpref.size() > 0))
    checkOperation("run repairWeightedInfeas", "Variable");

  std::vector<double> le = convertPreferences(attributes.getRows(), lepref);
  std::vector<double> ge = convertPreferences(attributes.getRows(), gepref);
  std::vector<double> lb = convertPreferences(attributes.getCols(), lbpref);
  std::vector<double> ub = convertPreferences(attributes.getCols(), ubpref);

  return XPRSProblem::repairWeightedInfeas(le, ge, lb, ub, phase2, delta,
                                           flags);
}
