/* *********************************************************************** *
 *                                                                         *
 * xpress.cpp                                                              *
 *                                                                         *
 * C++ API for Xpress-Optimizer                                            *
 *                                                                         *
 * (c) Copyright Fair Isaac Corporation 2024-2025. All rights reserved     *
 * For FICO Xpress Optimizer v45.01.02                                     *
 *                                                                         *
 * *********************************************************************** */

#ifndef XPRESS_CXX_H
#define XPRESS_DO_NOT_INCLUDE_CXX_SOURCE
#include "xpress.hpp"
#endif

#include "xpress_maps.hpp"
#include "xpress_objects.hpp"

;

template <typename T>
xpress::ExtendableArrayImplementation<T>::ExtendableArrayImplementation(
    xpress::SizedArray<T const> const &initial) {
  data.insert(data.end(), initial.begin(), initial.end());
  len = 0;
}

template <typename T>
auto xpress::ExtendableArrayImplementation<T>::append(T value) -> void {
  if (len >= xpress::toInt(data.size())) {

    int newcap = std::max(xpress::toInt(data.size()) * 2,
                          std::max(xpress::toInt(data.size()) + 1, 8));
    data.resize(newcap);
  }
  data[len++] = value;
}

template <typename T>
auto xpress::ExtendableArrayImplementation<T>::append(
    xpress::SizedArray<T const> const &values) -> void {
  int alen = xpress::toInt(values.size());
  if (alen == 0)
    return;
  if (len + alen >= xpress::toInt(data.size())) {
    int newcap = std::max(xpress::toInt(data.size()) * 2,
                          std::max(xpress::toInt(data.size()) + alen, 8));
    data.resize(newcap);
  }
  xpress::arrayCopy(values, 0, data, len, alen);
  len += alen;
}

template <typename T>
auto xpress::ExtendableArrayImplementation<T>::getLength() -> int {
  return len;
}

template <typename T>
auto xpress::ExtendableArrayImplementation<T>::getCapacity() -> int {
  return xpress::toInt(data.size());
}

template <typename T>
auto xpress::ExtendableArrayImplementation<T>::getData() -> std::vector<T> & {
  return data;
}

template <typename T>
auto xpress::ExtendableArrayImplementation<T>::reset() -> void {
  len = 0;
}

template <typename T>
auto xpress::ExtendableArrayImplementation<T>::setLength(int newLength)
    -> void {
  if (newLength < len) {
    len = std::max(newLength, 0);
  } else if (newLength > xpress::toInt(data.size())) {
    data.resize(newLength);
    len = newLength;
  } else {
    len = newLength;
  }
}

;

xpress::ExtendableDoubleArrayImplementation::
    ExtendableDoubleArrayImplementation(
        xpress::SizedArray<double const> const &initial) {
  data.insert(data.end(), initial.begin(), initial.end());
  len = 0;
}

auto xpress::ExtendableDoubleArrayImplementation::append(double value) -> void {
  if (len >= xpress::toInt(data.size())) {

    int newcap = std::max(xpress::toInt(data.size()) * 2,
                          std::max(xpress::toInt(data.size()) + 1, 8));
    data.resize(newcap);
  }
  data[len++] = value;
}

auto xpress::ExtendableDoubleArrayImplementation::append(
    xpress::SizedArray<double const> const &values) -> void {
  int alen = xpress::toInt(values.size());
  if (alen == 0)
    return;
  if (len + alen >= xpress::toInt(data.size())) {
    int newcap = std::max(xpress::toInt(data.size()) * 2,
                          std::max(xpress::toInt(data.size()) + alen, 8));
    data.resize(newcap);
  }
  xpress::arrayCopy(values, 0, data, len, alen);
  len += alen;
}

auto xpress::ExtendableDoubleArrayImplementation::getLength() -> int {
  return len;
}

auto xpress::ExtendableDoubleArrayImplementation::getCapacity() -> int {
  return xpress::toInt(data.size());
}

auto xpress::ExtendableDoubleArrayImplementation::getData()
    -> std::vector<double> & {
  return data;
}

auto xpress::ExtendableDoubleArrayImplementation::reset() -> void { len = 0; }

auto xpress::ExtendableDoubleArrayImplementation::setLength(int newLength)
    -> void {
  if (newLength < len) {
    len = std::max(newLength, 0);
  } else if (newLength > xpress::toInt(data.size())) {
    data.resize(newLength);
    len = newLength;
  } else {
    len = newLength;
  }
}

;

xpress::ExtendableIntArrayImplementation::ExtendableIntArrayImplementation(
    xpress::SizedArray<int const> const &initial) {
  data.insert(data.end(), initial.begin(), initial.end());
  len = 0;
}

auto xpress::ExtendableIntArrayImplementation::append(int value) -> void {
  if (len >= xpress::toInt(data.size())) {

    int newcap = std::max(xpress::toInt(data.size()) * 2,
                          std::max(xpress::toInt(data.size()) + 1, 8));
    data.resize(newcap);
  }
  data[len++] = value;
}

auto xpress::ExtendableIntArrayImplementation::append(
    xpress::SizedArray<int const> const &values) -> void {
  int alen = xpress::toInt(values.size());
  if (alen == 0)
    return;
  if (len + alen >= xpress::toInt(data.size())) {
    int newcap = std::max(xpress::toInt(data.size()) * 2,
                          std::max(xpress::toInt(data.size()) + alen, 8));
    data.resize(newcap);
  }
  xpress::arrayCopy(values, 0, data, len, alen);
  len += alen;
}

auto xpress::ExtendableIntArrayImplementation::getLength() -> int {
  return len;
}

auto xpress::ExtendableIntArrayImplementation::getCapacity() -> int {
  return xpress::toInt(data.size());
}

auto xpress::ExtendableIntArrayImplementation::getData() -> std::vector<int> & {
  return data;
}

auto xpress::ExtendableIntArrayImplementation::reset() -> void { len = 0; }

auto xpress::ExtendableIntArrayImplementation::setLength(int newLength)
    -> void {
  if (newLength < len) {
    len = std::max(newLength, 0);
  } else if (newLength > xpress::toInt(data.size())) {
    data.resize(newLength);
    len = newLength;
  } else {
    len = newLength;
  }
}

;

xpress::ExtendableTypeArrayImplementation::ExtendableTypeArrayImplementation(
    xpress::SizedArray<char const> const &initial) {
  data.insert(data.end(), initial.begin(), initial.end());
  len = 0;
}

auto xpress::ExtendableTypeArrayImplementation::append(char value) -> void {
  if (len >= xpress::toInt(data.size())) {

    int newcap = std::max(xpress::toInt(data.size()) * 2,
                          std::max(xpress::toInt(data.size()) + 1, 8));
    data.resize(newcap);
  }
  data[len++] = value;
}

auto xpress::ExtendableTypeArrayImplementation::append(
    xpress::SizedArray<char const> const &values) -> void {
  int alen = xpress::toInt(values.size());
  if (alen == 0)
    return;
  if (len + alen >= xpress::toInt(data.size())) {
    int newcap = std::max(xpress::toInt(data.size()) * 2,
                          std::max(xpress::toInt(data.size()) + alen, 8));
    data.resize(newcap);
  }
  xpress::arrayCopy(values, 0, data, len, alen);
  len += alen;
}

auto xpress::ExtendableTypeArrayImplementation::getLength() -> int {
  return len;
}

auto xpress::ExtendableTypeArrayImplementation::getCapacity() -> int {
  return xpress::toInt(data.size());
}

auto xpress::ExtendableTypeArrayImplementation::getData()
    -> std::vector<char> & {
  return data;
}

auto xpress::ExtendableTypeArrayImplementation::reset() -> void { len = 0; }

auto xpress::ExtendableTypeArrayImplementation::setLength(int newLength)
    -> void {
  if (newLength < len) {
    len = std::max(newLength, 0);
  } else if (newLength > xpress::toInt(data.size())) {
    data.resize(newLength);
    len = newLength;
  } else {
    len = newLength;
  }
}

;

;
xpress::IndicatorInfo::IndicatorInfo(int indicator, bool complemented, int row)
    : indicator(indicator), complemented(complemented), row(row) {}

xpress::PwlBreakpoint::PwlBreakpoint() : PwlBreakpoint(0.0, 0.0) {}

xpress::PwlBreakpoint::PwlBreakpoint(double x, double y) : x(x), y(y) {}

auto xpress::PwlBreakpoint::equals(PwlBreakpoint const &other) const -> bool {
  return getX() == other.getX() && getY() == other.getY();
}
auto xpress::PwlBreakpoint::compareTo(PwlBreakpoint const &other) const -> int {
  if (getX() < other.getX())
    return -1;
  else if (getX() > other.getX())
    return 1;
  else if (getY() < other.getY())
    return -1;
  else if (getY() > other.getY())
    return 1;
  else
    return 0;
}

auto xpress::PwlBreakpoint::operator==(PwlBreakpoint const &other) const
    -> bool {
  return getX() == other.getX() && getY() == other.getY();
}
auto xpress::PwlBreakpoint::operator!=(PwlBreakpoint const &other) const
    -> bool {
  return getX() == other.getX() && getY() == other.getY();
}

auto xpress::PwlBreakpoint::getHashCode() const -> std::size_t {
  return xpress::combineHash(getX(), getY());
}

auto xpress::PwlBreakpoint::convert(
    std::vector<xpress::PwlBreakpoint> breakpoints)
    -> std::vector<std::vector<double>> {
  std::vector<std::vector<double>> ret(
      {std::vector<double>(xpress::toInt(breakpoints.size())),
       std::vector<double>(xpress::toInt(breakpoints.size()))});
  int i = 0;
  for (auto &p : breakpoints) {
    ret[0][i] = p.getX();
    ret[1][i] = p.getY();
    ++i;
  }
  return ret;
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::PwlBreakpoint::convert(Strm0 const &breakpoints, int sizeHint)
    -> std::vector<std::vector<double>> {
  if (sizeHint >= 0) {

    std::vector<std::vector<double>> ret(
        {std::vector<double>(sizeHint), std::vector<double>(sizeHint)});
    int i = 0;
    for (auto &p : breakpoints) {
      ret[0][i] = p.getX();
      ret[1][i] = p.getY();
      ++i;
    }
    return ret;
  } else {

    std::vector<double> x;
    std::vector<double> y;
    for (auto &p : breakpoints) {
      x.push_back(p.getX());
      y.push_back(p.getY());
    }
    return std::vector<std::vector<double>>({x, y});
  }
}

template <typename Strm0, typename Strm0IsStream>
auto xpress::PwlBreakpoint::convert(Strm0 const &breakpoints)
    -> std::vector<std::vector<double>> {
  return convert(breakpoints, xpress::sizeHint(breakpoints));
}

auto xpress::PwlBreakpoint::convert(
    xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval)
    -> std::vector<xpress::PwlBreakpoint> {
  if (xpress::toInt(xval.size()) != xpress::toInt(yval.size()))
    throw std::invalid_argument("xval and yval have different length");
  std::vector<PwlBreakpoint> ret(xpress::toInt(xval.size()));
  for (int i = 0; i < xpress::toInt(xval.size()); ++i)
    ret[i] = PwlBreakpoint(xval[i], yval[i]);
  return ret;
}

;
;
;

template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::ArrayBuilderImplementation(C dim)
    : dims(std::vector<int>({xpress::toInt(dim)})), lb(nullptr), ub(nullptr),
      limit(nullptr), name(nullptr), type(nullptr) {}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::getDim(int dim) const -> int {
  return dims[dim];
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::getLB() const -> std::function<double(C)> const & {
  return lb;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::getUB() const -> std::function<double(C)> const & {
  return ub;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::getLimit() const -> std::function<double(C)> const & {
  return limit;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<C,
                                                         CIsIntegral>::getName()
    const -> std::function<std::optional<std::string>(C)> const & {
  return name;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<C,
                                                         CIsIntegral>::getType()
    const -> std::function<xpress::ColumnType(C)> const & {
  return type;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  lb = newLb;
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  lb = [=](C) { return newLb; };
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  ub = newUb;
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  ub = [=](C) { return newUb; };
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  limit = newLimit;
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  limit = [=](C) { return newLimit; };
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  name = newName;
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](C k1) { return xpress::format(newName.value(), k1); };
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  type = newType;
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<
    C, CIsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> & {
  type = [=](C) { return newType; };
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral>::build(
    XPRSProblem *prob) -> std::vector<int> {
  return prob->buildColumns(this);
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral>::build(
    xpress::objects::XpressProblem *prob)
    -> std::vector<xpress::objects::Variable> {
  return prob->buildVariables(this);
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::array(C dim)
    -> xpress::VariableBuilder::ArrayBuilder<C> {
  return ArrayBuilder<C>(xpress::toInt(dim));
}

;
;
;

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::Array2BuilderImplementation(C1 dim1,
                                                                     C2 dim2)
    : dims(std::vector<int>({xpress::toInt(dim1), xpress::toInt(dim2)})),
      lb(nullptr), ub(nullptr), limit(nullptr), name(nullptr), type(nullptr) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::getDim(int dim) const -> int {
  return dims[dim];
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<C1, C2, C1IsIntegral,
                                                          C2IsIntegral>::getLB()
    const -> std::function<double(C1, C2)> const & {
  return lb;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<C1, C2, C1IsIntegral,
                                                          C2IsIntegral>::getUB()
    const -> std::function<double(C1, C2)> const & {
  return ub;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::getLimit() const
    -> std::function<double(C1, C2)> const & {
  return limit;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C1, C2)> const & {
  return name;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C1, C2)> const & {
  return type;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  lb = newLb;
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  lb = [=](C1, C2) { return newLb; };
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  ub = newUb;
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  ub = [=](C1, C2) { return newUb; };
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  limit = newLimit;
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  limit = [=](C1, C2) { return newLimit; };
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  name = newName;
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(std::optional<std::string>
                                                      newName)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](C1 k1, C2 k2) {
      return xpress::format(newName.value(), k1, k2);
    };
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  type = newType;
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Array2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  type = [=](C1, C2) { return newType; };
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::build(XPRSProblem *prob)
    -> std::vector<std::vector<int>> {
  return prob->buildColumns(this);
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::build(xpress::objects::XpressProblem
                                                   *prob)
    -> std::vector<std::vector<xpress::objects::Variable>> {
  return prob->buildVariables(this);
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::array2(C1 dim1, C2 dim2)
    -> xpress::VariableBuilder::Array2Builder<C1, C2> {
  return Array2Builder<C1, C2>(xpress::toInt(dim1), xpress::toInt(dim2));
}

;
;
;

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::Array3BuilderImplementation(C1 dim1, C2 dim2, C3 dim3)
    : dims(std::vector<int>(
          {xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3)})),
      lb(nullptr), ub(nullptr), limit(nullptr), name(nullptr), type(nullptr) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getDim(int dim) const
    -> int {
  return dims[dim];
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getLB() const
    -> std::function<double(C1, C2, C3)> const & {
  return lb;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getUB() const
    -> std::function<double(C1, C2, C3)> const & {
  return ub;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getLimit() const
    -> std::function<double(C1, C2, C3)> const & {
  return limit;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C1, C2, C3)> const & {
  return name;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C1, C2, C3)> const & {
  return type;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  lb = newLb;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  lb = [=](C1, C2, C3) { return newLb; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  ub = newUb;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  ub = [=](C1, C2, C3) { return newUb; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  limit = newLimit;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  limit = [=](C1, C2, C3) { return newLimit; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withName(Func0
                                                                        newName)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  name = newName;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](C1 k1, C2 k2, C3 k3) {
      return xpress::format(newName.value(), k1, k2, k3);
    };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(Func0
                                                                        newType)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  type = newType;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(ColumnType
                                                                        newType)
    -> xpress::VariableBuilder::Array3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  type = [=](C1, C2, C3) { return newType; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::build(XPRSProblem
                                                                     *prob)
    -> std::vector<std::vector<std::vector<int>>> {
  return prob->buildColumns(this);
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::build(xpress::objects::XpressProblem *prob)
    -> std::vector<std::vector<std::vector<xpress::objects::Variable>>> {
  return prob->buildVariables(this);
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::array3(C1 dim1, C2 dim2, C3 dim3)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3> {
  return Array3Builder<C1, C2, C3>(xpress::toInt(dim1), xpress::toInt(dim2),
                                   xpress::toInt(dim3));
}

;
;
;

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::Array4BuilderImplementation(C1 dim1, C2 dim2, C3 dim3,
                                               C4 dim4)
    : dims(std::vector<int>({xpress::toInt(dim1), xpress::toInt(dim2),
                             xpress::toInt(dim3), xpress::toInt(dim4)})),
      lb(nullptr), ub(nullptr), limit(nullptr), name(nullptr), type(nullptr) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::getDim(int dim) const -> int {
  return dims[dim];
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::getLB() const
    -> std::function<double(C1, C2, C3, C4)> const & {
  return lb;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::getUB() const
    -> std::function<double(C1, C2, C3, C4)> const & {
  return ub;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::getLimit() const
    -> std::function<double(C1, C2, C3, C4)> const & {
  return limit;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C1, C2, C3, C4)> const & {
  return name;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C1, C2, C3, C4)> const & {
  return type;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  lb = newLb;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  lb = [=](C1, C2, C3, C4) { return newLb; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  ub = newUb;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  ub = [=](C1, C2, C3, C4) { return newUb; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  limit = newLimit;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  limit = [=](C1, C2, C3, C4) { return newLimit; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  name = newName;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](C1 k1, C2 k2, C3 k3, C4 k4) {
      return xpress::format(newName.value(), k1, k2, k3, k4);
    };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  type = newType;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  type = [=](C1, C2, C3, C4) { return newType; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::build(XPRSProblem *prob)
    -> std::vector<std::vector<std::vector<std::vector<int>>>> {
  return prob->buildColumns(this);
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::build(xpress::objects::XpressProblem *prob)
    -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>> {
  return prob->buildVariables(this);
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::array4(C1 dim1, C2 dim2, C3 dim3, C4 dim4)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4> {
  return Array4Builder<C1, C2, C3, C4>(xpress::toInt(dim1), xpress::toInt(dim2),
                                       xpress::toInt(dim3),
                                       xpress::toInt(dim4));
}

;
;
;

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::Array5BuilderImplementation(C1 dim1, C2 dim2, C3 dim3,
                                               C4 dim4, C5 dim5)
    : dims(std::vector<int>({xpress::toInt(dim1), xpress::toInt(dim2),
                             xpress::toInt(dim3), xpress::toInt(dim4),
                             xpress::toInt(dim5)})),
      lb(nullptr), ub(nullptr), limit(nullptr), name(nullptr), type(nullptr) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getDim(int dim) const -> int {
  return dims[dim];
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getLB() const
    -> std::function<double(C1, C2, C3, C4, C5)> const & {
  return lb;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getUB() const
    -> std::function<double(C1, C2, C3, C4, C5)> const & {
  return ub;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getLimit() const
    -> std::function<double(C1, C2, C3, C4, C5)> const & {
  return limit;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C1, C2, C3, C4, C5)> const & {
  return name;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C1, C2, C3, C4, C5)> const & {
  return type;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  lb = newLb;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  lb = [=](C1, C2, C3, C4, C5) { return newLb; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  ub = newUb;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  ub = [=](C1, C2, C3, C4, C5) { return newUb; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  limit = newLimit;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  limit = [=](C1, C2, C3, C4, C5) { return newLimit; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  name = newName;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](C1 k1, C2 k2, C3 k3, C4 k4, C5 k5) {
      return xpress::format(newName.value(), k1, k2, k3, k4, k5);
    };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  type = newType;
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  type = [=](C1, C2, C3, C4, C5) { return newType; };
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::build(XPRSProblem *prob)
    -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>> {
  return prob->buildColumns(this);
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::build(xpress::objects::XpressProblem *prob)
    -> std::vector<std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>> {
  return prob->buildVariables(this);
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::array5(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                     C5 dim5)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> {
  return Array5Builder<C1, C2, C3, C4, C5>(
      xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
      xpress::toInt(dim4), xpress::toInt(dim5));
}

;

template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::ColumnArrayBuilderImplementation(C dim,
                                                      xpress::XPRSProblem *prob)
    : wrapped(ArrayBuilder<C>(xpress::toInt(dim))), prob(prob) {}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::toArray() -> std::vector<int> {
  return prob->buildColumns(wrapped);
}

template <typename C, typename CIsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilderImplementation<
    C, CIsIntegral>::toArray(Func0 makeResult,
                             std::function<void(I, C, int)> addResult) -> I {
  return prob->buildColumns(wrapped, makeResult, addResult);
}

;

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral,
    C2IsIntegral>::ColumnArray2BuilderImplementation(C1 dim1, C2 dim2,
                                                     xpress::XPRSProblem *prob)
    : wrapped(Array2Builder<C1, C2>(xpress::toInt(dim1), xpress::toInt(dim2))),
      prob(prob) {}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(std::optional<std::string>
                                                      newName)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::toArray()
    -> std::vector<std::vector<int>> {
  return prob->buildColumns(wrapped);
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2BuilderImplementation<
    C1, C2, C1IsIntegral,
    C2IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, int)> addResult) -> I {
  return prob->buildColumns(wrapped, makeResult, addResult);
}

;

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::ColumnArray3BuilderImplementation(C1 dim1, C2 dim2, C3 dim3,
                                                     xpress::XPRSProblem *prob)
    : wrapped(Array3Builder<C1, C2, C3>(
          xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3))),
      prob(prob) {}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withName(Func0
                                                                        newName)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(Func0
                                                                        newType)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(ColumnType
                                                                        newType)
    -> xpress::VariableBuilder::ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::toArray()
    -> std::vector<std::vector<std::vector<int>>> {
  return prob->buildColumns(wrapped);
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, C3, int)> addResult)
    -> I {
  return prob->buildColumns(wrapped, makeResult, addResult);
}

;

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::ColumnArray4BuilderImplementation(C1 dim1, C2 dim2, C3 dim3,
                                                     C4 dim4,
                                                     xpress::XPRSProblem *prob)
    : wrapped(Array4Builder<C1, C2, C3, C4>(
          xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
          xpress::toInt(dim4))),
      prob(prob) {}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::toArray()
    -> std::vector<std::vector<std::vector<std::vector<int>>>> {
  return prob->buildColumns(wrapped);
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, C3, C4, int)>
                               addResult) -> I {
  return prob->buildColumns(wrapped, makeResult, addResult);
}

;

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::ColumnArray5BuilderImplementation(C1 dim1, C2 dim2, C3 dim3,
                                                     C4 dim4, C5 dim5,
                                                     xpress::XPRSProblem *prob)
    : wrapped(Array5Builder<C1, C2, C3, C4, C5>(
          xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
          xpress::toInt(dim4), xpress::toInt(dim5))),
      prob(prob) {}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::toArray()
    -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>> {
  return prob->buildColumns(wrapped);
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, C3, C4, C5, int)>
                               addResult) -> I {
  return prob->buildColumns(wrapped, makeResult, addResult);
}

;
template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::getIterable1() const
    -> xpress::Iterable<K1> const & {
  return iterable1;
}

;
;
;

template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::getLB() const
    -> std::function<double(K1)> const & {
  return lb;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::getUB() const
    -> std::function<double(K1)> const & {
  return ub;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::getLimit() const
    -> std::function<double(K1)> const & {
  return limit;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::getName() const
    -> std::function<std::optional<std::string>(K1)> const & {
  return name;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::getType() const
    -> std::function<xpress::ColumnType(K1)> const & {
  return type;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::getFilter() const
    -> std::function<bool(K1)> const & {
  return filter;
}

template <typename K1>
template <typename Iter0, typename Iter0IsIterable>
xpress::VariableBuilder::MapBuilderImplementation<K1>::MapBuilderImplementation(
    Iter0 const &iterable1)
    : iterable1(xpress::Iterable<K1>::fromSTLRef(&iterable1)), lb(nullptr),
      ub(nullptr), limit(nullptr), name(nullptr), type(nullptr),
      filter(nullptr) {}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  lb = newLb;
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withLB(double newLb)
    -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  lb = [=](K1) { return newLb; };
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  ub = newUb;
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withUB(double newUb)
    -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  ub = [=](K1) { return newUb; };
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withLimit(
    Func0 newLimit) -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  limit = newLimit;
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  limit = [=](K1) { return newLimit; };
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withName(
    Func0 newName) -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  name = newName;
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](K1 arg1) { return xpress::format(newName.value(), arg1); };
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withType(
    Func0 newType) -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  type = newType;
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  type = [=](K1) { return newType; };
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::MapBuilderImplementation<K1> & {
  filter = newFilter;
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::build(
    XPRSProblem *prob) -> std::unordered_map<K1, int> {
  return prob->buildColumns(*this);
}

template <typename K1>
auto xpress::VariableBuilder::MapBuilderImplementation<K1>::build(
    xpress::objects::XpressProblem *prob)
    -> std::unordered_map<K1, xpress::objects::Variable> {
  return prob->buildVariables(*this);
}

template <typename Iter0, typename K1, typename Iter0IsIterable>
auto xpress::VariableBuilder::map(Iter0 const &iterable1)
    -> xpress::VariableBuilder::MapBuilder<K1> {
  return MapBuilder<K1>(iterable1);
}

template <typename K1>
auto xpress::VariableBuilder::map(xpress::SizedArray<K1 const> const &arr1)
    -> xpress::VariableBuilder::MapBuilder<K1> {
  return MapBuilder<K1>(arr1);
}

;
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::getIterable1()
    const -> xpress::Iterable<K1> const & {
  return iterable1;
};
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::getIterable2()
    const -> xpress::Iterable<K2> const & {
  return iterable2;
}

;
;
;

template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::getLB() const
    -> std::function<double(K1, K2)> const & {
  return lb;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::getUB() const
    -> std::function<double(K1, K2)> const & {
  return ub;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::getLimit()
    const -> std::function<double(K1, K2)> const & {
  return limit;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::getName() const
    -> std::function<std::optional<std::string>(K1, K2)> const & {
  return name;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::getType() const
    -> std::function<xpress::ColumnType(K1, K2)> const & {
  return type;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::getFilter()
    const -> std::function<bool(K1, K2)> const & {
  return filter;
}

template <typename K1, typename K2>
template <typename Iter0, typename Iter1, typename Iter0IsIterable,
          typename Iter1IsIterable>
xpress::VariableBuilder::Map2BuilderImplementation<
    K1, K2>::Map2BuilderImplementation(Iter0 const &iterable1,
                                       Iter1 const &iterable2)
    : iterable1(xpress::Iterable<K1>::fromSTLRef(&iterable1)),
      iterable2(xpress::Iterable<K2>::fromSTLRef(&iterable2)), lb(nullptr),
      ub(nullptr), limit(nullptr), name(nullptr), type(nullptr),
      filter(nullptr) {}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  lb = newLb;
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withLB(
    double newLb)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  lb = [=](K1, K2) { return newLb; };
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  ub = newUb;
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withUB(
    double newUb)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  ub = [=](K1, K2) { return newUb; };
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  limit = newLimit;
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  limit = [=](K1, K2) { return newLimit; };
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  name = newName;
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](K1 arg1, K2 arg2) {
      return xpress::format(newName.value(), arg1, arg2);
    };
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  type = newType;
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  type = [=](K1, K2) { return newType; };
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> & {
  filter = newFilter;
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::build(
    XPRSProblem *prob) -> xpress::maps::HashMap2<K1, K2, int> {
  return prob->buildColumns(*this);
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2BuilderImplementation<K1, K2>::build(
    xpress::objects::XpressProblem *prob)
    -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable> {
  return prob->buildVariables(*this);
}

template <typename Iter0, typename Iter1, typename K1, typename K2,
          typename Iter0IsIterable, typename Iter1IsIterable>
auto xpress::VariableBuilder::map2(Iter0 const &iterable1,
                                   Iter1 const &iterable2)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> {
  return Map2Builder<K1, K2>(iterable1, iterable2);
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::map2(xpress::SizedArray<K1 const> const &arr1,
                                   xpress::SizedArray<K2 const> const &arr2)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> {
  return Map2Builder<K1, K2>(arr1, arr2);
}

;
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<
    K1, K2, K3>::getIterable1() const -> xpress::Iterable<K1> const & {
  return iterable1;
};
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<
    K1, K2, K3>::getIterable2() const -> xpress::Iterable<K2> const & {
  return iterable2;
};
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<
    K1, K2, K3>::getIterable3() const -> xpress::Iterable<K3> const & {
  return iterable3;
}

;
;
;

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::getLB()
    const -> std::function<double(K1, K2, K3)> const & {
  return lb;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::getUB()
    const -> std::function<double(K1, K2, K3)> const & {
  return ub;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::getLimit()
    const -> std::function<double(K1, K2, K3)> const & {
  return limit;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::getName()
    const -> std::function<std::optional<std::string>(K1, K2, K3)> const & {
  return name;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::getType()
    const -> std::function<xpress::ColumnType(K1, K2, K3)> const & {
  return type;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::getFilter()
    const -> std::function<bool(K1, K2, K3)> const & {
  return filter;
}

template <typename K1, typename K2, typename K3>
template <typename Iter0, typename Iter1, typename Iter2,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable>
xpress::VariableBuilder::Map3BuilderImplementation<
    K1, K2, K3>::Map3BuilderImplementation(Iter0 const &iterable1,
                                           Iter1 const &iterable2,
                                           Iter2 const &iterable3)
    : iterable1(xpress::Iterable<K1>::fromSTLRef(&iterable1)),
      iterable2(xpress::Iterable<K2>::fromSTLRef(&iterable2)),
      iterable3(xpress::Iterable<K3>::fromSTLRef(&iterable3)), lb(nullptr),
      ub(nullptr), limit(nullptr), name(nullptr), type(nullptr),
      filter(nullptr) {}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  lb = newLb;
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withLB(
    double newLb)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  lb = [=](K1, K2, K3) { return newLb; };
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  ub = newUb;
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withUB(
    double newUb)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  ub = [=](K1, K2, K3) { return newUb; };
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  limit = newLimit;
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  limit = [=](K1, K2, K3) { return newLimit; };
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  name = newName;
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](K1 arg1, K2 arg2, K3 arg3) {
      return xpress::format(newName.value(), arg1, arg2, arg3);
    };
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  type = newType;
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  type = [=](K1, K2, K3) { return newType; };
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> & {
  filter = newFilter;
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::build(
    XPRSProblem *prob) -> xpress::maps::HashMap3<K1, K2, K3, int> {
  return prob->buildColumns(*this);
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3>::build(
    xpress::objects::XpressProblem *prob)
    -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable> {
  return prob->buildVariables(*this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename K1,
          typename K2, typename K3, typename Iter0IsIterable,
          typename Iter1IsIterable, typename Iter2IsIterable>
auto xpress::VariableBuilder::map3(Iter0 const &iterable1,
                                   Iter1 const &iterable2,
                                   Iter2 const &iterable3)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> {
  return Map3Builder<K1, K2, K3>(iterable1, iterable2, iterable3);
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::map3(xpress::SizedArray<K1 const> const &arr1,
                                   xpress::SizedArray<K2 const> const &arr2,
                                   xpress::SizedArray<K3 const> const &arr3)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> {
  return Map3Builder<K1, K2, K3>(arr1, arr2, arr3);
}

;
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::getIterable1() const -> xpress::Iterable<K1> const & {
  return iterable1;
};
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::getIterable2() const -> xpress::Iterable<K2> const & {
  return iterable2;
};
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::getIterable3() const -> xpress::Iterable<K3> const & {
  return iterable3;
};
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::getIterable4() const -> xpress::Iterable<K4> const & {
  return iterable4;
}

;
;
;

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4>::getLB()
    const -> std::function<double(K1, K2, K3, K4)> const & {
  return lb;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4>::getUB()
    const -> std::function<double(K1, K2, K3, K4)> const & {
  return ub;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3,
                                                        K4>::getLimit() const
    -> std::function<double(K1, K2, K3, K4)> const & {
  return limit;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3,
                                                        K4>::getName() const
    -> std::function<std::optional<std::string>(K1, K2, K3, K4)> const & {
  return name;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3,
                                                        K4>::getType() const
    -> std::function<xpress::ColumnType(K1, K2, K3, K4)> const & {
  return type;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3,
                                                        K4>::getFilter() const
    -> std::function<bool(K1, K2, K3, K4)> const & {
  return filter;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::Map4BuilderImplementation(Iter0 const &iterable1,
                                               Iter1 const &iterable2,
                                               Iter2 const &iterable3,
                                               Iter3 const &iterable4)
    : iterable1(xpress::Iterable<K1>::fromSTLRef(&iterable1)),
      iterable2(xpress::Iterable<K2>::fromSTLRef(&iterable2)),
      iterable3(xpress::Iterable<K3>::fromSTLRef(&iterable3)),
      iterable4(xpress::Iterable<K4>::fromSTLRef(&iterable4)), lb(nullptr),
      ub(nullptr), limit(nullptr), name(nullptr), type(nullptr),
      filter(nullptr) {}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  lb = newLb;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4>::withLB(
    double newLb)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  lb = [=](K1, K2, K3, K4) { return newLb; };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  ub = newUb;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4>::withUB(
    double newUb)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  ub = [=](K1, K2, K3, K4) { return newUb; };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  limit = newLimit;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  limit = [=](K1, K2, K3, K4) { return newLimit; };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::withName(Func0 newName)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  name = newName;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](K1 arg1, K2 arg2, K3 arg3, K4 arg4) {
      return xpress::format(newName.value(), arg1, arg2, arg3, arg4);
    };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::withType(Func0 newType)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  type = newType;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  type = [=](K1, K2, K3, K4) { return newType; };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4BuilderImplementation<
    K1, K2, K3, K4>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> & {
  filter = newFilter;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4>::build(
    XPRSProblem *prob) -> xpress::maps::HashMap4<K1, K2, K3, K4, int> {
  return prob->buildColumns(*this);
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4>::build(
    xpress::objects::XpressProblem *prob)
    -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable> {
  return prob->buildVariables(*this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename K1, typename K2, typename K3, typename K4,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
auto xpress::VariableBuilder::map4(Iter0 const &iterable1,
                                   Iter1 const &iterable2,
                                   Iter2 const &iterable3,
                                   Iter3 const &iterable4)
    -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> {
  return Map4Builder<K1, K2, K3, K4>(iterable1, iterable2, iterable3,
                                     iterable4);
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::map4(xpress::SizedArray<K1 const> const &arr1,
                                   xpress::SizedArray<K2 const> const &arr2,
                                   xpress::SizedArray<K3 const> const &arr3,
                                   xpress::SizedArray<K4 const> const &arr4)
    -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> {
  return Map4Builder<K1, K2, K3, K4>(arr1, arr2, arr3, arr4);
}

;
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::getIterable1() const -> xpress::Iterable<K1> const & {
  return iterable1;
};
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::getIterable2() const -> xpress::Iterable<K2> const & {
  return iterable2;
};
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::getIterable3() const -> xpress::Iterable<K3> const & {
  return iterable3;
};
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::getIterable4() const -> xpress::Iterable<K4> const & {
  return iterable4;
};
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::getIterable5() const -> xpress::Iterable<K5> const & {
  return iterable5;
}

;
;
;

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4,
                                                        K5>::getLB() const
    -> std::function<double(K1, K2, K3, K4, K5)> const & {
  return lb;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4,
                                                        K5>::getUB() const
    -> std::function<double(K1, K2, K3, K4, K5)> const & {
  return ub;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4,
                                                        K5>::getLimit() const
    -> std::function<double(K1, K2, K3, K4, K5)> const & {
  return limit;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4,
                                                        K5>::getName() const
    -> std::function<std::optional<std::string>(K1, K2, K3, K4, K5)> const & {
  return name;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4,
                                                        K5>::getType() const
    -> std::function<xpress::ColumnType(K1, K2, K3, K4, K5)> const & {
  return type;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4,
                                                        K5>::getFilter() const
    -> std::function<bool(K1, K2, K3, K4, K5)> const & {
  return filter;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::Map5BuilderImplementation(Iter0 const &iterable1,
                                                   Iter1 const &iterable2,
                                                   Iter2 const &iterable3,
                                                   Iter3 const &iterable4,
                                                   Iter4 const &iterable5)
    : iterable1(xpress::Iterable<K1>::fromSTLRef(&iterable1)),
      iterable2(xpress::Iterable<K2>::fromSTLRef(&iterable2)),
      iterable3(xpress::Iterable<K3>::fromSTLRef(&iterable3)),
      iterable4(xpress::Iterable<K4>::fromSTLRef(&iterable4)),
      iterable5(xpress::Iterable<K5>::fromSTLRef(&iterable5)), lb(nullptr),
      ub(nullptr), limit(nullptr), name(nullptr), type(nullptr),
      filter(nullptr) {}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4,
                                                        K5>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  lb = newLb;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLB(double newLb)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  lb = [=](K1, K2, K3, K4, K5) { return newLb; };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4,
                                                        K5>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  ub = newUb;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withUB(double newUb)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  ub = [=](K1, K2, K3, K4, K5) { return newUb; };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  limit = newLimit;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  limit = [=](K1, K2, K3, K4, K5) { return newLimit; };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withName(Func0 newName)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  name = newName;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  if (!(newName))
    name = nullptr;
  else
    name = [=](K1 arg1, K2 arg2, K3 arg3, K4 arg4, K5 arg5) {
      return xpress::format(newName.value(), arg1, arg2, arg3, arg4, arg5);
    };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withType(Func0 newType)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  type = newType;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  type = [=](K1, K2, K3, K4, K5) { return newType; };
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5>
        & {
  filter = newFilter;
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::build(XPRSProblem *prob)
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int> {
  return prob->buildColumns(*this);
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5BuilderImplementation<
    K1, K2, K3, K4, K5>::build(xpress::objects::XpressProblem *prob)
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, xpress::objects::Variable> {
  return prob->buildVariables(*this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename K1, typename K2, typename K3, typename K4,
          typename K5, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
auto xpress::VariableBuilder::map5(Iter0 const &iterable1,
                                   Iter1 const &iterable2,
                                   Iter2 const &iterable3,
                                   Iter3 const &iterable4,
                                   Iter4 const &iterable5)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> {
  return Map5Builder<K1, K2, K3, K4, K5>(iterable1, iterable2, iterable3,
                                         iterable4, iterable5);
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::map5(xpress::SizedArray<K1 const> const &arr1,
                                   xpress::SizedArray<K2 const> const &arr2,
                                   xpress::SizedArray<K3 const> const &arr3,
                                   xpress::SizedArray<K4 const> const &arr4,
                                   xpress::SizedArray<K5 const> const &arr5)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> {
  return Map5Builder<K1, K2, K3, K4, K5>(arr1, arr2, arr3, arr4, arr5);
}

;

template <typename C, typename CIsIntegral>
xpress::VariableBuilder::VariableArrayBuilderImplementation<C, CIsIntegral>::
    VariableArrayBuilderImplementation(C dim,
                                       xpress::objects::XpressProblem *prob)
    : wrapped(ArrayBuilder<C>(xpress::toInt(dim))), prob(prob) {}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableArrayBuilderImplementation<
        C, CIsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilderImplementation<
    C, CIsIntegral>::toArray() -> std::vector<xpress::objects::Variable> {
  return prob->buildVariables(wrapped);
}

template <typename C, typename CIsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::
    VariableArrayBuilderImplementation<C, CIsIntegral>::toArray(
        Func0 makeResult,
        std::function<void(I, C, xpress::objects::Variable)> addResult) -> I {
  return prob->buildVariables(wrapped, makeResult, addResult);
}

;

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::
    VariableArray2BuilderImplementation(C1 dim1, C2 dim2,
                                        xpress::objects::XpressProblem *prob)
    : wrapped(Array2Builder<C1, C2>(xpress::toInt(dim1), xpress::toInt(dim2))),
      prob(prob) {}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(std::optional<std::string>
                                                      newName)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableArray2BuilderImplementation<
        C1, C2, C1IsIntegral, C2IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::toArray()
    -> std::vector<std::vector<xpress::objects::Variable>> {
  return prob->buildVariables(wrapped);
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral>::
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, xpress::objects::Variable)> addResult)
        -> I {
  return prob->buildVariables(wrapped, makeResult, addResult);
}

;

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::
    VariableArray3BuilderImplementation(C1 dim1, C2 dim2, C3 dim3,
                                        xpress::objects::XpressProblem *prob)
    : wrapped(Array3Builder<C1, C2, C3>(
          xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3))),
      prob(prob) {}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withName(Func0
                                                                        newName)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(Func0
                                                                        newType)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(ColumnType
                                                                        newType)
    -> xpress::VariableBuilder::VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::toArray()
    -> std::vector<std::vector<std::vector<xpress::objects::Variable>>> {
  return prob->buildVariables(wrapped);
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, C3, xpress::objects::Variable)>
                addResult) -> I {
  return prob->buildVariables(wrapped, makeResult, addResult);
}

;

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    VariableArray4BuilderImplementation(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                        xpress::objects::XpressProblem *prob)
    : wrapped(Array4Builder<C1, C2, C3, C4>(
          xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
          xpress::toInt(dim4))),
      prob(prob) {}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::toArray()
    -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>> {
  return prob->buildVariables(wrapped);
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, C3, C4, xpress::objects::Variable)>
                addResult) -> I {
  return prob->buildVariables(wrapped, makeResult, addResult);
}

;

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::
    VariableArray5BuilderImplementation(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                        C5 dim5,
                                        xpress::objects::XpressProblem *prob)
    : wrapped(Array5Builder<C1, C2, C3, C4, C5>(
          xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
          xpress::toInt(dim4), xpress::toInt(dim5))),
      prob(prob) {}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withName(newName);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  wrapped.withType(newType);
  return *this;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::toArray()
    -> std::vector<std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>> {
  return prob->buildVariables(wrapped);
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, C3, C4, C5,
                                              xpress::objects::Variable)>
                               addResult) -> I {
  return prob->buildVariables(wrapped, makeResult, addResult);
}

;

template <typename K1>
template <typename Iter0, typename Iter0IsIterable>
xpress::VariableBuilder::ColumnMapBuilderImplementation<
    K1>::ColumnMapBuilderImplementation(Iter0 const &iterable1,
                                        xpress::XPRSProblem *prob)
    : wrapped(MapBuilder<K1>(iterable1)), prob(prob) {}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withLB(
    double newLb)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withUB(
    double newUb)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::toMap()
    -> std::unordered_map<K1, int> {
  return prob->buildColumns(wrapped);
}

template <typename K1>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilderImplementation<K1>::toMap(
    Func0 makeResult, std::function<void(I &, K1, int)> addResult) -> I {
  return prob->buildColumns<I>(wrapped, makeResult, addResult);
}

;

template <typename K1, typename K2>
template <typename Iter0, typename Iter1, typename Iter0IsIterable,
          typename Iter1IsIterable>
xpress::VariableBuilder::ColumnMap2BuilderImplementation<
    K1, K2>::ColumnMap2BuilderImplementation(Iter0 const &iterable1,
                                             Iter1 const &iterable2,
                                             xpress::XPRSProblem *prob)
    : wrapped(Map2Builder<K1, K2>(iterable1, iterable2)), prob(prob) {}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::withLB(
    double newLb)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::withUB(
    double newUb)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<
    K1, K2>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<
    K1, K2>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<
    K1, K2>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::toMap()
    -> xpress::maps::HashMap2<K1, K2, int> {
  return prob->buildColumns(wrapped);
}

template <typename K1, typename K2>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2>::toMap(
    Func0 makeResult, std::function<void(I &, K1, K2, int)> addResult) -> I {
  return prob->buildColumns<I>(wrapped, makeResult, addResult);
}

;

template <typename K1, typename K2, typename K3>
template <typename Iter0, typename Iter1, typename Iter2,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable>
xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::ColumnMap3BuilderImplementation(Iter0 const &iterable1,
                                                 Iter1 const &iterable2,
                                                 Iter2 const &iterable3,
                                                 xpress::XPRSProblem *prob)
    : wrapped(Map3Builder<K1, K2, K3>(iterable1, iterable2, iterable3)),
      prob(prob) {}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::toMap() -> xpress::maps::HashMap3<K1, K2, K3, int> {
  return prob->buildColumns(wrapped);
}

template <typename K1, typename K2, typename K3>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3BuilderImplementation<
    K1, K2, K3>::toMap(Func0 makeResult,
                       std::function<void(I &, K1, K2, K3, int)> addResult)
    -> I {
  return prob->buildColumns<I>(wrapped, makeResult, addResult);
}

;

template <typename K1, typename K2, typename K3, typename K4>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::ColumnMap4BuilderImplementation(Iter0 const &iterable1,
                                                     Iter1 const &iterable2,
                                                     Iter2 const &iterable3,
                                                     Iter3 const &iterable4,
                                                     xpress::XPRSProblem *prob)
    : wrapped(Map4Builder<K1, K2, K3, K4>(iterable1, iterable2, iterable3,
                                          iterable4)),
      prob(prob) {}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>
        & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<
    K1, K2, K3, K4>::toMap() -> xpress::maps::HashMap4<K1, K2, K3, K4, int> {
  return prob->buildColumns(wrapped);
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4>::
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, K3, K4, int)> addResult) -> I {
  return prob->buildColumns<I>(wrapped, makeResult, addResult);
}

;

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5>::
    ColumnMap5BuilderImplementation(Iter0 const &iterable1,
                                    Iter1 const &iterable2,
                                    Iter2 const &iterable3,
                                    Iter3 const &iterable4,
                                    Iter4 const &iterable5,
                                    xpress::XPRSProblem *prob)
    : wrapped(Map5Builder<K1, K2, K3, K4, K5>(iterable1, iterable2, iterable3,
                                              iterable4, iterable5)),
      prob(prob) {}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5> & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                              K5>::toMap()
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int> {
  return prob->buildColumns(wrapped);
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::toMap(Func0 makeResult,
                               std::function<void(I &, K1, K2, K3, K4, K5, int)>
                                   addResult) -> I {
  return prob->buildColumns<I>(wrapped, makeResult, addResult);
}

;

template <typename K1>
template <typename Iter0, typename Iter0IsIterable>
xpress::VariableBuilder::VariableMapBuilderImplementation<
    K1>::VariableMapBuilderImplementation(Iter0 const &iterable1,
                                          xpress::objects::XpressProblem *prob)
    : wrapped(MapBuilder<K1>(iterable1)), prob(prob) {}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withLB(
    double newLb)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withUB(
    double newUb)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::VariableMapBuilderImplementation<K1> & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::toMap()
    -> std::unordered_map<K1, xpress::objects::Variable> {
  return prob->buildVariables(wrapped);
}

template <typename K1>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilderImplementation<K1>::toMap(
    Func0 makeResult,
    std::function<void(I &, K1, xpress::objects::Variable)> addResult) -> I {
  return prob->buildVariables<I>(wrapped, makeResult, addResult);
}

;

template <typename K1, typename K2>
template <typename Iter0, typename Iter1, typename Iter0IsIterable,
          typename Iter1IsIterable>
xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2>::
    VariableMap2BuilderImplementation(Iter0 const &iterable1,
                                      Iter1 const &iterable2,
                                      xpress::objects::XpressProblem *prob)
    : wrapped(Map2Builder<K1, K2>(iterable1, iterable2)), prob(prob) {}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2>::withLB(
    double newLb)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2>::withUB(
    double newUb)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<
    K1, K2>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<
    K1, K2>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<
    K1, K2>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<
    K1, K2>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<
    K1, K2>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<
    K1, K2>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<
    K1, K2>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2>::toMap()
    -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable> {
  return prob->buildVariables(wrapped);
}

template <typename K1, typename K2>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2>::toMap(
    Func0 makeResult,
    std::function<void(I &, K1, K2, xpress::objects::Variable)> addResult)
    -> I {
  return prob->buildVariables<I>(wrapped, makeResult, addResult);
}

;

template <typename K1, typename K2, typename K3>
template <typename Iter0, typename Iter1, typename Iter2,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable>
xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>::
    VariableMap3BuilderImplementation(Iter0 const &iterable1,
                                      Iter1 const &iterable2,
                                      Iter2 const &iterable3,
                                      xpress::objects::XpressProblem *prob)
    : wrapped(Map3Builder<K1, K2, K3>(iterable1, iterable2, iterable3)),
      prob(prob) {}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<
    K1, K2, K3>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>
        & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2,
                                                                K3>::toMap()
    -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable> {
  return prob->buildVariables(wrapped);
}

template <typename K1, typename K2, typename K3>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3>::
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, K3, xpress::objects::Variable)>
              addResult) -> I {
  return prob->buildVariables<I>(wrapped, makeResult, addResult);
}

;

template <typename K1, typename K2, typename K3, typename K4>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3, K4>::
    VariableMap4BuilderImplementation(Iter0 const &iterable1,
                                      Iter1 const &iterable2,
                                      Iter2 const &iterable3,
                                      Iter3 const &iterable4,
                                      xpress::objects::XpressProblem *prob)
    : wrapped(Map4Builder<K1, K2, K3, K4>(iterable1, iterable2, iterable3,
                                          iterable4)),
      prob(prob) {}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<
    K1, K2, K3, K4>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                  K4> & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                                K4>::toMap()
    -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable> {
  return prob->buildVariables(wrapped);
}

template <typename K1, typename K2, typename K3, typename K4>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::
    VariableMap4BuilderImplementation<K1, K2, K3, K4>::toMap(
        Func0 makeResult,
        std::function<void(I &, K1, K2, K3, K4, xpress::objects::Variable)>
            addResult) -> I {
  return prob->buildVariables<I>(wrapped, makeResult, addResult);
}

;

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3, K4, K5>::
    VariableMap5BuilderImplementation(Iter0 const &iterable1,
                                      Iter1 const &iterable2,
                                      Iter2 const &iterable3,
                                      Iter3 const &iterable4,
                                      Iter4 const &iterable5,
                                      xpress::objects::XpressProblem *prob)
    : wrapped(Map5Builder<K1, K2, K3, K4, K5>(iterable1, iterable2, iterable3,
                                              iterable4, iterable5)),
      prob(prob) {}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withLB(newLb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withUB(newUb);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withLimit(newLimit);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withName(newName);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withType(newType);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<
    K1, K2, K3, K4, K5>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3,
                                                                  K4, K5> & {
  wrapped.withFilter(newFilter);
  return *this;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3, K4,
                                                                K5>::toMap()
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, xpress::objects::Variable> {
  return prob->buildVariables(wrapped);
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::
    VariableMap5BuilderImplementation<K1, K2, K3, K4, K5>::toMap(
        Func0 makeResult,
        std::function<void(I &, K1, K2, K3, K4, K5, xpress::objects::Variable)>
            addResult) -> I {
  return prob->buildVariables<I>(wrapped, makeResult, addResult);
}

namespace xpress {

class AfterObjectiveCallbackHolder : public CallbackHolder {
  AfterObjectiveCallbackHolder(AfterObjectiveCallbackHolder const &) = delete;
  AfterObjectiveCallbackHolder &
  operator=(AfterObjectiveCallbackHolder const &) = delete;
  AfterObjectiveCallbackHolder(AfterObjectiveCallbackHolder &&) = delete;
  AfterObjectiveCallbackHolder &
  operator=(AfterObjectiveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  AfterObjectiveCallbackHolder(CallbackExceptionHandler *e,
                               std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapAfterObjectiveCallback(XPRSprob cbprob, void *context) {
    AfterObjectiveCallbackHolder *holder =
        reinterpret_cast<AfterObjectiveCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class BarIterationCallbackHolder : public CallbackHolder {
  BarIterationCallbackHolder(BarIterationCallbackHolder const &) = delete;
  BarIterationCallbackHolder &
  operator=(BarIterationCallbackHolder const &) = delete;
  BarIterationCallbackHolder(BarIterationCallbackHolder &&) = delete;
  BarIterationCallbackHolder &operator=(BarIterationCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int *)> callback;
  BarIterationCallbackHolder(CallbackExceptionHandler *e,
                             std::function<void(XPRSProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapBarIterationCallback(XPRSprob cbprob, void *context,
                                       int *p_action) {
    BarIterationCallbackHolder *holder =
        reinterpret_cast<BarIterationCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_action);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class BarlogCallbackHolder : public CallbackHolder {
  BarlogCallbackHolder(BarlogCallbackHolder const &) = delete;
  BarlogCallbackHolder &operator=(BarlogCallbackHolder const &) = delete;
  BarlogCallbackHolder(BarlogCallbackHolder &&) = delete;
  BarlogCallbackHolder &operator=(BarlogCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  BarlogCallbackHolder(CallbackExceptionHandler *e,
                       std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapBarlogCallback(XPRSprob cbprob, void *context) {
    BarlogCallbackHolder *holder =
        reinterpret_cast<BarlogCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class BeforeObjectiveCallbackHolder : public CallbackHolder {
  BeforeObjectiveCallbackHolder(BeforeObjectiveCallbackHolder const &) = delete;
  BeforeObjectiveCallbackHolder &
  operator=(BeforeObjectiveCallbackHolder const &) = delete;
  BeforeObjectiveCallbackHolder(BeforeObjectiveCallbackHolder &&) = delete;
  BeforeObjectiveCallbackHolder &
  operator=(BeforeObjectiveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  BeforeObjectiveCallbackHolder(CallbackExceptionHandler *e,
                                std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapBeforeObjectiveCallback(XPRSprob cbprob, void *context) {
    BeforeObjectiveCallbackHolder *holder =
        reinterpret_cast<BeforeObjectiveCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class BeforeSolveCallbackHolder : public CallbackHolder {
  BeforeSolveCallbackHolder(BeforeSolveCallbackHolder const &) = delete;
  BeforeSolveCallbackHolder &
  operator=(BeforeSolveCallbackHolder const &) = delete;
  BeforeSolveCallbackHolder(BeforeSolveCallbackHolder &&) = delete;
  BeforeSolveCallbackHolder &operator=(BeforeSolveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  BeforeSolveCallbackHolder(CallbackExceptionHandler *e,
                            std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapBeforeSolveCallback(XPRSprob cbprob, void *context) {
    BeforeSolveCallbackHolder *holder =
        reinterpret_cast<BeforeSolveCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class ChangeBranchObjectCallbackHolder : public CallbackHolder {
  ChangeBranchObjectCallbackHolder(ChangeBranchObjectCallbackHolder const &) =
      delete;
  ChangeBranchObjectCallbackHolder &
  operator=(ChangeBranchObjectCallbackHolder const &) = delete;
  ChangeBranchObjectCallbackHolder(ChangeBranchObjectCallbackHolder &&) =
      delete;
  ChangeBranchObjectCallbackHolder &
  operator=(ChangeBranchObjectCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, BranchObject *, BranchObject **)> callback;
  ChangeBranchObjectCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XPRSProblem &, BranchObject *, BranchObject **)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapChangeBranchObjectCallback(XPRSprob cbprob, void *context,
                                             XPRSbranchobject branch,
                                             XPRSbranchobject *p_newbranch) {
    ChangeBranchObjectCallbackHolder *holder =
        reinterpret_cast<ChangeBranchObjectCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      xpress::BranchObject *branchObject =
          branch ? new xpress::BranchObject(branch) : nullptr;
      xpress::BranchObject *p_newbranchObject = nullptr;
      holder->callback(*cbprobObject, branchObject, &p_newbranchObject);
      if (p_newbranchObject && p_newbranchObject != branchObject) {
        *p_newbranch = p_newbranchObject->ptr;
        p_newbranchObject->dtorFunction = nullptr;
        delete p_newbranchObject;
        ObjectMap<void>::removeObject<XPRSbranchobject>(*p_newbranch);
      }

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class CheckTimeCallbackHolder : public CallbackHolder {
  CheckTimeCallbackHolder(CheckTimeCallbackHolder const &) = delete;
  CheckTimeCallbackHolder &operator=(CheckTimeCallbackHolder const &) = delete;
  CheckTimeCallbackHolder(CheckTimeCallbackHolder &&) = delete;
  CheckTimeCallbackHolder &operator=(CheckTimeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  CheckTimeCallbackHolder(CallbackExceptionHandler *e,
                          std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapCheckTimeCallback(XPRSprob cbprob, void *context) {
    CheckTimeCallbackHolder *holder =
        reinterpret_cast<CheckTimeCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class ChgbranchCallbackHolder : public CallbackHolder {
  ChgbranchCallbackHolder(ChgbranchCallbackHolder const &) = delete;
  ChgbranchCallbackHolder &operator=(ChgbranchCallbackHolder const &) = delete;
  ChgbranchCallbackHolder(ChgbranchCallbackHolder &&) = delete;
  ChgbranchCallbackHolder &operator=(ChgbranchCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int *, int *, double *)> callback;
  ChgbranchCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XPRSProblem &, int *, int *, double *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapChgbranchCallback(XPRSprob cbprob, void *context,
                                    int *p_entity, int *p_up,
                                    double *p_estdeg) {
    ChgbranchCallbackHolder *holder =
        reinterpret_cast<ChgbranchCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_entity, p_up, p_estdeg);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class ChgnodeCallbackHolder : public CallbackHolder {
  ChgnodeCallbackHolder(ChgnodeCallbackHolder const &) = delete;
  ChgnodeCallbackHolder &operator=(ChgnodeCallbackHolder const &) = delete;
  ChgnodeCallbackHolder(ChgnodeCallbackHolder &&) = delete;
  ChgnodeCallbackHolder &operator=(ChgnodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int *)> callback;
  ChgnodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XPRSProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapChgnodeCallback(XPRSprob cbprob, void *context, int *p_node) {
    ChgnodeCallbackHolder *holder =
        reinterpret_cast<ChgnodeCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_node);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class ComputeRestartCallbackHolder : public CallbackHolder {
  ComputeRestartCallbackHolder(ComputeRestartCallbackHolder const &) = delete;
  ComputeRestartCallbackHolder &
  operator=(ComputeRestartCallbackHolder const &) = delete;
  ComputeRestartCallbackHolder(ComputeRestartCallbackHolder &&) = delete;
  ComputeRestartCallbackHolder &
  operator=(ComputeRestartCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  ComputeRestartCallbackHolder(CallbackExceptionHandler *e,
                               std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapComputeRestartCallback(XPRSprob cbprob, void *context) {
    ComputeRestartCallbackHolder *holder =
        reinterpret_cast<ComputeRestartCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class CutRoundCallbackHolder : public CallbackHolder {
  CutRoundCallbackHolder(CutRoundCallbackHolder const &) = delete;
  CutRoundCallbackHolder &operator=(CutRoundCallbackHolder const &) = delete;
  CutRoundCallbackHolder(CutRoundCallbackHolder &&) = delete;
  CutRoundCallbackHolder &operator=(CutRoundCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int, int *)> callback;
  CutRoundCallbackHolder(CallbackExceptionHandler *e,
                         std::function<void(XPRSProblem &, int, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapCutRoundCallback(XPRSprob cbprob, void *context,
                                   int ifxpresscuts, int *p_action) {
    CutRoundCallbackHolder *holder =
        reinterpret_cast<CutRoundCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, ifxpresscuts, p_action);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class CutlogCallbackHolder : public CallbackHolder {
  CutlogCallbackHolder(CutlogCallbackHolder const &) = delete;
  CutlogCallbackHolder &operator=(CutlogCallbackHolder const &) = delete;
  CutlogCallbackHolder(CutlogCallbackHolder &&) = delete;
  CutlogCallbackHolder &operator=(CutlogCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  CutlogCallbackHolder(CallbackExceptionHandler *e,
                       std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapCutlogCallback(XPRSprob cbprob, void *context) {
    CutlogCallbackHolder *holder =
        reinterpret_cast<CutlogCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class CutmgrCallbackHolder : public CallbackHolder {
  CutmgrCallbackHolder(CutmgrCallbackHolder const &) = delete;
  CutmgrCallbackHolder &operator=(CutmgrCallbackHolder const &) = delete;
  CutmgrCallbackHolder(CutmgrCallbackHolder &&) = delete;
  CutmgrCallbackHolder &operator=(CutmgrCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  CutmgrCallbackHolder(CallbackExceptionHandler *e,
                       std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapCutmgrCallback(XPRSprob cbprob, void *context) {
    CutmgrCallbackHolder *holder =
        reinterpret_cast<CutmgrCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class GapNotifyCallbackHolder : public CallbackHolder {
  GapNotifyCallbackHolder(GapNotifyCallbackHolder const &) = delete;
  GapNotifyCallbackHolder &operator=(GapNotifyCallbackHolder const &) = delete;
  GapNotifyCallbackHolder(GapNotifyCallbackHolder &&) = delete;
  GapNotifyCallbackHolder &operator=(GapNotifyCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, double *, double *, double *, double *)>
      callback;
  GapNotifyCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XPRSProblem &, double *, double *, double *, double *)>
          c)
      : exceptionHandler(e), callback(c) {}
  static void wrapGapNotifyCallback(XPRSprob cbprob, void *context,
                                    double *p_relgapnotifytarget,
                                    double *p_absgapnotifytarget,
                                    double *p_absgapnotifyobjtarget,
                                    double *p_absgapnotifyboundtarget) {
    GapNotifyCallbackHolder *holder =
        reinterpret_cast<GapNotifyCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_relgapnotifytarget,
                       p_absgapnotifytarget, p_absgapnotifyobjtarget,
                       p_absgapnotifyboundtarget);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class InfnodeCallbackHolder : public CallbackHolder {
  InfnodeCallbackHolder(InfnodeCallbackHolder const &) = delete;
  InfnodeCallbackHolder &operator=(InfnodeCallbackHolder const &) = delete;
  InfnodeCallbackHolder(InfnodeCallbackHolder &&) = delete;
  InfnodeCallbackHolder &operator=(InfnodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  InfnodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapInfnodeCallback(XPRSprob cbprob, void *context) {
    InfnodeCallbackHolder *holder =
        reinterpret_cast<InfnodeCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class IntsolCallbackHolder : public CallbackHolder {
  IntsolCallbackHolder(IntsolCallbackHolder const &) = delete;
  IntsolCallbackHolder &operator=(IntsolCallbackHolder const &) = delete;
  IntsolCallbackHolder(IntsolCallbackHolder &&) = delete;
  IntsolCallbackHolder &operator=(IntsolCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  IntsolCallbackHolder(CallbackExceptionHandler *e,
                       std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapIntsolCallback(XPRSprob cbprob, void *context) {
    IntsolCallbackHolder *holder =
        reinterpret_cast<IntsolCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class LplogCallbackHolder : public CallbackHolder {
  LplogCallbackHolder(LplogCallbackHolder const &) = delete;
  LplogCallbackHolder &operator=(LplogCallbackHolder const &) = delete;
  LplogCallbackHolder(LplogCallbackHolder &&) = delete;
  LplogCallbackHolder &operator=(LplogCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  LplogCallbackHolder(CallbackExceptionHandler *e,
                      std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapLplogCallback(XPRSprob cbprob, void *context) {
    LplogCallbackHolder *holder =
        reinterpret_cast<LplogCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class MessageCallbackHolder : public CallbackHolder {
  MessageCallbackHolder(MessageCallbackHolder const &) = delete;
  MessageCallbackHolder &operator=(MessageCallbackHolder const &) = delete;
  MessageCallbackHolder(MessageCallbackHolder &&) = delete;
  MessageCallbackHolder &operator=(MessageCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, char const *, int, int)> callback;
  MessageCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XPRSProblem &, char const *, int, int)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapMessageCallback(XPRSprob cbprob, void *context,
                                  char const *msg, int msglen, int msgtype) {
    MessageCallbackHolder *holder =
        reinterpret_cast<MessageCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, msg, msglen, msgtype);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class MipSolEnumHandlerCallbackHolder : public CallbackHolder {
  MipSolEnumHandlerCallbackHolder(MipSolEnumHandlerCallbackHolder const &) =
      delete;
  MipSolEnumHandlerCallbackHolder &
  operator=(MipSolEnumHandlerCallbackHolder const &) = delete;
  MipSolEnumHandlerCallbackHolder(MipSolEnumHandlerCallbackHolder &&) = delete;
  MipSolEnumHandlerCallbackHolder &
  operator=(MipSolEnumHandlerCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(MIPSolEnum &, XPRSProblem &, MIPSolPool &, int *,
                    double const *, int, double, double *, bool *, bool *)>
      callback;
  MipSolEnumHandlerCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(MIPSolEnum &, XPRSProblem &, MIPSolPool &, int *,
                        double const *, int, double, double *, bool *, bool *)>
          c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMipSolEnumHandlerCallback(
      XPRSmipsolenum mse, XPRSprob prob, XPRSmipsolpool msp, void *context,
      int *nMaxSols, double const *x_Zb, int nCols, double dMipObject,
      double *dModifiedObject, int *bRejectSoln,
      int *bUpdateMipAbsCutOffOnCurrentSet) {
    MipSolEnumHandlerCallbackHolder *holder =
        reinterpret_cast<MipSolEnumHandlerCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      MIPSolEnum *mseObject =
          ObjectMap<void>::findObject<XPRSmipsolenum, MIPSolEnum>(
              mse, [&mse](XPRSmipsolenum) { return new MIPSolEnum(mse); });
      XPRSProblem *probObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              prob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(prob);
              });
      MIPSolPool *mspObject =
          ObjectMap<void>::findObject<XPRSmipsolpool, MIPSolPool>(
              msp, [&msp](XPRSmipsolpool) { return new MIPSolPool(msp); });
      bool bool_bRejectSoln = bRejectSoln ? (*bRejectSoln != 0) : 0;
      bool bool_bUpdateMipAbsCutOffOnCurrentSet =
          bUpdateMipAbsCutOffOnCurrentSet
              ? (*bUpdateMipAbsCutOffOnCurrentSet != 0)
              : 0;
      callbackReturn = holder->callback(*mseObject, *probObject, *mspObject,
                                        nMaxSols, x_Zb, nCols, dMipObject,
                                        dModifiedObject, &bool_bRejectSoln,
                                        &bool_bUpdateMipAbsCutOffOnCurrentSet);
      if (bRejectSoln)
        *bRejectSoln = bool_bRejectSoln;
      if (bUpdateMipAbsCutOffOnCurrentSet)
        *bUpdateMipAbsCutOffOnCurrentSet = bool_bUpdateMipAbsCutOffOnCurrentSet;
    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
    }
    return callbackReturn;
  }
};

class MipThreadCallbackHolder : public CallbackHolder {
  MipThreadCallbackHolder(MipThreadCallbackHolder const &) = delete;
  MipThreadCallbackHolder &operator=(MipThreadCallbackHolder const &) = delete;
  MipThreadCallbackHolder(MipThreadCallbackHolder &&) = delete;
  MipThreadCallbackHolder &operator=(MipThreadCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, XPRSProblem &)> callback;
  MipThreadCallbackHolder(CallbackExceptionHandler *e,
                          std::function<void(XPRSProblem &, XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapMipThreadCallback(XPRSprob cbprob, void *context,
                                    XPRSprob threadprob) {
    MipThreadCallbackHolder *holder =
        reinterpret_cast<MipThreadCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      XPRSProblem *threadprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              threadprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(threadprob);
              });
      holder->callback(*cbprobObject, *threadprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class MipThreadDestroyCallbackHolder : public CallbackHolder {
  MipThreadDestroyCallbackHolder(MipThreadDestroyCallbackHolder const &) =
      delete;
  MipThreadDestroyCallbackHolder &
  operator=(MipThreadDestroyCallbackHolder const &) = delete;
  MipThreadDestroyCallbackHolder(MipThreadDestroyCallbackHolder &&) = delete;
  MipThreadDestroyCallbackHolder &
  operator=(MipThreadDestroyCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  MipThreadDestroyCallbackHolder(CallbackExceptionHandler *e,
                                 std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapMipThreadDestroyCallback(XPRSprob cbprob, void *context) {
    MipThreadDestroyCallbackHolder *holder =
        reinterpret_cast<MipThreadDestroyCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class MiplogCallbackHolder : public CallbackHolder {
  MiplogCallbackHolder(MiplogCallbackHolder const &) = delete;
  MiplogCallbackHolder &operator=(MiplogCallbackHolder const &) = delete;
  MiplogCallbackHolder(MiplogCallbackHolder &&) = delete;
  MiplogCallbackHolder &operator=(MiplogCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  MiplogCallbackHolder(CallbackExceptionHandler *e,
                       std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMiplogCallback(XPRSprob cbprob, void *context) {
    MiplogCallbackHolder *holder =
        reinterpret_cast<MiplogCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class MsJobEndCallbackHolder : public CallbackHolder {
  MsJobEndCallbackHolder(MsJobEndCallbackHolder const &) = delete;
  MsJobEndCallbackHolder &operator=(MsJobEndCallbackHolder const &) = delete;
  MsJobEndCallbackHolder(MsJobEndCallbackHolder &&) = delete;
  MsJobEndCallbackHolder &operator=(MsJobEndCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, void *, char const *, int *)> callback;
  MsJobEndCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(XPRSProblem &, void *, char const *, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMsJobEndCallback(XPRSprob cbprob, void *context, void *jobdata,
                                  char const *jobdesc, int *p_status) {
    MsJobEndCallbackHolder *holder =
        reinterpret_cast<MsJobEndCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn =
          holder->callback(*cbprobObject, jobdata, jobdesc, p_status);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class MsJobStartCallbackHolder : public CallbackHolder {
  MsJobStartCallbackHolder(MsJobStartCallbackHolder const &) = delete;
  MsJobStartCallbackHolder &
  operator=(MsJobStartCallbackHolder const &) = delete;
  MsJobStartCallbackHolder(MsJobStartCallbackHolder &&) = delete;
  MsJobStartCallbackHolder &operator=(MsJobStartCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, void *, char const *, int *)> callback;
  MsJobStartCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(XPRSProblem &, void *, char const *, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMsJobStartCallback(XPRSprob cbprob, void *context,
                                    void *jobdata, char const *jobdesc,
                                    int *p_status) {
    MsJobStartCallbackHolder *holder =
        reinterpret_cast<MsJobStartCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn =
          holder->callback(*cbprobObject, jobdata, jobdesc, p_status);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class MsWinnerCallbackHolder : public CallbackHolder {
  MsWinnerCallbackHolder(MsWinnerCallbackHolder const &) = delete;
  MsWinnerCallbackHolder &operator=(MsWinnerCallbackHolder const &) = delete;
  MsWinnerCallbackHolder(MsWinnerCallbackHolder &&) = delete;
  MsWinnerCallbackHolder &operator=(MsWinnerCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, void *, char const *)> callback;
  MsWinnerCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(XPRSProblem &, void *, char const *)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMsWinnerCallback(XPRSprob cbprob, void *context, void *jobdata,
                                  char const *jobdesc) {
    MsWinnerCallbackHolder *holder =
        reinterpret_cast<MsWinnerCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, jobdata, jobdesc);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class MseGetSolutionDiffCallbackHolder : public CallbackHolder {
  MseGetSolutionDiffCallbackHolder(MseGetSolutionDiffCallbackHolder const &) =
      delete;
  MseGetSolutionDiffCallbackHolder &
  operator=(MseGetSolutionDiffCallbackHolder const &) = delete;
  MseGetSolutionDiffCallbackHolder(MseGetSolutionDiffCallbackHolder &&) =
      delete;
  MseGetSolutionDiffCallbackHolder &
  operator=(MseGetSolutionDiffCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(MIPSolEnum &, int, int, int, double, double const *,
                    int const *, int, int, double, double const *, int const *,
                    double *)>
      callback;
  MseGetSolutionDiffCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<int(MIPSolEnum &, int, int, int, double, double const *,
                        int const *, int, int, double, double const *,
                        int const *, double *)>
          c)
      : exceptionHandler(e), callback(c) {}
  static int wrapMseGetSolutionDiffCallback(
      XPRSmipsolenum mse, void *context, int nCols, int iSolutionId_1,
      int iElemCount_1, double dMipObj_1, double const *Vals_1,
      int const *iSparseIndices_1, int iSolutionId_2, int iElemCount_2,
      double dMipObj_2, double const *Vals_2, int const *iSparseIndices_2,
      double *dDiffMetric) {
    MseGetSolutionDiffCallbackHolder *holder =
        reinterpret_cast<MseGetSolutionDiffCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      MIPSolEnum *mseObject =
          ObjectMap<void>::findObject<XPRSmipsolenum, MIPSolEnum>(
              mse, [&mse](XPRSmipsolenum) { return new MIPSolEnum(mse); });
      callbackReturn = holder->callback(
          *mseObject, nCols, iSolutionId_1, iElemCount_1, dMipObj_1, Vals_1,
          iSparseIndices_1, iSolutionId_2, iElemCount_2, dMipObj_2, Vals_2,
          iSparseIndices_2, dDiffMetric);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      callbackReturn = -1;
    }
    return callbackReturn;
  }
};

class NewnodeCallbackHolder : public CallbackHolder {
  NewnodeCallbackHolder(NewnodeCallbackHolder const &) = delete;
  NewnodeCallbackHolder &operator=(NewnodeCallbackHolder const &) = delete;
  NewnodeCallbackHolder(NewnodeCallbackHolder &&) = delete;
  NewnodeCallbackHolder &operator=(NewnodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int, int, int)> callback;
  NewnodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XPRSProblem &, int, int, int)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapNewnodeCallback(XPRSprob cbprob, void *context,
                                  int parentnode, int node, int branch) {
    NewnodeCallbackHolder *holder =
        reinterpret_cast<NewnodeCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, parentnode, node, branch);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class NlpCoefEvalErrorCallbackHolder : public CallbackHolder {
  NlpCoefEvalErrorCallbackHolder(NlpCoefEvalErrorCallbackHolder const &) =
      delete;
  NlpCoefEvalErrorCallbackHolder &
  operator=(NlpCoefEvalErrorCallbackHolder const &) = delete;
  NlpCoefEvalErrorCallbackHolder(NlpCoefEvalErrorCallbackHolder &&) = delete;
  NlpCoefEvalErrorCallbackHolder &
  operator=(NlpCoefEvalErrorCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, int, int)> callback;
  NlpCoefEvalErrorCallbackHolder(CallbackExceptionHandler *e,
                                 std::function<int(XPRSProblem &, int, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapNlpCoefEvalErrorCallback(XPRSprob cbprob, void *context,
                                          int col, int row) {
    NlpCoefEvalErrorCallbackHolder *holder =
        reinterpret_cast<NlpCoefEvalErrorCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, col, row);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class NodeLPSolvedCallbackHolder : public CallbackHolder {
  NodeLPSolvedCallbackHolder(NodeLPSolvedCallbackHolder const &) = delete;
  NodeLPSolvedCallbackHolder &
  operator=(NodeLPSolvedCallbackHolder const &) = delete;
  NodeLPSolvedCallbackHolder(NodeLPSolvedCallbackHolder &&) = delete;
  NodeLPSolvedCallbackHolder &operator=(NodeLPSolvedCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  NodeLPSolvedCallbackHolder(CallbackExceptionHandler *e,
                             std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapNodeLPSolvedCallback(XPRSprob cbprob, void *context) {
    NodeLPSolvedCallbackHolder *holder =
        reinterpret_cast<NodeLPSolvedCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class NodecutoffCallbackHolder : public CallbackHolder {
  NodecutoffCallbackHolder(NodecutoffCallbackHolder const &) = delete;
  NodecutoffCallbackHolder &
  operator=(NodecutoffCallbackHolder const &) = delete;
  NodecutoffCallbackHolder(NodecutoffCallbackHolder &&) = delete;
  NodecutoffCallbackHolder &operator=(NodecutoffCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int)> callback;
  NodecutoffCallbackHolder(CallbackExceptionHandler *e,
                           std::function<void(XPRSProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapNodecutoffCallback(XPRSprob cbprob, void *context, int node) {
    NodecutoffCallbackHolder *holder =
        reinterpret_cast<NodecutoffCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, node);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class OptnodeCallbackHolder : public CallbackHolder {
  OptnodeCallbackHolder(OptnodeCallbackHolder const &) = delete;
  OptnodeCallbackHolder &operator=(OptnodeCallbackHolder const &) = delete;
  OptnodeCallbackHolder(OptnodeCallbackHolder &&) = delete;
  OptnodeCallbackHolder &operator=(OptnodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int *)> callback;
  OptnodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XPRSProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapOptnodeCallback(XPRSprob cbprob, void *context,
                                  int *p_infeasible) {
    OptnodeCallbackHolder *holder =
        reinterpret_cast<OptnodeCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_infeasible);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class PreIntsolCallbackHolder : public CallbackHolder {
  PreIntsolCallbackHolder(PreIntsolCallbackHolder const &) = delete;
  PreIntsolCallbackHolder &operator=(PreIntsolCallbackHolder const &) = delete;
  PreIntsolCallbackHolder(PreIntsolCallbackHolder &&) = delete;
  PreIntsolCallbackHolder &operator=(PreIntsolCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int, int *, double *)> callback;
  PreIntsolCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XPRSProblem &, int, int *, double *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapPreIntsolCallback(XPRSprob cbprob, void *context, int soltype,
                                    int *p_reject, double *p_cutoff) {
    PreIntsolCallbackHolder *holder =
        reinterpret_cast<PreIntsolCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, soltype, p_reject, p_cutoff);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class PrenodeCallbackHolder : public CallbackHolder {
  PrenodeCallbackHolder(PrenodeCallbackHolder const &) = delete;
  PrenodeCallbackHolder &operator=(PrenodeCallbackHolder const &) = delete;
  PrenodeCallbackHolder(PrenodeCallbackHolder &&) = delete;
  PrenodeCallbackHolder &operator=(PrenodeCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, int *)> callback;
  PrenodeCallbackHolder(CallbackExceptionHandler *e,
                        std::function<void(XPRSProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapPrenodeCallback(XPRSprob cbprob, void *context,
                                  int *p_infeasible) {
    PrenodeCallbackHolder *holder =
        reinterpret_cast<PrenodeCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, p_infeasible);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class PresolveCallbackHolder : public CallbackHolder {
  PresolveCallbackHolder(PresolveCallbackHolder const &) = delete;
  PresolveCallbackHolder &operator=(PresolveCallbackHolder const &) = delete;
  PresolveCallbackHolder(PresolveCallbackHolder &&) = delete;
  PresolveCallbackHolder &operator=(PresolveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &)> callback;
  PresolveCallbackHolder(CallbackExceptionHandler *e,
                         std::function<void(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapPresolveCallback(XPRSprob cbprob, void *context) {
    PresolveCallbackHolder *holder =
        reinterpret_cast<PresolveCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};

class SlpCascadeEndCallbackHolder : public CallbackHolder {
  SlpCascadeEndCallbackHolder(SlpCascadeEndCallbackHolder const &) = delete;
  SlpCascadeEndCallbackHolder &
  operator=(SlpCascadeEndCallbackHolder const &) = delete;
  SlpCascadeEndCallbackHolder(SlpCascadeEndCallbackHolder &&) = delete;
  SlpCascadeEndCallbackHolder &
  operator=(SlpCascadeEndCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  SlpCascadeEndCallbackHolder(CallbackExceptionHandler *e,
                              std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpCascadeEndCallback(XPRSprob cbprob, void *context) {
    SlpCascadeEndCallbackHolder *holder =
        reinterpret_cast<SlpCascadeEndCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpCascadeStartCallbackHolder : public CallbackHolder {
  SlpCascadeStartCallbackHolder(SlpCascadeStartCallbackHolder const &) = delete;
  SlpCascadeStartCallbackHolder &
  operator=(SlpCascadeStartCallbackHolder const &) = delete;
  SlpCascadeStartCallbackHolder(SlpCascadeStartCallbackHolder &&) = delete;
  SlpCascadeStartCallbackHolder &
  operator=(SlpCascadeStartCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  SlpCascadeStartCallbackHolder(CallbackExceptionHandler *e,
                                std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpCascadeStartCallback(XPRSprob cbprob, void *context) {
    SlpCascadeStartCallbackHolder *holder =
        reinterpret_cast<SlpCascadeStartCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpCascadeVarCallbackHolder : public CallbackHolder {
  SlpCascadeVarCallbackHolder(SlpCascadeVarCallbackHolder const &) = delete;
  SlpCascadeVarCallbackHolder &
  operator=(SlpCascadeVarCallbackHolder const &) = delete;
  SlpCascadeVarCallbackHolder(SlpCascadeVarCallbackHolder &&) = delete;
  SlpCascadeVarCallbackHolder &
  operator=(SlpCascadeVarCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, int)> callback;
  SlpCascadeVarCallbackHolder(CallbackExceptionHandler *e,
                              std::function<int(XPRSProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpCascadeVarCallback(XPRSprob cbprob, void *context,
                                       int col) {
    SlpCascadeVarCallbackHolder *holder =
        reinterpret_cast<SlpCascadeVarCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, col);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpCascadeVarFailCallbackHolder : public CallbackHolder {
  SlpCascadeVarFailCallbackHolder(SlpCascadeVarFailCallbackHolder const &) =
      delete;
  SlpCascadeVarFailCallbackHolder &
  operator=(SlpCascadeVarFailCallbackHolder const &) = delete;
  SlpCascadeVarFailCallbackHolder(SlpCascadeVarFailCallbackHolder &&) = delete;
  SlpCascadeVarFailCallbackHolder &
  operator=(SlpCascadeVarFailCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, int)> callback;
  SlpCascadeVarFailCallbackHolder(CallbackExceptionHandler *e,
                                  std::function<int(XPRSProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpCascadeVarFailCallback(XPRSprob cbprob, void *context,
                                           int col) {
    SlpCascadeVarFailCallbackHolder *holder =
        reinterpret_cast<SlpCascadeVarFailCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, col);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpConstructCallbackHolder : public CallbackHolder {
  SlpConstructCallbackHolder(SlpConstructCallbackHolder const &) = delete;
  SlpConstructCallbackHolder &
  operator=(SlpConstructCallbackHolder const &) = delete;
  SlpConstructCallbackHolder(SlpConstructCallbackHolder &&) = delete;
  SlpConstructCallbackHolder &operator=(SlpConstructCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  SlpConstructCallbackHolder(CallbackExceptionHandler *e,
                             std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpConstructCallback(XPRSprob cbprob, void *context) {
    SlpConstructCallbackHolder *holder =
        reinterpret_cast<SlpConstructCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpDrColCallbackHolder : public CallbackHolder {
  SlpDrColCallbackHolder(SlpDrColCallbackHolder const &) = delete;
  SlpDrColCallbackHolder &operator=(SlpDrColCallbackHolder const &) = delete;
  SlpDrColCallbackHolder(SlpDrColCallbackHolder &&) = delete;
  SlpDrColCallbackHolder &operator=(SlpDrColCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, int, int, double, double *, double, double)>
      callback;
  SlpDrColCallbackHolder(CallbackExceptionHandler *e,
                         std::function<int(XPRSProblem &, int, int, double,
                                           double *, double, double)>
                             c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpDrColCallback(XPRSprob prob, void *context, int col,
                                  int detcol, double detval, double *p_value,
                                  double lb, double ub) {
    SlpDrColCallbackHolder *holder =
        reinterpret_cast<SlpDrColCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *probObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              prob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(prob);
              });
      callbackReturn =
          holder->callback(*probObject, col, detcol, detval, p_value, lb, ub);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(prob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpIntSolCallbackHolder : public CallbackHolder {
  SlpIntSolCallbackHolder(SlpIntSolCallbackHolder const &) = delete;
  SlpIntSolCallbackHolder &operator=(SlpIntSolCallbackHolder const &) = delete;
  SlpIntSolCallbackHolder(SlpIntSolCallbackHolder &&) = delete;
  SlpIntSolCallbackHolder &operator=(SlpIntSolCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  SlpIntSolCallbackHolder(CallbackExceptionHandler *e,
                          std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpIntSolCallback(XPRSprob cbprob, void *context) {
    SlpIntSolCallbackHolder *holder =
        reinterpret_cast<SlpIntSolCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpIterEndCallbackHolder : public CallbackHolder {
  SlpIterEndCallbackHolder(SlpIterEndCallbackHolder const &) = delete;
  SlpIterEndCallbackHolder &
  operator=(SlpIterEndCallbackHolder const &) = delete;
  SlpIterEndCallbackHolder(SlpIterEndCallbackHolder &&) = delete;
  SlpIterEndCallbackHolder &operator=(SlpIterEndCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  SlpIterEndCallbackHolder(CallbackExceptionHandler *e,
                           std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpIterEndCallback(XPRSprob cbprob, void *context) {
    SlpIterEndCallbackHolder *holder =
        reinterpret_cast<SlpIterEndCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class SlpIterStartCallbackHolder : public CallbackHolder {
  SlpIterStartCallbackHolder(SlpIterStartCallbackHolder const &) = delete;
  SlpIterStartCallbackHolder &
  operator=(SlpIterStartCallbackHolder const &) = delete;
  SlpIterStartCallbackHolder(SlpIterStartCallbackHolder &&) = delete;
  SlpIterStartCallbackHolder &operator=(SlpIterStartCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &)> callback;
  SlpIterStartCallbackHolder(CallbackExceptionHandler *e,
                             std::function<int(XPRSProblem &)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpIterStartCallback(XPRSprob cbprob, void *context) {
    SlpIterStartCallbackHolder *holder =
        reinterpret_cast<SlpIterStartCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
      callbackReturn = 1;
    }
    return callbackReturn;
  }
};

class SlpIterVarCallbackHolder : public CallbackHolder {
  SlpIterVarCallbackHolder(SlpIterVarCallbackHolder const &) = delete;
  SlpIterVarCallbackHolder &
  operator=(SlpIterVarCallbackHolder const &) = delete;
  SlpIterVarCallbackHolder(SlpIterVarCallbackHolder &&) = delete;
  SlpIterVarCallbackHolder &operator=(SlpIterVarCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, int)> callback;
  SlpIterVarCallbackHolder(CallbackExceptionHandler *e,
                           std::function<int(XPRSProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpIterVarCallback(XPRSprob cbprob, void *context, int col) {
    SlpIterVarCallbackHolder *holder =
        reinterpret_cast<SlpIterVarCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, col);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class SlpPreUpdateLinearizationCallbackHolder : public CallbackHolder {
  SlpPreUpdateLinearizationCallbackHolder(
      SlpPreUpdateLinearizationCallbackHolder const &) = delete;
  SlpPreUpdateLinearizationCallbackHolder &
  operator=(SlpPreUpdateLinearizationCallbackHolder const &) = delete;
  SlpPreUpdateLinearizationCallbackHolder(
      SlpPreUpdateLinearizationCallbackHolder &&) = delete;
  SlpPreUpdateLinearizationCallbackHolder &
  operator=(SlpPreUpdateLinearizationCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, int *)> callback;
  SlpPreUpdateLinearizationCallbackHolder(
      CallbackExceptionHandler *e, std::function<int(XPRSProblem &, int *)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapSlpPreUpdateLinearizationCallback(XPRSprob cbprob,
                                                   void *context,
                                                   int *ifRepeat) {
    SlpPreUpdateLinearizationCallbackHolder *holder =
        reinterpret_cast<SlpPreUpdateLinearizationCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, ifRepeat);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class StrongBranchSolveCallbackHolder : public CallbackHolder {
  StrongBranchSolveCallbackHolder(StrongBranchSolveCallbackHolder const &) =
      delete;
  StrongBranchSolveCallbackHolder &
  operator=(StrongBranchSolveCallbackHolder const &) = delete;
  StrongBranchSolveCallbackHolder(StrongBranchSolveCallbackHolder &&) = delete;
  StrongBranchSolveCallbackHolder &
  operator=(StrongBranchSolveCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<int(XPRSProblem &, int)> callback;
  StrongBranchSolveCallbackHolder(CallbackExceptionHandler *e,
                                  std::function<int(XPRSProblem &, int)> c)
      : exceptionHandler(e), callback(c) {}
  static int wrapStrongBranchSolveCallback(XPRSprob cbprob, void *context,
                                           int bndidx) {
    StrongBranchSolveCallbackHolder *holder =
        reinterpret_cast<StrongBranchSolveCallbackHolder *>(context);
    int callbackReturn = 0;
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      callbackReturn = holder->callback(*cbprobObject, bndidx);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
    return callbackReturn;
  }
};

class UserSolNotifyCallbackHolder : public CallbackHolder {
  UserSolNotifyCallbackHolder(UserSolNotifyCallbackHolder const &) = delete;
  UserSolNotifyCallbackHolder &
  operator=(UserSolNotifyCallbackHolder const &) = delete;
  UserSolNotifyCallbackHolder(UserSolNotifyCallbackHolder &&) = delete;
  UserSolNotifyCallbackHolder &
  operator=(UserSolNotifyCallbackHolder &&) = delete;

public:
  CallbackExceptionHandler *const exceptionHandler;
  std::function<void(XPRSProblem &, char const *, int)> callback;
  UserSolNotifyCallbackHolder(
      CallbackExceptionHandler *e,
      std::function<void(XPRSProblem &, char const *, int)> c)
      : exceptionHandler(e), callback(c) {}
  static void wrapUserSolNotifyCallback(XPRSprob cbprob, void *context,
                                        char const *solname, int status) {
    UserSolNotifyCallbackHolder *holder =
        reinterpret_cast<UserSolNotifyCallbackHolder *>(context);
    try {
      XPRSProblem *cbprobObject =
          ObjectMap<void>::findObject<XPRSprob, XPRSProblem>(
              cbprob, [&](XPRSprob) {
                return dynamic_cast<XPRSProblem *>(holder->exceptionHandler)
                    ->makeChild(cbprob);
              });
      holder->callback(*cbprobObject, solname, status);

    } catch (...) {
      holder->exceptionHandler->setCBException(std::current_exception());
      (void)XPRSinterrupt(cbprob, XPRS_STOP_GENERICERROR);
    }
  }
};
} // namespace xpress
void xpress::BranchObject::destroy() { check(XPRS_bo_destroy(ptr)); }

auto xpress::BranchObject::store() -> int {

  int p_status;
  check(XPRS_bo_store(ptr, &p_status));
  return p_status;
}

void xpress::BranchObject::addBranches(int nbranches) {

  check(XPRS_bo_addbranches(ptr, nbranches));
}

auto xpress::BranchObject::getBranches() -> int {

  int p_nbranches;
  check(XPRS_bo_getbranches(ptr, &p_nbranches));
  return p_nbranches;
}

void xpress::BranchObject::setPriority(int priority) {

  check(XPRS_bo_setpriority(ptr, priority));
}

void xpress::BranchObject::setPreferredBranch(int branch) {

  check(XPRS_bo_setpreferredbranch(ptr, branch));
}

void xpress::BranchObject::addBounds(int branch, int nbounds,
                                     Array<char const> const &bndtype,
                                     Array<int const> const &colind,
                                     Array<double const> const &bndval) {

  check(XPRS_bo_addbounds(ptr, branch, nbounds, bndtype.data, colind.data,
                          bndval.data));
}

void xpress::BranchObject::getBounds(int branch, int *p_nbounds, int maxbounds,
                                     Array<char> const &bndtype,
                                     Array<int> const &colind,
                                     Array<double> const &bndval) {

  check(XPRS_bo_getbounds(ptr, branch, p_nbounds, maxbounds, bndtype.data,
                          colind.data, bndval.data));
}

auto xpress::BranchObject::getBounds(int branch, int maxbounds,
                                     Array<char> const &bndtype,
                                     Array<int> const &colind,
                                     Array<double> const &bndval) -> int {

  int p_nbounds;
  check(XPRS_bo_getbounds(ptr, branch, &p_nbounds, maxbounds, bndtype.data,
                          colind.data, bndval.data));
  return p_nbounds;
}

auto xpress::BranchObject::getBounds(int branch) -> int {

  SizedArray<double> bndval(nullptr);
  Array<char> bndtype(nullptr);
  Array<int> colind(nullptr);
  int maxbounds = 0;
  int p_nbounds;
  check(XPRS_bo_getbounds(ptr, branch, &p_nbounds, maxbounds, bndtype.data,
                          colind.data, bndval.data));
  return p_nbounds;
}

void xpress::BranchObject::addRows(int branch, int nrows, int ncoefs,
                                   Array<char const> const &rowtype,
                                   Array<double const> const &rhs,
                                   Array<int const> const &start,
                                   Array<int const> const &colind,
                                   Array<double const> const &rowcoef) {

  check(XPRS_bo_addrows(ptr, branch, nrows, ncoefs, rowtype.data, rhs.data,
                        start.data, colind.data, rowcoef.data));
}

void xpress::BranchObject::getRows(int branch, int *p_nrows, int maxrows,
                                   int *p_ncoefs, int maxcoefs,
                                   Array<char> const &rowtype,
                                   Array<double> const &rhs,
                                   Array<int> const &start,
                                   Array<int> const &colind,
                                   Array<double> const &rowcoef) {

  check(XPRS_bo_getrows(ptr, branch, p_nrows, maxrows, p_ncoefs, maxcoefs,
                        rowtype.data, rhs.data, start.data, colind.data,
                        rowcoef.data));
}

auto xpress::BranchObject::getRows(int branch, int maxrows, int *p_ncoefs,
                                   int maxcoefs, Array<char> const &rowtype,
                                   Array<double> const &rhs,
                                   Array<int> const &start,
                                   Array<int> const &colind,
                                   Array<double> const &rowcoef) -> int {

  int p_nrows;
  check(XPRS_bo_getrows(ptr, branch, &p_nrows, maxrows, p_ncoefs, maxcoefs,
                        rowtype.data, rhs.data, start.data, colind.data,
                        rowcoef.data));
  return p_nrows;
}

auto xpress::BranchObject::getRows(int branch) -> int {

  SizedArray<double> rowcoef(nullptr);
  Array<double> rhs(nullptr);
  Array<char> rowtype(nullptr);
  int *p_ncoefs(nullptr);
  SizedArray<int> start(nullptr);
  Array<int> colind(nullptr);
  int maxcoefs = 0;
  int maxrows = 0;
  int p_nrows;
  check(XPRS_bo_getrows(ptr, branch, &p_nrows, maxrows, p_ncoefs, maxcoefs,
                        rowtype.data, rhs.data, start.data, colind.data,
                        rowcoef.data));
  return p_nrows;
}

void xpress::BranchObject::addCuts(int branch, int ncuts,
                                   Array<XPRScut const> const &cutind) {

  check(XPRS_bo_addcuts(ptr, branch, ncuts, cutind.data));
}

auto xpress::BranchObject::getID() -> int {

  int p_id;
  check(XPRS_bo_getid(ptr, &p_id));
  return p_id;
}

auto xpress::BranchObject::validate() -> int {

  int p_status;
  check(XPRS_bo_validate(ptr, &p_status));
  return p_status;
}

namespace xpress {
namespace objects {}
} // namespace xpress
void xpress::XPRSProblem::setProbName(
    std::optional<std::string> const &probname) {

  check(XPRSsetprobname(ptr, probname ? probname.value().c_str() : nullptr));
}

void xpress::XPRSProblem::setLogFile(
    std::optional<std::string> const &filename) {

  check(XPRSsetlogfile(ptr, filename ? filename.value().c_str() : nullptr));
}

void xpress::XPRSProblem::setDefaultControl(int control) {

  check(XPRSsetdefaultcontrol(ptr, control));
}

void xpress::XPRSProblem::setDefaults() { check(XPRSsetdefaults(ptr)); }

void xpress::XPRSProblem::readProb(std::optional<std::string> const &filename,
                                   std::optional<std::string> const &flags) {

  drop();
  check(XPRSreadprob(ptr, filename ? filename.value().c_str() : nullptr,
                     flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::readProb(std::optional<std::string> const &filename) {

  drop();
  char const *flags(nullptr);
  check(
      XPRSreadprob(ptr, filename ? filename.value().c_str() : nullptr, flags));
}

void xpress::XPRSProblem::loadLp(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<int const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub) {

  drop();
  check(XPRSloadlp(ptr, probname ? probname.value().c_str() : nullptr, ncols,
                   nrows, rowtype.data, rhs.data, rng.data, objcoef.data,
                   start.data, collen.data, rowind.data, rowcoef.data, lb.data,
                   ub.data));
}

void xpress::XPRSProblem::loadLp(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<XPRSint64 const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub) {

  drop();
  check(XPRSloadlp64(ptr, probname ? probname.value().c_str() : nullptr, ncols,
                     nrows, rowtype.data, rhs.data, rng.data, objcoef.data,
                     start.data, collen.data, rowind.data, rowcoef.data,
                     lb.data, ub.data));
}

void xpress::XPRSProblem::loadQP(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<int const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub,
    int nobjqcoefs, Array<int const> const &objqcol1,
    Array<int const> const &objqcol2, Array<double const> const &objqcoef) {

  drop();
  check(XPRSloadqp(ptr, probname ? probname.value().c_str() : nullptr, ncols,
                   nrows, rowtype.data, rhs.data, rng.data, objcoef.data,
                   start.data, collen.data, rowind.data, rowcoef.data, lb.data,
                   ub.data, nobjqcoefs, objqcol1.data, objqcol2.data,
                   objqcoef.data));
}

void xpress::XPRSProblem::loadQP(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<XPRSint64 const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub,
    XPRSint64 nobjqcoefs, Array<int const> const &objqcol1,
    Array<int const> const &objqcol2, Array<double const> const &objqcoef) {

  drop();
  check(XPRSloadqp64(ptr, probname ? probname.value().c_str() : nullptr, ncols,
                     nrows, rowtype.data, rhs.data, rng.data, objcoef.data,
                     start.data, collen.data, rowind.data, rowcoef.data,
                     lb.data, ub.data, nobjqcoefs, objqcol1.data, objqcol2.data,
                     objqcoef.data));
}

void xpress::XPRSProblem::loadMIQP(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<int const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub,
    int nobjqcoefs, Array<int const> const &objqcol1,
    Array<int const> const &objqcol2, Array<double const> const &objqcoef,
    int nentities, int nsets, Array<char const> const &coltype,
    Array<int const> const &entind, Array<double const> const &limit,
    Array<char const> const &settype, Array<int const> const &setstart,
    Array<int const> const &setind, Array<double const> const &refval) {

  drop();
  check(XPRSloadmiqp(
      ptr, probname ? probname.value().c_str() : nullptr, ncols, nrows,
      rowtype.data, rhs.data, rng.data, objcoef.data, start.data, collen.data,
      rowind.data, rowcoef.data, lb.data, ub.data, nobjqcoefs, objqcol1.data,
      objqcol2.data, objqcoef.data, nentities, nsets, coltype.data, entind.data,
      limit.data, settype.data, setstart.data, setind.data, refval.data));
}

void xpress::XPRSProblem::loadMIQP(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<XPRSint64 const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub,
    XPRSint64 nobjqcoefs, Array<int const> const &objqcol1,
    Array<int const> const &objqcol2, Array<double const> const &objqcoef,
    int nentities, int nsets, Array<char const> const &coltype,
    Array<int const> const &entind, Array<double const> const &limit,
    Array<char const> const &settype, Array<XPRSint64 const> const &setstart,
    Array<int const> const &setind, Array<double const> const &refval) {

  drop();
  check(XPRSloadmiqp64(
      ptr, probname ? probname.value().c_str() : nullptr, ncols, nrows,
      rowtype.data, rhs.data, rng.data, objcoef.data, start.data, collen.data,
      rowind.data, rowcoef.data, lb.data, ub.data, nobjqcoefs, objqcol1.data,
      objqcol2.data, objqcoef.data, nentities, nsets, coltype.data, entind.data,
      limit.data, settype.data, setstart.data, setind.data, refval.data));
}

void xpress::XPRSProblem::fixMipEntities(int options) {

  check(XPRSfixmipentities(ptr, options));
}

void xpress::XPRSProblem::loadModelCuts(int nrows,
                                        Array<int const> const &rowind) {

  check(XPRSloadmodelcuts(ptr, nrows, rowind.data));
}

void xpress::XPRSProblem::loadDelayedRows(int nrows,
                                          Array<int const> const &rowind) {

  check(XPRSloaddelayedrows(ptr, nrows, rowind.data));
}

void xpress::XPRSProblem::loadDirs(int ndirs, Array<int const> const &colind,
                                   Array<int const> const &priority,
                                   Array<char const> const &dir,
                                   Array<double const> const &uppseudo,
                                   Array<double const> const &downpseudo) {

  check(XPRSloaddirs(ptr, ndirs, colind.data, priority.data, dir.data,
                     uppseudo.data, downpseudo.data));
}

void xpress::XPRSProblem::loadBranchDirs(int ncols,
                                         Array<int const> const &colind,
                                         Array<int const> const &dir) {

  check(XPRSloadbranchdirs(ptr, ncols, colind.data, dir.data));
}

void xpress::XPRSProblem::loadPresolveDirs(
    int ndirs, Array<int const> const &colind, Array<int const> const &priority,
    Array<char const> const &dir, Array<double const> const &uppseudo,
    Array<double const> const &downpseudo) {

  check(XPRSloadpresolvedirs(ptr, ndirs, colind.data, priority.data, dir.data,
                             uppseudo.data, downpseudo.data));
}

void xpress::XPRSProblem::loadMip(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<int const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub, int nentities,
    int nsets, Array<char const> const &coltype, Array<int const> const &entind,
    Array<double const> const &limit, Array<char const> const &settype,
    Array<int const> const &setstart, Array<int const> const &setind,
    Array<double const> const &refval) {

  drop();
  check(XPRSloadmip(ptr, probname ? probname.value().c_str() : nullptr, ncols,
                    nrows, rowtype.data, rhs.data, rng.data, objcoef.data,
                    start.data, collen.data, rowind.data, rowcoef.data, lb.data,
                    ub.data, nentities, nsets, coltype.data, entind.data,
                    limit.data, settype.data, setstart.data, setind.data,
                    refval.data));
}

void xpress::XPRSProblem::loadMip(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<XPRSint64 const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub, int nentities,
    int nsets, Array<char const> const &coltype, Array<int const> const &entind,
    Array<double const> const &limit, Array<char const> const &settype,
    Array<XPRSint64 const> const &setstart, Array<int const> const &setind,
    Array<double const> const &refval) {

  drop();
  check(XPRSloadmip64(ptr, probname ? probname.value().c_str() : nullptr, ncols,
                      nrows, rowtype.data, rhs.data, rng.data, objcoef.data,
                      start.data, collen.data, rowind.data, rowcoef.data,
                      lb.data, ub.data, nentities, nsets, coltype.data,
                      entind.data, limit.data, settype.data, setstart.data,
                      setind.data, refval.data));
}

void xpress::XPRSProblem::addNames(int type,
                                   Array<std::string const> const &names,
                                   int first, int last) {

  std::vector<char> namesFlat;
  for (int i = 0; i < last - first + 1; ++i) {
    namesFlat.insert(namesFlat.end(), names.data[i].begin(),
                     names.data[i].end());
    namesFlat.push_back('\0');
  }
  check(XPRSaddnames(ptr, type, namesFlat.data(), first, last));
}

void xpress::XPRSProblem::addSetNames(Array<std::string const> const &names,
                                      int first, int last) {

  xpress_diags_push xpress_ignore_deprecated std::vector<char> namesFlat;
  for (int i = 0; i < last - first + 1; ++i) {
    namesFlat.insert(namesFlat.end(), names.data[i].begin(),
                     names.data[i].end());
    namesFlat.push_back('\0');
  }
  check(XPRSaddsetnames(ptr, namesFlat.data(), first, last));
  xpress_diags_pop
}

void xpress::XPRSProblem::scale(Array<int const> const &rowscale,
                                Array<int const> const &colscale) {

  check(XPRSscale(ptr, rowscale.data, colscale.data));
}

void xpress::XPRSProblem::readDirs(std::optional<std::string> const &filename) {

  check(XPRSreaddirs(ptr, filename ? filename.value().c_str() : nullptr));
}

void xpress::XPRSProblem::readDirs() {

  char const *filename(nullptr);
  check(XPRSreaddirs(ptr, filename));
}

void xpress::XPRSProblem::writeDirs(
    std::optional<std::string> const &filename) {

  check(XPRSwritedirs(ptr, filename ? filename.value().c_str() : nullptr));
}

void xpress::XPRSProblem::writeDirs() {

  char const *filename(nullptr);
  check(XPRSwritedirs(ptr, filename));
}

void xpress::XPRSProblem::unloadProb() {

  xpress_diags_push xpress_ignore_deprecated check(XPRSunloadprob(ptr));
  xpress_diags_pop
}

void xpress::XPRSProblem::setIndicators(int nrows,
                                        Array<int const> const &rowind,
                                        Array<int const> const &colind,
                                        Array<int const> const &complement) {

  check(
      XPRSsetindicators(ptr, nrows, rowind.data, colind.data, complement.data));
}

void xpress::XPRSProblem::addPwlCons(int npwls, int npoints,
                                     Array<int const> const &colind,
                                     Array<int const> const &resultant,
                                     Array<int const> const &start,
                                     Array<double const> const &xval,
                                     Array<double const> const &yval) {

  check(XPRSaddpwlcons(ptr, npwls, npoints, colind.data, resultant.data,
                       start.data, xval.data, yval.data));
}

void xpress::XPRSProblem::addPwlCons(int npwls, XPRSint64 npoints,
                                     Array<int const> const &colind,
                                     Array<int const> const &resultant,
                                     Array<XPRSint64 const> const &start,
                                     Array<double const> const &xval,
                                     Array<double const> const &yval) {

  check(XPRSaddpwlcons64(ptr, npwls, npoints, colind.data, resultant.data,
                         start.data, xval.data, yval.data));
}

void xpress::XPRSProblem::getPwlCons(Array<int> const &colind,
                                     Array<int> const &resultant,
                                     Array<int> const &start,
                                     Array<double> const &xval,
                                     Array<double> const &yval, int maxpoints,
                                     int *p_npoints, int first, int last) {

  check(XPRSgetpwlcons(ptr, colind.data, resultant.data, start.data, xval.data,
                       yval.data, maxpoints, p_npoints, first, last));
}

void xpress::XPRSProblem::getPwlCons(Array<int> const &colind,
                                     Array<int> const &resultant,
                                     Array<XPRSint64> const &start,
                                     Array<double> const &xval,
                                     Array<double> const &yval,
                                     XPRSint64 maxpoints, XPRSint64 *p_npoints,
                                     int first, int last) {

  check(XPRSgetpwlcons64(ptr, colind.data, resultant.data, start.data,
                         xval.data, yval.data, maxpoints, p_npoints, first,
                         last));
}

void xpress::XPRSProblem::addGenCons(int ncons, int ncols, int nvals,
                                     Array<GenConsType const> const &contype,
                                     Array<int const> const &resultant,
                                     Array<int const> const &colstart,
                                     Array<int const> const &colind,
                                     Array<int const> const &valstart,
                                     Array<double const> const &val) {

  static_assert(sizeof(Array<GenConsType>::value_type) == sizeof(int),
                "size mismatch");
  check(XPRSaddgencons(
      ptr, ncons, ncols, nvals, reinterpret_cast<int const *>(contype.data),
      resultant.data, colstart.data, colind.data, valstart.data, val.data));
}

void xpress::XPRSProblem::addGenCons(
    int ncons, XPRSint64 ncols, XPRSint64 nvals,
    Array<GenConsType const> const &contype, Array<int const> const &resultant,
    Array<XPRSint64 const> const &colstart, Array<int const> const &colind,
    Array<XPRSint64 const> const &valstart, Array<double const> const &val) {

  static_assert(sizeof(Array<GenConsType>::value_type) == sizeof(int),
                "size mismatch");
  check(XPRSaddgencons64(
      ptr, ncons, ncols, nvals, reinterpret_cast<int const *>(contype.data),
      resultant.data, colstart.data, colind.data, valstart.data, val.data));
}

void xpress::XPRSProblem::getGenCons(Array<GenConsType> const &contype,
                                     Array<int> const &resultant,
                                     Array<int> const &colstart,
                                     Array<int> const &colind, int maxcols,
                                     int *p_ncols, Array<int> const &valstart,
                                     Array<double> const &val, int maxvals,
                                     int *p_nvals, int first, int last) {

  static_assert(sizeof(Array<GenConsType>::value_type) == sizeof(int),
                "size mismatch");
  check(XPRSgetgencons(ptr, reinterpret_cast<int *>(contype.data),
                       resultant.data, colstart.data, colind.data, maxcols,
                       p_ncols, valstart.data, val.data, maxvals, p_nvals,
                       first, last));
}

void xpress::XPRSProblem::getGenCons(
    Array<GenConsType> const &contype, Array<int> const &resultant,
    Array<XPRSint64> const &colstart, Array<int> const &colind,
    XPRSint64 maxcols, XPRSint64 *p_ncols, Array<XPRSint64> const &valstart,
    Array<double> const &val, XPRSint64 maxvals, XPRSint64 *p_nvals, int first,
    int last) {

  static_assert(sizeof(Array<GenConsType>::value_type) == sizeof(int),
                "size mismatch");
  check(XPRSgetgencons64(ptr, reinterpret_cast<int *>(contype.data),
                         resultant.data, colstart.data, colind.data, maxcols,
                         p_ncols, valstart.data, val.data, maxvals, p_nvals,
                         first, last));
}

void xpress::XPRSProblem::delPwlCons(int npwls,
                                     Array<int const> const &pwlind) {

  check(XPRSdelpwlcons(ptr, npwls, pwlind.data));
}

void xpress::XPRSProblem::delGenCons(int ncons,
                                     Array<int const> const &conind) {

  check(XPRSdelgencons(ptr, ncons, conind.data));
}

void xpress::XPRSProblem::dumpControls() { check(XPRSdumpcontrols(ptr)); }

void xpress::XPRSProblem::getIndicators(Array<int> const &colind,
                                        Array<int> const &complement, int first,
                                        int last) {

  check(XPRSgetindicators(ptr, colind.data, complement.data, first, last));
}

void xpress::XPRSProblem::delIndicators(int first, int last) {

  check(XPRSdelindicators(ptr, first, last));
}

void xpress::XPRSProblem::lpOptimize(std::optional<std::string> const &flags) {

  check(XPRSlpoptimize(ptr, flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::lpOptimize() {

  char const *flags(nullptr);
  check(XPRSlpoptimize(ptr, flags));
}

void xpress::XPRSProblem::mipOptimize(std::optional<std::string> const &flags) {

  check(XPRSmipoptimize(ptr, flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::mipOptimize() {

  char const *flags(nullptr);
  check(XPRSmipoptimize(ptr, flags));
}

void xpress::XPRSProblem::optimize(std::optional<std::string> const &flags,
                                   int *solvestatus, int *solstatus) {

  check(XPRSoptimize(ptr, flags ? flags.value().c_str() : nullptr, solvestatus,
                     solstatus));
}

void xpress::XPRSProblem::readSlxSol(std::optional<std::string> const &filename,
                                     std::optional<std::string> const &flags) {

  check(XPRSreadslxsol(ptr, filename ? filename.value().c_str() : nullptr,
                       flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::readSlxSol(
    std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(XPRSreadslxsol(ptr, filename ? filename.value().c_str() : nullptr,
                       flags));
}

void xpress::XPRSProblem::readSlxSol() {

  char const *filename(nullptr);
  char const *flags(nullptr);
  check(XPRSreadslxsol(ptr, filename, flags));
}

void xpress::XPRSProblem::alter(std::optional<std::string> const &filename) {

  xpress_diags_push xpress_ignore_deprecated check(
      XPRSalter(ptr, filename ? filename.value().c_str() : nullptr));
  xpress_diags_pop
}

void xpress::XPRSProblem::readBasis(std::optional<std::string> const &filename,
                                    std::optional<std::string> const &flags) {

  check(XPRSreadbasis(ptr, filename ? filename.value().c_str() : nullptr,
                      flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::readBasis(
    std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(
      XPRSreadbasis(ptr, filename ? filename.value().c_str() : nullptr, flags));
}

void xpress::XPRSProblem::readBasis() {

  char const *filename(nullptr);
  char const *flags(nullptr);
  check(XPRSreadbasis(ptr, filename, flags));
}

void xpress::XPRSProblem::readBinSol(std::optional<std::string> const &filename,
                                     std::optional<std::string> const &flags) {

  check(XPRSreadbinsol(ptr, filename ? filename.value().c_str() : nullptr,
                       flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::readBinSol(
    std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(XPRSreadbinsol(ptr, filename ? filename.value().c_str() : nullptr,
                       flags));
}

void xpress::XPRSProblem::readBinSol() {

  char const *filename(nullptr);
  char const *flags(nullptr);
  check(XPRSreadbinsol(ptr, filename, flags));
}

void xpress::XPRSProblem::getInfeas(int *p_nprimalcols, int *p_nprimalrows,
                                    int *p_ndualrows, int *p_ndualcols,
                                    Array<int> const &x,
                                    Array<int> const &slack,
                                    Array<int> const &duals,
                                    Array<int> const &djs) {

  check(XPRSgetinfeas(ptr, p_nprimalcols, p_nprimalrows, p_ndualrows,
                      p_ndualcols, x.data, slack.data, duals.data, djs.data));
}

void xpress::XPRSProblem::getScaledInfeas(int *p_nprimalcols,
                                          int *p_nprimalrows, int *p_ndualrows,
                                          int *p_ndualcols, Array<int> const &x,
                                          Array<int> const &slack,
                                          Array<int> const &duals,
                                          Array<int> const &djs) {

  check(XPRSgetscaledinfeas(ptr, p_nprimalcols, p_nprimalrows, p_ndualrows,
                            p_ndualcols, x.data, slack.data, duals.data,
                            djs.data));
}

auto xpress::XPRSProblem::getUnbVec() -> int {

  int p_seq;
  check(XPRSgetunbvec(ptr, &p_seq));
  return p_seq;
}

auto xpress::XPRSProblem::crossoverLpSol() -> int {

  int p_status;
  check(XPRScrossoverlpsol(ptr, &p_status));
  return p_status;
}

void xpress::XPRSProblem::tune(std::optional<std::string> const &flags) {

  check(XPRStune(ptr, flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::tuneProbSetFile(
    std::optional<std::string> const &setfile, int ifmip, int sense) {

  check(XPRStuneprobsetfile(ptr, setfile ? setfile.value().c_str() : nullptr,
                            ifmip, sense));
}

void xpress::XPRSProblem::tunerWriteMethod(
    std::optional<std::string> const &methodfile) {

  check(XPRStunerwritemethod(ptr, methodfile ? methodfile.value().c_str()
                                             : nullptr));
}

void xpress::XPRSProblem::tunerReadMethod(
    std::optional<std::string> const &methodfile) {

  check(XPRStunerreadmethod(ptr,
                            methodfile ? methodfile.value().c_str() : nullptr));
}

void xpress::XPRSProblem::getBarNumStability(Array<int> const &colstab,
                                             Array<int> const &rowstab) {

  xpress_diags_push xpress_ignore_deprecated check(
      XPRSgetbarnumstability(ptr, colstab.data, rowstab.data));
  xpress_diags_pop
}

void xpress::XPRSProblem::getLastBarSol(Array<double> const &x,
                                        Array<double> const &slack,
                                        Array<double> const &duals,
                                        Array<double> const &djs,
                                        int *p_status) {

  check(XPRSgetlastbarsol(ptr, x.data, slack.data, duals.data, djs.data,
                          p_status));
}

void xpress::XPRSProblem::clearIIS() { check(XPRSiisclear(ptr)); }

auto xpress::XPRSProblem::firstIIS(int mode) -> int {

  int p_status;
  check(XPRSiisfirst(ptr, mode, &p_status));
  return p_status;
}

auto xpress::XPRSProblem::nextIIS() -> int {

  int p_status;
  check(XPRSiisnext(ptr, &p_status));
  return p_status;
}

void xpress::XPRSProblem::printIIS(int iis) { check(XPRSiisprint(ptr, iis)); }

void xpress::XPRSProblem::IISStatus(int *p_niis, Array<int> const &nrows,
                                    Array<int> const &ncols,
                                    Array<double> const &suminfeas,
                                    Array<int> const &numinfeas) {

  check(XPRSiisstatus(ptr, p_niis, nrows.data, ncols.data, suminfeas.data,
                      numinfeas.data));
}

void xpress::XPRSProblem::IISAll() { check(XPRSiisall(ptr)); }

void xpress::XPRSProblem::writeIIS(int iis,
                                   std::optional<std::string> const &filename,
                                   int filetype,
                                   std::optional<std::string> const &flags) {

  check(XPRSiiswrite(ptr, iis, filename ? filename.value().c_str() : nullptr,
                     filetype, flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::writeIIS(int iis,
                                   std::optional<std::string> const &filename,
                                   int filetype) {

  char const *flags(nullptr);
  check(XPRSiiswrite(ptr, iis, filename ? filename.value().c_str() : nullptr,
                     filetype, flags));
}

void xpress::XPRSProblem::IISIsolations(int iis) {

  check(XPRSiisisolations(ptr, iis));
}

void xpress::XPRSProblem::getIISData(
    int iis, int *p_nrows, int *p_ncols, Array<int> const &rowind,
    Array<int> const &colind, Array<char> const &contype,
    Array<char> const &bndtype, Array<double> const &duals,
    Array<double> const &djs, Array<char> const &isolationrows,
    Array<char> const &isolationcols) {

  check(XPRSgetiisdata(ptr, iis, p_nrows, p_ncols, rowind.data, colind.data,
                       contype.data, bndtype.data, duals.data, djs.data,
                       isolationrows.data, isolationcols.data));
}

void xpress::XPRSProblem::loadPresolveBasis(Array<int const> const &rowstat,
                                            Array<int const> const &colstat) {

  check(XPRSloadpresolvebasis(ptr, rowstat.data, colstat.data));
}

void xpress::XPRSProblem::loadSecureVecs(int nrows, int ncols,
                                         Array<int const> const &rowind,
                                         Array<int const> const &colind) {

  check(XPRSloadsecurevecs(ptr, nrows, ncols, rowind.data, colind.data));
}

void xpress::XPRSProblem::addRows(int nrows, int ncoefs,
                                  Array<char const> const &rowtype,
                                  Array<double const> const &rhs,
                                  Array<double const> const &rng,
                                  Array<int const> const &start,
                                  Array<int const> const &colind,
                                  Array<double const> const &rowcoef) {

  check(XPRSaddrows(ptr, nrows, ncoefs, rowtype.data, rhs.data, rng.data,
                    start.data, colind.data, rowcoef.data));
}

void xpress::XPRSProblem::addRows(int nrows, XPRSint64 ncoefs,
                                  Array<char const> const &rowtype,
                                  Array<double const> const &rhs,
                                  Array<double const> const &rng,
                                  Array<XPRSint64 const> const &start,
                                  Array<int const> const &colind,
                                  Array<double const> const &rowcoef) {

  check(XPRSaddrows64(ptr, nrows, ncoefs, rowtype.data, rhs.data, rng.data,
                      start.data, colind.data, rowcoef.data));
}

void xpress::XPRSProblem::addRows(int nrows, int ncoefs,
                                  Array<char const> const &rowtype,
                                  Array<double const> const &rhs,
                                  Array<int const> const &start,
                                  Array<int const> const &colind,
                                  Array<double const> const &rowcoef) {

  Array<double const> rng(nullptr);
  check(XPRSaddrows(ptr, nrows, ncoefs, rowtype.data, rhs.data, rng.data,
                    start.data, colind.data, rowcoef.data));
}

void xpress::XPRSProblem::addRows(int nrows, XPRSint64 ncoefs,
                                  Array<char const> const &rowtype,
                                  Array<double const> const &rhs,
                                  Array<XPRSint64 const> const &start,
                                  Array<int const> const &colind,
                                  Array<double const> const &rowcoef) {

  Array<double const> rng(nullptr);
  check(XPRSaddrows64(ptr, nrows, ncoefs, rowtype.data, rhs.data, rng.data,
                      start.data, colind.data, rowcoef.data));
}

void xpress::XPRSProblem::delRows(int nrows, Array<int const> const &rowind) {

  check(XPRSdelrows(ptr, nrows, rowind.data));
}

void xpress::XPRSProblem::addCols(int ncols, int ncoefs,
                                  Array<double const> const &objcoef,
                                  Array<int const> const &start,
                                  Array<int const> const &rowind,
                                  Array<double const> const &rowcoef,
                                  Array<double const> const &lb,
                                  Array<double const> const &ub) {

  check(XPRSaddcols(ptr, ncols, ncoefs, objcoef.data, start.data, rowind.data,
                    rowcoef.data, lb.data, ub.data));
}

void xpress::XPRSProblem::addCols(int ncols, XPRSint64 ncoefs,
                                  Array<double const> const &objcoef,
                                  Array<XPRSint64 const> const &start,
                                  Array<int const> const &rowind,
                                  Array<double const> const &rowcoef,
                                  Array<double const> const &lb,
                                  Array<double const> const &ub) {

  check(XPRSaddcols64(ptr, ncols, ncoefs, objcoef.data, start.data, rowind.data,
                      rowcoef.data, lb.data, ub.data));
}

void xpress::XPRSProblem::delCols(int ncols, Array<int const> const &colind) {

  check(XPRSdelcols(ptr, ncols, colind.data));
}

void xpress::XPRSProblem::chgColType(int ncols, Array<int const> const &colind,
                                     Array<char const> const &coltype) {

  check(XPRSchgcoltype(ptr, ncols, colind.data, coltype.data));
}

void xpress::XPRSProblem::loadBasis(Array<int const> const &rowstat,
                                    Array<int const> const &colstat) {

  check(XPRSloadbasis(ptr, rowstat.data, colstat.data));
}

void xpress::XPRSProblem::postSolve() { check(XPRSpostsolve(ptr)); }

void xpress::XPRSProblem::delSets(int nsets, Array<int const> const &setind) {

  check(XPRSdelsets(ptr, nsets, setind.data));
}

void xpress::XPRSProblem::addSets(int nsets, int nelems,
                                  Array<char const> const &settype,
                                  Array<int const> const &start,
                                  Array<int const> const &colind,
                                  Array<double const> const &refval) {

  check(XPRSaddsets(ptr, nsets, nelems, settype.data, start.data, colind.data,
                    refval.data));
}

void xpress::XPRSProblem::addSets(int nsets, XPRSint64 nelems,
                                  Array<char const> const &settype,
                                  Array<XPRSint64 const> const &start,
                                  Array<int const> const &colind,
                                  Array<double const> const &refval) {

  check(XPRSaddsets64(ptr, nsets, nelems, settype.data, start.data, colind.data,
                      refval.data));
}

void xpress::XPRSProblem::strongBranch(int nbounds,
                                       Array<int const> const &colind,
                                       Array<char const> const &bndtype,
                                       Array<double const> const &bndval,
                                       int iterlim, Array<double> const &objval,
                                       Array<int> const &status) {

  check(XPRSstrongbranch(ptr, nbounds, colind.data, bndtype.data, bndval.data,
                         iterlim, objval.data, status.data));
}

void xpress::XPRSProblem::estimateRowDualRanges(int nrows,
                                                Array<int const> const &rowind,
                                                int iterlim,
                                                Array<double> const &mindual,
                                                Array<double> const &maxdual) {

  check(XPRSestimaterowdualranges(ptr, nrows, rowind.data, iterlim,
                                  mindual.data, maxdual.data));
}

void xpress::XPRSProblem::setMessageStatus(int msgcode, int status) {

  check(XPRSsetmessagestatus(ptr, msgcode, status));
}

auto xpress::XPRSProblem::getMessageStatus(int msgcode) -> int {

  int p_status;
  check(XPRSgetmessagestatus(ptr, msgcode, &p_status));
  return p_status;
}

void xpress::XPRSProblem::chgObjSense(ObjSense objsense) {

  check(XPRSchgobjsense(ptr, static_cast<int>(objsense)));
}

void xpress::XPRSProblem::chgGlbLimit(int ncols, Array<int const> const &colind,
                                      Array<double const> const &limit) {

  check(XPRSchgglblimit(ptr, ncols, colind.data, limit.data));
}

void xpress::XPRSProblem::save() { check(XPRSsave(ptr)); }

void xpress::XPRSProblem::saveAs(
    std::optional<std::string> const &sSaveFileName) {

  check(
      XPRSsaveas(ptr, sSaveFileName ? sSaveFileName.value().c_str() : nullptr));
}

void xpress::XPRSProblem::restore(std::optional<std::string> const &probname,
                                  std::optional<std::string> const &flags) {

  check(XPRSrestore(ptr, probname ? probname.value().c_str() : nullptr,
                    flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::restore() {

  char const *probname(nullptr);
  char const *flags(nullptr);
  check(XPRSrestore(ptr, probname, flags));
}

void xpress::XPRSProblem::restore(std::optional<std::string> const &probname) {

  char const *flags(nullptr);
  check(XPRSrestore(ptr, probname ? probname.value().c_str() : nullptr, flags));
}

void xpress::XPRSProblem::pivot(int enter, int leave) {

  check(XPRSpivot(ptr, enter, leave));
}

auto xpress::XPRSProblem::loadLpSol(Array<double const> const &x,
                                    Array<double const> const &slack,
                                    Array<double const> const &duals,
                                    Array<double const> const &djs) -> int {

  int p_status;
  check(
      XPRSloadlpsol(ptr, x.data, slack.data, duals.data, djs.data, &p_status));
  return p_status;
}

auto xpress::XPRSProblem::repairWeightedInfeas(
    Array<double const> const &lepref, Array<double const> const &gepref,
    Array<double const> const &lbpref, Array<double const> const &ubpref,
    char phase2, double delta, std::optional<std::string> const &flags) -> int {

  int p_status;
  check(XPRSrepairweightedinfeas(ptr, &p_status, lepref.data, gepref.data,
                                 lbpref.data, ubpref.data, phase2, delta,
                                 flags ? flags.value().c_str() : nullptr));
  return p_status;
}

auto xpress::XPRSProblem::repairWeightedInfeasBounds(
    Array<double const> const &lepref, Array<double const> const &gepref,
    Array<double const> const &lbpref, Array<double const> const &ubpref,
    Array<double const> const &lerelax, Array<double const> const &gerelax,
    Array<double const> const &lbrelax, Array<double const> const &ubrelax,
    char phase2, double delta, std::optional<std::string> const &flags) -> int {

  int p_status;
  check(XPRSrepairweightedinfeasbounds(
      ptr, &p_status, lepref.data, gepref.data, lbpref.data, ubpref.data,
      lerelax.data, gerelax.data, lbrelax.data, ubrelax.data, phase2, delta,
      flags ? flags.value().c_str() : nullptr));
  return p_status;
}

auto xpress::XPRSProblem::repairInfeas(char penalty, char phase2, char flags,
                                       double lepref, double gepref,
                                       double lbpref, double ubpref,
                                       double delta) -> int {

  int p_status;
  check(XPRSrepairinfeas(ptr, &p_status, penalty, phase2, flags, lepref, gepref,
                         lbpref, ubpref, delta));
  return p_status;
}

auto xpress::XPRSProblem::basisStability(int type, int norm,
                                         int scaled) -> double {

  double p_value;
  check(XPRSbasisstability(ptr, type, norm, scaled, &p_value));
  return p_value;
}

void xpress::XPRSProblem::objSA(int ncols, Array<int const> const &colind,
                                Array<double> const &lower,
                                Array<double> const &upper) {

  check(XPRSobjsa(ptr, ncols, colind.data, lower.data, upper.data));
}

void xpress::XPRSProblem::bndSA(int ncols, Array<int const> const &colind,
                                Array<double> const &lblower,
                                Array<double> const &lbupper,
                                Array<double> const &ublower,
                                Array<double> const &ubupper) {

  check(XPRSbndsa(ptr, ncols, colind.data, lblower.data, lbupper.data,
                  ublower.data, ubupper.data));
}

void xpress::XPRSProblem::rhsSA(int nrows, Array<int const> const &rowind,
                                Array<double> const &lower,
                                Array<double> const &upper) {

  check(XPRSrhssa(ptr, nrows, rowind.data, lower.data, upper.data));
}

void xpress::XPRSProblem::addQMatrix(int row, int ncoefs,
                                     Array<int const> const &rowqcol1,
                                     Array<int const> const &rowqcol2,
                                     Array<double const> const &rowqcoef) {

  check(XPRSaddqmatrix(ptr, row, ncoefs, rowqcol1.data, rowqcol2.data,
                       rowqcoef.data));
}

void xpress::XPRSProblem::addQMatrix(int row, XPRSint64 ncoefs,
                                     Array<int const> const &rowqcol1,
                                     Array<int const> const &rowqcol2,
                                     Array<double const> const &rowqcoef) {

  check(XPRSaddqmatrix64(ptr, row, ncoefs, rowqcol1.data, rowqcol2.data,
                         rowqcoef.data));
}

void xpress::XPRSProblem::delQMatrix(int row) {

  check(XPRSdelqmatrix(ptr, row));
}

void xpress::XPRSProblem::loadQCQP(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<int const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub,
    int nobjqcoefs, Array<int const> const &objqcol1,
    Array<int const> const &objqcol2, Array<double const> const &objqcoef,
    int nqrows, Array<int const> const &qrowind,
    Array<int const> const &nrowqcoef, Array<int const> const &rowqcol1,
    Array<int const> const &rowqcol2, Array<double const> const &rowqcoef) {

  drop();
  check(XPRSloadqcqp(ptr, probname ? probname.value().c_str() : nullptr, ncols,
                     nrows, rowtype.data, rhs.data, rng.data, objcoef.data,
                     start.data, collen.data, rowind.data, rowcoef.data,
                     lb.data, ub.data, nobjqcoefs, objqcol1.data, objqcol2.data,
                     objqcoef.data, nqrows, qrowind.data, nrowqcoef.data,
                     rowqcol1.data, rowqcol2.data, rowqcoef.data));
}

void xpress::XPRSProblem::loadQCQP(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<XPRSint64 const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub,
    XPRSint64 nobjqcoefs, Array<int const> const &objqcol1,
    Array<int const> const &objqcol2, Array<double const> const &objqcoef,
    int nqrows, Array<int const> const &qrowind,
    Array<XPRSint64 const> const &nrowqcoef, Array<int const> const &rowqcol1,
    Array<int const> const &rowqcol2, Array<double const> const &rowqcoef) {

  drop();
  check(XPRSloadqcqp64(
      ptr, probname ? probname.value().c_str() : nullptr, ncols, nrows,
      rowtype.data, rhs.data, rng.data, objcoef.data, start.data, collen.data,
      rowind.data, rowcoef.data, lb.data, ub.data, nobjqcoefs, objqcol1.data,
      objqcol2.data, objqcoef.data, nqrows, qrowind.data, nrowqcoef.data,
      rowqcol1.data, rowqcol2.data, rowqcoef.data));
}

void xpress::XPRSProblem::loadMIQCQP(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<int const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub,
    int nobjqcoefs, Array<int const> const &objqcol1,
    Array<int const> const &objqcol2, Array<double const> const &objqcoef,
    int nqrows, Array<int const> const &qrowind,
    Array<int const> const &nrowqcoefs, Array<int const> const &rowqcol1,
    Array<int const> const &rowqcol2, Array<double const> const &rowqcoef,
    int nentities, int nsets, Array<char const> const &coltype,
    Array<int const> const &entind, Array<double const> const &limit,
    Array<char const> const &settype, Array<int const> const &setstart,
    Array<int const> const &setind, Array<double const> const &refval) {

  drop();
  check(XPRSloadmiqcqp(
      ptr, probname ? probname.value().c_str() : nullptr, ncols, nrows,
      rowtype.data, rhs.data, rng.data, objcoef.data, start.data, collen.data,
      rowind.data, rowcoef.data, lb.data, ub.data, nobjqcoefs, objqcol1.data,
      objqcol2.data, objqcoef.data, nqrows, qrowind.data, nrowqcoefs.data,
      rowqcol1.data, rowqcol2.data, rowqcoef.data, nentities, nsets,
      coltype.data, entind.data, limit.data, settype.data, setstart.data,
      setind.data, refval.data));
}

void xpress::XPRSProblem::loadMIQCQP(
    std::optional<std::string> const &probname, int ncols, int nrows,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<double const> const &rng, Array<double const> const &objcoef,
    Array<XPRSint64 const> const &start, Array<int const> const &collen,
    Array<int const> const &rowind, Array<double const> const &rowcoef,
    Array<double const> const &lb, Array<double const> const &ub,
    XPRSint64 nobjqcoefs, Array<int const> const &objqcol1,
    Array<int const> const &objqcol2, Array<double const> const &objqcoef,
    int nqrows, Array<int const> const &qrowind,
    Array<XPRSint64 const> const &nrowqcoefs, Array<int const> const &rowqcol1,
    Array<int const> const &rowqcol2, Array<double const> const &rowqcoef,
    int nentities, int nsets, Array<char const> const &coltype,
    Array<int const> const &entind, Array<double const> const &limit,
    Array<char const> const &settype, Array<XPRSint64 const> const &setstart,
    Array<int const> const &setind, Array<double const> const &refval) {

  drop();
  check(XPRSloadmiqcqp64(
      ptr, probname ? probname.value().c_str() : nullptr, ncols, nrows,
      rowtype.data, rhs.data, rng.data, objcoef.data, start.data, collen.data,
      rowind.data, rowcoef.data, lb.data, ub.data, nobjqcoefs, objqcol1.data,
      objqcol2.data, objqcoef.data, nqrows, qrowind.data, nrowqcoefs.data,
      rowqcol1.data, rowqcol2.data, rowqcoef.data, nentities, nsets,
      coltype.data, entind.data, limit.data, settype.data, setstart.data,
      setind.data, refval.data));
}

void xpress::XPRSProblem::addObj(int ncols, Array<int const> const &colind,
                                 Array<double const> const &objcoef,
                                 int priority, double weight) {

  check(XPRSaddobj(ptr, ncols, colind.data, objcoef.data, priority, weight));
}

void xpress::XPRSProblem::chgObjN(int objidx, int ncols,
                                  Array<int const> const &colind,
                                  Array<double const> const &objcoef) {

  check(XPRSchgobjn(ptr, objidx, ncols, colind.data, objcoef.data));
}

void xpress::XPRSProblem::delObj(int objidx) { check(XPRSdelobj(ptr, objidx)); }

void xpress::XPRSProblem::copyControls(XPRSProblem const &src) {

  check(XPRScopycontrols(ptr, src.ptr));
}

void xpress::XPRSProblem::copyProb(XPRSProblem const &src,
                                   std::optional<std::string> const &name) {

  check(XPRScopyprob(ptr, src.ptr, name ? name.value().c_str() : nullptr));
}

void xpress::XPRSProblem::destroyProb() { check(XPRSdestroyprob(ptr)); }

void xpress::XPRSProblem::interrupt(StopType reason) {

  check(XPRSinterrupt(ptr, static_cast<int>(reason)));
}

auto xpress::XPRSProblem::getProbName() -> std::string {

  std::vector<char> nameBuffer(std::max(0, 1025));

  check(XPRSgetprobname(ptr, nameBuffer.data()));
  nameBuffer.push_back('\0');
  return std::string(nameBuffer.data());
}

void xpress::XPRSProblem::setIntControl(int control, int value) {

  check(XPRSsetintcontrol(ptr, control, value));
}

void xpress::XPRSProblem::setLongControl(int control, XPRSint64 value) {

  check(XPRSsetintcontrol64(ptr, control, value));
}

void xpress::XPRSProblem::setDblControl(int control, double value) {

  check(XPRSsetdblcontrol(ptr, control, value));
}

void xpress::XPRSProblem::setStrControl(
    int control, std::optional<std::string> const &value) {

  check(
      XPRSsetstrcontrol(ptr, control, value ? value.value().c_str() : nullptr));
}

auto xpress::XPRSProblem::getIntControl(int control) const -> int {

  int p_value;
  check(XPRSgetintcontrol(ptr, control, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getLongControl(int control) const -> XPRSint64 {

  XPRSint64 p_value;
  check(XPRSgetintcontrol64(ptr, control, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getDblControl(int control) const -> double {

  double p_value;
  check(XPRSgetdblcontrol(ptr, control, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getStrControl(int control) const -> std::string {

  std::vector<char> valueBuffer(std::max(0, 1024));

  check(XPRSgetstrcontrol(ptr, control, valueBuffer.data()));
  valueBuffer.push_back('\0');
  return std::string(valueBuffer.data());
}

void xpress::XPRSProblem::getStringControl(int control, char *value,
                                           int maxbytes, int *p_nbytes) const {

  check(XPRSgetstringcontrol(ptr, control, value, maxbytes, p_nbytes));
}

auto xpress::XPRSProblem::getIntAttrib(int attrib) const -> int {

  int p_value;
  check(XPRSgetintattrib(ptr, attrib, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getLongAttrib(int attrib) const -> XPRSint64 {

  XPRSint64 p_value;
  check(XPRSgetintattrib64(ptr, attrib, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getStrAttrib(int attrib) const -> std::string {

  std::vector<char> valueBuffer(std::max(0, 1024));

  check(XPRSgetstrattrib(ptr, attrib, valueBuffer.data()));
  valueBuffer.push_back('\0');
  return std::string(valueBuffer.data());
}

void xpress::XPRSProblem::getStrStringAttrib(int attrib, char *value,
                                             int maxbytes,
                                             int *p_nbytes) const {

  check(XPRSgetstringattrib(ptr, attrib, value, maxbytes, p_nbytes));
}

auto xpress::XPRSProblem::getDblAttrib(int attrib) const -> double {

  double p_value;
  check(XPRSgetdblattrib(ptr, attrib, &p_value));
  return p_value;
}

void xpress::XPRSProblem::getControlInfo(std::optional<std::string> const &name,
                                         int *p_id, ParameterType *p_type) {

  int p_typeAsInt = p_type ? static_cast<int>(*p_type) : 0;
  check(XPRSgetcontrolinfo(ptr, name ? name.value().c_str() : nullptr, p_id,
                           &p_typeAsInt));
  if (p_type)
    *p_type = static_cast<ParameterType>(p_typeAsInt);
}

void xpress::XPRSProblem::getAttribInfo(std::optional<std::string> const &name,
                                        int *p_id, ParameterType *p_type) {

  int p_typeAsInt = p_type ? static_cast<int>(*p_type) : 0;
  check(XPRSgetattribinfo(ptr, name ? name.value().c_str() : nullptr, p_id,
                          &p_typeAsInt));
  if (p_type)
    *p_type = static_cast<ParameterType>(p_typeAsInt);
}

auto xpress::XPRSProblem::getIndex(
    int type, std::optional<std::string> const &name) -> int {

  int p_index;
  check(
      XPRSgetindex(ptr, type, name ? name.value().c_str() : nullptr, &p_index));
  return p_index;
}

void xpress::XPRSProblem::setObjIntControl(int objidx, ObjControl control,
                                           int value) {

  check(XPRSsetobjintcontrol(ptr, objidx, static_cast<int>(control), value));
}

void xpress::XPRSProblem::setObjDblControl(int objidx, ObjControl control,
                                           double value) {

  check(XPRSsetobjdblcontrol(ptr, objidx, static_cast<int>(control), value));
}

auto xpress::XPRSProblem::getObjIntControl(int objidx,
                                           ObjControl control) -> int {

  int p_value;
  check(XPRSgetobjintcontrol(ptr, objidx, static_cast<int>(control), &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getObjDblControl(int objidx,
                                           ObjControl control) -> double {

  double p_value;
  check(XPRSgetobjdblcontrol(ptr, objidx, static_cast<int>(control), &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getObjIntAttrib(int solveidx, int attrib) -> int {

  int p_value;
  check(XPRSgetobjintattrib(ptr, solveidx, attrib, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getObjLongAttrib(int solveidx,
                                           int attrib) -> XPRSint64 {

  XPRSint64 p_value;
  check(XPRSgetobjintattrib64(ptr, solveidx, attrib, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getObjDblAttrib(int solveidx, int attrib) -> double {

  double p_value;
  check(XPRSgetobjdblattrib(ptr, solveidx, attrib, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getQObj(int objqcol1, int objqcol2) -> double {

  double p_objqcoef;
  check(XPRSgetqobj(ptr, objqcol1, objqcol2, &p_objqcoef));
  return p_objqcoef;
}

void xpress::XPRSProblem::getDirs(int *p_ndir, Array<int> const &indices,
                                  Array<int> const &prios,
                                  Array<char> const &branchdirs,
                                  Array<double> const &uppseudo,
                                  Array<double> const &downpseudo) {

  check(XPRSgetdirs(ptr, p_ndir, indices.data, prios.data, branchdirs.data,
                    uppseudo.data, downpseudo.data));
}

auto xpress::XPRSProblem::getDirs(Array<int> const &indices,
                                  Array<int> const &prios,
                                  Array<char> const &branchdirs,
                                  Array<double> const &uppseudo,
                                  Array<double> const &downpseudo) -> int {

  int p_ndir;
  check(XPRSgetdirs(ptr, &p_ndir, indices.data, prios.data, branchdirs.data,
                    uppseudo.data, downpseudo.data));
  return p_ndir;
}

auto xpress::XPRSProblem::getDirs() -> int {

  Array<double> uppseudo(nullptr);
  Array<int> prios(nullptr);
  Array<char> branchdirs(nullptr);
  Array<int> indices(nullptr);
  Array<double> downpseudo(nullptr);
  int p_ndir;
  check(XPRSgetdirs(ptr, &p_ndir, indices.data, prios.data, branchdirs.data,
                    uppseudo.data, downpseudo.data));
  return p_ndir;
}

void xpress::XPRSProblem::getScale(Array<int> const &rowscale,
                                   Array<int> const &colscale) {

  check(XPRSgetscale(ptr, rowscale.data, colscale.data));
}

auto xpress::XPRSProblem::getPivotOrder() -> std::vector<int> {

  std::size_t const pivotorderLength =
      std::max<XPRSint64>(0, getLongAttrib(XPRS_ROWS));
  std::vector<int> pivotorder(pivotorderLength);
  check(XPRSgetpivotorder(ptr, pivotorder.data()));
  return pivotorder;
}

void xpress::XPRSProblem::getPresolveMap(Array<int> const &rowmap,
                                         Array<int> const &colmap) {

  check(XPRSgetpresolvemap(ptr, rowmap.data, colmap.data));
}

void xpress::XPRSProblem::bTran(Array<double> const &vec) {

  check(XPRSbtran(ptr, vec.data));
}

void xpress::XPRSProblem::fTran(Array<double> const &vec) {

  check(XPRSftran(ptr, vec.data));
}

void xpress::XPRSProblem::sparseBTran(Array<double> const &val,
                                      Array<int> const &ind, int *p_ncoefs) {

  check(XPRSsparsebtran(ptr, val.data, ind.data, p_ncoefs));
}

void xpress::XPRSProblem::sparseFTran(Array<double> const &val,
                                      Array<int> const &ind, int *p_ncoefs) {

  check(XPRSsparseftran(ptr, val.data, ind.data, p_ncoefs));
}

auto xpress::XPRSProblem::getObj(int first, int last) -> std::vector<double> {

  std::size_t const objcoefLength = std::max(0, (last - first + 1));
  std::vector<double> objcoef(objcoefLength);
  check(XPRSgetobj(ptr, objcoef.data(), first, last));
  return objcoef;
}

auto xpress::XPRSProblem::getObjN(int objidx, int first,
                                  int last) -> std::vector<double> {

  std::size_t const objcoefLength = std::max(0, (last - first + 1));
  std::vector<double> objcoef(objcoefLength);
  check(XPRSgetobjn(ptr, objidx, objcoef.data(), first, last));
  return objcoef;
}

auto xpress::XPRSProblem::getRhs(int first, int last) -> std::vector<double> {

  std::size_t const rhsLength = std::max(0, (last - first + 1));
  std::vector<double> rhs(rhsLength);
  check(XPRSgetrhs(ptr, rhs.data(), first, last));
  return rhs;
}

auto xpress::XPRSProblem::getRhsRange(int first,
                                      int last) -> std::vector<double> {

  std::size_t const rngLength = std::max(0, (last - first + 1));
  std::vector<double> rng(rngLength);
  check(XPRSgetrhsrange(ptr, rng.data(), first, last));
  return rng;
}

auto xpress::XPRSProblem::getLB(int first, int last) -> std::vector<double> {

  std::size_t const lbLength = std::max(0, (last - first + 1));
  std::vector<double> lb(lbLength);
  check(XPRSgetlb(ptr, lb.data(), first, last));
  return lb;
}

auto xpress::XPRSProblem::getUB(int first, int last) -> std::vector<double> {

  std::size_t const ubLength = std::max(0, (last - first + 1));
  std::vector<double> ub(ubLength);
  check(XPRSgetub(ptr, ub.data(), first, last));
  return ub;
}

void xpress::XPRSProblem::getCols(Array<int> const &start,
                                  Array<int> const &rowind,
                                  Array<double> const &rowcoef, int maxcoefs,
                                  int *p_ncoefs, int first, int last) {

  check(XPRSgetcols(ptr, start.data, rowind.data, rowcoef.data, maxcoefs,
                    p_ncoefs, first, last));
}

void xpress::XPRSProblem::getCols(Array<XPRSint64> const &start,
                                  Array<int> const &rowind,
                                  Array<double> const &rowcoef,
                                  XPRSint64 maxcoefs, XPRSint64 *p_ncoefs,
                                  int first, int last) {

  check(XPRSgetcols64(ptr, start.data, rowind.data, rowcoef.data, maxcoefs,
                      p_ncoefs, first, last));
}

auto xpress::XPRSProblem::getCols(Array<int> const &start,
                                  Array<int> const &rowind,
                                  Array<double> const &rowcoef, int maxcoefs,
                                  int first, int last) -> int {

  int p_ncoefs;
  check(XPRSgetcols(ptr, start.data, rowind.data, rowcoef.data, maxcoefs,
                    &p_ncoefs, first, last));
  return p_ncoefs;
}

auto xpress::XPRSProblem::getCols(Array<XPRSint64> const &start,
                                  Array<int> const &rowind,
                                  Array<double> const &rowcoef,
                                  XPRSint64 maxcoefs, int first,
                                  int last) -> XPRSint64 {

  XPRSint64 p_ncoefs;
  check(XPRSgetcols64(ptr, start.data, rowind.data, rowcoef.data, maxcoefs,
                      &p_ncoefs, first, last));
  return p_ncoefs;
}

void xpress::XPRSProblem::getRows(Array<int> const &start,
                                  Array<int> const &colind,
                                  Array<double> const &colcoef, int maxcoefs,
                                  int *p_ncoefs, int first, int last) {

  check(XPRSgetrows(ptr, start.data, colind.data, colcoef.data, maxcoefs,
                    p_ncoefs, first, last));
}

void xpress::XPRSProblem::getRows(Array<XPRSint64> const &start,
                                  Array<int> const &colind,
                                  Array<double> const &colcoef,
                                  XPRSint64 maxcoefs, XPRSint64 *p_ncoefs,
                                  int first, int last) {

  check(XPRSgetrows64(ptr, start.data, colind.data, colcoef.data, maxcoefs,
                      p_ncoefs, first, last));
}

auto xpress::XPRSProblem::getRows(Array<int> const &start,
                                  Array<int> const &colind,
                                  Array<double> const &colcoef, int maxcoefs,
                                  int first, int last) -> int {

  int p_ncoefs;
  check(XPRSgetrows(ptr, start.data, colind.data, colcoef.data, maxcoefs,
                    &p_ncoefs, first, last));
  return p_ncoefs;
}

auto xpress::XPRSProblem::getRows(Array<XPRSint64> const &start,
                                  Array<int> const &colind,
                                  Array<double> const &colcoef,
                                  XPRSint64 maxcoefs, int first,
                                  int last) -> XPRSint64 {

  XPRSint64 p_ncoefs;
  check(XPRSgetrows64(ptr, start.data, colind.data, colcoef.data, maxcoefs,
                      &p_ncoefs, first, last));
  return p_ncoefs;
}

void xpress::XPRSProblem::getMipEntities(
    int *p_nentities, int *p_nsets, Array<char> const &coltype,
    Array<int> const &colind, Array<double> const &limit,
    Array<char> const &settype, Array<int> const &start,
    Array<int> const &setcols, Array<double> const &refval) {

  check(XPRSgetmipentities(ptr, p_nentities, p_nsets, coltype.data, colind.data,
                           limit.data, settype.data, start.data, setcols.data,
                           refval.data));
}

void xpress::XPRSProblem::getMipEntities(
    int *p_nentities, int *p_nsets, Array<char> const &coltype,
    Array<int> const &colind, Array<double> const &limit,
    Array<char> const &settype, Array<XPRSint64> const &start,
    Array<int> const &setcols, Array<double> const &refval) {

  check(XPRSgetmipentities64(ptr, p_nentities, p_nsets, coltype.data,
                             colind.data, limit.data, settype.data, start.data,
                             setcols.data, refval.data));
}

void xpress::XPRSProblem::getMipEntities(int *p_nentities,
                                         Array<char> const &coltype,
                                         Array<int> const &colind,
                                         Array<double> const &limit) {

  Array<int> setcols(nullptr);
  Array<int> start(nullptr);
  Array<double> refval(nullptr);
  int *p_nsets(nullptr);
  Array<char> settype(nullptr);
  check(XPRSgetmipentities(ptr, p_nentities, p_nsets, coltype.data, colind.data,
                           limit.data, settype.data, start.data, setcols.data,
                           refval.data));
}

auto xpress::XPRSProblem::getMipEntities(Array<char> const &coltype,
                                         Array<int> const &colind,
                                         Array<double> const &limit) -> int {

  Array<int> setcols(nullptr);
  Array<int> start(nullptr);
  Array<double> refval(nullptr);
  int *p_nsets(nullptr);
  Array<char> settype(nullptr);
  int p_nentities;
  check(XPRSgetmipentities(ptr, &p_nentities, p_nsets, coltype.data,
                           colind.data, limit.data, settype.data, start.data,
                           setcols.data, refval.data));
  return p_nentities;
}

auto xpress::XPRSProblem::getRowFlags(int first, int last) -> std::vector<int> {

  std::size_t const flagsLength = std::max(0, (last - first + 1));
  std::vector<int> flags(flagsLength);
  check(XPRSgetrowflags(ptr, flags.data(), first, last));
  return flags;
}

void xpress::XPRSProblem::clearRowFlags(Array<int const> const &flags,
                                        int first, int last) {

  check(XPRSclearrowflags(ptr, flags.data, first, last));
}

auto xpress::XPRSProblem::getCoef(int row, int col) -> double {

  double p_coef;
  check(XPRSgetcoef(ptr, row, col, &p_coef));
  return p_coef;
}

void xpress::XPRSProblem::getMQObj(Array<int> const &start,
                                   Array<int> const &colind,
                                   Array<double> const &objqcoef, int maxcoefs,
                                   int *p_ncoefs, int first, int last) {

  check(XPRSgetmqobj(ptr, start.data, colind.data, objqcoef.data, maxcoefs,
                     p_ncoefs, first, last));
}

void xpress::XPRSProblem::getMQObj(Array<XPRSint64> const &start,
                                   Array<int> const &colind,
                                   Array<double> const &objqcoef,
                                   XPRSint64 maxcoefs, XPRSint64 *p_ncoefs,
                                   int first, int last) {

  check(XPRSgetmqobj64(ptr, start.data, colind.data, objqcoef.data, maxcoefs,
                       p_ncoefs, first, last));
}

auto xpress::XPRSProblem::getMQObj(Array<int> const &start,
                                   Array<int> const &colind,
                                   Array<double> const &objqcoef, int maxcoefs,
                                   int first, int last) -> int {

  int p_ncoefs;
  check(XPRSgetmqobj(ptr, start.data, colind.data, objqcoef.data, maxcoefs,
                     &p_ncoefs, first, last));
  return p_ncoefs;
}

auto xpress::XPRSProblem::getMQObj(Array<XPRSint64> const &start,
                                   Array<int> const &colind,
                                   Array<double> const &objqcoef,
                                   XPRSint64 maxcoefs, int first,
                                   int last) -> XPRSint64 {

  XPRSint64 p_ncoefs;
  check(XPRSgetmqobj64(ptr, start.data, colind.data, objqcoef.data, maxcoefs,
                       &p_ncoefs, first, last));
  return p_ncoefs;
}

void xpress::XPRSProblem::writeBasis(std::optional<std::string> const &filename,
                                     std::optional<std::string> const &flags) {

  check(XPRSwritebasis(ptr, filename ? filename.value().c_str() : nullptr,
                       flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::writeBasis(
    std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(XPRSwritebasis(ptr, filename ? filename.value().c_str() : nullptr,
                       flags));
}

void xpress::XPRSProblem::writeBasis() {

  char const *flags(nullptr);
  char const *filename(nullptr);
  check(XPRSwritebasis(ptr, filename, flags));
}

void xpress::XPRSProblem::writeSol(std::optional<std::string> const &filename,
                                   std::optional<std::string> const &flags) {

  check(XPRSwritesol(ptr, filename ? filename.value().c_str() : nullptr,
                     flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::writeSol(std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(
      XPRSwritesol(ptr, filename ? filename.value().c_str() : nullptr, flags));
}

void xpress::XPRSProblem::writeSol() {

  char const *flags(nullptr);
  char const *filename(nullptr);
  check(XPRSwritesol(ptr, filename, flags));
}

void xpress::XPRSProblem::writeBinSol(
    std::optional<std::string> const &filename,
    std::optional<std::string> const &flags) {

  check(XPRSwritebinsol(ptr, filename ? filename.value().c_str() : nullptr,
                        flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::writeBinSol(
    std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(XPRSwritebinsol(ptr, filename ? filename.value().c_str() : nullptr,
                        flags));
}

void xpress::XPRSProblem::writeBinSol() {

  char const *filename(nullptr);
  char const *flags(nullptr);
  check(XPRSwritebinsol(ptr, filename, flags));
}

void xpress::XPRSProblem::writePrtSol(
    std::optional<std::string> const &filename,
    std::optional<std::string> const &flags) {

  check(XPRSwriteprtsol(ptr, filename ? filename.value().c_str() : nullptr,
                        flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::writePrtSol(
    std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(XPRSwriteprtsol(ptr, filename ? filename.value().c_str() : nullptr,
                        flags));
}

void xpress::XPRSProblem::writePrtSol() {

  char const *flags(nullptr);
  char const *filename(nullptr);
  check(XPRSwriteprtsol(ptr, filename, flags));
}

void xpress::XPRSProblem::writeSlxSol(
    std::optional<std::string> const &filename,
    std::optional<std::string> const &flags) {

  check(XPRSwriteslxsol(ptr, filename ? filename.value().c_str() : nullptr,
                        flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::writeSlxSol(
    std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(XPRSwriteslxsol(ptr, filename ? filename.value().c_str() : nullptr,
                        flags));
}

void xpress::XPRSProblem::writeSlxSol() {

  char const *flags(nullptr);
  char const *filename(nullptr);
  check(XPRSwriteslxsol(ptr, filename, flags));
}

void xpress::XPRSProblem::getPrimalRay(Array<double> const &ray,
                                       int *p_hasray) {

  check(XPRSgetprimalray(ptr, ray.data, p_hasray));
}

void xpress::XPRSProblem::getDualRay(Array<double> const &ray, int *p_hasray) {

  check(XPRSgetdualray(ptr, ray.data, p_hasray));
}

void xpress::XPRSProblem::strongBranchCB(
    int nbounds, Array<int const> const &colind,
    Array<char const> const &bndtype, Array<double const> const &bndval,
    int iterlim, Array<double> const &objval, Array<int> const &status,
    std::function<int(XPRSProblem &, int)> callback) {

  StrongBranchSolveCallbackHolder callbackUserContextObject(this, callback);
  StrongBranchSolveCallbackHolder *callbackUserContext =
      &callbackUserContextObject;
  check(XPRSstrongbranchcb(
      ptr, nbounds, colind.data, bndtype.data, bndval.data, iterlim,
      objval.data, status.data,
      StrongBranchSolveCallbackHolder::wrapStrongBranchSolveCallback,
      callbackUserContext));
}

auto xpress::XPRSProblem::loadMipSol(Array<double const> const &x) -> int {

  int p_status;
  check(XPRSloadmipsol(ptr, x.data, &p_status));
  return p_status;
}

void xpress::XPRSProblem::getBasis(Array<int> const &rowstat,
                                   Array<int> const &colstat) {

  check(XPRSgetbasis(ptr, rowstat.data, colstat.data));
}

void xpress::XPRSProblem::getBasisVal(int row, int col, int *p_rowstat,
                                      int *p_colstat) {

  check(XPRSgetbasisval(ptr, row, col, p_rowstat, p_colstat));
}

void xpress::XPRSProblem::addManagedCuts(int globalvalid, int ncuts,
                                         Array<char const> const &rowtype,
                                         Array<double const> const &rhs,
                                         Array<int const> const &start,
                                         Array<int const> const &colind,
                                         Array<double const> const &cutcoef) {

  check(XPRSaddmanagedcuts(ptr, globalvalid, ncuts, rowtype.data, rhs.data,
                           start.data, colind.data, cutcoef.data));
}

void xpress::XPRSProblem::addManagedCuts(int globalvalid, int ncuts,
                                         Array<char const> const &rowtype,
                                         Array<double const> const &rhs,
                                         Array<XPRSint64 const> const &start,
                                         Array<int const> const &colind,
                                         Array<double const> const &cutcoef) {

  check(XPRSaddmanagedcuts64(ptr, globalvalid, ncuts, rowtype.data, rhs.data,
                             start.data, colind.data, cutcoef.data));
}

void xpress::XPRSProblem::addCuts(int ncuts, Array<int const> const &cuttype,
                                  Array<char const> const &rowtype,
                                  Array<double const> const &rhs,
                                  Array<int const> const &start,
                                  Array<int const> const &colind,
                                  Array<double const> const &cutcoef) {

  check(XPRSaddcuts(ptr, ncuts, cuttype.data, rowtype.data, rhs.data,
                    start.data, colind.data, cutcoef.data));
}

void xpress::XPRSProblem::addCuts(int ncuts, Array<int const> const &cuttype,
                                  Array<char const> const &rowtype,
                                  Array<double const> const &rhs,
                                  Array<XPRSint64 const> const &start,
                                  Array<int const> const &colind,
                                  Array<double const> const &cutcoef) {

  check(XPRSaddcuts64(ptr, ncuts, cuttype.data, rowtype.data, rhs.data,
                      start.data, colind.data, cutcoef.data));
}

void xpress::XPRSProblem::delCuts(int basis, int cuttype, int interp,
                                  double delta, int ncuts,
                                  Array<XPRScut const> const &cutind) {

  check(XPRSdelcuts(ptr, basis, cuttype, interp, delta, ncuts, cutind.data));
}

void xpress::XPRSProblem::delCuts(int basis) {

  SizedArray<XPRScut const> cutind(nullptr);
  int interp = 0;
  double delta = 0.0;
  int cuttype = 0;
  int ncuts = 0;
  check(XPRSdelcuts(ptr, basis, cuttype, interp, delta, ncuts, cutind.data));
}

void xpress::XPRSProblem::delCuts(int basis, int cuttype, int interp) {

  SizedArray<XPRScut const> cutind(nullptr);
  double delta = 0.0;
  int ncuts = 0;
  check(XPRSdelcuts(ptr, basis, cuttype, interp, delta, ncuts, cutind.data));
}

void xpress::XPRSProblem::delCuts(int basis, int cuttype, int interp,
                                  double delta) {

  SizedArray<XPRScut const> cutind(nullptr);
  int ncuts = 0;
  check(XPRSdelcuts(ptr, basis, cuttype, interp, delta, ncuts, cutind.data));
}

void xpress::XPRSProblem::delCuts(int basis, int ncuts,
                                  Array<XPRScut const> const &cutind) {

  int interp = 0;
  double delta = 0.0;
  int cuttype = 0;
  check(XPRSdelcuts(ptr, basis, cuttype, interp, delta, ncuts, cutind.data));
}

void xpress::XPRSProblem::delCPCuts(int cuttype, int interp, int ncuts,
                                    Array<XPRScut const> const &cutind) {

  check(XPRSdelcpcuts(ptr, cuttype, interp, ncuts, cutind.data));
}

void xpress::XPRSProblem::delCPCuts() {

  SizedArray<XPRScut const> cutind(nullptr);
  int interp = 0;
  int cuttype = 0;
  int ncuts = 0;
  check(XPRSdelcpcuts(ptr, cuttype, interp, ncuts, cutind.data));
}

void xpress::XPRSProblem::delCPCuts(int cuttype, int interp) {

  SizedArray<XPRScut const> cutind(nullptr);
  int ncuts = 0;
  check(XPRSdelcpcuts(ptr, cuttype, interp, ncuts, cutind.data));
}

void xpress::XPRSProblem::delCPCuts(int ncuts,
                                    Array<XPRScut const> const &cutind) {

  int interp = 0;
  int cuttype = 0;
  check(XPRSdelcpcuts(ptr, cuttype, interp, ncuts, cutind.data));
}

void xpress::XPRSProblem::getCutList(int cuttype, int interp, int *p_ncuts,
                                     int maxcuts,
                                     Array<XPRScut> const &cutind) {

  check(XPRSgetcutlist(ptr, cuttype, interp, p_ncuts, maxcuts, cutind.data));
}

auto xpress::XPRSProblem::getCutList(int cuttype, int interp, int maxcuts,
                                     Array<XPRScut> const &cutind) -> int {

  int p_ncuts;
  check(XPRSgetcutlist(ptr, cuttype, interp, &p_ncuts, maxcuts, cutind.data));
  return p_ncuts;
}

auto xpress::XPRSProblem::getCutList(int maxcuts,
                                     Array<XPRScut> const &cutind) -> int {

  int cuttype = 0;
  int interp = 0;
  int p_ncuts;
  check(XPRSgetcutlist(ptr, cuttype, interp, &p_ncuts, maxcuts, cutind.data));
  return p_ncuts;
}

void xpress::XPRSProblem::getCPCutList(int cuttype, int interp, double delta,
                                       int *p_ncuts, int maxcuts,
                                       Array<XPRScut> const &cutind,
                                       Array<double> const &viol) {

  check(XPRSgetcpcutlist(ptr, cuttype, interp, delta, p_ncuts, maxcuts,
                         cutind.data, viol.data));
}

auto xpress::XPRSProblem::getCPCutList(int cuttype, int interp, double delta,
                                       int maxcuts,
                                       Array<XPRScut> const &cutind,
                                       Array<double> const &viol) -> int {

  int p_ncuts;
  check(XPRSgetcpcutlist(ptr, cuttype, interp, delta, &p_ncuts, maxcuts,
                         cutind.data, viol.data));
  return p_ncuts;
}

auto xpress::XPRSProblem::getCPCutList(int maxcuts,
                                       Array<XPRScut> const &cutind,
                                       Array<double> const &viol) -> int {

  double delta = 0.0;
  int interp = 0;
  int cuttype = 0;
  int p_ncuts;
  check(XPRSgetcpcutlist(ptr, cuttype, interp, delta, &p_ncuts, maxcuts,
                         cutind.data, viol.data));
  return p_ncuts;
}

void xpress::XPRSProblem::getCPCuts(
    Array<XPRScut const> const &rowind, int ncuts, int maxcoefs,
    Array<int> const &cuttype, Array<char> const &rowtype,
    Array<int> const &start, Array<int> const &colind,
    Array<double> const &cutcoef, Array<double> const &rhs) {

  check(XPRSgetcpcuts(ptr, rowind.data, ncuts, maxcoefs, cuttype.data,
                      rowtype.data, start.data, colind.data, cutcoef.data,
                      rhs.data));
}

void xpress::XPRSProblem::getCPCuts(
    Array<XPRScut const> const &rowind, int ncuts, XPRSint64 maxcoefs,
    Array<int> const &cuttype, Array<char> const &rowtype,
    Array<XPRSint64> const &start, Array<int> const &colind,
    Array<double> const &cutcoef, Array<double> const &rhs) {

  check(XPRSgetcpcuts64(ptr, rowind.data, ncuts, maxcoefs, cuttype.data,
                        rowtype.data, start.data, colind.data, cutcoef.data,
                        rhs.data));
}

void xpress::XPRSProblem::loadCuts(int cuttype, int interp, int ncuts,
                                   Array<XPRScut const> const &cutind) {

  check(XPRSloadcuts(ptr, cuttype, interp, ncuts, cutind.data));
}

void xpress::XPRSProblem::loadCuts(int cuttype, int interp) {

  SizedArray<XPRScut const> cutind(nullptr);
  int ncuts = 0;
  check(XPRSloadcuts(ptr, cuttype, interp, ncuts, cutind.data));
}

void xpress::XPRSProblem::loadCuts(int ncuts,
                                   Array<XPRScut const> const &cutind) {

  int interp = 0;
  int cuttype = 0;
  check(XPRSloadcuts(ptr, cuttype, interp, ncuts, cutind.data));
}

auto xpress::XPRSProblem::storeCuts(
    int ncuts, int nodups, Array<int const> const &cuttype,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<int const> const &start, Array<int const> const &colind,
    Array<double const> const &cutcoef) -> std::vector<XPRScut> {

  std::size_t const cutindLength = ncuts;
  std::vector<XPRScut> cutind(cutindLength);
  check(XPRSstorecuts(ptr, ncuts, nodups, cuttype.data, rowtype.data, rhs.data,
                      start.data, cutind.data(), colind.data, cutcoef.data));
  return cutind;
}

auto xpress::XPRSProblem::storeCuts(
    int ncuts, int nodups, Array<int const> const &cuttype,
    Array<char const> const &rowtype, Array<double const> const &rhs,
    Array<XPRSint64 const> const &start, Array<int const> const &colind,
    Array<double const> const &cutcoef) -> std::vector<XPRScut> {

  std::size_t const cutindLength = ncuts;
  std::vector<XPRScut> cutind(cutindLength);
  check(XPRSstorecuts64(ptr, ncuts, nodups, cuttype.data, rowtype.data,
                        rhs.data, start.data, cutind.data(), colind.data,
                        cutcoef.data));
  return cutind;
}

void xpress::XPRSProblem::presolveRow(char rowtype, int norigcoefs,
                                      Array<int const> const &origcolind,
                                      Array<double const> const &origrowcoef,
                                      double origrhs, int maxcoefs,
                                      int *p_ncoefs, Array<int> const &colind,
                                      Array<double> const &rowcoef,
                                      double *p_rhs, int *p_status) {

  check(XPRSpresolverow(ptr, rowtype, norigcoefs, origcolind.data,
                        origrowcoef.data, origrhs, maxcoefs, p_ncoefs,
                        colind.data, rowcoef.data, p_rhs, p_status));
}

auto xpress::XPRSProblem::postSolveSol(Array<double const> const &prex)
    -> std::vector<double> {

  std::size_t const origxLength =
      std::max<XPRSint64>(0, getLongAttrib(XPRS_INPUTCOLS));
  std::vector<double> origx(origxLength);
  check(XPRSpostsolvesol(ptr, prex.data, origx.data()));
  return origx;
}

void xpress::XPRSProblem::getPivots(int enter, Array<int> const &outlist,
                                    Array<double> const &x, double *p_objval,
                                    int *p_npivots, int maxpivots) {

  check(XPRSgetpivots(ptr, enter, outlist.data, x.data, p_objval, p_npivots,
                      maxpivots));
}

void xpress::XPRSProblem::writeProb(std::optional<std::string> const &filename,
                                    std::optional<std::string> const &flags) {

  check(XPRSwriteprob(ptr, filename ? filename.value().c_str() : nullptr,
                      flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::writeProb(
    std::optional<std::string> const &filename) {

  char const *flags(nullptr);
  check(
      XPRSwriteprob(ptr, filename ? filename.value().c_str() : nullptr, flags));
}

void xpress::XPRSProblem::writeProb() {

  char const *filename(nullptr);
  char const *flags(nullptr);
  check(XPRSwriteprob(ptr, filename, flags));
}

auto xpress::XPRSProblem::calcSlacks(Array<double const> const &solution)
    -> std::vector<double> {

  std::size_t const slacksLength =
      std::max<XPRSint64>(0, getLongAttrib(XPRS_ORIGINALROWS));
  std::vector<double> slacks(slacksLength);
  check(XPRScalcslacks(ptr, solution.data, slacks.data()));
  return slacks;
}

auto xpress::XPRSProblem::calcReducedCosts(Array<double const> const &duals,
                                           Array<double const> const &solution)
    -> std::vector<double> {

  std::size_t const djsLength =
      std::max<XPRSint64>(0, getLongAttrib(XPRS_ORIGINALCOLS));
  std::vector<double> djs(djsLength);
  check(XPRScalcreducedcosts(ptr, duals.data, solution.data, djs.data()));
  return djs;
}

auto xpress::XPRSProblem::calcObjective(Array<double const> const &solution)
    -> double {

  double p_objval;
  check(XPRScalcobjective(ptr, solution.data, &p_objval));
  return p_objval;
}

auto xpress::XPRSProblem::calcObjN(
    int objidx, Array<double const> const &solution) -> double {

  double p_objval;
  check(XPRScalcobjn(ptr, objidx, solution.data, &p_objval));
  return p_objval;
}

auto xpress::XPRSProblem::calcSolInfo(Array<double const> const &solution,
                                      Array<double const> const &duals,
                                      int property) -> double {

  double p_value;
  check(XPRScalcsolinfo(ptr, solution.data, duals.data, property, &p_value));
  return p_value;
}

auto xpress::XPRSProblem::getRowType(int first, int last) -> std::vector<char> {

  std::size_t const rowtypeLength = std::max(0, (last - first + 1));
  std::vector<char> rowtype(rowtypeLength);
  check(XPRSgetrowtype(ptr, rowtype.data(), first, last));
  return rowtype;
}

void xpress::XPRSProblem::getPresolveBasis(Array<int> const &rowstat,
                                           Array<int> const &colstat) {

  check(XPRSgetpresolvebasis(ptr, rowstat.data, colstat.data));
}

auto xpress::XPRSProblem::getColType(int first, int last) -> std::vector<char> {

  std::size_t const coltypeLength = std::max(0, (last - first + 1));
  std::vector<char> coltype(coltypeLength);
  check(XPRSgetcoltype(ptr, coltype.data(), first, last));
  return coltype;
}

auto xpress::XPRSProblem::getQRowCoeff(int row, int rowqcol1,
                                       int rowqcol2) -> double {

  double p_rowqcoef;
  check(XPRSgetqrowcoeff(ptr, row, rowqcol1, rowqcol2, &p_rowqcoef));
  return p_rowqcoef;
}

void xpress::XPRSProblem::getQRowQMatrix(int row, Array<int> const &start,
                                         Array<int> const &colind,
                                         Array<double> const &rowqcoef,
                                         int maxcoefs, int *p_ncoefs, int first,
                                         int last) {

  check(XPRSgetqrowqmatrix(ptr, row, start.data, colind.data, rowqcoef.data,
                           maxcoefs, p_ncoefs, first, last));
}

auto xpress::XPRSProblem::getQRowQMatrix(int row, Array<int> const &start,
                                         Array<int> const &colind,
                                         Array<double> const &rowqcoef,
                                         int maxcoefs, int first,
                                         int last) -> int {

  int p_ncoefs;
  check(XPRSgetqrowqmatrix(ptr, row, start.data, colind.data, rowqcoef.data,
                           maxcoefs, &p_ncoefs, first, last));
  return p_ncoefs;
}

void xpress::XPRSProblem::getQRowQMatrixTriplets(
    int row, int *p_ncoefs, Array<int> const &rowqcol1,
    Array<int> const &rowqcol2, Array<double> const &rowqcoef) {

  check(XPRSgetqrowqmatrixtriplets(ptr, row, p_ncoefs, rowqcol1.data,
                                   rowqcol2.data, rowqcoef.data));
}

auto xpress::XPRSProblem::getQRowQMatrixTriplets(
    int row, Array<int> const &rowqcol1, Array<int> const &rowqcol2,
    Array<double> const &rowqcoef) -> int {

  int p_ncoefs;
  check(XPRSgetqrowqmatrixtriplets(ptr, row, &p_ncoefs, rowqcol1.data,
                                   rowqcol2.data, rowqcoef.data));
  return p_ncoefs;
}

void xpress::XPRSProblem::chgQRowCoeff(int row, int rowqcol1, int rowqcol2,
                                       double rowqcoef) {

  check(XPRSchgqrowcoeff(ptr, row, rowqcol1, rowqcol2, rowqcoef));
}

void xpress::XPRSProblem::getQRows(int *p_nrows, Array<int> const &rowind) {

  check(XPRSgetqrows(ptr, p_nrows, rowind.data));
}

auto xpress::XPRSProblem::getQRows(Array<int> const &rowind) -> int {

  int p_nrows;
  check(XPRSgetqrows(ptr, &p_nrows, rowind.data));
  return p_nrows;
}

auto xpress::XPRSProblem::getQRows() -> int {

  Array<int> rowind(nullptr);
  int p_nrows;
  check(XPRSgetqrows(ptr, &p_nrows, rowind.data));
  return p_nrows;
}

void xpress::XPRSProblem::chgBounds(int nbounds, Array<int const> const &colind,
                                    Array<char const> const &bndtype,
                                    Array<double const> const &bndval) {

  check(XPRSchgbounds(ptr, nbounds, colind.data, bndtype.data, bndval.data));
}

auto xpress::XPRSProblem::getNameList(int type, int first,
                                      int last) -> std::vector<std::string> {

  int namesLength = 0;
  check(XPRSgetnamelist(ptr, type, nullptr, 0, &namesLength, first, last));
  std::vector<char> namesBuffer(namesLength);
  int maxbytes = namesLength;
  int p_nbytesDummy = 0;
  int *p_nbytes = &p_nbytesDummy;
  std::vector<std::string> names(last - first + 1);
  check(XPRSgetnamelist(ptr, type, namesBuffer.data(), maxbytes, p_nbytes,
                        first, last));
  char const *namesStart = namesBuffer.data();
  for (int i = 0; i < last - first + 1; ++i) {
    names[i] = std::string(namesStart);
    namesStart += names[i].size() + 1;
  }
  return names;
}

void xpress::XPRSProblem::addMipSol(int length,
                                    Array<double const> const &solval,
                                    Array<int const> const &colind,
                                    std::optional<std::string> const &name) {

  check(XPRSaddmipsol(ptr, length, solval.data, colind.data,
                      name ? name.value().c_str() : nullptr));
}

auto xpress::XPRSProblem::getCutSlack(XPRScut cutind) -> double {

  double p_slack;
  check(XPRSgetcutslack(ptr, cutind, &p_slack));
  return p_slack;
}

auto xpress::XPRSProblem::getCutMap(
    int ncuts, Array<XPRScut const> const &cutind) -> std::vector<int> {

  std::size_t const cutmapLength = ncuts;
  std::vector<int> cutmap(cutmapLength);
  check(XPRSgetcutmap(ptr, ncuts, cutind.data, cutmap.data()));
  return cutmap;
}

void xpress::XPRSProblem::getPresolveSol(Array<double> const &x,
                                         Array<double> const &slack,
                                         Array<double> const &duals,
                                         Array<double> const &djs) {

  check(XPRSgetpresolvesol(ptr, x.data, slack.data, duals.data, djs.data));
}

void xpress::XPRSProblem::getPresolveSol(Array<double> const &x) {

  Array<double> duals(nullptr);
  Array<double> djs(nullptr);
  Array<double> slack(nullptr);
  check(XPRSgetpresolvesol(ptr, x.data, slack.data, duals.data, djs.data));
}

void xpress::XPRSProblem::getSolution(int *status, Array<double> const &x,
                                      int first, int last) const {

  check(XPRSgetsolution(ptr, status, x.data, first, last));
}

void xpress::XPRSProblem::getSlacks(int *status, Array<double> const &slacks,
                                    int first, int last) const {

  check(XPRSgetslacks(ptr, status, slacks.data, first, last));
}

void xpress::XPRSProblem::getDuals(int *status, Array<double> const &duals,
                                   int first, int last) const {

  check(XPRSgetduals(ptr, status, duals.data, first, last));
}

void xpress::XPRSProblem::getRedCosts(int *status, Array<double> const &djs,
                                      int first, int last) const {

  check(XPRSgetredcosts(ptr, status, djs.data, first, last));
}

void xpress::XPRSProblem::getLpSol(Array<double> const &x,
                                   Array<double> const &slack,
                                   Array<double> const &duals,
                                   Array<double> const &djs) {

  check(XPRSgetlpsol(ptr, x.data, slack.data, duals.data, djs.data));
}

void xpress::XPRSProblem::getLpSol(Array<double> const &x) {

  Array<double> djs(nullptr);
  Array<double> slack(nullptr);
  Array<double> duals(nullptr);
  check(XPRSgetlpsol(ptr, x.data, slack.data, duals.data, djs.data));
}

void xpress::XPRSProblem::getLpSolVal(int col, int row, double *p_x,
                                      double *p_slack, double *p_dual,
                                      double *p_dj) {

  xpress_diags_push xpress_ignore_deprecated check(
      XPRSgetlpsolval(ptr, col, row, p_x, p_slack, p_dual, p_dj));
  xpress_diags_pop
}

void xpress::XPRSProblem::getMipSol(Array<double> const &x,
                                    Array<double> const &slack) {

  xpress_diags_push xpress_ignore_deprecated check(
      XPRSgetmipsol(ptr, x.data, slack.data));
  xpress_diags_pop
}

void xpress::XPRSProblem::getMipSol(Array<double> const &x) {

  xpress_diags_push xpress_ignore_deprecated Array<double> slack(nullptr);
  check(XPRSgetmipsol(ptr, x.data, slack.data));
  xpress_diags_pop
}

void xpress::XPRSProblem::getMipSolVal(int col, int row, double *p_x,
                                       double *p_slack) {

  xpress_diags_push xpress_ignore_deprecated check(
      XPRSgetmipsolval(ptr, col, row, p_x, p_slack));
  xpress_diags_pop
}

void xpress::XPRSProblem::getCallbackSolution(bool *p_available,
                                              Array<double> const &x, int first,
                                              int last) const {

  int p_available_int = std::numeric_limits<int>::min();
  check(XPRSgetcallbacksolution(ptr, p_available ? &p_available_int : nullptr,
                                x.data, first, last));
  if (p_available)
    *p_available = p_available_int != 0;
}

void xpress::XPRSProblem::getCallbackSlacks(bool *p_available,
                                            Array<double> const &slacks,
                                            int first, int last) const {

  int p_available_int = std::numeric_limits<int>::min();
  check(XPRSgetcallbackslacks(ptr, p_available ? &p_available_int : nullptr,
                              slacks.data, first, last));
  if (p_available)
    *p_available = p_available_int != 0;
}

void xpress::XPRSProblem::getCallbackDuals(bool *p_available,
                                           Array<double> const &duals,
                                           int first, int last) const {

  int p_available_int = std::numeric_limits<int>::min();
  check(XPRSgetcallbackduals(ptr, p_available ? &p_available_int : nullptr,
                             duals.data, first, last));
  if (p_available)
    *p_available = p_available_int != 0;
}

void xpress::XPRSProblem::getCallbackRedCosts(bool *p_available,
                                              Array<double> const &djs,
                                              int first, int last) const {

  int p_available_int = std::numeric_limits<int>::min();
  check(XPRSgetcallbackredcosts(ptr, p_available ? &p_available_int : nullptr,
                                djs.data, first, last));
  if (p_available)
    *p_available = p_available_int != 0;
}

void xpress::XPRSProblem::getCallbackPresolveSolution(bool *p_available,
                                                      Array<double> const &x,
                                                      int first,
                                                      int last) const {

  int p_available_int = std::numeric_limits<int>::min();
  check(XPRSgetcallbackpresolvesolution(
      ptr, p_available ? &p_available_int : nullptr, x.data, first, last));
  if (p_available)
    *p_available = p_available_int != 0;
}

void xpress::XPRSProblem::getCallbackPresolveSlacks(bool *p_available,
                                                    Array<double> const &slacks,
                                                    int first, int last) const {

  int p_available_int = std::numeric_limits<int>::min();
  check(XPRSgetcallbackpresolveslacks(
      ptr, p_available ? &p_available_int : nullptr, slacks.data, first, last));
  if (p_available)
    *p_available = p_available_int != 0;
}

void xpress::XPRSProblem::getCallbackPresolveDuals(bool *p_available,
                                                   Array<double> const &duals,
                                                   int first, int last) const {

  int p_available_int = std::numeric_limits<int>::min();
  check(XPRSgetcallbackpresolveduals(
      ptr, p_available ? &p_available_int : nullptr, duals.data, first, last));
  if (p_available)
    *p_available = p_available_int != 0;
}

void xpress::XPRSProblem::getCallbackPresolveRedCosts(bool *p_available,
                                                      Array<double> const &djs,
                                                      int first,
                                                      int last) const {

  int p_available_int = std::numeric_limits<int>::min();
  check(XPRSgetcallbackpresolveredcosts(
      ptr, p_available ? &p_available_int : nullptr, djs.data, first, last));
  if (p_available)
    *p_available = p_available_int != 0;
}

void xpress::XPRSProblem::chgObj(int ncols, Array<int const> const &colind,
                                 Array<double const> const &objcoef) {

  check(XPRSchgobj(ptr, ncols, colind.data, objcoef.data));
}

void xpress::XPRSProblem::chgCoef(int row, int col, double coef) {

  check(XPRSchgcoef(ptr, row, col, coef));
}

void xpress::XPRSProblem::chgMCoef(int ncoefs, Array<int const> const &rowind,
                                   Array<int const> const &colind,
                                   Array<double const> const &rowcoef) {

  check(XPRSchgmcoef(ptr, ncoefs, rowind.data, colind.data, rowcoef.data));
}

void xpress::XPRSProblem::chgMCoef(XPRSint64 ncoefs,
                                   Array<int const> const &rowind,
                                   Array<int const> const &colind,
                                   Array<double const> const &rowcoef) {

  check(XPRSchgmcoef64(ptr, ncoefs, rowind.data, colind.data, rowcoef.data));
}

void xpress::XPRSProblem::chgMQObj(int ncoefs, Array<int const> const &objqcol1,
                                   Array<int const> const &objqcol2,
                                   Array<double const> const &objqcoef) {

  check(XPRSchgmqobj(ptr, ncoefs, objqcol1.data, objqcol2.data, objqcoef.data));
}

void xpress::XPRSProblem::chgMQObj(XPRSint64 ncoefs,
                                   Array<int const> const &objqcol1,
                                   Array<int const> const &objqcol2,
                                   Array<double const> const &objqcoef) {

  check(
      XPRSchgmqobj64(ptr, ncoefs, objqcol1.data, objqcol2.data, objqcoef.data));
}

void xpress::XPRSProblem::chgQObj(int objqcol1, int objqcol2, double objqcoef) {

  check(XPRSchgqobj(ptr, objqcol1, objqcol2, objqcoef));
}

void xpress::XPRSProblem::chgRhs(int nrows, Array<int const> const &rowind,
                                 Array<double const> const &rhs) {

  check(XPRSchgrhs(ptr, nrows, rowind.data, rhs.data));
}

void xpress::XPRSProblem::chgRhsRange(int nrows, Array<int const> const &rowind,
                                      Array<double const> const &rng) {

  check(XPRSchgrhsrange(ptr, nrows, rowind.data, rng.data));
}

void xpress::XPRSProblem::chgRowType(int nrows, Array<int const> const &rowind,
                                     Array<char const> const &rowtype) {

  check(XPRSchgrowtype(ptr, nrows, rowind.data, rowtype.data));
}

void xpress::XPRSProblem::msAddJob(
    std::optional<std::string> const &description, int ninitial,
    Array<int const> const &colind, Array<double const> const &initial,
    int nintcontrols, Array<int const> const &intcontrolid,
    Array<int const> const &intcontrolval, int ndblcontrols,
    Array<int const> const &dblcontrolid,
    Array<double const> const &dblcontrolval, void *data) {

  check(XPRSmsaddjob(ptr, description ? description.value().c_str() : nullptr,
                     ninitial, colind.data, initial.data, nintcontrols,
                     intcontrolid.data, intcontrolval.data, ndblcontrols,
                     dblcontrolid.data, dblcontrolval.data, data));
}

void xpress::XPRSProblem::msAddPreset(
    std::optional<std::string> const &description, int preset, int maxjobs,
    void *data) {

  check(XPRSmsaddpreset(ptr,
                        description ? description.value().c_str() : nullptr,
                        preset, maxjobs, data));
}

void xpress::XPRSProblem::msAddCustomPreset(
    std::optional<std::string> const &description, int preset, int maxjobs,
    int ninitial, Array<int const> const &colind,
    Array<double const> const &initial, int nintcontrols,
    Array<int const> const &intcontrolid, Array<int const> const &intcontrolval,
    int ndblcontrols, Array<int const> const &dblcontrolid,
    Array<double const> const &dblcontrolval, void *data) {

  check(XPRSmsaddcustompreset(
      ptr, description ? description.value().c_str() : nullptr, preset, maxjobs,
      ninitial, colind.data, initial.data, nintcontrols, intcontrolid.data,
      intcontrolval.data, ndblcontrols, dblcontrolid.data, dblcontrolval.data,
      data));
}

void xpress::XPRSProblem::nlpSetFunctionError() {

  check(XPRSnlpsetfunctionerror(ptr));
}

void xpress::XPRSProblem::nlpPrintEvalInfo() {

  check(XPRSnlpprintevalinfo(ptr));
}

void xpress::XPRSProblem::nlpValidate() { check(XPRSnlpvalidate(ptr)); }

void xpress::XPRSProblem::nlpOptimize(std::optional<std::string> const &flags) {

  check(XPRSnlpoptimize(ptr, flags ? flags.value().c_str() : nullptr));
}

void xpress::XPRSProblem::getNlpsol(Array<double> const &x,
                                    Array<double> const &slack,
                                    Array<double> const &duals,
                                    Array<double> const &djs) {

  xpress_diags_push xpress_ignore_deprecated check(
      XPRSgetnlpsol(ptr, x.data, slack.data, duals.data, djs.data));
  xpress_diags_pop
}

void xpress::XPRSProblem::nlpCurrentIV() { check(XPRSnlpsetcurrentiv(ptr)); }

void xpress::XPRSProblem::nlpValidateRow(int row) {

  check(XPRSnlpvalidaterow(ptr, row));
}

void xpress::XPRSProblem::nlpValidateKKT(int mode, int respectbasis,
                                         int updatemult, double violtarget) {

  check(XPRSnlpvalidatekkt(ptr, mode, respectbasis, updatemult, violtarget));
}

void xpress::XPRSProblem::msClear() { check(XPRSmsclear(ptr)); }

auto xpress::XPRSProblem::nlpEvaluateFormula(
    int parsed, Array<int const> const &type,
    Array<double const> const &values) -> double {

  double p_value;
  check(XPRSnlpevaluateformula(ptr, parsed, type.data, values.data, &p_value));
  return p_value;
}

void xpress::XPRSProblem::nlpValidateVector(Array<double const> const &solution,
                                            double *p_suminf,
                                            double *p_sumscaledinf,
                                            double *p_objval) {

  check(XPRSnlpvalidatevector(ptr, solution.data, p_suminf, p_sumscaledinf,
                              p_objval));
}

void xpress::XPRSProblem::nlpDelUserFunction(int type) {

  check(XPRSnlpdeluserfunction(ptr, type));
}

void xpress::XPRSProblem::nlpImportLibFunc(
    std::optional<std::string> const &libname,
    std::optional<std::string> const &funcname, XPRSfunctionptr *p_function,
    int *p_status) {

  check(XPRSnlpimportlibfunc(ptr, libname ? libname.value().c_str() : nullptr,
                             funcname ? funcname.value().c_str() : nullptr,
                             p_function, p_status));
}

void xpress::XPRSProblem::nlpAddFormulas(int ncoefs,
                                         Array<int const> const &rowind,
                                         Array<int const> const &formulastart,
                                         int parsed,
                                         Array<int const> const &type,
                                         Array<double const> const &value) {

  check(XPRSnlpaddformulas(ptr, ncoefs, rowind.data, formulastart.data, parsed,
                           type.data, value.data));
}

void xpress::XPRSProblem::nlpChgFormulaStr(
    int row, std::optional<std::string> const &formula) {

  check(XPRSnlpchgformulastr(ptr, row,
                             formula ? formula.value().c_str() : nullptr));
}

void xpress::XPRSProblem::nlpChgFormula(int row, int parsed,
                                        Array<int const> const &type,
                                        Array<double const> const &value) {

  check(XPRSnlpchgformula(ptr, row, parsed, type.data, value.data));
}

void xpress::XPRSProblem::nlpGetFormula(int row, int parsed, int maxtypes,
                                        int *p_ntypes, Array<int> const &type,
                                        Array<double> const &value) {

  check(XPRSnlpgetformula(ptr, row, parsed, maxtypes, p_ntypes, type.data,
                          value.data));
}

void xpress::XPRSProblem::nlpGetFormulaRows(int *p_nformulas,
                                            Array<int> const &rowind) {

  check(XPRSnlpgetformularows(ptr, p_nformulas, rowind.data));
}

void xpress::XPRSProblem::nlpLoadFormulas(int nnlpcoefs,
                                          Array<int const> const &rowind,
                                          Array<int const> const &formulastart,
                                          int parsed,
                                          Array<int const> const &type,
                                          Array<double const> const &value) {

  check(XPRSnlploadformulas(ptr, nnlpcoefs, rowind.data, formulastart.data,
                            parsed, type.data, value.data));
}

void xpress::XPRSProblem::nlpDelFormulas(int nformulas,
                                         Array<int const> const &rowind) {

  check(XPRSnlpdelformulas(ptr, nformulas, rowind.data));
}

void xpress::XPRSProblem::nlpGetFormulaStr(int row, char *formula, int maxbytes,
                                           int *p_nbytes) {

  check(XPRSnlpgetformulastr(ptr, row, formula, maxbytes, p_nbytes));
}

void xpress::XPRSProblem::nlpSetInitVal(int nvars,
                                        Array<int const> const &colind,
                                        Array<double const> const &initial) {

  check(XPRSnlpsetinitval(ptr, nvars, colind.data, initial.data));
}

void xpress::XPRSProblem::slpGetCoefFormula(int row, int col, double *p_factor,
                                            int parsed, int maxtypes,
                                            int *p_ntypes,
                                            Array<int> const &type,
                                            Array<double> const &value) {

  check(XPRSslpgetcoefformula(ptr, row, col, p_factor, parsed, maxtypes,
                              p_ntypes, type.data, value.data));
}

void xpress::XPRSProblem::slpGetCoefs(int *p_ncoefs, Array<int> const &rowind,
                                      Array<int> const &colind) {

  check(XPRSslpgetcoefs(ptr, p_ncoefs, rowind.data, colind.data));
}

void xpress::XPRSProblem::slpLoadCoefs(
    int ncoefs, Array<int const> const &rowind, Array<int const> const &colind,
    Array<double const> const &factor, Array<int const> const &formulastart,
    int parsed, Array<int const> const &type, Array<double const> const &coef) {

  check(XPRSslploadcoefs(ptr, ncoefs, rowind.data, colind.data, factor.data,
                         formulastart.data, parsed, type.data, coef.data));
}

void xpress::XPRSProblem::slpDelCoefs(int ncoefs,
                                      Array<int const> const &rowind,
                                      Array<int const> const &colind) {

  check(XPRSslpdelcoefs(ptr, ncoefs, rowind.data, colind.data));
}

void xpress::XPRSProblem::slpGetCoefStr(int row, int col, double *p_factor,
                                        char *formula, int maxbytes,
                                        int *p_nbytes) {

  check(
      XPRSslpgetcoefstr(ptr, row, col, p_factor, formula, maxbytes, p_nbytes));
}

void xpress::XPRSProblem::slpSetDetRow(int nvars,
                                       Array<int const> const &colind,
                                       Array<int const> const &rowind) {

  check(XPRSslpsetdetrow(ptr, nvars, colind.data, rowind.data));
}

void xpress::XPRSProblem::slpAddCoefs(int ncoefs,
                                      Array<int const> const &rowind,
                                      Array<int const> const &colind,
                                      Array<double const> const &factor,
                                      Array<int const> const &formulastart,
                                      int parsed, Array<int const> const &type,
                                      Array<double const> const &value) {

  check(XPRSslpaddcoefs(ptr, ncoefs, rowind.data, colind.data, factor.data,
                        formulastart.data, parsed, type.data, value.data));
}

void xpress::XPRSProblem::slpChgCoefStr(
    int row, int col, double *factor,
    std::optional<std::string> const &formula) {

  check(XPRSslpchgcoefstr(ptr, row, col, factor,
                          formula ? formula.value().c_str() : nullptr));
}

void xpress::XPRSProblem::slpChgCoef(int row, int col, double *factor,
                                     int parsed, Array<int const> const &type,
                                     Array<double const> const &value) {

  check(XPRSslpchgcoef(ptr, row, col, factor, parsed, type.data, value.data));
}

void xpress::XPRSProblem::slpCascadeSol() { check(XPRSslpcascade(ptr)); }

void xpress::XPRSProblem::slpCascadeOrder() { check(XPRSslpcascadeorder(ptr)); }

auto xpress::XPRSProblem::slpChgRowStatus(int row) -> int {

  int status;
  check(XPRSslpchgrowstatus(ptr, row, &status));
  return status;
}

void xpress::XPRSProblem::slpChgRowWt(int row, double *weight) {

  check(XPRSslpchgrowwt(ptr, row, weight));
}

void xpress::XPRSProblem::slpChgDeltaType(int nvars,
                                          Array<int const> const &varind,
                                          Array<int const> const &deltatypes,
                                          Array<double const> const &values) {

  check(XPRSslpchgdeltatype(ptr, nvars, varind.data, deltatypes.data,
                            values.data));
}

void xpress::XPRSProblem::slpChgCascadeNLimit(int col, int limit) {

  check(XPRSslpchgcascadenlimit(ptr, col, limit));
}

void xpress::XPRSProblem::slpConstruct() { check(XPRSslpconstruct(ptr)); }

void xpress::XPRSProblem::slpGetRowStatus(int row, int *p_status) {

  check(XPRSslpgetrowstatus(ptr, row, p_status));
}

auto xpress::XPRSProblem::slpGetRowWT(int row) -> double {

  double p_weight;
  check(XPRSslpgetrowwt(ptr, row, &p_weight));
  return p_weight;
}

auto xpress::XPRSProblem::slpEvaluateCoef(int row, int col) -> double {

  double p_value;
  check(XPRSslpevaluatecoef(ptr, row, col, &p_value));
  return p_value;
}

void xpress::XPRSProblem::slpReInitialize() { check(XPRSslpreinitialize(ptr)); }

void xpress::XPRSProblem::slpUnConstruct() { check(XPRSslpunconstruct(ptr)); }

void xpress::XPRSProblem::slpUpdateLinearization() {

  check(XPRSslpupdatelinearization(ptr));
}

void xpress::XPRSProblem::slpFixPenalties(int *p_status) {

  check(XPRSslpfixpenalties(ptr, p_status));
}

void xpress::XPRSProblem::nlpPostsolveProb() { check(XPRSnlppostsolve(ptr)); }

void xpress::XPRSProblem::nlpCalcSlacks(Array<double const> const &solution,
                                        Array<double> const &slack) {

  check(XPRSnlpcalcslacks(ptr, solution.data, slack.data));
}

auto xpress::XPRSProblem::addLplogCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeLplogCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::LplogCallbackHolder> wrapper =
      std::make_unique<xpress::LplogCallbackHolder>(this, callback);
  check(XPRSaddcblplog(ptr, xpress::LplogCallbackHolder::wrapLplogCallback,
                       wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeLplogCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeLplogCallbacks();
  else
    check(XPRSremovecblplog(ptr, LplogCallbackHolder::wrapLplogCallback,
                            callback.getHolder()));
}
void xpress::XPRSProblem::removeLplogCallbacks() {
  check(XPRSremovecblplog(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addMiplogCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMiplogCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::MiplogCallbackHolder> wrapper =
      std::make_unique<xpress::MiplogCallbackHolder>(this, callback);
  check(XPRSaddcbmiplog(ptr, xpress::MiplogCallbackHolder::wrapMiplogCallback,
                        wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeMiplogCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMiplogCallbacks();
  else
    check(XPRSremovecbmiplog(ptr, MiplogCallbackHolder::wrapMiplogCallback,
                             callback.getHolder()));
}
void xpress::XPRSProblem::removeMiplogCallbacks() {
  check(XPRSremovecbmiplog(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addCutlogCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeCutlogCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::CutlogCallbackHolder> wrapper =
      std::make_unique<xpress::CutlogCallbackHolder>(this, callback);
  check(XPRSaddcbcutlog(ptr, xpress::CutlogCallbackHolder::wrapCutlogCallback,
                        wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeCutlogCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeCutlogCallbacks();
  else
    check(XPRSremovecbcutlog(ptr, CutlogCallbackHolder::wrapCutlogCallback,
                             callback.getHolder()));
}
void xpress::XPRSProblem::removeCutlogCallbacks() {
  check(XPRSremovecbcutlog(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addBarlogCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeBarlogCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::BarlogCallbackHolder> wrapper =
      std::make_unique<xpress::BarlogCallbackHolder>(this, callback);
  check(XPRSaddcbbarlog(ptr, xpress::BarlogCallbackHolder::wrapBarlogCallback,
                        wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeBarlogCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeBarlogCallbacks();
  else
    check(XPRSremovecbbarlog(ptr, BarlogCallbackHolder::wrapBarlogCallback,
                             callback.getHolder()));
}
void xpress::XPRSProblem::removeBarlogCallbacks() {
  check(XPRSremovecbbarlog(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addOptnodeCallback(
    std::function<void(xpress::XPRSProblem &, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeOptnodeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::OptnodeCallbackHolder> wrapper =
      std::make_unique<xpress::OptnodeCallbackHolder>(this, callback);
  check(XPRSaddcboptnode(ptr,
                         xpress::OptnodeCallbackHolder::wrapOptnodeCallback,
                         wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeOptnodeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeOptnodeCallbacks();
  else
    check(XPRSremovecboptnode(ptr, OptnodeCallbackHolder::wrapOptnodeCallback,
                              callback.getHolder()));
}
void xpress::XPRSProblem::removeOptnodeCallbacks() {
  check(XPRSremovecboptnode(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addPrenodeCallback(
    std::function<void(xpress::XPRSProblem &, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removePrenodeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::PrenodeCallbackHolder> wrapper =
      std::make_unique<xpress::PrenodeCallbackHolder>(this, callback);
  check(XPRSaddcbprenode(ptr,
                         xpress::PrenodeCallbackHolder::wrapPrenodeCallback,
                         wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removePrenodeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removePrenodeCallbacks();
  else
    check(XPRSremovecbprenode(ptr, PrenodeCallbackHolder::wrapPrenodeCallback,
                              callback.getHolder()));
}
void xpress::XPRSProblem::removePrenodeCallbacks() {
  check(XPRSremovecbprenode(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addInfnodeCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeInfnodeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::InfnodeCallbackHolder> wrapper =
      std::make_unique<xpress::InfnodeCallbackHolder>(this, callback);
  check(XPRSaddcbinfnode(ptr,
                         xpress::InfnodeCallbackHolder::wrapInfnodeCallback,
                         wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeInfnodeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeInfnodeCallbacks();
  else
    check(XPRSremovecbinfnode(ptr, InfnodeCallbackHolder::wrapInfnodeCallback,
                              callback.getHolder()));
}
void xpress::XPRSProblem::removeInfnodeCallbacks() {
  check(XPRSremovecbinfnode(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addNodecutoffCallback(
    std::function<void(xpress::XPRSProblem &, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeNodecutoffCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::NodecutoffCallbackHolder> wrapper =
      std::make_unique<xpress::NodecutoffCallbackHolder>(this, callback);
  check(XPRSaddcbnodecutoff(
      ptr, xpress::NodecutoffCallbackHolder::wrapNodecutoffCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeNodecutoffCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeNodecutoffCallbacks();
  else
    check(XPRSremovecbnodecutoff(
        ptr, NodecutoffCallbackHolder::wrapNodecutoffCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeNodecutoffCallbacks() {
  check(XPRSremovecbnodecutoff(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addIntsolCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeIntsolCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::IntsolCallbackHolder> wrapper =
      std::make_unique<xpress::IntsolCallbackHolder>(this, callback);
  check(XPRSaddcbintsol(ptr, xpress::IntsolCallbackHolder::wrapIntsolCallback,
                        wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeIntsolCallback(CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeIntsolCallbacks();
  else
    check(XPRSremovecbintsol(ptr, IntsolCallbackHolder::wrapIntsolCallback,
                             callback.getHolder()));
}
void xpress::XPRSProblem::removeIntsolCallbacks() {
  check(XPRSremovecbintsol(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addPreIntsolCallback(
    std::function<void(xpress::XPRSProblem &, int, int *, double *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removePreIntsolCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::PreIntsolCallbackHolder> wrapper =
      std::make_unique<xpress::PreIntsolCallbackHolder>(this, callback);
  check(XPRSaddcbpreintsol(
      ptr, xpress::PreIntsolCallbackHolder::wrapPreIntsolCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removePreIntsolCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removePreIntsolCallbacks();
  else
    check(XPRSremovecbpreintsol(ptr,
                                PreIntsolCallbackHolder::wrapPreIntsolCallback,
                                callback.getHolder()));
}
void xpress::XPRSProblem::removePreIntsolCallbacks() {
  check(XPRSremovecbpreintsol(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addMessageCallback(
    std::function<void(xpress::XPRSProblem &, char const *, int, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMessageCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::MessageCallbackHolder> wrapper =
      std::make_unique<xpress::MessageCallbackHolder>(this, callback);
  check(XPRSaddcbmessage(ptr,
                         xpress::MessageCallbackHolder::wrapMessageCallback,
                         wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeMessageCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMessageCallbacks();
  else
    check(XPRSremovecbmessage(ptr, MessageCallbackHolder::wrapMessageCallback,
                              callback.getHolder()));
}
void xpress::XPRSProblem::removeMessageCallbacks() {
  check(XPRSremovecbmessage(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addMipThreadCallback(
    std::function<void(xpress::XPRSProblem &, xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMipThreadCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::MipThreadCallbackHolder> wrapper =
      std::make_unique<xpress::MipThreadCallbackHolder>(this, callback);
  check(XPRSaddcbmipthread(
      ptr, xpress::MipThreadCallbackHolder::wrapMipThreadCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeMipThreadCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMipThreadCallbacks();
  else
    check(XPRSremovecbmipthread(ptr,
                                MipThreadCallbackHolder::wrapMipThreadCallback,
                                callback.getHolder()));
}
void xpress::XPRSProblem::removeMipThreadCallbacks() {
  check(XPRSremovecbmipthread(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addMipThreadDestroyCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMipThreadDestroyCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::MipThreadDestroyCallbackHolder> wrapper =
      std::make_unique<xpress::MipThreadDestroyCallbackHolder>(this, callback);
  check(XPRSaddcbdestroymt(
      ptr, xpress::MipThreadDestroyCallbackHolder::wrapMipThreadDestroyCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeMipThreadDestroyCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMipThreadDestroyCallbacks();
  else
    check(XPRSremovecbdestroymt(
        ptr, MipThreadDestroyCallbackHolder::wrapMipThreadDestroyCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeMipThreadDestroyCallbacks() {
  check(XPRSremovecbdestroymt(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addNewnodeCallback(
    std::function<void(xpress::XPRSProblem &, int, int, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeNewnodeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::NewnodeCallbackHolder> wrapper =
      std::make_unique<xpress::NewnodeCallbackHolder>(this, callback);
  check(XPRSaddcbnewnode(ptr,
                         xpress::NewnodeCallbackHolder::wrapNewnodeCallback,
                         wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeNewnodeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeNewnodeCallbacks();
  else
    check(XPRSremovecbnewnode(ptr, NewnodeCallbackHolder::wrapNewnodeCallback,
                              callback.getHolder()));
}
void xpress::XPRSProblem::removeNewnodeCallbacks() {
  check(XPRSremovecbnewnode(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addBarIterationCallback(
    std::function<void(xpress::XPRSProblem &, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeBarIterationCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::BarIterationCallbackHolder> wrapper =
      std::make_unique<xpress::BarIterationCallbackHolder>(this, callback);
  check(XPRSaddcbbariteration(
      ptr, xpress::BarIterationCallbackHolder::wrapBarIterationCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeBarIterationCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeBarIterationCallbacks();
  else
    check(XPRSremovecbbariteration(
        ptr, BarIterationCallbackHolder::wrapBarIterationCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeBarIterationCallbacks() {
  check(XPRSremovecbbariteration(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addPresolveCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removePresolveCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::PresolveCallbackHolder> wrapper =
      std::make_unique<xpress::PresolveCallbackHolder>(this, callback);
  check(XPRSaddcbpresolve(ptr,
                          xpress::PresolveCallbackHolder::wrapPresolveCallback,
                          wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removePresolveCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removePresolveCallbacks();
  else
    check(XPRSremovecbpresolve(ptr,
                               PresolveCallbackHolder::wrapPresolveCallback,
                               callback.getHolder()));
}
void xpress::XPRSProblem::removePresolveCallbacks() {
  check(XPRSremovecbpresolve(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addChangeBranchObjectCallback(
    std::function<void(xpress::XPRSProblem &, BranchObject *, BranchObject **)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeChangeBranchObjectCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::ChangeBranchObjectCallbackHolder> wrapper =
      std::make_unique<xpress::ChangeBranchObjectCallbackHolder>(this,
                                                                 callback);
  check(XPRSaddcbchgbranchobject(
      ptr,
      xpress::ChangeBranchObjectCallbackHolder::wrapChangeBranchObjectCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeChangeBranchObjectCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeChangeBranchObjectCallbacks();
  else
    check(XPRSremovecbchgbranchobject(
        ptr, ChangeBranchObjectCallbackHolder::wrapChangeBranchObjectCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeChangeBranchObjectCallbacks() {
  check(XPRSremovecbchgbranchobject(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addComputeRestartCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeComputeRestartCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::ComputeRestartCallbackHolder> wrapper =
      std::make_unique<xpress::ComputeRestartCallbackHolder>(this, callback);
  check(XPRSaddcbcomputerestart(
      ptr, xpress::ComputeRestartCallbackHolder::wrapComputeRestartCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeComputeRestartCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeComputeRestartCallbacks();
  else
    check(XPRSremovecbcomputerestart(
        ptr, ComputeRestartCallbackHolder::wrapComputeRestartCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeComputeRestartCallbacks() {
  check(XPRSremovecbcomputerestart(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addNodeLPSolvedCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeNodeLPSolvedCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::NodeLPSolvedCallbackHolder> wrapper =
      std::make_unique<xpress::NodeLPSolvedCallbackHolder>(this, callback);
  check(XPRSaddcbnodelpsolved(
      ptr, xpress::NodeLPSolvedCallbackHolder::wrapNodeLPSolvedCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeNodeLPSolvedCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeNodeLPSolvedCallbacks();
  else
    check(XPRSremovecbnodelpsolved(
        ptr, NodeLPSolvedCallbackHolder::wrapNodeLPSolvedCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeNodeLPSolvedCallbacks() {
  check(XPRSremovecbnodelpsolved(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addGapNotifyCallback(
    std::function<void(xpress::XPRSProblem &, double *, double *, double *,
                       double *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeGapNotifyCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::GapNotifyCallbackHolder> wrapper =
      std::make_unique<xpress::GapNotifyCallbackHolder>(this, callback);
  check(XPRSaddcbgapnotify(
      ptr, xpress::GapNotifyCallbackHolder::wrapGapNotifyCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeGapNotifyCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeGapNotifyCallbacks();
  else
    check(XPRSremovecbgapnotify(ptr,
                                GapNotifyCallbackHolder::wrapGapNotifyCallback,
                                callback.getHolder()));
}
void xpress::XPRSProblem::removeGapNotifyCallbacks() {
  check(XPRSremovecbgapnotify(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addUserSolNotifyCallback(
    std::function<void(xpress::XPRSProblem &, char const *, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeUserSolNotifyCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::UserSolNotifyCallbackHolder> wrapper =
      std::make_unique<xpress::UserSolNotifyCallbackHolder>(this, callback);
  check(XPRSaddcbusersolnotify(
      ptr, xpress::UserSolNotifyCallbackHolder::wrapUserSolNotifyCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeUserSolNotifyCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeUserSolNotifyCallbacks();
  else
    check(XPRSremovecbusersolnotify(
        ptr, UserSolNotifyCallbackHolder::wrapUserSolNotifyCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeUserSolNotifyCallbacks() {
  check(XPRSremovecbusersolnotify(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addBeforeSolveCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeBeforeSolveCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::BeforeSolveCallbackHolder> wrapper =
      std::make_unique<xpress::BeforeSolveCallbackHolder>(this, callback);
  check(XPRSaddcbbeforesolve(
      ptr, xpress::BeforeSolveCallbackHolder::wrapBeforeSolveCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeBeforeSolveCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeBeforeSolveCallbacks();
  else
    check(XPRSremovecbbeforesolve(
        ptr, BeforeSolveCallbackHolder::wrapBeforeSolveCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeBeforeSolveCallbacks() {
  check(XPRSremovecbbeforesolve(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addBeforeObjectiveCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeBeforeObjectiveCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::BeforeObjectiveCallbackHolder> wrapper =
      std::make_unique<xpress::BeforeObjectiveCallbackHolder>(this, callback);
  check(XPRSaddcbbeforeobjective(
      ptr, xpress::BeforeObjectiveCallbackHolder::wrapBeforeObjectiveCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeBeforeObjectiveCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeBeforeObjectiveCallbacks();
  else
    check(XPRSremovecbbeforeobjective(
        ptr, BeforeObjectiveCallbackHolder::wrapBeforeObjectiveCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeBeforeObjectiveCallbacks() {
  check(XPRSremovecbbeforeobjective(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addAfterObjectiveCallback(
    std::function<void(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeAfterObjectiveCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::AfterObjectiveCallbackHolder> wrapper =
      std::make_unique<xpress::AfterObjectiveCallbackHolder>(this, callback);
  check(XPRSaddcbafterobjective(
      ptr, xpress::AfterObjectiveCallbackHolder::wrapAfterObjectiveCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeAfterObjectiveCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeAfterObjectiveCallbacks();
  else
    check(XPRSremovecbafterobjective(
        ptr, AfterObjectiveCallbackHolder::wrapAfterObjectiveCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeAfterObjectiveCallbacks() {
  check(XPRSremovecbafterobjective(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addCheckTimeCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeCheckTimeCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::CheckTimeCallbackHolder> wrapper =
      std::make_unique<xpress::CheckTimeCallbackHolder>(this, callback);
  check(XPRSaddcbchecktime(
      ptr, xpress::CheckTimeCallbackHolder::wrapCheckTimeCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeCheckTimeCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeCheckTimeCallbacks();
  else
    check(XPRSremovecbchecktime(ptr,
                                CheckTimeCallbackHolder::wrapCheckTimeCallback,
                                callback.getHolder()));
}
void xpress::XPRSProblem::removeCheckTimeCallbacks() {
  check(XPRSremovecbchecktime(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addCutRoundCallback(
    std::function<void(xpress::XPRSProblem &, int, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeCutRoundCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::CutRoundCallbackHolder> wrapper =
      std::make_unique<xpress::CutRoundCallbackHolder>(this, callback);
  check(XPRSaddcbcutround(ptr,
                          xpress::CutRoundCallbackHolder::wrapCutRoundCallback,
                          wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeCutRoundCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeCutRoundCallbacks();
  else
    check(XPRSremovecbcutround(ptr,
                               CutRoundCallbackHolder::wrapCutRoundCallback,
                               callback.getHolder()));
}
void xpress::XPRSProblem::removeCutRoundCallbacks() {
  check(XPRSremovecbcutround(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpCascadeEndCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpCascadeEndCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpCascadeEndCallbackHolder> wrapper =
      std::make_unique<xpress::SlpCascadeEndCallbackHolder>(this, callback);
  check(XPRSaddcbslpcascadeend(
      ptr, xpress::SlpCascadeEndCallbackHolder::wrapSlpCascadeEndCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpCascadeEndCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpCascadeEndCallbacks();
  else
    check(XPRSremovecbslpcascadeend(
        ptr, SlpCascadeEndCallbackHolder::wrapSlpCascadeEndCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpCascadeEndCallbacks() {
  check(XPRSremovecbslpcascadeend(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpCascadeStartCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpCascadeStartCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpCascadeStartCallbackHolder> wrapper =
      std::make_unique<xpress::SlpCascadeStartCallbackHolder>(this, callback);
  check(XPRSaddcbslpcascadestart(
      ptr, xpress::SlpCascadeStartCallbackHolder::wrapSlpCascadeStartCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpCascadeStartCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpCascadeStartCallbacks();
  else
    check(XPRSremovecbslpcascadestart(
        ptr, SlpCascadeStartCallbackHolder::wrapSlpCascadeStartCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpCascadeStartCallbacks() {
  check(XPRSremovecbslpcascadestart(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpCascadeVarCallback(
    std::function<int(xpress::XPRSProblem &, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpCascadeVarCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpCascadeVarCallbackHolder> wrapper =
      std::make_unique<xpress::SlpCascadeVarCallbackHolder>(this, callback);
  check(XPRSaddcbslpcascadevar(
      ptr, xpress::SlpCascadeVarCallbackHolder::wrapSlpCascadeVarCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpCascadeVarCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpCascadeVarCallbacks();
  else
    check(XPRSremovecbslpcascadevar(
        ptr, SlpCascadeVarCallbackHolder::wrapSlpCascadeVarCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpCascadeVarCallbacks() {
  check(XPRSremovecbslpcascadevar(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpCascadeVarFailCallback(
    std::function<int(xpress::XPRSProblem &, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpCascadeVarFailCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpCascadeVarFailCallbackHolder> wrapper =
      std::make_unique<xpress::SlpCascadeVarFailCallbackHolder>(this, callback);
  check(XPRSaddcbslpcascadevarfail(
      ptr,
      xpress::SlpCascadeVarFailCallbackHolder::wrapSlpCascadeVarFailCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpCascadeVarFailCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpCascadeVarFailCallbacks();
  else
    check(XPRSremovecbslpcascadevarfail(
        ptr, SlpCascadeVarFailCallbackHolder::wrapSlpCascadeVarFailCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpCascadeVarFailCallbacks() {
  check(XPRSremovecbslpcascadevarfail(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpConstructCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpConstructCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpConstructCallbackHolder> wrapper =
      std::make_unique<xpress::SlpConstructCallbackHolder>(this, callback);
  check(XPRSaddcbslpconstruct(
      ptr, xpress::SlpConstructCallbackHolder::wrapSlpConstructCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpConstructCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpConstructCallbacks();
  else
    check(XPRSremovecbslpconstruct(
        ptr, SlpConstructCallbackHolder::wrapSlpConstructCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpConstructCallbacks() {
  check(XPRSremovecbslpconstruct(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpIntSolCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpIntSolCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpIntSolCallbackHolder> wrapper =
      std::make_unique<xpress::SlpIntSolCallbackHolder>(this, callback);
  check(XPRSaddcbslpintsol(
      ptr, xpress::SlpIntSolCallbackHolder::wrapSlpIntSolCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpIntSolCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpIntSolCallbacks();
  else
    check(XPRSremovecbslpintsol(ptr,
                                SlpIntSolCallbackHolder::wrapSlpIntSolCallback,
                                callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpIntSolCallbacks() {
  check(XPRSremovecbslpintsol(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpIterEndCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpIterEndCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpIterEndCallbackHolder> wrapper =
      std::make_unique<xpress::SlpIterEndCallbackHolder>(this, callback);
  check(XPRSaddcbslpiterend(
      ptr, xpress::SlpIterEndCallbackHolder::wrapSlpIterEndCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpIterEndCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpIterEndCallbacks();
  else
    check(XPRSremovecbslpiterend(
        ptr, SlpIterEndCallbackHolder::wrapSlpIterEndCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpIterEndCallbacks() {
  check(XPRSremovecbslpiterend(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpIterStartCallback(
    std::function<int(xpress::XPRSProblem &)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpIterStartCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpIterStartCallbackHolder> wrapper =
      std::make_unique<xpress::SlpIterStartCallbackHolder>(this, callback);
  check(XPRSaddcbslpiterstart(
      ptr, xpress::SlpIterStartCallbackHolder::wrapSlpIterStartCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpIterStartCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpIterStartCallbacks();
  else
    check(XPRSremovecbslpiterstart(
        ptr, SlpIterStartCallbackHolder::wrapSlpIterStartCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpIterStartCallbacks() {
  check(XPRSremovecbslpiterstart(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpIterVarCallback(
    std::function<int(xpress::XPRSProblem &, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpIterVarCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpIterVarCallbackHolder> wrapper =
      std::make_unique<xpress::SlpIterVarCallbackHolder>(this, callback);
  check(XPRSaddcbslpitervar(
      ptr, xpress::SlpIterVarCallbackHolder::wrapSlpIterVarCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpIterVarCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpIterVarCallbacks();
  else
    check(XPRSremovecbslpitervar(
        ptr, SlpIterVarCallbackHolder::wrapSlpIterVarCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpIterVarCallbacks() {
  check(XPRSremovecbslpitervar(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpDrColCallback(
    std::function<int(xpress::XPRSProblem &, int, int, double, double *, double,
                      double)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpDrColCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpDrColCallbackHolder> wrapper =
      std::make_unique<xpress::SlpDrColCallbackHolder>(this, callback);
  check(XPRSaddcbslpdrcol(ptr,
                          xpress::SlpDrColCallbackHolder::wrapSlpDrColCallback,
                          wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpDrColCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpDrColCallbacks();
  else
    check(XPRSremovecbslpdrcol(ptr,
                               SlpDrColCallbackHolder::wrapSlpDrColCallback,
                               callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpDrColCallbacks() {
  check(XPRSremovecbslpdrcol(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addMsJobStartCallback(
    std::function<int(xpress::XPRSProblem &, void *, char const *, int *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMsJobStartCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::MsJobStartCallbackHolder> wrapper =
      std::make_unique<xpress::MsJobStartCallbackHolder>(this, callback);
  check(XPRSaddcbmsjobstart(
      ptr, xpress::MsJobStartCallbackHolder::wrapMsJobStartCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeMsJobStartCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMsJobStartCallbacks();
  else
    check(XPRSremovecbmsjobstart(
        ptr, MsJobStartCallbackHolder::wrapMsJobStartCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeMsJobStartCallbacks() {
  check(XPRSremovecbmsjobstart(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addMsJobEndCallback(
    std::function<int(xpress::XPRSProblem &, void *, char const *, int *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMsJobEndCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::MsJobEndCallbackHolder> wrapper =
      std::make_unique<xpress::MsJobEndCallbackHolder>(this, callback);
  check(XPRSaddcbmsjobend(ptr,
                          xpress::MsJobEndCallbackHolder::wrapMsJobEndCallback,
                          wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeMsJobEndCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMsJobEndCallbacks();
  else
    check(XPRSremovecbmsjobend(ptr,
                               MsJobEndCallbackHolder::wrapMsJobEndCallback,
                               callback.getHolder()));
}
void xpress::XPRSProblem::removeMsJobEndCallbacks() {
  check(XPRSremovecbmsjobend(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addMsWinnerCallback(
    std::function<int(xpress::XPRSProblem &, void *, char const *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMsWinnerCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::MsWinnerCallbackHolder> wrapper =
      std::make_unique<xpress::MsWinnerCallbackHolder>(this, callback);
  check(XPRSaddcbmswinner(ptr,
                          xpress::MsWinnerCallbackHolder::wrapMsWinnerCallback,
                          wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeMsWinnerCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMsWinnerCallbacks();
  else
    check(XPRSremovecbmswinner(ptr,
                               MsWinnerCallbackHolder::wrapMsWinnerCallback,
                               callback.getHolder()));
}
void xpress::XPRSProblem::removeMsWinnerCallbacks() {
  check(XPRSremovecbmswinner(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addNlpCoefEvalErrorCallback(
    std::function<int(xpress::XPRSProblem &, int, int)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeNlpCoefEvalErrorCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::NlpCoefEvalErrorCallbackHolder> wrapper =
      std::make_unique<xpress::NlpCoefEvalErrorCallbackHolder>(this, callback);
  check(XPRSaddcbnlpcoefevalerror(
      ptr, xpress::NlpCoefEvalErrorCallbackHolder::wrapNlpCoefEvalErrorCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeNlpCoefEvalErrorCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeNlpCoefEvalErrorCallbacks();
  else
    check(XPRSremovecbnlpcoefevalerror(
        ptr, NlpCoefEvalErrorCallbackHolder::wrapNlpCoefEvalErrorCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeNlpCoefEvalErrorCallbacks() {
  check(XPRSremovecbnlpcoefevalerror(ptr, nullptr, nullptr));
}
auto xpress::XPRSProblem::addSlpPreUpdateLinearizationCallback(
    std::function<int(xpress::XPRSProblem &, int *)> callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeSlpPreUpdateLinearizationCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::SlpPreUpdateLinearizationCallbackHolder> wrapper =
      std::make_unique<xpress::SlpPreUpdateLinearizationCallbackHolder>(
          this, callback);
  check(XPRSaddcbslppreupdatelinearization(
      ptr,
      xpress::SlpPreUpdateLinearizationCallbackHolder::
          wrapSlpPreUpdateLinearizationCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::XPRSProblem::removeSlpPreUpdateLinearizationCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeSlpPreUpdateLinearizationCallbacks();
  else
    check(XPRSremovecbslppreupdatelinearization(
        ptr,
        SlpPreUpdateLinearizationCallbackHolder::
            wrapSlpPreUpdateLinearizationCallback,
        callback.getHolder()));
}
void xpress::XPRSProblem::removeSlpPreUpdateLinearizationCallbacks() {
  check(XPRSremovecbslppreupdatelinearization(ptr, nullptr, nullptr));
}
namespace xpress {

;

;

;
xpress::XPRSProblem::GeneralConstraintInfo::GeneralConstraintInfo()
    : GeneralConstraintInfo(std::vector<xpress::GenConsType>(0),
                            std::vector<int>(0), std::vector<int>(0),
                            std::vector<int>(0), 0, 0) {}
xpress::XPRSProblem::GeneralConstraintInfo::GeneralConstraintInfo(
    xpress::SizedArray<xpress::GenConsType const> const &type,
    xpress::SizedArray<int const> const &resultant,
    xpress::SizedArray<int const> const &colstart,
    xpress::SizedArray<int const> const &valstart, int ncols, int nvals)
    : resultant(resultant.toVector()), type(type.toVector()),
      colstart(colstart.toVector()), colind(std::vector<int>(ncols)),
      valstart(valstart.toVector()), val(std::vector<double>(nvals)) {}

auto xpress::XPRSProblem::getGenCons(int first, int last)
    -> xpress::XPRSProblem::GeneralConstraintInfo {
  int count = last - first + 1;
  if (count <= 0)
    return GeneralConstraintInfo();
  int ncols;
  int nvals;
  std::vector<xpress::GenConsType> type(count);
  std::vector<int> resultant(count);
  std::vector<int> colstart(count + 1);
  std::vector<int> valstart(count + 1);
  getGenCons(type, resultant, colstart, nullptr, 0, &ncols, valstart, nullptr,
             0, &nvals, first, last);
  GeneralConstraintInfo ret =
      GeneralConstraintInfo(type, resultant, colstart, valstart, ncols, nvals);
  getGenCons(ret.type, ret.resultant, ret.colstart, ret.colind, ncols, &ncols,
             ret.valstart, ret.val, nvals, &nvals, first, last);
  return ret;
}

auto xpress::XPRSProblem::delIndicator(int row) -> void {
  delIndicators(row, row);
}

auto xpress::XPRSProblem::getIndicators(int first, int last)
    -> std::vector<xpress::IndicatorInfo> {
  std::vector<int> colind(std::max(0, last - first + 1));
  std::vector<int> complement(std::max(0, last - first + 1));
  getIndicators(colind, complement, first, last);
  std::vector<xpress::IndicatorInfo> ret;
  for (int i = 0; i < last - first + 1; ++i) {
    if (complement[i] > 0)
      ret.push_back(xpress::IndicatorInfo(colind[i], false, first + i));
    else if (complement[i] < 0)
      ret.push_back(xpress::IndicatorInfo(colind[i], true, first + i));
  }
  return ret;
}

auto xpress::XPRSProblem::getIndicator(int row)
    -> std::optional<xpress::IndicatorInfo> {
  std::vector<xpress::IndicatorInfo> d = getIndicators(row, row);
  if (xpress::toInt(d.size()) > 0) {
    return d[0];
  }
  return std::nullopt;
}

auto xpress::XPRSProblem::setIndicator(int rowind, int colind,
                                       int complement) -> void {
  setIndicators(1, std::vector<int>({rowind}), std::vector<int>({colind}),
                std::vector<int>({complement}));
}

xpress::XPRSProblem::InvalidStateException::InvalidStateException(
    char const *message)
    : std::runtime_error(message) {}

xpress::XPRSProblem::CannotReferenceVariablesException::
    CannotReferenceVariablesException()
    : InvalidStateException(
          "Cannot reference Variable instances when querying or modifying a "
          "presolved problem or from within a callback"){}

      ;
auto xpress::XPRSProblem::CannotPerformOperationException::mkMsg(
    char *buffer, char const *op, char const *cls) -> char const * {
  ::snprintf(buffer, sizeof(msgBuffer) - 1,
             "cannot %s based on %s for presolved model or from callback", op,
             cls);
  buffer[sizeof(msgBuffer) - 1] = '\0';
  return buffer;
}
auto xpress::XPRSProblem::CannotPerformOperationException::mkMsg(
    char *buffer, char const *msg) -> char const * {
#ifdef _WIN32
  ::strncpy_s(buffer, sizeof(msgBuffer) - 1, msg, _TRUNCATE);
#else
  ::strncpy(buffer, msg, sizeof(msgBuffer) - 1);
#endif
  buffer[sizeof(msgBuffer) - 1] = '\0';
  return buffer;
}

xpress::XPRSProblem::CannotPerformOperationException::
    CannotPerformOperationException(char const *op, char const *cls)
    : InvalidStateException(mkMsg(msgBuffer, op, cls)) {}

xpress::XPRSProblem::CannotPerformOperationException::
    CannotPerformOperationException(char const *msg)
    : InvalidStateException(mkMsg(msgBuffer, msg)){}

      ;

;

;

;

xpress::XPRSProblem::MIPEntityInfo::MIPEntityInfo() : MIPEntityInfo(0, 0, 0) {}

xpress::XPRSProblem::MIPEntityInfo::MIPEntityInfo(int ngents, int nsets,
                                                  int setmembers)
    : coltype(std::vector<char>(ngents)), colind(std::vector<int>(ngents)),
      limit(std::vector<double>(ngents)), settype(std::vector<char>(nsets)),
      setstart(std::vector<int>(nsets + 1)),
      setind(std::vector<int>(setmembers)),
      refval(std::vector<double>(setmembers)) {}

auto xpress::XPRSProblem::getMipEntities()
    -> xpress::XPRSProblem::MIPEntityInfo {
  int ngents;
  int nsets;
  getMipEntities(&ngents, &nsets, nullptr, static_cast<int *>(nullptr), nullptr,
                 nullptr, static_cast<int *>(nullptr),
                 static_cast<int *>(nullptr), nullptr);
  if (ngents <= 0 && nsets <= 0)
    return MIPEntityInfo();
  if (attributes.getSetMembers() > std::numeric_limits<int>::max())
    throw std::runtime_error("cannot handle 64bit set members");
  MIPEntityInfo ret =
      MIPEntityInfo(ngents, nsets, (int)attributes.getSetMembers());

  getMipEntities(&ngents, &nsets, ret.coltype, ret.colind, ret.limit,
                 ret.settype, ret.setstart, ret.setind, ret.refval);
  return ret;
}

auto xpress::XPRSProblem::getDiscreteCols()
    -> xpress::XPRSProblem::MIPEntityInfo {
  int ngents;
  int nsets;
  getMipEntities(&ngents, &nsets, nullptr, static_cast<int *>(nullptr), nullptr,
                 nullptr, static_cast<int *>(nullptr),
                 static_cast<int *>(nullptr), nullptr);
  if (ngents <= 0)
    return MIPEntityInfo();
  MIPEntityInfo ret = MIPEntityInfo(ngents, 0, 0);
  getMipEntities(&ngents, &nsets, ret.coltype, ret.colind, ret.limit, nullptr,
                 static_cast<int *>(nullptr), static_cast<int *>(nullptr),
                 nullptr);
  return ret;
}

auto xpress::XPRSProblem::getSetDefinitions()
    -> xpress::XPRSProblem::MIPEntityInfo {
  if (attributes.getSetMembers() > std::numeric_limits<int>::max())
    throw std::runtime_error("cannot handle 64bit set members");
  int ngents;
  int nsets;
  getMipEntities(&ngents, &nsets, nullptr, static_cast<int *>(nullptr), nullptr,
                 nullptr, static_cast<int *>(nullptr),
                 static_cast<int *>(nullptr), nullptr);
  if (nsets <= 0)
    return MIPEntityInfo();
  MIPEntityInfo ret = MIPEntityInfo(0, nsets, (int)attributes.getSetMembers());
  getMipEntities(&ngents, &nsets, nullptr, nullptr, nullptr, ret.settype,
                 ret.setstart, ret.setind, ret.refval);
  return ret;
}

auto xpress::XPRSProblem::addSets(
    int count, xpress::SizedArray<int const> const &start,
    xpress::SizedArray<SetType const> const &type,
    xpress::SizedArray<int const> const &setind,
    xpress::SizedArray<double const> const &setref,
    xpress::SizedArray<std::optional<std::string> const> const &names)
    -> std::vector<int> {
  int next = attributes.getSets();
  std::vector<int> idx(count);
  std::vector<char> rawType(count);
  std::vector<double> defaultRefs;
  if (!setref) {
    defaultRefs = std::vector<double>(xpress::toInt(setind.size()));
    for (int s = 0; s < count; ++s) {
      int end = (s == count - 1) ? xpress::toInt(setind.size()) : start[s + 1];
      for (int k = start[s], i = 0; k < end; ++k, ++i)
        defaultRefs[k] = i;
    }
  }
  SizedArray<double const> defaultRefsArray(defaultRefs);
  SizedArray<double const> const &refs = (!setref) ? defaultRefsArray : setref;
  for (int i = 0; i < xpress::toInt(rawType.size()); ++i) {
    idx[i] = next + i;
    rawType[i] = (char)type[i];
  }
  int created = 0;
  {
    xpress::RaiiHelper finally([&]() {
      if (created > 0) {
        delSets(created, idx);
      }
    });

    addSets(count, xpress::toInt(setind.size()), rawType, start, setind, refs);
    created = count;
    if (!(!names)) {
      addNamesWithNull(xpress::Namespaces::Set, next, created, names);
    }
    created = 0;
  }

  return idx;
}

auto xpress::XPRSProblem::addSet(
    xpress::SetType type, xpress::SizedArray<int const> const &elements,
    xpress::SizedArray<double const> const &weights,
    std::optional<std::string> name) -> int {
  std::vector<int> setIdx({-1});
  {
    xpress::RaiiHelper finally([&]() {
      if (setIdx[0] >= 0)
        delSets(1, setIdx);
    });

    if (!(name))
      addSets(1, std::vector<int>({0}), std::vector<xpress::SetType>({type}),
              elements, weights, nullptr);
    else
      addSets(1, std::vector<int>({0}), std::vector<xpress::SetType>({type}),
              elements, weights,
              std::vector<std::optional<std::string>>({name}));
    setIdx[0] = attributes.getSets() - 1;
  }

  return setIdx[0];
}

auto xpress::XPRSProblem::addSets(
    std::vector<xpress::SetType> type,
    xpress::SizedArray<std::vector<int>> const &elements,
    xpress::SizedArray<std::vector<double>> const &weights,
    xpress::SizedArray<std::optional<std::string> const> const &name)
    -> std::vector<int> {
  if (xpress::toInt(type.size()) != xpress::toInt(elements.size()))
    throw std::invalid_argument(
        "arrays type and elements have different length");
  if (!(!weights) &&
      xpress::toInt(elements.size()) != xpress::toInt(weights.size()))
    throw std::invalid_argument(
        "arrays type and weights have different length");
  if (!(!name) && xpress::toInt(elements.size()) != xpress::toInt(name.size()))
    throw std::invalid_argument("arrays type and name have different length");

  int count = xpress::toInt(type.size());
  std::vector<int> ret(count);
  if (count == 0)
    return ret;
  std::vector<int> colind;
  std::vector<double> refval;
  std::vector<int> start;
  std::vector<char> settype(count);
  int first = attributes.getSets();
  for (int i = 0; i < count; ++i) {
    start.push_back(xpress::toInt(colind.size()));
    ret[i] = first + i;
    settype[i] = (char)type[i];
    if (!(!weights)) {
      if (xpress::toInt(elements[i].size()) != xpress::toInt(weights[i].size()))
        throw std::invalid_argument(
            "arrays elements and weights have different length");
      for (int j = 0; j < xpress::toInt(elements[i].size()); ++j) {
        colind.push_back(elements[i][j]);
        refval.push_back(weights[i][j]);
      }
    } else {
      for (int j = 0; j < xpress::toInt(elements[i].size()); ++j) {
        colind.push_back(elements[i][j]);
        refval.push_back(j + 1.0);
      }
    }
  }
  addSets(count, xpress::toInt(colind.size()), settype, start, colind, refval);
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo) {
        delSets(count, ret);
      }
    });

    if (!(!name)) {
      addNamesWithNull(xpress::Namespaces::Set, first,
                       xpress::toInt(settype.size()), name);
    }
    undo = false;
  }

  return ret;
}

template <typename C, typename CIsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::ArrayBuilder<C> builder,
    std::function<void(std::vector<int> const &)> transformResult)
    -> std::vector<int> {
  std::vector<int> indices = xpress::makeArray<int>(builder.getDim(1 - 1));
  ColumnCreator creator =
      ColumnCreator(this, builder.getDim(1 - 1), !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    int nextIndex = attributes.getCols();
    std::function<double(int)> getColumnLB = builder.getLB();
    std::function<double(int)> getColumnUB = builder.getUB();
    std::function<double(int)> getColumnLimit = builder.getLimit();
    std::function<std::optional<std::string>(int)> getColumnName =
        builder.getName();
    std::function<ColumnType(int)> getColumnType = builder.getType();
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1) {
      creator.addColumn(this, !(!getColumnLB) ? getColumnLB(i1) : 0.0,
                        !(!getColumnUB)
                            ? getColumnUB(i1)
                            : std::numeric_limits<double>::infinity(),
                        !(!getColumnType) ? (char)getColumnType(i1) : 'C',
                        !(!getColumnLimit) ? getColumnLimit(i1) : 0.0,
                        !(!getColumnName) ? getColumnName(i1) : std::nullopt);
      indices[i1] = nextIndex++;
    }
    creator.commit(this);

    transformResult(indices);
    undo = false;
  }

  return indices;
}

template <typename C, typename CIsIntegral>
auto xpress::XPRSProblem::buildColumns(VariableBuilder::ArrayBuilder<C> builder)
    -> std::vector<int> {
  return buildColumns(builder, [=](std::vector<int> const &) {});
}

template <typename I, typename C, typename CIsIntegral, typename Func0,
          typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::ArrayBuilder<C> builder, Func0 makeResult,
    std::function<void(I &, C, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(builder, [&](std::vector<int> const &indices) {
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1) {
      addResult(ret, i1, indices[i1]);
    }
  });
  return ret;
}

template <typename K1>
auto xpress::XPRSProblem::buildColumns(VariableBuilder::MapBuilder<K1> builder,
                                       std::function<void(K1, int)> addKey,
                                       std::function<void()> makeResult)
    -> void {
  int next = attributes.getCols();
  int initial = 1;
  xpress::Iterable<K1> iterable1 = builder.getIterable1();
  initial *= std::max(xpress::sizeHint(iterable1), 1);

  ColumnCreator creator = ColumnCreator(this, initial, !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    std::function<double(K1)> getColumnLB = builder.getLB();
    std::function<double(K1)> getColumnUB = builder.getUB();
    std::function<ColumnType(K1)> getColumnType = builder.getType();
    std::function<double(K1)> getColumnLimit = builder.getLimit();
    std::function<std::optional<std::string>(K1)> getColumnName =
        builder.getName();
    std::function<bool(K1)> dataFilter = builder.getFilter();
    for (auto &v1 : iterable1) {
      if (!(!dataFilter) && !dataFilter(v1))
        continue;
      creator.addColumn(this, !(!getColumnLB) ? getColumnLB(v1) : 0.0,
                        !(!getColumnUB)
                            ? getColumnUB(v1)
                            : std::numeric_limits<double>::infinity(),
                        !(!getColumnType) ? (char)getColumnType(v1) : 'C',
                        !(!getColumnLimit) ? getColumnLimit(v1) : 0.0,
                        !(!getColumnName) ? getColumnName(v1) : std::nullopt);
      addKey(v1, next++);
    }
    creator.commit(this);

    makeResult();
    undo = false;
  }
}

template <typename K1>
auto xpress::XPRSProblem::buildColumns(VariableBuilder::MapBuilder<K1> builder)
    -> std::unordered_map<K1, int> {
  std::unordered_map<K1, int> ret = std::unordered_map<K1, int>();
  buildColumns(
      builder,
      [&](K1 k1, int idx) {
        ret[std::unordered_map<K1, int>::key_type(k1)] = idx;
      },
      [=]() {});
  return ret;
}

template <typename I, typename K1, typename Func0, typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::MapBuilder<K1> builder, Func0 makeResult,
    std::function<void(I &, K1, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(
      builder, [&](K1 k1, int idx) { addResult(ret, k1, idx); }, [=]() {});
  return ret;
}

template <typename C, typename CIsIntegral>
auto xpress::XPRSProblem::addColumns(C dim)
    -> VariableBuilder::ColumnArrayBuilder<C> {
  return VariableBuilder::ColumnArrayBuilder<C>(xpress::toInt(dim), this);
}

template <typename Iter0, typename K1, typename Iter0IsIterable>
auto xpress::XPRSProblem::addColumns(Iter0 const &iterable1)
    -> VariableBuilder::ColumnMapBuilder<K1> {
  return VariableBuilder::ColumnMapBuilder<K1>(iterable1, this);
}

template <typename K1>
auto xpress::XPRSProblem::addColumns(xpress::SizedArray<K1 const> const &arr1)
    -> VariableBuilder::ColumnMapBuilder<K1> {
  return VariableBuilder::ColumnMapBuilder<K1>(arr1, this);
}

auto xpress::XPRSProblem::addColumn(double lb, double ub, ColumnType type,
                                    std::optional<std::string> name) -> int {
  std::vector<double> lbs({lb});
  std::vector<double> ubs({ub});
  std::vector<char> types({(char)type});

  std::vector<int> var({-1});
  int idx = attributes.getCols();
  {
    xpress::RaiiHelper finally([&]() {
      if (var[0] != -1)
        delCols(1, var);
    });

    addCols(1, 0, nullptr, SizedArray<int>(nullptr), nullptr, nullptr, lbs,
            ubs);
    var[0] = idx;
    chgColType(1, var, types);

    chgLB(idx, lb);
    chgUB(idx, ub);
    if (!(!(name)))
      addNames(xpress::Namespaces::Column,
               std::vector<std::optional<std::string>>({name}), idx, idx);
    var[0] = -1;
  }

  return idx;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array2Builder<C1, C2> builder,
    std::function<void(std::vector<std::vector<int>> const &)> transformResult)
    -> std::vector<std::vector<int>> {
  std::vector<std::vector<int>> indices =
      xpress::makeArray<int>(builder.getDim(1 - 1), builder.getDim(2 - 1));
  ColumnCreator creator =
      ColumnCreator(this, builder.getDim(1 - 1) + builder.getDim(2 - 1),
                    !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    int nextIndex = attributes.getCols();
    std::function<double(int, int)> getColumnLB = builder.getLB();
    std::function<double(int, int)> getColumnUB = builder.getUB();
    std::function<double(int, int)> getColumnLimit = builder.getLimit();
    std::function<std::optional<std::string>(int, int)> getColumnName =
        builder.getName();
    std::function<ColumnType(int, int)> getColumnType = builder.getType();
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
      for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2) {
        creator.addColumn(
            this, !(!getColumnLB) ? getColumnLB(i1, i2) : 0.0,
            !(!getColumnUB) ? getColumnUB(i1, i2)
                            : std::numeric_limits<double>::infinity(),
            !(!getColumnType) ? (char)getColumnType(i1, i2) : 'C',
            !(!getColumnLimit) ? getColumnLimit(i1, i2) : 0.0,
            !(!getColumnName) ? getColumnName(i1, i2) : std::nullopt);
        indices[i1][i2] = nextIndex++;
      }
    creator.commit(this);

    transformResult(indices);
    undo = false;
  }

  return indices;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array2Builder<C1, C2> builder)
    -> std::vector<std::vector<int>> {
  return buildColumns(builder, [=](std::vector<std::vector<int>> const &) {});
}

template <typename I, typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral, typename Func0, typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array2Builder<C1, C2> builder, Func0 makeResult,
    std::function<void(I &, C1, C2, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(builder, [&](std::vector<std::vector<int>> const &indices) {
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
      for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2) {
        addResult(ret, i1, i2, indices[i1][i2]);
      }
  });
  return ret;
}

template <typename K1, typename K2>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map2Builder<K1, K2> builder,
    std::function<void(K1, K2, int)> addKey,
    std::function<void()> makeResult) -> void {
  int next = attributes.getCols();
  int initial = 1;
  xpress::Iterable<K1> iterable1 = builder.getIterable1();
  initial *= std::max(xpress::sizeHint(iterable1), 1);
  xpress::Iterable<K2> iterable2 = builder.getIterable2();
  initial *= std::max(xpress::sizeHint(iterable2), 1);

  ColumnCreator creator = ColumnCreator(this, initial, !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    std::function<double(K1, K2)> getColumnLB = builder.getLB();
    std::function<double(K1, K2)> getColumnUB = builder.getUB();
    std::function<ColumnType(K1, K2)> getColumnType = builder.getType();
    std::function<double(K1, K2)> getColumnLimit = builder.getLimit();
    std::function<std::optional<std::string>(K1, K2)> getColumnName =
        builder.getName();
    std::function<bool(K1, K2)> dataFilter = builder.getFilter();
    for (auto &v1 : iterable1)
      for (auto &v2 : iterable2) {
        if (!(!dataFilter) && !dataFilter(v1, v2))
          continue;
        creator.addColumn(
            this, !(!getColumnLB) ? getColumnLB(v1, v2) : 0.0,
            !(!getColumnUB) ? getColumnUB(v1, v2)
                            : std::numeric_limits<double>::infinity(),
            !(!getColumnType) ? (char)getColumnType(v1, v2) : 'C',
            !(!getColumnLimit) ? getColumnLimit(v1, v2) : 0.0,
            !(!getColumnName) ? getColumnName(v1, v2) : std::nullopt);
        addKey(v1, v2, next++);
      }
    creator.commit(this);

    makeResult();
    undo = false;
  }
}

template <typename K1, typename K2>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map2Builder<K1, K2> builder)
    -> xpress::maps::HashMap2<K1, K2, int> {
  xpress::maps::HashMap2<K1, K2, int> ret =
      xpress::maps::HashMap2<K1, K2, int>();
  buildColumns(
      builder,
      [&](K1 k1, K2 k2, int idx) {
        ret[xpress::maps::HashMap2<K1, K2, int>::key_type(k1, k2)] = idx;
      },
      [=]() {});
  return ret;
}

template <typename I, typename K1, typename K2, typename Func0,
          typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map2Builder<K1, K2> builder, Func0 makeResult,
    std::function<void(I &, K1, K2, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(
      builder, [&](K1 k1, K2 k2, int idx) { addResult(ret, k1, k2, idx); },
      [=]() {});
  return ret;
}

template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::XPRSProblem::addColumns(C1 dim1, C2 dim2)
    -> VariableBuilder::ColumnArray2Builder<C1, C2> {
  return VariableBuilder::ColumnArray2Builder<C1, C2>(
      xpress::toInt(dim1), xpress::toInt(dim2), this);
}

template <typename Iter0, typename Iter1, typename K1, typename K2,
          typename Iter0IsIterable, typename Iter1IsIterable>
auto xpress::XPRSProblem::addColumns(Iter0 const &iterable1,
                                     Iter1 const &iterable2)
    -> VariableBuilder::ColumnMap2Builder<K1, K2> {
  return VariableBuilder::ColumnMap2Builder<K1, K2>(iterable1, iterable2, this);
}

template <typename K1, typename K2>
auto xpress::XPRSProblem::addColumns(xpress::SizedArray<K1 const> const &arr1,
                                     xpress::SizedArray<K2 const> const &arr2)
    -> VariableBuilder::ColumnMap2Builder<K1, K2> {
  return VariableBuilder::ColumnMap2Builder<K1, K2>(arr1, arr2, this);
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array3Builder<C1, C2, C3> builder,
    std::function<void(std::vector<std::vector<std::vector<int>>> const &)>
        transformResult) -> std::vector<std::vector<std::vector<int>>> {
  std::vector<std::vector<std::vector<int>>> indices = xpress::makeArray<int>(
      builder.getDim(1 - 1), builder.getDim(2 - 1), builder.getDim(3 - 1));
  ColumnCreator creator = ColumnCreator(
      this,
      builder.getDim(1 - 1) + builder.getDim(2 - 1) + builder.getDim(3 - 1),
      !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    int nextIndex = attributes.getCols();
    std::function<double(int, int, int)> getColumnLB = builder.getLB();
    std::function<double(int, int, int)> getColumnUB = builder.getUB();
    std::function<double(int, int, int)> getColumnLimit = builder.getLimit();
    std::function<std::optional<std::string>(int, int, int)> getColumnName =
        builder.getName();
    std::function<ColumnType(int, int, int)> getColumnType = builder.getType();
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
      for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
        for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3) {
          creator.addColumn(
              this, !(!getColumnLB) ? getColumnLB(i1, i2, i3) : 0.0,
              !(!getColumnUB) ? getColumnUB(i1, i2, i3)
                              : std::numeric_limits<double>::infinity(),
              !(!getColumnType) ? (char)getColumnType(i1, i2, i3) : 'C',
              !(!getColumnLimit) ? getColumnLimit(i1, i2, i3) : 0.0,
              !(!getColumnName) ? getColumnName(i1, i2, i3) : std::nullopt);
          indices[i1][i2][i3] = nextIndex++;
        }
    creator.commit(this);

    transformResult(indices);
    undo = false;
  }

  return indices;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array3Builder<C1, C2, C3> builder)
    -> std::vector<std::vector<std::vector<int>>> {
  return buildColumns(
      builder, [=](std::vector<std::vector<std::vector<int>>> const &) {});
}

template <typename I, typename C1, typename C2, typename C3,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename Func0, typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array3Builder<C1, C2, C3> builder, Func0 makeResult,
    std::function<void(I &, C1, C2, C3, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(builder,
               [&](std::vector<std::vector<std::vector<int>>> const &indices) {
                 for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
                   for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
                     for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3) {
                       addResult(ret, i1, i2, i3, indices[i1][i2][i3]);
                     }
               });
  return ret;
}

template <typename K1, typename K2, typename K3>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map3Builder<K1, K2, K3> builder,
    std::function<void(K1, K2, K3, int)> addKey,
    std::function<void()> makeResult) -> void {
  int next = attributes.getCols();
  int initial = 1;
  xpress::Iterable<K1> iterable1 = builder.getIterable1();
  initial *= std::max(xpress::sizeHint(iterable1), 1);
  xpress::Iterable<K2> iterable2 = builder.getIterable2();
  initial *= std::max(xpress::sizeHint(iterable2), 1);
  xpress::Iterable<K3> iterable3 = builder.getIterable3();
  initial *= std::max(xpress::sizeHint(iterable3), 1);

  ColumnCreator creator = ColumnCreator(this, initial, !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    std::function<double(K1, K2, K3)> getColumnLB = builder.getLB();
    std::function<double(K1, K2, K3)> getColumnUB = builder.getUB();
    std::function<ColumnType(K1, K2, K3)> getColumnType = builder.getType();
    std::function<double(K1, K2, K3)> getColumnLimit = builder.getLimit();
    std::function<std::optional<std::string>(K1, K2, K3)> getColumnName =
        builder.getName();
    std::function<bool(K1, K2, K3)> dataFilter = builder.getFilter();
    for (auto &v1 : iterable1)
      for (auto &v2 : iterable2)
        for (auto &v3 : iterable3) {
          if (!(!dataFilter) && !dataFilter(v1, v2, v3))
            continue;
          creator.addColumn(
              this, !(!getColumnLB) ? getColumnLB(v1, v2, v3) : 0.0,
              !(!getColumnUB) ? getColumnUB(v1, v2, v3)
                              : std::numeric_limits<double>::infinity(),
              !(!getColumnType) ? (char)getColumnType(v1, v2, v3) : 'C',
              !(!getColumnLimit) ? getColumnLimit(v1, v2, v3) : 0.0,
              !(!getColumnName) ? getColumnName(v1, v2, v3) : std::nullopt);
          addKey(v1, v2, v3, next++);
        }
    creator.commit(this);

    makeResult();
    undo = false;
  }
}

template <typename K1, typename K2, typename K3>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map3Builder<K1, K2, K3> builder)
    -> xpress::maps::HashMap3<K1, K2, K3, int> {
  xpress::maps::HashMap3<K1, K2, K3, int> ret =
      xpress::maps::HashMap3<K1, K2, K3, int>();
  buildColumns(
      builder,
      [&](K1 k1, K2 k2, K3 k3, int idx) {
        ret[xpress::maps::HashMap3<K1, K2, K3, int>::key_type(k1, k2, k3)] =
            idx;
      },
      [=]() {});
  return ret;
}

template <typename I, typename K1, typename K2, typename K3, typename Func0,
          typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map3Builder<K1, K2, K3> builder, Func0 makeResult,
    std::function<void(I &, K1, K2, K3, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(
      builder,
      [&](K1 k1, K2 k2, K3 k3, int idx) { addResult(ret, k1, k2, k3, idx); },
      [=]() {});
  return ret;
}

template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::XPRSProblem::addColumns(C1 dim1, C2 dim2, C3 dim3)
    -> VariableBuilder::ColumnArray3Builder<C1, C2, C3> {
  return VariableBuilder::ColumnArray3Builder<C1, C2, C3>(
      xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3), this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename K1,
          typename K2, typename K3, typename Iter0IsIterable,
          typename Iter1IsIterable, typename Iter2IsIterable>
auto xpress::XPRSProblem::addColumns(
    Iter0 const &iterable1, Iter1 const &iterable2,
    Iter2 const &iterable3) -> VariableBuilder::ColumnMap3Builder<K1, K2, K3> {
  return VariableBuilder::ColumnMap3Builder<K1, K2, K3>(iterable1, iterable2,
                                                        iterable3, this);
}

template <typename K1, typename K2, typename K3>
auto xpress::XPRSProblem::addColumns(xpress::SizedArray<K1 const> const &arr1,
                                     xpress::SizedArray<K2 const> const &arr2,
                                     xpress::SizedArray<K3 const> const &arr3)
    -> VariableBuilder::ColumnMap3Builder<K1, K2, K3> {
  return VariableBuilder::ColumnMap3Builder<K1, K2, K3>(arr1, arr2, arr3, this);
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array4Builder<C1, C2, C3, C4> builder,
    std::function<
        void(std::vector<std::vector<std::vector<std::vector<int>>>> const &)>
        transformResult)
    -> std::vector<std::vector<std::vector<std::vector<int>>>> {
  std::vector<std::vector<std::vector<std::vector<int>>>> indices =
      xpress::makeArray<int>(builder.getDim(1 - 1), builder.getDim(2 - 1),
                             builder.getDim(3 - 1), builder.getDim(4 - 1));
  ColumnCreator creator =
      ColumnCreator(this,
                    builder.getDim(1 - 1) + builder.getDim(2 - 1) +
                        builder.getDim(3 - 1) + builder.getDim(4 - 1),
                    !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    int nextIndex = attributes.getCols();
    std::function<double(int, int, int, int)> getColumnLB = builder.getLB();
    std::function<double(int, int, int, int)> getColumnUB = builder.getUB();
    std::function<double(int, int, int, int)> getColumnLimit =
        builder.getLimit();
    std::function<std::optional<std::string>(int, int, int, int)>
        getColumnName = builder.getName();
    std::function<ColumnType(int, int, int, int)> getColumnType =
        builder.getType();
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
      for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
        for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3)
          for (int i4 = 0; i4 < builder.getDim(4 - 1); ++i4) {
            creator.addColumn(
                this, !(!getColumnLB) ? getColumnLB(i1, i2, i3, i4) : 0.0,
                !(!getColumnUB) ? getColumnUB(i1, i2, i3, i4)
                                : std::numeric_limits<double>::infinity(),
                !(!getColumnType) ? (char)getColumnType(i1, i2, i3, i4) : 'C',
                !(!getColumnLimit) ? getColumnLimit(i1, i2, i3, i4) : 0.0,
                !(!getColumnName) ? getColumnName(i1, i2, i3, i4)
                                  : std::nullopt);
            indices[i1][i2][i3][i4] = nextIndex++;
          }
    creator.commit(this);

    transformResult(indices);
    undo = false;
  }

  return indices;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array4Builder<C1, C2, C3, C4> builder)
    -> std::vector<std::vector<std::vector<std::vector<int>>>> {
  return buildColumns(
      builder,
      [=](std::vector<std::vector<std::vector<std::vector<int>>>> const &) {});
}

template <typename I, typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename Func0, typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array4Builder<C1, C2, C3, C4> builder, Func0 makeResult,
    std::function<void(I &, C1, C2, C3, C4, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(
      builder, [&](std::vector<std::vector<std::vector<std::vector<int>>>> const
                       &indices) {
        for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
          for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
            for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3)
              for (int i4 = 0; i4 < builder.getDim(4 - 1); ++i4) {
                addResult(ret, i1, i2, i3, i4, indices[i1][i2][i3][i4]);
              }
      });
  return ret;
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map4Builder<K1, K2, K3, K4> builder,
    std::function<void(K1, K2, K3, K4, int)> addKey,
    std::function<void()> makeResult) -> void {
  int next = attributes.getCols();
  int initial = 1;
  xpress::Iterable<K1> iterable1 = builder.getIterable1();
  initial *= std::max(xpress::sizeHint(iterable1), 1);
  xpress::Iterable<K2> iterable2 = builder.getIterable2();
  initial *= std::max(xpress::sizeHint(iterable2), 1);
  xpress::Iterable<K3> iterable3 = builder.getIterable3();
  initial *= std::max(xpress::sizeHint(iterable3), 1);
  xpress::Iterable<K4> iterable4 = builder.getIterable4();
  initial *= std::max(xpress::sizeHint(iterable4), 1);

  ColumnCreator creator = ColumnCreator(this, initial, !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    std::function<double(K1, K2, K3, K4)> getColumnLB = builder.getLB();
    std::function<double(K1, K2, K3, K4)> getColumnUB = builder.getUB();
    std::function<ColumnType(K1, K2, K3, K4)> getColumnType = builder.getType();
    std::function<double(K1, K2, K3, K4)> getColumnLimit = builder.getLimit();
    std::function<std::optional<std::string>(K1, K2, K3, K4)> getColumnName =
        builder.getName();
    std::function<bool(K1, K2, K3, K4)> dataFilter = builder.getFilter();
    for (auto &v1 : iterable1)
      for (auto &v2 : iterable2)
        for (auto &v3 : iterable3)
          for (auto &v4 : iterable4) {
            if (!(!dataFilter) && !dataFilter(v1, v2, v3, v4))
              continue;
            creator.addColumn(
                this, !(!getColumnLB) ? getColumnLB(v1, v2, v3, v4) : 0.0,
                !(!getColumnUB) ? getColumnUB(v1, v2, v3, v4)
                                : std::numeric_limits<double>::infinity(),
                !(!getColumnType) ? (char)getColumnType(v1, v2, v3, v4) : 'C',
                !(!getColumnLimit) ? getColumnLimit(v1, v2, v3, v4) : 0.0,
                !(!getColumnName) ? getColumnName(v1, v2, v3, v4)
                                  : std::nullopt);
            addKey(v1, v2, v3, v4, next++);
          }
    creator.commit(this);

    makeResult();
    undo = false;
  }
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map4Builder<K1, K2, K3, K4> builder)
    -> xpress::maps::HashMap4<K1, K2, K3, K4, int> {
  xpress::maps::HashMap4<K1, K2, K3, K4, int> ret =
      xpress::maps::HashMap4<K1, K2, K3, K4, int>();
  buildColumns(
      builder,
      [&](K1 k1, K2 k2, K3 k3, K4 k4, int idx) {
        ret[xpress::maps::HashMap4<K1, K2, K3, K4, int>::key_type(k1, k2, k3,
                                                                  k4)] = idx;
      },
      [=]() {});
  return ret;
}

template <typename I, typename K1, typename K2, typename K3, typename K4,
          typename Func0, typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map4Builder<K1, K2, K3, K4> builder, Func0 makeResult,
    std::function<void(I &, K1, K2, K3, K4, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(
      builder,
      [&](K1 k1, K2 k2, K3 k3, K4 k4, int idx) {
        addResult(ret, k1, k2, k3, k4, idx);
      },
      [=]() {});
  return ret;
}

template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::XPRSProblem::addColumns(C1 dim1, C2 dim2, C3 dim3, C4 dim4)
    -> VariableBuilder::ColumnArray4Builder<C1, C2, C3, C4> {
  return VariableBuilder::ColumnArray4Builder<C1, C2, C3, C4>(
      xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
      xpress::toInt(dim4), this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename K1, typename K2, typename K3, typename K4,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
auto xpress::XPRSProblem::addColumns(Iter0 const &iterable1,
                                     Iter1 const &iterable2,
                                     Iter2 const &iterable3,
                                     Iter3 const &iterable4)
    -> VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> {
  return VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>(
      iterable1, iterable2, iterable3, iterable4, this);
}

template <typename K1, typename K2, typename K3, typename K4>
auto xpress::XPRSProblem::addColumns(xpress::SizedArray<K1 const> const &arr1,
                                     xpress::SizedArray<K2 const> const &arr2,
                                     xpress::SizedArray<K3 const> const &arr3,
                                     xpress::SizedArray<K4 const> const &arr4)
    -> VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> {
  return VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>(arr1, arr2, arr3,
                                                            arr4, this);
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder,
    std::function<
        void(std::vector<
             std::vector<std::vector<std::vector<std::vector<int>>>>> const &)>
        transformResult)
    -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>> {
  std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>> indices =
      xpress::makeArray<int>(builder.getDim(1 - 1), builder.getDim(2 - 1),
                             builder.getDim(3 - 1), builder.getDim(4 - 1),
                             builder.getDim(5 - 1));
  ColumnCreator creator = ColumnCreator(
      this,
      builder.getDim(1 - 1) + builder.getDim(2 - 1) + builder.getDim(3 - 1) +
          builder.getDim(4 - 1) + builder.getDim(5 - 1),
      !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    int nextIndex = attributes.getCols();
    std::function<double(int, int, int, int, int)> getColumnLB =
        builder.getLB();
    std::function<double(int, int, int, int, int)> getColumnUB =
        builder.getUB();
    std::function<double(int, int, int, int, int)> getColumnLimit =
        builder.getLimit();
    std::function<std::optional<std::string>(int, int, int, int, int)>
        getColumnName = builder.getName();
    std::function<ColumnType(int, int, int, int, int)> getColumnType =
        builder.getType();
    for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
      for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
        for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3)
          for (int i4 = 0; i4 < builder.getDim(4 - 1); ++i4)
            for (int i5 = 0; i5 < builder.getDim(5 - 1); ++i5) {
              creator.addColumn(
                  this, !(!getColumnLB) ? getColumnLB(i1, i2, i3, i4, i5) : 0.0,
                  !(!getColumnUB) ? getColumnUB(i1, i2, i3, i4, i5)
                                  : std::numeric_limits<double>::infinity(),
                  !(!getColumnType) ? (char)getColumnType(i1, i2, i3, i4, i5)
                                    : 'C',
                  !(!getColumnLimit) ? getColumnLimit(i1, i2, i3, i4, i5) : 0.0,
                  !(!getColumnName) ? getColumnName(i1, i2, i3, i4, i5)
                                    : std::nullopt);
              indices[i1][i2][i3][i4][i5] = nextIndex++;
            }
    creator.commit(this);

    transformResult(indices);
    undo = false;
  }

  return indices;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder)
    -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>> {
  return buildColumns(
      builder,
      [=](std::vector<
          std::vector<std::vector<std::vector<std::vector<int>>>>> const &) {});
}

template <typename I, typename C1, typename C2, typename C3, typename C4,
          typename C5, typename C1IsIntegral, typename C2IsIntegral,
          typename C3IsIntegral, typename C4IsIntegral, typename C5IsIntegral,
          typename Func0, typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder,
    Func0 makeResult,
    std::function<void(I &, C1, C2, C3, C4, C5, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(
      builder,
      [&](std::vector<std::vector<
              std::vector<std::vector<std::vector<int>>>>> const &indices) {
        for (int i1 = 0; i1 < builder.getDim(1 - 1); ++i1)
          for (int i2 = 0; i2 < builder.getDim(2 - 1); ++i2)
            for (int i3 = 0; i3 < builder.getDim(3 - 1); ++i3)
              for (int i4 = 0; i4 < builder.getDim(4 - 1); ++i4)
                for (int i5 = 0; i5 < builder.getDim(5 - 1); ++i5) {
                  addResult(ret, i1, i2, i3, i4, i5,
                            indices[i1][i2][i3][i4][i5]);
                }
      });
  return ret;
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder,
    std::function<void(K1, K2, K3, K4, K5, int)> addKey,
    std::function<void()> makeResult) -> void {
  int next = attributes.getCols();
  int initial = 1;
  xpress::Iterable<K1> iterable1 = builder.getIterable1();
  initial *= std::max(xpress::sizeHint(iterable1), 1);
  xpress::Iterable<K2> iterable2 = builder.getIterable2();
  initial *= std::max(xpress::sizeHint(iterable2), 1);
  xpress::Iterable<K3> iterable3 = builder.getIterable3();
  initial *= std::max(xpress::sizeHint(iterable3), 1);
  xpress::Iterable<K4> iterable4 = builder.getIterable4();
  initial *= std::max(xpress::sizeHint(iterable4), 1);
  xpress::Iterable<K5> iterable5 = builder.getIterable5();
  initial *= std::max(xpress::sizeHint(iterable5), 1);

  ColumnCreator creator = ColumnCreator(this, initial, !(!builder.getName()));
  bool undo = true;
  {
    xpress::RaiiHelper finally([&]() {
      if (undo)
        creator.undo(this);
    });

    std::function<double(K1, K2, K3, K4, K5)> getColumnLB = builder.getLB();
    std::function<double(K1, K2, K3, K4, K5)> getColumnUB = builder.getUB();
    std::function<ColumnType(K1, K2, K3, K4, K5)> getColumnType =
        builder.getType();
    std::function<double(K1, K2, K3, K4, K5)> getColumnLimit =
        builder.getLimit();
    std::function<std::optional<std::string>(K1, K2, K3, K4, K5)>
        getColumnName = builder.getName();
    std::function<bool(K1, K2, K3, K4, K5)> dataFilter = builder.getFilter();
    for (auto &v1 : iterable1)
      for (auto &v2 : iterable2)
        for (auto &v3 : iterable3)
          for (auto &v4 : iterable4)
            for (auto &v5 : iterable5) {
              if (!(!dataFilter) && !dataFilter(v1, v2, v3, v4, v5))
                continue;
              creator.addColumn(
                  this, !(!getColumnLB) ? getColumnLB(v1, v2, v3, v4, v5) : 0.0,
                  !(!getColumnUB) ? getColumnUB(v1, v2, v3, v4, v5)
                                  : std::numeric_limits<double>::infinity(),
                  !(!getColumnType) ? (char)getColumnType(v1, v2, v3, v4, v5)
                                    : 'C',
                  !(!getColumnLimit) ? getColumnLimit(v1, v2, v3, v4, v5) : 0.0,
                  !(!getColumnName) ? getColumnName(v1, v2, v3, v4, v5)
                                    : std::nullopt);
              addKey(v1, v2, v3, v4, v5, next++);
            }
    creator.commit(this);

    makeResult();
    undo = false;
  }
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder)
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int> {
  xpress::maps::HashMap5<K1, K2, K3, K4, K5, int> ret =
      xpress::maps::HashMap5<K1, K2, K3, K4, K5, int>();
  buildColumns(
      builder,
      [&](K1 k1, K2 k2, K3 k3, K4 k4, K5 k5, int idx) {
        ret[xpress::maps::HashMap5<K1, K2, K3, K4, K5, int>::key_type(
            k1, k2, k3, k4, k5)] = idx;
      },
      [=]() {});
  return ret;
}

template <typename I, typename K1, typename K2, typename K3, typename K4,
          typename K5, typename Func0, typename Func0IsInvocable>
auto xpress::XPRSProblem::buildColumns(
    VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder, Func0 makeResult,
    std::function<void(I &, K1, K2, K3, K4, K5, int)> addResult) -> I {
  I ret = makeResult();
  buildColumns(
      builder,
      [&](K1 k1, K2 k2, K3 k3, K4 k4, K5 k5, int idx) {
        addResult(ret, k1, k2, k3, k4, k5, idx);
      },
      [=]() {});
  return ret;
}

template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::XPRSProblem::addColumns(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                     C5 dim5)
    -> VariableBuilder::ColumnArray5Builder<C1, C2, C3, C4, C5> {
  return VariableBuilder::ColumnArray5Builder<C1, C2, C3, C4, C5>(
      xpress::toInt(dim1), xpress::toInt(dim2), xpress::toInt(dim3),
      xpress::toInt(dim4), xpress::toInt(dim5), this);
}

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename K1, typename K2, typename K3, typename K4,
          typename K5, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
auto xpress::XPRSProblem::addColumns(Iter0 const &iterable1,
                                     Iter1 const &iterable2,
                                     Iter2 const &iterable3,
                                     Iter3 const &iterable4,
                                     Iter4 const &iterable5)
    -> VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> {
  return VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>(
      iterable1, iterable2, iterable3, iterable4, iterable5, this);
}

template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::XPRSProblem::addColumns(xpress::SizedArray<K1 const> const &arr1,
                                     xpress::SizedArray<K2 const> const &arr2,
                                     xpress::SizedArray<K3 const> const &arr3,
                                     xpress::SizedArray<K4 const> const &arr4,
                                     xpress::SizedArray<K5 const> const &arr5)
    -> VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> {
  return VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>(
      arr1, arr2, arr3, arr4, arr5, this);
}

auto xpress::XPRSProblem::columnTypeToArray(
    xpress::SizedArray<ColumnType const> const &types) -> std::vector<char> {

  assert(!(!types));
  int count = xpress::toInt(types.size());
  if (count == 0)
    return std::vector<char>(0);
  std::vector<char> ret(count);
  for (int i = 0; i < count; ++i)
    ret[i] = (char)types[i];
  return ret;
}

;

;
;
;
;

;

;

;

;

xpress::XPRSProblem::ColumnCreator::ColumnCreator(XPRSProblem *prob,
                                                  int initialSize,
                                                  bool hasNames)
    : firstCol(prob->attributes.getCols()), nextCol(firstCol), created(0),
      lbs(ExtendableDoubleArray(std::vector<double>(std::max(initialSize, 1)))),
      ubs(ExtendableDoubleArray(std::vector<double>(std::max(initialSize, 1)))),
      types(ExtendableTypeArray(std::vector<char>(std::max(initialSize, 1)))),
      limits(
          ExtendableDoubleArray(std::vector<double>(std::max(initialSize, 1)))),
      names(hasNames ? ExtendableArray<std::optional<std::string>>(
                           std::vector<std::optional<std::string>>(
                               std::max(initialSize, 1)))
                     : nullptr),
      indices(ExtendableIntArray(std::vector<int>(std::max(initialSize, 1)))),
      flags(0), nondefIdx(nullptr), nondefBounds(nullptr), nondefVals(nullptr) {
}

auto xpress::XPRSProblem::ColumnCreator::finishColumn(XPRSProblem *prob)
    -> void {
  indices.append(nextCol++);
  conditionalCommit(prob);
}
auto xpress::XPRSProblem::ColumnCreator::conditionalCommit(XPRSProblem *prob)
    -> void {
  if (lbs.getLength() >= 1024 * 1024 && lbs.getLength() >= lbs.getCapacity())
    commit(prob);
}
auto xpress::XPRSProblem::ColumnCreator::commit(XPRSProblem *prob) -> void {
  if (lbs.getLength() > 0) {
    {
      xpress::RaiiHelper finally([&]() {
        lbs.reset();
        ubs.reset();
        types.reset();
        limits.reset();
        if (!(!names))
          names.reset();
        indices.reset();
        if (!(!nondefIdx)) {
          nondefIdx.reset();
          nondefBounds.reset();
          nondefVals.reset();
        }
        flags = 0;
      });

      prob->addCols(lbs.getLength(), 0, nullptr, SizedArray<int>(nullptr),
                    nullptr, nullptr,
                    ((flags & FLAG_LB) != 0) ? lbs.getData()
                                             : SizedArray<double>(nullptr),
                    ((flags & FLAG_UB) != 0) ? ubs.getData()
                                             : SizedArray<double>(nullptr));
      created += lbs.getLength();
      std::vector<int> idx = indices.getData();
      if ((flags & FLAG_TYPE) != 0) {
        prob->chgColType(types.getLength(), idx, types.getData());

        if (!(!nondefIdx) && nondefIdx.getLength() > 0) {
          prob->chgBounds(nondefIdx.getLength(), nondefIdx.getData(),
                          nondefBounds.getData(), nondefVals.getData());
        }
      }
      if ((flags & FLAG_LIMIT) != 0)
        prob->chgGlbLimit(limits.getLength(), idx, limits.getData());
      if (!(!names) && (flags & FLAG_NAME) != 0)
        prob->addNamesWithNull(xpress::Namespaces::Column, idx[0],
                               nextCol - idx[0], names.getData());
    }
  }
}

auto xpress::XPRSProblem::ColumnCreator::appendNondef(int idx, char type,
                                                      double value) -> void {
  if (!nondefIdx) {
    nondefIdx = ExtendableIntArray(std::vector<int>(8));
    nondefBounds = ExtendableTypeArray(std::vector<char>(8));
    nondefVals = ExtendableDoubleArray(std::vector<double>(8));
  }
  nondefIdx.append(idx);
  nondefBounds.append(type);
  nondefVals.append(value);
}

auto xpress::XPRSProblem::ColumnCreator::addColumn(
    XPRSProblem *prob, double lb, double ub, char type, double limit,
    std::optional<std::string> name) -> int {
  lbs.append(lb);
  flags |= (lb != 0.0) ? FLAG_LB : 0;
  ubs.append(ub);
  flags |= (ub < XPRS_PLUSINFINITY) ? FLAG_UB : 0;
  types.append(type);
  if (type != (char)'C') {
    flags |= FLAG_TYPE;

    switch (type) {
    case (char)'B':
      if (ub == std::numeric_limits<double>::infinity()) {

        ub = 1.0;
      }

      if (lb < 0.0 || lb > 1.0 || ub < 0.0 || ub > 1.0)
        throw std::invalid_argument(
            "bounds for binary variables cannot be outside [0,1]");
      if (lb != 0.0)
        appendNondef(nextCol, (char)'L', std::ceil(lb));
      if (ub != 1.0)
        appendNondef(nextCol, (char)'U', std::floor(ub));
      break;
    case (char)'I':
      break;
    case (char)'S':
      if (lb != 0.0)
        appendNondef(nextCol, (char)'L', lb);
      break;
    case (char)'R':
      if (lb != 0.0)
        appendNondef(nextCol, (char)'L', std::ceil(lb));
      break;
    case (char)'P':
      break;
    }
  }
  limits.append(limit);
  flags |= (limit != 0.0) ? FLAG_LIMIT : 0;
  if (!(!names)) {
    names.append(name);
    flags |= (!(!(name))) ? FLAG_NAME : 0;
  }
  finishColumn(prob);
  return nextCol - 1;
}

auto xpress::XPRSProblem::ColumnCreator::undo(XPRSProblem *prob) -> void {
  if (created > 0) {

    std::vector<int> idx(128);
    int k = 0;
    for (int j = firstCol + created - 1; j >= firstCol; --j) {
      idx[k++] = j;
      if (k == xpress::toInt(idx.size())) {
        prob->delCols(k, idx);
        k = 0;
      }
    }
    if (k > 0)
      prob->delCols(k, idx);
  }
}

auto xpress::XPRSProblem::addOneColumn(
    double obj, double lb, double ub, char type,
    xpress::SizedArray<int const> const &rowind,
    xpress::SizedArray<double const> const &rowval,
    std::optional<std::string> name) -> int {
  std::vector<int> var({-1});
  int idx = attributes.getCols();
  {
    xpress::RaiiHelper finally([&]() {
      if (var[0] != -1)
        delCols(1, var);
    });

    int rows = 0;
    if (!(!rowind) != (!(!rowval)))
      throw std::invalid_argument("rowind and rowval must either both be "
                                  "nullptr or both be non-nullptr");
    if (!(!rowind) && !(!rowval)) {
      if (xpress::toInt(rowind.size()) != xpress::toInt(rowval.size()))
        throw std::invalid_argument("array length mismatch");
      rows = xpress::toInt(rowind.size());
      std::vector<int> start({0, rows});
      addCols(1, rows, std::vector<double>({obj}), start, rowind, rowval,
              std::vector<double>({lb}), std::vector<double>({ub}));
    } else {
      addCols(1, rows, std::vector<double>({obj}), SizedArray<int>(nullptr),
              nullptr, nullptr, std::vector<double>({lb}),
              std::vector<double>({ub}));
    }

    var[0] = idx;

    chgColType(1, var, std::vector<char>({(char)type}));
    if (!(!(name)))
      addNames(xpress::Namespaces::Column,
               std::vector<std::optional<std::string>>({name}), idx, idx);
    var[0] = -1;
  }

  return idx;
}

auto xpress::XPRSProblem::addCol(double obj, double lb, double ub, char type,
                                 xpress::SizedArray<int const> const &rowind,
                                 xpress::SizedArray<double const> const &rowval,
                                 std::optional<std::string> name) -> int {
  return addOneColumn(obj, lb, ub, type, rowind, rowval, name);
}

auto xpress::XPRSProblem::addCol(double obj, double lb, double ub, char type,
                                 xpress::SizedArray<int const> const &rowind,
                                 xpress::SizedArray<double const> const &rowval)
    -> int {
  return addOneColumn(obj, lb, ub, type, rowind, rowval, nullptr);
}

auto xpress::XPRSProblem::addCol(double obj, double lb, double ub) -> int {
  return addOneColumn(obj, lb, ub, 'C', SizedArray<int>(nullptr), nullptr,
                      nullptr);
}

auto xpress::XPRSProblem::addCol(double obj, double lb, double ub,
                                 char type) -> int {
  return addOneColumn(obj, lb, ub, type, SizedArray<int>(nullptr), nullptr,
                      nullptr);
}

auto xpress::XPRSProblem::addCol(double obj, double lb, double ub,
                                 std::optional<std::string> name) -> int {
  return addOneColumn(obj, lb, ub, 'C', SizedArray<int>(nullptr), nullptr,
                      name);
}

auto xpress::XPRSProblem::addCol(double obj, double lb, double ub, char type,
                                 std::optional<std::string> name) -> int {
  return addOneColumn(obj, lb, ub, type, SizedArray<int>(nullptr), nullptr,
                      name);
}

auto xpress::XPRSProblem::addMipSol(xpress::SizedArray<double const> const &val,
                                    xpress::SizedArray<int const> const &ind,
                                    std::optional<std::string> name) -> void {
  if (xpress::toInt(ind.size()) != xpress::toInt(val.size()))
    throw std::invalid_argument("array length mismatch");
  addMipSol(xpress::toInt(ind.size()), val, ind, name);
}

auto xpress::XPRSProblem::addMipSol(xpress::SizedArray<double const> const &val,
                                    xpress::SizedArray<int const> const &ind)
    -> void {
  if (xpress::toInt(ind.size()) != xpress::toInt(val.size()))
    throw std::invalid_argument("array length mismatch");
  addMipSol(xpress::toInt(ind.size()), val, ind, nullptr);
}

auto xpress::XPRSProblem::addNamesWithNull(
    xpress::Namespaces which, int firstIndex, int count,
    xpress::SizedArray<std::optional<std::string> const> const &namesToAdd)
    -> void {
  int begin = 0;
  int end = 0;
  while (end < count) {
    if (!(namesToAdd[end])) {

      if (end > begin) {
        addNames(static_cast<int>(which), arrayChunk(namesToAdd, begin, end),
                 firstIndex + begin, firstIndex + end - 1);
      }
      begin = end + 1;
    }
    ++end;
  }
  if (begin == 0 && end == count) {

    std::vector<std::string> names;
    names.reserve(count);
    for (int i = 0; i < count; ++i)
      names.push_back(namesToAdd[i].value());
    addNames(static_cast<int>(which), names, firstIndex,
             firstIndex + count - 1);
  } else if (end > begin) {

    addNames(static_cast<int>(which), arrayChunk(namesToAdd, begin, end),
             firstIndex + begin, firstIndex + end - 1);
  }
}

auto xpress::XPRSProblem::addNames(
    xpress::Namespaces type,
    xpress::SizedArray<std::optional<std::string> const> const &names,
    int first, int last) -> void {
  addNamesWithNull(type, first, last - first + 1, names);
}

auto xpress::XPRSProblem::rowTypeToArray(
    xpress::SizedArray<RowType const> const &types) -> std::vector<char> {

  assert(!(!types));
  int count = xpress::toInt(xpress::toInt(types.size()));
  if (count == 0)
    return std::vector<char>(0);
  std::vector<char> ret(count);
  for (int i = 0; i < count; ++i)
    ret[i] = (char)types[i];
  return ret;
}

;

;

auto xpress::XPRSProblem::arrayChunk(
    xpress::SizedArray<std::optional<std::string> const> const &src, int begin,
    int end) -> std::vector<std::string> {
  int count = end - begin;
  std::vector<std::string> chunk(count);
  for (int i = 0; i < count; ++i)
    chunk[i] = src[i + begin].value();
  return chunk;
}

;

;

;

;

;

;

;

;

;

;

;

;

;

xpress::XPRSProblem::RowCreator::RowCreator(XPRSProblem *prob,
                                            int initialInequalitiesSize,
                                            int initialNzsSize, bool hasNames,
                                            bool errorIfVariables)
    : prob(prob), firstRow(prob->attributes.getRows()),
      nextInequality(firstRow), created(0), errorIfVariables(errorIfVariables),
      start(ExtendableIntArray(std::vector<int>(initialInequalitiesSize + 1))),
      rngvals(
          ExtendableDoubleArray(std::vector<double>(initialInequalitiesSize))),
      rowcoefs(ExtendableDoubleArray(std::vector<double>(initialNzsSize))),
      colinds(ExtendableIntArray(std::vector<int>(initialNzsSize))),
      types(ExtendableTypeArray(std::vector<char>(initialInequalitiesSize))),
      rhsvals(
          ExtendableDoubleArray(std::vector<double>(initialInequalitiesSize))),
      names(hasNames ? ExtendableArray<std::optional<std::string>>(
                           std::vector<std::optional<std::string>>(
                               initialInequalitiesSize))
                     : ExtendableArray<std::optional<std::string>>(nullptr)),
      qind1(nullptr), qind2(nullptr), qcoef(nullptr), flags(0),
      formulaRows(nullptr), formulaStart(nullptr), formulaType(nullptr),
      formulaValue(nullptr), formulasStarted(0), formulasInCurrentRow(0),
      indices(hasNames ? ExtendableIntArray(
                             std::vector<int>(initialInequalitiesSize))
                       : ExtendableIntArray(nullptr)) {
  start.append(0);
  rhsvals.append(0.0);
  rngvals.append(0.0);
  if (!(!names))
    names.append(std::nullopt);
}

auto xpress::XPRSProblem::RowCreator::checkForVariables() -> void {
  if (errorIfVariables)
    throw CannotReferenceVariablesException();
}

auto xpress::XPRSProblem::RowCreator::normalizeCurrentRow() -> void {

  int offset = start.getData()[start.getLength() - 1];
  int length = rowcoefs.getLength() - offset;
  if (length > 1) {
    std::vector<int> &c = colinds.getData();
    std::vector<double> &r = rowcoefs.getData();
    xpress::objects::InternalUtils::sortParallelArrays(c, r, offset, length);

    int last = std::numeric_limits<int>::min();
    int outIdx = offset;
    for (int inIdx = offset; inIdx < offset + length; ++inIdx) {
      int idx = c[inIdx];
      if (idx != last) {
        c[outIdx] = idx;
        r[outIdx++] = r[inIdx];
        last = idx;
      } else {
        r[outIdx - 1] += r[inIdx];
      }
    }
    colinds.setLength(outIdx);
    rowcoefs.setLength(outIdx);
  }

  if (!(!qind1)) {
    length = qind1.getLength();
    if (length > 1) {
      std::vector<int> &c1 = qind1.getData();
      std::vector<int> &c2 = qind2.getData();
      std::vector<double> &r = qcoef.getData();
      xpress::objects::InternalUtils::sortParallelArrays(c1, c2, r, 0, length);
      int last1 = std::numeric_limits<int>::min();
      int last2 = std::numeric_limits<int>::min();
      int outIdx = offset;
      for (int inIdx = offset; inIdx < offset + length; ++inIdx) {
        int idx1 = c1[inIdx];
        int idx2 = c2[inIdx];
        if (idx1 != last1 || idx2 != last2) {
          c1[outIdx] = idx1;
          c2[outIdx] = idx2;
          r[outIdx++] = r[inIdx];
          last1 = idx1;
          last2 = idx2;
        } else {
          r[outIdx - 1] += r[inIdx];
        }
      }
      qind1.setLength(outIdx);
      qind2.setLength(outIdx);
      qcoef.setLength(outIdx);
    }
  }
}

auto xpress::XPRSProblem::RowCreator::finishRow(bool normalize) -> void {
  assert(formulasStarted == 0);
  if (normalize) {
    normalizeCurrentRow();
  }
  if (!(!indices))
    indices.append(nextInequality);
  ++nextInequality;
  start.append(rowcoefs.getLength());
  rhsvals.append(0.0);
  rngvals.append(0.0);
  if (!(!names))
    names.append(std::nullopt);
  assert(start.getLength() == types.getLength() + 1);
  assert(rhsvals.getLength() == types.getLength() + 1);
  assert(rngvals.getLength() == types.getLength() + 1);
  assert((!names) || names.getLength() == types.getLength() + 1);
  assert(colinds.getLength() == rowcoefs.getLength());
  if (!(!formulaStart) && formulasInCurrentRow > 0) {
    if (formulasInCurrentRow > 1) {

      assert(
          formulaType
              .getData()[formulaStart.getData()[formulaRows.getLength() - 1]] ==
          XPRS_TOK_RB);
      formulaType.append(XPRS_TOK_IFUN);
      formulaValue.append(XPRS_IFUN_SUM);
    } else {

      assert(
          formulaType
              .getData()[formulaStart.getData()[formulaRows.getLength() - 1]] ==
          XPRS_TOK_RB);
      formulaStart.getData()[formulaRows.getLength() - 1] += 1;
    }

    formulaType.append(XPRS_TOK_EOF);
    formulaValue.append(XPRS_TOK_EOF);

    formulaStart.append(formulaType.getLength());
  }
  formulasInCurrentRow = 0;
  conditionalCommit();
}

;

auto xpress::XPRSProblem::RowCreator::conditionalCommit() -> void {
  if (rowcoefs.getLength() > nonzeroThreshold ||
      (!(!formulaType) && formulaType.getLength() > nonzeroThreshold) ||

      (!(!qind1) && qind1.getLength() > 0))
    commit();
}

auto xpress::XPRSProblem::RowCreator::commit() -> void {
  if (types.getLength() > 0) {
    int qrow = -1;
    {
      xpress::RaiiHelper finally([&]() {
        start.reset();
        rngvals.reset();
        rowcoefs.reset();
        colinds.reset();
        types.reset();
        rhsvals.reset();
        if (!(!names))
          names.reset();
        if (!(!indices))
          indices.reset();
        start.append(0);
        rhsvals.append(0.0);
        rngvals.append(0.0);
        if (!(!names))
          names.append(std::nullopt);
        if (qrow >= 0) {
          qind1.reset();
          qind2.reset();
          qcoef.reset();
        }
        if (!(!formulaRows)) {
          formulaRows.reset();
          formulaStart.reset();
          formulaStart.append(0);
          formulaType.reset();
          formulaValue.reset();
        }
        flags = 0;
      });

      if (!(!qind1) && qind1.getLength() > 0) {

        assert(types.getLength() == 1);
        qrow = prob->attributes.getRows();
      }
      assert(start.getLength() > 0);
      prob->addRows(types.getLength(), rowcoefs.getLength(), types.getData(),
                    rhsvals.getData(),
                    ((flags & FLAG_RANGE) != 0)
                        ? rngvals.getData()
                        : xpress::SizedArray<double>(nullptr, 0),
                    start.getData(), colinds.getData(), rowcoefs.getData());
      created += types.getLength();
      if (!(!names) && (flags & FLAG_NAME) != 0) {
        std::vector<int> idx = indices.getData();
        prob->addNamesWithNull(xpress::Namespaces::Row, idx[0],
                               nextInequality - idx[0], names.getData());
      }
      if (qrow >= 0) {
        assert(!(!qind1));
        assert(!(!qind2));
        prob->addQMatrix(qrow, qind1.getLength(), qind1.getData(),
                         qind2.getData(), qcoef.getData());
      }
      if (!(!formulaRows) && formulaRows.getLength() > 0) {
        assert(!(!formulaRows));
        assert(formulaRows.getLength() > 0);
        assert(formulaStart.getLength() == formulaRows.getLength() + 1);

        prob->nlpAddFormulas(formulaRows.getLength(), formulaRows.getData(),
                             formulaStart.getData(), 1, formulaType.getData(),
                             formulaValue.getData());
      }
    }
  }
}

auto xpress::XPRSProblem::RowCreator::addRow(
    xpress::SizedArray<int const> const &colind,
    xpress::SizedArray<double const> const &rowcoef, double rngval, char type,
    double rhs, std::optional<std::string> name, bool normalize) -> void {
  assert(xpress::toInt(colind.size()) == xpress::toInt(rowcoef.size()));
  colinds.append(colind);
  rowcoefs.append(rowcoef);
  rngvals.append(rngval);
  types.append(type);
  rhsvals.append(rhs);
  if (!(!names)) {
    if (!(!(name)))
      flags |= FLAG_NAME;
    names.append(name);
  }
  finishRow(normalize);
}

auto xpress::XPRSProblem::RowCreator::undo() -> void {
  if (created > 0) {

    std::vector<int> idx(128);
    int k = 0;
    for (int j = firstRow + created - 1; j >= firstRow; --j) {
      idx[k++] = j;
      if (k == xpress::toInt(xpress::toInt(idx.size()))) {
        if (!(!formulaStart)) {

          prob->nlpDelFormulas(k, idx);
        }
        prob->delRows(k, idx);
        k = 0;
      }
    }
    if (k > 0) {
      if (!(!formulaStart)) {

        prob->nlpDelFormulas(k, idx);
      }
      prob->delRows(k, idx);
    }
  }
}

auto xpress::XPRSProblem::RowCreator::addRhs(double add) -> void {
  rhsvals.getData()[rhsvals.getLength() - 1] += add;
}

auto xpress::XPRSProblem::RowCreator::addNonzero(int idx, double coef) -> void {
  colinds.append(idx);
  rowcoefs.append(coef);
}

auto xpress::XPRSProblem::RowCreator::addNonzero(int idx1, int idx2,
                                                 double coef) -> void {
  assert(idx1 >= 0);
  assert(idx2 >= 0);
  if (!qind1) {
    qind1 = ExtendableIntArray(std::vector<int>(8));
    qind2 = ExtendableIntArray(std::vector<int>(8));
    qcoef = ExtendableDoubleArray(std::vector<double>(8));
  }
  qind1.append(idx1);
  qind2.append(idx2);
  qcoef.append(coef);
}

auto xpress::XPRSProblem::RowCreator::addNonzeros(
    xpress::SizedArray<int const> const &idx,
    xpress::SizedArray<double const> const &coef, double factor) -> void {
  int oldLength = colinds.getLength();
  colinds.append(idx);
  rowcoefs.append(coef);
  if (factor != 1.0 && oldLength > 0) {
    std::vector<double> &r = rowcoefs.getData();
    int l = colinds.getLength();
    for (int i = oldLength; i < l; ++i)
      r[i] *= factor;
  }
}

auto xpress::XPRSProblem::RowCreator::getPostfixExtractor()
    -> xpress::objects::PostfixExtractor * {
  return this;
}

auto xpress::XPRSProblem::RowCreator::dereferenceVariable(
    xpress::objects::Variable x) -> int {
  checkForVariables();
  return x.getIndexForProb((xpress::objects::XpressProblem *)prob);
}

auto xpress::XPRSProblem::RowCreator::addToken(int type, double value)
    -> xpress::objects::PostfixExtractor & {
  assert(formulasStarted > 0);
  if (!formulaRows) {

    formulaRows = ExtendableIntArray(std::vector<int>(8));
    formulaStart = ExtendableIntArray(std::vector<int>(8));
    formulaType = ExtendableIntArray(std::vector<int>(8));
    formulaValue = ExtendableDoubleArray(std::vector<double>(8));
    formulaRows.append(nextInequality);
    formulaStart.append(0);
  } else {

    if (formulaRows.getLength() == 0 ||
        formulaRows.getData()[formulaRows.getLength() - 1] != nextInequality)
      formulaRows.append(nextInequality);
  }
  formulaType.append(type);
  formulaValue.append(value);
  return *this;
}

auto xpress::XPRSProblem::RowCreator::startExpression() -> void {
  ++formulasStarted;
  if (formulasInCurrentRow == 0) {

    addToken(XPRS_TOK_RB, 0.0);
  }
  if (formulasStarted == 1) {
    if (formulasInCurrentRow > 0) {

      addToken(XPRS_TOK_DEL, XPRS_DEL_COMMA);
    }
    ++formulasInCurrentRow;
  }
}

auto xpress::XPRSProblem::RowCreator::endExpression() -> void {
  assert(formulasStarted > 0);
  --formulasStarted;
}

auto xpress::XPRSProblem::RowCreator::setType(char type) -> void {
  if (types.getLength() == rhsvals.getLength()) {

    types.getData()[types.getLength() - 1] = type;
  } else {

    types.append(type);
  }
}

auto xpress::XPRSProblem::RowCreator::applyRange(double lb, double ub) -> void {
  if (lb > ub)
    throw std::invalid_argument("range lower bound exceeds upper bound");
  setType((char)'R');

  rhsvals.getData()[rhsvals.getLength() - 1] += ub;
  rngvals.getData()[rngvals.getLength() - 1] = ub - lb;
  flags |= FLAG_RANGE;
}

auto xpress::XPRSProblem::RowCreator::setName(std::optional<std::string> name)
    -> void {
  names.getData()[names.getLength() - 1] = name;
  if (!(!(name)))
    flags |= FLAG_NAME;
}

auto xpress::XPRSProblem::RowCreator::getRowCount() -> int {
  return firstRow + created;
}

;

auto xpress::XPRSProblem::RowCreator::estimateLinearNonzeros(
    int rows, XPRSProblem *prob) -> int {
  if (rows < 1)
    return 0;
  int estimate = nonzerosPerInequality * rows;

  if (estimate > 1000000 || estimate < 0) {
    int cols = prob->attributes.getCols();
    if (cols > 1000000 || estimate < 0)
      estimate = cols;
  }
  return estimate;
}

;

;

xpress::XPRSProblem::RowInfo::RowInfo(
    xpress::SizedArray<int const> const &ind,
    xpress::SizedArray<double const> const &val, char type, double rhs)
    : ind(ind.toVector()), val(val.toVector()), type(type), rhs(rhs) {

  for (auto &i : ind) {
    if (i < 0)
      throw std::invalid_argument("negative variable index");
  }
}

auto xpress::XPRSProblem::addRow(xpress::SizedArray<int const> const &colind,
                                 xpress::SizedArray<double const> const &colval,
                                 char rowtype, double rhs, double rng,
                                 std::optional<std::string> name) -> int {

  int idx = attributes.getRows();
  std::vector<int> row({-1});
  {
    xpress::RaiiHelper finally([&]() {
      if (row[0] != -1)
        delRows(1, row);
    });

    int cols = 0;
    if (!(!colind) != (!(!colval)))
      throw std::invalid_argument(
          "colind and colval must be either both nullptr or both non-nullptr");
    if (!(!colind) && !(!colval)) {
      if (xpress::toInt(colind.size()) != xpress::toInt(colval.size()))
        throw std::invalid_argument("array length mismatch");
      cols = xpress::toInt(xpress::toInt(colind.size()));
      addRows(1, cols, std::vector<char>({rowtype}), std::vector<double>({rhs}),
              std::vector<double>({rng}), std::vector<int>({0, cols}), colind,
              colval);
    } else {
      addRows(1, cols, std::vector<char>({rowtype}), std::vector<double>({rhs}),
              std::vector<double>({rng}), std::vector<int>({0, 0}), nullptr,
              nullptr);
    }
    row[0] = idx;
    if (!(!(name)))
      addNames(xpress::Namespaces::Row,
               std::vector<std::optional<std::string>>({name}), idx, idx);
    row[0] = -1;
  }

  return idx;
}

auto xpress::XPRSProblem::addRow(xpress::SizedArray<int const> const &colind,
                                 xpress::SizedArray<double const> const &colval,
                                 char rowtype, double rhs,
                                 std::optional<std::string> name) -> int {

  int idx = attributes.getRows();
  std::vector<int> row({-1});
  {
    xpress::RaiiHelper finally([&]() {
      if (row[0] != -1)
        delRows(1, row);
    });

    int cols = 0;
    if (!(!colind) != (!(!colval)))
      throw std::invalid_argument(
          "colind and colval must be either both nullptr or both non-nullptr");
    if (!(!colind) && !(!colval)) {
      if (xpress::toInt(colind.size()) != xpress::toInt(colval.size()))
        throw std::invalid_argument("array length mismatch");
      cols = xpress::toInt(xpress::toInt(colind.size()));
      addRows(1, cols, std::vector<char>({rowtype}), std::vector<double>({rhs}),
              nullptr, std::vector<int>({0, cols}), colind, colval);
    } else {
      addRows(1, cols, std::vector<char>({rowtype}), std::vector<double>({rhs}),
              nullptr, std::vector<int>({0, 0}), nullptr, nullptr);
    }
    row[0] = idx;
    if (!(!(name)))
      addNames(xpress::Namespaces::Row,
               std::vector<std::optional<std::string>>({name}), idx, idx);
    row[0] = -1;
  }

  return idx;
}

auto xpress::XPRSProblem::addRow(xpress::SizedArray<int const> const &colind,
                                 xpress::SizedArray<double const> const &colval,
                                 char rowtype, double rhs) -> int {
  return addRow(colind, colval, rowtype, rhs,
                (std::optional<std::string>)nullptr);
}

auto xpress::XPRSProblem::addRow(RowInfo row) -> int {
  return addRow(row.ind, row.val, row.type, row.rhs);
}

auto xpress::XPRSProblem::addRow(RowInfo row,
                                 std::optional<std::string> name) -> int {
  return addRow(row.ind, row.val, row.type, row.rhs, name);
}

auto xpress::XPRSProblem::addCut(int cuttype,
                                 xpress::SizedArray<int const> const &colind,
                                 xpress::SizedArray<double const> const &colval,
                                 char rowtype, double rhs) -> void {
  if (xpress::toInt(colind.size()) != xpress::toInt(colval.size()))
    throw std::invalid_argument("array length mismatch");

  addCuts(1, std::vector<int>({cuttype}), std::vector<char>({rowtype}),
          std::vector<double>({rhs}),
          std::vector<int>({0, xpress::toInt(xpress::toInt(colind.size()))}),
          colind, colval);
}

auto xpress::XPRSProblem::addCut(int cuttype, RowInfo cut) -> void {
  addCut(cuttype, cut.ind, cut.val, cut.type, cut.rhs);
}

auto xpress::XPRSProblem::presolveRow(
    xpress::SizedArray<int const> const &ind,
    xpress::SizedArray<double const> const &val, char type,
    double rhs) -> std::optional<xpress::XPRSProblem::RowInfo> {
  if (xpress::toInt(ind.size()) != xpress::toInt(val.size()))
    throw std::invalid_argument("array length mismatch");

  double prhs;
  int pnzs;
  int status;
  int cols = attributes.getCols();
  std::vector<int> pind(cols);
  std::vector<double> pval(cols);

  presolveRow(type, xpress::toInt(xpress::toInt(ind.size())), ind, val, rhs,
              cols, &pnzs, pind, pval, &prhs, &status);
  if (status != 0)
    return std::nullopt;
  return RowInfo(xpress::copyOfArray(pind, pnzs),
                 xpress::copyOfArray(pval, pnzs), type, prhs);
}

auto xpress::XPRSProblem::presolveRow(xpress::XPRSProblem::RowInfo row)
    -> std::optional<xpress::XPRSProblem::RowInfo> {
  return presolveRow(row.ind, row.val, row.type, row.rhs);
}

auto xpress::XPRSProblem::clearObjective() -> void {

  int cols = attributes.getCols();
  std::vector<int> ind(cols + 1);
  std::vector<double> val(cols + 1);
  for (int i = 0; i < cols; ++i)
    ind[i] = i;
  ind[cols] = -1;
  chgObj(cols + 1, ind, val);

  delQMatrix(-1);
}

auto xpress::XPRSProblem::setObjective(
    xpress::SizedArray<int const> const &ind,
    xpress::SizedArray<double const> const &val) -> void {
  clearObjective();
  chgObj(xpress::toInt(xpress::toInt(ind.size())), ind, val);
}

auto xpress::XPRSProblem::setObjective(
    xpress::SizedArray<int const> const &ind,
    xpress::SizedArray<double const> const &val,
    xpress::ObjSense sense) -> void {
  clearObjective();
  chgObj(xpress::toInt(xpress::toInt(ind.size())), ind, val);
  chgObjSense(sense);
}

auto xpress::XPRSProblem::chgLB(int j, double lb) -> void {
  chgBounds(1, std::vector<int>({j}), std::vector<char>({(char)'L'}),
            std::vector<double>({lb}));
}

auto xpress::XPRSProblem::chgUB(int j, double ub) -> void {
  chgBounds(1, std::vector<int>({j}), std::vector<char>({(char)'U'}),
            std::vector<double>({ub}));
}

auto xpress::XPRSProblem::chgBounds(int j, double lb, double ub) -> void {
  chgBounds(2, std::vector<int>({j, j}),
            std::vector<char>({(char)'L', (char)'U'}),
            std::vector<double>({lb, ub}));
}

auto xpress::XPRSProblem::drop() -> void {}

auto xpress::XPRSProblem::getNames(int type, int first,
                                   int last) -> std::vector<std::string> {
  if (last < first)
    return std::vector<std::string>(0);
  return getNameList(type, first, last);
}

auto xpress::XPRSProblem::getNames(xpress::Namespaces type, int first,
                                   int last) -> std::vector<std::string> {
  if (last < first)
    return std::vector<std::string>(0);
  return getNames((int)type, first, last);
}

auto xpress::XPRSProblem::getName(int type, int elt) -> std::string {
  return getNames(type, elt, elt)[0];
}

auto xpress::XPRSProblem::getName(xpress::Namespaces type,
                                  int elt) -> std::string {
  return getNames((int)type, elt, elt)[0];
}

auto xpress::XPRSProblem::getRowNames(int first,
                                      int last) -> std::vector<std::string> {
  return getNames(1, first, last);
}

auto xpress::XPRSProblem::getRowName(int index) -> std::string {
  return getRowNames(index, index)[0];
}

auto xpress::XPRSProblem::getColumnNames(int first,
                                         int last) -> std::vector<std::string> {
  return getNames(2, first, last);
}

auto xpress::XPRSProblem::getColumnName(int index) -> std::string {
  return getColumnNames(index, index)[0];
}

auto xpress::XPRSProblem::getSetNames(int first,
                                      int last) -> std::vector<std::string> {
  return getNames(3, first, last);
}

auto xpress::XPRSProblem::getSetName(int index) -> std::string {
  return getSetNames(index, index)[0];
}

auto xpress::XPRSProblem::getPWLNames(int first,
                                      int last) -> std::vector<std::string> {
  return getNames(4, first, last);
}

auto xpress::XPRSProblem::getPWLName(int index) -> std::string {
  return getPWLNames(index, index)[0];
}

auto xpress::XPRSProblem::getGenConsNames(int first, int last)
    -> std::vector<std::string> {
  return getNames(5, first, last);
}

auto xpress::XPRSProblem::getGenConsName(int index) -> std::string {
  return getGenConsNames(index, index)[0];
}

;

;
xpress::XPRSProblem::MatrixInfo::MatrixInfo()
    : start(std::vector<int>(0)), ind(std::vector<int>(0)),
      val(std::vector<double>(0)) {}

auto xpress::XPRSProblem::getCols(int first,
                                  int last) -> xpress::XPRSProblem::MatrixInfo {
  MatrixInfo ret = MatrixInfo();
  int count = last - first + 1;
  if (count <= 0)
    return ret;
  int ncoefs;
  ret.start = std::vector<int>(count + 1);
  getCols(ret.start, nullptr, nullptr, 0, &ncoefs, first, last);
  ret.ind = std::vector<int>(ncoefs);
  ret.val = std::vector<double>(ncoefs);
  getCols(ret.start, ret.ind, ret.val, ncoefs, &ncoefs, first, last);
  return ret;
}

auto xpress::XPRSProblem::getRows(int first,
                                  int last) -> xpress::XPRSProblem::MatrixInfo {
  MatrixInfo ret = MatrixInfo();
  int count = last - first + 1;
  if (count <= 0)
    return ret;
  int ncoefs;
  ret.start = std::vector<int>(count + 1);
  getRows(ret.start, nullptr, nullptr, 0, &ncoefs, first, last);
  ret.ind = std::vector<int>(ncoefs);
  ret.val = std::vector<double>(ncoefs);
  getRows(ret.start, ret.ind, ret.val, ncoefs, &ncoefs, first, last);
  return ret;
}

auto xpress::XPRSProblem::getMQObj(int first, int last)
    -> xpress::XPRSProblem::MatrixInfo {
  MatrixInfo ret = MatrixInfo();
  int count = last - first + 1;
  if (count <= 0)
    return ret;
  int ncoefs;
  ret.start = std::vector<int>(count + 1);
  getMQObj(ret.start, nullptr, nullptr, 0, &ncoefs, first, last);
  ret.ind = std::vector<int>(ncoefs);
  ret.val = std::vector<double>(ncoefs);
  getMQObj(ret.start, ret.ind, ret.val, ncoefs, &ncoefs, first, last);
  return ret;
}

;

;

;

;

xpress::XPRSProblem::IISData::IISData(int rownumber, int colnumber)
    : rowind(std::vector<int>(rownumber)), colind(std::vector<int>(colnumber)),
      contype(std::vector<char>(rownumber)),
      bndtype(std::vector<char>(colnumber)),
      duals(std::vector<double>(rownumber)),
      djs(std::vector<double>(colnumber)),
      isolationrows(std::vector<char>(rownumber)),
      isolationcols(std::vector<char>(colnumber)) {}

auto xpress::XPRSProblem::getIISData(int number)
    -> xpress::XPRSProblem::IISData {
  int rownumber;
  int colnumber;
  getIISData(number, &rownumber, &colnumber, nullptr, nullptr, nullptr, nullptr,
             nullptr, nullptr, nullptr, nullptr);
  IISData ret = IISData(rownumber, colnumber);
  getIISData(number, &rownumber, &colnumber, ret.rowind, ret.colind,
             ret.contype, ret.bndtype, ret.duals, ret.djs, ret.isolationrows,
             ret.isolationcols);
  return ret;
}

;

;

xpress::XPRSProblem::IISStatusInfo::IISStatusInfo()
    : rowsizes(std::vector<int>(0)), colsizes(std::vector<int>(0)),
      suminfeas(std::vector<double>(0)), numinfeas(std::vector<int>(0)) {}

xpress::XPRSProblem::IISStatusInfo::IISStatusInfo(int iiscount)
    : rowsizes(std::vector<int>(iiscount + 1)),
      colsizes(std::vector<int>(iiscount + 1)),
      suminfeas(std::vector<double>(iiscount + 1)),
      numinfeas(std::vector<int>(iiscount + 1)) {}

auto xpress::XPRSProblem::IISStatus() -> xpress::XPRSProblem::IISStatusInfo {
  int iiscount;
  IISStatus(&iiscount, nullptr, nullptr, nullptr, nullptr);
  if (iiscount <= 0)
    return IISStatusInfo();
  IISStatusInfo ret = IISStatusInfo(iiscount);
  IISStatus(&iiscount, ret.rowsizes, ret.colsizes, ret.suminfeas,
            ret.numinfeas);
  return ret;
}

auto xpress::XPRSProblem::optimize(std::optional<std::string> flags) -> int {
  int solvestatus;
  int solstatus;
  optimize(flags, &solvestatus, &solstatus);
  return solvestatus;
}

auto xpress::XPRSProblem::optimize() -> int {
  return optimize("");
}

;

;
xpress::XPRSProblem::Solution::Solution()
    : x({}), slack({}), duals({}), djs({}) {}
xpress::XPRSProblem::Solution::Solution(int rows, int cols, bool hasDuals)
    : x(std::vector<double>(cols)), slack(std::vector<double>(rows)),
      duals(hasDuals ? std::vector<double>(rows) : std::vector<double>()),
      djs(hasDuals ? std::vector<double>(cols) : std::vector<double>()){}

;
xpress::XPRSProblem::StatusSolution::StatusSolution()
    : Solution(), status(-1) {}
xpress::XPRSProblem::StatusSolution::StatusSolution(int rows, int cols,
                                                    bool duals)
    : Solution(rows, cols, duals), status(-1){}

;

;

;

;

;

auto xpress::XPRSProblem::AbstractUserFunction::triggerEvalError() -> void {
  prob->interrupt(xpress::StopType::GenericError);
  prob->nlpSetFunctionError();
}

auto xpress::XPRSProblem::AbstractUserFunction::triggerEvalError(
    std::exception_ptr e) -> void {
  prob->setCBException(e);
  prob->interrupt(xpress::StopType::GenericError);
  prob->nlpSetFunctionError();
}
xpress::XPRSProblem::AbstractUserFunction::AbstractUserFunction(
    XPRSProblem *owner, std::string name)
    : id(-1), name(name), prob(owner) {}
auto xpress::XPRSProblem::AbstractUserFunction::setId(int functionId) -> void {

  this->id = functionId;
}

auto xpress::XPRSProblem::AbstractUserFunction::getId() -> int { return id; }

auto xpress::XPRSProblem::AbstractUserFunction::getName() const -> std::string {
  return name;
}

;

auto xpress::XPRSProblem::AbstractUserFunction::evaluate(
    xpress::SizedArray<double const> const &args) -> double {
  return evaluate(1, args);
}

;

auto xpress::XPRSProblem::AbstractUserFunction::call(
    std::vector<xpress::objects::Expression> arguments)
    -> xpress::objects::Expression {
  return call(1, arguments);
}

;
xpress::XPRSProblem::MapFunction::MapFunction(XPRSProblem *prob,
                                              std::string name,
                                              MapFunctor functor)
    : AbstractUserFunction(prob, name), functor(functor) {}

auto xpress::XPRSProblem::MapFunction::isMultiOutput() -> bool { return false; }
auto xpress::XPRSProblem::MapFunction::call(double value) -> double {
  return functor(value);
}

auto xpress::XPRSProblem::MapFunction::evaluate(
    int, xpress::SizedArray<double const> const &args) -> double {
  if (xpress::toInt(args.size()) != 1)
    throw std::invalid_argument("function requires exactly one argument");
  return call(args[0]);
}

auto xpress::XPRSProblem::MapFunction::call(
    int, std::vector<xpress::objects::Expression> arguments)
    -> xpress::objects::Expression {
  if (xpress::toInt(arguments.size()) != 1)
    throw std::invalid_argument("function requires exactly one argument");
  return xpress::objects::UserFunctionExpression(this, arguments);
}

;
xpress::XPRSProblem::VecMapFunction::VecMapFunction(XPRSProblem *prob,
                                                    std::string name,
                                                    VecMapFunctor functor,
                                                    int nIn)
    : AbstractUserFunction(prob, name), nIn(nIn), functor(functor) {}

auto xpress::XPRSProblem::VecMapFunction::isMultiOutput() -> bool {
  return false;
}
auto xpress::XPRSProblem::VecMapFunction::call(
    xpress::SizedArray<double const> const &values) -> double {
  return functor(values);
}

auto xpress::XPRSProblem::VecMapFunction::evaluate(
    int, xpress::SizedArray<double const> const &args) -> double {
  if (xpress::toInt(args.size()) != nIn)
    throw std::invalid_argument("incorrect argument count");
  return call(args);
}

auto xpress::XPRSProblem::VecMapFunction::call(
    int, std::vector<xpress::objects::Expression> arguments)
    -> xpress::objects::Expression {
  if (xpress::toInt(arguments.size()) != nIn)
    throw std::invalid_argument("incorrect argument count");
  return xpress::objects::UserFunctionExpression(this, arguments);
}

;

;
xpress::XPRSProblem::MultiMapFunction::MultiMapFunction(XPRSProblem *prob,
                                                        std::string name,
                                                        MultiMapFunctor functor,
                                                        int nIn, int nOut)
    : AbstractUserFunction(prob, name), functor(functor), nIn(nIn), nOut(nOut) {
}

auto xpress::XPRSProblem::MultiMapFunction::isMultiOutput() -> bool {
  return true;
}
auto xpress::XPRSProblem::MultiMapFunction::call(
    xpress::SizedArray<double const> const &values) -> std::vector<double> {
  return functor(values);
}

auto xpress::XPRSProblem::MultiMapFunction::evaluate(
    int output, xpress::SizedArray<double const> const &args) -> double {
  if (xpress::toInt(args.size()) != nIn)
    throw std::invalid_argument("incorrect argument count");
  return call(args)[output - 1];
}

auto xpress::XPRSProblem::MultiMapFunction::call(
    int output, std::vector<xpress::objects::Expression> arguments)
    -> xpress::objects::Expression {
  if (xpress::toInt(arguments.size()) != nIn)
    throw std::invalid_argument("incorrect argument count");
  return xpress::objects::UserFunctionExpression(this, output, arguments);
}

;
xpress::XPRSProblem::MapDeltaFunction::MapDeltaFunction(XPRSProblem *prob,
                                                        std::string name,
                                                        MapDeltaFunctor functor)
    : AbstractUserFunction(prob, name), functor(functor) {}

auto xpress::XPRSProblem::MapDeltaFunction::isMultiOutput() -> bool {
  return false;
}
auto xpress::XPRSProblem::MapDeltaFunction::call(
    double value, double delta,
    xpress::SizedArray<double> const &partial) -> double {
  return functor(value, delta, partial);
}

auto xpress::XPRSProblem::MapDeltaFunction::evaluate(
    int, xpress::SizedArray<double const> const &args) -> double {
  if (xpress::toInt(args.size()) != 1)
    throw std::invalid_argument("function requires exactly one argument");
  return call(args[0], 0.0, nullptr);
}

auto xpress::XPRSProblem::MapDeltaFunction::call(
    int, std::vector<xpress::objects::Expression> arguments)
    -> xpress::objects::Expression {
  if (xpress::toInt(arguments.size()) != 1)
    throw std::invalid_argument("function requires exactly one argument");
  return xpress::objects::UserFunctionExpression(this, arguments);
}

;
xpress::XPRSProblem::VecMapDeltaFunction::VecMapDeltaFunction(
    XPRSProblem *prob, std::string name, VecMapDeltaFunctor functor, int nIn)
    : AbstractUserFunction(prob, name), functor(functor), nIn(nIn) {}

auto xpress::XPRSProblem::VecMapDeltaFunction::isMultiOutput() -> bool {
  return false;
}
auto xpress::XPRSProblem::VecMapDeltaFunction::call(
    xpress::SizedArray<double const> const &values,
    xpress::SizedArray<double const> const &deltas,
    xpress::SizedArray<double> const &partials) -> double {
  return functor(values, deltas, partials);
}

auto xpress::XPRSProblem::VecMapDeltaFunction::evaluate(
    int, xpress::SizedArray<double const> const &args) -> double {
  if (xpress::toInt(args.size()) != nIn)
    throw std::invalid_argument("incorrect argument count");
  return call(args, nullptr, nullptr);
}

auto xpress::XPRSProblem::VecMapDeltaFunction::call(
    int, std::vector<xpress::objects::Expression> arguments)
    -> xpress::objects::Expression {
  if (xpress::toInt(arguments.size()) != nIn)
    throw std::invalid_argument("incorrect argument count");
  return xpress::objects::UserFunctionExpression(this, arguments);
}

;

;
xpress::XPRSProblem::MultiMapDeltaFunction::MultiMapDeltaFunction(
    XPRSProblem *prob, std::string name, MultiMapDeltaFunctor functor, int nIn,
    int nOut)
    : AbstractUserFunction(prob, name), functor(functor), nIn(nIn), nOut(nOut) {
}

auto xpress::XPRSProblem::MultiMapDeltaFunction::isMultiOutput() -> bool {
  return true;
}

auto xpress::XPRSProblem::MultiMapDeltaFunction::call(
    xpress::SizedArray<double const> const &values,
    xpress::SizedArray<double const> const &deltas,
    xpress::SizedArray<double> const &partials) -> std::vector<double> {
  return functor(values, deltas, partials);
}

auto xpress::XPRSProblem::MultiMapDeltaFunction::evaluate(
    int output, xpress::SizedArray<double const> const &args) -> double {
  if (xpress::toInt(args.size()) != nIn)
    throw std::invalid_argument("incorrect argument count");
  return call(args, nullptr, nullptr)[output - 1];
}

auto xpress::XPRSProblem::MultiMapDeltaFunction::call(
    int output, std::vector<xpress::objects::Expression> arguments)
    -> xpress::objects::Expression {
  if (xpress::toInt(arguments.size()) != nIn)
    throw std::invalid_argument("incorrect argument count");
  return xpress::objects::UserFunctionExpression(this, output, arguments);
}

extern "C" {
static inline double xpress_ufunmap(double value, void *ctx) {
  xpress::XPRSProblem::MapFunction *f =
      reinterpret_cast<xpress::XPRSProblem::MapFunction *>(ctx);
  try {
    return f->call(value);
  } catch (...) {
    f->triggerEvalError(std::current_exception());
    return std::numeric_limits<double>::quiet_NaN();
  }
}
static inline double xpress_ufunvecmap(double const *value, void *ctx) {
  xpress::XPRSProblem::VecMapFunction *f =
      reinterpret_cast<xpress::XPRSProblem::VecMapFunction *>(ctx);
  try {
    return f->call(xpress::SizedArray<double const>(value, f->nIn));
  } catch (...) {
    f->triggerEvalError(std::current_exception());
    return std::numeric_limits<double>::quiet_NaN();
  }
}
static inline int xpress_ufunmultimap(double const *value, double *evaluation,
                                      void *ctx) {
  xpress::XPRSProblem::MultiMapFunction *f =
      reinterpret_cast<xpress::XPRSProblem::MultiMapFunction *>(ctx);
  try {
    std::vector<double> res =
        f->call(xpress::SizedArray<double const>(value, f->nIn));
    std::memcpy(evaluation, res.data(), sizeof(*evaluation) * f->nOut);
    return 0;
  } catch (...) {
    f->triggerEvalError(std::current_exception());
    return -1;
  }
}
static inline int xpress_ufunmapdelta(double value, double delta,
                                      double *p_evaluation, double *p_partial,
                                      void *ctx) {
  xpress::XPRSProblem::MapDeltaFunction *f =
      reinterpret_cast<xpress::XPRSProblem::MapDeltaFunction *>(ctx);
  try {
    if (p_partial)
      *p_evaluation =
          f->call(value, delta, xpress::SizedArray<double>(p_partial, 1));
    else
      *p_evaluation = f->call(value, delta, nullptr);
    return 0;
  } catch (...) {
    f->triggerEvalError(std::current_exception());
    return -1;
  }
}
static inline int xpress_ufunvecmapdelta(double const *values,
                                         double const *deltas,
                                         double *p_evaluation, double *partials,
                                         void *ctx) {
  xpress::XPRSProblem::VecMapDeltaFunction *f =
      reinterpret_cast<xpress::XPRSProblem::VecMapDeltaFunction *>(ctx);
  try {
    *p_evaluation =
        f->call(xpress::SizedArray<double const>(values, f->nIn),
                deltas ? xpress::SizedArray<double const>(deltas, f->nIn)
                       : xpress::SizedArray<double const>(nullptr),
                partials ? xpress::SizedArray<double>(partials, f->nIn)
                         : xpress::SizedArray<double>(nullptr));
    return 0;
  } catch (...) {
    f->triggerEvalError(std::current_exception());
    return -1;
  }
}
static inline int xpress_ufunmultimapdelta(double const *value,
                                           double const *deltas, double *output,
                                           void *ctx) {
  xpress::XPRSProblem::MultiMapDeltaFunction *f =
      reinterpret_cast<xpress::XPRSProblem::MultiMapDeltaFunction *>(ctx);
  try {
    std::vector<double> partials(deltas ? (f->nOut * f->nIn) : 0);
    std::vector<double> res = f->call(
        xpress::SizedArray<double const>(value, f->nIn),
        deltas ? xpress::SizedArray<double const>(deltas, f->nIn)
               : xpress::SizedArray<double const>(nullptr),
        deltas ? xpress::SizedArray<double>(partials.data(), f->nOut * f->nIn)
               : xpress::SizedArray<double>(nullptr));

    for (int out = 0; out < f->nOut; ++out) {
      output[out * (f->nIn + 1)] = res[out];
      if (deltas)
        std::memcpy(output + out * (f->nIn + 1) + 1,
                    partials.data() + out * f->nIn,
                    sizeof(*output) * out * f->nIn);
    }
    return 0;
  } catch (...) {
    f->triggerEvalError(std::current_exception());
    return -1;
  }
}
}

auto xpress::XPRSProblem::nlpAddUserFunction(
    std::optional<std::string> funcname, int options,
    MapFunctor functor) -> xpress::XPRSProblem::MapFunction * {
  if (!(funcname))
    throw std::invalid_argument("cannot add function without name");
  std::unique_ptr<MapFunction> f =
      std::make_unique<MapFunction>(this, funcname.value(), functor);
  int fid = -1;
  check(XPRSnlpadduserfunction(
      ptr, funcname.value().c_str(), XPRS_USERFUNCTION_MAP, 1, 1, options,
      reinterpret_cast<XPRSfunctionptr>(xpress_ufunmap),
      reinterpret_cast<void *>(f.get()), &fid));
  f->setId(fid);
  return f.release();
}

auto xpress::XPRSProblem::nlpAddUserFunction(
    std::optional<std::string> funcname, int nIn, int options,
    VecMapFunctor functor) -> xpress::XPRSProblem::VecMapFunction * {
  if (!(funcname))
    throw std::invalid_argument("cannot add function without name");
  std::unique_ptr<VecMapFunction> f =
      std::make_unique<VecMapFunction>(this, funcname.value(), functor, nIn);
  int fid = -1;
  check(XPRSnlpadduserfunction(
      ptr, funcname.value().c_str(), XPRS_USERFUNCTION_VECMAP, nIn, 1, options,
      reinterpret_cast<XPRSfunctionptr>(xpress_ufunvecmap),
      reinterpret_cast<void *>(f.get()), &fid));
  f->setId(fid);
  return f.release();
}

auto xpress::XPRSProblem::nlpAddUserFunction(
    std::optional<std::string> funcname, int nIn, int nOut, int options,
    MultiMapFunctor functor) -> xpress::XPRSProblem::MultiMapFunction * {
  if (!(funcname))
    throw std::invalid_argument("cannot add function without name");
  std::unique_ptr<MultiMapFunction> f = std::make_unique<MultiMapFunction>(
      this, funcname.value(), functor, nIn, nOut);
  int fid = -1;
  check(XPRSnlpadduserfunction(
      ptr, funcname.value().c_str(), XPRS_USERFUNCTION_MULTIMAP, nIn, nOut,
      options, reinterpret_cast<XPRSfunctionptr>(xpress_ufunmultimap),
      reinterpret_cast<void *>(f.get()), &fid));
  f->setId(fid);
  return f.release();
}

auto xpress::XPRSProblem::nlpAddUserFunction(
    std::optional<std::string> funcname, int options,
    MapDeltaFunctor functor) -> xpress::XPRSProblem::MapDeltaFunction * {
  if (!(funcname))
    throw std::invalid_argument("cannot add function without name");
  std::unique_ptr<MapDeltaFunction> f =
      std::make_unique<MapDeltaFunction>(this, funcname.value(), functor);
  int fid = -1;
  check(XPRSnlpadduserfunction(
      ptr, funcname.value().c_str(), XPRS_USERFUNCTION_MAPDELTA, 1, 1, options,
      reinterpret_cast<XPRSfunctionptr>(xpress_ufunmapdelta),
      reinterpret_cast<void *>(f.get()), &fid));
  f->setId(fid);
  return f.release();
}

auto xpress::XPRSProblem::nlpAddUserFunction(
    std::optional<std::string> funcname, int nIn, int options,
    VecMapDeltaFunctor functor) -> xpress::XPRSProblem::VecMapDeltaFunction * {
  if (!(funcname))
    throw std::invalid_argument("cannot add function without name");
  std::unique_ptr<VecMapDeltaFunction> f =
      std::make_unique<VecMapDeltaFunction>(this, funcname.value(), functor,
                                            nIn);
  int fid = -1;
  check(XPRSnlpadduserfunction(
      ptr, funcname.value().c_str(), XPRS_USERFUNCTION_VECMAPDELTA, nIn, 1,
      options, reinterpret_cast<XPRSfunctionptr>(xpress_ufunvecmapdelta),
      reinterpret_cast<void *>(f.get()), &fid));
  f->setId(fid);
  return f.release();
}

auto xpress::XPRSProblem::nlpAddUserFunction(
    std::optional<std::string> funcname, int nIn, int nOut, int options,
    MultiMapDeltaFunctor functor)
    -> xpress::XPRSProblem::MultiMapDeltaFunction * {
  if (!(funcname))
    throw std::invalid_argument("cannot add function without name");
  std::unique_ptr<MultiMapDeltaFunction> f =
      std::make_unique<MultiMapDeltaFunction>(this, funcname.value(), functor,
                                              nIn, nOut);
  int fid = -1;
  check(XPRSnlpadduserfunction(
      ptr, funcname.value().c_str(), XPRS_USERFUNCTION_MULTIMAPDELTA, nIn, nOut,
      options, reinterpret_cast<XPRSfunctionptr>(xpress_ufunmultimapdelta),
      reinterpret_cast<void *>(f.get()), &fid));
  f->setId(fid);
  return f.release();
}

auto xpress::XPRSProblem::mapToArrays(
    std::unordered_map<int, double> const &map)
    -> std::pair<std::vector<int>, std::vector<double>> {
  std::vector<int> key(0);
  std::vector<double> val(0);
  if (map.size() > 0) {
    key = std::vector<int>(map.size());
    val = std::vector<double>(map.size());
    int idx = 0;
    for (auto &e : map) {
      key[idx] = e.first;
      val[idx] = e.second;
      ++idx;
    }
  }
  return std::pair<std::vector<int>, std::vector<double>>(key, val);
}

;

;

;

;

;

xpress::XPRSProblem::ConstraintCreator::PWLData::PWLData(XPRSProblem *xprob,
                                                         bool hasNames)
    : firstPWL(xprob->attributes.getPwlCons()), nextPWL(firstPWL), created(0),
      arguments(ExtendableIntArray(std::vector<int>(8))),
      resultants(ExtendableIntArray(std::vector<int>(8))),
      start(ExtendableIntArray(std::vector<int>(8))),
      xvals(ExtendableDoubleArray(std::vector<double>(8))),
      yvals(ExtendableDoubleArray(std::vector<double>(8))),
      names(hasNames ? ExtendableArray<std::optional<std::string>>(
                           std::vector<std::optional<std::string>>(8))
                     : nullptr),
      anyName(false) {}

auto xpress::XPRSProblem::ConstraintCreator::PWLData::addPWL(
    int argument, int resultant, xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval,
    std::optional<std::string> name) -> void {
  assert(xpress::toInt(xval.size()) == xpress::toInt(yval.size()));
  arguments.append(argument);
  resultants.append(resultant);
  start.append(xvals.getLength());
  xvals.append(xval);
  yvals.append(yval);
  if (!(!names)) {
    if (!(!(name)))
      anyName = true;
    names.append(name);
  }
}

auto xpress::XPRSProblem::ConstraintCreator::PWLData::commit(XPRSProblem *xprob)
    -> void {
  if (start.getLength() == 0)
    return;
  int oldCreated = created;
  xprob->addPwlCons(start.getLength(), xvals.getLength(), arguments.getData(),
                    resultants.getData(), start.getData(), xvals.getData(),
                    yvals.getData());
  created += start.getLength();
  if (anyName) {
    assert(!(!names));
    xprob->addNamesWithNull(xpress::Namespaces::PwlCons, firstPWL + oldCreated,
                            created - oldCreated, names.getData());
  }
  arguments.reset();
  resultants.reset();
  start.reset();
  xvals.reset();
  yvals.reset();
  if (!(!names))
    names.reset();
  anyName = false;
}

auto xpress::XPRSProblem::ConstraintCreator::PWLData::undo(XPRSProblem *xprob)
    -> void {

  start.setLength(start.getCapacity());
  int chunkLen = start.getCapacity();
  start.reset();
  for (int i = 0; i < created; ++i) {
    start.append(firstPWL + i);
    if (start.getLength() == chunkLen) {
      xprob->delPwlCons(start.getLength(), start.getData());
      start.reset();
    }
  }
  if (start.getLength() > 0) {
    xprob->delPwlCons(start.getLength(), start.getData());
  }
}

;

;

;

;

;

;

xpress::XPRSProblem::ConstraintCreator::GenConsData::GenConsData(
    XPRSProblem *prob, bool hasNames)
    : firstCon(prob->attributes.getGenCons()), created(0),
      colstart(ExtendableIntArray(std::vector<int>(8))),
      valstart(ExtendableIntArray(std::vector<int>(8))),
      resultants(ExtendableIntArray(std::vector<int>(8))),
      types(ExtendableArray<xpress::GenConsType>(
          std::vector<xpress::GenConsType>(8))),
      colinds(ExtendableIntArray(std::vector<int>(8))),
      values(ExtendableDoubleArray(std::vector<double>(8))),
      names(hasNames ? ExtendableArray<std::optional<std::string>>(
                           std::vector<std::optional<std::string>>(8))
                     : nullptr),
      nameCached(false) {}

auto xpress::XPRSProblem::ConstraintCreator::GenConsData::commit(
    XPRSProblem *xprob) -> void {
  if (types.getLength() < 1)
    return;
  xprob->addGenCons(types.getLength(), colinds.getLength(), values.getLength(),
                    types.getData(), resultants.getData(), colstart.getData(),
                    colinds.getData(), valstart.getData(), values.getData());
  int oldCreated = created;
  created += types.getLength();
  if (nameCached) {
    assert(!(!names));
    xprob->addNamesWithNull(xpress::Namespaces::GenCons, firstCon + oldCreated,
                            created - oldCreated, names.getData());
    names.reset();
    nameCached = false;
  }
  types.reset();
  resultants.reset();
  colstart.reset();
  valstart.reset();
  colinds.reset();
  values.reset();
}

auto xpress::XPRSProblem::ConstraintCreator::GenConsData::addGeneralConstraint(
    int resultant, xpress::GenConsType type,
    xpress::SizedArray<int const> const &colind,
    xpress::SizedArray<double const> const &value,
    std::optional<std::string> name) -> void {
  colstart.append(colinds.getLength());
  valstart.append(values.getLength());
  resultants.append(resultant);
  types.append(type);
  if (!(!colind))
    colinds.append(colind);
  if (!(!value))
    values.append(value);
  if (!(!names)) {
    if (!(!(name)))
      nameCached = true;
    names.append(name);
  }
}

auto xpress::XPRSProblem::ConstraintCreator::GenConsData::undo(
    XPRSProblem *xprob) -> void {
  if (created > 0) {

    ExtendableIntArray toDelete = colstart;
    toDelete.setLength(toDelete.getCapacity());
    int capacity = toDelete.getCapacity();
    assert(capacity > 0);
    toDelete.reset();
    for (int i = 0; i < created; ++i) {
      toDelete.append(firstCon + i);
      if (toDelete.getLength() == capacity) {
        xprob->delGenCons(toDelete.getLength(), toDelete.getData());
        toDelete.reset();
      }
    }
    if (toDelete.getLength() > 0) {
      xprob->delGenCons(toDelete.getLength(), toDelete.getData());
    }
  }
}

auto xpress::XPRSProblem::ConstraintCreator::GenConsData::shouldCommit(
    int threshold) -> bool {
  return std::max(colinds.getLength(), values.getLength()) >= threshold;
}

;

;

;

;

;

xpress::XPRSProblem::ConstraintCreator::SetData::SetData(XPRSProblem *prob,
                                                         bool hasNames)
    : firstSet(prob->attributes.getSets()), setsAdded(0),
      start(ExtendableIntArray(std::vector<int>(8))),
      types(ExtendableTypeArray(std::vector<char>(8))),
      sosElems(ExtendableIntArray(std::vector<int>(8))),
      sosWeights(ExtendableDoubleArray(std::vector<double>(8))),
      names(hasNames ? ExtendableArray<std::optional<std::string>>(
                           std::vector<std::optional<std::string>>(8))
                     : nullptr),
      nameCached(false) {}

auto xpress::XPRSProblem::ConstraintCreator::SetData::addSet(
    SetType type, xpress::SizedArray<int const> const &elements,
    xpress::SizedArray<double const> const &weights,
    std::optional<std::string> name) -> void {
  start.append(sosElems.getLength());
  types.append((char)type);
  sosElems.append(elements);
  if (!weights)
    for (int i = 1; i <= xpress::toInt(elements.size()); ++i)
      sosWeights.append(i);
  else
    sosWeights.append(weights);
  if (!(!names)) {
    if (!(!(name)))
      nameCached = true;
    names.append(name);
  }
}

auto xpress::XPRSProblem::ConstraintCreator::SetData::getNZs() -> int {
  return sosElems.getLength();
}

auto xpress::XPRSProblem::ConstraintCreator::SetData::commit(XPRSProblem *xprob)
    -> void {
  if (start.getLength() < 1)
    return;
  int oldAdded = setsAdded;
  xprob->addSets(start.getLength(), sosElems.getLength(), types.getData(),
                 start.getData(), sosElems.getData(), sosWeights.getData());
  setsAdded += start.getLength();
  if (nameCached) {
    assert(!(!names));
    xprob->addNamesWithNull(xpress::Namespaces::Set, firstSet + oldAdded,
                            setsAdded - oldAdded, names.getData());
  }
  start.reset();
  types.reset();
  sosElems.reset();
  sosWeights.reset();
  if (!(!names))
    names.reset();
  nameCached = false;
}

auto xpress::XPRSProblem::ConstraintCreator::SetData::undo(XPRSProblem *xprob)
    -> void {

  start.setLength(start.getCapacity());
  int maxChunk = start.getCapacity();
  assert(maxChunk > 0);
  start.reset();
  for (int i = 0; i < setsAdded; ++i) {
    start.append(firstSet + i);
    if (start.getLength() == maxChunk) {
      xprob->delSets(start.getLength(), start.getData());
      start.reset();
    }
  }
  if (start.getLength() > 0)
    xprob->delSets(start.getLength(), start.getData());
}

;

xpress::XPRSProblem::ConstraintCreator::ConstraintCreator(XPRSProblem *prob,
                                                          bool hasNames,
                                                          bool errorIfVariables)
    : xpress::XPRSProblem::RowCreator(prob, 16, 32, hasNames, errorIfVariables),
      pwl(nullptr), gencons(nullptr), setData(nullptr) {}

xpress::XPRSProblem::ConstraintCreator::ConstraintCreator(XPRSProblem *prob,
                                                          int initialRowsSize,
                                                          int initialNzsSize,
                                                          bool hasNames,
                                                          bool errorIfVariables)
    : xpress::XPRSProblem::RowCreator(prob, initialRowsSize, initialNzsSize,
                                      hasNames, errorIfVariables),
      pwl(nullptr), gencons(nullptr), setData(nullptr) {}

auto xpress::XPRSProblem::ConstraintCreator::undo() -> void {
  RowCreator::undo();
  if (!(!pwl))
    pwl->undo(prob);
  if (!(!gencons))
    gencons->undo(prob);
  if (!(!setData))
    setData->undo(prob);
}

auto xpress::XPRSProblem::ConstraintCreator::conditionalCommit() -> void {
  if ((!(!gencons) && gencons->shouldCommit(nonzeroThreshold)) ||
      (!(!setData) && setData->getNZs() >= nonzeroThreshold) ||
      (!(!pwl) && pwl->xvals.getLength() >= nonzeroThreshold)) {
    commit();
    return;
  }
  RowCreator::conditionalCommit();
}

auto xpress::XPRSProblem::ConstraintCreator::commit() -> void {
  RowCreator::commit();
  if (!(!pwl))
    pwl->commit(prob);
  if (!(!gencons))
    gencons->commit(prob);
  if (!(!setData))
    setData->commit(prob);
}

auto xpress::XPRSProblem::ConstraintCreator::addPWL(
    int argument, int resultant, xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval,
    std::optional<std::string> name) -> void {
  if (!pwl)
    pwl = std::make_shared<PWLData>(prob, !(!names));
  pwl->addPWL(argument, resultant, xval, yval, name);
  conditionalCommit();
}

auto xpress::XPRSProblem::ConstraintCreator::addGeneralConstraint(
    int resultant, xpress::GenConsType type,
    xpress::SizedArray<int const> const &colind,
    xpress::SizedArray<double const> const &value,
    std::optional<std::string> name) -> void {
  if (!gencons)
    gencons = std::make_shared<GenConsData>(prob, !(!names));
  gencons->addGeneralConstraint(resultant, type, colind, value, name);
  conditionalCommit();
}

auto xpress::XPRSProblem::ConstraintCreator::addSet(
    SetType type, xpress::SizedArray<int const> const &elems,
    xpress::SizedArray<double const> const &weight,
    std::optional<std::string> name) -> void {
  if (!setData)
    setData = std::make_shared<SetData>(prob, !(!names));
  setData->addSet(type, elems, weight, name);
  conditionalCommit();
}

} // namespace xpress

xpress_diags_push xpress_ignore_deprecated xpress_diags_pop void
xpress::MIPSolPool::destroy() {

  check(XPRS_msp_destroy(ptr));
}

void xpress::MIPSolPool::probAttach(XPRSProblem const &prob) {

  check(XPRS_msp_probattach(ptr, prob.ptr));
}

void xpress::MIPSolPool::probDetach(XPRSProblem const &prob) {

  check(XPRS_msp_probdetach(ptr, prob.ptr));
}

void xpress::MIPSolPool::getSolList(XPRSProblem const &prob_to_rank_against,
                                    int iRankAttrib, int bRankAscending,
                                    int iRankFirstIndex_Ob,
                                    int iRankLastIndex_Ob,
                                    Array<int> const &iSolutionIds_Zb,
                                    int *nReturnedSolIds, int *nSols) {

  check(XPRS_msp_getsollist(ptr, prob_to_rank_against.ptr, iRankAttrib,
                            bRankAscending, iRankFirstIndex_Ob,
                            iRankLastIndex_Ob, iSolutionIds_Zb.data,
                            nReturnedSolIds, nSols));
}

void xpress::MIPSolPool::getSolList2(
    XPRSProblem const &prob_to_rank_against, int iRankAttrib,
    int bRankAscending, int iRankFirstIndex_Ob, int iRankLastIndex_Ob,
    int bUseUserBitFilter, int iUserBitMask, int iUserBitPattern,
    int bUseInternalBitFilter, int iInternalBitMask, int iInternalBitPattern,
    Array<int> const &iSolutionIds_Zb, int *nReturnedSolIds, int *nSols) {

  check(XPRS_msp_getsollist2(
      ptr, prob_to_rank_against.ptr, iRankAttrib, bRankAscending,
      iRankFirstIndex_Ob, iRankLastIndex_Ob, bUseUserBitFilter, iUserBitMask,
      iUserBitPattern, bUseInternalBitFilter, iInternalBitMask,
      iInternalBitPattern, iSolutionIds_Zb.data, nReturnedSolIds, nSols));
}

void xpress::MIPSolPool::getSol(int iSolutionId, int *iSolutionIdStatus_,
                                Array<double> const &x, int iColFirst,
                                int iColLast, int *nValuesReturned) {

  check(XPRS_msp_getsol(ptr, iSolutionId, iSolutionIdStatus_, x.data, iColFirst,
                        iColLast, nValuesReturned));
}

void xpress::MIPSolPool::getSlack(XPRSProblem const &prob_to_rank_against,
                                  int iSolutionId, int *iSolutionIdStatus_,
                                  Array<double> const &slack, int iRowFirst,
                                  int iRowLast, int *nValuesReturned) {

  check(XPRS_msp_getslack(ptr, prob_to_rank_against.ptr, iSolutionId,
                          iSolutionIdStatus_, slack.data, iRowFirst, iRowLast,
                          nValuesReturned));
}

void xpress::MIPSolPool::loadSol(
    int *iSolutionId, Array<double const> const &x, int nCols,
    std::optional<std::string> const &sSolutionName,
    int *bNameModifiedForUniqueness,
    int *iSolutionIdOfExistingDuplicatePreventedLoad) {

  check(XPRS_msp_loadsol(
      ptr, iSolutionId, x.data, nCols,
      sSolutionName ? sSolutionName.value().c_str() : nullptr,
      bNameModifiedForUniqueness, iSolutionIdOfExistingDuplicatePreventedLoad));
}

auto xpress::MIPSolPool::delSol(int iSolutionId) -> int {

  int iSolutionIdStatus_;
  check(XPRS_msp_delsol(ptr, iSolutionId, &iSolutionIdStatus_));
  return iSolutionIdStatus_;
}

void xpress::MIPSolPool::getIntAttribProbSol(
    XPRSProblem const &prob_to_rank_against, int iSolutionId,
    int *iSolutionIdStatus_, int iAttribId, int *Dst) const {

  check(XPRS_msp_getintattribprobsol(ptr, prob_to_rank_against.ptr, iSolutionId,
                                     iSolutionIdStatus_, iAttribId, Dst));
}

void xpress::MIPSolPool::getDblAttribProbSol(
    XPRSProblem const &prob_to_rank_against, int iSolutionId,
    int *iSolutionIdStatus_, int iAttribId, double *Dst) const {

  check(XPRS_msp_getdblattribprobsol(ptr, prob_to_rank_against.ptr, iSolutionId,
                                     iSolutionIdStatus_, iAttribId, Dst));
}

auto xpress::MIPSolPool::getIntAttribProb(XPRSProblem const &prob,
                                          int iAttribId) const -> int {

  int Dst;
  check(XPRS_msp_getintattribprob(ptr, prob.ptr, iAttribId, &Dst));
  return Dst;
}

auto xpress::MIPSolPool::getDblAttribProb(XPRSProblem const &prob,
                                          int iAttribId) const -> double {

  double Dst;
  check(XPRS_msp_getdblattribprob(ptr, prob.ptr, iAttribId, &Dst));
  return Dst;
}

void xpress::MIPSolPool::getIntAttribSol(int iSolutionId,
                                         int *iSolutionIdStatus_, int iAttribId,
                                         int *Dst) const {

  check(XPRS_msp_getintattribsol(ptr, iSolutionId, iSolutionIdStatus_,
                                 iAttribId, Dst));
}

auto xpress::MIPSolPool::getIntAttribSol(int iSolutionId,
                                         int *iSolutionIdStatus_,
                                         int iAttribId) const -> int {

  int Dst;
  check(XPRS_msp_getintattribsol(ptr, iSolutionId, iSolutionIdStatus_,
                                 iAttribId, &Dst));
  return Dst;
}

auto xpress::MIPSolPool::getIntAttribSol(int iSolutionId,
                                         int iAttribId) const -> int {

  int *iSolutionIdStatus_(nullptr);
  int Dst;
  check(XPRS_msp_getintattribsol(ptr, iSolutionId, iSolutionIdStatus_,
                                 iAttribId, &Dst));
  return Dst;
}

void xpress::MIPSolPool::getDblAttribSol(int iSolutionId,
                                         int *iSolutionIdStatus_, int iAttribId,
                                         double *Dst) const {

  check(XPRS_msp_getdblattribsol(ptr, iSolutionId, iSolutionIdStatus_,
                                 iAttribId, Dst));
}

auto xpress::MIPSolPool::getDblAttribSol(int iSolutionId,
                                         int *iSolutionIdStatus_,
                                         int iAttribId) const -> double {

  double Dst;
  check(XPRS_msp_getdblattribsol(ptr, iSolutionId, iSolutionIdStatus_,
                                 iAttribId, &Dst));
  return Dst;
}

auto xpress::MIPSolPool::getDblAttribSol(int iSolutionId,
                                         int iAttribId) const -> double {

  int *iSolutionIdStatus_(nullptr);
  double Dst;
  check(XPRS_msp_getdblattribsol(ptr, iSolutionId, iSolutionIdStatus_,
                                 iAttribId, &Dst));
  return Dst;
}

void xpress::MIPSolPool::getIntControlSol(int iSolutionId,
                                          int *iSolutionIdStatus_,
                                          int iControlId, int *Val) const {

  check(XPRS_msp_getintcontrolsol(ptr, iSolutionId, iSolutionIdStatus_,
                                  iControlId, Val));
}

auto xpress::MIPSolPool::getIntControlSol(int iSolutionId,
                                          int *iSolutionIdStatus_,
                                          int iControlId) const -> int {

  int Val;
  check(XPRS_msp_getintcontrolsol(ptr, iSolutionId, iSolutionIdStatus_,
                                  iControlId, &Val));
  return Val;
}

auto xpress::MIPSolPool::getIntControlSol(int iSolutionId,
                                          int iControlId) const -> int {

  int *iSolutionIdStatus_(nullptr);
  int Val;
  check(XPRS_msp_getintcontrolsol(ptr, iSolutionId, iSolutionIdStatus_,
                                  iControlId, &Val));
  return Val;
}

void xpress::MIPSolPool::getDblControlSol(int iSolutionId,
                                          int *iSolutionIdStatus_,
                                          int iControlId, double *Val) const {

  check(XPRS_msp_getdblcontrolsol(ptr, iSolutionId, iSolutionIdStatus_,
                                  iControlId, Val));
}

auto xpress::MIPSolPool::getDblControlSol(int iSolutionId,
                                          int *iSolutionIdStatus_,
                                          int iControlId) const -> double {

  double Val;
  check(XPRS_msp_getdblcontrolsol(ptr, iSolutionId, iSolutionIdStatus_,
                                  iControlId, &Val));
  return Val;
}

auto xpress::MIPSolPool::getDblControlSol(int iSolutionId,
                                          int iControlId) const -> double {

  int *iSolutionIdStatus_(nullptr);
  double Val;
  check(XPRS_msp_getdblcontrolsol(ptr, iSolutionId, iSolutionIdStatus_,
                                  iControlId, &Val));
  return Val;
}

auto xpress::MIPSolPool::setIntControlSol(int iSolutionId, int iControlId,
                                          int Val) -> int {

  int iSolutionIdStatus_;
  check(XPRS_msp_setintcontrolsol(ptr, iSolutionId, &iSolutionIdStatus_,
                                  iControlId, Val));
  return iSolutionIdStatus_;
}

auto xpress::MIPSolPool::setDblControlSol(int iSolutionId, int iControlId,
                                          double Val) -> int {

  int iSolutionIdStatus_;
  check(XPRS_msp_setdblcontrolsol(ptr, iSolutionId, &iSolutionIdStatus_,
                                  iControlId, Val));
  return iSolutionIdStatus_;
}

void xpress::MIPSolPool::getIntAttribProbExtreme(
    XPRSProblem const &prob_to_rank_against, int bGet_Max_Otherwise_Min,
    int *iSolutionId, int iAttribId, int *ExtremeVal) const {

  check(XPRS_msp_getintattribprobextreme(ptr, prob_to_rank_against.ptr,
                                         bGet_Max_Otherwise_Min, iSolutionId,
                                         iAttribId, ExtremeVal));
}

void xpress::MIPSolPool::getDblAttribProbExtreme(
    XPRSProblem const &prob_to_rank_against, int bGet_Max_Otherwise_Min,
    int *iSolutionId, int iAttribId, double *ExtremeVal) const {

  check(XPRS_msp_getdblattribprobextreme(ptr, prob_to_rank_against.ptr,
                                         bGet_Max_Otherwise_Min, iSolutionId,
                                         iAttribId, ExtremeVal));
}

auto xpress::MIPSolPool::getIntAttrib(int iAttribId) const -> int {

  int Val;
  check(XPRS_msp_getintattrib(ptr, iAttribId, &Val));
  return Val;
}

auto xpress::MIPSolPool::getDblAttrib(int iAttribId) const -> double {

  double Val;
  check(XPRS_msp_getdblattrib(ptr, iAttribId, &Val));
  return Val;
}

auto xpress::MIPSolPool::getIntControl(int iControlId) const -> int {

  int Val;
  check(XPRS_msp_getintcontrol(ptr, iControlId, &Val));
  return Val;
}

auto xpress::MIPSolPool::getDblControl(int iControlId) const -> double {

  double Val;
  check(XPRS_msp_getdblcontrol(ptr, iControlId, &Val));
  return Val;
}

void xpress::MIPSolPool::setIntControl(int iControlId, int Val) {

  check(XPRS_msp_setintcontrol(ptr, iControlId, Val));
}

void xpress::MIPSolPool::setDblControl(int iControlId, double Val) {

  check(XPRS_msp_setdblcontrol(ptr, iControlId, Val));
}

void xpress::MIPSolPool::setSolName(
    int iSolutionId, std::optional<std::string> const &sNewSolutionBaseName,
    int *bNameModifiedForUniqueness, int *iSolutionIdStatus_) {

  check(XPRS_msp_setsolname(
      ptr, iSolutionId,
      sNewSolutionBaseName ? sNewSolutionBaseName.value().c_str() : nullptr,
      bNameModifiedForUniqueness, iSolutionIdStatus_));
}

void xpress::MIPSolPool::getSolName(int iSolutionId, char *_sname,
                                    int _iStringBufferBytes,
                                    int *_iBytesInInternalString,
                                    int *iSolutionIdStatus_) {

  check(XPRS_msp_getsolname(ptr, iSolutionId, _sname, _iStringBufferBytes,
                            _iBytesInInternalString, iSolutionIdStatus_));
}

auto xpress::MIPSolPool::getSolName(int iSolutionId,
                                    int *iSolutionIdStatus_) -> std::string {

  int *_iBytesInInternalString(nullptr);
  int _iStringBufferBytes = 0;
  std::vector<char> _snameBuffer(std::max(0, _iStringBufferBytes));

  check(XPRS_msp_getsolname(ptr, iSolutionId, _snameBuffer.data(),
                            _iStringBufferBytes, _iBytesInInternalString,
                            iSolutionIdStatus_));
  _snameBuffer.push_back('\0');
  return std::string(_snameBuffer.data());
}

auto xpress::MIPSolPool::getSolName(int iSolutionId) -> std::string {

  int *_iBytesInInternalString(nullptr);
  int _iStringBufferBytes = 0;
  int *iSolutionIdStatus_(nullptr);
  std::vector<char> _snameBuffer(std::max(0, _iStringBufferBytes));

  check(XPRS_msp_getsolname(ptr, iSolutionId, _snameBuffer.data(),
                            _iStringBufferBytes, _iBytesInInternalString,
                            iSolutionIdStatus_));
  _snameBuffer.push_back('\0');
  return std::string(_snameBuffer.data());
}

auto xpress::MIPSolPool::findSolByName(
    std::optional<std::string> const &sSolutionName) -> int {

  int iSolutionId;
  check(XPRS_msp_findsolbyname(
      ptr, sSolutionName ? sSolutionName.value().c_str() : nullptr,
      &iSolutionId));
  return iSolutionId;
}

auto xpress::MIPSolPool::writeSlxSol(
    XPRSProblem const &prob_context, int iSolutionId,
    std::optional<std::string> const &sFileName,
    std::optional<std::string> const &sFlags) -> int {

  int iSolutionIdStatus_;
  check(XPRS_msp_writeslxsol(ptr, prob_context.ptr, iSolutionId,
                             &iSolutionIdStatus_,
                             sFileName ? sFileName.value().c_str() : nullptr,
                             sFlags ? sFlags.value().c_str() : nullptr));
  return iSolutionIdStatus_;
}

void xpress::MIPSolPool::readSlxSol(NameList const &col_name_list,
                                    std::optional<std::string> const &sFileName,
                                    std::optional<std::string> const &sFlags,
                                    int *iSolutionId_Beg,
                                    int *iSolutionId_End) {

  check(XPRS_msp_readslxsol(ptr, col_name_list.ptr,
                            sFileName ? sFileName.value().c_str() : nullptr,
                            sFlags ? sFlags.value().c_str() : nullptr,
                            iSolutionId_Beg, iSolutionId_End));
}

void xpress::MIPSolEnum::destroy() { check(XPRS_mse_destroy(ptr)); }

void xpress::MIPSolEnum::minim(
    XPRSProblem const &prob, MIPSolPool const &msp,
    std::function<int(MIPSolEnum &, XPRSProblem &, MIPSolPool &, int *,
                      double const *, int, double, double *, bool *, bool *)>
        f_mse_handler,
    int *nMaxSols) {

  MipSolEnumHandlerCallbackHolder callbackUserContextObject(this,
                                                            f_mse_handler);
  MipSolEnumHandlerCallbackHolder *callbackUserContext =
      &callbackUserContextObject;
  check(XPRS_mse_minim(
      ptr, prob.ptr, msp.ptr,
      MipSolEnumHandlerCallbackHolder::wrapMipSolEnumHandlerCallback,
      callbackUserContext, nMaxSols));
}

void xpress::MIPSolEnum::maxim(
    XPRSProblem const &prob, MIPSolPool const &msp,
    std::function<int(MIPSolEnum &, XPRSProblem &, MIPSolPool &, int *,
                      double const *, int, double, double *, bool *, bool *)>
        f_mse_handler,
    int *nMaxSols) {

  MipSolEnumHandlerCallbackHolder callbackUserContextObject(this,
                                                            f_mse_handler);
  MipSolEnumHandlerCallbackHolder *callbackUserContext =
      &callbackUserContextObject;
  check(XPRS_mse_maxim(
      ptr, prob.ptr, msp.ptr,
      MipSolEnumHandlerCallbackHolder::wrapMipSolEnumHandlerCallback,
      callbackUserContext, nMaxSols));
}

void xpress::MIPSolEnum::opt(
    XPRSProblem const &prob, MIPSolPool const &msp,
    std::function<int(MIPSolEnum &, XPRSProblem &, MIPSolPool &, int *,
                      double const *, int, double, double *, bool *, bool *)>
        f_mse_handler,
    int *nMaxSols) {

  MipSolEnumHandlerCallbackHolder callbackUserContextObject(this,
                                                            f_mse_handler);
  MipSolEnumHandlerCallbackHolder *callbackUserContext =
      &callbackUserContextObject;
  check(XPRS_mse_opt(
      ptr, prob.ptr, msp.ptr,
      MipSolEnumHandlerCallbackHolder::wrapMipSolEnumHandlerCallback,
      callbackUserContext, nMaxSols));
}

void xpress::MIPSolEnum::getSolList(int iMetricId, int iRankFirstIndex_Ob,
                                    int iRankLastIndex_Ob,
                                    Array<int> const &iSolutionIds,
                                    int *nReturnedSolIds, int *nSols) {

  check(XPRS_mse_getsollist(ptr, iMetricId, iRankFirstIndex_Ob,
                            iRankLastIndex_Ob, iSolutionIds.data,
                            nReturnedSolIds, nSols));
}

void xpress::MIPSolEnum::getSolMetric(int iSolutionId, int *iSolutionIdStatus,
                                      int iMetricId, double *dMetric) {

  check(XPRS_mse_getsolmetric(ptr, iSolutionId, iSolutionIdStatus, iMetricId,
                              dMetric));
}

void xpress::MIPSolEnum::getCullChoice(int iMetricId,
                                       Array<int> const &cull_sol_id_list,
                                       int nMaxSolsToCull, int *nSolsToCull,
                                       double dNewSolMetric,
                                       Array<double const> const &x, int nCols,
                                       bool *bRejectSoln) {

  int bRejectSoln_int = std::numeric_limits<int>::min();
  check(XPRS_mse_getcullchoice(
      ptr, iMetricId, cull_sol_id_list.data, nMaxSolsToCull, nSolsToCull,
      dNewSolMetric, x.data, nCols, bRejectSoln ? &bRejectSoln_int : nullptr));
  if (bRejectSoln)
    *bRejectSoln = bRejectSoln_int != 0;
}

auto xpress::MIPSolEnum::getCullChoice(int iMetricId,
                                       Array<int> const &cull_sol_id_list,
                                       int nMaxSolsToCull, double dNewSolMetric,
                                       Array<double const> const &x, int nCols,
                                       bool *bRejectSoln) -> int {

  int nSolsToCull;
  int bRejectSoln_int = std::numeric_limits<int>::min();
  check(XPRS_mse_getcullchoice(
      ptr, iMetricId, cull_sol_id_list.data, nMaxSolsToCull, &nSolsToCull,
      dNewSolMetric, x.data, nCols, bRejectSoln ? &bRejectSoln_int : nullptr));
  if (bRejectSoln)
    *bRejectSoln = bRejectSoln_int != 0;
  return nSolsToCull;
}

auto xpress::MIPSolEnum::getCullChoice(int iMetricId,
                                       Array<int> const &cull_sol_id_list,
                                       int nMaxSolsToCull, double dNewSolMetric,
                                       bool *bRejectSoln) -> int {

  SizedArray<double const> x(nullptr);
  int nCols = 0;
  int nSolsToCull;
  int bRejectSoln_int = std::numeric_limits<int>::min();
  check(XPRS_mse_getcullchoice(
      ptr, iMetricId, cull_sol_id_list.data, nMaxSolsToCull, &nSolsToCull,
      dNewSolMetric, x.data, nCols, bRejectSoln ? &bRejectSoln_int : nullptr));
  if (bRejectSoln)
    *bRejectSoln = bRejectSoln_int != 0;
  return nSolsToCull;
}

auto xpress::MIPSolEnum::getIntAttrib(int iAttribId) const -> int {

  int Val;
  check(XPRS_mse_getintattrib(ptr, iAttribId, &Val));
  return Val;
}

auto xpress::MIPSolEnum::getDblAttrib(int iAttribId) const -> double {

  double Val;
  check(XPRS_mse_getdblattrib(ptr, iAttribId, &Val));
  return Val;
}

auto xpress::MIPSolEnum::getIntControl(int iAttribId) const -> int {

  int Val;
  check(XPRS_mse_getintcontrol(ptr, iAttribId, &Val));
  return Val;
}

auto xpress::MIPSolEnum::getDblControl(int iAttribId) const -> double {

  double Val;
  check(XPRS_mse_getdblcontrol(ptr, iAttribId, &Val));
  return Val;
}

void xpress::MIPSolEnum::setIntControl(int iAttribId, int Val) {

  check(XPRS_mse_setintcontrol(ptr, iAttribId, Val));
}

void xpress::MIPSolEnum::setDblControl(int iAttribId, double Val) {

  check(XPRS_mse_setdblcontrol(ptr, iAttribId, Val));
}

void xpress::MIPSolEnum::setSolBaseName(
    std::optional<std::string> const &sSolutionBaseName) {

  check(XPRS_mse_setsolbasename(
      ptr, sSolutionBaseName ? sSolutionBaseName.value().c_str() : nullptr));
}

void xpress::MIPSolEnum::getSolBaseName(char *_sname, int _iStringBufferBytes,
                                        int *_iBytesInInternalString) {

  check(XPRS_mse_getsolbasename(ptr, _sname, _iStringBufferBytes,
                                _iBytesInInternalString));
}

auto xpress::MIPSolEnum::addMseGetSolutionDiffCallback(
    std::function<int(MIPSolEnum &, int, int, int, double, double const *,
                      int const *, int, int, double, double const *,
                      int const *, double *)>
        callback,
    int prio) -> CallbackHandle {

  if (!callback) {
    removeMseGetSolutionDiffCallbacks();
    return nullptr;
  }
  std::unique_ptr<xpress::MseGetSolutionDiffCallbackHolder> wrapper =
      std::make_unique<xpress::MseGetSolutionDiffCallbackHolder>(this,
                                                                 callback);
  check(XPRS_mse_addcbgetsolutiondiff(
      ptr,
      xpress::MseGetSolutionDiffCallbackHolder::wrapMseGetSolutionDiffCallback,
      wrapper.get(), prio));
  return wrapper.release();
}
void xpress::MIPSolEnum::removeMseGetSolutionDiffCallback(
    CallbackHandle const &callback) {
  if (!callback.getHolder())
    removeMseGetSolutionDiffCallbacks();
  else
    check(XPRS_mse_removecbgetsolutiondiff(
        ptr, MseGetSolutionDiffCallbackHolder::wrapMseGetSolutionDiffCallback,
        callback.getHolder()));
}
void xpress::MIPSolEnum::removeMseGetSolutionDiffCallbacks() {
  check(XPRS_mse_removecbgetsolutiondiff(ptr, nullptr, nullptr));
}
