// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KTASKSERIALIZER_H
#define __KTASKSERIALIZER_H

#include "Globals.h"
#include "KBranchingScheme.h"
#include "kalis_compat.h"

/**
 * Task-based branching strategy
 *
 * A custom scheduling optimization strategy can be specified by using the
 * KTaskSerializer branching scheme to select the task to be scheduled and value
 * choice heuristics for its start, duration and assignments variables.
 *
 * \image html taskSerializer.png
 *
 * @see KSmallestEarliestStartTime KSmallestEarliestCompletionTime
 *           KLargestEarliestStartTime KLargestEarliestCompletionTime
 *           KSmallestLatestStartTime KSmallestLatestCompletionTime
 *           KLargestLatestStartTime KLargestLatestCompletionTime
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KTaskSerializer : public KBranchingScheme {
public:
    /**
     * Variable branching orders for each task
     */
    enum varOrder {
        AFF_DUR_START = 0,  ///< Variable branching order: <ol><li>assignments <li>duration <li>start time</ol>
        AFF_START_DUR = 1,  ///< Variable branching order: <ol><li>assignments <li>start time <li>duration</ol>
        DUR_START_AFF = 2,  ///< Variable branching order: <ol><li>duration <li>start time <li>assignments</ol>
        DUR_AFF_START = 3,  ///< Variable branching order: <ol><li>duration <li>assignments <li>start time</ol>
        START_DUR_AFF = 4,  ///< Variable branching order: <ol><li>start time <li>duration <li>assignments</ol>
        START_AFF_DUR = 5   ///< Variable branching order: <ol><li>start time <li>assignments <li>duration</ol>
    };
    /// Primary constructor (default strategy)
    KTaskSerializer(KTaskArray& tasks,
                    int discrepancyLimit = MAX_INT,
                    varOrder varOrder = DUR_START_AFF);
    /**
     * Constructor with a set of tasks, a task selector, a value selector for
     * the duration and start variables, an optional discrepancy limit and an
     * optional order for branching on variables.
     */
    KTaskSerializer(const KTaskArray& tasks,
                    const KTaskSelector& tsel,
                    const KValueSelector& durationSelector,
                    const KValueSelector& startSelector,
                    int discrepancyLimit = MAX_INT,
                    varOrder varOrder = DUR_START_AFF);

    /**
     * Constructor with a set of tasks, a task selector, a value selector for
     * the duration and start variables, a variable selector for assignments (
     * including providing, producing, requiring and consumming variables),
     * an optional discrepancy limit and an optional order for branching on
     * variables.
     *
     */
    KTaskSerializer(const KTaskArray& tasks,
                    const KTaskSelector& tsel,
                    const KValueSelector& durationSelector,
                    const KValueSelector& startSelector,
                    const KVariableSelector& assignmentVariableSelector,
                    const KValueSelector& assignmentValueSelector,
                    int discrepancyLimit = MAX_INT,
                    varOrder varOrder = AFF_START_DUR);

    /// Constructor with KTaskSerializer_I*
    KTaskSerializer(KTaskSerializer_I* taskSerializer);
    /// Copy constructor
    KTaskSerializer(const KTaskSerializer& taskSerializer);
    /// Destructor
    virtual ~KTaskSerializer();
    /// Get a copy pointer
    virtual KBranchingScheme* getCopyPtr() const;
    virtual KBranchingScheme* getInstanceCopyPtr(const KProblem& problem) const;
};

#endif
