// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KTASK_H
#define __KTASK_H

#include "Globals.h"
#include "kalis_compat.h"

/**
 * Tasks (processing operations, activities) are represented by the class KTask. This object contains three variables :
 *     - a start variable representing the starting time of the task
 *     - an end variable representing the ending time of the task
 *     - a duration variable representing the duration of the task
 *
 * These three structural variables are linked by Artelys-Kalis with the following constraint :
 *
 * \f$ start + duration = end \f$
 *
 * The starting time variable represents two specific parameters of the task:
 *     - the Earliest Starting Time (EST represented by the start variable lower bound)
 *     - and its Latest Starting Time (LST represented by the start variable upper bound)
 *
 * The end variable represents two specific parameters of the task:
 *     - the Earliest Completion Time (ECT represented by the end variable lower bound)
 *     - and its Latest Completion Time (LCT represented by the end variable upper bound)
 *
 * The duration variable represents two specific parameters of the task:
 *     - the minimum task duration (Dmin represented by the duration variable lower bound)
 *     - the maximum task duration (Dmax represented by the duration variable upper bound)
 *
 * \image html task.png
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KTask : public KPtrArray<KTask> {
   private:
    KTask_I *task_IPtr;
    // Pointer to KSchedule instance
    KSchedule *_schedule;

   public:
    /**
     * Empty constructor
     */
    KTask();
    /**
     * Basic constructor
     */
    KTask(KSchedule& schedule);
    /**
     * Primary constructor
     *
     * @param s the schedule
     * @param name string representing the name of the task
     */
    KTask(KSchedule& s, const char* name);
    /**
     * Constructor with min/max value for start time and duration
     *
     * @param s the schedule
     * @param name string representing the name of the task
     * @param smin Minimum possible start of the task
     * @param smin Maximum possible start of the task
     * @param dmin Minimum possible duration of the task
     * @param dmax Maximum possible duration of the task
     */
    KTask(KSchedule& s, const char* name, int smin, int smax, int dmin, int dmax);
    /**
     * Constructor with constant duration
     *
     * @param s the schedule
     * @param name string representing the name of the task
     * @param duration fixed duration of the task
     */
    KTask(KSchedule& s, const char* name, int duration);
    /**
     * Constructor with fixed duration and unary resource usage
     *
     * @param s the schedule
     * @param name string representing the name of the task
     * @param duration fixed task duration
     * @param r the resource required by the task
     */
    KTask(KSchedule& s, const char* name, int duration, KResource& r);
    /**
     * @param s the schedule
     * @param name string representing the name of the task
     * @param r the resource required by the task
     * @param size the surface (duration * resourceConsumption) of the task
     */
    KTask(KSchedule& s, const char* name, KResource& r, int size);
    /**
     * Copy constructor
     */
    KTask(const KTask& toCopy);
    /**
     * Destructor
     */
    virtual ~KTask();
    /**
     * Pretty printing of the task to stdout
     */
    KSchedule *getSchedule() const;

    /// Pretty printing the task
    virtual void print() const;
    /**
     * Pretty printing of the task with a PrintFunctionPtr
     */
    virtual void print(void* ctx, PrintFunctionPtr* pfp) const;
    /**
     * Set the sequence dependent setup time between the current task,and the one passed in parameters to before/after
     */
    void setSetupTime(KTask& task, int before, int after);
    /**
     * Return the setup time between the current task and the one passed in parameter
     */
    int getSetupTime(KTask& task);

    /// Add a resource usage requirement for this task
    void requires(KResourceUsage& resusage);

    /// Add a resource usage provision for this task
    void provides(KResourceUsage& resusage);

    /// Add a resource usage consumption for this task
    void consumes(KResourceUsage& resusage);

    /// Add a resource usage production for this task
    void produces(KResourceUsage& resusage);

    /**
     * Add optional resources usages requirements for this task and ensure that
     * between [min..max] of theses requirements are satisfied
     */
    void requires(KResourceUsageArray& resusagearray, int min = 1, int max = 1);
    /**
     * Add optional resources usages provisions for this task and ensure that
     * between [min..max] of theses requirements are satisfied
     */
    void provides(KResourceUsageArray& resusagearray, int min = 1, int max = 1);
    /**
     * Add optional resources usages consumptions for this task and ensure that
     * between [min..max] of theses requirements are satisfied
     */
    void consumes(KResourceUsageArray& resusagearray, int min = 1, int max = 1);
    /**
     * Add optional resources usages productions for this task and ensure that
     * between [min..max] of theses requirements are satisfied
     */
    void produces(KResourceUsageArray& resusagearray, int min = 1, int max = 1);
    /**
     * Return the minimal resource requirement for this task and the resource in
     * parameters at time step `tslot`
     */
    int getMinimalRequirement(KResource* resource, int tslot);
    /**
     * Return the minimal resource consumption for this task and the resource in
     * parameters at time step `tslot`
     */
    int getMinimalConsumption(KResource* resource, int tslot);
    /**
     * Return the maximal resource provision for this task and the resource in
     * parameters at time step `tslot`
     */
    int getMaximalProvision(KResource* resource, int tslot);
    /**
     * Return the maximal resource production for this task and the resource in
     * parameters at time step `tslot`
     */
    int getMaximalProduction(KResource* resource, int tslot);
    /**
     * State that this ressource requires ( renewable ) `requirement` unit of
     * resource `resource`
     *
     * @param resource the resource
     * @param requirement the resource requirement
     */
    void requires(KResource& resource, int requirement);
    /**
     * State that this ressource consumes ( non-renewable ) `consumption` unit
     * of resource `resource`
     *
     * @param resource the involved resource
     * @param consumption the resource consumption
     */
    void consumes(KResource& resource, int consumption);
    /**
     * State that this ressource provides ( renewable ) `provision` unit of
     * resource `resource`
     *
     * @param resource the resource
     * @param provision the resource provision
     */
    void provides(KResource& resource, int provision);
    /**
     * State that this ressource produces ( non-renewable ) `production` unit of
     * resource `resource`
     *
     * @param resource the resource
     * @param production the resource production
     */
    void produces(KResource& resource, int production);
    /**
     * State that this ressource requires ( renewable ) between `requirementmin`
     * and 'requirementmax unit of resource `resource`
     *
     * @param resource the resource
     * @param requirementmin the minimal resource requirement
     * @param requirementmax the maximal resource requirement
     */
    void requires(KResource& resource, int requirementmin, int requirementmax);
    /**
     * State that this ressource consumes ( non-renewable ) between
     * `consumptionmin` and `consumptionmax` unit of resource `resource`
     *
     * @param resource the resource
     * @param consumptionmin the minimal resource consumption
     * @param consumptionmax the maximal resource consumption
     */
    void consumes(KResource& resource, int consumptionmin, int consumptionmax);
    /**
     * State that this ressource provides ( renewable ) between `provisionmin`
     * and 'provisionmax unit of resource `resource`
     *
     * @param resource the resource
     * @param provisionmin the minimal resource provision
     * @param provisionmax the maximal resource provision
     */
    void provides(KResource& resource, int provisionmin, int provisionmax);
    /**
     * State that this ressource produces ( non-renewable ) between
     * `productionmin` and `productionmax` unit of resource `resource`
     *
     * @param resource the resource
     * @param productionmin the minimal resource production
     * @param productionmax the maximal resource production
     */
    void produces(KResource& resource, int productionmin, int productionmax);
    /**
     * State that this ressource requires ( renewable ) one unit of resource
     * `resource`
     */
    int requires(KResource& resource);
    /**
     * State that this ressource consumes ( non-renewable ) one unit of resource
     * `resource`
     */
    int consumes(KResource& resource);
    /**
     * State that this ressource provides ( renewable ) one unit of resource
     * `resource`
     */
    int provides(KResource& resource);
    /**
     * State that this ressource produces ( non-renewable ) one unit of resource
     * `resource`
     */
    int produces(KResource& resource);
    /**
     * State that this task starts `delay` time unit after the completion of
     * the given `task`
     *
     * @param task the task before the current task
     * @param delay the time distance between the two tasks
     */
    void startsAfter(KTask& task, int delay = 0);
    /**
     * State that this task ends `delay` time unit before the completion of
     * the given `task`
     *
     * @param task the task before the current task
     * @param delay the time distance between the two tasks
     */
    void endsBefore(KTask& task, int delay = 0);
    /**
     * State that the distance between the completion of this task and the
     * completion of task `task` cannot exceed `Max` time units
     *
     * @param task the task
     * @param Max the distance between the completion of this task and the completion of task `task`
     */
    void postEndToEndMaxC(KTask& task, int Max = 0);
    /**
     * State that the distance between the completion of this task and the start
     * of task `task` cannot exceed `Max` time units
     *
     * @param task the task
     * @param Max the distance between the completion of this task and the start of task `task`
     */
    void postEndToStartMaxC(KTask& task, int Max = 0);
    /**
     * State that the distance between the start of this task and the start of
     * task `task` must exceed `Min` time units
     *
     * @param task the task
     * @param Min the distance between the start of this task and the start of task `task`
     */
    void postStartToStartMinC(KTask& task, int Min = 0);
    /**
     * State that the distance between the completion of this task and the start
     * of task `task` must exceed `Min` time units
     *
     * @param task the task
     * @param Min the distance between the completion of this task and the start of task `task`
     */
    void postEndToStartMinC(KTask& task, int Min = 0);

    // Internal use only
    void setIndex(int index);

    /// Return the id of this task
    int getIndex();

    /// Return the name of this task
    const char* getName() const;

    /// Set the name of this task
    void setName(const char* name);
    /**
     * Return a pointer to the KIntVar representing the starting date of this
     * task
     */
    KIntVar* getStartDateVar();
    /**
     * Return a pointer to the KIntVar representing the ending date of this task
     */
    KIntVar* getEndDateVar();
    /**
     * Return a pointer to the KIntVar representing the duration of this task
     */
    KIntVar* getDurationVar();
    /**
     * Return a pointer to the KIntVar representing the resource requirement of
     * this task for resource `r` if any or nullptr
     */
    KIntVar* getRequiresVar(KResource* r);
    /**
     * Return a pointer to the KIntVar representing the resource consumtion of
     * this task for resource `r` if any or nullptr
     */
    KIntVar* getConsumesVar(KResource* r);
    /**
     * Return a pointer to the KIntVar representing the resource provision of
     * this task for resource `r` if any or nullptr
     */
    KIntVar* getProvidesVar(KResource* r);
    /**
     * Return a pointer to the KIntVar representing the resource production of
     * this task for resource `r` if any or nullptr
     */
    KIntVar* getProducesVar(KResource* r);
    /**
     * Return a pointer to the KIntVar representing the product requirement * duration
     */
    KIntVar* getSizeVar(KResource* r);
    /**
     * Return a pointer to the KIntVar representing the resource requirement of
     * this task for resource `r` if any or nullptr
     */
    KIntVar* getAssignmentVar(KResource* r);
    /**
     * Return the constant duration of this task or the lowerbound if duration
     * is not constant
     */
    int getDurationValue();

    // Internal use
    int getResourceConsumption();

    /// Return the earliest starting time of this task
    int getEarliestStartTime();

    /// Return the latest starting time of this task
    int getLatestStartTime();

    /// Return the earliest completion time of this task
    int getEarliestCompletionTime();

    /// Return the latest completion time of this task
    int getLatestCompletionTime();

    /// Return the minimum duration of this task
    int getMinimumDuration();

    /// Return the maximum duration of this task
    int getMaximumDuration();

    /// Set the earliest starting time of this task
    void setEarliestStartTime(int time);

    /// Set the latest starting time of this task
    void setLatestStartTime(int time);

    /// Set the earliest completion time of this task
    void setEarliestCompletionTime(int time);

    /// Set the latest completion time of this task
    void setLatestCompletionTime(int time);

    /// Set the minimum duration of this task
    void setMinimumDuration(int durationmin);

    /// Set the maximum duration of this task
    void setMaximumDuration(int durationmax);

    /// Set the starting time of this task to `startTime`
    void setStartTime(int startTime);

    /// Set the ending time of this task to `endTime`
    void setEndTime(int endTime);

    /// Set the duration of this task to `duration`
    void setDuration(int duration);

    /**
     * Return <b>true</b> IFF this task is fixed (Start,End,Duration, and
     * resource utilizations variables are instantiated)
     */
    bool isFixed();

    /**
     * Returns the earliest time at which this task can be started on the
     * given resource. Throws an exception if the task has not been
     * declared on the resource.
     */
    int getEarliestStartTime(KResource& r);

    // Internal use only
    void constrainVars(void);
    // Internal use only
    virtual KTask* getCopyPtr() const;
    // Internal use only
    KTask_I* getTaskIPtr();
    // Internal use only
    const KTask_I* getTaskIPtr() const;
    // Internal use only
    void setTaskIPtr(KTask_I* taskIPtr);
};

#endif
