// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KRELAXATIONSOLVER_H
#define __KRELAXATIONSOLVER_H

#include "Globals.h"
#include "KProblem.h"

/**
 * This class is intended as a superclass for linear relaxation solvers.
 *
 * Such a solver must be provided with
 *     - a linear relaxation (KLinearRelaxation)
 *     - an objective variable (KNumVar)
 *     - a sense for optimization (KProblem::Sense).
 *
 * It relies on a LP/MIP solver to provide the following information:
 *     - a value (a bound for the relaxed problem, cf method getBound())
 *     - a solution, possibly not feasible for the original problem, but which can be used to guide the search for a feasible solution
 *     - if the problem is LP, reduced costs (that can be used for instance in the "reduced cost fixing" procedure).
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KRelaxationSolver : public KAbstractPtrArray<KRelaxationSolver> {
  protected:
    KRelaxationSolver_I* _KRelaxationSolver_IPtr;
    KRelaxationSolverConfigurator_I* _configurator;
  public:
    // destructor
    virtual ~KRelaxationSolver();
    // Internal use
    KRelaxationSolver_I* getInternalPtr (void);
    /**
     * Set objective variable.
     *
     * @param var the new objective variable
     */
    virtual void setObjective (const KNumVar& var) = 0;

    /**
     * Solve the relaxed optimization problem.
     *
     * This methods returns the following error codes :
     *    - 0 : optimal
     *    - 1 : infeasible
     *    - 2 : search interrupted prematurely, a solution was found
     *    - 3 : search interrupted prematurely, no solution was found
     *    - 4 : other problem
     */
    virtual int solve() = 0;

    /**
     * Get the (lower for minimization, upper for maximization) bound computed by solve().
     *
     * Note that :
     *    - solve() method must be called before the getBound() method
     *    - moreover, the return code provided by solve() must be checked before using the value
     *      returned by getBound().
     */
    virtual double getBound(void) const = 0;

    virtual double getBestBound(void) const = 0;

    // Print the internal state of the solver
    virtual void print(void) const = 0;

    /// Return true if the given variable is set as global
    bool isGlobal(const KIntVar&);

    /// Return true if the given variable is set as global
    bool isGlobal(const KAuxVar&);

    /**
     * Set (or unset) a KIntVar as global.
     *
     * @param var variable to modify
     * @param isGlobal new global status
     */
    void setGlobal(const KIntVar& var, bool isGlobal);

    /**
     * Set (or unset) a KAuxVar global.
     *
     * @param var variable to check
     * @param isGlobal new global status
     */
    void setGlobal(const KAuxVar& var, bool isGlobal);

    /** (Un)set variables as global.
     *
     * Set or unset as "global" all KIntVar and KAuxVar with global type (note
     * that a KFloatVar variables are not modified, since it would make little
     * sense to set them global.)
     *
     * @param isGlobal new global status
     */
    void setAllGlobal(bool isGlobal);

    /**
     * Set all indicator auxiliary variables global.
     *
     * @param isGlobal new global status
     */
    void setIndicatorsGlobal(bool isGlobal);

    /**
     * Set the sense of optimization (maximize, minimize).
     *
     * @param sense new sense for optimization
     */
    void setSense(KProblem::Sense);

    /**
     * Get the current MIP solution for a given KNumVar variable.
     *
     * @param var variable whose solution is checked
     * @return value of var in the current MIP solution
     */
    virtual double getMIPSolution(const KNumVar& var) const = 0;

    /**
     * Get the current MIP solution for a given KAuxVar variable.
     *
     * @param var variable whose solution is checked
     * @return value of var in the current MIP solution
     */
    virtual double getMIPSolution(const KAuxVar& var) const = 0;

    /**
     * Get the current LP solution for a given KNumVar variable.
     *
     * @param var variable whose value is checked
     * @return value of var in the current MIP solution
     */
    virtual double getLPSolution(const KNumVar& var) const = 0;

    /** Get the current relaxed solution for a given KAuxVar variable.
     *
     * @param var variable whose value is checked
     * @return value of var in the current solution
     */
    virtual double getLPSolution(const KAuxVar& var) const = 0;

    /// Get the total number of global variables.
    virtual int getNumberGlobals (void) const = 0;

    /// Get the total amount of time spent in computations since the object was built.
    double getTotalTimeSpentInComputation (void) const;

    /// Get the amount of time spent during the last call to solve().
    double getTimeSpentInLastComputation (void) const;

    /**
     * Set this flag to 0 if you want to solve as LP a linear relaxation
     * containing global entities (1 is the default).
     *
     * @param flag New flag value
     *
     * \deprecated The new way of doing this is to use a configurator
     */
    void setSolveAsMIP(bool flag);

    /// Return true if the flag "solveAsMIP" is set
    bool solveAsMIP();

    /**
     * Stop global search after maxMIPSol feasible solutions found.
     *
     * 0 for no limit, this is the default.
     * If the limit is low, this is likely to cause optimization to end before
     * optimality.
     */
    void setMaxMIPSol(int);

    /// Set the resolution algorithm
    void setAlgorithm(int alg);

    /// Get the resolution algorithm
    int getAlgorithm();

    /// Instantiate variables to current solution obtained by linear relaxation solver
    virtual void instantiateNumVarsToCurrentSol(void) = 0;

    /// Instantiate a variables to current solution obtained by linear relaxation solver
    virtual void instantiateNumVarToCurrentSol(KNumVar* var) = 0;

    /// Cut generation
    virtual void generateCuts(KLinearRelaxation* relaxation) = 0;

    /// Set the configurator of a KRelaxationSolver
    void setConfigurator(KRelaxationSolverConfigurator* configurator);

    /// Get the configurator of a KRelaxationSolver
    KRelaxationSolverConfigurator* getConfigurator();
};

#endif
