// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Description : Optimality tolerance checkers                      *
// ********************************************************************

#ifndef __KOPTIMALITYTOLERANCECHECKER_H
#define __KOPTIMALITYTOLERANCECHECKER_H

#include "Globals.h"
#include "kalis_compat.h"

/**
 * This interface sets a framework for objects providing method to check if the
 * current solution is close enough to the optimum, and, if not, to give a new
 * bound to set on the objective variable.
 */
class DLLIMPORTEXPORT KOptimalityToleranceChecker {
  public:
    virtual ~KOptimalityToleranceChecker();

    /**
    * Check for the optimality tolearance
    *
    * @param bestSolutionObj
    * @param bestBound
    * @return <tt>true</tt> is the best solution is close enough - for some criteria - to the optimum
    */
    virtual bool isGoodEnough(double bestSolutionObj, double bestBound) const = 0;

    /**
    * Returns a bound to set on the objective, in order to look for solution which are not too close from the
    * current best known solution. This prevent from storing too many solutions which are very similar.
    *
    * @param bestSolutionObj the best objective value of already found solutions.
    * @return a bound to set on the objective.
    */
    virtual double nextBoundToTry(double bestSolutionObj) const = 0;


};

/**
 * An <tt>OptimalityToleranceChecker</tt> to use with integer objective only.
 */
class DLLIMPORTEXPORT KIntegerObjectiveOptimalityChecker : public KOptimalityToleranceChecker {
  private :
    int increment;
  public :
    KIntegerObjectiveOptimalityChecker(bool maximize);
    virtual ~KIntegerObjectiveOptimalityChecker();
    bool isGoodEnough(double bestSolutionObj, double bestBound) const;
    double nextBoundToTry(double bestSolutionObj) const;
};

/**
 * An <tt>OptimalityToleranceChecker</tt> to use with any type of
 * <tt>KNumVar</tt> objective, which use an absolute difference criteria.
 */
class DLLIMPORTEXPORT KAbsoluteToleranceOptimalityChecker : public KOptimalityToleranceChecker {
  private :
    double positiveTolerance;
    double nextBoundDelta;
  public :
    /** Primary constructor */
    KAbsoluteToleranceOptimalityChecker(bool maximize, double tolerance);
    virtual ~KAbsoluteToleranceOptimalityChecker();
    bool isGoodEnough(double bestSolutionObj, double bestBound) const;
    double nextBoundToTry(double bestSolutionObj) const;
};

/**
 * An <tt>OptimalityToleranceChecker</tt> to use with any type of
 * <tt>KNumVar</tt> objective, which use a relative difference criteria.
 */
class DLLIMPORTEXPORT KRelativeToleranceOptimalityChecker : public KOptimalityToleranceChecker {
  private :
    double ratio; // always positive, and less than 1
    double nextBoundRatio;
  public :
    KRelativeToleranceOptimalityChecker(bool maximize, double tolerance);
    virtual ~KRelativeToleranceOptimalityChecker();
    bool isGoodEnough(double bestSolutionObj, double bestBound) const;
    double nextBoundToTry(double bestSolutionObj) const;
};

/**
 * An <tt>OptimalityToleranceChecker</tt> to use with any type of
 * <tt>KNumVar</tt> objective, which use both a relative and absolute
 * difference criteria.
 */
class DLLIMPORTEXPORT KNumObjectiveOptimalityChecker : public KOptimalityToleranceChecker {
  private :
    bool _maximize;
    double ratio; // always positive, and less than 1
    double nextBoundRatio;
    double positiveTolerance;
    double nextBoundDelta;
  public :
    KNumObjectiveOptimalityChecker(bool maximize, double absoluteTolerance, double relativeTolerance);
    virtual ~KNumObjectiveOptimalityChecker();
    bool isGoodEnough(double bestSolutionObj, double bestBound) const;
    double nextBoundToTry(double bestSolutionObj) const;
};

#endif
