// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KNUMVAR_H
#define __KNUMVAR_H

#include <iostream>

#include "Globals.h"
#include "KPtrArray.h"
#include "kalis_compat.h"


/**
 * Superclass of decision variables
 */
class DLLIMPORTEXPORT KNumVar : public KPtrArray<KNumVar> {
  protected :
    // Pointer to the problem
    KProblem * _problem;
    // Internal use
    KNumVar_I * _numVarIPtr;
  public:
    /// Default constructor
    KNumVar();
    /// main constructor
    KNumVar(KProblem& problem);
    /// Constructor with KProblem and KNumVar_I
    KNumVar(KProblem* problem, KNumVar_I* numVar);
    //#ifdef ACK_LIB
    /// Constructor with KNumVar_I
    KNumVar(KNumVar_I* numVar);
    //#endif
    /// Copy constructor
    KNumVar(const KNumVar& toCopy);
    /// Assignment copy constructor
    KNumVar& operator=(const KNumVar& toCopy);
    /// Destructor
    virtual ~KNumVar();
    /// Instantiate the variable to value
    virtual void instantiate(const double value);
    /// returns the KProblem associated with this variable
    KProblem * getProblem() const;
    KProblem * getKProblem() const;
    // get methods
    virtual const char * getName() const;
    /// Set the name of the variable
    virtual void setName(const char * name);
    /// returns lower bound of this variable
    virtual double getInf() const;
    /// returns upper bound of this variable
    virtual double getSup() const;
    /// returns current instantiation of the variable (when the variable is not instantiated the returned value is undefined)
    virtual double getValue(void) const;
    /// returns true if the variable has been assigned a value, false otherwhise
    virtual bool getIsInstantiated() const;
    /// returns the number of constraints where this variable appears
    virtual int getDegree() const;
    /// get target value
    virtual double getTarget() const;
    /// set the lower bound to value
    virtual void setInf(double value);
    /// set the upper bound to value
    virtual void setSup(double value);
    /// set the target value
    virtual void setTarget(double value);
    /// Return true if this variable can be instantiated to 'value'
    virtual bool canBeInstantiatedTo(double value);
    /// activate shaving Y/N
    virtual void useShaving(bool use);
    /// Hidden variable Y/N
    virtual void setHidden(bool hidden);
    /// Return true iff this variable is hidden
    virtual bool isHidden(void);
    // Internal use
    virtual KNumVar_I *getNumVar_I_ptr() const;
    // Internal use
    KNumVar_I *getInternalObject() const;
    // Internal use
    void setNumVar_I_ptr(KNumVar_I * numVar);
    /// Return a copy of this object
    KNumVar* getCopyPtr() const;
    /// Pretty printing
    virtual void print(std::ostream & fout) const;
    virtual void print(void) const;
    /// Pretty printing
    virtual void print(void* ctx, PrintFunctionPtr* pfp) const;
    /// Pretty print its constraints
    void printConstraints() const;
    void printConstraints(void* ctx, PrintFunctionPtr* pfp) const;
    /**
    * Return the type of this variable
    * @param KNumVar::IsKNumVar for an instance of the class KNumVar
    * @param KNumVar::IsKIntVar for an instance of the class KNumVar
    * @param KNumVar::IsKFloatVar for an instance of the class KNumVar
    */
    virtual int instanceof(void) const;

    /// Return the index of the variable
    virtual int getIndex() const;

    /**
    * Variable types
    */
    enum Type {
        IsKNumVar,		///< Numeric variables
        IsKIntVar,		///< Floating-point (continuous) variables
        IsKFloatVar		///< Integer variables
    };
}; // class KNumVar

DLLIMPORTEXPORT std::ostream& operator<<(std::ostream& os, const KNumVar& v);

#endif
