// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KNUMLINCOMB_H
#define __KNUMLINCOMB_H

#include "Globals.h"
#include "KConstraint.h"

/**
 * This class creates a  Sum(ai.Xi) { <= , != , == } C constraint
 *
 * Example :
 * @code
 * KNumVarArray X(...);
 * //...
 * problem.post(2 * X[1] + 3 * X[2] + 5 * X[3] + ... + 7 * X[n] == 3);
 * // or
 * problem.post(2 * X[1] + 3 * X[2] + 5 * X[3] + ... + 7 * X[n] <= 3);
 * // or
 * problem.post(2 * X[1] + 3 * X[2] + 5 * X[3] + ... + 7 * X[n] >= 3);
 * // or
 * problem.post(2 * X[1] + 3 * X[2] + 5 * X[3] + ... + 7 * X[n] != 3);
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumLinComb : public KConstraint {
   public:
    /**
     Primary Constructor
     @param name name of the constraint
     @param coeffs array of coefficients for the variables in the linear combination
     @param vars array of variables involved in the linear combination
     @param cste constant in the linear combination
     @param linCombOperator operator of the linear constraint {<>,<=,>=,==}
    */
    KNumLinComb(const char* name, KDoubleArray& coeffs, KNumVarArray& vars, double cste, int linCombOperator);
    /// Copy Constructor
    KNumLinComb(const KNumLinComb& toCopy);
    virtual ~KNumLinComb();
    virtual void* getConstraintIPtr();
    enum LinCombOperator {
        Equal,           ///< Equality relation
        GreaterOrEqual,  ///< Greater or Equal relation
        NotEqual,        ///< Not equal relation
        LessOrEqual      ///< Lest or Equal relation
    };
};

/**
 * Conditionnal numeric linear combination constraint.
 *
 * This constraint can be represented as a linear combination
 * Sum(a_i * X_i * f(X_i)) { <= , != , == } C where the function f(X_i) is an
 * indicator (1 or 0) function to specify.
 */
class DLLIMPORTEXPORT KConditionNumLinComb : public KConstraint {
   public:
    /**
     Primary Constructor
     @param name name of the constraint
     @param coeffs array of coefficients for the variables in the linear combination
     @param vars array of variables involved in the linear combination
     @param cste constant in the linear combination
     @param linCombOperator operator of the linear constraint {<=,>=,==}
    */
    KConditionNumLinComb(const char* name, KDoubleArray& coeffs, KNumVarArray& vars, double cste, int linCombOperator);
    /// Copy Constructor
    KConditionNumLinComb(const KConditionNumLinComb& toCopy);
    virtual ~KConditionNumLinComb();
    virtual void* getConstraintIPtr();
    /// Method to overload for indicator function
    virtual int conditionTest(int varIndex) = 0;
    /// Available operators for the combination
    enum LinCombOperator { Equal, GreaterOrEqual, LessOrEqual };
};

#endif
