// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KNUMARITHMETICCONSTRAINTS_H
#define __KNUMARITHMETICCONSTRAINTS_H

#include "Globals.h"
#include "KConstraint.h"

// ********************************************************************
//
//                UNARY ARITHMETIC CONSTRAINTS X op (int)
//
// ********************************************************************

/**
 * This class creates a  `X == C` constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * // ...
 * problem.post(X == 5);
 * // or
 * problem.post(KNumEqualXc(X,5));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumEqualXc : public KConstraint {
   public:
    /** Primary constructor of v1 == C
     * @param v1 the variable
     * @param c the constant
     */
    KNumEqualXc(const KNumVar& v1, double c);
    /// Copy constructor
    KNumEqualXc(const KNumEqualXc& toCopy);
    // Destructor
    virtual ~KNumEqualXc();
};

/**
 * This class creates a  `X >= C`  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * // ...
 * problem.post(X >= 3);
 * // or
 * problem.post(KNumGreaterOrEqualXc(X, 3));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumGreaterOrEqualXc : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 >= C`
     *
     * @param v1 the variable
     * @param c the constant
     */
    KNumGreaterOrEqualXc(const KNumVar& v1, double c);
    /// Copy constructor
    KNumGreaterOrEqualXc(const KNumGreaterOrEqualXc& toCopy);
    // Destructor
    virtual ~KNumGreaterOrEqualXc();
};

/**
 * This class creates a  `X <= C`  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * // ...
 * problem.post(X <= 3);
 * // or
 * problem.post(KNumLessOrEqualXc(X, 3));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumLessOrEqualXc : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 <= C`
     *
     * @param v1 the variable
     * @param c the constant
     */
    KNumLessOrEqualXc(const KNumVar& v1, double c);
    /// Copy constructor
    KNumLessOrEqualXc(const KNumLessOrEqualXc& toCopy);
    // Destructor
    virtual ~KNumLessOrEqualXc();
};

// ********************************************************************
//
//            BINARY ARITHMETIC CONSTRAINTS X op Y + (int)
//
// ********************************************************************

/**
 * This class creates a `X == Y + C` constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(X == Y + 5);
 * // or
 * problem.post(KNumEqualXyc(X, Y, 5));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumEqualXYc : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 == v2 + C`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param c the constant
     */
    KNumEqualXYc(const KNumVar& v1, const KNumVar& v2, double c);
    /// Copy constructor
    KNumEqualXYc(const KNumEqualXYc& toCopy);
    // Destructor
    virtual ~KNumEqualXYc();
};

/**
 * This class creates a  `X >= Y + C`  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(X >= Y + 5);
 * // or
 * problem.post(KNumGreaterOrEqualXyc(X, Y, 5));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumGreaterOrEqualXyc : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 >= v2 + C`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param c the constant
     */
    KNumGreaterOrEqualXyc(const KNumVar& v1, const KNumVar& v2, double c);
    /// Copy constructor
    KNumGreaterOrEqualXyc(const KNumGreaterOrEqualXyc& toCopy);
    // Destructor
    virtual ~KNumGreaterOrEqualXyc();
};

/**
 * This class creates a  `X <= Y + C`  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(X <= Y + 5);
 * // or
 * problem.post(KNumLowerOrEqualXyc(X, Y, 5));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumLowerOrEqualXyc : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 <= v2 + C`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param  c the constant
     */
    KNumLowerOrEqualXyc(const KNumVar& v1, const KNumVar& v2, double c);
    /// Copy constructor
    KNumLowerOrEqualXyc(const KNumLowerOrEqualXyc& toCopy);
    // Destructor
    virtual ~KNumLowerOrEqualXyc();
};

/**
 * This class creates a `X = Y^2` constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ..
 * problem.post(KNumXEqualsYSquared(X, Y));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXEqualsYSquared : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 = v2^2`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     */
    KNumXEqualsYSquared(const KNumVar& v1, const KNumVar& v2);
    /// Copy constructor
    KNumXEqualsYSquared(const KNumXEqualsYSquared& toCopy);
    // Destructor
    virtual ~KNumXEqualsYSquared();
};

/**
 * This class creates a `X = Y * C` constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXEqualsYTimesC(X, Y, 5));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXEqualsYTimesC : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 = v2 * c`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param  c the constant
     */
    KNumXEqualsYTimesC(const KNumVar& v1, const KNumVar& v2, double c);
    /// Copy constructor
    KNumXEqualsYTimesC(const KNumXEqualsYTimesC& toCopy);
    // Destructor
    virtual ~KNumXEqualsYTimesC();
};

/**
 * This class creates a  X = Y ^ C  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXEqualsYArithPowC(X, Y, 5));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXEqualsYArithPowC : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 = v2 ^ c`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param  c the constant
     */
    KNumXEqualsYArithPowC(const KNumVar& v1, const KNumVar& v2, int c);
    /// Copy constructor
    KNumXEqualsYArithPowC(const KNumXEqualsYArithPowC& toCopy);
    // Destructor
    virtual ~KNumXEqualsYArithPowC();
};

/**
 * This class creates a  X = ln(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXEqualsLnY(X, Y));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXEqualsLnY : public KConstraint {
   public:
    /**
     * Primary constructor of `v1 = ln( v2 )`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     */
    KNumXEqualsLnY(const KNumVar& v1, const KNumVar& v2);
    // Copy constructor
    KNumXEqualsLnY(const KNumXEqualsLnY& toCopy);
    // Destructor
    virtual ~KNumXEqualsLnY();
};

/**
 * This class creates a  `X = |Y|`  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXEqualsAbsY(X, Y));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXEqualsAbsY : public KConstraint {
   public:
    /**
     * Primary constructor of v1 = ln( v2 )
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     */
    KNumXEqualsAbsY(const KNumVar& v1, const KNumVar& v2);
    // Copy constructor
    KNumXEqualsAbsY(const KNumXEqualsAbsY& toCopy);
    // Destructor
    virtual ~KNumXEqualsAbsY();
};

// ********************************************************************
//
//					Usual Trigonometric functions
//
// ********************************************************************

/**
 * This class creates a  X {==,<=,>=} cos(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXOperatorCosY(X, Y, GEQ));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXOperatorCosY : public KConstraint {
   public:
    /**
     * Primary constructor of v1 [op] cos( v2 )
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param op operator for the constraint (GEQ, LEQ, EQ)
     */
    KNumXOperatorCosY(const KNumVar& v1, const KNumVar& v2, int op);
    /// Copy constructor
    KNumXOperatorCosY(const KNumXOperatorCosY& toCopy);
    // Destructor
    virtual ~KNumXOperatorCosY();
};

/**
 * This class creates a  X {==,<=,>=} sin(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXOperatorSinY(X, Y, GEQ));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXOperatorSinY : public KConstraint {
   public:
    /**
     * Primary constructor of v1 [op] sin( v2 )
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param op operator for the constraint (GEQ, LEQ, EQ)
     */
    KNumXOperatorSinY(const KNumVar& v1, const KNumVar& v2, int op);
    /// Copy constructor
    KNumXOperatorSinY(const KNumXOperatorSinY& toCopy);
    // Destructor
    virtual ~KNumXOperatorSinY();
};

/**
 * This class creates a  X {==,<=,>=} tan(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXOperatorTanY(X, Y, GEQ));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXOperatorTanY : public KConstraint {
   public:
    /**
     * Primary constructor of v1 [op] tan( v2 )
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param op operator for the constraint (GEQ, LEQ, EQ)
     */
    KNumXOperatorTanY(const KNumVar& v1, const KNumVar& v2, int op);
    /// Copy constructor
    KNumXOperatorTanY(const KNumXOperatorTanY& toCopy);
    // Destructor
    virtual ~KNumXOperatorTanY();
};

// ********************************************************************
//
//             Inverse trigonometric functions
//
// ********************************************************************

/**
 * This class creates a  X {==,<=,>=} acos(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXOperatorACosY(X, Y, GEQ));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXOperatorACosY : public KConstraint {
   public:
    /**
     * Primary constructor of v1 [op] acos( v2 )
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param op operator for the constraint (GEQ, LEQ, EQ)
     */
    KNumXOperatorACosY(const KNumVar& v1, const KNumVar& v2, int op);
    /// Copy constructor
    KNumXOperatorACosY(const KNumXOperatorACosY& toCopy);
    // Destructor
    virtual ~KNumXOperatorACosY();
};

/**
 * This class creates a  X {==,<=,>=} asin(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXOperatorASinY(X, Y, GEQ));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXOperatorASinY : public KConstraint {
   public:
    /** Primary constructor of v1 [op] asin( v2 )
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param op operator for the constraint (GEQ, LEQ, EQ)
     */
    KNumXOperatorASinY(const KNumVar& v1, const KNumVar& v2, int op);
    /// Copy constructor
    KNumXOperatorASinY(const KNumXOperatorASinY& toCopy);
    // Destructor
    virtual ~KNumXOperatorASinY();
};

/**
 * This class creates a  X {==,<=,>=} atan(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXOperatorATanY(X, Y, GEQ));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXOperatorATanY : public KConstraint {
   public:
    /** Primary constructor of v1 [op] atan( v2 )
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param op operator for the constraint (GEQ, LEQ, EQ)
     */
    KNumXOperatorATanY(const KNumVar& v1, const KNumVar& v2, int OPERATOR);
    /// Copy constructor
    KNumXOperatorATanY(const KNumXOperatorATanY& toCopy);
    // Destructor
    virtual ~KNumXOperatorATanY();
};

/**
 * This class creates a  X {==,<=,>=} ln(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXOperatorLnY(X, Y, GEQ));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXOperatorLnY : public KConstraint {
   public:
    /** Primary constructor of v1 [op] ln( v2 )
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param op operator for the constraint (GEQ, LEQ, EQ)
     */
    KNumXOperatorLnY(const KNumVar& v1, const KNumVar& v2, int op);
    /// Copy constructor
    KNumXOperatorLnY(const KNumXOperatorLnY& toCopy);
    // Destructor
    virtual ~KNumXOperatorLnY();
};

/**
 * This class creates a  X {==,<=,>=} exp(Y)  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumXOperatorExpY(X, Y, GEQ));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXOperatorExpY : public KConstraint {
   public:
    /** Primary constructor of v1 [op] ln( v2 )
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param op operator for the constraint (GEQ, LEQ, EQ)
     */
    KNumXOperatorExpY(const KNumVar& v1, const KNumVar& v2, int op);
    /// Copy constructor
    KNumXOperatorExpY(const KNumXOperatorExpY& toCopy);
    // Destructor
    virtual ~KNumXOperatorExpY();
};

// ********************************************************************
//
//          TERNARY ARITHMETIC CONSTRAINTS X op Y op Z + (int)
//
// ********************************************************************

/**
 * This class creates a `X == Y + Z` constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * KNumVar Z(...);
 * // ...
 * problem.post(KNumEqualXYZ(X, Y, Z));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumEqualXYZ : public KConstraint {
   public:
    KNumEqualXYZ(const KNumVar& v1, const KNumVar& v2, const KNumVar& v3);
    KNumEqualXYZ(const KNumEqualXYZ& toCopy);
    virtual ~KNumEqualXYZ();
};

/**
 * This class creates a  `X == Y * Z` constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * KNumVar Z(...);
 * // ...
 * problem.post(X == Y * Z);
 * // or
 * problem.post(KNumXEqualsYTimesZ(X, Y, Z));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumXEqualsYTimesZ : public KConstraint {
   public:
    KNumXEqualsYTimesZ(const KNumVar& v1, const KNumVar& v2, const KNumVar& v3);
    KNumXEqualsYTimesZ(const KNumXEqualsYTimesZ& toCopy);
    virtual ~KNumXEqualsYTimesZ();
};

/**
 * This class creates a `abs(X-Y) <= C` constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KDistanceLowerThanXyc(X, Y, 3));	// |X-Y| <= 3
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumDistanceLowerThanXyc : public KConstraint {
   public:
    /**
     * Primary constructor of `abs(v1-v2) <= C`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param c the constant
     */
    KNumDistanceLowerThanXyc(const KNumVar& v1, const KNumVar& v2, double c);
    /// Copy Constructor
    KNumDistanceLowerThanXyc(const KNumDistanceLowerThanXyc& toCopy);
    // Destructor
    virtual ~KNumDistanceLowerThanXyc();
};

/**
 * This class creates a `abs(X-Y) >= C` constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KDistanceGreaterThanXyc(X, Y, 3));	// |X-Y| >= 3
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumDistanceGreaterThanXyc : public KConstraint {
   public:
    /**
     * Primary constructor of `abs(v1-v2) >= C`
     *
     * @param v1 the v1 variable
     * @param v2 the v2 variable
     * @param c the constant
     */
    KNumDistanceGreaterThanXyc(const KNumVar& v1, const KNumVar& v2, double c);
    /// Copy constructor
    KNumDistanceGreaterThanXyc(const KNumDistanceGreaterThanXyc& toCopy);
    // Destructor
    virtual ~KNumDistanceGreaterThanXyc();
};

/**
 * This class creates a  abs(X-Y) == C  constraint
 *
 * Example :
 * @code
 * KNumVar X(...);
 * KNumVar Y(...);
 * // ...
 * problem.post(KNumDistanceEqualXyc(X, Y, 3));	// |X-Y| == 3
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumDistanceEqualXyc : public KConstraint {
   public:
    /** Primary constructor of abs(v1-v2) == C
        @param v1 the v1 variable
        @param v2 the v2 variable
        @param c the constant
    */
    KNumDistanceEqualXyc(const KNumVar& v1, const KNumVar& v2, double c);
    /// Copy constructor
    KNumDistanceEqualXyc(const KNumDistanceEqualXyc& toCopy);
    // Destructor
    virtual ~KNumDistanceEqualXyc();
};

/**
 * This class creates a  X = atan2(Y, Z) constraint.
 * Atan2(Y, Z) is defined as follow :
 *     -  atan(Y/Z) if Z > 0
 *     -  atan(Y/Z) + PI if Z < 0 and Y >= 0
 *     -  atan(Y/Z) - PI if Z < 0 and Y < 0
 *     - (+ PI / 2) if Z = 0 and Y > 0
 *     - (- PI / 2) if Z = 0 and Y < 0
 *     -  undefined if Z = 0 and Y = 0
 *
 * Domain of X variable is at least (-PI, PI].
 *
 * Example :
 * @code
 * KFloatVar X(...);
 * KFloatVar Y(...);
 * KFloatVar Z(...);
 * // ...
 * problem.post(KNumXEqualsAtan2YZ(X, Y, Z));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2020.1
 */
class DLLIMPORTEXPORT KNumXEqualsAtan2YZ : public KConstraint {
   public:
    /** Primary constructor of X = atan2(Y, Z)
     * @param v1 the result variable
     * @param v2 the first atan2 argument variable
     * @param v3 the second atan2 argument variable
     */
    KNumXEqualsAtan2YZ(const KNumVar& v1, const KNumVar& v2, const KNumVar& v3);
    /// Copy constructor
    KNumXEqualsAtan2YZ(const KNumXEqualsAtan2YZ& toCopy);
    // Destructor
    virtual ~KNumXEqualsAtan2YZ();
};

#endif
