// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Description : SubClass of KValueSelector                         *
// * looks for the nearest value of a target (middle value by default)*
// ********************************************************************

#ifndef __KNEARESTVALUE_H
#define __KNEARESTVALUE_H

#include "Globals.h"
#include "KValueSelector.h"
#include "KNumValueSelector.h"

/**
 * This class implements a value selector that selects the nearest value from
 * target in the domain of the variable.
 *
 * Example :
 * @code
 * KBranchingSchemeArray myBranchingSchemeArray;
 * myBranchingSchemeArray += KAssignVar(KSmallestDomain(), KNearestValue());
 * @endcode
 *
 * @see KValueSelector
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNearestValue : public KValueSelector {
  public:
    /// Default Constructor
    KNearestValue();
    /// Copy constructor
    KNearestValue(const KNearestValue& nearestToCopy);
    /// Destructor
    virtual ~KNearestValue();
    // Methods
    virtual int selectNextValue(KIntVar* intVar);
    // Internal use only
    virtual KValueSelector* getCopyPtr() const;
    virtual const char * getName() const {
        return "NEAREST_VALUE";
    }
}; // class KNearestValue

/**
 * This class implements a value selector that selects the nearest value from target in the domain of the variable .
 *
 * Example :
 * @code
 * KBranchingSchemeArray myBranchingSchemeArray;
 * myBranchingSchemeArray += KSplitDomain(KWidestDomain(), KNumNearestValue());
 * @endcode
 *
 * @see KValueSelector
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumNearestValue : public KNumValueSelector {
  public:
    // Default Constructor
    KNumNearestValue();
    // Copy constructor
    KNumNearestValue(const KNumNearestValue& nearestToCopy);
    // Destructor
    virtual ~KNumNearestValue();
    //methods
    virtual double selectNextValue(KNumVar* intVar); // get Next Value
    virtual KNumValueSelector* getCopyPtr() const;
    virtual const char * getName() const {
        return "NUM_NEAREST_VALUE";
    }
}; // class KNearestValue

/**
 * A nearest neighboor branching scheme based on a distance matrix.
 */
class DLLIMPORTEXPORT KNearestNeighbor : public KValueSelector {
   public:
    /// Default Constructor with successors variable and distance matrix
    KNearestNeighbor(KIntVarArray& successors, KIntMatrix& distanceMatrix);
    // Copy constructor
    KNearestNeighbor(const KNearestNeighbor& toCopy);
    // Destructor
    virtual ~KNearestNeighbor();
    // methods
    virtual int selectNextValue(KIntVar* intVar);
    virtual KValueSelector* getCopyPtr() const;
    virtual const char* getName() const {
        return "NEAREST_NEIGHBOOR";
    }
};  // class KNearestNeighbor

#endif
