// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Description : SubClass of KValueSelector                         *
// * looks for the middle of current domain as next value             *
// ********************************************************************

#ifndef __KMIDDLE_H
#define __KMIDDLE_H

#include "Globals.h"
#include "KValueSelector.h"
#include "kalis_compat.h"

/**
 * This class implements a value selector that selects the nearest value from
 * the middle value in the domain of the variable.
 *
 * Example :
 * @code
 * KBranchingSchemeArray myBranchingSchemeArray;
 * myBranchingSchemeArray += KAssignVar(KSmallestDomain(), KMiddle());
 * @endcode
 *
 * @see KValueSelector
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KMiddle : public KValueSelector {
  public:
    /// Constructor
    KMiddle();
    /// Copy constructor
    KMiddle(const KMiddle& middleToCopy);
    // Destructor
    virtual ~KMiddle();
    //methods
    virtual int selectNextValue(KIntVar* intVar); // get Next Value
    virtual KValueSelector* getCopyPtr() const;
}; // class KMiddle

/**
 * Value selector that selects the value of a variable that implies the best
 * problem size reduction when instantiated.
 *
 * For each possible value of the domain of the variable, the variable is
 * instantiated and the problem size reduction is evaluated.
 */
class DLLIMPORTEXPORT KMinMaxConflict : public KValueSelector {
  protected :
    /**
     * True for min conflict heuristic , false for max propagate
     */
    bool _minconflict;
  public:
    /**
     * Primary Constructor
     *
     * @param minconflict : true for minimizing the induced problem size, false
     *  for maximizing.
     */
    KMinMaxConflict(bool minconflict=true);
    /** Copy constructor */
    KMinMaxConflict(const KMinMaxConflict& toCopy);
    /**
     * Selects the value of the given variable that induces the best problem
     * size once instantiated to this value.
     */
    virtual int selectNextValue(KIntVar* intVar);
    /** Return an allocated copy of the selector */
    virtual KValueSelector* getCopyPtr() const;
}; // class KMinMaxConflict


/**
 * Value selector that selects the value of a variable that implies the best
 * bound for the objective.
 *
 * For each possible value in the domain of a given variable, the variable
 * is instantiated on this value and the propagation is launched. The selected
 * value will be the value that impacted the objective in the best way.
 *
 * If the lower bound is used, the best value will be the value that induces the
 * minimal lower bound on the objective. If the upper bound is used, the
 * best value will be the value that induces the maximal upper bound on the
 * objective.
  */
class DLLIMPORTEXPORT KBestBoundValue : public KValueSelector {
  protected :
    /**
     * True for bound value on lower bound, False for bound value on upper bound
     */
    bool _lowerbound;
  public:
    /**
     * Primary Constructor
     *
     * @param lowerbound : True for bound value on lower bound, False for bound value on upper bound
     */
    KBestBoundValue(bool lowerbound);
    /**
     * Copy constructor
     */
    KBestBoundValue(const KBestBoundValue& toCopy);
    /**
     * Selects the next objective best bound value for the given variable.
     */
    virtual int selectNextValue(KIntVar* intVar);

    /** Return an allocated copy of the selector */
    virtual KValueSelector* getCopyPtr() const;
}; // class KBestBoundValue

#endif
