// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Description : user class which represents a branching scheme (the*
// * way branching could be done during resolution)                   *
// * especially for a KIntVar branching object                        *
// ********************************************************************

#ifndef __KINTVARBRANCHINGSCHEME_H
#define __KINTVARBRANCHINGSCHEME_H

#include "Globals.h"
#include "KBranchingScheme.h"

/**
 * Abstract class for  Branching scheme. Search is made thanks to a tree search algorithm. At each node, propagation is
 * made and if no solution exists, Artelys Kalis needs to split your problem in smaller subproblems covering (or not)
 * all the initial problem. This partition is made following a branching scheme.
 *
 * Different types of branching schemes exist. For example, a classical way is to choose a variable which has not been
 * instantiated so far and to build a sub-problem for each remaining value in the variable's domains, this sub-problem
 * being the original problem where the variable has been instantiated to this value. And then, you can continue the
 * search with these new nodes. Choosing the right branching schemes to be used with your particular problem could
 * greatly improve the performance of the tree search. Artelys Kalis allows you to choose between many classical
 * branching schemes provided with the library and to easily program yourself the more specialized branching schemes
 * that you suppose to be useful for your own problems. This branching scheme is suited for branching on KIntVar objects
 * only.
 *
 * @see KBranchingScheme KFloatVarBranchingScheme KAssignAndForbidd KSplitDomain KSettleDisjunction KProbe
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KIntVarBranchingScheme : public KBranchingScheme {
public:
    /// Constructor
    KIntVarBranchingScheme();
    /// Constructor with KProblem
    KIntVarBranchingScheme(KProblem* problem);
    /// Copy constructor
    KIntVarBranchingScheme(const KIntVarBranchingScheme& toCopy);
    // Destructor
    virtual ~KIntVarBranchingScheme();
    //methods
    KProblem* getProblem() const;
    /**
     * Select the next KIntVar to branch on when one branch has been explored
     */
    virtual KIntVar* selectNextBranchingVar();
    /**
     * Return true IFF branching is completed on one specific branch of the branch and bound
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual bool finishedBranching(KIntVar* branchingObject, int* branchingInformation, int currentBranchNumber);
    /**
     * Return the next branch
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual int* getNextBranch(KIntVar* branchingObject, int* branchingInformation, int currentBranchNumber);
    /**
     * This method is called once a branch has been selected and a decision must be taken
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual void goDownBranch(KIntVar* branchingObject, int* branchingInformation, int currentBranchNumber);
    /**
     * This method is called upon backtrack on a specific branch
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual void goUpBranch(KIntVar* branchingObject, int* branchingInformation, int currentBranchNumber);
    /**
     * This method is called upon finishing branching for the current node and
     * allows freeing objects created at the current node.
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     */
    virtual void freeAllocatedObjectsForBranching(KIntVar* branchingObject, int* branchingInformation);
    virtual std::string getGoDownDescription(KIntVar* branchingObject, int* branchingInformation, int currentBranchNumber);
    //methods from KBranchingScheme
    virtual void* selectNextBranchingObject(); // get Next Branching Object
    virtual bool finishedBranching(void* branchingObject,void* branchingInformation,int currentBranchNumber);
    virtual void* getNextBranch(void* branchingObject,void* branchingInformation,int currentBranchNumber);
    virtual void goDownBranch(void* branchingObject,void* branchingInformation,int currentBranchNumber);
    virtual void goUpBranch(void* branchingObject,void* branchingInformation,int currentBranchNumber);
    virtual void freeAllocatedObjectsForBranching(void* branchingObject,void* branchingInformation);
    /// Return a string representation of the branching decision
    virtual std::string getGoDownDescription(void* branchingObject, void* branchingInformation, int currentBranchNumber);
    virtual KBranchingScheme* getCopyPtr() const;
    virtual KBranchingScheme* getInstanceCopyPtr(const KProblem& problem) const;

}; // class KIntVarBranchingScheme

#endif
