// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KINTVAR_H
#define __KINTVAR_H

#include <iostream>

#include "Globals.h"
#include "KNumVar.h"
#include "kalis_compat.h"

class KProblem;
class KIntArray;

/**
 * This class implements an integer variable with enumerated (finite) domain.
 * Decision variables are the variable quantities that we are trying to
 * instantiate in order to satisfy the constraints of our problem.
 * In this version, Artelys Kalis works with integer variables : decision
 * variables which are constrained to take only integer values.
 * These integer variables are represented by instances of the class KIntVar.
 *
 * Example :
 * @code
 * KProblem  p(...);
 * // X is an integer variable that can take value 0,1,2,3,4,5,6,7,8,9,10
 * KIntVar X(p, "X", 0, 10);
 * // Y is an integer variable that can take value 7,8,10 (3 different values)
 * KIntVar Y(p, "Y", KIntArray(3, 7, 8, 10));
 *
 * // Z is an integer variable that can take value 3,4,5
 * KIntVar Z;
 * Z = KIntVar(p,3,5);
 * @endcode
 *
 * @see  KIntArray KIntVarArray
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KIntVar : public KNumVar {
  private :
    // Pointer to the internal object implementing this KIntVar
    KIntVar_I * _intVar_I_ptr;
  public:
    /// Default constructor
    KIntVar();
    /**
     * Main constructor : minValue and maxValue are the bounds of the KIntVar's domain
     *
     * @param problem Involved problem
     * @param name Name of the variable
     * @param minValue lower bound for the variable
     * @param maxValue upper bound for the variable
     * @param isBoundIntVar false if the domain may have holes, true otherwise
     */
    KIntVar(KProblem& problem, const char* name, const int minValue = 0, const int maxValue = 1,
            bool isBoundIntVar = false);
    /// Main constructor : enumerated domain
    KIntVar(KProblem &problem, const char * name, KIntArray &domain);

    KIntVar(KProblem& problem, const char * name, const int minValue, const int maxValue, bool isBoundIntVar, KIntVar& boolVar);
    KIntVar(KProblem& problem, const char * name, KIntArray &domain, KIntVar& boolVar);
    /// Basic constructor
    KIntVar(KProblem &problem);
#ifdef ACK_LIB
    // Constructor with KNumVar_I
    KIntVar(KNumVar_I* numVar);
#endif
    /// Copy constructor
    KIntVar(const KIntVar& toCopy);
    // constructor using a KIntVar_I
    KIntVar(KIntVar_I& var_I) {
        _intVar_I_ptr = &var_I;
    };
    //destructor
    virtual ~KIntVar();
    virtual KIntVar* getInstance(unsigned int pb) const;
    /// Instantiate the variable to a value
    void instantiate(const int value);
    /// Remove value from the variable's domain
    void remVal(const int value);
    /// Remove the interval [rinf, rsup] (both bounds included) from the variable's domain
    void removeInterval(const int rinf, const int rsup);
    /// Return the name of the variable
    const char* getName() const;
    /// Set the name of the variable
    void setName(const char * name);
    /// Returns lower bound of this variable
    double getInf() const;
    /// Returns integer lower bound of this variable
    int getIntInf() const;
    /// Returns value in variable's domain and close to the middle
    double getMiddle() const;
    /// Returns upper bound of this variable
    double getSup() const;
    /// Returns integer upper bound of this variable
    int getIntSup() const;
    /// Returns current domain size of the variable
    int getDomainSize() const;
    /// Returns current instantiation of the variable (when the variable is not instantiated the returned value is undefined)
    double getValue(void) const;
    /// Returns current instantiation of the variable (when the variable is not instantiated the returned value is undefined)
    int getIntValue(void) const;
    /// Returns true if the variable has been assigned a value, false otherwise
    bool getIsInstantiated() const;
    /// Returns the number of constraints where this variable appears
    int getDegree() const;
    /// Get target value
    double getTarget() const;
    /// Get a random value in the domain of the variable
    int getRandomValue() const;
    /// Get value immediatly after "next" in the domain of the variable and put it into next
    void getNextDomainValue(int &next);
    /// Get value immediatly before "prev" in the domain of the variable and put it into prev
    void getPrevDomainValue(int &prev);
    /// Shave lower bound of variable
    bool shaveFromLeft(void);
    /// Shave upper bound of variable
    bool shaveFromRight(void);
    /// Shave the value 'val'
    bool shaveOnValue(int val);

    /// Set the lower bound to value
    void setInf(int value);
    /// Set the upper bound to value
    void setSup(int value);
    /// Set the target value
    void setTarget(int value);
    /// Optimize the internal representation of the domain
    void optimizeDomainRepresentation();
    /// Check if value is in the domain
    bool canBeInstantiatedTo(int value);
    /// Check if equal to x
    bool isEqualTo(KIntVar &x);
    /// Return a copy of this KIntVar object
    KIntVar* getCopyPtr() const;
    /// overloaded operator =
    KIntVar& operator=(const KIntVar &toCopy);
    // Internal use only
    KNumVar_I *getNumVar_I_ptr() const;
    // Internal use only
    KIntVar_I *getIntVar_I_ptr() const;
    // Internal use only
    void setIntVar_I_ptr(KIntVar_I * intVar);
    /// Pretty printing of the variable to an ouput stream
    virtual void print(std::ostream & fout) const;
    /// Pretty printing to the standard ouput stream
    virtual void print(void) const;
    // Internal use only
    virtual void print(void * ctx, PrintFunctionPtr*pfp) const;
    // Internal use only
    int instanceof(void) const;
    // Kalis hybrid
    // create indicators (KauxVar objects associated with a value in the domain of "*this")
    void createIndicators (void);
    // update the indicators with respect to the current domain of the KIntVar_I
    void updateIndicators (void) const;
    // get a pointer to the aux var associated with a value (nullptr if undefined)
    KAuxVar * getIndicator (int value) const;
}; // class KIntVar

#endif
