// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Description : SubClass of KVariableSelector                      *
// * looks for first unassigned variable                              *
// ********************************************************************

#ifndef __KINPUTORDER_H
#define __KINPUTORDER_H

#include "Globals.h"
#include "KVariableSelector.h"
#include "KNumVariableSelector.h"
#include "kalis_compat.h"

/**
 * This class implements a variable selector that selects the first uninstantiated variable in the input order.
 *
 * Example :
 * @code
 * KBranchingSchemeArray myBranchingSchemeArray;
 * myBranchingSchemeArray += KAssignVar(KInputOrder(),KMaxToMin());
 * @endcode
 *
 * @see KVariableSelector
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KInputOrder : public KVariableSelector {
  public:
    /// Constructor
    KInputOrder();
    /// Copy constructor
    KInputOrder(const KInputOrder& inputOrderToCopy);
    // Destructor
    virtual ~KInputOrder();
    //methods
    /// return the first uninstantiated variable in the order of creation
    virtual KIntVar* selectNextVariable(KIntVarArray* intVarArray);
    virtual KVariableSelector* getCopyPtr() const;
}; // class KInputOrder

/**
 * This class implements a variable selector that selects the first uninstantiated variable in the input order.
 *
 * Example :
 * @code
 * KBranchingSchemeArray bsa;
 * bsa += KAssignVar(KNumInputOrder(), KMaxToMin());
 * @endcode
 *
 * @see KVariableSelector
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KNumInputOrder : public KNumVariableSelector {
  public:
    /// Constructor
    KNumInputOrder();
    /// Copy constructor
    KNumInputOrder(const KNumInputOrder& toCopy);
    // Destructor
    virtual ~KNumInputOrder();
    //methods
    /// return the first uninstantiated variable in the order of creation
    virtual KNumVariableSelector*  getCopyPtr() const;

}; // class KNumInputOrder


/**
 * A variable selector based on a path order.
 *
 * The initial successor is chosen randomly. Then, the following chosen
 * variable will the designated successor in the previous branching choice.
 *
 * @see KVariableSelector
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KPathOrder : public KVariableSelector {
  public:
    KPathOrder();
    KPathOrder(KIntVarArray& successors);
    /// Constructor
    KPathOrder(const KIntVarArray& successors, const KIntMatrix& distanceMatrix);
    /// Copy constructor
    KPathOrder(const KPathOrder& toCopy);
    // Destructor
    virtual ~KPathOrder();
    //methods
    /// return the first uninstantiated variable in the order of creation
    virtual KVariableSelector* getCopyPtr() const;
}; // class KPathOrder

#endif
