// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KHYBRIDSOLUTION_H
#define __KHYBRIDSOLUTION_H

#include "Globals.h"
#include "kalis_compat.h"

/**
 * This class represents a solution of a relaxation solver, that is, a mapping
 * from variables (KNumVar and/or KAuxVar) to their value.
 *
 * Example :
 * @code
 * KXPRSLinearRelaxationSolver mySolver(...);
 * mySolver.solve();
 * solverMIP.updateSolution();
 * KHybridSolution * mySol = solverMIP.getSolutionPtr();
 * mySol->print();
 * @endcode
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KHybridSolution {
  protected:
    KHybridSolution_I * _InternalPtr;
  public:
    /// Default constructor.
    KHybridSolution();
    // Constructor reserved for internal use.
    KHybridSolution(KHybridSolution_I*);
    /// Copy constructor (forbidden).
    KHybridSolution(const KHybridSolution&);
    /// Destructor.
    ~KHybridSolution();
    // ----- methods -----
    /// Print solution.
    void print (void) const;
    // ----- accessors -----
    // Reserved for internal use.
    KHybridSolution_I* getInternalPtr (void) const;
    /**
     * Set the value of a KNumVar.
     *
     * @param variable whose value is modified
     * @param new value
     */
    void setVal(const KNumVar* var, double val);
    /**
     * Set the value of a KAuxVar.
     *
     * @param var varaible to modify
     * @param new value
     */
    void setVal(const KAuxVar* var, double val);
    /// Get the value of a KNumVar.
    double getVal(const KNumVar*) const;
    /// Get the value of a KAuxVar.
    double getVal(const KAuxVar*) const;
};

#endif
