// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Description : user class which represents a branching scheme (the*
// * way branching could be done during resolution)                   *
// * especially for a KFloatVar branching object                      *
// ********************************************************************

#ifndef __KFLOATVARBRANCHINGSCHEME_H
#define __KFLOATVARBRANCHINGSCHEME_H

#include "Globals.h"
#include "KBranchingScheme.h"

/**
 * This branching scheme is suited for branching on KFloatVar objects.
 *
 * @see KBranchingScheme KIntVarBranchingScheme KAssignAndForbidd KSplitDomain KSettleDisjunction KProbe
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KFloatVarBranchingScheme : public KBranchingScheme {
   public:
    /// Constructor
    KFloatVarBranchingScheme();
    /// Constructor with KProblem
    KFloatVarBranchingScheme(KProblem* problem);
    /// Copy constructor
    KFloatVarBranchingScheme(const KFloatVarBranchingScheme& toCopy);
    /// Destructor
    virtual ~KFloatVarBranchingScheme();
    /// Problem getter
    KProblem* getProblem() const;

    /**
     * Select the next KNumVar to branch on when one branch has been explored
     */
    virtual KNumVar* selectNextBranchingVar();
    /**
     * Return true IFF branching is completed on one specific branch of the branch and bound
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual bool finishedBranching(KNumVar* branchingObject, double* branchingInformation, int currentBranchNumber);
    /**
     * Return the next branch
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual double* getNextBranch(KNumVar* branchingObject, double* branchingInformation, int currentBranchNumber);
    /**
     * This method is called once a branch has been selected and a decision must be taken
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual void goDownBranch(KNumVar* branchingObject, double* branchingInformation, int currentBranchNumber);
    /**
     * This method is called upon backtrack on a specific branch
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual void goUpBranch(KNumVar* branchingObject, double* branchingInformation, int currentBranchNumber);
    /**
     * This method is called upon finishing branching for the current node and allows freeing objects created at the
     * current node
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     */
    virtual void freeAllocatedObjectsForBranching(KNumVar* branchingObject, double* branchingInformation);
    // methods from KBranchingScheme
    virtual void* selectNextBranchingObject();  // get Next Branching Object
    virtual bool finishedBranching(void* branchingObject, void* branchingInformation, int currentBranchNumber);
    virtual void* getNextBranch(void* branchingObject, void* branchingInformation, int currentBranchNumber);
    virtual void goDownBranch(void* branchingObject, void* branchingInformation, int currentBranchNumber);
    virtual void goUpBranch(void* branchingObject, void* branchingInformation, int currentBranchNumber);
    virtual void freeAllocatedObjectsForBranching(void* branchingObject, void* branchingInformation);
    virtual KBranchingScheme* getCopyPtr() const;
    virtual KBranchingScheme* getInstanceCopyPtr(const KProblem& problem) const;
};  // class KFloatVarBranchingScheme

#endif
