// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KELEMENT_H
#define __KELEMENT_H

#include "Globals.h"
#include "KConstraint.h"

/**
 * This class creates a x == tab[i + cste] constraint
 *
 * Example :
 * @code
 * KIntArray tab(...);
 * KIntVar x(...);
 * KIntVar i(...);
 * // ...
 * problem.post(KElement(tab,i,x,4,"x == tab[i + 4]"));
 * @endcode
 *
 * @see KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KElement : public KConstraint {
   public:
    /**
     * Constructor for `x == tab[i + cste]`
     * @param tab the values
     * @param i the index variable
     * @param x the value variable
     * @param offset constant offset of index
     * @param name name of the constraint
     */
    KElement(KIntArray& tab, const KIntVar& i, const KIntVar& x, const int offset, char* name = 0);

    /**
     * Constructor for `x == tab[i]`
     * @param tab the values
     * @param i the index variable
     * @param x the value variable
     * @param name name of the constraint
     */
    KElement(KIntArray& tab, const KIntVar& i, const KIntVar& x, char* name = 0);

    /**
     * Constructor for `x == tab[i + cste]`
     * @param tab the values
     * @param i the index variable
     * @param x the value constant
     * @param offset constant offset of index
     * @param name name of the constraint
     */
    KElement(KIntArray& tab, const KIntVar& i, const int x, const int offset, char* name = 0);

    /**
     * Constructor for `x == tab[i]`
     * @param tab the values
     * @param i the index variable
     * @param x the value constant
     * @param name name of the constraint
     */
    KElement(KIntArray& tab, const KIntVar& i, const int x, char* name = 0);

    /**
     * Constructor for `x = D[i + offset]` with `D` being a sparse interval map.
     *
     * Given a list of intervals `[s_i, e_i)` with associated values `v_i`, then
     * `s_i <= y < e_i => D[y] = v_i`
     *
     * @param i t
     * @param x the value variable
     * @param starts Interval starts `s_i`
     * @param ends Interval ends `e_i`
     * @param values Interval values `v_i`
     * @param offset potential offset for the indxing variable
     * @param name name of the constraint
     */
    KElement(const KIntVar& i, const KIntVar& x,
             const KIntArray& starts,
             const KIntArray& ends,
             const KIntArray& values,
             const int offset = 0, const char* name = 0);

    // Internal constructors
    KElement(const KEltTerm& e, const KIntVar& x, char* name = 0);
    KElement(const KEltTerm& e, const int x, char* name = 0);
    // Value function from index
    virtual int getValueForIndex(int index);
    // Use overloaded "getValueForIndex" function instead of integer array
    void setUseValueFunction(bool useValueFunction);
    // Copy constructor
    KElement(const KElement& toCopy);
    // Destructor
    virtual ~KElement();
    // Internal use only
    virtual void* getConstraintIPtr();
};

#endif
