// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Description : class which is in charge of founding a variable    *
// * following specific parameteres (used in KBranchingScheme (mainly *
// * KAssignVar))                                                     *
// ********************************************************************

#ifndef __KDISJUNCTIONSELECTOR_H
#define __KDISJUNCTIONSELECTOR_H

#include <iostream>

#include "Globals.h"
#include "kalis_compat.h"

/**
* Abstract interface class for disjunction selection heuristic
* @since 2016.1
*/
class DLLIMPORTEXPORT KDisjunctionSelector {
  private:
    KProblem* _problem;
  public:
    /// Constructor
    KDisjunctionSelector();
    /// Constructor with KProblem
    KDisjunctionSelector(KProblem* problem);
    /// Copy constructor
    KDisjunctionSelector(const KDisjunctionSelector& selectorToCopy);
    /// Destructor
    virtual ~KDisjunctionSelector();

    KProblem* getProblem() const;

    /**
    * Virtual interface method to overload for definition of your own disjunction selection heuristics
    * @param intVarArray Array of variable from wich selecting  a variable
    */
    virtual KDisjunction * selectNextDisjunction(KDisjunctionArray* disjunctionArray);
    /// Return a copy of this object
    virtual KDisjunctionSelector* getCopyPtr() const;
    /// Print the name of this disjunction selector
    virtual void printName() const {
      std::cout << "KDisjunctionSelector" << std::endl;
    }
    /// Return the name of this disjunction selector
    virtual const char * getName() const {
        return "KDisjunctionSelector";
    }
}; // class KVariableSelector

/**
* This class implements a disjunction selector that selects the disjunction in the input order.
* @code
*  Example :
*
*	KBranchingSchemeArray myBranchingSchemeArray;
*	myBranchingSchemeArray += KSettleDisjunction(new KDisjunctionInputOrder());
*
* @endcode
* @see  KDisjunctionSelector
* @since 2016.1
*/
class DLLIMPORTEXPORT KDisjunctionInputOrder : public KDisjunctionSelector {
  public:
    /// Constructor
    KDisjunctionInputOrder();
    /// Copy constructor
    KDisjunctionInputOrder(const KDisjunctionInputOrder& toCopy);
    // Destructor
    virtual ~KDisjunctionInputOrder();
    //methods
    virtual KDisjunction * selectNextDisjunction(KDisjunctionArray* disjunctionArray);
    virtual KDisjunctionSelector* getCopyPtr() const;
};

/**
 * This class implements a disjunction selector that selects first the disjunction ith the highest priority
 *
 * Example :
 * @code
 * KBranchingSchemeArray myBranchingSchemeArray;
 * myBranchingSchemeArray += KSettleDisjunction(new KDisjunctionPriorityOrder());
 * @endcode
 *
 * @see  KDisjunctionSelector
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KDisjunctionPriorityOrder : public KDisjunctionSelector {
  public:
    /// Constructor
    KDisjunctionPriorityOrder();
    /// Copy constructor
    KDisjunctionPriorityOrder(const KDisjunctionPriorityOrder& toCopy);
    // Destructor
    virtual ~KDisjunctionPriorityOrder();
    //methods
    virtual KDisjunction * selectNextDisjunction(KDisjunctionArray* disjunctionArray);
    virtual KDisjunctionSelector* getCopyPtr() const;
};

#endif
