// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Description : user class which represents a branching scheme (the*
// * way branching could be done during resolution)                   *
// ********************************************************************

#ifndef __KBRANCHINGSCHEME_H
#define __KBRANCHINGSCHEME_H

#include <string>

#include "Globals.h"

/**
 * Abstract class defining branching schemes.
 * Search is made thanks to a tree search algorithm. At each node, propagation
 * is made and if no solution exists, Artelys Kalis needs to split your problem
 * in smaller subproblems covering (or not) all the initial problem. This
 * partition is made following a branching scheme.
 *
 * Different types of branching schemes exist. For example, a classical way is
 * to choose a variable which has not been instantiated so far and to build a
 * sub-problem for each remaining value in the variable's domains, this
 * sub-problem being the original problem where the variable has been
 * instantiated to this value. And then, you can continue the search with these
 * new nodes.
 *
 * Choosing the right branching schemes to be used with your particular problem
 * could greatly improve the performance of the tree search. Artelys Kalis
 * allows you to choose between many classical branching schemes provided with
 * the library and to easily program yourself the more specialized branching
 * schemes that you suppose to be useful for your own problems.
 *
 * @see KAssignAndForbid KSplitDomain KSettleDisjunction KProbe
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KBranchingScheme : public KAutoExternalObject<KBranchingScheme, KBranchingScheme_I>,
                                         public KPtrArray<KBranchingScheme> {
   protected:
    KProblem* _problem;

   public:
    /// Constructor
    KBranchingScheme();
    /// Constructor with a given problem
    KBranchingScheme(KProblem* problem);
    // Internal use
    KBranchingScheme(KBranchingScheme_I* branchingScheme);
    // Internal use
    KBranchingScheme(const KBranchingScheme& toCopy);
    // Destructor
    virtual ~KBranchingScheme();
    /// Return the current problem
    KProblem* getProblem() const;

    /**
     * Select the next object (variable in general) to branch on when one branch
     * has been explored.
     */
    virtual void* selectNextBranchingObject();

    /**
     * Return true IFF branching is completed on one specific branch of the
     * branch and bound.
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual bool finishedBranching(void* branchingObject, void* branchingInformation, int currentBranchNumber);

    /**
     * Return the next branch to explore
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual void* getNextBranch(void* branchingObject, void* branchingInformation, int currentBranchNumber);

    /**
     * This method is called once a branch has been selected and a decision must
     * be taken.
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual void goDownBranch(void* branchingObject, void* branchingInformation, int currentBranchNumber);

    /**
     * This method is called upon backtrack on a specific branch
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     * @param currentBranchNumber the current branch number
     */
    virtual void goUpBranch(void* branchingObject, void* branchingInformation, int currentBranchNumber);

    /**
     * This method is called upon finishing branching for the current node and
     * allows freeing objects created at the current node.
     *
     * @param branchingObject the branching object
     * @param branchingInformation the branching information
     */
    virtual void freeAllocatedObjectsForBranching(void* branchingObject, void* branchingInformation);

    /// Pretty printing of the branching scheme
    virtual void printName() const;
    /// Return the name of the branching scheme
    virtual const char* getName() const;
    /// Return a string representation of the branching decision
    virtual std::string getGoDownDescription(void* branchingObject, void* branchingInformation, int currentBranchNumber);

    // Internal use
    virtual KBranchingScheme* getCopyPtr() const;
    // Internal use
    virtual KBranchingScheme* getInstanceCopyPtr(const KProblem& problem) const;
    // Internal use
    virtual void setSolver_I_ptr(KSolver_I* solver_I);

};  // class KBranchingScheme

EXTTEMPL template class DLLIMPORTEXPORT KPtrArray<KBranchingScheme>;

#endif
