// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// *                                                                  *
// * Auxiliary variables used in relaxations                          *
// ********************************************************************

#ifndef __KAUXVAR_H
#define __KAUXVAR_H

#include "Globals.h"
#include "kalis_compat.h"

/**
 * This class represents an auxiliary variable to use in relaxations.
 *
 * KAuxVar objects represent auxiliary variables, consisting of a name, lower
 * and upper bounds, and a type that is either "global" or "continuous". They
 * are intended to be used in relaxations, as new variables that are not needed
 * in the CP problem but that can be necessary in the LP/MIP formulation.
 *
 * Example (creation of a boolean auxiliary variable) :
 * @code
 * KAuxVar auxVar(0, 1, true, "bool aux var");
 * @endcode
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KAuxVar : public KAutoExternalObject<KAuxVar,KAuxVar_I>, public KPtrArray<KAuxVar> {
public:
    /** \brief main constructor.

        @param problem current problem
        @param inf lower bound
    	@param sup upper bound
    	@param isGlobal variable type (global or continuous)
    	@param name name used for printing
    */
    KAuxVar(const KProblem& problem, double inf, double sup, bool isGlobal = true, const char* name = nullptr);
    //
    KAuxVar(KAuxVar_I* auxVar);
    /// copy constructor
    KAuxVar (const KAuxVar& toCopy);
    /// destructor
    virtual ~KAuxVar ();
    /// get the lower bound
    double getInf (void) const;
    /// get the upper bound
    double getSup (void) const;
    /// Get the name of this auxiliary variable
    const char * getName(void) const;
    /** set the lower bound.
    @param inf new lower bound
    @param chain if this flag is set, Kalis attempts to chain the change in the lower bound of the auxiliary variable
    to the "real" variables (KIntVar and KFloatVar). Useful only for KAuxVar automatically generated by Kalis, such as
    indicator auxiliary variables.
    @return a flag indicating if the domain was reduced
    */
    virtual bool setInf (double inf, bool chain = true);
    /** set the upper bound.
    @param inf new upper bound
    @param chain if this flag is set, Kalis attempts to chain the change in the upper bound of the auxiliary variable
    to the "real" variables (KIntVar and KFloatVar). Useful only for KAuxVar automatically generated by Kalis, such as
    indicator auxiliary variables.
    @return a flag indicating if the domain was reduced
    */
    virtual bool setSup (double sup, bool chain = true);
    /** set both bounds
    @param inf new value
    @param chain if this flag is set, Kalis attempts to chain the change in the auxiliary variable
    to the "real" variables (KIntVar and KFloatVar). Useful only for KAuxVar automatically generated by Kalis, such as
    indicator auxiliary variables.
    @return true iff the variable was not already instantiated to val
    */
    // Internal use
    KAuxVar_I * getInternalObject() const;
    void setInternalObject(KAuxVar_I * auxVar);

    virtual bool instantiate (double val, bool chain = true);
    /// check variable type
    bool isGlobal(void) const;
    /// print the variable
    virtual void print(void) const;
    virtual void print(void * ctx,PrintFunctionPtr*pfp) const;
    // Internal use
    KAuxVar * getCopyPtr() const;
    const KProblem* getKProblem() const;
    virtual double getTarget() const;
    virtual void setTarget(const double target);
  private:
    /// for automatic name creation
    static int nextAvailableNumber;
    const KProblem* _kProblem;
  protected:
    KAuxVar_I * _auxVarIPtr;
};

#endif
