// ********************************************************************
// * Artelys Kalis                                                    *
// * Copyright (C) 2001-2024 by Artelys                               *
// * All Rights Reserved                                              *
// ********************************************************************

#ifndef __KACBINCONSTRAINT_H
#define __KACBINCONSTRAINT_H

#include "Globals.h"
#include "KConstraint.h"

/**
 * This class implements a generic class for propagation of any binary constraint by local 2-consistency (arc consistency) \\
 * Two algorithms (AC3 and AC2001) are available for propagation of the constraint.
 *
 * Example : X == Y + C
 * @code
 *   class XEqualYC : public KACBinConstraint {
 *     int _C;
 *     public:
 *      XEqualYC(const char* name, KIntVar& v1, KIntVar& v2, int cst)
 *           : KACBinConstraint(v1, v2, KACBinConstraint::ALGORITHM_AC2001, "XEqualYC") {
 *         _C = cst;
 *      }
 *      virtual bool testIfSatisfied(int valX, int valY) {
 *         return (valX == valY + _C);	// the constraint is true if only iff valX == valY + C
 *      }
 *   };
 * @endcode
 *
 * @see KACBinTableConstraint KConstraint
 *
 * @since 2016.1
 */
class DLLIMPORTEXPORT KACBinConstraint : public KConstraint {
  public :
    /** Constructor.
     * This constructor takes threee arguments
     *
     * @param v1 the first variable of the constraint
     * @param v2 the second variable of the constraint
     * @param acAlgorithm ALGORITHM_AC2001 (default value) for propagation by the AC2001 algorithm , ALGORITHM_AC3 for propagation by the AC3 algorithm
     * @param name label for pretty printing of the constraint
     */
    KACBinConstraint(KIntVar& v1, KIntVar& v2, int acAlgorithm = ALGORITHM_AC2001, const char * name=0);
    // Copy Constructor
    KACBinConstraint(const KACBinConstraint &toCopy);
    // Destructor
    virtual ~KACBinConstraint();
    // Internal use
    virtual KConstraint* getInstance(unsigned int pb) const;
    /**
     * Abstract interface for generic propagation of any binary constraint
     *
     * @return true if and only if the constraint is satisfied when v1 == val1 & v2 == val2
     */
    virtual bool testIfSatisfied(int val1,int val2);
    /**
     * Virtual copy method.
     * Each modeling elements stored (and used) in the binary constraint must be copied using the KProblem::getCopyPtr() method.
     */
    virtual KACBinConstraint* getInstanceCopyPtr(const KProblem& problem) const;
    virtual void print() const;
    virtual void print(void*ctx,PrintFunctionPtr*pfp) const;
    /// Differents level of propagation for the constraints
    enum acAlgorithms {
        ALGORITHM_AC3,
        ALGORITHM_AC2001
    };
};

#endif
