/* *********************************************************************** *
 *                                                                         *
 * xpress_objects.hpp                                                      *
 *                                                                         *
 * C++ API for Xpress-Optimizer                                            *
 *                                                                         *
 * (c) Copyright Fair Isaac Corporation 2024-2025. All rights reserved     *
 * For FICO Xpress Optimizer v45.01.02                                     *
 *                                                                         *
 * *********************************************************************** */

#ifndef XPRESS_OBJECTS_H
#define XPRESS_OBJECTS_H 1
#include "xpress.hpp"
#include "xpress_maps.hpp"
namespace xpress {

template <typename AlwaysVoid> class GlobalConstants {
public:
  static xpress::objects::Token const xpress_objects_Token_EOF;
  static xpress::objects::Token const xpress_objects_Token_LB;
  static xpress::objects::Token const xpress_objects_Token_RB;
  static xpress::objects::Token const xpress_objects_Token_UMINUS;
  static xpress::objects::Token const xpress_objects_Token_EXPONENT;
  static xpress::objects::Token const xpress_objects_Token_MULTIPLY;
  static xpress::objects::Token const xpress_objects_Token_DIVIDE;
  static xpress::objects::Token const xpress_objects_Token_PLUS;
  static xpress::objects::Token const xpress_objects_Token_MINUS;
  static xpress::objects::Token const xpress_objects_Token_COMMA;
  static xpress::objects::Token const xpress_objects_Token_COLON;
  static xpress::objects::Token const xpress_objects_Token_LOG10;
  static xpress::objects::Token const xpress_objects_Token_LN;
  static xpress::objects::Token const xpress_objects_Token_EXP;
  static xpress::objects::Token const xpress_objects_Token_ABS;
  static xpress::objects::Token const xpress_objects_Token_SQRT;
  static xpress::objects::Token const xpress_objects_Token_SIN;
  static xpress::objects::Token const xpress_objects_Token_COS;
  static xpress::objects::Token const xpress_objects_Token_TAN;
  static xpress::objects::Token const xpress_objects_Token_ARCSIN;
  static xpress::objects::Token const xpress_objects_Token_ARCCOS;
  static xpress::objects::Token const xpress_objects_Token_ARCTAN;
  static xpress::objects::Token const xpress_objects_Token_MIN;
  static xpress::objects::Token const xpress_objects_Token_MAX;
  static xpress::objects::Token const xpress_objects_Token_PWL;
  static xpress::objects::Token const xpress_objects_Token_SUM;
  static xpress::objects::Token const xpress_objects_Token_PROD;
  static xpress::objects::Token const xpress_objects_Token_SIGN;
  static bool xpress_objects_LinTermList_autoCompress;
  static bool xpress_objects_QuadTermList_autoCompress;
};
} // namespace xpress
namespace xpress {

namespace objects {

class SymbolicEvalStack final {
private:
  std::vector<Expression> stack;

  int pos;

public:
  inline SymbolicEvalStack(int capacity);

  inline auto push(Expression e) -> void;

  inline auto pop() -> xpress::objects::Expression;

  inline auto peek() -> xpress::objects::Expression;

  inline auto size() -> int;
};

class Token final {
  friend class xpress::GlobalConstants<void>;
private:
  int type;

public:
  auto getType() const -> int { return type; }

private:
  double value;

public:
  auto getValue() const -> double { return value; }

private:
  inline Token(int type, double value);

  inline Token(int type);

public:
  inline Token();

  inline auto equals(Token const &other) const -> bool;

  inline auto compareTo(Token const &other) const -> int;

  inline auto getHashCode() const -> std::size_t;

  inline auto operator==(Token const &other) const -> bool;

  inline auto operator!=(Token const &other) const -> bool;

  static inline constexpr Token const &EOFORMULA =
      xpress::GlobalConstants<void>::xpress_objects_Token_EOF;

  static inline constexpr Token const &LB =
      xpress::GlobalConstants<void>::xpress_objects_Token_LB;

  static inline constexpr Token const &RB =
      xpress::GlobalConstants<void>::xpress_objects_Token_RB;

  static inline constexpr Token const &UMINUS =
      xpress::GlobalConstants<void>::xpress_objects_Token_UMINUS;

  static inline constexpr Token const &EXPONENT =
      xpress::GlobalConstants<void>::xpress_objects_Token_EXPONENT;

  static inline constexpr Token const &MULTIPLY =
      xpress::GlobalConstants<void>::xpress_objects_Token_MULTIPLY;

  static inline constexpr Token const &DIVIDE =
      xpress::GlobalConstants<void>::xpress_objects_Token_DIVIDE;

  static inline constexpr Token const &PLUS =
      xpress::GlobalConstants<void>::xpress_objects_Token_PLUS;

  static inline constexpr Token const &MINUS =
      xpress::GlobalConstants<void>::xpress_objects_Token_MINUS;

  static inline constexpr Token const &COMMA =
      xpress::GlobalConstants<void>::xpress_objects_Token_COMMA;

  static inline constexpr Token const &COLON =
      xpress::GlobalConstants<void>::xpress_objects_Token_COLON;

  static inline constexpr Token const &LOG10 =
      xpress::GlobalConstants<void>::xpress_objects_Token_LOG10;

  static inline constexpr Token const &LN =
      xpress::GlobalConstants<void>::xpress_objects_Token_LN;

  static inline constexpr Token const &EXP =
      xpress::GlobalConstants<void>::xpress_objects_Token_EXP;

  static inline constexpr Token const &ABS =
      xpress::GlobalConstants<void>::xpress_objects_Token_ABS;

  static inline constexpr Token const &SQRT =
      xpress::GlobalConstants<void>::xpress_objects_Token_SQRT;

  static inline constexpr Token const &SIN =
      xpress::GlobalConstants<void>::xpress_objects_Token_SIN;

  static inline constexpr Token const &COS =
      xpress::GlobalConstants<void>::xpress_objects_Token_COS;

  static inline constexpr Token const &TAN =
      xpress::GlobalConstants<void>::xpress_objects_Token_TAN;

  static inline constexpr Token const &ARCSIN =
      xpress::GlobalConstants<void>::xpress_objects_Token_ARCSIN;

  static inline constexpr Token const &ARCCOS =
      xpress::GlobalConstants<void>::xpress_objects_Token_ARCCOS;

  static inline constexpr Token const &ARCTAN =
      xpress::GlobalConstants<void>::xpress_objects_Token_ARCTAN;

  static inline constexpr Token const &MIN =
      xpress::GlobalConstants<void>::xpress_objects_Token_MIN;

  static inline constexpr Token const &MAX =
      xpress::GlobalConstants<void>::xpress_objects_Token_MAX;

  static inline constexpr Token const &PWL =
      xpress::GlobalConstants<void>::xpress_objects_Token_PWL;

  static inline constexpr Token const &SUM =
      xpress::GlobalConstants<void>::xpress_objects_Token_SUM;

  static inline constexpr Token const &PROD =
      xpress::GlobalConstants<void>::xpress_objects_Token_PROD;

  static inline constexpr Token const &SIGN =
      xpress::GlobalConstants<void>::xpress_objects_Token_SIGN;

  static inline auto constant(double value) -> xpress::objects::Token;

  static inline auto column(int col) -> xpress::objects::Token;

  static inline auto function(int id) -> xpress::objects::Token;

  static inline auto internalFunction(int id) -> xpress::objects::Token;

  static inline auto of(int type, double value) -> xpress::objects::Token;

  inline auto toString() const -> std::string;
};
template <typename C> class ConstraintDefinitionImplementation;

template <typename C> class ConstraintDefinition {
protected:
  std::shared_ptr<ConstraintDefinitionImplementation<C>> implPointer = nullptr;

public:
  bool isSameImpl(ConstraintDefinition const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(ConstraintDefinitionImplementation<C> const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline ConstraintDefinition(
      std::shared_ptr<ConstraintDefinitionImplementation<C>> init);

public:
  inline ConstraintDefinition(
      ConstraintDefinitionImplementation<C> const *init);
  inline virtual ~ConstraintDefinition(){};

private:
  inline ConstraintDefinitionImplementation<C> *impl();
  inline ConstraintDefinitionImplementation<C> const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(ConstraintDefinition const &other) const {
    return impl() == other.impl();
  }
  typedef C constraint_type;

protected:
  virtual inline auto checkModify() -> void final;

public:
  virtual inline auto getName() const -> std::optional<std::string>;

  virtual inline auto setConstraintName(std::optional<std::string> newName)
      -> ConstraintDefinition<C> & final;

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const -> void;

public:
  virtual inline auto
  create(xpress::XPRSProblem::ConstraintCreator *creator) const -> void final;

  virtual inline auto getIndexHandler(XpressProblem *prob) const
      -> xpress::objects::IndexHandler<C> const *;
  inline ConstraintDefinition();
};

enum struct ExpressionRTTI {

  Constant,
  Variable,
  LinTerm,
  QuadTerm,
  LinExpression,
  QuadExpression,
  Formula,
  Sum
};
inline bool operator==(ExpressionRTTI a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(ExpressionRTTI a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, ExpressionRTTI b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, ExpressionRTTI b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(ExpressionRTTI e) {
  switch (e) {
  case ExpressionRTTI::Constant:
    return "Constant";
  case ExpressionRTTI::Variable:
    return "Variable";
  case ExpressionRTTI::LinTerm:
    return "LinTerm";
  case ExpressionRTTI::QuadTerm:
    return "QuadTerm";
  case ExpressionRTTI::LinExpression:
    return "LinExpression";
  case ExpressionRTTI::QuadExpression:
    return "QuadExpression";
  case ExpressionRTTI::Formula:
    return "Formula";
  case ExpressionRTTI::Sum:
    return "Sum";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, ExpressionRTTI e) {
  os << to_string(e);
  return os;
}

class IIS final {

public:
  enum struct Isolation {

    NotAvailable,
    NotIsolated,
    Isolated
  };

private:
  std::vector<IISConstraint> constraints;

public:
  inline auto getConstraints() const -> std::vector<IISConstraint>;

private:
  std::vector<IISVariable> variables;

public:
  inline auto getVariables() const -> std::vector<IISVariable>;

private:
  template <typename T> static inline auto get_begin(T const &t) -> auto;

  template <typename T> static inline auto get_end(T const &t) -> auto;

public:
  inline IIS(xpress::SizedArray<IISConstraint const> const &constraints,
             xpress::SizedArray<IISVariable const> const &variables);
};
inline bool operator==(xpress::objects::IIS::Isolation a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(xpress::objects::IIS::Isolation a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, xpress::objects::IIS::Isolation b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, xpress::objects::IIS::Isolation b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(xpress::objects::IIS::Isolation e) {
  switch (e) {
  case xpress::objects::IIS::Isolation::NotAvailable:
    return "NotAvailable";
  case xpress::objects::IIS::Isolation::NotIsolated:
    return "NotIsolated";
  case xpress::objects::IIS::Isolation::Isolated:
    return "Isolated";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os,
                                xpress::objects::IIS::Isolation e) {
  os << to_string(e);
  return os;
}

class Index {
public:
  virtual auto getName() const -> std::string = 0;

  virtual auto getIndex() const -> int = 0;

  virtual auto updateIndex(int delta) -> void = 0;

  virtual auto getIndexForProb(XpressProblem const *xprob) const -> int = 0;

  virtual auto getProblem() const -> XpressProblem * = 0;

  virtual auto getSerial() const -> XPRSint64 = 0;

  virtual auto markDeleted() -> void = 0;
  virtual ~Index() {}
};

template <typename T> class IndexHandler {
public:
  int const attribute;

private:
  long mutable serial;

  std::vector<T> mutable objects;

  std::recursive_mutex mutable synchronizedMutex;

public:
  inline IndexHandler(int attribute);

protected:
  virtual auto makeInstance(XpressProblem *prob, long serial, int index) const
      -> T = 0;

  virtual auto delIndices(XpressProblem *prob, int count,
                          xpress::Array<int const> const &indices) -> void = 0;

public:
  inline auto objectForIndex(XpressProblem *prob, int index) const -> T;

  inline auto fillObjects(XpressProblem *prob, int first, int last,
                          std::vector<T> &objs) const -> void;

  virtual auto objectsForIndices(XpressProblem *prob, int first, int last) const
      -> std::vector<T> = 0;

  inline auto deleteIndices(XpressProblem *prob, int count,
                            xpress::Array<int const> const &indices) -> void;

  template <typename O = xpress::objects::Index>
  inline auto deleteObjects(XpressProblem *prob,
                            xpress::SizedArray<O const> const &objs) -> void;

  inline auto drop() -> void;
  virtual ~IndexHandler() {}
};

class LinQuadUtils final {
public:
  static inline auto index4Var(xpress::objects::XpressProblem *prob,
                               xpress::objects::Variable v) -> int;

  static inline auto index4Row(xpress::objects::XpressProblem *prob,
                               xpress::objects::Inequality r) -> int;

  static inline auto unifyConstantKey(xpress::objects::Variable v)
      -> xpress::objects::Variable;

  static inline auto isVariablePairOrdered(xpress::objects::Variable v1,
                                           xpress::objects::Variable v2)
      -> bool;

  static inline auto isLinear(Variable v1, Variable v2) -> bool;

  static inline auto isConstant(Variable x) -> bool;

  static inline auto isConstant(Variable v1, Variable v2) -> bool;

  static inline auto getQConstantKey() -> xpress::objects::QPair const &;
};

class LinearView : public std::enable_shared_from_this<LinearView> {
public:
  typedef LinCoef value_type;
  typedef xpress::WrappedIterator<LinCoef> iterator;

  virtual auto begin() const
      -> xpress::WrappedIterator<xpress::objects::LinCoef> = 0;

  virtual auto end() const
      -> xpress::WrappedIterator<xpress::objects::LinCoef> = 0;
  virtual ~LinearView() {}
};

class NotInProblemException final : public xpress::XPRSException {
private:
  int index;

public:
  auto getIndex() const -> int { return index; }

  void setIndex(int newIndex) { index = newIndex; }

  inline NotInProblemException(std::string message);

  inline NotInProblemException(std::string message, int index);
};

class QuadView : public std::enable_shared_from_this<QuadView> {
public:
  typedef QuadCoef value_type;
  typedef xpress::WrappedIterator<QuadCoef> iterator;

  virtual auto begin() const
      -> xpress::WrappedIterator<xpress::objects::QuadCoef> = 0;

  virtual auto end() const
      -> xpress::WrappedIterator<xpress::objects::QuadCoef> = 0;
  virtual ~QuadView() {}
};

namespace utils {

template <typename A = Expression, typename I, xpress_isintegral(I),
          typename Func0, xpress_isinvocable(Func0, A, Func0, I)>
inline auto sum(I const &count, Func0 makeTerm)
    -> xpress::objects::SumExpression;

template <typename Strm0, typename A = Expression,
          typename T = typename xpress_elementtype(Strm0), typename Func0,
          xpress_isstream(Strm0),
          xpress_isinvocable(Func0, A, Func0, T const &)>
inline auto sum(Strm0 const &data, Func0 makeTerm)
    -> xpress::objects::SumExpression;

template <typename Strm0, typename T = typename xpress_elementtype(Strm0),
          xpress_isstream(Strm0)>
inline auto sum(Strm0 const &terms) -> xpress::objects::SumExpression;

template <typename... MORETERMS,
          typename RestrictPack = typename std::enable_if<
              xpress::AllConvertible<Expression, MORETERMS...>::value>::type>
inline auto sum(Expression firstTerm, MORETERMS... moreTerms)
    -> xpress::objects::SumExpression;

inline auto uminus(Expression value) -> xpress::objects::Expression;

inline auto plus(Expression augend, Expression addend)
    -> xpress::objects::Expression;

inline auto plus(double augend, Expression addend)
    -> xpress::objects::Expression;

inline auto plus(Expression augend, double addend)
    -> xpress::objects::Expression;

inline auto minus(Expression minuend, Expression subtrahend)
    -> xpress::objects::Expression;

inline auto minus(double minuend, Expression subtrahend)
    -> xpress::objects::Expression;

inline auto minus(Expression minuend, double subtrahend)
    -> xpress::objects::Expression;

inline auto div(Expression dividend, Expression divisor)
    -> xpress::objects::Expression;

inline auto div(double dividend, Expression divisor)
    -> xpress::objects::Expression;

inline auto div(Expression dividend, double divisor)
    -> xpress::objects::Expression;

inline auto mul(Expression factor1, Expression factor2)
    -> xpress::objects::Expression;

inline auto mul(double factor1, Expression factor2)
    -> xpress::objects::Expression;

inline auto mul(Expression factor1, double factor2)
    -> xpress::objects::Expression;

inline auto pow(Expression basis, Expression exp)
    -> xpress::objects::BinaryExpression;

inline auto pow(double basis, Expression exp)
    -> xpress::objects::BinaryExpression;

inline auto pow(Expression basis, double exp)
    -> xpress::objects::BinaryExpression;

inline auto log10(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto ln(Expression value) -> xpress::objects::InternalFunctionExpression;

inline auto exp(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto abs(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto sqrt(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto sin(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto cos(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto tan(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto arcsin(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto arccos(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto arctan(Expression value)
    -> xpress::objects::InternalFunctionExpression;

inline auto sign(Expression value)
    -> xpress::objects::InternalFunctionExpression;

template <typename... VALUE,
          typename RestrictPack = typename std::enable_if<
              xpress::AllConvertible<Expression, VALUE...>::value>::type>
inline auto min(VALUE... value) -> xpress::objects::InternalFunctionExpression;

inline auto min(xpress::SizedArray<Expression const> const &value)
    -> xpress::objects::InternalFunctionExpression;

template <typename... VALUE,
          typename RestrictPack = typename std::enable_if<
              xpress::AllConvertible<Expression, VALUE...>::value>::type>
inline auto max(VALUE... value) -> xpress::objects::InternalFunctionExpression;

inline auto max(xpress::SizedArray<Expression const> const &value)
    -> xpress::objects::InternalFunctionExpression;

template <typename... VALUE,
          typename RestrictPack = typename std::enable_if<
              xpress::AllConvertible<Expression, VALUE...>::value>::type>
inline auto prod(VALUE... value) -> xpress::objects::Expression;

inline auto prod(xpress::SizedArray<Expression const> const &value)
    -> xpress::objects::Expression;

template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
          xpress_isstream(Strm1)>
inline auto scalarProduct(Strm0 const &variables, Strm1 const &coefs)
    -> xpress::objects::LinExpression;

template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
          xpress_isstream(Strm1)>
inline auto scalarProduct(LinExpression result, Strm0 const &variables,
                          Strm1 const &coefs) -> xpress::objects::LinExpression;

template <typename K>
inline auto scalarProduct(std::unordered_map<K, Variable> variables,
                          std::unordered_map<K, double> coefs)
    -> xpress::objects::LinExpression;

template <typename K>
inline auto scalarProduct(LinExpression result,
                          std::unordered_map<K, Variable> variables,
                          std::unordered_map<K, double> coefs)
    -> xpress::objects::LinExpression;

template <typename Strm0, typename Strm1, typename Strm2,
          xpress_isstream(Strm0), xpress_isstream(Strm1),
          xpress_isstream(Strm2)>
inline auto scalarProduct(Strm0 const &variables1, Strm1 const &variables2,
                          Strm2 const &coefs)
    -> xpress::objects::QuadExpression;

template <typename Strm0, typename Strm1, typename Strm2,
          xpress_isstream(Strm0), xpress_isstream(Strm1),
          xpress_isstream(Strm2)>
inline auto scalarProduct(QuadExpression result, Strm0 const &variables1,
                          Strm1 const &variables2, Strm2 const &coefs)
    -> xpress::objects::QuadExpression;

template <typename A = Expression, typename C1, typename C2,
          xpress_isintegral(C1), xpress_isintegral(C2), typename Func0,
          xpress_isinvocable(Func0, A, Func0, C1, C2)>
inline auto sum(C1 count1, C2 count2, Func0 makeTerm)
    -> xpress::objects::SumExpression;

template <typename Iter0, typename Iter1, typename A = Expression,
          typename T1 = typename xpress_elementtype(Iter0),
          typename T2 = typename xpress_elementtype(Iter1), typename Func0,
          xpress_isiterable(Iter0), xpress_isiterable(Iter1),
          xpress_isinvocable(Func0, A, Func0, T1, T2)>
inline auto sum(Iter0 const &data1, Iter1 const &data2, Func0 makeTerm)
    -> xpress::objects::SumExpression;

template <typename A = Expression, typename C1, typename C2, typename C3,
          xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
          typename Func0, xpress_isinvocable(Func0, A, Func0, C1, C2, C3)>
inline auto sum(C1 count1, C2 count2, C3 count3, Func0 makeTerm)
    -> xpress::objects::SumExpression;

template <
    typename Iter0, typename Iter1, typename Iter2, typename A = Expression,
    typename T1 = typename xpress_elementtype(Iter0),
    typename T2 = typename xpress_elementtype(Iter1),
    typename T3 = typename xpress_elementtype(Iter2), typename Func0,
    xpress_isiterable(Iter0), xpress_isiterable(Iter1),
    xpress_isiterable(Iter2), xpress_isinvocable(Func0, A, Func0, T1, T2, T3)>
inline auto sum(Iter0 const &data1, Iter1 const &data2, Iter2 const &data3,
                Func0 makeTerm) -> xpress::objects::SumExpression;

template <typename A = Expression, typename C1, typename C2, typename C3,
          typename C4, xpress_isintegral(C1), xpress_isintegral(C2),
          xpress_isintegral(C3), xpress_isintegral(C4), typename Func0,
          xpress_isinvocable(Func0, A, Func0, C1, C2, C3, C4)>
inline auto sum(C1 count1, C2 count2, C3 count3, C4 count4, Func0 makeTerm)
    -> xpress::objects::SumExpression;

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename A = Expression,
          typename T1 = typename xpress_elementtype(Iter0),
          typename T2 = typename xpress_elementtype(Iter1),
          typename T3 = typename xpress_elementtype(Iter2),
          typename T4 = typename xpress_elementtype(Iter3), typename Func0,
          xpress_isiterable(Iter0), xpress_isiterable(Iter1),
          xpress_isiterable(Iter2), xpress_isiterable(Iter3),
          xpress_isinvocable(Func0, A, Func0, T1, T2, T3, T4)>
inline auto sum(Iter0 const &data1, Iter1 const &data2, Iter2 const &data3,
                Iter3 const &data4, Func0 makeTerm)
    -> xpress::objects::SumExpression;

template <typename A = Expression, typename C1, typename C2, typename C3,
          typename C4, typename C5, xpress_isintegral(C1),
          xpress_isintegral(C2), xpress_isintegral(C3), xpress_isintegral(C4),
          xpress_isintegral(C5), typename Func0,
          xpress_isinvocable(Func0, A, Func0, C1, C2, C3, C4, C5)>
inline auto sum(C1 count1, C2 count2, C3 count3, C4 count4, C5 count5,
                Func0 makeTerm) -> xpress::objects::SumExpression;

template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename A = Expression,
          typename T1 = typename xpress_elementtype(Iter0),
          typename T2 = typename xpress_elementtype(Iter1),
          typename T3 = typename xpress_elementtype(Iter2),
          typename T4 = typename xpress_elementtype(Iter3),
          typename T5 = typename xpress_elementtype(Iter4), typename Func0,
          xpress_isiterable(Iter0), xpress_isiterable(Iter1),
          xpress_isiterable(Iter2), xpress_isiterable(Iter3),
          xpress_isiterable(Iter4),
          xpress_isinvocable(Func0, A, Func0, T1, T2, T3, T4, T5)>
inline auto sum(Iter0 const &data1, Iter1 const &data2, Iter2 const &data3,
                Iter3 const &data4, Iter4 const &data5, Func0 makeTerm)
    -> xpress::objects::SumExpression;
}; // namespace utils
class ExpressionImplementation;

class Expression {
public:
  typedef ExpressionImplementation impl_type;

protected:
  std::shared_ptr<ExpressionImplementation> implPointer = nullptr;

public:
  bool isSameImpl(Expression const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(ExpressionImplementation const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline Expression(std::shared_ptr<ExpressionImplementation> init);

public:
  inline Expression(ExpressionImplementation const *init);
  inline virtual ~Expression(){};

private:
  inline ExpressionImplementation *impl();
  inline ExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(Expression const &other) const { return impl() == other.impl(); }
  virtual inline auto leq(double rhs) const
      -> xpress::objects::InequalityDefinition;

  virtual inline auto leq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition;

  virtual inline auto geq(double rhs) const
      -> xpress::objects::InequalityDefinition;

  virtual inline auto geq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition;

  virtual inline auto eq(double rhs) const
      -> xpress::objects::InequalityDefinition;

  virtual inline auto eq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition;

  virtual inline auto in(double lb, double ub)
      -> xpress::objects::InequalityDefinition;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool;

  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const -> double;

  virtual inline auto getRTTI() -> xpress::objects::ExpressionRTTI;

  virtual inline auto getMaxDegree() const -> int;

  virtual inline auto getConstantView() const -> double;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView>;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView>;

  virtual inline auto toString() const -> std::string;

  virtual inline auto uminus() const -> xpress::objects::Expression;

  virtual inline auto plus(xpress::objects::Expression arg) const
      -> xpress::objects::Expression;

  virtual inline auto plus(double arg) const -> xpress::objects::Expression;

  virtual inline auto minus(xpress::objects::Expression arg) const
      -> xpress::objects::Expression;

  virtual inline auto minus(double arg) const -> xpress::objects::Expression;

  virtual inline auto mul(Expression arg) const -> xpress::objects::Expression;

  virtual inline auto mul(double arg) const -> xpress::objects::Expression;

  virtual inline auto div(Expression arg) const -> xpress::objects::Expression;

  virtual inline auto div(double arg) const -> xpress::objects::Expression;
  inline Expression();
};
inline xpress::objects::InequalityDefinition operator<=(Expression const &lhs,
                                                        double rhs);
inline xpress::objects::InequalityDefinition operator>=(Expression const &lhs,
                                                        double rhs);
inline xpress::objects::InequalityDefinition operator<=(double lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator>=(double lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator<=(Expression const &lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator>=(Expression const &lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator==(Expression const &lhs,
                                                        double rhs);
inline xpress::objects::InequalityDefinition operator!=(Expression const &,
                                                        double const &);
inline xpress::objects::InequalityDefinition operator==(double lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator!=(double,
                                                        Expression const &);
inline xpress::objects::InequalityDefinition operator==(Expression const &lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator!=(Expression const &,
                                                        Expression const &);
inline xpress::objects::Expression operator-(Expression const &expr);
inline xpress::objects::Expression operator+(Expression const &expr,
                                             Expression const &e);
inline xpress::objects::Expression operator+(Expression const &expr, double d);
inline xpress::objects::Expression operator+(double d, Expression const &expr);
inline xpress::objects::Expression operator-(Expression const &expr,
                                             Expression const &e);
inline xpress::objects::Expression operator-(Expression const &expr, double d);
inline xpress::objects::Expression operator-(double d, Expression const &expr);
inline xpress::objects::Expression operator*(Expression const &expr,
                                             Expression const &e);
inline xpress::objects::Expression operator*(Expression const &expr, double d);
inline xpress::objects::Expression operator*(double d, Expression const &expr);
inline xpress::objects::Expression operator/(Expression const &expr,
                                             Expression const &e);
inline xpress::objects::Expression operator/(Expression const &expr, double d);
inline xpress::objects::Expression operator/(double d, Expression const &expr);
class InequalityImplementation;

class Inequality final : public xpress::objects::Index {
public:
  typedef InequalityImplementation impl_type;

protected:
  std::shared_ptr<InequalityImplementation> implPointer = nullptr;

public:
  bool isSameImpl(Inequality const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(InequalityImplementation const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline Inequality(std::shared_ptr<InequalityImplementation> init);

public:
  inline Inequality(InequalityImplementation const *init);
  inline virtual ~Inequality(){};

private:
  inline InequalityImplementation *impl();
  inline InequalityImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(Inequality const &other) const { return impl() == other.impl(); }
  inline auto getIndex() const -> int override;

  inline auto updateIndex(int delta) -> void override;

  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;

  inline auto getProblem() const -> xpress::objects::XpressProblem * override;

  inline auto getSerial() const -> XPRSint64 override;

  inline auto markDeleted() -> void override;

  virtual inline auto compareTo(Inequality const &other) const -> int final;

  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline Inequality(XpressProblem *prob, long serial, int index);

  inline auto getRhs() -> double;

  inline auto getRhsRange() -> double;

  inline auto getName() const -> std::string override;

  inline auto getType() -> xpress::RowType;

  inline auto setRhs(double newRhs) -> Inequality &;

  inline auto setRhsRange(double newRange) -> Inequality &;

  inline auto setType(xpress::RowType newType) -> Inequality &;

  inline auto setName(std::optional<std::string> newName) -> Inequality &;

  inline auto getValue(xpress::SizedArray<double const> const &data) const
      -> double;

  inline auto getSlack() const -> double;

  inline auto getDual() const -> double;

  inline auto getCallbackSlack() -> double;

  inline auto getCallbackDual() -> double;

  inline auto chgCoef(Variable v, double newCoef) -> Inequality &;

  inline auto chgCoefs(xpress::SizedArray<Variable const> const &v,
                       xpress::SizedArray<double const> const &newCoef)
      -> Inequality &;

  inline auto getLhs() -> xpress::objects::Expression;
  inline Inequality();
};
class AbstractExpressionImplementation;

class AbstractExpression : public xpress::objects::Expression {
public:
  typedef AbstractExpressionImplementation impl_type;

protected:
  inline AbstractExpression(
      std::shared_ptr<AbstractExpressionImplementation> init);

public:
  inline AbstractExpression(AbstractExpressionImplementation const *init);
  inline virtual ~AbstractExpression(){};

private:
  inline AbstractExpressionImplementation *impl();
  inline AbstractExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(AbstractExpression const &other) const {
    return impl() == other.impl();
  }
  virtual inline auto getConstantView() const -> double override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;

  virtual inline auto leq(double rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto leq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto geq(double rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto geq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto eq(double rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto eq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto in(double lb, double ub)
      -> xpress::objects::InequalityDefinition override;
  inline AbstractExpression();
};
class InequalityDefinitionImplementation;

class InequalityDefinition final : public ConstraintDefinition<Inequality> {
public:
  typedef InequalityDefinitionImplementation impl_type;

protected:
  inline InequalityDefinition(
      std::shared_ptr<InequalityDefinitionImplementation> init);

public:
  inline InequalityDefinition(InequalityDefinitionImplementation const *init);
  inline virtual ~InequalityDefinition(){};

private:
  inline InequalityDefinitionImplementation *impl();
  inline InequalityDefinitionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(InequalityDefinition const &other) const {
    return impl() == other.impl();
  }
  inline auto getLhs() const -> xpress::objects::Expression;

  inline auto getType() const -> xpress::RowType;

  inline auto getRhs() const -> xpress::objects::Expression;

  inline auto getRange() const -> std::vector<double>;

  inline InequalityDefinition(xpress::objects::Expression lhs,
                              xpress::RowType type,
                              xpress::objects::Expression rhs,
                              std::optional<std::string> name);

  inline InequalityDefinition(xpress::objects::Expression lhs,
                              xpress::RowType type, double rhs,
                              std::optional<std::string> name);

  inline InequalityDefinition(double lhs, xpress::RowType type,
                              xpress::objects::Expression rhs,
                              std::optional<std::string> name);

  inline InequalityDefinition(Expression expr, double lb, double ub,
                              std::optional<std::string> name);

  inline auto createRow(xpress::XPRSProblem::RowCreator *creator) const -> void;

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  inline auto setName(std::optional<std::string> newName)
      -> InequalityDefinition &;

  virtual inline auto getIndexHandler(XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::Inequality> const
          * override;
  inline InequalityDefinition();
};
class VariableImplementation;

class Variable final : public xpress::objects::Expression,
                       public xpress::objects::Index {
public:
  typedef VariableImplementation impl_type;

protected:
  inline Variable(std::shared_ptr<VariableImplementation> init);

public:
  inline Variable(VariableImplementation const *init);
  inline virtual ~Variable(){};

private:
  inline VariableImplementation *impl();
  inline VariableImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(Variable const &other) const { return impl() == other.impl(); }
  friend class xpress::GlobalConstants<void>;
  inline auto getIndex() const -> int override;

  inline auto updateIndex(int delta) -> void override;

  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;

  inline auto getProblem() const -> xpress::objects::XpressProblem * override;

  inline auto getSerial() const -> XPRSint64 override;

  inline auto markDeleted() -> void override;

  virtual inline auto compareTo(Variable const &other) const -> int final;

  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline Variable(XpressProblem *prob, long serial, int index);

  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;

  virtual inline auto leq(double rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto leq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto geq(double rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto geq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto eq(double rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto eq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;

  virtual inline auto in(double lb, double ub)
      -> xpress::objects::InequalityDefinition override;

  inline auto square() -> xpress::objects::QuadTerm;

  inline auto getLB() const -> double;

  inline auto getUB() const -> double;

  inline auto getName() const -> std::string override;

  inline auto getType() const -> xpress::ColumnType;

  inline auto fix(double value) -> Variable &;

  inline auto setLB(double newLB) -> Variable &;

  inline auto setUB(double newUB) -> Variable &;

  inline auto setType(xpress::ColumnType newType) -> Variable &;

  inline auto setLimit(double newLimit) -> Variable &;

  inline auto setName(std::optional<std::string> newName) -> Variable &;

  inline auto getValue(xpress::SizedArray<double const> const &data) const
      -> double;

  inline auto getSolution() const -> double;

  inline auto getRedCost() const -> double;

  inline auto getCallbackSolution() -> double;

  inline auto getCallbackRedCost() -> double;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  inline auto chgCoef(Inequality r, double newCoef) -> void;

  inline auto chgCoefs(xpress::SizedArray<Inequality const> const &r,
                       xpress::SizedArray<double const> const &newCoef) -> void;

  inline auto chgObj(double newCoef) -> Variable &;

  inline auto getObj() const -> double;

  inline auto chgObjN(int objidx, double newCoef) -> Variable &;

  inline auto getObjN(int objIdx) const -> double;

  inline auto pwlOf(Variable input,
                    xpress::SizedArray<double const> const &xval,
                    xpress::SizedArray<double const> const &yval) const
      -> xpress::objects::PWLDefinition;

  inline auto pwlOf(Variable input,
                    xpress::SizedArray<double const> const &xval,
                    xpress::SizedArray<double const> const &yval,
                    std::optional<std::string> name) const
      -> xpress::objects::PWLDefinition;

  template <
      typename... BREAKPOINTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::PwlBreakpoint, BREAKPOINTS...>::value>::type>
  inline auto pwlOf(Variable input, BREAKPOINTS... breakpoints) const
      -> xpress::objects::PWLDefinition;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto pwlOf(Variable input, Strm0 const &breakpoints) const
      -> xpress::objects::PWLDefinition;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto pwlOf(Variable input, Strm0 const &breakpoints,
                    std::optional<std::string> name) const
      -> xpress::objects::PWLDefinition;

  template <typename... VALUES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<double, VALUES...>::value>::type>
  inline auto maxOf(xpress::SizedArray<Variable const> const &variables,
                    VALUES... values) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto maxOf(Strm0 const &variables, Strm1 const &values,
                    std::optional<std::string> name) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto maxOf(Strm0 const &variables, Strm1 const &values) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline auto maxOf(VARIABLES... variables) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VALUES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<double, VALUES...>::value>::type>
  inline auto minOf(xpress::SizedArray<Variable const> const &variables,
                    VALUES... values) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto minOf(Strm0 const &variables, Strm1 const &values,
                    std::optional<std::string> name) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto minOf(Strm0 const &variables, Strm1 const &values) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline auto minOf(VARIABLES... variables) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto orOf(Strm0 const &variables,
                   std::optional<std::string> name) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto orOf(Strm0 const &variables) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline auto orOf(VARIABLES... variables) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto andOf(Strm0 const &variables,
                    std::optional<std::string> name) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto andOf(Strm0 const &variables) const
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline auto andOf(VARIABLES... variables) const
      -> xpress::objects::GeneralConstraintDefinition;

  inline auto absOf(Variable x) const
      -> xpress::objects::GeneralConstraintDefinition;

  inline auto ifThen(InequalityDefinition implied) const
      -> xpress::objects::IndicatorDescription;

  inline auto ifNotThen(InequalityDefinition implied) const
      -> xpress::objects::IndicatorDescription;

  virtual inline auto toString() const -> std::string override;

  virtual inline auto getMaxDegree() const -> int override;

  virtual inline auto getConstantView() const -> double override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;

  virtual inline auto uminus() const -> xpress::objects::Expression override;

  virtual inline auto mul(Expression arg) const
      -> xpress::objects::Expression override;

  inline auto mul(ConstantExpression arg) const -> xpress::objects::LinTerm;

  virtual inline auto mul(double arg) const
      -> xpress::objects::Expression override;

  inline auto mul(Variable arg) const -> xpress::objects::QuadTerm;

  inline auto mul(LinTerm arg) const -> xpress::objects::QuadTerm;

  virtual inline auto div(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;

  inline auto div(ConstantExpression arg) const -> xpress::objects::LinTerm;

  virtual inline auto div(double arg) const
      -> xpress::objects::Expression override;
  inline Variable();
};
inline xpress::objects::InequalityDefinition operator<=(Variable const &lhs,
                                                        double rhs);
inline xpress::objects::InequalityDefinition operator>=(Variable const &lhs,
                                                        double rhs);
inline xpress::objects::InequalityDefinition operator<=(double lhs,
                                                        Variable const &rhs);
inline xpress::objects::InequalityDefinition operator>=(double lhs,
                                                        Variable const &rhs);
inline xpress::objects::InequalityDefinition operator<=(Variable const &lhs,
                                                        Variable const &rhs);
inline xpress::objects::InequalityDefinition operator>=(Variable const &lhs,
                                                        Variable const &rhs);
inline xpress::objects::InequalityDefinition operator<=(Variable const &lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator>=(Variable const &lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator==(Variable const &lhs,
                                                        double rhs);
inline xpress::objects::InequalityDefinition operator!=(Variable, double);
inline xpress::objects::InequalityDefinition operator==(double lhs,
                                                        Variable const &rhs);
inline xpress::objects::InequalityDefinition operator!=(double,
                                                        Variable const &);
inline xpress::objects::InequalityDefinition operator==(Variable const &lhs,
                                                        Variable const &rhs);
inline xpress::objects::InequalityDefinition operator!=(Variable const &,
                                                        Variable const &);
inline xpress::objects::InequalityDefinition operator==(Variable const &lhs,
                                                        Expression const &rhs);
inline xpress::objects::InequalityDefinition operator!=(Variable const &,
                                                        Expression const &);
inline xpress::objects::LinTerm operator-(Variable v);
inline xpress::objects::LinTerm operator*(Variable const &v, double d);
inline xpress::objects::LinTerm operator*(double d, Variable const &v);
inline xpress::objects::LinTerm operator*(Variable const &v,
                                          ConstantExpression const &d);
inline xpress::objects::QuadTerm operator*(Variable const &v,
                                           Variable const &v2);
inline xpress::objects::QuadTerm operator*(Variable const &v, LinTerm const &t);
inline xpress::objects::LinTerm operator/(Variable const &v, double d);
inline xpress::objects::LinTerm operator/(Variable const &v,
                                          ConstantExpression const &d);
class ConstantExpressionImplementation;

class ConstantExpression final : public AbstractExpression {
public:
  typedef ConstantExpressionImplementation impl_type;

protected:
  inline ConstantExpression(
      std::shared_ptr<ConstantExpressionImplementation> init);

public:
  inline ConstantExpression(ConstantExpressionImplementation const *init);
  inline virtual ~ConstantExpression(){};

private:
  inline ConstantExpressionImplementation *impl();
  inline ConstantExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(ConstantExpression const &other) const {
    return impl() == other.impl();
  }
  inline auto getValue() const -> double;

  inline ConstantExpression(double value);

  static inline auto constant(double cnst)
      -> xpress::objects::ConstantExpression;

  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto toString() const -> std::string override;

  virtual inline auto getMaxDegree() const -> int override;

  virtual inline auto getConstantView() const -> double override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;

  virtual inline auto uminus() const -> xpress::objects::Expression override;

  virtual inline auto plus(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;

  inline auto plus(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::ConstantExpression;

  virtual inline auto plus(double arg) const
      -> xpress::objects::Expression override;

  virtual inline auto minus(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;

  inline auto minus(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::ConstantExpression;

  virtual inline auto minus(double arg) const
      -> xpress::objects::Expression override;

  virtual inline auto mul(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;

  inline auto mul(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::ConstantExpression;

  virtual inline auto mul(double arg) const
      -> xpress::objects::Expression override;

  inline auto mul(xpress::objects::Variable arg) const
      -> xpress::objects::LinTerm;

  inline auto mul(xpress::objects::LinTerm arg) const
      -> xpress::objects::LinTerm;

  inline auto mul(xpress::objects::QuadTerm arg) const
      -> xpress::objects::QuadTerm;

  virtual inline auto div(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;

  inline auto div(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::ConstantExpression;

  virtual inline auto div(double arg) const
      -> xpress::objects::Expression override;
  inline ConstantExpression();
};
inline xpress::objects::ConstantExpression
operator-(ConstantExpression const &c);
inline xpress::objects::ConstantExpression
operator+(ConstantExpression const &v, double d);
inline xpress::objects::ConstantExpression
operator+(double d, ConstantExpression const &v);
inline xpress::objects::ConstantExpression
operator+(ConstantExpression const &v, ConstantExpression const &d);
inline xpress::objects::ConstantExpression
operator-(ConstantExpression const &v, double d);
inline xpress::objects::ConstantExpression
operator-(double d, ConstantExpression const &v);
inline xpress::objects::ConstantExpression
operator-(ConstantExpression const &v, ConstantExpression const &d);
inline xpress::objects::ConstantExpression
operator*(ConstantExpression const &v, double d);
inline xpress::objects::ConstantExpression
operator*(double d, ConstantExpression const &v);
inline xpress::objects::ConstantExpression
operator*(ConstantExpression const &v, ConstantExpression const &d);
inline xpress::objects::LinTerm operator*(ConstantExpression const &c,
                                          Variable const &v);
inline xpress::objects::LinTerm operator*(ConstantExpression const &v,
                                          LinTerm const &t);
inline xpress::objects::QuadTerm operator*(ConstantExpression const &v,
                                           QuadTerm const &t);
inline xpress::objects::ConstantExpression
operator/(ConstantExpression const &v, double d);
inline xpress::objects::ConstantExpression
operator/(double d, ConstantExpression const &v);
inline xpress::objects::ConstantExpression
operator/(ConstantExpression const &v, ConstantExpression const &d);
class GeneralConstraintImplementation;

class GeneralConstraint final : public xpress::objects::Index {
public:
  typedef GeneralConstraintImplementation impl_type;

protected:
  std::shared_ptr<GeneralConstraintImplementation> implPointer = nullptr;

public:
  bool isSameImpl(GeneralConstraint const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(GeneralConstraintImplementation const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline GeneralConstraint(
      std::shared_ptr<GeneralConstraintImplementation> init);

public:
  inline GeneralConstraint(GeneralConstraintImplementation const *init);
  inline virtual ~GeneralConstraint(){};

private:
  inline GeneralConstraintImplementation *impl();
  inline GeneralConstraintImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(GeneralConstraint const &other) const {
    return impl() == other.impl();
  }
  inline auto getIndex() const -> int override;

  inline auto updateIndex(int delta) -> void override;

  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;

  inline auto getProblem() const -> xpress::objects::XpressProblem * override;

  inline auto getSerial() const -> XPRSint64 override;

  inline auto markDeleted() -> void override;

  virtual inline auto compareTo(GeneralConstraint const &other) const
      -> int final;

  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline GeneralConstraint(XpressProblem *prob, long serial, int index);

  inline auto getType() -> xpress::GenConsType;

  inline auto getResultant() -> xpress::objects::Variable;

  inline auto getVariables() -> std::vector<xpress::objects::Variable>;

  inline auto getValues() -> std::vector<double>;

  inline auto getName() const -> std::string override;

  inline auto setName(std::optional<std::string> newName)
      -> GeneralConstraint &;

  static inline auto Abs(Variable resultant, Variable variable)
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  static inline auto And(Variable resultant, VARIABLES... variables)
      -> xpress::objects::GeneralConstraintDefinition;

  static inline auto And(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables)
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  static inline auto Or(Variable resultant, VARIABLES... variables)
      -> xpress::objects::GeneralConstraintDefinition;

  static inline auto Or(Variable resultant,
                        xpress::SizedArray<Variable const> const &variables)
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  static inline auto Max(Variable resultant, VARIABLES... variables)
      -> xpress::objects::GeneralConstraintDefinition;

  static inline auto Max(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables)
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VALUES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<double, VALUES...>::value>::type>
  static inline auto Max(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables,
                         VALUES... values)
      -> xpress::objects::GeneralConstraintDefinition;

  static inline auto Max(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables,
                         xpress::SizedArray<double const> const &values)
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  static inline auto Min(Variable resultant, VARIABLES... variables)
      -> xpress::objects::GeneralConstraintDefinition;

  static inline auto Min(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables)
      -> xpress::objects::GeneralConstraintDefinition;

  template <typename... VALUES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<double, VALUES...>::value>::type>
  static inline auto Min(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables,
                         VALUES... values)
      -> xpress::objects::GeneralConstraintDefinition;

  static inline auto Min(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables,
                         xpress::SizedArray<double const> const &values)
      -> xpress::objects::GeneralConstraintDefinition;
  inline GeneralConstraint();
};
class GeneralConstraintDefinitionImplementation;

class GeneralConstraintDefinition final
    : public ConstraintDefinition<GeneralConstraint> {
public:
  typedef GeneralConstraintDefinitionImplementation impl_type;

protected:
  inline GeneralConstraintDefinition(
      std::shared_ptr<GeneralConstraintDefinitionImplementation> init);

public:
  inline GeneralConstraintDefinition(
      GeneralConstraintDefinitionImplementation const *init);
  inline virtual ~GeneralConstraintDefinition(){};

private:
  inline GeneralConstraintDefinitionImplementation *impl();
  inline GeneralConstraintDefinitionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(GeneralConstraintDefinition const &other) const {
    return impl() == other.impl();
  }
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline GeneralConstraintDefinition(Variable resultant,
                                     xpress::GenConsType type,
                                     Strm0 const &variables, Strm1 const &value,
                                     std::optional<std::string> name);

  template <typename Strm0, xpress_isstream(Strm0)>
  inline GeneralConstraintDefinition(Variable resultant,
                                     xpress::GenConsType type,
                                     Strm0 const &variables,
                                     std::optional<std::string> name);

  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline GeneralConstraintDefinition(Variable resultant,
                                     xpress::GenConsType type,
                                     VARIABLES... variables);

  inline GeneralConstraintDefinition(
      Variable resultant, xpress::GenConsType type,
      xpress::SizedArray<Variable const> const &variables);

  inline GeneralConstraintDefinition(
      Variable resultant, xpress::GenConsType type,
      xpress::SizedArray<Variable const> const &variables,
      xpress::SizedArray<double const> const &values);

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  inline auto setName(std::optional<std::string> newName)
      -> GeneralConstraintDefinition &;

  virtual inline auto
  getIndexHandler(xpress::objects::XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::GeneralConstraint> const
          * override;
  inline GeneralConstraintDefinition();
};

class IISVariable final {

public:
  enum struct InfeasibleDomain {

    UpperBound,
    LowerBound,
    Fixing,
    Binary,
    Integral,
    PartialInteger,
    SemiContinuous,
    SemiInteger
  };

private:
  InfeasibleDomain domain;

public:
  auto getDomain() const -> InfeasibleDomain { return domain; }

private:
  Variable variable;

public:
  auto getVariable() const -> Variable { return variable; }

private:
  double dj;

public:
  auto getDj() const -> double { return dj; }

private:
  IIS::Isolation isolation;

public:
  auto getIsolation() const -> IIS::Isolation { return isolation; }

  inline IISVariable(InfeasibleDomain domain, Variable variable, double dj,
                     IIS::Isolation isolation);
};
inline bool operator==(xpress::objects::IISVariable::InfeasibleDomain a,
                       int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(xpress::objects::IISVariable::InfeasibleDomain a,
                       int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a,
                       xpress::objects::IISVariable::InfeasibleDomain b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a,
                       xpress::objects::IISVariable::InfeasibleDomain b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(xpress::objects::IISVariable::InfeasibleDomain e) {
  switch (e) {
  case xpress::objects::IISVariable::InfeasibleDomain::UpperBound:
    return "UpperBound";
  case xpress::objects::IISVariable::InfeasibleDomain::LowerBound:
    return "LowerBound";
  case xpress::objects::IISVariable::InfeasibleDomain::Fixing:
    return "Fixing";
  case xpress::objects::IISVariable::InfeasibleDomain::Binary:
    return "Binary";
  case xpress::objects::IISVariable::InfeasibleDomain::Integral:
    return "Integral";
  case xpress::objects::IISVariable::InfeasibleDomain::PartialInteger:
    return "PartialInteger";
  case xpress::objects::IISVariable::InfeasibleDomain::SemiContinuous:
    return "SemiContinuous";
  case xpress::objects::IISVariable::InfeasibleDomain::SemiInteger:
    return "SemiInteger";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &
operator<<(std::ostream &os, xpress::objects::IISVariable::InfeasibleDomain e) {
  os << to_string(e);
  return os;
}

class IndicatorDescriptionImplementation;

class IndicatorDescription final : public ConstraintDefinition<Inequality> {
public:
  typedef IndicatorDescriptionImplementation impl_type;

protected:
  inline IndicatorDescription(
      std::shared_ptr<IndicatorDescriptionImplementation> init);

public:
  inline IndicatorDescription(IndicatorDescriptionImplementation const *init);
  inline virtual ~IndicatorDescription(){};

private:
  inline IndicatorDescriptionImplementation *impl();
  inline IndicatorDescriptionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(IndicatorDescription const &other) const {
    return impl() == other.impl();
  }
  inline IndicatorDescription(Variable indicatorVariable, bool indicatorValue,
                              InequalityDefinition implied);

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  virtual inline auto getName() const -> std::optional<std::string> override;

  inline auto setName(std::optional<std::string> newName)
      -> IndicatorDescription &;

  virtual inline auto
  getIndexHandler(xpress::objects::XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::Inequality> const
          * override;
  inline IndicatorDescription();
};

class IndicatorObjects final {
private:
  Variable indicatorVariable;

public:
  auto getIndicatorVariable() const -> Variable { return indicatorVariable; }

private:
  bool indicatorValue;

public:
  auto getIndicatorValue() const -> bool { return indicatorValue; }

private:
  Inequality row;

public:
  auto getRow() const -> Inequality { return row; }

  inline IndicatorObjects(Variable indicatorVariable, bool indicatorValue,
                          Inequality row);
};

class LinCoef final {
public:
  LinCoef &operator=(LinCoef const &) = default;

private:
  Variable variable;

public:
  auto getVariable() const -> Variable { return variable; }

private:
  double coefficient;

public:
  auto getCoefficient() const -> double { return coefficient; }

  inline LinCoef(Variable variable, double coefficient);

  inline LinCoef(LinCoef const &other);

  inline LinCoef();
};
class LinExpressionImplementation;

class LinExpression : public xpress::objects::AbstractExpression {
public:
  typedef LinExpressionImplementation impl_type;

protected:
  inline LinExpression(std::shared_ptr<LinExpressionImplementation> init);

public:
  inline LinExpression(LinExpressionImplementation const *init);
  inline virtual ~LinExpression(){};

private:
  inline LinExpressionImplementation *impl();
  inline LinExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(LinExpression const &other) const {
    return impl() == other.impl();
  }
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;

  virtual inline auto normalizeTerms() -> LinExpression &;

  virtual inline auto deepCopy() -> xpress::objects::LinExpression;

  virtual inline auto deepCopy(double factor) -> xpress::objects::LinExpression;

  virtual inline auto getConstant() const -> double;

  virtual inline auto addConstant(double c) -> LinExpression &;

  virtual inline auto setConstant(double c) -> LinExpression &;

  virtual inline auto reset() -> LinExpression &;

protected:
  virtual inline auto prepareToAdd(int newTerms) -> void;

public:
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto addTerms(Strm0 const &variables, Strm1 const &coefficients)
      -> LinExpression &;

  inline auto addTerms(xpress::SizedArray<Variable const> const &variables,
                       xpress::SizedArray<double const> const &coefficients,
                       int offset, int count) -> LinExpression &;

  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient) -> LinExpression &;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable) -> LinExpression &;

  inline auto addTerm(LinTerm const &term) -> LinExpression &;

  virtual inline auto addTerm(Variable const &x) -> LinExpression &;

  virtual inline auto addTerms(xpress::objects::LinExpression const &expr)
      -> LinExpression &;

  virtual inline auto addTerms(xpress::objects::LinExpression const &expr,
                               double factor) -> LinExpression &;

  inline auto square() -> xpress::objects::Expression;

  static inline auto create() -> xpress::objects::LinExpression;

  static inline auto create(double constant) -> xpress::objects::LinExpression;

  virtual inline auto getMaxDegree() const -> int override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  inline LinExpression();
};
class LinTermImplementation;

class LinTerm final : public xpress::objects::AbstractExpression {
public:
  typedef LinTermImplementation impl_type;

protected:
  inline LinTerm(std::shared_ptr<LinTermImplementation> init);

public:
  inline LinTerm(LinTermImplementation const *init);
  inline virtual ~LinTerm(){};

private:
  inline LinTermImplementation *impl();
  inline LinTermImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(LinTerm const &other) const { return impl() == other.impl(); }
  inline auto getVariable() const -> xpress::objects::Variable;

  inline auto getCoefficient() const -> double;

  inline LinTerm(Variable variable, double coefficient);

  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  static inline auto lTerm(xpress::objects::Variable variable,
                           double coefficient) -> xpress::objects::LinTerm;

  static inline auto lTerm(Variable variable) -> xpress::objects::LinTerm;

  inline auto square() const -> xpress::objects::QuadTerm;

  virtual inline auto toString() const -> std::string override;

  static inline auto toString(xpress::objects::Variable v, double coef)
      -> std::string;

  virtual inline auto getMaxDegree() const -> int override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;

  virtual inline auto uminus() const -> xpress::objects::Expression override;

  virtual inline auto mul(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;

  inline auto mul(ConstantExpression arg) const -> xpress::objects::LinTerm;

  virtual inline auto mul(double arg) const
      -> xpress::objects::Expression override;

  inline auto mul(Variable arg) const -> xpress::objects::QuadTerm;

  inline auto mul(LinTerm arg) const -> xpress::objects::QuadTerm;

  virtual inline auto div(Expression arg) const
      -> xpress::objects::Expression override;

  inline auto div(ConstantExpression arg) const -> xpress::objects::LinTerm;

  virtual inline auto div(double arg) const
      -> xpress::objects::Expression override;
  inline LinTerm();
};
inline xpress::objects::LinTerm operator-(LinTerm const &term);
inline xpress::objects::LinTerm operator*(LinTerm const &l, double d);
inline xpress::objects::LinTerm operator*(double d, LinTerm const &l);
inline xpress::objects::LinTerm operator*(LinTerm const &l,
                                          ConstantExpression const &d);
inline xpress::objects::QuadTerm operator*(LinTerm const &l, Variable const &v);
inline xpress::objects::QuadTerm operator*(LinTerm const &l, LinTerm const &t);
inline xpress::objects::LinTerm operator/(LinTerm const &l, double d);
inline xpress::objects::LinTerm operator/(LinTerm const &l,
                                          ConstantExpression const &d);
class PWLImplementation;

class PWL final : public xpress::objects::Index {
public:
  typedef PWLImplementation impl_type;

protected:
  std::shared_ptr<PWLImplementation> implPointer = nullptr;

public:
  bool isSameImpl(PWL const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(PWLImplementation const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline PWL(std::shared_ptr<PWLImplementation> init);

public:
  inline PWL(PWLImplementation const *init);
  inline virtual ~PWL(){};

private:
  inline PWLImplementation *impl();
  inline PWLImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(PWL const &other) const { return impl() == other.impl(); }
  inline auto getIndex() const -> int override;

  inline auto updateIndex(int delta) -> void override;

  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;

  inline auto getProblem() const -> xpress::objects::XpressProblem * override;

  inline auto getSerial() const -> XPRSint64 override;

  inline auto markDeleted() -> void override;

  virtual inline auto compareTo(PWL const &other) const -> int final;

  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline PWL(XpressProblem *prob, long serial, int index);

  inline auto getInput() -> xpress::objects::Variable;

  inline auto getResultant() -> xpress::objects::Variable;

  inline auto getBreakpoints() -> std::vector<xpress::PwlBreakpoint>;

  inline auto getName() const -> std::string override;

  inline auto setName(std::optional<std::string> newName) -> PWL &;
  inline PWL();
};
class PWLDefinitionImplementation;

class PWLDefinition final : public ConstraintDefinition<PWL> {
public:
  typedef PWLDefinitionImplementation impl_type;

protected:
  inline PWLDefinition(std::shared_ptr<PWLDefinitionImplementation> init);

public:
  inline PWLDefinition(PWLDefinitionImplementation const *init);
  inline virtual ~PWLDefinition(){};

private:
  inline PWLDefinitionImplementation *impl();
  inline PWLDefinitionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(PWLDefinition const &other) const {
    return impl() == other.impl();
  }
  inline PWLDefinition(xpress::objects::Variable input,
                       xpress::objects::Variable resultant,
                       xpress::SizedArray<double const> const &xval,
                       xpress::SizedArray<double const> const &yval,
                       std::optional<std::string> name);

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  inline auto setName(std::optional<std::string> newName) -> PWLDefinition &;

  virtual inline auto getIndexHandler(XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::PWL> const * override;
  inline PWLDefinition();
};

class QPair final
    : public xpress::maps::ComparableMapKey2<xpress::objects::Variable,
                                             xpress::objects::Variable> {
public:
  QPair &operator=(QPair const &) = default;

  inline QPair(xpress::objects::Variable v1, xpress::objects::Variable v2);

  inline QPair(QPair const &q);

  inline QPair();

  inline auto getVar1() const -> xpress::objects::Variable;

  inline auto getVar2() const -> xpress::objects::Variable;

  inline auto isLinear() -> bool;

  inline auto isConstant() -> bool;

  inline auto toString() const -> std::string;
};

class QuadCoef final {
public:
  QuadCoef &operator=(QuadCoef const &) = default;

private:
  QPair variables;

public:
  auto getVariables() const -> QPair { return variables; }

private:
  double coefficient;

public:
  auto getCoefficient() const -> double { return coefficient; }

  inline QuadCoef(QPair variables, double coefficient);

  inline QuadCoef(Variable var1, Variable var2, double coefficient);

  inline QuadCoef(QuadCoef const &other);

  inline QuadCoef();

  inline auto getVar1() -> xpress::objects::Variable;

  inline auto getVar2() -> xpress::objects::Variable;
};
class QuadExpressionImplementation;

class QuadExpression : public AbstractExpression {
public:
  typedef QuadExpressionImplementation impl_type;

protected:
  inline QuadExpression(std::shared_ptr<QuadExpressionImplementation> init);

public:
  inline QuadExpression(QuadExpressionImplementation const *init);
  inline virtual ~QuadExpression(){};

private:
  inline QuadExpressionImplementation *impl();
  inline QuadExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(QuadExpression const &other) const {
    return impl() == other.impl();
  }
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;

  virtual inline auto normalizeTerms() -> QuadExpression &;

  virtual inline auto deepCopy() -> xpress::objects::QuadExpression;

  virtual inline auto deepCopy(double factor)
      -> xpress::objects::QuadExpression;

  virtual inline auto getConstant() const -> double;

  virtual inline auto addConstant(double c) -> QuadExpression &;

  virtual inline auto setConstant(double c) -> QuadExpression &;

  virtual inline auto reset() -> QuadExpression &;

protected:
  virtual inline auto prepareToAdd(int newTerms) -> void;

public:
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto addTerms(Strm0 const &variables, Strm1 const &coefficients)
      -> QuadExpression &;

  template <typename Strm0, typename Strm1, typename Strm2,
            xpress_isstream(Strm0), xpress_isstream(Strm1),
            xpress_isstream(Strm2)>
  inline auto addTerms(Strm0 const &variables1, Strm1 const &variables2,
                       Strm2 const &coefficients) -> QuadExpression &;

  inline auto addTerms(xpress::SizedArray<Variable const> const &variables,
                       xpress::SizedArray<double const> const &coefficients,
                       int offset, int count) -> QuadExpression &;

  inline auto addTerms(xpress::SizedArray<Variable const> const &variables1,
                       xpress::SizedArray<Variable const> const &variables2,
                       xpress::SizedArray<double const> const &coefficients,
                       int offset, int count) -> QuadExpression &;

  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient) -> QuadExpression &;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable) -> QuadExpression &;

  virtual inline auto addTerm(Variable const &x) -> QuadExpression &;

  inline auto addTerm(LinTerm const &term) -> QuadExpression &;

  inline auto addTerm(QuadTerm const &term) -> QuadExpression &;

  virtual inline auto addTerm(Variable const &variable1,
                              Variable const &variable2,
                              double const &coefficient) -> QuadExpression &;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable1,
                              Variable const &variable2) -> QuadExpression &;

  virtual inline auto addTerm(Variable const &x1, Variable const &x2)
      -> QuadExpression &;

  virtual inline auto addTerms(LinExpression const &expr) -> QuadExpression &;

  virtual inline auto addTerms(QuadExpression const &expr) -> QuadExpression &;

  virtual inline auto addTerms(LinExpression const &expr, double const &factor)
      -> QuadExpression &;

  virtual inline auto addTerms(QuadExpression const &expr, double const &factor)
      -> QuadExpression &;

  static inline auto create() -> xpress::objects::QuadExpression;

  static inline auto create(double constant) -> xpress::objects::QuadExpression;

  virtual inline auto getMaxDegree() const -> int override;
  inline QuadExpression();
};
class QuadTermImplementation;

class QuadTerm final : public xpress::objects::AbstractExpression {
public:
  typedef QuadTermImplementation impl_type;

protected:
  inline QuadTerm(std::shared_ptr<QuadTermImplementation> init);

public:
  inline QuadTerm(QuadTermImplementation const *init);
  inline virtual ~QuadTerm(){};

private:
  inline QuadTermImplementation *impl();
  inline QuadTermImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(QuadTerm const &other) const { return impl() == other.impl(); }
  inline auto getVariable1() const -> xpress::objects::Variable;

  inline auto getVariable2() const -> xpress::objects::Variable;

  inline auto getCoefficient() const -> double;

  inline QuadTerm(xpress::objects::Variable variable1,
                  xpress::objects::Variable variable2, double coefficient);

  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  static inline auto qTerm(xpress::objects::Variable variable1,
                           xpress::objects::Variable variable2,
                           double coefficient) -> xpress::objects::QuadTerm;

  static inline auto qTerm(xpress::objects::Variable variable1,
                           xpress::objects::Variable variable2)
      -> xpress::objects::QuadTerm;

  virtual inline auto toString() const -> std::string override;

  static inline auto toString(xpress::objects::Variable v1,
                              xpress::objects::Variable v2, double coef)
      -> std::string;

  virtual inline auto getMaxDegree() const -> int override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;

  virtual inline auto uminus() const -> xpress::objects::Expression override;

  virtual inline auto mul(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;

  inline auto mul(ConstantExpression arg) const -> xpress::objects::QuadTerm;

  virtual inline auto mul(double arg) const
      -> xpress::objects::Expression override;

  virtual inline auto div(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;

  inline auto div(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::QuadTerm;

  virtual inline auto div(double arg) const
      -> xpress::objects::Expression override;
  inline QuadTerm();
};
inline xpress::objects::QuadTerm operator-(QuadTerm const &term);
inline xpress::objects::QuadTerm operator*(QuadTerm const &q, double d);
inline xpress::objects::QuadTerm operator*(double d, QuadTerm const &q);
inline xpress::objects::QuadTerm operator*(QuadTerm const &q,
                                           ConstantExpression const &d);
inline xpress::objects::QuadTerm operator/(QuadTerm const &q, double d);
class SOSImplementation;

class SOS final : public xpress::objects::Index {
public:
  typedef SOSImplementation impl_type;

protected:
  std::shared_ptr<SOSImplementation> implPointer = nullptr;

public:
  bool isSameImpl(SOS const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(SOSImplementation const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline SOS(std::shared_ptr<SOSImplementation> init);

public:
  inline SOS(SOSImplementation const *init);
  inline virtual ~SOS(){};

private:
  inline SOSImplementation *impl();
  inline SOSImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(SOS const &other) const { return impl() == other.impl(); }
  inline auto getIndex() const -> int override;

  inline auto updateIndex(int delta) -> void override;

  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;

  inline auto getProblem() const -> xpress::objects::XpressProblem * override;

  inline auto getSerial() const -> XPRSint64 override;

  inline auto markDeleted() -> void override;

  virtual inline auto compareTo(SOS const &other) const -> int final;

  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline SOS(XpressProblem *prob, long serial, int index);

  inline auto getName() const -> std::string override;

  inline auto setName(std::optional<std::string> newName) -> SOS &;

  static inline auto sos(xpress::SetType type,
                         xpress::SizedArray<Variable const> const &items,
                         xpress::SizedArray<double const> const &weights,
                         std::optional<std::string> name)
      -> xpress::objects::SOSDefinition;

  static inline auto sos1(xpress::SizedArray<Variable const> const &items,
                          xpress::SizedArray<double const> const &weights,
                          std::optional<std::string> name)
      -> xpress::objects::SOSDefinition;

  static inline auto sos2(xpress::SizedArray<Variable const> const &items,
                          xpress::SizedArray<double const> const &weights,
                          std::optional<std::string> name)
      -> xpress::objects::SOSDefinition;
  inline SOS();
};
class SOSDefinitionImplementation;

class SOSDefinition final : public ConstraintDefinition<SOS> {
public:
  typedef SOSDefinitionImplementation impl_type;

protected:
  inline SOSDefinition(std::shared_ptr<SOSDefinitionImplementation> init);

public:
  inline SOSDefinition(SOSDefinitionImplementation const *init);
  inline virtual ~SOSDefinition(){};

private:
  inline SOSDefinitionImplementation *impl();
  inline SOSDefinitionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(SOSDefinition const &other) const {
    return impl() == other.impl();
  }
  inline SOSDefinition(xpress::SetType type,
                       xpress::SizedArray<Variable const> const &items,
                       xpress::SizedArray<double const> const &weights,
                       std::optional<std::string> name);

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  inline auto setName(std::optional<std::string> newName) -> SOSDefinition &;

  virtual inline auto getIndexHandler(XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::SOS> const * override;
  inline SOSDefinition();
};
class SumExpressionImplementation;

class SumExpression final : public AbstractExpression {
public:
  typedef SumExpressionImplementation impl_type;

protected:
  inline SumExpression(std::shared_ptr<SumExpressionImplementation> init);

public:
  inline SumExpression(SumExpressionImplementation const *init);
  inline virtual ~SumExpression(){};

private:
  inline SumExpressionImplementation *impl();
  inline SumExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(SumExpression const &other) const {
    return impl() == other.impl();
  }
  inline SumExpression(xpress::SizedArray<Expression const> const &terms);

  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto toString() const -> std::string override;

  virtual inline auto getMaxDegree() const -> int override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  inline SumExpression();
};
class FormulaExpressionImplementation;

class FormulaExpression : public xpress::objects::AbstractExpression {
public:
  typedef FormulaExpressionImplementation impl_type;

protected:
  inline FormulaExpression(
      std::shared_ptr<FormulaExpressionImplementation> init);

public:
  inline FormulaExpression(FormulaExpressionImplementation const *init);
  inline virtual ~FormulaExpression(){};

private:
  inline FormulaExpressionImplementation *impl();
  inline FormulaExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(FormulaExpression const &other) const {
    return impl() == other.impl();
  }
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;

  virtual inline auto extract(PostfixExtractor *extractor) const -> void;

  virtual inline auto extract(double factor, PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool final override;

private:
  template <typename T>
  static inline auto parseUnaryFunction(SymbolicEvalStack &stack, T type)
      -> xpress::objects::InternalFunctionExpression;

  template <typename T, typename Func0,
            xpress_isinvocable(Func0, std::vector<T>, Func0, int)>
  static inline auto reverseArgs(std::vector<T> const &args, Func0 mkArray)
      -> std::vector<T>;

  template <typename T>
  static inline auto parseNaryFunction(SymbolicEvalStack &stack, T type)
      -> xpress::objects::InternalFunctionExpression;

public:
  static inline auto parse(XpressProblem *prob,
                           xpress::SizedArray<int const> const &type,
                           xpress::SizedArray<double const> const &value)
      -> xpress::objects::Expression;
  inline FormulaExpression();
};

class LinExprProductView final : public QuadView {
private:
  std::shared_ptr<LinearView> const e1;

  std::shared_ptr<LinearView> const e2;

public:
  inline LinExprProductView(std::shared_ptr<LinearView> e1,
                            std::shared_ptr<LinearView> e2);

private:
  class WrappedIteratorImpl : public xpress::WrappedIteratorImplementationBase<
                                  xpress::objects::QuadCoef> {
  private:
    std::shared_ptr<LinearView> outer;

    WrappedIterator<LinCoef> outerCurrent;

    WrappedIterator<LinCoef> outerEnd;

    std::shared_ptr<LinearView> inner;

    WrappedIterator<LinCoef> innerCurrent;

    WrappedIterator<LinCoef> innerEnd;

    QuadCoef value;

  public:
    inline WrappedIteratorImpl(std::shared_ptr<LinearView> outer,
                               std::shared_ptr<LinearView> inner, bool isEnd);

    virtual inline auto operator!=(WrappedIterator<QuadCoef> const &other) const
        -> bool override;

    virtual inline auto operator*() -> xpress::objects::QuadCoef & override;

    virtual inline auto operator++() -> xpress::objects::QuadCoef & override;
  };

public:
  virtual inline auto begin() const
      -> xpress::WrappedIterator<xpress::objects::QuadCoef> override;

  virtual inline auto end() const
      -> xpress::WrappedIterator<xpress::objects::QuadCoef> override;
};

class IISConstraint final {
private:
  std::variant<xpress::objects::Inequality, xpress::objects::SOS,
               xpress::objects::PWL, xpress::objects::GeneralConstraint>
      constraint;

public:
  auto getConstraint() const
      -> std::variant<xpress::objects::Inequality, xpress::objects::SOS,
                      xpress::objects::PWL,
                      xpress::objects::GeneralConstraint> {
    return constraint;
  }

private:
  std::optional<xpress::RowType> direction;

public:
  auto getDirection() const -> std::optional<xpress::RowType> {
    return direction;
  }

private:
  double dual;

public:
  auto getDual() const -> double { return dual; }

private:
  IIS::Isolation isolation;

public:
  auto getIsolation() const -> IIS::Isolation { return isolation; }

  template <typename I>
  inline IISConstraint(I constraint, std::optional<xpress::RowType> direction,
                       double dual, IIS::Isolation isolation);
};

class InternalUtils final {
public:
  template <typename M, typename T>
  static inline auto mergeCoefficient(M &m, T const &key, double value) -> void;

  static inline auto incsForSort() -> std::vector<int> const &;

  static inline auto sortParallelArrays(xpress::SizedArray<int> const &mvec,
                                        xpress::SizedArray<double> const &dvec,
                                        int offset, int size) -> void;

  static inline auto sortParallelArrays(xpress::SizedArray<int> const &key1,
                                        xpress::SizedArray<int> const &key2,
                                        xpress::SizedArray<double> const &dvec,
                                        int offset, int size) -> void;

  template <typename T, typename NoPrimitive = typename std::enable_if<
                            !std::is_integral<T>::value &&
                            !std::is_convertible<T, double>::value>::type>
  static inline auto sortParallelArrays(std::vector<T> &mvec,
                                        xpress::SizedArray<double> const &dvec,
                                        int offset, int size) -> void;

  template <typename T, typename NoPrimitive = typename std::enable_if<
                            !std::is_integral<T>::value &&
                            !std::is_convertible<T, double>::value>::type>
  static inline auto sortParallelArrays(std::vector<T> &key1,
                                        std::vector<T> &key2,
                                        xpress::SizedArray<double> const &dvec,
                                        int offset, int size) -> void;

  static inline auto
  sortParallelArraysLex(xpress::SizedArray<double> const &key1,
                        xpress::SizedArray<double> const &key2, int offset,
                        int size) -> void;

private:
  static inline auto calcExtendedArraySize(int old, int minAdd) -> int;

public:
  template <typename T>
  static inline auto extendArray(std::vector<T> &old, int minAdd, bool)
      -> std::vector<T> &;

  static inline auto getSingletonLinView(Variable v, double c)
      -> std::shared_ptr<xpress::objects::LinearView>;

  static inline auto getSingletonQuadView(Variable v1, Variable v2, double c)
      -> std::shared_ptr<xpress::objects::QuadView>;

  static inline auto floatToString(double value) -> std::string;
};
class LinTermListImplementation;

class LinTermList final : public LinExpression {
public:
  typedef LinTermListImplementation impl_type;

protected:
  inline LinTermList(std::shared_ptr<LinTermListImplementation> init);

public:
  inline LinTermList(LinTermListImplementation const *init);
  inline virtual ~LinTermList(){};

private:
  inline LinTermListImplementation *impl();
  inline LinTermListImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(LinTermList const &other) const { return impl() == other.impl(); }
  using LinExpression::addTerm;
  using LinExpression::addTerms;

  static inline auto setAutoCompress(bool compress) -> void;

  static inline auto isAutoCompress() -> bool;

  inline LinTermList();

  inline LinTermList(double c);

  inline LinTermList(double c, int capacity);

  virtual inline auto deepCopy() -> xpress::objects::LinExpression override;

  virtual inline auto deepCopy(double factor)
      -> xpress::objects::LinExpression override;

protected:
  virtual inline auto prepareToAdd(int newTerms) -> void override;

public:
  inline auto reserve(int l) -> LinTermList &;

  virtual inline auto reset() -> LinTermList & override;

  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient)
      -> LinTermList & override;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable)
      -> LinTermList & override;

  virtual inline auto addTerm(Variable const &x) -> LinTermList & override;

  virtual inline auto addTerms(LinExpression const &expr)
      -> LinTermList & override;

  virtual inline auto addTerms(LinExpression const &expr, double factor)
      -> LinTermList & override;

  virtual inline auto addConstant(double c) -> LinTermList & override;

  virtual inline auto setConstant(double c) -> LinTermList & override;

  virtual inline auto getConstant() const -> double override;

  virtual inline auto normalizeTerms() -> LinTermList & override;

  inline auto compress() const -> void;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto toString() const -> std::string override;
};
class LinTermMapImplementation;

class LinTermMap final : public LinExpression {
public:
  typedef LinTermMapImplementation impl_type;

protected:
  inline LinTermMap(std::shared_ptr<LinTermMapImplementation> init);

public:
  inline LinTermMap(LinTermMapImplementation const *init);
  inline virtual ~LinTermMap(){};

private:
  inline LinTermMapImplementation *impl();
  inline LinTermMapImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(LinTermMap const &other) const { return impl() == other.impl(); }
  using LinExpression::addTerm;
  using LinExpression::addTerms;

  inline LinTermMap();

  inline LinTermMap(double constant);

  virtual inline auto deepCopy() -> xpress::objects::LinExpression override;

  virtual inline auto deepCopy(double factor)
      -> xpress::objects::LinExpression override;

  virtual inline auto reset() -> LinTermMap & override;

  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient)
      -> LinTermMap & override;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable)
      -> LinTermMap & override;

  virtual inline auto addTerm(Variable const &x) -> LinTermMap & override;

  virtual inline auto addTerms(LinExpression const &expr)
      -> LinTermMap & override;

  virtual inline auto addTerms(LinExpression const &expr, double factor)
      -> LinTermMap & override;

  inline auto setCoefficient(Variable variable, double coefficient)
      -> LinTermMap &;

  inline auto getCoefficient(Variable variable) const -> double;

  virtual inline auto addConstant(double c) -> LinTermMap & override;

  virtual inline auto setConstant(double c) -> LinTermMap & override;

  virtual inline auto getConstant() const -> double override;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto toString() const -> std::string override;
};
class QuadTermListImplementation;

class QuadTermList final : public QuadExpression {
public:
  typedef QuadTermListImplementation impl_type;

protected:
  inline QuadTermList(std::shared_ptr<QuadTermListImplementation> init);

public:
  inline QuadTermList(QuadTermListImplementation const *init);
  inline virtual ~QuadTermList(){};

private:
  inline QuadTermListImplementation *impl();
  inline QuadTermListImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(QuadTermList const &other) const {
    return impl() == other.impl();
  }
  using QuadExpression::addTerm;
  using QuadExpression::addTerms;

  static inline auto setAutoCompress(bool compress) -> void;

  static inline auto isAutoCompress() -> bool;

  inline QuadTermList();

  inline QuadTermList(double c);

  inline QuadTermList(double c, int capacity);

  virtual inline auto deepCopy() -> xpress::objects::QuadExpression override;

  virtual inline auto deepCopy(double factor)
      -> xpress::objects::QuadExpression override;

  inline auto reserve(int l) -> QuadTermList &;

  virtual inline auto reset() -> QuadTermList & override;

  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient)
      -> QuadTermList & override;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable)
      -> QuadTermList & override;

  virtual inline auto addTerm(Variable const &x) -> QuadTermList & override;

  virtual inline auto addTerm(Variable const &variable1,
                              Variable const &variable2,
                              double const &coefficient)
      -> QuadTermList & override;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable1,
                              Variable const &variable2)
      -> QuadTermList & override;

  virtual inline auto addTerm(Variable const &x1, Variable const &x2)
      -> QuadTermList & override;

  virtual inline auto addTerms(LinExpression const &expr)
      -> QuadTermList & override;

  virtual inline auto addTerms(QuadExpression const &expr)
      -> QuadTermList & override;

  virtual inline auto addTerms(LinExpression const &expr, double const &factor)
      -> QuadTermList & override;

  virtual inline auto addTerms(QuadExpression const &expr, double const &factor)
      -> QuadTermList & override;

  virtual inline auto addConstant(double c) -> QuadTermList & override;

  virtual inline auto setConstant(double c) -> QuadTermList & override;

  virtual inline auto getConstant() const -> double override;

  virtual inline auto normalizeTerms() -> QuadTermList & override;

  inline auto compress() const -> void;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto extract(double factor, PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;

  virtual inline auto toString() const -> std::string override;
};
class QuadTermMapImplementation;

class QuadTermMap final : public QuadExpression {
public:
  typedef QuadTermMapImplementation impl_type;

protected:
  inline QuadTermMap(std::shared_ptr<QuadTermMapImplementation> init);

public:
  inline QuadTermMap(QuadTermMapImplementation const *init);
  inline virtual ~QuadTermMap(){};

private:
  inline QuadTermMapImplementation *impl();
  inline QuadTermMapImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(QuadTermMap const &other) const { return impl() == other.impl(); }
  using QuadExpression::addTerm;
  using QuadExpression::addTerms;

  inline QuadTermMap();

  inline QuadTermMap(double constant);

  virtual inline auto deepCopy() -> xpress::objects::QuadExpression override;

  virtual inline auto deepCopy(double factor)
      -> xpress::objects::QuadExpression override;

  virtual inline auto reset() -> QuadTermMap & override;

  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient)
      -> QuadTermMap & override;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable)
      -> QuadTermMap & override;

  virtual inline auto addTerm(Variable const &x) -> QuadTermMap & override;

  virtual inline auto addTerm(Variable const &variable1,
                              Variable const &variable2,
                              double const &coefficient)
      -> QuadTermMap & override;

  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable1,
                              Variable const &variable2)
      -> QuadTermMap & override;

  virtual inline auto addTerm(Variable const &x1, Variable const &x2)
      -> QuadTermMap & override;

  virtual inline auto addTerms(LinExpression const &expr)
      -> QuadTermMap & override;

  virtual inline auto addTerms(QuadExpression const &expr)
      -> QuadTermMap & override;

  virtual inline auto addTerms(LinExpression const &expr, double const &factor)
      -> QuadTermMap & override;

  virtual inline auto addTerms(QuadExpression const &expr, double const &factor)
      -> QuadTermMap & override;

  inline auto setCoefficient(Variable variable, double coefficient)
      -> QuadTermMap &;

  inline auto setCoefficient(Variable variable1, Variable variable2,
                             double coefficient) -> QuadTermMap &;

  inline auto getCoefficient(Variable variable) const -> double;

  inline auto getCoefficient(Variable variable1, Variable variable2) const
      -> double;

  virtual inline auto addConstant(double c) -> QuadTermMap & override;

  virtual inline auto setConstant(double c) -> QuadTermMap & override;

  virtual inline auto getConstant() const -> double override;

  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;

  virtual inline auto extract(double factor, PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;

  virtual inline auto toString() const -> std::string override;
};
class BinaryExpressionImplementation;

class BinaryExpression final : public FormulaExpression {
public:
  typedef BinaryExpressionImplementation impl_type;

protected:
  inline BinaryExpression(std::shared_ptr<BinaryExpressionImplementation> init);

public:
  inline BinaryExpression(BinaryExpressionImplementation const *init);
  inline virtual ~BinaryExpression(){};

private:
  inline BinaryExpressionImplementation *impl();
  inline BinaryExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(BinaryExpression const &other) const {
    return impl() == other.impl();
  }
  enum struct Operator : int {

    Exponent = XPRS_OP_EXPONENT,
    Multiply = XPRS_OP_MULTIPLY,
    Divide = XPRS_OP_DIVIDE,
    Plus = XPRS_OP_PLUS,
    Minus = XPRS_OP_MINUS
  };

  inline auto getLeft() const -> xpress::objects::Expression;

  inline auto getOp() const -> xpress::objects::BinaryExpression::Operator;

  inline auto getRight() const -> xpress::objects::Expression;

  inline BinaryExpression(xpress::objects::Expression left,
                          xpress::objects::BinaryExpression::Operator op,
                          xpress::objects::Expression right);

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto toString() const -> std::string override;

  virtual inline auto getMaxDegree() const -> int override;

  virtual inline auto getConstantView() const -> double override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  inline BinaryExpression();
};
inline bool operator==(xpress::objects::BinaryExpression::Operator a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(xpress::objects::BinaryExpression::Operator a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, xpress::objects::BinaryExpression::Operator b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, xpress::objects::BinaryExpression::Operator b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(xpress::objects::BinaryExpression::Operator e) {
  switch (e) {
  case xpress::objects::BinaryExpression::Operator::Exponent:
    return "Exponent";
  case xpress::objects::BinaryExpression::Operator::Multiply:
    return "Multiply";
  case xpress::objects::BinaryExpression::Operator::Divide:
    return "Divide";
  case xpress::objects::BinaryExpression::Operator::Plus:
    return "Plus";
  case xpress::objects::BinaryExpression::Operator::Minus:
    return "Minus";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os,
                                xpress::objects::BinaryExpression::Operator e) {
  os << to_string(e);
  return os;
}

class InternalFunctionExpressionImplementation;

class InternalFunctionExpression final : public FormulaExpression {
public:
  typedef InternalFunctionExpressionImplementation impl_type;

protected:
  inline InternalFunctionExpression(
      std::shared_ptr<InternalFunctionExpressionImplementation> init);

public:
  inline InternalFunctionExpression(
      InternalFunctionExpressionImplementation const *init);
  inline virtual ~InternalFunctionExpression(){};

private:
  inline InternalFunctionExpressionImplementation *impl();
  inline InternalFunctionExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(InternalFunctionExpression const &other) const {
    return impl() == other.impl();
  }
  enum struct FunctionType : int {

    Log10 = XPRS_IFUN_LOG10,
    Ln = XPRS_IFUN_LN,
    Exp = XPRS_IFUN_EXP,
    Abs = XPRS_IFUN_ABS,
    Sqrt = XPRS_IFUN_SQRT,
    Sin = XPRS_IFUN_SIN,
    Cos = XPRS_IFUN_COS,
    Tan = XPRS_IFUN_TAN,
    ArcSin = XPRS_IFUN_ARCSIN,
    ArcCos = XPRS_IFUN_ARCCOS,
    ArcTan = XPRS_IFUN_ARCTAN,
    Min = XPRS_IFUN_MIN,
    Max = XPRS_IFUN_MAX,
    Sum = XPRS_IFUN_SUM,
    Prod = XPRS_IFUN_PROD,
    Sign = XPRS_IFUN_SIGN
  };

  inline auto getFunction() const
      -> xpress::objects::InternalFunctionExpression::FunctionType;

  inline auto getArguments() const -> std::vector<xpress::objects::Expression>;

  inline InternalFunctionExpression(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      std::vector<xpress::objects::Expression> arguments);

  inline InternalFunctionExpression(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      xpress::objects::Expression argument);

  inline InternalFunctionExpression(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      xpress::objects::Expression firstArgument,
      xpress::objects::Expression secondArgument);

private:
  static inline auto
  collectArguments(Expression firstArgument, Expression secondArgument,
                   xpress::SizedArray<Expression const> const &more)
      -> std::vector<xpress::objects::Expression>;

public:
  template <
      typename... MOREARGUMENTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::objects::Expression, MOREARGUMENTS...>::value>::type>
  inline InternalFunctionExpression(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      xpress::objects::Expression firstArgument,
      xpress::objects::Expression secondArgument,
      MOREARGUMENTS... moreArguments);

  inline InternalFunctionExpression(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      xpress::SizedArray<Expression const> const &args);

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto toString() const -> std::string override;

  virtual inline auto getMaxDegree() const -> int override;
  inline InternalFunctionExpression();
};
inline bool
operator==(xpress::objects::InternalFunctionExpression::FunctionType a, int b) {
  return static_cast<int>(a) == b;
}
inline bool
operator!=(xpress::objects::InternalFunctionExpression::FunctionType a, int b) {
  return static_cast<int>(a) != b;
}
inline bool
operator==(int a, xpress::objects::InternalFunctionExpression::FunctionType b) {
  return a == static_cast<int>(b);
}
inline bool
operator!=(int a, xpress::objects::InternalFunctionExpression::FunctionType b) {
  return a != static_cast<int>(b);
}
inline std::string
to_string(xpress::objects::InternalFunctionExpression::FunctionType e) {
  switch (e) {
  case xpress::objects::InternalFunctionExpression::FunctionType::Log10:
    return "Log10";
  case xpress::objects::InternalFunctionExpression::FunctionType::Ln:
    return "Ln";
  case xpress::objects::InternalFunctionExpression::FunctionType::Exp:
    return "Exp";
  case xpress::objects::InternalFunctionExpression::FunctionType::Abs:
    return "Abs";
  case xpress::objects::InternalFunctionExpression::FunctionType::Sqrt:
    return "Sqrt";
  case xpress::objects::InternalFunctionExpression::FunctionType::Sin:
    return "Sin";
  case xpress::objects::InternalFunctionExpression::FunctionType::Cos:
    return "Cos";
  case xpress::objects::InternalFunctionExpression::FunctionType::Tan:
    return "Tan";
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcSin:
    return "ArcSin";
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcCos:
    return "ArcCos";
  case xpress::objects::InternalFunctionExpression::FunctionType::ArcTan:
    return "ArcTan";
  case xpress::objects::InternalFunctionExpression::FunctionType::Min:
    return "Min";
  case xpress::objects::InternalFunctionExpression::FunctionType::Max:
    return "Max";
  case xpress::objects::InternalFunctionExpression::FunctionType::Sum:
    return "Sum";
  case xpress::objects::InternalFunctionExpression::FunctionType::Prod:
    return "Prod";
  case xpress::objects::InternalFunctionExpression::FunctionType::Sign:
    return "Sign";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &
operator<<(std::ostream &os,
           xpress::objects::InternalFunctionExpression::FunctionType e) {
  os << to_string(e);
  return os;
}

class PWLExpressionImplementation;

class PWLExpression : public FormulaExpression {
public:
  typedef PWLExpressionImplementation impl_type;

protected:
  inline PWLExpression(std::shared_ptr<PWLExpressionImplementation> init);

public:
  inline PWLExpression(PWLExpressionImplementation const *init);
  inline virtual ~PWLExpression(){};

private:
  inline PWLExpressionImplementation *impl();
  inline PWLExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(PWLExpression const &other) const {
    return impl() == other.impl();
  }
private:
  static inline auto
  breakToX(xpress::SizedArray<PwlBreakpoint const> const &points)
      -> std::vector<double>;

  static inline auto
  breakToY(xpress::SizedArray<PwlBreakpoint const> const &points)
      -> std::vector<double>;

public:
  inline auto getArgumentColumn() const -> int;

  inline auto getArgumentVariable() const -> xpress::objects::Variable;

  inline auto getBreakX() const -> std::vector<double>;

  inline auto getBreakY() const -> std::vector<double>;

  inline PWLExpression(Variable argument,
                       xpress::SizedArray<double const> const &breakX,
                       xpress::SizedArray<double const> const &breakY);

  template <
      typename... POINTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::PwlBreakpoint, POINTS...>::value>::type>
  inline PWLExpression(Variable argument, POINTS... points);

  inline PWLExpression(Variable argument,
                       xpress::SizedArray<PwlBreakpoint const> const &points);

  inline PWLExpression(int argument,
                       xpress::SizedArray<double const> const &breakX,
                       xpress::SizedArray<double const> const &breakY);

  template <
      typename... POINTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::PwlBreakpoint, POINTS...>::value>::type>
  inline PWLExpression(int argument, POINTS... points);

  inline PWLExpression(int argument,
                       xpress::SizedArray<PwlBreakpoint const> const &points);

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto toString() const -> std::string override;

  virtual inline auto getMaxDegree() const -> int override;
  inline PWLExpression();
};
class UnaryExpressionImplementation;

class UnaryExpression final : public FormulaExpression {
public:
  typedef UnaryExpressionImplementation impl_type;

protected:
  inline UnaryExpression(std::shared_ptr<UnaryExpressionImplementation> init);

public:
  inline UnaryExpression(UnaryExpressionImplementation const *init);
  inline virtual ~UnaryExpression(){};

private:
  inline UnaryExpressionImplementation *impl();
  inline UnaryExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(UnaryExpression const &other) const {
    return impl() == other.impl();
  }
  enum struct Operator : int {

    UMinus = XPRS_OP_UMINUS
  };

  inline auto getOp() const -> xpress::objects::UnaryExpression::Operator;

  inline auto getArgument() const -> xpress::objects::Expression;

  inline UnaryExpression(xpress::objects::UnaryExpression::Operator op,
                         xpress::objects::Expression argument);

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto toString() const -> std::string override;

  virtual inline auto getMaxDegree() const -> int override;

  virtual inline auto getConstantView() const -> double override;

  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;

  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;

  static inline auto scaledLinView(std::shared_ptr<LinearView> data,
                                   double factor)
      -> std::shared_ptr<xpress::objects::LinearView>;

  static inline auto scaledQuadView(std::shared_ptr<QuadView> data,
                                    double factor)
      -> std::shared_ptr<xpress::objects::QuadView>;
  inline UnaryExpression();
};
inline bool operator==(xpress::objects::UnaryExpression::Operator a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(xpress::objects::UnaryExpression::Operator a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, xpress::objects::UnaryExpression::Operator b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, xpress::objects::UnaryExpression::Operator b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(xpress::objects::UnaryExpression::Operator e) {
  switch (e) {
  case xpress::objects::UnaryExpression::Operator::UMinus:
    return "UMinus";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os,
                                xpress::objects::UnaryExpression::Operator e) {
  os << to_string(e);
  return os;
}

class UserFunctionExpressionImplementation;

class UserFunctionExpression : public FormulaExpression {
public:
  typedef UserFunctionExpressionImplementation impl_type;

protected:
  inline UserFunctionExpression(
      std::shared_ptr<UserFunctionExpressionImplementation> init);

public:
  inline UserFunctionExpression(
      UserFunctionExpressionImplementation const *init);
  inline virtual ~UserFunctionExpression(){};

private:
  inline UserFunctionExpressionImplementation *impl();
  inline UserFunctionExpressionImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(UserFunctionExpression const &other) const {
    return impl() == other.impl();
  }
  inline auto getFunction() const
      -> xpress::XPRSProblem::AbstractUserFunction *;

  inline auto getArguments() const -> std::vector<xpress::objects::Expression>;

  inline auto getOutput() const -> int;

  template <
      typename... ARGUMENTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::objects::Expression, ARGUMENTS...>::value>::type>
  inline UserFunctionExpression(
      xpress::XPRSProblem::AbstractUserFunction *function,
      ARGUMENTS... arguments);

  template <
      typename... ARGUMENTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::objects::Expression, ARGUMENTS...>::value>::type>
  inline UserFunctionExpression(
      xpress::XPRSProblem::AbstractUserFunction *function, int output,
      ARGUMENTS... arguments);

  inline UserFunctionExpression(
      xpress::XPRSProblem::AbstractUserFunction *function,
      SizedArray<Expression> const &arguments);

  inline UserFunctionExpression(
      xpress::XPRSProblem::AbstractUserFunction *function, int output,
      SizedArray<Expression> const &arguments);

  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;

  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;

  virtual inline auto toString() const -> std::string override;

  virtual inline auto getMaxDegree() const -> int override;
  inline UserFunctionExpression();
};
template <typename C>
class ConstraintDefinitionImplementation
    : public std::enable_shared_from_this<
          ConstraintDefinitionImplementation<C>> {
  friend class ConstraintDefinition<C>;

public:
  virtual ~ConstraintDefinitionImplementation() {}

private:
  std::optional<std::string> name;

  bool mutable extracted;

protected:
  inline ConstraintDefinitionImplementation(std::optional<std::string> name);
  virtual inline auto checkModify() -> void final;

public:
  virtual inline auto getName() const -> std::optional<std::string>;
  virtual inline auto setConstraintName(std::optional<std::string> newName)
      -> ConstraintDefinitionImplementation<C> & final;

protected:
  virtual auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void = 0;

public:
  virtual inline auto
  create(xpress::XPRSProblem::ConstraintCreator *creator) const -> void final;
  virtual auto getIndexHandler(XpressProblem *prob) const
      -> xpress::objects::IndexHandler<C> const * = 0;
};
class ExpressionImplementation
    : public std::enable_shared_from_this<ExpressionImplementation> {
  friend class Expression;

public:
  virtual ~ExpressionImplementation() {}
  virtual auto leq(double rhs) const
      -> xpress::objects::InequalityDefinition = 0;
  virtual auto leq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition = 0;
  virtual auto geq(double rhs) const
      -> xpress::objects::InequalityDefinition = 0;
  virtual auto geq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition = 0;
  virtual auto eq(double rhs) const
      -> xpress::objects::InequalityDefinition = 0;
  virtual auto eq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition = 0;
  virtual auto in(double lb, double ub)
      -> xpress::objects::InequalityDefinition = 0;
  virtual auto extract(double factor,
                       xpress::XPRSProblem::RowCreator *creator) const
      -> bool = 0;
  virtual auto extract(double factor,
                       xpress::objects::PostfixExtractor *extractor) const
      -> void = 0;
  virtual auto evaluate(xpress::SizedArray<double const> const &solution) const
      -> double = 0;
  virtual auto getRTTI() -> xpress::objects::ExpressionRTTI = 0;
  virtual auto getMaxDegree() const -> int = 0;
  virtual auto getConstantView() const -> double = 0;
  virtual auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> = 0;
  virtual auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> = 0;
  virtual auto toString() const -> std::string = 0;
  virtual inline auto uminus() const -> xpress::objects::Expression;
  virtual inline auto plus(xpress::objects::Expression arg) const
      -> xpress::objects::Expression;
  virtual inline auto plus(double arg) const -> xpress::objects::Expression;
  virtual inline auto minus(xpress::objects::Expression arg) const
      -> xpress::objects::Expression;
  virtual inline auto minus(double arg) const -> xpress::objects::Expression;
  virtual inline auto mul(Expression arg) const -> xpress::objects::Expression;
  virtual inline auto mul(double arg) const -> xpress::objects::Expression;
  virtual inline auto div(Expression arg) const -> xpress::objects::Expression;
  virtual inline auto div(double arg) const -> xpress::objects::Expression;
};
class InequalityImplementation final
    : public xpress::objects::Index,
      public std::enable_shared_from_this<InequalityImplementation> {
  friend class Inequality;

public:
  virtual ~InequalityImplementation() {}

  XpressProblem *prob;

  XPRSint64 const serial;

private:
  int index;

public:
  inline auto getIndex() const -> int override;
  inline auto updateIndex(int delta) -> void override;
  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;
  inline auto getProblem() const -> xpress::objects::XpressProblem * override;
  inline auto getSerial() const -> XPRSint64 override;
  inline auto markDeleted() -> void override;
  virtual inline auto compareTo(Inequality const &other) const -> int final;
  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline InequalityImplementation(XpressProblem *prob, long serial, int index);
  inline auto getRhs() -> double;
  inline auto getRhsRange() -> double;
  inline auto getName() const -> std::string override;
  inline auto getType() -> xpress::RowType;
  inline auto setRhs(double newRhs) -> InequalityImplementation &;
  inline auto setRhsRange(double newRange) -> InequalityImplementation &;
  inline auto setType(xpress::RowType newType) -> InequalityImplementation &;
  inline auto setName(std::optional<std::string> newName)
      -> InequalityImplementation &;
  inline auto getValue(xpress::SizedArray<double const> const &data) const
      -> double;
  inline auto getSlack() const -> double;
  inline auto getDual() const -> double;
  inline auto getCallbackSlack() -> double;
  inline auto getCallbackDual() -> double;
  inline auto chgCoef(Variable v, double newCoef) -> InequalityImplementation &;
  inline auto chgCoefs(xpress::SizedArray<Variable const> const &v,
                       xpress::SizedArray<double const> const &newCoef)
      -> InequalityImplementation &;
  inline auto getLhs() -> xpress::objects::Expression;
};
class AbstractExpressionImplementation
    : public xpress::objects::ExpressionImplementation {
  friend class AbstractExpression;

public:
  virtual ~AbstractExpressionImplementation() {}
  virtual inline auto getConstantView() const -> double override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  virtual inline auto leq(double rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto leq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto geq(double rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto geq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto eq(double rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto eq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto in(double lb, double ub)
      -> xpress::objects::InequalityDefinition override;
};
class InequalityDefinitionImplementation final
    : public ConstraintDefinitionImplementation<Inequality> {
  friend class InequalityDefinition;

public:
  virtual ~InequalityDefinitionImplementation() {}

private:
  xpress::objects::Expression lhs;

public:
  auto getLhs() const -> xpress::objects::Expression { return lhs; }

private:
  xpress::RowType type;

public:
  auto getType() const -> xpress::RowType { return type; }

private:
  xpress::objects::Expression rhs;

public:
  auto getRhs() const -> xpress::objects::Expression { return rhs; }

private:
  std::vector<double> range;

public:
  auto getRange() const -> std::vector<double> { return range; }
  inline InequalityDefinitionImplementation(xpress::objects::Expression lhs,
                                            xpress::RowType type,
                                            xpress::objects::Expression rhs,
                                            std::optional<std::string> name);
  inline InequalityDefinitionImplementation(xpress::objects::Expression lhs,
                                            xpress::RowType type, double rhs,
                                            std::optional<std::string> name);
  inline InequalityDefinitionImplementation(double lhs, xpress::RowType type,
                                            xpress::objects::Expression rhs,
                                            std::optional<std::string> name);
  inline InequalityDefinitionImplementation(Expression expr, double lb,
                                            double ub,
                                            std::optional<std::string> name);
  inline auto createRow(xpress::XPRSProblem::RowCreator *creator) const -> void;

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  inline auto setName(std::optional<std::string> newName)
      -> InequalityDefinitionImplementation &;
  virtual inline auto getIndexHandler(XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::Inequality> const
          * override;
};
class VariableImplementation final
    : public xpress::objects::ExpressionImplementation,
      public xpress::objects::Index {
  friend class Variable;

public:
  virtual ~VariableImplementation() {}

  XpressProblem *prob;

  XPRSint64 const serial;

private:
  int index;

public:
  inline auto getIndex() const -> int override;
  inline auto updateIndex(int delta) -> void override;
  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;
  inline auto getProblem() const -> xpress::objects::XpressProblem * override;
  inline auto getSerial() const -> XPRSint64 override;
  inline auto markDeleted() -> void override;
  virtual inline auto compareTo(Variable const &other) const -> int final;
  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline VariableImplementation(XpressProblem *prob, long serial, int index);
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;
  virtual inline auto leq(double rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto leq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto geq(double rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto geq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto eq(double rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto eq(Expression const &rhs) const
      -> xpress::objects::InequalityDefinition override;
  virtual inline auto in(double lb, double ub)
      -> xpress::objects::InequalityDefinition override;
  inline auto square() -> xpress::objects::QuadTerm;
  inline auto getLB() const -> double;
  inline auto getUB() const -> double;
  inline auto getName() const -> std::string override;
  inline auto getType() const -> xpress::ColumnType;
  inline auto fix(double value) -> VariableImplementation &;
  inline auto setLB(double newLB) -> VariableImplementation &;
  inline auto setUB(double newUB) -> VariableImplementation &;
  inline auto setType(xpress::ColumnType newType) -> VariableImplementation &;
  inline auto setLimit(double newLimit) -> VariableImplementation &;
  inline auto setName(std::optional<std::string> newName)
      -> VariableImplementation &;
  inline auto getValue(xpress::SizedArray<double const> const &data) const
      -> double;
  inline auto getSolution() const -> double;
  inline auto getRedCost() const -> double;
  inline auto getCallbackSolution() -> double;
  inline auto getCallbackRedCost() -> double;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  inline auto chgCoef(Inequality r, double newCoef) -> void;
  inline auto chgCoefs(xpress::SizedArray<Inequality const> const &r,
                       xpress::SizedArray<double const> const &newCoef) -> void;
  inline auto chgObj(double newCoef) -> VariableImplementation &;
  inline auto getObj() const -> double;
  inline auto chgObjN(int objidx, double newCoef) -> VariableImplementation &;
  inline auto getObjN(int objIdx) const -> double;
  inline auto pwlOf(Variable input,
                    xpress::SizedArray<double const> const &xval,
                    xpress::SizedArray<double const> const &yval) const
      -> xpress::objects::PWLDefinition;
  inline auto pwlOf(Variable input,
                    xpress::SizedArray<double const> const &xval,
                    xpress::SizedArray<double const> const &yval,
                    std::optional<std::string> name) const
      -> xpress::objects::PWLDefinition;
  template <
      typename... BREAKPOINTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::PwlBreakpoint, BREAKPOINTS...>::value>::type>
  inline auto pwlOf(Variable input, BREAKPOINTS... breakpoints) const
      -> xpress::objects::PWLDefinition;
  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto pwlOf(Variable input, Strm0 const &breakpoints) const
      -> xpress::objects::PWLDefinition;
  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto pwlOf(Variable input, Strm0 const &breakpoints,
                    std::optional<std::string> name) const
      -> xpress::objects::PWLDefinition;
  template <typename... VALUES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<double, VALUES...>::value>::type>
  inline auto maxOf(xpress::SizedArray<Variable const> const &variables,
                    VALUES... values) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto maxOf(Strm0 const &variables, Strm1 const &values,
                    std::optional<std::string> name) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto maxOf(Strm0 const &variables, Strm1 const &values) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline auto maxOf(VARIABLES... variables) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VALUES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<double, VALUES...>::value>::type>
  inline auto minOf(xpress::SizedArray<Variable const> const &variables,
                    VALUES... values) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto minOf(Strm0 const &variables, Strm1 const &values,
                    std::optional<std::string> name) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto minOf(Strm0 const &variables, Strm1 const &values) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline auto minOf(VARIABLES... variables) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto orOf(Strm0 const &variables,
                   std::optional<std::string> name) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto orOf(Strm0 const &variables) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline auto orOf(VARIABLES... variables) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto andOf(Strm0 const &variables,
                    std::optional<std::string> name) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto andOf(Strm0 const &variables) const
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline auto andOf(VARIABLES... variables) const
      -> xpress::objects::GeneralConstraintDefinition;
  inline auto absOf(Variable x) const
      -> xpress::objects::GeneralConstraintDefinition;
  inline auto ifThen(InequalityDefinition implied) const
      -> xpress::objects::IndicatorDescription;
  inline auto ifNotThen(InequalityDefinition implied) const
      -> xpress::objects::IndicatorDescription;
  virtual inline auto toString() const -> std::string override;
  virtual inline auto getMaxDegree() const -> int override;
  virtual inline auto getConstantView() const -> double override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  virtual inline auto uminus() const -> xpress::objects::Expression override;
  virtual inline auto mul(Expression arg) const
      -> xpress::objects::Expression override;
  inline auto mul(ConstantExpression arg) const -> xpress::objects::LinTerm;
  virtual inline auto mul(double arg) const
      -> xpress::objects::Expression override;
  inline auto mul(Variable arg) const -> xpress::objects::QuadTerm;
  inline auto mul(LinTerm arg) const -> xpress::objects::QuadTerm;
  virtual inline auto div(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;
  inline auto div(ConstantExpression arg) const -> xpress::objects::LinTerm;
  virtual inline auto div(double arg) const
      -> xpress::objects::Expression override;
};
class ConstantExpressionImplementation final
    : public AbstractExpressionImplementation {
  friend class ConstantExpression;

public:
  virtual ~ConstantExpressionImplementation() {}

private:
  double value;

public:
  auto getValue() const -> double { return value; }
  inline ConstantExpressionImplementation(double value);
  static inline auto constant(double cnst)
      -> xpress::objects::ConstantExpression;
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto toString() const -> std::string override;
  virtual inline auto getMaxDegree() const -> int override;
  virtual inline auto getConstantView() const -> double override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  virtual inline auto uminus() const -> xpress::objects::Expression override;
  virtual inline auto plus(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;
  inline auto plus(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::ConstantExpression;
  virtual inline auto plus(double arg) const
      -> xpress::objects::Expression override;
  virtual inline auto minus(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;
  inline auto minus(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::ConstantExpression;
  virtual inline auto minus(double arg) const
      -> xpress::objects::Expression override;
  virtual inline auto mul(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;
  inline auto mul(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::ConstantExpression;
  virtual inline auto mul(double arg) const
      -> xpress::objects::Expression override;
  inline auto mul(xpress::objects::Variable arg) const
      -> xpress::objects::LinTerm;
  inline auto mul(xpress::objects::LinTerm arg) const
      -> xpress::objects::LinTerm;
  inline auto mul(xpress::objects::QuadTerm arg) const
      -> xpress::objects::QuadTerm;
  virtual inline auto div(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;
  inline auto div(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::ConstantExpression;
  virtual inline auto div(double arg) const
      -> xpress::objects::Expression override;
};
class GeneralConstraintImplementation final
    : public xpress::objects::Index,
      public std::enable_shared_from_this<GeneralConstraintImplementation> {
  friend class GeneralConstraint;

public:
  virtual ~GeneralConstraintImplementation() {}

  XpressProblem *prob;

  XPRSint64 const serial;

private:
  int index;

public:
  inline auto getIndex() const -> int override;
  inline auto updateIndex(int delta) -> void override;
  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;
  inline auto getProblem() const -> xpress::objects::XpressProblem * override;
  inline auto getSerial() const -> XPRSint64 override;
  inline auto markDeleted() -> void override;
  virtual inline auto compareTo(GeneralConstraint const &other) const
      -> int final;
  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline GeneralConstraintImplementation(XpressProblem *prob, long serial,
                                         int index);
  inline auto getType() -> xpress::GenConsType;
  inline auto getResultant() -> xpress::objects::Variable;
  inline auto getVariables() -> std::vector<xpress::objects::Variable>;
  inline auto getValues() -> std::vector<double>;
  inline auto getName() const -> std::string override;
  inline auto setName(std::optional<std::string> newName)
      -> GeneralConstraintImplementation &;
  static inline auto Abs(Variable resultant, Variable variable)
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  static inline auto And(Variable resultant, VARIABLES... variables)
      -> xpress::objects::GeneralConstraintDefinition;
  static inline auto And(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables)
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  static inline auto Or(Variable resultant, VARIABLES... variables)
      -> xpress::objects::GeneralConstraintDefinition;
  static inline auto Or(Variable resultant,
                        xpress::SizedArray<Variable const> const &variables)
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  static inline auto Max(Variable resultant, VARIABLES... variables)
      -> xpress::objects::GeneralConstraintDefinition;
  static inline auto Max(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables)
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VALUES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<double, VALUES...>::value>::type>
  static inline auto Max(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables,
                         VALUES... values)
      -> xpress::objects::GeneralConstraintDefinition;
  static inline auto Max(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables,
                         xpress::SizedArray<double const> const &values)
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  static inline auto Min(Variable resultant, VARIABLES... variables)
      -> xpress::objects::GeneralConstraintDefinition;
  static inline auto Min(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables)
      -> xpress::objects::GeneralConstraintDefinition;
  template <typename... VALUES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<double, VALUES...>::value>::type>
  static inline auto Min(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables,
                         VALUES... values)
      -> xpress::objects::GeneralConstraintDefinition;
  static inline auto Min(Variable resultant,
                         xpress::SizedArray<Variable const> const &variables,
                         xpress::SizedArray<double const> const &values)
      -> xpress::objects::GeneralConstraintDefinition;
};
class GeneralConstraintDefinitionImplementation final
    : public ConstraintDefinitionImplementation<GeneralConstraint> {
  friend class GeneralConstraintDefinition;

public:
  virtual ~GeneralConstraintDefinitionImplementation() {}

private:
  std::vector<Variable> const variables;

  std::vector<double> const value;

  Variable const resultant;

  xpress::GenConsType const type;

public:
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline GeneralConstraintDefinitionImplementation(
      Variable resultant, xpress::GenConsType type, Strm0 const &variables,
      Strm1 const &value, std::optional<std::string> name);
  template <typename Strm0, xpress_isstream(Strm0)>
  inline GeneralConstraintDefinitionImplementation(
      Variable resultant, xpress::GenConsType type, Strm0 const &variables,
      std::optional<std::string> name);
  template <typename... VARIABLES,
            typename RestrictPack = typename std::enable_if<
                xpress::AllConvertible<Variable, VARIABLES...>::value>::type>
  inline GeneralConstraintDefinitionImplementation(Variable resultant,
                                                   xpress::GenConsType type,
                                                   VARIABLES... variables);
  inline GeneralConstraintDefinitionImplementation(
      Variable resultant, xpress::GenConsType type,
      xpress::SizedArray<Variable const> const &variables);
  inline GeneralConstraintDefinitionImplementation(
      Variable resultant, xpress::GenConsType type,
      xpress::SizedArray<Variable const> const &variables,
      xpress::SizedArray<double const> const &values);

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  inline auto setName(std::optional<std::string> newName)
      -> GeneralConstraintDefinitionImplementation &;
  virtual inline auto
  getIndexHandler(xpress::objects::XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::GeneralConstraint> const
          * override;
};
class IndicatorDescriptionImplementation final
    : public ConstraintDefinitionImplementation<Inequality> {
  friend class IndicatorDescription;

public:
  virtual ~IndicatorDescriptionImplementation() {}

private:
  Variable indicatorVariable;

  bool indicatorValue;

  xpress::objects::InequalityDefinition implied;

public:
  inline IndicatorDescriptionImplementation(Variable indicatorVariable,
                                            bool indicatorValue,
                                            InequalityDefinition implied);

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  virtual inline auto getName() const -> std::optional<std::string> override;
  inline auto setName(std::optional<std::string> newName)
      -> IndicatorDescriptionImplementation &;
  virtual inline auto
  getIndexHandler(xpress::objects::XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::Inequality> const
          * override;
};
class LinExpressionImplementation
    : public xpress::objects::AbstractExpressionImplementation {
  friend class LinExpression;

public:
  virtual ~LinExpressionImplementation() {}
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;
  virtual inline auto normalizeTerms() -> LinExpressionImplementation &;
  virtual auto deepCopy() -> xpress::objects::LinExpression = 0;
  virtual auto deepCopy(double factor) -> xpress::objects::LinExpression = 0;
  virtual auto getConstant() const -> double = 0;
  virtual auto addConstant(double c) -> LinExpressionImplementation & = 0;
  virtual auto setConstant(double c) -> LinExpressionImplementation & = 0;
  virtual auto reset() -> LinExpressionImplementation & = 0;

protected:
  virtual inline auto prepareToAdd(int newTerms) -> void;

public:
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto addTerms(Strm0 const &variables, Strm1 const &coefficients)
      -> LinExpressionImplementation &;
  inline auto addTerms(xpress::SizedArray<Variable const> const &variables,
                       xpress::SizedArray<double const> const &coefficients,
                       int offset, int count) -> LinExpressionImplementation &;
  virtual auto addTerm(Variable const &variable, double const &coefficient)
      -> LinExpressionImplementation & = 0;
  virtual auto addTerm(double const &coefficient, Variable const &variable)
      -> LinExpressionImplementation & = 0;
  inline auto addTerm(LinTerm const &term) -> LinExpressionImplementation &;
  virtual auto addTerm(Variable const &x) -> LinExpressionImplementation & = 0;
  virtual auto addTerms(xpress::objects::LinExpression const &expr)
      -> LinExpressionImplementation & = 0;
  virtual auto addTerms(xpress::objects::LinExpression const &expr,
                        double factor) -> LinExpressionImplementation & = 0;
  inline auto square() -> xpress::objects::Expression;
  static inline auto create() -> xpress::objects::LinExpression;
  static inline auto create(double constant) -> xpress::objects::LinExpression;
  virtual inline auto getMaxDegree() const -> int override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
};
class LinTermImplementation final
    : public xpress::objects::AbstractExpressionImplementation {
  friend class LinTerm;

public:
  virtual ~LinTermImplementation() {}

private:
  xpress::objects::Variable variable;

public:
  auto getVariable() const -> xpress::objects::Variable { return variable; }

private:
  double coefficient;

public:
  auto getCoefficient() const -> double { return coefficient; }
  inline LinTermImplementation(Variable variable, double coefficient);
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  static inline auto lTerm(xpress::objects::Variable variable,
                           double coefficient) -> xpress::objects::LinTerm;
  static inline auto lTerm(Variable variable) -> xpress::objects::LinTerm;
  inline auto square() const -> xpress::objects::QuadTerm;
  virtual inline auto toString() const -> std::string override;
  static inline auto toString(xpress::objects::Variable v, double coef)
      -> std::string;
  virtual inline auto getMaxDegree() const -> int override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  virtual inline auto uminus() const -> xpress::objects::Expression override;
  virtual inline auto mul(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;
  inline auto mul(ConstantExpression arg) const -> xpress::objects::LinTerm;
  virtual inline auto mul(double arg) const
      -> xpress::objects::Expression override;
  inline auto mul(Variable arg) const -> xpress::objects::QuadTerm;
  inline auto mul(LinTerm arg) const -> xpress::objects::QuadTerm;
  virtual inline auto div(Expression arg) const
      -> xpress::objects::Expression override;
  inline auto div(ConstantExpression arg) const -> xpress::objects::LinTerm;
  virtual inline auto div(double arg) const
      -> xpress::objects::Expression override;
};
class PWLImplementation final
    : public xpress::objects::Index,
      public std::enable_shared_from_this<PWLImplementation> {
  friend class PWL;

public:
  virtual ~PWLImplementation() {}

  XpressProblem *prob;

  XPRSint64 const serial;

private:
  int index;

public:
  inline auto getIndex() const -> int override;
  inline auto updateIndex(int delta) -> void override;
  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;
  inline auto getProblem() const -> xpress::objects::XpressProblem * override;
  inline auto getSerial() const -> XPRSint64 override;
  inline auto markDeleted() -> void override;
  virtual inline auto compareTo(PWL const &other) const -> int final;
  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline PWLImplementation(XpressProblem *prob, long serial, int index);
  inline auto getInput() -> xpress::objects::Variable;
  inline auto getResultant() -> xpress::objects::Variable;
  inline auto getBreakpoints() -> std::vector<xpress::PwlBreakpoint>;
  inline auto getName() const -> std::string override;
  inline auto setName(std::optional<std::string> newName)
      -> PWLImplementation &;
};
class PWLDefinitionImplementation final
    : public ConstraintDefinitionImplementation<PWL> {
  friend class PWLDefinition;

public:
  virtual ~PWLDefinitionImplementation() {}

private:
  xpress::objects::Variable input;

  xpress::objects::Variable resultant;

  std::vector<double> xval;

  std::vector<double> yval;

public:
  inline PWLDefinitionImplementation(
      xpress::objects::Variable input, xpress::objects::Variable resultant,
      xpress::SizedArray<double const> const &xval,
      xpress::SizedArray<double const> const &yval,
      std::optional<std::string> name);

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  inline auto setName(std::optional<std::string> newName)
      -> PWLDefinitionImplementation &;
  virtual inline auto getIndexHandler(XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::PWL> const * override;
};
class QuadExpressionImplementation : public AbstractExpressionImplementation {
  friend class QuadExpression;

public:
  virtual ~QuadExpressionImplementation() {}
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;
  virtual inline auto normalizeTerms() -> QuadExpressionImplementation &;
  virtual auto deepCopy() -> xpress::objects::QuadExpression = 0;
  virtual auto deepCopy(double factor) -> xpress::objects::QuadExpression = 0;
  virtual auto getConstant() const -> double = 0;
  virtual auto addConstant(double c) -> QuadExpressionImplementation & = 0;
  virtual auto setConstant(double c) -> QuadExpressionImplementation & = 0;
  virtual auto reset() -> QuadExpressionImplementation & = 0;

protected:
  virtual inline auto prepareToAdd(int newTerms) -> void;

public:
  template <typename Strm0, typename Strm1, xpress_isstream(Strm0),
            xpress_isstream(Strm1)>
  inline auto addTerms(Strm0 const &variables, Strm1 const &coefficients)
      -> QuadExpressionImplementation &;
  template <typename Strm0, typename Strm1, typename Strm2,
            xpress_isstream(Strm0), xpress_isstream(Strm1),
            xpress_isstream(Strm2)>
  inline auto addTerms(Strm0 const &variables1, Strm1 const &variables2,
                       Strm2 const &coefficients)
      -> QuadExpressionImplementation &;
  inline auto addTerms(xpress::SizedArray<Variable const> const &variables,
                       xpress::SizedArray<double const> const &coefficients,
                       int offset, int count) -> QuadExpressionImplementation &;
  inline auto addTerms(xpress::SizedArray<Variable const> const &variables1,
                       xpress::SizedArray<Variable const> const &variables2,
                       xpress::SizedArray<double const> const &coefficients,
                       int offset, int count) -> QuadExpressionImplementation &;
  virtual auto addTerm(Variable const &variable, double const &coefficient)
      -> QuadExpressionImplementation & = 0;
  virtual auto addTerm(double const &coefficient, Variable const &variable)
      -> QuadExpressionImplementation & = 0;
  virtual auto addTerm(Variable const &x) -> QuadExpressionImplementation & = 0;
  inline auto addTerm(LinTerm const &term) -> QuadExpressionImplementation &;
  inline auto addTerm(QuadTerm const &term) -> QuadExpressionImplementation &;
  virtual auto addTerm(Variable const &variable1, Variable const &variable2,
                       double const &coefficient)
      -> QuadExpressionImplementation & = 0;
  virtual auto addTerm(double const &coefficient, Variable const &variable1,
                       Variable const &variable2)
      -> QuadExpressionImplementation & = 0;
  virtual auto addTerm(Variable const &x1, Variable const &x2)
      -> QuadExpressionImplementation & = 0;
  virtual auto addTerms(LinExpression const &expr)
      -> QuadExpressionImplementation & = 0;
  virtual auto addTerms(QuadExpression const &expr)
      -> QuadExpressionImplementation & = 0;
  virtual auto addTerms(LinExpression const &expr, double const &factor)
      -> QuadExpressionImplementation & = 0;
  virtual auto addTerms(QuadExpression const &expr, double const &factor)
      -> QuadExpressionImplementation & = 0;
  static inline auto create() -> xpress::objects::QuadExpression;
  static inline auto create(double constant) -> xpress::objects::QuadExpression;
  virtual inline auto getMaxDegree() const -> int override;
};
class QuadTermImplementation final
    : public xpress::objects::AbstractExpressionImplementation {
  friend class QuadTerm;

public:
  virtual ~QuadTermImplementation() {}

private:
  xpress::objects::Variable variable1;

public:
  auto getVariable1() const -> xpress::objects::Variable { return variable1; }

private:
  xpress::objects::Variable variable2;

public:
  auto getVariable2() const -> xpress::objects::Variable { return variable2; }

private:
  double coefficient;

public:
  auto getCoefficient() const -> double { return coefficient; }
  inline QuadTermImplementation(xpress::objects::Variable variable1,
                                xpress::objects::Variable variable2,
                                double coefficient);
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  static inline auto qTerm(xpress::objects::Variable variable1,
                           xpress::objects::Variable variable2,
                           double coefficient) -> xpress::objects::QuadTerm;
  static inline auto qTerm(xpress::objects::Variable variable1,
                           xpress::objects::Variable variable2)
      -> xpress::objects::QuadTerm;
  virtual inline auto toString() const -> std::string override;
  static inline auto toString(xpress::objects::Variable v1,
                              xpress::objects::Variable v2, double coef)
      -> std::string;
  virtual inline auto getMaxDegree() const -> int override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  virtual inline auto uminus() const -> xpress::objects::Expression override;
  virtual inline auto mul(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;
  inline auto mul(ConstantExpression arg) const -> xpress::objects::QuadTerm;
  virtual inline auto mul(double arg) const
      -> xpress::objects::Expression override;
  virtual inline auto div(xpress::objects::Expression arg) const
      -> xpress::objects::Expression override;
  inline auto div(xpress::objects::ConstantExpression arg) const
      -> xpress::objects::QuadTerm;
  virtual inline auto div(double arg) const
      -> xpress::objects::Expression override;
};
class SOSImplementation final
    : public xpress::objects::Index,
      public std::enable_shared_from_this<SOSImplementation> {
  friend class SOS;

public:
  virtual ~SOSImplementation() {}

  XpressProblem *prob;

  XPRSint64 const serial;

private:
  int index;

public:
  inline auto getIndex() const -> int override;
  inline auto updateIndex(int delta) -> void override;
  inline auto getIndexForProb(XpressProblem const *xprob) const -> int override;
  inline auto getProblem() const -> xpress::objects::XpressProblem * override;
  inline auto getSerial() const -> XPRSint64 override;
  inline auto markDeleted() -> void override;
  virtual inline auto compareTo(SOS const &other) const -> int final;
  virtual inline auto getHashCode() const -> std::size_t final;

private:
  virtual inline auto check(char const *operation) const -> void final;

public:
  inline SOSImplementation(XpressProblem *prob, long serial, int index);
  inline auto getName() const -> std::string override;
  inline auto setName(std::optional<std::string> newName)
      -> SOSImplementation &;
  static inline auto sos(xpress::SetType type,
                         xpress::SizedArray<Variable const> const &items,
                         xpress::SizedArray<double const> const &weights,
                         std::optional<std::string> name)
      -> xpress::objects::SOSDefinition;
  static inline auto sos1(xpress::SizedArray<Variable const> const &items,
                          xpress::SizedArray<double const> const &weights,
                          std::optional<std::string> name)
      -> xpress::objects::SOSDefinition;
  static inline auto sos2(xpress::SizedArray<Variable const> const &items,
                          xpress::SizedArray<double const> const &weights,
                          std::optional<std::string> name)
      -> xpress::objects::SOSDefinition;
};
class SOSDefinitionImplementation final
    : public ConstraintDefinitionImplementation<SOS> {
  friend class SOSDefinition;

public:
  virtual ~SOSDefinitionImplementation() {}

private:
  xpress::SetType type;

  std::vector<xpress::objects::Variable> sosElems;

  std::vector<double> sosWeights;

  bool defaultWeights;

public:
  inline SOSDefinitionImplementation(
      xpress::SetType type, xpress::SizedArray<Variable const> const &items,
      xpress::SizedArray<double const> const &weights,
      std::optional<std::string> name);

protected:
  virtual inline auto
  createInternal(xpress::XPRSProblem::ConstraintCreator *creator) const
      -> void override;

public:
  inline auto setName(std::optional<std::string> newName)
      -> SOSDefinitionImplementation &;
  virtual inline auto getIndexHandler(XpressProblem *prob) const
      -> xpress::objects::IndexHandler<xpress::objects::SOS> const * override;
};
class SumExpressionImplementation final
    : public AbstractExpressionImplementation {
  friend class SumExpression;

public:
  virtual ~SumExpressionImplementation() {}

private:
  std::vector<Expression> const terms;

  std::atomic<int> mutable maxDegree;

public:
  inline SumExpressionImplementation(
      xpress::SizedArray<Expression const> const &terms);
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto toString() const -> std::string override;
  virtual inline auto getMaxDegree() const -> int override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
};
class FormulaExpressionImplementation
    : public xpress::objects::AbstractExpressionImplementation {
  friend class FormulaExpression;

public:
  virtual ~FormulaExpressionImplementation() {}

protected:
  inline FormulaExpressionImplementation();

public:
  virtual inline auto getRTTI()
      -> xpress::objects::ExpressionRTTI final override;
  virtual auto extract(PostfixExtractor *extractor) const -> void = 0;
  virtual inline auto extract(double factor, PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool final override;

private:
  static inline ConstantExpression const dummyRB =
      std::numeric_limits<double>::quiet_NaN();

  static inline ConstantExpression const dummyCOLON =
      std::numeric_limits<double>::quiet_NaN();
  template <typename T>
  static inline auto parseUnaryFunction(SymbolicEvalStack &stack, T type)
      -> xpress::objects::InternalFunctionExpression;
  template <typename T, typename Func0,
            xpress_isinvocable(Func0, std::vector<T>, Func0, int)>
  static inline auto reverseArgs(std::vector<T> const &args, Func0 mkArray)
      -> std::vector<T>;
  template <typename T>
  static inline auto parseNaryFunction(SymbolicEvalStack &stack, T type)
      -> xpress::objects::InternalFunctionExpression;

public:
  static inline auto parse(XpressProblem *prob,
                           xpress::SizedArray<int const> const &type,
                           xpress::SizedArray<double const> const &value)
      -> xpress::objects::Expression;
};
class LinTermListImplementation final : public LinExpressionImplementation {
  friend class LinTermList;

public:
  virtual ~LinTermListImplementation() {}
  using LinExpressionImplementation::addTerm;
  using LinExpressionImplementation::addTerms;

private:
  static inline constexpr bool &autoCompress =
      xpress::GlobalConstants<void>::xpress_objects_LinTermList_autoCompress;

public:
  static inline auto setAutoCompress(bool compress) -> void;
  static inline auto isAutoCompress() -> bool;

private:
  double mutable constant;

  int mutable length;

  std::vector<xpress::objects::Variable> mutable ind;

  std::vector<double> mutable val;

public:
  inline LinTermListImplementation();
  inline LinTermListImplementation(double c);
  inline LinTermListImplementation(double c, int capacity);
  virtual inline auto deepCopy() -> xpress::objects::LinExpression override;
  virtual inline auto deepCopy(double factor)
      -> xpress::objects::LinExpression override;

protected:
  virtual inline auto prepareToAdd(int newTerms) -> void override;

public:
  inline auto reserve(int l) -> LinTermListImplementation &;
  virtual inline auto reset() -> LinTermListImplementation & override;
  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient)
      -> LinTermListImplementation & override;
  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable)
      -> LinTermListImplementation & override;
  virtual inline auto addTerm(Variable const &x)
      -> LinTermListImplementation & override;
  virtual inline auto addTerms(LinExpression const &expr)
      -> LinTermListImplementation & override;
  virtual inline auto addTerms(LinExpression const &expr, double factor)
      -> LinTermListImplementation & override;
  virtual inline auto addConstant(double c)
      -> LinTermListImplementation & override;
  virtual inline auto setConstant(double c)
      -> LinTermListImplementation & override;
  virtual inline auto getConstant() const -> double override;
  virtual inline auto normalizeTerms() -> LinTermListImplementation & override;
  inline auto compress() const -> void;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto toString() const -> std::string override;
};
class LinTermMapImplementation final : public LinExpressionImplementation {
  friend class LinTermMap;

public:
  virtual ~LinTermMapImplementation() {}
  using LinExpressionImplementation::addTerm;
  using LinExpressionImplementation::addTerms;

private:
  static inline double const ZERO = 0.0;

  std::unordered_map<xpress::objects::Variable, double> map;

public:
  inline LinTermMapImplementation();
  inline LinTermMapImplementation(double constant);
  virtual inline auto deepCopy() -> xpress::objects::LinExpression override;
  virtual inline auto deepCopy(double factor)
      -> xpress::objects::LinExpression override;
  virtual inline auto reset() -> LinTermMapImplementation & override;
  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient)
      -> LinTermMapImplementation & override;
  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable)
      -> LinTermMapImplementation & override;
  virtual inline auto addTerm(Variable const &x)
      -> LinTermMapImplementation & override;
  virtual inline auto addTerms(LinExpression const &expr)
      -> LinTermMapImplementation & override;
  virtual inline auto addTerms(LinExpression const &expr, double factor)
      -> LinTermMapImplementation & override;
  inline auto setCoefficient(Variable variable, double coefficient)
      -> LinTermMapImplementation &;
  inline auto getCoefficient(Variable variable) const -> double;
  virtual inline auto addConstant(double c)
      -> LinTermMapImplementation & override;
  virtual inline auto setConstant(double c)
      -> LinTermMapImplementation & override;
  virtual inline auto getConstant() const -> double override;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto
  extract(double factor, xpress::objects::PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto toString() const -> std::string override;
};
class QuadTermListImplementation final : public QuadExpressionImplementation {
  friend class QuadTermList;

public:
  virtual ~QuadTermListImplementation() {}
  using QuadExpressionImplementation::addTerm;
  using QuadExpressionImplementation::addTerms;

private:
  static inline constexpr bool &autoCompress =
      xpress::GlobalConstants<void>::xpress_objects_QuadTermList_autoCompress;

public:
  static inline auto setAutoCompress(bool compress) -> void;
  static inline auto isAutoCompress() -> bool;

private:
  double mutable constant;

  int mutable length;

  std::vector<xpress::objects::Variable> mutable ind1;

  std::vector<xpress::objects::Variable> mutable ind2;

  std::vector<double> mutable val;

public:
  inline QuadTermListImplementation();
  inline QuadTermListImplementation(double c);
  inline QuadTermListImplementation(double c, int capacity);
  virtual inline auto deepCopy() -> xpress::objects::QuadExpression override;
  virtual inline auto deepCopy(double factor)
      -> xpress::objects::QuadExpression override;
  inline auto reserve(int l) -> QuadTermListImplementation &;
  virtual inline auto reset() -> QuadTermListImplementation & override;
  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerm(Variable const &x)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerm(Variable const &variable1,
                              Variable const &variable2,
                              double const &coefficient)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable1,
                              Variable const &variable2)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerm(Variable const &x1, Variable const &x2)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerms(LinExpression const &expr)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerms(QuadExpression const &expr)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerms(LinExpression const &expr, double const &factor)
      -> QuadTermListImplementation & override;
  virtual inline auto addTerms(QuadExpression const &expr, double const &factor)
      -> QuadTermListImplementation & override;
  virtual inline auto addConstant(double c)
      -> QuadTermListImplementation & override;
  virtual inline auto setConstant(double c)
      -> QuadTermListImplementation & override;
  virtual inline auto getConstant() const -> double override;
  virtual inline auto normalizeTerms() -> QuadTermListImplementation & override;
  inline auto compress() const -> void;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto extract(double factor, PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  virtual inline auto toString() const -> std::string override;
};
class QuadTermMapImplementation final : public QuadExpressionImplementation {
  friend class QuadTermMap;

public:
  virtual ~QuadTermMapImplementation() {}
  using QuadExpressionImplementation::addTerm;
  using QuadExpressionImplementation::addTerms;

private:
  static inline double const ZERO = 0.0;

  std::unordered_map<QPair, double> map;

public:
  inline QuadTermMapImplementation();
  inline QuadTermMapImplementation(double constant);
  virtual inline auto deepCopy() -> xpress::objects::QuadExpression override;
  virtual inline auto deepCopy(double factor)
      -> xpress::objects::QuadExpression override;
  virtual inline auto reset() -> QuadTermMapImplementation & override;
  virtual inline auto addTerm(Variable const &variable,
                              double const &coefficient)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerm(Variable const &x)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerm(Variable const &variable1,
                              Variable const &variable2,
                              double const &coefficient)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerm(double const &coefficient,
                              Variable const &variable1,
                              Variable const &variable2)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerm(Variable const &x1, Variable const &x2)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerms(LinExpression const &expr)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerms(QuadExpression const &expr)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerms(LinExpression const &expr, double const &factor)
      -> QuadTermMapImplementation & override;
  virtual inline auto addTerms(QuadExpression const &expr, double const &factor)
      -> QuadTermMapImplementation & override;
  inline auto setCoefficient(Variable variable, double coefficient)
      -> QuadTermMapImplementation &;
  inline auto setCoefficient(Variable variable1, Variable variable2,
                             double coefficient) -> QuadTermMapImplementation &;
  inline auto getCoefficient(Variable variable) const -> double;
  inline auto getCoefficient(Variable variable1, Variable variable2) const
      -> double;
  virtual inline auto addConstant(double c)
      -> QuadTermMapImplementation & override;
  virtual inline auto setConstant(double c)
      -> QuadTermMapImplementation & override;
  virtual inline auto getConstant() const -> double override;
  virtual inline auto extract(double factor,
                              xpress::XPRSProblem::RowCreator *creator) const
      -> bool override;
  virtual inline auto extract(double factor, PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  virtual inline auto toString() const -> std::string override;
};
class BinaryExpressionImplementation final
    : public FormulaExpressionImplementation {
  friend class BinaryExpression;

public:
  virtual ~BinaryExpressionImplementation() {}

private:
  xpress::objects::Expression left;

public:
  auto getLeft() const -> xpress::objects::Expression { return left; }

private:
  xpress::objects::BinaryExpression::Operator op;

public:
  auto getOp() const -> xpress::objects::BinaryExpression::Operator {
    return op;
  }

private:
  xpress::objects::Expression right;

public:
  auto getRight() const -> xpress::objects::Expression { return right; }

private:
  std::atomic<int> mutable maxDegree;

public:
  inline BinaryExpressionImplementation(
      xpress::objects::Expression left,
      xpress::objects::BinaryExpression::Operator op,
      xpress::objects::Expression right);
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto toString() const -> std::string override;
  virtual inline auto getMaxDegree() const -> int override;
  virtual inline auto getConstantView() const -> double override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
};
class InternalFunctionExpressionImplementation final
    : public FormulaExpressionImplementation {
  friend class InternalFunctionExpression;

public:
  virtual ~InternalFunctionExpressionImplementation() {}

private:
  static inline constexpr XPRSint64 const unary =
      (((XPRSint64)1) << XPRS_IFUN_LOG10) | (((XPRSint64)1) << XPRS_IFUN_LN) |
      (((XPRSint64)1) << XPRS_IFUN_EXP) | (((XPRSint64)1) << XPRS_IFUN_ABS) |
      (((XPRSint64)1) << XPRS_IFUN_SQRT) | (((XPRSint64)1) << XPRS_IFUN_SIN) |
      (((XPRSint64)1) << XPRS_IFUN_COS) | (((XPRSint64)1) << XPRS_IFUN_TAN) |
      (((XPRSint64)1) << XPRS_IFUN_ARCSIN) |
      (((XPRSint64)1) << XPRS_IFUN_ARCCOS) |
      (((XPRSint64)1) << XPRS_IFUN_ARCTAN) | (((XPRSint64)1) << XPRS_IFUN_SIGN);
  xpress::objects::InternalFunctionExpression::FunctionType function;

public:
  auto getFunction() const
      -> xpress::objects::InternalFunctionExpression::FunctionType {
    return function;
  }

private:
  std::vector<xpress::objects::Expression> arguments;

public:
  auto getArguments() const -> std::vector<xpress::objects::Expression> {
    return arguments;
  }
  inline InternalFunctionExpressionImplementation(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      std::vector<xpress::objects::Expression> arguments);
  inline InternalFunctionExpressionImplementation(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      xpress::objects::Expression argument);
  inline InternalFunctionExpressionImplementation(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      xpress::objects::Expression firstArgument,
      xpress::objects::Expression secondArgument);

private:
  static inline auto
  collectArguments(Expression firstArgument, Expression secondArgument,
                   xpress::SizedArray<Expression const> const &more)
      -> std::vector<xpress::objects::Expression>;

public:
  template <
      typename... MOREARGUMENTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::objects::Expression, MOREARGUMENTS...>::value>::type>
  inline InternalFunctionExpressionImplementation(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      xpress::objects::Expression firstArgument,
      xpress::objects::Expression secondArgument,
      MOREARGUMENTS... moreArguments);
  inline InternalFunctionExpressionImplementation(
      xpress::objects::InternalFunctionExpression::FunctionType function,
      xpress::SizedArray<Expression const> const &args);
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto toString() const -> std::string override;
  virtual inline auto getMaxDegree() const -> int override;
};
class PWLExpressionImplementation : public FormulaExpressionImplementation {
  friend class PWLExpression;

public:
  virtual ~PWLExpressionImplementation() {}

private:
  static inline auto
  breakToX(xpress::SizedArray<PwlBreakpoint const> const &points)
      -> std::vector<double>;
  static inline auto
  breakToY(xpress::SizedArray<PwlBreakpoint const> const &points)
      -> std::vector<double>;
  int argumentColumn;

public:
  auto getArgumentColumn() const -> int { return argumentColumn; }

private:
  xpress::objects::Variable argumentVariable;

public:
  auto getArgumentVariable() const -> xpress::objects::Variable {
    return argumentVariable;
  }

private:
  std::vector<double> breakX;

public:
  auto getBreakX() const -> std::vector<double> { return breakX; }

private:
  std::vector<double> breakY;

public:
  auto getBreakY() const -> std::vector<double> { return breakY; }
  inline PWLExpressionImplementation(
      Variable argument, xpress::SizedArray<double const> const &breakX,
      xpress::SizedArray<double const> const &breakY);
  template <
      typename... POINTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::PwlBreakpoint, POINTS...>::value>::type>
  inline PWLExpressionImplementation(Variable argument, POINTS... points);
  inline PWLExpressionImplementation(
      Variable argument, xpress::SizedArray<PwlBreakpoint const> const &points);
  inline PWLExpressionImplementation(
      int argument, xpress::SizedArray<double const> const &breakX,
      xpress::SizedArray<double const> const &breakY);
  template <
      typename... POINTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::PwlBreakpoint, POINTS...>::value>::type>
  inline PWLExpressionImplementation(int argument, POINTS... points);
  inline PWLExpressionImplementation(
      int argument, xpress::SizedArray<PwlBreakpoint const> const &points);
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto toString() const -> std::string override;
  virtual inline auto getMaxDegree() const -> int override;
};
class UnaryExpressionImplementation final
    : public FormulaExpressionImplementation {
  friend class UnaryExpression;

public:
  virtual ~UnaryExpressionImplementation() {}

private:
  xpress::objects::UnaryExpression::Operator op;

public:
  auto getOp() const -> xpress::objects::UnaryExpression::Operator {
    return op;
  }

private:
  xpress::objects::Expression argument;

public:
  auto getArgument() const -> xpress::objects::Expression { return argument; }
  inline UnaryExpressionImplementation(
      xpress::objects::UnaryExpression::Operator op,
      xpress::objects::Expression argument);
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto toString() const -> std::string override;
  virtual inline auto getMaxDegree() const -> int override;
  virtual inline auto getConstantView() const -> double override;
  virtual inline auto getLinearView() const
      -> std::shared_ptr<xpress::objects::LinearView> override;
  virtual inline auto getQuadView() const
      -> std::shared_ptr<xpress::objects::QuadView> override;
  static inline auto scaledLinView(std::shared_ptr<LinearView> data,
                                   double factor)
      -> std::shared_ptr<xpress::objects::LinearView>;
  static inline auto scaledQuadView(std::shared_ptr<QuadView> data,
                                    double factor)
      -> std::shared_ptr<xpress::objects::QuadView>;
};
class UserFunctionExpressionImplementation
    : public FormulaExpressionImplementation {
  friend class UserFunctionExpression;

public:
  virtual ~UserFunctionExpressionImplementation() {}

private:
  xpress::XPRSProblem::AbstractUserFunction *function;

public:
  auto getFunction() const -> xpress::XPRSProblem::AbstractUserFunction * {
    return function;
  }

private:
  std::vector<xpress::objects::Expression> arguments;

public:
  auto getArguments() const -> std::vector<xpress::objects::Expression> {
    return arguments;
  }

private:
  int output;

public:
  auto getOutput() const -> int { return output; }
  template <
      typename... ARGUMENTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::objects::Expression, ARGUMENTS...>::value>::type>
  inline UserFunctionExpressionImplementation(
      xpress::XPRSProblem::AbstractUserFunction *function,
      ARGUMENTS... arguments);
  template <
      typename... ARGUMENTS,
      typename RestrictPack = typename std::enable_if<xpress::AllConvertible<
          xpress::objects::Expression, ARGUMENTS...>::value>::type>
  inline UserFunctionExpressionImplementation(
      xpress::XPRSProblem::AbstractUserFunction *function, int output,
      ARGUMENTS... arguments);
  inline UserFunctionExpressionImplementation(
      xpress::XPRSProblem::AbstractUserFunction *function,
      SizedArray<Expression> const &arguments);
  inline UserFunctionExpressionImplementation(
      xpress::XPRSProblem::AbstractUserFunction *function, int output,
      SizedArray<Expression> const &arguments);
  virtual inline auto
  evaluate(xpress::SizedArray<double const> const &solution) const
      -> double override;
  virtual inline auto extract(PostfixExtractor *extractor) const
      -> void override;
  virtual inline auto toString() const -> std::string override;
  virtual inline auto getMaxDegree() const -> int override;
};
} // namespace objects
} // namespace xpress
template <typename C>
xpress::objects::ConstraintDefinition<C>::ConstraintDefinition(
    std::shared_ptr<ConstraintDefinitionImplementation<C>> init)
    : implPointer(init) {}
template <typename C>
xpress::objects::ConstraintDefinition<C>::ConstraintDefinition(
    ConstraintDefinitionImplementation<C> const *init)
    : implPointer(
          init ? const_cast<ConstraintDefinitionImplementation<C> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ConstraintDefinitionImplementation<C>>(
                     nullptr)) {}
template <typename C>
xpress::objects::ConstraintDefinitionImplementation<C> *
xpress::objects::ConstraintDefinition<C>::impl() {
  return dynamic_cast<ConstraintDefinitionImplementation<C> *>(
      implPointer.get());
}
template <typename C>
xpress::objects::ConstraintDefinitionImplementation<C> const *
xpress::objects::ConstraintDefinition<C>::impl() const {
  return dynamic_cast<ConstraintDefinitionImplementation<C> *>(
      implPointer.get());
}
template <typename C>
auto xpress::objects::ConstraintDefinition<C>::checkModify() -> void {
  impl()->checkModify();
}
template <typename C>
auto xpress::objects::ConstraintDefinition<C>::getName() const
    -> std::optional<std::string> {
  return impl()->getName();
}
template <typename C>
auto xpress::objects::ConstraintDefinition<C>::setConstraintName(
    std::optional<std::string> newName)
    -> xpress::objects::ConstraintDefinition<C> & {
  impl()->setConstraintName(newName);
  return *this;
}
template <typename C>
auto xpress::objects::ConstraintDefinition<C>::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  impl()->createInternal(creator);
}
template <typename C>
auto xpress::objects::ConstraintDefinition<C>::create(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  impl()->create(creator);
}
template <typename C>
auto xpress::objects::ConstraintDefinition<C>::getIndexHandler(
    XpressProblem *prob) const -> xpress::objects::IndexHandler<C> const * {
  return impl()->getIndexHandler(prob);
}
template <typename C>
xpress::objects::ConstraintDefinition<C>::ConstraintDefinition()
    : implPointer(nullptr) {}
xpress::objects::Expression::Expression(
    std::shared_ptr<ExpressionImplementation> init)
    : implPointer(init) {}
xpress::objects::Expression::Expression(ExpressionImplementation const *init)
    : implPointer(
          init
              ? const_cast<ExpressionImplementation *>(init)->shared_from_this()
              : std::shared_ptr<ExpressionImplementation>(nullptr)) {}
xpress::objects::ExpressionImplementation *xpress::objects::Expression::impl() {
  return dynamic_cast<ExpressionImplementation *>(implPointer.get());
}
xpress::objects::ExpressionImplementation const *
xpress::objects::Expression::impl() const {
  return dynamic_cast<ExpressionImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::Expression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::Expression::leq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->leq(rhs);
}
auto xpress::objects::Expression::leq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->leq(rhs);
}
auto xpress::objects::Expression::geq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->geq(rhs);
}
auto xpress::objects::Expression::geq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->geq(rhs);
}
auto xpress::objects::Expression::eq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->eq(rhs);
}
auto xpress::objects::Expression::eq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->eq(rhs);
}
auto xpress::objects::Expression::in(double lb, double ub)
    -> xpress::objects::InequalityDefinition {
  return impl()->in(lb, ub);
}
auto xpress::objects::Expression::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::Expression::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::Expression::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::Expression::getRTTI() -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::Expression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::Expression::getConstantView() const -> double {
  return impl()->getConstantView();
}
auto xpress::objects::Expression::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::Expression::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::Expression::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::Expression::uminus() const
    -> xpress::objects::Expression {
  return impl()->uminus();
}
auto xpress::objects::Expression::plus(xpress::objects::Expression arg) const
    -> xpress::objects::Expression {
  return impl()->plus(arg);
}
auto xpress::objects::Expression::plus(double arg) const
    -> xpress::objects::Expression {
  return impl()->plus(arg);
}
auto xpress::objects::Expression::minus(xpress::objects::Expression arg) const
    -> xpress::objects::Expression {
  return impl()->minus(arg);
}
auto xpress::objects::Expression::minus(double arg) const
    -> xpress::objects::Expression {
  return impl()->minus(arg);
}
auto xpress::objects::Expression::mul(Expression arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::Expression::mul(double arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::Expression::div(Expression arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
auto xpress::objects::Expression::div(double arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
xpress::objects::Expression::Expression() : implPointer(nullptr) {}
xpress::objects::Inequality::Inequality(
    std::shared_ptr<InequalityImplementation> init)
    : implPointer(init) {}
xpress::objects::Inequality::Inequality(InequalityImplementation const *init)
    : implPointer(
          init
              ? const_cast<InequalityImplementation *>(init)->shared_from_this()
              : std::shared_ptr<InequalityImplementation>(nullptr)) {}
xpress::objects::InequalityImplementation *xpress::objects::Inequality::impl() {
  return dynamic_cast<InequalityImplementation *>(implPointer.get());
}
xpress::objects::InequalityImplementation const *
xpress::objects::Inequality::impl() const {
  return dynamic_cast<InequalityImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::Inequality::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::Inequality::getIndex() const -> int {
  return impl()->getIndex();
}
auto xpress::objects::Inequality::updateIndex(int delta) -> void {
  impl()->updateIndex(delta);
}
auto xpress::objects::Inequality::getIndexForProb(
    XpressProblem const *xprob) const -> int {
  return impl()->getIndexForProb(xprob);
}
auto xpress::objects::Inequality::getProblem() const
    -> xpress::objects::XpressProblem * {
  return impl()->getProblem();
}
auto xpress::objects::Inequality::getSerial() const -> XPRSint64 {
  return impl()->getSerial();
}
auto xpress::objects::Inequality::markDeleted() -> void {
  impl()->markDeleted();
}
auto xpress::objects::Inequality::compareTo(Inequality const &other) const
    -> int {
  return impl()->compareTo(other);
}
auto xpress::objects::Inequality::getHashCode() const -> std::size_t {
  return impl()->getHashCode();
}
auto xpress::objects::Inequality::check(char const *operation) const -> void {
  impl()->check(operation);
}
xpress::objects::Inequality::Inequality(XpressProblem *prob, long serial,
                                        int index)
    : implPointer(
          std::make_shared<InequalityImplementation>(prob, serial, index)) {}
auto xpress::objects::Inequality::getRhs() -> double {
  return impl()->getRhs();
}
auto xpress::objects::Inequality::getRhsRange() -> double {
  return impl()->getRhsRange();
}
auto xpress::objects::Inequality::getName() const -> std::string {
  return impl()->getName();
}
auto xpress::objects::Inequality::getType() -> xpress::RowType {
  return impl()->getType();
}
auto xpress::objects::Inequality::setRhs(double newRhs)
    -> xpress::objects::Inequality & {
  impl()->setRhs(newRhs);
  return *this;
}
auto xpress::objects::Inequality::setRhsRange(double newRange)
    -> xpress::objects::Inequality & {
  impl()->setRhsRange(newRange);
  return *this;
}
auto xpress::objects::Inequality::setType(xpress::RowType newType)
    -> xpress::objects::Inequality & {
  impl()->setType(newType);
  return *this;
}
auto xpress::objects::Inequality::setName(std::optional<std::string> newName)
    -> xpress::objects::Inequality & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::Inequality::getValue(
    xpress::SizedArray<double const> const &data) const -> double {
  return impl()->getValue(data);
}
auto xpress::objects::Inequality::getSlack() const -> double {
  return impl()->getSlack();
}
auto xpress::objects::Inequality::getDual() const -> double {
  return impl()->getDual();
}
auto xpress::objects::Inequality::getCallbackSlack() -> double {
  return impl()->getCallbackSlack();
}
auto xpress::objects::Inequality::getCallbackDual() -> double {
  return impl()->getCallbackDual();
}
auto xpress::objects::Inequality::chgCoef(Variable v, double newCoef)
    -> xpress::objects::Inequality & {
  impl()->chgCoef(v, newCoef);
  return *this;
}
auto xpress::objects::Inequality::chgCoefs(
    xpress::SizedArray<Variable const> const &v,
    xpress::SizedArray<double const> const &newCoef)
    -> xpress::objects::Inequality & {
  impl()->chgCoefs(v, newCoef);
  return *this;
}
auto xpress::objects::Inequality::getLhs() -> xpress::objects::Expression {
  return impl()->getLhs();
}
xpress::objects::Inequality::Inequality() : implPointer(nullptr) {}
xpress::objects::AbstractExpression::AbstractExpression(
    std::shared_ptr<AbstractExpressionImplementation> init)
    : xpress::objects::Expression(init) {}
xpress::objects::AbstractExpression::AbstractExpression(
    AbstractExpressionImplementation const *init)
    : xpress::objects::Expression(init) {}
xpress::objects::AbstractExpressionImplementation *
xpress::objects::AbstractExpression::impl() {
  return dynamic_cast<AbstractExpressionImplementation *>(implPointer.get());
}
xpress::objects::AbstractExpressionImplementation const *
xpress::objects::AbstractExpression::impl() const {
  return dynamic_cast<AbstractExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::AbstractExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::AbstractExpression::getConstantView() const -> double {
  return impl()->getConstantView();
}
auto xpress::objects::AbstractExpression::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::AbstractExpression::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::AbstractExpression::leq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->leq(rhs);
}
auto xpress::objects::AbstractExpression::leq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->leq(rhs);
}
auto xpress::objects::AbstractExpression::geq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->geq(rhs);
}
auto xpress::objects::AbstractExpression::geq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->geq(rhs);
}
auto xpress::objects::AbstractExpression::eq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->eq(rhs);
}
auto xpress::objects::AbstractExpression::eq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->eq(rhs);
}
auto xpress::objects::AbstractExpression::in(double lb, double ub)
    -> xpress::objects::InequalityDefinition {
  return impl()->in(lb, ub);
}
xpress::objects::AbstractExpression::AbstractExpression()
    : xpress::objects::Expression(nullptr) {}
xpress::objects::InequalityDefinition::InequalityDefinition(
    std::shared_ptr<InequalityDefinitionImplementation> init)
    : ConstraintDefinition<Inequality>(init) {}
xpress::objects::InequalityDefinition::InequalityDefinition(
    InequalityDefinitionImplementation const *init)
    : ConstraintDefinition<Inequality>(init) {}
xpress::objects::InequalityDefinitionImplementation *
xpress::objects::InequalityDefinition::impl() {
  return dynamic_cast<InequalityDefinitionImplementation *>(implPointer.get());
}
xpress::objects::InequalityDefinitionImplementation const *
xpress::objects::InequalityDefinition::impl() const {
  return dynamic_cast<InequalityDefinitionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::InequalityDefinition::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::InequalityDefinition::getLhs() const
    -> xpress::objects::Expression {
  return impl()->getLhs();
}
auto xpress::objects::InequalityDefinition::getType() const -> xpress::RowType {
  return impl()->getType();
}
auto xpress::objects::InequalityDefinition::getRhs() const
    -> xpress::objects::Expression {
  return impl()->getRhs();
}
auto xpress::objects::InequalityDefinition::getRange() const
    -> std::vector<double> {
  return impl()->getRange();
}
xpress::objects::InequalityDefinition::InequalityDefinition(
    xpress::objects::Expression lhs, xpress::RowType type,
    xpress::objects::Expression rhs, std::optional<std::string> name)
    : ConstraintDefinition<Inequality>(
          std::make_shared<InequalityDefinitionImplementation>(lhs, type, rhs,
                                                               name)) {}
xpress::objects::InequalityDefinition::InequalityDefinition(
    xpress::objects::Expression lhs, xpress::RowType type, double rhs,
    std::optional<std::string> name)
    : ConstraintDefinition<Inequality>(
          std::make_shared<InequalityDefinitionImplementation>(lhs, type, rhs,
                                                               name)) {}
xpress::objects::InequalityDefinition::InequalityDefinition(
    double lhs, xpress::RowType type, xpress::objects::Expression rhs,
    std::optional<std::string> name)
    : ConstraintDefinition<Inequality>(
          std::make_shared<InequalityDefinitionImplementation>(lhs, type, rhs,
                                                               name)) {}
xpress::objects::InequalityDefinition::InequalityDefinition(
    Expression expr, double lb, double ub, std::optional<std::string> name)
    : ConstraintDefinition<Inequality>(
          std::make_shared<InequalityDefinitionImplementation>(expr, lb, ub,
                                                               name)) {}
auto xpress::objects::InequalityDefinition::createRow(
    xpress::XPRSProblem::RowCreator *creator) const -> void {
  impl()->createRow(creator);
}
auto xpress::objects::InequalityDefinition::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  impl()->createInternal(creator);
}
auto xpress::objects::InequalityDefinition::setName(
    std::optional<std::string> newName)
    -> xpress::objects::InequalityDefinition & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::InequalityDefinition::getIndexHandler(
    XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::Inequality> const * {
  return impl()->getIndexHandler(prob);
}
xpress::objects::InequalityDefinition::InequalityDefinition()
    : ConstraintDefinition<Inequality>(nullptr) {}
xpress::objects::Variable::Variable(
    std::shared_ptr<VariableImplementation> init)
    : xpress::objects::Expression(init) {}
xpress::objects::Variable::Variable(VariableImplementation const *init)
    : xpress::objects::Expression(init) {}
xpress::objects::VariableImplementation *xpress::objects::Variable::impl() {
  return dynamic_cast<VariableImplementation *>(implPointer.get());
}
xpress::objects::VariableImplementation const *
xpress::objects::Variable::impl() const {
  return dynamic_cast<VariableImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::Variable::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::Variable::getIndex() const -> int {
  return impl()->getIndex();
}
auto xpress::objects::Variable::updateIndex(int delta) -> void {
  impl()->updateIndex(delta);
}
auto xpress::objects::Variable::getIndexForProb(
    XpressProblem const *xprob) const -> int {
  return impl()->getIndexForProb(xprob);
}
auto xpress::objects::Variable::getProblem() const
    -> xpress::objects::XpressProblem * {
  return impl()->getProblem();
}
auto xpress::objects::Variable::getSerial() const -> XPRSint64 {
  return impl()->getSerial();
}
auto xpress::objects::Variable::markDeleted() -> void { impl()->markDeleted(); }
auto xpress::objects::Variable::compareTo(Variable const &other) const -> int {
  return impl()->compareTo(other);
}
auto xpress::objects::Variable::getHashCode() const -> std::size_t {
  return impl()->getHashCode();
}
auto xpress::objects::Variable::check(char const *operation) const -> void {
  impl()->check(operation);
}
xpress::objects::Variable::Variable(XpressProblem *prob, long serial, int index)
    : xpress::objects::Expression(
          std::make_shared<VariableImplementation>(prob, serial, index)) {}
auto xpress::objects::Variable::getRTTI() -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::Variable::leq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->leq(rhs);
}
auto xpress::objects::Variable::leq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->leq(rhs);
}
auto xpress::objects::Variable::geq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->geq(rhs);
}
auto xpress::objects::Variable::geq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->geq(rhs);
}
auto xpress::objects::Variable::eq(double rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->eq(rhs);
}
auto xpress::objects::Variable::eq(Expression const &rhs) const
    -> xpress::objects::InequalityDefinition {
  return impl()->eq(rhs);
}
auto xpress::objects::Variable::in(double lb, double ub)
    -> xpress::objects::InequalityDefinition {
  return impl()->in(lb, ub);
}
auto xpress::objects::Variable::square() -> xpress::objects::QuadTerm {
  return impl()->square();
}
auto xpress::objects::Variable::getLB() const -> double {
  return impl()->getLB();
}
auto xpress::objects::Variable::getUB() const -> double {
  return impl()->getUB();
}
auto xpress::objects::Variable::getName() const -> std::string {
  return impl()->getName();
}
auto xpress::objects::Variable::getType() const -> xpress::ColumnType {
  return impl()->getType();
}
auto xpress::objects::Variable::fix(double value)
    -> xpress::objects::Variable & {
  impl()->fix(value);
  return *this;
}
auto xpress::objects::Variable::setLB(double newLB)
    -> xpress::objects::Variable & {
  impl()->setLB(newLB);
  return *this;
}
auto xpress::objects::Variable::setUB(double newUB)
    -> xpress::objects::Variable & {
  impl()->setUB(newUB);
  return *this;
}
auto xpress::objects::Variable::setType(xpress::ColumnType newType)
    -> xpress::objects::Variable & {
  impl()->setType(newType);
  return *this;
}
auto xpress::objects::Variable::setLimit(double newLimit)
    -> xpress::objects::Variable & {
  impl()->setLimit(newLimit);
  return *this;
}
auto xpress::objects::Variable::setName(std::optional<std::string> newName)
    -> xpress::objects::Variable & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::Variable::getValue(
    xpress::SizedArray<double const> const &data) const -> double {
  return impl()->getValue(data);
}
auto xpress::objects::Variable::getSolution() const -> double {
  return impl()->getSolution();
}
auto xpress::objects::Variable::getRedCost() const -> double {
  return impl()->getRedCost();
}
auto xpress::objects::Variable::getCallbackSolution() -> double {
  return impl()->getCallbackSolution();
}
auto xpress::objects::Variable::getCallbackRedCost() -> double {
  return impl()->getCallbackRedCost();
}
auto xpress::objects::Variable::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::Variable::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::Variable::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::Variable::chgCoef(Inequality r, double newCoef) -> void {
  impl()->chgCoef(r, newCoef);
}
auto xpress::objects::Variable::chgCoefs(
    xpress::SizedArray<Inequality const> const &r,
    xpress::SizedArray<double const> const &newCoef) -> void {
  impl()->chgCoefs(r, newCoef);
}
auto xpress::objects::Variable::chgObj(double newCoef)
    -> xpress::objects::Variable & {
  impl()->chgObj(newCoef);
  return *this;
}
auto xpress::objects::Variable::getObj() const -> double {
  return impl()->getObj();
}
auto xpress::objects::Variable::chgObjN(int objidx, double newCoef)
    -> xpress::objects::Variable & {
  impl()->chgObjN(objidx, newCoef);
  return *this;
}
auto xpress::objects::Variable::getObjN(int objIdx) const -> double {
  return impl()->getObjN(objIdx);
}
auto xpress::objects::Variable::pwlOf(
    Variable input, xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval) const
    -> xpress::objects::PWLDefinition {
  return impl()->pwlOf(input, xval, yval);
}
auto xpress::objects::Variable::pwlOf(
    Variable input, xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval,
    std::optional<std::string> name) const -> xpress::objects::PWLDefinition {
  return impl()->pwlOf(input, xval, yval, name);
}
template <typename... BREAKPOINTS, typename RestrictPack>
auto xpress::objects::Variable::pwlOf(Variable input,
                                      BREAKPOINTS... breakpoints) const
    -> xpress::objects::PWLDefinition {
  return impl()->pwlOf(input, breakpoints...);
}
template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::Variable::pwlOf(Variable input,
                                      Strm0 const &breakpoints) const
    -> xpress::objects::PWLDefinition {
  return impl()->pwlOf(input, breakpoints);
}
template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::Variable::pwlOf(Variable input, Strm0 const &breakpoints,
                                      std::optional<std::string> name) const
    -> xpress::objects::PWLDefinition {
  return impl()->pwlOf(input, breakpoints, name);
}
template <typename... VALUES, typename RestrictPack>
auto xpress::objects::Variable::maxOf(
    xpress::SizedArray<Variable const> const &variables, VALUES... values) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->maxOf(variables, values...);
}
template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::Variable::maxOf(Strm0 const &variables,
                                      Strm1 const &values,
                                      std::optional<std::string> name) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->maxOf(variables, values, name);
}
template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::Variable::maxOf(Strm0 const &variables,
                                      Strm1 const &values) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->maxOf(variables, values);
}
template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::Variable::maxOf(VARIABLES... variables) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->maxOf(variables...);
}
template <typename... VALUES, typename RestrictPack>
auto xpress::objects::Variable::minOf(
    xpress::SizedArray<Variable const> const &variables, VALUES... values) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->minOf(variables, values...);
}
template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::Variable::minOf(Strm0 const &variables,
                                      Strm1 const &values,
                                      std::optional<std::string> name) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->minOf(variables, values, name);
}
template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::Variable::minOf(Strm0 const &variables,
                                      Strm1 const &values) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->minOf(variables, values);
}
template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::Variable::minOf(VARIABLES... variables) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->minOf(variables...);
}
template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::Variable::orOf(Strm0 const &variables,
                                     std::optional<std::string> name) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->orOf(variables, name);
}
template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::Variable::orOf(Strm0 const &variables) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->orOf(variables);
}
template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::Variable::orOf(VARIABLES... variables) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->orOf(variables...);
}
template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::Variable::andOf(Strm0 const &variables,
                                      std::optional<std::string> name) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->andOf(variables, name);
}
template <typename Strm0, typename Strm0IsStream>
auto xpress::objects::Variable::andOf(Strm0 const &variables) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->andOf(variables);
}
template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::Variable::andOf(VARIABLES... variables) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->andOf(variables...);
}
auto xpress::objects::Variable::absOf(Variable x) const
    -> xpress::objects::GeneralConstraintDefinition {
  return impl()->absOf(x);
}
auto xpress::objects::Variable::ifThen(InequalityDefinition implied) const
    -> xpress::objects::IndicatorDescription {
  return impl()->ifThen(implied);
}
auto xpress::objects::Variable::ifNotThen(InequalityDefinition implied) const
    -> xpress::objects::IndicatorDescription {
  return impl()->ifNotThen(implied);
}
auto xpress::objects::Variable::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::Variable::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::Variable::getConstantView() const -> double {
  return impl()->getConstantView();
}
auto xpress::objects::Variable::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::Variable::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::Variable::uminus() const -> xpress::objects::Expression {
  return impl()->uminus();
}
auto xpress::objects::Variable::mul(Expression arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::Variable::mul(ConstantExpression arg) const
    -> xpress::objects::LinTerm {
  return impl()->mul(arg);
}
auto xpress::objects::Variable::mul(double arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::Variable::mul(Variable arg) const
    -> xpress::objects::QuadTerm {
  return impl()->mul(arg);
}
auto xpress::objects::Variable::mul(LinTerm arg) const
    -> xpress::objects::QuadTerm {
  return impl()->mul(arg);
}
auto xpress::objects::Variable::div(xpress::objects::Expression arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
auto xpress::objects::Variable::div(ConstantExpression arg) const
    -> xpress::objects::LinTerm {
  return impl()->div(arg);
}
auto xpress::objects::Variable::div(double arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
xpress::objects::Variable::Variable() : xpress::objects::Expression(nullptr) {}
xpress::objects::ConstantExpression::ConstantExpression(
    std::shared_ptr<ConstantExpressionImplementation> init)
    : AbstractExpression(init) {}
xpress::objects::ConstantExpression::ConstantExpression(
    ConstantExpressionImplementation const *init)
    : AbstractExpression(init) {}
xpress::objects::ConstantExpressionImplementation *
xpress::objects::ConstantExpression::impl() {
  return dynamic_cast<ConstantExpressionImplementation *>(implPointer.get());
}
xpress::objects::ConstantExpressionImplementation const *
xpress::objects::ConstantExpression::impl() const {
  return dynamic_cast<ConstantExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::ConstantExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::ConstantExpression::getValue() const -> double {
  return impl()->getValue();
}
xpress::objects::ConstantExpression::ConstantExpression(double value)
    : AbstractExpression(
          std::make_shared<ConstantExpressionImplementation>(value)) {}
auto xpress::objects::ConstantExpression::constant(double cnst)
    -> xpress::objects::ConstantExpression {
  return ConstantExpressionImplementation::constant(cnst);
}
auto xpress::objects::ConstantExpression::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::ConstantExpression::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::ConstantExpression::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::ConstantExpression::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::ConstantExpression::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::ConstantExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::ConstantExpression::getConstantView() const -> double {
  return impl()->getConstantView();
}
auto xpress::objects::ConstantExpression::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::ConstantExpression::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::ConstantExpression::uminus() const
    -> xpress::objects::Expression {
  return impl()->uminus();
}
auto xpress::objects::ConstantExpression::plus(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  return impl()->plus(arg);
}
auto xpress::objects::ConstantExpression::plus(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::ConstantExpression {
  return impl()->plus(arg);
}
auto xpress::objects::ConstantExpression::plus(double arg) const
    -> xpress::objects::Expression {
  return impl()->plus(arg);
}
auto xpress::objects::ConstantExpression::minus(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  return impl()->minus(arg);
}
auto xpress::objects::ConstantExpression::minus(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::ConstantExpression {
  return impl()->minus(arg);
}
auto xpress::objects::ConstantExpression::minus(double arg) const
    -> xpress::objects::Expression {
  return impl()->minus(arg);
}
auto xpress::objects::ConstantExpression::mul(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::ConstantExpression::mul(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::ConstantExpression {
  return impl()->mul(arg);
}
auto xpress::objects::ConstantExpression::mul(double arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::ConstantExpression::mul(
    xpress::objects::Variable arg) const -> xpress::objects::LinTerm {
  return impl()->mul(arg);
}
auto xpress::objects::ConstantExpression::mul(
    xpress::objects::LinTerm arg) const -> xpress::objects::LinTerm {
  return impl()->mul(arg);
}
auto xpress::objects::ConstantExpression::mul(
    xpress::objects::QuadTerm arg) const -> xpress::objects::QuadTerm {
  return impl()->mul(arg);
}
auto xpress::objects::ConstantExpression::div(
    xpress::objects::Expression arg) const -> xpress::objects::Expression {
  return impl()->div(arg);
}
auto xpress::objects::ConstantExpression::div(
    xpress::objects::ConstantExpression arg) const
    -> xpress::objects::ConstantExpression {
  return impl()->div(arg);
}
auto xpress::objects::ConstantExpression::div(double arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
xpress::objects::ConstantExpression::ConstantExpression()
    : AbstractExpression(nullptr) {}
xpress::objects::GeneralConstraint::GeneralConstraint(
    std::shared_ptr<GeneralConstraintImplementation> init)
    : implPointer(init) {}
xpress::objects::GeneralConstraint::GeneralConstraint(
    GeneralConstraintImplementation const *init)
    : implPointer(
          init ? const_cast<GeneralConstraintImplementation *>(init)
                     ->shared_from_this()
               : std::shared_ptr<GeneralConstraintImplementation>(nullptr)) {}
xpress::objects::GeneralConstraintImplementation *
xpress::objects::GeneralConstraint::impl() {
  return dynamic_cast<GeneralConstraintImplementation *>(implPointer.get());
}
xpress::objects::GeneralConstraintImplementation const *
xpress::objects::GeneralConstraint::impl() const {
  return dynamic_cast<GeneralConstraintImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::GeneralConstraint::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::GeneralConstraint::getIndex() const -> int {
  return impl()->getIndex();
}
auto xpress::objects::GeneralConstraint::updateIndex(int delta) -> void {
  impl()->updateIndex(delta);
}
auto xpress::objects::GeneralConstraint::getIndexForProb(
    XpressProblem const *xprob) const -> int {
  return impl()->getIndexForProb(xprob);
}
auto xpress::objects::GeneralConstraint::getProblem() const
    -> xpress::objects::XpressProblem * {
  return impl()->getProblem();
}
auto xpress::objects::GeneralConstraint::getSerial() const -> XPRSint64 {
  return impl()->getSerial();
}
auto xpress::objects::GeneralConstraint::markDeleted() -> void {
  impl()->markDeleted();
}
auto xpress::objects::GeneralConstraint::compareTo(
    GeneralConstraint const &other) const -> int {
  return impl()->compareTo(other);
}
auto xpress::objects::GeneralConstraint::getHashCode() const -> std::size_t {
  return impl()->getHashCode();
}
auto xpress::objects::GeneralConstraint::check(char const *operation) const
    -> void {
  impl()->check(operation);
}
xpress::objects::GeneralConstraint::GeneralConstraint(XpressProblem *prob,
                                                      long serial, int index)
    : implPointer(std::make_shared<GeneralConstraintImplementation>(
          prob, serial, index)) {}
auto xpress::objects::GeneralConstraint::getType() -> xpress::GenConsType {
  return impl()->getType();
}
auto xpress::objects::GeneralConstraint::getResultant()
    -> xpress::objects::Variable {
  return impl()->getResultant();
}
auto xpress::objects::GeneralConstraint::getVariables()
    -> std::vector<xpress::objects::Variable> {
  return impl()->getVariables();
}
auto xpress::objects::GeneralConstraint::getValues() -> std::vector<double> {
  return impl()->getValues();
}
auto xpress::objects::GeneralConstraint::getName() const -> std::string {
  return impl()->getName();
}
auto xpress::objects::GeneralConstraint::setName(
    std::optional<std::string> newName)
    -> xpress::objects::GeneralConstraint & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::GeneralConstraint::Abs(Variable resultant,
                                             Variable variable)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Abs(resultant, variable);
}
template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::GeneralConstraint::And(Variable resultant,
                                             VARIABLES... variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::And(resultant, variables...);
}
auto xpress::objects::GeneralConstraint::And(
    Variable resultant, xpress::SizedArray<Variable const> const &variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::And(resultant, variables);
}
template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::GeneralConstraint::Or(Variable resultant,
                                            VARIABLES... variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Or(resultant, variables...);
}
auto xpress::objects::GeneralConstraint::Or(
    Variable resultant, xpress::SizedArray<Variable const> const &variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Or(resultant, variables);
}
template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::GeneralConstraint::Max(Variable resultant,
                                             VARIABLES... variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Max(resultant, variables...);
}
auto xpress::objects::GeneralConstraint::Max(
    Variable resultant, xpress::SizedArray<Variable const> const &variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Max(resultant, variables);
}
template <typename... VALUES, typename RestrictPack>
auto xpress::objects::GeneralConstraint::Max(
    Variable resultant, xpress::SizedArray<Variable const> const &variables,
    VALUES... values) -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Max(resultant, variables, values...);
}
auto xpress::objects::GeneralConstraint::Max(
    Variable resultant, xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &values)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Max(resultant, variables, values);
}
template <typename... VARIABLES, typename RestrictPack>
auto xpress::objects::GeneralConstraint::Min(Variable resultant,
                                             VARIABLES... variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Min(resultant, variables...);
}
auto xpress::objects::GeneralConstraint::Min(
    Variable resultant, xpress::SizedArray<Variable const> const &variables)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Min(resultant, variables);
}
template <typename... VALUES, typename RestrictPack>
auto xpress::objects::GeneralConstraint::Min(
    Variable resultant, xpress::SizedArray<Variable const> const &variables,
    VALUES... values) -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Min(resultant, variables, values...);
}
auto xpress::objects::GeneralConstraint::Min(
    Variable resultant, xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &values)
    -> xpress::objects::GeneralConstraintDefinition {
  return GeneralConstraintImplementation::Min(resultant, variables, values);
}
xpress::objects::GeneralConstraint::GeneralConstraint()
    : implPointer(nullptr) {}
xpress::objects::GeneralConstraintDefinition::GeneralConstraintDefinition(
    std::shared_ptr<GeneralConstraintDefinitionImplementation> init)
    : ConstraintDefinition<GeneralConstraint>(init) {}
xpress::objects::GeneralConstraintDefinition::GeneralConstraintDefinition(
    GeneralConstraintDefinitionImplementation const *init)
    : ConstraintDefinition<GeneralConstraint>(init) {}
xpress::objects::GeneralConstraintDefinitionImplementation *
xpress::objects::GeneralConstraintDefinition::impl() {
  return dynamic_cast<GeneralConstraintDefinitionImplementation *>(
      implPointer.get());
}
xpress::objects::GeneralConstraintDefinitionImplementation const *
xpress::objects::GeneralConstraintDefinition::impl() const {
  return dynamic_cast<GeneralConstraintDefinitionImplementation *>(
      implPointer.get());
}
template <typename Derived>
Derived xpress::objects::GeneralConstraintDefinition::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
xpress::objects::GeneralConstraintDefinition::GeneralConstraintDefinition(
    Variable resultant, xpress::GenConsType type, Strm0 const &variables,
    Strm1 const &value, std::optional<std::string> name)
    : ConstraintDefinition<GeneralConstraint>(
          std::make_shared<GeneralConstraintDefinitionImplementation>(
              resultant, type, variables, value, name)) {}
template <typename Strm0, typename Strm0IsStream>
xpress::objects::GeneralConstraintDefinition::GeneralConstraintDefinition(
    Variable resultant, xpress::GenConsType type, Strm0 const &variables,
    std::optional<std::string> name)
    : ConstraintDefinition<GeneralConstraint>(
          std::make_shared<GeneralConstraintDefinitionImplementation>(
              resultant, type, variables, name)) {}
template <typename... VARIABLES, typename RestrictPack>
xpress::objects::GeneralConstraintDefinition::GeneralConstraintDefinition(
    Variable resultant, xpress::GenConsType type, VARIABLES... variables)
    : ConstraintDefinition<GeneralConstraint>(
          std::make_shared<GeneralConstraintDefinitionImplementation>(
              resultant, type, variables...)) {}
xpress::objects::GeneralConstraintDefinition::GeneralConstraintDefinition(
    Variable resultant, xpress::GenConsType type,
    xpress::SizedArray<Variable const> const &variables)
    : ConstraintDefinition<GeneralConstraint>(
          std::make_shared<GeneralConstraintDefinitionImplementation>(
              resultant, type, variables)) {}
xpress::objects::GeneralConstraintDefinition::GeneralConstraintDefinition(
    Variable resultant, xpress::GenConsType type,
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &values)
    : ConstraintDefinition<GeneralConstraint>(
          std::make_shared<GeneralConstraintDefinitionImplementation>(
              resultant, type, variables, values)) {}
auto xpress::objects::GeneralConstraintDefinition::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  impl()->createInternal(creator);
}
auto xpress::objects::GeneralConstraintDefinition::setName(
    std::optional<std::string> newName)
    -> xpress::objects::GeneralConstraintDefinition & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::GeneralConstraintDefinition::getIndexHandler(
    xpress::objects::XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::GeneralConstraint> const
        * {
  return impl()->getIndexHandler(prob);
}
xpress::objects::GeneralConstraintDefinition::GeneralConstraintDefinition()
    : ConstraintDefinition<GeneralConstraint>(nullptr) {}
xpress::objects::IndicatorDescription::IndicatorDescription(
    std::shared_ptr<IndicatorDescriptionImplementation> init)
    : ConstraintDefinition<Inequality>(init) {}
xpress::objects::IndicatorDescription::IndicatorDescription(
    IndicatorDescriptionImplementation const *init)
    : ConstraintDefinition<Inequality>(init) {}
xpress::objects::IndicatorDescriptionImplementation *
xpress::objects::IndicatorDescription::impl() {
  return dynamic_cast<IndicatorDescriptionImplementation *>(implPointer.get());
}
xpress::objects::IndicatorDescriptionImplementation const *
xpress::objects::IndicatorDescription::impl() const {
  return dynamic_cast<IndicatorDescriptionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::IndicatorDescription::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::objects::IndicatorDescription::IndicatorDescription(
    Variable indicatorVariable, bool indicatorValue,
    InequalityDefinition implied)
    : ConstraintDefinition<Inequality>(
          std::make_shared<IndicatorDescriptionImplementation>(
              indicatorVariable, indicatorValue, implied)) {}
auto xpress::objects::IndicatorDescription::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  impl()->createInternal(creator);
}
auto xpress::objects::IndicatorDescription::getName() const
    -> std::optional<std::string> {
  return impl()->getName();
}
auto xpress::objects::IndicatorDescription::setName(
    std::optional<std::string> newName)
    -> xpress::objects::IndicatorDescription & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::IndicatorDescription::getIndexHandler(
    xpress::objects::XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::Inequality> const * {
  return impl()->getIndexHandler(prob);
}
xpress::objects::IndicatorDescription::IndicatorDescription()
    : ConstraintDefinition<Inequality>(nullptr) {}
xpress::objects::LinExpression::LinExpression(
    std::shared_ptr<LinExpressionImplementation> init)
    : xpress::objects::AbstractExpression(init) {}
xpress::objects::LinExpression::LinExpression(
    LinExpressionImplementation const *init)
    : xpress::objects::AbstractExpression(init) {}
xpress::objects::LinExpressionImplementation *
xpress::objects::LinExpression::impl() {
  return dynamic_cast<LinExpressionImplementation *>(implPointer.get());
}
xpress::objects::LinExpressionImplementation const *
xpress::objects::LinExpression::impl() const {
  return dynamic_cast<LinExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::LinExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::LinExpression::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::LinExpression::normalizeTerms()
    -> xpress::objects::LinExpression & {
  impl()->normalizeTerms();
  return *this;
}
auto xpress::objects::LinExpression::deepCopy()
    -> xpress::objects::LinExpression {
  return impl()->deepCopy();
}
auto xpress::objects::LinExpression::deepCopy(double factor)
    -> xpress::objects::LinExpression {
  return impl()->deepCopy(factor);
}
auto xpress::objects::LinExpression::getConstant() const -> double {
  return impl()->getConstant();
}
auto xpress::objects::LinExpression::addConstant(double c)
    -> xpress::objects::LinExpression & {
  impl()->addConstant(c);
  return *this;
}
auto xpress::objects::LinExpression::setConstant(double c)
    -> xpress::objects::LinExpression & {
  impl()->setConstant(c);
  return *this;
}
auto xpress::objects::LinExpression::reset()
    -> xpress::objects::LinExpression & {
  impl()->reset();
  return *this;
}
auto xpress::objects::LinExpression::prepareToAdd(int newTerms) -> void {
  impl()->prepareToAdd(newTerms);
}
template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::LinExpression::addTerms(Strm0 const &variables,
                                              Strm1 const &coefficients)
    -> xpress::objects::LinExpression & {
  impl()->addTerms(variables, coefficients);
  return *this;
}
auto xpress::objects::LinExpression::addTerms(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &coefficients, int offset, int count)
    -> xpress::objects::LinExpression & {
  impl()->addTerms(variables, coefficients, offset, count);
  return *this;
}
auto xpress::objects::LinExpression::addTerm(Variable const &variable,
                                             double const &coefficient)
    -> xpress::objects::LinExpression & {
  impl()->addTerm(variable, coefficient);
  return *this;
}
auto xpress::objects::LinExpression::addTerm(double const &coefficient,
                                             Variable const &variable)
    -> xpress::objects::LinExpression & {
  impl()->addTerm(coefficient, variable);
  return *this;
}
auto xpress::objects::LinExpression::addTerm(LinTerm const &term)
    -> xpress::objects::LinExpression & {
  impl()->addTerm(term);
  return *this;
}
auto xpress::objects::LinExpression::addTerm(Variable const &x)
    -> xpress::objects::LinExpression & {
  impl()->addTerm(x);
  return *this;
}
auto xpress::objects::LinExpression::addTerms(
    xpress::objects::LinExpression const &expr)
    -> xpress::objects::LinExpression & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::LinExpression::addTerms(
    xpress::objects::LinExpression const &expr, double factor)
    -> xpress::objects::LinExpression & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::LinExpression::square() -> xpress::objects::Expression {
  return impl()->square();
}
auto xpress::objects::LinExpression::create()
    -> xpress::objects::LinExpression {
  return LinExpressionImplementation::create();
}
auto xpress::objects::LinExpression::create(double constant)
    -> xpress::objects::LinExpression {
  return LinExpressionImplementation::create(constant);
}
auto xpress::objects::LinExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::LinExpression::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
xpress::objects::LinExpression::LinExpression()
    : xpress::objects::AbstractExpression(nullptr) {}
xpress::objects::LinTerm::LinTerm(std::shared_ptr<LinTermImplementation> init)
    : xpress::objects::AbstractExpression(init) {}
xpress::objects::LinTerm::LinTerm(LinTermImplementation const *init)
    : xpress::objects::AbstractExpression(init) {}
xpress::objects::LinTermImplementation *xpress::objects::LinTerm::impl() {
  return dynamic_cast<LinTermImplementation *>(implPointer.get());
}
xpress::objects::LinTermImplementation const *
xpress::objects::LinTerm::impl() const {
  return dynamic_cast<LinTermImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::LinTerm::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::LinTerm::getVariable() const
    -> xpress::objects::Variable {
  return impl()->getVariable();
}
auto xpress::objects::LinTerm::getCoefficient() const -> double {
  return impl()->getCoefficient();
}
xpress::objects::LinTerm::LinTerm(Variable variable, double coefficient)
    : xpress::objects::AbstractExpression(
          std::make_shared<LinTermImplementation>(variable, coefficient)) {}
auto xpress::objects::LinTerm::getRTTI() -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::LinTerm::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::LinTerm::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::LinTerm::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::LinTerm::lTerm(xpress::objects::Variable variable,
                                     double coefficient)
    -> xpress::objects::LinTerm {
  return LinTermImplementation::lTerm(variable, coefficient);
}
auto xpress::objects::LinTerm::lTerm(Variable variable)
    -> xpress::objects::LinTerm {
  return LinTermImplementation::lTerm(variable);
}
auto xpress::objects::LinTerm::square() const -> xpress::objects::QuadTerm {
  return impl()->square();
}
auto xpress::objects::LinTerm::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::LinTerm::toString(xpress::objects::Variable v,
                                        double coef) -> std::string {
  return LinTermImplementation::toString(v, coef);
}
auto xpress::objects::LinTerm::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::LinTerm::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::LinTerm::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::LinTerm::uminus() const -> xpress::objects::Expression {
  return impl()->uminus();
}
auto xpress::objects::LinTerm::mul(xpress::objects::Expression arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::LinTerm::mul(ConstantExpression arg) const
    -> xpress::objects::LinTerm {
  return impl()->mul(arg);
}
auto xpress::objects::LinTerm::mul(double arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::LinTerm::mul(Variable arg) const
    -> xpress::objects::QuadTerm {
  return impl()->mul(arg);
}
auto xpress::objects::LinTerm::mul(LinTerm arg) const
    -> xpress::objects::QuadTerm {
  return impl()->mul(arg);
}
auto xpress::objects::LinTerm::div(Expression arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
auto xpress::objects::LinTerm::div(ConstantExpression arg) const
    -> xpress::objects::LinTerm {
  return impl()->div(arg);
}
auto xpress::objects::LinTerm::div(double arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
xpress::objects::LinTerm::LinTerm()
    : xpress::objects::AbstractExpression(nullptr) {}
xpress::objects::PWL::PWL(std::shared_ptr<PWLImplementation> init)
    : implPointer(init) {}
xpress::objects::PWL::PWL(PWLImplementation const *init)
    : implPointer(
          init ? const_cast<PWLImplementation *>(init)->shared_from_this()
               : std::shared_ptr<PWLImplementation>(nullptr)) {}
xpress::objects::PWLImplementation *xpress::objects::PWL::impl() {
  return dynamic_cast<PWLImplementation *>(implPointer.get());
}
xpress::objects::PWLImplementation const *xpress::objects::PWL::impl() const {
  return dynamic_cast<PWLImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::PWL::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::PWL::getIndex() const -> int {
  return impl()->getIndex();
}
auto xpress::objects::PWL::updateIndex(int delta) -> void {
  impl()->updateIndex(delta);
}
auto xpress::objects::PWL::getIndexForProb(XpressProblem const *xprob) const
    -> int {
  return impl()->getIndexForProb(xprob);
}
auto xpress::objects::PWL::getProblem() const
    -> xpress::objects::XpressProblem * {
  return impl()->getProblem();
}
auto xpress::objects::PWL::getSerial() const -> XPRSint64 {
  return impl()->getSerial();
}
auto xpress::objects::PWL::markDeleted() -> void { impl()->markDeleted(); }
auto xpress::objects::PWL::compareTo(PWL const &other) const -> int {
  return impl()->compareTo(other);
}
auto xpress::objects::PWL::getHashCode() const -> std::size_t {
  return impl()->getHashCode();
}
auto xpress::objects::PWL::check(char const *operation) const -> void {
  impl()->check(operation);
}
xpress::objects::PWL::PWL(XpressProblem *prob, long serial, int index)
    : implPointer(std::make_shared<PWLImplementation>(prob, serial, index)) {}
auto xpress::objects::PWL::getInput() -> xpress::objects::Variable {
  return impl()->getInput();
}
auto xpress::objects::PWL::getResultant() -> xpress::objects::Variable {
  return impl()->getResultant();
}
auto xpress::objects::PWL::getBreakpoints()
    -> std::vector<xpress::PwlBreakpoint> {
  return impl()->getBreakpoints();
}
auto xpress::objects::PWL::getName() const -> std::string {
  return impl()->getName();
}
auto xpress::objects::PWL::setName(std::optional<std::string> newName)
    -> xpress::objects::PWL & {
  impl()->setName(newName);
  return *this;
}
xpress::objects::PWL::PWL() : implPointer(nullptr) {}
xpress::objects::PWLDefinition::PWLDefinition(
    std::shared_ptr<PWLDefinitionImplementation> init)
    : ConstraintDefinition<PWL>(init) {}
xpress::objects::PWLDefinition::PWLDefinition(
    PWLDefinitionImplementation const *init)
    : ConstraintDefinition<PWL>(init) {}
xpress::objects::PWLDefinitionImplementation *
xpress::objects::PWLDefinition::impl() {
  return dynamic_cast<PWLDefinitionImplementation *>(implPointer.get());
}
xpress::objects::PWLDefinitionImplementation const *
xpress::objects::PWLDefinition::impl() const {
  return dynamic_cast<PWLDefinitionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::PWLDefinition::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::objects::PWLDefinition::PWLDefinition(
    xpress::objects::Variable input, xpress::objects::Variable resultant,
    xpress::SizedArray<double const> const &xval,
    xpress::SizedArray<double const> const &yval,
    std::optional<std::string> name)
    : ConstraintDefinition<PWL>(std::make_shared<PWLDefinitionImplementation>(
          input, resultant, xval, yval, name)) {}
auto xpress::objects::PWLDefinition::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  impl()->createInternal(creator);
}
auto xpress::objects::PWLDefinition::setName(std::optional<std::string> newName)
    -> xpress::objects::PWLDefinition & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::PWLDefinition::getIndexHandler(XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::PWL> const * {
  return impl()->getIndexHandler(prob);
}
xpress::objects::PWLDefinition::PWLDefinition()
    : ConstraintDefinition<PWL>(nullptr) {}
xpress::objects::QuadExpression::QuadExpression(
    std::shared_ptr<QuadExpressionImplementation> init)
    : AbstractExpression(init) {}
xpress::objects::QuadExpression::QuadExpression(
    QuadExpressionImplementation const *init)
    : AbstractExpression(init) {}
xpress::objects::QuadExpressionImplementation *
xpress::objects::QuadExpression::impl() {
  return dynamic_cast<QuadExpressionImplementation *>(implPointer.get());
}
xpress::objects::QuadExpressionImplementation const *
xpress::objects::QuadExpression::impl() const {
  return dynamic_cast<QuadExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::QuadExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::QuadExpression::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::QuadExpression::normalizeTerms()
    -> xpress::objects::QuadExpression & {
  impl()->normalizeTerms();
  return *this;
}
auto xpress::objects::QuadExpression::deepCopy()
    -> xpress::objects::QuadExpression {
  return impl()->deepCopy();
}
auto xpress::objects::QuadExpression::deepCopy(double factor)
    -> xpress::objects::QuadExpression {
  return impl()->deepCopy(factor);
}
auto xpress::objects::QuadExpression::getConstant() const -> double {
  return impl()->getConstant();
}
auto xpress::objects::QuadExpression::addConstant(double c)
    -> xpress::objects::QuadExpression & {
  impl()->addConstant(c);
  return *this;
}
auto xpress::objects::QuadExpression::setConstant(double c)
    -> xpress::objects::QuadExpression & {
  impl()->setConstant(c);
  return *this;
}
auto xpress::objects::QuadExpression::reset()
    -> xpress::objects::QuadExpression & {
  impl()->reset();
  return *this;
}
auto xpress::objects::QuadExpression::prepareToAdd(int newTerms) -> void {
  impl()->prepareToAdd(newTerms);
}
template <typename Strm0, typename Strm1, typename Strm0IsStream,
          typename Strm1IsStream>
auto xpress::objects::QuadExpression::addTerms(Strm0 const &variables,
                                               Strm1 const &coefficients)
    -> xpress::objects::QuadExpression & {
  impl()->addTerms(variables, coefficients);
  return *this;
}
template <typename Strm0, typename Strm1, typename Strm2,
          typename Strm0IsStream, typename Strm1IsStream,
          typename Strm2IsStream>
auto xpress::objects::QuadExpression::addTerms(Strm0 const &variables1,
                                               Strm1 const &variables2,
                                               Strm2 const &coefficients)
    -> xpress::objects::QuadExpression & {
  impl()->addTerms(variables1, variables2, coefficients);
  return *this;
}
auto xpress::objects::QuadExpression::addTerms(
    xpress::SizedArray<Variable const> const &variables,
    xpress::SizedArray<double const> const &coefficients, int offset, int count)
    -> xpress::objects::QuadExpression & {
  impl()->addTerms(variables, coefficients, offset, count);
  return *this;
}
auto xpress::objects::QuadExpression::addTerms(
    xpress::SizedArray<Variable const> const &variables1,
    xpress::SizedArray<Variable const> const &variables2,
    xpress::SizedArray<double const> const &coefficients, int offset, int count)
    -> xpress::objects::QuadExpression & {
  impl()->addTerms(variables1, variables2, coefficients, offset, count);
  return *this;
}
auto xpress::objects::QuadExpression::addTerm(Variable const &variable,
                                              double const &coefficient)
    -> xpress::objects::QuadExpression & {
  impl()->addTerm(variable, coefficient);
  return *this;
}
auto xpress::objects::QuadExpression::addTerm(double const &coefficient,
                                              Variable const &variable)
    -> xpress::objects::QuadExpression & {
  impl()->addTerm(coefficient, variable);
  return *this;
}
auto xpress::objects::QuadExpression::addTerm(Variable const &x)
    -> xpress::objects::QuadExpression & {
  impl()->addTerm(x);
  return *this;
}
auto xpress::objects::QuadExpression::addTerm(LinTerm const &term)
    -> xpress::objects::QuadExpression & {
  impl()->addTerm(term);
  return *this;
}
auto xpress::objects::QuadExpression::addTerm(QuadTerm const &term)
    -> xpress::objects::QuadExpression & {
  impl()->addTerm(term);
  return *this;
}
auto xpress::objects::QuadExpression::addTerm(Variable const &variable1,
                                              Variable const &variable2,
                                              double const &coefficient)
    -> xpress::objects::QuadExpression & {
  impl()->addTerm(variable1, variable2, coefficient);
  return *this;
}
auto xpress::objects::QuadExpression::addTerm(double const &coefficient,
                                              Variable const &variable1,
                                              Variable const &variable2)
    -> xpress::objects::QuadExpression & {
  impl()->addTerm(coefficient, variable1, variable2);
  return *this;
}
auto xpress::objects::QuadExpression::addTerm(Variable const &x1,
                                              Variable const &x2)
    -> xpress::objects::QuadExpression & {
  impl()->addTerm(x1, x2);
  return *this;
}
auto xpress::objects::QuadExpression::addTerms(LinExpression const &expr)
    -> xpress::objects::QuadExpression & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::QuadExpression::addTerms(QuadExpression const &expr)
    -> xpress::objects::QuadExpression & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::QuadExpression::addTerms(LinExpression const &expr,
                                               double const &factor)
    -> xpress::objects::QuadExpression & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::QuadExpression::addTerms(QuadExpression const &expr,
                                               double const &factor)
    -> xpress::objects::QuadExpression & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::QuadExpression::create()
    -> xpress::objects::QuadExpression {
  return QuadExpressionImplementation::create();
}
auto xpress::objects::QuadExpression::create(double constant)
    -> xpress::objects::QuadExpression {
  return QuadExpressionImplementation::create(constant);
}
auto xpress::objects::QuadExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
xpress::objects::QuadExpression::QuadExpression()
    : AbstractExpression(nullptr) {}
xpress::objects::QuadTerm::QuadTerm(
    std::shared_ptr<QuadTermImplementation> init)
    : xpress::objects::AbstractExpression(init) {}
xpress::objects::QuadTerm::QuadTerm(QuadTermImplementation const *init)
    : xpress::objects::AbstractExpression(init) {}
xpress::objects::QuadTermImplementation *xpress::objects::QuadTerm::impl() {
  return dynamic_cast<QuadTermImplementation *>(implPointer.get());
}
xpress::objects::QuadTermImplementation const *
xpress::objects::QuadTerm::impl() const {
  return dynamic_cast<QuadTermImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::QuadTerm::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::QuadTerm::getVariable1() const
    -> xpress::objects::Variable {
  return impl()->getVariable1();
}
auto xpress::objects::QuadTerm::getVariable2() const
    -> xpress::objects::Variable {
  return impl()->getVariable2();
}
auto xpress::objects::QuadTerm::getCoefficient() const -> double {
  return impl()->getCoefficient();
}
xpress::objects::QuadTerm::QuadTerm(xpress::objects::Variable variable1,
                                    xpress::objects::Variable variable2,
                                    double coefficient)
    : xpress::objects::AbstractExpression(
          std::make_shared<QuadTermImplementation>(variable1, variable2,
                                                   coefficient)) {}
auto xpress::objects::QuadTerm::getRTTI() -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::QuadTerm::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::QuadTerm::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::QuadTerm::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::QuadTerm::qTerm(xpress::objects::Variable variable1,
                                      xpress::objects::Variable variable2,
                                      double coefficient)
    -> xpress::objects::QuadTerm {
  return QuadTermImplementation::qTerm(variable1, variable2, coefficient);
}
auto xpress::objects::QuadTerm::qTerm(xpress::objects::Variable variable1,
                                      xpress::objects::Variable variable2)
    -> xpress::objects::QuadTerm {
  return QuadTermImplementation::qTerm(variable1, variable2);
}
auto xpress::objects::QuadTerm::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::QuadTerm::toString(xpress::objects::Variable v1,
                                         xpress::objects::Variable v2,
                                         double coef) -> std::string {
  return QuadTermImplementation::toString(v1, v2, coef);
}
auto xpress::objects::QuadTerm::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::QuadTerm::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::QuadTerm::uminus() const -> xpress::objects::Expression {
  return impl()->uminus();
}
auto xpress::objects::QuadTerm::mul(xpress::objects::Expression arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::QuadTerm::mul(ConstantExpression arg) const
    -> xpress::objects::QuadTerm {
  return impl()->mul(arg);
}
auto xpress::objects::QuadTerm::mul(double arg) const
    -> xpress::objects::Expression {
  return impl()->mul(arg);
}
auto xpress::objects::QuadTerm::div(xpress::objects::Expression arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
auto xpress::objects::QuadTerm::div(xpress::objects::ConstantExpression arg)
    const -> xpress::objects::QuadTerm {
  return impl()->div(arg);
}
auto xpress::objects::QuadTerm::div(double arg) const
    -> xpress::objects::Expression {
  return impl()->div(arg);
}
xpress::objects::QuadTerm::QuadTerm()
    : xpress::objects::AbstractExpression(nullptr) {}
xpress::objects::SOS::SOS(std::shared_ptr<SOSImplementation> init)
    : implPointer(init) {}
xpress::objects::SOS::SOS(SOSImplementation const *init)
    : implPointer(
          init ? const_cast<SOSImplementation *>(init)->shared_from_this()
               : std::shared_ptr<SOSImplementation>(nullptr)) {}
xpress::objects::SOSImplementation *xpress::objects::SOS::impl() {
  return dynamic_cast<SOSImplementation *>(implPointer.get());
}
xpress::objects::SOSImplementation const *xpress::objects::SOS::impl() const {
  return dynamic_cast<SOSImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::SOS::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::SOS::getIndex() const -> int {
  return impl()->getIndex();
}
auto xpress::objects::SOS::updateIndex(int delta) -> void {
  impl()->updateIndex(delta);
}
auto xpress::objects::SOS::getIndexForProb(XpressProblem const *xprob) const
    -> int {
  return impl()->getIndexForProb(xprob);
}
auto xpress::objects::SOS::getProblem() const
    -> xpress::objects::XpressProblem * {
  return impl()->getProblem();
}
auto xpress::objects::SOS::getSerial() const -> XPRSint64 {
  return impl()->getSerial();
}
auto xpress::objects::SOS::markDeleted() -> void { impl()->markDeleted(); }
auto xpress::objects::SOS::compareTo(SOS const &other) const -> int {
  return impl()->compareTo(other);
}
auto xpress::objects::SOS::getHashCode() const -> std::size_t {
  return impl()->getHashCode();
}
auto xpress::objects::SOS::check(char const *operation) const -> void {
  impl()->check(operation);
}
xpress::objects::SOS::SOS(XpressProblem *prob, long serial, int index)
    : implPointer(std::make_shared<SOSImplementation>(prob, serial, index)) {}
auto xpress::objects::SOS::getName() const -> std::string {
  return impl()->getName();
}
auto xpress::objects::SOS::setName(std::optional<std::string> newName)
    -> xpress::objects::SOS & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::SOS::sos(xpress::SetType type,
                               xpress::SizedArray<Variable const> const &items,
                               xpress::SizedArray<double const> const &weights,
                               std::optional<std::string> name)
    -> xpress::objects::SOSDefinition {
  return SOSImplementation::sos(type, items, weights, name);
}
auto xpress::objects::SOS::sos1(xpress::SizedArray<Variable const> const &items,
                                xpress::SizedArray<double const> const &weights,
                                std::optional<std::string> name)
    -> xpress::objects::SOSDefinition {
  return SOSImplementation::sos1(items, weights, name);
}
auto xpress::objects::SOS::sos2(xpress::SizedArray<Variable const> const &items,
                                xpress::SizedArray<double const> const &weights,
                                std::optional<std::string> name)
    -> xpress::objects::SOSDefinition {
  return SOSImplementation::sos2(items, weights, name);
}
xpress::objects::SOS::SOS() : implPointer(nullptr) {}
xpress::objects::SOSDefinition::SOSDefinition(
    std::shared_ptr<SOSDefinitionImplementation> init)
    : ConstraintDefinition<SOS>(init) {}
xpress::objects::SOSDefinition::SOSDefinition(
    SOSDefinitionImplementation const *init)
    : ConstraintDefinition<SOS>(init) {}
xpress::objects::SOSDefinitionImplementation *
xpress::objects::SOSDefinition::impl() {
  return dynamic_cast<SOSDefinitionImplementation *>(implPointer.get());
}
xpress::objects::SOSDefinitionImplementation const *
xpress::objects::SOSDefinition::impl() const {
  return dynamic_cast<SOSDefinitionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::SOSDefinition::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::objects::SOSDefinition::SOSDefinition(
    xpress::SetType type, xpress::SizedArray<Variable const> const &items,
    xpress::SizedArray<double const> const &weights,
    std::optional<std::string> name)
    : ConstraintDefinition<SOS>(std::make_shared<SOSDefinitionImplementation>(
          type, items, weights, name)) {}
auto xpress::objects::SOSDefinition::createInternal(
    xpress::XPRSProblem::ConstraintCreator *creator) const -> void {
  impl()->createInternal(creator);
}
auto xpress::objects::SOSDefinition::setName(std::optional<std::string> newName)
    -> xpress::objects::SOSDefinition & {
  impl()->setName(newName);
  return *this;
}
auto xpress::objects::SOSDefinition::getIndexHandler(XpressProblem *prob) const
    -> xpress::objects::IndexHandler<xpress::objects::SOS> const * {
  return impl()->getIndexHandler(prob);
}
xpress::objects::SOSDefinition::SOSDefinition()
    : ConstraintDefinition<SOS>(nullptr) {}
xpress::objects::SumExpression::SumExpression(
    std::shared_ptr<SumExpressionImplementation> init)
    : AbstractExpression(init) {}
xpress::objects::SumExpression::SumExpression(
    SumExpressionImplementation const *init)
    : AbstractExpression(init) {}
xpress::objects::SumExpressionImplementation *
xpress::objects::SumExpression::impl() {
  return dynamic_cast<SumExpressionImplementation *>(implPointer.get());
}
xpress::objects::SumExpressionImplementation const *
xpress::objects::SumExpression::impl() const {
  return dynamic_cast<SumExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::SumExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::objects::SumExpression::SumExpression(
    xpress::SizedArray<Expression const> const &terms)
    : AbstractExpression(std::make_shared<SumExpressionImplementation>(terms)) {
}
auto xpress::objects::SumExpression::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::SumExpression::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::SumExpression::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::SumExpression::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::SumExpression::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::SumExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::SumExpression::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::SumExpression::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
xpress::objects::SumExpression::SumExpression() : AbstractExpression(nullptr) {}
xpress::objects::FormulaExpression::FormulaExpression(
    std::shared_ptr<FormulaExpressionImplementation> init)
    : xpress::objects::AbstractExpression(init) {}
xpress::objects::FormulaExpression::FormulaExpression(
    FormulaExpressionImplementation const *init)
    : xpress::objects::AbstractExpression(init) {}
xpress::objects::FormulaExpressionImplementation *
xpress::objects::FormulaExpression::impl() {
  return dynamic_cast<FormulaExpressionImplementation *>(implPointer.get());
}
xpress::objects::FormulaExpressionImplementation const *
xpress::objects::FormulaExpression::impl() const {
  return dynamic_cast<FormulaExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::FormulaExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::FormulaExpression::getRTTI()
    -> xpress::objects::ExpressionRTTI {
  return impl()->getRTTI();
}
auto xpress::objects::FormulaExpression::extract(
    PostfixExtractor *extractor) const -> void {
  impl()->extract(extractor);
}
auto xpress::objects::FormulaExpression::extract(
    double factor, PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::FormulaExpression::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
template <typename T>
auto xpress::objects::FormulaExpression::parseUnaryFunction(
    SymbolicEvalStack &stack, T type)
    -> xpress::objects::InternalFunctionExpression {
  return FormulaExpressionImplementation::parseUnaryFunction(stack, type);
}
template <typename T, typename Func0, typename Func0IsInvocable>
auto xpress::objects::FormulaExpression::reverseArgs(std::vector<T> const &args,
                                                     Func0 mkArray)
    -> std::vector<T> {
  return FormulaExpressionImplementation::reverseArgs(args, mkArray);
}
template <typename T>
auto xpress::objects::FormulaExpression::parseNaryFunction(
    SymbolicEvalStack &stack, T type)
    -> xpress::objects::InternalFunctionExpression {
  return FormulaExpressionImplementation::parseNaryFunction(stack, type);
}
auto xpress::objects::FormulaExpression::parse(
    XpressProblem *prob, xpress::SizedArray<int const> const &type,
    xpress::SizedArray<double const> const &value)
    -> xpress::objects::Expression {
  return FormulaExpressionImplementation::parse(prob, type, value);
}
xpress::objects::FormulaExpression::FormulaExpression()
    : xpress::objects::AbstractExpression(nullptr) {}
xpress::objects::LinTermList::LinTermList(
    std::shared_ptr<LinTermListImplementation> init)
    : LinExpression(init) {}
xpress::objects::LinTermList::LinTermList(LinTermListImplementation const *init)
    : LinExpression(init) {}
xpress::objects::LinTermListImplementation *
xpress::objects::LinTermList::impl() {
  return dynamic_cast<LinTermListImplementation *>(implPointer.get());
}
xpress::objects::LinTermListImplementation const *
xpress::objects::LinTermList::impl() const {
  return dynamic_cast<LinTermListImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::LinTermList::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::LinTermList::setAutoCompress(bool compress) -> void {
  LinTermListImplementation::setAutoCompress(compress);
}
auto xpress::objects::LinTermList::isAutoCompress() -> bool {
  return LinTermListImplementation::isAutoCompress();
}
xpress::objects::LinTermList::LinTermList()
    : LinExpression(std::make_shared<LinTermListImplementation>()) {}
xpress::objects::LinTermList::LinTermList(double c)
    : LinExpression(std::make_shared<LinTermListImplementation>(c)) {}
xpress::objects::LinTermList::LinTermList(double c, int capacity)
    : LinExpression(std::make_shared<LinTermListImplementation>(c, capacity)) {}
auto xpress::objects::LinTermList::deepCopy()
    -> xpress::objects::LinExpression {
  return impl()->deepCopy();
}
auto xpress::objects::LinTermList::deepCopy(double factor)
    -> xpress::objects::LinExpression {
  return impl()->deepCopy(factor);
}
auto xpress::objects::LinTermList::prepareToAdd(int newTerms) -> void {
  impl()->prepareToAdd(newTerms);
}
auto xpress::objects::LinTermList::reserve(int l)
    -> xpress::objects::LinTermList & {
  impl()->reserve(l);
  return *this;
}
auto xpress::objects::LinTermList::reset() -> xpress::objects::LinTermList & {
  impl()->reset();
  return *this;
}
auto xpress::objects::LinTermList::addTerm(Variable const &variable,
                                           double const &coefficient)
    -> xpress::objects::LinTermList & {
  impl()->addTerm(variable, coefficient);
  return *this;
}
auto xpress::objects::LinTermList::addTerm(double const &coefficient,
                                           Variable const &variable)
    -> xpress::objects::LinTermList & {
  impl()->addTerm(coefficient, variable);
  return *this;
}
auto xpress::objects::LinTermList::addTerm(Variable const &x)
    -> xpress::objects::LinTermList & {
  impl()->addTerm(x);
  return *this;
}
auto xpress::objects::LinTermList::addTerms(LinExpression const &expr)
    -> xpress::objects::LinTermList & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::LinTermList::addTerms(LinExpression const &expr,
                                            double factor)
    -> xpress::objects::LinTermList & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::LinTermList::addConstant(double c)
    -> xpress::objects::LinTermList & {
  impl()->addConstant(c);
  return *this;
}
auto xpress::objects::LinTermList::setConstant(double c)
    -> xpress::objects::LinTermList & {
  impl()->setConstant(c);
  return *this;
}
auto xpress::objects::LinTermList::getConstant() const -> double {
  return impl()->getConstant();
}
auto xpress::objects::LinTermList::normalizeTerms()
    -> xpress::objects::LinTermList & {
  impl()->normalizeTerms();
  return *this;
}
auto xpress::objects::LinTermList::compress() const -> void {
  impl()->compress();
}
auto xpress::objects::LinTermList::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::LinTermList::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::LinTermList::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::LinTermList::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::LinTermList::toString() const -> std::string {
  return impl()->toString();
}
xpress::objects::LinTermMap::LinTermMap(
    std::shared_ptr<LinTermMapImplementation> init)
    : LinExpression(init) {}
xpress::objects::LinTermMap::LinTermMap(LinTermMapImplementation const *init)
    : LinExpression(init) {}
xpress::objects::LinTermMapImplementation *xpress::objects::LinTermMap::impl() {
  return dynamic_cast<LinTermMapImplementation *>(implPointer.get());
}
xpress::objects::LinTermMapImplementation const *
xpress::objects::LinTermMap::impl() const {
  return dynamic_cast<LinTermMapImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::LinTermMap::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::objects::LinTermMap::LinTermMap()
    : LinExpression(std::make_shared<LinTermMapImplementation>()) {}
xpress::objects::LinTermMap::LinTermMap(double constant)
    : LinExpression(std::make_shared<LinTermMapImplementation>(constant)) {}
auto xpress::objects::LinTermMap::deepCopy() -> xpress::objects::LinExpression {
  return impl()->deepCopy();
}
auto xpress::objects::LinTermMap::deepCopy(double factor)
    -> xpress::objects::LinExpression {
  return impl()->deepCopy(factor);
}
auto xpress::objects::LinTermMap::reset() -> xpress::objects::LinTermMap & {
  impl()->reset();
  return *this;
}
auto xpress::objects::LinTermMap::addTerm(Variable const &variable,
                                          double const &coefficient)
    -> xpress::objects::LinTermMap & {
  impl()->addTerm(variable, coefficient);
  return *this;
}
auto xpress::objects::LinTermMap::addTerm(double const &coefficient,
                                          Variable const &variable)
    -> xpress::objects::LinTermMap & {
  impl()->addTerm(coefficient, variable);
  return *this;
}
auto xpress::objects::LinTermMap::addTerm(Variable const &x)
    -> xpress::objects::LinTermMap & {
  impl()->addTerm(x);
  return *this;
}
auto xpress::objects::LinTermMap::addTerms(LinExpression const &expr)
    -> xpress::objects::LinTermMap & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::LinTermMap::addTerms(LinExpression const &expr,
                                           double factor)
    -> xpress::objects::LinTermMap & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::LinTermMap::setCoefficient(Variable variable,
                                                 double coefficient)
    -> xpress::objects::LinTermMap & {
  impl()->setCoefficient(variable, coefficient);
  return *this;
}
auto xpress::objects::LinTermMap::getCoefficient(Variable variable) const
    -> double {
  return impl()->getCoefficient(variable);
}
auto xpress::objects::LinTermMap::addConstant(double c)
    -> xpress::objects::LinTermMap & {
  impl()->addConstant(c);
  return *this;
}
auto xpress::objects::LinTermMap::setConstant(double c)
    -> xpress::objects::LinTermMap & {
  impl()->setConstant(c);
  return *this;
}
auto xpress::objects::LinTermMap::getConstant() const -> double {
  return impl()->getConstant();
}
auto xpress::objects::LinTermMap::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::LinTermMap::extract(
    double factor, xpress::objects::PostfixExtractor *extractor) const -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::LinTermMap::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::LinTermMap::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::LinTermMap::toString() const -> std::string {
  return impl()->toString();
}
xpress::objects::QuadTermList::QuadTermList(
    std::shared_ptr<QuadTermListImplementation> init)
    : QuadExpression(init) {}
xpress::objects::QuadTermList::QuadTermList(
    QuadTermListImplementation const *init)
    : QuadExpression(init) {}
xpress::objects::QuadTermListImplementation *
xpress::objects::QuadTermList::impl() {
  return dynamic_cast<QuadTermListImplementation *>(implPointer.get());
}
xpress::objects::QuadTermListImplementation const *
xpress::objects::QuadTermList::impl() const {
  return dynamic_cast<QuadTermListImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::QuadTermList::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::QuadTermList::setAutoCompress(bool compress) -> void {
  QuadTermListImplementation::setAutoCompress(compress);
}
auto xpress::objects::QuadTermList::isAutoCompress() -> bool {
  return QuadTermListImplementation::isAutoCompress();
}
xpress::objects::QuadTermList::QuadTermList()
    : QuadExpression(std::make_shared<QuadTermListImplementation>()) {}
xpress::objects::QuadTermList::QuadTermList(double c)
    : QuadExpression(std::make_shared<QuadTermListImplementation>(c)) {}
xpress::objects::QuadTermList::QuadTermList(double c, int capacity)
    : QuadExpression(
          std::make_shared<QuadTermListImplementation>(c, capacity)) {}
auto xpress::objects::QuadTermList::deepCopy()
    -> xpress::objects::QuadExpression {
  return impl()->deepCopy();
}
auto xpress::objects::QuadTermList::deepCopy(double factor)
    -> xpress::objects::QuadExpression {
  return impl()->deepCopy(factor);
}
auto xpress::objects::QuadTermList::reserve(int l)
    -> xpress::objects::QuadTermList & {
  impl()->reserve(l);
  return *this;
}
auto xpress::objects::QuadTermList::reset() -> xpress::objects::QuadTermList & {
  impl()->reset();
  return *this;
}
auto xpress::objects::QuadTermList::addTerm(Variable const &variable,
                                            double const &coefficient)
    -> xpress::objects::QuadTermList & {
  impl()->addTerm(variable, coefficient);
  return *this;
}
auto xpress::objects::QuadTermList::addTerm(double const &coefficient,
                                            Variable const &variable)
    -> xpress::objects::QuadTermList & {
  impl()->addTerm(coefficient, variable);
  return *this;
}
auto xpress::objects::QuadTermList::addTerm(Variable const &x)
    -> xpress::objects::QuadTermList & {
  impl()->addTerm(x);
  return *this;
}
auto xpress::objects::QuadTermList::addTerm(Variable const &variable1,
                                            Variable const &variable2,
                                            double const &coefficient)
    -> xpress::objects::QuadTermList & {
  impl()->addTerm(variable1, variable2, coefficient);
  return *this;
}
auto xpress::objects::QuadTermList::addTerm(double const &coefficient,
                                            Variable const &variable1,
                                            Variable const &variable2)
    -> xpress::objects::QuadTermList & {
  impl()->addTerm(coefficient, variable1, variable2);
  return *this;
}
auto xpress::objects::QuadTermList::addTerm(Variable const &x1,
                                            Variable const &x2)
    -> xpress::objects::QuadTermList & {
  impl()->addTerm(x1, x2);
  return *this;
}
auto xpress::objects::QuadTermList::addTerms(LinExpression const &expr)
    -> xpress::objects::QuadTermList & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::QuadTermList::addTerms(QuadExpression const &expr)
    -> xpress::objects::QuadTermList & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::QuadTermList::addTerms(LinExpression const &expr,
                                             double const &factor)
    -> xpress::objects::QuadTermList & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::QuadTermList::addTerms(QuadExpression const &expr,
                                             double const &factor)
    -> xpress::objects::QuadTermList & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::QuadTermList::addConstant(double c)
    -> xpress::objects::QuadTermList & {
  impl()->addConstant(c);
  return *this;
}
auto xpress::objects::QuadTermList::setConstant(double c)
    -> xpress::objects::QuadTermList & {
  impl()->setConstant(c);
  return *this;
}
auto xpress::objects::QuadTermList::getConstant() const -> double {
  return impl()->getConstant();
}
auto xpress::objects::QuadTermList::normalizeTerms()
    -> xpress::objects::QuadTermList & {
  impl()->normalizeTerms();
  return *this;
}
auto xpress::objects::QuadTermList::compress() const -> void {
  impl()->compress();
}
auto xpress::objects::QuadTermList::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::QuadTermList::extract(double factor,
                                            PostfixExtractor *extractor) const
    -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::QuadTermList::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::QuadTermList::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::QuadTermList::toString() const -> std::string {
  return impl()->toString();
}
xpress::objects::QuadTermMap::QuadTermMap(
    std::shared_ptr<QuadTermMapImplementation> init)
    : QuadExpression(init) {}
xpress::objects::QuadTermMap::QuadTermMap(QuadTermMapImplementation const *init)
    : QuadExpression(init) {}
xpress::objects::QuadTermMapImplementation *
xpress::objects::QuadTermMap::impl() {
  return dynamic_cast<QuadTermMapImplementation *>(implPointer.get());
}
xpress::objects::QuadTermMapImplementation const *
xpress::objects::QuadTermMap::impl() const {
  return dynamic_cast<QuadTermMapImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::objects::QuadTermMap::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::objects::QuadTermMap::QuadTermMap()
    : QuadExpression(std::make_shared<QuadTermMapImplementation>()) {}
xpress::objects::QuadTermMap::QuadTermMap(double constant)
    : QuadExpression(std::make_shared<QuadTermMapImplementation>(constant)) {}
auto xpress::objects::QuadTermMap::deepCopy()
    -> xpress::objects::QuadExpression {
  return impl()->deepCopy();
}
auto xpress::objects::QuadTermMap::deepCopy(double factor)
    -> xpress::objects::QuadExpression {
  return impl()->deepCopy(factor);
}
auto xpress::objects::QuadTermMap::reset() -> xpress::objects::QuadTermMap & {
  impl()->reset();
  return *this;
}
auto xpress::objects::QuadTermMap::addTerm(Variable const &variable,
                                           double const &coefficient)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerm(variable, coefficient);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerm(double const &coefficient,
                                           Variable const &variable)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerm(coefficient, variable);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerm(Variable const &x)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerm(x);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerm(Variable const &variable1,
                                           Variable const &variable2,
                                           double const &coefficient)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerm(variable1, variable2, coefficient);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerm(double const &coefficient,
                                           Variable const &variable1,
                                           Variable const &variable2)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerm(coefficient, variable1, variable2);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerm(Variable const &x1,
                                           Variable const &x2)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerm(x1, x2);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerms(LinExpression const &expr)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerms(QuadExpression const &expr)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerms(expr);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerms(LinExpression const &expr,
                                            double const &factor)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::QuadTermMap::addTerms(QuadExpression const &expr,
                                            double const &factor)
    -> xpress::objects::QuadTermMap & {
  impl()->addTerms(expr, factor);
  return *this;
}
auto xpress::objects::QuadTermMap::setCoefficient(Variable variable,
                                                  double coefficient)
    -> xpress::objects::QuadTermMap & {
  impl()->setCoefficient(variable, coefficient);
  return *this;
}
auto xpress::objects::QuadTermMap::setCoefficient(Variable variable1,
                                                  Variable variable2,
                                                  double coefficient)
    -> xpress::objects::QuadTermMap & {
  impl()->setCoefficient(variable1, variable2, coefficient);
  return *this;
}
auto xpress::objects::QuadTermMap::getCoefficient(Variable variable) const
    -> double {
  return impl()->getCoefficient(variable);
}
auto xpress::objects::QuadTermMap::getCoefficient(Variable variable1,
                                                  Variable variable2) const
    -> double {
  return impl()->getCoefficient(variable1, variable2);
}
auto xpress::objects::QuadTermMap::addConstant(double c)
    -> xpress::objects::QuadTermMap & {
  impl()->addConstant(c);
  return *this;
}
auto xpress::objects::QuadTermMap::setConstant(double c)
    -> xpress::objects::QuadTermMap & {
  impl()->setConstant(c);
  return *this;
}
auto xpress::objects::QuadTermMap::getConstant() const -> double {
  return impl()->getConstant();
}
auto xpress::objects::QuadTermMap::extract(
    double factor, xpress::XPRSProblem::RowCreator *creator) const -> bool {
  return impl()->extract(factor, creator);
}
auto xpress::objects::QuadTermMap::extract(double factor,
                                           PostfixExtractor *extractor) const
    -> void {
  impl()->extract(factor, extractor);
}
auto xpress::objects::QuadTermMap::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::QuadTermMap::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::QuadTermMap::toString() const -> std::string {
  return impl()->toString();
}
xpress::objects::BinaryExpression::BinaryExpression(
    std::shared_ptr<BinaryExpressionImplementation> init)
    : FormulaExpression(init) {}
xpress::objects::BinaryExpression::BinaryExpression(
    BinaryExpressionImplementation const *init)
    : FormulaExpression(init) {}
xpress::objects::BinaryExpressionImplementation *
xpress::objects::BinaryExpression::impl() {
  return dynamic_cast<BinaryExpressionImplementation *>(implPointer.get());
}
xpress::objects::BinaryExpressionImplementation const *
xpress::objects::BinaryExpression::impl() const {
  return dynamic_cast<BinaryExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::BinaryExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::BinaryExpression::getLeft() const
    -> xpress::objects::Expression {
  return impl()->getLeft();
}
auto xpress::objects::BinaryExpression::getOp() const
    -> xpress::objects::BinaryExpression::Operator {
  return impl()->getOp();
}
auto xpress::objects::BinaryExpression::getRight() const
    -> xpress::objects::Expression {
  return impl()->getRight();
}
xpress::objects::BinaryExpression::BinaryExpression(
    xpress::objects::Expression left,
    xpress::objects::BinaryExpression::Operator op,
    xpress::objects::Expression right)
    : FormulaExpression(
          std::make_shared<BinaryExpressionImplementation>(left, op, right)) {}
auto xpress::objects::BinaryExpression::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::BinaryExpression::extract(
    PostfixExtractor *extractor) const -> void {
  impl()->extract(extractor);
}
auto xpress::objects::BinaryExpression::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::BinaryExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::BinaryExpression::getConstantView() const -> double {
  return impl()->getConstantView();
}
auto xpress::objects::BinaryExpression::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::BinaryExpression::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
xpress::objects::BinaryExpression::BinaryExpression()
    : FormulaExpression(nullptr) {}
xpress::objects::InternalFunctionExpression::InternalFunctionExpression(
    std::shared_ptr<InternalFunctionExpressionImplementation> init)
    : FormulaExpression(init) {}
xpress::objects::InternalFunctionExpression::InternalFunctionExpression(
    InternalFunctionExpressionImplementation const *init)
    : FormulaExpression(init) {}
xpress::objects::InternalFunctionExpressionImplementation *
xpress::objects::InternalFunctionExpression::impl() {
  return dynamic_cast<InternalFunctionExpressionImplementation *>(
      implPointer.get());
}
xpress::objects::InternalFunctionExpressionImplementation const *
xpress::objects::InternalFunctionExpression::impl() const {
  return dynamic_cast<InternalFunctionExpressionImplementation *>(
      implPointer.get());
}
template <typename Derived>
Derived xpress::objects::InternalFunctionExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::InternalFunctionExpression::getFunction() const
    -> xpress::objects::InternalFunctionExpression::FunctionType {
  return impl()->getFunction();
}
auto xpress::objects::InternalFunctionExpression::getArguments() const
    -> std::vector<xpress::objects::Expression> {
  return impl()->getArguments();
}
xpress::objects::InternalFunctionExpression::InternalFunctionExpression(
    xpress::objects::InternalFunctionExpression::FunctionType function,
    std::vector<xpress::objects::Expression> arguments)
    : FormulaExpression(
          std::make_shared<InternalFunctionExpressionImplementation>(
              function, arguments)) {}
xpress::objects::InternalFunctionExpression::InternalFunctionExpression(
    xpress::objects::InternalFunctionExpression::FunctionType function,
    xpress::objects::Expression argument)
    : FormulaExpression(
          std::make_shared<InternalFunctionExpressionImplementation>(
              function, argument)) {}
xpress::objects::InternalFunctionExpression::InternalFunctionExpression(
    xpress::objects::InternalFunctionExpression::FunctionType function,
    xpress::objects::Expression firstArgument,
    xpress::objects::Expression secondArgument)
    : FormulaExpression(
          std::make_shared<InternalFunctionExpressionImplementation>(
              function, firstArgument, secondArgument)) {}
auto xpress::objects::InternalFunctionExpression::collectArguments(
    Expression firstArgument, Expression secondArgument,
    xpress::SizedArray<Expression const> const &more)
    -> std::vector<xpress::objects::Expression> {
  return InternalFunctionExpressionImplementation::collectArguments(
      firstArgument, secondArgument, more);
}
template <typename... MOREARGUMENTS, typename RestrictPack>
xpress::objects::InternalFunctionExpression::InternalFunctionExpression(
    xpress::objects::InternalFunctionExpression::FunctionType function,
    xpress::objects::Expression firstArgument,
    xpress::objects::Expression secondArgument, MOREARGUMENTS... moreArguments)
    : FormulaExpression(
          std::make_shared<InternalFunctionExpressionImplementation>(
              function, firstArgument, secondArgument, moreArguments...)) {}
xpress::objects::InternalFunctionExpression::InternalFunctionExpression(
    xpress::objects::InternalFunctionExpression::FunctionType function,
    xpress::SizedArray<Expression const> const &args)
    : FormulaExpression(
          std::make_shared<InternalFunctionExpressionImplementation>(function,
                                                                     args)) {}
auto xpress::objects::InternalFunctionExpression::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::InternalFunctionExpression::extract(
    PostfixExtractor *extractor) const -> void {
  impl()->extract(extractor);
}
auto xpress::objects::InternalFunctionExpression::toString() const
    -> std::string {
  return impl()->toString();
}
auto xpress::objects::InternalFunctionExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
xpress::objects::InternalFunctionExpression::InternalFunctionExpression()
    : FormulaExpression(nullptr) {}
xpress::objects::PWLExpression::PWLExpression(
    std::shared_ptr<PWLExpressionImplementation> init)
    : FormulaExpression(init) {}
xpress::objects::PWLExpression::PWLExpression(
    PWLExpressionImplementation const *init)
    : FormulaExpression(init) {}
xpress::objects::PWLExpressionImplementation *
xpress::objects::PWLExpression::impl() {
  return dynamic_cast<PWLExpressionImplementation *>(implPointer.get());
}
xpress::objects::PWLExpressionImplementation const *
xpress::objects::PWLExpression::impl() const {
  return dynamic_cast<PWLExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::PWLExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::PWLExpression::breakToX(
    xpress::SizedArray<PwlBreakpoint const> const &points)
    -> std::vector<double> {
  return PWLExpressionImplementation::breakToX(points);
}
auto xpress::objects::PWLExpression::breakToY(
    xpress::SizedArray<PwlBreakpoint const> const &points)
    -> std::vector<double> {
  return PWLExpressionImplementation::breakToY(points);
}
auto xpress::objects::PWLExpression::getArgumentColumn() const -> int {
  return impl()->getArgumentColumn();
}
auto xpress::objects::PWLExpression::getArgumentVariable() const
    -> xpress::objects::Variable {
  return impl()->getArgumentVariable();
}
auto xpress::objects::PWLExpression::getBreakX() const -> std::vector<double> {
  return impl()->getBreakX();
}
auto xpress::objects::PWLExpression::getBreakY() const -> std::vector<double> {
  return impl()->getBreakY();
}
xpress::objects::PWLExpression::PWLExpression(
    Variable argument, xpress::SizedArray<double const> const &breakX,
    xpress::SizedArray<double const> const &breakY)
    : FormulaExpression(std::make_shared<PWLExpressionImplementation>(
          argument, breakX, breakY)) {}
template <typename... POINTS, typename RestrictPack>
xpress::objects::PWLExpression::PWLExpression(Variable argument,
                                              POINTS... points)
    : FormulaExpression(
          std::make_shared<PWLExpressionImplementation>(argument, points...)) {}
xpress::objects::PWLExpression::PWLExpression(
    Variable argument, xpress::SizedArray<PwlBreakpoint const> const &points)
    : FormulaExpression(
          std::make_shared<PWLExpressionImplementation>(argument, points)) {}
xpress::objects::PWLExpression::PWLExpression(
    int argument, xpress::SizedArray<double const> const &breakX,
    xpress::SizedArray<double const> const &breakY)
    : FormulaExpression(std::make_shared<PWLExpressionImplementation>(
          argument, breakX, breakY)) {}
template <typename... POINTS, typename RestrictPack>
xpress::objects::PWLExpression::PWLExpression(int argument, POINTS... points)
    : FormulaExpression(
          std::make_shared<PWLExpressionImplementation>(argument, points...)) {}
xpress::objects::PWLExpression::PWLExpression(
    int argument, xpress::SizedArray<PwlBreakpoint const> const &points)
    : FormulaExpression(
          std::make_shared<PWLExpressionImplementation>(argument, points)) {}
auto xpress::objects::PWLExpression::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::PWLExpression::extract(PostfixExtractor *extractor) const
    -> void {
  impl()->extract(extractor);
}
auto xpress::objects::PWLExpression::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::PWLExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
xpress::objects::PWLExpression::PWLExpression() : FormulaExpression(nullptr) {}
xpress::objects::UnaryExpression::UnaryExpression(
    std::shared_ptr<UnaryExpressionImplementation> init)
    : FormulaExpression(init) {}
xpress::objects::UnaryExpression::UnaryExpression(
    UnaryExpressionImplementation const *init)
    : FormulaExpression(init) {}
xpress::objects::UnaryExpressionImplementation *
xpress::objects::UnaryExpression::impl() {
  return dynamic_cast<UnaryExpressionImplementation *>(implPointer.get());
}
xpress::objects::UnaryExpressionImplementation const *
xpress::objects::UnaryExpression::impl() const {
  return dynamic_cast<UnaryExpressionImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::objects::UnaryExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::UnaryExpression::getOp() const
    -> xpress::objects::UnaryExpression::Operator {
  return impl()->getOp();
}
auto xpress::objects::UnaryExpression::getArgument() const
    -> xpress::objects::Expression {
  return impl()->getArgument();
}
xpress::objects::UnaryExpression::UnaryExpression(
    xpress::objects::UnaryExpression::Operator op,
    xpress::objects::Expression argument)
    : FormulaExpression(
          std::make_shared<UnaryExpressionImplementation>(op, argument)) {}
auto xpress::objects::UnaryExpression::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::UnaryExpression::extract(
    PostfixExtractor *extractor) const -> void {
  impl()->extract(extractor);
}
auto xpress::objects::UnaryExpression::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::UnaryExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
auto xpress::objects::UnaryExpression::getConstantView() const -> double {
  return impl()->getConstantView();
}
auto xpress::objects::UnaryExpression::getLinearView() const
    -> std::shared_ptr<xpress::objects::LinearView> {
  return impl()->getLinearView();
}
auto xpress::objects::UnaryExpression::getQuadView() const
    -> std::shared_ptr<xpress::objects::QuadView> {
  return impl()->getQuadView();
}
auto xpress::objects::UnaryExpression::scaledLinView(
    std::shared_ptr<LinearView> data, double factor)
    -> std::shared_ptr<xpress::objects::LinearView> {
  return UnaryExpressionImplementation::scaledLinView(data, factor);
}
auto xpress::objects::UnaryExpression::scaledQuadView(
    std::shared_ptr<QuadView> data, double factor)
    -> std::shared_ptr<xpress::objects::QuadView> {
  return UnaryExpressionImplementation::scaledQuadView(data, factor);
}
xpress::objects::UnaryExpression::UnaryExpression()
    : FormulaExpression(nullptr) {}
xpress::objects::UserFunctionExpression::UserFunctionExpression(
    std::shared_ptr<UserFunctionExpressionImplementation> init)
    : FormulaExpression(init) {}
xpress::objects::UserFunctionExpression::UserFunctionExpression(
    UserFunctionExpressionImplementation const *init)
    : FormulaExpression(init) {}
xpress::objects::UserFunctionExpressionImplementation *
xpress::objects::UserFunctionExpression::impl() {
  return dynamic_cast<UserFunctionExpressionImplementation *>(
      implPointer.get());
}
xpress::objects::UserFunctionExpressionImplementation const *
xpress::objects::UserFunctionExpression::impl() const {
  return dynamic_cast<UserFunctionExpressionImplementation *>(
      implPointer.get());
}
template <typename Derived>
Derived xpress::objects::UserFunctionExpression::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
auto xpress::objects::UserFunctionExpression::getFunction() const
    -> xpress::XPRSProblem::AbstractUserFunction * {
  return impl()->getFunction();
}
auto xpress::objects::UserFunctionExpression::getArguments() const
    -> std::vector<xpress::objects::Expression> {
  return impl()->getArguments();
}
auto xpress::objects::UserFunctionExpression::getOutput() const -> int {
  return impl()->getOutput();
}
template <typename... ARGUMENTS, typename RestrictPack>
xpress::objects::UserFunctionExpression::UserFunctionExpression(
    xpress::XPRSProblem::AbstractUserFunction *function, ARGUMENTS... arguments)
    : FormulaExpression(std::make_shared<UserFunctionExpressionImplementation>(
          function, arguments...)) {}
template <typename... ARGUMENTS, typename RestrictPack>
xpress::objects::UserFunctionExpression::UserFunctionExpression(
    xpress::XPRSProblem::AbstractUserFunction *function, int output,
    ARGUMENTS... arguments)
    : FormulaExpression(std::make_shared<UserFunctionExpressionImplementation>(
          function, output, arguments...)) {}
xpress::objects::UserFunctionExpression::UserFunctionExpression(
    xpress::XPRSProblem::AbstractUserFunction *function,
    SizedArray<Expression> const &arguments)
    : FormulaExpression(std::make_shared<UserFunctionExpressionImplementation>(
          function, arguments)) {}
xpress::objects::UserFunctionExpression::UserFunctionExpression(
    xpress::XPRSProblem::AbstractUserFunction *function, int output,
    SizedArray<Expression> const &arguments)
    : FormulaExpression(std::make_shared<UserFunctionExpressionImplementation>(
          function, output, arguments)) {}
auto xpress::objects::UserFunctionExpression::evaluate(
    xpress::SizedArray<double const> const &solution) const -> double {
  return impl()->evaluate(solution);
}
auto xpress::objects::UserFunctionExpression::extract(
    PostfixExtractor *extractor) const -> void {
  impl()->extract(extractor);
}
auto xpress::objects::UserFunctionExpression::toString() const -> std::string {
  return impl()->toString();
}
auto xpress::objects::UserFunctionExpression::getMaxDegree() const -> int {
  return impl()->getMaxDegree();
}
xpress::objects::UserFunctionExpression::UserFunctionExpression()
    : FormulaExpression(nullptr) {}
namespace xpress {
namespace objects {

class XpressProblem : public XPRSProblem {
public:
  static void console(XpressProblem &prob, char const *msg, int msglen,
                      int msgtype) {
    (void)prob;
    if (!msglen)
      msg = "";
    if (msgtype < 0) {
      std::cout << std::flush;
      std::cerr << std::flush;
    } else {
      switch (msgtype) {
      case 1:
        std::cout << msg << std::endl;
        break;
      case 3:
        std::cout << msg << std::endl;
        break;
      case 4:
        std::cerr << msg << std::endl;
        break;
      }
    }
  }

  class CallbackAPI {
  private:
    XpressProblem *prob;

  public:
    CallbackAPI(XpressProblem *prob) : prob(prob) {}

    inline auto addLplogCallback(std::function<int(XpressProblem &)> callback,
                                 int prio = 0) -> CallbackHandle;

    inline void removeLplogCallback(CallbackHandle const &callback);

    inline void removeLplogCallbacks();

    inline auto addMiplogCallback(std::function<int(XpressProblem &)> callback,
                                  int prio = 0) -> CallbackHandle;

    inline void removeMiplogCallback(CallbackHandle const &callback);

    inline void removeMiplogCallbacks();

    inline auto addCutlogCallback(std::function<int(XpressProblem &)> callback,
                                  int prio = 0) -> CallbackHandle;

    inline void removeCutlogCallback(CallbackHandle const &callback);

    inline void removeCutlogCallbacks();

    inline auto addBarlogCallback(std::function<int(XpressProblem &)> callback,
                                  int prio = 0) -> CallbackHandle;

    inline void removeBarlogCallback(CallbackHandle const &callback);

    inline void removeBarlogCallbacks();

    inline auto
    addOptnodeCallback(std::function<void(XpressProblem &, int *)> callback,
                       int prio = 0) -> CallbackHandle;

    inline void removeOptnodeCallback(CallbackHandle const &callback);

    inline void removeOptnodeCallbacks();

    inline auto
    addPrenodeCallback(std::function<void(XpressProblem &, int *)> callback,
                       int prio = 0) -> CallbackHandle;

    inline void removePrenodeCallback(CallbackHandle const &callback);

    inline void removePrenodeCallbacks();

    inline auto
    addInfnodeCallback(std::function<void(XpressProblem &)> callback,
                       int prio = 0) -> CallbackHandle;

    inline void removeInfnodeCallback(CallbackHandle const &callback);

    inline void removeInfnodeCallbacks();

    inline auto
    addNodecutoffCallback(std::function<void(XpressProblem &, int)> callback,
                          int prio = 0) -> CallbackHandle;

    inline void removeNodecutoffCallback(CallbackHandle const &callback);

    inline void removeNodecutoffCallbacks();

    inline auto addIntsolCallback(std::function<void(XpressProblem &)> callback,
                                  int prio = 0) -> CallbackHandle;

    inline void removeIntsolCallback(CallbackHandle const &callback);

    inline void removeIntsolCallbacks();

    inline auto addPreIntsolCallback(
        std::function<void(XpressProblem &, int, int *, double *)> callback,
        int prio = 0) -> CallbackHandle;

    inline void removePreIntsolCallback(CallbackHandle const &callback);

    inline void removePreIntsolCallbacks();

    inline auto addMessageCallback(
        std::function<void(XpressProblem &, char const *, int, int)> callback,
        int prio = 0) -> CallbackHandle;

    inline void removeMessageCallback(CallbackHandle const &callback);

    inline void removeMessageCallbacks();

    inline auto addMipThreadCallback(
        std::function<void(XpressProblem &, XpressProblem &)> callback,
        int prio = 0) -> CallbackHandle;

    inline void removeMipThreadCallback(CallbackHandle const &callback);

    inline void removeMipThreadCallbacks();

    inline auto
    addMipThreadDestroyCallback(std::function<void(XpressProblem &)> callback,
                                int prio = 0) -> CallbackHandle;

    inline void removeMipThreadDestroyCallback(CallbackHandle const &callback);

    inline void removeMipThreadDestroyCallbacks();

    inline auto addNewnodeCallback(
        std::function<void(XpressProblem &, int, int, int)> callback,
        int prio = 0) -> CallbackHandle;

    inline void removeNewnodeCallback(CallbackHandle const &callback);

    inline void removeNewnodeCallbacks();

    inline auto addBarIterationCallback(
        std::function<void(XpressProblem &, int *)> callback, int prio = 0)
        -> CallbackHandle;

    inline void removeBarIterationCallback(CallbackHandle const &callback);

    inline void removeBarIterationCallbacks();

    inline auto
    addPresolveCallback(std::function<void(XpressProblem &)> callback,
                        int prio = 0) -> CallbackHandle;

    inline void removePresolveCallback(CallbackHandle const &callback);

    inline void removePresolveCallbacks();

    inline auto addChangeBranchObjectCallback(
        std::function<void(XpressProblem &, xpress::BranchObject *,
                           xpress::BranchObject **)>
            callback,
        int prio = 0) -> CallbackHandle;

    inline void
    removeChangeBranchObjectCallback(CallbackHandle const &callback);

    inline void removeChangeBranchObjectCallbacks();

    inline auto
    addComputeRestartCallback(std::function<void(XpressProblem &)> callback,
                              int prio = 0) -> CallbackHandle;

    inline void removeComputeRestartCallback(CallbackHandle const &callback);

    inline void removeComputeRestartCallbacks();

    inline auto
    addNodeLPSolvedCallback(std::function<void(XpressProblem &)> callback,
                            int prio = 0) -> CallbackHandle;

    inline void removeNodeLPSolvedCallback(CallbackHandle const &callback);

    inline void removeNodeLPSolvedCallbacks();

    inline auto
    addGapNotifyCallback(std::function<void(XpressProblem &, double *, double *,
                                            double *, double *)>
                             callback,
                         int prio = 0) -> CallbackHandle;

    inline void removeGapNotifyCallback(CallbackHandle const &callback);

    inline void removeGapNotifyCallbacks();

    inline auto addUserSolNotifyCallback(
        std::function<void(XpressProblem &, char const *, int)> callback,
        int prio = 0) -> CallbackHandle;

    inline void removeUserSolNotifyCallback(CallbackHandle const &callback);

    inline void removeUserSolNotifyCallbacks();

    inline auto
    addBeforeSolveCallback(std::function<void(XpressProblem &)> callback,
                           int prio = 0) -> CallbackHandle;

    inline void removeBeforeSolveCallback(CallbackHandle const &callback);

    inline void removeBeforeSolveCallbacks();

    inline auto
    addBeforeObjectiveCallback(std::function<void(XpressProblem &)> callback,
                               int prio = 0) -> CallbackHandle;

    inline void removeBeforeObjectiveCallback(CallbackHandle const &callback);

    inline void removeBeforeObjectiveCallbacks();

    inline auto
    addAfterObjectiveCallback(std::function<void(XpressProblem &)> callback,
                              int prio = 0) -> CallbackHandle;

    inline void removeAfterObjectiveCallback(CallbackHandle const &callback);

    inline void removeAfterObjectiveCallbacks();

    inline auto
    addCheckTimeCallback(std::function<int(XpressProblem &)> callback,
                         int prio = 0) -> CallbackHandle;

    inline void removeCheckTimeCallback(CallbackHandle const &callback);

    inline void removeCheckTimeCallbacks();

    inline auto addCutRoundCallback(
        std::function<void(XpressProblem &, int, int *)> callback, int prio = 0)
        -> CallbackHandle;

    inline void removeCutRoundCallback(CallbackHandle const &callback);

    inline void removeCutRoundCallbacks();

    inline auto
    addSlpCascadeEndCallback(std::function<int(XpressProblem &)> callback,
                             int prio = 0) -> CallbackHandle;

    inline void removeSlpCascadeEndCallback(CallbackHandle const &callback);

    inline void removeSlpCascadeEndCallbacks();

    inline auto
    addSlpCascadeStartCallback(std::function<int(XpressProblem &)> callback,
                               int prio = 0) -> CallbackHandle;

    inline void removeSlpCascadeStartCallback(CallbackHandle const &callback);

    inline void removeSlpCascadeStartCallbacks();

    inline auto
    addSlpCascadeVarCallback(std::function<int(XpressProblem &, int)> callback,
                             int prio = 0) -> CallbackHandle;

    inline void removeSlpCascadeVarCallback(CallbackHandle const &callback);

    inline void removeSlpCascadeVarCallbacks();

    inline auto addSlpCascadeVarFailCallback(
        std::function<int(XpressProblem &, int)> callback, int prio = 0)
        -> CallbackHandle;

    inline void removeSlpCascadeVarFailCallback(CallbackHandle const &callback);

    inline void removeSlpCascadeVarFailCallbacks();

    inline auto
    addSlpConstructCallback(std::function<int(XpressProblem &)> callback,
                            int prio = 0) -> CallbackHandle;

    inline void removeSlpConstructCallback(CallbackHandle const &callback);

    inline void removeSlpConstructCallbacks();

    inline auto
    addSlpIntSolCallback(std::function<int(XpressProblem &)> callback,
                         int prio = 0) -> CallbackHandle;

    inline void removeSlpIntSolCallback(CallbackHandle const &callback);

    inline void removeSlpIntSolCallbacks();

    inline auto
    addSlpIterEndCallback(std::function<int(XpressProblem &)> callback,
                          int prio = 0) -> CallbackHandle;

    inline void removeSlpIterEndCallback(CallbackHandle const &callback);

    inline void removeSlpIterEndCallbacks();

    inline auto
    addSlpIterStartCallback(std::function<int(XpressProblem &)> callback,
                            int prio = 0) -> CallbackHandle;

    inline void removeSlpIterStartCallback(CallbackHandle const &callback);

    inline void removeSlpIterStartCallbacks();

    inline auto
    addSlpIterVarCallback(std::function<int(XpressProblem &, int)> callback,
                          int prio = 0) -> CallbackHandle;

    inline void removeSlpIterVarCallback(CallbackHandle const &callback);

    inline void removeSlpIterVarCallbacks();

    inline auto
    addSlpDrColCallback(std::function<int(XpressProblem &, int, int, double,
                                          double *, double, double)>
                            callback,
                        int prio = 0) -> CallbackHandle;

    inline void removeSlpDrColCallback(CallbackHandle const &callback);

    inline void removeSlpDrColCallbacks();

    inline auto addMsJobStartCallback(
        std::function<int(XpressProblem &, void *, char const *, int *)>
            callback,
        int prio = 0) -> CallbackHandle;

    inline void removeMsJobStartCallback(CallbackHandle const &callback);

    inline void removeMsJobStartCallbacks();

    inline auto addMsJobEndCallback(
        std::function<int(XpressProblem &, void *, char const *, int *)>
            callback,
        int prio = 0) -> CallbackHandle;

    inline void removeMsJobEndCallback(CallbackHandle const &callback);

    inline void removeMsJobEndCallbacks();

    inline auto addMsWinnerCallback(
        std::function<int(XpressProblem &, void *, char const *)> callback,
        int prio = 0) -> CallbackHandle;

    inline void removeMsWinnerCallback(CallbackHandle const &callback);

    inline void removeMsWinnerCallbacks();

    inline auto addNlpCoefEvalErrorCallback(
        std::function<int(XpressProblem &, int, int)> callback, int prio = 0)
        -> CallbackHandle;

    inline void removeNlpCoefEvalErrorCallback(CallbackHandle const &callback);

    inline void removeNlpCoefEvalErrorCallbacks();

    inline auto addSlpPreUpdateLinearizationCallback(
        std::function<int(XpressProblem &, int *)> callback, int prio = 0)
        -> CallbackHandle;

    inline void
    removeSlpPreUpdateLinearizationCallback(CallbackHandle const &callback);

    inline void removeSlpPreUpdateLinearizationCallbacks();
  };

  CallbackAPI callbacks;

public:
  using XPRSProblem::addMipSol;

  using XPRSProblem::bndSA;

  using XPRSProblem::chgBounds;

  using XPRSProblem::chgCoef;

  using XPRSProblem::chgColType;

  using XPRSProblem::chgObj;

  using XPRSProblem::chgObjN;

  using XPRSProblem::chgRhs;

  using XPRSProblem::chgRhsRange;

  using XPRSProblem::chgRowType;

  using XPRSProblem::delCols;

  using XPRSProblem::delGenCons;

  using XPRSProblem::delPwlCons;

  using XPRSProblem::delRows;

  using XPRSProblem::delSets;

  using XPRSProblem::getCallbackDual;

  using XPRSProblem::getCallbackDuals;

  using XPRSProblem::getCallbackRedCost;

  using XPRSProblem::getCallbackRedCosts;

  using XPRSProblem::getCallbackSlack;

  using XPRSProblem::getCallbackSlacks;

  using XPRSProblem::getCallbackSolution;

  using XPRSProblem::getCoef;

  using XPRSProblem::getDual;

  using XPRSProblem::getDuals;

  using XPRSProblem::getObj;

  using XPRSProblem::getObjN;

  using XPRSProblem::getRedCost;

  using XPRSProblem::getRedCosts;

  using XPRSProblem::getSlack;

  using XPRSProblem::getSlacks;

  using XPRSProblem::getSolution;

  using XPRSProblem::loadDelayedRows;

  using XPRSProblem::loadModelCuts;

  using XPRSProblem::nlpSetInitVal;

  using XPRSProblem::objSA;

  using XPRSProblem::repairWeightedInfeas;

  using XPRSProblem::rhsSA;

  using XPRSProblem::setIndicators;

  using XPRSProblem::slpSetDetRow;

public:
  XpressProblem *const toplevel;

  virtual inline auto getTopLevel() -> xpress::objects::XpressProblem * final;

  inline XpressProblem();

  inline XpressProblem(std::optional<std::string> problemName);

  inline XpressProblem(std::optional<std::string> problemName,
                       std::optional<std::string> licensePath);

  inline XpressProblem(XPRSprob callbackProblem, XpressProblem *toplevel);

  virtual inline auto makeChild(XPRSprob childPtr)
      -> xpress::objects::XpressProblem * override;

  inline auto isOriginal() -> bool;

private:
  class NoCommitRowCreator final : public RowCreator {
  public:
    inline NoCommitRowCreator(XPRSProblem *prob, int initialInequalitiesSize,
                              int initialNzsSize, bool hasNames,
                              bool errorIfVariables);

    virtual inline auto commit() -> void override;
  };

public:
  inline auto addCut(int cuttype, InequalityDefinition cut) -> int;

  inline auto addManagedCut(bool globallyValid, InequalityDefinition cut)
      -> void;

  class BranchObject : public xpress::BranchObject {
    friend class ChangeBranchObjectCallbackHolder;
  private:
    XpressProblem *const prob;

  public:
    inline BranchObject(XpressProblem *prob);

    inline auto addBounds(int branch, int nbounds,
                          xpress::SizedArray<char const> const &bndtype,
                          std::vector<xpress::objects::Variable> variables,
                          xpress::SizedArray<double const> const &bndval)
        -> void;

    inline auto addBound(int branch, char bndtype,
                         xpress::objects::Variable variable, double bndval)
        -> void;

    inline auto addConstraint(int branch, InequalityDefinition constraint)
        -> void;
  };

  inline auto createBranchObject()
      -> xpress::objects::XpressProblem::BranchObject *;

private:
  static inline auto
  fixupQObjForExpression(int length, xpress::SizedArray<double> const &qcoef)
      -> void;

public:
  inline auto setObjective(xpress::objects::Expression obj) -> void;

  inline auto setObjective(xpress::objects::Expression obj,
                           xpress::ObjSense sense) -> void;

private:
  virtual inline auto extractObjective(xpress::objects::Expression obj,
                                       bool isFirst)
      -> std::shared_ptr<xpress::XPRSProblem::RowCreator> final;

public:
  inline auto addObjective(xpress::objects::Expression obj, int priority,
                           double weight) -> void;

  inline auto setObjective(int idx, xpress::objects::Expression obj,
                           int priority, double weight, double abstol,
                           double reltol) -> void;

  inline auto nlpSetInitVal(std::unordered_map<Variable, double> values)
      -> void;

  inline auto nlpSetInitVal(xpress::SizedArray<Variable const> const &variables,
                            xpress::SizedArray<double const> const &values)
      -> void;

  inline auto addToRow(double factor, xpress::objects::Expression expr,
                       xpress::XPRSProblem::RowCreator *creator) -> bool;

private:
  template <typename T = xpress::objects::Index>
  static inline auto toIndexArray(XpressProblem *prob,
                                  xpress::SizedArray<T const> const &indices)
      -> std::vector<int>;

  template <typename T = xpress::objects::Index>
  static inline auto toIndexArray(int len, XpressProblem *prob,
                                  xpress::SizedArray<T const> const &indices)
      -> std::vector<int>;

  template <typename Strm0, typename T = typename xpress_elementtype(Strm0),
            xpress_isstream(Strm0)>
  static inline auto toIndexArray(XpressProblem *prob, Strm0 const &indices,
                                  int indicesLength) -> std::vector<int>;

  template <typename Strm0, typename T = typename xpress_elementtype(Strm0),
            xpress_isstream(Strm0)>
  static inline auto toIndexArray(XpressProblem *prob, Strm0 const &indices)
      -> std::vector<int>;

  template <typename Strm0, typename I = typename xpress_elementtype(Strm0),
            xpress_isstream(Strm0)>
  static inline auto getIndicesForProb(XpressProblem *prob, Strm0 const &objs,
                                       int objsSize) -> std::vector<int>;

  template <typename Strm0, typename I = typename xpress_elementtype(Strm0),
            xpress_isstream(Strm0)>
  static inline auto getIndicesForProb(XpressProblem *prob, Strm0 const &objs)
      -> std::vector<int>;

public:
  inline auto getLhsExpression(int row) -> xpress::objects::Expression;

private:
  inline auto checkOperation(char const *operation, char const *clazz) -> void;

private:
  template <typename T>
  static inline auto appendArray(std::vector<T> &array,
                                 xpress::SizedArray<T const> const &data)
      -> void;

  template <typename Strm0, typename C,
            typename T = typename xpress_elementtype(Strm0),
            xpress_isstream(Strm0)>
  static inline auto dereferenceIndices(XpressProblem *prob,
                                        IndexHandler<C> const *indexHandler,
                                        int oldCount, int newCount,
                                        Strm0 const &) -> xpress::Iterable<C>;

  template <typename R, typename C, typename I, typename Func0, typename Func1,
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0,
                               typename xpress_iteratorvaluetype(I) const &),
            xpress_isinvocable(
                Func1, R, Func1, XpressProblem *, IndexHandler<C> const *, int,
                int, xpress::Iterable<typename xpress_iteratorvaluetype(I)>)>
  inline auto makeConstraints(I dataBegin, I dataEnd, Func0 generator,
                              Func1 makeResult, int keyHint) -> R;

  template <typename C, typename I,
            typename T = typename xpress_iteratorvaluetype(I)>
  inline auto makeConstraints(I dataBegin, I dataEnd) -> xpress::Iterable<C>;

public:
  template <typename I, typename T = typename xpress_iteratorvaluetype(I),
            typename Func0,
            typename C = typename xpress_resultof(Func0, T)::constraint_type,
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, T)>
  inline auto addConstraints(I dataBegin, I dataEnd, Func0 defs)
      -> xpress::Iterable<std::pair<T, C>>;

  template <typename C>
  inline auto addConstraint(ConstraintDefinition<C> def) -> C;

  template <typename Strm0,
            typename C = typename Strm0::value_type::constraint_type,
            xpress_isstream(Strm0)>
  inline auto addConstraints(Strm0 const &defs) -> xpress::Iterable<C>;

  template <typename I, xpress_isintegral(I), typename Func0,
            typename C = typename xpress_resultof(Func0, I)::constraint_type,
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, I)>
  inline auto addConstraints(I const &count, Func0 defs) -> xpress::Iterable<C>;

  template <typename Strm0, typename T = typename xpress_elementtype(Strm0),
            typename Func0,
            typename C = typename xpress_resultof(Func0, T)::constraint_type,
            xpress_isstream(Strm0),
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0,
                               T const &)>
  inline auto addConstraints(Strm0 const &data, Func0 defs)
      -> xpress::Iterable<C>;

public:
  inline auto getGeneralConstraints()
      -> std::vector<xpress::objects::GeneralConstraint>;

  class GeneralConstraintHandler final
      : public IndexHandler<GeneralConstraint> {
  public:
    inline GeneralConstraintHandler();

  protected:
    virtual inline auto makeInstance(XpressProblem *prob, long serialNo,
                                     int index) const
        -> xpress::objects::GeneralConstraint final override;

  public:
    static inline auto lowLevelDelete(XpressProblem *prob, int count,
                                      xpress::Array<int const> const &indices)
        -> void;

  protected:
    virtual inline auto delIndices(XpressProblem *prob, int count,
                                   xpress::Array<int const> const &indices)
        -> void final override;

  public:
    virtual inline auto objectsForIndices(XpressProblem *prob, int first,
                                          int last) const
        -> std::vector<xpress::objects::GeneralConstraint> override;
  };

  std::shared_ptr<GeneralConstraintHandler> const generalConstraintHandler;

  inline auto generalConstraintForIndex(int index) const
      -> xpress::objects::GeneralConstraint;

  inline auto generalConstraintForIndices(int first, int last) const
      -> std::vector<xpress::objects::GeneralConstraint>;

  virtual inline auto delGenCons(int nconstraints,
                                 xpress::Array<int const> const &conind)
      -> void override;

  inline auto delGeneralConstraints(
      xpress::SizedArray<GeneralConstraint const> const &constraints) -> void;

private:
  typedef std::function<std::vector<double>(XPRSProblem *prob, int first,
                                            int last)>
      GetValuesFunctor;

  template <typename Strm0, typename T = typename xpress_elementtype(Strm0),
            xpress_isstream(Strm0)>
  inline auto getValues(std::shared_ptr<IndexHandler<T>>, Strm0 const &objs,
                        GetValuesFunctor functor, int objsSize)
      -> std::vector<double>;

  template <typename Strm0, typename T = typename xpress_elementtype(Strm0),
            xpress_isstream(Strm0)>
  inline auto getValues(std::shared_ptr<IndexHandler<T>> handler,
                        Strm0 const &objs, GetValuesFunctor functor)
      -> std::vector<double>;

public:
  inline auto delIndicator(Inequality row) -> void;

  inline auto getIndicator(Inequality row)
      -> std::optional<xpress::objects::IndicatorObjects>;

  inline auto setIndicator(Variable indicatorVariable, bool indicatorValue,
                           Inequality row) -> void;

  inline auto
  setIndicators(xpress::SizedArray<Variable const> const &indicatorVariable,
                std::vector<bool> const &indicatorValue,
                xpress::SizedArray<Inequality const> const &row) -> void;

private:
  template <typename Strm0, typename T = typename xpress_elementtype(Strm0),
            typename Func0, typename Func1, typename Func2,
            xpress_isstream(Strm0),
            xpress_isinvocable(Func0, Variable, Func0, T),
            xpress_isinvocable(Func1, bool, Func1, T),
            xpress_isinvocable(Func2, Inequality, Func2, T)>
  inline auto setIndicators(Strm0 const &data, int dataLength,
                            Func0 indicatorVariable, Func1 indicatorValue,
                            Func2 row) -> void;

public:
  template <typename Strm0, typename T = typename xpress_elementtype(Strm0),
            typename Func0, typename Func1, typename Func2,
            xpress_isstream(Strm0),
            xpress_isinvocable(Func0, Variable, Func0, T),
            xpress_isinvocable(Func1, bool, Func1, T),
            xpress_isinvocable(Func2, Inequality, Func2, T)>
  inline auto setIndicators(Strm0 const &data, Func0 indicatorVariable,
                            Func1 indicatorValue, Func2 row) -> void;

  template <typename I, xpress_isintegral(I), typename Func0, typename Func1,
            typename Func2, xpress_isinvocable(Func0, Variable, Func0, int),
            xpress_isinvocable(Func1, bool, Func1, int),
            xpress_isinvocable(Func2, Inequality, Func2, int)>
  inline auto setIndicators(I count, Func0 indicatorVariable,
                            Func1 indicatorValue, Func2 row) -> void;

public:
  class InequalityHandler final
      : public xpress::objects::IndexHandler<xpress::objects::Inequality> {
  public:
    inline InequalityHandler();

  protected:
    virtual inline auto makeInstance(xpress::objects::XpressProblem *prob,
                                     long serialNo, int index) const
        -> xpress::objects::Inequality final override;

  public:
    static inline auto lowLevelDelete(xpress::objects::XpressProblem *prob,
                                      int count,
                                      xpress::Array<int const> const &indices)
        -> void;

  protected:
    virtual inline auto delIndices(xpress::objects::XpressProblem *prob,
                                   int count,
                                   xpress::Array<int const> const &indices)
        -> void final override;

  public:
    virtual inline auto objectsForIndices(XpressProblem *prob, int first,
                                          int last) const
        -> std::vector<xpress::objects::Inequality> override;
  };

  std::shared_ptr<InequalityHandler> const rowHandler;

  inline auto inequalityForIndex(int index) const
      -> xpress::objects::Inequality;

  inline auto inequalitiesForIndices(int first, int last) const
      -> std::vector<xpress::objects::Inequality>;

  inline auto getInequalities() -> std::vector<xpress::objects::Inequality>;

  virtual inline auto delRows(int nrows, xpress::Array<int const> const &rowind)
      -> void override;

  inline auto delInequalities(xpress::SizedArray<Inequality const> const &rows)
      -> void;

private:
  static inline auto getValuesSlacks(XPRSProblem *prob, int first, int last)
      -> std::vector<double>;

  static inline auto getValuesDuals(XPRSProblem *prob, int first, int last)
      -> std::vector<double>;

  static inline auto getValuesCallbackSlacks(XPRSProblem *prob, int first,
                                             int last) -> std::vector<double>;

  static inline auto getValuesCallbackDuals(XPRSProblem *prob, int first,
                                            int last) -> std::vector<double>;

public:
  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto getSlacks(Strm0 const &rows) -> std::vector<double>;

  inline auto getSlack(Inequality r) -> double;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto getDuals(Strm0 const &rows) -> std::vector<double>;

  inline auto getDual(Inequality r) -> double;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto getCallbackSlacks(Strm0 const &rows) -> std::vector<double>;

  inline auto getCallbackSlack(Inequality r) -> double;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto getCallbackDuals(Strm0 const &rows) -> std::vector<double>;

  inline auto getCallbackDual(Inequality r) -> double;

  inline auto rhsSA(int len, xpress::SizedArray<Inequality const> const &rows,
                    xpress::SizedArray<double> const &lower,
                    xpress::SizedArray<double> const &upper) -> void;

  inline auto rhsSA(xpress::SizedArray<Inequality const> const &rows,
                    xpress::SizedArray<double> const &lower,
                    xpress::SizedArray<double> const &upper) -> void;

  inline auto chgRhs(xpress::SizedArray<Inequality const> const &rows,
                     xpress::SizedArray<double const> const &newRHS) -> void;

  inline auto chgRowType(xpress::SizedArray<Inequality const> const &rows,
                         xpress::SizedArray<RowType const> const &rowType)
      -> void;

  inline auto chgRhsRange(xpress::SizedArray<Inequality const> const &rows,
                          xpress::SizedArray<double const> const &newRange)
      -> void;

  inline auto loadDelayedRows(xpress::SizedArray<Inequality const> const &rows)
      -> void;

  inline auto loadModelCuts(xpress::SizedArray<Inequality const> const &rows)
      -> void;

public:
  inline auto chgObj(Variable variable, double coefficient)
      -> xpress::objects::XpressProblem &;

  inline auto chgObj(xpress::SizedArray<Variable const> const &variables,
                     xpress::SizedArray<double const> const &coefficients)
      -> xpress::objects::XpressProblem &;

  inline auto chgObjN(int objidx, Variable variable, double coefficient)
      -> xpress::objects::XpressProblem &;

  inline auto chgObjN(int objidx,
                      xpress::SizedArray<Variable const> const &variables,
                      xpress::SizedArray<double const> const &coefficients)
      -> xpress::objects::XpressProblem &;

  inline auto chgCoef(Inequality row, Variable variable, double coefficient)
      -> xpress::objects::XpressProblem &;

  inline auto chgCoefs(xpress::SizedArray<Inequality const> const &row,
                       xpress::SizedArray<Variable const> const &variable,
                       xpress::SizedArray<double const> const &coefficient)
      -> xpress::objects::XpressProblem &;

public:
  inline auto getPWLs() -> std::vector<xpress::objects::PWL>;

  class PWLHandler final : public IndexHandler<PWL> {
  public:
    inline PWLHandler();

  protected:
    virtual inline auto makeInstance(xpress::objects::XpressProblem *prob,
                                     long serialNo, int index) const
        -> xpress::objects::PWL final override;

  public:
    static inline auto lowLevelDelete(xpress::objects::XpressProblem *prob,
                                      int count,
                                      xpress::Array<int const> const &indices)
        -> void;

  protected:
    virtual inline auto delIndices(xpress::objects::XpressProblem *prob,
                                   int count,
                                   xpress::Array<int const> const &indices)
        -> void final override;

  public:
    virtual inline auto objectsForIndices(XpressProblem *prob, int first,
                                          int last) const
        -> std::vector<xpress::objects::PWL> override;
  };

  std::shared_ptr<PWLHandler> const pwlHandler;

  inline auto pwlForIndex(int index) const -> xpress::objects::PWL;

  inline auto pwlsForIndices(int first, int last) const
      -> std::vector<xpress::objects::PWL>;

  virtual inline auto delPwlCons(int npwls,
                                 xpress::Array<int const> const &pwlind)
      -> void override;

  inline auto delPwlConstraints(xpress::SizedArray<PWL const> const &pwls)
      -> void;

public:
  inline auto getCoef(Inequality row, Variable variable) -> double;

  inline auto getObj(Variable variable) -> double;

  inline auto getObjN(int objidx, Variable variable) -> double;

public:
  inline auto getSOSs() -> std::vector<xpress::objects::SOS>;

  class SOSHandler final : public xpress::objects::IndexHandler<SOS> {
  public:
    inline SOSHandler();

  protected:
    virtual inline auto makeInstance(XpressProblem *prob, long serialNo,
                                     int index) const
        -> xpress::objects::SOS final override;

  public:
    static inline auto lowLevelDelete(XpressProblem *prob, int count,
                                      xpress::Array<int const> const &indices)
        -> void;

  protected:
    virtual inline auto delIndices(XpressProblem *prob, int count,
                                   xpress::Array<int const> const &indices)
        -> void final override;

  public:
    virtual inline auto objectsForIndices(XpressProblem *prob, int first,
                                          int last) const
        -> std::vector<xpress::objects::SOS> override;
  };

  std::shared_ptr<SOSHandler> const sosHandler;

  inline auto sosForIndex(int index) const -> xpress::objects::SOS;

  inline auto sosForIndices(int first, int last) const
      -> std::vector<xpress::objects::SOS>;

  virtual inline auto delSets(int nsets, xpress::Array<int const> const &sosind)
      -> void override;

  inline auto delSOS(xpress::SizedArray<SOS const> const &soss) -> void;

  inline auto delSOS(SOS sos) -> void;

public:
  static inline constexpr int const NULL_VARIABLE_INDEX = -2;

  static inline Variable const NULL_VARIABLE = Variable(
      nullptr, -1, xpress::objects::XpressProblem::NULL_VARIABLE_INDEX);

private:
  class VariableHandler final : public xpress::objects::IndexHandler<Variable> {
  public:
    inline VariableHandler();

  protected:
    virtual inline auto makeInstance(XpressProblem *prob, long serialNo,
                                     int index) const
        -> xpress::objects::Variable final override;

  public:
    static inline auto lowLevelDelete(XpressProblem *prob, int count,
                                      xpress::Array<int const> const &indices)
        -> void;

  protected:
    virtual inline auto delIndices(XpressProblem *prob, int count,
                                   xpress::Array<int const> const &indices)
        -> void final override;

  public:
    virtual inline auto objectsForIndices(XpressProblem *prob, int first,
                                          int last) const
        -> std::vector<xpress::objects::Variable> override;
  };

  std::shared_ptr<VariableHandler> const variableHandler;

public:
  inline auto variableForIndex(int index) const -> xpress::objects::Variable;

  inline auto variablesForIndices(int first, int last) const
      -> std::vector<xpress::objects::Variable>;

  virtual inline auto delCols(int ncols, xpress::Array<int const> const &colind)
      -> void override;

  inline auto delVariables(xpress::SizedArray<Variable const> const &vars)
      -> void;

private:
  static inline auto getValuesSolution(XPRSProblem *prob, int first, int last)
      -> std::vector<double>;

  static inline auto getValuesRedCost(XPRSProblem *prob, int first, int last)
      -> std::vector<double>;

  static inline auto getValuesCallbackSolution(XPRSProblem *prob, int first,
                                               int last) -> std::vector<double>;

  static inline auto getValuesCallbackRedCost(XPRSProblem *prob, int first,
                                              int last) -> std::vector<double>;

public:
  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto getSolution(Strm0 const &vars) -> std::vector<double>;

  inline auto getSolution(Variable v) -> double;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto getRedCosts(Strm0 const &vars) -> std::vector<double>;

  inline auto getRedCost(Variable v) -> double;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto getCallbackSolution(Strm0 const &vars) -> std::vector<double>;

  inline auto getCallbackSolution(Variable v) -> double;

  template <typename Strm0, xpress_isstream(Strm0)>
  inline auto getCallbackRedCosts(Strm0 const &vars) -> std::vector<double>;

  inline auto getCallbackRedCost(Variable v) -> double;

  inline auto objSA(int len,
                    xpress::SizedArray<Variable const> const &variables,
                    xpress::SizedArray<double> const &lower,
                    xpress::SizedArray<double> const &upper) -> void;

  inline auto objSA(xpress::SizedArray<Variable const> const &variables,
                    xpress::SizedArray<double> const &lower,
                    xpress::SizedArray<double> const &upper) -> void;

  inline auto bndSA(int len,
                    xpress::SizedArray<Variable const> const &variables,
                    xpress::SizedArray<double> const &lblower,
                    xpress::SizedArray<double> const &lbupper,
                    xpress::SizedArray<double> const &ublower,
                    xpress::SizedArray<double> const &ubupper) -> void;

  inline auto bndSA(xpress::SizedArray<Variable const> const &variables,
                    xpress::SizedArray<double> const &lblower,
                    xpress::SizedArray<double> const &lbupper,
                    xpress::SizedArray<double> const &ublower,
                    xpress::SizedArray<double> const &ubupper) -> void;

  inline auto addMipSol(xpress::SizedArray<double const> const &values,
                        xpress::SizedArray<Variable const> const &variables,
                        std::optional<std::string> name) -> void;

  inline auto getVariables() -> std::vector<xpress::objects::Variable>;

  inline auto chgBounds(xpress::SizedArray<Variable const> const &variables,
                        xpress::SizedArray<char const> const &bndType,
                        xpress::SizedArray<double const> const &bndValue)
      -> void;

  inline auto chgColType(xpress::SizedArray<Variable const> const &variables,
                         xpress::SizedArray<ColumnType const> const &colType)
      -> void;

  static inline auto isNullVariable(Variable v) -> bool;

  inline auto slpSetDetRow(Variable v, Inequality i) -> void;

public:
  template <typename C, xpress_isintegral(C)>
  inline auto buildVariables(xpress::VariableBuilder::ArrayBuilder<C> builder)
      -> std::vector<xpress::objects::Variable>;

  template <typename I, typename C, xpress_isintegral(C), typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::ArrayBuilder<C> builder, Func0 makeResult,
      std::function<void(I &, C, xpress::objects::Variable)> addResult) -> I;

  template <typename K1>
  inline auto buildVariables(xpress::VariableBuilder::MapBuilder<K1> builder)
      -> std::unordered_map<K1, xpress::objects::Variable>;

  template <typename I, typename K1, typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::MapBuilder<K1> builder, Func0 makeResult,
      std::function<void(I &, K1, xpress::objects::Variable)> addResult) -> I;

  template <typename C, xpress_isintegral(C)>
  inline auto addVariables(C dim)
      -> xpress::VariableBuilder::VariableArrayBuilder<C>;

  template <typename Iter0, typename K1 = typename xpress_elementtype(Iter0),
            xpress_isiterable(Iter0)>
  inline auto addVariables(Iter0 const &iterable1)
      -> xpress::VariableBuilder::VariableMapBuilder<K1>;

  inline auto addVariable(double lb, double ub, ColumnType type,
                          std::optional<std::string> name)
      -> xpress::objects::Variable;

  inline auto addVariable(double lb, double ub, ColumnType type, double limit,
                          std::optional<std::string> name)
      -> xpress::objects::Variable;

  inline auto addVariable(double lb, double ub, ColumnType type)
      -> xpress::objects::Variable;

  inline auto addVariable(std::optional<std::string> name)
      -> xpress::objects::Variable;

  inline auto addVariable(ColumnType type) -> xpress::objects::Variable;

  inline auto addVariable(ColumnType type, std::optional<std::string> name)
      -> xpress::objects::Variable;

  inline auto addVariable() -> xpress::objects::Variable;

public:
  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  inline auto
  buildVariables(xpress::VariableBuilder::Array2Builder<C1, C2> builder)
      -> std::vector<std::vector<xpress::objects::Variable>>;

  template <typename I, typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2), typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::Array2Builder<C1, C2> builder, Func0 makeResult,
      std::function<void(I &, C1, C2, xpress::objects::Variable)> addResult)
      -> I;

  template <typename K1, typename K2>
  inline auto
  buildVariables(xpress::VariableBuilder::Map2Builder<K1, K2> builder)
      -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable>;

  template <typename I, typename K1, typename K2, typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::Map2Builder<K1, K2> builder, Func0 makeResult,
      std::function<void(I &, K1, K2, xpress::objects::Variable)> addResult)
      -> I;

  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  inline auto addVariables(C1 dim1, C2 dim2)
      -> xpress::VariableBuilder::VariableArray2Builder<C1, C2>;

  template <typename Iter0, typename Iter1,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1)>
  inline auto addVariables(Iter0 const &iterable1, Iter1 const &iterable2)
      -> xpress::VariableBuilder::VariableMap2Builder<K1, K2>;

public:
  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  inline auto
  buildVariables(xpress::VariableBuilder::Array3Builder<C1, C2, C3> builder)
      -> std::vector<std::vector<std::vector<xpress::objects::Variable>>>;

  template <typename I, typename C1, typename C2, typename C3,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            typename Func0, xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::Array3Builder<C1, C2, C3> builder,
      Func0 makeResult,
      std::function<void(I &, C1, C2, C3, xpress::objects::Variable)> addResult)
      -> I;

  template <typename K1, typename K2, typename K3>
  inline auto
  buildVariables(xpress::VariableBuilder::Map3Builder<K1, K2, K3> builder)
      -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable>;

  template <typename I, typename K1, typename K2, typename K3, typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::Map3Builder<K1, K2, K3> builder,
      Func0 makeResult,
      std::function<void(I &, K1, K2, K3, xpress::objects::Variable)> addResult)
      -> I;

  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  inline auto addVariables(C1 dim1, C2 dim2, C3 dim3)
      -> xpress::VariableBuilder::VariableArray3Builder<C1, C2, C3>;

  template <typename Iter0, typename Iter1, typename Iter2,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2)>
  inline auto addVariables(Iter0 const &iterable1, Iter1 const &iterable2,
                           Iter2 const &iterable3)
      -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>;

public:
  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  inline auto
  buildVariables(xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4> builder)
      -> std::vector<
          std::vector<std::vector<std::vector<xpress::objects::Variable>>>>;

  template <typename I, typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4> builder,
      Func0 makeResult,
      std::function<void(I &, C1, C2, C3, C4, xpress::objects::Variable)>
          addResult) -> I;

  template <typename K1, typename K2, typename K3, typename K4>
  inline auto
  buildVariables(xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> builder)
      -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable>;

  template <typename I, typename K1, typename K2, typename K3, typename K4,
            typename Func0, xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> builder,
      Func0 makeResult,
      std::function<void(I &, K1, K2, K3, K4, xpress::objects::Variable)>
          addResult) -> I;

  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  inline auto addVariables(C1 dim1, C2 dim2, C3 dim3, C4 dim4)
      -> xpress::VariableBuilder::VariableArray4Builder<C1, C2, C3, C4>;

  template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            typename K4 = typename xpress_elementtype(Iter3),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
  inline auto addVariables(Iter0 const &iterable1, Iter1 const &iterable2,
                           Iter2 const &iterable3, Iter3 const &iterable4)
      -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>;

public:
  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  inline auto buildVariables(
      xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder)
      -> std::vector<std::vector<
          std::vector<std::vector<std::vector<xpress::objects::Variable>>>>>;

  template <typename I, typename C1, typename C2, typename C3, typename C4,
            typename C5, xpress_isintegral(C1), xpress_isintegral(C2),
            xpress_isintegral(C3), xpress_isintegral(C4), xpress_isintegral(C5),
            typename Func0, xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder,
      Func0 makeResult,
      std::function<void(I &, C1, C2, C3, C4, C5, xpress::objects::Variable)>
          addResult) -> I;

  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  inline auto buildVariables(
      xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder)
      -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, xpress::objects::Variable>;

  template <typename I, typename K1, typename K2, typename K3, typename K4,
            typename K5, typename Func0, xpress_isinvocable(Func0, I, Func0)>
  inline auto buildVariables(
      xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder,
      Func0 makeResult,
      std::function<void(I &, K1, K2, K3, K4, K5, xpress::objects::Variable)>
          addResult) -> I;

  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  inline auto addVariables(C1 dim1, C2 dim2, C3 dim3, C4 dim4, C5 dim5)
      -> xpress::VariableBuilder::VariableArray5Builder<C1, C2, C3, C4, C5>;

  template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
            typename Iter4, typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            typename K4 = typename xpress_elementtype(Iter3),
            typename K5 = typename xpress_elementtype(Iter4),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2), xpress_isiterable(Iter3),
            xpress_isiterable(Iter4)>
  inline auto addVariables(Iter0 const &iterable1, Iter1 const &iterable2,
                           Iter2 const &iterable3, Iter3 const &iterable4,
                           Iter4 const &iterable5)
      -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>;

protected:
  inline auto makeConstraintCreator()
      -> std::shared_ptr<xpress::XPRSProblem::ConstraintCreator>;

  inline auto makeConstraintCreator(bool hasNames)
      -> std::shared_ptr<xpress::XPRSProblem::ConstraintCreator>;

public:
  template <typename I1, typename I2, xpress_isintegral(I1),
            xpress_isintegral(I2), typename Func0,
            typename C = typename xpress_resultof(Func0, I1,
                                                  I2)::constraint_type,
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, I1, I2)>
  inline auto addConstraints(I1 count1, I2 count2, Func0 makeConstraint)
      -> void;

  template <typename Iter0, typename Iter1,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1), typename Func0,
            typename C = typename xpress_resultof(Func0, K1,
                                                  K2)::constraint_type,
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, K1, K2)>
  inline auto addConstraints(Iter0 const &iterable1, Iter1 const &iterable2,
                             Func0 makeConstraint) -> void;

public:
  template <
      typename I1, typename I2, typename I3, xpress_isintegral(I1),
      xpress_isintegral(I2), xpress_isintegral(I3), typename Func0,
      typename C = typename xpress_resultof(Func0, I1, I2, I3)::constraint_type,
      xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, I1, I2, I3)>
  inline auto addConstraints(I1 count1, I2 count2, I3 count3,
                             Func0 makeConstraint) -> void;

  template <
      typename Iter0, typename Iter1, typename Iter2,
      typename K1 = typename xpress_elementtype(Iter0),
      typename K2 = typename xpress_elementtype(Iter1),
      typename K3 = typename xpress_elementtype(Iter2), typename Func0,
      typename C = typename xpress_resultof(Func0, K1, K2, K3)::constraint_type,
      xpress_isiterable(Iter0), xpress_isiterable(Iter1),
      xpress_isiterable(Iter2),
      xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, K1, K2, K3)>
  inline auto addConstraints(Iter0 const &iterable1, Iter1 const &iterable2,
                             Iter2 const &iterable3, Func0 makeConstraint)
      -> void;

public:
  template <typename I1, typename I2, typename I3, typename I4,
            xpress_isintegral(I1), xpress_isintegral(I2), xpress_isintegral(I3),
            xpress_isintegral(I4), typename Func0,
            typename C = typename xpress_resultof(Func0, I1, I2, I3,
                                                  I4)::constraint_type,
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, I1, I2,
                               I3, I4)>
  inline auto addConstraints(I1 count1, I2 count2, I3 count3, I4 count4,
                             Func0 makeConstraint) -> void;

  template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            typename K4 = typename xpress_elementtype(Iter3), typename Func0,
            typename C = typename xpress_resultof(Func0, K1, K2, K3,
                                                  K4)::constraint_type,
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2), xpress_isiterable(Iter3),
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, K1, K2,
                               K3, K4)>
  inline auto addConstraints(Iter0 const &iterable1, Iter1 const &iterable2,
                             Iter2 const &iterable3, Iter3 const &iterable4,
                             Func0 makeConstraint) -> void;

public:
  template <typename I1, typename I2, typename I3, typename I4, typename I5,
            xpress_isintegral(I1), xpress_isintegral(I2), xpress_isintegral(I3),
            xpress_isintegral(I4), xpress_isintegral(I5), typename Func0,
            typename C = typename xpress_resultof(Func0, I1, I2, I3, I4,
                                                  I5)::constraint_type,
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, I1, I2,
                               I3, I4, I5)>
  inline auto addConstraints(I1 count1, I2 count2, I3 count3, I4 count4,
                             I5 count5, Func0 makeConstraint) -> void;

  template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
            typename Iter4, typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            typename K4 = typename xpress_elementtype(Iter3),
            typename K5 = typename xpress_elementtype(Iter4), typename Func0,
            typename C = typename xpress_resultof(Func0, K1, K2, K3, K4,
                                                  K5)::constraint_type,
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2), xpress_isiterable(Iter3),
            xpress_isiterable(Iter4),
            xpress_isinvocable(Func0, ConstraintDefinition<C>, Func0, K1, K2,
                               K3, K4, K5)>
  inline auto addConstraints(Iter0 const &iterable1, Iter1 const &iterable2,
                             Iter2 const &iterable3, Iter3 const &iterable4,
                             Iter4 const &iterable5, Func0 makeConstraint)
      -> void;

protected:
  virtual inline auto drop() -> void override;

private:
  static inline auto mapIsolation(char isolation)
      -> xpress::objects::IIS::Isolation;

public:
  inline auto getIIS(int iis) -> xpress::objects::IIS;

private:
  template <typename M>
  static inline auto convertPreferences(int count, M prefs)
      -> std::vector<double>;

public:
  template <typename IMap, typename VMap>
  inline auto repairWeightedInfeas(VMap const &lepref, VMap const &gepref,
                                   IMap const &lbpref, IMap const &ubpref,
                                   char phase2, double delta,
                                   std::optional<std::string> flags) -> int;
};
} // namespace objects
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_EOF(XPRS_TOK_EOF);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_LB(XPRS_TOK_LB);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_RB(XPRS_TOK_RB);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_UMINUS(
        XPRS_TOK_OP, XPRS_OP_UMINUS);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_EXPONENT(
        XPRS_TOK_OP, XPRS_OP_EXPONENT);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_MULTIPLY(
        XPRS_TOK_OP, XPRS_OP_MULTIPLY);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_DIVIDE(
        XPRS_TOK_OP, XPRS_OP_DIVIDE);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_PLUS(
        XPRS_TOK_OP, XPRS_OP_PLUS);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_MINUS(
        XPRS_TOK_OP, XPRS_OP_MINUS);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_COMMA(
        XPRS_TOK_DEL, XPRS_DEL_COMMA);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_COLON(
        XPRS_TOK_DEL, XPRS_DEL_COLON);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_LOG10(
        XPRS_TOK_IFUN, XPRS_IFUN_LOG10);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_LN(XPRS_TOK_IFUN,
                                                                 XPRS_IFUN_LN);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_EXP(
        XPRS_TOK_IFUN, XPRS_IFUN_EXP);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_ABS(
        XPRS_TOK_IFUN, XPRS_IFUN_ABS);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_SQRT(
        XPRS_TOK_IFUN, XPRS_IFUN_SQRT);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_SIN(
        XPRS_TOK_IFUN, XPRS_IFUN_SIN);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_COS(
        XPRS_TOK_IFUN, XPRS_IFUN_COS);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_TAN(
        XPRS_TOK_IFUN, XPRS_IFUN_TAN);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_ARCSIN(
        XPRS_TOK_IFUN, XPRS_IFUN_ARCSIN);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_ARCCOS(
        XPRS_TOK_IFUN, XPRS_IFUN_ARCCOS);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_ARCTAN(
        XPRS_TOK_IFUN, XPRS_IFUN_ARCTAN);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_MIN(
        XPRS_TOK_IFUN, XPRS_IFUN_MIN);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_MAX(
        XPRS_TOK_IFUN, XPRS_IFUN_MAX);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_PWL(
        XPRS_TOK_IFUN, XPRS_IFUN_PWL);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_SUM(
        XPRS_TOK_IFUN, XPRS_IFUN_SUM);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_PROD(
        XPRS_TOK_IFUN, XPRS_IFUN_PROD);
template <typename AlwaysVoid>
xpress::objects::Token const
    xpress::GlobalConstants<AlwaysVoid>::xpress_objects_Token_SIGN(
        XPRS_TOK_IFUN, XPRS_IFUN_SIGN);
template <typename AlwaysVoid>
bool xpress::GlobalConstants<
    AlwaysVoid>::xpress_objects_LinTermList_autoCompress(false);
template <typename AlwaysVoid>
bool xpress::GlobalConstants<
    AlwaysVoid>::xpress_objects_QuadTermList_autoCompress(false);
} // namespace xpress
template <typename K1, typename K2>
std::size_t std::hash<xpress::maps::MapKey2<K1, K2>>::operator()(
    xpress::maps::MapKey2<K1, K2> const &arg) const {
  return arg.getHashCode();
}
template <typename K1, typename K2>
std::size_t std::hash<xpress::maps::ComparableMapKey2<K1, K2>>::operator()(
    xpress::maps::ComparableMapKey2<K1, K2> const &arg) const {
  return arg.getHashCode();
}
template <typename K1, typename K2, typename K3>
std::size_t std::hash<xpress::maps::MapKey3<K1, K2, K3>>::operator()(
    xpress::maps::MapKey3<K1, K2, K3> const &arg) const {
  return arg.getHashCode();
}
template <typename K1, typename K2, typename K3>
std::size_t std::hash<xpress::maps::ComparableMapKey3<K1, K2, K3>>::operator()(
    xpress::maps::ComparableMapKey3<K1, K2, K3> const &arg) const {
  return arg.getHashCode();
}
template <typename K1, typename K2, typename K3, typename K4>
std::size_t std::hash<xpress::maps::MapKey4<K1, K2, K3, K4>>::operator()(
    xpress::maps::MapKey4<K1, K2, K3, K4> const &arg) const {
  return arg.getHashCode();
}
template <typename K1, typename K2, typename K3, typename K4>
std::size_t
std::hash<xpress::maps::ComparableMapKey4<K1, K2, K3, K4>>::operator()(
    xpress::maps::ComparableMapKey4<K1, K2, K3, K4> const &arg) const {
  return arg.getHashCode();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
std::size_t std::hash<xpress::maps::MapKey5<K1, K2, K3, K4, K5>>::operator()(
    xpress::maps::MapKey5<K1, K2, K3, K4, K5> const &arg) const {
  return arg.getHashCode();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
std::size_t
std::hash<xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>>::operator()(
    xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5> const &arg) const {
  return arg.getHashCode();
}
std::size_t std::hash<xpress::objects::QPair>::operator()(
    xpress::objects::QPair const &arg) const {
  return arg.getHashCode();
}
std::size_t std::hash<xpress::PwlBreakpoint>::operator()(
    xpress::PwlBreakpoint const &arg) const {
  return arg.getHashCode();
}
std::size_t std::hash<xpress::objects::GeneralConstraint>::operator()(
    xpress::objects::GeneralConstraint const &arg) const {
  return arg.getHashCode();
}
std::size_t std::hash<xpress::objects::Inequality>::operator()(
    xpress::objects::Inequality const &arg) const {
  return arg.getHashCode();
}
std::size_t std::hash<xpress::objects::PWL>::operator()(
    xpress::objects::PWL const &arg) const {
  return arg.getHashCode();
}
std::size_t std::hash<xpress::objects::SOS>::operator()(
    xpress::objects::SOS const &arg) const {
  return arg.getHashCode();
}
std::size_t std::hash<xpress::objects::Token>::operator()(
    xpress::objects::Token const &arg) const {
  return arg.getHashCode();
}
std::size_t std::hash<xpress::objects::Variable>::operator()(
    xpress::objects::Variable const &arg) const {
  return arg.getHashCode();
}
template <typename K1, typename K2>
bool std::less<xpress::maps::ComparableMapKey2<K1, K2>>::operator()(
    xpress::maps::ComparableMapKey2<K1, K2> const &a,
    xpress::maps::ComparableMapKey2<K1, K2> const &b) const {
  return a.compareTo(b) < 0;
}
template <typename K1, typename K2>
bool std::greater<xpress::maps::ComparableMapKey2<K1, K2>>::operator()(
    xpress::maps::ComparableMapKey2<K1, K2> const &a,
    xpress::maps::ComparableMapKey2<K1, K2> const &b) const {
  return a.compareTo(b) > 0;
}
template <typename K1, typename K2>
bool std::equal_to<xpress::maps::ComparableMapKey2<K1, K2>>::operator()(
    xpress::maps::ComparableMapKey2<K1, K2> const &a,
    xpress::maps::ComparableMapKey2<K1, K2> const &b) const {
  return a.compareTo(b) == 0;
}
template <typename K1, typename K2, typename K3>
bool std::less<xpress::maps::ComparableMapKey3<K1, K2, K3>>::operator()(
    xpress::maps::ComparableMapKey3<K1, K2, K3> const &a,
    xpress::maps::ComparableMapKey3<K1, K2, K3> const &b) const {
  return a.compareTo(b) < 0;
}
template <typename K1, typename K2, typename K3>
bool std::greater<xpress::maps::ComparableMapKey3<K1, K2, K3>>::operator()(
    xpress::maps::ComparableMapKey3<K1, K2, K3> const &a,
    xpress::maps::ComparableMapKey3<K1, K2, K3> const &b) const {
  return a.compareTo(b) > 0;
}
template <typename K1, typename K2, typename K3>
bool std::equal_to<xpress::maps::ComparableMapKey3<K1, K2, K3>>::operator()(
    xpress::maps::ComparableMapKey3<K1, K2, K3> const &a,
    xpress::maps::ComparableMapKey3<K1, K2, K3> const &b) const {
  return a.compareTo(b) == 0;
}
template <typename K1, typename K2, typename K3, typename K4>
bool std::less<xpress::maps::ComparableMapKey4<K1, K2, K3, K4>>::operator()(
    xpress::maps::ComparableMapKey4<K1, K2, K3, K4> const &a,
    xpress::maps::ComparableMapKey4<K1, K2, K3, K4> const &b) const {
  return a.compareTo(b) < 0;
}
template <typename K1, typename K2, typename K3, typename K4>
bool std::greater<xpress::maps::ComparableMapKey4<K1, K2, K3, K4>>::operator()(
    xpress::maps::ComparableMapKey4<K1, K2, K3, K4> const &a,
    xpress::maps::ComparableMapKey4<K1, K2, K3, K4> const &b) const {
  return a.compareTo(b) > 0;
}
template <typename K1, typename K2, typename K3, typename K4>
bool std::equal_to<xpress::maps::ComparableMapKey4<K1, K2, K3, K4>>::operator()(
    xpress::maps::ComparableMapKey4<K1, K2, K3, K4> const &a,
    xpress::maps::ComparableMapKey4<K1, K2, K3, K4> const &b) const {
  return a.compareTo(b) == 0;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
bool std::less<xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>>::operator()(
    xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5> const &a,
    xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5> const &b) const {
  return a.compareTo(b) < 0;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
bool std::greater<xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>>::
operator()(xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5> const &a,
           xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5> const &b) const {
  return a.compareTo(b) > 0;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
bool std::equal_to<xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>>::
operator()(xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5> const &a,
           xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5> const &b) const {
  return a.compareTo(b) == 0;
}
bool std::less<xpress::objects::QPair>::operator()(
    xpress::objects::QPair const &a, xpress::objects::QPair const &b) const {
  return a.compareTo(b) < 0;
}
bool std::greater<xpress::objects::QPair>::operator()(
    xpress::objects::QPair const &a, xpress::objects::QPair const &b) const {
  return a.compareTo(b) > 0;
}
bool std::equal_to<xpress::objects::QPair>::operator()(
    xpress::objects::QPair const &a, xpress::objects::QPair const &b) const {
  return a.compareTo(b) == 0;
}
bool std::less<xpress::PwlBreakpoint>::operator()(
    xpress::PwlBreakpoint const &a, xpress::PwlBreakpoint const &b) const {
  return a.compareTo(b) < 0;
}
bool std::greater<xpress::PwlBreakpoint>::operator()(
    xpress::PwlBreakpoint const &a, xpress::PwlBreakpoint const &b) const {
  return a.compareTo(b) > 0;
}
bool std::equal_to<xpress::PwlBreakpoint>::operator()(
    xpress::PwlBreakpoint const &a, xpress::PwlBreakpoint const &b) const {
  return a.compareTo(b) == 0;
}
bool std::less<xpress::objects::GeneralConstraint>::operator()(
    xpress::objects::GeneralConstraint const &a,
    xpress::objects::GeneralConstraint const &b) const {
  return a.compareTo(b) < 0;
}
bool std::greater<xpress::objects::GeneralConstraint>::operator()(
    xpress::objects::GeneralConstraint const &a,
    xpress::objects::GeneralConstraint const &b) const {
  return a.compareTo(b) > 0;
}
bool std::equal_to<xpress::objects::GeneralConstraint>::operator()(
    xpress::objects::GeneralConstraint const &a,
    xpress::objects::GeneralConstraint const &b) const {
  return a.compareTo(b) == 0;
}
bool std::less<xpress::objects::Inequality>::operator()(
    xpress::objects::Inequality const &a,
    xpress::objects::Inequality const &b) const {
  return a.compareTo(b) < 0;
}
bool std::greater<xpress::objects::Inequality>::operator()(
    xpress::objects::Inequality const &a,
    xpress::objects::Inequality const &b) const {
  return a.compareTo(b) > 0;
}
bool std::equal_to<xpress::objects::Inequality>::operator()(
    xpress::objects::Inequality const &a,
    xpress::objects::Inequality const &b) const {
  return a.compareTo(b) == 0;
}
bool std::less<xpress::objects::PWL>::operator()(
    xpress::objects::PWL const &a, xpress::objects::PWL const &b) const {
  return a.compareTo(b) < 0;
}
bool std::greater<xpress::objects::PWL>::operator()(
    xpress::objects::PWL const &a, xpress::objects::PWL const &b) const {
  return a.compareTo(b) > 0;
}
bool std::equal_to<xpress::objects::PWL>::operator()(
    xpress::objects::PWL const &a, xpress::objects::PWL const &b) const {
  return a.compareTo(b) == 0;
}
bool std::less<xpress::objects::SOS>::operator()(
    xpress::objects::SOS const &a, xpress::objects::SOS const &b) const {
  return a.compareTo(b) < 0;
}
bool std::greater<xpress::objects::SOS>::operator()(
    xpress::objects::SOS const &a, xpress::objects::SOS const &b) const {
  return a.compareTo(b) > 0;
}
bool std::equal_to<xpress::objects::SOS>::operator()(
    xpress::objects::SOS const &a, xpress::objects::SOS const &b) const {
  return a.compareTo(b) == 0;
}
bool std::less<xpress::objects::Token>::operator()(
    xpress::objects::Token const &a, xpress::objects::Token const &b) const {
  return a.compareTo(b) < 0;
}
bool std::greater<xpress::objects::Token>::operator()(
    xpress::objects::Token const &a, xpress::objects::Token const &b) const {
  return a.compareTo(b) > 0;
}
bool std::equal_to<xpress::objects::Token>::operator()(
    xpress::objects::Token const &a, xpress::objects::Token const &b) const {
  return a.compareTo(b) == 0;
}
bool std::less<xpress::objects::Variable>::operator()(
    xpress::objects::Variable const &a,
    xpress::objects::Variable const &b) const {
  return a.compareTo(b) < 0;
}
bool std::greater<xpress::objects::Variable>::operator()(
    xpress::objects::Variable const &a,
    xpress::objects::Variable const &b) const {
  return a.compareTo(b) > 0;
}
bool std::equal_to<xpress::objects::Variable>::operator()(
    xpress::objects::Variable const &a,
    xpress::objects::Variable const &b) const {
  return a.compareTo(b) == 0;
}
template <typename K1, typename K2>
bool std::equal_to<xpress::maps::MapKey2<K1, K2>>::operator()(
    xpress::maps::MapKey2<K1, K2> const &a,
    xpress::maps::MapKey2<K1, K2> const &b) const {
  return a.equals(b);
}
template <typename K1, typename K2, typename K3>
bool std::equal_to<xpress::maps::MapKey3<K1, K2, K3>>::operator()(
    xpress::maps::MapKey3<K1, K2, K3> const &a,
    xpress::maps::MapKey3<K1, K2, K3> const &b) const {
  return a.equals(b);
}
template <typename K1, typename K2, typename K3, typename K4>
bool std::equal_to<xpress::maps::MapKey4<K1, K2, K3, K4>>::operator()(
    xpress::maps::MapKey4<K1, K2, K3, K4> const &a,
    xpress::maps::MapKey4<K1, K2, K3, K4> const &b) const {
  return a.equals(b);
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
bool std::equal_to<xpress::maps::MapKey5<K1, K2, K3, K4, K5>>::operator()(
    xpress::maps::MapKey5<K1, K2, K3, K4, K5> const &a,
    xpress::maps::MapKey5<K1, K2, K3, K4, K5> const &b) const {
  return a.equals(b);
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::BinaryExpression const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::ConstantExpression const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::Expression const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::InternalFunctionExpression const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os, xpress::objects::LinTerm const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::LinTermList const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::LinTermMap const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::PWLExpression const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os, xpress::objects::QPair const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os, xpress::objects::QuadTerm const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::QuadTermList const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::QuadTermMap const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::SumExpression const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os, xpress::objects::Token const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::UnaryExpression const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os,
                         xpress::objects::UserFunctionExpression const &o) {
  os << o.toString();
  return os;
}
std::ostream &operator<<(std::ostream &os, xpress::objects::Variable const &o) {
  os << o.toString();
  return os;
}
#ifndef XPRESS_DO_NOT_INCLUDE_CXX_SOURCE
#include "xpress_objects.cpp"
#endif
#endif
