/* *********************************************************************** *
 *                                                                         *
 * xpress_maps.hpp                                                         *
 *                                                                         *
 * C++ API for Xpress-Optimizer                                            *
 *                                                                         *
 * (c) Copyright Fair Isaac Corporation 2024-2025. All rights reserved     *
 * For FICO Xpress Optimizer v45.01.02                                     *
 *                                                                         *
 * *********************************************************************** */

#ifndef XPRESS_MAPS_H
#define XPRESS_MAPS_H 1
#include <map>
#include <tuple>
#include <unordered_map>
namespace xpress {

namespace maps {

template <typename K1, typename K2> class MapKey2 : public std::tuple<K1, K2> {
public:
  MapKey2() : std::tuple<K1, K2>(K1(), K2()) {}

  K1 &getKey1() { return std::get<0>(*this); }
  K1 const &getKey1() const { return std::get<0>(*this); }

  K2 &getKey2() { return std::get<1>(*this); }
  K2 const &getKey2() const { return std::get<1>(*this); }

  MapKey2(K1 const &key1, K2 const &key2) : std::tuple<K1, K2>(key1, key2) {}
  std::size_t getHashCode() const {
    return xpress::combineHash(std::get<0>(*this), std::get<1>(*this));
  }
  bool operator==(MapKey2<K1, K2> const &other) const {
    return std::equal_to<K1>()(std::get<0>(*this), std::get<0>(other)) &&
           std::equal_to<K2>()(std::get<1>(*this), std::get<1>(other));
  }
  bool operator!=(MapKey2<K1, K2> const &other) const {
    return !operator==(other);
  }
  bool equals(MapKey2<K1, K2> const &other) const { return operator==(other); }
};

template <typename K1, typename K2, typename K3>
class MapKey3 : public std::tuple<K1, K2, K3> {
public:
  MapKey3() : std::tuple<K1, K2, K3>(K1(), K2(), K3()) {}

  K1 &getKey1() { return std::get<0>(*this); }
  K1 const &getKey1() const { return std::get<0>(*this); }

  K2 &getKey2() { return std::get<1>(*this); }
  K2 const &getKey2() const { return std::get<1>(*this); }

  K3 &getKey3() { return std::get<2>(*this); }
  K3 const &getKey3() const { return std::get<2>(*this); }

  MapKey3(K1 const &key1, K2 const &key2, K3 const &key3)
      : std::tuple<K1, K2, K3>(key1, key2, key3) {}
  std::size_t getHashCode() const {
    return xpress::combineHash(std::get<0>(*this), std::get<1>(*this),
                               std::get<2>(*this));
  }
  bool operator==(MapKey3<K1, K2, K3> const &other) const {
    return std::equal_to<K1>()(std::get<0>(*this), std::get<0>(other)) &&
           std::equal_to<K2>()(std::get<1>(*this), std::get<1>(other)) &&
           std::equal_to<K3>()(std::get<2>(*this), std::get<2>(other));
  }
  bool operator!=(MapKey3<K1, K2, K3> const &other) const {
    return !operator==(other);
  }
  bool equals(MapKey3<K1, K2, K3> const &other) const {
    return operator==(other);
  }
};

template <typename K1, typename K2, typename K3, typename K4>
class MapKey4 : public std::tuple<K1, K2, K3, K4> {
public:
  MapKey4() : std::tuple<K1, K2, K3, K4>(K1(), K2(), K3(), K4()) {}

  K1 &getKey1() { return std::get<0>(*this); }
  K1 const &getKey1() const { return std::get<0>(*this); }

  K2 &getKey2() { return std::get<1>(*this); }
  K2 const &getKey2() const { return std::get<1>(*this); }

  K3 &getKey3() { return std::get<2>(*this); }
  K3 const &getKey3() const { return std::get<2>(*this); }

  K4 &getKey4() { return std::get<3>(*this); }
  K4 const &getKey4() const { return std::get<3>(*this); }

  MapKey4(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4)
      : std::tuple<K1, K2, K3, K4>(key1, key2, key3, key4) {}
  std::size_t getHashCode() const {
    return xpress::combineHash(std::get<0>(*this), std::get<1>(*this),
                               std::get<2>(*this), std::get<3>(*this));
  }
  bool operator==(MapKey4<K1, K2, K3, K4> const &other) const {
    return std::equal_to<K1>()(std::get<0>(*this), std::get<0>(other)) &&
           std::equal_to<K2>()(std::get<1>(*this), std::get<1>(other)) &&
           std::equal_to<K3>()(std::get<2>(*this), std::get<2>(other)) &&
           std::equal_to<K4>()(std::get<3>(*this), std::get<3>(other));
  }
  bool operator!=(MapKey4<K1, K2, K3, K4> const &other) const {
    return !operator==(other);
  }
  bool equals(MapKey4<K1, K2, K3, K4> const &other) const {
    return operator==(other);
  }
};

template <typename K1, typename K2, typename K3, typename K4, typename K5>
class MapKey5 : public std::tuple<K1, K2, K3, K4, K5> {
public:
  MapKey5() : std::tuple<K1, K2, K3, K4, K5>(K1(), K2(), K3(), K4(), K5()) {}

  K1 &getKey1() { return std::get<0>(*this); }
  K1 const &getKey1() const { return std::get<0>(*this); }

  K2 &getKey2() { return std::get<1>(*this); }
  K2 const &getKey2() const { return std::get<1>(*this); }

  K3 &getKey3() { return std::get<2>(*this); }
  K3 const &getKey3() const { return std::get<2>(*this); }

  K4 &getKey4() { return std::get<3>(*this); }
  K4 const &getKey4() const { return std::get<3>(*this); }

  K5 &getKey5() { return std::get<4>(*this); }
  K5 const &getKey5() const { return std::get<4>(*this); }

  MapKey5(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
          K5 const &key5)
      : std::tuple<K1, K2, K3, K4, K5>(key1, key2, key3, key4, key5) {}
  std::size_t getHashCode() const {
    return xpress::combineHash(std::get<0>(*this), std::get<1>(*this),
                               std::get<2>(*this), std::get<3>(*this),
                               std::get<4>(*this));
  }
  bool operator==(MapKey5<K1, K2, K3, K4, K5> const &other) const {
    return std::equal_to<K1>()(std::get<0>(*this), std::get<0>(other)) &&
           std::equal_to<K2>()(std::get<1>(*this), std::get<1>(other)) &&
           std::equal_to<K3>()(std::get<2>(*this), std::get<2>(other)) &&
           std::equal_to<K4>()(std::get<3>(*this), std::get<3>(other)) &&
           std::equal_to<K5>()(std::get<4>(*this), std::get<4>(other));
  }
  bool operator!=(MapKey5<K1, K2, K3, K4, K5> const &other) const {
    return !operator==(other);
  }
  bool equals(MapKey5<K1, K2, K3, K4, K5> const &other) const {
    return operator==(other);
  }
};

template <typename K1, typename K2>
class ComparableMapKey2 : public MapKey2<K1, K2> {
public:
  ComparableMapKey2() : MapKey2<K1, K2>() {}

  ComparableMapKey2(K1 const &key1, K2 const &key2)
      : MapKey2<K1, K2>(key1, key2) {}
  int compareTo(ComparableMapKey2<K1, K2> const &other) const {
    if (std::less<K1>()(std::get<0>(*this), std::get<0>(other)))
      return -1;
    if (std::less<K1>()(std::get<0>(other), std::get<0>(*this)))
      return 1;
    if (std::less<K2>()(std::get<1>(*this), std::get<1>(other)))
      return -1;
    if (std::less<K2>()(std::get<1>(other), std::get<1>(*this)))
      return 1;
    return 0;
  }
};

template <typename K1, typename K2, typename K3>
class ComparableMapKey3 : public MapKey3<K1, K2, K3> {
public:
  ComparableMapKey3() : MapKey3<K1, K2, K3>() {}

  ComparableMapKey3(K1 const &key1, K2 const &key2, K3 const &key3)
      : MapKey3<K1, K2, K3>(key1, key2, key3) {}
  int compareTo(ComparableMapKey3<K1, K2, K3> const &other) const {
    if (std::less<K1>()(std::get<0>(*this), std::get<0>(other)))
      return -1;
    if (std::less<K1>()(std::get<0>(other), std::get<0>(*this)))
      return 1;
    if (std::less<K2>()(std::get<1>(*this), std::get<1>(other)))
      return -1;
    if (std::less<K2>()(std::get<1>(other), std::get<1>(*this)))
      return 1;
    if (std::less<K3>()(std::get<2>(*this), std::get<2>(other)))
      return -1;
    if (std::less<K3>()(std::get<2>(other), std::get<2>(*this)))
      return 1;
    return 0;
  }
};

template <typename K1, typename K2, typename K3, typename K4>
class ComparableMapKey4 : public MapKey4<K1, K2, K3, K4> {
public:
  ComparableMapKey4() : MapKey4<K1, K2, K3, K4>() {}

  ComparableMapKey4(K1 const &key1, K2 const &key2, K3 const &key3,
                    K4 const &key4)
      : MapKey4<K1, K2, K3, K4>(key1, key2, key3, key4) {}
  int compareTo(ComparableMapKey4<K1, K2, K3, K4> const &other) const {
    if (std::less<K1>()(std::get<0>(*this), std::get<0>(other)))
      return -1;
    if (std::less<K1>()(std::get<0>(other), std::get<0>(*this)))
      return 1;
    if (std::less<K2>()(std::get<1>(*this), std::get<1>(other)))
      return -1;
    if (std::less<K2>()(std::get<1>(other), std::get<1>(*this)))
      return 1;
    if (std::less<K3>()(std::get<2>(*this), std::get<2>(other)))
      return -1;
    if (std::less<K3>()(std::get<2>(other), std::get<2>(*this)))
      return 1;
    if (std::less<K4>()(std::get<3>(*this), std::get<3>(other)))
      return -1;
    if (std::less<K4>()(std::get<3>(other), std::get<3>(*this)))
      return 1;
    return 0;
  }
};

template <typename K1, typename K2, typename K3, typename K4, typename K5>
class ComparableMapKey5 : public MapKey5<K1, K2, K3, K4, K5> {
public:
  ComparableMapKey5() : MapKey5<K1, K2, K3, K4, K5>() {}

  ComparableMapKey5(K1 const &key1, K2 const &key2, K3 const &key3,
                    K4 const &key4, K5 const &key5)
      : MapKey5<K1, K2, K3, K4, K5>(key1, key2, key3, key4, key5) {}
  int compareTo(ComparableMapKey5<K1, K2, K3, K4, K5> const &other) const {
    if (std::less<K1>()(std::get<0>(*this), std::get<0>(other)))
      return -1;
    if (std::less<K1>()(std::get<0>(other), std::get<0>(*this)))
      return 1;
    if (std::less<K2>()(std::get<1>(*this), std::get<1>(other)))
      return -1;
    if (std::less<K2>()(std::get<1>(other), std::get<1>(*this)))
      return 1;
    if (std::less<K3>()(std::get<2>(*this), std::get<2>(other)))
      return -1;
    if (std::less<K3>()(std::get<2>(other), std::get<2>(*this)))
      return 1;
    if (std::less<K4>()(std::get<3>(*this), std::get<3>(other)))
      return -1;
    if (std::less<K4>()(std::get<3>(other), std::get<3>(*this)))
      return 1;
    if (std::less<K5>()(std::get<4>(*this), std::get<4>(other)))
      return -1;
    if (std::less<K5>()(std::get<4>(other), std::get<4>(*this)))
      return 1;
    return 0;
  }
};

template <typename K1, typename K2, typename V>
class HashMap2 final
    : public std::unordered_map<xpress::maps::MapKey2<K1, K2>, V> {
public:
  V get(K1 const &key1, K2 const &key2) const {
    auto it = this->find(xpress::maps::MapKey2<K1, K2>(key1, key2));
    if (it == this->end()) {
      throw std::invalid_argument("no such key");
    }
    return it->second;
  }

  void put(K1 const &key1, K2 const &key2, V value) {
    this->insert(std::pair<xpress::maps::MapKey2<K1, K2>, V>(
        xpress::maps::MapKey2<K1, K2>(key1, key2), value));
  }

  bool containsKey(K1 const &key1, K2 const &key2) const {
    return this->find(xpress::maps::MapKey2<K1, K2>(key1, key2)) != this->end;
  }

  bool contains(K1 const &key1, K2 const &key2) const {
    return this->containsKey(key1, key2);
  }

  V operator()(K1 const &key1, K2 const &key2) const {
    return this->get(key1, key2);
  }
};

template <typename K1, typename K2, typename K3, typename V>
class HashMap3 final
    : public std::unordered_map<xpress::maps::MapKey3<K1, K2, K3>, V> {
public:
  V get(K1 const &key1, K2 const &key2, K3 const &key3) const {
    auto it = this->find(xpress::maps::MapKey3<K1, K2, K3>(key1, key2, key3));
    if (it == this->end()) {
      throw std::invalid_argument("no such key");
    }
    return it->second;
  }

  void put(K1 const &key1, K2 const &key2, K3 const &key3, V value) {
    this->insert(std::pair<xpress::maps::MapKey3<K1, K2, K3>, V>(
        xpress::maps::MapKey3<K1, K2, K3>(key1, key2, key3), value));
  }

  bool containsKey(K1 const &key1, K2 const &key2, K3 const &key3) const {
    return this->find(xpress::maps::MapKey3<K1, K2, K3>(key1, key2, key3)) !=
           this->end;
  }

  bool contains(K1 const &key1, K2 const &key2, K3 const &key3) const {
    return this->containsKey(key1, key2, key3);
  }

  V operator()(K1 const &key1, K2 const &key2, K3 const &key3) const {
    return this->get(key1, key2, key3);
  }
};

template <typename K1, typename K2, typename K3, typename K4, typename V>
class HashMap4 final
    : public std::unordered_map<xpress::maps::MapKey4<K1, K2, K3, K4>, V> {
public:
  V get(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4) const {
    auto it = this->find(
        xpress::maps::MapKey4<K1, K2, K3, K4>(key1, key2, key3, key4));
    if (it == this->end()) {
      throw std::invalid_argument("no such key");
    }
    return it->second;
  }

  void put(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
           V value) {
    this->insert(std::pair<xpress::maps::MapKey4<K1, K2, K3, K4>, V>(
        xpress::maps::MapKey4<K1, K2, K3, K4>(key1, key2, key3, key4), value));
  }

  bool containsKey(K1 const &key1, K2 const &key2, K3 const &key3,
                   K4 const &key4) const {
    return this->find(xpress::maps::MapKey4<K1, K2, K3, K4>(key1, key2, key3,
                                                            key4)) != this->end;
  }

  bool contains(K1 const &key1, K2 const &key2, K3 const &key3,
                K4 const &key4) const {
    return this->containsKey(key1, key2, key3, key4);
  }

  V operator()(K1 const &key1, K2 const &key2, K3 const &key3,
               K4 const &key4) const {
    return this->get(key1, key2, key3, key4);
  }
};

template <typename K1, typename K2, typename K3, typename K4, typename K5,
          typename V>
class HashMap5 final
    : public std::unordered_map<xpress::maps::MapKey5<K1, K2, K3, K4, K5>, V> {
public:
  V get(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
        K5 const &key5) const {
    auto it = this->find(xpress::maps::MapKey5<K1, K2, K3, K4, K5>(
        key1, key2, key3, key4, key5));
    if (it == this->end()) {
      throw std::invalid_argument("no such key");
    }
    return it->second;
  }

  void put(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
           K5 const &key5, V value) {
    this->insert(std::pair<xpress::maps::MapKey5<K1, K2, K3, K4, K5>, V>(
        xpress::maps::MapKey5<K1, K2, K3, K4, K5>(key1, key2, key3, key4, key5),
        value));
  }

  bool containsKey(K1 const &key1, K2 const &key2, K3 const &key3,
                   K4 const &key4, K5 const &key5) const {
    return this->find(xpress::maps::MapKey5<K1, K2, K3, K4, K5>(
               key1, key2, key3, key4, key5)) != this->end;
  }

  bool contains(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
                K5 const &key5) const {
    return this->containsKey(key1, key2, key3, key4, key5);
  }

  V operator()(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
               K5 const &key5) const {
    return this->get(key1, key2, key3, key4, key5);
  }
};

template <typename K1, typename K2, typename V>
class TreeMap2 final
    : public std::map<xpress::maps::ComparableMapKey2<K1, K2>, V> {
public:
  V get(K1 const &key1, K2 const &key2) const {
    auto it = this->find(xpress::maps::ComparableMapKey2<K1, K2>(key1, key2));
    if (it == this->end()) {
      throw std::invalid_argument("no such key");
    }
    return it->second;
  }

  V put(K1 const &key1, K2 const &key2, V value) {
    this->insert(std::pair<xpress::maps::ComparableMapKey2<K1, K2>, V>(
        xpress::maps::ComparableMapKey2<K1, K2>(key1, key2), value));
    return value;
  }

  bool containsKey(K1 const &key1, K2 const &key2) const {
    return this->find(xpress::maps::ComparableMapKey2<K1, K2>(key1, key2)) !=
           this->end;
  }

  bool contains(K1 const &key1, K2 const &key2) const {
    return this->containsKey(key1, key2);
  }

  V operator()(K1 const &key1, K2 const &key2) const {
    return this->get(key1, key2);
  }
};

template <typename K1, typename K2, typename K3, typename V>
class TreeMap3 final
    : public std::map<xpress::maps::ComparableMapKey3<K1, K2, K3>, V> {
public:
  V get(K1 const &key1, K2 const &key2, K3 const &key3) const {
    auto it = this->find(
        xpress::maps::ComparableMapKey3<K1, K2, K3>(key1, key2, key3));
    if (it == this->end()) {
      throw std::invalid_argument("no such key");
    }
    return it->second;
  }

  V put(K1 const &key1, K2 const &key2, K3 const &key3, V value) {
    this->insert(std::pair<xpress::maps::ComparableMapKey3<K1, K2, K3>, V>(
        xpress::maps::ComparableMapKey3<K1, K2, K3>(key1, key2, key3), value));
    return value;
  }

  bool containsKey(K1 const &key1, K2 const &key2, K3 const &key3) const {
    return this->find(xpress::maps::ComparableMapKey3<K1, K2, K3>(
               key1, key2, key3)) != this->end;
  }

  bool contains(K1 const &key1, K2 const &key2, K3 const &key3) const {
    return this->containsKey(key1, key2, key3);
  }

  V operator()(K1 const &key1, K2 const &key2, K3 const &key3) const {
    return this->get(key1, key2, key3);
  }
};

template <typename K1, typename K2, typename K3, typename K4, typename V>
class TreeMap4 final
    : public std::map<xpress::maps::ComparableMapKey4<K1, K2, K3, K4>, V> {
public:
  V get(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4) const {
    auto it = this->find(xpress::maps::ComparableMapKey4<K1, K2, K3, K4>(
        key1, key2, key3, key4));
    if (it == this->end()) {
      throw std::invalid_argument("no such key");
    }
    return it->second;
  }

  V put(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
        V value) {
    this->insert(std::pair<xpress::maps::ComparableMapKey4<K1, K2, K3, K4>, V>(
        xpress::maps::ComparableMapKey4<K1, K2, K3, K4>(key1, key2, key3, key4),
        value));
    return value;
  }

  bool containsKey(K1 const &key1, K2 const &key2, K3 const &key3,
                   K4 const &key4) const {
    return this->find(xpress::maps::ComparableMapKey4<K1, K2, K3, K4>(
               key1, key2, key3, key4)) != this->end;
  }

  bool contains(K1 const &key1, K2 const &key2, K3 const &key3,
                K4 const &key4) const {
    return this->containsKey(key1, key2, key3, key4);
  }

  V operator()(K1 const &key1, K2 const &key2, K3 const &key3,
               K4 const &key4) const {
    return this->get(key1, key2, key3, key4);
  }
};

template <typename K1, typename K2, typename K3, typename K4, typename K5,
          typename V>
class TreeMap5 final
    : public std::map<xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>, V> {
public:
  V get(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
        K5 const &key5) const {
    auto it = this->find(xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>(
        key1, key2, key3, key4, key5));
    if (it == this->end()) {
      throw std::invalid_argument("no such key");
    }
    return it->second;
  }

  V put(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
        K5 const &key5, V value) {
    this->insert(
        std::pair<xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>, V>(
            xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>(
                key1, key2, key3, key4, key5),
            value));
    return value;
  }

  bool containsKey(K1 const &key1, K2 const &key2, K3 const &key3,
                   K4 const &key4, K5 const &key5) const {
    return this->find(xpress::maps::ComparableMapKey5<K1, K2, K3, K4, K5>(
               key1, key2, key3, key4, key5)) != this->end;
  }

  bool contains(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
                K5 const &key5) const {
    return this->containsKey(key1, key2, key3, key4, key5);
  }

  V operator()(K1 const &key1, K2 const &key2, K3 const &key3, K4 const &key4,
               K5 const &key5) const {
    return this->get(key1, key2, key3, key4, key5);
  }
};

} // namespace maps
} // namespace xpress
#endif
