/* *********************************************************************** *
 *                                                                         *
 * xpress.hpp                                                              *
 *                                                                         *
 * C++ API for Xpress-Optimizer                                            *
 *                                                                         *
 * (c) Copyright Fair Isaac Corporation 2024-2025. All rights reserved     *
 * For FICO Xpress Optimizer v45.01.02                                     *
 *                                                                         *
 * *********************************************************************** */

#ifndef XPRESS_CXX_H
#define XPRESS_CXX_H 1

#if __cplusplus < 201703L
#error "This file requires at least C++ 17"

#endif

#include <xprs.h>

#include <algorithm>
#include <array>
#include <atomic>
#include <cmath>
#include <cstddef>
#include <cstring>
#include <exception>
#include <functional>
#include <iomanip>
#include <iostream>
#include <memory>
#include <mutex>
#include <optional>
#include <sstream>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <unordered_map>
#include <variant>
#include <vector>

#include <xpress_classes.hpp>

#ifndef XPRESS_ASSERT_DISABLE
#include <cassert>
#define xpress_assert assert
#endif

#define xpress_isstream(s)                                                     \
  typename s##IsStream =                                                       \
      typename std::enable_if<xpress::is_stream<s>::value>::type

#define xpress_iscollection(c)                                                 \
  typename c##IsCollection =                                                   \
      typename std::enable_if<xpress::is_collection<c>::value>::type

#define xpress_isiterable(i)                                                   \
  typename i##IsIterable =                                                     \
      typename std::enable_if<xpress::is_iterable<i>::value>::type

#define xpress_isinvocable(f, r, ...)                                          \
  typename f##IsInvocable = typename std::enable_if<std::is_convertible<       \
      typename std::invoke_result<__VA_ARGS__>::type, r>::value>::type

#define xpress_isintegral(x)                                                   \
  typename x##IsIntegral =                                                     \
      typename std::enable_if<std::is_integral<x>::value>::type

#define xpress_resultof(...) std::invoke_result<__VA_ARGS__>::type

#define xpress_elementtype(x) x::value_type

#define xpress_iteratorvaluetype(i) std::iterator_traits<i>::value_type

extern "C" {

int XPRS_CC XPRScxxinit(char const *licpath,
                        void(XPRS_CC *onObjectDestroyed)(void *),
                        void(XPRS_CC *onCallbackAdded)(void *),
                        void(XPRS_CC *onCallbackRemoved)(void *));

int XPRS_CC XPRScxxfree(void);
}

#ifndef xpress_stringify
#define xpress_stringify(x) #x
#endif
#if defined(__clang__)
#define xpress_diags_push _Pragma(xpress_stringify(clang diagnostic push))
#define xpress_diags_pop _Pragma(xpress_stringify(clang diagnostic pop))
#define xpress_ignore_deprecated                                               \
  _Pragma(                                                                     \
      xpress_stringify(clang diagnostic ignored "-Wdeprecated-declarations"))
#elif defined(__GNUC__)
#define xpress_diags_push _Pragma(xpress_stringify(GCC diagnostic push))
#define xpress_diags_pop _Pragma(xpress_stringify(GCC diagnostic pop))
#define xpress_ignore_deprecated                                               \
  _Pragma(xpress_stringify(GCC diagnostic ignored "-Wdeprecated-"              \
                                                  "declarations"))
#elif defined(_MSC_VER)
#define xpress_diags_push __pragma(warning(push))
#define xpress_diags_pop __pragma(warning(pop))
#define xpress_ignore_deprecated                                               \
  __pragma(warning(disable : 4995)) __pragma(warning(disable : 4996))
#else
#ifndef xpress_diags_push
#define xpress_diags_push
#endif
#ifndef xpress_diags_pop
#define xpress_diags_pop
#endif
#ifndef xpress_ignore_deprecated
#define xpress_ignore_deprecated
#endif
#endif

namespace xpress {

class RaiiHelper {
  RaiiHelper(RaiiHelper const &) = delete;
  RaiiHelper(RaiiHelper &&) = delete;
  RaiiHelper &operator=(RaiiHelper const &) = delete;
  RaiiHelper &operator=(RaiiHelper &&) = delete;

  std::function<void()> onExitScope;

public:
  RaiiHelper(std::function<void()> code) : onExitScope(code) {}

  ~RaiiHelper() { onExitScope(); }
};

namespace objects {}

template <typename T> class has_value_type {
  struct Yes {
    char a[1];
  };
  struct No {
    char a[2];
  };

  template <typename X> static Yes test(typename X::value_type *);

  template <typename X> static No test(...);

public:
  static constexpr bool value = sizeof(test<T>(nullptr)) == sizeof(Yes);
};

template <typename C> struct has_size {
private:
  template <typename T>
  static constexpr typename std::enable_if<
      std::is_convertible<decltype(std::declval<T>().size()),
                          std::size_t>::value,
      std::true_type>::type
  test(T *);

  template <typename> static constexpr std::false_type test(...);

  typedef decltype(test<C>(nullptr)) type;

public:
  static constexpr bool value = type::value;
};

template <typename T, typename = void>
struct has_begin_function : std::false_type {};
template <typename T>
struct has_begin_function<T, std::void_t<decltype(begin(std::declval<T>()))>>
    : std::true_type {};

template <typename T, typename = void>
struct has_end_function : std::false_type {};
template <typename T>
struct has_end_function<T, std::void_t<decltype(end(std::declval<T>()))>>
    : std::true_type {};

template <typename C> struct has_begin_member {
private:
  struct No {
    char a[0];
  };
  template <typename T>
  static constexpr decltype(std::declval<T>().begin()) check(T *);

  template <typename> static constexpr No check(...);

  typedef std::is_same<decltype(check<C>(nullptr)), No> type;

public:
  static constexpr bool value = !type::value;
};

template <typename C> struct has_end_member {
private:
  struct No {
    char a[0];
  };
  template <typename T>
  static constexpr decltype(std::declval<T>().end()) check(T *);

  template <typename> static constexpr No check(...);

  typedef std::is_same<decltype(check<C>(nullptr)), No> type;

public:
  static constexpr bool value = !type::value;
};

template <typename C> struct has_begin {

  static constexpr bool value =
      has_begin_function<C>::value || has_begin_member<C>::value;
};

template <typename C> struct has_end {

  static constexpr bool value =
      has_end_function<C>::value || has_end_member<C>::value;
};

template <typename T> struct is_collection {

  static constexpr bool value =
      xpress::has_value_type<T>::value && xpress::has_size<T>::value &&
      xpress::has_begin<T>::value && xpress::has_end<T>::value;
};

template <typename T> struct is_stream {

  static constexpr bool value = xpress::has_value_type<T>::value &&
                                xpress::has_begin<T>::value &&
                                xpress::has_end<T>::value;
};

template <typename T> struct is_iterable {

  static constexpr bool value = xpress::has_value_type<T>::value &&
                                xpress::has_begin<T>::value &&
                                xpress::has_end<T>::value;
};

template <typename T> struct WrapInfo;

template <typename T> struct ImplInfo;

template <typename... T>
auto format(std::string const &format, T... args) -> std::string {

  char const *fmt = format.c_str();

  std::string result;
  result.resize(result.capacity());
  int bytes = std::snprintf(result.data(), result.size(), fmt, args...);
  if (bytes < 0)
    throw std::invalid_argument("invalid format");
  if (static_cast<unsigned int>(bytes) >= result.size()) {

    result.resize(bytes + 1);
    std::snprintf(result.data(), result.size(), fmt, args...);
  }
  result.resize(bytes);
  return result;
}

template <typename T>
inline auto toInt(T const &t) ->
    typename std::enable_if<std::numeric_limits<T>::is_signed, int>::type {
  if (t > std::numeric_limits<int>::max() ||
      t < std::numeric_limits<int>::min())
    throw std::range_error("number cannot be represented as 32bit integer");
  return static_cast<int>(t);
}

template <typename T>
inline auto toInt(T const &t) ->
    typename std::enable_if<!std::numeric_limits<T>::is_signed, int>::type {
  if (t > static_cast<unsigned int>(std::numeric_limits<int>::max()))
    throw std::range_error("number cannot be represented as 32bit integer");
  return static_cast<int>(t);
}

template <> inline int toInt<int>(int const &t) { return t; }

template <typename T, std::size_t D> struct MakeArray;

template <typename T> struct MakeArray<T, 1> {

  typedef std::vector<T> array_type;

  template <typename I> static void init(array_type &a, I const &i) {
    a.resize(i);
  }
};

template <typename T, std::size_t D> struct MakeArray {

  typedef std::vector<typename MakeArray<T, D - 1>::array_type> array_type;

  template <typename I, typename... C>
  static void init(array_type &a, I const &i, C... dims) {
    a.resize(typename array_type::size_type(i));
    for (auto j = 0; j < i; ++j) {
      MakeArray<T, D - 1>::init(a[j], dims...);
    }
  }
};

template <typename T, typename... D>
auto makeArray(D... dims) -> typename MakeArray<T, sizeof...(D)>::array_type {
  typename MakeArray<T, sizeof...(D)>::array_type a;
  MakeArray<T, sizeof...(D)>::init(a, dims...);
  return a;
}

template <typename T, typename C> auto toArray(C const &c) -> std::vector<T> {
  using std::begin;
  using std::end;
  return std::vector<T>(begin(c), end(c));
}

template <typename T> void packToArrayHelper(std::vector<T> &) {}

template <typename T, typename Arg, typename... Args>
void packToArrayHelper(std::vector<T> &v, Arg const &arg, Args... args) {

  v.push_back(arg);
  packToArrayHelper<T, Args...>(v, args...);
}

template <typename T, typename... Args>
auto packToArray(Args... args) -> std::vector<T> {
  std::vector<T> v;
  packToArrayHelper<T, Args...>(v, args...);
  return v;
}

template <bool...> struct BooleanPack;

template <bool... bs>
using AllTrue =
    std::is_same<BooleanPack<bs..., true>, BooleanPack<true, bs...>>;

template <typename R, typename... Ts>
using AllConvertible = AllTrue<std::is_convertible<Ts, R>::value...>;

template <typename T> class WrappedIterator;

template <typename T>
struct WrappedIteratorImplementationBase
    : public std::enable_shared_from_this<
          WrappedIteratorImplementationBase<T>> {

  virtual ~WrappedIteratorImplementationBase() {}

  virtual bool operator!=(WrappedIterator<T> const &other) const = 0;

  virtual T &operator*() = 0;

  virtual T &operator++() = 0;
};

template <typename T>
class WrappedIterator
    : public std::shared_ptr<WrappedIteratorImplementationBase<T>> {
  T tmp;

public:
  explicit WrappedIterator(
      std::shared_ptr<WrappedIteratorImplementationBase<T>> init)
      : std::shared_ptr<WrappedIteratorImplementationBase<T>>(init) {}
  WrappedIterator()
      : std::shared_ptr<WrappedIteratorImplementationBase<T>>(nullptr) {}

  typedef T value_type;
  typedef T *pointer;
  typedef T &reference;
  typedef std::input_iterator_tag iterator_category;
  typedef std::ptrdiff_t difference_type;

  bool operator!=(WrappedIterator<T> const &other) const {
    return this->get()->operator!=(other);
  }
  reference operator*() { return this->get()->operator*(); }
  reference operator->() { return operator*(); }

  reference operator++() { return this->get()->operator++(); }

  value_type operator++(int) {
    tmp = this->get()->operator*();
    this->get()->operator++();
    return tmp;
  }
};

template <typename T, typename I, typename X = T>
class WrappedIteratorImplementation
    : public WrappedIteratorImplementationBase<T> {
  I i;
  I iend;
  std::function<T(X const &)> xform;
  T value;

public:
  WrappedIteratorImplementation(I const &i, I const &iend)
      : i(i), iend(iend),
        xform([](X const &unchangedValue) { return unchangedValue; }), value() {
  }
  WrappedIteratorImplementation(I const &i, I const &iend,
                                std::function<T(X const &)> xform)
      : i(i), iend(iend), xform(xform), value() {}
  virtual bool
  operator!=(WrappedIterator<T> const &other) const final override {
    return i != dynamic_cast<WrappedIteratorImplementation<T, I, X> const *>(
                    other.get())
                    ->i;
  }
  virtual T &operator*() final override { return (value = xform(*i)); }
  virtual T &operator++() final override {

    ++i;
    if (i != iend)
      return operator*();
    else
      return (value = T());
  }
};

template <typename T>
struct IterableImplementationBase
    : public std::enable_shared_from_this<IterableImplementationBase<T>> {

  virtual ~IterableImplementationBase() {}

  virtual WrappedIterator<T> begin() const = 0;

  virtual WrappedIterator<T> end() const = 0;
};

template <typename T>
struct Iterable : public std::shared_ptr<IterableImplementationBase<T>> {
  typedef WrappedIterator<T> iterator;
  typedef WrappedIterator<T> const_iterator;
  typedef T value_type;

private:
  template <typename C> static auto get_begin(C const &container) {
    using std::begin;
    return begin(container);
  }
  template <typename C> static auto get_end(C const &container) {
    using std::end;
    return end(container);
  }

  struct DeepCopyContainer : public IterableImplementationBase<T> {
    std::vector<T> data;
    template <typename C>
    DeepCopyContainer(C const &container)
        : data(get_begin(container), get_end(container)) {}
    virtual WrappedIterator<T> begin() const override {
      return WrappedIterator<T>(
          std::make_shared<WrappedIteratorImplementation<
              T, typename std::vector<T>::const_iterator>>(data.cbegin(),
                                                           data.cend()));
    }
    virtual WrappedIterator<T> end() const override {
      return WrappedIterator<T>(
          std::make_shared<WrappedIteratorImplementation<
              T, typename std::vector<T>::const_iterator>>(data.cend(),
                                                           data.cend()));
    }
  };

  template <typename C>
  struct STLRefContainer : public IterableImplementationBase<T> {
    C const *c;
    STLRefContainer(C const *container) : c(container) {}
    virtual WrappedIterator<T> begin() const override {
      return WrappedIterator<T>(
          std::make_shared<
              WrappedIteratorImplementation<T, typename C::const_iterator>>(
              std::cbegin(*c), std::cend(*c)));
    }
    virtual WrappedIterator<T> end() const override {
      return WrappedIterator<T>(
          std::make_shared<
              WrappedIteratorImplementation<T, typename C::const_iterator>>(
              std::end(*c), std::cend(*c)));
    }
  };

  template <typename Out>
  class MappedIterable final : public IterableImplementationBase<Out> {
    Iterable<T> source;
    std::function<Out(T const &)> xform;

  public:
    MappedIterable(Iterable<T> source, std::function<Out(T const &)> xform)
        : source(source), xform(xform) {}
    virtual xpress::WrappedIterator<Out> begin() const override {
      return xpress::WrappedIterator<Out>(
          std::make_shared<xpress::WrappedIteratorImplementation<
              Out, xpress::WrappedIterator<T>, T>>(source.begin(), source.end(),
                                                   xform));
    }
    virtual xpress::WrappedIterator<Out> end() const override {
      return xpress::WrappedIterator<Out>(
          std::make_shared<xpress::WrappedIteratorImplementation<
              Out, xpress::WrappedIterator<T>, T>>(source.end(), source.end(),
                                                   xform));
    }
  };

public:
  Iterable(std::shared_ptr<IterableImplementationBase<T>> init)
      : std::shared_ptr<IterableImplementationBase<T>>(init) {}
  Iterable() : std::shared_ptr<IterableImplementationBase<T>>(nullptr) {}

  template <typename C> static auto fromArray(C const &c) -> Iterable<T> {
    return Iterable<T>(std::shared_ptr<IterableImplementationBase<T>>(
        std::make_shared<DeepCopyContainer>(c)));
  }

  template <typename C> static auto fromSTLRef(C const *c) -> Iterable<T> {
    return Iterable<T>(std::shared_ptr<IterableImplementationBase<T>>(
        std::make_shared<STLRefContainer<C>>(c)));
  }

  template <typename C> static auto fromSTLRef(C const &c) -> Iterable<T> {
    return Iterable<T>(std::shared_ptr<IterableImplementationBase<T>>(
        std::make_shared<STLRefContainer<C>>(&c)));
  }

  static auto makeNull() -> Iterable<T> {
    return Iterable<T>(std::shared_ptr<IterableImplementationBase<T>>(nullptr));
  }

  auto begin() const -> const_iterator { return this->get()->begin(); }

  auto end() const -> const_iterator { return this->get()->end(); }

  template <typename Out>
  auto map(std::function<Out(T const &)> xform) const -> Iterable<Out> {
    return Iterable<Out>(std::make_shared<MappedIterable<Out>>(
        *const_cast<Iterable<T> *>(this), xform));
  }
};

template <typename T>
class SingletonIterable final : public IterableImplementationBase<T> {
  T value;

  struct STLIterator final : public WrappedIteratorImplementationBase<T> {
    T value;
    bool consumed;
    STLIterator(T const &value) : value(value), consumed(false) {}
    STLIterator(T const &value, bool consumed)
        : value(value), consumed(consumed) {}
    virtual bool operator!=(WrappedIterator<T> const &other) const override {
      return dynamic_cast<STLIterator const *>(other.get())->consumed !=
             consumed;
    }
    virtual T &operator*() override { return value; }
    virtual T &operator++() override {
      consumed = true;
      return value;
    }
  };

public:
  SingletonIterable(T const &value) : value(value) {}

  virtual WrappedIterator<T> begin() const override {
    return WrappedIterator<T>(std::make_shared<STLIterator>(value));
  }
  virtual WrappedIterator<T> end() const override {
    return WrappedIterator<T>(std::make_shared<STLIterator>(value, true));
  }
};

template <typename I>
class RangeIterable final : public IterableImplementationBase<I> {
  I startVal;
  I endVal;

  struct STLIterator final : public WrappedIteratorImplementationBase<I> {
    I current;
    I value;
    STLIterator(I current) : current(current), value(current) {}
    virtual bool operator!=(WrappedIterator<I> const &other) const override {
      return dynamic_cast<STLIterator const *>(other.get())->current != current;
    }
    virtual I &operator*() override { return value; }
    virtual I &operator++() override { return (value = ++current); }
  };

public:
  RangeIterable(I startVal, I endVal) : startVal(startVal), endVal(endVal) {}

  typedef WrappedIterator<I> iterator;
  typedef WrappedIterator<I> const_iterator;

  virtual iterator begin() const override {
    return WrappedIterator<I>(std::make_shared<STLIterator>(startVal));
  }
  virtual iterator end() const override {
    return WrappedIterator<I>(std::make_shared<STLIterator>(endVal));
  }
};

template <typename V> struct SingletonView final : public V {
  typedef typename V::value_type value_type;

private:
  std::function<value_type()> makeValue;

  struct WrappedIteratorImpl final
      : public WrappedIteratorImplementationBase<value_type> {
    value_type value;
    bool isEnd;

    WrappedIteratorImpl(value_type const &v, bool e) : value(v), isEnd(e) {}

    virtual bool
    operator!=(WrappedIterator<value_type> const &other) const override {
      return dynamic_cast<WrappedIteratorImpl const *>(other.get())->isEnd !=
             isEnd;
    }
    virtual value_type &operator*() override { return value; }
    virtual value_type &operator++() override {
      isEnd = true;
      return value;
    }
  };

public:
  SingletonView(std::function<value_type()> makeValue) : makeValue(makeValue) {}
  virtual WrappedIterator<value_type> begin() const override {
    return WrappedIterator<value_type>(
        std::make_shared<WrappedIteratorImpl>(makeValue(), false));
  }
  virtual WrappedIterator<value_type> end() const override {
    return WrappedIterator<value_type>(
        std::make_shared<WrappedIteratorImpl>(makeValue(), true));
  }
};

template <typename V, typename D> struct IndexedView final : public V {
  typedef typename V::value_type value_type;

private:
  std::shared_ptr<D> keepAlive;

  int count;

  value_type initial;

  bool haveInitial;

  std::function<value_type(int)> makeValue;

  struct WrappedIteratorImpl final
      : public WrappedIteratorImplementationBase<value_type> {
    std::shared_ptr<D> keepAlive;
    value_type value;
    int currentIndex;
    int endIndex;
    std::function<value_type(int)> makeValue;

    WrappedIteratorImpl(std::shared_ptr<D> keepAlive, int endIndex,
                        std::function<value_type(int)> makeValue)
        : keepAlive(keepAlive), value(), currentIndex(0), endIndex(endIndex),
          makeValue(makeValue) {
      if (endIndex > 0)
        value = makeValue(0);
    }
    WrappedIteratorImpl(std::shared_ptr<D> keepAlive, value_type initial,
                        int endIndex, std::function<value_type(int)> makeValue)
        : keepAlive(keepAlive), value(initial), currentIndex(-1),
          endIndex(endIndex), makeValue(makeValue) {}
    WrappedIteratorImpl(std::shared_ptr<D> keepAlive, int endIndex)
        : keepAlive(keepAlive), value(), currentIndex(endIndex),
          endIndex(endIndex), makeValue(nullptr) {}

    virtual bool
    operator!=(WrappedIterator<value_type> const &other) const override {
      return dynamic_cast<WrappedIteratorImpl const *>(other.get())
                 ->currentIndex != currentIndex;
    }
    virtual value_type &operator*() override { return value; }
    virtual value_type &operator++() override {
      if (currentIndex < endIndex) {
        ++currentIndex;
        if (currentIndex < endIndex)
          value = makeValue(currentIndex);
      }
      return value;
    }
  };

public:
  IndexedView(std::shared_ptr<D> data, int count,
              std::function<value_type(int)> makeValue)
      : keepAlive(data), count(count), initial(), haveInitial(false),
        makeValue(makeValue) {}
  IndexedView(std::shared_ptr<D> data, int count, value_type const &initial,
              std::function<value_type(int)> makeValue)
      : keepAlive(data), count(count), initial(initial), haveInitial(true),
        makeValue(makeValue) {}

  virtual WrappedIterator<value_type> begin() const override {
    return WrappedIterator<value_type>(
        haveInitial ? std::make_shared<WrappedIteratorImpl>(keepAlive, initial,
                                                            count, makeValue)
                    : std::make_shared<WrappedIteratorImpl>(keepAlive, count,
                                                            makeValue));
  }
  virtual WrappedIterator<value_type> end() const override {
    return WrappedIterator<value_type>(
        std::make_shared<WrappedIteratorImpl>(keepAlive, count));
  }
};

template <typename V, typename C, typename D> struct STLView final : public V {
  typedef typename V::value_type value_type;

private:
  std::shared_ptr<D> keepAlive;
  C const *stl;
  std::function<value_type(typename C::value_type)> makeValue;

public:
  STLView(std::shared_ptr<D> keepAlive, C const *stl,
          std::function<value_type(typename C::value_type)> makeValue)
      : keepAlive(keepAlive), stl(stl), makeValue(makeValue) {}

private:
  struct WrappedIteratorImpl final
      : public WrappedIteratorImplementationBase<value_type> {
    std::shared_ptr<D> keepAlive;
    value_type value;
    typename C::const_iterator itCurrent;
    typename C::const_iterator itEnd;
    std::function<value_type(typename C::value_type)> makeValue;
    WrappedIteratorImpl(
        std::shared_ptr<D> keepAlive, typename C::const_iterator itCurrent,
        typename C::const_iterator itEnd,
        std::function<value_type(typename C::value_type)> makeValue)
        : keepAlive(keepAlive), value(), itCurrent(itCurrent), itEnd(itEnd),
          makeValue(makeValue) {
      if (itCurrent != itEnd)
        value = makeValue(*itCurrent);
    }
    virtual bool
    operator!=(WrappedIterator<value_type> const &other) const override {
      return dynamic_cast<WrappedIteratorImpl const *>(other.get())
                 ->itCurrent != itCurrent;
    }
    virtual value_type &operator*() override { return value; }
    virtual value_type &operator++() override {
      if (itCurrent != itEnd) {
        ++itCurrent;
        if (itCurrent != itEnd)
          value = makeValue(*itCurrent);
      }
      return value;
    }
  };

public:
  virtual WrappedIterator<value_type> begin() const override {
    return WrappedIterator<value_type>(std::make_shared<WrappedIteratorImpl>(
        keepAlive, stl->begin(), stl->end(), makeValue));
  }
  virtual WrappedIterator<value_type> end() const override {
    return WrappedIterator<value_type>(std::make_shared<WrappedIteratorImpl>(
        keepAlive, stl->end(), stl->end(), makeValue));
  }
};

template <typename Vin, typename Vout>
struct TransformedView final : public Vout {
  typedef typename Vout::value_type value_type;

private:
  std::shared_ptr<Vin> vin;
  std::function<value_type(typename Vin::value_type)> makeValue;

public:
  TransformedView(
      std::shared_ptr<Vin> vin,
      std::function<value_type(typename Vin::value_type const &)> makeValue)
      : vin(vin), makeValue(makeValue) {}

private:
  struct WrappedIteratorImpl final
      : public WrappedIteratorImplementationBase<value_type> {
    std::shared_ptr<Vin> keepAlive;
    value_type value;
    typename Vin::iterator it;
    typename Vin::iterator itEnd;
    std::function<value_type(typename Vin::value_type)> makeValue;
    WrappedIteratorImpl(
        std::shared_ptr<Vin> keepAlive, typename Vin::iterator it,
        typename Vin::iterator itEnd,
        std::function<value_type(typename Vin::value_type)> makeValue)
        : keepAlive(keepAlive), value(), it(it), itEnd(itEnd),
          makeValue(makeValue) {
      if (it != itEnd)
        value = makeValue(*it);
    }

    virtual bool
    operator!=(WrappedIterator<value_type> const &other) const override {
      return dynamic_cast<WrappedIteratorImpl const *>(other.get())->it != it;
    }
    virtual value_type &operator*() override { return value; }
    virtual value_type &operator++() override {
      if (it != itEnd) {
        ++it;
        value = makeValue(*it);
      }
      return value;
    }
  };

public:
  virtual WrappedIterator<value_type> begin() const override {
    return WrappedIterator<value_type>(std::make_shared<WrappedIteratorImpl>(
        vin, vin->begin(), vin->end(), makeValue));
  }
  virtual WrappedIterator<value_type> end() const override {
    return WrappedIterator<value_type>(std::make_shared<WrappedIteratorImpl>(
        vin, vin->end(), vin->end(), makeValue));
  }
};

template <typename V, typename E, typename S> struct SumView final : public V {
  typedef typename V::value_type value_type;
  typedef E element_type;

private:
  std::shared_ptr<S> keepAlive;
  std::vector<E> persistent;
  E const *data;
  int len;
  std::function<std::shared_ptr<V>(E const &)> getView;

public:
  SumView(std::shared_ptr<S> keepAlive, E const *data, int len,
          std::function<std::shared_ptr<V>(E const &)> getView)
      : keepAlive(keepAlive), persistent(), data(data), len(len),
        getView(getView) {}
  template <typename... Args, typename = typename std::enable_if<
                                  AllConvertible<E, Args...>::value>::type>
  SumView(std::function<std::shared_ptr<V>(E const &)> getView, Args... args)
      : keepAlive(nullptr), persistent{args...}, data(persistent.data()),
        len(xpress::toInt(persistent.size())), getView(getView) {}

private:
  struct WrappedIteratorImpl final
      : public WrappedIteratorImplementationBase<value_type> {
    std::shared_ptr<SumView> view;
    int dataIndex;
    std::shared_ptr<V> currentView;
    WrappedIterator<value_type> currentIt;
    WrappedIterator<value_type> currentEnd;
    bool atEnd;

    WrappedIteratorImpl(std::shared_ptr<SumView> view, bool isEnd)
        : view(view), dataIndex(0), currentView(nullptr), currentIt(nullptr),
          currentEnd(nullptr), atEnd(isEnd) {
      if (!isEnd) {
        atEnd = true;
        while (dataIndex < view->len) {
          currentView = view->getView(view->data[dataIndex]);
          currentIt = currentView->begin();
          currentEnd = currentView->end();
          if (currentIt != currentEnd) {
            atEnd = false;
            break;
          }
          ++dataIndex;
        }
      }
    }

    virtual bool
    operator!=(WrappedIterator<value_type> const &other) const override {
      return dynamic_cast<WrappedIteratorImpl const *>(other.get())->atEnd !=
             atEnd;
    }
    virtual value_type &operator*() override { return *currentIt; }
    virtual value_type &operator++() override {
      if (!atEnd) {
        ++currentIt;
        if (!(currentIt != currentEnd)) {
          atEnd = true;
          ++dataIndex;
          while (dataIndex < view->len) {
            currentView = view->getView(view->data[dataIndex]);
            currentIt = currentView->begin();
            currentEnd = currentView->end();
            if (currentIt != currentEnd) {
              atEnd = false;
              break;
            }
            ++dataIndex;
          }
        }
      }
      return *currentIt;
    }
  };

public:
  virtual WrappedIterator<value_type> begin() const override {
    return WrappedIterator<value_type>(std::make_shared<WrappedIteratorImpl>(
        std::dynamic_pointer_cast<SumView>(
            const_cast<SumView *>(this)->shared_from_this()),
        false));
  }
  virtual WrappedIterator<value_type> end() const override {
    return WrappedIterator<value_type>(std::make_shared<WrappedIteratorImpl>(
        std::dynamic_pointer_cast<SumView>(
            const_cast<SumView *>(this)->shared_from_this()),
        true));
  }
};

template <typename C>
auto sortMap(C const &c, bool reverse) -> std::vector<std::pair<
    typename std::remove_const<typename C::value_type::first_type>::type,
    typename C::value_type::second_type>> {
  typedef std::pair<
      typename std::remove_const<typename C::value_type::first_type>::type,
      typename C::value_type::second_type>
      pair_type;
  std::vector<pair_type> data(c.size());
  std::transform(c.cbegin(), c.cend(), data.begin(),
                 [](auto x) { return pair_type(x.first, x.second); });
  if (reverse)
    std::sort(
        data.begin(), data.end(), [](pair_type const &a, pair_type const &b) {
          return std::less<typename pair_type::first_type>()(b.first, a.first);
        });
  else
    std::sort(
        data.begin(), data.end(), [](pair_type const &a, pair_type const &b) {
          return std::less<typename pair_type::first_type>()(a.first, b.first);
        });
  return data;
}

template <typename S, typename D, typename I1, typename I2, typename L>
void arrayCopy(S const &src, I1 const &srcPos, D &dest, I2 const &destPos,
               L const &count) {
  for (L i = 0; i < count; ++i)
    dest[i + destPos] = src[i + srcPos];
}

template <typename... Args> auto concatStrings(Args... args) -> std::string {
  std::stringstream s;
  (s << ... << args);
  return s.str();
}

template <typename T>
inline auto combineHashHelper(std::size_t init, T const &t) -> std::size_t {
  return init * 23 + std::hash<T>()(t);
}

template <typename T, typename... More>
inline auto combineHashHelper(std::size_t init, T const &t, More const &...more)
    -> std::size_t {
  return combineHashHelper(combineHashHelper(init, t), more...);
}

template <typename... Args>
inline auto combineHash(Args const &...args) -> std::size_t {

  return combineHashHelper(17, args...);
}

template <typename T,
          typename std::enable_if<has_size<T>::value, bool>::type = true>
auto sizeHint(T const &t) -> int {
  return toInt(t.size());
}

template <typename T,
          typename std::enable_if<!has_size<T>::value, bool>::type = true>
auto sizeHint(T const &) -> int {
  return -1;
}

class XPRS;
class Object;
class XPRSProblem;
class NameList;
class MIPSolPool;
class MIPSolEnum;
class BranchObject;
class UserFunctionMap;
class UserFunctionMapDelta;
class UserFunctionVecMap;
class UserFunctionVecMapDelta;
class UserFunctionMultiMap;
class UserFunctionMultiMapDelta;

class XPRSException : public std::exception {
  char msg[XPRS_MAXMESSAGELENGTH];
  int errorCode;

private:
  template <typename T>
  static auto objectError(T obj, int code,
                          int (*f)(T, int *, char *, int, int *))
      -> XPRSException {
    XPRSException ret(code);
    int subcode, bytes;
    if (f(obj, &subcode, ret.msg, sizeof(ret.msg), &bytes))
      return XPRSException(code);
    else {
      ret.msg[sizeof(ret.msg) - 1] = '\0';
      return ret;
    }
  }

public:
  static inline constexpr int const UNKNOWN_ERROR =
      std::numeric_limits<int>::min();

  XPRSException(int error) : errorCode(error) {
    ::snprintf(msg, sizeof(msg) - 1, "error %d", error);
    msg[sizeof(msg) - 1] = '\0';
  }

  XPRSException(char const *message) : errorCode(UNKNOWN_ERROR) {
#ifdef _WIN32
    ::strncpy_s(msg, sizeof(msg) - 1, message, _TRUNCATE);
#else
    ::strncpy(msg, message, sizeof(msg) - 1);
#endif
    msg[sizeof(msg) - 1] = '\0';
  }

  XPRSException(std::string const &message) : errorCode(UNKNOWN_ERROR) {
    std::size_t len = message.length();
    if (len >= sizeof(msg))
      len = sizeof(msg) - 1;
    ::memcpy(msg, message.data(), len);
    msg[len] = '\0';
  }

  virtual char const *what() const noexcept override { return msg; }

  int getErrorCode() const { return errorCode; }

  static auto licenseError(int code) -> XPRSException {
    XPRSException ret(code);
    if (XPRSgetlicerrmsg(ret.msg, sizeof(ret.msg))) {
      return ret;
    } else {
      ret.msg[sizeof(ret.msg) - 1] = '\0';
    }
    return ret;
  }

  static auto error(XPRSprob prob, int code) -> XPRSException {
    XPRSException ret(code);
    if (XPRSgetlasterror(prob, ret.msg))
      return XPRSException(code);
    else {
      ret.msg[sizeof(ret.msg) - 1] = '\0';
      return ret;
    }
  }

  static auto error(XPRSmipsolpool pool, int code) -> XPRSException {
    return objectError(pool, code, XPRS_msp_getlasterror);
  }

  static auto error(XPRSmipsolenum enumerator, int code) -> XPRSException {
    return objectError(enumerator, code, XPRS_mse_getlasterror);
  }

  static auto error(XPRSnamelist namelist, int code) -> XPRSException {
    xpress_diags_push xpress_ignore_deprecated return objectError(
        namelist, code, XPRS_nml_getlasterror);
    xpress_diags_pop
  }

  static auto error(XPRSbranchobject branchobject, int code) -> XPRSException {
    return objectError(branchobject, code, XPRS_bo_getlasterror);
  }
};

template <typename T> class Array {
public:
  typedef T value_type;
  value_type *data;

  Array(T *d) : data(d) {}

  Array(std::vector<T> &d) : data(d.data()) {}

  Array(std::vector<typename std::remove_const<T>::type> const &d)
      : data(const_cast<T *>(d.data())) {}

  template <std::size_t N> Array(std::array<T, N> &d) : data(d.data()) {}

  template <std::size_t N>
  Array(std::array<typename std::remove_const<T>::type, N> const &d)
      : data(d.data()) {}

  Array(Array<typename std::remove_const<T>::type> const &a) : data(a.data) {}

  Array(std::nullptr_t) : data(nullptr) {}

  Array() : data(nullptr) {}

  bool operator==(std::nullptr_t) const { return data == nullptr; }

  bool operator!=(std::nullptr_t) const { return data != nullptr; }

  explicit operator bool() const { return data != nullptr; }

  template <typename I> value_type &operator[](I i) const { return data[i]; }

  operator value_type *() const { return data; }
};

template <typename T> class SizedArray : public Array<T> {
public:
  typedef std::size_t size_type;

  size_type length;

  SizedArray(std::vector<T> &d) : Array<T>(d), length(d.size()) {}

  SizedArray(std::vector<typename std::remove_const<T>::type> const &d)
      : Array<T>(d), length(d.size()) {}

  template <size_type N>
  SizedArray(std::array<T, N> &d) : Array<T>(d), length(N) {}

  template <size_type N>
  SizedArray(std::array<typename std::remove_const<T>::type, N> const &d)
      : Array<T>(d), length(N) {}

  SizedArray(SizedArray<typename std::remove_const<T>::type> const &s)
      : Array<T>(s), length(s.length) {}

  SizedArray(T *d, size_type l) : Array<T>(d), length(l) {}

  SizedArray(std::nullptr_t) : Array<T>(nullptr), length(0) {}

  SizedArray() : Array<T>(nullptr), length(0) {}

  auto size() const -> size_type { return length; }

  auto begin() const -> typename Array<T>::value_type * { return this->data; }

  auto end() const -> typename Array<T>::value_type * {
    return this->data + length;
  }

  auto toVector() const -> std::vector<typename std::remove_const<T>::type> {
    return std::vector<typename std::remove_const<T>::type>(begin(), end());
  }
};

template <typename T, typename I>
auto copyOfArray(Array<T> const &s, I const &len)
    -> std::vector<typename std::remove_const<T>::type> {
  return std::vector<typename std::remove_const<T>::type>(s.data, s.data + len);
}

template <typename T, typename I>
auto copyOfArray(std::vector<T> const &s, I const &len)
    -> std::vector<typename std::remove_const<T>::type> {
  return std::vector<typename std::remove_const<T>::type>(s.data(),
                                                          s.data() + len);
}

class CallbackExceptionHandler {
protected:
  CallbackExceptionHandler() {}

public:
  virtual ~CallbackExceptionHandler() {}

  virtual void setCBException(std::exception_ptr e) const noexcept = 0;
};

class CallbackHolder {
  CallbackHolder(CallbackHolder const &) = delete;
  CallbackHolder &operator=(CallbackHolder const &) = delete;
  CallbackHolder(CallbackHolder &&) = delete;
  CallbackHolder &operator=(CallbackHolder &&) = delete;

  std::atomic<int> references;

protected:
  CallbackHolder() : references(0) {}

public:
  virtual ~CallbackHolder() {}

  int incRef() { return ++references; }

  int decRef() { return --references; }

  int refCount() const { return references; }
};

class CallbackHandle {

  CallbackHolder *holder;

public:
  CallbackHandle(CallbackHolder *cbHolder = nullptr) : holder(cbHolder) {}

  CallbackHolder *getHolder() const { return holder; }
};

enum struct ColumnType : char {

  Continuous = (char)'C',
  Integer = (char)'I',
  Binary = (char)'B',
  SemiContinuous = (char)'S',
  SemiInteger = (char)'R',
  PartialInteger = (char)'P'
};
inline bool operator==(ColumnType a, char b) {
  return static_cast<char>(a) == b;
}
inline bool operator!=(ColumnType a, char b) {
  return static_cast<char>(a) != b;
}
inline bool operator==(char a, ColumnType b) {
  return a == static_cast<char>(b);
}
inline bool operator!=(char a, ColumnType b) {
  return a != static_cast<char>(b);
}
inline std::string to_string(ColumnType e) {
  switch (e) {
  case ColumnType::Continuous:
    return "Continuous";
  case ColumnType::Integer:
    return "Integer";
  case ColumnType::Binary:
    return "Binary";
  case ColumnType::SemiContinuous:
    return "SemiContinuous";
  case ColumnType::SemiInteger:
    return "SemiInteger";
  case ColumnType::PartialInteger:
    return "PartialInteger";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<char>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, ColumnType e) {
  os << to_string(e);
  return os;
}

template <typename T> class ExtendableArrayImplementation;

template <typename T> class ExtendableArray final {
protected:
  std::shared_ptr<ExtendableArrayImplementation<T>> implPointer = nullptr;

public:
  bool isSameImpl(ExtendableArray const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(ExtendableArrayImplementation<T> const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline ExtendableArray(
      std::shared_ptr<ExtendableArrayImplementation<T>> init);

public:
  inline ExtendableArray(ExtendableArrayImplementation<T> const *init);
  inline virtual ~ExtendableArray(){};

private:
  inline ExtendableArrayImplementation<T> *impl();
  inline ExtendableArrayImplementation<T> const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(ExtendableArray const &other) const {
    return impl() == other.impl();
  }
  inline ExtendableArray(xpress::SizedArray<T const> const &initial);

  inline auto append(T value) -> void;

  inline auto append(xpress::SizedArray<T const> const &values) -> void;

  inline auto getLength() -> int;

  inline auto getCapacity() -> int;

  inline auto getData() -> std::vector<T> &;

  inline auto reset() -> void;

  inline auto setLength(int newLength) -> void;
  inline ExtendableArray();
};
class ExtendableDoubleArrayImplementation;

class ExtendableDoubleArray final {
public:
  typedef ExtendableDoubleArrayImplementation impl_type;

protected:
  std::shared_ptr<ExtendableDoubleArrayImplementation> implPointer = nullptr;

public:
  bool isSameImpl(ExtendableDoubleArray const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(ExtendableDoubleArrayImplementation const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline ExtendableDoubleArray(
      std::shared_ptr<ExtendableDoubleArrayImplementation> init);

public:
  inline ExtendableDoubleArray(ExtendableDoubleArrayImplementation const *init);
  inline virtual ~ExtendableDoubleArray(){};

private:
  inline ExtendableDoubleArrayImplementation *impl();
  inline ExtendableDoubleArrayImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(ExtendableDoubleArray const &other) const {
    return impl() == other.impl();
  }
  inline ExtendableDoubleArray(xpress::SizedArray<double const> const &initial);

  inline auto append(double value) -> void;

  inline auto append(xpress::SizedArray<double const> const &values) -> void;

  inline auto getLength() -> int;

  inline auto getCapacity() -> int;

  inline auto getData() -> std::vector<double> &;

  inline auto reset() -> void;

  inline auto setLength(int newLength) -> void;
  inline ExtendableDoubleArray();
};
class ExtendableIntArrayImplementation;

class ExtendableIntArray final {
public:
  typedef ExtendableIntArrayImplementation impl_type;

protected:
  std::shared_ptr<ExtendableIntArrayImplementation> implPointer = nullptr;

public:
  bool isSameImpl(ExtendableIntArray const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(ExtendableIntArrayImplementation const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline ExtendableIntArray(
      std::shared_ptr<ExtendableIntArrayImplementation> init);

public:
  inline ExtendableIntArray(ExtendableIntArrayImplementation const *init);
  inline virtual ~ExtendableIntArray(){};

private:
  inline ExtendableIntArrayImplementation *impl();
  inline ExtendableIntArrayImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(ExtendableIntArray const &other) const {
    return impl() == other.impl();
  }
  inline ExtendableIntArray(xpress::SizedArray<int const> const &initial);

  inline auto append(int value) -> void;

  inline auto append(xpress::SizedArray<int const> const &values) -> void;

  inline auto getLength() -> int;

  inline auto getCapacity() -> int;

  inline auto getData() -> std::vector<int> &;

  inline auto reset() -> void;

  inline auto setLength(int newLength) -> void;
  inline ExtendableIntArray();
};
class ExtendableTypeArrayImplementation;

class ExtendableTypeArray final {
public:
  typedef ExtendableTypeArrayImplementation impl_type;

protected:
  std::shared_ptr<ExtendableTypeArrayImplementation> implPointer = nullptr;

public:
  bool isSameImpl(ExtendableTypeArray const &other) const {
    return implPointer.get() == other.implPointer.get();
  }
  bool isSameImpl(ExtendableTypeArrayImplementation const *other) const {
    return implPointer.get() == other;
  }

protected:
  inline ExtendableTypeArray(
      std::shared_ptr<ExtendableTypeArrayImplementation> init);

public:
  inline ExtendableTypeArray(ExtendableTypeArrayImplementation const *init);
  inline virtual ~ExtendableTypeArray(){};

private:
  inline ExtendableTypeArrayImplementation *impl();
  inline ExtendableTypeArrayImplementation const *impl() const;

public:
  template <typename Type> bool isInstance() const {
    return dynamic_cast<Type const *>(impl()) != nullptr;
  }
  template <typename Derived> inline Derived cast() const;
  explicit operator bool() const { return implPointer != nullptr; }
  bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
  bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
  bool equals(ExtendableTypeArray const &other) const {
    return impl() == other.impl();
  }
  inline ExtendableTypeArray(xpress::SizedArray<char const> const &initial);

  inline auto append(char value) -> void;

  inline auto append(xpress::SizedArray<char const> const &values) -> void;

  inline auto getLength() -> int;

  inline auto getCapacity() -> int;

  inline auto getData() -> std::vector<char> &;

  inline auto reset() -> void;

  inline auto setLength(int newLength) -> void;
  inline ExtendableTypeArray();
};

class IndicatorInfo final {
public:
  int indicator;

  bool complemented;

  int row;

  inline IndicatorInfo(int indicator, bool complemented, int row);
};

class PwlBreakpoint final {
private:
  double x;

public:
  auto getX() const -> double { return x; }

  void setX(double newX) { x = newX; }

private:
  double y;

public:
  auto getY() const -> double { return y; }

  void setY(double newY) { y = newY; }

  inline PwlBreakpoint();

  inline PwlBreakpoint(double x, double y);

  inline auto equals(PwlBreakpoint const &other) const -> bool;

  inline auto compareTo(PwlBreakpoint const &other) const -> int;

  inline auto operator==(PwlBreakpoint const &other) const -> bool;

  inline auto operator!=(PwlBreakpoint const &other) const -> bool;

  inline auto getHashCode() const -> std::size_t;

  static inline auto convert(std::vector<xpress::PwlBreakpoint> breakpoints)
      -> std::vector<std::vector<double>>;

private:
  template <typename Strm0, xpress_isstream(Strm0)>
  static inline auto convert(Strm0 const &breakpoints, int sizeHint)
      -> std::vector<std::vector<double>>;

public:
  template <typename Strm0, xpress_isstream(Strm0)>
  static inline auto convert(Strm0 const &breakpoints)
      -> std::vector<std::vector<double>>;

  static inline auto convert(xpress::SizedArray<double const> const &xval,
                             xpress::SizedArray<double const> const &yval)
      -> std::vector<xpress::PwlBreakpoint>;
};

enum struct RowType : char {

  LEQ = (char)'L',
  EQ = (char)'E',
  GEQ = (char)'G',
  Range = (char)'R',
  N = (char)'N'
};
inline bool operator==(RowType a, char b) { return static_cast<char>(a) == b; }
inline bool operator!=(RowType a, char b) { return static_cast<char>(a) != b; }
inline bool operator==(char a, RowType b) { return a == static_cast<char>(b); }
inline bool operator!=(char a, RowType b) { return a != static_cast<char>(b); }
inline std::string to_string(RowType e) {
  switch (e) {
  case RowType::LEQ:
    return "LEQ";
  case RowType::EQ:
    return "EQ";
  case RowType::GEQ:
    return "GEQ";
  case RowType::Range:
    return "Range";
  case RowType::N:
    return "N";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<char>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, RowType e) {
  os << to_string(e);
  return os;
}

enum struct SetType : char {

  SOS1 = (char)'1',
  SOS2 = (char)'2'
};
inline bool operator==(SetType a, char b) { return static_cast<char>(a) == b; }
inline bool operator!=(SetType a, char b) { return static_cast<char>(a) != b; }
inline bool operator==(char a, SetType b) { return a == static_cast<char>(b); }
inline bool operator!=(char a, SetType b) { return a != static_cast<char>(b); }
inline std::string to_string(SetType e) {
  switch (e) {
  case SetType::SOS1:
    return "SOS1";
  case SetType::SOS2:
    return "SOS2";
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<char>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, SetType e) {
  os << to_string(e);
  return os;
}

template <typename T>
class ExtendableArrayImplementation final
    : public std::enable_shared_from_this<ExtendableArrayImplementation<T>> {
  friend class ExtendableArray<T>;

public:
  virtual ~ExtendableArrayImplementation() {}

private:
  std::vector<T> data;

  int len;

public:
  inline ExtendableArrayImplementation(
      xpress::SizedArray<T const> const &initial);
  inline auto append(T value) -> void;
  inline auto append(xpress::SizedArray<T const> const &values) -> void;
  inline auto getLength() -> int;
  inline auto getCapacity() -> int;
  inline auto getData() -> std::vector<T> &;
  inline auto reset() -> void;
  inline auto setLength(int newLength) -> void;
};
class ExtendableDoubleArrayImplementation final
    : public std::enable_shared_from_this<ExtendableDoubleArrayImplementation> {
  friend class ExtendableDoubleArray;

public:
  virtual ~ExtendableDoubleArrayImplementation() {}

private:
  std::vector<double> data;

  int len;

public:
  inline ExtendableDoubleArrayImplementation(
      xpress::SizedArray<double const> const &initial);
  inline auto append(double value) -> void;
  inline auto append(xpress::SizedArray<double const> const &values) -> void;
  inline auto getLength() -> int;
  inline auto getCapacity() -> int;
  inline auto getData() -> std::vector<double> &;
  inline auto reset() -> void;
  inline auto setLength(int newLength) -> void;
};
class ExtendableIntArrayImplementation final
    : public std::enable_shared_from_this<ExtendableIntArrayImplementation> {
  friend class ExtendableIntArray;

public:
  virtual ~ExtendableIntArrayImplementation() {}

private:
  std::vector<int> data;

  int len;

public:
  inline ExtendableIntArrayImplementation(
      xpress::SizedArray<int const> const &initial);
  inline auto append(int value) -> void;
  inline auto append(xpress::SizedArray<int const> const &values) -> void;
  inline auto getLength() -> int;
  inline auto getCapacity() -> int;
  inline auto getData() -> std::vector<int> &;
  inline auto reset() -> void;
  inline auto setLength(int newLength) -> void;
};
class ExtendableTypeArrayImplementation final
    : public std::enable_shared_from_this<ExtendableTypeArrayImplementation> {
  friend class ExtendableTypeArray;

public:
  virtual ~ExtendableTypeArrayImplementation() {}

private:
  std::vector<char> data;

  int len;

public:
  inline ExtendableTypeArrayImplementation(
      xpress::SizedArray<char const> const &initial);
  inline auto append(char value) -> void;
  inline auto append(xpress::SizedArray<char const> const &values) -> void;
  inline auto getLength() -> int;
  inline auto getCapacity() -> int;
  inline auto getData() -> std::vector<char> &;
  inline auto reset() -> void;
  inline auto setLength(int newLength) -> void;
};
template <typename T>
xpress::ExtendableArray<T>::ExtendableArray(
    std::shared_ptr<ExtendableArrayImplementation<T>> init)
    : implPointer(init) {}
template <typename T>
xpress::ExtendableArray<T>::ExtendableArray(
    ExtendableArrayImplementation<T> const *init)
    : implPointer(
          init ? const_cast<ExtendableArrayImplementation<T> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ExtendableArrayImplementation<T>>(nullptr)) {}
template <typename T>
xpress::ExtendableArrayImplementation<T> *xpress::ExtendableArray<T>::impl() {
  return dynamic_cast<ExtendableArrayImplementation<T> *>(implPointer.get());
}
template <typename T>
xpress::ExtendableArrayImplementation<T> const *
xpress::ExtendableArray<T>::impl() const {
  return dynamic_cast<ExtendableArrayImplementation<T> *>(implPointer.get());
}
template <typename T>
xpress::ExtendableArray<T>::ExtendableArray(
    xpress::SizedArray<T const> const &initial)
    : implPointer(std::make_shared<ExtendableArrayImplementation<T>>(initial)) {
}
template <typename T> auto xpress::ExtendableArray<T>::append(T value) -> void {
  impl()->append(value);
}
template <typename T>
auto xpress::ExtendableArray<T>::append(
    xpress::SizedArray<T const> const &values) -> void {
  impl()->append(values);
}
template <typename T> auto xpress::ExtendableArray<T>::getLength() -> int {
  return impl()->getLength();
}
template <typename T> auto xpress::ExtendableArray<T>::getCapacity() -> int {
  return impl()->getCapacity();
}
template <typename T>
auto xpress::ExtendableArray<T>::getData() -> std::vector<T> & {
  return impl()->getData();
}
template <typename T> auto xpress::ExtendableArray<T>::reset() -> void {
  impl()->reset();
}
template <typename T>
auto xpress::ExtendableArray<T>::setLength(int newLength) -> void {
  impl()->setLength(newLength);
}
template <typename T>
xpress::ExtendableArray<T>::ExtendableArray() : implPointer(nullptr) {}
xpress::ExtendableDoubleArray::ExtendableDoubleArray(
    std::shared_ptr<ExtendableDoubleArrayImplementation> init)
    : implPointer(init) {}
xpress::ExtendableDoubleArray::ExtendableDoubleArray(
    ExtendableDoubleArrayImplementation const *init)
    : implPointer(init ? const_cast<ExtendableDoubleArrayImplementation *>(init)
                             ->shared_from_this()
                       : std::shared_ptr<ExtendableDoubleArrayImplementation>(
                             nullptr)) {}
xpress::ExtendableDoubleArrayImplementation *
xpress::ExtendableDoubleArray::impl() {
  return dynamic_cast<ExtendableDoubleArrayImplementation *>(implPointer.get());
}
xpress::ExtendableDoubleArrayImplementation const *
xpress::ExtendableDoubleArray::impl() const {
  return dynamic_cast<ExtendableDoubleArrayImplementation *>(implPointer.get());
}
template <typename Derived>
Derived xpress::ExtendableDoubleArray::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::ExtendableDoubleArray::ExtendableDoubleArray(
    xpress::SizedArray<double const> const &initial)
    : implPointer(
          std::make_shared<ExtendableDoubleArrayImplementation>(initial)) {}
auto xpress::ExtendableDoubleArray::append(double value) -> void {
  impl()->append(value);
}
auto xpress::ExtendableDoubleArray::append(
    xpress::SizedArray<double const> const &values) -> void {
  impl()->append(values);
}
auto xpress::ExtendableDoubleArray::getLength() -> int {
  return impl()->getLength();
}
auto xpress::ExtendableDoubleArray::getCapacity() -> int {
  return impl()->getCapacity();
}
auto xpress::ExtendableDoubleArray::getData() -> std::vector<double> & {
  return impl()->getData();
}
auto xpress::ExtendableDoubleArray::reset() -> void { impl()->reset(); }
auto xpress::ExtendableDoubleArray::setLength(int newLength) -> void {
  impl()->setLength(newLength);
}
xpress::ExtendableDoubleArray::ExtendableDoubleArray() : implPointer(nullptr) {}
xpress::ExtendableIntArray::ExtendableIntArray(
    std::shared_ptr<ExtendableIntArrayImplementation> init)
    : implPointer(init) {}
xpress::ExtendableIntArray::ExtendableIntArray(
    ExtendableIntArrayImplementation const *init)
    : implPointer(
          init ? const_cast<ExtendableIntArrayImplementation *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ExtendableIntArrayImplementation>(nullptr)) {}
xpress::ExtendableIntArrayImplementation *xpress::ExtendableIntArray::impl() {
  return dynamic_cast<ExtendableIntArrayImplementation *>(implPointer.get());
}
xpress::ExtendableIntArrayImplementation const *
xpress::ExtendableIntArray::impl() const {
  return dynamic_cast<ExtendableIntArrayImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::ExtendableIntArray::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::ExtendableIntArray::ExtendableIntArray(
    xpress::SizedArray<int const> const &initial)
    : implPointer(std::make_shared<ExtendableIntArrayImplementation>(initial)) {
}
auto xpress::ExtendableIntArray::append(int value) -> void {
  impl()->append(value);
}
auto xpress::ExtendableIntArray::append(
    xpress::SizedArray<int const> const &values) -> void {
  impl()->append(values);
}
auto xpress::ExtendableIntArray::getLength() -> int {
  return impl()->getLength();
}
auto xpress::ExtendableIntArray::getCapacity() -> int {
  return impl()->getCapacity();
}
auto xpress::ExtendableIntArray::getData() -> std::vector<int> & {
  return impl()->getData();
}
auto xpress::ExtendableIntArray::reset() -> void { impl()->reset(); }
auto xpress::ExtendableIntArray::setLength(int newLength) -> void {
  impl()->setLength(newLength);
}
xpress::ExtendableIntArray::ExtendableIntArray() : implPointer(nullptr) {}
xpress::ExtendableTypeArray::ExtendableTypeArray(
    std::shared_ptr<ExtendableTypeArrayImplementation> init)
    : implPointer(init) {}
xpress::ExtendableTypeArray::ExtendableTypeArray(
    ExtendableTypeArrayImplementation const *init)
    : implPointer(
          init ? const_cast<ExtendableTypeArrayImplementation *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ExtendableTypeArrayImplementation>(nullptr)) {}
xpress::ExtendableTypeArrayImplementation *xpress::ExtendableTypeArray::impl() {
  return dynamic_cast<ExtendableTypeArrayImplementation *>(implPointer.get());
}
xpress::ExtendableTypeArrayImplementation const *
xpress::ExtendableTypeArray::impl() const {
  return dynamic_cast<ExtendableTypeArrayImplementation *>(implPointer.get());
}
template <typename Derived> Derived xpress::ExtendableTypeArray::cast() const {
  return const_cast<typename Derived::impl_type *>(
      dynamic_cast<typename Derived::impl_type const *>(impl()));
}
xpress::ExtendableTypeArray::ExtendableTypeArray(
    xpress::SizedArray<char const> const &initial)
    : implPointer(
          std::make_shared<ExtendableTypeArrayImplementation>(initial)) {}
auto xpress::ExtendableTypeArray::append(char value) -> void {
  impl()->append(value);
}
auto xpress::ExtendableTypeArray::append(
    xpress::SizedArray<char const> const &values) -> void {
  impl()->append(values);
}
auto xpress::ExtendableTypeArray::getLength() -> int {
  return impl()->getLength();
}
auto xpress::ExtendableTypeArray::getCapacity() -> int {
  return impl()->getCapacity();
}
auto xpress::ExtendableTypeArray::getData() -> std::vector<char> & {
  return impl()->getData();
}
auto xpress::ExtendableTypeArray::reset() -> void { impl()->reset(); }
auto xpress::ExtendableTypeArray::setLength(int newLength) -> void {
  impl()->setLength(newLength);
}
xpress::ExtendableTypeArray::ExtendableTypeArray() : implPointer(nullptr) {}

class VariableBuilder {

public:
  template <typename C, xpress_isintegral(C)> class ArrayBuilderImplementation;

  template <typename C, xpress_isintegral(C)> class ArrayBuilder final {
  protected:
    std::shared_ptr<ArrayBuilderImplementation<C, CIsIntegral>> implPointer =
        nullptr;

  public:
    bool isSameImpl(ArrayBuilder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool
    isSameImpl(ArrayBuilderImplementation<C, CIsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ArrayBuilder(
        std::shared_ptr<ArrayBuilderImplementation<C, CIsIntegral>> init);

  public:
    inline ArrayBuilder(ArrayBuilderImplementation<C, CIsIntegral> const *init);
    inline virtual ~ArrayBuilder(){};

  private:
    inline ArrayBuilderImplementation<C, CIsIntegral> *impl();
    inline ArrayBuilderImplementation<C, CIsIntegral> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ArrayBuilder const &other) const {
      return impl() == other.impl();
    }
    inline ArrayBuilder(C dim);

    inline auto getDim(int dim) const -> int;

    inline auto getLB() const -> std::function<double(C)> const &;

    inline auto getUB() const -> std::function<double(C)> const &;

    inline auto getLimit() const -> std::function<double(C)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(C)> const &;

    inline auto getType() const -> std::function<xpress::ColumnType(C)> const &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLB(Func0 newLb) -> ArrayBuilder<C, CIsIntegral> &;

    inline auto withLB(double newLb) -> ArrayBuilder<C, CIsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withUB(Func0 newUb) -> ArrayBuilder<C, CIsIntegral> &;

    inline auto withUB(double newUb) -> ArrayBuilder<C, CIsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLimit(Func0 newLimit) -> ArrayBuilder<C, CIsIntegral> &;

    inline auto withLimit(double newLimit) -> ArrayBuilder<C, CIsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C)>
    inline auto withName(Func0 newName) -> ArrayBuilder<C, CIsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> ArrayBuilder<C, CIsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C)>
    inline auto withType(Func0 newType) -> ArrayBuilder<C, CIsIntegral> &;

    inline auto withType(ColumnType newType) -> ArrayBuilder<C, CIsIntegral> &;

    inline auto build(XPRSProblem *prob) -> std::vector<int>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::vector<xpress::objects::Variable>;
    inline ArrayBuilder();
  };

  template <typename C, xpress_isintegral(C)>
  static inline auto array(C dim) -> xpress::VariableBuilder::ArrayBuilder<C>;

public:
  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  class Array2BuilderImplementation;

  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  class Array2Builder final {
  protected:
    std::shared_ptr<
        Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(Array2Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> const
            *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline Array2Builder(
        std::shared_ptr<
            Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>>
            init);

  public:
    inline Array2Builder(Array2BuilderImplementation<C1, C2, C1IsIntegral,
                                                     C2IsIntegral> const *init);
    inline virtual ~Array2Builder(){};

  private:
    inline Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> *
    impl();
    inline Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> const
        *
        impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(Array2Builder const &other) const {
      return impl() == other.impl();
    }
    inline Array2Builder(C1 dim1, C2 dim2);

    inline auto getDim(int dim) const -> int;

    inline auto getLB() const -> std::function<double(C1, C2)> const &;

    inline auto getUB() const -> std::function<double(C1, C2)> const &;

    inline auto getLimit() const -> std::function<double(C1, C2)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(C1, C2)> const &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(C1, C2)> const &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLB(Func0 newLb)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withLB(double newLb)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withUB(Func0 newUb)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withUB(double newUb)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLimit(Func0 newLimit)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2)>
    inline auto withName(Func0 newName)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2)>
    inline auto withType(Func0 newType)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto build(XPRSProblem *prob) -> std::vector<std::vector<int>>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::vector<std::vector<xpress::objects::Variable>>;
    inline Array2Builder();
  };

  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  static inline auto array2(C1 dim1, C2 dim2)
      -> xpress::VariableBuilder::Array2Builder<C1, C2>;

public:
  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  class Array3BuilderImplementation;

  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  class Array3Builder final {
  protected:
    std::shared_ptr<Array3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                                C2IsIntegral, C3IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(Array3Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                    C3IsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline Array3Builder(
        std::shared_ptr<Array3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                                    C2IsIntegral, C3IsIntegral>>
            init);

  public:
    inline Array3Builder(
        Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                    C3IsIntegral> const *init);
    inline virtual ~Array3Builder(){};

  private:
    inline Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> *
    impl();
    inline Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(Array3Builder const &other) const {
      return impl() == other.impl();
    }
    inline Array3Builder(C1 dim1, C2 dim2, C3 dim3);

    inline auto getDim(int dim) const -> int;

    inline auto getLB() const -> std::function<double(C1, C2, C3)> const &;

    inline auto getUB() const -> std::function<double(C1, C2, C3)> const &;

    inline auto getLimit() const -> std::function<double(C1, C2, C3)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(C1, C2, C3)> const &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(C1, C2, C3)> const &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLB(Func0 newLb)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    inline auto withLB(double newLb)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withUB(Func0 newUb)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    inline auto withUB(double newUb)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLimit(Func0 newLimit)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    inline auto withLimit(double newLimit)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3)>
    inline auto withName(Func0 newName)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    inline auto withName(std::optional<std::string> newName)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2, C3)>
    inline auto withType(Func0 newType)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    inline auto withType(ColumnType newType)
        -> Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>
            &;

    inline auto build(XPRSProblem *prob)
        -> std::vector<std::vector<std::vector<int>>>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::vector<std::vector<std::vector<xpress::objects::Variable>>>;
    inline Array3Builder();
  };

  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  static inline auto array3(C1 dim1, C2 dim2, C3 dim3)
      -> xpress::VariableBuilder::Array3Builder<C1, C2, C3>;

public:
  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  class Array4BuilderImplementation;

  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  class Array4Builder final {
  protected:
    std::shared_ptr<Array4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(Array4Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                    C3IsIntegral, C4IsIntegral> const *other)
        const {
      return implPointer.get() == other;
    }

  protected:
    inline Array4Builder(std::shared_ptr<Array4BuilderImplementation<
                             C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                             C3IsIntegral, C4IsIntegral>>
                             init);

  public:
    inline Array4Builder(
        Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                    C3IsIntegral, C4IsIntegral> const *init);
    inline virtual ~Array4Builder(){};

  private:
    inline Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
        *impl();
    inline Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral,
                                       C4IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(Array4Builder const &other) const {
      return impl() == other.impl();
    }
    inline Array4Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4);

    inline auto getDim(int dim) const -> int;

    inline auto getLB() const -> std::function<double(C1, C2, C3, C4)> const &;

    inline auto getUB() const -> std::function<double(C1, C2, C3, C4)> const &;

    inline auto getLimit() const
        -> std::function<double(C1, C2, C3, C4)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(C1, C2, C3, C4)> const &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(C1, C2, C3, C4)> const &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLB(Func0 newLb)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    inline auto withLB(double newLb)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withUB(Func0 newUb)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    inline auto withUB(double newUb)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLimit(Func0 newLimit)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4)>
    inline auto withName(Func0 newName)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4)>
    inline auto withType(Func0 newType)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral> &;

    inline auto build(XPRSProblem *prob)
        -> std::vector<std::vector<std::vector<std::vector<int>>>>;

    inline auto build(xpress::objects::XpressProblem *prob) -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>;
    inline Array4Builder();
  };

  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  static inline auto array4(C1 dim1, C2 dim2, C3 dim3, C4 dim4)
      -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4>;

public:
  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  class Array5BuilderImplementation;

  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  class Array5Builder final {
  protected:
    std::shared_ptr<Array5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(Array5Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                    C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                    C5IsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline Array5Builder(std::shared_ptr<Array5BuilderImplementation<
                             C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                             C3IsIntegral, C4IsIntegral, C5IsIntegral>>
                             init);

  public:
    inline Array5Builder(Array5BuilderImplementation<
                         C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> const *init);
    inline virtual ~Array5Builder(){};

  private:
    inline Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> *
    impl();
    inline Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(Array5Builder const &other) const {
      return impl() == other.impl();
    }
    inline Array5Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4, C5 dim5);

    inline auto getDim(int dim) const -> int;

    inline auto getLB() const
        -> std::function<double(C1, C2, C3, C4, C5)> const &;

    inline auto getUB() const
        -> std::function<double(C1, C2, C3, C4, C5)> const &;

    inline auto getLimit() const
        -> std::function<double(C1, C2, C3, C4, C5)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(C1, C2, C3, C4, C5)> const
            &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(C1, C2, C3, C4, C5)> const &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLB(Func0 newLb)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withLB(double newLb)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withUB(Func0 newUb)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withUB(double newUb)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLimit(Func0 newLimit)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4, C5)>
    inline auto withName(Func0 newName)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4, C5)>
    inline auto withType(Func0 newType)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                         C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto build(XPRSProblem *prob)
        -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::vector<std::vector<
            std::vector<std::vector<std::vector<xpress::objects::Variable>>>>>;
    inline Array5Builder();
  };

  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  static inline auto array5(C1 dim1, C2 dim2, C3 dim3, C4 dim4, C5 dim5)
      -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5>;

public:
  template <typename C, xpress_isintegral(C)>
  class ColumnArrayBuilderImplementation;

  template <typename C, xpress_isintegral(C)> class ColumnArrayBuilder final {
  protected:
    std::shared_ptr<ColumnArrayBuilderImplementation<C, CIsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(ColumnArrayBuilder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        ColumnArrayBuilderImplementation<C, CIsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnArrayBuilder(
        std::shared_ptr<ColumnArrayBuilderImplementation<C, CIsIntegral>> init);

  public:
    inline ColumnArrayBuilder(
        ColumnArrayBuilderImplementation<C, CIsIntegral> const *init);
    inline virtual ~ColumnArrayBuilder(){};

  private:
    inline ColumnArrayBuilderImplementation<C, CIsIntegral> *impl();
    inline ColumnArrayBuilderImplementation<C, CIsIntegral> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnArrayBuilder const &other) const {
      return impl() == other.impl();
    }
    inline ColumnArrayBuilder(C dim, xpress::XPRSProblem *prob);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLB(Func0 newLb) -> ColumnArrayBuilder<C, CIsIntegral> &;

    inline auto withLB(double newLb) -> ColumnArrayBuilder<C, CIsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withUB(Func0 newUb) -> ColumnArrayBuilder<C, CIsIntegral> &;

    inline auto withUB(double newUb) -> ColumnArrayBuilder<C, CIsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArrayBuilder<C, CIsIntegral> &;

    inline auto withLimit(double newLimit)
        -> ColumnArrayBuilder<C, CIsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C)>
    inline auto withName(Func0 newName) -> ColumnArrayBuilder<C, CIsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnArrayBuilder<C, CIsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C)>
    inline auto withType(Func0 newType) -> ColumnArrayBuilder<C, CIsIntegral> &;

    inline auto withType(ColumnType newType)
        -> ColumnArrayBuilder<C, CIsIntegral> &;

    inline auto toArray() -> std::vector<int>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(Func0 makeResult,
                        std::function<void(I, C, int)> addResult) -> I;
    inline ColumnArrayBuilder();
  };

public:
  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  class ColumnArray2BuilderImplementation;

  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  class ColumnArray2Builder final {
  protected:
    std::shared_ptr<
        ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(ColumnArray2Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(ColumnArray2BuilderImplementation<
                    C1, C2, C1IsIntegral, C2IsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnArray2Builder(
        std::shared_ptr<ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                                          C2IsIntegral>>
            init);

  public:
    inline ColumnArray2Builder(ColumnArray2BuilderImplementation<
                               C1, C2, C1IsIntegral, C2IsIntegral> const *init);
    inline virtual ~ColumnArray2Builder(){};

  private:
    inline ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
        *impl();
    inline ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                             C2IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnArray2Builder const &other) const {
      return impl() == other.impl();
    }
    inline ColumnArray2Builder(C1 dim1, C2 dim2, xpress::XPRSProblem *prob);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLB(Func0 newLb)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withLB(double newLb)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withUB(Func0 newUb)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withUB(double newUb)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2)>
    inline auto withName(Func0 newName)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2)>
    inline auto withType(Func0 newType)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto toArray() -> std::vector<std::vector<int>>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(Func0 makeResult,
                        std::function<void(I, C1, C2, int)> addResult) -> I;
    inline ColumnArray2Builder();
  };

public:
  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  class ColumnArray3BuilderImplementation;

  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  class ColumnArray3Builder final {
  protected:
    std::shared_ptr<ColumnArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(ColumnArray3Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(ColumnArray3BuilderImplementation<
                    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const
                        *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnArray3Builder(
        std::shared_ptr<ColumnArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>
            init);

  public:
    inline ColumnArray3Builder(
        ColumnArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const *init);
    inline virtual ~ColumnArray3Builder(){};

  private:
    inline ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> *
    impl();
    inline ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnArray3Builder const &other) const {
      return impl() == other.impl();
    }
    inline ColumnArray3Builder(C1 dim1, C2 dim2, C3 dim3,
                               xpress::XPRSProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLB(Func0 newLb)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    inline auto withLB(double newLb)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withUB(Func0 newUb)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    inline auto withUB(double newUb)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3)>
    inline auto withName(Func0 newName)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2, C3)>
    inline auto withType(Func0 newType)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral> &;

    inline auto toArray() -> std::vector<std::vector<std::vector<int>>>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(Func0 makeResult,
                        std::function<void(I, C1, C2, C3, int)> addResult) -> I;
    inline ColumnArray3Builder();
  };

public:
  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  class ColumnArray4BuilderImplementation;

  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  class ColumnArray4Builder final {
  protected:
    std::shared_ptr<ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(ColumnArray4Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(ColumnArray4BuilderImplementation<
                    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
                    C4IsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnArray4Builder(
        std::shared_ptr<ColumnArray4BuilderImplementation<
            C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral>>
            init);

  public:
    inline ColumnArray4Builder(ColumnArray4BuilderImplementation<
                               C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> const *init);
    inline virtual ~ColumnArray4Builder(){};

  private:
    inline ColumnArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        *impl();
    inline ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnArray4Builder const &other) const {
      return impl() == other.impl();
    }
    inline ColumnArray4Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                               xpress::XPRSProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLB(Func0 newLb)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    inline auto withLB(double newLb)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withUB(Func0 newUb)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    inline auto withUB(double newUb)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4)>
    inline auto withName(Func0 newName)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4)>
    inline auto withType(Func0 newType)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral> &;

    inline auto toArray()
        -> std::vector<std::vector<std::vector<std::vector<int>>>>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(Func0 makeResult,
                        std::function<void(I, C1, C2, C3, C4, int)> addResult)
        -> I;
    inline ColumnArray4Builder();
  };

public:
  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  class ColumnArray5BuilderImplementation;

  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  class ColumnArray5Builder final {
  protected:
    std::shared_ptr<ColumnArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(ColumnArray5Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool
    isSameImpl(ColumnArray5BuilderImplementation<
               C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
               C4IsIntegral, C5IsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnArray5Builder(
        std::shared_ptr<ColumnArray5BuilderImplementation<
            C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral, C5IsIntegral>>
            init);

  public:
    inline ColumnArray5Builder(
        ColumnArray5BuilderImplementation<
            C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral, C5IsIntegral> const *init);
    inline virtual ~ColumnArray5Builder(){};

  private:
    inline ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> *
    impl();
    inline ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnArray5Builder const &other) const {
      return impl() == other.impl();
    }
    inline ColumnArray5Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4, C5 dim5,
                               xpress::XPRSProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLB(Func0 newLb)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withLB(double newLb)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withUB(Func0 newUb)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withUB(double newUb)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4, C5)>
    inline auto withName(Func0 newName)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4, C5)>
    inline auto withType(Func0 newType)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto toArray()
        -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, C3, C4, C5, int)> addResult) -> I;
    inline ColumnArray5Builder();
  };

public:
  template <typename K1> class MapBuilderImplementation;

  template <typename K1> class MapBuilder {
  protected:
    std::shared_ptr<MapBuilderImplementation<K1>> implPointer = nullptr;

  public:
    bool isSameImpl(MapBuilder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(MapBuilderImplementation<K1> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline MapBuilder(std::shared_ptr<MapBuilderImplementation<K1>> init);

  public:
    inline MapBuilder(MapBuilderImplementation<K1> const *init);
    inline virtual ~MapBuilder(){};

  private:
    inline MapBuilderImplementation<K1> *impl();
    inline MapBuilderImplementation<K1> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(MapBuilder const &other) const {
      return impl() == other.impl();
    }
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

    inline auto getLB() const -> std::function<double(K1)> const &;

    inline auto getUB() const -> std::function<double(K1)> const &;

    inline auto getLimit() const -> std::function<double(K1)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(K1)> const &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(K1)> const &;

    inline auto getFilter() const -> std::function<bool(K1)> const &;

    template <typename Iter0, xpress_isiterable(Iter0)>
    inline MapBuilder(Iter0 const &iterable1);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLB(Func0 newLb) -> MapBuilder<K1> &;

    inline auto withLB(double newLb) -> MapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withUB(Func0 newUb) -> MapBuilder<K1> &;

    inline auto withUB(double newUb) -> MapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLimit(Func0 newLimit) -> MapBuilder<K1> &;

    inline auto withLimit(double newLimit) -> MapBuilder<K1> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1)>
    inline auto withName(Func0 newName) -> MapBuilder<K1> &;

    inline auto withName(std::optional<std::string> newName)
        -> MapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, ColumnType, Func0, K1)>
    inline auto withType(Func0 newType) -> MapBuilder<K1> &;

    inline auto withType(ColumnType newType) -> MapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1)>
    inline auto withFilter(Func0 newFilter) -> MapBuilder<K1> &;

    inline auto build(XPRSProblem *prob) -> std::unordered_map<K1, int>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::unordered_map<K1, xpress::objects::Variable>;
    inline MapBuilder();
  };

  template <typename Iter0, typename K1 = typename xpress_elementtype(Iter0),
            xpress_isiterable(Iter0)>
  static inline auto map(Iter0 const &iterable1)
      -> xpress::VariableBuilder::MapBuilder<K1>;

  template <typename K1>
  static inline auto map(xpress::SizedArray<K1 const> const &arr1)
      -> xpress::VariableBuilder::MapBuilder<K1>;

public:
  template <typename K1, typename K2> class Map2BuilderImplementation;

  template <typename K1, typename K2> class Map2Builder {
  protected:
    std::shared_ptr<Map2BuilderImplementation<K1, K2>> implPointer = nullptr;

  public:
    bool isSameImpl(Map2Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(Map2BuilderImplementation<K1, K2> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline Map2Builder(std::shared_ptr<Map2BuilderImplementation<K1, K2>> init);

  public:
    inline Map2Builder(Map2BuilderImplementation<K1, K2> const *init);
    inline virtual ~Map2Builder(){};

  private:
    inline Map2BuilderImplementation<K1, K2> *impl();
    inline Map2BuilderImplementation<K1, K2> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(Map2Builder const &other) const {
      return impl() == other.impl();
    }
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

    inline auto getIterable2() const -> xpress::Iterable<K2> const &;

    inline auto getLB() const -> std::function<double(K1, K2)> const &;

    inline auto getUB() const -> std::function<double(K1, K2)> const &;

    inline auto getLimit() const -> std::function<double(K1, K2)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(K1, K2)> const &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(K1, K2)> const &;

    inline auto getFilter() const -> std::function<bool(K1, K2)> const &;

    template <typename Iter0, typename Iter1, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1)>
    inline Map2Builder(Iter0 const &iterable1, Iter1 const &iterable2);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLB(Func0 newLb) -> Map2Builder<K1, K2> &;

    inline auto withLB(double newLb) -> Map2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withUB(Func0 newUb) -> Map2Builder<K1, K2> &;

    inline auto withUB(double newUb) -> Map2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLimit(Func0 newLimit) -> Map2Builder<K1, K2> &;

    inline auto withLimit(double newLimit) -> Map2Builder<K1, K2> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2)>
    inline auto withName(Func0 newName) -> Map2Builder<K1, K2> &;

    inline auto withName(std::optional<std::string> newName)
        -> Map2Builder<K1, K2> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2)>
    inline auto withType(Func0 newType) -> Map2Builder<K1, K2> &;

    inline auto withType(ColumnType newType) -> Map2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1, K2)>
    inline auto withFilter(Func0 newFilter) -> Map2Builder<K1, K2> &;

    inline auto build(XPRSProblem *prob) -> xpress::maps::HashMap2<K1, K2, int>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable>;
    inline Map2Builder();
  };

  template <typename Iter0, typename Iter1,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1)>
  static inline auto map2(Iter0 const &iterable1, Iter1 const &iterable2)
      -> xpress::VariableBuilder::Map2Builder<K1, K2>;

  template <typename K1, typename K2>
  static inline auto map2(xpress::SizedArray<K1 const> const &arr1,
                          xpress::SizedArray<K2 const> const &arr2)
      -> xpress::VariableBuilder::Map2Builder<K1, K2>;

public:
  template <typename K1, typename K2, typename K3>
  class Map3BuilderImplementation;

  template <typename K1, typename K2, typename K3> class Map3Builder {
  protected:
    std::shared_ptr<Map3BuilderImplementation<K1, K2, K3>> implPointer =
        nullptr;

  public:
    bool isSameImpl(Map3Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(Map3BuilderImplementation<K1, K2, K3> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline Map3Builder(
        std::shared_ptr<Map3BuilderImplementation<K1, K2, K3>> init);

  public:
    inline Map3Builder(Map3BuilderImplementation<K1, K2, K3> const *init);
    inline virtual ~Map3Builder(){};

  private:
    inline Map3BuilderImplementation<K1, K2, K3> *impl();
    inline Map3BuilderImplementation<K1, K2, K3> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(Map3Builder const &other) const {
      return impl() == other.impl();
    }
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

    inline auto getIterable2() const -> xpress::Iterable<K2> const &;

    inline auto getIterable3() const -> xpress::Iterable<K3> const &;

    inline auto getLB() const -> std::function<double(K1, K2, K3)> const &;

    inline auto getUB() const -> std::function<double(K1, K2, K3)> const &;

    inline auto getLimit() const -> std::function<double(K1, K2, K3)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(K1, K2, K3)> const &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(K1, K2, K3)> const &;

    inline auto getFilter() const -> std::function<bool(K1, K2, K3)> const &;

    template <typename Iter0, typename Iter1, typename Iter2,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2)>
    inline Map3Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                       Iter2 const &iterable3);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLB(Func0 newLb) -> Map3Builder<K1, K2, K3> &;

    inline auto withLB(double newLb) -> Map3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withUB(Func0 newUb) -> Map3Builder<K1, K2, K3> &;

    inline auto withUB(double newUb) -> Map3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLimit(Func0 newLimit) -> Map3Builder<K1, K2, K3> &;

    inline auto withLimit(double newLimit) -> Map3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3)>
    inline auto withName(Func0 newName) -> Map3Builder<K1, K2, K3> &;

    inline auto withName(std::optional<std::string> newName)
        -> Map3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3)>
    inline auto withType(Func0 newType) -> Map3Builder<K1, K2, K3> &;

    inline auto withType(ColumnType newType) -> Map3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3)>
    inline auto withFilter(Func0 newFilter) -> Map3Builder<K1, K2, K3> &;

    inline auto build(XPRSProblem *prob)
        -> xpress::maps::HashMap3<K1, K2, K3, int>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable>;
    inline Map3Builder();
  };

  template <typename Iter0, typename Iter1, typename Iter2,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2)>
  static inline auto map3(Iter0 const &iterable1, Iter1 const &iterable2,
                          Iter2 const &iterable3)
      -> xpress::VariableBuilder::Map3Builder<K1, K2, K3>;

  template <typename K1, typename K2, typename K3>
  static inline auto map3(xpress::SizedArray<K1 const> const &arr1,
                          xpress::SizedArray<K2 const> const &arr2,
                          xpress::SizedArray<K3 const> const &arr3)
      -> xpress::VariableBuilder::Map3Builder<K1, K2, K3>;

public:
  template <typename K1, typename K2, typename K3, typename K4>
  class Map4BuilderImplementation;

  template <typename K1, typename K2, typename K3, typename K4>
  class Map4Builder {
  protected:
    std::shared_ptr<Map4BuilderImplementation<K1, K2, K3, K4>> implPointer =
        nullptr;

  public:
    bool isSameImpl(Map4Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool
    isSameImpl(Map4BuilderImplementation<K1, K2, K3, K4> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline Map4Builder(
        std::shared_ptr<Map4BuilderImplementation<K1, K2, K3, K4>> init);

  public:
    inline Map4Builder(Map4BuilderImplementation<K1, K2, K3, K4> const *init);
    inline virtual ~Map4Builder(){};

  private:
    inline Map4BuilderImplementation<K1, K2, K3, K4> *impl();
    inline Map4BuilderImplementation<K1, K2, K3, K4> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(Map4Builder const &other) const {
      return impl() == other.impl();
    }
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

    inline auto getIterable2() const -> xpress::Iterable<K2> const &;

    inline auto getIterable3() const -> xpress::Iterable<K3> const &;

    inline auto getIterable4() const -> xpress::Iterable<K4> const &;

    inline auto getLB() const -> std::function<double(K1, K2, K3, K4)> const &;

    inline auto getUB() const -> std::function<double(K1, K2, K3, K4)> const &;

    inline auto getLimit() const
        -> std::function<double(K1, K2, K3, K4)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(K1, K2, K3, K4)> const &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(K1, K2, K3, K4)> const &;

    inline auto getFilter() const
        -> std::function<bool(K1, K2, K3, K4)> const &;

    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
    inline Map4Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                       Iter2 const &iterable3, Iter3 const &iterable4);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLB(Func0 newLb) -> Map4Builder<K1, K2, K3, K4> &;

    inline auto withLB(double newLb) -> Map4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withUB(Func0 newUb) -> Map4Builder<K1, K2, K3, K4> &;

    inline auto withUB(double newUb) -> Map4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLimit(Func0 newLimit) -> Map4Builder<K1, K2, K3, K4> &;

    inline auto withLimit(double newLimit) -> Map4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4)>
    inline auto withName(Func0 newName) -> Map4Builder<K1, K2, K3, K4> &;

    inline auto withName(std::optional<std::string> newName)
        -> Map4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4)>
    inline auto withType(Func0 newType) -> Map4Builder<K1, K2, K3, K4> &;

    inline auto withType(ColumnType newType) -> Map4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4)>
    inline auto withFilter(Func0 newFilter) -> Map4Builder<K1, K2, K3, K4> &;

    inline auto build(XPRSProblem *prob)
        -> xpress::maps::HashMap4<K1, K2, K3, K4, int>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable>;
    inline Map4Builder();
  };

  template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            typename K4 = typename xpress_elementtype(Iter3),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
  static inline auto map4(Iter0 const &iterable1, Iter1 const &iterable2,
                          Iter2 const &iterable3, Iter3 const &iterable4)
      -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>;

  template <typename K1, typename K2, typename K3, typename K4>
  static inline auto map4(xpress::SizedArray<K1 const> const &arr1,
                          xpress::SizedArray<K2 const> const &arr2,
                          xpress::SizedArray<K3 const> const &arr3,
                          xpress::SizedArray<K4 const> const &arr4)
      -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>;

public:
  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class Map5BuilderImplementation;

  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class Map5Builder {
  protected:
    std::shared_ptr<Map5BuilderImplementation<K1, K2, K3, K4, K5>> implPointer =
        nullptr;

  public:
    bool isSameImpl(Map5Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        Map5BuilderImplementation<K1, K2, K3, K4, K5> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline Map5Builder(
        std::shared_ptr<Map5BuilderImplementation<K1, K2, K3, K4, K5>> init);

  public:
    inline Map5Builder(
        Map5BuilderImplementation<K1, K2, K3, K4, K5> const *init);
    inline virtual ~Map5Builder(){};

  private:
    inline Map5BuilderImplementation<K1, K2, K3, K4, K5> *impl();
    inline Map5BuilderImplementation<K1, K2, K3, K4, K5> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(Map5Builder const &other) const {
      return impl() == other.impl();
    }
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

    inline auto getIterable2() const -> xpress::Iterable<K2> const &;

    inline auto getIterable3() const -> xpress::Iterable<K3> const &;

    inline auto getIterable4() const -> xpress::Iterable<K4> const &;

    inline auto getIterable5() const -> xpress::Iterable<K5> const &;

    inline auto getLB() const
        -> std::function<double(K1, K2, K3, K4, K5)> const &;

    inline auto getUB() const
        -> std::function<double(K1, K2, K3, K4, K5)> const &;

    inline auto getLimit() const
        -> std::function<double(K1, K2, K3, K4, K5)> const &;

    inline auto getName() const
        -> std::function<std::optional<std::string>(K1, K2, K3, K4, K5)> const
            &;

    inline auto getType() const
        -> std::function<xpress::ColumnType(K1, K2, K3, K4, K5)> const &;

    inline auto getFilter() const
        -> std::function<bool(K1, K2, K3, K4, K5)> const &;

    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              typename Iter4, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1), xpress_isiterable(Iter2),
              xpress_isiterable(Iter3), xpress_isiterable(Iter4)>
    inline Map5Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                       Iter2 const &iterable3, Iter3 const &iterable4,
                       Iter4 const &iterable5);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLB(Func0 newLb) -> Map5Builder<K1, K2, K3, K4, K5> &;

    inline auto withLB(double newLb) -> Map5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withUB(Func0 newUb) -> Map5Builder<K1, K2, K3, K4, K5> &;

    inline auto withUB(double newUb) -> Map5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLimit(Func0 newLimit) -> Map5Builder<K1, K2, K3, K4, K5> &;

    inline auto withLimit(double newLimit) -> Map5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4, K5)>
    inline auto withName(Func0 newName) -> Map5Builder<K1, K2, K3, K4, K5> &;

    inline auto withName(std::optional<std::string> newName)
        -> Map5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4, K5)>
    inline auto withType(Func0 newType) -> Map5Builder<K1, K2, K3, K4, K5> &;

    inline auto withType(ColumnType newType)
        -> Map5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4, K5)>
    inline auto withFilter(Func0 newFilter)
        -> Map5Builder<K1, K2, K3, K4, K5> &;

    inline auto build(XPRSProblem *prob)
        -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int>;

    inline auto build(xpress::objects::XpressProblem *prob)
        -> xpress::maps::HashMap5<K1, K2, K3, K4, K5,
                                  xpress::objects::Variable>;
    inline Map5Builder();
  };

  template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
            typename Iter4, typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            typename K4 = typename xpress_elementtype(Iter3),
            typename K5 = typename xpress_elementtype(Iter4),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2), xpress_isiterable(Iter3),
            xpress_isiterable(Iter4)>
  static inline auto map5(Iter0 const &iterable1, Iter1 const &iterable2,
                          Iter2 const &iterable3, Iter3 const &iterable4,
                          Iter4 const &iterable5)
      -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>;

  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  static inline auto map5(xpress::SizedArray<K1 const> const &arr1,
                          xpress::SizedArray<K2 const> const &arr2,
                          xpress::SizedArray<K3 const> const &arr3,
                          xpress::SizedArray<K4 const> const &arr4,
                          xpress::SizedArray<K5 const> const &arr5)
      -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>;

public:
  template <typename C, xpress_isintegral(C)>
  class VariableArrayBuilderImplementation;

  template <typename C, xpress_isintegral(C)> class VariableArrayBuilder final {
  protected:
    std::shared_ptr<VariableArrayBuilderImplementation<C, CIsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(VariableArrayBuilder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        VariableArrayBuilderImplementation<C, CIsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableArrayBuilder(
        std::shared_ptr<VariableArrayBuilderImplementation<C, CIsIntegral>>
            init);

  public:
    inline VariableArrayBuilder(
        VariableArrayBuilderImplementation<C, CIsIntegral> const *init);
    inline virtual ~VariableArrayBuilder(){};

  private:
    inline VariableArrayBuilderImplementation<C, CIsIntegral> *impl();
    inline VariableArrayBuilderImplementation<C, CIsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableArrayBuilder const &other) const {
      return impl() == other.impl();
    }
    inline VariableArrayBuilder(C dim, xpress::objects::XpressProblem *prob);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLB(Func0 newLb) -> VariableArrayBuilder<C, CIsIntegral> &;

    inline auto withLB(double newLb) -> VariableArrayBuilder<C, CIsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withUB(Func0 newUb) -> VariableArrayBuilder<C, CIsIntegral> &;

    inline auto withUB(double newUb) -> VariableArrayBuilder<C, CIsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArrayBuilder<C, CIsIntegral> &;

    inline auto withLimit(double newLimit)
        -> VariableArrayBuilder<C, CIsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C)>
    inline auto withName(Func0 newName)
        -> VariableArrayBuilder<C, CIsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableArrayBuilder<C, CIsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C)>
    inline auto withType(Func0 newType)
        -> VariableArrayBuilder<C, CIsIntegral> &;

    inline auto withType(ColumnType newType)
        -> VariableArrayBuilder<C, CIsIntegral> &;

    inline auto toArray() -> std::vector<xpress::objects::Variable>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toArray(Func0 makeResult,
            std::function<void(I, C, xpress::objects::Variable)> addResult)
        -> I;
    inline VariableArrayBuilder();
  };

public:
  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  class VariableArray2BuilderImplementation;

  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  class VariableArray2Builder final {
  protected:
    std::shared_ptr<
        VariableArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(VariableArray2Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(VariableArray2BuilderImplementation<
                    C1, C2, C1IsIntegral, C2IsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableArray2Builder(
        std::shared_ptr<VariableArray2BuilderImplementation<
            C1, C2, C1IsIntegral, C2IsIntegral>>
            init);

  public:
    inline VariableArray2Builder(
        VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                            C2IsIntegral> const *init);
    inline virtual ~VariableArray2Builder(){};

  private:
    inline VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> *
    impl();
    inline VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableArray2Builder const &other) const {
      return impl() == other.impl();
    }
    inline VariableArray2Builder(C1 dim1, C2 dim2,
                                 xpress::objects::XpressProblem *prob);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLB(Func0 newLb)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withLB(double newLb)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withUB(Func0 newUb)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withUB(double newUb)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2)>
    inline auto withName(Func0 newName)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2)>
    inline auto withType(Func0 newType)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral> &;

    inline auto toArray()
        -> std::vector<std::vector<xpress::objects::Variable>>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, xpress::objects::Variable)> addResult)
        -> I;
    inline VariableArray2Builder();
  };

public:
  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  class VariableArray3BuilderImplementation;

  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  class VariableArray3Builder final {
  protected:
    std::shared_ptr<VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(VariableArray3Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(VariableArray3BuilderImplementation<
                    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const
                        *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableArray3Builder(
        std::shared_ptr<VariableArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>
            init);

  public:
    inline VariableArray3Builder(
        VariableArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const *init);
    inline virtual ~VariableArray3Builder(){};

  private:
    inline VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> *
    impl();
    inline VariableArray3BuilderImplementation<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableArray3Builder const &other) const {
      return impl() == other.impl();
    }
    inline VariableArray3Builder(C1 dim1, C2 dim2, C3 dim3,
                                 xpress::objects::XpressProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLB(Func0 newLb)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    inline auto withLB(double newLb)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withUB(Func0 newUb)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    inline auto withUB(double newUb)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3)>
    inline auto withName(Func0 newName)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2, C3)>
    inline auto withType(Func0 newType)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral> &;

    inline auto toArray()
        -> std::vector<std::vector<std::vector<xpress::objects::Variable>>>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(
        Func0 makeResult,
        std::function<void(I, C1, C2, C3, xpress::objects::Variable)> addResult)
        -> I;
    inline VariableArray3Builder();
  };

public:
  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  class VariableArray4BuilderImplementation;

  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  class VariableArray4Builder final {
  protected:
    std::shared_ptr<VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(VariableArray4Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(VariableArray4BuilderImplementation<
                    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
                    C4IsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableArray4Builder(
        std::shared_ptr<VariableArray4BuilderImplementation<
            C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral>>
            init);

  public:
    inline VariableArray4Builder(VariableArray4BuilderImplementation<
                                 C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> const *init);
    inline virtual ~VariableArray4Builder(){};

  private:
    inline VariableArray4BuilderImplementation<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        *impl();
    inline VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableArray4Builder const &other) const {
      return impl() == other.impl();
    }
    inline VariableArray4Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                 xpress::objects::XpressProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLB(Func0 newLb)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    inline auto withLB(double newLb)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withUB(Func0 newUb)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    inline auto withUB(double newUb)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4)>
    inline auto withName(Func0 newName)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4)>
    inline auto withType(Func0 newType)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral> &;

    inline auto toArray() -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, C3, C4, xpress::objects::Variable)>
                addResult) -> I;
    inline VariableArray4Builder();
  };

public:
  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  class VariableArray5BuilderImplementation;

  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  class VariableArray5Builder final {
  protected:
    std::shared_ptr<VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral>>
        implPointer = nullptr;

  public:
    bool isSameImpl(VariableArray5Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool
    isSameImpl(VariableArray5BuilderImplementation<
               C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
               C4IsIntegral, C5IsIntegral> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableArray5Builder(
        std::shared_ptr<VariableArray5BuilderImplementation<
            C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral, C5IsIntegral>>
            init);

  public:
    inline VariableArray5Builder(
        VariableArray5BuilderImplementation<
            C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral, C5IsIntegral> const *init);
    inline virtual ~VariableArray5Builder(){};

  private:
    inline VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> *
    impl();
    inline VariableArray5BuilderImplementation<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableArray5Builder const &other) const {
      return impl() == other.impl();
    }
    inline VariableArray5Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4, C5 dim5,
                                 xpress::objects::XpressProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLB(Func0 newLb)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withLB(double newLb)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withUB(Func0 newUb)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withUB(double newUb)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withLimit(double newLimit)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4, C5)>
    inline auto withName(Func0 newName)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4, C5)>
    inline auto withType(Func0 newType)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto withType(ColumnType newType)
        -> VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                                 C3IsIntegral, C4IsIntegral, C5IsIntegral> &;

    inline auto toArray() -> std::vector<std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(
        Func0 makeResult,
        std::function<void(I, C1, C2, C3, C4, C5, xpress::objects::Variable)>
            addResult) -> I;
    inline VariableArray5Builder();
  };

public:
  template <typename K1> class ColumnMapBuilderImplementation;

  template <typename K1> class ColumnMapBuilder {
  protected:
    std::shared_ptr<ColumnMapBuilderImplementation<K1>> implPointer = nullptr;

  public:
    bool isSameImpl(ColumnMapBuilder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(ColumnMapBuilderImplementation<K1> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnMapBuilder(
        std::shared_ptr<ColumnMapBuilderImplementation<K1>> init);

  public:
    inline ColumnMapBuilder(ColumnMapBuilderImplementation<K1> const *init);
    inline virtual ~ColumnMapBuilder(){};

  private:
    inline ColumnMapBuilderImplementation<K1> *impl();
    inline ColumnMapBuilderImplementation<K1> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnMapBuilder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, xpress_isiterable(Iter0)>
    inline ColumnMapBuilder(Iter0 const &iterable1, xpress::XPRSProblem *prob);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLB(Func0 newLb) -> ColumnMapBuilder<K1> &;

    inline auto withLB(double newLb) -> ColumnMapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withUB(Func0 newUb) -> ColumnMapBuilder<K1> &;

    inline auto withUB(double newUb) -> ColumnMapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLimit(Func0 newLimit) -> ColumnMapBuilder<K1> &;

    inline auto withLimit(double newLimit) -> ColumnMapBuilder<K1> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1)>
    inline auto withName(Func0 newName) -> ColumnMapBuilder<K1> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnMapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, ColumnType, Func0, K1)>
    inline auto withType(Func0 newType) -> ColumnMapBuilder<K1> &;

    inline auto withType(ColumnType newType) -> ColumnMapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1)>
    inline auto withFilter(Func0 newFilter) -> ColumnMapBuilder<K1> &;

    inline auto toMap() -> std::unordered_map<K1, int>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(Func0 makeResult,
                      std::function<void(I &, K1, int)> addResult) -> I;
    inline ColumnMapBuilder();
  };

public:
  template <typename K1, typename K2> class ColumnMap2BuilderImplementation;

  template <typename K1, typename K2> class ColumnMap2Builder {
  protected:
    std::shared_ptr<ColumnMap2BuilderImplementation<K1, K2>> implPointer =
        nullptr;

  public:
    bool isSameImpl(ColumnMap2Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool
    isSameImpl(ColumnMap2BuilderImplementation<K1, K2> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnMap2Builder(
        std::shared_ptr<ColumnMap2BuilderImplementation<K1, K2>> init);

  public:
    inline ColumnMap2Builder(
        ColumnMap2BuilderImplementation<K1, K2> const *init);
    inline virtual ~ColumnMap2Builder(){};

  private:
    inline ColumnMap2BuilderImplementation<K1, K2> *impl();
    inline ColumnMap2BuilderImplementation<K1, K2> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnMap2Builder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, typename Iter1, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1)>
    inline ColumnMap2Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                             xpress::XPRSProblem *prob);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLB(Func0 newLb) -> ColumnMap2Builder<K1, K2> &;

    inline auto withLB(double newLb) -> ColumnMap2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withUB(Func0 newUb) -> ColumnMap2Builder<K1, K2> &;

    inline auto withUB(double newUb) -> ColumnMap2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLimit(Func0 newLimit) -> ColumnMap2Builder<K1, K2> &;

    inline auto withLimit(double newLimit) -> ColumnMap2Builder<K1, K2> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2)>
    inline auto withName(Func0 newName) -> ColumnMap2Builder<K1, K2> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnMap2Builder<K1, K2> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2)>
    inline auto withType(Func0 newType) -> ColumnMap2Builder<K1, K2> &;

    inline auto withType(ColumnType newType) -> ColumnMap2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1, K2)>
    inline auto withFilter(Func0 newFilter) -> ColumnMap2Builder<K1, K2> &;

    inline auto toMap() -> xpress::maps::HashMap2<K1, K2, int>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(Func0 makeResult,
                      std::function<void(I &, K1, K2, int)> addResult) -> I;
    inline ColumnMap2Builder();
  };

public:
  template <typename K1, typename K2, typename K3>
  class ColumnMap3BuilderImplementation;

  template <typename K1, typename K2, typename K3> class ColumnMap3Builder {
  protected:
    std::shared_ptr<ColumnMap3BuilderImplementation<K1, K2, K3>> implPointer =
        nullptr;

  public:
    bool isSameImpl(ColumnMap3Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool
    isSameImpl(ColumnMap3BuilderImplementation<K1, K2, K3> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnMap3Builder(
        std::shared_ptr<ColumnMap3BuilderImplementation<K1, K2, K3>> init);

  public:
    inline ColumnMap3Builder(
        ColumnMap3BuilderImplementation<K1, K2, K3> const *init);
    inline virtual ~ColumnMap3Builder(){};

  private:
    inline ColumnMap3BuilderImplementation<K1, K2, K3> *impl();
    inline ColumnMap3BuilderImplementation<K1, K2, K3> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnMap3Builder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, typename Iter1, typename Iter2,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2)>
    inline ColumnMap3Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                             Iter2 const &iterable3, xpress::XPRSProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLB(Func0 newLb) -> ColumnMap3Builder<K1, K2, K3> &;

    inline auto withLB(double newLb) -> ColumnMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withUB(Func0 newUb) -> ColumnMap3Builder<K1, K2, K3> &;

    inline auto withUB(double newUb) -> ColumnMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLimit(Func0 newLimit) -> ColumnMap3Builder<K1, K2, K3> &;

    inline auto withLimit(double newLimit) -> ColumnMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3)>
    inline auto withName(Func0 newName) -> ColumnMap3Builder<K1, K2, K3> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3)>
    inline auto withType(Func0 newType) -> ColumnMap3Builder<K1, K2, K3> &;

    inline auto withType(ColumnType newType) -> ColumnMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3)>
    inline auto withFilter(Func0 newFilter) -> ColumnMap3Builder<K1, K2, K3> &;

    inline auto toMap() -> xpress::maps::HashMap3<K1, K2, K3, int>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(Func0 makeResult,
                      std::function<void(I &, K1, K2, K3, int)> addResult) -> I;
    inline ColumnMap3Builder();
  };

public:
  template <typename K1, typename K2, typename K3, typename K4>
  class ColumnMap4BuilderImplementation;

  template <typename K1, typename K2, typename K3, typename K4>
  class ColumnMap4Builder {
  protected:
    std::shared_ptr<ColumnMap4BuilderImplementation<K1, K2, K3, K4>>
        implPointer = nullptr;

  public:
    bool isSameImpl(ColumnMap4Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        ColumnMap4BuilderImplementation<K1, K2, K3, K4> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnMap4Builder(
        std::shared_ptr<ColumnMap4BuilderImplementation<K1, K2, K3, K4>> init);

  public:
    inline ColumnMap4Builder(
        ColumnMap4BuilderImplementation<K1, K2, K3, K4> const *init);
    inline virtual ~ColumnMap4Builder(){};

  private:
    inline ColumnMap4BuilderImplementation<K1, K2, K3, K4> *impl();
    inline ColumnMap4BuilderImplementation<K1, K2, K3, K4> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnMap4Builder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
    inline ColumnMap4Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                             Iter2 const &iterable3, Iter3 const &iterable4,
                             xpress::XPRSProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLB(Func0 newLb) -> ColumnMap4Builder<K1, K2, K3, K4> &;

    inline auto withLB(double newLb) -> ColumnMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withUB(Func0 newUb) -> ColumnMap4Builder<K1, K2, K3, K4> &;

    inline auto withUB(double newUb) -> ColumnMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnMap4Builder<K1, K2, K3, K4> &;

    inline auto withLimit(double newLimit)
        -> ColumnMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4)>
    inline auto withName(Func0 newName) -> ColumnMap4Builder<K1, K2, K3, K4> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4)>
    inline auto withType(Func0 newType) -> ColumnMap4Builder<K1, K2, K3, K4> &;

    inline auto withType(ColumnType newType)
        -> ColumnMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4)>
    inline auto withFilter(Func0 newFilter)
        -> ColumnMap4Builder<K1, K2, K3, K4> &;

    inline auto toMap() -> xpress::maps::HashMap4<K1, K2, K3, K4, int>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(Func0 makeResult,
                      std::function<void(I &, K1, K2, K3, K4, int)> addResult)
        -> I;
    inline ColumnMap4Builder();
  };

public:
  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class ColumnMap5BuilderImplementation;

  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class ColumnMap5Builder {
  protected:
    std::shared_ptr<ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5>>
        implPointer = nullptr;

  public:
    bool isSameImpl(ColumnMap5Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> const
                        *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline ColumnMap5Builder(
        std::shared_ptr<ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5>>
            init);

  public:
    inline ColumnMap5Builder(
        ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> const *init);
    inline virtual ~ColumnMap5Builder(){};

  private:
    inline ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> *impl();
    inline ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(ColumnMap5Builder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              typename Iter4, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1), xpress_isiterable(Iter2),
              xpress_isiterable(Iter3), xpress_isiterable(Iter4)>
    inline ColumnMap5Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                             Iter2 const &iterable3, Iter3 const &iterable4,
                             Iter4 const &iterable5, xpress::XPRSProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLB(Func0 newLb) -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withLB(double newLb) -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withUB(Func0 newUb) -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withUB(double newUb) -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withLimit(double newLimit)
        -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4, K5)>
    inline auto withName(Func0 newName)
        -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withName(std::optional<std::string> newName)
        -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4, K5)>
    inline auto withType(Func0 newType)
        -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withType(ColumnType newType)
        -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4, K5)>
    inline auto withFilter(Func0 newFilter)
        -> ColumnMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto toMap() -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, K3, K4, K5, int)> addResult) -> I;
    inline ColumnMap5Builder();
  };

public:
  template <typename K1> class VariableMapBuilderImplementation;

  template <typename K1> class VariableMapBuilder {
  protected:
    std::shared_ptr<VariableMapBuilderImplementation<K1>> implPointer = nullptr;

  public:
    bool isSameImpl(VariableMapBuilder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(VariableMapBuilderImplementation<K1> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableMapBuilder(
        std::shared_ptr<VariableMapBuilderImplementation<K1>> init);

  public:
    inline VariableMapBuilder(VariableMapBuilderImplementation<K1> const *init);
    inline virtual ~VariableMapBuilder(){};

  private:
    inline VariableMapBuilderImplementation<K1> *impl();
    inline VariableMapBuilderImplementation<K1> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableMapBuilder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, xpress_isiterable(Iter0)>
    inline VariableMapBuilder(Iter0 const &iterable1,
                              xpress::objects::XpressProblem *prob);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLB(Func0 newLb) -> VariableMapBuilder<K1> &;

    inline auto withLB(double newLb) -> VariableMapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withUB(Func0 newUb) -> VariableMapBuilder<K1> &;

    inline auto withUB(double newUb) -> VariableMapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLimit(Func0 newLimit) -> VariableMapBuilder<K1> &;

    inline auto withLimit(double newLimit) -> VariableMapBuilder<K1> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1)>
    inline auto withName(Func0 newName) -> VariableMapBuilder<K1> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableMapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, ColumnType, Func0, K1)>
    inline auto withType(Func0 newType) -> VariableMapBuilder<K1> &;

    inline auto withType(ColumnType newType) -> VariableMapBuilder<K1> &;

    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1)>
    inline auto withFilter(Func0 newFilter) -> VariableMapBuilder<K1> &;

    inline auto toMap() -> std::unordered_map<K1, xpress::objects::Variable>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, xpress::objects::Variable)> addResult)
        -> I;
    inline VariableMapBuilder();
  };

public:
  template <typename K1, typename K2> class VariableMap2BuilderImplementation;

  template <typename K1, typename K2> class VariableMap2Builder {
  protected:
    std::shared_ptr<VariableMap2BuilderImplementation<K1, K2>> implPointer =
        nullptr;

  public:
    bool isSameImpl(VariableMap2Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool
    isSameImpl(VariableMap2BuilderImplementation<K1, K2> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableMap2Builder(
        std::shared_ptr<VariableMap2BuilderImplementation<K1, K2>> init);

  public:
    inline VariableMap2Builder(
        VariableMap2BuilderImplementation<K1, K2> const *init);
    inline virtual ~VariableMap2Builder(){};

  private:
    inline VariableMap2BuilderImplementation<K1, K2> *impl();
    inline VariableMap2BuilderImplementation<K1, K2> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableMap2Builder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, typename Iter1, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1)>
    inline VariableMap2Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                               xpress::objects::XpressProblem *prob);

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLB(Func0 newLb) -> VariableMap2Builder<K1, K2> &;

    inline auto withLB(double newLb) -> VariableMap2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withUB(Func0 newUb) -> VariableMap2Builder<K1, K2> &;

    inline auto withUB(double newUb) -> VariableMap2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLimit(Func0 newLimit) -> VariableMap2Builder<K1, K2> &;

    inline auto withLimit(double newLimit) -> VariableMap2Builder<K1, K2> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2)>
    inline auto withName(Func0 newName) -> VariableMap2Builder<K1, K2> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableMap2Builder<K1, K2> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2)>
    inline auto withType(Func0 newType) -> VariableMap2Builder<K1, K2> &;

    inline auto withType(ColumnType newType) -> VariableMap2Builder<K1, K2> &;

    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1, K2)>
    inline auto withFilter(Func0 newFilter) -> VariableMap2Builder<K1, K2> &;

    inline auto toMap()
        -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, xpress::objects::Variable)> addResult)
        -> I;
    inline VariableMap2Builder();
  };

public:
  template <typename K1, typename K2, typename K3>
  class VariableMap3BuilderImplementation;

  template <typename K1, typename K2, typename K3> class VariableMap3Builder {
  protected:
    std::shared_ptr<VariableMap3BuilderImplementation<K1, K2, K3>> implPointer =
        nullptr;

  public:
    bool isSameImpl(VariableMap3Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        VariableMap3BuilderImplementation<K1, K2, K3> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableMap3Builder(
        std::shared_ptr<VariableMap3BuilderImplementation<K1, K2, K3>> init);

  public:
    inline VariableMap3Builder(
        VariableMap3BuilderImplementation<K1, K2, K3> const *init);
    inline virtual ~VariableMap3Builder(){};

  private:
    inline VariableMap3BuilderImplementation<K1, K2, K3> *impl();
    inline VariableMap3BuilderImplementation<K1, K2, K3> const *impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableMap3Builder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, typename Iter1, typename Iter2,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2)>
    inline VariableMap3Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                               Iter2 const &iterable3,
                               xpress::objects::XpressProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLB(Func0 newLb) -> VariableMap3Builder<K1, K2, K3> &;

    inline auto withLB(double newLb) -> VariableMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withUB(Func0 newUb) -> VariableMap3Builder<K1, K2, K3> &;

    inline auto withUB(double newUb) -> VariableMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLimit(Func0 newLimit) -> VariableMap3Builder<K1, K2, K3> &;

    inline auto withLimit(double newLimit) -> VariableMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3)>
    inline auto withName(Func0 newName) -> VariableMap3Builder<K1, K2, K3> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3)>
    inline auto withType(Func0 newType) -> VariableMap3Builder<K1, K2, K3> &;

    inline auto withType(ColumnType newType)
        -> VariableMap3Builder<K1, K2, K3> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3)>
    inline auto withFilter(Func0 newFilter)
        -> VariableMap3Builder<K1, K2, K3> &;

    inline auto toMap()
        -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, K3, xpress::objects::Variable)>
              addResult) -> I;
    inline VariableMap3Builder();
  };

public:
  template <typename K1, typename K2, typename K3, typename K4>
  class VariableMap4BuilderImplementation;

  template <typename K1, typename K2, typename K3, typename K4>
  class VariableMap4Builder {
  protected:
    std::shared_ptr<VariableMap4BuilderImplementation<K1, K2, K3, K4>>
        implPointer = nullptr;

  public:
    bool isSameImpl(VariableMap4Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(
        VariableMap4BuilderImplementation<K1, K2, K3, K4> const *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableMap4Builder(
        std::shared_ptr<VariableMap4BuilderImplementation<K1, K2, K3, K4>>
            init);

  public:
    inline VariableMap4Builder(
        VariableMap4BuilderImplementation<K1, K2, K3, K4> const *init);
    inline virtual ~VariableMap4Builder(){};

  private:
    inline VariableMap4BuilderImplementation<K1, K2, K3, K4> *impl();
    inline VariableMap4BuilderImplementation<K1, K2, K3, K4> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableMap4Builder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
    inline VariableMap4Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                               Iter2 const &iterable3, Iter3 const &iterable4,
                               xpress::objects::XpressProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLB(Func0 newLb) -> VariableMap4Builder<K1, K2, K3, K4> &;

    inline auto withLB(double newLb) -> VariableMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withUB(Func0 newUb) -> VariableMap4Builder<K1, K2, K3, K4> &;

    inline auto withUB(double newUb) -> VariableMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLimit(Func0 newLimit)
        -> VariableMap4Builder<K1, K2, K3, K4> &;

    inline auto withLimit(double newLimit)
        -> VariableMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4)>
    inline auto withName(Func0 newName)
        -> VariableMap4Builder<K1, K2, K3, K4> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4)>
    inline auto withType(Func0 newType)
        -> VariableMap4Builder<K1, K2, K3, K4> &;

    inline auto withType(ColumnType newType)
        -> VariableMap4Builder<K1, K2, K3, K4> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4)>
    inline auto withFilter(Func0 newFilter)
        -> VariableMap4Builder<K1, K2, K3, K4> &;

    inline auto toMap()
        -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, K3, K4, xpress::objects::Variable)>
              addResult) -> I;
    inline VariableMap4Builder();
  };

public:
  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class VariableMap5BuilderImplementation;

  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class VariableMap5Builder {
  protected:
    std::shared_ptr<VariableMap5BuilderImplementation<K1, K2, K3, K4, K5>>
        implPointer = nullptr;

  public:
    bool isSameImpl(VariableMap5Builder const &other) const {
      return implPointer.get() == other.implPointer.get();
    }
    bool isSameImpl(VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> const
                        *other) const {
      return implPointer.get() == other;
    }

  protected:
    inline VariableMap5Builder(
        std::shared_ptr<VariableMap5BuilderImplementation<K1, K2, K3, K4, K5>>
            init);

  public:
    inline VariableMap5Builder(
        VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> const *init);
    inline virtual ~VariableMap5Builder(){};

  private:
    inline VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> *impl();
    inline VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> const *
    impl() const;

  public:
    template <typename Type> bool isInstance() const {
      return dynamic_cast<Type const *>(impl()) != nullptr;
    }
    explicit operator bool() const { return implPointer != nullptr; }
    bool operator==(std::nullptr_t) const { return implPointer == nullptr; }
    bool operator!=(std::nullptr_t) const { return implPointer != nullptr; }
    bool equals(VariableMap5Builder const &other) const {
      return impl() == other.impl();
    }
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              typename Iter4, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1), xpress_isiterable(Iter2),
              xpress_isiterable(Iter3), xpress_isiterable(Iter4)>
    inline VariableMap5Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                               Iter2 const &iterable3, Iter3 const &iterable4,
                               Iter4 const &iterable5,
                               xpress::objects::XpressProblem *prob);

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLB(Func0 newLb)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withLB(double newLb)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withUB(Func0 newUb)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withUB(double newUb)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLimit(Func0 newLimit)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withLimit(double newLimit)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4, K5)>
    inline auto withName(Func0 newName)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withName(std::optional<std::string> newName)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4, K5)>
    inline auto withType(Func0 newType)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto withType(ColumnType newType)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4, K5)>
    inline auto withFilter(Func0 newFilter)
        -> VariableMap5Builder<K1, K2, K3, K4, K5> &;

    inline auto toMap() -> xpress::maps::HashMap5<K1, K2, K3, K4, K5,
                                                  xpress::objects::Variable>;

    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(
        Func0 makeResult,
        std::function<void(I &, K1, K2, K3, K4, K5, xpress::objects::Variable)>
            addResult) -> I;
    inline VariableMap5Builder();
  };

public:
  template <typename C, typename CIsIntegral>
  class ArrayBuilderImplementation final
      : public std::enable_shared_from_this<
            ArrayBuilderImplementation<C, CIsIntegral>> {
    friend class ArrayBuilder<C, CIsIntegral>;

  public:
    virtual ~ArrayBuilderImplementation() {}

  private:
    std::vector<int> const dims;

  public:
    std::function<double(C)> lb;

    std::function<double(C)> ub;

    std::function<double(C)> limit;

    std::function<std::optional<std::string>(C)> name;

    std::function<xpress::ColumnType(C)> type;
    inline ArrayBuilderImplementation(C dim);
    inline auto getDim(int dim) const -> int;
    inline auto getLB() const -> std::function<double(C)> const &;
    inline auto getUB() const -> std::function<double(C)> const &;
    inline auto getLimit() const -> std::function<double(C)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(C)> const &;
    inline auto getType() const -> std::function<xpress::ColumnType(C)> const &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLB(Func0 newLb)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withLB(double newLb)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withUB(Func0 newUb)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withUB(double newUb)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLimit(Func0 newLimit)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withLimit(double newLimit)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C)>
    inline auto withName(Func0 newName)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C)>
    inline auto withType(Func0 newType)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withType(ColumnType newType)
        -> ArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto build(XPRSProblem *prob) -> std::vector<int>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::vector<xpress::objects::Variable>;
  };

public:
  template <typename C1, typename C2, typename C1IsIntegral,
            typename C2IsIntegral>
  class Array2BuilderImplementation final
      : public std::enable_shared_from_this<
            Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>> {
    friend class Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral>;

  public:
    virtual ~Array2BuilderImplementation() {}

  private:
    std::vector<int> const dims;

  public:
    std::function<double(C1, C2)> lb;

    std::function<double(C1, C2)> ub;

    std::function<double(C1, C2)> limit;

    std::function<std::optional<std::string>(C1, C2)> name;

    std::function<xpress::ColumnType(C1, C2)> type;
    inline Array2BuilderImplementation(C1 dim1, C2 dim2);
    inline auto getDim(int dim) const -> int;
    inline auto getLB() const -> std::function<double(C1, C2)> const &;
    inline auto getUB() const -> std::function<double(C1, C2)> const &;
    inline auto getLimit() const -> std::function<double(C1, C2)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(C1, C2)> const &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(C1, C2)> const &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLB(Func0 newLb)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    inline auto withLB(double newLb)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withUB(Func0 newUb)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    inline auto withUB(double newUb)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLimit(Func0 newLimit)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2)>
    inline auto withName(Func0 newName)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2)>
    inline auto withType(Func0 newType)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> &;
    inline auto build(XPRSProblem *prob) -> std::vector<std::vector<int>>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::vector<std::vector<xpress::objects::Variable>>;
  };

public:
  template <typename C1, typename C2, typename C3, typename C1IsIntegral,
            typename C2IsIntegral, typename C3IsIntegral>
  class Array3BuilderImplementation final
      : public std::enable_shared_from_this<Array3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>> {
    friend class Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral>;

  public:
    virtual ~Array3BuilderImplementation() {}

  private:
    std::vector<int> const dims;

  public:
    std::function<double(C1, C2, C3)> lb;

    std::function<double(C1, C2, C3)> ub;

    std::function<double(C1, C2, C3)> limit;

    std::function<std::optional<std::string>(C1, C2, C3)> name;

    std::function<xpress::ColumnType(C1, C2, C3)> type;
    inline Array3BuilderImplementation(C1 dim1, C2 dim2, C3 dim3);
    inline auto getDim(int dim) const -> int;
    inline auto getLB() const -> std::function<double(C1, C2, C3)> const &;
    inline auto getUB() const -> std::function<double(C1, C2, C3)> const &;
    inline auto getLimit() const -> std::function<double(C1, C2, C3)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(C1, C2, C3)> const &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(C1, C2, C3)> const &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLB(Func0 newLb)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    inline auto withLB(double newLb)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withUB(Func0 newUb)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    inline auto withUB(double newUb)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLimit(Func0 newLimit)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3)>
    inline auto withName(Func0 newName)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2, C3)>
    inline auto withType(Func0 newType)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral> &;
    inline auto build(XPRSProblem *prob)
        -> std::vector<std::vector<std::vector<int>>>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::vector<std::vector<std::vector<xpress::objects::Variable>>>;
  };

public:
  template <typename C1, typename C2, typename C3, typename C4,
            typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
            typename C4IsIntegral>
  class Array4BuilderImplementation final
      : public std::enable_shared_from_this<Array4BuilderImplementation<
            C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral>> {
    friend class Array4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral>;

  public:
    virtual ~Array4BuilderImplementation() {}

  private:
    std::vector<int> const dims;

  public:
    std::function<double(C1, C2, C3, C4)> lb;

    std::function<double(C1, C2, C3, C4)> ub;

    std::function<double(C1, C2, C3, C4)> limit;

    std::function<std::optional<std::string>(C1, C2, C3, C4)> name;

    std::function<xpress::ColumnType(C1, C2, C3, C4)> type;
    inline Array4BuilderImplementation(C1 dim1, C2 dim2, C3 dim3, C4 dim4);
    inline auto getDim(int dim) const -> int;
    inline auto getLB() const -> std::function<double(C1, C2, C3, C4)> const &;
    inline auto getUB() const -> std::function<double(C1, C2, C3, C4)> const &;
    inline auto getLimit() const
        -> std::function<double(C1, C2, C3, C4)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(C1, C2, C3, C4)> const &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(C1, C2, C3, C4)> const &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLB(Func0 newLb)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    inline auto withLB(double newLb)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withUB(Func0 newUb)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    inline auto withUB(double newUb)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLimit(Func0 newLimit)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    inline auto withLimit(double newLimit)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4)>
    inline auto withName(Func0 newName)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    inline auto withName(std::optional<std::string> newName)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4)>
    inline auto withType(Func0 newType)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    inline auto withType(ColumnType newType)
        -> Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral>
            &;
    inline auto build(XPRSProblem *prob)
        -> std::vector<std::vector<std::vector<std::vector<int>>>>;
    inline auto build(xpress::objects::XpressProblem *prob) -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>;
  };

public:
  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
            typename C4IsIntegral, typename C5IsIntegral>
  class Array5BuilderImplementation final
      : public std::enable_shared_from_this<Array5BuilderImplementation<
            C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral, C5IsIntegral>> {
    friend class Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                               C3IsIntegral, C4IsIntegral, C5IsIntegral>;

  public:
    virtual ~Array5BuilderImplementation() {}

  private:
    std::vector<int> const dims;

  public:
    std::function<double(C1, C2, C3, C4, C5)> lb;

    std::function<double(C1, C2, C3, C4, C5)> ub;

    std::function<double(C1, C2, C3, C4, C5)> limit;

    std::function<std::optional<std::string>(C1, C2, C3, C4, C5)> name;

    std::function<xpress::ColumnType(C1, C2, C3, C4, C5)> type;
    inline Array5BuilderImplementation(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                       C5 dim5);
    inline auto getDim(int dim) const -> int;
    inline auto getLB() const
        -> std::function<double(C1, C2, C3, C4, C5)> const &;
    inline auto getUB() const
        -> std::function<double(C1, C2, C3, C4, C5)> const &;
    inline auto getLimit() const
        -> std::function<double(C1, C2, C3, C4, C5)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(C1, C2, C3, C4, C5)> const
            &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(C1, C2, C3, C4, C5)> const &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLB(Func0 newLb)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    inline auto withLB(double newLb)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withUB(Func0 newUb)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    inline auto withUB(double newUb)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLimit(Func0 newLimit)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4, C5)>
    inline auto withName(Func0 newName)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4, C5)>
    inline auto withType(Func0 newType)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> Array5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral> &;
    inline auto build(XPRSProblem *prob)
        -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::vector<std::vector<
            std::vector<std::vector<std::vector<xpress::objects::Variable>>>>>;
  };

public:
  template <typename C, typename CIsIntegral>
  class ColumnArrayBuilderImplementation final
      : public std::enable_shared_from_this<
            ColumnArrayBuilderImplementation<C, CIsIntegral>> {
    friend class ColumnArrayBuilder<C, CIsIntegral>;

  public:
    virtual ~ColumnArrayBuilderImplementation() {}

  private:
    ArrayBuilder<C> wrapped;

    xpress::XPRSProblem *prob;

  public:
    inline ColumnArrayBuilderImplementation(C dim, xpress::XPRSProblem *prob);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLB(Func0 newLb)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withLB(double newLb)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withUB(Func0 newUb)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withUB(double newUb)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withLimit(double newLimit)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C)>
    inline auto withName(Func0 newName)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C)>
    inline auto withType(Func0 newType)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withType(ColumnType newType)
        -> ColumnArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto toArray() -> std::vector<int>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(Func0 makeResult,
                        std::function<void(I, C, int)> addResult) -> I;
  };

public:
  template <typename C1, typename C2, typename C1IsIntegral,
            typename C2IsIntegral>
  class ColumnArray2BuilderImplementation final
      : public std::enable_shared_from_this<ColumnArray2BuilderImplementation<
            C1, C2, C1IsIntegral, C2IsIntegral>> {
    friend class ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral>;

  public:
    virtual ~ColumnArray2BuilderImplementation() {}

  private:
    Array2Builder<C1, C2> wrapped;

    xpress::XPRSProblem *prob;

  public:
    inline ColumnArray2BuilderImplementation(C1 dim1, C2 dim2,
                                             xpress::XPRSProblem *prob);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLB(Func0 newLb)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    inline auto withLB(double newLb)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withUB(Func0 newUb)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    inline auto withUB(double newUb)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    inline auto withLimit(double newLimit)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2)>
    inline auto withName(Func0 newName)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2)>
    inline auto withType(Func0 newType)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    inline auto withType(ColumnType newType)
        -> ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>
            &;
    inline auto toArray() -> std::vector<std::vector<int>>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(Func0 makeResult,
                        std::function<void(I, C1, C2, int)> addResult) -> I;
  };

public:
  template <typename C1, typename C2, typename C3, typename C1IsIntegral,
            typename C2IsIntegral, typename C3IsIntegral>
  class ColumnArray3BuilderImplementation final
      : public std::enable_shared_from_this<ColumnArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>> {
    friend class ColumnArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                     C3IsIntegral>;

  public:
    virtual ~ColumnArray3BuilderImplementation() {}

  private:
    Array3Builder<C1, C2, C3> wrapped;

    xpress::XPRSProblem *prob;

  public:
    inline ColumnArray3BuilderImplementation(C1 dim1, C2 dim2, C3 dim3,
                                             xpress::XPRSProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLB(Func0 newLb)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    inline auto withLB(double newLb)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withUB(Func0 newUb)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    inline auto withUB(double newUb)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3)>
    inline auto withName(Func0 newName)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2, C3)>
    inline auto withType(Func0 newType)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> ColumnArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral> &;
    inline auto toArray() -> std::vector<std::vector<std::vector<int>>>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(Func0 makeResult,
                        std::function<void(I, C1, C2, C3, int)> addResult) -> I;
  };

public:
  template <typename C1, typename C2, typename C3, typename C4,
            typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
            typename C4IsIntegral>
  class ColumnArray4BuilderImplementation final
      : public std::enable_shared_from_this<ColumnArray4BuilderImplementation<
            C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral>> {
    friend class ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                     C3IsIntegral, C4IsIntegral>;

  public:
    virtual ~ColumnArray4BuilderImplementation() {}

  private:
    Array4Builder<C1, C2, C3, C4> wrapped;

    xpress::XPRSProblem *prob;

  public:
    inline ColumnArray4BuilderImplementation(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                             xpress::XPRSProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLB(Func0 newLb)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    inline auto withLB(double newLb)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withUB(Func0 newUb)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    inline auto withUB(double newUb)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4)>
    inline auto withName(Func0 newName)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4)>
    inline auto withType(Func0 newType)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> ColumnArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral> &;
    inline auto toArray()
        -> std::vector<std::vector<std::vector<std::vector<int>>>>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(Func0 makeResult,
                        std::function<void(I, C1, C2, C3, C4, int)> addResult)
        -> I;
  };

public:
  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
            typename C4IsIntegral, typename C5IsIntegral>
  class ColumnArray5BuilderImplementation final
      : public std::enable_shared_from_this<ColumnArray5BuilderImplementation<
            C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral, C5IsIntegral>> {
    friend class ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                     C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                     C5IsIntegral>;

  public:
    virtual ~ColumnArray5BuilderImplementation() {}

  private:
    Array5Builder<C1, C2, C3, C4, C5> wrapped;

    xpress::XPRSProblem *prob;

  public:
    inline ColumnArray5BuilderImplementation(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                             C5 dim5,
                                             xpress::XPRSProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLB(Func0 newLb)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    inline auto withLB(double newLb)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withUB(Func0 newUb)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    inline auto withUB(double newUb)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4, C5)>
    inline auto withName(Func0 newName)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4, C5)>
    inline auto withType(Func0 newType)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> ColumnArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral> &;
    inline auto toArray()
        -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, C3, C4, C5, int)> addResult) -> I;
  };

public:
  template <typename K1>
  class MapBuilderImplementation
      : public std::enable_shared_from_this<MapBuilderImplementation<K1>> {
    friend class MapBuilder<K1>;

  public:
    virtual ~MapBuilderImplementation() {}

  private:
    xpress::Iterable<K1> iterable1;

  public:
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

  private:
    std::function<double(K1)> lb;

    std::function<double(K1)> ub;

    std::function<double(K1)> limit;

    std::function<std::optional<std::string>(K1)> name;

    std::function<xpress::ColumnType(K1)> type;

    std::function<bool(K1)> filter;

  public:
    inline auto getLB() const -> std::function<double(K1)> const &;
    inline auto getUB() const -> std::function<double(K1)> const &;
    inline auto getLimit() const -> std::function<double(K1)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(K1)> const &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(K1)> const &;
    inline auto getFilter() const -> std::function<bool(K1)> const &;
    template <typename Iter0, xpress_isiterable(Iter0)>
    inline MapBuilderImplementation(Iter0 const &iterable1);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLB(Func0 newLb) -> MapBuilderImplementation<K1> &;
    inline auto withLB(double newLb) -> MapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withUB(Func0 newUb) -> MapBuilderImplementation<K1> &;
    inline auto withUB(double newUb) -> MapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLimit(Func0 newLimit) -> MapBuilderImplementation<K1> &;
    inline auto withLimit(double newLimit) -> MapBuilderImplementation<K1> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1)>
    inline auto withName(Func0 newName) -> MapBuilderImplementation<K1> &;
    inline auto withName(std::optional<std::string> newName)
        -> MapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, ColumnType, Func0, K1)>
    inline auto withType(Func0 newType) -> MapBuilderImplementation<K1> &;
    inline auto withType(ColumnType newType) -> MapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1)>
    inline auto withFilter(Func0 newFilter) -> MapBuilderImplementation<K1> &;
    inline auto build(XPRSProblem *prob) -> std::unordered_map<K1, int>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> std::unordered_map<K1, xpress::objects::Variable>;
  };

public:
  template <typename K1, typename K2>
  class Map2BuilderImplementation
      : public std::enable_shared_from_this<Map2BuilderImplementation<K1, K2>> {
    friend class Map2Builder<K1, K2>;

  public:
    virtual ~Map2BuilderImplementation() {}

  private:
    xpress::Iterable<K1> iterable1;

  public:
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

  private:
    xpress::Iterable<K2> iterable2;

  public:
    inline auto getIterable2() const -> xpress::Iterable<K2> const &;

  private:
    std::function<double(K1, K2)> lb;

    std::function<double(K1, K2)> ub;

    std::function<double(K1, K2)> limit;

    std::function<std::optional<std::string>(K1, K2)> name;

    std::function<xpress::ColumnType(K1, K2)> type;

    std::function<bool(K1, K2)> filter;

  public:
    inline auto getLB() const -> std::function<double(K1, K2)> const &;
    inline auto getUB() const -> std::function<double(K1, K2)> const &;
    inline auto getLimit() const -> std::function<double(K1, K2)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(K1, K2)> const &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(K1, K2)> const &;
    inline auto getFilter() const -> std::function<bool(K1, K2)> const &;
    template <typename Iter0, typename Iter1, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1)>
    inline Map2BuilderImplementation(Iter0 const &iterable1,
                                     Iter1 const &iterable2);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLB(Func0 newLb) -> Map2BuilderImplementation<K1, K2> &;
    inline auto withLB(double newLb) -> Map2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withUB(Func0 newUb) -> Map2BuilderImplementation<K1, K2> &;
    inline auto withUB(double newUb) -> Map2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLimit(Func0 newLimit)
        -> Map2BuilderImplementation<K1, K2> &;
    inline auto withLimit(double newLimit)
        -> Map2BuilderImplementation<K1, K2> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2)>
    inline auto withName(Func0 newName) -> Map2BuilderImplementation<K1, K2> &;
    inline auto withName(std::optional<std::string> newName)
        -> Map2BuilderImplementation<K1, K2> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2)>
    inline auto withType(Func0 newType) -> Map2BuilderImplementation<K1, K2> &;
    inline auto withType(ColumnType newType)
        -> Map2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1, K2)>
    inline auto withFilter(Func0 newFilter)
        -> Map2BuilderImplementation<K1, K2> &;
    inline auto build(XPRSProblem *prob) -> xpress::maps::HashMap2<K1, K2, int>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable>;
  };

public:
  template <typename K1, typename K2, typename K3>
  class Map3BuilderImplementation : public std::enable_shared_from_this<
                                        Map3BuilderImplementation<K1, K2, K3>> {
    friend class Map3Builder<K1, K2, K3>;

  public:
    virtual ~Map3BuilderImplementation() {}

  private:
    xpress::Iterable<K1> iterable1;

  public:
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

  private:
    xpress::Iterable<K2> iterable2;

  public:
    inline auto getIterable2() const -> xpress::Iterable<K2> const &;

  private:
    xpress::Iterable<K3> iterable3;

  public:
    inline auto getIterable3() const -> xpress::Iterable<K3> const &;

  private:
    std::function<double(K1, K2, K3)> lb;

    std::function<double(K1, K2, K3)> ub;

    std::function<double(K1, K2, K3)> limit;

    std::function<std::optional<std::string>(K1, K2, K3)> name;

    std::function<xpress::ColumnType(K1, K2, K3)> type;

    std::function<bool(K1, K2, K3)> filter;

  public:
    inline auto getLB() const -> std::function<double(K1, K2, K3)> const &;
    inline auto getUB() const -> std::function<double(K1, K2, K3)> const &;
    inline auto getLimit() const -> std::function<double(K1, K2, K3)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(K1, K2, K3)> const &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(K1, K2, K3)> const &;
    inline auto getFilter() const -> std::function<bool(K1, K2, K3)> const &;
    template <typename Iter0, typename Iter1, typename Iter2,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2)>
    inline Map3BuilderImplementation(Iter0 const &iterable1,
                                     Iter1 const &iterable2,
                                     Iter2 const &iterable3);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLB(Func0 newLb) -> Map3BuilderImplementation<K1, K2, K3> &;
    inline auto withLB(double newLb) -> Map3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withUB(Func0 newUb) -> Map3BuilderImplementation<K1, K2, K3> &;
    inline auto withUB(double newUb) -> Map3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLimit(Func0 newLimit)
        -> Map3BuilderImplementation<K1, K2, K3> &;
    inline auto withLimit(double newLimit)
        -> Map3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3)>
    inline auto withName(Func0 newName)
        -> Map3BuilderImplementation<K1, K2, K3> &;
    inline auto withName(std::optional<std::string> newName)
        -> Map3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3)>
    inline auto withType(Func0 newType)
        -> Map3BuilderImplementation<K1, K2, K3> &;
    inline auto withType(ColumnType newType)
        -> Map3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3)>
    inline auto withFilter(Func0 newFilter)
        -> Map3BuilderImplementation<K1, K2, K3> &;
    inline auto build(XPRSProblem *prob)
        -> xpress::maps::HashMap3<K1, K2, K3, int>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable>;
  };

public:
  template <typename K1, typename K2, typename K3, typename K4>
  class Map4BuilderImplementation
      : public std::enable_shared_from_this<
            Map4BuilderImplementation<K1, K2, K3, K4>> {
    friend class Map4Builder<K1, K2, K3, K4>;

  public:
    virtual ~Map4BuilderImplementation() {}

  private:
    xpress::Iterable<K1> iterable1;

  public:
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

  private:
    xpress::Iterable<K2> iterable2;

  public:
    inline auto getIterable2() const -> xpress::Iterable<K2> const &;

  private:
    xpress::Iterable<K3> iterable3;

  public:
    inline auto getIterable3() const -> xpress::Iterable<K3> const &;

  private:
    xpress::Iterable<K4> iterable4;

  public:
    inline auto getIterable4() const -> xpress::Iterable<K4> const &;

  private:
    std::function<double(K1, K2, K3, K4)> lb;

    std::function<double(K1, K2, K3, K4)> ub;

    std::function<double(K1, K2, K3, K4)> limit;

    std::function<std::optional<std::string>(K1, K2, K3, K4)> name;

    std::function<xpress::ColumnType(K1, K2, K3, K4)> type;

    std::function<bool(K1, K2, K3, K4)> filter;

  public:
    inline auto getLB() const -> std::function<double(K1, K2, K3, K4)> const &;
    inline auto getUB() const -> std::function<double(K1, K2, K3, K4)> const &;
    inline auto getLimit() const
        -> std::function<double(K1, K2, K3, K4)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(K1, K2, K3, K4)> const &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(K1, K2, K3, K4)> const &;
    inline auto getFilter() const
        -> std::function<bool(K1, K2, K3, K4)> const &;
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
    inline Map4BuilderImplementation(Iter0 const &iterable1,
                                     Iter1 const &iterable2,
                                     Iter2 const &iterable3,
                                     Iter3 const &iterable4);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLB(Func0 newLb)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withLB(double newLb)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withUB(Func0 newUb)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withUB(double newUb)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLimit(Func0 newLimit)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withLimit(double newLimit)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4)>
    inline auto withName(Func0 newName)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withName(std::optional<std::string> newName)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4)>
    inline auto withType(Func0 newType)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withType(ColumnType newType)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4)>
    inline auto withFilter(Func0 newFilter)
        -> Map4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto build(XPRSProblem *prob)
        -> xpress::maps::HashMap4<K1, K2, K3, K4, int>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable>;
  };

public:
  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class Map5BuilderImplementation
      : public std::enable_shared_from_this<
            Map5BuilderImplementation<K1, K2, K3, K4, K5>> {
    friend class Map5Builder<K1, K2, K3, K4, K5>;

  public:
    virtual ~Map5BuilderImplementation() {}

  private:
    xpress::Iterable<K1> iterable1;

  public:
    inline auto getIterable1() const -> xpress::Iterable<K1> const &;

  private:
    xpress::Iterable<K2> iterable2;

  public:
    inline auto getIterable2() const -> xpress::Iterable<K2> const &;

  private:
    xpress::Iterable<K3> iterable3;

  public:
    inline auto getIterable3() const -> xpress::Iterable<K3> const &;

  private:
    xpress::Iterable<K4> iterable4;

  public:
    inline auto getIterable4() const -> xpress::Iterable<K4> const &;

  private:
    xpress::Iterable<K5> iterable5;

  public:
    inline auto getIterable5() const -> xpress::Iterable<K5> const &;

  private:
    std::function<double(K1, K2, K3, K4, K5)> lb;

    std::function<double(K1, K2, K3, K4, K5)> ub;

    std::function<double(K1, K2, K3, K4, K5)> limit;

    std::function<std::optional<std::string>(K1, K2, K3, K4, K5)> name;

    std::function<xpress::ColumnType(K1, K2, K3, K4, K5)> type;

    std::function<bool(K1, K2, K3, K4, K5)> filter;

  public:
    inline auto getLB() const
        -> std::function<double(K1, K2, K3, K4, K5)> const &;
    inline auto getUB() const
        -> std::function<double(K1, K2, K3, K4, K5)> const &;
    inline auto getLimit() const
        -> std::function<double(K1, K2, K3, K4, K5)> const &;
    inline auto getName() const
        -> std::function<std::optional<std::string>(K1, K2, K3, K4, K5)> const
            &;
    inline auto getType() const
        -> std::function<xpress::ColumnType(K1, K2, K3, K4, K5)> const &;
    inline auto getFilter() const
        -> std::function<bool(K1, K2, K3, K4, K5)> const &;
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              typename Iter4, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1), xpress_isiterable(Iter2),
              xpress_isiterable(Iter3), xpress_isiterable(Iter4)>
    inline Map5BuilderImplementation(Iter0 const &iterable1,
                                     Iter1 const &iterable2,
                                     Iter2 const &iterable3,
                                     Iter3 const &iterable4,
                                     Iter4 const &iterable5);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLB(Func0 newLb)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withLB(double newLb)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withUB(Func0 newUb)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withUB(double newUb)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLimit(Func0 newLimit)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withLimit(double newLimit)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4, K5)>
    inline auto withName(Func0 newName)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withName(std::optional<std::string> newName)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4, K5)>
    inline auto withType(Func0 newType)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withType(ColumnType newType)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4, K5)>
    inline auto withFilter(Func0 newFilter)
        -> Map5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto build(XPRSProblem *prob)
        -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int>;
    inline auto build(xpress::objects::XpressProblem *prob)
        -> xpress::maps::HashMap5<K1, K2, K3, K4, K5,
                                  xpress::objects::Variable>;
  };

public:
  template <typename C, typename CIsIntegral>
  class VariableArrayBuilderImplementation final
      : public std::enable_shared_from_this<
            VariableArrayBuilderImplementation<C, CIsIntegral>> {
    friend class VariableArrayBuilder<C, CIsIntegral>;

  public:
    virtual ~VariableArrayBuilderImplementation() {}

  private:
    ArrayBuilder<C> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    inline VariableArrayBuilderImplementation(
        C dim, xpress::objects::XpressProblem *prob);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLB(Func0 newLb)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withLB(double newLb)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withUB(Func0 newUb)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withUB(double newUb)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withLimit(double newLimit)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C)>
    inline auto withName(Func0 newName)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C)>
    inline auto withType(Func0 newType)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto withType(ColumnType newType)
        -> VariableArrayBuilderImplementation<C, CIsIntegral> &;
    inline auto toArray() -> std::vector<xpress::objects::Variable>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toArray(Func0 makeResult,
            std::function<void(I, C, xpress::objects::Variable)> addResult)
        -> I;
  };

public:
  template <typename C1, typename C2, typename C1IsIntegral,
            typename C2IsIntegral>
  class VariableArray2BuilderImplementation final
      : public std::enable_shared_from_this<VariableArray2BuilderImplementation<
            C1, C2, C1IsIntegral, C2IsIntegral>> {
    friend class VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral>;

  public:
    virtual ~VariableArray2BuilderImplementation() {}

  private:
    Array2Builder<C1, C2> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    inline VariableArray2BuilderImplementation(
        C1 dim1, C2 dim2, xpress::objects::XpressProblem *prob);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLB(Func0 newLb)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    inline auto withLB(double newLb)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withUB(Func0 newUb)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    inline auto withUB(double newUb)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, C1, C2)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2)>
    inline auto withName(Func0 newName)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2)>
    inline auto withType(Func0 newType)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                               C2IsIntegral> &;
    inline auto toArray()
        -> std::vector<std::vector<xpress::objects::Variable>>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, xpress::objects::Variable)> addResult)
        -> I;
  };

public:
  template <typename C1, typename C2, typename C3, typename C1IsIntegral,
            typename C2IsIntegral, typename C3IsIntegral>
  class VariableArray3BuilderImplementation final
      : public std::enable_shared_from_this<VariableArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>> {
    friend class VariableArray3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral>;

  public:
    virtual ~VariableArray3BuilderImplementation() {}

  private:
    Array3Builder<C1, C2, C3> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    inline VariableArray3BuilderImplementation(
        C1 dim1, C2 dim2, C3 dim3, xpress::objects::XpressProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLB(Func0 newLb)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    inline auto withLB(double newLb)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withUB(Func0 newUb)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    inline auto withUB(double newUb)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3)>
    inline auto withName(Func0 newName)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, xpress::ColumnType, Func0, C1, C2, C3)>
    inline auto withType(Func0 newType)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> VariableArray3BuilderImplementation<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral> &;
    inline auto toArray()
        -> std::vector<std::vector<std::vector<xpress::objects::Variable>>>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(
        Func0 makeResult,
        std::function<void(I, C1, C2, C3, xpress::objects::Variable)> addResult)
        -> I;
  };

public:
  template <typename C1, typename C2, typename C3, typename C4,
            typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
            typename C4IsIntegral>
  class VariableArray4BuilderImplementation final
      : public std::enable_shared_from_this<VariableArray4BuilderImplementation<
            C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral>> {
    friend class VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>;

  public:
    virtual ~VariableArray4BuilderImplementation() {}

  private:
    Array4Builder<C1, C2, C3, C4> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    inline VariableArray4BuilderImplementation(
        C1 dim1, C2 dim2, C3 dim3, C4 dim4,
        xpress::objects::XpressProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLB(Func0 newLb)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    inline auto withLB(double newLb)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withUB(Func0 newUb)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    inline auto withUB(double newUb)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4)>
    inline auto withName(Func0 newName)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4)>
    inline auto withType(Func0 newType)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> VariableArray4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral> &;
    inline auto toArray() -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, C3, C4, xpress::objects::Variable)>
                addResult) -> I;
  };

public:
  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
            typename C4IsIntegral, typename C5IsIntegral>
  class VariableArray5BuilderImplementation final
      : public std::enable_shared_from_this<VariableArray5BuilderImplementation<
            C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
            C4IsIntegral, C5IsIntegral>> {
    friend class VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral>;

  public:
    virtual ~VariableArray5BuilderImplementation() {}

  private:
    Array5Builder<C1, C2, C3, C4, C5> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    inline VariableArray5BuilderImplementation(
        C1 dim1, C2 dim2, C3 dim3, C4 dim4, C5 dim5,
        xpress::objects::XpressProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLB(Func0 newLb)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    inline auto withLB(double newLb)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withUB(Func0 newUb)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    inline auto withUB(double newUb)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, C1, C2, C3, C4, C5)>
    inline auto withLimit(Func0 newLimit)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    inline auto withLimit(double newLimit)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, C1,
                                 C2, C3, C4, C5)>
    inline auto withName(Func0 newName)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    template <typename Func0, xpress_isinvocable(Func0, xpress::ColumnType,
                                                 Func0, C1, C2, C3, C4, C5)>
    inline auto withType(Func0 newType)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    inline auto withType(ColumnType newType)
        -> VariableArray5BuilderImplementation<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral> &;
    inline auto toArray() -> std::vector<std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toArray(
        Func0 makeResult,
        std::function<void(I, C1, C2, C3, C4, C5, xpress::objects::Variable)>
            addResult) -> I;
  };

public:
  template <typename K1>
  class ColumnMapBuilderImplementation
      : public std::enable_shared_from_this<
            ColumnMapBuilderImplementation<K1>> {
    friend class ColumnMapBuilder<K1>;

  public:
    virtual ~ColumnMapBuilderImplementation() {}

  private:
    MapBuilder<K1> wrapped;

    xpress::XPRSProblem *prob;

  public:
    template <typename Iter0, xpress_isiterable(Iter0)>
    inline ColumnMapBuilderImplementation(Iter0 const &iterable1,
                                          xpress::XPRSProblem *prob);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLB(Func0 newLb) -> ColumnMapBuilderImplementation<K1> &;
    inline auto withLB(double newLb) -> ColumnMapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withUB(Func0 newUb) -> ColumnMapBuilderImplementation<K1> &;
    inline auto withUB(double newUb) -> ColumnMapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnMapBuilderImplementation<K1> &;
    inline auto withLimit(double newLimit)
        -> ColumnMapBuilderImplementation<K1> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1)>
    inline auto withName(Func0 newName) -> ColumnMapBuilderImplementation<K1> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnMapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, ColumnType, Func0, K1)>
    inline auto withType(Func0 newType) -> ColumnMapBuilderImplementation<K1> &;
    inline auto withType(ColumnType newType)
        -> ColumnMapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1)>
    inline auto withFilter(Func0 newFilter)
        -> ColumnMapBuilderImplementation<K1> &;
    inline auto toMap() -> std::unordered_map<K1, int>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(Func0 makeResult,
                      std::function<void(I &, K1, int)> addResult) -> I;
  };

public:
  template <typename K1, typename K2>
  class ColumnMap2BuilderImplementation
      : public std::enable_shared_from_this<
            ColumnMap2BuilderImplementation<K1, K2>> {
    friend class ColumnMap2Builder<K1, K2>;

  public:
    virtual ~ColumnMap2BuilderImplementation() {}

  private:
    Map2Builder<K1, K2> wrapped;

    xpress::XPRSProblem *prob;

  public:
    template <typename Iter0, typename Iter1, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1)>
    inline ColumnMap2BuilderImplementation(Iter0 const &iterable1,
                                           Iter1 const &iterable2,
                                           xpress::XPRSProblem *prob);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLB(Func0 newLb)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    inline auto withLB(double newLb)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withUB(Func0 newUb)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    inline auto withUB(double newUb)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    inline auto withLimit(double newLimit)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2)>
    inline auto withName(Func0 newName)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2)>
    inline auto withType(Func0 newType)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    inline auto withType(ColumnType newType)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1, K2)>
    inline auto withFilter(Func0 newFilter)
        -> ColumnMap2BuilderImplementation<K1, K2> &;
    inline auto toMap() -> xpress::maps::HashMap2<K1, K2, int>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(Func0 makeResult,
                      std::function<void(I &, K1, K2, int)> addResult) -> I;
  };

public:
  template <typename K1, typename K2, typename K3>
  class ColumnMap3BuilderImplementation
      : public std::enable_shared_from_this<
            ColumnMap3BuilderImplementation<K1, K2, K3>> {
    friend class ColumnMap3Builder<K1, K2, K3>;

  public:
    virtual ~ColumnMap3BuilderImplementation() {}

  private:
    Map3Builder<K1, K2, K3> wrapped;

    xpress::XPRSProblem *prob;

  public:
    template <typename Iter0, typename Iter1, typename Iter2,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2)>
    inline ColumnMap3BuilderImplementation(Iter0 const &iterable1,
                                           Iter1 const &iterable2,
                                           Iter2 const &iterable3,
                                           xpress::XPRSProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLB(Func0 newLb)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withLB(double newLb)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withUB(Func0 newUb)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withUB(double newUb)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withLimit(double newLimit)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3)>
    inline auto withName(Func0 newName)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3)>
    inline auto withType(Func0 newType)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withType(ColumnType newType)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3)>
    inline auto withFilter(Func0 newFilter)
        -> ColumnMap3BuilderImplementation<K1, K2, K3> &;
    inline auto toMap() -> xpress::maps::HashMap3<K1, K2, K3, int>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(Func0 makeResult,
                      std::function<void(I &, K1, K2, K3, int)> addResult) -> I;
  };

public:
  template <typename K1, typename K2, typename K3, typename K4>
  class ColumnMap4BuilderImplementation
      : public std::enable_shared_from_this<
            ColumnMap4BuilderImplementation<K1, K2, K3, K4>> {
    friend class ColumnMap4Builder<K1, K2, K3, K4>;

  public:
    virtual ~ColumnMap4BuilderImplementation() {}

  private:
    Map4Builder<K1, K2, K3, K4> wrapped;

    xpress::XPRSProblem *prob;

  public:
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
    inline ColumnMap4BuilderImplementation(Iter0 const &iterable1,
                                           Iter1 const &iterable2,
                                           Iter2 const &iterable3,
                                           Iter3 const &iterable4,
                                           xpress::XPRSProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLB(Func0 newLb)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withLB(double newLb)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withUB(Func0 newUb)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withUB(double newUb)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withLimit(double newLimit)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4)>
    inline auto withName(Func0 newName)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4)>
    inline auto withType(Func0 newType)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withType(ColumnType newType)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4)>
    inline auto withFilter(Func0 newFilter)
        -> ColumnMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto toMap() -> xpress::maps::HashMap4<K1, K2, K3, K4, int>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(Func0 makeResult,
                      std::function<void(I &, K1, K2, K3, K4, int)> addResult)
        -> I;
  };

public:
  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class ColumnMap5BuilderImplementation
      : public std::enable_shared_from_this<
            ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5>> {
    friend class ColumnMap5Builder<K1, K2, K3, K4, K5>;

  public:
    virtual ~ColumnMap5BuilderImplementation() {}

  private:
    Map5Builder<K1, K2, K3, K4, K5> wrapped;

    xpress::XPRSProblem *prob;

  public:
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              typename Iter4, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1), xpress_isiterable(Iter2),
              xpress_isiterable(Iter3), xpress_isiterable(Iter4)>
    inline ColumnMap5BuilderImplementation(Iter0 const &iterable1,
                                           Iter1 const &iterable2,
                                           Iter2 const &iterable3,
                                           Iter3 const &iterable4,
                                           Iter4 const &iterable5,
                                           xpress::XPRSProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLB(Func0 newLb)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withLB(double newLb)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withUB(Func0 newUb)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withUB(double newUb)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLimit(Func0 newLimit)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withLimit(double newLimit)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4, K5)>
    inline auto withName(Func0 newName)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withName(std::optional<std::string> newName)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4, K5)>
    inline auto withType(Func0 newType)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withType(ColumnType newType)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4, K5)>
    inline auto withFilter(Func0 newFilter)
        -> ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto toMap() -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, K3, K4, K5, int)> addResult) -> I;
  };

public:
  template <typename K1>
  class VariableMapBuilderImplementation
      : public std::enable_shared_from_this<
            VariableMapBuilderImplementation<K1>> {
    friend class VariableMapBuilder<K1>;

  public:
    virtual ~VariableMapBuilderImplementation() {}

  private:
    MapBuilder<K1> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    template <typename Iter0, xpress_isiterable(Iter0)>
    inline VariableMapBuilderImplementation(
        Iter0 const &iterable1, xpress::objects::XpressProblem *prob);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLB(Func0 newLb) -> VariableMapBuilderImplementation<K1> &;
    inline auto withLB(double newLb) -> VariableMapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withUB(Func0 newUb) -> VariableMapBuilderImplementation<K1> &;
    inline auto withUB(double newUb) -> VariableMapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1)>
    inline auto withLimit(Func0 newLimit)
        -> VariableMapBuilderImplementation<K1> &;
    inline auto withLimit(double newLimit)
        -> VariableMapBuilderImplementation<K1> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1)>
    inline auto withName(Func0 newName)
        -> VariableMapBuilderImplementation<K1> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableMapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, ColumnType, Func0, K1)>
    inline auto withType(Func0 newType)
        -> VariableMapBuilderImplementation<K1> &;
    inline auto withType(ColumnType newType)
        -> VariableMapBuilderImplementation<K1> &;
    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1)>
    inline auto withFilter(Func0 newFilter)
        -> VariableMapBuilderImplementation<K1> &;
    inline auto toMap() -> std::unordered_map<K1, xpress::objects::Variable>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, xpress::objects::Variable)> addResult)
        -> I;
  };

public:
  template <typename K1, typename K2>
  class VariableMap2BuilderImplementation
      : public std::enable_shared_from_this<
            VariableMap2BuilderImplementation<K1, K2>> {
    friend class VariableMap2Builder<K1, K2>;

  public:
    virtual ~VariableMap2BuilderImplementation() {}

  private:
    Map2Builder<K1, K2> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    template <typename Iter0, typename Iter1, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1)>
    inline VariableMap2BuilderImplementation(
        Iter0 const &iterable1, Iter1 const &iterable2,
        xpress::objects::XpressProblem *prob);
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLB(Func0 newLb)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    inline auto withLB(double newLb)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withUB(Func0 newUb)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    inline auto withUB(double newUb)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, double, Func0, K1, K2)>
    inline auto withLimit(Func0 newLimit)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    inline auto withLimit(double newLimit)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2)>
    inline auto withName(Func0 newName)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2)>
    inline auto withType(Func0 newType)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    inline auto withType(ColumnType newType)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    template <typename Func0, xpress_isinvocable(Func0, bool, Func0, K1, K2)>
    inline auto withFilter(Func0 newFilter)
        -> VariableMap2BuilderImplementation<K1, K2> &;
    inline auto toMap()
        -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, xpress::objects::Variable)> addResult)
        -> I;
  };

public:
  template <typename K1, typename K2, typename K3>
  class VariableMap3BuilderImplementation
      : public std::enable_shared_from_this<
            VariableMap3BuilderImplementation<K1, K2, K3>> {
    friend class VariableMap3Builder<K1, K2, K3>;

  public:
    virtual ~VariableMap3BuilderImplementation() {}

  private:
    Map3Builder<K1, K2, K3> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    template <typename Iter0, typename Iter1, typename Iter2,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2)>
    inline VariableMap3BuilderImplementation(
        Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
        xpress::objects::XpressProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLB(Func0 newLb)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withLB(double newLb)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withUB(Func0 newUb)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withUB(double newUb)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3)>
    inline auto withLimit(Func0 newLimit)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withLimit(double newLimit)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3)>
    inline auto withName(Func0 newName)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3)>
    inline auto withType(Func0 newType)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    inline auto withType(ColumnType newType)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3)>
    inline auto withFilter(Func0 newFilter)
        -> VariableMap3BuilderImplementation<K1, K2, K3> &;
    inline auto toMap()
        -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, K3, xpress::objects::Variable)>
              addResult) -> I;
  };

public:
  template <typename K1, typename K2, typename K3, typename K4>
  class VariableMap4BuilderImplementation
      : public std::enable_shared_from_this<
            VariableMap4BuilderImplementation<K1, K2, K3, K4>> {
    friend class VariableMap4Builder<K1, K2, K3, K4>;

  public:
    virtual ~VariableMap4BuilderImplementation() {}

  private:
    Map4Builder<K1, K2, K3, K4> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              xpress_isiterable(Iter0), xpress_isiterable(Iter1),
              xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
    inline VariableMap4BuilderImplementation(
        Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
        Iter3 const &iterable4, xpress::objects::XpressProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLB(Func0 newLb)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withLB(double newLb)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withUB(Func0 newUb)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withUB(double newUb)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4)>
    inline auto withLimit(Func0 newLimit)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withLimit(double newLimit)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4)>
    inline auto withName(Func0 newName)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4)>
    inline auto withType(Func0 newType)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto withType(ColumnType newType)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4)>
    inline auto withFilter(Func0 newFilter)
        -> VariableMap4BuilderImplementation<K1, K2, K3, K4> &;
    inline auto toMap()
        -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto
    toMap(Func0 makeResult,
          std::function<void(I &, K1, K2, K3, K4, xpress::objects::Variable)>
              addResult) -> I;
  };

public:
  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  class VariableMap5BuilderImplementation
      : public std::enable_shared_from_this<
            VariableMap5BuilderImplementation<K1, K2, K3, K4, K5>> {
    friend class VariableMap5Builder<K1, K2, K3, K4, K5>;

  public:
    virtual ~VariableMap5BuilderImplementation() {}

  private:
    Map5Builder<K1, K2, K3, K4, K5> wrapped;

    xpress::objects::XpressProblem *prob;

  public:
    template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
              typename Iter4, xpress_isiterable(Iter0),
              xpress_isiterable(Iter1), xpress_isiterable(Iter2),
              xpress_isiterable(Iter3), xpress_isiterable(Iter4)>
    inline VariableMap5BuilderImplementation(
        Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
        Iter3 const &iterable4, Iter4 const &iterable5,
        xpress::objects::XpressProblem *prob);
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLB(Func0 newLb)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withLB(double newLb)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withUB(Func0 newUb)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withUB(double newUb)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, double, Func0, K1, K2, K3, K4, K5)>
    inline auto withLimit(Func0 newLimit)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withLimit(double newLimit)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, std::optional<std::string>, Func0, K1,
                                 K2, K3, K4, K5)>
    inline auto withName(Func0 newName)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withName(std::optional<std::string> newName)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, ColumnType, Func0, K1, K2, K3, K4, K5)>
    inline auto withType(Func0 newType)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto withType(ColumnType newType)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    template <typename Func0,
              xpress_isinvocable(Func0, bool, Func0, K1, K2, K3, K4, K5)>
    inline auto withFilter(Func0 newFilter)
        -> VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> &;
    inline auto toMap() -> xpress::maps::HashMap5<K1, K2, K3, K4, K5,
                                                  xpress::objects::Variable>;
    template <typename I, typename Func0, xpress_isinvocable(Func0, I, Func0)>
    inline auto toMap(
        Func0 makeResult,
        std::function<void(I &, K1, K2, K3, K4, K5, xpress::objects::Variable)>
            addResult) -> I;
  };
};
} // namespace xpress
namespace xpress {
xpress_diags_push xpress_ignore_deprecated

    enum class SolStatus : int {
      NotFound = XPRS_SOLSTATUS_NOTFOUND,
      Optimal = XPRS_SOLSTATUS_OPTIMAL,
      Feasible = XPRS_SOLSTATUS_FEASIBLE,
      Infeasible = XPRS_SOLSTATUS_INFEASIBLE,
      Unbounded = XPRS_SOLSTATUS_UNBOUNDED
    };
xpress_diags_pop inline bool operator==(SolStatus a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(SolStatus a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, SolStatus b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, SolStatus b) { return a != static_cast<int>(b); }
inline std::string to_string(SolStatus e) {

  switch (e) {
  case SolStatus::NotFound:
    return "NotFound";
    break;
  case SolStatus::Optimal:
    return "Optimal";
    break;
  case SolStatus::Feasible:
    return "Feasible";
    break;
  case SolStatus::Infeasible:
    return "Infeasible";
    break;
  case SolStatus::Unbounded:
    return "Unbounded";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, SolStatus e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class SolveStatus : int {
      Unstarted = XPRS_SOLVESTATUS_UNSTARTED,
      Stopped = XPRS_SOLVESTATUS_STOPPED,
      Failed = XPRS_SOLVESTATUS_FAILED,
      Completed = XPRS_SOLVESTATUS_COMPLETED
    };
xpress_diags_pop inline bool operator==(SolveStatus a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(SolveStatus a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, SolveStatus b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, SolveStatus b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(SolveStatus e) {

  switch (e) {
  case SolveStatus::Unstarted:
    return "Unstarted";
    break;
  case SolveStatus::Stopped:
    return "Stopped";
    break;
  case SolveStatus::Failed:
    return "Failed";
    break;
  case SolveStatus::Completed:
    return "Completed";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, SolveStatus e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class LPStatus : int {
      Unstarted = XPRS_LP_UNSTARTED,
      Optimal = XPRS_LP_OPTIMAL,
      Infeasible = XPRS_LP_INFEAS,
      CutOff = XPRS_LP_CUTOFF,
      Unfinished = XPRS_LP_UNFINISHED,
      Unbounded = XPRS_LP_UNBOUNDED,
      CutOffInDual = XPRS_LP_CUTOFF_IN_DUAL,
      Unsolved = XPRS_LP_UNSOLVED,
      Nonconvex = XPRS_LP_NONCONVEX
    };
xpress_diags_pop inline bool operator==(LPStatus a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(LPStatus a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, LPStatus b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, LPStatus b) { return a != static_cast<int>(b); }
inline std::string to_string(LPStatus e) {

  switch (e) {
  case LPStatus::Unstarted:
    return "Unstarted";
    break;
  case LPStatus::Optimal:
    return "Optimal";
    break;
  case LPStatus::Infeasible:
    return "Infeasible";
    break;
  case LPStatus::CutOff:
    return "CutOff";
    break;
  case LPStatus::Unfinished:
    return "Unfinished";
    break;
  case LPStatus::Unbounded:
    return "Unbounded";
    break;
  case LPStatus::CutOffInDual:
    return "CutOffInDual";
    break;
  case LPStatus::Unsolved:
    return "Unsolved";
    break;
  case LPStatus::Nonconvex:
    return "Nonconvex";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, LPStatus e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class MIPStatus : int {
      NotLoaded = XPRS_MIP_NOT_LOADED,
      LPNotOptimal = XPRS_MIP_LP_NOT_OPTIMAL,
      LPOptimal = XPRS_MIP_LP_OPTIMAL,
      NoSolutionFound = XPRS_MIP_NO_SOL_FOUND,
      Solution = XPRS_MIP_SOLUTION,
      Infeasible = XPRS_MIP_INFEAS,
      Optimal = XPRS_MIP_OPTIMAL,
      Unbounded = XPRS_MIP_UNBOUNDED
    };
xpress_diags_pop inline bool operator==(MIPStatus a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(MIPStatus a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, MIPStatus b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, MIPStatus b) { return a != static_cast<int>(b); }
inline std::string to_string(MIPStatus e) {

  switch (e) {
  case MIPStatus::NotLoaded:
    return "NotLoaded";
    break;
  case MIPStatus::LPNotOptimal:
    return "LPNotOptimal";
    break;
  case MIPStatus::LPOptimal:
    return "LPOptimal";
    break;
  case MIPStatus::NoSolutionFound:
    return "NoSolutionFound";
    break;
  case MIPStatus::Solution:
    return "Solution";
    break;
  case MIPStatus::Infeasible:
    return "Infeasible";
    break;
  case MIPStatus::Optimal:
    return "Optimal";
    break;
  case MIPStatus::Unbounded:
    return "Unbounded";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, MIPStatus e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class IISSolStatus : int {
      Unstarted = XPRS_IIS_UNSTARTED,
      Feasible = XPRS_IIS_FEASIBLE,
      Completed = XPRS_IIS_COMPLETED,
      Unfinished = XPRS_IIS_UNFINISHED
    };
xpress_diags_pop inline bool operator==(IISSolStatus a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(IISSolStatus a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, IISSolStatus b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, IISSolStatus b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(IISSolStatus e) {

  switch (e) {
  case IISSolStatus::Unstarted:
    return "Unstarted";
    break;
  case IISSolStatus::Feasible:
    return "Feasible";
    break;
  case IISSolStatus::Completed:
    return "Completed";
    break;
  case IISSolStatus::Unfinished:
    return "Unfinished";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, IISSolStatus e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class SolAvailable : int {
      NotFound = XPRS_SOLAVAILABLE_NOTFOUND,
      Optimal = XPRS_SOLAVAILABLE_OPTIMAL,
      Feasible = XPRS_SOLAVAILABLE_FEASIBLE
    };
xpress_diags_pop inline bool operator==(SolAvailable a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(SolAvailable a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, SolAvailable b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, SolAvailable b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(SolAvailable e) {

  switch (e) {
  case SolAvailable::NotFound:
    return "NotFound";
    break;
  case SolAvailable::Optimal:
    return "Optimal";
    break;
  case SolAvailable::Feasible:
    return "Feasible";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, SolAvailable e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class OptimizeType : int {
      None = XPRS_OPTIMIZETYPE_NONE,
      Lp = XPRS_OPTIMIZETYPE_LP,
      Mip = XPRS_OPTIMIZETYPE_MIP,
      Local = XPRS_OPTIMIZETYPE_LOCAL,
      Global = XPRS_OPTIMIZETYPE_GLOBAL
    };
xpress_diags_pop inline bool operator==(OptimizeType a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(OptimizeType a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, OptimizeType b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, OptimizeType b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(OptimizeType e) {

  switch (e) {
  case OptimizeType::None:
    return "None";
    break;
  case OptimizeType::Lp:
    return "Lp";
    break;
  case OptimizeType::Mip:
    return "Mip";
    break;
  case OptimizeType::Local:
    return "Local";
    break;
  case OptimizeType::Global:
    return "Global";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, OptimizeType e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class BarOrder : int {
      Default = XPRS_BAR_DEFAULT,
      MinDegree = XPRS_BAR_MIN_DEGREE,
      MinLocalFill = XPRS_BAR_MIN_LOCAL_FILL,
      NestedDissection = XPRS_BAR_NESTED_DISSECTION
    };
xpress_diags_pop inline bool operator==(BarOrder a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(BarOrder a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, BarOrder b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, BarOrder b) { return a != static_cast<int>(b); }
inline std::string to_string(BarOrder e) {

  switch (e) {
  case BarOrder::Default:
    return "Default";
    break;
  case BarOrder::MinDegree:
    return "MinDegree";
    break;
  case BarOrder::MinLocalFill:
    return "MinLocalFill";
    break;
  case BarOrder::NestedDissection:
    return "NestedDissection";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, BarOrder e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class DefaultAlg : int {
      Default = XPRS_ALG_DEFAULT,
      Dual = XPRS_ALG_DUAL,
      Primal = XPRS_ALG_PRIMAL,
      Barrier = XPRS_ALG_BARRIER,
      Network = XPRS_ALG_NETWORK
    };
xpress_diags_pop inline bool operator==(DefaultAlg a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(DefaultAlg a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, DefaultAlg b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, DefaultAlg b) { return a != static_cast<int>(b); }
inline std::string to_string(DefaultAlg e) {

  switch (e) {
  case DefaultAlg::Default:
    return "Default";
    break;
  case DefaultAlg::Dual:
    return "Dual";
    break;
  case DefaultAlg::Primal:
    return "Primal";
    break;
  case DefaultAlg::Barrier:
    return "Barrier";
    break;
  case DefaultAlg::Network:
    return "Network";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, DefaultAlg e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class StopType : int {
      None = XPRS_STOP_NONE,
      TimeLimit = XPRS_STOP_TIMELIMIT,
      CtrlC = XPRS_STOP_CTRLC,
      NodeLimit = XPRS_STOP_NODELIMIT,
      IterLimit = XPRS_STOP_ITERLIMIT,
      MIPGap = XPRS_STOP_MIPGAP,
      SolLimit = XPRS_STOP_SOLLIMIT,
      GenericError = XPRS_STOP_GENERICERROR,
      MemoryError = XPRS_STOP_MEMORYERROR,
      User = XPRS_STOP_USER,
      SolveComplete = XPRS_STOP_SOLVECOMPLETE,
      LicenseLost = XPRS_STOP_LICENSELOST,
      NumericalError = XPRS_STOP_NUMERICALERROR,
      WorkLimit = XPRS_STOP_WORKLIMIT
    };
xpress_diags_pop inline bool operator==(StopType a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(StopType a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, StopType b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, StopType b) { return a != static_cast<int>(b); }
inline std::string to_string(StopType e) {

  switch (e) {
  case StopType::None:
    return "None";
    break;
  case StopType::TimeLimit:
    return "TimeLimit";
    break;
  case StopType::CtrlC:
    return "CtrlC";
    break;
  case StopType::NodeLimit:
    return "NodeLimit";
    break;
  case StopType::IterLimit:
    return "IterLimit";
    break;
  case StopType::MIPGap:
    return "MIPGap";
    break;
  case StopType::SolLimit:
    return "SolLimit";
    break;
  case StopType::GenericError:
    return "GenericError";
    break;
  case StopType::MemoryError:
    return "MemoryError";
    break;
  case StopType::User:
    return "User";
    break;
  case StopType::SolveComplete:
    return "SolveComplete";
    break;
  case StopType::LicenseLost:
    return "LicenseLost";
    break;
  case StopType::NumericalError:
    return "NumericalError";
    break;
  case StopType::WorkLimit:
    return "WorkLimit";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, StopType e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class AlwaysNeverAutomatic : int {
      Automatic = XPRS_ANA_AUTOMATIC,
      Never = XPRS_ANA_NEVER,
      Always = XPRS_ANA_ALWAYS
    };
xpress_diags_pop inline bool operator==(AlwaysNeverAutomatic a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(AlwaysNeverAutomatic a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, AlwaysNeverAutomatic b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, AlwaysNeverAutomatic b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(AlwaysNeverAutomatic e) {

  switch (e) {
  case AlwaysNeverAutomatic::Automatic:
    return "Automatic";
    break;
  case AlwaysNeverAutomatic::Never:
    return "Never";
    break;
  case AlwaysNeverAutomatic::Always:
    return "Always";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, AlwaysNeverAutomatic e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class OnOff : int {
      Off = XPRS_BOOL_OFF,
      On = XPRS_BOOL_ON
    };
xpress_diags_pop inline bool operator==(OnOff a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(OnOff a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, OnOff b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, OnOff b) { return a != static_cast<int>(b); }
inline std::string to_string(OnOff e) {

  switch (e) {
  case OnOff::Off:
    return "Off";
    break;
  case OnOff::On:
    return "On";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, OnOff e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class BacktrackAlg : int {
      BestEstimate = XPRS_BACKTRACKALG_BEST_ESTIMATE,
      BestBound = XPRS_BACKTRACKALG_BEST_BOUND,
      DeepestNode = XPRS_BACKTRACKALG_DEEPEST_NODE,
      HighestNode = XPRS_BACKTRACKALG_HIGHEST_NODE,
      EarliestNode = XPRS_BACKTRACKALG_EARLIEST_NODE,
      LatestNode = XPRS_BACKTRACKALG_LATEST_NODE,
      Random = XPRS_BACKTRACKALG_RANDOM,
      MinimumInfeas = XPRS_BACKTRACKALG_MIN_INFEAS,
      BestEstimateWithMinimumInfeas =
          XPRS_BACKTRACKALG_BEST_ESTIMATE_MIN_INFEAS,
      DeepestBestEstimate = XPRS_BACKTRACKALG_DEEPEST_BEST_ESTIMATE
    };
xpress_diags_pop inline bool operator==(BacktrackAlg a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(BacktrackAlg a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, BacktrackAlg b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, BacktrackAlg b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(BacktrackAlg e) {

  switch (e) {
  case BacktrackAlg::BestEstimate:
    return "BestEstimate";
    break;
  case BacktrackAlg::BestBound:
    return "BestBound";
    break;
  case BacktrackAlg::DeepestNode:
    return "DeepestNode";
    break;
  case BacktrackAlg::HighestNode:
    return "HighestNode";
    break;
  case BacktrackAlg::EarliestNode:
    return "EarliestNode";
    break;
  case BacktrackAlg::LatestNode:
    return "LatestNode";
    break;
  case BacktrackAlg::Random:
    return "Random";
    break;
  case BacktrackAlg::MinimumInfeas:
    return "MinimumInfeas";
    break;
  case BacktrackAlg::BestEstimateWithMinimumInfeas:
    return "BestEstimateWithMinimumInfeas";
    break;
  case BacktrackAlg::DeepestBestEstimate:
    return "DeepestBestEstimate";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, BacktrackAlg e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class BranchChoice : int {
      MinimumEstimateFirst = XPRS_BRANCH_MIN_EST_FIRST,
      MaximumEstimateFirst = XPRS_BRANCH_MAX_EST_FIRST
    };
xpress_diags_pop inline bool operator==(BranchChoice a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(BranchChoice a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, BranchChoice b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, BranchChoice b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(BranchChoice e) {

  switch (e) {
  case BranchChoice::MinimumEstimateFirst:
    return "MinimumEstimateFirst";
    break;
  case BranchChoice::MaximumEstimateFirst:
    return "MaximumEstimateFirst";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, BranchChoice e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class CholeskyAlgorithm : int {
      PullCholesky = XPRS_ALG_PULL_CHOLESKY,
      PushCholesky = XPRS_ALG_PUSH_CHOLESKY
    };
xpress_diags_pop inline bool operator==(CholeskyAlgorithm a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(CholeskyAlgorithm a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, CholeskyAlgorithm b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, CholeskyAlgorithm b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(CholeskyAlgorithm e) {

  switch (e) {
  case CholeskyAlgorithm::PullCholesky:
    return "PullCholesky";
    break;
  case CholeskyAlgorithm::PushCholesky:
    return "PushCholesky";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, CholeskyAlgorithm e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class CrossoverDynamicReduction : int {
      BeforeCrossover = XPRS_XDRPBEFORE_CROSSOVER,
      InsideCrossover = XPRS_XDRPINSIDE_CROSSOVER,
      AggressiveBeforeCrossover = XPRS_XDRPAGGRESSIVE_BEFORE_CROSSOVER
    };
xpress_diags_pop inline bool operator==(CrossoverDynamicReduction a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(CrossoverDynamicReduction a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, CrossoverDynamicReduction b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, CrossoverDynamicReduction b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(CrossoverDynamicReduction e) {

  switch (e) {
  case CrossoverDynamicReduction::BeforeCrossover:
    return "BeforeCrossover";
    break;
  case CrossoverDynamicReduction::InsideCrossover:
    return "InsideCrossover";
    break;
  case CrossoverDynamicReduction::AggressiveBeforeCrossover:
    return "AggressiveBeforeCrossover";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, CrossoverDynamicReduction e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class DualGradient : int {
      Automatic = XPRS_DUALGRADIENT_AUTOMATIC,
      Devex = XPRS_DUALGRADIENT_DEVEX,
      SteepestEdge = XPRS_DUALGRADIENT_STEEPESTEDGE
    };
xpress_diags_pop inline bool operator==(DualGradient a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(DualGradient a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, DualGradient b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, DualGradient b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(DualGradient e) {

  switch (e) {
  case DualGradient::Automatic:
    return "Automatic";
    break;
  case DualGradient::Devex:
    return "Devex";
    break;
  case DualGradient::SteepestEdge:
    return "SteepestEdge";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, DualGradient e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class DualStrategy : int {
      RemoveInfeasibilitiesWithPrimal =
          XPRS_DUALSTRATEGY_REMOVE_INFEAS_WITH_PRIMAL,
      RemoveInfeasibilitiesWithDual = XPRS_DUALSTRATEGY_REMOVE_INFEAS_WITH_DUAL
    };
xpress_diags_pop inline bool operator==(DualStrategy a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(DualStrategy a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, DualStrategy b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, DualStrategy b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(DualStrategy e) {

  switch (e) {
  case DualStrategy::RemoveInfeasibilitiesWithPrimal:
    return "RemoveInfeasibilitiesWithPrimal";
    break;
  case DualStrategy::RemoveInfeasibilitiesWithDual:
    return "RemoveInfeasibilitiesWithDual";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, DualStrategy e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class FeasibilityPump : int {
      Automatic = XPRS_FEASIBILITYPUMP_AUTOMATIC,
      Never = XPRS_FEASIBILITYPUMP_NEVER,
      Always = XPRS_FEASIBILITYPUMP_ALWAYS,
      LastResort = XPRS_FEASIBILITYPUMP_LASTRESORT
    };
xpress_diags_pop inline bool operator==(FeasibilityPump a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(FeasibilityPump a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, FeasibilityPump b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, FeasibilityPump b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(FeasibilityPump e) {

  switch (e) {
  case FeasibilityPump::Automatic:
    return "Automatic";
    break;
  case FeasibilityPump::Never:
    return "Never";
    break;
  case FeasibilityPump::Always:
    return "Always";
    break;
  case FeasibilityPump::LastResort:
    return "LastResort";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, FeasibilityPump e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class HeuristicSearchSelect : int {
      LocalSearchLargeNeighbourhood =
          XPRS_HEURSEARCH_LOCAL_SEARCH_LARGE_NEIGHBOURHOOD,
      LocalSearchNodeNeighbourhood =
          XPRS_HEURSEARCH_LOCAL_SEARCH_NODE_NEIGHBOURHOOD,
      LocalSearchSolutionNeighbourhood =
          XPRS_HEURSEARCH_LOCAL_SEARCH_SOLUTION_NEIGHBOURHOOD
    };
xpress_diags_pop inline bool operator==(HeuristicSearchSelect a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(HeuristicSearchSelect a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, HeuristicSearchSelect b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, HeuristicSearchSelect b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(HeuristicSearchSelect e) {

  switch (e) {
  case HeuristicSearchSelect::LocalSearchLargeNeighbourhood:
    return "LocalSearchLargeNeighbourhood";
    break;
  case HeuristicSearchSelect::LocalSearchNodeNeighbourhood:
    return "LocalSearchNodeNeighbourhood";
    break;
  case HeuristicSearchSelect::LocalSearchSolutionNeighbourhood:
    return "LocalSearchSolutionNeighbourhood";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, HeuristicSearchSelect e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class HeuristicStrategy : int {
      Automatic = XPRS_HEURSTRATEGY_AUTOMATIC,
      None = XPRS_HEURSTRATEGY_NONE,
      Basic = XPRS_HEURSTRATEGY_BASIC,
      Enhanced = XPRS_HEURSTRATEGY_ENHANCED,
      Extensive = XPRS_HEURSTRATEGY_EXTENSIVE
    };
xpress_diags_pop inline bool operator==(HeuristicStrategy a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(HeuristicStrategy a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, HeuristicStrategy b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, HeuristicStrategy b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(HeuristicStrategy e) {

  switch (e) {
  case HeuristicStrategy::Automatic:
    return "Automatic";
    break;
  case HeuristicStrategy::None:
    return "None";
    break;
  case HeuristicStrategy::Basic:
    return "Basic";
    break;
  case HeuristicStrategy::Enhanced:
    return "Enhanced";
    break;
  case HeuristicStrategy::Extensive:
    return "Extensive";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, HeuristicStrategy e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class NodeSelectionCriteria : int {
      LocalFirst = XPRS_NODESELECTION_LOCAL_FIRST,
      BestFirst = XPRS_NODESELECTION_BEST_FIRST,
      LocalDepthFirst = XPRS_NODESELECTION_LOCAL_DEPTH_FIRST,
      BestFirstThenLocalFirst = XPRS_NODESELECTION_BEST_FIRST_THEN_LOCAL_FIRST,
      DepthFirst = XPRS_NODESELECTION_DEPTH_FIRST
    };
xpress_diags_pop inline bool operator==(NodeSelectionCriteria a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(NodeSelectionCriteria a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, NodeSelectionCriteria b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, NodeSelectionCriteria b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(NodeSelectionCriteria e) {

  switch (e) {
  case NodeSelectionCriteria::LocalFirst:
    return "LocalFirst";
    break;
  case NodeSelectionCriteria::BestFirst:
    return "BestFirst";
    break;
  case NodeSelectionCriteria::LocalDepthFirst:
    return "LocalDepthFirst";
    break;
  case NodeSelectionCriteria::BestFirstThenLocalFirst:
    return "BestFirstThenLocalFirst";
    break;
  case NodeSelectionCriteria::DepthFirst:
    return "DepthFirst";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, NodeSelectionCriteria e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class OutputDetail : int {
      NoOutput = XPRS_OUTPUTLOG_NO_OUTPUT,
      FullOutput = XPRS_OUTPUTLOG_FULL_OUTPUT,
      ErrorsAndWarnings = XPRS_OUTPUTLOG_ERRORS_AND_WARNINGS,
      ErrorsOnly = XPRS_OUTPUTLOG_ERRORS
    };
xpress_diags_pop inline bool operator==(OutputDetail a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(OutputDetail a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, OutputDetail b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, OutputDetail b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(OutputDetail e) {

  switch (e) {
  case OutputDetail::NoOutput:
    return "NoOutput";
    break;
  case OutputDetail::FullOutput:
    return "FullOutput";
    break;
  case OutputDetail::ErrorsAndWarnings:
    return "ErrorsAndWarnings";
    break;
  case OutputDetail::ErrorsOnly:
    return "ErrorsOnly";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, OutputDetail e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class PreProbing : int {
      Automatic = XPRS_PREPROBING_AUTOMATIC,
      Disabled = XPRS_PREPROBING_DISABLED,
      LightProbing = XPRS_PREPROBING_LIGHT,
      FullProbing = XPRS_PREPROBING_FULL,
      FullProbingAndRepeat = XPRS_PREPROBING_FULL_AND_REPEAT
    };
xpress_diags_pop inline bool operator==(PreProbing a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(PreProbing a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, PreProbing b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, PreProbing b) { return a != static_cast<int>(b); }
inline std::string to_string(PreProbing e) {

  switch (e) {
  case PreProbing::Automatic:
    return "Automatic";
    break;
  case PreProbing::Disabled:
    return "Disabled";
    break;
  case PreProbing::LightProbing:
    return "LightProbing";
    break;
  case PreProbing::FullProbing:
    return "FullProbing";
    break;
  case PreProbing::FullProbingAndRepeat:
    return "FullProbingAndRepeat";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, PreProbing e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class PresolveOperations : int {
      SingletonColumnRemoval = XPRS_PRESOLVEOPS_SINGLETONCOLUMNREMOVAL,
      SingletonRowRemoval = XPRS_PRESOLVEOPS_SINGLETONROWREMOVAL,
      ForcingRowRemoval = XPRS_PRESOLVEOPS_FORCINGROWREMOVAL,
      DualReductions = XPRS_PRESOLVEOPS_DUALREDUCTIONS,
      RedundantRowRemoval = XPRS_PRESOLVEOPS_REDUNDANTROWREMOVAL,
      DuplicateColumnRemoval = XPRS_PRESOLVEOPS_DUPLICATECOLUMNREMOVAL,
      DuplicateRowRemoval = XPRS_PRESOLVEOPS_DUPLICATEROWREMOVAL,
      StrongDualReductions = XPRS_PRESOLVEOPS_STRONGDUALREDUCTIONS,
      VariableEliminations = XPRS_PRESOLVEOPS_VARIABLEELIMINATIONS,
      NoIPReductions = XPRS_PRESOLVEOPS_NOIPREDUCTIONS,
      NoGlobalChange = XPRS_PRESOLVEOPS_NOGLOBALDOMAINCHANGE,
      NoAdvancedIPReductions = XPRS_PRESOLVEOPS_NOADVANCEDIPREDUCTIONS,
      NoIntegerEliminations = XPRS_PRESOLVEOPS_NOINTEGERELIMINATIONS,
      NoSolutionEnumeration = XPRS_PRESOLVEOPS_NOSOLUTIONENUMERATION,
      LinearlyDependantRowRemoval =
          XPRS_PRESOLVEOPS_LINEARLYDEPENDANTROWREMOVAL,
      NoIntegerVariableAndSOSDetection =
          XPRS_PRESOLVEOPS_NOINTEGERVARIABLEANDSOSDETECTION,
      NoImpliedBounds = XPRS_PRESOLVEOPS_NOIMPLIEDBOUNDS,
      NoCliquePresolve = XPRS_PRESOLVEOPS_NOCLIQUEPRESOLVE,
      NoMod2Reductions = XPRS_PRESOLVEOPS_NOMOD2REDUCTIONS,
      NoDualReductionsOnGlobals = XPRS_PRESOLVEOPS_NODUALREDONGLOBALS
    };
xpress_diags_pop inline bool operator==(PresolveOperations a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(PresolveOperations a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, PresolveOperations b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, PresolveOperations b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(PresolveOperations e) {

  switch (e) {
  case PresolveOperations::SingletonColumnRemoval:
    return "SingletonColumnRemoval";
    break;
  case PresolveOperations::SingletonRowRemoval:
    return "SingletonRowRemoval";
    break;
  case PresolveOperations::ForcingRowRemoval:
    return "ForcingRowRemoval";
    break;
  case PresolveOperations::DualReductions:
    return "DualReductions";
    break;
  case PresolveOperations::RedundantRowRemoval:
    return "RedundantRowRemoval";
    break;
  case PresolveOperations::DuplicateColumnRemoval:
    return "DuplicateColumnRemoval";
    break;
  case PresolveOperations::DuplicateRowRemoval:
    return "DuplicateRowRemoval";
    break;
  case PresolveOperations::StrongDualReductions:
    return "StrongDualReductions";
    break;
  case PresolveOperations::VariableEliminations:
    return "VariableEliminations";
    break;
  case PresolveOperations::NoIPReductions:
    return "NoIPReductions";
    break;
  case PresolveOperations::NoGlobalChange:
    return "NoGlobalChange";
    break;
  case PresolveOperations::NoAdvancedIPReductions:
    return "NoAdvancedIPReductions";
    break;
  case PresolveOperations::NoIntegerEliminations:
    return "NoIntegerEliminations";
    break;
  case PresolveOperations::NoSolutionEnumeration:
    return "NoSolutionEnumeration";
    break;
  case PresolveOperations::LinearlyDependantRowRemoval:
    return "LinearlyDependantRowRemoval";
    break;
  case PresolveOperations::NoIntegerVariableAndSOSDetection:
    return "NoIntegerVariableAndSOSDetection";
    break;
  case PresolveOperations::NoImpliedBounds:
    return "NoImpliedBounds";
    break;
  case PresolveOperations::NoCliquePresolve:
    return "NoCliquePresolve";
    break;
  case PresolveOperations::NoMod2Reductions:
    return "NoMod2Reductions";
    break;
  case PresolveOperations::NoDualReductionsOnGlobals:
    return "NoDualReductionsOnGlobals";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, PresolveOperations e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class PresolveState : int {
      ProblemLoaded = XPRS_PRESOLVESTATE_PROBLEMLOADED,
      ProblemLPPresolved = XPRS_PRESOLVESTATE_PROBLEMLPPRESOLVED,
      ProblemMIPPresolved = XPRS_PRESOLVESTATE_PROBLEMMIPPRESOLVED,
      SolutionValid = XPRS_PRESOLVESTATE_SOLUTIONVALID
    };
xpress_diags_pop inline bool operator==(PresolveState a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(PresolveState a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, PresolveState b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, PresolveState b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(PresolveState e) {

  switch (e) {
  case PresolveState::ProblemLoaded:
    return "ProblemLoaded";
    break;
  case PresolveState::ProblemLPPresolved:
    return "ProblemLPPresolved";
    break;
  case PresolveState::ProblemMIPPresolved:
    return "ProblemMIPPresolved";
    break;
  case PresolveState::SolutionValid:
    return "SolutionValid";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, PresolveState e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class MipPresolve : int {
      ReducedCostFixing = XPRS_MIPPRESOLVE_REDUCED_COST_FIXING,
      LogicPreprossessing = XPRS_MIPPRESOLVE_LOGIC_PREPROCESSING,
      AllowChangeBounds = XPRS_MIPPRESOLVE_ALLOW_CHANGE_BOUNDS,
      DualReductions = XPRS_MIPPRESOLVE_DUAL_REDUCTIONS,
      GlobalCoefficientTightening =
          XPRS_MIPPRESOLVE_GLOBAL_COEFFICIENT_TIGHTENING,
      ObjectiveBasedReductions = XPRS_MIPPRESOLVE_OBJECTIVE_BASED_REDUCTIONS,
      AllowTreeRestart = XPRS_MIPPRESOLVE_ALLOW_TREE_RESTART,
      SymmetryReductions = XPRS_MIPPRESOLVE_SYMMETRY_REDUCTIONS
    };
xpress_diags_pop inline bool operator==(MipPresolve a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(MipPresolve a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, MipPresolve b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, MipPresolve b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(MipPresolve e) {

  switch (e) {
  case MipPresolve::ReducedCostFixing:
    return "ReducedCostFixing";
    break;
  case MipPresolve::LogicPreprossessing:
    return "LogicPreprossessing";
    break;
  case MipPresolve::AllowChangeBounds:
    return "AllowChangeBounds";
    break;
  case MipPresolve::DualReductions:
    return "DualReductions";
    break;
  case MipPresolve::GlobalCoefficientTightening:
    return "GlobalCoefficientTightening";
    break;
  case MipPresolve::ObjectiveBasedReductions:
    return "ObjectiveBasedReductions";
    break;
  case MipPresolve::AllowTreeRestart:
    return "AllowTreeRestart";
    break;
  case MipPresolve::SymmetryReductions:
    return "SymmetryReductions";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, MipPresolve e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class Presolve : int {
      NoPrimalInfeasibility = XPRS_PRESOLVE_NOPRIMALINFEASIBILITY,
      None = XPRS_PRESOLVE_NONE,
      Default = XPRS_PRESOLVE_DEFAULT,
      KeepRedundantBounds = XPRS_PRESOLVE_KEEPREDUNDANTBOUNDS
    };
xpress_diags_pop inline bool operator==(Presolve a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(Presolve a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, Presolve b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, Presolve b) { return a != static_cast<int>(b); }
inline std::string to_string(Presolve e) {

  switch (e) {
  case Presolve::NoPrimalInfeasibility:
    return "NoPrimalInfeasibility";
    break;
  case Presolve::None:
    return "None";
    break;
  case Presolve::Default:
    return "Default";
    break;
  case Presolve::KeepRedundantBounds:
    return "KeepRedundantBounds";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, Presolve e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class Pricing : int {
      Partial = XPRS_PRICING_PARTIAL,
      Default = XPRS_PRICING_DEFAULT,
      Devex = XPRS_PRICING_DEVEX
    };
xpress_diags_pop inline bool operator==(Pricing a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(Pricing a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, Pricing b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, Pricing b) { return a != static_cast<int>(b); }
inline std::string to_string(Pricing e) {

  switch (e) {
  case Pricing::Partial:
    return "Partial";
    break;
  case Pricing::Default:
    return "Default";
    break;
  case Pricing::Devex:
    return "Devex";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, Pricing e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class CutStrategy : int {
      Default = XPRS_CUTSTRATEGY_DEFAULT,
      None = XPRS_CUTSTRATEGY_NONE,
      Conservative = XPRS_CUTSTRATEGY_CONSERVATIVE,
      Moderate = XPRS_CUTSTRATEGY_MODERATE,
      Aggressive = XPRS_CUTSTRATEGY_AGGRESSIVE
    };
xpress_diags_pop inline bool operator==(CutStrategy a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(CutStrategy a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, CutStrategy b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, CutStrategy b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(CutStrategy e) {

  switch (e) {
  case CutStrategy::Default:
    return "Default";
    break;
  case CutStrategy::None:
    return "None";
    break;
  case CutStrategy::Conservative:
    return "Conservative";
    break;
  case CutStrategy::Moderate:
    return "Moderate";
    break;
  case CutStrategy::Aggressive:
    return "Aggressive";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, CutStrategy e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class VariableSelection : int {
      Automatic = XPRS_VARSELECTION_AUTOMATIC,
      MinimumUpDownPseudoCosts = XPRS_VARSELECTION_MIN_UPDOWN_PSEUDO_COSTS,
      SumupDownPseudoCosts = XPRS_VARSELECTION_SUM_UPDOWN_PSEUDO_COSTS,
      MaximumUpDownPseudoCostsPlusTwiceMinimum =
          XPRS_VARSELECTION_MAX_UPDOWN_PSEUDO_COSTS_PLUS_TWICE_MIN,
      MaximumUpDownPseudoCosts = XPRS_VARSELECTION_MAX_UPDOWN_PSEUDO_COSTS,
      DownPseudoCost = XPRS_VARSELECTION_DOWN_PSEUDO_COST,
      UpPseudoCost = XPRS_VARSELECTION_UP_PSEUDO_COST
    };
xpress_diags_pop inline bool operator==(VariableSelection a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(VariableSelection a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, VariableSelection b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, VariableSelection b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(VariableSelection e) {

  switch (e) {
  case VariableSelection::Automatic:
    return "Automatic";
    break;
  case VariableSelection::MinimumUpDownPseudoCosts:
    return "MinimumUpDownPseudoCosts";
    break;
  case VariableSelection::SumupDownPseudoCosts:
    return "SumupDownPseudoCosts";
    break;
  case VariableSelection::MaximumUpDownPseudoCostsPlusTwiceMinimum:
    return "MaximumUpDownPseudoCostsPlusTwiceMinimum";
    break;
  case VariableSelection::MaximumUpDownPseudoCosts:
    return "MaximumUpDownPseudoCosts";
    break;
  case VariableSelection::DownPseudoCost:
    return "DownPseudoCost";
    break;
  case VariableSelection::UpPseudoCost:
    return "UpPseudoCost";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, VariableSelection e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class Scaling : int {
      RowScaling = XPRS_SCALING_ROW_SCALING,
      ColumnScaling = XPRS_SCALING_COLUMN_SCALING,
      RowScalingAgain = XPRS_SCALING_ROW_SCALING_AGAIN,
      Maximum = XPRS_SCALING_MAXIMUM,
      CurtisReid = XPRS_SCALING_CURTIS_REID,
      ByMaxElemNotGeoMean = XPRS_SCALING_BY_MAX_ELEM_NOT_GEO_MEAN,
      BigMScaling = XPRS_SCALING_BIGM,
      SimplexObjectiveScaling = XPRS_SCALING_SIMPLEX_OBJECTIVE_SCALING,
      IgnoreQuadraticRowPart = XPRS_SCALING_IGNORE_QUADRATIC_ROW_PART,
      ScaleBeforePresolve = XPRS_SCALING_BEFORE_PRESOLVE,
      NoScalingRowsUp = XPRS_SCALING_NO_SCALING_ROWS_UP,
      NoScalingColumnsDown = XPRS_SCALING_NO_SCALING_COLUMNS_DOWN,
      ObjectiveScaling = XPRS_SCALING_DISABLE_GLOBAL_OBJECTIVE_SCALING,
      RhsScaling = XPRS_SCALING_RHS_SCALING,
      NoAggressiveQScaling = XPRS_SCALING_NO_AGGRESSIVE_Q_SCALING,
      SlackScaling = XPRS_SCALING_SLACK_SCALING,
      Ruiz = XPRS_SCALING_RUIZ,
      DogLeg = XPRS_SCALING_DOGLEG,
      BeforeAfterPresolve = XPRS_SCALING_BEFORE_AND_AFTER_PRESOLVE
    };
xpress_diags_pop inline bool operator==(Scaling a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(Scaling a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, Scaling b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, Scaling b) { return a != static_cast<int>(b); }
inline std::string to_string(Scaling e) {

  switch (e) {
  case Scaling::RowScaling:
    return "RowScaling";
    break;
  case Scaling::ColumnScaling:
    return "ColumnScaling";
    break;
  case Scaling::RowScalingAgain:
    return "RowScalingAgain";
    break;
  case Scaling::Maximum:
    return "Maximum";
    break;
  case Scaling::CurtisReid:
    return "CurtisReid";
    break;
  case Scaling::ByMaxElemNotGeoMean:
    return "ByMaxElemNotGeoMean";
    break;
  case Scaling::BigMScaling:
    return "BigMScaling";
    break;
  case Scaling::SimplexObjectiveScaling:
    return "SimplexObjectiveScaling";
    break;
  case Scaling::IgnoreQuadraticRowPart:
    return "IgnoreQuadraticRowPart";
    break;
  case Scaling::ScaleBeforePresolve:
    return "ScaleBeforePresolve";
    break;
  case Scaling::NoScalingRowsUp:
    return "NoScalingRowsUp";
    break;
  case Scaling::NoScalingColumnsDown:
    return "NoScalingColumnsDown";
    break;
  case Scaling::ObjectiveScaling:
    return "ObjectiveScaling";
    break;
  case Scaling::RhsScaling:
    return "RhsScaling";
    break;
  case Scaling::NoAggressiveQScaling:
    return "NoAggressiveQScaling";
    break;
  case Scaling::SlackScaling:
    return "SlackScaling";
    break;
  case Scaling::Ruiz:
    return "Ruiz";
    break;
  case Scaling::DogLeg:
    return "DogLeg";
    break;
  case Scaling::BeforeAfterPresolve:
    return "BeforeAfterPresolve";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, Scaling e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class CutSelect : int {
      Clique = XPRS_CUTSELECT_CLIQUE,
      Mir = XPRS_CUTSELECT_MIR,
      Cover = XPRS_CUTSELECT_COVER,
      Flowpath = XPRS_CUTSELECT_FLOWPATH,
      Implication = XPRS_CUTSELECT_IMPLICATION,
      LiftAndProject = XPRS_CUTSELECT_LIFT_AND_PROJECT,
      DisableCutRows = XPRS_CUTSELECT_DISABLE_CUT_ROWS,
      GubCover = XPRS_CUTSELECT_GUB_COVER,
      Default = XPRS_CUTSELECT_DEFAULT
    };
xpress_diags_pop inline bool operator==(CutSelect a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(CutSelect a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, CutSelect b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, CutSelect b) { return a != static_cast<int>(b); }
inline std::string to_string(CutSelect e) {

  switch (e) {
  case CutSelect::Clique:
    return "Clique";
    break;
  case CutSelect::Mir:
    return "Mir";
    break;
  case CutSelect::Cover:
    return "Cover";
    break;
  case CutSelect::Flowpath:
    return "Flowpath";
    break;
  case CutSelect::Implication:
    return "Implication";
    break;
  case CutSelect::LiftAndProject:
    return "LiftAndProject";
    break;
  case CutSelect::DisableCutRows:
    return "DisableCutRows";
    break;
  case CutSelect::GubCover:
    return "GubCover";
    break;
  case CutSelect::Default:
    return "Default";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, CutSelect e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class RefineOps : int {
      LPOptimal = XPRS_REFINEOPS_LPOPTIMAL,
      MIPSolution = XPRS_REFINEOPS_MIPSOLUTION,
      LP = XPRS_REFINEOPS_MIPNODELP,
      LPPresolve = XPRS_REFINEOPS_LPPRESOLVE,
      IterativeRefiner = XPRS_REFINEOPS_ITERATIVEREFINER,
      RefinerPrecision = XPRS_REFINEOPS_REFINERPRECISION,
      RefinerUsePrimal = XPRS_REFINEOPS_REFINERUSEPRIMAL,
      RefinerUseDual = XPRS_REFINEOPS_REFINERUSEDUAL,
      MIPfixglobals = XPRS_REFINEOPS_MIPFIXGLOBALS,
      MIPfixglobalstarget = XPRS_REFINEOPS_MIPFIXGLOBALSTARGET
    };
xpress_diags_pop inline bool operator==(RefineOps a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(RefineOps a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, RefineOps b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, RefineOps b) { return a != static_cast<int>(b); }
inline std::string to_string(RefineOps e) {

  switch (e) {
  case RefineOps::LPOptimal:
    return "LPOptimal";
    break;
  case RefineOps::MIPSolution:
    return "MIPSolution";
    break;
  case RefineOps::LP:
    return "LP";
    break;
  case RefineOps::LPPresolve:
    return "LPPresolve";
    break;
  case RefineOps::IterativeRefiner:
    return "IterativeRefiner";
    break;
  case RefineOps::RefinerPrecision:
    return "RefinerPrecision";
    break;
  case RefineOps::RefinerUsePrimal:
    return "RefinerUsePrimal";
    break;
  case RefineOps::RefinerUseDual:
    return "RefinerUseDual";
    break;
  case RefineOps::MIPfixglobals:
    return "MIPfixglobals";
    break;
  case RefineOps::MIPfixglobalstarget:
    return "MIPfixglobalstarget";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, RefineOps e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class DualizeOps : int {
      SwitchAlgorithm = XPRS_DUALIZEOPS_SWITCHALGORITHM
    };
xpress_diags_pop inline bool operator==(DualizeOps a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(DualizeOps a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, DualizeOps b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, DualizeOps b) { return a != static_cast<int>(b); }
inline std::string to_string(DualizeOps e) {

  switch (e) {
  case DualizeOps::SwitchAlgorithm:
    return "SwitchAlgorithm";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, DualizeOps e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class TreeDiagnostics : int {
      MemoryUsageSummaries = XPRS_TREEDIAGNOSTICS_MEMORY_USAGE_SUMMARIES,
      MemorySavedReports = XPRS_TREEDIAGNOSTICS_MEMORY_SAVED_REPORTS
    };
xpress_diags_pop inline bool operator==(TreeDiagnostics a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(TreeDiagnostics a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, TreeDiagnostics b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, TreeDiagnostics b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(TreeDiagnostics e) {

  switch (e) {
  case TreeDiagnostics::MemoryUsageSummaries:
    return "MemoryUsageSummaries";
    break;
  case TreeDiagnostics::MemorySavedReports:
    return "MemorySavedReports";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, TreeDiagnostics e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class BarPresolveOps : int {
      StandardPresolve = XPRS_BARPRESOLVEOPS_STANDARD_PRESOLVE,
      ExtraBarrierPresolve = XPRS_BARPRESOLVEOPS_EXTRA_BARRIER_PRESOLVE
    };
xpress_diags_pop inline bool operator==(BarPresolveOps a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(BarPresolveOps a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, BarPresolveOps b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, BarPresolveOps b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(BarPresolveOps e) {

  switch (e) {
  case BarPresolveOps::StandardPresolve:
    return "StandardPresolve";
    break;
  case BarPresolveOps::ExtraBarrierPresolve:
    return "ExtraBarrierPresolve";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, BarPresolveOps e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class MipRestart : int {
      Default = XPRS_MIPRESTART_DEFAULT,
      Off = XPRS_MIPRESTART_OFF,
      Moderate = XPRS_MIPRESTART_MODERATE,
      Aggressive = XPRS_MIPRESTART_AGGRESSIVE
    };
xpress_diags_pop inline bool operator==(MipRestart a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(MipRestart a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, MipRestart b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, MipRestart b) { return a != static_cast<int>(b); }
inline std::string to_string(MipRestart e) {

  switch (e) {
  case MipRestart::Default:
    return "Default";
    break;
  case MipRestart::Off:
    return "Off";
    break;
  case MipRestart::Moderate:
    return "Moderate";
    break;
  case MipRestart::Aggressive:
    return "Aggressive";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, MipRestart e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class PresolveCoefElim : int {
      Disabled = XPRS_PRECOEFELIM_DISABLED,
      Aggressive = XPRS_PRECOEFELIM_AGGRESSIVE,
      Cautious = XPRS_PRECOEFELIM_CAUTIOUS
    };
xpress_diags_pop inline bool operator==(PresolveCoefElim a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(PresolveCoefElim a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, PresolveCoefElim b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, PresolveCoefElim b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(PresolveCoefElim e) {

  switch (e) {
  case PresolveCoefElim::Disabled:
    return "Disabled";
    break;
  case PresolveCoefElim::Aggressive:
    return "Aggressive";
    break;
  case PresolveCoefElim::Cautious:
    return "Cautious";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, PresolveCoefElim e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class PresolveDomRow : int {
      Automatic = XPRS_PREDOMROW_AUTOMATIC,
      Disabled = XPRS_PREDOMROW_DISABLED,
      Cautious = XPRS_PREDOMROW_CAUTIOUS,
      Medium = XPRS_PREDOMROW_MEDIUM,
      Aggressive = XPRS_PREDOMROW_AGGRESSIVE
    };
xpress_diags_pop inline bool operator==(PresolveDomRow a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(PresolveDomRow a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, PresolveDomRow b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, PresolveDomRow b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(PresolveDomRow e) {

  switch (e) {
  case PresolveDomRow::Automatic:
    return "Automatic";
    break;
  case PresolveDomRow::Disabled:
    return "Disabled";
    break;
  case PresolveDomRow::Cautious:
    return "Cautious";
    break;
  case PresolveDomRow::Medium:
    return "Medium";
    break;
  case PresolveDomRow::Aggressive:
    return "Aggressive";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, PresolveDomRow e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class PresolveDomColumn : int {
      Automatic = XPRS_PREDOMCOL_AUTOMATIC,
      Disabled = XPRS_PREDOMCOL_DISABLED,
      Cautious = XPRS_PREDOMCOL_CAUTIOUS,
      Aggressive = XPRS_PREDOMCOL_AGGRESSIVE
    };
xpress_diags_pop inline bool operator==(PresolveDomColumn a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(PresolveDomColumn a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, PresolveDomColumn b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, PresolveDomColumn b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(PresolveDomColumn e) {

  switch (e) {
  case PresolveDomColumn::Automatic:
    return "Automatic";
    break;
  case PresolveDomColumn::Disabled:
    return "Disabled";
    break;
  case PresolveDomColumn::Cautious:
    return "Cautious";
    break;
  case PresolveDomColumn::Aggressive:
    return "Aggressive";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, PresolveDomColumn e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class PrimalUnshift : int {
      AllowDualUnshift = XPRS_PRIMALUNSHIFT_ALLOW_DUAL_UNSHIFT,
      NoDualUnshift = XPRS_PRIMALUNSHIFT_NO_DUAL_UNSHIFT
    };
xpress_diags_pop inline bool operator==(PrimalUnshift a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(PrimalUnshift a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, PrimalUnshift b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, PrimalUnshift b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(PrimalUnshift e) {

  switch (e) {
  case PrimalUnshift::AllowDualUnshift:
    return "AllowDualUnshift";
    break;
  case PrimalUnshift::NoDualUnshift:
    return "NoDualUnshift";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, PrimalUnshift e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class RepairIndefiniteQuadratic : int {
      RepairIfPossible = XPRS_REPAIRINDEFINITEQ_REPAIR_IF_POSSIBLE,
      NoRepair = XPRS_REPAIRINDEFINITEQ_NO_REPAIR
    };
xpress_diags_pop inline bool operator==(RepairIndefiniteQuadratic a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(RepairIndefiniteQuadratic a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, RepairIndefiniteQuadratic b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, RepairIndefiniteQuadratic b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(RepairIndefiniteQuadratic e) {

  switch (e) {
  case RepairIndefiniteQuadratic::RepairIfPossible:
    return "RepairIfPossible";
    break;
  case RepairIndefiniteQuadratic::NoRepair:
    return "NoRepair";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, RepairIndefiniteQuadratic e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class ObjSense : int {
      Minimize = XPRS_OBJ_MINIMIZE,
      Maximize = XPRS_OBJ_MAXIMIZE
    };
xpress_diags_pop inline bool operator==(ObjSense a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(ObjSense a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, ObjSense b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, ObjSense b) { return a != static_cast<int>(b); }
inline std::string to_string(ObjSense e) {

  switch (e) {
  case ObjSense::Minimize:
    return "Minimize";
    break;
  case ObjSense::Maximize:
    return "Maximize";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, ObjSense e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class ParameterType : int {
      NotDefined = XPRS_TYPE_NOTDEFINED,
      Integer = XPRS_TYPE_INT,
      Integer64 = XPRS_TYPE_INT64,
      Double = XPRS_TYPE_DOUBLE,
      String = XPRS_TYPE_STRING
    };
xpress_diags_pop inline bool operator==(ParameterType a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(ParameterType a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, ParameterType b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, ParameterType b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(ParameterType e) {

  switch (e) {
  case ParameterType::NotDefined:
    return "NotDefined";
    break;
  case ParameterType::Integer:
    return "Integer";
    break;
  case ParameterType::Integer64:
    return "Integer64";
    break;
  case ParameterType::Double:
    return "Double";
    break;
  case ParameterType::String:
    return "String";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, ParameterType e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class QConvexity : int {
      Unknown = XPRS_QCONVEXITY_UNKNOWN,
      NonConvex = XPRS_QCONVEXITY_NONCONVEX,
      Convex = XPRS_QCONVEXITY_CONVEX,
      Repairable = XPRS_QCONVEXITY_REPAIRABLE,
      ConvexCone = XPRS_QCONVEXITY_CONVEXCONE,
      ConeConvertable = XPRS_QCONVEXITY_CONECONVERTABLE
    };
xpress_diags_pop inline bool operator==(QConvexity a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(QConvexity a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, QConvexity b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, QConvexity b) { return a != static_cast<int>(b); }
inline std::string to_string(QConvexity e) {

  switch (e) {
  case QConvexity::Unknown:
    return "Unknown";
    break;
  case QConvexity::NonConvex:
    return "NonConvex";
    break;
  case QConvexity::Convex:
    return "Convex";
    break;
  case QConvexity::Repairable:
    return "Repairable";
    break;
  case QConvexity::ConvexCone:
    return "ConvexCone";
    break;
  case QConvexity::ConeConvertable:
    return "ConeConvertable";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, QConvexity e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class SolInfo : int {
      AbsPrimalInfeas = XPRS_SOLINFO_ABSPRIMALINFEAS,
      RelPrimalInfeas = XPRS_SOLINFO_RELPRIMALINFEAS,
      AbsDualInfeas = XPRS_SOLINFO_ABSDUALINFEAS,
      RelDualInfeas = XPRS_SOLINFO_RELDUALINFEAS,
      MaxMIPFractional = XPRS_SOLINFO_MAXMIPFRACTIONAL,
      AbsMIPInfeas = XPRS_SOLINFO_ABSMIPINFEAS,
      RelMIPInfeas = XPRS_SOLINFO_RELMIPINFEAS
    };
xpress_diags_pop inline bool operator==(SolInfo a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(SolInfo a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, SolInfo b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, SolInfo b) { return a != static_cast<int>(b); }
inline std::string to_string(SolInfo e) {

  switch (e) {
  case SolInfo::AbsPrimalInfeas:
    return "AbsPrimalInfeas";
    break;
  case SolInfo::RelPrimalInfeas:
    return "RelPrimalInfeas";
    break;
  case SolInfo::AbsDualInfeas:
    return "AbsDualInfeas";
    break;
  case SolInfo::RelDualInfeas:
    return "RelDualInfeas";
    break;
  case SolInfo::MaxMIPFractional:
    return "MaxMIPFractional";
    break;
  case SolInfo::AbsMIPInfeas:
    return "AbsMIPInfeas";
    break;
  case SolInfo::RelMIPInfeas:
    return "RelMIPInfeas";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, SolInfo e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class TunerMode : int {
      Automatic = XPRS_TUNERMODE_AUTOMATIC,
      Off = XPRS_TUNERMODE_OFF,
      On = XPRS_TUNERMODE_ON
    };
xpress_diags_pop inline bool operator==(TunerMode a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(TunerMode a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, TunerMode b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, TunerMode b) { return a != static_cast<int>(b); }
inline std::string to_string(TunerMode e) {

  switch (e) {
  case TunerMode::Automatic:
    return "Automatic";
    break;
  case TunerMode::Off:
    return "Off";
    break;
  case TunerMode::On:
    return "On";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, TunerMode e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class TunerMethod : int {
      Automatic = XPRS_TUNERMETHOD_AUTOMATIC,
      LpQuick = XPRS_TUNERMETHOD_LPQUICK,
      MipQuick = XPRS_TUNERMETHOD_MIPQUICK,
      MipComprehensive = XPRS_TUNERMETHOD_MIPCOMPREHENSIVE,
      MipRootFocus = XPRS_TUNERMETHOD_MIPROOTFOCUS,
      MipTreeFocus = XPRS_TUNERMETHOD_MIPTREEFOCUS,
      MipSimple = XPRS_TUNERMETHOD_MIPSIMPLE,
      SlpQuick = XPRS_TUNERMETHOD_SLPQUICK,
      MiSlpQuick = XPRS_TUNERMETHOD_MISLPQUICK,
      MipHeuristics = XPRS_TUNERMETHOD_MIPHEURISTICS,
      GlobalQuick = XPRS_TUNERMETHOD_GLOBALQUICK,
      LpNumerics = XPRS_TUNERMETHOD_LPNUMERICS
    };
xpress_diags_pop inline bool operator==(TunerMethod a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(TunerMethod a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, TunerMethod b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, TunerMethod b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(TunerMethod e) {

  switch (e) {
  case TunerMethod::Automatic:
    return "Automatic";
    break;
  case TunerMethod::LpQuick:
    return "LpQuick";
    break;
  case TunerMethod::MipQuick:
    return "MipQuick";
    break;
  case TunerMethod::MipComprehensive:
    return "MipComprehensive";
    break;
  case TunerMethod::MipRootFocus:
    return "MipRootFocus";
    break;
  case TunerMethod::MipTreeFocus:
    return "MipTreeFocus";
    break;
  case TunerMethod::MipSimple:
    return "MipSimple";
    break;
  case TunerMethod::SlpQuick:
    return "SlpQuick";
    break;
  case TunerMethod::MiSlpQuick:
    return "MiSlpQuick";
    break;
  case TunerMethod::MipHeuristics:
    return "MipHeuristics";
    break;
  case TunerMethod::GlobalQuick:
    return "GlobalQuick";
    break;
  case TunerMethod::LpNumerics:
    return "LpNumerics";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, TunerMethod e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class TunerTarget : int {
      Automatic = XPRS_TUNERTARGET_AUTOMATIC,
      TimeGap = XPRS_TUNERTARGET_TIMEGAP,
      TimeBound = XPRS_TUNERTARGET_TIMEBOUND,
      TimeObjval = XPRS_TUNERTARGET_TIMEOBJVAL,
      Integral = XPRS_TUNERTARGET_INTEGRAL,
      SlpTime = XPRS_TUNERTARGET_SLPTIME,
      SlpObjval = XPRS_TUNERTARGET_SLPOBJVAL,
      SlpValidation = XPRS_TUNERTARGET_SLPVALIDATION,
      Gap = XPRS_TUNERTARGET_GAP,
      Bound = XPRS_TUNERTARGET_BOUND,
      Objval = XPRS_TUNERTARGET_OBJVAL,
      PrimalIntegral = XPRS_TUNERTARGET_PRIMALINTEGRAL
    };
xpress_diags_pop inline bool operator==(TunerTarget a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(TunerTarget a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, TunerTarget b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, TunerTarget b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(TunerTarget e) {

  switch (e) {
  case TunerTarget::Automatic:
    return "Automatic";
    break;
  case TunerTarget::TimeGap:
    return "TimeGap";
    break;
  case TunerTarget::TimeBound:
    return "TimeBound";
    break;
  case TunerTarget::TimeObjval:
    return "TimeObjval";
    break;
  case TunerTarget::Integral:
    return "Integral";
    break;
  case TunerTarget::SlpTime:
    return "SlpTime";
    break;
  case TunerTarget::SlpObjval:
    return "SlpObjval";
    break;
  case TunerTarget::SlpValidation:
    return "SlpValidation";
    break;
  case TunerTarget::Gap:
    return "Gap";
    break;
  case TunerTarget::Bound:
    return "Bound";
    break;
  case TunerTarget::Objval:
    return "Objval";
    break;
  case TunerTarget::PrimalIntegral:
    return "PrimalIntegral";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, TunerTarget e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class TunerHistory : int {
      Ignore = XPRS_TUNERHISTORY_IGNORE,
      Append = XPRS_TUNERHISTORY_APPEND,
      Reuse = XPRS_TUNERHISTORY_REUSE
    };
xpress_diags_pop inline bool operator==(TunerHistory a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(TunerHistory a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, TunerHistory b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, TunerHistory b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(TunerHistory e) {

  switch (e) {
  case TunerHistory::Ignore:
    return "Ignore";
    break;
  case TunerHistory::Append:
    return "Append";
    break;
  case TunerHistory::Reuse:
    return "Reuse";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, TunerHistory e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class TunerRootAlg : int {
      Dual = XPRS_TUNERROOTALG_DUAL,
      Primal = XPRS_TUNERROOTALG_PRIMAL,
      Barrier = XPRS_TUNERROOTALG_BARRIER,
      Network = XPRS_TUNERROOTALG_NETWORK
    };
xpress_diags_pop inline bool operator==(TunerRootAlg a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(TunerRootAlg a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, TunerRootAlg b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, TunerRootAlg b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(TunerRootAlg e) {

  switch (e) {
  case TunerRootAlg::Dual:
    return "Dual";
    break;
  case TunerRootAlg::Primal:
    return "Primal";
    break;
  case TunerRootAlg::Barrier:
    return "Barrier";
    break;
  case TunerRootAlg::Network:
    return "Network";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, TunerRootAlg e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class LPFlags : int {
      Dual = XPRS_LPFLAGS_DUAL,
      Primal = XPRS_LPFLAGS_PRIMAL,
      Barrier = XPRS_LPFLAGS_BARRIER,
      Network = XPRS_LPFLAGS_NETWORK
    };
xpress_diags_pop inline bool operator==(LPFlags a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(LPFlags a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, LPFlags b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, LPFlags b) { return a != static_cast<int>(b); }
inline std::string to_string(LPFlags e) {

  switch (e) {
  case LPFlags::Dual:
    return "Dual";
    break;
  case LPFlags::Primal:
    return "Primal";
    break;
  case LPFlags::Barrier:
    return "Barrier";
    break;
  case LPFlags::Network:
    return "Network";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, LPFlags e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class GenConsType : int {
      Max = XPRS_GENCONS_MAX,
      Min = XPRS_GENCONS_MIN,
      And = XPRS_GENCONS_AND,
      Or = XPRS_GENCONS_OR,
      Abs = XPRS_GENCONS_ABS
    };
xpress_diags_pop inline bool operator==(GenConsType a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(GenConsType a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, GenConsType b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, GenConsType b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(GenConsType e) {

  switch (e) {
  case GenConsType::Max:
    return "Max";
    break;
  case GenConsType::Min:
    return "Min";
    break;
  case GenConsType::And:
    return "And";
    break;
  case GenConsType::Or:
    return "Or";
    break;
  case GenConsType::Abs:
    return "Abs";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, GenConsType e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class Clamping : int {
      Primal = XPRS_CLAMPING_PRIMAL,
      Dual = XPRS_CLAMPING_DUAL,
      Slacks = XPRS_CLAMPING_SLACKS,
      RDJ = XPRS_CLAMPING_RDJ
    };
xpress_diags_pop inline bool operator==(Clamping a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(Clamping a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, Clamping b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, Clamping b) { return a != static_cast<int>(b); }
inline std::string to_string(Clamping e) {

  switch (e) {
  case Clamping::Primal:
    return "Primal";
    break;
  case Clamping::Dual:
    return "Dual";
    break;
  case Clamping::Slacks:
    return "Slacks";
    break;
  case Clamping::RDJ:
    return "RDJ";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, Clamping e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class RowFlag : int {
      Quadratic = XPRS_ROWFLAG_QUADRATIC,
      Delayed = XPRS_ROWFLAG_DELAYED,
      ModelCut = XPRS_ROWFLAG_MODELCUT,
      Indicator = XPRS_ROWFLAG_INDICATOR,
      Nonlinear = XPRS_ROWFLAG_NONLINEAR
    };
xpress_diags_pop inline bool operator==(RowFlag a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(RowFlag a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, RowFlag b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, RowFlag b) { return a != static_cast<int>(b); }
inline std::string to_string(RowFlag e) {

  switch (e) {
  case RowFlag::Quadratic:
    return "Quadratic";
    break;
  case RowFlag::Delayed:
    return "Delayed";
    break;
  case RowFlag::ModelCut:
    return "ModelCut";
    break;
  case RowFlag::Indicator:
    return "Indicator";
    break;
  case RowFlag::Nonlinear:
    return "Nonlinear";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, RowFlag e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class ObjControl : int {
      Priority = XPRS_OBJECTIVE_PRIORITY,
      Weight = XPRS_OBJECTIVE_WEIGHT,
      AbsTol = XPRS_OBJECTIVE_ABSTOL,
      RelTol = XPRS_OBJECTIVE_RELTOL,
      Rhs = XPRS_OBJECTIVE_RHS
    };
xpress_diags_pop inline bool operator==(ObjControl a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(ObjControl a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, ObjControl b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, ObjControl b) { return a != static_cast<int>(b); }
inline std::string to_string(ObjControl e) {

  switch (e) {
  case ObjControl::Priority:
    return "Priority";
    break;
  case ObjControl::Weight:
    return "Weight";
    break;
  case ObjControl::AbsTol:
    return "AbsTol";
    break;
  case ObjControl::RelTol:
    return "RelTol";
    break;
  case ObjControl::Rhs:
    return "Rhs";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, ObjControl e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class AllowCompute : int {
      Always = XPRS_ALLOW_COMPUTE_ALWAYS,
      Never = XPRS_ALLOW_COMPUTE_NEVER,
      Default = XPRS_ALLOW_COMPUTE_DEFAULT
    };
xpress_diags_pop inline bool operator==(AllowCompute a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(AllowCompute a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, AllowCompute b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, AllowCompute b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(AllowCompute e) {

  switch (e) {
  case AllowCompute::Always:
    return "Always";
    break;
  case AllowCompute::Never:
    return "Never";
    break;
  case AllowCompute::Default:
    return "Default";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, AllowCompute e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class ComputeLog : int {
      Never = XPRS_COMPUTELOG_NEVER,
      RealTime = XPRS_COMPUTELOG_REALTIME,
      OnCompletion = XPRS_COMPUTELOG_ONCOMPLETION,
      OnError = XPRS_COMPUTELOG_ONERROR
    };
xpress_diags_pop inline bool operator==(ComputeLog a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(ComputeLog a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, ComputeLog b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, ComputeLog b) { return a != static_cast<int>(b); }
inline std::string to_string(ComputeLog e) {

  switch (e) {
  case ComputeLog::Never:
    return "Never";
    break;
  case ComputeLog::RealTime:
    return "RealTime";
    break;
  case ComputeLog::OnCompletion:
    return "OnCompletion";
    break;
  case ComputeLog::OnError:
    return "OnError";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, ComputeLog e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class Namespaces : int {
      Row = XPRS_NAMES_ROW,
      Column = XPRS_NAMES_COLUMN,
      Set = XPRS_NAMES_SET,
      PwlCons = XPRS_NAMES_PWLCONS,
      GenCons = XPRS_NAMES_GENCONS,
      Objective = XPRS_NAMES_OBJECTIVE,
      Userfunc = XPRS_NAMES_USERFUNC,
      InternalFunc = XPRS_NAMES_INTERNALFUNC,
      UserFuncNoCase = XPRS_NAMES_USERFUNCNOCASE,
      InternalFuncNoCase = XPRS_NAMES_INTERNALFUNCNOCASE
    };
xpress_diags_pop inline bool operator==(Namespaces a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(Namespaces a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, Namespaces b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, Namespaces b) { return a != static_cast<int>(b); }
inline std::string to_string(Namespaces e) {

  switch (e) {
  case Namespaces::Row:
    return "Row";
    break;
  case Namespaces::Column:
    return "Column";
    break;
  case Namespaces::Set:
    return "Set";
    break;
  case Namespaces::PwlCons:
    return "PwlCons";
    break;
  case Namespaces::GenCons:
    return "GenCons";
    break;
  case Namespaces::Objective:
    return "Objective";
    break;
  case Namespaces::Userfunc:
    return "Userfunc";
    break;
  case Namespaces::InternalFunc:
    return "InternalFunc";
    break;
  case Namespaces::UserFuncNoCase:
    return "UserFuncNoCase";
    break;
  case Namespaces::InternalFuncNoCase:
    return "InternalFuncNoCase";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, Namespaces e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class Globalboundingbox : int {
      NotApplied = XPRS_GLOBALBOUNDINGBOX_NOT_APPLIED,
      Original = XPRS_GLOBALBOUNDINGBOX_ORIGINAL,
      Auxiliary = XPRS_GLOBALBOUNDINGBOX_AUXILIARY
    };
xpress_diags_pop inline bool operator==(Globalboundingbox a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(Globalboundingbox a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, Globalboundingbox b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, Globalboundingbox b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(Globalboundingbox e) {

  switch (e) {
  case Globalboundingbox::NotApplied:
    return "NotApplied";
    break;
  case Globalboundingbox::Original:
    return "Original";
    break;
  case Globalboundingbox::Auxiliary:
    return "Auxiliary";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, Globalboundingbox e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class MultiObjOps : int {
      Enabled = XPRS_MULTIOBJOPS_ENABLED,
      Presolve = XPRS_MULTIOBJOPS_PRESOLVE,
      RCFixing = XPRS_MULTIOBJOPS_RCFIXING
    };
xpress_diags_pop inline bool operator==(MultiObjOps a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(MultiObjOps a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, MultiObjOps b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, MultiObjOps b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(MultiObjOps e) {

  switch (e) {
  case MultiObjOps::Enabled:
    return "Enabled";
    break;
  case MultiObjOps::Presolve:
    return "Presolve";
    break;
  case MultiObjOps::RCFixing:
    return "RCFixing";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, MultiObjOps e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class IISOps : int {
      Binary = XPRS_IISOPS_BINARY,
      ZeroLower = XPRS_IISOPS_ZEROLOWER,
      FixedVar = XPRS_IISOPS_FIXEDVAR,
      Bound = XPRS_IISOPS_BOUND,
      GenIntegrality = XPRS_IISOPS_GENINTEGRALITY,
      Integrality = XPRS_IISOPS_INTEGRALITY,
      Variable = XPRS_IISOPS_VARIABLE,
      Equality = XPRS_IISOPS_EQUALITY,
      General = XPRS_IISOPS_GENERAL,
      PwL = XPRS_IISOPS_PWL,
      SET = XPRS_IISOPS_SET,
      Indicator = XPRS_IISOPS_INDICATOR,
      Delayed = XPRS_IISOPS_DELAYED,
      Constraint = XPRS_IISOPS_CONSTRAINT
    };
xpress_diags_pop inline bool operator==(IISOps a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(IISOps a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, IISOps b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, IISOps b) { return a != static_cast<int>(b); }
inline std::string to_string(IISOps e) {

  switch (e) {
  case IISOps::Binary:
    return "Binary";
    break;
  case IISOps::ZeroLower:
    return "ZeroLower";
    break;
  case IISOps::FixedVar:
    return "FixedVar";
    break;
  case IISOps::Bound:
    return "Bound";
    break;
  case IISOps::GenIntegrality:
    return "GenIntegrality";
    break;
  case IISOps::Integrality:
    return "Integrality";
    break;
  case IISOps::Variable:
    return "Variable";
    break;
  case IISOps::Equality:
    return "Equality";
    break;
  case IISOps::General:
    return "General";
    break;
  case IISOps::PwL:
    return "PwL";
    break;
  case IISOps::SET:
    return "SET";
    break;
  case IISOps::Indicator:
    return "Indicator";
    break;
  case IISOps::Delayed:
    return "Delayed";
    break;
  case IISOps::Constraint:
    return "Constraint";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, IISOps e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class UserSolStatus : int {
      NotChecked = XPRS_USERSOLSTATUS_NOT_CHECKED,
      AcceptedFeasible = XPRS_USERSOLSTATUS_ACCEPTED_FEASIBLE,
      AcceptedOptimized = XPRS_USERSOLSTATUS_ACCEPTED_OPTIMIZED,
      SearchedSol = XPRS_USERSOLSTATUS_SEARCHED_SOL,
      SearchedNoSol = XPRS_USERSOLSTATUS_SEARCHED_NOSOL,
      RejectedInfeasNoSearch = XPRS_USERSOLSTATUS_REJECTED_INFEAS_NOSEARCH,
      RejectedPartialNoSearch = XPRS_USERSOLSTATUS_REJECTED_PARTIAL_NOSEARCH,
      RejectedFailedOptimize = XPRS_USERSOLSTATUS_REJECTED_FAILED_OPTIMIZE,
      Dropped = XPRS_USERSOLSTATUS_DROPPED,
      RejectedCutoff = XPRS_USERSOLSTATUS_REJECTED_CUTOFF
    };
xpress_diags_pop inline bool operator==(UserSolStatus a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(UserSolStatus a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, UserSolStatus b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, UserSolStatus b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(UserSolStatus e) {

  switch (e) {
  case UserSolStatus::NotChecked:
    return "NotChecked";
    break;
  case UserSolStatus::AcceptedFeasible:
    return "AcceptedFeasible";
    break;
  case UserSolStatus::AcceptedOptimized:
    return "AcceptedOptimized";
    break;
  case UserSolStatus::SearchedSol:
    return "SearchedSol";
    break;
  case UserSolStatus::SearchedNoSol:
    return "SearchedNoSol";
    break;
  case UserSolStatus::RejectedInfeasNoSearch:
    return "RejectedInfeasNoSearch";
    break;
  case UserSolStatus::RejectedPartialNoSearch:
    return "RejectedPartialNoSearch";
    break;
  case UserSolStatus::RejectedFailedOptimize:
    return "RejectedFailedOptimize";
    break;
  case UserSolStatus::Dropped:
    return "Dropped";
    break;
  case UserSolStatus::RejectedCutoff:
    return "RejectedCutoff";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, UserSolStatus e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class GlobalLSHEURStrategy : int {
      Default = XPRS_GLOBALLSHEURSTRATEGY_DEFAULT,
      None = XPRS_GLOBALLSHEURSTRATEGY_NONE,
      Conservative = XPRS_GLOBALLSHEURSTRATEGY_CONSERVATIVE,
      Moderate = XPRS_GLOBALLSHEURSTRATEGY_MODERATE,
      Aggressive = XPRS_GLOBALLSHEURSTRATEGY_AGGRESSIVE
    };
xpress_diags_pop inline bool operator==(GlobalLSHEURStrategy a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(GlobalLSHEURStrategy a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, GlobalLSHEURStrategy b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, GlobalLSHEURStrategy b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(GlobalLSHEURStrategy e) {

  switch (e) {
  case GlobalLSHEURStrategy::Default:
    return "Default";
    break;
  case GlobalLSHEURStrategy::None:
    return "None";
    break;
  case GlobalLSHEURStrategy::Conservative:
    return "Conservative";
    break;
  case GlobalLSHEURStrategy::Moderate:
    return "Moderate";
    break;
  case GlobalLSHEURStrategy::Aggressive:
    return "Aggressive";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, GlobalLSHEURStrategy e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class BARHGOps : int {
      AsymAvg = XPRS_BARHGOPS_ASYM_AVG,
      StartL1 = XPRS_BARHGOPS_START_L1,
      StartL2 = XPRS_BARHGOPS_START_L2,
      StartLInf = XPRS_BARHGOPS_START_LINF,
      OmegaContract = XPRS_BARHGOPS_OMEGA_CONTRACT,
      OmegaInf = XPRS_BARHGOPS_OMEGA_INF,
      MaxObjScale = XPRS_BARHGOPS_MAX_OBJSCALE,
      NoObjScale = XPRS_BARHGOPS_NO_OBJSCALE,
      HPDHG = XPRS_BARHGOPS_HPDHG,
      HBASE = XPRS_BARHGOPS_HBASE
    };
xpress_diags_pop inline bool operator==(BARHGOps a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(BARHGOps a, int b) { return static_cast<int>(a) != b; }
inline bool operator==(int a, BARHGOps b) { return a == static_cast<int>(b); }
inline bool operator!=(int a, BARHGOps b) { return a != static_cast<int>(b); }
inline std::string to_string(BARHGOps e) {

  switch (e) {
  case BARHGOps::AsymAvg:
    return "AsymAvg";
    break;
  case BARHGOps::StartL1:
    return "StartL1";
    break;
  case BARHGOps::StartL2:
    return "StartL2";
    break;
  case BARHGOps::StartLInf:
    return "StartLInf";
    break;
  case BARHGOps::OmegaContract:
    return "OmegaContract";
    break;
  case BARHGOps::OmegaInf:
    return "OmegaInf";
    break;
  case BARHGOps::MaxObjScale:
    return "MaxObjScale";
    break;
  case BARHGOps::NoObjScale:
    return "NoObjScale";
    break;
  case BARHGOps::HPDHG:
    return "HPDHG";
    break;
  case BARHGOps::HBASE:
    return "HBASE";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, BARHGOps e) {
  os << to_string(e);
  return os;
}
xpress_diags_push xpress_ignore_deprecated

    enum class BasisStatus : int {
      NonbasicLower = XPRS_BASISSTATUS_NONBASIC_LOWER,
      Basic = XPRS_BASISSTATUS_BASIC,
      NonbasicUpper = XPRS_BASISSTATUS_NONBASIC_UPPER,
      Superbasic = XPRS_BASISSTATUS_SUPERBASIC
    };
xpress_diags_pop inline bool operator==(BasisStatus a, int b) {
  return static_cast<int>(a) == b;
}
inline bool operator!=(BasisStatus a, int b) {
  return static_cast<int>(a) != b;
}
inline bool operator==(int a, BasisStatus b) {
  return a == static_cast<int>(b);
}
inline bool operator!=(int a, BasisStatus b) {
  return a != static_cast<int>(b);
}
inline std::string to_string(BasisStatus e) {

  switch (e) {
  case BasisStatus::NonbasicLower:
    return "NonbasicLower";
    break;
  case BasisStatus::Basic:
    return "Basic";
    break;
  case BasisStatus::NonbasicUpper:
    return "NonbasicUpper";
    break;
  case BasisStatus::Superbasic:
    return "Superbasic";
    break;
  default:
    std::stringstream s;
    s.imbue(std::locale("C"));
    s << static_cast<int>(e);
    return s.str();
  }
}
inline std::ostream &operator<<(std::ostream &os, BasisStatus e) {
  os << to_string(e);
  return os;
}

class XPRS {
  friend class Object;
  XPRS() = delete;
  XPRS(XPRS const &) = delete;
  XPRS &operator=(XPRS const &) = delete;
  XPRS(XPRS &&) = delete;
  XPRS &operator=(XPRS &&) = delete;

  bool const doFree;

  static inline void callInit(char const *licpath);

  static inline void callFree();

  static void callbackAdded(void *arg) {
    CallbackHolder *holder = reinterpret_cast<CallbackHolder *>(arg);
    if (holder)
      holder->incRef();
  }

  static void callbackRemoved(void *arg) {
    CallbackHolder *holder = reinterpret_cast<CallbackHolder *>(arg);
    if (holder && holder->decRef() == 0)
      delete holder;
  }

  XPRS(bool initialize, char const *licpath) : doFree(initialize) {
    if (initialize)
      callInit(licpath);
  }

public:
  XPRS(char const *licpath = nullptr) : doFree(true) { callInit(licpath); }

  ~XPRS() noexcept {
    if (doFree) {
      callFree();
    }
  }
};

class Object : public CallbackExceptionHandler {
  Object(Object const &) = delete;
  Object &operator=(Object const &) = delete;
  Object(Object &&) = delete;
  Object &operator=(Object &&) = delete;
  XPRS const license;

protected:
  std::recursive_mutex synchronizedMutex;
  Object(bool initLicense, char const *licpath)
      : license(initLicense, licpath) {}

public:
  virtual ~Object() noexcept {}

  virtual bool autoDelete() const noexcept = 0;

  virtual char const *getTypeName() const = 0;
};

template <typename AlwaysVoid> class ObjectMap {

  static std::recursive_mutex mtx;

  static std::unordered_map<void *, Object *> objectMap;
  friend class XPRS;

public:
  template <typename P>
  static auto
  removeObject(typename std::enable_if<std::is_pointer<P>::value, P>::type ptr)
      -> Object * {
    std::lock_guard<std::recursive_mutex> guard(mtx);
    std::unordered_map<void *, Object *>::iterator it =
        objectMap.find(reinterpret_cast<void *>(ptr));
    if (it == objectMap.end()) {

      return nullptr;
    }
    Object *o = it->second;
    objectMap.erase(it);
    return o;
  }

private:
  static void objectDestroyed(void *obj) {
    Object *destroy = removeObject<void *>(obj);
    if (destroy) {
      xpress_assert(!destroy->autoDelete());
      delete destroy;
    }
  }

public:
  template <typename P, typename O>
  static auto
  findObject(typename std::enable_if<std::is_pointer<P>::value &&
                                         std::is_base_of<Object, O>::value,
                                     P>::type ptr) -> O * {
    std::lock_guard<std::recursive_mutex> guard(mtx);
    std::unordered_map<void *, Object *>::iterator it =
        objectMap.find(reinterpret_cast<void *>(ptr));
    if (it == objectMap.end())
      return nullptr;
    O *o = dynamic_cast<O *>(it->second);
    if (!o)
      throw XPRSException("object has wrong type");
    return o;
  }

  template <typename P, typename O>
  static auto
  findObject(typename std::enable_if<std::is_pointer<P>::value &&
                                         std::is_base_of<Object, O>::value,
                                     P>::type ptr,
             std::function<O *(P)> makeObject) -> O * {
    O *o = findObject<P, O>(ptr);
    if (o)
      return o;
    return makeObject(ptr);
  }

  template <typename P, typename O>
  static void addObject(
      typename std::enable_if<std::is_pointer<P>::value, P>::type ptr,
      typename std::enable_if<std::is_base_of<Object, O>::value, O>::type *o) {
    std::lock_guard<std::recursive_mutex> guard(mtx);

    xpress_assert(objectMap.find(reinterpret_cast<void *>(ptr)) ==
                  objectMap.end());
    objectMap.insert(
        std::pair<void *, Object *>(reinterpret_cast<void *>(ptr), o));
  }
};
template <typename AlwaysVoid> std::recursive_mutex ObjectMap<AlwaysVoid>::mtx;
template <typename AlwaysVoid>
std::unordered_map<void *, Object *> ObjectMap<AlwaysVoid>::objectMap;

template <typename T> class TypedObject : public Object {
  TypedObject() = delete;
  TypedObject(TypedObject const &) = delete;
  TypedObject &operator=(TypedObject const &) = delete;
  TypedObject(TypedObject &&) = delete;
  TypedObject &operator=(TypedObject &&) = delete;

  friend class XPRSProblem;
  friend class NameList;
  friend class MIPSolPool;
  friend class MIPSolEnum;

protected:
  typedef void DestructorFunction(T);

  DestructorFunction *dtorFunction;

  T const ptr;

  std::atomic<bool> mutable callbackExceptionPending;

  std::exception_ptr mutable callbackException;

  TypedObject(T p)
      : Object(false, nullptr), dtorFunction(nullptr), ptr(p),
        callbackExceptionPending(false), callbackException() {
    ObjectMap<void>::addObject<T, TypedObject>(ptr, this);
  }

  TypedObject(std::function<T()> newPointer, DestructorFunction *df,
              bool initLicense, char const *licpath)
      : Object(initLicense, licpath), dtorFunction(df), ptr(newPointer()),
        callbackExceptionPending(false), callbackException() {
    ObjectMap<void>::addObject<T, TypedObject>(ptr, this);
  }

public:
  auto getRawPointer() const -> T { return ptr; }

  void setCBException(std::exception_ptr e) const noexcept override {

    bool f = false;
    if (callbackExceptionPending.compare_exchange_strong(f, true))
      callbackException = e;
  }

  void consumeCBException() const {

    std::exception_ptr e = callbackException;
    callbackException = nullptr;
    callbackExceptionPending.store(false);
    if (e)
      std::rethrow_exception(e);
  }

protected:
  void checkNullObject() const {

    if (!ptr)
      throw std::runtime_error("use of deleted object");
  }

  void check(int error) const {

    consumeCBException();
    if (error)
      throw XPRSException::error(ptr, error);
  }

public:
  virtual ~TypedObject() noexcept {
    if (ptr && dtorFunction) {
      dtorFunction(ptr);
      ObjectMap<void>::removeObject<T>(ptr);
    }
  }

  virtual bool autoDelete() const noexcept override { return dtorFunction; }

  virtual char const *getTypeName() const override {
    char const *typeName = nullptr;
    check(XPRSgetobjecttypename(reinterpret_cast<XPRSobject>(ptr), &typeName));
    return typeName;
  }

public:
};

class ChangeBranchObjectCallbackHolder;
namespace objects {

class ChangeBranchObjectCallbackHolder;
}

class BranchObject : public TypedObject<XPRSbranchobject> {
  friend class XPRSProblem;
  friend class xpress::ChangeBranchObjectCallbackHolder;
  friend class xpress::objects::ChangeBranchObjectCallbackHolder;
  BranchObject(BranchObject const &) = delete;
  BranchObject &operator=(BranchObject const &) = delete;
  BranchObject(BranchObject &&) = delete;
  BranchObject &operator=(BranchObject &&) = delete;

  static void destroyPointer(XPRSbranchobject bo) { XPRS_bo_destroy(bo); }

  BranchObject(XPRSbranchobject p)
      : TypedObject([&]() { return p; }, nullptr, false, nullptr) {}

protected:
  BranchObject(XPRSprob prob, bool isOriginal)
      : TypedObject(
            [=]() {
              XPRSbranchobject bo;
              int code = XPRS_bo_create(&bo, prob, isOriginal ? 1 : 0);
              if (code) {
                throw XPRSException::error(prob, code);
              }
              return bo;
            },
            &destroyPointer, false, nullptr) {}

public:
  inline void destroy();

  inline auto store() -> int;

  inline void addBranches(int nbranches);

  inline auto getBranches() -> int;

  inline void setPriority(int priority);

  inline void setPreferredBranch(int branch);

  inline void addBounds(int branch, int nbounds,
                        Array<char const> const &bndtype,
                        Array<int const> const &colind,
                        Array<double const> const &bndval);

  inline void getBounds(int branch, int *p_nbounds, int maxbounds,
                        Array<char> const &bndtype, Array<int> const &colind,
                        Array<double> const &bndval);

  inline auto getBounds(int branch, int maxbounds, Array<char> const &bndtype,
                        Array<int> const &colind, Array<double> const &bndval)
      -> int;

  inline auto getBounds(int branch) -> int;

  inline void
  addRows(int branch, int nrows, int ncoefs, Array<char const> const &rowtype,
          Array<double const> const &rhs, Array<int const> const &start,
          Array<int const> const &colind, Array<double const> const &rowcoef);

  inline void getRows(int branch, int *p_nrows, int maxrows, int *p_ncoefs,
                      int maxcoefs, Array<char> const &rowtype,
                      Array<double> const &rhs, Array<int> const &start,
                      Array<int> const &colind, Array<double> const &rowcoef);

  inline auto getRows(int branch, int maxrows, int *p_ncoefs, int maxcoefs,
                      Array<char> const &rowtype, Array<double> const &rhs,
                      Array<int> const &start, Array<int> const &colind,
                      Array<double> const &rowcoef) -> int;

  inline auto getRows(int branch) -> int;

  inline void addCuts(int branch, int ncuts,
                      Array<XPRScut const> const &cutind);

  inline auto getID() -> int;

  inline auto validate() -> int;
};
namespace objects {

class PostfixExtractor {
public:
  virtual auto addToken(int type, double value) -> PostfixExtractor & = 0;

  virtual auto dereferenceVariable(Variable x) -> int = 0;

  virtual auto startExpression() -> void = 0;

  virtual auto endExpression() -> void = 0;
  virtual ~PostfixExtractor() {}
};
} // namespace objects

class XPRSProblem : public TypedObject<XPRSprob> {
  XPRSProblem(XPRSProblem const &) = delete;
  XPRSProblem &operator=(XPRSProblem const &) = delete;
  XPRSProblem(XPRSProblem &&) = delete;
  XPRSProblem &operator=(XPRSProblem &&) = delete;

  static auto createPointer() -> XPRSprob {
    XPRSprob prob;
    int code = XPRScreateprob(&prob);
    if (code) {
      throw XPRSException(code);
    }
    return prob;
  }

  static void destroyPointer(XPRSprob p) { XPRSdestroyprob(p); }

public:
  virtual auto makeChild(XPRSprob pointer) -> XPRSProblem * {
    return new XPRSProblem(pointer);
  }

  auto newBranchObject(bool isOriginal) -> BranchObject * {
    return new BranchObject(ptr, isOriginal);
  }

  inline void setProbName(std::optional<std::string> const &probname);

  inline void setLogFile(std::optional<std::string> const &filename);

  inline void setDefaultControl(int control);

  inline void setDefaults();

  inline void readProb(std::optional<std::string> const &filename,
                       std::optional<std::string> const &flags);

  inline void readProb(std::optional<std::string> const &filename);

  inline void
  loadLp(std::optional<std::string> const &probname, int ncols, int nrows,
         Array<char const> const &rowtype, Array<double const> const &rhs,
         Array<double const> const &rng, Array<double const> const &objcoef,
         Array<int const> const &start, Array<int const> const &collen,
         Array<int const> const &rowind, Array<double const> const &rowcoef,
         Array<double const> const &lb, Array<double const> const &ub);

  inline void
  loadLp(std::optional<std::string> const &probname, int ncols, int nrows,
         Array<char const> const &rowtype, Array<double const> const &rhs,
         Array<double const> const &rng, Array<double const> const &objcoef,
         Array<XPRSint64 const> const &start, Array<int const> const &collen,
         Array<int const> const &rowind, Array<double const> const &rowcoef,
         Array<double const> const &lb, Array<double const> const &ub);

  inline void
  loadQP(std::optional<std::string> const &probname, int ncols, int nrows,
         Array<char const> const &rowtype, Array<double const> const &rhs,
         Array<double const> const &rng, Array<double const> const &objcoef,
         Array<int const> const &start, Array<int const> const &collen,
         Array<int const> const &rowind, Array<double const> const &rowcoef,
         Array<double const> const &lb, Array<double const> const &ub,
         int nobjqcoefs, Array<int const> const &objqcol1,
         Array<int const> const &objqcol2, Array<double const> const &objqcoef);

  inline void
  loadQP(std::optional<std::string> const &probname, int ncols, int nrows,
         Array<char const> const &rowtype, Array<double const> const &rhs,
         Array<double const> const &rng, Array<double const> const &objcoef,
         Array<XPRSint64 const> const &start, Array<int const> const &collen,
         Array<int const> const &rowind, Array<double const> const &rowcoef,
         Array<double const> const &lb, Array<double const> const &ub,
         XPRSint64 nobjqcoefs, Array<int const> const &objqcol1,
         Array<int const> const &objqcol2, Array<double const> const &objqcoef);

  inline void
  loadMIQP(std::optional<std::string> const &probname, int ncols, int nrows,
           Array<char const> const &rowtype, Array<double const> const &rhs,
           Array<double const> const &rng, Array<double const> const &objcoef,
           Array<int const> const &start, Array<int const> const &collen,
           Array<int const> const &rowind, Array<double const> const &rowcoef,
           Array<double const> const &lb, Array<double const> const &ub,
           int nobjqcoefs, Array<int const> const &objqcol1,
           Array<int const> const &objqcol2,
           Array<double const> const &objqcoef, int nentities, int nsets,
           Array<char const> const &coltype, Array<int const> const &entind,
           Array<double const> const &limit, Array<char const> const &settype,
           Array<int const> const &setstart, Array<int const> const &setind,
           Array<double const> const &refval);

  inline void
  loadMIQP(std::optional<std::string> const &probname, int ncols, int nrows,
           Array<char const> const &rowtype, Array<double const> const &rhs,
           Array<double const> const &rng, Array<double const> const &objcoef,
           Array<XPRSint64 const> const &start, Array<int const> const &collen,
           Array<int const> const &rowind, Array<double const> const &rowcoef,
           Array<double const> const &lb, Array<double const> const &ub,
           XPRSint64 nobjqcoefs, Array<int const> const &objqcol1,
           Array<int const> const &objqcol2,
           Array<double const> const &objqcoef, int nentities, int nsets,
           Array<char const> const &coltype, Array<int const> const &entind,
           Array<double const> const &limit, Array<char const> const &settype,
           Array<XPRSint64 const> const &setstart,
           Array<int const> const &setind, Array<double const> const &refval);

  inline void fixMipEntities(int options);

  inline void loadModelCuts(int nrows, Array<int const> const &rowind);

  inline void loadDelayedRows(int nrows, Array<int const> const &rowind);

  inline void loadDirs(int ndirs, Array<int const> const &colind,
                       Array<int const> const &priority,
                       Array<char const> const &dir,
                       Array<double const> const &uppseudo,
                       Array<double const> const &downpseudo);

  inline void loadBranchDirs(int ncols, Array<int const> const &colind,
                             Array<int const> const &dir);
  void loadBranchDirs(int index, int value) {
    loadBranchDirs(1, &index, &value);
  }

  inline void loadPresolveDirs(int ndirs, Array<int const> const &colind,
                               Array<int const> const &priority,
                               Array<char const> const &dir,
                               Array<double const> const &uppseudo,
                               Array<double const> const &downpseudo);

  inline void
  loadMip(std::optional<std::string> const &probname, int ncols, int nrows,
          Array<char const> const &rowtype, Array<double const> const &rhs,
          Array<double const> const &rng, Array<double const> const &objcoef,
          Array<int const> const &start, Array<int const> const &collen,
          Array<int const> const &rowind, Array<double const> const &rowcoef,
          Array<double const> const &lb, Array<double const> const &ub,
          int nentities, int nsets, Array<char const> const &coltype,
          Array<int const> const &entind, Array<double const> const &limit,
          Array<char const> const &settype, Array<int const> const &setstart,
          Array<int const> const &setind, Array<double const> const &refval);

  inline void
  loadMip(std::optional<std::string> const &probname, int ncols, int nrows,
          Array<char const> const &rowtype, Array<double const> const &rhs,
          Array<double const> const &rng, Array<double const> const &objcoef,
          Array<XPRSint64 const> const &start, Array<int const> const &collen,
          Array<int const> const &rowind, Array<double const> const &rowcoef,
          Array<double const> const &lb, Array<double const> const &ub,
          int nentities, int nsets, Array<char const> const &coltype,
          Array<int const> const &entind, Array<double const> const &limit,
          Array<char const> const &settype,
          Array<XPRSint64 const> const &setstart,
          Array<int const> const &setind, Array<double const> const &refval);

  inline void addNames(int type, Array<std::string const> const &names,
                       int first, int last);

  [[deprecated("since 45.00")]] inline void
  addSetNames(Array<std::string const> const &names, int first, int last);

  inline void scale(Array<int const> const &rowscale,
                    Array<int const> const &colscale);

  inline void readDirs(std::optional<std::string> const &filename);

  inline void readDirs();

  inline void writeDirs(std::optional<std::string> const &filename);

  inline void writeDirs();

  [[deprecated("since 45.00")]] inline void unloadProb();

  inline void setIndicators(int nrows, Array<int const> const &rowind,
                            Array<int const> const &colind,
                            Array<int const> const &complement);

  inline void addPwlCons(int npwls, int npoints, Array<int const> const &colind,
                         Array<int const> const &resultant,
                         Array<int const> const &start,
                         Array<double const> const &xval,
                         Array<double const> const &yval);

  inline void addPwlCons(int npwls, XPRSint64 npoints,
                         Array<int const> const &colind,
                         Array<int const> const &resultant,
                         Array<XPRSint64 const> const &start,
                         Array<double const> const &xval,
                         Array<double const> const &yval);

  inline void getPwlCons(Array<int> const &colind, Array<int> const &resultant,
                         Array<int> const &start, Array<double> const &xval,
                         Array<double> const &yval, int maxpoints,
                         int *p_npoints, int first, int last);

  inline void getPwlCons(Array<int> const &colind, Array<int> const &resultant,
                         Array<XPRSint64> const &start,
                         Array<double> const &xval, Array<double> const &yval,
                         XPRSint64 maxpoints, XPRSint64 *p_npoints, int first,
                         int last);

  inline void addGenCons(int ncons, int ncols, int nvals,
                         Array<GenConsType const> const &contype,
                         Array<int const> const &resultant,
                         Array<int const> const &colstart,
                         Array<int const> const &colind,
                         Array<int const> const &valstart,
                         Array<double const> const &val);

  inline void addGenCons(int ncons, XPRSint64 ncols, XPRSint64 nvals,
                         Array<GenConsType const> const &contype,
                         Array<int const> const &resultant,
                         Array<XPRSint64 const> const &colstart,
                         Array<int const> const &colind,
                         Array<XPRSint64 const> const &valstart,
                         Array<double const> const &val);

  inline void getGenCons(Array<GenConsType> const &contype,
                         Array<int> const &resultant,
                         Array<int> const &colstart, Array<int> const &colind,
                         int maxcols, int *p_ncols, Array<int> const &valstart,
                         Array<double> const &val, int maxvals, int *p_nvals,
                         int first, int last);

  inline void getGenCons(Array<GenConsType> const &contype,
                         Array<int> const &resultant,
                         Array<XPRSint64> const &colstart,
                         Array<int> const &colind, XPRSint64 maxcols,
                         XPRSint64 *p_ncols, Array<XPRSint64> const &valstart,
                         Array<double> const &val, XPRSint64 maxvals,
                         XPRSint64 *p_nvals, int first, int last);

  virtual inline void delPwlCons(int npwls, Array<int const> const &pwlind);

  virtual inline void delGenCons(int ncons, Array<int const> const &conind);

  inline void dumpControls();

  inline void getIndicators(Array<int> const &colind,
                            Array<int> const &complement, int first, int last);

  inline void delIndicators(int first, int last);

  inline void lpOptimize(std::optional<std::string> const &flags);

  inline void lpOptimize();

  inline void mipOptimize(std::optional<std::string> const &flags);

  inline void mipOptimize();

  inline void optimize(std::optional<std::string> const &flags,
                       int *solvestatus, int *solstatus);

  inline void readSlxSol(std::optional<std::string> const &filename,
                         std::optional<std::string> const &flags);

  inline void readSlxSol(std::optional<std::string> const &filename);

  inline void readSlxSol();

  [[deprecated("since 45.00, use functions to directly manipulate matrix "
               "data")]] inline void
  alter(std::optional<std::string> const &filename);

  inline void readBasis(std::optional<std::string> const &filename,
                        std::optional<std::string> const &flags);

  inline void readBasis(std::optional<std::string> const &filename);

  inline void readBasis();

  inline void readBinSol(std::optional<std::string> const &filename,
                         std::optional<std::string> const &flags);

  inline void readBinSol(std::optional<std::string> const &filename);

  inline void readBinSol();

  inline void getInfeas(int *p_nprimalcols, int *p_nprimalrows,
                        int *p_ndualrows, int *p_ndualcols, Array<int> const &x,
                        Array<int> const &slack, Array<int> const &duals,
                        Array<int> const &djs);

  inline void getScaledInfeas(int *p_nprimalcols, int *p_nprimalrows,
                              int *p_ndualrows, int *p_ndualcols,
                              Array<int> const &x, Array<int> const &slack,
                              Array<int> const &duals, Array<int> const &djs);

  inline auto getUnbVec() -> int;

  inline auto crossoverLpSol() -> int;

  inline void tune(std::optional<std::string> const &flags);

  inline void tuneProbSetFile(std::optional<std::string> const &setfile,
                              int ifmip, int sense);

  inline void tunerWriteMethod(std::optional<std::string> const &methodfile);

  inline void tunerReadMethod(std::optional<std::string> const &methodfile);

  [[deprecated("since 45.00")]] inline void
  getBarNumStability(Array<int> const &colstab, Array<int> const &rowstab);

  inline void getLastBarSol(Array<double> const &x, Array<double> const &slack,
                            Array<double> const &duals,
                            Array<double> const &djs, int *p_status);

  inline void clearIIS();

  inline auto firstIIS(int mode) -> int;

  inline auto nextIIS() -> int;

  inline void printIIS(int iis);

  inline void IISStatus(int *p_niis, Array<int> const &nrows,
                        Array<int> const &ncols, Array<double> const &suminfeas,
                        Array<int> const &numinfeas);

  inline void IISAll();

  inline void writeIIS(int iis, std::optional<std::string> const &filename,
                       int filetype, std::optional<std::string> const &flags);

  inline void writeIIS(int iis, std::optional<std::string> const &filename,
                       int filetype);

  inline void IISIsolations(int iis);

  inline void getIISData(int iis, int *p_nrows, int *p_ncols,
                         Array<int> const &rowind, Array<int> const &colind,
                         Array<char> const &contype, Array<char> const &bndtype,
                         Array<double> const &duals, Array<double> const &djs,
                         Array<char> const &isolationrows,
                         Array<char> const &isolationcols);

  inline void loadPresolveBasis(Array<int const> const &rowstat,
                                Array<int const> const &colstat);

  inline void loadSecureVecs(int nrows, int ncols,
                             Array<int const> const &rowind,
                             Array<int const> const &colind);

  inline void addRows(int nrows, int ncoefs, Array<char const> const &rowtype,
                      Array<double const> const &rhs,
                      Array<double const> const &rng,
                      Array<int const> const &start,
                      Array<int const> const &colind,
                      Array<double const> const &rowcoef);

  inline void
  addRows(int nrows, XPRSint64 ncoefs, Array<char const> const &rowtype,
          Array<double const> const &rhs, Array<double const> const &rng,
          Array<XPRSint64 const> const &start, Array<int const> const &colind,
          Array<double const> const &rowcoef);

  inline void addRows(int nrows, int ncoefs, Array<char const> const &rowtype,
                      Array<double const> const &rhs,
                      Array<int const> const &start,
                      Array<int const> const &colind,
                      Array<double const> const &rowcoef);

  inline void
  addRows(int nrows, XPRSint64 ncoefs, Array<char const> const &rowtype,
          Array<double const> const &rhs, Array<XPRSint64 const> const &start,
          Array<int const> const &colind, Array<double const> const &rowcoef);

  virtual inline void delRows(int nrows, Array<int const> const &rowind);

  inline void addCols(int ncols, int ncoefs, Array<double const> const &objcoef,
                      Array<int const> const &start,
                      Array<int const> const &rowind,
                      Array<double const> const &rowcoef,
                      Array<double const> const &lb,
                      Array<double const> const &ub);

  inline void
  addCols(int ncols, XPRSint64 ncoefs, Array<double const> const &objcoef,
          Array<XPRSint64 const> const &start, Array<int const> const &rowind,
          Array<double const> const &rowcoef, Array<double const> const &lb,
          Array<double const> const &ub);

  virtual inline void delCols(int ncols, Array<int const> const &colind);

  inline void chgColType(int ncols, Array<int const> const &colind,
                         Array<char const> const &coltype);
  void chgColType(int index, char value) { chgColType(1, &index, &value); }

  inline void loadBasis(Array<int const> const &rowstat,
                        Array<int const> const &colstat);

  inline void postSolve();

  virtual inline void delSets(int nsets, Array<int const> const &setind);

  inline void addSets(int nsets, int nelems, Array<char const> const &settype,
                      Array<int const> const &start,
                      Array<int const> const &colind,
                      Array<double const> const &refval);

  inline void addSets(int nsets, XPRSint64 nelems,
                      Array<char const> const &settype,
                      Array<XPRSint64 const> const &start,
                      Array<int const> const &colind,
                      Array<double const> const &refval);

  inline void strongBranch(int nbounds, Array<int const> const &colind,
                           Array<char const> const &bndtype,
                           Array<double const> const &bndval, int iterlim,
                           Array<double> const &objval,
                           Array<int> const &status);

  inline void estimateRowDualRanges(int nrows, Array<int const> const &rowind,
                                    int iterlim, Array<double> const &mindual,
                                    Array<double> const &maxdual);

  inline void setMessageStatus(int msgcode, int status);

  inline auto getMessageStatus(int msgcode) -> int;

  inline void chgObjSense(ObjSense objsense);

  inline void chgGlbLimit(int ncols, Array<int const> const &colind,
                          Array<double const> const &limit);
  void chgGlbLimit(int index, double value) { chgGlbLimit(1, &index, &value); }

  inline void save();

  inline void saveAs(std::optional<std::string> const &sSaveFileName);

  inline void restore(std::optional<std::string> const &probname,
                      std::optional<std::string> const &flags);

  inline void restore();

  inline void restore(std::optional<std::string> const &probname);

  inline void pivot(int enter, int leave);

  inline auto loadLpSol(Array<double const> const &x,
                        Array<double const> const &slack,
                        Array<double const> const &duals,
                        Array<double const> const &djs) -> int;

  inline auto repairWeightedInfeas(Array<double const> const &lepref,
                                   Array<double const> const &gepref,
                                   Array<double const> const &lbpref,
                                   Array<double const> const &ubpref,
                                   char phase2, double delta,
                                   std::optional<std::string> const &flags)
      -> int;

  inline auto repairWeightedInfeasBounds(
      Array<double const> const &lepref, Array<double const> const &gepref,
      Array<double const> const &lbpref, Array<double const> const &ubpref,
      Array<double const> const &lerelax, Array<double const> const &gerelax,
      Array<double const> const &lbrelax, Array<double const> const &ubrelax,
      char phase2, double delta, std::optional<std::string> const &flags)
      -> int;

  inline auto repairInfeas(char penalty, char phase2, char flags, double lepref,
                           double gepref, double lbpref, double ubpref,
                           double delta) -> int;

  inline auto basisStability(int type, int norm, int scaled) -> double;

  inline void objSA(int ncols, Array<int const> const &colind,
                    Array<double> const &lower, Array<double> const &upper);

  inline void bndSA(int ncols, Array<int const> const &colind,
                    Array<double> const &lblower, Array<double> const &lbupper,
                    Array<double> const &ublower, Array<double> const &ubupper);

  inline void rhsSA(int nrows, Array<int const> const &rowind,
                    Array<double> const &lower, Array<double> const &upper);

  inline void addQMatrix(int row, int ncoefs, Array<int const> const &rowqcol1,
                         Array<int const> const &rowqcol2,
                         Array<double const> const &rowqcoef);

  inline void addQMatrix(int row, XPRSint64 ncoefs,
                         Array<int const> const &rowqcol1,
                         Array<int const> const &rowqcol2,
                         Array<double const> const &rowqcoef);

  inline void delQMatrix(int row);

  inline void
  loadQCQP(std::optional<std::string> const &probname, int ncols, int nrows,
           Array<char const> const &rowtype, Array<double const> const &rhs,
           Array<double const> const &rng, Array<double const> const &objcoef,
           Array<int const> const &start, Array<int const> const &collen,
           Array<int const> const &rowind, Array<double const> const &rowcoef,
           Array<double const> const &lb, Array<double const> const &ub,
           int nobjqcoefs, Array<int const> const &objqcol1,
           Array<int const> const &objqcol2,
           Array<double const> const &objqcoef, int nqrows,
           Array<int const> const &qrowind, Array<int const> const &nrowqcoef,
           Array<int const> const &rowqcol1, Array<int const> const &rowqcol2,
           Array<double const> const &rowqcoef);

  inline void loadQCQP(
      std::optional<std::string> const &probname, int ncols, int nrows,
      Array<char const> const &rowtype, Array<double const> const &rhs,
      Array<double const> const &rng, Array<double const> const &objcoef,
      Array<XPRSint64 const> const &start, Array<int const> const &collen,
      Array<int const> const &rowind, Array<double const> const &rowcoef,
      Array<double const> const &lb, Array<double const> const &ub,
      XPRSint64 nobjqcoefs, Array<int const> const &objqcol1,
      Array<int const> const &objqcol2, Array<double const> const &objqcoef,
      int nqrows, Array<int const> const &qrowind,
      Array<XPRSint64 const> const &nrowqcoef, Array<int const> const &rowqcol1,
      Array<int const> const &rowqcol2, Array<double const> const &rowqcoef);

  inline void loadMIQCQP(
      std::optional<std::string> const &probname, int ncols, int nrows,
      Array<char const> const &rowtype, Array<double const> const &rhs,
      Array<double const> const &rng, Array<double const> const &objcoef,
      Array<int const> const &start, Array<int const> const &collen,
      Array<int const> const &rowind, Array<double const> const &rowcoef,
      Array<double const> const &lb, Array<double const> const &ub,
      int nobjqcoefs, Array<int const> const &objqcol1,
      Array<int const> const &objqcol2, Array<double const> const &objqcoef,
      int nqrows, Array<int const> const &qrowind,
      Array<int const> const &nrowqcoefs, Array<int const> const &rowqcol1,
      Array<int const> const &rowqcol2, Array<double const> const &rowqcoef,
      int nentities, int nsets, Array<char const> const &coltype,
      Array<int const> const &entind, Array<double const> const &limit,
      Array<char const> const &settype, Array<int const> const &setstart,
      Array<int const> const &setind, Array<double const> const &refval);

  inline void
  loadMIQCQP(std::optional<std::string> const &probname, int ncols, int nrows,
             Array<char const> const &rowtype, Array<double const> const &rhs,
             Array<double const> const &rng, Array<double const> const &objcoef,
             Array<XPRSint64 const> const &start,
             Array<int const> const &collen, Array<int const> const &rowind,
             Array<double const> const &rowcoef, Array<double const> const &lb,
             Array<double const> const &ub, XPRSint64 nobjqcoefs,
             Array<int const> const &objqcol1, Array<int const> const &objqcol2,
             Array<double const> const &objqcoef, int nqrows,
             Array<int const> const &qrowind,
             Array<XPRSint64 const> const &nrowqcoefs,
             Array<int const> const &rowqcol1, Array<int const> const &rowqcol2,
             Array<double const> const &rowqcoef, int nentities, int nsets,
             Array<char const> const &coltype, Array<int const> const &entind,
             Array<double const> const &limit, Array<char const> const &settype,
             Array<XPRSint64 const> const &setstart,
             Array<int const> const &setind, Array<double const> const &refval);

  inline void addObj(int ncols, Array<int const> const &colind,
                     Array<double const> const &objcoef, int priority,
                     double weight);

  inline void chgObjN(int objidx, int ncols, Array<int const> const &colind,
                      Array<double const> const &objcoef);

  inline void delObj(int objidx);

  inline void copyControls(XPRSProblem const &src);

  inline void copyProb(XPRSProblem const &src,
                       std::optional<std::string> const &name);

  inline void destroyProb();

  inline void interrupt(StopType reason);

  inline auto getProbName() -> std::string;

  inline void setIntControl(int control, int value);

  inline void setLongControl(int control, XPRSint64 value);

  inline void setDblControl(int control, double value);

  inline void setStrControl(int control,
                            std::optional<std::string> const &value);

  inline auto getIntControl(int control) const -> int;

  inline auto getLongControl(int control) const -> XPRSint64;

  inline auto getDblControl(int control) const -> double;

  inline auto getStrControl(int control) const -> std::string;

  inline void getStringControl(int control, char *value, int maxbytes,
                               int *p_nbytes) const;

  inline auto getIntAttrib(int attrib) const -> int;

  inline auto getLongAttrib(int attrib) const -> XPRSint64;

  inline auto getStrAttrib(int attrib) const -> std::string;

  inline void getStrStringAttrib(int attrib, char *value, int maxbytes,
                                 int *p_nbytes) const;

  inline auto getDblAttrib(int attrib) const -> double;

  inline void getControlInfo(std::optional<std::string> const &name, int *p_id,
                             ParameterType *p_type);

  inline void getAttribInfo(std::optional<std::string> const &name, int *p_id,
                            ParameterType *p_type);

  inline auto getIndex(int type, std::optional<std::string> const &name) -> int;

  inline void setObjIntControl(int objidx, ObjControl control, int value);

  inline void setObjDblControl(int objidx, ObjControl control, double value);

  inline auto getObjIntControl(int objidx, ObjControl control) -> int;

  inline auto getObjDblControl(int objidx, ObjControl control) -> double;

  inline auto getObjIntAttrib(int solveidx, int attrib) -> int;

  inline auto getObjLongAttrib(int solveidx, int attrib) -> XPRSint64;

  inline auto getObjDblAttrib(int solveidx, int attrib) -> double;

  inline auto getQObj(int objqcol1, int objqcol2) -> double;

  inline void getDirs(int *p_ndir, Array<int> const &indices,
                      Array<int> const &prios, Array<char> const &branchdirs,
                      Array<double> const &uppseudo,
                      Array<double> const &downpseudo);

  inline auto getDirs(Array<int> const &indices, Array<int> const &prios,
                      Array<char> const &branchdirs,
                      Array<double> const &uppseudo,
                      Array<double> const &downpseudo) -> int;

  inline auto getDirs() -> int;

  inline void getScale(Array<int> const &rowscale, Array<int> const &colscale);

  inline auto getPivotOrder() -> std::vector<int>;

  inline void getPresolveMap(Array<int> const &rowmap,
                             Array<int> const &colmap);

  inline void bTran(Array<double> const &vec);

  inline void fTran(Array<double> const &vec);

  inline void sparseBTran(Array<double> const &val, Array<int> const &ind,
                          int *p_ncoefs);

  inline void sparseFTran(Array<double> const &val, Array<int> const &ind,
                          int *p_ncoefs);

  inline auto getObj(int first, int last) -> std::vector<double>;
  auto getObj(int index) -> double { return getObj(index, index)[0]; }

  inline auto getObjN(int objidx, int first, int last) -> std::vector<double>;

  inline auto getRhs(int first, int last) -> std::vector<double>;
  auto getRhs(int index) -> double { return getRhs(index, index)[0]; }

  inline auto getRhsRange(int first, int last) -> std::vector<double>;
  auto getRhsRange(int index) -> double { return getRhsRange(index, index)[0]; }

  inline auto getLB(int first, int last) -> std::vector<double>;
  auto getLB(int index) -> double { return getLB(index, index)[0]; }

  inline auto getUB(int first, int last) -> std::vector<double>;
  auto getUB(int index) -> double { return getUB(index, index)[0]; }

  inline void getCols(Array<int> const &start, Array<int> const &rowind,
                      Array<double> const &rowcoef, int maxcoefs, int *p_ncoefs,
                      int first, int last);

  inline void getCols(Array<XPRSint64> const &start, Array<int> const &rowind,
                      Array<double> const &rowcoef, XPRSint64 maxcoefs,
                      XPRSint64 *p_ncoefs, int first, int last);

  inline auto getCols(Array<int> const &start, Array<int> const &rowind,
                      Array<double> const &rowcoef, int maxcoefs, int first,
                      int last) -> int;

  inline auto getCols(Array<XPRSint64> const &start, Array<int> const &rowind,
                      Array<double> const &rowcoef, XPRSint64 maxcoefs,
                      int first, int last) -> XPRSint64;

  inline void getRows(Array<int> const &start, Array<int> const &colind,
                      Array<double> const &colcoef, int maxcoefs, int *p_ncoefs,
                      int first, int last);

  inline void getRows(Array<XPRSint64> const &start, Array<int> const &colind,
                      Array<double> const &colcoef, XPRSint64 maxcoefs,
                      XPRSint64 *p_ncoefs, int first, int last);

  inline auto getRows(Array<int> const &start, Array<int> const &colind,
                      Array<double> const &colcoef, int maxcoefs, int first,
                      int last) -> int;

  inline auto getRows(Array<XPRSint64> const &start, Array<int> const &colind,
                      Array<double> const &colcoef, XPRSint64 maxcoefs,
                      int first, int last) -> XPRSint64;

  inline void
  getMipEntities(int *p_nentities, int *p_nsets, Array<char> const &coltype,
                 Array<int> const &colind, Array<double> const &limit,
                 Array<char> const &settype, Array<int> const &start,
                 Array<int> const &setcols, Array<double> const &refval);

  inline void
  getMipEntities(int *p_nentities, int *p_nsets, Array<char> const &coltype,
                 Array<int> const &colind, Array<double> const &limit,
                 Array<char> const &settype, Array<XPRSint64> const &start,
                 Array<int> const &setcols, Array<double> const &refval);

  inline void getMipEntities(int *p_nentities, Array<char> const &coltype,
                             Array<int> const &colind,
                             Array<double> const &limit);

  inline auto getMipEntities(Array<char> const &coltype,
                             Array<int> const &colind,
                             Array<double> const &limit) -> int;

  inline auto getRowFlags(int first, int last) -> std::vector<int>;
  auto getRowFlags(int index) -> int { return getRowFlags(index, index)[0]; }

  inline void clearRowFlags(Array<int const> const &flags, int first, int last);

  inline auto getCoef(int row, int col) -> double;

  inline void getMQObj(Array<int> const &start, Array<int> const &colind,
                       Array<double> const &objqcoef, int maxcoefs,
                       int *p_ncoefs, int first, int last);

  inline void getMQObj(Array<XPRSint64> const &start, Array<int> const &colind,
                       Array<double> const &objqcoef, XPRSint64 maxcoefs,
                       XPRSint64 *p_ncoefs, int first, int last);

  inline auto getMQObj(Array<int> const &start, Array<int> const &colind,
                       Array<double> const &objqcoef, int maxcoefs, int first,
                       int last) -> int;

  inline auto getMQObj(Array<XPRSint64> const &start, Array<int> const &colind,
                       Array<double> const &objqcoef, XPRSint64 maxcoefs,
                       int first, int last) -> XPRSint64;

  inline void writeBasis(std::optional<std::string> const &filename,
                         std::optional<std::string> const &flags);

  inline void writeBasis(std::optional<std::string> const &filename);

  inline void writeBasis();

  inline void writeSol(std::optional<std::string> const &filename,
                       std::optional<std::string> const &flags);

  inline void writeSol(std::optional<std::string> const &filename);

  inline void writeSol();

  inline void writeBinSol(std::optional<std::string> const &filename,
                          std::optional<std::string> const &flags);

  inline void writeBinSol(std::optional<std::string> const &filename);

  inline void writeBinSol();

  inline void writePrtSol(std::optional<std::string> const &filename,
                          std::optional<std::string> const &flags);

  inline void writePrtSol(std::optional<std::string> const &filename);

  inline void writePrtSol();

  inline void writeSlxSol(std::optional<std::string> const &filename,
                          std::optional<std::string> const &flags);

  inline void writeSlxSol(std::optional<std::string> const &filename);

  inline void writeSlxSol();

  inline void getPrimalRay(Array<double> const &ray, int *p_hasray);

  inline void getDualRay(Array<double> const &ray, int *p_hasray);

  inline void strongBranchCB(int nbounds, Array<int const> const &colind,
                             Array<char const> const &bndtype,
                             Array<double const> const &bndval, int iterlim,
                             Array<double> const &objval,
                             Array<int> const &status,
                             std::function<int(XPRSProblem &, int)> callback);

  inline auto loadMipSol(Array<double const> const &x) -> int;

  inline void getBasis(Array<int> const &rowstat, Array<int> const &colstat);

  inline void getBasisVal(int row, int col, int *p_rowstat, int *p_colstat);

  inline void addManagedCuts(int globalvalid, int ncuts,
                             Array<char const> const &rowtype,
                             Array<double const> const &rhs,
                             Array<int const> const &start,
                             Array<int const> const &colind,
                             Array<double const> const &cutcoef);

  inline void addManagedCuts(int globalvalid, int ncuts,
                             Array<char const> const &rowtype,
                             Array<double const> const &rhs,
                             Array<XPRSint64 const> const &start,
                             Array<int const> const &colind,
                             Array<double const> const &cutcoef);

  inline void addCuts(int ncuts, Array<int const> const &cuttype,
                      Array<char const> const &rowtype,
                      Array<double const> const &rhs,
                      Array<int const> const &start,
                      Array<int const> const &colind,
                      Array<double const> const &cutcoef);

  inline void addCuts(int ncuts, Array<int const> const &cuttype,
                      Array<char const> const &rowtype,
                      Array<double const> const &rhs,
                      Array<XPRSint64 const> const &start,
                      Array<int const> const &colind,
                      Array<double const> const &cutcoef);

  inline void delCuts(int basis, int cuttype, int interp, double delta,
                      int ncuts, Array<XPRScut const> const &cutind);

  inline void delCuts(int basis);

  inline void delCuts(int basis, int cuttype, int interp);

  inline void delCuts(int basis, int cuttype, int interp, double delta);

  inline void delCuts(int basis, int ncuts, Array<XPRScut const> const &cutind);

  inline void delCPCuts(int cuttype, int interp, int ncuts,
                        Array<XPRScut const> const &cutind);

  inline void delCPCuts();

  inline void delCPCuts(int cuttype, int interp);

  inline void delCPCuts(int ncuts, Array<XPRScut const> const &cutind);

  inline void getCutList(int cuttype, int interp, int *p_ncuts, int maxcuts,
                         Array<XPRScut> const &cutind);

  inline auto getCutList(int cuttype, int interp, int maxcuts,
                         Array<XPRScut> const &cutind) -> int;

  inline auto getCutList(int maxcuts, Array<XPRScut> const &cutind) -> int;

  inline void getCPCutList(int cuttype, int interp, double delta, int *p_ncuts,
                           int maxcuts, Array<XPRScut> const &cutind,
                           Array<double> const &viol);

  inline auto getCPCutList(int cuttype, int interp, double delta, int maxcuts,
                           Array<XPRScut> const &cutind,
                           Array<double> const &viol) -> int;

  inline auto getCPCutList(int maxcuts, Array<XPRScut> const &cutind,
                           Array<double> const &viol) -> int;

  inline void getCPCuts(Array<XPRScut const> const &rowind, int ncuts,
                        int maxcoefs, Array<int> const &cuttype,
                        Array<char> const &rowtype, Array<int> const &start,
                        Array<int> const &colind, Array<double> const &cutcoef,
                        Array<double> const &rhs);

  inline void getCPCuts(Array<XPRScut const> const &rowind, int ncuts,
                        XPRSint64 maxcoefs, Array<int> const &cuttype,
                        Array<char> const &rowtype,
                        Array<XPRSint64> const &start, Array<int> const &colind,
                        Array<double> const &cutcoef, Array<double> const &rhs);

  inline void loadCuts(int cuttype, int interp, int ncuts,
                       Array<XPRScut const> const &cutind);

  inline void loadCuts(int cuttype, int interp);

  inline void loadCuts(int ncuts, Array<XPRScut const> const &cutind);

  inline auto
  storeCuts(int ncuts, int nodups, Array<int const> const &cuttype,
            Array<char const> const &rowtype, Array<double const> const &rhs,
            Array<int const> const &start, Array<int const> const &colind,
            Array<double const> const &cutcoef) -> std::vector<XPRScut>;

  inline auto
  storeCuts(int ncuts, int nodups, Array<int const> const &cuttype,
            Array<char const> const &rowtype, Array<double const> const &rhs,
            Array<XPRSint64 const> const &start, Array<int const> const &colind,
            Array<double const> const &cutcoef) -> std::vector<XPRScut>;

  inline void
  presolveRow(char rowtype, int norigcoefs, Array<int const> const &origcolind,
              Array<double const> const &origrowcoef, double origrhs,
              int maxcoefs, int *p_ncoefs, Array<int> const &colind,
              Array<double> const &rowcoef, double *p_rhs, int *p_status);

  inline auto postSolveSol(Array<double const> const &prex)
      -> std::vector<double>;

  inline void getPivots(int enter, Array<int> const &outlist,
                        Array<double> const &x, double *p_objval,
                        int *p_npivots, int maxpivots);

  inline void writeProb(std::optional<std::string> const &filename,
                        std::optional<std::string> const &flags);

  inline void writeProb(std::optional<std::string> const &filename);

  inline void writeProb();

  inline auto calcSlacks(Array<double const> const &solution)
      -> std::vector<double>;

  inline auto calcReducedCosts(Array<double const> const &duals,
                               Array<double const> const &solution)
      -> std::vector<double>;

  inline auto calcObjective(Array<double const> const &solution) -> double;

  inline auto calcObjN(int objidx, Array<double const> const &solution)
      -> double;

  inline auto calcSolInfo(Array<double const> const &solution,
                          Array<double const> const &duals, int property)
      -> double;

  inline auto getRowType(int first, int last) -> std::vector<char>;
  auto getRowType(int index) -> char { return getRowType(index, index)[0]; }

  inline void getPresolveBasis(Array<int> const &rowstat,
                               Array<int> const &colstat);

  inline auto getColType(int first, int last) -> std::vector<char>;
  auto getColType(int index) -> char { return getColType(index, index)[0]; }

  inline auto getQRowCoeff(int row, int rowqcol1, int rowqcol2) -> double;

  inline void getQRowQMatrix(int row, Array<int> const &start,
                             Array<int> const &colind,
                             Array<double> const &rowqcoef, int maxcoefs,
                             int *p_ncoefs, int first, int last);

  inline auto getQRowQMatrix(int row, Array<int> const &start,
                             Array<int> const &colind,
                             Array<double> const &rowqcoef, int maxcoefs,
                             int first, int last) -> int;

  inline void getQRowQMatrixTriplets(int row, int *p_ncoefs,
                                     Array<int> const &rowqcol1,
                                     Array<int> const &rowqcol2,
                                     Array<double> const &rowqcoef);

  inline auto getQRowQMatrixTriplets(int row, Array<int> const &rowqcol1,
                                     Array<int> const &rowqcol2,
                                     Array<double> const &rowqcoef) -> int;

  inline void chgQRowCoeff(int row, int rowqcol1, int rowqcol2,
                           double rowqcoef);

  inline void getQRows(int *p_nrows, Array<int> const &rowind);

  inline auto getQRows(Array<int> const &rowind) -> int;

  inline auto getQRows() -> int;

  inline void chgBounds(int nbounds, Array<int const> const &colind,
                        Array<char const> const &bndtype,
                        Array<double const> const &bndval);

  inline auto getNameList(int type, int first, int last)
      -> std::vector<std::string>;

  inline void addMipSol(int length, Array<double const> const &solval,
                        Array<int const> const &colind,
                        std::optional<std::string> const &name);

  inline auto getCutSlack(XPRScut cutind) -> double;

  inline auto getCutMap(int ncuts, Array<XPRScut const> const &cutind)
      -> std::vector<int>;

  inline void getPresolveSol(Array<double> const &x, Array<double> const &slack,
                             Array<double> const &duals,
                             Array<double> const &djs);

  inline void getPresolveSol(Array<double> const &x);

  inline void getSolution(int *status, Array<double> const &x, int first,
                          int last) const;

  auto getSolution(int *status, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getSolution(status, ret, first, last);
    return ret;
  }

  auto getSolution(int first, int last) const -> std::vector<double> {
    int status;
    std::vector<double> ret(std::max(0, last - first + 1));
    getSolution(&status, ret, first, last);
    return ret;
  }

  auto getSolution(int *status, int index) const -> double {
    double ret;
    getSolution(status, &ret, index, index);
    return ret;
  }

  auto getSolution(int index) const -> double {
    int status;
    double ret;
    getSolution(&status, &ret, index, index);
    return ret;
  }

  auto getSolution(int *status) const -> std::vector<double> {
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getSolution(status, ret, 0, length - 1);
    return ret;
  }

  auto getSolution() const -> std::vector<double> {
    int status;
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getSolution(&status, ret, 0, length - 1);
    return ret;
  }

  inline void getSlacks(int *status, Array<double> const &slacks, int first,
                        int last) const;

  auto getSlacks(int *status, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getSlacks(status, ret, first, last);
    return ret;
  }

  auto getSlacks(int first, int last) const -> std::vector<double> {
    int status;
    std::vector<double> ret(std::max(0, last - first + 1));
    getSlacks(&status, ret, first, last);
    return ret;
  }

  auto getSlack(int *status, int index) const -> double {
    double ret;
    getSlacks(status, &ret, index, index);
    return ret;
  }

  auto getSlack(int index) const -> double {
    int status;
    double ret;
    getSlacks(&status, &ret, index, index);
    return ret;
  }

  auto getSlacks(int *status) const -> std::vector<double> {
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getSlacks(status, ret, 0, length - 1);
    return ret;
  }

  auto getSlacks() const -> std::vector<double> {
    int status;
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getSlacks(&status, ret, 0, length - 1);
    return ret;
  }

  inline void getDuals(int *status, Array<double> const &duals, int first,
                       int last) const;

  auto getDuals(int *status, int first, int last) const -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getDuals(status, ret, first, last);
    return ret;
  }

  auto getDuals(int first, int last) const -> std::vector<double> {
    int status;
    std::vector<double> ret(std::max(0, last - first + 1));
    getDuals(&status, ret, first, last);
    return ret;
  }

  auto getDual(int *status, int index) const -> double {
    double ret;
    getDuals(status, &ret, index, index);
    return ret;
  }

  auto getDual(int index) const -> double {
    int status;
    double ret;
    getDuals(&status, &ret, index, index);
    return ret;
  }

  auto getDuals(int *status) const -> std::vector<double> {
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getDuals(status, ret, 0, length - 1);
    return ret;
  }

  auto getDuals() const -> std::vector<double> {
    int status;
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getDuals(&status, ret, 0, length - 1);
    return ret;
  }

  inline void getRedCosts(int *status, Array<double> const &djs, int first,
                          int last) const;

  auto getRedCosts(int *status, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getRedCosts(status, ret, first, last);
    return ret;
  }

  auto getRedCosts(int first, int last) const -> std::vector<double> {
    int status;
    std::vector<double> ret(std::max(0, last - first + 1));
    getRedCosts(&status, ret, first, last);
    return ret;
  }

  auto getRedCost(int *status, int index) const -> double {
    double ret;
    getRedCosts(status, &ret, index, index);
    return ret;
  }

  auto getRedCost(int index) const -> double {
    int status;
    double ret;
    getRedCosts(&status, &ret, index, index);
    return ret;
  }

  auto getRedCosts(int *status) const -> std::vector<double> {
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getRedCosts(status, ret, 0, length - 1);
    return ret;
  }

  auto getRedCosts() const -> std::vector<double> {
    int status;
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getRedCosts(&status, ret, 0, length - 1);
    return ret;
  }

  inline void getLpSol(Array<double> const &x, Array<double> const &slack,
                       Array<double> const &duals, Array<double> const &djs);

  inline void getLpSol(Array<double> const &x);

  [[deprecated(
      "since 44.00, use getSolution(), getCallbackSolution(), getSlacks(), or "
      "getCallbackSlacks(), getRedCosts(), getCallbackRedCosts(), getDuals(), "
      "or getCallbackDuals()")]] inline void
  getLpSolVal(int col, int row, double *p_x, double *p_slack, double *p_dual,
              double *p_dj);

  [[deprecated("since 44.00, use getSolution(), getCallbackSolution(), "
               "getSlacks(), or getCallbackSlacks()")]] inline void
  getMipSol(Array<double> const &x, Array<double> const &slack);

  [[deprecated("since 44.00, use getSolution(), getCallbackSolution(), "
               "getSlacks(), or getCallbackSlacks()")]] inline void
  getMipSol(Array<double> const &x);

  [[deprecated("since 44.00, use getSolution(), getCallbackSolution(), "
               "getSlacks(), or getCallbackSlacks()")]] inline void
  getMipSolVal(int col, int row, double *p_x, double *p_slack);

  inline void getCallbackSolution(bool *p_available, Array<double> const &x,
                                  int first, int last) const;

  auto getCallbackSolution(bool *p_available, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackSolution(p_available, ret, first, last);
    return ret;
  }

  auto getCallbackSolution(int first, int last) const -> std::vector<double> {
    bool p_available;
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackSolution(&p_available, ret, first, last);
    return ret;
  }

  auto getCallbackSolution(bool *p_available, int index) const -> double {
    double ret;
    getCallbackSolution(p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackSolution(int index) const -> double {
    bool p_available;
    double ret;
    getCallbackSolution(&p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackSolution(bool *p_available) const -> std::vector<double> {
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getCallbackSolution(p_available, ret, 0, length - 1);
    return ret;
  }

  auto getCallbackSolution() const -> std::vector<double> {
    bool p_available;
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getCallbackSolution(&p_available, ret, 0, length - 1);
    return ret;
  }

  inline void getCallbackSlacks(bool *p_available, Array<double> const &slacks,
                                int first, int last) const;

  auto getCallbackSlacks(bool *p_available, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackSlacks(p_available, ret, first, last);
    return ret;
  }

  auto getCallbackSlacks(int first, int last) const -> std::vector<double> {
    bool p_available;
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackSlacks(&p_available, ret, first, last);
    return ret;
  }

  auto getCallbackSlack(bool *p_available, int index) const -> double {
    double ret;
    getCallbackSlacks(p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackSlack(int index) const -> double {
    bool p_available;
    double ret;
    getCallbackSlacks(&p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackSlacks(bool *p_available) const -> std::vector<double> {
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getCallbackSlacks(p_available, ret, 0, length - 1);
    return ret;
  }

  auto getCallbackSlacks() const -> std::vector<double> {
    bool p_available;
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getCallbackSlacks(&p_available, ret, 0, length - 1);
    return ret;
  }

  inline void getCallbackDuals(bool *p_available, Array<double> const &duals,
                               int first, int last) const;

  auto getCallbackDuals(bool *p_available, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackDuals(p_available, ret, first, last);
    return ret;
  }

  auto getCallbackDuals(int first, int last) const -> std::vector<double> {
    bool p_available;
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackDuals(&p_available, ret, first, last);
    return ret;
  }

  auto getCallbackDual(bool *p_available, int index) const -> double {
    double ret;
    getCallbackDuals(p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackDual(int index) const -> double {
    bool p_available;
    double ret;
    getCallbackDuals(&p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackDuals(bool *p_available) const -> std::vector<double> {
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getCallbackDuals(p_available, ret, 0, length - 1);
    return ret;
  }

  auto getCallbackDuals() const -> std::vector<double> {
    bool p_available;
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getCallbackDuals(&p_available, ret, 0, length - 1);
    return ret;
  }

  inline void getCallbackRedCosts(bool *p_available, Array<double> const &djs,
                                  int first, int last) const;

  auto getCallbackRedCosts(bool *p_available, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackRedCosts(p_available, ret, first, last);
    return ret;
  }

  auto getCallbackRedCosts(int first, int last) const -> std::vector<double> {
    bool p_available;
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackRedCosts(&p_available, ret, first, last);
    return ret;
  }

  auto getCallbackRedCost(bool *p_available, int index) const -> double {
    double ret;
    getCallbackRedCosts(p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackRedCost(int index) const -> double {
    bool p_available;
    double ret;
    getCallbackRedCosts(&p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackRedCosts(bool *p_available) const -> std::vector<double> {
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getCallbackRedCosts(p_available, ret, 0, length - 1);
    return ret;
  }

  auto getCallbackRedCosts() const -> std::vector<double> {
    bool p_available;
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getCallbackRedCosts(&p_available, ret, 0, length - 1);
    return ret;
  }

  inline void getCallbackPresolveSolution(bool *p_available,
                                          Array<double> const &x, int first,
                                          int last) const;

  auto getCallbackPresolveSolution(bool *p_available, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackPresolveSolution(p_available, ret, first, last);
    return ret;
  }

  auto getCallbackPresolveSolution(int first, int last) const
      -> std::vector<double> {
    bool p_available;
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackPresolveSolution(&p_available, ret, first, last);
    return ret;
  }

  auto getCallbackPresolveSolution(bool *p_available, int index) const
      -> double {
    double ret;
    getCallbackPresolveSolution(p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackPresolveSolution(int index) const -> double {
    bool p_available;
    double ret;
    getCallbackPresolveSolution(&p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackPresolveSolution(bool *p_available) const
      -> std::vector<double> {
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getCallbackPresolveSolution(p_available, ret, 0, length - 1);
    return ret;
  }

  auto getCallbackPresolveSolution() const -> std::vector<double> {
    bool p_available;
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getCallbackPresolveSolution(&p_available, ret, 0, length - 1);
    return ret;
  }

  inline void getCallbackPresolveSlacks(bool *p_available,
                                        Array<double> const &slacks, int first,
                                        int last) const;

  auto getCallbackPresolveSlacks(bool *p_available, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackPresolveSlacks(p_available, ret, first, last);
    return ret;
  }

  auto getCallbackPresolveSlacks(int first, int last) const
      -> std::vector<double> {
    bool p_available;
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackPresolveSlacks(&p_available, ret, first, last);
    return ret;
  }

  auto getCallbackPresolveSlack(bool *p_available, int index) const -> double {
    double ret;
    getCallbackPresolveSlacks(p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackPresolveSlack(int index) const -> double {
    bool p_available;
    double ret;
    getCallbackPresolveSlacks(&p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackPresolveSlacks(bool *p_available) const
      -> std::vector<double> {
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getCallbackPresolveSlacks(p_available, ret, 0, length - 1);
    return ret;
  }

  auto getCallbackPresolveSlacks() const -> std::vector<double> {
    bool p_available;
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getCallbackPresolveSlacks(&p_available, ret, 0, length - 1);
    return ret;
  }

  inline void getCallbackPresolveDuals(bool *p_available,
                                       Array<double> const &duals, int first,
                                       int last) const;

  auto getCallbackPresolveDuals(bool *p_available, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackPresolveDuals(p_available, ret, first, last);
    return ret;
  }

  auto getCallbackPresolveDuals(int first, int last) const
      -> std::vector<double> {
    bool p_available;
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackPresolveDuals(&p_available, ret, first, last);
    return ret;
  }

  auto getCallbackPresolveDual(bool *p_available, int index) const -> double {
    double ret;
    getCallbackPresolveDuals(p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackPresolveDual(int index) const -> double {
    bool p_available;
    double ret;
    getCallbackPresolveDuals(&p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackPresolveDuals(bool *p_available) const
      -> std::vector<double> {
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getCallbackPresolveDuals(p_available, ret, 0, length - 1);
    return ret;
  }

  auto getCallbackPresolveDuals() const -> std::vector<double> {
    bool p_available;
    int length = attributes.getInputRows();
    std::vector<double> ret(length);
    getCallbackPresolveDuals(&p_available, ret, 0, length - 1);
    return ret;
  }

  inline void getCallbackPresolveRedCosts(bool *p_available,
                                          Array<double> const &djs, int first,
                                          int last) const;

  auto getCallbackPresolveRedCosts(bool *p_available, int first, int last) const
      -> std::vector<double> {
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackPresolveRedCosts(p_available, ret, first, last);
    return ret;
  }

  auto getCallbackPresolveRedCosts(int first, int last) const
      -> std::vector<double> {
    bool p_available;
    std::vector<double> ret(std::max(0, last - first + 1));
    getCallbackPresolveRedCosts(&p_available, ret, first, last);
    return ret;
  }

  auto getCallbackPresolveRedCost(bool *p_available, int index) const
      -> double {
    double ret;
    getCallbackPresolveRedCosts(p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackPresolveRedCost(int index) const -> double {
    bool p_available;
    double ret;
    getCallbackPresolveRedCosts(&p_available, &ret, index, index);
    return ret;
  }

  auto getCallbackPresolveRedCosts(bool *p_available) const
      -> std::vector<double> {
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getCallbackPresolveRedCosts(p_available, ret, 0, length - 1);
    return ret;
  }

  auto getCallbackPresolveRedCosts() const -> std::vector<double> {
    bool p_available;
    int length = attributes.getInputCols();
    std::vector<double> ret(length);
    getCallbackPresolveRedCosts(&p_available, ret, 0, length - 1);
    return ret;
  }

  inline void chgObj(int ncols, Array<int const> const &colind,
                     Array<double const> const &objcoef);
  void chgObj(int index, double value) { chgObj(1, &index, &value); }

  inline void chgCoef(int row, int col, double coef);

  inline void chgMCoef(int ncoefs, Array<int const> const &rowind,
                       Array<int const> const &colind,
                       Array<double const> const &rowcoef);

  inline void chgMCoef(XPRSint64 ncoefs, Array<int const> const &rowind,
                       Array<int const> const &colind,
                       Array<double const> const &rowcoef);

  inline void chgMQObj(int ncoefs, Array<int const> const &objqcol1,
                       Array<int const> const &objqcol2,
                       Array<double const> const &objqcoef);

  inline void chgMQObj(XPRSint64 ncoefs, Array<int const> const &objqcol1,
                       Array<int const> const &objqcol2,
                       Array<double const> const &objqcoef);

  inline void chgQObj(int objqcol1, int objqcol2, double objqcoef);

  inline void chgRhs(int nrows, Array<int const> const &rowind,
                     Array<double const> const &rhs);
  void chgRhs(int index, double value) { chgRhs(1, &index, &value); }

  inline void chgRhsRange(int nrows, Array<int const> const &rowind,
                          Array<double const> const &rng);
  void chgRhsRange(int index, double value) { chgRhsRange(1, &index, &value); }

  inline void chgRowType(int nrows, Array<int const> const &rowind,
                         Array<char const> const &rowtype);
  void chgRowType(int index, char value) { chgRowType(1, &index, &value); }

  inline void msAddJob(std::optional<std::string> const &description,
                       int ninitial, Array<int const> const &colind,
                       Array<double const> const &initial, int nintcontrols,
                       Array<int const> const &intcontrolid,
                       Array<int const> const &intcontrolval, int ndblcontrols,
                       Array<int const> const &dblcontrolid,
                       Array<double const> const &dblcontrolval, void *data);

  inline void msAddPreset(std::optional<std::string> const &description,
                          int preset, int maxjobs, void *data);

  inline void
  msAddCustomPreset(std::optional<std::string> const &description, int preset,
                    int maxjobs, int ninitial, Array<int const> const &colind,
                    Array<double const> const &initial, int nintcontrols,
                    Array<int const> const &intcontrolid,
                    Array<int const> const &intcontrolval, int ndblcontrols,
                    Array<int const> const &dblcontrolid,
                    Array<double const> const &dblcontrolval, void *data);

  inline void nlpSetFunctionError();

  inline void nlpPrintEvalInfo();

  inline void nlpValidate();

  inline void nlpOptimize(std::optional<std::string> const &flags);

  [[deprecated(
      "since 44.00, use getSolution(), getCallbackSolution(), getSlacks(), or "
      "getCallbackSlacks(), getRedCosts(), getCallbackRedCosts(), getDuals(), "
      "or getCallbackDuals()")]] inline void
  getNlpsol(Array<double> const &x, Array<double> const &slack,
            Array<double> const &duals, Array<double> const &djs);

  inline void nlpCurrentIV();

  inline void nlpValidateRow(int row);

  inline void nlpValidateKKT(int mode, int respectbasis, int updatemult,
                             double violtarget);

  inline void msClear();

  inline auto nlpEvaluateFormula(int parsed, Array<int const> const &type,
                                 Array<double const> const &values) -> double;

  inline void nlpValidateVector(Array<double const> const &solution,
                                double *p_suminf, double *p_sumscaledinf,
                                double *p_objval);

  inline void nlpDelUserFunction(int type);

  inline void nlpImportLibFunc(std::optional<std::string> const &libname,
                               std::optional<std::string> const &funcname,
                               XPRSfunctionptr *p_function, int *p_status);

  inline void nlpAddFormulas(int ncoefs, Array<int const> const &rowind,
                             Array<int const> const &formulastart, int parsed,
                             Array<int const> const &type,
                             Array<double const> const &value);

  inline void nlpChgFormulaStr(int row,
                               std::optional<std::string> const &formula);

  inline void nlpChgFormula(int row, int parsed, Array<int const> const &type,
                            Array<double const> const &value);

  inline void nlpGetFormula(int row, int parsed, int maxtypes, int *p_ntypes,
                            Array<int> const &type, Array<double> const &value);

  inline void nlpGetFormulaRows(int *p_nformulas, Array<int> const &rowind);

  inline void nlpLoadFormulas(int nnlpcoefs, Array<int const> const &rowind,
                              Array<int const> const &formulastart, int parsed,
                              Array<int const> const &type,
                              Array<double const> const &value);

  inline void nlpDelFormulas(int nformulas, Array<int const> const &rowind);

  inline void nlpGetFormulaStr(int row, char *formula, int maxbytes,
                               int *p_nbytes);

  inline void nlpSetInitVal(int nvars, Array<int const> const &colind,
                            Array<double const> const &initial);
  void nlpSetInitVal(int index, double value) {
    nlpSetInitVal(1, &index, &value);
  }

  inline void slpGetCoefFormula(int row, int col, double *p_factor, int parsed,
                                int maxtypes, int *p_ntypes,
                                Array<int> const &type,
                                Array<double> const &value);

  inline void slpGetCoefs(int *p_ncoefs, Array<int> const &rowind,
                          Array<int> const &colind);

  inline void slpLoadCoefs(int ncoefs, Array<int const> const &rowind,
                           Array<int const> const &colind,
                           Array<double const> const &factor,
                           Array<int const> const &formulastart, int parsed,
                           Array<int const> const &type,
                           Array<double const> const &coef);

  inline void slpDelCoefs(int ncoefs, Array<int const> const &rowind,
                          Array<int const> const &colind);
  void slpDelCoefs(int index, int value) { slpDelCoefs(1, &index, &value); }

  inline void slpGetCoefStr(int row, int col, double *p_factor, char *formula,
                            int maxbytes, int *p_nbytes);

  inline void slpSetDetRow(int nvars, Array<int const> const &colind,
                           Array<int const> const &rowind);
  void slpSetDetRow(int index, int value) { slpSetDetRow(1, &index, &value); }

  inline void slpAddCoefs(int ncoefs, Array<int const> const &rowind,
                          Array<int const> const &colind,
                          Array<double const> const &factor,
                          Array<int const> const &formulastart, int parsed,
                          Array<int const> const &type,
                          Array<double const> const &value);

  inline void slpChgCoefStr(int row, int col, double *factor,
                            std::optional<std::string> const &formula);

  inline void slpChgCoef(int row, int col, double *factor, int parsed,
                         Array<int const> const &type,
                         Array<double const> const &value);

  inline void slpCascadeSol();

  inline void slpCascadeOrder();

  inline auto slpChgRowStatus(int row) -> int;

  inline void slpChgRowWt(int row, double *weight);

  inline void slpChgDeltaType(int nvars, Array<int const> const &varind,
                              Array<int const> const &deltatypes,
                              Array<double const> const &values);

  inline void slpChgCascadeNLimit(int col, int limit);

  inline void slpConstruct();

  inline void slpGetRowStatus(int row, int *p_status);

  inline auto slpGetRowWT(int row) -> double;

  inline auto slpEvaluateCoef(int row, int col) -> double;

  inline void slpReInitialize();

  inline void slpUnConstruct();

  inline void slpUpdateLinearization();

  inline void slpFixPenalties(int *p_status);

  inline void nlpPostsolveProb();

  inline void nlpCalcSlacks(Array<double const> const &solution,
                            Array<double> const &slack);

  inline auto addLplogCallback(std::function<int(XPRSProblem &)> callback,
                               int prio = 0) -> CallbackHandle;

  inline void removeLplogCallback(CallbackHandle const &callback);

  inline void removeLplogCallbacks();

  inline auto addMiplogCallback(std::function<int(XPRSProblem &)> callback,
                                int prio = 0) -> CallbackHandle;

  inline void removeMiplogCallback(CallbackHandle const &callback);

  inline void removeMiplogCallbacks();

  inline auto addCutlogCallback(std::function<int(XPRSProblem &)> callback,
                                int prio = 0) -> CallbackHandle;

  inline void removeCutlogCallback(CallbackHandle const &callback);

  inline void removeCutlogCallbacks();

  inline auto addBarlogCallback(std::function<int(XPRSProblem &)> callback,
                                int prio = 0) -> CallbackHandle;

  inline void removeBarlogCallback(CallbackHandle const &callback);

  inline void removeBarlogCallbacks();

  inline auto
  addOptnodeCallback(std::function<void(XPRSProblem &, int *)> callback,
                     int prio = 0) -> CallbackHandle;

  inline void removeOptnodeCallback(CallbackHandle const &callback);

  inline void removeOptnodeCallbacks();

  inline auto
  addPrenodeCallback(std::function<void(XPRSProblem &, int *)> callback,
                     int prio = 0) -> CallbackHandle;

  inline void removePrenodeCallback(CallbackHandle const &callback);

  inline void removePrenodeCallbacks();

  inline auto addInfnodeCallback(std::function<void(XPRSProblem &)> callback,
                                 int prio = 0) -> CallbackHandle;

  inline void removeInfnodeCallback(CallbackHandle const &callback);

  inline void removeInfnodeCallbacks();

  inline auto
  addNodecutoffCallback(std::function<void(XPRSProblem &, int)> callback,
                        int prio = 0) -> CallbackHandle;

  inline void removeNodecutoffCallback(CallbackHandle const &callback);

  inline void removeNodecutoffCallbacks();

  inline auto addIntsolCallback(std::function<void(XPRSProblem &)> callback,
                                int prio = 0) -> CallbackHandle;

  inline void removeIntsolCallback(CallbackHandle const &callback);

  inline void removeIntsolCallbacks();

  inline auto addPreIntsolCallback(
      std::function<void(XPRSProblem &, int, int *, double *)> callback,
      int prio = 0) -> CallbackHandle;

  inline void removePreIntsolCallback(CallbackHandle const &callback);

  inline void removePreIntsolCallbacks();

  inline auto addMessageCallback(
      std::function<void(XPRSProblem &, char const *, int, int)> callback,
      int prio = 0) -> CallbackHandle;

  inline void removeMessageCallback(CallbackHandle const &callback);

  inline void removeMessageCallbacks();

  inline auto addMipThreadCallback(
      std::function<void(XPRSProblem &, XPRSProblem &)> callback, int prio = 0)
      -> CallbackHandle;

  inline void removeMipThreadCallback(CallbackHandle const &callback);

  inline void removeMipThreadCallbacks();

  inline auto
  addMipThreadDestroyCallback(std::function<void(XPRSProblem &)> callback,
                              int prio = 0) -> CallbackHandle;

  inline void removeMipThreadDestroyCallback(CallbackHandle const &callback);

  inline void removeMipThreadDestroyCallbacks();

  inline auto
  addNewnodeCallback(std::function<void(XPRSProblem &, int, int, int)> callback,
                     int prio = 0) -> CallbackHandle;

  inline void removeNewnodeCallback(CallbackHandle const &callback);

  inline void removeNewnodeCallbacks();

  inline auto
  addBarIterationCallback(std::function<void(XPRSProblem &, int *)> callback,
                          int prio = 0) -> CallbackHandle;

  inline void removeBarIterationCallback(CallbackHandle const &callback);

  inline void removeBarIterationCallbacks();

  inline auto addPresolveCallback(std::function<void(XPRSProblem &)> callback,
                                  int prio = 0) -> CallbackHandle;

  inline void removePresolveCallback(CallbackHandle const &callback);

  inline void removePresolveCallbacks();

  inline auto addChangeBranchObjectCallback(
      std::function<void(XPRSProblem &, BranchObject *, BranchObject **)>
          callback,
      int prio = 0) -> CallbackHandle;

  inline void removeChangeBranchObjectCallback(CallbackHandle const &callback);

  inline void removeChangeBranchObjectCallbacks();

  inline auto
  addComputeRestartCallback(std::function<void(XPRSProblem &)> callback,
                            int prio = 0) -> CallbackHandle;

  inline void removeComputeRestartCallback(CallbackHandle const &callback);

  inline void removeComputeRestartCallbacks();

  inline auto
  addNodeLPSolvedCallback(std::function<void(XPRSProblem &)> callback,
                          int prio = 0) -> CallbackHandle;

  inline void removeNodeLPSolvedCallback(CallbackHandle const &callback);

  inline void removeNodeLPSolvedCallbacks();

  inline auto addGapNotifyCallback(
      std::function<void(XPRSProblem &, double *, double *, double *, double *)>
          callback,
      int prio = 0) -> CallbackHandle;

  inline void removeGapNotifyCallback(CallbackHandle const &callback);

  inline void removeGapNotifyCallbacks();

  inline auto addUserSolNotifyCallback(
      std::function<void(XPRSProblem &, char const *, int)> callback,
      int prio = 0) -> CallbackHandle;

  inline void removeUserSolNotifyCallback(CallbackHandle const &callback);

  inline void removeUserSolNotifyCallbacks();

  inline auto
  addBeforeSolveCallback(std::function<void(XPRSProblem &)> callback,
                         int prio = 0) -> CallbackHandle;

  inline void removeBeforeSolveCallback(CallbackHandle const &callback);

  inline void removeBeforeSolveCallbacks();

  inline auto
  addBeforeObjectiveCallback(std::function<void(XPRSProblem &)> callback,
                             int prio = 0) -> CallbackHandle;

  inline void removeBeforeObjectiveCallback(CallbackHandle const &callback);

  inline void removeBeforeObjectiveCallbacks();

  inline auto
  addAfterObjectiveCallback(std::function<void(XPRSProblem &)> callback,
                            int prio = 0) -> CallbackHandle;

  inline void removeAfterObjectiveCallback(CallbackHandle const &callback);

  inline void removeAfterObjectiveCallbacks();

  inline auto addCheckTimeCallback(std::function<int(XPRSProblem &)> callback,
                                   int prio = 0) -> CallbackHandle;

  inline void removeCheckTimeCallback(CallbackHandle const &callback);

  inline void removeCheckTimeCallbacks();

  inline auto
  addCutRoundCallback(std::function<void(XPRSProblem &, int, int *)> callback,
                      int prio = 0) -> CallbackHandle;

  inline void removeCutRoundCallback(CallbackHandle const &callback);

  inline void removeCutRoundCallbacks();

  inline auto
  addSlpCascadeEndCallback(std::function<int(XPRSProblem &)> callback,
                           int prio = 0) -> CallbackHandle;

  inline void removeSlpCascadeEndCallback(CallbackHandle const &callback);

  inline void removeSlpCascadeEndCallbacks();

  inline auto
  addSlpCascadeStartCallback(std::function<int(XPRSProblem &)> callback,
                             int prio = 0) -> CallbackHandle;

  inline void removeSlpCascadeStartCallback(CallbackHandle const &callback);

  inline void removeSlpCascadeStartCallbacks();

  inline auto
  addSlpCascadeVarCallback(std::function<int(XPRSProblem &, int)> callback,
                           int prio = 0) -> CallbackHandle;

  inline void removeSlpCascadeVarCallback(CallbackHandle const &callback);

  inline void removeSlpCascadeVarCallbacks();

  inline auto
  addSlpCascadeVarFailCallback(std::function<int(XPRSProblem &, int)> callback,
                               int prio = 0) -> CallbackHandle;

  inline void removeSlpCascadeVarFailCallback(CallbackHandle const &callback);

  inline void removeSlpCascadeVarFailCallbacks();

  inline auto
  addSlpConstructCallback(std::function<int(XPRSProblem &)> callback,
                          int prio = 0) -> CallbackHandle;

  inline void removeSlpConstructCallback(CallbackHandle const &callback);

  inline void removeSlpConstructCallbacks();

  inline auto addSlpIntSolCallback(std::function<int(XPRSProblem &)> callback,
                                   int prio = 0) -> CallbackHandle;

  inline void removeSlpIntSolCallback(CallbackHandle const &callback);

  inline void removeSlpIntSolCallbacks();

  inline auto addSlpIterEndCallback(std::function<int(XPRSProblem &)> callback,
                                    int prio = 0) -> CallbackHandle;

  inline void removeSlpIterEndCallback(CallbackHandle const &callback);

  inline void removeSlpIterEndCallbacks();

  inline auto
  addSlpIterStartCallback(std::function<int(XPRSProblem &)> callback,
                          int prio = 0) -> CallbackHandle;

  inline void removeSlpIterStartCallback(CallbackHandle const &callback);

  inline void removeSlpIterStartCallbacks();

  inline auto
  addSlpIterVarCallback(std::function<int(XPRSProblem &, int)> callback,
                        int prio = 0) -> CallbackHandle;

  inline void removeSlpIterVarCallback(CallbackHandle const &callback);

  inline void removeSlpIterVarCallbacks();

  inline auto
  addSlpDrColCallback(std::function<int(XPRSProblem &, int, int, double,
                                        double *, double, double)>
                          callback,
                      int prio = 0) -> CallbackHandle;

  inline void removeSlpDrColCallback(CallbackHandle const &callback);

  inline void removeSlpDrColCallbacks();

  inline auto addMsJobStartCallback(
      std::function<int(XPRSProblem &, void *, char const *, int *)> callback,
      int prio = 0) -> CallbackHandle;

  inline void removeMsJobStartCallback(CallbackHandle const &callback);

  inline void removeMsJobStartCallbacks();

  inline auto addMsJobEndCallback(
      std::function<int(XPRSProblem &, void *, char const *, int *)> callback,
      int prio = 0) -> CallbackHandle;

  inline void removeMsJobEndCallback(CallbackHandle const &callback);

  inline void removeMsJobEndCallbacks();

  inline auto addMsWinnerCallback(
      std::function<int(XPRSProblem &, void *, char const *)> callback,
      int prio = 0) -> CallbackHandle;

  inline void removeMsWinnerCallback(CallbackHandle const &callback);

  inline void removeMsWinnerCallbacks();

  inline auto addNlpCoefEvalErrorCallback(
      std::function<int(XPRSProblem &, int, int)> callback, int prio = 0)
      -> CallbackHandle;

  inline void removeNlpCoefEvalErrorCallback(CallbackHandle const &callback);

  inline void removeNlpCoefEvalErrorCallbacks();

  inline auto addSlpPreUpdateLinearizationCallback(
      std::function<int(XPRSProblem &, int *)> callback, int prio = 0)
      -> CallbackHandle;

  inline void
  removeSlpPreUpdateLinearizationCallback(CallbackHandle const &callback);

  inline void removeSlpPreUpdateLinearizationCallbacks();

public:
  class Attributes {
    XPRSProblem *p;
    friend class XPRSProblem;
    Attributes(XPRSProblem *p) : p(p) {}

  public:
    auto getRows() const -> int { return p->getIntAttrib(XPRS_ROWS); }

    auto getSets() const -> int { return p->getIntAttrib(XPRS_SETS); }

    auto getSetMembers() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_SETMEMBERS);
    }

    auto getElems() const -> XPRSint64 { return p->getLongAttrib(XPRS_ELEMS); }

    auto getPrimalInfeas() const -> int {
      return p->getIntAttrib(XPRS_PRIMALINFEAS);
    }

    auto getDualInfeas() const -> int {
      return p->getIntAttrib(XPRS_DUALINFEAS);
    }

    auto getSimplexIter() const -> int {
      return p->getIntAttrib(XPRS_SIMPLEXITER);
    }

    auto getLpStatus() const -> xpress::LPStatus {
      return static_cast<xpress::LPStatus>(p->getIntAttrib(XPRS_LPSTATUS));
    }

    auto getMipStatus() const -> xpress::MIPStatus {
      return static_cast<xpress::MIPStatus>(p->getIntAttrib(XPRS_MIPSTATUS));
    }

    auto getCuts() const -> int { return p->getIntAttrib(XPRS_CUTS); }

    auto getNodes() const -> int { return p->getIntAttrib(XPRS_NODES); }

    auto getNodeDepth() const -> int { return p->getIntAttrib(XPRS_NODEDEPTH); }

    auto getActiveNodes() const -> int {
      return p->getIntAttrib(XPRS_ACTIVENODES);
    }

    auto getMipSolNode() const -> int {
      return p->getIntAttrib(XPRS_MIPSOLNODE);
    }

    auto getMipSols() const -> int { return p->getIntAttrib(XPRS_MIPSOLS); }

    auto getCols() const -> int { return p->getIntAttrib(XPRS_COLS); }

    auto getSpareRows() const -> int { return p->getIntAttrib(XPRS_SPAREROWS); }

    auto getSpareCols() const -> int { return p->getIntAttrib(XPRS_SPARECOLS); }

    auto getSpareElems() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_SPAREELEMS);
    }

    auto getSpareMipEnts() const -> int {
      return p->getIntAttrib(XPRS_SPAREMIPENTS);
    }

    auto getErrorCode() const -> int { return p->getIntAttrib(XPRS_ERRORCODE); }

    auto getMipInfeas() const -> int { return p->getIntAttrib(XPRS_MIPINFEAS); }

    auto getPresolveState() const -> int {
      return p->getIntAttrib(XPRS_PRESOLVESTATE);
    }

    auto getParentNode() const -> int {
      return p->getIntAttrib(XPRS_PARENTNODE);
    }

    auto getNameLength() const -> int {
      return p->getIntAttrib(XPRS_NAMELENGTH);
    }

    auto getQElems() const -> int { return p->getIntAttrib(XPRS_QELEMS); }

    auto getNumIIS() const -> int { return p->getIntAttrib(XPRS_NUMIIS); }

    auto getMipEnts() const -> int { return p->getIntAttrib(XPRS_MIPENTS); }

    auto getBranchVar() const -> int { return p->getIntAttrib(XPRS_BRANCHVAR); }

    auto getMipThreadID() const -> int {
      return p->getIntAttrib(XPRS_MIPTHREADID);
    }

    auto getAlgorithm() const -> int { return p->getIntAttrib(XPRS_ALGORITHM); }

    auto getCrossoverIter() const -> int {
      return p->getIntAttrib(XPRS_CROSSOVERITER);
    }

    auto getSolStatus() const -> xpress::SolStatus {
      return static_cast<xpress::SolStatus>(p->getIntAttrib(XPRS_SOLSTATUS));
    }

    auto getCutRounds() const -> int { return p->getIntAttrib(XPRS_CUTROUNDS); }

    auto getOriginalRows() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALROWS);
    }

    auto getCallbackCount_OptNode() const -> int {
      return p->getIntAttrib(XPRS_CALLBACKCOUNT_OPTNODE);
    }

    auto getCallbackCount_CutMgr() const -> int {
      return p->getIntAttrib(XPRS_CALLBACKCOUNT_CUTMGR);
    }

    auto getSystemMemory() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_SYSTEMMEMORY);
    }

    auto getOriginalQElems() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALQELEMS);
    }

    auto getMaxProbNameLength() const -> int {
      return p->getIntAttrib(XPRS_MAXPROBNAMELENGTH);
    }

    auto getStopStatus() const -> int {
      return p->getIntAttrib(XPRS_STOPSTATUS);
    }

    auto getOriginalMipEnts() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALMIPENTS);
    }

    auto getOriginalSets() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALSETS);
    }

    auto getOriginalSetMembers() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_ORIGINALSETMEMBERS);
    }

    auto getSpareSets() const -> int { return p->getIntAttrib(XPRS_SPARESETS); }

    auto getSpareSetElems() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_SPARESETELEMS);
    }

    auto getChecksOnMaxTime() const -> int {
      return p->getIntAttrib(XPRS_CHECKSONMAXTIME);
    }

    auto getChecksOnMaxCutTime() const -> int {
      return p->getIntAttrib(XPRS_CHECKSONMAXCUTTIME);
    }

    auto getOriginalCols() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALCOLS);
    }

    auto getQCElems() const -> int { return p->getIntAttrib(XPRS_QCELEMS); }

    auto getQConstraints() const -> int {
      return p->getIntAttrib(XPRS_QCONSTRAINTS);
    }

    auto getOriginalQCElems() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALQCELEMS);
    }

    auto getOriginalQConstraints() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALQCONSTRAINTS);
    }

    auto getPeakTotalTreeMemoryUsage() const -> int {
      return p->getIntAttrib(XPRS_PEAKTOTALTREEMEMORYUSAGE);
    }

    auto getCurrentNode() const -> int {
      return p->getIntAttrib(XPRS_CURRENTNODE);
    }

    auto getTreeMemoryUsage() const -> int {
      return p->getIntAttrib(XPRS_TREEMEMORYUSAGE);
    }

    auto getTreeFileSize() const -> int {
      return p->getIntAttrib(XPRS_TREEFILESIZE);
    }

    auto getTreeFileUsage() const -> int {
      return p->getIntAttrib(XPRS_TREEFILEUSAGE);
    }

    auto getIndicators() const -> int {
      return p->getIntAttrib(XPRS_INDICATORS);
    }

    auto getOriginalIndicators() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALINDICATORS);
    }

    auto getCoresPerCPUDetected() const -> int {
      return p->getIntAttrib(XPRS_CORESPERCPUDETECTED);
    }

    auto getCPUsDetected() const -> int {
      return p->getIntAttrib(XPRS_CPUSDETECTED);
    }

    auto getCoresDetected() const -> int {
      return p->getIntAttrib(XPRS_CORESDETECTED);
    }

    auto getPhysicalCoresDetected() const -> int {
      return p->getIntAttrib(XPRS_PHYSICALCORESDETECTED);
    }

    auto getPhysicalCoresPerCPUDetected() const -> int {
      return p->getIntAttrib(XPRS_PHYSICALCORESPERCPUDETECTED);
    }

    auto getOptimizeTypeUsed() const -> xpress::OptimizeType {
      return static_cast<xpress::OptimizeType>(
          p->getIntAttrib(XPRS_OPTIMIZETYPEUSED));
    }

    auto getBarSing() const -> int { return p->getIntAttrib(XPRS_BARSING); }

    auto getBarSingR() const -> int { return p->getIntAttrib(XPRS_BARSINGR); }

    auto getPresolveIndex() const -> int {
      return p->getIntAttrib(XPRS_PRESOLVEINDEX);
    }

    auto getCurrentMemory() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_CURRENTMEMORY);
    }

    auto getPeakMemory() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_PEAKMEMORY);
    }

    auto getCones() const -> int { return p->getIntAttrib(XPRS_CONES); }

    auto getConeElems() const -> int { return p->getIntAttrib(XPRS_CONEELEMS); }

    auto getTotalMemory() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_TOTALMEMORY);
    }

    auto getAvailableMemory() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_AVAILABLEMEMORY);
    }

    auto getPwlCons() const -> int { return p->getIntAttrib(XPRS_PWLCONS); }

    auto getPwlPoints() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_PWLPOINTS);
    }

    auto getGenCons() const -> int { return p->getIntAttrib(XPRS_GENCONS); }

    auto getGenConCols() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_GENCONCOLS);
    }

    auto getGenConVals() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_GENCONVALS);
    }

    auto getTreeRestarts() const -> int {
      return p->getIntAttrib(XPRS_TREERESTARTS);
    }

    auto getOriginalPwls() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALPWLS);
    }

    auto getOriginalPwlpoints() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_ORIGINALPWLPOINTS);
    }

    auto getOriginalGencons() const -> int {
      return p->getIntAttrib(XPRS_ORIGINALGENCONS);
    }

    auto getOriginalGenconCols() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_ORIGINALGENCONCOLS);
    }

    auto getOriginalGenconVals() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_ORIGINALGENCONVALS);
    }

    auto getComputeExecutions() const -> int {
      return p->getIntAttrib(XPRS_COMPUTEEXECUTIONS);
    }

    auto getMemoryLimitDetected() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_MEMORYLIMITDETECTED);
    }

    auto getRestarts() const -> int { return p->getIntAttrib(XPRS_RESTARTS); }

    auto getSolveStatus() const -> xpress::SolveStatus {
      return static_cast<xpress::SolveStatus>(
          p->getIntAttrib(XPRS_SOLVESTATUS));
    }

    auto getGlobalBoundingboxApplied() const -> int {
      return p->getIntAttrib(XPRS_GLOBALBOUNDINGBOXAPPLIED);
    }

    auto getObjectives() const -> int {
      return p->getIntAttrib(XPRS_OBJECTIVES);
    }

    auto getSolvedObjs() const -> int {
      return p->getIntAttrib(XPRS_SOLVEDOBJS);
    }

    auto getObjsToSolve() const -> int {
      return p->getIntAttrib(XPRS_OBJSTOSOLVE);
    }

    auto getGlobalNlpInfeas() const -> int {
      return p->getIntAttrib(XPRS_GLOBALNLPINFEAS);
    }

    auto getIISSolStatus() const -> xpress::IISSolStatus {
      return static_cast<xpress::IISSolStatus>(
          p->getIntAttrib(XPRS_IISSOLSTATUS));
    }

    auto getInputRows() const -> int { return p->getIntAttrib(XPRS_INPUTROWS); }

    auto getInputCols() const -> int { return p->getIntAttrib(XPRS_INPUTCOLS); }

    auto getMipSolTime() const -> double {
      return p->getDblAttrib(XPRS_MIPSOLTIME);
    }

    auto getTime() const -> double { return p->getDblAttrib(XPRS_TIME); }

    auto getLpObjVal() const -> double {
      return p->getDblAttrib(XPRS_LPOBJVAL);
    }

    auto getSumPrimalInf() const -> double {
      return p->getDblAttrib(XPRS_SUMPRIMALINF);
    }

    auto getMipObjVal() const -> double {
      return p->getDblAttrib(XPRS_MIPOBJVAL);
    }

    auto getBestBound() const -> double {
      return p->getDblAttrib(XPRS_BESTBOUND);
    }

    auto getObjRhs() const -> double { return p->getDblAttrib(XPRS_OBJRHS); }

    auto getMipBestObjVal() const -> double {
      return p->getDblAttrib(XPRS_MIPBESTOBJVAL);
    }

    auto getOBJSense() const -> double {
      return p->getDblAttrib(XPRS_OBJSENSE);
    }

    auto getBranchValue() const -> double {
      return p->getDblAttrib(XPRS_BRANCHVALUE);
    }

    auto getPenaltyValue() const -> double {
      return p->getDblAttrib(XPRS_PENALTYVALUE);
    }

    auto getCurrMipCutOff() const -> double {
      return p->getDblAttrib(XPRS_CURRMIPCUTOFF);
    }

    auto getBarCondA() const -> double {
      return p->getDblAttrib(XPRS_BARCONDA);
    }

    auto getBarCondD() const -> double {
      return p->getDblAttrib(XPRS_BARCONDD);
    }

    auto getMaxAbsPrimalInfeas() const -> double {
      return p->getDblAttrib(XPRS_MAXABSPRIMALINFEAS);
    }

    auto getMaxRelPrimalInfeas() const -> double {
      return p->getDblAttrib(XPRS_MAXRELPRIMALINFEAS);
    }

    auto getMaxAbsDualInfeas() const -> double {
      return p->getDblAttrib(XPRS_MAXABSDUALINFEAS);
    }

    auto getMaxRelDualInfeas() const -> double {
      return p->getDblAttrib(XPRS_MAXRELDUALINFEAS);
    }

    auto getPrimalDualIntegral() const -> double {
      return p->getDblAttrib(XPRS_PRIMALDUALINTEGRAL);
    }

    auto getMaxMipInfeas() const -> double {
      return p->getDblAttrib(XPRS_MAXMIPINFEAS);
    }

    auto getAttentionLevel() const -> double {
      return p->getDblAttrib(XPRS_ATTENTIONLEVEL);
    }

    auto getMaxKappa() const -> double {
      return p->getDblAttrib(XPRS_MAXKAPPA);
    }

    auto getTreeCompletion() const -> double {
      return p->getDblAttrib(XPRS_TREECOMPLETION);
    }

    auto getPredictedAttLevel() const -> double {
      return p->getDblAttrib(XPRS_PREDICTEDATTLEVEL);
    }

    auto getObservedPrimalIntegral() const -> double {
      return p->getDblAttrib(XPRS_OBSERVEDPRIMALINTEGRAL);
    }

    auto getCpiScaleFactor() const -> double {
      return p->getDblAttrib(XPRS_CPISCALEFACTOR);
    }

    auto getObjVal() const -> double { return p->getDblAttrib(XPRS_OBJVAL); }

    auto getWork() const -> double { return p->getDblAttrib(XPRS_WORK); }

    auto getMatrixName() const -> std::string {
      int count;
      p->getStrStringAttrib(XPRS_MATRIXNAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStrStringAttrib(XPRS_MATRIXNAME, bytes.data(), count, &count);
      return bytes.data();
    }

    auto getBoundName() const -> std::string {
      int count;
      p->getStrStringAttrib(XPRS_BOUNDNAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStrStringAttrib(XPRS_BOUNDNAME, bytes.data(), count, &count);
      return bytes.data();
    }

    auto getObjName() const -> std::string {
      int count;
      xpress_diags_push xpress_ignore_deprecated p->getStrStringAttrib(
          XPRS_OBJNAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStrStringAttrib(XPRS_OBJNAME, bytes.data(), count, &count);
      return bytes.data();
      xpress_diags_pop
    }

    auto getRhsName() const -> std::string {
      int count;
      p->getStrStringAttrib(XPRS_RHSNAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStrStringAttrib(XPRS_RHSNAME, bytes.data(), count, &count);
      return bytes.data();
    }

    auto getRangeName() const -> std::string {
      int count;
      p->getStrStringAttrib(XPRS_RANGENAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStrStringAttrib(XPRS_RANGENAME, bytes.data(), count, &count);
      return bytes.data();
    }

    auto getXpressVersion() const -> std::string {
      int count;
      p->getStrStringAttrib(XPRS_XPRESSVERSION, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStrStringAttrib(XPRS_XPRESSVERSION, bytes.data(), count, &count);
      return bytes.data();
    }

    auto getUUID() const -> std::string {
      int count;
      p->getStrStringAttrib(XPRS_UUID, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStrStringAttrib(XPRS_UUID, bytes.data(), count, &count);
      return bytes.data();
    }

    auto getBarPrimalObj() const -> double {
      return p->getDblAttrib(XPRS_BARPRIMALOBJ);
    }

    auto getBarDualObj() const -> double {
      return p->getDblAttrib(XPRS_BARDUALOBJ);
    }

    auto getBarPrimalInf() const -> double {
      return p->getDblAttrib(XPRS_BARPRIMALINF);
    }

    auto getBarDualInf() const -> double {
      return p->getDblAttrib(XPRS_BARDUALINF);
    }

    auto getBarCGap() const -> double { return p->getDblAttrib(XPRS_BARCGAP); }

    auto getBarIter() const -> int { return p->getIntAttrib(XPRS_BARITER); }

    auto getBarAASize() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_BARAASIZE);
    }

    auto getBarLSize() const -> XPRSint64 {
      return p->getLongAttrib(XPRS_BARLSIZE);
    }

    auto getBarDenseCol() const -> int {
      return p->getIntAttrib(XPRS_BARDENSECOL);
    }

    auto getBarCrossover() const -> int {
      return p->getIntAttrib(XPRS_BARCROSSOVER);
    }

    auto getMspSolutions() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOLUTIONS);
    }

    auto getMspPrb_ValidSols() const -> int {
      return p->getIntAttrib(XPRS_MSP_PRB_VALIDSOLS);
    }

    auto getMspPrb_FeasibleSols() const -> int {
      return p->getIntAttrib(XPRS_MSP_PRB_FEASIBLESOLS);
    }

    auto getMspSol_Cols() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOL_COLS);
    }

    auto getMspSol_NonZeros() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOL_NONZEROS);
    }

    auto getMspSol_IsUserSolution() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOL_ISUSERSOLUTION);
    }

    auto getMspSol_IsReProcessedUserSolution() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOL_ISREPROCESSEDUSERSOLUTION);
    }

    auto getMspSol_BitFieldsSys() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOL_BITFIELDSSYS);
    }

    auto getMspSolPrb_Obj() const -> double {
      return p->getDblAttrib(XPRS_MSP_SOLPRB_OBJ);
    }

    auto getMspSolPrb_InfeasCount() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOLPRB_INFEASCOUNT);
    }

    auto getMspSolPrb_InfSum_Primal() const -> double {
      return p->getDblAttrib(XPRS_MSP_SOLPRB_INFSUM_PRIMAL);
    }

    auto getMspSolPrb_InfCnt_Primal() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOLPRB_INFCNT_PRIMAL);
    }

    auto getMspSolPrb_InfSum_Mip() const -> double {
      return p->getDblAttrib(XPRS_MSP_SOLPRB_INFSUM_MIP);
    }

    auto getMspSolPrb_InfCnt_Mip() const -> int {
      return p->getIntAttrib(XPRS_MSP_SOLPRB_INFCNT_MIP);
    }

    auto getMseSolutions() const -> int {
      return p->getIntAttrib(XPRS_MSE_SOLUTIONS);
    }

    auto getMseMetric_MipObject() const -> int {
      return p->getIntAttrib(XPRS_MSE_METRIC_MIPOBJECT);
    }

    auto getMseMetric_Diversity() const -> int {
      return p->getIntAttrib(XPRS_MSE_METRIC_DIVERSITY);
    }

    auto getMseMetric_ModObject() const -> int {
      return p->getIntAttrib(XPRS_MSE_METRIC_MODOBJECT);
    }

    auto getMseDiversitySum() const -> double {
      return p->getDblAttrib(XPRS_MSE_DIVERSITYSUM);
    }

    auto getNlpOptTime() const -> double {
      return p->getDblAttrib(XPRS_NLPOPTTIME);
    }

    auto getNlpValidationIndex_A() const -> double {
      return p->getDblAttrib(XPRS_NLPVALIDATIONINDEX_A);
    }

    auto getNlpValidationIndex_R() const -> double {
      return p->getDblAttrib(XPRS_NLPVALIDATIONINDEX_R);
    }

    auto getNlpObjVal() const -> double {
      return p->getDblAttrib(XPRS_NLPOBJVAL);
    }

    auto getSlpErrorCosts() const -> double {
      return p->getDblAttrib(XPRS_SLPERRORCOSTS);
    }

    auto getNlpOriginalRows() const -> int {
      return p->getIntAttrib(XPRS_NLPORIGINALROWS);
    }

    auto getNlpOriginalCols() const -> int {
      return p->getIntAttrib(XPRS_NLPORIGINALCOLS);
    }

    auto getNlpUFs() const -> int { return p->getIntAttrib(XPRS_NLPUFS); }

    auto getNlpIfs() const -> int { return p->getIntAttrib(XPRS_NLPIFS); }

    auto getNlpEqualsColumn() const -> int {
      return p->getIntAttrib(XPRS_NLPEQUALSCOLUMN);
    }

    auto getNlpVariables() const -> int {
      return p->getIntAttrib(XPRS_NLPVARIABLES);
    }

    auto getNlpImplicitVariables() const -> int {
      return p->getIntAttrib(XPRS_NLPIMPLICITVARIABLES);
    }

    auto getNonLinearConstraints() const -> int {
      return p->getIntAttrib(XPRS_NONLINEARCONSTRAINTS);
    }

    auto getNlpUserFuncCalls() const -> int {
      return p->getIntAttrib(XPRS_NLPUSERFUNCCALLS);
    }

    auto getNlpUseDerivatives() const -> int {
      return p->getIntAttrib(XPRS_NLPUSEDERIVATIVES);
    }

    auto getNlpKeepBestIter() const -> int {
      return p->getIntAttrib(XPRS_NLPKEEPBESTITER);
    }

    auto getNlpStatus() const -> int { return p->getIntAttrib(XPRS_NLPSTATUS); }

    auto getLocalSolverSelected() const -> int {
      return p->getIntAttrib(XPRS_LOCALSOLVERSELECTED);
    }

    auto getNlpModelRows() const -> int {
      return p->getIntAttrib(XPRS_NLPMODELROWS);
    }

    auto getNlpModelCols() const -> int {
      return p->getIntAttrib(XPRS_NLPMODELCOLS);
    }

    auto getNlpJobID() const -> int { return p->getIntAttrib(XPRS_NLPJOBID); }

    auto getNlpStopStatus() const -> int {
      return p->getIntAttrib(XPRS_NLPSTOPSTATUS);
    }

    auto getNlpPresolveEliminations() const -> int {
      return p->getIntAttrib(XPRS_NLPPRESOLVEELIMINATIONS);
    }

    auto getSlpIter() const -> int { return p->getIntAttrib(XPRS_SLPITER); }

    auto getSlpStatus() const -> int { return p->getIntAttrib(XPRS_SLPSTATUS); }

    auto getSlpUnConverged() const -> int {
      return p->getIntAttrib(XPRS_SLPUNCONVERGED);
    }

    auto getSlpSbxConverged() const -> int {
      return p->getIntAttrib(XPRS_SLPSBXCONVERGED);
    }

    auto getSlpPenaltyDeltaRow() const -> int {
      return p->getIntAttrib(XPRS_SLPPENALTYDELTAROW);
    }

    auto getSlpPenaltyDeltaColumn() const -> int {
      return p->getIntAttrib(XPRS_SLPPENALTYDELTACOLUMN);
    }

    auto getSlpPenaltyErrorRow() const -> int {
      return p->getIntAttrib(XPRS_SLPPENALTYERRORROW);
    }

    auto getSlpPenaltyErrorColumn() const -> int {
      return p->getIntAttrib(XPRS_SLPPENALTYERRORCOLUMN);
    }

    auto getSlpCoefficients() const -> int {
      return p->getIntAttrib(XPRS_SLPCOEFFICIENTS);
    }

    auto getSlpPenaltyDeltas() const -> int {
      return p->getIntAttrib(XPRS_SLPPENALTYDELTAS);
    }

    auto getSlpPenaltyErrors() const -> int {
      return p->getIntAttrib(XPRS_SLPPENALTYERRORS);
    }

    auto getSlpPlusPenaltyErrors() const -> int {
      return p->getIntAttrib(XPRS_SLPPLUSPENALTYERRORS);
    }

    auto getSlpMinusPenaltyErrors() const -> int {
      return p->getIntAttrib(XPRS_SLPMINUSPENALTYERRORS);
    }

    auto getSlpUCConstrainedCount() const -> int {
      return p->getIntAttrib(XPRS_SLPUCCONSTRAINEDCOUNT);
    }

    auto getSlpMipNodes() const -> int {
      return p->getIntAttrib(XPRS_SLPMIPNODES);
    }

    auto getSlpMipIter() const -> int {
      return p->getIntAttrib(XPRS_SLPMIPITER);
    }

    auto getSlpTolSets() const -> int {
      return p->getIntAttrib(XPRS_SLPTOLSETS);
    }

    auto getSlpECFCount() const -> int {
      return p->getIntAttrib(XPRS_SLPECFCOUNT);
    }

    auto getSlpDeltas() const -> int { return p->getIntAttrib(XPRS_SLPDELTAS); }

    auto getSlpZeroesReset() const -> int {
      return p->getIntAttrib(XPRS_SLPZEROESRESET);
    }

    auto getSlpZeroesTotal() const -> int {
      return p->getIntAttrib(XPRS_SLPZEROESTOTAL);
    }

    auto getSlpZeroesRetained() const -> int {
      return p->getIntAttrib(XPRS_SLPZEROESRETAINED);
    }

    auto getSlpNonConstantCoeffs() const -> int {
      return p->getIntAttrib(XPRS_SLPNONCONSTANTCOEFFS);
    }

    auto getSlpMipSols() const -> int {
      return p->getIntAttrib(XPRS_SLPMIPSOLS);
    }

    auto getNlpValidationIndex_K() const -> double {
      return p->getDblAttrib(XPRS_NLPVALIDATIONINDEX_K);
    }

    auto getSlpCurrentDeltaCost() const -> double {
      return p->getDblAttrib(XPRS_SLPCURRENTDELTACOST);
    }

    auto getSlpCurrentErrorCost() const -> double {
      return p->getDblAttrib(XPRS_SLPCURRENTERRORCOST);
    }

    auto getSlpPenaltyErrorTotal() const -> double {
      return p->getDblAttrib(XPRS_SLPPENALTYERRORTOTAL);
    }

    auto getSlpPenaltyErrorValue() const -> double {
      return p->getDblAttrib(XPRS_SLPPENALTYERRORVALUE);
    }

    auto getSlpPenaltyDeltaTotal() const -> double {
      return p->getDblAttrib(XPRS_SLPPENALTYDELTATOTAL);
    }

    auto getSlpPenaltyDeltaValue() const -> double {
      return p->getDblAttrib(XPRS_SLPPENALTYDELTAVALUE);
    }
  };

  Attributes attributes;

public:
  class Controls {
    XPRSProblem *p;
    friend class XPRSProblem;
    Controls(XPRSProblem *p) : p(p) {}

  public:
    auto getExtraRows() const -> int {
      return p->getIntControl(XPRS_EXTRAROWS);
    }

    void setExtraRows(int newValue) {
      p->setIntControl(XPRS_EXTRAROWS, newValue);
    }

    auto getExtraCols() const -> int {
      return p->getIntControl(XPRS_EXTRACOLS);
    }

    void setExtraCols(int newValue) {
      p->setIntControl(XPRS_EXTRACOLS, newValue);
    }

    auto getExtraElems() const -> XPRSint64 {
      return p->getLongControl(XPRS_EXTRAELEMS);
    }

    void setExtraElems(XPRSint64 newValue) {
      p->setLongControl(XPRS_EXTRAELEMS, newValue);
    }

    auto getLpIterLimit() const -> int {
      return p->getIntControl(XPRS_LPITERLIMIT);
    }

    void setLpIterLimit(int newValue) {
      p->setIntControl(XPRS_LPITERLIMIT, newValue);
    }

    auto getLpLog() const -> int { return p->getIntControl(XPRS_LPLOG); }

    void setLpLog(int newValue) { p->setIntControl(XPRS_LPLOG, newValue); }

    auto getScaling() const -> int { return p->getIntControl(XPRS_SCALING); }

    void setScaling(int newValue) { p->setIntControl(XPRS_SCALING, newValue); }

    auto getPresolve() const -> int { return p->getIntControl(XPRS_PRESOLVE); }

    void setPresolve(int newValue) {
      p->setIntControl(XPRS_PRESOLVE, newValue);
    }

    auto getCrash() const -> int { return p->getIntControl(XPRS_CRASH); }

    void setCrash(int newValue) { p->setIntControl(XPRS_CRASH, newValue); }

    auto getPricingAlg() const -> int {
      return p->getIntControl(XPRS_PRICINGALG);
    }

    void setPricingAlg(int newValue) {
      p->setIntControl(XPRS_PRICINGALG, newValue);
    }

    auto getInvertFreq() const -> int {
      return p->getIntControl(XPRS_INVERTFREQ);
    }

    void setInvertFreq(int newValue) {
      p->setIntControl(XPRS_INVERTFREQ, newValue);
    }

    auto getInvertMin() const -> int {
      return p->getIntControl(XPRS_INVERTMIN);
    }

    void setInvertMin(int newValue) {
      p->setIntControl(XPRS_INVERTMIN, newValue);
    }

    auto getMaxNode() const -> int { return p->getIntControl(XPRS_MAXNODE); }

    void setMaxNode(int newValue) { p->setIntControl(XPRS_MAXNODE, newValue); }

    auto getMaxTime() const -> int {
      xpress_diags_push xpress_ignore_deprecated return p->getIntControl(
          XPRS_MAXTIME);
      xpress_diags_pop
    }

    void setMaxTime(int newValue) {
      xpress_diags_push xpress_ignore_deprecated p->setIntControl(XPRS_MAXTIME,
                                                                  newValue);
      xpress_diags_pop
    }

    auto getMaxMipSol() const -> int {
      return p->getIntControl(XPRS_MAXMIPSOL);
    }

    void setMaxMipSol(int newValue) {
      p->setIntControl(XPRS_MAXMIPSOL, newValue);
    }

    auto getSiftPasses() const -> int {
      return p->getIntControl(XPRS_SIFTPASSES);
    }

    void setSiftPasses(int newValue) {
      p->setIntControl(XPRS_SIFTPASSES, newValue);
    }

    auto getDefaultAlg() const -> xpress::DefaultAlg {
      return static_cast<xpress::DefaultAlg>(p->getIntControl(XPRS_DEFAULTALG));
    }

    void setDefaultAlg(xpress::DefaultAlg newValue) {
      p->setIntControl(XPRS_DEFAULTALG, static_cast<int>(newValue));
    }

    auto getVarSelection() const -> int {
      return p->getIntControl(XPRS_VARSELECTION);
    }

    void setVarSelection(int newValue) {
      p->setIntControl(XPRS_VARSELECTION, newValue);
    }

    auto getNodeSelection() const -> int {
      return p->getIntControl(XPRS_NODESELECTION);
    }

    void setNodeSelection(int newValue) {
      p->setIntControl(XPRS_NODESELECTION, newValue);
    }

    auto getBackTrack() const -> int {
      return p->getIntControl(XPRS_BACKTRACK);
    }

    void setBackTrack(int newValue) {
      p->setIntControl(XPRS_BACKTRACK, newValue);
    }

    auto getMipLog() const -> int { return p->getIntControl(XPRS_MIPLOG); }

    void setMipLog(int newValue) { p->setIntControl(XPRS_MIPLOG, newValue); }

    auto getKeepNRows() const -> int {
      return p->getIntControl(XPRS_KEEPNROWS);
    }

    void setKeepNRows(int newValue) {
      p->setIntControl(XPRS_KEEPNROWS, newValue);
    }

    auto getMpsEcho() const -> int { return p->getIntControl(XPRS_MPSECHO); }

    void setMpsEcho(int newValue) { p->setIntControl(XPRS_MPSECHO, newValue); }

    auto getMaxPageLines() const -> int {
      return p->getIntControl(XPRS_MAXPAGELINES);
    }

    void setMaxPageLines(int newValue) {
      p->setIntControl(XPRS_MAXPAGELINES, newValue);
    }

    auto getOutputLog() const -> int {
      return p->getIntControl(XPRS_OUTPUTLOG);
    }

    void setOutputLog(int newValue) {
      p->setIntControl(XPRS_OUTPUTLOG, newValue);
    }

    auto getBarSolution() const -> int {
      return p->getIntControl(XPRS_BARSOLUTION);
    }

    void setBarSolution(int newValue) {
      p->setIntControl(XPRS_BARSOLUTION, newValue);
    }

    auto getCacheSize() const -> int {
      xpress_diags_push xpress_ignore_deprecated return p->getIntControl(
          XPRS_CACHESIZE);
      xpress_diags_pop
    }

    void setCacheSize(int newValue) {
      xpress_diags_push xpress_ignore_deprecated p->setIntControl(
          XPRS_CACHESIZE, newValue);
      xpress_diags_pop
    }

    auto getCrossOver() const -> int {
      return p->getIntControl(XPRS_CROSSOVER);
    }

    void setCrossOver(int newValue) {
      p->setIntControl(XPRS_CROSSOVER, newValue);
    }

    auto getBarIterLimit() const -> int {
      return p->getIntControl(XPRS_BARITERLIMIT);
    }

    void setBarIterLimit(int newValue) {
      p->setIntControl(XPRS_BARITERLIMIT, newValue);
    }

    auto getCholeskyAlg() const -> int {
      return p->getIntControl(XPRS_CHOLESKYALG);
    }

    void setCholeskyAlg(int newValue) {
      p->setIntControl(XPRS_CHOLESKYALG, newValue);
    }

    auto getBarOutput() const -> int {
      return p->getIntControl(XPRS_BAROUTPUT);
    }

    void setBarOutput(int newValue) {
      p->setIntControl(XPRS_BAROUTPUT, newValue);
    }

    auto getExtraMipEnts() const -> int {
      return p->getIntControl(XPRS_EXTRAMIPENTS);
    }

    void setExtraMipEnts(int newValue) {
      p->setIntControl(XPRS_EXTRAMIPENTS, newValue);
    }

    auto getRefactor() const -> int { return p->getIntControl(XPRS_REFACTOR); }

    void setRefactor(int newValue) {
      p->setIntControl(XPRS_REFACTOR, newValue);
    }

    auto getBarThreads() const -> int {
      return p->getIntControl(XPRS_BARTHREADS);
    }

    void setBarThreads(int newValue) {
      p->setIntControl(XPRS_BARTHREADS, newValue);
    }

    auto getKeepBasis() const -> int {
      return p->getIntControl(XPRS_KEEPBASIS);
    }

    void setKeepBasis(int newValue) {
      p->setIntControl(XPRS_KEEPBASIS, newValue);
    }

    auto getCrossoverOps() const -> int {
      return p->getIntControl(XPRS_CROSSOVEROPS);
    }

    void setCrossoverOps(int newValue) {
      p->setIntControl(XPRS_CROSSOVEROPS, newValue);
    }

    auto getVersion() const -> int { return p->getIntControl(XPRS_VERSION); }

    void setVersion(int newValue) { p->setIntControl(XPRS_VERSION, newValue); }

    auto getCrossoverThreads() const -> int {
      return p->getIntControl(XPRS_CROSSOVERTHREADS);
    }

    void setCrossoverThreads(int newValue) {
      p->setIntControl(XPRS_CROSSOVERTHREADS, newValue);
    }

    auto getBigmMethod() const -> int {
      return p->getIntControl(XPRS_BIGMMETHOD);
    }

    void setBigmMethod(int newValue) {
      p->setIntControl(XPRS_BIGMMETHOD, newValue);
    }

    auto getMpsNameLength() const -> int {
      return p->getIntControl(XPRS_MPSNAMELENGTH);
    }

    void setMpsNameLength(int newValue) {
      p->setIntControl(XPRS_MPSNAMELENGTH, newValue);
    }

    auto getElimFillIn() const -> int {
      return p->getIntControl(XPRS_ELIMFILLIN);
    }

    void setElimFillIn(int newValue) {
      p->setIntControl(XPRS_ELIMFILLIN, newValue);
    }

    auto getPresolveOps() const -> int {
      return p->getIntControl(XPRS_PRESOLVEOPS);
    }

    void setPresolveOps(int newValue) {
      p->setIntControl(XPRS_PRESOLVEOPS, newValue);
    }

    auto getMipPresolve() const -> int {
      return p->getIntControl(XPRS_MIPPRESOLVE);
    }

    void setMipPresolve(int newValue) {
      p->setIntControl(XPRS_MIPPRESOLVE, newValue);
    }

    auto getMipThreads() const -> int {
      return p->getIntControl(XPRS_MIPTHREADS);
    }

    void setMipThreads(int newValue) {
      p->setIntControl(XPRS_MIPTHREADS, newValue);
    }

    auto getBarOrder() const -> xpress::BarOrder {
      return static_cast<xpress::BarOrder>(p->getIntControl(XPRS_BARORDER));
    }

    void setBarOrder(xpress::BarOrder newValue) {
      p->setIntControl(XPRS_BARORDER, static_cast<int>(newValue));
    }

    auto getBreadthFirst() const -> int {
      return p->getIntControl(XPRS_BREADTHFIRST);
    }

    void setBreadthFirst(int newValue) {
      p->setIntControl(XPRS_BREADTHFIRST, newValue);
    }

    auto getAutoPerturb() const -> int {
      return p->getIntControl(XPRS_AUTOPERTURB);
    }

    void setAutoPerturb(int newValue) {
      p->setIntControl(XPRS_AUTOPERTURB, newValue);
    }

    auto getDenseColLimit() const -> int {
      return p->getIntControl(XPRS_DENSECOLLIMIT);
    }

    void setDenseColLimit(int newValue) {
      p->setIntControl(XPRS_DENSECOLLIMIT, newValue);
    }

    auto getCallbackFromMasterThread() const -> int {
      return p->getIntControl(XPRS_CALLBACKFROMMASTERTHREAD);
    }

    void setCallbackFromMasterThread(int newValue) {
      p->setIntControl(XPRS_CALLBACKFROMMASTERTHREAD, newValue);
    }

    auto getMaxMCoeffBufferElems() const -> int {
      return p->getIntControl(XPRS_MAXMCOEFFBUFFERELEMS);
    }

    void setMaxMCoeffBufferElems(int newValue) {
      p->setIntControl(XPRS_MAXMCOEFFBUFFERELEMS, newValue);
    }

    auto getRefineOps() const -> int {
      return p->getIntControl(XPRS_REFINEOPS);
    }

    void setRefineOps(int newValue) {
      p->setIntControl(XPRS_REFINEOPS, newValue);
    }

    auto getLpRefineIterLimit() const -> int {
      return p->getIntControl(XPRS_LPREFINEITERLIMIT);
    }

    void setLpRefineIterLimit(int newValue) {
      p->setIntControl(XPRS_LPREFINEITERLIMIT, newValue);
    }

    auto getMipRefineIterLimit() const -> int {
      return p->getIntControl(XPRS_MIPREFINEITERLIMIT);
    }

    void setMipRefineIterLimit(int newValue) {
      p->setIntControl(XPRS_MIPREFINEITERLIMIT, newValue);
    }

    auto getDualizeOps() const -> int {
      return p->getIntControl(XPRS_DUALIZEOPS);
    }

    void setDualizeOps(int newValue) {
      p->setIntControl(XPRS_DUALIZEOPS, newValue);
    }

    auto getCrossoverIterLimit() const -> int {
      return p->getIntControl(XPRS_CROSSOVERITERLIMIT);
    }

    void setCrossoverIterLimit(int newValue) {
      p->setIntControl(XPRS_CROSSOVERITERLIMIT, newValue);
    }

    auto getPreBasisRed() const -> int {
      return p->getIntControl(XPRS_PREBASISRED);
    }

    void setPreBasisRed(int newValue) {
      p->setIntControl(XPRS_PREBASISRED, newValue);
    }

    auto getPreSort() const -> int { return p->getIntControl(XPRS_PRESORT); }

    void setPreSort(int newValue) { p->setIntControl(XPRS_PRESORT, newValue); }

    auto getPrePermute() const -> int {
      return p->getIntControl(XPRS_PREPERMUTE);
    }

    void setPrePermute(int newValue) {
      p->setIntControl(XPRS_PREPERMUTE, newValue);
    }

    auto getPrePermuteSeed() const -> int {
      return p->getIntControl(XPRS_PREPERMUTESEED);
    }

    void setPrePermuteSeed(int newValue) {
      p->setIntControl(XPRS_PREPERMUTESEED, newValue);
    }

    auto getMaxMemorySoft() const -> int {
      return p->getIntControl(XPRS_MAXMEMORYSOFT);
    }

    void setMaxMemorySoft(int newValue) {
      p->setIntControl(XPRS_MAXMEMORYSOFT, newValue);
    }

    auto getCutFreq() const -> int { return p->getIntControl(XPRS_CUTFREQ); }

    void setCutFreq(int newValue) { p->setIntControl(XPRS_CUTFREQ, newValue); }

    auto getSymSelect() const -> int {
      return p->getIntControl(XPRS_SYMSELECT);
    }

    void setSymSelect(int newValue) {
      p->setIntControl(XPRS_SYMSELECT, newValue);
    }

    auto getSymmetry() const -> int { return p->getIntControl(XPRS_SYMMETRY); }

    void setSymmetry(int newValue) {
      p->setIntControl(XPRS_SYMMETRY, newValue);
    }

    auto getMaxMemoryHard() const -> int {
      return p->getIntControl(XPRS_MAXMEMORYHARD);
    }

    void setMaxMemoryHard(int newValue) {
      p->setIntControl(XPRS_MAXMEMORYHARD, newValue);
    }

    auto getMIQCPAlg() const -> int { return p->getIntControl(XPRS_MIQCPALG); }

    void setMIQCPAlg(int newValue) {
      p->setIntControl(XPRS_MIQCPALG, newValue);
    }

    auto getQCCuts() const -> int { return p->getIntControl(XPRS_QCCUTS); }

    void setQCCuts(int newValue) { p->setIntControl(XPRS_QCCUTS, newValue); }

    auto getQCRootAlg() const -> int {
      return p->getIntControl(XPRS_QCROOTALG);
    }

    void setQCRootAlg(int newValue) {
      p->setIntControl(XPRS_QCROOTALG, newValue);
    }

    auto getPreConvertSeparable() const -> int {
      return p->getIntControl(XPRS_PRECONVERTSEPARABLE);
    }

    void setPreConvertSeparable(int newValue) {
      p->setIntControl(XPRS_PRECONVERTSEPARABLE, newValue);
    }

    auto getAlgAfterNetwork() const -> int {
      return p->getIntControl(XPRS_ALGAFTERNETWORK);
    }

    void setAlgAfterNetwork(int newValue) {
      p->setIntControl(XPRS_ALGAFTERNETWORK, newValue);
    }

    auto getTrace() const -> int { return p->getIntControl(XPRS_TRACE); }

    void setTrace(int newValue) { p->setIntControl(XPRS_TRACE, newValue); }

    auto getMaxIIS() const -> int { return p->getIntControl(XPRS_MAXIIS); }

    void setMaxIIS(int newValue) { p->setIntControl(XPRS_MAXIIS, newValue); }

    auto getCPUTime() const -> int { return p->getIntControl(XPRS_CPUTIME); }

    void setCPUTime(int newValue) { p->setIntControl(XPRS_CPUTIME, newValue); }

    auto getCoverCuts() const -> int {
      return p->getIntControl(XPRS_COVERCUTS);
    }

    void setCoverCuts(int newValue) {
      p->setIntControl(XPRS_COVERCUTS, newValue);
    }

    auto getGomCuts() const -> int { return p->getIntControl(XPRS_GOMCUTS); }

    void setGomCuts(int newValue) { p->setIntControl(XPRS_GOMCUTS, newValue); }

    auto getLpFolding() const -> int {
      return p->getIntControl(XPRS_LPFOLDING);
    }

    void setLpFolding(int newValue) {
      p->setIntControl(XPRS_LPFOLDING, newValue);
    }

    auto getMpsFormat() const -> int {
      return p->getIntControl(XPRS_MPSFORMAT);
    }

    void setMpsFormat(int newValue) {
      p->setIntControl(XPRS_MPSFORMAT, newValue);
    }

    auto getCutStrategy() const -> int {
      return p->getIntControl(XPRS_CUTSTRATEGY);
    }

    void setCutStrategy(int newValue) {
      p->setIntControl(XPRS_CUTSTRATEGY, newValue);
    }

    auto getCutDepth() const -> int { return p->getIntControl(XPRS_CUTDEPTH); }

    void setCutDepth(int newValue) {
      p->setIntControl(XPRS_CUTDEPTH, newValue);
    }

    auto getTreeCoverCuts() const -> int {
      return p->getIntControl(XPRS_TREECOVERCUTS);
    }

    void setTreeCoverCuts(int newValue) {
      p->setIntControl(XPRS_TREECOVERCUTS, newValue);
    }

    auto getTreeGomCuts() const -> int {
      return p->getIntControl(XPRS_TREEGOMCUTS);
    }

    void setTreeGomCuts(int newValue) {
      p->setIntControl(XPRS_TREEGOMCUTS, newValue);
    }

    auto getCutSelect() const -> int {
      return p->getIntControl(XPRS_CUTSELECT);
    }

    void setCutSelect(int newValue) {
      p->setIntControl(XPRS_CUTSELECT, newValue);
    }

    auto getTreeCutSelect() const -> int {
      return p->getIntControl(XPRS_TREECUTSELECT);
    }

    void setTreeCutSelect(int newValue) {
      p->setIntControl(XPRS_TREECUTSELECT, newValue);
    }

    auto getDualize() const -> int { return p->getIntControl(XPRS_DUALIZE); }

    void setDualize(int newValue) { p->setIntControl(XPRS_DUALIZE, newValue); }

    auto getDualGradient() const -> int {
      return p->getIntControl(XPRS_DUALGRADIENT);
    }

    void setDualGradient(int newValue) {
      p->setIntControl(XPRS_DUALGRADIENT, newValue);
    }

    auto getSBIterLimit() const -> int {
      return p->getIntControl(XPRS_SBITERLIMIT);
    }

    void setSBIterLimit(int newValue) {
      p->setIntControl(XPRS_SBITERLIMIT, newValue);
    }

    auto getSBBest() const -> int { return p->getIntControl(XPRS_SBBEST); }

    void setSBBest(int newValue) { p->setIntControl(XPRS_SBBEST, newValue); }

    auto getBarIndefLimit() const -> int {
      return p->getIntControl(XPRS_BARINDEFLIMIT);
    }

    void setBarIndefLimit(int newValue) {
      p->setIntControl(XPRS_BARINDEFLIMIT, newValue);
    }

    auto getHeurFreq() const -> int { return p->getIntControl(XPRS_HEURFREQ); }

    void setHeurFreq(int newValue) {
      p->setIntControl(XPRS_HEURFREQ, newValue);
    }

    auto getHeurDepth() const -> int {
      return p->getIntControl(XPRS_HEURDEPTH);
    }

    void setHeurDepth(int newValue) {
      p->setIntControl(XPRS_HEURDEPTH, newValue);
    }

    auto getHeurMaxSol() const -> int {
      return p->getIntControl(XPRS_HEURMAXSOL);
    }

    void setHeurMaxSol(int newValue) {
      p->setIntControl(XPRS_HEURMAXSOL, newValue);
    }

    auto getHeurNodes() const -> int {
      return p->getIntControl(XPRS_HEURNODES);
    }

    void setHeurNodes(int newValue) {
      p->setIntControl(XPRS_HEURNODES, newValue);
    }

    auto getLNPBest() const -> int { return p->getIntControl(XPRS_LNPBEST); }

    void setLNPBest(int newValue) { p->setIntControl(XPRS_LNPBEST, newValue); }

    auto getLNPIterLimit() const -> int {
      return p->getIntControl(XPRS_LNPITERLIMIT);
    }

    void setLNPIterLimit(int newValue) {
      p->setIntControl(XPRS_LNPITERLIMIT, newValue);
    }

    auto getBranchChoice() const -> int {
      return p->getIntControl(XPRS_BRANCHCHOICE);
    }

    void setBranchChoice(int newValue) {
      p->setIntControl(XPRS_BRANCHCHOICE, newValue);
    }

    auto getBarRegularize() const -> int {
      return p->getIntControl(XPRS_BARREGULARIZE);
    }

    void setBarRegularize(int newValue) {
      p->setIntControl(XPRS_BARREGULARIZE, newValue);
    }

    auto getSBSelect() const -> int { return p->getIntControl(XPRS_SBSELECT); }

    void setSBSelect(int newValue) {
      p->setIntControl(XPRS_SBSELECT, newValue);
    }

    auto getIISLog() const -> int { return p->getIntControl(XPRS_IISLOG); }

    void setIISLog(int newValue) { p->setIntControl(XPRS_IISLOG, newValue); }

    auto getLocalChoice() const -> int {
      return p->getIntControl(XPRS_LOCALCHOICE);
    }

    void setLocalChoice(int newValue) {
      p->setIntControl(XPRS_LOCALCHOICE, newValue);
    }

    auto getLocalBacktrack() const -> int {
      return p->getIntControl(XPRS_LOCALBACKTRACK);
    }

    void setLocalBacktrack(int newValue) {
      p->setIntControl(XPRS_LOCALBACKTRACK, newValue);
    }

    auto getDualStrategy() const -> int {
      return p->getIntControl(XPRS_DUALSTRATEGY);
    }

    void setDualStrategy(int newValue) {
      p->setIntControl(XPRS_DUALSTRATEGY, newValue);
    }

    auto getL1Cache() const -> int {
      xpress_diags_push xpress_ignore_deprecated return p->getIntControl(
          XPRS_L1CACHE);
      xpress_diags_pop
    }

    void setL1Cache(int newValue) {
      xpress_diags_push xpress_ignore_deprecated p->setIntControl(XPRS_L1CACHE,
                                                                  newValue);
      xpress_diags_pop
    }

    auto getHeurDiveStrategy() const -> int {
      return p->getIntControl(XPRS_HEURDIVESTRATEGY);
    }

    void setHeurDiveStrategy(int newValue) {
      p->setIntControl(XPRS_HEURDIVESTRATEGY, newValue);
    }

    auto getHeurSelect() const -> int {
      return p->getIntControl(XPRS_HEURSELECT);
    }

    void setHeurSelect(int newValue) {
      p->setIntControl(XPRS_HEURSELECT, newValue);
    }

    auto getBarStart() const -> int { return p->getIntControl(XPRS_BARSTART); }

    void setBarStart(int newValue) {
      p->setIntControl(XPRS_BARSTART, newValue);
    }

    auto getPresolvePasses() const -> int {
      return p->getIntControl(XPRS_PRESOLVEPASSES);
    }

    void setPresolvePasses(int newValue) {
      p->setIntControl(XPRS_PRESOLVEPASSES, newValue);
    }

    auto getBarNumStability() const -> int {
      xpress_diags_push xpress_ignore_deprecated return p->getIntControl(
          XPRS_BARNUMSTABILITY);
      xpress_diags_pop
    }

    void setBarNumStability(int newValue) {
      xpress_diags_push xpress_ignore_deprecated p->setIntControl(
          XPRS_BARNUMSTABILITY, newValue);
      xpress_diags_pop
    }

    auto getBarOrderThreads() const -> int {
      return p->getIntControl(XPRS_BARORDERTHREADS);
    }

    void setBarOrderThreads(int newValue) {
      p->setIntControl(XPRS_BARORDERTHREADS, newValue);
    }

    auto getExtraSets() const -> int {
      return p->getIntControl(XPRS_EXTRASETS);
    }

    void setExtraSets(int newValue) {
      p->setIntControl(XPRS_EXTRASETS, newValue);
    }

    auto getExtraSetElems() const -> XPRSint64 {
      return p->getLongControl(XPRS_EXTRASETELEMS);
    }

    void setExtraSetElems(XPRSint64 newValue) {
      p->setLongControl(XPRS_EXTRASETELEMS, newValue);
    }

    auto getFeasibilityPump() const -> int {
      return p->getIntControl(XPRS_FEASIBILITYPUMP);
    }

    void setFeasibilityPump(int newValue) {
      p->setIntControl(XPRS_FEASIBILITYPUMP, newValue);
    }

    auto getPreCoefElim() const -> int {
      return p->getIntControl(XPRS_PRECOEFELIM);
    }

    void setPreCoefElim(int newValue) {
      p->setIntControl(XPRS_PRECOEFELIM, newValue);
    }

    auto getPreDomCol() const -> int {
      return p->getIntControl(XPRS_PREDOMCOL);
    }

    void setPreDomCol(int newValue) {
      p->setIntControl(XPRS_PREDOMCOL, newValue);
    }

    auto getHeurSearchFreq() const -> int {
      return p->getIntControl(XPRS_HEURSEARCHFREQ);
    }

    void setHeurSearchFreq(int newValue) {
      p->setIntControl(XPRS_HEURSEARCHFREQ, newValue);
    }

    auto getHeurDiveSpeedUp() const -> int {
      return p->getIntControl(XPRS_HEURDIVESPEEDUP);
    }

    void setHeurDiveSpeedUp(int newValue) {
      p->setIntControl(XPRS_HEURDIVESPEEDUP, newValue);
    }

    auto getSBEstimate() const -> int {
      return p->getIntControl(XPRS_SBESTIMATE);
    }

    void setSBEstimate(int newValue) {
      p->setIntControl(XPRS_SBESTIMATE, newValue);
    }

    auto getBarCores() const -> int { return p->getIntControl(XPRS_BARCORES); }

    void setBarCores(int newValue) {
      p->setIntControl(XPRS_BARCORES, newValue);
    }

    auto getMaxChecksOnMaxTime() const -> int {
      return p->getIntControl(XPRS_MAXCHECKSONMAXTIME);
    }

    void setMaxChecksOnMaxTime(int newValue) {
      p->setIntControl(XPRS_MAXCHECKSONMAXTIME, newValue);
    }

    auto getMaxChecksOnMaxCutTime() const -> int {
      return p->getIntControl(XPRS_MAXCHECKSONMAXCUTTIME);
    }

    void setMaxChecksOnMaxCutTime(int newValue) {
      p->setIntControl(XPRS_MAXCHECKSONMAXCUTTIME, newValue);
    }

    auto getHistoryCosts() const -> int {
      return p->getIntControl(XPRS_HISTORYCOSTS);
    }

    void setHistoryCosts(int newValue) {
      p->setIntControl(XPRS_HISTORYCOSTS, newValue);
    }

    auto getAlgAfterCrossOver() const -> int {
      return p->getIntControl(XPRS_ALGAFTERCROSSOVER);
    }

    void setAlgAfterCrossOver(int newValue) {
      p->setIntControl(XPRS_ALGAFTERCROSSOVER, newValue);
    }

    auto getMutexCallBacks() const -> int {
      return p->getIntControl(XPRS_MUTEXCALLBACKS);
    }

    void setMutexCallBacks(int newValue) {
      p->setIntControl(XPRS_MUTEXCALLBACKS, newValue);
    }

    auto getBarCrash() const -> int { return p->getIntControl(XPRS_BARCRASH); }

    void setBarCrash(int newValue) {
      p->setIntControl(XPRS_BARCRASH, newValue);
    }

    auto getHeurDiveSoftRounding() const -> int {
      return p->getIntControl(XPRS_HEURDIVESOFTROUNDING);
    }

    void setHeurDiveSoftRounding(int newValue) {
      p->setIntControl(XPRS_HEURDIVESOFTROUNDING, newValue);
    }

    auto getHeurSearchRootSelect() const -> int {
      return p->getIntControl(XPRS_HEURSEARCHROOTSELECT);
    }

    void setHeurSearchRootSelect(int newValue) {
      p->setIntControl(XPRS_HEURSEARCHROOTSELECT, newValue);
    }

    auto getHeurSearchTreeSelect() const -> int {
      return p->getIntControl(XPRS_HEURSEARCHTREESELECT);
    }

    void setHeurSearchTreeSelect(int newValue) {
      p->setIntControl(XPRS_HEURSEARCHTREESELECT, newValue);
    }

    auto getMps18Compatible() const -> int {
      return p->getIntControl(XPRS_MPS18COMPATIBLE);
    }

    void setMps18Compatible(int newValue) {
      p->setIntControl(XPRS_MPS18COMPATIBLE, newValue);
    }

    auto getRootPresolve() const -> int {
      return p->getIntControl(XPRS_ROOTPRESOLVE);
    }

    void setRootPresolve(int newValue) {
      p->setIntControl(XPRS_ROOTPRESOLVE, newValue);
    }

    auto getCrossOverDRP() const -> int {
      return p->getIntControl(XPRS_CROSSOVERDRP);
    }

    void setCrossOverDRP(int newValue) {
      p->setIntControl(XPRS_CROSSOVERDRP, newValue);
    }

    auto getForceOutput() const -> int {
      return p->getIntControl(XPRS_FORCEOUTPUT);
    }

    void setForceOutput(int newValue) {
      p->setIntControl(XPRS_FORCEOUTPUT, newValue);
    }

    auto getPrimalOps() const -> int {
      return p->getIntControl(XPRS_PRIMALOPS);
    }

    void setPrimalOps(int newValue) {
      p->setIntControl(XPRS_PRIMALOPS, newValue);
    }

    auto getDeterministic() const -> int {
      return p->getIntControl(XPRS_DETERMINISTIC);
    }

    void setDeterministic(int newValue) {
      p->setIntControl(XPRS_DETERMINISTIC, newValue);
    }

    auto getPreProbing() const -> int {
      return p->getIntControl(XPRS_PREPROBING);
    }

    void setPreProbing(int newValue) {
      p->setIntControl(XPRS_PREPROBING, newValue);
    }

    auto getTreeMemoryLimit() const -> int {
      return p->getIntControl(XPRS_TREEMEMORYLIMIT);
    }

    void setTreeMemoryLimit(int newValue) {
      p->setIntControl(XPRS_TREEMEMORYLIMIT, newValue);
    }

    auto getTreeCompression() const -> int {
      return p->getIntControl(XPRS_TREECOMPRESSION);
    }

    void setTreeCompression(int newValue) {
      p->setIntControl(XPRS_TREECOMPRESSION, newValue);
    }

    auto getTreeDiagnostics() const -> int {
      return p->getIntControl(XPRS_TREEDIAGNOSTICS);
    }

    void setTreeDiagnostics(int newValue) {
      p->setIntControl(XPRS_TREEDIAGNOSTICS, newValue);
    }

    auto getMaxTreeFileSize() const -> int {
      return p->getIntControl(XPRS_MAXTREEFILESIZE);
    }

    void setMaxTreeFileSize(int newValue) {
      p->setIntControl(XPRS_MAXTREEFILESIZE, newValue);
    }

    auto getPreCliqueStrategy() const -> int {
      return p->getIntControl(XPRS_PRECLIQUESTRATEGY);
    }

    void setPreCliqueStrategy(int newValue) {
      p->setIntControl(XPRS_PRECLIQUESTRATEGY, newValue);
    }

    auto getRepairInfeasMaxTime() const -> int {
      xpress_diags_push xpress_ignore_deprecated return p->getIntControl(
          XPRS_REPAIRINFEASMAXTIME);
      xpress_diags_pop
    }

    void setRepairInfeasMaxTime(int newValue) {
      xpress_diags_push xpress_ignore_deprecated p->setIntControl(
          XPRS_REPAIRINFEASMAXTIME, newValue);
      xpress_diags_pop
    }

    auto getIfCheckConvexity() const -> int {
      return p->getIntControl(XPRS_IFCHECKCONVEXITY);
    }

    void setIfCheckConvexity(int newValue) {
      p->setIntControl(XPRS_IFCHECKCONVEXITY, newValue);
    }

    auto getPrimalUnshift() const -> int {
      return p->getIntControl(XPRS_PRIMALUNSHIFT);
    }

    void setPrimalUnshift(int newValue) {
      p->setIntControl(XPRS_PRIMALUNSHIFT, newValue);
    }

    auto getRepairIndefiniteQ() const -> xpress::RepairIndefiniteQuadratic {
      return static_cast<xpress::RepairIndefiniteQuadratic>(
          p->getIntControl(XPRS_REPAIRINDEFINITEQ));
    }

    void setRepairIndefiniteQ(xpress::RepairIndefiniteQuadratic newValue) {
      p->setIntControl(XPRS_REPAIRINDEFINITEQ, static_cast<int>(newValue));
    }

    auto getMipRampup() const -> int {
      return p->getIntControl(XPRS_MIPRAMPUP);
    }

    void setMipRampup(int newValue) {
      p->setIntControl(XPRS_MIPRAMPUP, newValue);
    }

    auto getMaxLocalBacktrack() const -> int {
      return p->getIntControl(XPRS_MAXLOCALBACKTRACK);
    }

    void setMaxLocalBacktrack(int newValue) {
      p->setIntControl(XPRS_MAXLOCALBACKTRACK, newValue);
    }

    auto getUserSolHeuristic() const -> int {
      return p->getIntControl(XPRS_USERSOLHEURISTIC);
    }

    void setUserSolHeuristic(int newValue) {
      p->setIntControl(XPRS_USERSOLHEURISTIC, newValue);
    }

    auto getPreConvertObjToCons() const -> int {
      return p->getIntControl(XPRS_PRECONVERTOBJTOCONS);
    }

    void setPreConvertObjToCons(int newValue) {
      p->setIntControl(XPRS_PRECONVERTOBJTOCONS, newValue);
    }

    auto getForceParallelDual() const -> int {
      return p->getIntControl(XPRS_FORCEPARALLELDUAL);
    }

    void setForceParallelDual(int newValue) {
      p->setIntControl(XPRS_FORCEPARALLELDUAL, newValue);
    }

    auto getBacktrackTie() const -> int {
      return p->getIntControl(XPRS_BACKTRACKTIE);
    }

    void setBacktrackTie(int newValue) {
      p->setIntControl(XPRS_BACKTRACKTIE, newValue);
    }

    auto getBranchDisj() const -> int {
      return p->getIntControl(XPRS_BRANCHDISJ);
    }

    void setBranchDisj(int newValue) {
      p->setIntControl(XPRS_BRANCHDISJ, newValue);
    }

    auto getMipFracReduce() const -> int {
      return p->getIntControl(XPRS_MIPFRACREDUCE);
    }

    void setMipFracReduce(int newValue) {
      p->setIntControl(XPRS_MIPFRACREDUCE, newValue);
    }

    auto getConcurrentThreads() const -> int {
      return p->getIntControl(XPRS_CONCURRENTTHREADS);
    }

    void setConcurrentThreads(int newValue) {
      p->setIntControl(XPRS_CONCURRENTTHREADS, newValue);
    }

    auto getMaxScaleFactor() const -> int {
      return p->getIntControl(XPRS_MAXSCALEFACTOR);
    }

    void setMaxScaleFactor(int newValue) {
      p->setIntControl(XPRS_MAXSCALEFACTOR, newValue);
    }

    auto getHeurThreads() const -> int {
      return p->getIntControl(XPRS_HEURTHREADS);
    }

    void setHeurThreads(int newValue) {
      p->setIntControl(XPRS_HEURTHREADS, newValue);
    }

    auto getThreads() const -> int { return p->getIntControl(XPRS_THREADS); }

    void setThreads(int newValue) { p->setIntControl(XPRS_THREADS, newValue); }

    auto getHeurBeforeLp() const -> int {
      return p->getIntControl(XPRS_HEURBEFORELP);
    }

    void setHeurBeforeLp(int newValue) {
      p->setIntControl(XPRS_HEURBEFORELP, newValue);
    }

    auto getPreDomRow() const -> int {
      return p->getIntControl(XPRS_PREDOMROW);
    }

    void setPreDomRow(int newValue) {
      p->setIntControl(XPRS_PREDOMROW, newValue);
    }

    auto getBranchStructural() const -> int {
      return p->getIntControl(XPRS_BRANCHSTRUCTURAL);
    }

    void setBranchStructural(int newValue) {
      p->setIntControl(XPRS_BRANCHSTRUCTURAL, newValue);
    }

    auto getQuadraticUnshift() const -> int {
      return p->getIntControl(XPRS_QUADRATICUNSHIFT);
    }

    void setQuadraticUnshift(int newValue) {
      p->setIntControl(XPRS_QUADRATICUNSHIFT, newValue);
    }

    auto getBarPresolveOps() const -> int {
      return p->getIntControl(XPRS_BARPRESOLVEOPS);
    }

    void setBarPresolveOps(int newValue) {
      p->setIntControl(XPRS_BARPRESOLVEOPS, newValue);
    }

    auto getQSimplexOps() const -> int {
      return p->getIntControl(XPRS_QSIMPLEXOPS);
    }

    void setQSimplexOps(int newValue) {
      p->setIntControl(XPRS_QSIMPLEXOPS, newValue);
    }

    auto getMipRestart() const -> int {
      return p->getIntControl(XPRS_MIPRESTART);
    }

    void setMipRestart(int newValue) {
      p->setIntControl(XPRS_MIPRESTART, newValue);
    }

    auto getConflictCuts() const -> int {
      return p->getIntControl(XPRS_CONFLICTCUTS);
    }

    void setConflictCuts(int newValue) {
      p->setIntControl(XPRS_CONFLICTCUTS, newValue);
    }

    auto getPreProtectDual() const -> int {
      return p->getIntControl(XPRS_PREPROTECTDUAL);
    }

    void setPreProtectDual(int newValue) {
      p->setIntControl(XPRS_PREPROTECTDUAL, newValue);
    }

    auto getCoresPerCPU() const -> int {
      return p->getIntControl(XPRS_CORESPERCPU);
    }

    void setCoresPerCPU(int newValue) {
      p->setIntControl(XPRS_CORESPERCPU, newValue);
    }

    auto getResourceStrategy() const -> int {
      return p->getIntControl(XPRS_RESOURCESTRATEGY);
    }

    void setResourceStrategy(int newValue) {
      p->setIntControl(XPRS_RESOURCESTRATEGY, newValue);
    }

    auto getClamping() const -> int { return p->getIntControl(XPRS_CLAMPING); }

    void setClamping(int newValue) {
      p->setIntControl(XPRS_CLAMPING, newValue);
    }

    auto getSleepOnThreadWait() const -> int {
      xpress_diags_push xpress_ignore_deprecated return p->getIntControl(
          XPRS_SLEEPONTHREADWAIT);
      xpress_diags_pop
    }

    void setSleepOnThreadWait(int newValue) {
      xpress_diags_push xpress_ignore_deprecated p->setIntControl(
          XPRS_SLEEPONTHREADWAIT, newValue);
      xpress_diags_pop
    }

    auto getPreDupRow() const -> int {
      return p->getIntControl(XPRS_PREDUPROW);
    }

    void setPreDupRow(int newValue) {
      p->setIntControl(XPRS_PREDUPROW, newValue);
    }

    auto getCPUPlatform() const -> int {
      return p->getIntControl(XPRS_CPUPLATFORM);
    }

    void setCPUPlatform(int newValue) {
      p->setIntControl(XPRS_CPUPLATFORM, newValue);
    }

    auto getBarAlg() const -> int { return p->getIntControl(XPRS_BARALG); }

    void setBarAlg(int newValue) { p->setIntControl(XPRS_BARALG, newValue); }

    auto getSifting() const -> int { return p->getIntControl(XPRS_SIFTING); }

    void setSifting(int newValue) { p->setIntControl(XPRS_SIFTING, newValue); }

    auto getBarKeepLastSol() const -> int {
      return p->getIntControl(XPRS_BARKEEPLASTSOL);
    }

    void setBarKeepLastSol(int newValue) {
      p->setIntControl(XPRS_BARKEEPLASTSOL, newValue);
    }

    auto getLpLogStyle() const -> int {
      return p->getIntControl(XPRS_LPLOGSTYLE);
    }

    void setLpLogStyle(int newValue) {
      p->setIntControl(XPRS_LPLOGSTYLE, newValue);
    }

    auto getRandomSeed() const -> int {
      return p->getIntControl(XPRS_RANDOMSEED);
    }

    void setRandomSeed(int newValue) {
      p->setIntControl(XPRS_RANDOMSEED, newValue);
    }

    auto getTreeQCCuts() const -> int {
      return p->getIntControl(XPRS_TREEQCCUTS);
    }

    void setTreeQCCuts(int newValue) {
      p->setIntControl(XPRS_TREEQCCUTS, newValue);
    }

    auto getPreLinDep() const -> int {
      return p->getIntControl(XPRS_PRELINDEP);
    }

    void setPreLinDep(int newValue) {
      p->setIntControl(XPRS_PRELINDEP, newValue);
    }

    auto getDualThreads() const -> int {
      return p->getIntControl(XPRS_DUALTHREADS);
    }

    void setDualThreads(int newValue) {
      p->setIntControl(XPRS_DUALTHREADS, newValue);
    }

    auto getPreObjCutDetect() const -> int {
      return p->getIntControl(XPRS_PREOBJCUTDETECT);
    }

    void setPreObjCutDetect(int newValue) {
      p->setIntControl(XPRS_PREOBJCUTDETECT, newValue);
    }

    auto getPreBndRedQuad() const -> int {
      return p->getIntControl(XPRS_PREBNDREDQUAD);
    }

    void setPreBndRedQuad(int newValue) {
      p->setIntControl(XPRS_PREBNDREDQUAD, newValue);
    }

    auto getPreBndRedCone() const -> int {
      return p->getIntControl(XPRS_PREBNDREDCONE);
    }

    void setPreBndRedCone(int newValue) {
      p->setIntControl(XPRS_PREBNDREDCONE, newValue);
    }

    auto getPreComponents() const -> int {
      return p->getIntControl(XPRS_PRECOMPONENTS);
    }

    void setPreComponents(int newValue) {
      p->setIntControl(XPRS_PRECOMPONENTS, newValue);
    }

    auto getMaxMipTasks() const -> int {
      return p->getIntControl(XPRS_MAXMIPTASKS);
    }

    void setMaxMipTasks(int newValue) {
      p->setIntControl(XPRS_MAXMIPTASKS, newValue);
    }

    auto getMipTerminationMethod() const -> int {
      return p->getIntControl(XPRS_MIPTERMINATIONMETHOD);
    }

    void setMipTerminationMethod(int newValue) {
      p->setIntControl(XPRS_MIPTERMINATIONMETHOD, newValue);
    }

    auto getPreConeDecomp() const -> int {
      return p->getIntControl(XPRS_PRECONEDECOMP);
    }

    void setPreConeDecomp(int newValue) {
      p->setIntControl(XPRS_PRECONEDECOMP, newValue);
    }

    auto getHeurForceSpecialObj() const -> int {
      return p->getIntControl(XPRS_HEURFORCESPECIALOBJ);
    }

    void setHeurForceSpecialObj(int newValue) {
      p->setIntControl(XPRS_HEURFORCESPECIALOBJ, newValue);
    }

    auto getHeurSearchRootCutFreq() const -> int {
      return p->getIntControl(XPRS_HEURSEARCHROOTCUTFREQ);
    }

    void setHeurSearchRootCutFreq(int newValue) {
      p->setIntControl(XPRS_HEURSEARCHROOTCUTFREQ, newValue);
    }

    auto getPreElimQuad() const -> int {
      return p->getIntControl(XPRS_PREELIMQUAD);
    }

    void setPreElimQuad(int newValue) {
      p->setIntControl(XPRS_PREELIMQUAD, newValue);
    }

    auto getPreImplications() const -> int {
      return p->getIntControl(XPRS_PREIMPLICATIONS);
    }

    void setPreImplications(int newValue) {
      p->setIntControl(XPRS_PREIMPLICATIONS, newValue);
    }

    auto getTunerMode() const -> xpress::TunerMode {
      return static_cast<xpress::TunerMode>(p->getIntControl(XPRS_TUNERMODE));
    }

    void setTunerMode(xpress::TunerMode newValue) {
      p->setIntControl(XPRS_TUNERMODE, static_cast<int>(newValue));
    }

    auto getTunerMethod() const -> xpress::TunerMethod {
      return static_cast<xpress::TunerMethod>(
          p->getIntControl(XPRS_TUNERMETHOD));
    }

    void setTunerMethod(xpress::TunerMethod newValue) {
      p->setIntControl(XPRS_TUNERMETHOD, static_cast<int>(newValue));
    }

    auto getTunerTarget() const -> xpress::TunerTarget {
      return static_cast<xpress::TunerTarget>(
          p->getIntControl(XPRS_TUNERTARGET));
    }

    void setTunerTarget(xpress::TunerTarget newValue) {
      p->setIntControl(XPRS_TUNERTARGET, static_cast<int>(newValue));
    }

    auto getTunerThreads() const -> int {
      return p->getIntControl(XPRS_TUNERTHREADS);
    }

    void setTunerThreads(int newValue) {
      p->setIntControl(XPRS_TUNERTHREADS, newValue);
    }

    auto getTunerHistory() const -> xpress::TunerHistory {
      return static_cast<xpress::TunerHistory>(
          p->getIntControl(XPRS_TUNERHISTORY));
    }

    void setTunerHistory(xpress::TunerHistory newValue) {
      p->setIntControl(XPRS_TUNERHISTORY, static_cast<int>(newValue));
    }

    auto getTunerPermute() const -> int {
      return p->getIntControl(XPRS_TUNERPERMUTE);
    }

    void setTunerPermute(int newValue) {
      p->setIntControl(XPRS_TUNERPERMUTE, newValue);
    }

    auto getTunerVerbose() const -> int {
      return p->getIntControl(XPRS_TUNERVERBOSE);
    }

    void setTunerVerbose(int newValue) {
      p->setIntControl(XPRS_TUNERVERBOSE, newValue);
    }

    auto getTunerOutput() const -> int {
      return p->getIntControl(XPRS_TUNEROUTPUT);
    }

    void setTunerOutput(int newValue) {
      p->setIntControl(XPRS_TUNEROUTPUT, newValue);
    }

    auto getPreAnalyticcenter() const -> int {
      return p->getIntControl(XPRS_PREANALYTICCENTER);
    }

    void setPreAnalyticcenter(int newValue) {
      p->setIntControl(XPRS_PREANALYTICCENTER, newValue);
    }

    auto getNetCuts() const -> int {
      xpress_diags_push xpress_ignore_deprecated return p->getIntControl(
          XPRS_NETCUTS);
      xpress_diags_pop
    }

    void setNetCuts(int newValue) {
      xpress_diags_push xpress_ignore_deprecated p->setIntControl(XPRS_NETCUTS,
                                                                  newValue);
      xpress_diags_pop
    }

    auto getLpFlags() const -> xpress::LPFlags {
      return static_cast<xpress::LPFlags>(p->getIntControl(XPRS_LPFLAGS));
    }

    void setLpFlags(xpress::LPFlags newValue) {
      p->setIntControl(XPRS_LPFLAGS, static_cast<int>(newValue));
    }

    auto getMipKappaFreq() const -> int {
      return p->getIntControl(XPRS_MIPKAPPAFREQ);
    }

    void setMipKappaFreq(int newValue) {
      p->setIntControl(XPRS_MIPKAPPAFREQ, newValue);
    }

    auto getObjScaleFactor() const -> int {
      return p->getIntControl(XPRS_OBJSCALEFACTOR);
    }

    void setObjScaleFactor(int newValue) {
      p->setIntControl(XPRS_OBJSCALEFACTOR, newValue);
    }

    auto getTreeFileLogInterval() const -> int {
      return p->getIntControl(XPRS_TREEFILELOGINTERVAL);
    }

    void setTreeFileLogInterval(int newValue) {
      p->setIntControl(XPRS_TREEFILELOGINTERVAL, newValue);
    }

    auto getIgnoreContainerCpuLimit() const -> int {
      return p->getIntControl(XPRS_IGNORECONTAINERCPULIMIT);
    }

    void setIgnoreContainerCpuLimit(int newValue) {
      p->setIntControl(XPRS_IGNORECONTAINERCPULIMIT, newValue);
    }

    auto getIgnoreContainerMemoryLimit() const -> int {
      return p->getIntControl(XPRS_IGNORECONTAINERMEMORYLIMIT);
    }

    void setIgnoreContainerMemoryLimit(int newValue) {
      p->setIntControl(XPRS_IGNORECONTAINERMEMORYLIMIT, newValue);
    }

    auto getMipDualReductions() const -> int {
      return p->getIntControl(XPRS_MIPDUALREDUCTIONS);
    }

    void setMipDualReductions(int newValue) {
      p->setIntControl(XPRS_MIPDUALREDUCTIONS, newValue);
    }

    auto getGenconsDualReductions() const -> int {
      return p->getIntControl(XPRS_GENCONSDUALREDUCTIONS);
    }

    void setGenconsDualReductions(int newValue) {
      p->setIntControl(XPRS_GENCONSDUALREDUCTIONS, newValue);
    }

    auto getPwlDualReductions() const -> int {
      return p->getIntControl(XPRS_PWLDUALREDUCTIONS);
    }

    void setPwlDualReductions(int newValue) {
      p->setIntControl(XPRS_PWLDUALREDUCTIONS, newValue);
    }

    auto getBarFailIterLimit() const -> int {
      return p->getIntControl(XPRS_BARFAILITERLIMIT);
    }

    void setBarFailIterLimit(int newValue) {
      p->setIntControl(XPRS_BARFAILITERLIMIT, newValue);
    }

    auto getAutoScaling() const -> int {
      return p->getIntControl(XPRS_AUTOSCALING);
    }

    void setAutoScaling(int newValue) {
      p->setIntControl(XPRS_AUTOSCALING, newValue);
    }

    auto getGenconsAbsTransformation() const -> int {
      return p->getIntControl(XPRS_GENCONSABSTRANSFORMATION);
    }

    void setGenconsAbsTransformation(int newValue) {
      p->setIntControl(XPRS_GENCONSABSTRANSFORMATION, newValue);
    }

    auto getComputeJobPriority() const -> int {
      return p->getIntControl(XPRS_COMPUTEJOBPRIORITY);
    }

    void setComputeJobPriority(int newValue) {
      p->setIntControl(XPRS_COMPUTEJOBPRIORITY, newValue);
    }

    auto getPreFolding() const -> int {
      return p->getIntControl(XPRS_PREFOLDING);
    }

    void setPreFolding(int newValue) {
      p->setIntControl(XPRS_PREFOLDING, newValue);
    }

    auto getCompute() const -> int { return p->getIntControl(XPRS_COMPUTE); }

    void setCompute(int newValue) { p->setIntControl(XPRS_COMPUTE, newValue); }

    auto getNetStallLimit() const -> int {
      return p->getIntControl(XPRS_NETSTALLLIMIT);
    }

    void setNetStallLimit(int newValue) {
      p->setIntControl(XPRS_NETSTALLLIMIT, newValue);
    }

    auto getSerializePreIntSol() const -> int {
      return p->getIntControl(XPRS_SERIALIZEPREINTSOL);
    }

    void setSerializePreIntSol(int newValue) {
      p->setIntControl(XPRS_SERIALIZEPREINTSOL, newValue);
    }

    auto getNumericalEmphasis() const -> int {
      return p->getIntControl(XPRS_NUMERICALEMPHASIS);
    }

    void setNumericalEmphasis(int newValue) {
      p->setIntControl(XPRS_NUMERICALEMPHASIS, newValue);
    }

    auto getPwlNonConvexTransformation() const -> int {
      return p->getIntControl(XPRS_PWLNONCONVEXTRANSFORMATION);
    }

    void setPwlNonConvexTransformation(int newValue) {
      p->setIntControl(XPRS_PWLNONCONVEXTRANSFORMATION, newValue);
    }

    auto getMipComponents() const -> int {
      return p->getIntControl(XPRS_MIPCOMPONENTS);
    }

    void setMipComponents(int newValue) {
      p->setIntControl(XPRS_MIPCOMPONENTS, newValue);
    }

    auto getMipConcurrentNodes() const -> int {
      return p->getIntControl(XPRS_MIPCONCURRENTNODES);
    }

    void setMipConcurrentNodes(int newValue) {
      p->setIntControl(XPRS_MIPCONCURRENTNODES, newValue);
    }

    auto getMipConcurrentSolves() const -> int {
      return p->getIntControl(XPRS_MIPCONCURRENTSOLVES);
    }

    void setMipConcurrentSolves(int newValue) {
      p->setIntControl(XPRS_MIPCONCURRENTSOLVES, newValue);
    }

    auto getOutputControls() const -> int {
      return p->getIntControl(XPRS_OUTPUTCONTROLS);
    }

    void setOutputControls(int newValue) {
      p->setIntControl(XPRS_OUTPUTCONTROLS, newValue);
    }

    auto getSiftSwitch() const -> int {
      return p->getIntControl(XPRS_SIFTSWITCH);
    }

    void setSiftSwitch(int newValue) {
      p->setIntControl(XPRS_SIFTSWITCH, newValue);
    }

    auto getHeurEmphasis() const -> int {
      return p->getIntControl(XPRS_HEUREMPHASIS);
    }

    void setHeurEmphasis(int newValue) {
      p->setIntControl(XPRS_HEUREMPHASIS, newValue);
    }

    auto getBarRefIter() const -> int {
      return p->getIntControl(XPRS_BARREFITER);
    }

    void setBarRefIter(int newValue) {
      p->setIntControl(XPRS_BARREFITER, newValue);
    }

    auto getComputeLog() const -> xpress::ComputeLog {
      return static_cast<xpress::ComputeLog>(p->getIntControl(XPRS_COMPUTELOG));
    }

    void setComputeLog(xpress::ComputeLog newValue) {
      p->setIntControl(XPRS_COMPUTELOG, static_cast<int>(newValue));
    }

    auto getSiftPresolveOps() const -> int {
      return p->getIntControl(XPRS_SIFTPRESOLVEOPS);
    }

    void setSiftPresolveOps(int newValue) {
      p->setIntControl(XPRS_SIFTPRESOLVEOPS, newValue);
    }

    auto getCheckInputData() const -> int {
      return p->getIntControl(XPRS_CHECKINPUTDATA);
    }

    void setCheckInputData(int newValue) {
      p->setIntControl(XPRS_CHECKINPUTDATA, newValue);
    }

    auto getEscapeNames() const -> int {
      return p->getIntControl(XPRS_ESCAPENAMES);
    }

    void setEscapeNames(int newValue) {
      p->setIntControl(XPRS_ESCAPENAMES, newValue);
    }

    auto getIOTimeout() const -> int {
      return p->getIntControl(XPRS_IOTIMEOUT);
    }

    void setIOTimeout(int newValue) {
      p->setIntControl(XPRS_IOTIMEOUT, newValue);
    }

    auto getAutoCutting() const -> int {
      return p->getIntControl(XPRS_AUTOCUTTING);
    }

    void setAutoCutting(int newValue) {
      p->setIntControl(XPRS_AUTOCUTTING, newValue);
    }

    auto getGlobalNumInitNlpCuts() const -> int {
      return p->getIntControl(XPRS_GLOBALNUMINITNLPCUTS);
    }

    void setGlobalNumInitNlpCuts(int newValue) {
      p->setIntControl(XPRS_GLOBALNUMINITNLPCUTS, newValue);
    }

    auto getCallbackCheckTimeDelay() const -> int {
      return p->getIntControl(XPRS_CALLBACKCHECKTIMEDELAY);
    }

    void setCallbackCheckTimeDelay(int newValue) {
      p->setIntControl(XPRS_CALLBACKCHECKTIMEDELAY, newValue);
    }

    auto getMultiObjOps() const -> int {
      return p->getIntControl(XPRS_MULTIOBJOPS);
    }

    void setMultiObjOps(int newValue) {
      p->setIntControl(XPRS_MULTIOBJOPS, newValue);
    }

    auto getMultiObjLog() const -> int {
      return p->getIntControl(XPRS_MULTIOBJLOG);
    }

    void setMultiObjLog(int newValue) {
      p->setIntControl(XPRS_MULTIOBJLOG, newValue);
    }

    auto getBackgroundMaxThreads() const -> int {
      return p->getIntControl(XPRS_BACKGROUNDMAXTHREADS);
    }

    void setBackgroundMaxThreads(int newValue) {
      p->setIntControl(XPRS_BACKGROUNDMAXTHREADS, newValue);
    }

    auto getBackgroundSelect() const -> XPRSint64 {
      return p->getLongControl(XPRS_BACKGROUNDSELECT);
    }

    void setBackgroundSelect(XPRSint64 newValue) {
      p->setLongControl(XPRS_BACKGROUNDSELECT, newValue);
    }

    auto getGlobalLSHeurstrategy() const -> int {
      return p->getIntControl(XPRS_GLOBALLSHEURSTRATEGY);
    }

    void setGlobalLSHeurstrategy(int newValue) {
      p->setIntControl(XPRS_GLOBALLSHEURSTRATEGY, newValue);
    }

    auto getGlobalSpatialBranchIfPreferOrig() const -> int {
      return p->getIntControl(XPRS_GLOBALSPATIALBRANCHIFPREFERORIG);
    }

    void setGlobalSpatialBranchIfPreferOrig(int newValue) {
      p->setIntControl(XPRS_GLOBALSPATIALBRANCHIFPREFERORIG, newValue);
    }

    auto getPreConfiguration() const -> int {
      return p->getIntControl(XPRS_PRECONFIGURATION);
    }

    void setPreConfiguration(int newValue) {
      p->setIntControl(XPRS_PRECONFIGURATION, newValue);
    }

    auto getFeasibilityJump() const -> int {
      return p->getIntControl(XPRS_FEASIBILITYJUMP);
    }

    void setFeasibilityJump(int newValue) {
      p->setIntControl(XPRS_FEASIBILITYJUMP, newValue);
    }

    auto getIISOps() const -> xpress::IISOps {
      return static_cast<xpress::IISOps>(p->getIntControl(XPRS_IISOPS));
    }

    void setIISOps(xpress::IISOps newValue) {
      p->setIntControl(XPRS_IISOPS, static_cast<int>(newValue));
    }

    auto getRLTCuts() const -> int { return p->getIntControl(XPRS_RLTCUTS); }

    void setRLTCuts(int newValue) { p->setIntControl(XPRS_RLTCUTS, newValue); }

    auto getHeursearchBackgroundSelect() const -> XPRSint64 {
      return p->getLongControl(XPRS_HEURSEARCHBACKGROUNDSELECT);
    }

    void setHeursearchBackgroundSelect(XPRSint64 newValue) {
      p->setLongControl(XPRS_HEURSEARCHBACKGROUNDSELECT, newValue);
    }

    auto getAlternativeRedCosts() const -> int {
      return p->getIntControl(XPRS_ALTERNATIVEREDCOSTS);
    }

    void setAlternativeRedCosts(int newValue) {
      p->setIntControl(XPRS_ALTERNATIVEREDCOSTS, newValue);
    }

    auto getHeurShiftProp() const -> int {
      return p->getIntControl(XPRS_HEURSHIFTPROP);
    }

    void setHeurShiftProp(int newValue) {
      p->setIntControl(XPRS_HEURSHIFTPROP, newValue);
    }

    auto getHeurSearchCopyControls() const -> int {
      return p->getIntControl(XPRS_HEURSEARCHCOPYCONTROLS);
    }

    void setHeurSearchCopyControls(int newValue) {
      p->setIntControl(XPRS_HEURSEARCHCOPYCONTROLS, newValue);
    }

    auto getGlobalNlpCuts() const -> int {
      return p->getIntControl(XPRS_GLOBALNLPCUTS);
    }

    void setGlobalNlpCuts(int newValue) {
      p->setIntControl(XPRS_GLOBALNLPCUTS, newValue);
    }

    auto getGlobalTreeNlpCuts() const -> int {
      return p->getIntControl(XPRS_GLOBALTREENLPCUTS);
    }

    void setGlobalTreeNlpCuts(int newValue) {
      p->setIntControl(XPRS_GLOBALTREENLPCUTS, newValue);
    }

    auto getBarhgOps() const -> xpress::BARHGOps {
      return static_cast<xpress::BARHGOps>(p->getIntControl(XPRS_BARHGOPS));
    }

    void setBarhgOps(xpress::BARHGOps newValue) {
      p->setIntControl(XPRS_BARHGOPS, static_cast<int>(newValue));
    }

    auto getBarhgMaxRestarts() const -> int {
      return p->getIntControl(XPRS_BARHGMAXRESTARTS);
    }

    void setBarhgMaxRestarts(int newValue) {
      p->setIntControl(XPRS_BARHGMAXRESTARTS, newValue);
    }

    auto getMCFCutStrategy() const -> int {
      return p->getIntControl(XPRS_MCFCUTSTRATEGY);
    }

    void setMCFCutStrategy(int newValue) {
      p->setIntControl(XPRS_MCFCUTSTRATEGY, newValue);
    }

    auto getPreRootThreads() const -> int {
      return p->getIntControl(XPRS_PREROOTTHREADS);
    }

    void setPreRootThreads(int newValue) {
      p->setIntControl(XPRS_PREROOTTHREADS, newValue);
    }

    auto getBarIterative() const -> int {
      return p->getIntControl(XPRS_BARITERATIVE);
    }

    void setBarIterative(int newValue) {
      p->setIntControl(XPRS_BARITERATIVE, newValue);
    }

    auto getMaxCutTime() const -> double {
      return p->getDblControl(XPRS_MAXCUTTIME);
    }

    void setMaxCutTime(double newValue) {
      p->setDblControl(XPRS_MAXCUTTIME, newValue);
    }

    auto getMaxStallTime() const -> double {
      return p->getDblControl(XPRS_MAXSTALLTIME);
    }

    void setMaxStallTime(double newValue) {
      p->setDblControl(XPRS_MAXSTALLTIME, newValue);
    }

    auto getTunerMaxTime() const -> double {
      return p->getDblControl(XPRS_TUNERMAXTIME);
    }

    void setTunerMaxTime(double newValue) {
      p->setDblControl(XPRS_TUNERMAXTIME, newValue);
    }

    auto getMatrixTol() const -> double {
      return p->getDblControl(XPRS_MATRIXTOL);
    }

    void setMatrixTol(double newValue) {
      p->setDblControl(XPRS_MATRIXTOL, newValue);
    }

    auto getPivotTol() const -> double {
      return p->getDblControl(XPRS_PIVOTTOL);
    }

    void setPivotTol(double newValue) {
      p->setDblControl(XPRS_PIVOTTOL, newValue);
    }

    auto getFeasTol() const -> double { return p->getDblControl(XPRS_FEASTOL); }

    void setFeasTol(double newValue) {
      p->setDblControl(XPRS_FEASTOL, newValue);
    }

    auto getOutputTol() const -> double {
      return p->getDblControl(XPRS_OUTPUTTOL);
    }

    void setOutputTol(double newValue) {
      p->setDblControl(XPRS_OUTPUTTOL, newValue);
    }

    auto getSosRefTol() const -> double {
      return p->getDblControl(XPRS_SOSREFTOL);
    }

    void setSosRefTol(double newValue) {
      p->setDblControl(XPRS_SOSREFTOL, newValue);
    }

    auto getOptimalityTol() const -> double {
      return p->getDblControl(XPRS_OPTIMALITYTOL);
    }

    void setOptimalityTol(double newValue) {
      p->setDblControl(XPRS_OPTIMALITYTOL, newValue);
    }

    auto getEtaTol() const -> double { return p->getDblControl(XPRS_ETATOL); }

    void setEtaTol(double newValue) { p->setDblControl(XPRS_ETATOL, newValue); }

    auto getRelPivotTol() const -> double {
      return p->getDblControl(XPRS_RELPIVOTTOL);
    }

    void setRelPivotTol(double newValue) {
      p->setDblControl(XPRS_RELPIVOTTOL, newValue);
    }

    auto getMipTol() const -> double { return p->getDblControl(XPRS_MIPTOL); }

    void setMipTol(double newValue) { p->setDblControl(XPRS_MIPTOL, newValue); }

    auto getMipTolTarget() const -> double {
      return p->getDblControl(XPRS_MIPTOLTARGET);
    }

    void setMipTolTarget(double newValue) {
      p->setDblControl(XPRS_MIPTOLTARGET, newValue);
    }

    auto getBarPerturb() const -> double {
      return p->getDblControl(XPRS_BARPERTURB);
    }

    void setBarPerturb(double newValue) {
      p->setDblControl(XPRS_BARPERTURB, newValue);
    }

    auto getMipAddCutoff() const -> double {
      return p->getDblControl(XPRS_MIPADDCUTOFF);
    }

    void setMipAddCutoff(double newValue) {
      p->setDblControl(XPRS_MIPADDCUTOFF, newValue);
    }

    auto getMipAbsCutoff() const -> double {
      return p->getDblControl(XPRS_MIPABSCUTOFF);
    }

    void setMipAbsCutoff(double newValue) {
      p->setDblControl(XPRS_MIPABSCUTOFF, newValue);
    }

    auto getMipRelCutoff() const -> double {
      return p->getDblControl(XPRS_MIPRELCUTOFF);
    }

    void setMipRelCutoff(double newValue) {
      p->setDblControl(XPRS_MIPRELCUTOFF, newValue);
    }

    auto getPseudoCost() const -> double {
      return p->getDblControl(XPRS_PSEUDOCOST);
    }

    void setPseudoCost(double newValue) {
      p->setDblControl(XPRS_PSEUDOCOST, newValue);
    }

    auto getPenalty() const -> double { return p->getDblControl(XPRS_PENALTY); }

    void setPenalty(double newValue) {
      p->setDblControl(XPRS_PENALTY, newValue);
    }

    auto getBigM() const -> double { return p->getDblControl(XPRS_BIGM); }

    void setBigM(double newValue) { p->setDblControl(XPRS_BIGM, newValue); }

    auto getMipAbsStop() const -> double {
      return p->getDblControl(XPRS_MIPABSSTOP);
    }

    void setMipAbsStop(double newValue) {
      p->setDblControl(XPRS_MIPABSSTOP, newValue);
    }

    auto getMipRelStop() const -> double {
      return p->getDblControl(XPRS_MIPRELSTOP);
    }

    void setMipRelStop(double newValue) {
      p->setDblControl(XPRS_MIPRELSTOP, newValue);
    }

    auto getCrossoverAccuracyTol() const -> double {
      return p->getDblControl(XPRS_CROSSOVERACCURACYTOL);
    }

    void setCrossoverAccuracyTol(double newValue) {
      p->setDblControl(XPRS_CROSSOVERACCURACYTOL, newValue);
    }

    auto getPrimalPerturb() const -> double {
      return p->getDblControl(XPRS_PRIMALPERTURB);
    }

    void setPrimalPerturb(double newValue) {
      p->setDblControl(XPRS_PRIMALPERTURB, newValue);
    }

    auto getDualPerturb() const -> double {
      return p->getDblControl(XPRS_DUALPERTURB);
    }

    void setDualPerturb(double newValue) {
      p->setDblControl(XPRS_DUALPERTURB, newValue);
    }

    auto getBarObjScale() const -> double {
      return p->getDblControl(XPRS_BAROBJSCALE);
    }

    void setBarObjScale(double newValue) {
      p->setDblControl(XPRS_BAROBJSCALE, newValue);
    }

    auto getBarRhsScale() const -> double {
      return p->getDblControl(XPRS_BARRHSSCALE);
    }

    void setBarRhsScale(double newValue) {
      p->setDblControl(XPRS_BARRHSSCALE, newValue);
    }

    auto getCholeskyTol() const -> double {
      return p->getDblControl(XPRS_CHOLESKYTOL);
    }

    void setCholeskyTol(double newValue) {
      p->setDblControl(XPRS_CHOLESKYTOL, newValue);
    }

    auto getBarGapStop() const -> double {
      return p->getDblControl(XPRS_BARGAPSTOP);
    }

    void setBarGapStop(double newValue) {
      p->setDblControl(XPRS_BARGAPSTOP, newValue);
    }

    auto getBarDualStop() const -> double {
      return p->getDblControl(XPRS_BARDUALSTOP);
    }

    void setBarDualStop(double newValue) {
      p->setDblControl(XPRS_BARDUALSTOP, newValue);
    }

    auto getBarPrimalStop() const -> double {
      return p->getDblControl(XPRS_BARPRIMALSTOP);
    }

    void setBarPrimalStop(double newValue) {
      p->setDblControl(XPRS_BARPRIMALSTOP, newValue);
    }

    auto getBarStepStop() const -> double {
      return p->getDblControl(XPRS_BARSTEPSTOP);
    }

    void setBarStepStop(double newValue) {
      p->setDblControl(XPRS_BARSTEPSTOP, newValue);
    }

    auto getElimTol() const -> double { return p->getDblControl(XPRS_ELIMTOL); }

    void setElimTol(double newValue) {
      p->setDblControl(XPRS_ELIMTOL, newValue);
    }

    auto getMarkowitzTol() const -> double {
      return p->getDblControl(XPRS_MARKOWITZTOL);
    }

    void setMarkowitzTol(double newValue) {
      p->setDblControl(XPRS_MARKOWITZTOL, newValue);
    }

    auto getMipAbsGapNotify() const -> double {
      return p->getDblControl(XPRS_MIPABSGAPNOTIFY);
    }

    void setMipAbsGapNotify(double newValue) {
      p->setDblControl(XPRS_MIPABSGAPNOTIFY, newValue);
    }

    auto getMipRelGapNotify() const -> double {
      return p->getDblControl(XPRS_MIPRELGAPNOTIFY);
    }

    void setMipRelGapNotify(double newValue) {
      p->setDblControl(XPRS_MIPRELGAPNOTIFY, newValue);
    }

    auto getBarLargeBound() const -> double {
      return p->getDblControl(XPRS_BARLARGEBOUND);
    }

    void setBarLargeBound(double newValue) {
      p->setDblControl(XPRS_BARLARGEBOUND, newValue);
    }

    auto getPPFactor() const -> double {
      return p->getDblControl(XPRS_PPFACTOR);
    }

    void setPPFactor(double newValue) {
      p->setDblControl(XPRS_PPFACTOR, newValue);
    }

    auto getRepairIndefiniteQMax() const -> double {
      return p->getDblControl(XPRS_REPAIRINDEFINITEQMAX);
    }

    void setRepairIndefiniteQMax(double newValue) {
      p->setDblControl(XPRS_REPAIRINDEFINITEQMAX, newValue);
    }

    auto getBarGapTarget() const -> double {
      return p->getDblControl(XPRS_BARGAPTARGET);
    }

    void setBarGapTarget(double newValue) {
      p->setDblControl(XPRS_BARGAPTARGET, newValue);
    }

    auto getDummyControl() const -> double {
      return p->getDblControl(XPRS_DUMMYCONTROL);
    }

    void setDummyControl(double newValue) {
      p->setDblControl(XPRS_DUMMYCONTROL, newValue);
    }

    auto getBarStartWeight() const -> double {
      return p->getDblControl(XPRS_BARSTARTWEIGHT);
    }

    void setBarStartWeight(double newValue) {
      p->setDblControl(XPRS_BARSTARTWEIGHT, newValue);
    }

    auto getBarFreeScale() const -> double {
      return p->getDblControl(XPRS_BARFREESCALE);
    }

    void setBarFreeScale(double newValue) {
      p->setDblControl(XPRS_BARFREESCALE, newValue);
    }

    auto getSbEffort() const -> double {
      return p->getDblControl(XPRS_SBEFFORT);
    }

    void setSbEffort(double newValue) {
      p->setDblControl(XPRS_SBEFFORT, newValue);
    }

    auto getHeurDiveRandomize() const -> double {
      return p->getDblControl(XPRS_HEURDIVERANDOMIZE);
    }

    void setHeurDiveRandomize(double newValue) {
      p->setDblControl(XPRS_HEURDIVERANDOMIZE, newValue);
    }

    auto getHeurSearchEffort() const -> double {
      return p->getDblControl(XPRS_HEURSEARCHEFFORT);
    }

    void setHeurSearchEffort(double newValue) {
      p->setDblControl(XPRS_HEURSEARCHEFFORT, newValue);
    }

    auto getCutFactor() const -> double {
      return p->getDblControl(XPRS_CUTFACTOR);
    }

    void setCutFactor(double newValue) {
      p->setDblControl(XPRS_CUTFACTOR, newValue);
    }

    auto getEigenValueTol() const -> double {
      return p->getDblControl(XPRS_EIGENVALUETOL);
    }

    void setEigenValueTol(double newValue) {
      p->setDblControl(XPRS_EIGENVALUETOL, newValue);
    }

    auto getIndLinBigM() const -> double {
      return p->getDblControl(XPRS_INDLINBIGM);
    }

    void setIndLinBigM(double newValue) {
      p->setDblControl(XPRS_INDLINBIGM, newValue);
    }

    auto getTreeMemorySavingTarget() const -> double {
      return p->getDblControl(XPRS_TREEMEMORYSAVINGTARGET);
    }

    void setTreeMemorySavingTarget(double newValue) {
      p->setDblControl(XPRS_TREEMEMORYSAVINGTARGET, newValue);
    }

    auto getIndPreLinBigM() const -> double {
      return p->getDblControl(XPRS_INDPRELINBIGM);
    }

    void setIndPreLinBigM(double newValue) {
      p->setDblControl(XPRS_INDPRELINBIGM, newValue);
    }

    auto getRelaxTreeMemoryLimit() const -> double {
      return p->getDblControl(XPRS_RELAXTREEMEMORYLIMIT);
    }

    void setRelaxTreeMemoryLimit(double newValue) {
      p->setDblControl(XPRS_RELAXTREEMEMORYLIMIT, newValue);
    }

    auto getMipAbsGapNotifyObj() const -> double {
      return p->getDblControl(XPRS_MIPABSGAPNOTIFYOBJ);
    }

    void setMipAbsGapNotifyObj(double newValue) {
      p->setDblControl(XPRS_MIPABSGAPNOTIFYOBJ, newValue);
    }

    auto getMipAbsGapNotifyBound() const -> double {
      return p->getDblControl(XPRS_MIPABSGAPNOTIFYBOUND);
    }

    void setMipAbsGapNotifyBound(double newValue) {
      p->setDblControl(XPRS_MIPABSGAPNOTIFYBOUND, newValue);
    }

    auto getPresolveMaxGrow() const -> double {
      return p->getDblControl(XPRS_PRESOLVEMAXGROW);
    }

    void setPresolveMaxGrow(double newValue) {
      p->setDblControl(XPRS_PRESOLVEMAXGROW, newValue);
    }

    auto getHeurSearchTargetSize() const -> double {
      return p->getDblControl(XPRS_HEURSEARCHTARGETSIZE);
    }

    void setHeurSearchTargetSize(double newValue) {
      p->setDblControl(XPRS_HEURSEARCHTARGETSIZE, newValue);
    }

    auto getCrossOverRelPivotTol() const -> double {
      return p->getDblControl(XPRS_CROSSOVERRELPIVOTTOL);
    }

    void setCrossOverRelPivotTol(double newValue) {
      p->setDblControl(XPRS_CROSSOVERRELPIVOTTOL, newValue);
    }

    auto getCrossOverRelPivotTolSafe() const -> double {
      return p->getDblControl(XPRS_CROSSOVERRELPIVOTTOLSAFE);
    }

    void setCrossOverRelPivotTolSafe(double newValue) {
      p->setDblControl(XPRS_CROSSOVERRELPIVOTTOLSAFE, newValue);
    }

    auto getDetLogFreq() const -> double {
      return p->getDblControl(XPRS_DETLOGFREQ);
    }

    void setDetLogFreq(double newValue) {
      p->setDblControl(XPRS_DETLOGFREQ, newValue);
    }

    auto getMaxImpliedBound() const -> double {
      return p->getDblControl(XPRS_MAXIMPLIEDBOUND);
    }

    void setMaxImpliedBound(double newValue) {
      p->setDblControl(XPRS_MAXIMPLIEDBOUND, newValue);
    }

    auto getFeasTolTarget() const -> double {
      return p->getDblControl(XPRS_FEASTOLTARGET);
    }

    void setFeasTolTarget(double newValue) {
      p->setDblControl(XPRS_FEASTOLTARGET, newValue);
    }

    auto getOptimalityTolTarget() const -> double {
      return p->getDblControl(XPRS_OPTIMALITYTOLTARGET);
    }

    void setOptimalityTolTarget(double newValue) {
      p->setDblControl(XPRS_OPTIMALITYTOLTARGET, newValue);
    }

    auto getPreComponentsEffort() const -> double {
      return p->getDblControl(XPRS_PRECOMPONENTSEFFORT);
    }

    void setPreComponentsEffort(double newValue) {
      p->setDblControl(XPRS_PRECOMPONENTSEFFORT, newValue);
    }

    auto getLpLogDelay() const -> double {
      return p->getDblControl(XPRS_LPLOGDELAY);
    }

    void setLpLogDelay(double newValue) {
      p->setDblControl(XPRS_LPLOGDELAY, newValue);
    }

    auto getHeurDiveIterLimit() const -> double {
      return p->getDblControl(XPRS_HEURDIVEITERLIMIT);
    }

    void setHeurDiveIterLimit(double newValue) {
      p->setDblControl(XPRS_HEURDIVEITERLIMIT, newValue);
    }

    auto getBarKernel() const -> double {
      return p->getDblControl(XPRS_BARKERNEL);
    }

    void setBarKernel(double newValue) {
      p->setDblControl(XPRS_BARKERNEL, newValue);
    }

    auto getFeasTolPerturb() const -> double {
      return p->getDblControl(XPRS_FEASTOLPERTURB);
    }

    void setFeasTolPerturb(double newValue) {
      p->setDblControl(XPRS_FEASTOLPERTURB, newValue);
    }

    auto getCrossOverFeasWeight() const -> double {
      return p->getDblControl(XPRS_CROSSOVERFEASWEIGHT);
    }

    void setCrossOverFeasWeight(double newValue) {
      p->setDblControl(XPRS_CROSSOVERFEASWEIGHT, newValue);
    }

    auto getLUPivotTol() const -> double {
      return p->getDblControl(XPRS_LUPIVOTTOL);
    }

    void setLUPivotTol(double newValue) {
      p->setDblControl(XPRS_LUPIVOTTOL, newValue);
    }

    auto getMipRestartGapThreshold() const -> double {
      return p->getDblControl(XPRS_MIPRESTARTGAPTHRESHOLD);
    }

    void setMipRestartGapThreshold(double newValue) {
      p->setDblControl(XPRS_MIPRESTARTGAPTHRESHOLD, newValue);
    }

    auto getNodeProbingEffort() const -> double {
      return p->getDblControl(XPRS_NODEPROBINGEFFORT);
    }

    void setNodeProbingEffort(double newValue) {
      p->setDblControl(XPRS_NODEPROBINGEFFORT, newValue);
    }

    auto getInputtol() const -> double {
      return p->getDblControl(XPRS_INPUTTOL);
    }

    void setInputtol(double newValue) {
      p->setDblControl(XPRS_INPUTTOL, newValue);
    }

    auto getMipRestartFactor() const -> double {
      return p->getDblControl(XPRS_MIPRESTARTFACTOR);
    }

    void setMipRestartFactor(double newValue) {
      p->setDblControl(XPRS_MIPRESTARTFACTOR, newValue);
    }

    auto getBarObjPerturb() const -> double {
      return p->getDblControl(XPRS_BAROBJPERTURB);
    }

    void setBarObjPerturb(double newValue) {
      p->setDblControl(XPRS_BAROBJPERTURB, newValue);
    }

    auto getCpiAlpha() const -> double {
      return p->getDblControl(XPRS_CPIALPHA);
    }

    void setCpiAlpha(double newValue) {
      p->setDblControl(XPRS_CPIALPHA, newValue);
    }

    auto getGlobalSpatialBranchPropagationEffort() const -> double {
      return p->getDblControl(XPRS_GLOBALSPATIALBRANCHPROPAGATIONEFFORT);
    }

    void setGlobalSpatialBranchPropagationEffort(double newValue) {
      p->setDblControl(XPRS_GLOBALSPATIALBRANCHPROPAGATIONEFFORT, newValue);
    }

    auto getGlobalSpatialBranchCuttingEffort() const -> double {
      return p->getDblControl(XPRS_GLOBALSPATIALBRANCHCUTTINGEFFORT);
    }

    void setGlobalSpatialBranchCuttingEffort(double newValue) {
      p->setDblControl(XPRS_GLOBALSPATIALBRANCHCUTTINGEFFORT, newValue);
    }

    auto getGlobalBoundingBox() const -> double {
      return p->getDblControl(XPRS_GLOBALBOUNDINGBOX);
    }

    void setGlobalBoundingBox(double newValue) {
      p->setDblControl(XPRS_GLOBALBOUNDINGBOX, newValue);
    }

    auto getTimeLimit() const -> double {
      return p->getDblControl(XPRS_TIMELIMIT);
    }

    void setTimeLimit(double newValue) {
      p->setDblControl(XPRS_TIMELIMIT, newValue);
    }

    auto getSolTimeLimit() const -> double {
      return p->getDblControl(XPRS_SOLTIMELIMIT);
    }

    void setSolTimeLimit(double newValue) {
      p->setDblControl(XPRS_SOLTIMELIMIT, newValue);
    }

    auto getRepairInfeasTimeLimit() const -> double {
      return p->getDblControl(XPRS_REPAIRINFEASTIMELIMIT);
    }

    void setRepairInfeasTimeLimit(double newValue) {
      p->setDblControl(XPRS_REPAIRINFEASTIMELIMIT, newValue);
    }

    auto getBarhgExtrapolate() const -> double {
      return p->getDblControl(XPRS_BARHGEXTRAPOLATE);
    }

    void setBarhgExtrapolate(double newValue) {
      p->setDblControl(XPRS_BARHGEXTRAPOLATE, newValue);
    }

    auto getWorkLimit() const -> double {
      return p->getDblControl(XPRS_WORKLIMIT);
    }

    void setWorkLimit(double newValue) {
      p->setDblControl(XPRS_WORKLIMIT, newValue);
    }

    auto getCallbackCheckTimeWorkDelay() const -> double {
      return p->getDblControl(XPRS_CALLBACKCHECKTIMEWORKDELAY);
    }

    void setCallbackCheckTimeWorkDelay(double newValue) {
      p->setDblControl(XPRS_CALLBACKCHECKTIMEWORKDELAY, newValue);
    }

    auto getPreRootWorkLimit() const -> double {
      return p->getDblControl(XPRS_PREROOTWORKLIMIT);
    }

    void setPreRootWorkLimit(double newValue) {
      p->setDblControl(XPRS_PREROOTWORKLIMIT, newValue);
    }

    auto getPreRootEffort() const -> double {
      return p->getDblControl(XPRS_PREROOTEFFORT);
    }

    void setPreRootEffort(double newValue) {
      p->setDblControl(XPRS_PREROOTEFFORT, newValue);
    }

    auto getMpsRhsName() const -> std::string {
      int count;
      p->getStringControl(XPRS_MPSRHSNAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_MPSRHSNAME, bytes.data(), count, &count);
      return bytes.data();
    }

    void setMpsRhsName(std::string const &newValue) {
      p->setStrControl(XPRS_MPSRHSNAME, newValue);
    }

    auto getMpsObjName() const -> std::string {
      int count;
      p->getStringControl(XPRS_MPSOBJNAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_MPSOBJNAME, bytes.data(), count, &count);
      return bytes.data();
    }

    void setMpsObjName(std::string const &newValue) {
      p->setStrControl(XPRS_MPSOBJNAME, newValue);
    }

    auto getMpsRangeName() const -> std::string {
      int count;
      p->getStringControl(XPRS_MPSRANGENAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_MPSRANGENAME, bytes.data(), count, &count);
      return bytes.data();
    }

    void setMpsRangeName(std::string const &newValue) {
      p->setStrControl(XPRS_MPSRANGENAME, newValue);
    }

    auto getMpsBoundName() const -> std::string {
      int count;
      p->getStringControl(XPRS_MPSBOUNDNAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_MPSBOUNDNAME, bytes.data(), count, &count);
      return bytes.data();
    }

    void setMpsBoundName(std::string const &newValue) {
      p->setStrControl(XPRS_MPSBOUNDNAME, newValue);
    }

    auto getOutputMask() const -> std::string {
      int count;
      p->getStringControl(XPRS_OUTPUTMASK, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_OUTPUTMASK, bytes.data(), count, &count);
      return bytes.data();
    }

    void setOutputMask(std::string const &newValue) {
      p->setStrControl(XPRS_OUTPUTMASK, newValue);
    }

    auto getTunerMethodFile() const -> std::string {
      int count;
      p->getStringControl(XPRS_TUNERMETHODFILE, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_TUNERMETHODFILE, bytes.data(), count, &count);
      return bytes.data();
    }

    void setTunerMethodFile(std::string const &newValue) {
      p->setStrControl(XPRS_TUNERMETHODFILE, newValue);
    }

    auto getTunerOutputPath() const -> std::string {
      int count;
      p->getStringControl(XPRS_TUNEROUTPUTPATH, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_TUNEROUTPUTPATH, bytes.data(), count, &count);
      return bytes.data();
    }

    void setTunerOutputPath(std::string const &newValue) {
      p->setStrControl(XPRS_TUNEROUTPUTPATH, newValue);
    }

    auto getTunerSessionName() const -> std::string {
      int count;
      p->getStringControl(XPRS_TUNERSESSIONNAME, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_TUNERSESSIONNAME, bytes.data(), count, &count);
      return bytes.data();
    }

    void setTunerSessionName(std::string const &newValue) {
      p->setStrControl(XPRS_TUNERSESSIONNAME, newValue);
    }

    auto getComputeExecService() const -> std::string {
      int count;
      p->getStringControl(XPRS_COMPUTEEXECSERVICE, nullptr, 0, &count);
      std::vector<char> bytes(count);
      p->getStringControl(XPRS_COMPUTEEXECSERVICE, bytes.data(), count, &count);
      return bytes.data();
    }

    void setComputeExecService(std::string const &newValue) {
      p->setStrControl(XPRS_COMPUTEEXECSERVICE, newValue);
    }

    auto getMspDuplicateSolutionsPolicy() const -> int {
      return p->getIntControl(XPRS_MSP_DUPLICATESOLUTIONSPOLICY);
    }

    void setMspDuplicateSolutionsPolicy(int newValue) {
      p->setIntControl(XPRS_MSP_DUPLICATESOLUTIONSPOLICY, newValue);
    }

    auto getMspDefaultUserSolFeasTol() const -> double {
      return p->getDblControl(XPRS_MSP_DEFAULTUSERSOLFEASTOL);
    }

    void setMspDefaultUserSolFeasTol(double newValue) {
      p->setDblControl(XPRS_MSP_DEFAULTUSERSOLFEASTOL, newValue);
    }

    auto getMspDefaultUserSolMipTol() const -> double {
      return p->getDblControl(XPRS_MSP_DEFAULTUSERSOLMIPTOL);
    }

    void setMspDefaultUserSolMipTol(double newValue) {
      p->setDblControl(XPRS_MSP_DEFAULTUSERSOLMIPTOL, newValue);
    }

    auto getMspIncludeProbNameInLogging() const -> int {
      return p->getIntControl(XPRS_MSP_INCLUDEPROBNAMEINLOGGING);
    }

    void setMspIncludeProbNameInLogging(int newValue) {
      p->setIntControl(XPRS_MSP_INCLUDEPROBNAMEINLOGGING, newValue);
    }

    auto getMspWriteSlxSolLogging() const -> int {
      return p->getIntControl(XPRS_MSP_WRITESLXSOLLOGGING);
    }

    void setMspWriteSlxSolLogging(int newValue) {
      p->setIntControl(XPRS_MSP_WRITESLXSOLLOGGING, newValue);
    }

    auto getMspEnableSlackStorage() const -> int {
      return p->getIntControl(XPRS_MSP_ENABLESLACKSTORAGE);
    }

    void setMspEnableSlackStorage(int newValue) {
      p->setIntControl(XPRS_MSP_ENABLESLACKSTORAGE, newValue);
    }

    auto getMspOutputLog() const -> int {
      return p->getIntControl(XPRS_MSP_OUTPUTLOG);
    }

    void setMspOutputLog(int newValue) {
      p->setIntControl(XPRS_MSP_OUTPUTLOG, newValue);
    }

    auto getMspSol_FeasTol() const -> double {
      return p->getDblControl(XPRS_MSP_SOL_FEASTOL);
    }

    void setMspSol_FeasTol(double newValue) {
      p->setDblControl(XPRS_MSP_SOL_FEASTOL, newValue);
    }

    auto getMspSol_MipTol() const -> double {
      return p->getDblControl(XPRS_MSP_SOL_MIPTOL);
    }

    void setMspSol_MipTol(double newValue) {
      p->setDblControl(XPRS_MSP_SOL_MIPTOL, newValue);
    }

    auto getMspSol_BitFieldsUsr() const -> int {
      return p->getIntControl(XPRS_MSP_SOL_BITFIELDSUSR);
    }

    void setMspSol_BitFieldsUsr(int newValue) {
      p->setIntControl(XPRS_MSP_SOL_BITFIELDSUSR, newValue);
    }

    auto getMseCallbackCullSols_MipObject() const -> int {
      return p->getIntControl(XPRS_MSE_CALLBACKCULLSOLS_MIPOBJECT);
    }

    void setMseCallbackCullSols_MipObject(int newValue) {
      p->setIntControl(XPRS_MSE_CALLBACKCULLSOLS_MIPOBJECT, newValue);
    }

    auto getMseCallbackCullSols_Diversity() const -> int {
      return p->getIntControl(XPRS_MSE_CALLBACKCULLSOLS_DIVERSITY);
    }

    void setMseCallbackCullSols_Diversity(int newValue) {
      p->setIntControl(XPRS_MSE_CALLBACKCULLSOLS_DIVERSITY, newValue);
    }

    auto getMseCallbackCullSols_ModObject() const -> int {
      return p->getIntControl(XPRS_MSE_CALLBACKCULLSOLS_MODOBJECT);
    }

    void setMseCallbackCullSols_ModObject(int newValue) {
      p->setIntControl(XPRS_MSE_CALLBACKCULLSOLS_MODOBJECT, newValue);
    }

    auto getMseOptimizeDiversity() const -> int {
      return p->getIntControl(XPRS_MSE_OPTIMIZEDIVERSITY);
    }

    void setMseOptimizeDiversity(int newValue) {
      p->setIntControl(XPRS_MSE_OPTIMIZEDIVERSITY, newValue);
    }

    auto getMseOutputTol() const -> double {
      return p->getDblControl(XPRS_MSE_OUTPUTTOL);
    }

    void setMseOutputTol(double newValue) {
      p->setDblControl(XPRS_MSE_OUTPUTTOL, newValue);
    }

    auto getMseOutputLog() const -> int {
      return p->getIntControl(XPRS_MSE_OUTPUTLOG);
    }

    void setMseOutputLog(int newValue) {
      p->setIntControl(XPRS_MSE_OUTPUTLOG, newValue);
    }

    auto getNlpFuncEval() const -> int {
      return p->getIntControl(XPRS_NLPFUNCEVAL);
    }

    void setNlpFuncEval(int newValue) {
      p->setIntControl(XPRS_NLPFUNCEVAL, newValue);
    }

    auto getNlpLog() const -> int { return p->getIntControl(XPRS_NLPLOG); }

    void setNlpLog(int newValue) { p->setIntControl(XPRS_NLPLOG, newValue); }

    auto getNlpEvaluate() const -> int {
      return p->getIntControl(XPRS_NLPEVALUATE);
    }

    void setNlpEvaluate(int newValue) {
      p->setIntControl(XPRS_NLPEVALUATE, newValue);
    }

    auto getNlpPresolve() const -> int {
      return p->getIntControl(XPRS_NLPPRESOLVE);
    }

    void setNlpPresolve(int newValue) {
      p->setIntControl(XPRS_NLPPRESOLVE, newValue);
    }

    auto getLocalSolver() const -> int {
      return p->getIntControl(XPRS_LOCALSOLVER);
    }

    void setLocalSolver(int newValue) {
      p->setIntControl(XPRS_LOCALSOLVER, newValue);
    }

    auto getNlpStopOutOfRange() const -> int {
      return p->getIntControl(XPRS_NLPSTOPOUTOFRANGE);
    }

    void setNlpStopOutOfRange(int newValue) {
      p->setIntControl(XPRS_NLPSTOPOUTOFRANGE, newValue);
    }

    auto getNlpThreadSafeUserFunc() const -> int {
      return p->getIntControl(XPRS_NLPTHREADSAFEUSERFUNC);
    }

    void setNlpThreadSafeUserFunc(int newValue) {
      p->setIntControl(XPRS_NLPTHREADSAFEUSERFUNC, newValue);
    }

    auto getNlpJacobian() const -> int {
      return p->getIntControl(XPRS_NLPJACOBIAN);
    }

    void setNlpJacobian(int newValue) {
      p->setIntControl(XPRS_NLPJACOBIAN, newValue);
    }

    auto getNlpHessian() const -> int {
      return p->getIntControl(XPRS_NLPHESSIAN);
    }

    void setNlpHessian(int newValue) {
      p->setIntControl(XPRS_NLPHESSIAN, newValue);
    }

    auto getMultiStart() const -> int {
      return p->getIntControl(XPRS_MULTISTART);
    }

    void setMultiStart(int newValue) {
      p->setIntControl(XPRS_MULTISTART, newValue);
    }

    auto getMultiStart_Threads() const -> int {
      return p->getIntControl(XPRS_MULTISTART_THREADS);
    }

    void setMultiStart_Threads(int newValue) {
      p->setIntControl(XPRS_MULTISTART_THREADS, newValue);
    }

    auto getMultiStart_MaxSolves() const -> int {
      return p->getIntControl(XPRS_MULTISTART_MAXSOLVES);
    }

    void setMultiStart_MaxSolves(int newValue) {
      p->setIntControl(XPRS_MULTISTART_MAXSOLVES, newValue);
    }

    auto getMultiStart_MaxTime() const -> int {
      return p->getIntControl(XPRS_MULTISTART_MAXTIME);
    }

    void setMultiStart_MaxTime(int newValue) {
      p->setIntControl(XPRS_MULTISTART_MAXTIME, newValue);
    }

    auto getNlpMaxTime() const -> int {
      return p->getIntControl(XPRS_NLPMAXTIME);
    }

    void setNlpMaxTime(int newValue) {
      p->setIntControl(XPRS_NLPMAXTIME, newValue);
    }

    auto getNlpDerivatives() const -> int {
      return p->getIntControl(XPRS_NLPDERIVATIVES);
    }

    void setNlpDerivatives(int newValue) {
      p->setIntControl(XPRS_NLPDERIVATIVES, newValue);
    }

    auto getNlpReformulate() const -> int {
      return p->getIntControl(XPRS_NLPREFORMULATE);
    }

    void setNlpReformulate(int newValue) {
      p->setIntControl(XPRS_NLPREFORMULATE, newValue);
    }

    auto getNlpPresolveOps() const -> int {
      return p->getIntControl(XPRS_NLPPRESOLVEOPS);
    }

    void setNlpPresolveOps(int newValue) {
      p->setIntControl(XPRS_NLPPRESOLVEOPS, newValue);
    }

    auto getMultiStart_Log() const -> int {
      return p->getIntControl(XPRS_MULTISTART_LOG);
    }

    void setMultiStart_Log(int newValue) {
      p->setIntControl(XPRS_MULTISTART_LOG, newValue);
    }

    auto getMultiStart_Seed() const -> int {
      return p->getIntControl(XPRS_MULTISTART_SEED);
    }

    void setMultiStart_Seed(int newValue) {
      p->setIntControl(XPRS_MULTISTART_SEED, newValue);
    }

    auto getMultiStart_PoolSize() const -> int {
      return p->getIntControl(XPRS_MULTISTART_POOLSIZE);
    }

    void setMultiStart_PoolSize(int newValue) {
      p->setIntControl(XPRS_MULTISTART_POOLSIZE, newValue);
    }

    auto getNlpPostsolve() const -> int {
      return p->getIntControl(XPRS_NLPPOSTSOLVE);
    }

    void setNlpPostsolve(int newValue) {
      p->setIntControl(XPRS_NLPPOSTSOLVE, newValue);
    }

    auto getNlpDeterministic() const -> int {
      return p->getIntControl(XPRS_NLPDETERMINISTIC);
    }

    void setNlpDeterministic(int newValue) {
      p->setIntControl(XPRS_NLPDETERMINISTIC, newValue);
    }

    auto getNlpPresolveLevel() const -> int {
      return p->getIntControl(XPRS_NLPPRESOLVELEVEL);
    }

    void setNlpPresolveLevel(int newValue) {
      p->setIntControl(XPRS_NLPPRESOLVELEVEL, newValue);
    }

    auto getNlpProbing() const -> int {
      return p->getIntControl(XPRS_NLPPROBING);
    }

    void setNlpProbing(int newValue) {
      p->setIntControl(XPRS_NLPPROBING, newValue);
    }

    auto getNlpCalcThreads() const -> int {
      return p->getIntControl(XPRS_NLPCALCTHREADS);
    }

    void setNlpCalcThreads(int newValue) {
      p->setIntControl(XPRS_NLPCALCTHREADS, newValue);
    }

    auto getNlpThreads() const -> int {
      return p->getIntControl(XPRS_NLPTHREADS);
    }

    void setNlpThreads(int newValue) {
      p->setIntControl(XPRS_NLPTHREADS, newValue);
    }

    auto getNlpFindIV() const -> int {
      return p->getIntControl(XPRS_NLPFINDIV);
    }

    void setNlpFindIV(int newValue) {
      p->setIntControl(XPRS_NLPFINDIV, newValue);
    }

    auto getNlpLinQuadBR() const -> int {
      return p->getIntControl(XPRS_NLPLINQUADBR);
    }

    void setNlpLinQuadBR(int newValue) {
      p->setIntControl(XPRS_NLPLINQUADBR, newValue);
    }

    auto getNlpSolver() const -> int {
      return p->getIntControl(XPRS_NLPSOLVER);
    }

    void setNlpSolver(int newValue) {
      p->setIntControl(XPRS_NLPSOLVER, newValue);
    }

    auto getSlpAlgorithm() const -> int {
      return p->getIntControl(XPRS_SLPALGORITHM);
    }

    void setSlpAlgorithm(int newValue) {
      p->setIntControl(XPRS_SLPALGORITHM, newValue);
    }

    auto getSlpAugmentation() const -> int {
      return p->getIntControl(XPRS_SLPAUGMENTATION);
    }

    void setSlpAugmentation(int newValue) {
      p->setIntControl(XPRS_SLPAUGMENTATION, newValue);
    }

    auto getSlpBarLimit() const -> int {
      return p->getIntControl(XPRS_SLPBARLIMIT);
    }

    void setSlpBarLimit(int newValue) {
      p->setIntControl(XPRS_SLPBARLIMIT, newValue);
    }

    auto getSlpCascade() const -> int {
      return p->getIntControl(XPRS_SLPCASCADE);
    }

    void setSlpCascade(int newValue) {
      p->setIntControl(XPRS_SLPCASCADE, newValue);
    }

    auto getSlpCascadeNLimit() const -> int {
      return p->getIntControl(XPRS_SLPCASCADENLIMIT);
    }

    void setSlpCascadeNLimit(int newValue) {
      p->setIntControl(XPRS_SLPCASCADENLIMIT, newValue);
    }

    auto getSlpDampStart() const -> int {
      return p->getIntControl(XPRS_SLPDAMPSTART);
    }

    void setSlpDampStart(int newValue) {
      p->setIntControl(XPRS_SLPDAMPSTART, newValue);
    }

    auto getSlpCutStrategy() const -> int {
      return p->getIntControl(XPRS_SLPCUTSTRATEGY);
    }

    void setSlpCutStrategy(int newValue) {
      p->setIntControl(XPRS_SLPCUTSTRATEGY, newValue);
    }

    auto getSlpDeltaZLimit() const -> int {
      return p->getIntControl(XPRS_SLPDELTAZLIMIT);
    }

    void setSlpDeltaZLimit(int newValue) {
      p->setIntControl(XPRS_SLPDELTAZLIMIT, newValue);
    }

    auto getSlpInfeasLimit() const -> int {
      return p->getIntControl(XPRS_SLPINFEASLIMIT);
    }

    void setSlpInfeasLimit(int newValue) {
      p->setIntControl(XPRS_SLPINFEASLIMIT, newValue);
    }

    auto getSlpIterLimit() const -> int {
      return p->getIntControl(XPRS_SLPITERLIMIT);
    }

    void setSlpIterLimit(int newValue) {
      p->setIntControl(XPRS_SLPITERLIMIT, newValue);
    }

    auto getSlpSameCount() const -> int {
      return p->getIntControl(XPRS_SLPSAMECOUNT);
    }

    void setSlpSameCount(int newValue) {
      p->setIntControl(XPRS_SLPSAMECOUNT, newValue);
    }

    auto getSlpSameDamp() const -> int {
      return p->getIntControl(XPRS_SLPSAMEDAMP);
    }

    void setSlpSameDamp(int newValue) {
      p->setIntControl(XPRS_SLPSAMEDAMP, newValue);
    }

    auto getSlpSBStart() const -> int {
      return p->getIntControl(XPRS_SLPSBSTART);
    }

    void setSlpSBStart(int newValue) {
      p->setIntControl(XPRS_SLPSBSTART, newValue);
    }

    auto getSlpXCount() const -> int {
      return p->getIntControl(XPRS_SLPXCOUNT);
    }

    void setSlpXCount(int newValue) {
      p->setIntControl(XPRS_SLPXCOUNT, newValue);
    }

    auto getSlpXLimit() const -> int {
      return p->getIntControl(XPRS_SLPXLIMIT);
    }

    void setSlpXLimit(int newValue) {
      p->setIntControl(XPRS_SLPXLIMIT, newValue);
    }

    auto getSlpDelayUpdateRows() const -> int {
      return p->getIntControl(XPRS_SLPDELAYUPDATEROWS);
    }

    void setSlpDelayUpdateRows(int newValue) {
      p->setIntControl(XPRS_SLPDELAYUPDATEROWS, newValue);
    }

    auto getSlpAutoSave() const -> int {
      return p->getIntControl(XPRS_SLPAUTOSAVE);
    }

    void setSlpAutoSave(int newValue) {
      p->setIntControl(XPRS_SLPAUTOSAVE, newValue);
    }

    auto getSlpAnalyze() const -> int {
      return p->getIntControl(XPRS_SLPANALYZE);
    }

    void setSlpAnalyze(int newValue) {
      p->setIntControl(XPRS_SLPANALYZE, newValue);
    }

    auto getSlpOCount() const -> int {
      return p->getIntControl(XPRS_SLPOCOUNT);
    }

    void setSlpOCount(int newValue) {
      p->setIntControl(XPRS_SLPOCOUNT, newValue);
    }

    auto getSlpMipAlgorithm() const -> int {
      return p->getIntControl(XPRS_SLPMIPALGORITHM);
    }

    void setSlpMipAlgorithm(int newValue) {
      p->setIntControl(XPRS_SLPMIPALGORITHM, newValue);
    }

    auto getSlpMipRelaxStepBounds() const -> int {
      return p->getIntControl(XPRS_SLPMIPRELAXSTEPBOUNDS);
    }

    void setSlpMipRelaxStepBounds(int newValue) {
      p->setIntControl(XPRS_SLPMIPRELAXSTEPBOUNDS, newValue);
    }

    auto getSlpMipFixStepBounds() const -> int {
      return p->getIntControl(XPRS_SLPMIPFIXSTEPBOUNDS);
    }

    void setSlpMipFixStepBounds(int newValue) {
      p->setIntControl(XPRS_SLPMIPFIXSTEPBOUNDS, newValue);
    }

    auto getSlpMipIterLimit() const -> int {
      return p->getIntControl(XPRS_SLPMIPITERLIMIT);
    }

    void setSlpMipIterLimit(int newValue) {
      p->setIntControl(XPRS_SLPMIPITERLIMIT, newValue);
    }

    auto getSlpMipCutoffLimit() const -> int {
      return p->getIntControl(XPRS_SLPMIPCUTOFFLIMIT);
    }

    void setSlpMipCutoffLimit(int newValue) {
      p->setIntControl(XPRS_SLPMIPCUTOFFLIMIT, newValue);
    }

    auto getSlpMipOCount() const -> int {
      return p->getIntControl(XPRS_SLPMIPOCOUNT);
    }

    void setSlpMipOCount(int newValue) {
      p->setIntControl(XPRS_SLPMIPOCOUNT, newValue);
    }

    auto getSlpMipDefaultAlgorithm() const -> int {
      return p->getIntControl(XPRS_SLPMIPDEFAULTALGORITHM);
    }

    void setSlpMipDefaultAlgorithm(int newValue) {
      p->setIntControl(XPRS_SLPMIPDEFAULTALGORITHM, newValue);
    }

    auto getSlpMipLog() const -> int {
      return p->getIntControl(XPRS_SLPMIPLOG);
    }

    void setSlpMipLog(int newValue) {
      p->setIntControl(XPRS_SLPMIPLOG, newValue);
    }

    auto getSlpDeltaOffset() const -> int {
      return p->getIntControl(XPRS_SLPDELTAOFFSET);
    }

    void setSlpDeltaOffset(int newValue) {
      p->setIntControl(XPRS_SLPDELTAOFFSET, newValue);
    }

    auto getSlpUpdateOffset() const -> int {
      return p->getIntControl(XPRS_SLPUPDATEOFFSET);
    }

    void setSlpUpdateOffset(int newValue) {
      p->setIntControl(XPRS_SLPUPDATEOFFSET, newValue);
    }

    auto getSlpErrorOffset() const -> int {
      return p->getIntControl(XPRS_SLPERROROFFSET);
    }

    void setSlpErrorOffset(int newValue) {
      p->setIntControl(XPRS_SLPERROROFFSET, newValue);
    }

    auto getSlpSBRowOffset() const -> int {
      return p->getIntControl(XPRS_SLPSBROWOFFSET);
    }

    void setSlpSBRowOffset(int newValue) {
      p->setIntControl(XPRS_SLPSBROWOFFSET, newValue);
    }

    auto getSlpVCount() const -> int {
      return p->getIntControl(XPRS_SLPVCOUNT);
    }

    void setSlpVCount(int newValue) {
      p->setIntControl(XPRS_SLPVCOUNT, newValue);
    }

    auto getSlpVLimit() const -> int {
      return p->getIntControl(XPRS_SLPVLIMIT);
    }

    void setSlpVLimit(int newValue) {
      p->setIntControl(XPRS_SLPVLIMIT, newValue);
    }

    auto getSlpECFCheck() const -> int {
      return p->getIntControl(XPRS_SLPECFCHECK);
    }

    void setSlpECFCheck(int newValue) {
      p->setIntControl(XPRS_SLPECFCHECK, newValue);
    }

    auto getSlpMipCutOffCount() const -> int {
      return p->getIntControl(XPRS_SLPMIPCUTOFFCOUNT);
    }

    void setSlpMipCutOffCount(int newValue) {
      p->setIntControl(XPRS_SLPMIPCUTOFFCOUNT, newValue);
    }

    auto getSlpWCount() const -> int {
      return p->getIntControl(XPRS_SLPWCOUNT);
    }

    void setSlpWCount(int newValue) {
      p->setIntControl(XPRS_SLPWCOUNT, newValue);
    }

    auto getSlpUnFinishedLimit() const -> int {
      return p->getIntControl(XPRS_SLPUNFINISHEDLIMIT);
    }

    void setSlpUnFinishedLimit(int newValue) {
      p->setIntControl(XPRS_SLPUNFINISHEDLIMIT, newValue);
    }

    auto getSlpConvergenceOps() const -> int {
      return p->getIntControl(XPRS_SLPCONVERGENCEOPS);
    }

    void setSlpConvergenceOps(int newValue) {
      p->setIntControl(XPRS_SLPCONVERGENCEOPS, newValue);
    }

    auto getSlpZeroCriterion() const -> int {
      return p->getIntControl(XPRS_SLPZEROCRITERION);
    }

    void setSlpZeroCriterion(int newValue) {
      p->setIntControl(XPRS_SLPZEROCRITERION, newValue);
    }

    auto getSlpZeroCriterionStart() const -> int {
      return p->getIntControl(XPRS_SLPZEROCRITERIONSTART);
    }

    void setSlpZeroCriterionStart(int newValue) {
      p->setIntControl(XPRS_SLPZEROCRITERIONSTART, newValue);
    }

    auto getSlpZeroCriterionCount() const -> int {
      return p->getIntControl(XPRS_SLPZEROCRITERIONCOUNT);
    }

    void setSlpZeroCriterionCount(int newValue) {
      p->setIntControl(XPRS_SLPZEROCRITERIONCOUNT, newValue);
    }

    auto getSlpLSPatternLimit() const -> int {
      return p->getIntControl(XPRS_SLPLSPATTERNLIMIT);
    }

    void setSlpLSPatternLimit(int newValue) {
      p->setIntControl(XPRS_SLPLSPATTERNLIMIT, newValue);
    }

    auto getSlpLSIterLimit() const -> int {
      return p->getIntControl(XPRS_SLPLSITERLIMIT);
    }

    void setSlpLSIterLimit(int newValue) {
      p->setIntControl(XPRS_SLPLSITERLIMIT, newValue);
    }

    auto getSlpLSStart() const -> int {
      return p->getIntControl(XPRS_SLPLSSTART);
    }

    void setSlpLSStart(int newValue) {
      p->setIntControl(XPRS_SLPLSSTART, newValue);
    }

    auto getSlpPenaltyInfoStart() const -> int {
      return p->getIntControl(XPRS_SLPPENALTYINFOSTART);
    }

    void setSlpPenaltyInfoStart(int newValue) {
      p->setIntControl(XPRS_SLPPENALTYINFOSTART, newValue);
    }

    auto getSlpFilter() const -> int {
      return p->getIntControl(XPRS_SLPFILTER);
    }

    void setSlpFilter(int newValue) {
      p->setIntControl(XPRS_SLPFILTER, newValue);
    }

    auto getSlpTraceMaskOps() const -> int {
      return p->getIntControl(XPRS_SLPTRACEMASKOPS);
    }

    void setSlpTraceMaskOps(int newValue) {
      p->setIntControl(XPRS_SLPTRACEMASKOPS, newValue);
    }

    auto getSlpLSZeroLimit() const -> int {
      return p->getIntControl(XPRS_SLPLSZEROLIMIT);
    }

    void setSlpLSZeroLimit(int newValue) {
      p->setIntControl(XPRS_SLPLSZEROLIMIT, newValue);
    }

    auto getSlpHeurStrategy() const -> int {
      return p->getIntControl(XPRS_SLPHEURSTRATEGY);
    }

    void setSlpHeurStrategy(int newValue) {
      p->setIntControl(XPRS_SLPHEURSTRATEGY, newValue);
    }

    auto getSlpBarCrossoverStart() const -> int {
      return p->getIntControl(XPRS_SLPBARCROSSOVERSTART);
    }

    void setSlpBarCrossoverStart(int newValue) {
      p->setIntControl(XPRS_SLPBARCROSSOVERSTART, newValue);
    }

    auto getSlpBarStallingLimit() const -> int {
      return p->getIntControl(XPRS_SLPBARSTALLINGLIMIT);
    }

    void setSlpBarStallingLimit(int newValue) {
      p->setIntControl(XPRS_SLPBARSTALLINGLIMIT, newValue);
    }

    auto getSlpBarStallingObjLimit() const -> int {
      return p->getIntControl(XPRS_SLPBARSTALLINGOBJLIMIT);
    }

    void setSlpBarStallingObjLimit(int newValue) {
      p->setIntControl(XPRS_SLPBARSTALLINGOBJLIMIT, newValue);
    }

    auto getSlpBarStartOps() const -> int {
      return p->getIntControl(XPRS_SLPBARSTARTOPS);
    }

    void setSlpBarStartOps(int newValue) {
      p->setIntControl(XPRS_SLPBARSTARTOPS, newValue);
    }

    auto getSlpGridHeurSelect() const -> int {
      return p->getIntControl(XPRS_SLPGRIDHEURSELECT);
    }

    void setSlpGridHeurSelect(int newValue) {
      p->setIntControl(XPRS_SLPGRIDHEURSELECT, newValue);
    }

    auto getNlpInfinity() const -> double {
      return p->getDblControl(XPRS_NLPINFINITY);
    }

    void setNlpInfinity(double newValue) {
      p->setDblControl(XPRS_NLPINFINITY, newValue);
    }

    auto getNlpZero() const -> double { return p->getDblControl(XPRS_NLPZERO); }

    void setNlpZero(double newValue) {
      p->setDblControl(XPRS_NLPZERO, newValue);
    }

    auto getNlpDefaultIV() const -> double {
      return p->getDblControl(XPRS_NLPDEFAULTIV);
    }

    void setNlpDefaultIV(double newValue) {
      p->setDblControl(XPRS_NLPDEFAULTIV, newValue);
    }

    auto getNlpValidationTol_A() const -> double {
      return p->getDblControl(XPRS_NLPVALIDATIONTOL_A);
    }

    void setNlpValidationTol_A(double newValue) {
      p->setDblControl(XPRS_NLPVALIDATIONTOL_A, newValue);
    }

    auto getNlpValidationTol_R() const -> double {
      return p->getDblControl(XPRS_NLPVALIDATIONTOL_R);
    }

    void setNlpValidationTol_R(double newValue) {
      p->setDblControl(XPRS_NLPVALIDATIONTOL_R, newValue);
    }

    auto getNlpPrimalIntegralRef() const -> double {
      return p->getDblControl(XPRS_NLPPRIMALINTEGRALREF);
    }

    void setNlpPrimalIntegralRef(double newValue) {
      p->setDblControl(XPRS_NLPPRIMALINTEGRALREF, newValue);
    }

    auto getNlpPrimalIntegralAlpha() const -> double {
      return p->getDblControl(XPRS_NLPPRIMALINTEGRALALPHA);
    }

    void setNlpPrimalIntegralAlpha(double newValue) {
      p->setDblControl(XPRS_NLPPRIMALINTEGRALALPHA, newValue);
    }

    auto getNlpPresolveZero() const -> double {
      return p->getDblControl(XPRS_NLPPRESOLVEZERO);
    }

    void setNlpPresolveZero(double newValue) {
      p->setDblControl(XPRS_NLPPRESOLVEZERO, newValue);
    }

    auto getNlpMeritLambda() const -> double {
      return p->getDblControl(XPRS_NLPMERITLAMBDA);
    }

    void setNlpMeritLambda(double newValue) {
      p->setDblControl(XPRS_NLPMERITLAMBDA, newValue);
    }

    auto getMsMaxBoundRange() const -> double {
      return p->getDblControl(XPRS_MSMAXBOUNDRANGE);
    }

    void setMsMaxBoundRange(double newValue) {
      p->setDblControl(XPRS_MSMAXBOUNDRANGE, newValue);
    }

    auto getNlpValidationTol_K() const -> double {
      return p->getDblControl(XPRS_NLPVALIDATIONTOL_K);
    }

    void setNlpValidationTol_K(double newValue) {
      p->setDblControl(XPRS_NLPVALIDATIONTOL_K, newValue);
    }

    auto getNlpPresolve_ElimTol() const -> double {
      return p->getDblControl(XPRS_NLPPRESOLVE_ELIMTOL);
    }

    void setNlpPresolve_ElimTol(double newValue) {
      p->setDblControl(XPRS_NLPPRESOLVE_ELIMTOL, newValue);
    }

    auto getNlpValidationTarget_R() const -> double {
      return p->getDblControl(XPRS_NLPVALIDATIONTARGET_R);
    }

    void setNlpValidationTarget_R(double newValue) {
      p->setDblControl(XPRS_NLPVALIDATIONTARGET_R, newValue);
    }

    auto getNlpValidationTarget_K() const -> double {
      return p->getDblControl(XPRS_NLPVALIDATIONTARGET_K);
    }

    void setNlpValidationTarget_K(double newValue) {
      p->setDblControl(XPRS_NLPVALIDATIONTARGET_K, newValue);
    }

    auto getNlpValidationFactor() const -> double {
      return p->getDblControl(XPRS_NLPVALIDATIONFACTOR);
    }

    void setNlpValidationFactor(double newValue) {
      p->setDblControl(XPRS_NLPVALIDATIONFACTOR, newValue);
    }

    auto getNlpRelTolBoundThreshold() const -> double {
      return p->getDblControl(XPRS_NLPRELTOLBOUNDTHRESHOLD);
    }

    void setNlpRelTolBoundThreshold(double newValue) {
      p->setDblControl(XPRS_NLPRELTOLBOUNDTHRESHOLD, newValue);
    }

    auto getSlpDamp() const -> double { return p->getDblControl(XPRS_SLPDAMP); }

    void setSlpDamp(double newValue) {
      p->setDblControl(XPRS_SLPDAMP, newValue);
    }

    auto getSlpDampExpand() const -> double {
      return p->getDblControl(XPRS_SLPDAMPEXPAND);
    }

    void setSlpDampExpand(double newValue) {
      p->setDblControl(XPRS_SLPDAMPEXPAND, newValue);
    }

    auto getSlpDampShrink() const -> double {
      return p->getDblControl(XPRS_SLPDAMPSHRINK);
    }

    void setSlpDampShrink(double newValue) {
      p->setDblControl(XPRS_SLPDAMPSHRINK, newValue);
    }

    auto getSlpDelta_A() const -> double {
      return p->getDblControl(XPRS_SLPDELTA_A);
    }

    void setSlpDelta_A(double newValue) {
      p->setDblControl(XPRS_SLPDELTA_A, newValue);
    }

    auto getSlpDelta_R() const -> double {
      return p->getDblControl(XPRS_SLPDELTA_R);
    }

    void setSlpDelta_R(double newValue) {
      p->setDblControl(XPRS_SLPDELTA_R, newValue);
    }

    auto getSlpDelta_Z() const -> double {
      return p->getDblControl(XPRS_SLPDELTA_Z);
    }

    void setSlpDelta_Z(double newValue) {
      p->setDblControl(XPRS_SLPDELTA_Z, newValue);
    }

    auto getSlpDeltaCost() const -> double {
      return p->getDblControl(XPRS_SLPDELTACOST);
    }

    void setSlpDeltaCost(double newValue) {
      p->setDblControl(XPRS_SLPDELTACOST, newValue);
    }

    auto getSlpDeltaMaxCost() const -> double {
      return p->getDblControl(XPRS_SLPDELTAMAXCOST);
    }

    void setSlpDeltaMaxCost(double newValue) {
      p->setDblControl(XPRS_SLPDELTAMAXCOST, newValue);
    }

    auto getSlpDJTol() const -> double {
      return p->getDblControl(XPRS_SLPDJTOL);
    }

    void setSlpDJTol(double newValue) {
      p->setDblControl(XPRS_SLPDJTOL, newValue);
    }

    auto getSlpErrorCost() const -> double {
      return p->getDblControl(XPRS_SLPERRORCOST);
    }

    void setSlpErrorCost(double newValue) {
      p->setDblControl(XPRS_SLPERRORCOST, newValue);
    }

    auto getSlpErrorMaxCost() const -> double {
      return p->getDblControl(XPRS_SLPERRORMAXCOST);
    }

    void setSlpErrorMaxCost(double newValue) {
      p->setDblControl(XPRS_SLPERRORMAXCOST, newValue);
    }

    auto getSlpErrorTol_A() const -> double {
      return p->getDblControl(XPRS_SLPERRORTOL_A);
    }

    void setSlpErrorTol_A(double newValue) {
      p->setDblControl(XPRS_SLPERRORTOL_A, newValue);
    }

    auto getSlpExpand() const -> double {
      return p->getDblControl(XPRS_SLPEXPAND);
    }

    void setSlpExpand(double newValue) {
      p->setDblControl(XPRS_SLPEXPAND, newValue);
    }

    auto getSlpMaxWeight() const -> double {
      return p->getDblControl(XPRS_SLPMAXWEIGHT);
    }

    void setSlpMaxWeight(double newValue) {
      p->setDblControl(XPRS_SLPMAXWEIGHT, newValue);
    }

    auto getSlpMinWeight() const -> double {
      return p->getDblControl(XPRS_SLPMINWEIGHT);
    }

    void setSlpMinWeight(double newValue) {
      p->setDblControl(XPRS_SLPMINWEIGHT, newValue);
    }

    auto getSlpShrink() const -> double {
      return p->getDblControl(XPRS_SLPSHRINK);
    }

    void setSlpShrink(double newValue) {
      p->setDblControl(XPRS_SLPSHRINK, newValue);
    }

    auto getSlpCTol() const -> double { return p->getDblControl(XPRS_SLPCTOL); }

    void setSlpCTol(double newValue) {
      p->setDblControl(XPRS_SLPCTOL, newValue);
    }

    auto getSlpATol_A() const -> double {
      return p->getDblControl(XPRS_SLPATOL_A);
    }

    void setSlpATol_A(double newValue) {
      p->setDblControl(XPRS_SLPATOL_A, newValue);
    }

    auto getSlpATol_R() const -> double {
      return p->getDblControl(XPRS_SLPATOL_R);
    }

    void setSlpATol_R(double newValue) {
      p->setDblControl(XPRS_SLPATOL_R, newValue);
    }

    auto getSlpMTol_A() const -> double {
      return p->getDblControl(XPRS_SLPMTOL_A);
    }

    void setSlpMTol_A(double newValue) {
      p->setDblControl(XPRS_SLPMTOL_A, newValue);
    }

    auto getSlpMTol_R() const -> double {
      return p->getDblControl(XPRS_SLPMTOL_R);
    }

    void setSlpMTol_R(double newValue) {
      p->setDblControl(XPRS_SLPMTOL_R, newValue);
    }

    auto getSlpItol_A() const -> double {
      return p->getDblControl(XPRS_SLPITOL_A);
    }

    void setSlpItol_A(double newValue) {
      p->setDblControl(XPRS_SLPITOL_A, newValue);
    }

    auto getSlpITol_R() const -> double {
      return p->getDblControl(XPRS_SLPITOL_R);
    }

    void setSlpITol_R(double newValue) {
      p->setDblControl(XPRS_SLPITOL_R, newValue);
    }

    auto getSlpSTol_A() const -> double {
      return p->getDblControl(XPRS_SLPSTOL_A);
    }

    void setSlpSTol_A(double newValue) {
      p->setDblControl(XPRS_SLPSTOL_A, newValue);
    }

    auto getSlpSTol_R() const -> double {
      return p->getDblControl(XPRS_SLPSTOL_R);
    }

    void setSlpSTol_R(double newValue) {
      p->setDblControl(XPRS_SLPSTOL_R, newValue);
    }

    auto getSlpMVTol() const -> double {
      return p->getDblControl(XPRS_SLPMVTOL);
    }

    void setSlpMVTol(double newValue) {
      p->setDblControl(XPRS_SLPMVTOL, newValue);
    }

    auto getSlpXTol_A() const -> double {
      return p->getDblControl(XPRS_SLPXTOL_A);
    }

    void setSlpXTol_A(double newValue) {
      p->setDblControl(XPRS_SLPXTOL_A, newValue);
    }

    auto getSlpXTol_R() const -> double {
      return p->getDblControl(XPRS_SLPXTOL_R);
    }

    void setSlpXTol_R(double newValue) {
      p->setDblControl(XPRS_SLPXTOL_R, newValue);
    }

    auto getSlpDefaultStepBound() const -> double {
      return p->getDblControl(XPRS_SLPDEFAULTSTEPBOUND);
    }

    void setSlpDefaultStepBound(double newValue) {
      p->setDblControl(XPRS_SLPDEFAULTSTEPBOUND, newValue);
    }

    auto getSlpDampMax() const -> double {
      return p->getDblControl(XPRS_SLPDAMPMAX);
    }

    void setSlpDampMax(double newValue) {
      p->setDblControl(XPRS_SLPDAMPMAX, newValue);
    }

    auto getSlpDampMin() const -> double {
      return p->getDblControl(XPRS_SLPDAMPMIN);
    }

    void setSlpDampMin(double newValue) {
      p->setDblControl(XPRS_SLPDAMPMIN, newValue);
    }

    auto getSlpDeltaCostFactor() const -> double {
      return p->getDblControl(XPRS_SLPDELTACOSTFACTOR);
    }

    void setSlpDeltaCostFactor(double newValue) {
      p->setDblControl(XPRS_SLPDELTACOSTFACTOR, newValue);
    }

    auto getSlpErrorCostFactor() const -> double {
      return p->getDblControl(XPRS_SLPERRORCOSTFACTOR);
    }

    void setSlpErrorCostFactor(double newValue) {
      p->setDblControl(XPRS_SLPERRORCOSTFACTOR, newValue);
    }

    auto getSlpErrorTol_P() const -> double {
      return p->getDblControl(XPRS_SLPERRORTOL_P);
    }

    void setSlpErrorTol_P(double newValue) {
      p->setDblControl(XPRS_SLPERRORTOL_P, newValue);
    }

    auto getSlpCascadeTol_PA() const -> double {
      return p->getDblControl(XPRS_SLPCASCADETOL_PA);
    }

    void setSlpCascadeTol_PA(double newValue) {
      p->setDblControl(XPRS_SLPCASCADETOL_PA, newValue);
    }

    auto getSlpCascadeTol_PR() const -> double {
      return p->getDblControl(XPRS_SLPCASCADETOL_PR);
    }

    void setSlpCascadeTol_PR(double newValue) {
      p->setDblControl(XPRS_SLPCASCADETOL_PR, newValue);
    }

    auto getSlpOTol_A() const -> double {
      return p->getDblControl(XPRS_SLPOTOL_A);
    }

    void setSlpOTol_A(double newValue) {
      p->setDblControl(XPRS_SLPOTOL_A, newValue);
    }

    auto getSlpOTol_R() const -> double {
      return p->getDblControl(XPRS_SLPOTOL_R);
    }

    void setSlpOTol_R(double newValue) {
      p->setDblControl(XPRS_SLPOTOL_R, newValue);
    }

    auto getSlpDelta_X() const -> double {
      return p->getDblControl(XPRS_SLPDELTA_X);
    }

    void setSlpDelta_X(double newValue) {
      p->setDblControl(XPRS_SLPDELTA_X, newValue);
    }

    auto getSlpGranularity() const -> double {
      return p->getDblControl(XPRS_SLPGRANULARITY);
    }

    void setSlpGranularity(double newValue) {
      p->setDblControl(XPRS_SLPGRANULARITY, newValue);
    }

    auto getSlpMipCutoff_A() const -> double {
      return p->getDblControl(XPRS_SLPMIPCUTOFF_A);
    }

    void setSlpMipCutoff_A(double newValue) {
      p->setDblControl(XPRS_SLPMIPCUTOFF_A, newValue);
    }

    auto getSlpMipCutoff_R() const -> double {
      return p->getDblControl(XPRS_SLPMIPCUTOFF_R);
    }

    void setSlpMipCutoff_R(double newValue) {
      p->setDblControl(XPRS_SLPMIPCUTOFF_R, newValue);
    }

    auto getSlpMipOtol_A() const -> double {
      return p->getDblControl(XPRS_SLPMIPOTOL_A);
    }

    void setSlpMipOtol_A(double newValue) {
      p->setDblControl(XPRS_SLPMIPOTOL_A, newValue);
    }

    auto getSlpMipOtol_R() const -> double {
      return p->getDblControl(XPRS_SLPMIPOTOL_R);
    }

    void setSlpMipOtol_R(double newValue) {
      p->setDblControl(XPRS_SLPMIPOTOL_R, newValue);
    }

    auto getSlpEscalation() const -> double {
      return p->getDblControl(XPRS_SLPESCALATION);
    }

    void setSlpEscalation(double newValue) {
      p->setDblControl(XPRS_SLPESCALATION, newValue);
    }

    auto getSlpObjToPenaltyCost() const -> double {
      return p->getDblControl(XPRS_SLPOBJTOPENALTYCOST);
    }

    void setSlpObjToPenaltyCost(double newValue) {
      p->setDblControl(XPRS_SLPOBJTOPENALTYCOST, newValue);
    }

    auto getSlpShrinkBias() const -> double {
      return p->getDblControl(XPRS_SLPSHRINKBIAS);
    }

    void setSlpShrinkBias(double newValue) {
      p->setDblControl(XPRS_SLPSHRINKBIAS, newValue);
    }

    auto getSlpFeastolTarget() const -> double {
      return p->getDblControl(XPRS_SLPFEASTOLTARGET);
    }

    void setSlpFeastolTarget(double newValue) {
      p->setDblControl(XPRS_SLPFEASTOLTARGET, newValue);
    }

    auto getSlpOptimalityTolTarget() const -> double {
      return p->getDblControl(XPRS_SLPOPTIMALITYTOLTARGET);
    }

    void setSlpOptimalityTolTarget(double newValue) {
      p->setDblControl(XPRS_SLPOPTIMALITYTOLTARGET, newValue);
    }

    auto getSlpDelta_Infinity() const -> double {
      return p->getDblControl(XPRS_SLPDELTA_INFINITY);
    }

    void setSlpDelta_Infinity(double newValue) {
      p->setDblControl(XPRS_SLPDELTA_INFINITY, newValue);
    }

    auto getSlpVTol_A() const -> double {
      return p->getDblControl(XPRS_SLPVTOL_A);
    }

    void setSlpVTol_A(double newValue) {
      p->setDblControl(XPRS_SLPVTOL_A, newValue);
    }

    auto getSlpVTol_R() const -> double {
      return p->getDblControl(XPRS_SLPVTOL_R);
    }

    void setSlpVTol_R(double newValue) {
      p->setDblControl(XPRS_SLPVTOL_R, newValue);
    }

    auto getSlpETol_A() const -> double {
      return p->getDblControl(XPRS_SLPETOL_A);
    }

    void setSlpETol_A(double newValue) {
      p->setDblControl(XPRS_SLPETOL_A, newValue);
    }

    auto getSlpETol_R() const -> double {
      return p->getDblControl(XPRS_SLPETOL_R);
    }

    void setSlpETol_R(double newValue) {
      p->setDblControl(XPRS_SLPETOL_R, newValue);
    }

    auto getSlpEVTol_A() const -> double {
      return p->getDblControl(XPRS_SLPEVTOL_A);
    }

    void setSlpEVTol_A(double newValue) {
      p->setDblControl(XPRS_SLPEVTOL_A, newValue);
    }

    auto getSlpEVTol_R() const -> double {
      return p->getDblControl(XPRS_SLPEVTOL_R);
    }

    void setSlpEVTol_R(double newValue) {
      p->setDblControl(XPRS_SLPEVTOL_R, newValue);
    }

    auto getSlpDelta_Zero() const -> double {
      return p->getDblControl(XPRS_SLPDELTA_ZERO);
    }

    void setSlpDelta_Zero(double newValue) {
      p->setDblControl(XPRS_SLPDELTA_ZERO, newValue);
    }

    auto getSlpMinSBFactor() const -> double {
      return p->getDblControl(XPRS_SLPMINSBFACTOR);
    }

    void setSlpMinSBFactor(double newValue) {
      p->setDblControl(XPRS_SLPMINSBFACTOR, newValue);
    }

    auto getSlpClampValidationTol_A() const -> double {
      return p->getDblControl(XPRS_SLPCLAMPVALIDATIONTOL_A);
    }

    void setSlpClampValidationTol_A(double newValue) {
      p->setDblControl(XPRS_SLPCLAMPVALIDATIONTOL_A, newValue);
    }

    auto getSlpClampValidationTol_R() const -> double {
      return p->getDblControl(XPRS_SLPCLAMPVALIDATIONTOL_R);
    }

    void setSlpClampValidationTol_R(double newValue) {
      p->setDblControl(XPRS_SLPCLAMPVALIDATIONTOL_R, newValue);
    }

    auto getSlpClampShrink() const -> double {
      return p->getDblControl(XPRS_SLPCLAMPSHRINK);
    }

    void setSlpClampShrink(double newValue) {
      p->setDblControl(XPRS_SLPCLAMPSHRINK, newValue);
    }

    auto getSlpEcfTol_A() const -> double {
      return p->getDblControl(XPRS_SLPECFTOL_A);
    }

    void setSlpEcfTol_A(double newValue) {
      p->setDblControl(XPRS_SLPECFTOL_A, newValue);
    }

    auto getSlpEcfTol_R() const -> double {
      return p->getDblControl(XPRS_SLPECFTOL_R);
    }

    void setSlpEcfTol_R(double newValue) {
      p->setDblControl(XPRS_SLPECFTOL_R, newValue);
    }

    auto getSlpWTol_A() const -> double {
      return p->getDblControl(XPRS_SLPWTOL_A);
    }

    void setSlpWTol_A(double newValue) {
      p->setDblControl(XPRS_SLPWTOL_A, newValue);
    }

    auto getSlpWTol_R() const -> double {
      return p->getDblControl(XPRS_SLPWTOL_R);
    }

    void setSlpWTol_R(double newValue) {
      p->setDblControl(XPRS_SLPWTOL_R, newValue);
    }

    auto getSlpMatrixTol() const -> double {
      return p->getDblControl(XPRS_SLPMATRIXTOL);
    }

    void setSlpMatrixTol(double newValue) {
      p->setDblControl(XPRS_SLPMATRIXTOL, newValue);
    }

    auto getSlpDRFixRange() const -> double {
      return p->getDblControl(XPRS_SLPDRFIXRANGE);
    }

    void setSlpDRFixRange(double newValue) {
      p->setDblControl(XPRS_SLPDRFIXRANGE, newValue);
    }

    auto getSlpDRColTol() const -> double {
      return p->getDblControl(XPRS_SLPDRCOLTOL);
    }

    void setSlpDRColTol(double newValue) {
      p->setDblControl(XPRS_SLPDRCOLTOL, newValue);
    }

    auto getSlpMipErrorTol_A() const -> double {
      return p->getDblControl(XPRS_SLPMIPERRORTOL_A);
    }

    void setSlpMipErrorTol_A(double newValue) {
      p->setDblControl(XPRS_SLPMIPERRORTOL_A, newValue);
    }

    auto getSlpMipErrorTol_R() const -> double {
      return p->getDblControl(XPRS_SLPMIPERRORTOL_R);
    }

    void setSlpMipErrorTol_R(double newValue) {
      p->setDblControl(XPRS_SLPMIPERRORTOL_R, newValue);
    }

    auto getSlpCDTol_A() const -> double {
      return p->getDblControl(XPRS_SLPCDTOL_A);
    }

    void setSlpCDTol_A(double newValue) {
      p->setDblControl(XPRS_SLPCDTOL_A, newValue);
    }

    auto getSlpCDTol_R() const -> double {
      return p->getDblControl(XPRS_SLPCDTOL_R);
    }

    void setSlpCDTol_R(double newValue) {
      p->setDblControl(XPRS_SLPCDTOL_R, newValue);
    }

    auto getSlpEnforceMaxCost() const -> double {
      return p->getDblControl(XPRS_SLPENFORCEMAXCOST);
    }

    void setSlpEnforceMaxCost(double newValue) {
      p->setDblControl(XPRS_SLPENFORCEMAXCOST, newValue);
    }

    auto getSlpEnforceCostShrink() const -> double {
      return p->getDblControl(XPRS_SLPENFORCECOSTSHRINK);
    }

    void setSlpEnforceCostShrink(double newValue) {
      p->setDblControl(XPRS_SLPENFORCECOSTSHRINK, newValue);
    }

    auto getSlpDRColDjTol() const -> double {
      return p->getDblControl(XPRS_SLPDRCOLDJTOL);
    }

    void setSlpDRColDjTol(double newValue) {
      p->setDblControl(XPRS_SLPDRCOLDJTOL, newValue);
    }

    auto getSlpBarStallingTol() const -> double {
      return p->getDblControl(XPRS_SLPBARSTALLINGTOL);
    }

    void setSlpBarStallingTol(double newValue) {
      p->setDblControl(XPRS_SLPBARSTALLINGTOL, newValue);
    }

    auto getSlpObjThreshold() const -> double {
      return p->getDblControl(XPRS_SLPOBJTHRESHOLD);
    }

    void setSlpObjThreshold(double newValue) {
      p->setDblControl(XPRS_SLPOBJTHRESHOLD, newValue);
    }

    auto getSlpBoundThreshold() const -> double {
      return p->getDblControl(XPRS_SLPBOUNDTHRESHOLD);
    }

    void setSlpBoundThreshold(double newValue) {
      p->setDblControl(XPRS_SLPBOUNDTHRESHOLD, newValue);
    }

    auto getKnitroParamNewPoint() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_NEWPOINT);
    }

    void setKnitroParamNewPoint(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_NEWPOINT, newValue);
    }

    auto getKnitroParamHonorBbnds() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_HONORBNDS);
    }

    void setKnitroParamHonorBbnds(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_HONORBNDS, newValue);
    }

    auto getKnitroParamAlgorithm() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_ALGORITHM);
    }

    void setKnitroParamAlgorithm(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_ALGORITHM, newValue);
    }

    auto getKnitroParamBarMuRule() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_MURULE);
    }

    void setKnitroParamBarMuRule(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_MURULE, newValue);
    }

    auto getKnitroParamBarFeasible() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_FEASIBLE);
    }

    void setKnitroParamBarFeasible(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_FEASIBLE, newValue);
    }

    auto getKnitroParamGradOpt() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_GRADOPT);
    }

    void setKnitroParamGradOpt(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_GRADOPT, newValue);
    }

    auto getKnitroParamHessOpt() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_HESSOPT);
    }

    void setKnitroParamHessOpt(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_HESSOPT, newValue);
    }

    auto getKnitroParamBarInitPt() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_INITPT);
    }

    void setKnitroParamBarInitPt(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_INITPT, newValue);
    }

    auto getKnitroParamMaxCGIt() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MAXCGIT);
    }

    void setKnitroParamMaxCGIt(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MAXCGIT, newValue);
    }

    auto getKnitroParamMaxIt() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MAXIT);
    }

    void setKnitroParamMaxIt(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MAXIT, newValue);
    }

    auto getKnitroParamOutLev() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_OUTLEV);
    }

    void setKnitroParamOutLev(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_OUTLEV, newValue);
    }

    auto getKnitroParamScale() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_SCALE);
    }

    void setKnitroParamScale(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_SCALE, newValue);
    }

    auto getKnitroParamSOC() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_SOC);
    }

    void setKnitroParamSOC(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_SOC, newValue);
    }

    auto getKnitroParamDelta() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_DELTA);
    }

    void setKnitroParamDelta(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_DELTA, newValue);
    }

    auto getKnitroParamBarFeasModeTol() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_BAR_FEASMODETOL);
    }

    void setKnitroParamBarFeasModeTol(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_BAR_FEASMODETOL, newValue);
    }

    auto getKnitroParamFeastol() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_FEASTOL);
    }

    void setKnitroParamFeastol(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_FEASTOL, newValue);
    }

    auto getKnitroParamFeasTolAbs() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_FEASTOLABS);
    }

    void setKnitroParamFeasTolAbs(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_FEASTOLABS, newValue);
    }

    auto getKnitroParamBarInitMu() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_BAR_INITMU);
    }

    void setKnitroParamBarInitMu(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_BAR_INITMU, newValue);
    }

    auto getKnitroParamObjRange() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_OBJRANGE);
    }

    void setKnitroParamObjRange(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_OBJRANGE, newValue);
    }

    auto getKnitroParamOptTol() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_OPTTOL);
    }

    void setKnitroParamOptTol(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_OPTTOL, newValue);
    }

    auto getKnitroParamOptTolAbs() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_OPTTOLABS);
    }

    void setKnitroParamOptTolAbs(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_OPTTOLABS, newValue);
    }

    auto getKnitroParamPivot() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_PIVOT);
    }

    void setKnitroParamPivot(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_PIVOT, newValue);
    }

    auto getKnitroParamXTol() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_XTOL);
    }

    void setKnitroParamXTol(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_XTOL, newValue);
    }

    auto getKnitroParamDebug() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_DEBUG);
    }

    void setKnitroParamDebug(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_DEBUG, newValue);
    }

    auto getKnitroParamMultiStart() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MULTISTART);
    }

    void setKnitroParamMultiStart(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MULTISTART, newValue);
    }

    auto getKnitroParamMSMaxSolves() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MSMAXSOLVES);
    }

    void setKnitroParamMSMaxSolves(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MSMAXSOLVES, newValue);
    }

    auto getKnitroParamMsMaxBndRange() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_MSMAXBNDRANGE);
    }

    void setKnitroParamMsMaxBndRange(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_MSMAXBNDRANGE, newValue);
    }

    auto getKnitroParamLMSize() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_LMSIZE);
    }

    void setKnitroParamLMSize(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_LMSIZE, newValue);
    }

    auto getKnitroParamMaxCrossIt() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_MAXCROSSIT);
    }

    void setKnitroParamMaxCrossIt(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_MAXCROSSIT, newValue);
    }

    auto getKnitroParamBLASOption() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BLASOPTION);
    }

    void setKnitroParamBLASOption(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BLASOPTION, newValue);
    }

    auto getKnitroParamBarMaxRefactor() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_MAXREFACTOR);
    }

    void setKnitroParamBarMaxRefactor(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_MAXREFACTOR, newValue);
    }

    auto getKnitroParamBarMaxBacktrack() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_MAXBACKTRACK);
    }

    void setKnitroParamBarMaxBacktrack(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_MAXBACKTRACK, newValue);
    }

    auto getKnitroParamBarPenRule() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_PENRULE);
    }

    void setKnitroParamBarPenRule(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_PENRULE, newValue);
    }

    auto getKnitroParamBarPenCons() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_PENCONS);
    }

    void setKnitroParamBarPenCons(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_PENCONS, newValue);
    }

    auto getKnitroParamMSNumToSave() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MSNUMTOSAVE);
    }

    void setKnitroParamMSNumToSave(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MSNUMTOSAVE, newValue);
    }

    auto getKnitroParamMSSaveTol() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_MSSAVETOL);
    }

    void setKnitroParamMSSaveTol(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_MSSAVETOL, newValue);
    }

    auto getKnitroParamMSTerminate() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MSTERMINATE);
    }

    void setKnitroParamMSTerminate(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MSTERMINATE, newValue);
    }

    auto getKnitroParamMSStartPtRange() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_MSSTARTPTRANGE);
    }

    void setKnitroParamMSStartPtRange(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_MSSTARTPTRANGE, newValue);
    }

    auto getKnitroParamInfeasTol() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_INFEASTOL);
    }

    void setKnitroParamInfeasTol(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_INFEASTOL, newValue);
    }

    auto getKnitroParamLinSolver() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_LINSOLVER);
    }

    void setKnitroParamLinSolver(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_LINSOLVER, newValue);
    }

    auto getKnitroParamBarDirectInterval() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_DIRECTINTERVAL);
    }

    void setKnitroParamBarDirectInterval(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_DIRECTINTERVAL, newValue);
    }

    auto getKnitroParamPresolve() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_PRESOLVE);
    }

    void setKnitroParamPresolve(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_PRESOLVE, newValue);
    }

    auto getKnitroParamPresolveTol() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_PRESOLVE_TOL);
    }

    void setKnitroParamPresolveTol(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_PRESOLVE_TOL, newValue);
    }

    auto getKnitroParamBarSwitchRule() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_SWITCHRULE);
    }

    void setKnitroParamBarSwitchRule(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_SWITCHRULE, newValue);
    }

    auto getKnitroParamMATerminate() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MA_TERMINATE);
    }

    void setKnitroParamMATerminate(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MA_TERMINATE, newValue);
    }

    auto getKnitroParamMSSeed() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MSSEED);
    }

    void setKnitroParamMSSeed(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MSSEED, newValue);
    }

    auto getKnitroParamBarRelaxCons() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_BAR_RELAXCONS);
    }

    void setKnitroParamBarRelaxCons(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_BAR_RELAXCONS, newValue);
    }

    auto getKnitroParamMipMethod() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_METHOD);
    }

    void setKnitroParamMipMethod(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_METHOD, newValue);
    }

    auto getKnitroParamMipBranchRule() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_BRANCHRULE);
    }

    void setKnitroParamMipBranchRule(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_BRANCHRULE, newValue);
    }

    auto getKnitroParamMipSelectRule() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_SELECTRULE);
    }

    void setKnitroParamMipSelectRule(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_SELECTRULE, newValue);
    }

    auto getKnitroParamMipIntGapAbs() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_MIP_INTGAPABS);
    }

    void setKnitroParamMipIntGapAbs(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_MIP_INTGAPABS, newValue);
    }

    auto getKnitroParamMipIntGapRel() const -> double {
      return p->getDblControl(XPRS_KNITRO_PARAM_MIP_INTGAPREL);
    }

    void setKnitroParamMipIntGapRel(double newValue) {
      p->setDblControl(XPRS_KNITRO_PARAM_MIP_INTGAPREL, newValue);
    }

    auto getKnitroParamMipOutLevel() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_OUTLEVEL);
    }

    void setKnitroParamMipOutLevel(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_OUTLEVEL, newValue);
    }

    auto getKnitroParamMipOutInterval() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_OUTINTERVAL);
    }

    void setKnitroParamMipOutInterval(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_OUTINTERVAL, newValue);
    }

    auto getKnitroParamMipDebug() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_DEBUG);
    }

    void setKnitroParamMipDebug(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_DEBUG, newValue);
    }

    auto getKnitroParamMipImplicatns() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_IMPLICATNS);
    }

    void setKnitroParamMipImplicatns(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_IMPLICATNS, newValue);
    }

    auto getKnitroParamMipGUBBranch() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_GUB_BRANCH);
    }

    void setKnitroParamMipGUBBranch(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_GUB_BRANCH, newValue);
    }

    auto getKnitroParamMipKnapsack() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_KNAPSACK);
    }

    void setKnitroParamMipKnapsack(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_KNAPSACK, newValue);
    }

    auto getKnitroParamMipRounding() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_ROUNDING);
    }

    void setKnitroParamMipRounding(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_ROUNDING, newValue);
    }

    auto getKnitroParamMipRootAlg() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_ROOTALG);
    }

    void setKnitroParamMipRootAlg(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_ROOTALG, newValue);
    }

    auto getKnitroParamMipLpAlg() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_LPALG);
    }

    void setKnitroParamMipLpAlg(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_LPALG, newValue);
    }

    auto getKnitroParamMipMaxNodes() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_MAXNODES);
    }

    void setKnitroParamMipMaxNodes(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_MAXNODES, newValue);
    }

    auto getKnitroParamMipHeuristic() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_HEURISTIC);
    }

    void setKnitroParamMipHeuristic(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_HEURISTIC, newValue);
    }

    auto getKnitroParamMipHeurMaxIt() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_HEUR_MAXIT);
    }

    void setKnitroParamMipHeurMaxIt(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_HEUR_MAXIT, newValue);
    }

    auto getKnitroParamMipPseudoint() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_PSEUDOINIT);
    }

    void setKnitroParamMipPseudoint(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_PSEUDOINIT, newValue);
    }

    auto getKnitroParamMipStringMaxIt() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_STRONG_MAXIT);
    }

    void setKnitroParamMipStringMaxIt(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_STRONG_MAXIT, newValue);
    }

    auto getKnitroParamMipStrongCandLim() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_STRONG_CANDLIM);
    }

    void setKnitroParamMipStrongCandLim(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_STRONG_CANDLIM, newValue);
    }

    auto getKnitroParamMipStrongLevel() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_MIP_STRONG_LEVEL);
    }

    void setKnitroParamMipStrongLevel(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_MIP_STRONG_LEVEL, newValue);
    }

    auto getKnitroParamParNumThreads() const -> int {
      return p->getIntControl(XPRS_KNITRO_PARAM_PAR_NUMTHREADS);
    }

    void setKnitroParamParNumThreads(int newValue) {
      p->setIntControl(XPRS_KNITRO_PARAM_PAR_NUMTHREADS, newValue);
    }
  };

  Controls controls;

public:
  class GeneralConstraintInfo final {
  public:
    std::vector<int> resultant;

    std::vector<xpress::GenConsType> type;

    std::vector<int> colstart;

    std::vector<int> colind;

    std::vector<int> valstart;

    std::vector<double> val;

    inline GeneralConstraintInfo();

    inline GeneralConstraintInfo(
        xpress::SizedArray<xpress::GenConsType const> const &type,
        xpress::SizedArray<int const> const &resultant,
        xpress::SizedArray<int const> const &colstart,
        xpress::SizedArray<int const> const &valstart, int ncols, int nvals);
  };

  inline auto getGenCons(int first, int last)
      -> xpress::XPRSProblem::GeneralConstraintInfo;

public:
  inline auto delIndicator(int row) -> void;

  inline auto getIndicators(int first, int last)
      -> std::vector<xpress::IndicatorInfo>;

  inline auto getIndicator(int row) -> std::optional<xpress::IndicatorInfo>;

  inline auto setIndicator(int rowind, int colind, int complement) -> void;

public:
  class InvalidStateException : public std::runtime_error {
  public:
    inline InvalidStateException(char const *message);
  };

  class CannotReferenceVariablesException : public InvalidStateException {
  public:
    inline CannotReferenceVariablesException();
  };

  class CannotPerformOperationException : public InvalidStateException {
  private:
    char msgBuffer[128];

    static inline auto mkMsg(char *buffer, char const *op, char const *cls)
        -> char const *;

    static inline auto mkMsg(char *buffer, char const *msg) -> char const *;

  public:
    inline CannotPerformOperationException(char const *op, char const *cls);

    inline CannotPerformOperationException(char const *msg);
  };

public:
  class MIPEntityInfo final {
  public:
    std::vector<char> coltype;

    std::vector<int> colind;

    std::vector<double> limit;

    std::vector<char> settype;

    std::vector<int> setstart;

    std::vector<int> setind;

    std::vector<double> refval;

    inline MIPEntityInfo();

    inline MIPEntityInfo(int ngents, int nsets, int setmembers);
  };

  inline auto getMipEntities() -> xpress::XPRSProblem::MIPEntityInfo;

  inline auto getDiscreteCols() -> xpress::XPRSProblem::MIPEntityInfo;

  inline auto getSetDefinitions() -> xpress::XPRSProblem::MIPEntityInfo;

public:
  inline auto
  addSets(int count, xpress::SizedArray<int const> const &start,
          xpress::SizedArray<SetType const> const &type,
          xpress::SizedArray<int const> const &setind,
          xpress::SizedArray<double const> const &setref,
          xpress::SizedArray<std::optional<std::string> const> const &names)
      -> std::vector<int>;

  inline auto addSet(xpress::SetType type,
                     xpress::SizedArray<int const> const &elements,
                     xpress::SizedArray<double const> const &weights,
                     std::optional<std::string> name) -> int;

  inline auto
  addSets(std::vector<xpress::SetType> type,
          xpress::SizedArray<std::vector<int>> const &elements,
          xpress::SizedArray<std::vector<double>> const &weights,
          xpress::SizedArray<std::optional<std::string> const> const &name)
      -> std::vector<int>;

public:
  template <typename C, xpress_isintegral(C)>
  inline auto
  buildColumns(VariableBuilder::ArrayBuilder<C> builder,
               std::function<void(std::vector<int> const &)> transformResult)
      -> std::vector<int>;

  template <typename C, xpress_isintegral(C)>
  inline auto buildColumns(VariableBuilder::ArrayBuilder<C> builder)
      -> std::vector<int>;

  template <typename I, typename C, xpress_isintegral(C), typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildColumns(VariableBuilder::ArrayBuilder<C> builder,
                           Func0 makeResult,
                           std::function<void(I &, C, int)> addResult) -> I;

  template <typename K1>
  inline auto buildColumns(VariableBuilder::MapBuilder<K1> builder,
                           std::function<void(K1, int)> addKey,
                           std::function<void()> makeResult) -> void;

  template <typename K1>
  inline auto buildColumns(VariableBuilder::MapBuilder<K1> builder)
      -> std::unordered_map<K1, int>;

  template <typename I, typename K1, typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildColumns(VariableBuilder::MapBuilder<K1> builder,
                           Func0 makeResult,
                           std::function<void(I &, K1, int)> addResult) -> I;

  template <typename C, xpress_isintegral(C)>
  inline auto addColumns(C dim) -> VariableBuilder::ColumnArrayBuilder<C>;

  template <typename Iter0, typename K1 = typename xpress_elementtype(Iter0),
            xpress_isiterable(Iter0)>
  inline auto addColumns(Iter0 const &iterable1)
      -> VariableBuilder::ColumnMapBuilder<K1>;

  template <typename K1>
  inline auto addColumns(xpress::SizedArray<K1 const> const &arr1)
      -> VariableBuilder::ColumnMapBuilder<K1>;

  inline auto addColumn(double lb, double ub, ColumnType type,
                        std::optional<std::string> name) -> int;

public:
  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  inline auto
  buildColumns(VariableBuilder::Array2Builder<C1, C2> builder,
               std::function<void(std::vector<std::vector<int>> const &)>
                   transformResult) -> std::vector<std::vector<int>>;

  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  inline auto buildColumns(VariableBuilder::Array2Builder<C1, C2> builder)
      -> std::vector<std::vector<int>>;

  template <typename I, typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2), typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildColumns(VariableBuilder::Array2Builder<C1, C2> builder,
                           Func0 makeResult,
                           std::function<void(I &, C1, C2, int)> addResult)
      -> I;

  template <typename K1, typename K2>
  inline auto buildColumns(VariableBuilder::Map2Builder<K1, K2> builder,
                           std::function<void(K1, K2, int)> addKey,
                           std::function<void()> makeResult) -> void;

  template <typename K1, typename K2>
  inline auto buildColumns(VariableBuilder::Map2Builder<K1, K2> builder)
      -> xpress::maps::HashMap2<K1, K2, int>;

  template <typename I, typename K1, typename K2, typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildColumns(VariableBuilder::Map2Builder<K1, K2> builder,
                           Func0 makeResult,
                           std::function<void(I &, K1, K2, int)> addResult)
      -> I;

  template <typename C1, typename C2, xpress_isintegral(C1),
            xpress_isintegral(C2)>
  inline auto addColumns(C1 dim1, C2 dim2)
      -> VariableBuilder::ColumnArray2Builder<C1, C2>;

  template <typename Iter0, typename Iter1,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1)>
  inline auto addColumns(Iter0 const &iterable1, Iter1 const &iterable2)
      -> VariableBuilder::ColumnMap2Builder<K1, K2>;

  template <typename K1, typename K2>
  inline auto addColumns(xpress::SizedArray<K1 const> const &arr1,
                         xpress::SizedArray<K2 const> const &arr2)
      -> VariableBuilder::ColumnMap2Builder<K1, K2>;

public:
  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  inline auto buildColumns(
      VariableBuilder::Array3Builder<C1, C2, C3> builder,
      std::function<void(std::vector<std::vector<std::vector<int>>> const &)>
          transformResult) -> std::vector<std::vector<std::vector<int>>>;

  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  inline auto buildColumns(VariableBuilder::Array3Builder<C1, C2, C3> builder)
      -> std::vector<std::vector<std::vector<int>>>;

  template <typename I, typename C1, typename C2, typename C3,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            typename Func0, xpress_isinvocable(Func0, I, Func0)>
  inline auto buildColumns(VariableBuilder::Array3Builder<C1, C2, C3> builder,
                           Func0 makeResult,
                           std::function<void(I &, C1, C2, C3, int)> addResult)
      -> I;

  template <typename K1, typename K2, typename K3>
  inline auto buildColumns(VariableBuilder::Map3Builder<K1, K2, K3> builder,
                           std::function<void(K1, K2, K3, int)> addKey,
                           std::function<void()> makeResult) -> void;

  template <typename K1, typename K2, typename K3>
  inline auto buildColumns(VariableBuilder::Map3Builder<K1, K2, K3> builder)
      -> xpress::maps::HashMap3<K1, K2, K3, int>;

  template <typename I, typename K1, typename K2, typename K3, typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto buildColumns(VariableBuilder::Map3Builder<K1, K2, K3> builder,
                           Func0 makeResult,
                           std::function<void(I &, K1, K2, K3, int)> addResult)
      -> I;

  template <typename C1, typename C2, typename C3, xpress_isintegral(C1),
            xpress_isintegral(C2), xpress_isintegral(C3)>
  inline auto addColumns(C1 dim1, C2 dim2, C3 dim3)
      -> VariableBuilder::ColumnArray3Builder<C1, C2, C3>;

  template <typename Iter0, typename Iter1, typename Iter2,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2)>
  inline auto addColumns(Iter0 const &iterable1, Iter1 const &iterable2,
                         Iter2 const &iterable3)
      -> VariableBuilder::ColumnMap3Builder<K1, K2, K3>;

  template <typename K1, typename K2, typename K3>
  inline auto addColumns(xpress::SizedArray<K1 const> const &arr1,
                         xpress::SizedArray<K2 const> const &arr2,
                         xpress::SizedArray<K3 const> const &arr3)
      -> VariableBuilder::ColumnMap3Builder<K1, K2, K3>;

public:
  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  inline auto buildColumns(
      VariableBuilder::Array4Builder<C1, C2, C3, C4> builder,
      std::function<
          void(std::vector<std::vector<std::vector<std::vector<int>>>> const &)>
          transformResult)
      -> std::vector<std::vector<std::vector<std::vector<int>>>>;

  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  inline auto
  buildColumns(VariableBuilder::Array4Builder<C1, C2, C3, C4> builder)
      -> std::vector<std::vector<std::vector<std::vector<int>>>>;

  template <typename I, typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), typename Func0,
            xpress_isinvocable(Func0, I, Func0)>
  inline auto
  buildColumns(VariableBuilder::Array4Builder<C1, C2, C3, C4> builder,
               Func0 makeResult,
               std::function<void(I &, C1, C2, C3, C4, int)> addResult) -> I;

  template <typename K1, typename K2, typename K3, typename K4>
  inline auto buildColumns(VariableBuilder::Map4Builder<K1, K2, K3, K4> builder,
                           std::function<void(K1, K2, K3, K4, int)> addKey,
                           std::function<void()> makeResult) -> void;

  template <typename K1, typename K2, typename K3, typename K4>
  inline auto buildColumns(VariableBuilder::Map4Builder<K1, K2, K3, K4> builder)
      -> xpress::maps::HashMap4<K1, K2, K3, K4, int>;

  template <typename I, typename K1, typename K2, typename K3, typename K4,
            typename Func0, xpress_isinvocable(Func0, I, Func0)>
  inline auto
  buildColumns(VariableBuilder::Map4Builder<K1, K2, K3, K4> builder,
               Func0 makeResult,
               std::function<void(I &, K1, K2, K3, K4, int)> addResult) -> I;

  template <typename C1, typename C2, typename C3, typename C4,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4)>
  inline auto addColumns(C1 dim1, C2 dim2, C3 dim3, C4 dim4)
      -> VariableBuilder::ColumnArray4Builder<C1, C2, C3, C4>;

  template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
            typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            typename K4 = typename xpress_elementtype(Iter3),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2), xpress_isiterable(Iter3)>
  inline auto addColumns(Iter0 const &iterable1, Iter1 const &iterable2,
                         Iter2 const &iterable3, Iter3 const &iterable4)
      -> VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>;

  template <typename K1, typename K2, typename K3, typename K4>
  inline auto addColumns(xpress::SizedArray<K1 const> const &arr1,
                         xpress::SizedArray<K2 const> const &arr2,
                         xpress::SizedArray<K3 const> const &arr3,
                         xpress::SizedArray<K4 const> const &arr4)
      -> VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>;

public:
  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  inline auto
  buildColumns(VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder,
               std::function<void(std::vector<std::vector<std::vector<
                                      std::vector<std::vector<int>>>>> const &)>
                   transformResult)
      -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>>;

  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  inline auto
  buildColumns(VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder)
      -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>>;

  template <typename I, typename C1, typename C2, typename C3, typename C4,
            typename C5, xpress_isintegral(C1), xpress_isintegral(C2),
            xpress_isintegral(C3), xpress_isintegral(C4), xpress_isintegral(C5),
            typename Func0, xpress_isinvocable(Func0, I, Func0)>
  inline auto
  buildColumns(VariableBuilder::Array5Builder<C1, C2, C3, C4, C5> builder,
               Func0 makeResult,
               std::function<void(I &, C1, C2, C3, C4, C5, int)> addResult)
      -> I;

  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  inline auto
  buildColumns(VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder,
               std::function<void(K1, K2, K3, K4, K5, int)> addKey,
               std::function<void()> makeResult) -> void;

  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  inline auto
  buildColumns(VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder)
      -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int>;

  template <typename I, typename K1, typename K2, typename K3, typename K4,
            typename K5, typename Func0, xpress_isinvocable(Func0, I, Func0)>
  inline auto
  buildColumns(VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> builder,
               Func0 makeResult,
               std::function<void(I &, K1, K2, K3, K4, K5, int)> addResult)
      -> I;

  template <typename C1, typename C2, typename C3, typename C4, typename C5,
            xpress_isintegral(C1), xpress_isintegral(C2), xpress_isintegral(C3),
            xpress_isintegral(C4), xpress_isintegral(C5)>
  inline auto addColumns(C1 dim1, C2 dim2, C3 dim3, C4 dim4, C5 dim5)
      -> VariableBuilder::ColumnArray5Builder<C1, C2, C3, C4, C5>;

  template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
            typename Iter4, typename K1 = typename xpress_elementtype(Iter0),
            typename K2 = typename xpress_elementtype(Iter1),
            typename K3 = typename xpress_elementtype(Iter2),
            typename K4 = typename xpress_elementtype(Iter3),
            typename K5 = typename xpress_elementtype(Iter4),
            xpress_isiterable(Iter0), xpress_isiterable(Iter1),
            xpress_isiterable(Iter2), xpress_isiterable(Iter3),
            xpress_isiterable(Iter4)>
  inline auto addColumns(Iter0 const &iterable1, Iter1 const &iterable2,
                         Iter2 const &iterable3, Iter3 const &iterable4,
                         Iter4 const &iterable5)
      -> VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>;

  template <typename K1, typename K2, typename K3, typename K4, typename K5>
  inline auto addColumns(xpress::SizedArray<K1 const> const &arr1,
                         xpress::SizedArray<K2 const> const &arr2,
                         xpress::SizedArray<K3 const> const &arr3,
                         xpress::SizedArray<K4 const> const &arr4,
                         xpress::SizedArray<K5 const> const &arr5)
      -> VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>;

protected:
  inline auto
  columnTypeToArray(xpress::SizedArray<ColumnType const> const &types)
      -> std::vector<char>;

  class ColumnCreator final {
  public:
    int const firstCol;

  private:
    int nextCol;

  public:
    int created;

  private:
    ExtendableDoubleArray lbs;

    ExtendableDoubleArray ubs;

    ExtendableTypeArray types;

    ExtendableDoubleArray limits;

    ExtendableArray<std::optional<std::string>> names;

    ExtendableIntArray indices;

    int flags;

    static inline constexpr int const FLAG_LB = 0x0001;

    static inline constexpr int const FLAG_UB = 0x0002;

    static inline constexpr int const FLAG_NAME = 0x0004;

    static inline constexpr int const FLAG_TYPE = 0x0008;

    static inline constexpr int const FLAG_LIMIT = 0x0010;

    ExtendableIntArray nondefIdx;

    ExtendableTypeArray nondefBounds;

    ExtendableDoubleArray nondefVals;

  public:
    inline ColumnCreator(XPRSProblem *prob, int initialSize, bool hasNames);

    inline auto finishColumn(XPRSProblem *prob) -> void;

    inline auto conditionalCommit(XPRSProblem *prob) -> void;

    inline auto commit(XPRSProblem *prob) -> void;

  private:
    inline auto appendNondef(int idx, char type, double value) -> void;

  public:
    inline auto addColumn(XPRSProblem *prob, double lb, double ub, char type,
                          double limit, std::optional<std::string> name) -> int;

    inline auto undo(XPRSProblem *prob) -> void;
  };

private:
  inline auto addOneColumn(double obj, double lb, double ub, char type,
                           xpress::SizedArray<int const> const &rowind,
                           xpress::SizedArray<double const> const &rowval,
                           std::optional<std::string> name) -> int;

public:
  [[deprecated("deprecated since 45.00 use addColumn() instead")]] inline auto
  addCol(double obj, double lb, double ub, char type,
         xpress::SizedArray<int const> const &rowind,
         xpress::SizedArray<double const> const &rowval,
         std::optional<std::string> name) -> int;

  [[deprecated("deprecated since 45.00 use addColumn() instead")]] inline auto
  addCol(double obj, double lb, double ub, char type,
         xpress::SizedArray<int const> const &rowind,
         xpress::SizedArray<double const> const &rowval) -> int;

  [[deprecated("deprecated since 45.00 use addColumn() instead")]] inline auto
  addCol(double obj, double lb, double ub) -> int;

  [[deprecated("deprecated since 45.00 use addColumn() instead")]] inline auto
  addCol(double obj, double lb, double ub, char type) -> int;

  [[deprecated("deprecated since 45.00 use addColumn() instead")]] inline auto
  addCol(double obj, double lb, double ub, std::optional<std::string> name)
      -> int;

  [[deprecated("deprecated since 45.00 use addColumn() instead")]] inline auto
  addCol(double obj, double lb, double ub, char type,
         std::optional<std::string> name) -> int;

public:
  inline auto addMipSol(xpress::SizedArray<double const> const &val,
                        xpress::SizedArray<int const> const &ind,
                        std::optional<std::string> name) -> void;

  inline auto addMipSol(xpress::SizedArray<double const> const &val,
                        xpress::SizedArray<int const> const &ind) -> void;

private:
  inline auto addNamesWithNull(
      xpress::Namespaces which, int firstIndex, int count,
      xpress::SizedArray<std::optional<std::string> const> const &namesToAdd)
      -> void;

public:
  inline auto
  addNames(xpress::Namespaces type,
           xpress::SizedArray<std::optional<std::string> const> const &names,
           int first, int last) -> void;

protected:
  inline auto rowTypeToArray(xpress::SizedArray<RowType const> const &types)
      -> std::vector<char>;

public:
  static inline constexpr char const LEQ = 'L';

  static inline constexpr char const GEQ = 'G';

  static inline constexpr char const EQ = 'E';

private:
  static inline auto
  arrayChunk(xpress::SizedArray<std::optional<std::string> const> const &src,
             int begin, int end) -> std::vector<std::string>;

public:
  class RowCreator : public xpress::objects::PostfixExtractor {
  public:
    XPRSProblem *const prob;

    int const firstRow;

  private:
    int nextInequality;

  public:
    int created;

    bool const errorIfVariables;

    ExtendableIntArray start;

    ExtendableDoubleArray rngvals;

    ExtendableDoubleArray rowcoefs;

    ExtendableIntArray colinds;

    ExtendableTypeArray types;

    ExtendableDoubleArray rhsvals;

    ExtendableArray<std::optional<std::string>> names;

    ExtendableIntArray qind1;

    ExtendableIntArray qind2;

    ExtendableDoubleArray qcoef;

  protected:
    int flags;

  public:
    ExtendableIntArray formulaRows;

    ExtendableIntArray formulaStart;

    ExtendableIntArray formulaType;

    ExtendableDoubleArray formulaValue;

  private:
    int formulasStarted;

    int formulasInCurrentRow;

    static inline constexpr int const FLAG_NAME = 0x01;

    static inline constexpr int const FLAG_RANGE = 0x02;

  public:
    ExtendableIntArray indices;

    inline RowCreator(XPRSProblem *prob, int initialInequalitiesSize,
                      int initialNzsSize, bool hasNames, bool errorIfVariables);

    inline auto checkForVariables() -> void;

    inline auto normalizeCurrentRow() -> void;

    inline auto finishRow(bool normalize) -> void;

    static inline constexpr int const nonzeroThreshold = 1048576;

    virtual inline auto conditionalCommit() -> void;

    virtual inline auto commit() -> void;

    inline auto addRow(xpress::SizedArray<int const> const &colind,
                       xpress::SizedArray<double const> const &rowcoef,
                       double rngval, char type, double rhs,
                       std::optional<std::string> name, bool normalize) -> void;

    virtual inline auto undo() -> void;

    inline auto addRhs(double add) -> void;

    inline auto addNonzero(int idx, double coef) -> void;

    inline auto addNonzero(int idx1, int idx2, double coef) -> void;

    inline auto addNonzeros(xpress::SizedArray<int const> const &idx,
                            xpress::SizedArray<double const> const &coef,
                            double factor) -> void;

    inline auto getPostfixExtractor() -> xpress::objects::PostfixExtractor *;

    inline auto dereferenceVariable(xpress::objects::Variable x)
        -> int override;

    inline auto addToken(int type, double value)
        -> xpress::objects::PostfixExtractor & override;

    inline auto startExpression() -> void override;

    inline auto endExpression() -> void override;

    inline auto setType(char type) -> void;

    inline auto applyRange(double lb, double ub) -> void;

    inline auto setName(std::optional<std::string> name) -> void;

    inline auto getRowCount() -> int;

  private:
    static inline constexpr int const nonzerosPerInequality = 16;

  public:
    static inline auto estimateLinearNonzeros(int rows, XPRSProblem *prob)
        -> int;
    virtual ~RowCreator() {}
  };

  class RowInfo {
  public:
    std::vector<int> const ind;

    std::vector<double> const val;

    char const type;

    double const rhs;

    inline RowInfo(xpress::SizedArray<int const> const &ind,
                   xpress::SizedArray<double const> const &val, char type,
                   double rhs);
  };

  inline auto addRow(xpress::SizedArray<int const> const &colind,
                     xpress::SizedArray<double const> const &colval,
                     char rowtype, double rhs, double rng,
                     std::optional<std::string> name) -> int;

  inline auto addRow(xpress::SizedArray<int const> const &colind,
                     xpress::SizedArray<double const> const &colval,
                     char rowtype, double rhs, std::optional<std::string> name)
      -> int;

  inline auto addRow(xpress::SizedArray<int const> const &colind,
                     xpress::SizedArray<double const> const &colval,
                     char rowtype, double rhs) -> int;

  inline auto addRow(RowInfo row) -> int;

  inline auto addRow(RowInfo row, std::optional<std::string> name) -> int;

  inline auto addCut(int cuttype, xpress::SizedArray<int const> const &colind,
                     xpress::SizedArray<double const> const &colval,
                     char rowtype, double rhs) -> void;

  inline auto addCut(int cuttype, RowInfo cut) -> void;

  inline auto presolveRow(xpress::SizedArray<int const> const &ind,
                          xpress::SizedArray<double const> const &val,
                          char type, double rhs)
      -> std::optional<xpress::XPRSProblem::RowInfo>;

  inline auto presolveRow(xpress::XPRSProblem::RowInfo row)
      -> std::optional<xpress::XPRSProblem::RowInfo>;

  inline auto clearObjective() -> void;

  inline auto setObjective(xpress::SizedArray<int const> const &ind,
                           xpress::SizedArray<double const> const &val) -> void;

  inline auto setObjective(xpress::SizedArray<int const> const &ind,
                           xpress::SizedArray<double const> const &val,
                           xpress::ObjSense sense) -> void;

public:
  inline auto chgLB(int j, double lb) -> void;

  inline auto chgUB(int j, double ub) -> void;

  inline auto chgBounds(int j, double lb, double ub) -> void;

protected:
  virtual inline auto drop() -> void;

public:
  inline auto getNames(int type, int first, int last)
      -> std::vector<std::string>;

  inline auto getNames(xpress::Namespaces type, int first, int last)
      -> std::vector<std::string>;

  inline auto getName(int type, int elt) -> std::string;

  inline auto getName(xpress::Namespaces type, int elt) -> std::string;

  inline auto getRowNames(int first, int last) -> std::vector<std::string>;

  inline auto getRowName(int index) -> std::string;

  inline auto getColumnNames(int first, int last) -> std::vector<std::string>;

  inline auto getColumnName(int index) -> std::string;

  inline auto getSetNames(int first, int last) -> std::vector<std::string>;

  inline auto getSetName(int index) -> std::string;

  inline auto getPWLNames(int first, int last) -> std::vector<std::string>;

  inline auto getPWLName(int index) -> std::string;

  inline auto getGenConsNames(int first, int last) -> std::vector<std::string>;

  inline auto getGenConsName(int index) -> std::string;

public:
  class MatrixInfo final {
  public:
    std::vector<int> start;

    std::vector<int> ind;

    std::vector<double> val;

    inline MatrixInfo();
  };

  inline auto getCols(int first, int last) -> xpress::XPRSProblem::MatrixInfo;

  inline auto getRows(int first, int last) -> xpress::XPRSProblem::MatrixInfo;

  inline auto getMQObj(int first, int last) -> xpress::XPRSProblem::MatrixInfo;

public:
  class IISData {
  public:
    std::vector<int> rowind;

    std::vector<int> colind;

    std::vector<char> contype;

    std::vector<char> bndtype;

    std::vector<double> duals;

    std::vector<double> djs;

    std::vector<char> isolationrows;

    std::vector<char> isolationcols;

    inline IISData(int rownumber, int colnumber);
  };

  inline auto getIISData(int number) -> xpress::XPRSProblem::IISData;

  class IISStatusInfo final {
  public:
    std::vector<int> rowsizes;

    std::vector<int> colsizes;

    std::vector<double> suminfeas;

    std::vector<int> numinfeas;

    inline IISStatusInfo();

    inline IISStatusInfo(int iiscount);
  };

  inline auto IISStatus() -> xpress::XPRSProblem::IISStatusInfo;

public:
  inline auto optimize(std::optional<std::string> flags) -> int;

  inline auto optimize() -> int;

public:
  class Solution {
  public:
    std::vector<double> x;

    std::vector<double> slack;

    std::vector<double> duals;

    std::vector<double> djs;

    inline Solution();

    inline Solution(int rows, int cols, bool hasDuals);
  };

  class StatusSolution : public Solution {
  public:
    int status;

    inline StatusSolution();

    inline StatusSolution(int rows, int cols, bool duals);
  };

public:
  typedef std::function<double(double)> MapFunctor;

  typedef std::function<double(double const *)> VecMapFunctor;

  typedef std::function<std::vector<double>(double const *)> MultiMapFunctor;

  typedef std::function<double(double, double, double *)> MapDeltaFunctor;

  typedef std::function<double(double const *, double const *, double *)>
      VecMapDeltaFunctor;

  typedef std::function<std::vector<double>(double const *, double const *,
                                            double *)>
      MultiMapDeltaFunctor;

  class AbstractUserFunction : public CallbackHolder {
  private:
    int id;

    std::string const name;

    XPRSProblem *const prob;

  public:
    inline auto triggerEvalError() -> void;

    inline auto triggerEvalError(std::exception_ptr e) -> void;

  protected:
    inline AbstractUserFunction(XPRSProblem *owner, std::string name);

  public:
    inline auto setId(int functionId) -> void;

    inline auto getId() -> int;

    inline auto getName() const -> std::string;

    virtual auto evaluate(int output,
                          xpress::SizedArray<double const> const &args)
        -> double = 0;

    inline auto evaluate(xpress::SizedArray<double const> const &args)
        -> double;

    virtual auto isMultiOutput() -> bool = 0;

    virtual auto call(int output,
                      std::vector<xpress::objects::Expression> arguments)
        -> xpress::objects::Expression = 0;

    inline auto call(std::vector<xpress::objects::Expression> arguments)
        -> xpress::objects::Expression;
    virtual ~AbstractUserFunction() {}
  };

  class MapFunction final : public AbstractUserFunction {
  public:
    MapFunctor const functor;

    inline MapFunction(XPRSProblem *prob, std::string name, MapFunctor functor);

    virtual inline auto isMultiOutput() -> bool override;

    inline auto call(double value) -> double;

    virtual inline auto evaluate(int,
                                 xpress::SizedArray<double const> const &args)
        -> double override;

    virtual inline auto call(int,
                             std::vector<xpress::objects::Expression> arguments)
        -> xpress::objects::Expression override;
  };

  class VecMapFunction final : public AbstractUserFunction {
  public:
    int const nIn;

    VecMapFunctor const functor;

    inline VecMapFunction(XPRSProblem *prob, std::string name,
                          VecMapFunctor functor, int nIn);

    virtual inline auto isMultiOutput() -> bool override;

    inline auto call(xpress::SizedArray<double const> const &values) -> double;

    virtual inline auto evaluate(int,
                                 xpress::SizedArray<double const> const &args)
        -> double override;

    virtual inline auto call(int,
                             std::vector<xpress::objects::Expression> arguments)
        -> xpress::objects::Expression override;
  };

  class MultiMapFunction final : public AbstractUserFunction {
  public:
    MultiMapFunctor const functor;

    int const nIn;

    int const nOut;

    inline MultiMapFunction(XPRSProblem *prob, std::string name,
                            MultiMapFunctor functor, int nIn, int nOut);

    virtual inline auto isMultiOutput() -> bool override;

    inline auto call(xpress::SizedArray<double const> const &values)
        -> std::vector<double>;

    virtual inline auto evaluate(int output,
                                 xpress::SizedArray<double const> const &args)
        -> double override;

    virtual inline auto call(int output,
                             std::vector<xpress::objects::Expression> arguments)
        -> xpress::objects::Expression override;
  };

  class MapDeltaFunction final : public AbstractUserFunction {
  public:
    MapDeltaFunctor const functor;

    inline MapDeltaFunction(XPRSProblem *prob, std::string name,
                            MapDeltaFunctor functor);

    virtual inline auto isMultiOutput() -> bool override;

    inline auto call(double value, double delta,
                     xpress::SizedArray<double> const &partial) -> double;

    virtual inline auto evaluate(int,
                                 xpress::SizedArray<double const> const &args)
        -> double override;

    virtual inline auto call(int,
                             std::vector<xpress::objects::Expression> arguments)
        -> xpress::objects::Expression override;
  };

  class VecMapDeltaFunction final : public AbstractUserFunction {
  public:
    VecMapDeltaFunctor const functor;

    int const nIn;

    inline VecMapDeltaFunction(XPRSProblem *prob, std::string name,
                               VecMapDeltaFunctor functor, int nIn);

    virtual inline auto isMultiOutput() -> bool override;

    inline auto call(xpress::SizedArray<double const> const &values,
                     xpress::SizedArray<double const> const &deltas,
                     xpress::SizedArray<double> const &partials) -> double;

    virtual inline auto evaluate(int,
                                 xpress::SizedArray<double const> const &args)
        -> double override;

    virtual inline auto call(int,
                             std::vector<xpress::objects::Expression> arguments)
        -> xpress::objects::Expression override;
  };

  class MultiMapDeltaFunction final : public AbstractUserFunction {
  public:
    MultiMapDeltaFunctor const functor;

    int const nIn;

    int const nOut;

    inline MultiMapDeltaFunction(XPRSProblem *prob, std::string name,
                                 MultiMapDeltaFunctor functor, int nIn,
                                 int nOut);

    virtual inline auto isMultiOutput() -> bool override;

    inline auto call(xpress::SizedArray<double const> const &values,
                     xpress::SizedArray<double const> const &deltas,
                     xpress::SizedArray<double> const &partials)
        -> std::vector<double>;

    virtual inline auto evaluate(int output,
                                 xpress::SizedArray<double const> const &args)
        -> double override;

    virtual inline auto call(int output,
                             std::vector<xpress::objects::Expression> arguments)
        -> xpress::objects::Expression override;
  };

  inline auto nlpAddUserFunction(std::optional<std::string> funcname,
                                 int options, MapFunctor functor)
      -> xpress::XPRSProblem::MapFunction *;

  inline auto nlpAddUserFunction(std::optional<std::string> funcname, int nIn,
                                 int options, VecMapFunctor functor)
      -> xpress::XPRSProblem::VecMapFunction *;

  inline auto nlpAddUserFunction(std::optional<std::string> funcname, int nIn,
                                 int nOut, int options, MultiMapFunctor functor)
      -> xpress::XPRSProblem::MultiMapFunction *;

  inline auto nlpAddUserFunction(std::optional<std::string> funcname,
                                 int options, MapDeltaFunctor functor)
      -> xpress::XPRSProblem::MapDeltaFunction *;

  inline auto nlpAddUserFunction(std::optional<std::string> funcname, int nIn,
                                 int options, VecMapDeltaFunctor functor)
      -> xpress::XPRSProblem::VecMapDeltaFunction *;

  inline auto nlpAddUserFunction(std::optional<std::string> funcname, int nIn,
                                 int nOut, int options,
                                 MultiMapDeltaFunctor functor)
      -> xpress::XPRSProblem::MultiMapDeltaFunction *;

private:
  static inline auto mapToArrays(std::unordered_map<int, double> const &map)
      -> std::pair<std::vector<int>, std::vector<double>>;

public:
  class ConstraintCreator final : public xpress::XPRSProblem::RowCreator {

  private:
    class PWLData final {
    public:
      int const firstPWL;

      int nextPWL;

      int created;

      ExtendableIntArray arguments;

      ExtendableIntArray resultants;

      ExtendableIntArray start;

      ExtendableDoubleArray xvals;

      ExtendableDoubleArray yvals;

      ExtendableArray<std::optional<std::string>> names;

    private:
      bool anyName;

    public:
      inline PWLData(XPRSProblem *xprob, bool hasNames);

      inline auto addPWL(int argument, int resultant,
                         xpress::SizedArray<double const> const &xval,
                         xpress::SizedArray<double const> const &yval,
                         std::optional<std::string> name) -> void;

      inline auto commit(XPRSProblem *xprob) -> void;

      inline auto undo(XPRSProblem *xprob) -> void;
    };

    std::shared_ptr<PWLData> pwl;

    class GenConsData final {
    public:
      int const firstCon;

      int created;

      ExtendableIntArray colstart;

      ExtendableIntArray valstart;

      ExtendableIntArray resultants;

      ExtendableArray<xpress::GenConsType> types;

      ExtendableIntArray colinds;

      ExtendableDoubleArray values;

      ExtendableArray<std::optional<std::string>> names;

    private:
      bool nameCached;

    public:
      inline GenConsData(XPRSProblem *prob, bool hasNames);

      inline auto commit(XPRSProblem *xprob) -> void;

      inline auto
      addGeneralConstraint(int resultant, xpress::GenConsType type,
                           xpress::SizedArray<int const> const &colind,
                           xpress::SizedArray<double const> const &value,
                           std::optional<std::string> name) -> void;

      inline auto undo(XPRSProblem *xprob) -> void;

      inline auto shouldCommit(int threshold) -> bool;
    };

    std::shared_ptr<GenConsData> gencons;

    class SetData final {
    private:
      int const firstSet;

      int setsAdded;

      ExtendableIntArray start;

      ExtendableTypeArray types;

      ExtendableIntArray sosElems;

      ExtendableDoubleArray sosWeights;

      ExtendableArray<std::optional<std::string>> names;

      bool nameCached;

    public:
      inline SetData(XPRSProblem *prob, bool hasNames);

      inline auto addSet(SetType type,
                         xpress::SizedArray<int const> const &elements,
                         xpress::SizedArray<double const> const &weights,
                         std::optional<std::string> name) -> void;

      inline auto getNZs() -> int;

      inline auto commit(XPRSProblem *xprob) -> void;

      inline auto undo(XPRSProblem *xprob) -> void;
    };

    std::shared_ptr<SetData> setData;

  public:
    inline ConstraintCreator(XPRSProblem *prob, bool hasNames,
                             bool errorIfVariables);

    inline ConstraintCreator(XPRSProblem *prob, int initialRowsSize,
                             int initialNzsSize, bool hasNames,
                             bool errorIfVariables);

    virtual inline auto undo() -> void override;

    virtual inline auto conditionalCommit() -> void override;

    virtual inline auto commit() -> void override;

    inline auto addPWL(int argument, int resultant,
                       xpress::SizedArray<double const> const &xval,
                       xpress::SizedArray<double const> const &yval,
                       std::optional<std::string> name) -> void;

    inline auto
    addGeneralConstraint(int resultant, xpress::GenConsType type,
                         xpress::SizedArray<int const> const &colind,
                         xpress::SizedArray<double const> const &value,
                         std::optional<std::string> name) -> void;

    inline auto addSet(SetType type, xpress::SizedArray<int const> const &elems,
                       xpress::SizedArray<double const> const &weight,
                       std::optional<std::string> name) -> void;
  };

public:
  static void console(XPRSProblem &prob, char const *msg, int msglen,
                      int msgtype) {
    (void)prob;
    if (!msglen)
      msg = "";
    if (msgtype < 0) {
      std::cout << std::flush;
      std::cerr << std::flush;
    } else {
      switch (msgtype) {
      case 1:
        std::cout << msg << std::endl;
        break;
      case 3:
        std::cout << msg << std::endl;
        break;
      case 4:
        std::cerr << msg << std::endl;
        break;
      }
    }
  }

protected:
  XPRSProblem(XPRSprob p)
      : TypedObject([&p]() { return p; }, nullptr, false, nullptr),
        attributes(this), controls(this) {}

public:
  XPRSProblem(bool)
      : TypedObject(&createPointer, &destroyPointer, false, nullptr),
        attributes(this), controls(this) {}

  XPRSProblem(char const *name = nullptr, char const *licpath = nullptr)
      : TypedObject(&createPointer, &destroyPointer, true, licpath),
        attributes(this), controls(this) {
    if (name != nullptr) {
      int code = XPRSsetprobname(ptr, name);
      if (code) {
        XPRSException ex = XPRSException::error(ptr, code);
        XPRSdestroyprob(ptr);
        throw ex;
      }
    }
  }

  ~XPRSProblem() noexcept {}
};

class NameList : public TypedObject<XPRSnamelist> {
  NameList(NameList const &) = delete;
  NameList &operator=(NameList const &) = delete;
  NameList(NameList &&) = delete;
  NameList &operator=(NameList &&) = delete;

  static auto createPointer() -> XPRSnamelist {
    xpress_diags_push xpress_ignore_deprecated XPRSnamelist namelist;
    int code = XPRS_nml_create(&namelist);
    if (code) {
      throw XPRSException(code);
    }
    return namelist;
    xpress_diags_pop
  }

  static void destroyPointer(XPRSnamelist n) {
    xpress_diags_push xpress_ignore_deprecated XPRS_nml_destroy(n);
    xpress_diags_pop
  }

public:
  NameList() : TypedObject(&createPointer, &destroyPointer, false, nullptr) {}
};

class MIPSolPool : public TypedObject<XPRSmipsolpool> {
  friend class MipSolEnumHandlerCallbackHolder;
  MIPSolPool(MIPSolPool const &) = delete;
  MIPSolPool &operator=(MIPSolPool const &) = delete;
  MIPSolPool(MIPSolPool &&) = delete;
  MIPSolPool &operator=(MIPSolPool &&) = delete;

  static auto createPointer() -> XPRSmipsolpool {
    XPRSmipsolpool pool;
    int code = XPRS_msp_create(&pool);
    if (code) {
      throw XPRSException(code);
    }
    return pool;
  }

  static void destroyPointer(XPRSmipsolpool p) { XPRS_msp_destroy(p); }

protected:
  MIPSolPool(XPRSmipsolpool e)
      : TypedObject([=]() { return e; }, nullptr, false, nullptr) {}

public:
  MIPSolPool() : TypedObject(&createPointer, &destroyPointer, false, nullptr) {}

  inline void destroy();

  inline void probAttach(XPRSProblem const &prob);

  inline void probDetach(XPRSProblem const &prob);

  inline void getSolList(XPRSProblem const &prob_to_rank_against,
                         int iRankAttrib, int bRankAscending,
                         int iRankFirstIndex_Ob, int iRankLastIndex_Ob,
                         Array<int> const &iSolutionIds_Zb,
                         int *nReturnedSolIds, int *nSols);

  inline void getSolList2(XPRSProblem const &prob_to_rank_against,
                          int iRankAttrib, int bRankAscending,
                          int iRankFirstIndex_Ob, int iRankLastIndex_Ob,
                          int bUseUserBitFilter, int iUserBitMask,
                          int iUserBitPattern, int bUseInternalBitFilter,
                          int iInternalBitMask, int iInternalBitPattern,
                          Array<int> const &iSolutionIds_Zb,
                          int *nReturnedSolIds, int *nSols);

  inline void getSol(int iSolutionId, int *iSolutionIdStatus_,
                     Array<double> const &x, int iColFirst, int iColLast,
                     int *nValuesReturned);

  inline void getSlack(XPRSProblem const &prob_to_rank_against, int iSolutionId,
                       int *iSolutionIdStatus_, Array<double> const &slack,
                       int iRowFirst, int iRowLast, int *nValuesReturned);

  inline void loadSol(int *iSolutionId, Array<double const> const &x, int nCols,
                      std::optional<std::string> const &sSolutionName,
                      int *bNameModifiedForUniqueness,
                      int *iSolutionIdOfExistingDuplicatePreventedLoad);

  inline auto delSol(int iSolutionId) -> int;

  inline void getIntAttribProbSol(XPRSProblem const &prob_to_rank_against,
                                  int iSolutionId, int *iSolutionIdStatus_,
                                  int iAttribId, int *Dst) const;

  inline void getDblAttribProbSol(XPRSProblem const &prob_to_rank_against,
                                  int iSolutionId, int *iSolutionIdStatus_,
                                  int iAttribId, double *Dst) const;

  inline auto getIntAttribProb(XPRSProblem const &prob, int iAttribId) const
      -> int;

  inline auto getDblAttribProb(XPRSProblem const &prob, int iAttribId) const
      -> double;

  inline void getIntAttribSol(int iSolutionId, int *iSolutionIdStatus_,
                              int iAttribId, int *Dst) const;

  inline auto getIntAttribSol(int iSolutionId, int *iSolutionIdStatus_,
                              int iAttribId) const -> int;

  inline auto getIntAttribSol(int iSolutionId, int iAttribId) const -> int;

  inline void getDblAttribSol(int iSolutionId, int *iSolutionIdStatus_,
                              int iAttribId, double *Dst) const;

  inline auto getDblAttribSol(int iSolutionId, int *iSolutionIdStatus_,
                              int iAttribId) const -> double;

  inline auto getDblAttribSol(int iSolutionId, int iAttribId) const -> double;

  inline void getIntControlSol(int iSolutionId, int *iSolutionIdStatus_,
                               int iControlId, int *Val) const;

  inline auto getIntControlSol(int iSolutionId, int *iSolutionIdStatus_,
                               int iControlId) const -> int;

  inline auto getIntControlSol(int iSolutionId, int iControlId) const -> int;

  inline void getDblControlSol(int iSolutionId, int *iSolutionIdStatus_,
                               int iControlId, double *Val) const;

  inline auto getDblControlSol(int iSolutionId, int *iSolutionIdStatus_,
                               int iControlId) const -> double;

  inline auto getDblControlSol(int iSolutionId, int iControlId) const -> double;

  inline auto setIntControlSol(int iSolutionId, int iControlId, int Val) -> int;

  inline auto setDblControlSol(int iSolutionId, int iControlId, double Val)
      -> int;

  inline void getIntAttribProbExtreme(XPRSProblem const &prob_to_rank_against,
                                      int bGet_Max_Otherwise_Min,
                                      int *iSolutionId, int iAttribId,
                                      int *ExtremeVal) const;

  inline void getDblAttribProbExtreme(XPRSProblem const &prob_to_rank_against,
                                      int bGet_Max_Otherwise_Min,
                                      int *iSolutionId, int iAttribId,
                                      double *ExtremeVal) const;

  inline auto getIntAttrib(int iAttribId) const -> int;

  inline auto getDblAttrib(int iAttribId) const -> double;

  inline auto getIntControl(int iControlId) const -> int;

  inline auto getDblControl(int iControlId) const -> double;

  inline void setIntControl(int iControlId, int Val);

  inline void setDblControl(int iControlId, double Val);

  inline void setSolName(int iSolutionId,
                         std::optional<std::string> const &sNewSolutionBaseName,
                         int *bNameModifiedForUniqueness,
                         int *iSolutionIdStatus_);

  inline void getSolName(int iSolutionId, char *_sname, int _iStringBufferBytes,
                         int *_iBytesInInternalString, int *iSolutionIdStatus_);

  inline auto getSolName(int iSolutionId, int *iSolutionIdStatus_)
      -> std::string;

  inline auto getSolName(int iSolutionId) -> std::string;

  inline auto findSolByName(std::optional<std::string> const &sSolutionName)
      -> int;

  inline auto writeSlxSol(XPRSProblem const &prob_context, int iSolutionId,
                          std::optional<std::string> const &sFileName,
                          std::optional<std::string> const &sFlags) -> int;

  inline void readSlxSol(NameList const &col_name_list,
                         std::optional<std::string> const &sFileName,
                         std::optional<std::string> const &sFlags,
                         int *iSolutionId_Beg, int *iSolutionId_End);
};

class MIPSolEnum : public TypedObject<XPRSmipsolenum> {
  friend class MipSolEnumHandlerCallbackHolder;
  friend class MseGetSolutionDiffCallbackHolder;
  MIPSolEnum(MIPSolEnum const &) = delete;
  MIPSolEnum &operator=(MIPSolEnum const &) = delete;
  MIPSolEnum(MIPSolEnum &&) = delete;
  MIPSolEnum &operator=(MIPSolEnum &&) = delete;

  static auto createPointer() -> XPRSmipsolenum {
    XPRSmipsolenum enumerator;
    int code = XPRS_mse_create(&enumerator);
    if (code) {
      throw XPRSException(code);
    }
    return enumerator;
  }

  static void destroyPointer(XPRSmipsolenum e) { XPRS_mse_destroy(e); }

protected:
  MIPSolEnum(XPRSmipsolenum e)
      : TypedObject([=]() { return e; }, nullptr, false, nullptr) {}

public:
  MIPSolEnum() : TypedObject(&createPointer, &destroyPointer, false, nullptr) {}

  inline void destroy();

  inline void minim(
      XPRSProblem const &prob, MIPSolPool const &msp,
      std::function<int(MIPSolEnum &, XPRSProblem &, MIPSolPool &, int *,
                        double const *, int, double, double *, bool *, bool *)>
          f_mse_handler,
      int *nMaxSols);

  inline void maxim(
      XPRSProblem const &prob, MIPSolPool const &msp,
      std::function<int(MIPSolEnum &, XPRSProblem &, MIPSolPool &, int *,
                        double const *, int, double, double *, bool *, bool *)>
          f_mse_handler,
      int *nMaxSols);

  inline void
  opt(XPRSProblem const &prob, MIPSolPool const &msp,
      std::function<int(MIPSolEnum &, XPRSProblem &, MIPSolPool &, int *,
                        double const *, int, double, double *, bool *, bool *)>
          f_mse_handler,
      int *nMaxSols);

  inline void getSolList(int iMetricId, int iRankFirstIndex_Ob,
                         int iRankLastIndex_Ob, Array<int> const &iSolutionIds,
                         int *nReturnedSolIds, int *nSols);

  inline void getSolMetric(int iSolutionId, int *iSolutionIdStatus,
                           int iMetricId, double *dMetric);

  inline void getCullChoice(int iMetricId, Array<int> const &cull_sol_id_list,
                            int nMaxSolsToCull, int *nSolsToCull,
                            double dNewSolMetric, Array<double const> const &x,
                            int nCols, bool *bRejectSoln);

  inline auto getCullChoice(int iMetricId, Array<int> const &cull_sol_id_list,
                            int nMaxSolsToCull, double dNewSolMetric,
                            Array<double const> const &x, int nCols,
                            bool *bRejectSoln) -> int;

  inline auto getCullChoice(int iMetricId, Array<int> const &cull_sol_id_list,
                            int nMaxSolsToCull, double dNewSolMetric,
                            bool *bRejectSoln) -> int;

  inline auto getIntAttrib(int iAttribId) const -> int;

  inline auto getDblAttrib(int iAttribId) const -> double;

  inline auto getIntControl(int iAttribId) const -> int;

  inline auto getDblControl(int iAttribId) const -> double;

  inline void setIntControl(int iAttribId, int Val);

  inline void setDblControl(int iAttribId, double Val);

  inline void
  setSolBaseName(std::optional<std::string> const &sSolutionBaseName);

  inline void getSolBaseName(char *_sname, int _iStringBufferBytes,
                             int *_iBytesInInternalString);

  inline auto addMseGetSolutionDiffCallback(
      std::function<int(MIPSolEnum &, int, int, int, double, double const *,
                        int const *, int, int, double, double const *,
                        int const *, double *)>
          callback,
      int prio = 0) -> CallbackHandle;

  inline void removeMseGetSolutionDiffCallback(CallbackHandle const &callback);

  inline void removeMseGetSolutionDiffCallbacks();
};

template <typename AlwaysVoid> class InitOnce {
  friend class XPRS;
  static std::atomic<unsigned long long> initCount;
};

template <typename AlwaysVoid>
std::atomic<unsigned long long> InitOnce<AlwaysVoid>::initCount(0);

inline void XPRS::callInit(char const *licpath) {
  int code = XPRScxxinit(licpath, ObjectMap<void>::objectDestroyed,
                         callbackAdded, callbackRemoved);
  if (code != 0) {
    throw XPRSException::licenseError(code);
  }
}
inline void XPRS::callFree() {
  int code = XPRScxxfree();
  if (code != 0) {
    throw XPRSException(code);
  }
}
} // namespace xpress
#include <xpress_objects.hpp>
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::ArrayBuilder(
    std::shared_ptr<ArrayBuilderImplementation<C, CIsIntegral>> init)
    : implPointer(init) {}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::ArrayBuilder(
    ArrayBuilderImplementation<C, CIsIntegral> const *init)
    : implPointer(
          init ? const_cast<ArrayBuilderImplementation<C, CIsIntegral> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ArrayBuilderImplementation<C, CIsIntegral>>(
                     nullptr)) {}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> *
xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::impl() {
  return dynamic_cast<ArrayBuilderImplementation<C, CIsIntegral> *>(
      implPointer.get());
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ArrayBuilderImplementation<C, CIsIntegral> const *
xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::impl() const {
  return dynamic_cast<ArrayBuilderImplementation<C, CIsIntegral> *>(
      implPointer.get());
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::ArrayBuilder(C dim)
    : implPointer(
          std::make_shared<ArrayBuilderImplementation<C, CIsIntegral>>(dim)) {}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::getDim(
    int dim) const -> int {
  return impl()->getDim(dim);
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::getLB() const
    -> std::function<double(C)> const & {
  return impl()->getLB();
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::getUB() const
    -> std::function<double(C)> const & {
  return impl()->getUB();
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::getLimit() const
    -> std::function<double(C)> const & {
  return impl()->getLimit();
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C)> const & {
  return impl()->getName();
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C)> const & {
  return impl()->getType();
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withLimit(
    Func0 newLimit) -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withName(
    Func0 newName) -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withType(
    Func0 newType) -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::build(
    XPRSProblem *prob) -> std::vector<int> {
  return impl()->build(prob);
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::build(
    xpress::objects::XpressProblem *prob)
    -> std::vector<xpress::objects::Variable> {
  return impl()->build(prob);
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ArrayBuilder<C, CIsIntegral>::ArrayBuilder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral>::
    Array2Builder(
        std::shared_ptr<
            Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>>
            init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral, C2IsIntegral>::
    Array2Builder(Array2BuilderImplementation<C1, C2, C1IsIntegral,
                                              C2IsIntegral> const *init)
    : implPointer(
          init ? const_cast<Array2BuilderImplementation<C1, C2, C1IsIntegral,
                                                        C2IsIntegral> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<Array2BuilderImplementation<
                     C1, C2, C1IsIntegral, C2IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::Array2BuilderImplementation<C1, C2, C1IsIntegral,
                                                     C2IsIntegral> *
xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                       C2IsIntegral>::impl() {
  return dynamic_cast<
      Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::Array2BuilderImplementation<C1, C2, C1IsIntegral,
                                                     C2IsIntegral> const *
xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                       C2IsIntegral>::impl() const {
  return dynamic_cast<
      Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                       C2IsIntegral>::Array2Builder(C1 dim1,
                                                                    C2 dim2)
    : implPointer(
          std::make_shared<
              Array2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral>>(
              dim1, dim2)) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::getDim(int dim) const
    -> int {
  return impl()->getDim(dim);
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::getLB() const
    -> std::function<double(C1, C2)> const & {
  return impl()->getLB();
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::getUB() const
    -> std::function<double(C1, C2)> const & {
  return impl()->getUB();
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::getLimit() const
    -> std::function<double(C1, C2)> const & {
  return impl()->getLimit();
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C1, C2)> const & {
  return impl()->getName();
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C1, C2)> const & {
  return impl()->getType();
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                            C2IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(std::optional<std::string>
                                                      newName)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                              C2IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::build(XPRSProblem *prob)
    -> std::vector<std::vector<int>> {
  return impl()->build(prob);
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::Array2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::build(xpress::objects::XpressProblem
                                                   *prob)
    -> std::vector<std::vector<xpress::objects::Variable>> {
  return impl()->build(prob);
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::Array2Builder<C1, C2, C1IsIntegral,
                                       C2IsIntegral>::Array2Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral>::
    Array3Builder(std::shared_ptr<Array3BuilderImplementation<
                      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>
                      init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral>::
    Array3Builder(
        Array3BuilderImplementation<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                    C3IsIntegral> const *init)
    : implPointer(
          init ? const_cast<Array3BuilderImplementation<
                     C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
                     init)
                     ->shared_from_this()
               : std::shared_ptr<Array3BuilderImplementation<
                     C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>(
                     nullptr)) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *
xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral>::impl() {
  return dynamic_cast<Array3BuilderImplementation<
      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::Array3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const *
xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral>::impl() const {
  return dynamic_cast<Array3BuilderImplementation<
      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral>::Array3Builder(C1 dim1,
                                                                    C2 dim2,
                                                                    C3 dim3)
    : implPointer(std::make_shared<Array3BuilderImplementation<
                      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>(
          dim1, dim2, dim3)) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getDim(int dim) const
    -> int {
  return impl()->getDim(dim);
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getLB() const
    -> std::function<double(C1, C2, C3)> const & {
  return impl()->getLB();
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getUB() const
    -> std::function<double(C1, C2, C3)> const & {
  return impl()->getUB();
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getLimit() const
    -> std::function<double(C1, C2, C3)> const & {
  return impl()->getLimit();
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C1, C2, C3)> const & {
  return impl()->getName();
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C1, C2, C3)> const & {
  return impl()->getType();
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withName(Func0
                                                                        newName)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(Func0
                                                                        newType)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(ColumnType
                                                                        newType)
    -> xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::build(XPRSProblem
                                                                     *prob)
    -> std::vector<std::vector<std::vector<int>>> {
  return impl()->build(prob);
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::Array3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::build(xpress::objects::XpressProblem *prob)
    -> std::vector<std::vector<std::vector<xpress::objects::Variable>>> {
  return impl()->build(prob);
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::Array3Builder<C1, C2, C3, C1IsIntegral, C2IsIntegral,
                                       C3IsIntegral>::Array3Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    Array4Builder(std::shared_ptr<Array4BuilderImplementation<
                      C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
                      C4IsIntegral>>
                      init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    Array4Builder(
        Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                    C3IsIntegral, C4IsIntegral> const *init)
    : implPointer(init ? const_cast<Array4BuilderImplementation<
                             C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                             C3IsIntegral, C4IsIntegral> *>(init)
                             ->shared_from_this()
                       : std::shared_ptr<Array4BuilderImplementation<
                             C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                             C3IsIntegral, C4IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral> *
xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral,
                                       C4IsIntegral>::impl() {
  return dynamic_cast<
      Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                  C3IsIntegral, C4IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::Array4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral> const *
xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral,
                                       C4IsIntegral>::impl() const {
  return dynamic_cast<
      Array4BuilderImplementation<C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                                  C3IsIntegral, C4IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::Array4Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4)
    : implPointer(std::make_shared<Array4BuilderImplementation<
                      C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
                      C4IsIntegral>>(dim1, dim2, dim3, dim4)) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::getDim(int dim) const
    -> int {
  return impl()->getDim(dim);
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::getLB() const
    -> std::function<double(C1, C2, C3, C4)> const & {
  return impl()->getLB();
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::getUB() const
    -> std::function<double(C1, C2, C3, C4)> const & {
  return impl()->getUB();
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::getLimit() const
    -> std::function<double(C1, C2, C3, C4)> const & {
  return impl()->getLimit();
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C1, C2, C3, C4)> const & {
  return impl()->getName();
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C1, C2, C3, C4)> const & {
  return impl()->getType();
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                            C2IsIntegral, C3IsIntegral,
                                            C4IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::build(XPRSProblem *prob)
    -> std::vector<std::vector<std::vector<std::vector<int>>>> {
  return impl()->build(prob);
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::Array4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::build(xpress::objects::XpressProblem *prob)
    -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>> {
  return impl()->build(prob);
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::Array4Builder<C1, C2, C3, C4, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral,
                                       C4IsIntegral>::Array4Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral>::
    Array5Builder(std::shared_ptr<Array5BuilderImplementation<
                      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                      C3IsIntegral, C4IsIntegral, C5IsIntegral>>
                      init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral>::
    Array5Builder(Array5BuilderImplementation<
                  C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
                  C4IsIntegral, C5IsIntegral> const *init)
    : implPointer(
          init ? const_cast<Array5BuilderImplementation<
                     C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                     C3IsIntegral, C4IsIntegral, C5IsIntegral> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<Array5BuilderImplementation<
                     C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                     C3IsIntegral, C4IsIntegral, C5IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral> *
xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral>::impl() {
  return dynamic_cast<Array5BuilderImplementation<
      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
      C4IsIntegral, C5IsIntegral> *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::Array5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral> const *
xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral>::impl() const {
  return dynamic_cast<Array5BuilderImplementation<
      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
      C4IsIntegral, C5IsIntegral> *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::Array5Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4, C5 dim5)
    : implPointer(
          std::make_shared<Array5BuilderImplementation<
              C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
              C4IsIntegral, C5IsIntegral>>(dim1, dim2, dim3, dim4, dim5)) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getDim(int dim) const -> int {
  return impl()->getDim(dim);
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getLB() const
    -> std::function<double(C1, C2, C3, C4, C5)> const & {
  return impl()->getLB();
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getUB() const
    -> std::function<double(C1, C2, C3, C4, C5)> const & {
  return impl()->getUB();
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getLimit() const
    -> std::function<double(C1, C2, C3, C4, C5)> const & {
  return impl()->getLimit();
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getName() const
    -> std::function<std::optional<std::string>(C1, C2, C3, C4, C5)> const & {
  return impl()->getName();
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::getType() const
    -> std::function<xpress::ColumnType(C1, C2, C3, C4, C5)> const & {
  return impl()->getType();
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                              C2IsIntegral, C3IsIntegral,
                                              C4IsIntegral, C5IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::build(XPRSProblem *prob)
    -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>> {
  return impl()->build(prob);
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::Array5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::build(xpress::objects::XpressProblem *prob)
    -> std::vector<std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>> {
  return impl()->build(prob);
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::Array5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                       C2IsIntegral, C3IsIntegral, C4IsIntegral,
                                       C5IsIntegral>::Array5Builder()
    : implPointer(nullptr) {}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::ColumnArrayBuilder(
    std::shared_ptr<ColumnArrayBuilderImplementation<C, CIsIntegral>> init)
    : implPointer(init) {}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::ColumnArrayBuilder(
    ColumnArrayBuilderImplementation<C, CIsIntegral> const *init)
    : implPointer(
          init ? const_cast<ColumnArrayBuilderImplementation<C, CIsIntegral> *>(
                     init)
                     ->shared_from_this()
               : std::shared_ptr<
                     ColumnArrayBuilderImplementation<C, CIsIntegral>>(
                     nullptr)) {}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ColumnArrayBuilderImplementation<C, CIsIntegral> *
xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::impl() {
  return dynamic_cast<ColumnArrayBuilderImplementation<C, CIsIntegral> *>(
      implPointer.get());
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ColumnArrayBuilderImplementation<C,
                                                          CIsIntegral> const *
xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::impl() const {
  return dynamic_cast<ColumnArrayBuilderImplementation<C, CIsIntegral> *>(
      implPointer.get());
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::ColumnArrayBuilder(
    C dim, xpress::XPRSProblem *prob)
    : implPointer(
          std::make_shared<ColumnArrayBuilderImplementation<C, CIsIntegral>>(
              dim, prob)) {}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withLB(
    double newLb)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withUB(
    double newUb)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::toArray()
    -> std::vector<int> {
  return impl()->toArray();
}
template <typename C, typename CIsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArrayBuilder<C, CIsIntegral>::toArray(
    Func0 makeResult, std::function<void(I, C, int)> addResult) -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::ColumnArrayBuilder<C,
                                            CIsIntegral>::ColumnArrayBuilder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                             C2IsIntegral>::
    ColumnArray2Builder(std::shared_ptr<ColumnArray2BuilderImplementation<
                            C1, C2, C1IsIntegral, C2IsIntegral>>
                            init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                             C2IsIntegral>::
    ColumnArray2Builder(ColumnArray2BuilderImplementation<
                        C1, C2, C1IsIntegral, C2IsIntegral> const *init)
    : implPointer(init ? const_cast<ColumnArray2BuilderImplementation<
                             C1, C2, C1IsIntegral, C2IsIntegral> *>(init)
                             ->shared_from_this()
                       : std::shared_ptr<ColumnArray2BuilderImplementation<
                             C1, C2, C1IsIntegral, C2IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                                           C2IsIntegral> *
xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                             C2IsIntegral>::impl() {
  return dynamic_cast<
      ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                                           C2IsIntegral> const *
xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                             C2IsIntegral>::impl() const {
  return dynamic_cast<
      ColumnArray2BuilderImplementation<C1, C2, C1IsIntegral, C2IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::ColumnArray2Builder(C1 dim1, C2 dim2,
                                                             xpress::XPRSProblem
                                                                 *prob)
    : implPointer(std::make_shared<ColumnArray2BuilderImplementation<
                      C1, C2, C1IsIntegral, C2IsIntegral>>(dim1, dim2, prob)) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(std::optional<std::string>
                                                      newName)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::ColumnArray2Builder<C1, C2, C1IsIntegral,
                                                  C2IsIntegral>::toArray()
    -> std::vector<std::vector<int>> {
  return impl()->toArray();
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::
    ColumnArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral>::toArray(
        Func0 makeResult, std::function<void(I, C1, C2, int)> addResult) -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::ColumnArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::ColumnArray2Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::ColumnArray3Builder<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral>::
    ColumnArray3Builder(
        std::shared_ptr<ColumnArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>
            init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::ColumnArray3Builder<C1, C2, C3, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral>::
    ColumnArray3Builder(
        ColumnArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const *init)
    : implPointer(
          init ? const_cast<ColumnArray3BuilderImplementation<
                     C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
                     init)
                     ->shared_from_this()
               : std::shared_ptr<ColumnArray3BuilderImplementation<
                     C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>(
                     nullptr)) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *
xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::impl() {
  return dynamic_cast<ColumnArray3BuilderImplementation<
      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::ColumnArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const *
xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::impl() const {
  return dynamic_cast<ColumnArray3BuilderImplementation<
      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::ColumnArray3Builder(C1 dim1, C2 dim2, C3 dim3,
                                       xpress::XPRSProblem *prob)
    : implPointer(std::make_shared<ColumnArray3BuilderImplementation<
                      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>(
          dim1, dim2, dim3, prob)) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withName(Func0
                                                                        newName)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(Func0
                                                                        newType)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(ColumnType
                                                                        newType)
    -> xpress::VariableBuilder::ColumnArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::toArray()
    -> std::vector<std::vector<std::vector<int>>> {
  return impl()->toArray();
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, C3, int)> addResult)
    -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::ColumnArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::ColumnArray3Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    ColumnArray4Builder(std::shared_ptr<ColumnArray4BuilderImplementation<
                            C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                            C3IsIntegral, C4IsIntegral>>
                            init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    ColumnArray4Builder(ColumnArray4BuilderImplementation<
                        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                        C3IsIntegral, C4IsIntegral> const *init)
    : implPointer(init ? const_cast<ColumnArray4BuilderImplementation<
                             C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                             C3IsIntegral, C4IsIntegral> *>(init)
                             ->shared_from_this()
                       : std::shared_ptr<ColumnArray4BuilderImplementation<
                             C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                             C3IsIntegral, C4IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral> *
xpress::VariableBuilder::ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral>::impl() {
  return dynamic_cast<ColumnArray4BuilderImplementation<
      C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
                          *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::ColumnArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral> const *
xpress::VariableBuilder::ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral>::impl() const {
  return dynamic_cast<ColumnArray4BuilderImplementation<
      C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
                          *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::ColumnArray4Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                       xpress::XPRSProblem *prob)
    : implPointer(std::make_shared<ColumnArray4BuilderImplementation<
                      C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
                      C4IsIntegral>>(dim1, dim2, dim3, dim4, prob)) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::ColumnArray4Builder<C1, C2, C3, C4, C1IsIntegral,
                                                  C2IsIntegral, C3IsIntegral,
                                                  C4IsIntegral>::toArray()
    -> std::vector<std::vector<std::vector<std::vector<int>>>> {
  return impl()->toArray();
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, C3, C4, int)>
                               addResult) -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::ColumnArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::ColumnArray4Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral>::
    ColumnArray5Builder(std::shared_ptr<ColumnArray5BuilderImplementation<
                            C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                            C3IsIntegral, C4IsIntegral, C5IsIntegral>>
                            init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::ColumnArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                             C2IsIntegral, C3IsIntegral,
                                             C4IsIntegral, C5IsIntegral>::
    ColumnArray5Builder(ColumnArray5BuilderImplementation<
                        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                        C3IsIntegral, C4IsIntegral, C5IsIntegral> const *init)
    : implPointer(
          init ? const_cast<ColumnArray5BuilderImplementation<
                     C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                     C3IsIntegral, C4IsIntegral, C5IsIntegral> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ColumnArray5BuilderImplementation<
                     C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                     C3IsIntegral, C4IsIntegral, C5IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral> *
xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::impl() {
  return dynamic_cast<ColumnArray5BuilderImplementation<
      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
      C4IsIntegral, C5IsIntegral> *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::ColumnArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral> const *
xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::impl() const {
  return dynamic_cast<ColumnArray5BuilderImplementation<
      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
      C4IsIntegral, C5IsIntegral> *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::ColumnArray5Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                       C5 dim5, xpress::XPRSProblem *prob)
    : implPointer(std::make_shared<ColumnArray5BuilderImplementation<
                      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                      C3IsIntegral, C4IsIntegral, C5IsIntegral>>(
          dim1, dim2, dim3, dim4, dim5, prob)) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::toArray()
    -> std::vector<std::vector<std::vector<std::vector<std::vector<int>>>>> {
  return impl()->toArray();
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, C3, C4, C5, int)>
                               addResult) -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::ColumnArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::ColumnArray5Builder()
    : implPointer(nullptr) {}
template <typename K1>
xpress::VariableBuilder::MapBuilder<K1>::MapBuilder(
    std::shared_ptr<MapBuilderImplementation<K1>> init)
    : implPointer(init) {}
template <typename K1>
xpress::VariableBuilder::MapBuilder<K1>::MapBuilder(
    MapBuilderImplementation<K1> const *init)
    : implPointer(
          init ? const_cast<MapBuilderImplementation<K1> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<MapBuilderImplementation<K1>>(nullptr)) {}
template <typename K1>
xpress::VariableBuilder::MapBuilderImplementation<K1> *
xpress::VariableBuilder::MapBuilder<K1>::impl() {
  return dynamic_cast<MapBuilderImplementation<K1> *>(implPointer.get());
}
template <typename K1>
xpress::VariableBuilder::MapBuilderImplementation<K1> const *
xpress::VariableBuilder::MapBuilder<K1>::impl() const {
  return dynamic_cast<MapBuilderImplementation<K1> *>(implPointer.get());
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::getIterable1() const
    -> xpress::Iterable<K1> const & {
  return impl()->getIterable1();
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::getLB() const
    -> std::function<double(K1)> const & {
  return impl()->getLB();
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::getUB() const
    -> std::function<double(K1)> const & {
  return impl()->getUB();
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::getLimit() const
    -> std::function<double(K1)> const & {
  return impl()->getLimit();
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::getName() const
    -> std::function<std::optional<std::string>(K1)> const & {
  return impl()->getName();
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::getType() const
    -> std::function<xpress::ColumnType(K1)> const & {
  return impl()->getType();
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::getFilter() const
    -> std::function<bool(K1)> const & {
  return impl()->getFilter();
}
template <typename K1>
template <typename Iter0, typename Iter0IsIterable>
xpress::VariableBuilder::MapBuilder<K1>::MapBuilder(Iter0 const &iterable1)
    : implPointer(std::make_shared<MapBuilderImplementation<K1>>(iterable1)) {}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilder<K1>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::withLB(double newLb)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilder<K1>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::withUB(double newUb)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilder<K1>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::withLimit(double newLimit)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilder<K1>::withName(Func0 newName)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilder<K1>::withType(Func0 newType)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::withType(ColumnType newType)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::MapBuilder<K1>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::MapBuilder<K1> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::build(XPRSProblem *prob)
    -> std::unordered_map<K1, int> {
  return impl()->build(prob);
}
template <typename K1>
auto xpress::VariableBuilder::MapBuilder<K1>::build(
    xpress::objects::XpressProblem *prob)
    -> std::unordered_map<K1, xpress::objects::Variable> {
  return impl()->build(prob);
}
template <typename K1>
xpress::VariableBuilder::MapBuilder<K1>::MapBuilder() : implPointer(nullptr) {}
template <typename K1, typename K2>
xpress::VariableBuilder::Map2Builder<K1, K2>::Map2Builder(
    std::shared_ptr<Map2BuilderImplementation<K1, K2>> init)
    : implPointer(init) {}
template <typename K1, typename K2>
xpress::VariableBuilder::Map2Builder<K1, K2>::Map2Builder(
    Map2BuilderImplementation<K1, K2> const *init)
    : implPointer(
          init ? const_cast<Map2BuilderImplementation<K1, K2> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<Map2BuilderImplementation<K1, K2>>(nullptr)) {}
template <typename K1, typename K2>
xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> *
xpress::VariableBuilder::Map2Builder<K1, K2>::impl() {
  return dynamic_cast<Map2BuilderImplementation<K1, K2> *>(implPointer.get());
}
template <typename K1, typename K2>
xpress::VariableBuilder::Map2BuilderImplementation<K1, K2> const *
xpress::VariableBuilder::Map2Builder<K1, K2>::impl() const {
  return dynamic_cast<Map2BuilderImplementation<K1, K2> *>(implPointer.get());
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::getIterable1() const
    -> xpress::Iterable<K1> const & {
  return impl()->getIterable1();
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::getIterable2() const
    -> xpress::Iterable<K2> const & {
  return impl()->getIterable2();
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::getLB() const
    -> std::function<double(K1, K2)> const & {
  return impl()->getLB();
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::getUB() const
    -> std::function<double(K1, K2)> const & {
  return impl()->getUB();
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::getLimit() const
    -> std::function<double(K1, K2)> const & {
  return impl()->getLimit();
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::getName() const
    -> std::function<std::optional<std::string>(K1, K2)> const & {
  return impl()->getName();
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::getType() const
    -> std::function<xpress::ColumnType(K1, K2)> const & {
  return impl()->getType();
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::getFilter() const
    -> std::function<bool(K1, K2)> const & {
  return impl()->getFilter();
}
template <typename K1, typename K2>
template <typename Iter0, typename Iter1, typename Iter0IsIterable,
          typename Iter1IsIterable>
xpress::VariableBuilder::Map2Builder<K1, K2>::Map2Builder(
    Iter0 const &iterable1, Iter1 const &iterable2)
    : implPointer(std::make_shared<Map2BuilderImplementation<K1, K2>>(
          iterable1, iterable2)) {}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withLB(double newLb)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withUB(double newUb)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withLimit(double newLimit)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withName(Func0 newName)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withType(Func0 newType)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withType(ColumnType newType)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::Map2Builder<K1, K2> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::build(XPRSProblem *prob)
    -> xpress::maps::HashMap2<K1, K2, int> {
  return impl()->build(prob);
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::Map2Builder<K1, K2>::build(
    xpress::objects::XpressProblem *prob)
    -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable> {
  return impl()->build(prob);
}
template <typename K1, typename K2>
xpress::VariableBuilder::Map2Builder<K1, K2>::Map2Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::Map3Builder<K1, K2, K3>::Map3Builder(
    std::shared_ptr<Map3BuilderImplementation<K1, K2, K3>> init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::Map3Builder<K1, K2, K3>::Map3Builder(
    Map3BuilderImplementation<K1, K2, K3> const *init)
    : implPointer(
          init ? const_cast<Map3BuilderImplementation<K1, K2, K3> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<Map3BuilderImplementation<K1, K2, K3>>(
                     nullptr)) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> *
xpress::VariableBuilder::Map3Builder<K1, K2, K3>::impl() {
  return dynamic_cast<Map3BuilderImplementation<K1, K2, K3> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::Map3BuilderImplementation<K1, K2, K3> const *
xpress::VariableBuilder::Map3Builder<K1, K2, K3>::impl() const {
  return dynamic_cast<Map3BuilderImplementation<K1, K2, K3> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getIterable1() const
    -> xpress::Iterable<K1> const & {
  return impl()->getIterable1();
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getIterable2() const
    -> xpress::Iterable<K2> const & {
  return impl()->getIterable2();
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getIterable3() const
    -> xpress::Iterable<K3> const & {
  return impl()->getIterable3();
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getLB() const
    -> std::function<double(K1, K2, K3)> const & {
  return impl()->getLB();
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getUB() const
    -> std::function<double(K1, K2, K3)> const & {
  return impl()->getUB();
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getLimit() const
    -> std::function<double(K1, K2, K3)> const & {
  return impl()->getLimit();
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getName() const
    -> std::function<std::optional<std::string>(K1, K2, K3)> const & {
  return impl()->getName();
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getType() const
    -> std::function<xpress::ColumnType(K1, K2, K3)> const & {
  return impl()->getType();
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::getFilter() const
    -> std::function<bool(K1, K2, K3)> const & {
  return impl()->getFilter();
}
template <typename K1, typename K2, typename K3>
template <typename Iter0, typename Iter1, typename Iter2,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable>
xpress::VariableBuilder::Map3Builder<K1, K2, K3>::Map3Builder(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3)
    : implPointer(std::make_shared<Map3BuilderImplementation<K1, K2, K3>>(
          iterable1, iterable2, iterable3)) {}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withLB(double newLb)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withUB(double newUb)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withLimit(
    double newLimit) -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withName(Func0 newName)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withType(Func0 newType)
    -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withType(
    ColumnType newType) -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::withFilter(
    Func0 newFilter) -> xpress::VariableBuilder::Map3Builder<K1, K2, K3> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::build(XPRSProblem *prob)
    -> xpress::maps::HashMap3<K1, K2, K3, int> {
  return impl()->build(prob);
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::Map3Builder<K1, K2, K3>::build(
    xpress::objects::XpressProblem *prob)
    -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable> {
  return impl()->build(prob);
}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::Map3Builder<K1, K2, K3>::Map3Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::Map4Builder(
    std::shared_ptr<Map4BuilderImplementation<K1, K2, K3, K4>> init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::Map4Builder(
    Map4BuilderImplementation<K1, K2, K3, K4> const *init)
    : implPointer(
          init ? const_cast<Map4BuilderImplementation<K1, K2, K3, K4> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<Map4BuilderImplementation<K1, K2, K3, K4>>(
                     nullptr)) {}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> *
xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::impl() {
  return dynamic_cast<Map4BuilderImplementation<K1, K2, K3, K4> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::Map4BuilderImplementation<K1, K2, K3, K4> const *
xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::impl() const {
  return dynamic_cast<Map4BuilderImplementation<K1, K2, K3, K4> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getIterable1() const
    -> xpress::Iterable<K1> const & {
  return impl()->getIterable1();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getIterable2() const
    -> xpress::Iterable<K2> const & {
  return impl()->getIterable2();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getIterable3() const
    -> xpress::Iterable<K3> const & {
  return impl()->getIterable3();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getIterable4() const
    -> xpress::Iterable<K4> const & {
  return impl()->getIterable4();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getLB() const
    -> std::function<double(K1, K2, K3, K4)> const & {
  return impl()->getLB();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getUB() const
    -> std::function<double(K1, K2, K3, K4)> const & {
  return impl()->getUB();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getLimit() const
    -> std::function<double(K1, K2, K3, K4)> const & {
  return impl()->getLimit();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getName() const
    -> std::function<std::optional<std::string>(K1, K2, K3, K4)> const & {
  return impl()->getName();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getType() const
    -> std::function<xpress::ColumnType(K1, K2, K3, K4)> const & {
  return impl()->getType();
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::getFilter() const
    -> std::function<bool(K1, K2, K3, K4)> const & {
  return impl()->getFilter();
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::Map4Builder(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
    Iter3 const &iterable4)
    : implPointer(std::make_shared<Map4BuilderImplementation<K1, K2, K3, K4>>(
          iterable1, iterable2, iterable3, iterable4)) {}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withLB(double newLb)
    -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withUB(double newUb)
    -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withLimit(
    Func0 newLimit) -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withLimit(
    double newLimit) -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withName(
    Func0 newName) -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withType(
    Func0 newType) -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::withFilter(
    Func0 newFilter) -> xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::build(
    XPRSProblem *prob) -> xpress::maps::HashMap4<K1, K2, K3, K4, int> {
  return impl()->build(prob);
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::build(
    xpress::objects::XpressProblem *prob)
    -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable> {
  return impl()->build(prob);
}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::Map4Builder<K1, K2, K3, K4>::Map4Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::Map5Builder(
    std::shared_ptr<Map5BuilderImplementation<K1, K2, K3, K4, K5>> init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::Map5Builder(
    Map5BuilderImplementation<K1, K2, K3, K4, K5> const *init)
    : implPointer(
          init ? const_cast<Map5BuilderImplementation<K1, K2, K3, K4, K5> *>(
                     init)
                     ->shared_from_this()
               : std::shared_ptr<Map5BuilderImplementation<K1, K2, K3, K4, K5>>(
                     nullptr)) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5> *
xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::impl() {
  return dynamic_cast<Map5BuilderImplementation<K1, K2, K3, K4, K5> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::Map5BuilderImplementation<K1, K2, K3, K4, K5> const *
xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::impl() const {
  return dynamic_cast<Map5BuilderImplementation<K1, K2, K3, K4, K5> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getIterable1()
    const -> xpress::Iterable<K1> const & {
  return impl()->getIterable1();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getIterable2()
    const -> xpress::Iterable<K2> const & {
  return impl()->getIterable2();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getIterable3()
    const -> xpress::Iterable<K3> const & {
  return impl()->getIterable3();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getIterable4()
    const -> xpress::Iterable<K4> const & {
  return impl()->getIterable4();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getIterable5()
    const -> xpress::Iterable<K5> const & {
  return impl()->getIterable5();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getLB() const
    -> std::function<double(K1, K2, K3, K4, K5)> const & {
  return impl()->getLB();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getUB() const
    -> std::function<double(K1, K2, K3, K4, K5)> const & {
  return impl()->getUB();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getLimit() const
    -> std::function<double(K1, K2, K3, K4, K5)> const & {
  return impl()->getLimit();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getName() const
    -> std::function<std::optional<std::string>(K1, K2, K3, K4, K5)> const & {
  return impl()->getName();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getType() const
    -> std::function<xpress::ColumnType(K1, K2, K3, K4, K5)> const & {
  return impl()->getType();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::getFilter() const
    -> std::function<bool(K1, K2, K3, K4, K5)> const & {
  return impl()->getFilter();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::Map5Builder(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
    Iter3 const &iterable4, Iter4 const &iterable5)
    : implPointer(
          std::make_shared<Map5BuilderImplementation<K1, K2, K3, K4, K5>>(
              iterable1, iterable2, iterable3, iterable4, iterable5)) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withLB(
    Func0 newLb) -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withLB(
    double newLb)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withUB(
    Func0 newUb) -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withUB(
    double newUb)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::build(
    XPRSProblem *prob) -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int> {
  return impl()->build(prob);
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::build(
    xpress::objects::XpressProblem *prob)
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, xpress::objects::Variable> {
  return impl()->build(prob);
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::Map5Builder<K1, K2, K3, K4, K5>::Map5Builder()
    : implPointer(nullptr) {}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::
    VariableArrayBuilder(
        std::shared_ptr<VariableArrayBuilderImplementation<C, CIsIntegral>>
            init)
    : implPointer(init) {}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::
    VariableArrayBuilder(
        VariableArrayBuilderImplementation<C, CIsIntegral> const *init)
    : implPointer(
          init ? const_cast<
                     VariableArrayBuilderImplementation<C, CIsIntegral> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<
                     VariableArrayBuilderImplementation<C, CIsIntegral>>(
                     nullptr)) {}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::VariableArrayBuilderImplementation<C, CIsIntegral> *
xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::impl() {
  return dynamic_cast<VariableArrayBuilderImplementation<C, CIsIntegral> *>(
      implPointer.get());
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::VariableArrayBuilderImplementation<C,
                                                            CIsIntegral> const *
xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::impl() const {
  return dynamic_cast<VariableArrayBuilderImplementation<C, CIsIntegral> *>(
      implPointer.get());
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::VariableArrayBuilder<
    C, CIsIntegral>::VariableArrayBuilder(C dim,
                                          xpress::objects::XpressProblem *prob)
    : implPointer(
          std::make_shared<VariableArrayBuilderImplementation<C, CIsIntegral>>(
              dim, prob)) {}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withLB(
    double newLb)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withUB(
    double newUb)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C, typename CIsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C, typename CIsIntegral>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::toArray()
    -> std::vector<xpress::objects::Variable> {
  return impl()->toArray();
}
template <typename C, typename CIsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArrayBuilder<C, CIsIntegral>::toArray(
    Func0 makeResult,
    std::function<void(I, C, xpress::objects::Variable)> addResult) -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C, typename CIsIntegral>
xpress::VariableBuilder::VariableArrayBuilder<
    C, CIsIntegral>::VariableArrayBuilder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                               C2IsIntegral>::
    VariableArray2Builder(std::shared_ptr<VariableArray2BuilderImplementation<
                              C1, C2, C1IsIntegral, C2IsIntegral>>
                              init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                               C2IsIntegral>::
    VariableArray2Builder(VariableArray2BuilderImplementation<
                          C1, C2, C1IsIntegral, C2IsIntegral> const *init)
    : implPointer(init ? const_cast<VariableArray2BuilderImplementation<
                             C1, C2, C1IsIntegral, C2IsIntegral> *>(init)
                             ->shared_from_this()
                       : std::shared_ptr<VariableArray2BuilderImplementation<
                             C1, C2, C1IsIntegral, C2IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral> *
xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                               C2IsIntegral>::impl() {
  return dynamic_cast<VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                                          C2IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::VariableArray2BuilderImplementation<
    C1, C2, C1IsIntegral, C2IsIntegral> const *
xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                               C2IsIntegral>::impl() const {
  return dynamic_cast<VariableArray2BuilderImplementation<C1, C2, C1IsIntegral,
                                                          C2IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral,
    C2IsIntegral>::VariableArray2Builder(C1 dim1, C2 dim2,
                                         xpress::objects::XpressProblem *prob)
    : implPointer(std::make_shared<VariableArray2BuilderImplementation<
                      C1, C2, C1IsIntegral, C2IsIntegral>>(dim1, dim2, prob)) {}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withName(std::optional<std::string>
                                                      newName)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                      C2IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
auto xpress::VariableBuilder::VariableArray2Builder<C1, C2, C1IsIntegral,
                                                    C2IsIntegral>::toArray()
    -> std::vector<std::vector<xpress::objects::Variable>> {
  return impl()->toArray();
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::
    VariableArray2Builder<C1, C2, C1IsIntegral, C2IsIntegral>::toArray(
        Func0 makeResult,
        std::function<void(I, C1, C2, xpress::objects::Variable)> addResult)
        -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C1, typename C2, typename C1IsIntegral,
          typename C2IsIntegral>
xpress::VariableBuilder::VariableArray2Builder<
    C1, C2, C1IsIntegral, C2IsIntegral>::VariableArray2Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::VariableArray3Builder<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral>::
    VariableArray3Builder(
        std::shared_ptr<VariableArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>
            init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::VariableArray3Builder<C1, C2, C3, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral>::
    VariableArray3Builder(
        VariableArray3BuilderImplementation<
            C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const *init)
    : implPointer(
          init ? const_cast<VariableArray3BuilderImplementation<
                     C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
                     init)
                     ->shared_from_this()
               : std::shared_ptr<VariableArray3BuilderImplementation<
                     C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>(
                     nullptr)) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *
xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::impl() {
  return dynamic_cast<VariableArray3BuilderImplementation<
      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::VariableArray3BuilderImplementation<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> const *
xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::impl() const {
  return dynamic_cast<VariableArray3BuilderImplementation<
      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> *>(
      implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::VariableArray3Builder(C1 dim1, C2 dim2, C3 dim3,
                                         xpress::objects::XpressProblem *prob)
    : implPointer(std::make_shared<VariableArray3BuilderImplementation<
                      C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>>(
          dim1, dim2, dim3, prob)) {}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withName(Func0
                                                                        newName)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(Func0
                                                                        newType)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::withType(ColumnType
                                                                        newType)
    -> xpress::VariableBuilder::VariableArray3Builder<
        C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::toArray()
    -> std::vector<std::vector<std::vector<xpress::objects::Variable>>> {
  return impl()->toArray();
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral, C3IsIntegral>::
    toArray(
        Func0 makeResult,
        std::function<void(I, C1, C2, C3, xpress::objects::Variable)> addResult)
        -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C1, typename C2, typename C3, typename C1IsIntegral,
          typename C2IsIntegral, typename C3IsIntegral>
xpress::VariableBuilder::VariableArray3Builder<
    C1, C2, C3, C1IsIntegral, C2IsIntegral,
    C3IsIntegral>::VariableArray3Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    VariableArray4Builder(std::shared_ptr<VariableArray4BuilderImplementation<
                              C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                              C3IsIntegral, C4IsIntegral>>
                              init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    VariableArray4Builder(VariableArray4BuilderImplementation<
                          C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                          C3IsIntegral, C4IsIntegral> const *init)
    : implPointer(init ? const_cast<VariableArray4BuilderImplementation<
                             C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                             C3IsIntegral, C4IsIntegral> *>(init)
                             ->shared_from_this()
                       : std::shared_ptr<VariableArray4BuilderImplementation<
                             C1, C2, C3, C4, C1IsIntegral, C2IsIntegral,
                             C3IsIntegral, C4IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral> *
xpress::VariableBuilder::VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral>::impl() {
  return dynamic_cast<VariableArray4BuilderImplementation<
      C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
                          *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::VariableArray4BuilderImplementation<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral> const *
xpress::VariableBuilder::VariableArray4Builder<C1, C2, C3, C4, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral>::impl() const {
  return dynamic_cast<VariableArray4BuilderImplementation<
      C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
                          *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::VariableArray4Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                         xpress::objects::XpressProblem *prob)
    : implPointer(std::make_shared<VariableArray4BuilderImplementation<
                      C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
                      C4IsIntegral>>(dim1, dim2, dim3, dim4, prob)) {}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableArray4Builder<
        C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>
        & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::toArray()
    -> std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>> {
  return impl()->toArray();
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral>::
    toArray(Func0 makeResult,
            std::function<void(I, C1, C2, C3, C4, xpress::objects::Variable)>
                addResult) -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C1, typename C2, typename C3, typename C4,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral>
xpress::VariableBuilder::VariableArray4Builder<
    C1, C2, C3, C4, C1IsIntegral, C2IsIntegral, C3IsIntegral,
    C4IsIntegral>::VariableArray4Builder()
    : implPointer(nullptr) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral>::
    VariableArray5Builder(std::shared_ptr<VariableArray5BuilderImplementation<
                              C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                              C3IsIntegral, C4IsIntegral, C5IsIntegral>>
                              init)
    : implPointer(init) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::VariableArray5Builder<C1, C2, C3, C4, C5, C1IsIntegral,
                                               C2IsIntegral, C3IsIntegral,
                                               C4IsIntegral, C5IsIntegral>::
    VariableArray5Builder(VariableArray5BuilderImplementation<
                          C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                          C3IsIntegral, C4IsIntegral, C5IsIntegral> const *init)
    : implPointer(
          init ? const_cast<VariableArray5BuilderImplementation<
                     C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                     C3IsIntegral, C4IsIntegral, C5IsIntegral> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<VariableArray5BuilderImplementation<
                     C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                     C3IsIntegral, C4IsIntegral, C5IsIntegral>>(nullptr)) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral> *
xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::impl() {
  return dynamic_cast<VariableArray5BuilderImplementation<
      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
      C4IsIntegral, C5IsIntegral> *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::VariableArray5BuilderImplementation<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral> const *
xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::impl() const {
  return dynamic_cast<VariableArray5BuilderImplementation<
      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
      C4IsIntegral, C5IsIntegral> *>(implPointer.get());
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::VariableArray5Builder(C1 dim1, C2 dim2, C3 dim3, C4 dim4,
                                         C5 dim5,
                                         xpress::objects::XpressProblem *prob)
    : implPointer(std::make_shared<VariableArray5BuilderImplementation<
                      C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral,
                      C3IsIntegral, C4IsIntegral, C5IsIntegral>>(
          dim1, dim2, dim3, dim4, dim5, prob)) {}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withName(std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withName(newName);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::withType(ColumnType newType)
    -> xpress::VariableBuilder::VariableArray5Builder<
        C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral,
        C4IsIntegral, C5IsIntegral> & {
  impl()->withType(newType);
  return *this;
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::toArray()
    -> std::vector<std::vector<
        std::vector<std::vector<std::vector<xpress::objects::Variable>>>>> {
  return impl()->toArray();
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::toArray(Func0 makeResult,
                           std::function<void(I, C1, C2, C3, C4, C5,
                                              xpress::objects::Variable)>
                               addResult) -> I {
  return impl()->toArray(makeResult, addResult);
}
template <typename C1, typename C2, typename C3, typename C4, typename C5,
          typename C1IsIntegral, typename C2IsIntegral, typename C3IsIntegral,
          typename C4IsIntegral, typename C5IsIntegral>
xpress::VariableBuilder::VariableArray5Builder<
    C1, C2, C3, C4, C5, C1IsIntegral, C2IsIntegral, C3IsIntegral, C4IsIntegral,
    C5IsIntegral>::VariableArray5Builder()
    : implPointer(nullptr) {}
template <typename K1>
xpress::VariableBuilder::ColumnMapBuilder<K1>::ColumnMapBuilder(
    std::shared_ptr<ColumnMapBuilderImplementation<K1>> init)
    : implPointer(init) {}
template <typename K1>
xpress::VariableBuilder::ColumnMapBuilder<K1>::ColumnMapBuilder(
    ColumnMapBuilderImplementation<K1> const *init)
    : implPointer(
          init ? const_cast<ColumnMapBuilderImplementation<K1> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ColumnMapBuilderImplementation<K1>>(nullptr)) {
}
template <typename K1>
xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> *
xpress::VariableBuilder::ColumnMapBuilder<K1>::impl() {
  return dynamic_cast<ColumnMapBuilderImplementation<K1> *>(implPointer.get());
}
template <typename K1>
xpress::VariableBuilder::ColumnMapBuilderImplementation<K1> const *
xpress::VariableBuilder::ColumnMapBuilder<K1>::impl() const {
  return dynamic_cast<ColumnMapBuilderImplementation<K1> *>(implPointer.get());
}
template <typename K1>
template <typename Iter0, typename Iter0IsIterable>
xpress::VariableBuilder::ColumnMapBuilder<K1>::ColumnMapBuilder(
    Iter0 const &iterable1, xpress::XPRSProblem *prob)
    : implPointer(std::make_shared<ColumnMapBuilderImplementation<K1>>(
          iterable1, prob)) {}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withLimit(double newLimit)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withType(ColumnType newType)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMapBuilder<K1> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::toMap()
    -> std::unordered_map<K1, int> {
  return impl()->toMap();
}
template <typename K1>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMapBuilder<K1>::toMap(
    Func0 makeResult, std::function<void(I &, K1, int)> addResult) -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1>
xpress::VariableBuilder::ColumnMapBuilder<K1>::ColumnMapBuilder()
    : implPointer(nullptr) {}
template <typename K1, typename K2>
xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::ColumnMap2Builder(
    std::shared_ptr<ColumnMap2BuilderImplementation<K1, K2>> init)
    : implPointer(init) {}
template <typename K1, typename K2>
xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::ColumnMap2Builder(
    ColumnMap2BuilderImplementation<K1, K2> const *init)
    : implPointer(
          init ? const_cast<ColumnMap2BuilderImplementation<K1, K2> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ColumnMap2BuilderImplementation<K1, K2>>(
                     nullptr)) {}
template <typename K1, typename K2>
xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> *
xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::impl() {
  return dynamic_cast<ColumnMap2BuilderImplementation<K1, K2> *>(
      implPointer.get());
}
template <typename K1, typename K2>
xpress::VariableBuilder::ColumnMap2BuilderImplementation<K1, K2> const *
xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::impl() const {
  return dynamic_cast<ColumnMap2BuilderImplementation<K1, K2> *>(
      implPointer.get());
}
template <typename K1, typename K2>
template <typename Iter0, typename Iter1, typename Iter0IsIterable,
          typename Iter1IsIterable>
xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::ColumnMap2Builder(
    Iter0 const &iterable1, Iter1 const &iterable2, xpress::XPRSProblem *prob)
    : implPointer(std::make_shared<ColumnMap2BuilderImplementation<K1, K2>>(
          iterable1, iterable2, prob)) {}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withLB(double newLb)
    -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withUB(double newUb)
    -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withLimit(
    Func0 newLimit) -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withLimit(
    double newLimit) -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withName(Func0 newName)
    -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withType(Func0 newType)
    -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::withFilter(
    Func0 newFilter) -> xpress::VariableBuilder::ColumnMap2Builder<K1, K2> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::toMap()
    -> xpress::maps::HashMap2<K1, K2, int> {
  return impl()->toMap();
}
template <typename K1, typename K2>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::toMap(
    Func0 makeResult, std::function<void(I &, K1, K2, int)> addResult) -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1, typename K2>
xpress::VariableBuilder::ColumnMap2Builder<K1, K2>::ColumnMap2Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::ColumnMap3Builder(
    std::shared_ptr<ColumnMap3BuilderImplementation<K1, K2, K3>> init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::ColumnMap3Builder(
    ColumnMap3BuilderImplementation<K1, K2, K3> const *init)
    : implPointer(
          init ? const_cast<ColumnMap3BuilderImplementation<K1, K2, K3> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<ColumnMap3BuilderImplementation<K1, K2, K3>>(
                     nullptr)) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> *
xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::impl() {
  return dynamic_cast<ColumnMap3BuilderImplementation<K1, K2, K3> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::ColumnMap3BuilderImplementation<K1, K2, K3> const *
xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::impl() const {
  return dynamic_cast<ColumnMap3BuilderImplementation<K1, K2, K3> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3>
template <typename Iter0, typename Iter1, typename Iter2,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable>
xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::ColumnMap3Builder(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
    xpress::XPRSProblem *prob)
    : implPointer(std::make_shared<ColumnMap3BuilderImplementation<K1, K2, K3>>(
          iterable1, iterable2, iterable3, prob)) {}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withLB(
    double newLb) -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withUB(
    double newUb) -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withName(
    Func0 newName) -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withType(
    Func0 newType) -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::toMap()
    -> xpress::maps::HashMap3<K1, K2, K3, int> {
  return impl()->toMap();
}
template <typename K1, typename K2, typename K3>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::toMap(
    Func0 makeResult, std::function<void(I &, K1, K2, K3, int)> addResult)
    -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::ColumnMap3Builder<K1, K2, K3>::ColumnMap3Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::ColumnMap4Builder(
    std::shared_ptr<ColumnMap4BuilderImplementation<K1, K2, K3, K4>> init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::ColumnMap4Builder(
    ColumnMap4BuilderImplementation<K1, K2, K3, K4> const *init)
    : implPointer(
          init
              ? const_cast<ColumnMap4BuilderImplementation<K1, K2, K3, K4> *>(
                    init)
                    ->shared_from_this()
              : std::shared_ptr<
                    ColumnMap4BuilderImplementation<K1, K2, K3, K4>>(nullptr)) {
}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4> *
xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::impl() {
  return dynamic_cast<ColumnMap4BuilderImplementation<K1, K2, K3, K4> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::ColumnMap4BuilderImplementation<K1, K2, K3, K4> const *
xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::impl() const {
  return dynamic_cast<ColumnMap4BuilderImplementation<K1, K2, K3, K4> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::ColumnMap4Builder(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
    Iter3 const &iterable4, xpress::XPRSProblem *prob)
    : implPointer(
          std::make_shared<ColumnMap4BuilderImplementation<K1, K2, K3, K4>>(
              iterable1, iterable2, iterable3, iterable4, prob)) {}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withLB(
    double newLb)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withUB(
    double newUb)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::toMap()
    -> xpress::maps::HashMap4<K1, K2, K3, K4, int> {
  return impl()->toMap();
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::toMap(
    Func0 makeResult, std::function<void(I &, K1, K2, K3, K4, int)> addResult)
    -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::ColumnMap4Builder<K1, K2, K3, K4>::ColumnMap4Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::
    ColumnMap5Builder(
        std::shared_ptr<ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5>>
            init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::
    ColumnMap5Builder(
        ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> const *init)
    : implPointer(
          init
              ? const_cast<
                    ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> *>(init)
                    ->shared_from_this()
              : std::shared_ptr<
                    ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5>>(
                    nullptr)) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> *
xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::impl() {
  return dynamic_cast<ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::ColumnMap5BuilderImplementation<K1, K2, K3, K4,
                                                         K5> const *
xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::impl() const {
  return dynamic_cast<ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
xpress::VariableBuilder::ColumnMap5Builder<
    K1, K2, K3, K4, K5>::ColumnMap5Builder(Iter0 const &iterable1,
                                           Iter1 const &iterable2,
                                           Iter2 const &iterable3,
                                           Iter3 const &iterable4,
                                           Iter4 const &iterable5,
                                           xpress::XPRSProblem *prob)
    : implPointer(
          std::make_shared<ColumnMap5BuilderImplementation<K1, K2, K3, K4, K5>>(
              iterable1, iterable2, iterable3, iterable4, iterable5, prob)) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withLB(
    double newLb)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withUB(
    double newUb)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::toMap()
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, int> {
  return impl()->toMap();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4, K5>::toMap(
    Func0 makeResult,
    std::function<void(I &, K1, K2, K3, K4, K5, int)> addResult) -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::ColumnMap5Builder<K1, K2, K3, K4,
                                           K5>::ColumnMap5Builder()
    : implPointer(nullptr) {}
template <typename K1>
xpress::VariableBuilder::VariableMapBuilder<K1>::VariableMapBuilder(
    std::shared_ptr<VariableMapBuilderImplementation<K1>> init)
    : implPointer(init) {}
template <typename K1>
xpress::VariableBuilder::VariableMapBuilder<K1>::VariableMapBuilder(
    VariableMapBuilderImplementation<K1> const *init)
    : implPointer(init
                      ? const_cast<VariableMapBuilderImplementation<K1> *>(init)
                            ->shared_from_this()
                      : std::shared_ptr<VariableMapBuilderImplementation<K1>>(
                            nullptr)) {}
template <typename K1>
xpress::VariableBuilder::VariableMapBuilderImplementation<K1> *
xpress::VariableBuilder::VariableMapBuilder<K1>::impl() {
  return dynamic_cast<VariableMapBuilderImplementation<K1> *>(
      implPointer.get());
}
template <typename K1>
xpress::VariableBuilder::VariableMapBuilderImplementation<K1> const *
xpress::VariableBuilder::VariableMapBuilder<K1>::impl() const {
  return dynamic_cast<VariableMapBuilderImplementation<K1> *>(
      implPointer.get());
}
template <typename K1>
template <typename Iter0, typename Iter0IsIterable>
xpress::VariableBuilder::VariableMapBuilder<K1>::VariableMapBuilder(
    Iter0 const &iterable1, xpress::objects::XpressProblem *prob)
    : implPointer(std::make_shared<VariableMapBuilderImplementation<K1>>(
          iterable1, prob)) {}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withName(Func0 newName)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withType(Func0 newType)
    -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withType(
    ColumnType newType) -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::withFilter(
    Func0 newFilter) -> xpress::VariableBuilder::VariableMapBuilder<K1> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::toMap()
    -> std::unordered_map<K1, xpress::objects::Variable> {
  return impl()->toMap();
}
template <typename K1>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMapBuilder<K1>::toMap(
    Func0 makeResult,
    std::function<void(I &, K1, xpress::objects::Variable)> addResult) -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1>
xpress::VariableBuilder::VariableMapBuilder<K1>::VariableMapBuilder()
    : implPointer(nullptr) {}
template <typename K1, typename K2>
xpress::VariableBuilder::VariableMap2Builder<K1, K2>::VariableMap2Builder(
    std::shared_ptr<VariableMap2BuilderImplementation<K1, K2>> init)
    : implPointer(init) {}
template <typename K1, typename K2>
xpress::VariableBuilder::VariableMap2Builder<K1, K2>::VariableMap2Builder(
    VariableMap2BuilderImplementation<K1, K2> const *init)
    : implPointer(
          init ? const_cast<VariableMap2BuilderImplementation<K1, K2> *>(init)
                     ->shared_from_this()
               : std::shared_ptr<VariableMap2BuilderImplementation<K1, K2>>(
                     nullptr)) {}
template <typename K1, typename K2>
xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> *
xpress::VariableBuilder::VariableMap2Builder<K1, K2>::impl() {
  return dynamic_cast<VariableMap2BuilderImplementation<K1, K2> *>(
      implPointer.get());
}
template <typename K1, typename K2>
xpress::VariableBuilder::VariableMap2BuilderImplementation<K1, K2> const *
xpress::VariableBuilder::VariableMap2Builder<K1, K2>::impl() const {
  return dynamic_cast<VariableMap2BuilderImplementation<K1, K2> *>(
      implPointer.get());
}
template <typename K1, typename K2>
template <typename Iter0, typename Iter1, typename Iter0IsIterable,
          typename Iter1IsIterable>
xpress::VariableBuilder::VariableMap2Builder<K1, K2>::VariableMap2Builder(
    Iter0 const &iterable1, Iter1 const &iterable2,
    xpress::objects::XpressProblem *prob)
    : implPointer(std::make_shared<VariableMap2BuilderImplementation<K1, K2>>(
          iterable1, iterable2, prob)) {}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withLB(Func0 newLb)
    -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withLB(double newLb)
    -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withUB(Func0 newUb)
    -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withUB(double newUb)
    -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withLimit(
    Func0 newLimit) -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withLimit(
    double newLimit) -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withName(
    Func0 newName) -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withType(
    Func0 newType) -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::withFilter(
    Func0 newFilter) -> xpress::VariableBuilder::VariableMap2Builder<K1, K2> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::toMap()
    -> xpress::maps::HashMap2<K1, K2, xpress::objects::Variable> {
  return impl()->toMap();
}
template <typename K1, typename K2>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap2Builder<K1, K2>::toMap(
    Func0 makeResult,
    std::function<void(I &, K1, K2, xpress::objects::Variable)> addResult)
    -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1, typename K2>
xpress::VariableBuilder::VariableMap2Builder<K1, K2>::VariableMap2Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::VariableMap3Builder(
    std::shared_ptr<VariableMap3BuilderImplementation<K1, K2, K3>> init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::VariableMap3Builder(
    VariableMap3BuilderImplementation<K1, K2, K3> const *init)
    : implPointer(
          init ? const_cast<VariableMap3BuilderImplementation<K1, K2, K3> *>(
                     init)
                     ->shared_from_this()
               : std::shared_ptr<VariableMap3BuilderImplementation<K1, K2, K3>>(
                     nullptr)) {}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3> *
xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::impl() {
  return dynamic_cast<VariableMap3BuilderImplementation<K1, K2, K3> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::VariableMap3BuilderImplementation<K1, K2, K3> const *
xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::impl() const {
  return dynamic_cast<VariableMap3BuilderImplementation<K1, K2, K3> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3>
template <typename Iter0, typename Iter1, typename Iter2,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable>
xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::VariableMap3Builder(
    Iter0 const &iterable1, Iter1 const &iterable2, Iter2 const &iterable3,
    xpress::objects::XpressProblem *prob)
    : implPointer(
          std::make_shared<VariableMap3BuilderImplementation<K1, K2, K3>>(
              iterable1, iterable2, iterable3, prob)) {}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withLB(
    Func0 newLb) -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withLB(
    double newLb)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withUB(
    Func0 newUb) -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withUB(
    double newUb)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::toMap()
    -> xpress::maps::HashMap3<K1, K2, K3, xpress::objects::Variable> {
  return impl()->toMap();
}
template <typename K1, typename K2, typename K3>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::toMap(
    Func0 makeResult,
    std::function<void(I &, K1, K2, K3, xpress::objects::Variable)> addResult)
    -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1, typename K2, typename K3>
xpress::VariableBuilder::VariableMap3Builder<K1, K2, K3>::VariableMap3Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::
    VariableMap4Builder(
        std::shared_ptr<VariableMap4BuilderImplementation<K1, K2, K3, K4>> init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::
    VariableMap4Builder(
        VariableMap4BuilderImplementation<K1, K2, K3, K4> const *init)
    : implPointer(
          init
              ? const_cast<VariableMap4BuilderImplementation<K1, K2, K3, K4> *>(
                    init)
                    ->shared_from_this()
              : std::shared_ptr<
                    VariableMap4BuilderImplementation<K1, K2, K3, K4>>(
                    nullptr)) {}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3, K4> *
xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::impl() {
  return dynamic_cast<VariableMap4BuilderImplementation<K1, K2, K3, K4> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::VariableMap4BuilderImplementation<K1, K2, K3,
                                                           K4> const *
xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::impl() const {
  return dynamic_cast<VariableMap4BuilderImplementation<K1, K2, K3, K4> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable>
xpress::VariableBuilder::VariableMap4Builder<
    K1, K2, K3, K4>::VariableMap4Builder(Iter0 const &iterable1,
                                         Iter1 const &iterable2,
                                         Iter2 const &iterable3,
                                         Iter3 const &iterable4,
                                         xpress::objects::XpressProblem *prob)
    : implPointer(
          std::make_shared<VariableMap4BuilderImplementation<K1, K2, K3, K4>>(
              iterable1, iterable2, iterable3, iterable4, prob)) {}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withLB(
    double newLb)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withUB(
    double newUb)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withLimit(
    Func0 newLimit)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withLimit(
    double newLimit)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::withFilter(
    Func0 newFilter)
    -> xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::toMap()
    -> xpress::maps::HashMap4<K1, K2, K3, K4, xpress::objects::Variable> {
  return impl()->toMap();
}
template <typename K1, typename K2, typename K3, typename K4>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3, K4>::toMap(
    Func0 makeResult,
    std::function<void(I &, K1, K2, K3, K4, xpress::objects::Variable)>
        addResult) -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1, typename K2, typename K3, typename K4>
xpress::VariableBuilder::VariableMap4Builder<K1, K2, K3,
                                             K4>::VariableMap4Builder()
    : implPointer(nullptr) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::
    VariableMap5Builder(
        std::shared_ptr<VariableMap5BuilderImplementation<K1, K2, K3, K4, K5>>
            init)
    : implPointer(init) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::
    VariableMap5Builder(
        VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> const *init)
    : implPointer(
          init ? const_cast<
                     VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> *>(
                     init)
                     ->shared_from_this()
               : std::shared_ptr<
                     VariableMap5BuilderImplementation<K1, K2, K3, K4, K5>>(
                     nullptr)) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> *
xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::impl() {
  return dynamic_cast<VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::VariableMap5BuilderImplementation<K1, K2, K3, K4,
                                                           K5> const *
xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::impl() const {
  return dynamic_cast<VariableMap5BuilderImplementation<K1, K2, K3, K4, K5> *>(
      implPointer.get());
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Iter0, typename Iter1, typename Iter2, typename Iter3,
          typename Iter4, typename Iter0IsIterable, typename Iter1IsIterable,
          typename Iter2IsIterable, typename Iter3IsIterable,
          typename Iter4IsIterable>
xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::
    VariableMap5Builder(Iter0 const &iterable1, Iter1 const &iterable2,
                        Iter2 const &iterable3, Iter3 const &iterable4,
                        Iter4 const &iterable5,
                        xpress::objects::XpressProblem *prob)
    : implPointer(std::make_shared<
                  VariableMap5BuilderImplementation<K1, K2, K3, K4, K5>>(
          iterable1, iterable2, iterable3, iterable4, iterable5, prob)) {}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::withLB(
    Func0 newLb)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::withLB(
    double newLb)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLB(newLb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::withUB(
    Func0 newUb)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::withUB(
    double newUb)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withUB(newUb);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4,
                                                  K5>::withLimit(Func0 newLimit)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5Builder<
    K1, K2, K3, K4, K5>::withLimit(double newLimit)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withLimit(newLimit);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::withName(
    Func0 newName)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::withName(
    std::optional<std::string> newName)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withName(newName);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::withType(
    Func0 newType)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::withType(
    ColumnType newType)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withType(newType);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5Builder<
    K1, K2, K3, K4, K5>::withFilter(Func0 newFilter)
    -> xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5> & {
  impl()->withFilter(newFilter);
  return *this;
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::toMap()
    -> xpress::maps::HashMap5<K1, K2, K3, K4, K5, xpress::objects::Variable> {
  return impl()->toMap();
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
template <typename I, typename Func0, typename Func0IsInvocable>
auto xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4, K5>::toMap(
    Func0 makeResult,
    std::function<void(I &, K1, K2, K3, K4, K5, xpress::objects::Variable)>
        addResult) -> I {
  return impl()->toMap(makeResult, addResult);
}
template <typename K1, typename K2, typename K3, typename K4, typename K5>
xpress::VariableBuilder::VariableMap5Builder<K1, K2, K3, K4,
                                             K5>::VariableMap5Builder()
    : implPointer(nullptr) {}
#ifndef XPRESS_DO_NOT_INCLUDE_CXX_SOURCE
#include "xpress.cpp"
#endif
#endif
